/***************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,        *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.   *
 *                                                                         *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael          *
 *  Chastain, Michael Quan, and Mitchell Tse.                              *
 *                                                                         *
 *  In order to use any part of this Merc Diku Mud, you must comply with   *
 *  both the original Diku license in 'license.doc' as well the Merc       *
 *  license in 'license.txt'.  In particular, you may not remove either of *
 *  these copyright notices.                                               *
 *                                                                         *
 *  Much time and thought has gone into this software and you are          *
 *  benefitting.  We hope that you share your changes too.  What goes      *
 *  around, comes around.                                                  *
 ***************************************************************************/

#if defined(macintosh)
#include <types.h>
#else
#include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "mud.h"
#include "skills.h"
#include "script.h"
#include "fight.h"
#include "defaults.h"




#define FGT_HIT        1
#define FGT_WOUNDED    2
#define FGT_STUNNED    4
#define FGT_CMP_HIT    8
#define FGT_DODGE     16
#define FGT_PARRY     32
#define FGT_PRIMARY   64





struct fight_mode
{
    const char * name;
    int tohit;
    int dodge;
    int todam;
};

const struct fight_mode fight_mode_table[] =
{
    { "elusive",       -15, 15, -5 },
    { "cautious",      -10, 10,  0 },
    { "normal",          0,  0,  0 },
    { "bold",           10,-10,  0 },
    { "berserk",        15,-15,  5 },
    { "",                0,  0,  0 }
};

const int combat_int[26] =
{ 0,                         /* 0 */
  0,
  0,
/*
 * Creatures with int > 2 know when they've been injured
 */
  FGT_HIT,
  FGT_HIT,
/*
 * Creatures with int > 4 know when they've been seriously injured
 */
  FGT_HIT|FGT_WOUNDED,   /* 5 */
  FGT_HIT|FGT_WOUNDED,
  FGT_HIT|FGT_WOUNDED,
  FGT_HIT|FGT_WOUNDED,
/*
 * Creatures with int > 8 know when their opponent is unable to
 * attack them
 */
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED,   /* 10 */
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED,
/*
 * Creatures with int > 12 know when to change combat tactic due to damage
 */
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,   /* 15 */
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT, /* 20 */
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT,
  FGT_HIT|FGT_WOUNDED|FGT_STUNNED|FGT_CMP_HIT /* 25 */
};

    

/*
 * Local functions.
 */

bool flee               args( ( CHAR_DATA *ch ) );
void retreat            args( ( CHAR_DATA *ch ) );
bool disarm             args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
bool bash               args( ( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed ) );
bool grapple            args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
bool kick               args( ( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed ) );
bool sweep              args( ( CHAR_DATA *ch ) );
bool trip               args( ( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed ) );
int  number_fighting    args( ( CHAR_DATA *ch ) );
void auto_fight		args( ( CHAR_DATA *ch, CHAR_DATA *victim) );
void auto_flee		args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
bool armed_attack	args( ( CHAR_DATA *ch, CHAR_DATA *victim) );
bool unarmed_attack	args( ( CHAR_DATA *ch, CHAR_DATA *victim) );


int fight_mode_lookup( const char *name )
{
    int sn;

    if ( name == NULL ) return -1;

    for (sn = 0; fight_mode_table[sn].name[0] != '\0'; sn++ )
    {
	if ( LOWER(name[0]) == LOWER(fight_mode_table[sn].name[0])
	&&   !str_prefix( name, fight_mode_table[sn].name ) )
	    return sn;
    }

    return -1;
}

void MSG_weapon_hit( CHAR_DATA *ch, OBJ_DATA *obj, CHAR_DATA *vi, int dam )
{
    char w[MAX_STRING_LENGTH];
    char *i;

    if ( obj->item_type != ITEM_WEAPON )
    {
        bug( "MSG_weapon_hit: Supplied object is not a weapon.", 0 );
        return;
    }

    sprintf( w, "%s", smash_article(STR(obj,short_descr)) );
    i = "$n successfully hit$v $N with $s $t.";

    switch ( obj->value[3] )
    {
    default:
    if ( obj->value[3] > MAX_ATTACK )
    bug("MSG_weapon_hit: Invalid attack (vnum %d).", obj->pIndexData->vnum);

    case 0:  /* "hit" */
    if ( dam >= 100 ) i = "$n massacre$v $N with $s powerful hit!";
    else
    if ( dam < 5 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n$y $t barely grazes $N.";                                         break;
        case 1: i = "$n$y badly placed hit merely scratches $N.";                        break;
        case 2: i = "$n attack$v $N with $s $t, but it does little damage.";             break;
        case 3: i = "$N $K clipped by $n$y $t.";                                         break;
        case 4: i = "$n hit$v $N lightly.";                                              break;
        case 5: i = "$n manage$v to hit $N, though the damage is slight.";               break;
        case 6: i = "The $t that $n $k wielding hardly hits $N at all.";                 break;
        case 7: i = "$N$Y $K distracted, but $n fail$v to hit $M with any real force.";  break;
        case 8: i = "$n pounce$v on $N$Y weak spot, but $e fail$v to do much damage.";   break;
        case 9: i = "$n only skim$v $n$y weapon across $N$Y body.";                      break;
        }
    }
    else
    if ( dam < 40 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n hit$v $N with $s $t.";                                                break;
        case 1: i = "$N fail$V to escape $n$y $t.";                                           break;
        case 2: i = "$n hit$v $N.";                                                           break;
        case 3: i = "$N $K unable to dodge $n$y attack; $e hurt$v $M.";                       break;
        case 4: i = "Having surprised $N, $n bound$v forward and hit$v $M.";                  break;
        case 5: i = "$n lunge$v at $N with $s $t at the ready, hitting $M.";                  break;
        case 6: i = "$N $K hit by $n$y $t when $e attack$v.";                                 break;
        case 7: i = "$n hit$v $N.";                                                           break;
        case 8: i = "$N $K suddenly hit by $n$y $t.";                                         break;
        case 9: i = "With $s $t, $n succeed$v to harm $N.";                                   break;
        }
    }
    else
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n$y $t shakes $N with a powerful hit.";                                 break;
        case 1: i = "$N contort$V in pain as $n$y $t connects with $M.";                      break;
        case 2: i = "$n deal$v a heavy blow to $N with $s $t.";                               break;
        case 3: i = "$N $K overcome with a pained look as $n hit$v $M.";                      break;
        case 4: i = "$N $K surprised by $n$y ferocious attack.";                              break;
        case 5: i = "$n lunge$v at $N, dealing a heavy blow with $s $t.";                     break;
        case 6: i = "Raising $s weapon, $n hit$v $N with the full force of $s attack.";       break;
        case 7: i = "$N double$V over with the strength of $n$y attack!";                     break;
        case 8: i = "$n quite spontaneously leap$v forward, hitting $N with $s $t!";          break;
        case 9: i = "With skill and finesse, $N offer$V $Mself to $n$y awaiting $t.";         break;
        }
    }
    break;
    case 1: /* "slice" */
    if ( dam >= 100 )
    {
        switch( number_range(0, 4) )
        {
        case 0: i = "$n shred$v $N with $s powerful slice!";                                   break;
        case 1: i = "$n swing$v $s $t, bisecting $N$Y body with incredible feriocity!";        break;
        case 2: i = "$N $K vivisected by $n$y $t!";                                           break;
        case 3: i = "$n slice$v $N$Y body to shreds with $s $t!";                 break;
        case 4: i = "$n$y $t slices through $N$Y vital organs!";           break;
        }
    }
    else
    if ( dam < 5 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n$y $t barely slices $N.";                                              break;
        case 1: i = "$n$y badly placed attack merely scratches $N.";                          break;
        case 2: i = "$n slice$v $N with $s $t, but it does little damage.";                   break;
        case 3: i = "$n$y $t slices away a small piece of $N$Y flesh.";                       break;
        case 4: i = "$n leave$v a small wound in $N with $s $t.";                             break;
        case 5: i = "$n manage$v to slice $N, though the damage is slight.";                  break;
        case 6: i = "$N seem$V unaffected by the slice $n$y $t has left.";                    break;
        case 7: i = "$N $K caught off guard, but $n fail$v to use $s $t effectively.";        break;
        case 8: i = "$N$Y flesh is only slightly marred by $n$y slice.";                      break;
        case 9: i = "$n swing$v $n$y $t short, leaving only a scrape along $N$Y body.";       break;
        }
    }
    else
    if ( dam < 40 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n make$v a jagged rocking motion with $s $t, tearing $N$Y flesh!";      break;
        case 1: i = "$n$y $t leaves a long, curved wound on $N$Y body.";                      break;
        case 2: i = "With $s $t, $n slice$v $N gruesomely.";                                  break;
        case 3: i = "$N $K unable to avoid $n$y attack; $e slice$v $M.";                      break;
        case 4: i = "With a skillful execution, $n rend$v $N$Y flesh with $s $t.";            break;
        case 5: i = "The blade of $n$y $t slides smoothly along $N$Y body.";                  break;
        case 6: i = "$n$y $t leaves a small wound in $N$Y flesh.";                            break;
        case 7: i = "$n bring$v $s weapon's blade in contact with $N!";                       break;
        case 8: i = "Lunging forward, $n manage$v to slice $N.";                              break;
        case 9: i = "$N lose$V a bit of flesh to $n$y deadly $t!";                            break;
        }
    }
    else
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n tear$v a chunk of flesh from $N with $s $t!";                         break;
        case 1: i = "$N shriek$V as $n$y $t connects with $S flailing body!";                 break;
        case 2: i = "$n deal$v a mighty wound to $N with $s $t.";                             break;
        case 3: i = "$N $K momentarily stunned as $n$y $t streaks across $S flesh.";          break;
        case 4: i = "$N $K about to duck $n$y attack when $s $t connects!";                   break;
        case 5: i = "$n lunge$v at $N, slicing away with deadly accuracy at $N$Y frame.";     break;
        case 6: i = "$n swing$v $s $t in a lethal arc, rending $N$Y flesh horridly!";         break;
        case 7: i = "$N frantically reach$W for the wound that $n$y $t created.";             break;
        case 8: i = "$n suddenly find$v $s $t buried in $N.";                                 break;
        case 9: i = "$n slice$v easily through the hard and soft parts of $N$Y anatomy.";     break;
        }
    }
    break;
    case 2: /* "stab" */
    if ( dam >= 100 )
    {
        switch( number_range(0, 4) )
        {
        case 0: i = "$n perforate$v $N with a series of lightning stabs!";              break;
        case 1: i = "$n bur$x $s $t into $N, slaughtering $M!";                     break;
        case 2: i = "$n rearrange$v $N$Y anatomy with $s final stabs!";                       break;
        case 3: i = "With a vicious stab, $n$y $t punctures $N$Y body!";                                 break;
        case 4: i = "$N is impaled on $n$y weapon!";                                          break;
        }
    }
    else
    if ( dam < 5 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n stab$v $N, but fail$v to penetrate deeply.";                          break;
        case 1: i = "$n arc$v $s weapon a trifle high, missing $N$Y vitals.";                 break;
        case 2: i = "$n drive$v $s $t toward $N, stabbing $M only slightly.";                 break;
        case 3: i = "Fumbling with $s $t, $n ineptly graze$v $N.";                            break;
        case 4: i = "$n$y $t meets heavy resistance from $N$Y body.";                         break;
        case 5: i = "$N sidestep$V, but $n$y attack hits $M lightly anyway.";                  break;
        case 6: i = "The $t $n $k wielding only scratches $N.";                               break;
        case 7: i = "Using only the tip of $s $t, $n lightly cut$v $N.";                      break;
        case 8: i = "$N$Y flesh is only slightly marred by $n$y slice.";                      break;
        case 9: i = "$n swing$v $n$y $t short, leaving only a scrape along $N$Y body.";       break;
        }
    }
    else
    if ( dam < 40 )
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n bur$x $s $t up to $s hand in $N$Y body!";                             break;
        case 1: i = "With $s $t, $n open$v a gaping wound in $N.";                            break;
        case 2: i = "$n stab$v $N gruesomely.";                                               break;
        case 3: i = "$n leave$v a gory wound in $N with $s attack.";                          break;
        case 4: i = "$n poke$v a large hole in $N$Y flesh!";                                  break;
        case 5: i = "$N suddenly find$V $n$y $t tearing through $S vitals!";                  break;
        case 6: i = "$n stab$v $N with $s $t!";                                               break;
        case 7: i = "$n stick$v $s weapon far into $N.";                                      break;
        case 8: i = "With a sudden lunge, $n drive$v $s $t into $N$Y awaiting flesh!";          break;
        case 9: i = "$n$y $t tears a bit of flesh from $N, leaving behind a sickly wound!";   break;
        }
    }
    else
    {
        switch( number_range(0, 9) )
        {
        case 0: i = "$n suddenly find$v $s $t covered in $N$Y bodily fluids.";                break;
        case 1: i = "$N let$V out a small scream as $n stab$v $M!";                           break;
        case 2: i = "$n whirl$v $s weapon in the air, then wound$v $N with vicious ferver!";  break;
        case 3: i = "$N $K taken by surprise as $n tr$x to shred $M to pieces!";              break;
        case 4: i = "$n fumble$v with $s $t, but manage$v to eviscerate $N anyway!";          break;
        case 5: i = "$N meet$V $n$y attack with full force.";                                 break;
        case 6: i = "In a powerful maneuver, $n arc$v $s weapon perfectly into $N$Y vitals!"; break;
        case 7: i = "A sickening sound fills your ears as $n horribly stab$v $N.";            break;
        case 8: i = "$n suddenly find$v $s $t buried in $N.";                                 break;
        case 9: i = "With grotesque ease, $n slides $s $t into $N$Y gullet.";                 break;
        }
    }
    break;
    }
    act( i, ch, w, vi, TO_ALL );
    return;
}



void MSG_weapon_partial_dodge( CHAR_DATA *ch, OBJ_DATA*obj, CHAR_DATA *victim )
{
/*    bug( "MSG_weapon_partial_dodge: msg",0 );    */
    return;
}



void MSG_weapon_parry( CHAR_DATA *ch, OBJ_DATA *obj, OBJ_DATA *vo,
                       CHAR_DATA *victim )
{
    char w[MAX_STRING_LENGTH];
    char *i = NULL;

    sprintf( w, "%s", smash_article(STR(vo,short_descr)) );
    switch( number_range(0,7) )
    {
    case 0: i = "$N parr$X $n$y attack with $S $t."; break;
    case 1: i = "$N block$V $n$y strike with $S $t."; break;
    case 2: i = "$N bring$V $S $t up to parry $n$y blow."; break;
    case 3: i = "$N lift$V $S $t and manage$V to parry $S strike."; break;
    case 4: i = "$n$y attack is brushed aside by $N$Y parry."; break;
    case 5: i = "$N deflect$V $n$y attack with $S $t."; break;
    case 6: i = "$n$y blow glances harmlessly off from $N$Y $t."; break;
    case 7: i = "$n$y strike thuds into $N$Y $t, causing no damage."; break;
    }
    act( i, ch, w, victim, TO_ALL );
    return;
}



void MSG_weapon_dodge( CHAR_DATA *ch, OBJ_DATA *obj, CHAR_DATA *victim )
{
    char w[MAX_STRING_LENGTH];
    char *i = NULL;

    sprintf( w, "%s", smash_article(STR(obj,short_descr)) );
    switch(number_range(0,7))
    {
	case 0: i = "$N dodge$V $n$y $t."; break;
	case 1: i = "$N feint$V and manage$V to avoid being hit by $n."; break;
	case 2: i = "$N step$V back just in time to avoid $n$y blow."; break;
	case 3: i = "$N leap$V to the left, barely dodging $n$y $t."; break;
	case 4: i = "$N leap$V to the right, avoiding $n$y strike."; break;
	case 5: i = "$N duck$V under $n$y $t."; break;
	case 6: i = "$N swerve$V to the left, avoiding easily $n$y $t."; break;
	case 7: i = "$N swerve$V to the right, dodging $n$y $t with little effort."; break;
    }
    act( i, ch, w, victim, TO_ALL );
    return;
}

void MSG_weapon_miss( CHAR_DATA *ch, OBJ_DATA *obj, CHAR_DATA *victim )
{
    char w[MAX_STRING_LENGTH];
    char *i = NULL;

    sprintf( w, "%s", smash_article(STR(obj,short_descr)) );
    switch(number_range(0,9))
    {
        case 0: i = "$n swing$v $s $t wide and miss$w $N."; break;
	case 1: i = "$n$y haphazard lunge misses $N."; break;
	case 2: i = "$n thrust$v $s $t at $N but miss$w."; break;
	case 3: i = "$n$y badly aimed attack misses $N."; break;
	case 4: i = "$n$y feeble strike misses $N."; break;
	case 5: i = "$n nearly lose$v $s balance, missing $s blow."; break;
	case 6: i = "$n stretch$w forward to strike and fall$v out of balance."; break;
	case 7: i = "$n$y blow falls short of $N by several duspan."; break;
        case 8: i = "$n$y strike connects only thin air."; break;
        case 9: i = "$n find$v $mself in an awkward position and misses."; break;
    }
    act( i, ch, w, victim, TO_ALL );
    return;
}



void MSG_mobile_hit( CHAR_DATA *ch, int idx, CHAR_DATA *victim, int dam )
{
/*
    char *i = NULL;

    switch( idx )
    {
        case 0:
    { "hit",      TYPE_HIT,     &gsn_wp_other,   NULL              },
    { "slice",    TYPE_SLASH,   &gsn_wp_slash,   NULL              },
    { "stab",     TYPE_PIERCE,  &gsn_wp_pierce,  NULL              },
    { "slash",    TYPE_SLASH,   &gsn_wp_slash,   NULL              },
    { "whip",     TYPE_WHIP,    &gsn_wp_whip,    NULL              },
    { "cleave",   TYPE_SLASH,   &gsn_wp_slash,   NULL              },
    { "vorpal",   TYPE_SLASH,   &gsn_wp_slash,   hit_vorpal        },
    { "pound",    TYPE_POUND,   &gsn_wp_pound,   NULL              },
    { "pierce",   TYPE_PIERCE,  &gsn_wp_pierce,  NULL              },
    { "lash",     TYPE_WHIP,    &gsn_wp_whip,    NULL              },
    { "claw",     TYPE_SLASH,   &gsn_wp_slash,   NULL              },
    { "blast",    TYPE_EXPLODE, &gsn_wp_other,   NULL              },
    { "crush",    TYPE_POUND,   &gsn_wp_pound,   NULL              },
    { "gore",     TYPE_PIERCE,  &gsn_wp_pierce,  NULL              },
    { "bite",     TYPE_PIERCE,  &gsn_wp_pierce,  NULL              },
    { "suction",  TYPE_SUCTION, &gsn_wp_other,   hit_suck_disarm   },
      bug( "MSG_mobile_hit: msg",0 );              */
    return;
}



void MSG_partial_dodge( CHAR_DATA *ch, int idx, CHAR_DATA *victim )
{
/*    bug( "MSG_partial_dodge: msg", 0 ); */
    return;
}

void MSG_hand_hit( CHAR_DATA *ch, CHAR_DATA *victim, int dam )
{
    char *i = NULL;

    if ( dam >= 100 ) i = "$n slay$v $N with $s powerful hit!";
    else
    if ( dam < 5 )
    {
        switch( number_range(0, 9) )
        {
	case 0: i = "$n$y punch barely grazes $N.";                                         break;
	case 1: i = "$n$y badly placed punch thuds harmlessly on $N.";                        break;
	case 2: i = "$n punch$w $N, but does little damage.";             break;
	case 3: i = "$n$y punch grazes $N, but the damage is slight.";                                         break;
	case 4: i = "$n punch$w $N lightly.";                                              break;
	case 5: i = "$n manage$v to punch $N, though the damage is slight.";               break;
	case 6: i = "$n$y feeble punch hardly hits $N at all.";                 break;
	case 7: i = "$N$Y $K distracted, but $n fail$v to do $M with any real damage.";  break;
	case 8: i = "$n punch$w $N$Y in a weak spot, but fail$v to do much damage.";   break;
	case 9: i = "$n$y pound connects $N, but without visible effect.";                      break;
	}
    }
    else
    if ( dam < 12 )
    {
        switch( number_range(0, 9) )
        {
	case 0: i = "$n punch$w $N viciously.";                                                break;
	case 1: i = "$N fail$V to escape $n$y hard uppercut.";                                           break;
	case 2: i = "$n pound$v $N.";                                                           break;
	case 3: i = "$N $K unable to dodge $n$y attack; $e hurt$v $M.";                       break;
	case 4: i = "Having surprised $N, $n step$v forward and punch$w $M.";                  break;
	case 5: i = "$n$y left hook thuds into $N.";                  break;
	case 6: i = "$N $K winded by $n$y right hook.";                                 break;
	case 7: i = "$n$y hard straight connects $N.";                                                           break;
	case 8: i = "$N $K suddenly hit by $n$y fast jab.";                                         break;
	case 9: i = "$n$y sudden knee jerk bruises $N.";                                   break;
        }
    }
    else
    {
        switch( number_range(0, 9) )
        {
	case 0: i = "$n$y fast left thuds painfully into $N.";                                 break;
	case 1: i = "$N contort$V in pain as $n$y uppercut connects with $M.";                      break;
	case 2: i = "$n deal$v a heavy right hook to $N.";                               break;
	case 3: i = "$N $K overcome with a pained look as $n punch$w $M.";                      break;
	case 4: i = "$N $K surprised by $n$y ferocious attack.";                              break;
	case 5: i = "$n feint$v and throw$v a forceful blow into $N.";                     break;
	case 6: i = "$N groan$V in pain as $n$y heavy punch hits $M.";       break;
	case 7: i = "$N double$V over with the strength of $n$y hard jab!";                     break;
	case 8: i = "$n quite spontaneously leap$v forward, delivering a knee jerk into $N$Y groin!";          break;
	case 9: i = "$N leap$v forward, straight into $n$y outstretched fist!";         break;
        }
    }
    act( i, ch, NULL, victim, TO_ALL );
    return;
}

void MSG_hand_miss( CHAR_DATA *ch, CHAR_DATA *victim )
{
    char *i = NULL;
    switch(number_range(0,7))
    {
	case 0: i = "$n swing$v $s uppercut wide and miss$w $N."; break;
	case 1: i = "$n$y haphazard straight miss$w $N."; break;
	case 2: i = "$n jab$v at $N but miss$w."; break;
	case 3: i = "$n$y badly aimed attack misses $N."; break;
	case 4: i = "$n$y feeble punch misses $N."; break;
	case 5: i = "$n nearly lose$v $s balance!"; break;
	case 6: i = "$n lean$v forward to punch and nearly fall$v on $s nose."; break;
	case 7: i = "$n$y blow falls short of $N by several duspan."; break;
    }
    act( i, ch, NULL, victim, TO_ALL );
    return;
}

void MSG_dodge( CHAR_DATA *ch, int idx, CHAR_DATA *victim )
{
    const char *i;
    i = "$N dodge$V $n$y attack.";
    act( i, ch, NULL, victim, TO_ALL );
    return;
}



/*
 * Control the fights going on.
 * Called periodically by update_handler.
 */
void violence_update( void )
{
    CHAR_DATA *ch;
    CHAR_DATA *ch_next;
    CHAR_DATA *victim;

    for ( ch = char_list; ch != NULL; ch = ch->next )
    {
        ch_next = ch->next;

        ch->fbits = 0;

        if ( ( victim = ch->fighting ) == NULL || ch->in_room == NULL )
            continue;
 
	if ( ch->position < POS_SLEEPING )
	{
	    stop_fighting( ch, FALSE );
	    continue;
	}

	else if ( ch->position < POS_FIGHTING )
	{
	    if ( ch->position == POS_SLEEPING && !IS_AFFECTED(ch, AFF_SLEEP) )
	    	send_to_char( "You jolt awake and rejoin the fray.\n\r", ch );
	    do_stand( ch, "" );
	}

        if ( IS_NPC(ch) )
        {
            if( get_eq_char( ch, WEAR_WIELD_1 ) == NULL
             && get_eq_char( ch, WEAR_WIELD_2 ) == NULL )
            do_draw( ch, "" );
	    /* If disarmed, try to find a weapon? */
        }

        script_update( ch, TYPE_MOB,           TRIG_COMBAT, victim, NULL, NULL, NULL );
        script_update( ch->in_room, TYPE_ROOM, TRIG_COMBAT, ch,     NULL, NULL, NULL );
        trigger_list( ch->carrying,            TRIG_COMBAT, victim, NULL, NULL, NULL );
        trigger_list( ch->in_room->contents,   TRIG_COMBAT, victim, NULL, NULL, NULL );

        /* No actions for chars in WAIT_STATE. Handle wait for mobiles */
	if (ch->wait > 0)
        {
	    if ( ch->desc == NULL )
	       ch->wait -= PULSE_VIOLENCE;
	    continue;
        }

        if ( IS_AWAKE(ch)
          && ch->in_room == victim->in_room
	  && ch != victim )
	{
	    /* Exchange blows */
	    oroc( ch, victim );

	    /* Maybe change combat tactic? */
	    if ( IS_NPC(ch) )
		auto_fight( ch, victim );

	    /* Maybe flee? -- This has been moved here from damage(),
	     * because fleeing may cause damage to the victim and we 
	     * don't want to be caught in an infinite loop
	     */
	    if ( IS_NPC(victim) )
		auto_flee( ch, victim );
	}
        else
            stop_fighting( ch, FALSE );
    }

    return;
}



/*
 * Do one group of attacks.
 * (One Round One Char)
 */
void oroc( CHAR_DATA *ch, CHAR_DATA *victim )
{
    OBJ_DATA *primary_wpn, *secondary_wpn;

    primary_wpn = get_eq_char( ch, WEAR_WIELD_1 );
    secondary_wpn = get_eq_char( ch, WEAR_WIELD_2 );

    weapon_attack( ch, primary_wpn, victim );

    /*
     * Weapon_attack() does a hand attack if no weapon found. However,
     * we want to make a dual wield attack only if ch is actually
     * wielding a weapon, so we must make a check here...
     * Also, if the victim died, there's no point attacking...
     */
    if (  ch->fighting != NULL
     && skill_check( ch, gsn_dual_wield, 25 )
     && secondary_wpn != NULL
     && secondary_wpn->item_type == ITEM_WEAPON )
    	weapon_attack( ch, secondary_wpn, victim );

    if ( ch->fighting != NULL
      && skill_check( ch, gsn_second_attack, 40 )
      && !IS_SET(ch->fbits, FGT_PARRY) )
    	weapon_attack( ch, primary_wpn, victim );

    if ( ch->fighting != NULL
      && skill_check( ch, gsn_third_attack, 65 )
      && !IS_SET(ch->fbits, FGT_PARRY) )
    	weapon_attack( ch, primary_wpn, victim );
    
    if ( IS_NPC(ch) && ch->fighting != NULL )
	mobile_attack( ch, victim );

    return;
}

/*----------------------
 * Attack modifier funcs
 *----------------------*/

/*
 * Calculates effects of hit to the armor: armor wear and
 * deflection. Note that beat_armor() has been moved here (it's
 * more logical that the wearing out occurs simultaneously to
 * the blow...
 */
int armor_deflect( CHAR_DATA *victim, int idx )
{

    OBJ_DATA *armor;
    int deflect;

    /* Add Check for Material Later */

    armor = get_eq_char( victim, number_range( 0, MAX_WEAR-1 ) );

    if ( armor != NULL && armor->item_type == ITEM_ARMOR )
    {
    	if ( armor->value[1] > 0
          && armor->value[2] > 0 )
           --armor->value[1];

    	/* Gradually worsen the armor condition... */
    	if ( armor->value[1] == 0 )
        armor->value[1] = (--armor->value[0] > 0) ? armor->value[2] : -1;
    }
    /*
     * INTERPOLATE() does not work, because victim_ac can be negative
     * Here we convert victim_ac to a percentage value and reverse
     * it so that 0 (no deflection) - 100 (complete deflection)
     */

    deflect = 100 - (GET_AC(victim) + 100)/2 + (15-number_range(0, 30));
    return( URANGE(0,deflect,100) );
}

/*
 * Calculates whether a hit really connects the victim
 */
int chance_to_hit( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int chance;
    chance = 25;

    /*
     * Size adjustment.
     */
    if ( ch->size + 2 < victim->size
    && (ch->size > 0 && victim->size > 0) )
    chance += 10;

    /*
     * STR vs. DEX adjustment
     */
    if ( get_curr_str( ch ) > get_curr_dex( victim ) )
	chance += 5;

    /*
     * Fight mode adjustment
     * Victim position adjustment
     */
    chance += fight_mode_table[ch->fmode].tohit;
    if ( victim->position < POS_FIGHTING )
	chance += 40;
    else
       chance -= fight_mode_table[victim->fmode].dodge;

    /*
     * Drunkeness adjustment
     */
    if ( !IS_NPC(ch) )
	chance -= PC(ch, condition)[COND_DRUNK];
    if ( !IS_NPC(victim) && PC(victim, condition)[COND_DRUNK] > 0 )
	chance += PC(victim, condition)[COND_DRUNK];
    /*
     * Blindness/invis adjustment
     */
    if ( !can_see(ch,victim) )
	chance -= 40;

    /*
     * Health adjustment.
     * Hurt characters have less chance to hit, and are hit easier.
     */
    chance += ((PERCENTAGE(ch->hit, MAXHIT(ch))+25)/2)-25;
    chance += ((PERCENTAGE(victim->hit, MAXHIT(victim))+25)/2)-25;

    return( chance );
}

/*
 * Calculates modifiers to the damage
 */
int damage_adjust( CHAR_DATA *ch, int dam, int deflected )
{
    /*
     * Deflected is again the percentage of damage deflected,
     * so to find out damage applied we must subtract the
     * value from 100 and convert to percentage.
     */
    dam = dam * (100 - deflected) / 100 
	+ fight_mode_table[ch->fmode].todam
	+ str_app[get_curr_str(ch)].todam;
    return UMAX(0,dam);
}

void mobile_attack( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int count;
    int dam;
    int oldhit;
    int deflected;

    if ( IS_NPC(ch) )
    {
    for ( count = 0;  count < MAX_ATTACK_DATA;  count++ )
    {
        ATTACK_DATA *attack;

        if ( (attack = ch->pIndexData->attacks[count]) != NULL )
        {
        /*
         * Hit or Miss?
	 */
	if ( number_percent( ) < chance_to_hit( ch, victim ) )
	{
	    /* Hit, check armor. */
	    deflected = armor_deflect( victim, attack->idx );
	    dam = number_range( attack->dam1, attack->dam2 );
	    dam = damage_adjust( ch, dam, deflected );
	}
        else
	{
	    damage( ch, victim, 0 );
	    continue;
	}

        if ( can_see(victim,ch) && IS_AWAKE(victim) )
        {
        if ( skill_check( victim, gsn_dodge, 50 ) )
        {
            MSG_dodge( ch, attack->idx, victim );
	    SET_BIT( victim->fbits, FGT_DODGE );
            continue;
        }

        if ( skill_check( victim, gsn_dodge, 75 ) )
        {
            MSG_partial_dodge( ch, attack->idx, victim );
            damage( ch, victim, dam / 2 );
	    SET_BIT( victim->fbits, FGT_PARRY );
            continue;
        }
        }

        oldhit = victim->hit;
        MSG_mobile_hit( ch, attack->idx, victim, PERCENTAGE(dam,oldhit) );
        damage( ch, victim, dam );
	}
    }
    }

    return;
}


/*
 * Hand-to-Hand Routines
 */
void hand_attack( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int dam = 0, oldhit;
    /*
     * Can't beat a dead char!
     * Guard against weird room-leavings.
     */
    if ( victim->position == POS_DEAD || ch->in_room != victim->in_room )
    return;

    /*
     * If we a used special skill, do nothing more
     */
    if ( unarmed_attack( ch, victim ) )
	return;
    /*
     * Hit or Miss?
     */

    if ( number_percent( ) < chance_to_hit( ch, victim ) )
    {
	/* Hit, check armor. */
	dam = damage_adjust( ch, 1, armor_deflect( victim, 0 ) );
    }
    else
    {
	MSG_hand_miss( ch, victim );
	if ( !IS_NPC(ch) && IS_NPC( victim ) )
	    damage( ch, victim, 0 );
        return;
    }

    if ( skill_check( victim, gsn_dodge, 50 )
     && can_see(victim,ch) && IS_AWAKE(victim) )
    {
	MSG_dodge( ch, 0, victim );
	SET_BIT( victim->fbits, FGT_DODGE );
        return;
    }

    oldhit = victim->hit;
    MSG_hand_hit( ch, victim, PERCENTAGE(dam,oldhit) );
    damage( ch, victim, dam );
    tail_chain( );
    return;
}


/*
 * Hit one guy once.
 */
void weapon_attack( CHAR_DATA *ch, OBJ_DATA *obj, CHAR_DATA *victim )
{
    int dam = -1;   /* miss */
    int oldhit;
    int chance;
    int deflected;
    int wp;

    if ( obj == NULL
      || victim == NULL
      || obj->item_type != ITEM_WEAPON )
    {
        hand_attack( ch, victim );
        return;
    }

    /*
     * Can't beat a dead char!
     * Guard against weird room-leavings.
     */
    if ( victim->position == POS_DEAD || ch->in_room != victim->in_room )
    return;

    /*
     * If we a used special skill, do nothing more
     */
    if ( armed_attack( ch, victim ) )
	return;

    /*
     * Hit or Miss?
     */
    chance = chance_to_hit( ch, victim );
    /*
     * Weapon proficiencies.
     */
    wp = *attack_table[URANGE(0,obj->value[3],MAX_ATTACK)].wpgsn;
    /*
     * Rather than use a flat modifier, increase the chance in
     * relation to the weapon skill
     */
    chance += (LEARNED(ch,wp)*30)/100;

    if ( number_percent( ) < chance )
    {
	/* Hit, check armor. */
	deflected = armor_deflect( victim, obj->value[3] );
        if ( skill_check(ch, wp, 0) )
            deflected /= 2;

	dam = number_range( UMIN(obj->value[1],obj->value[2]),
			    UMAX(obj->value[1],obj->value[2])  );
	dam = damage_adjust( ch, dam, deflected );
        SET_BIT( ch->fbits, FGT_PRIMARY );
    }
    else
    {
        MSG_weapon_miss( ch, obj, victim );
	if ( !IS_NPC(ch) && IS_NPC( victim ) )
	    damage( ch, victim, 0 );
        return;
    }

    if ( victim->wait <= 0 && can_see( victim, ch ) && IS_AWAKE( victim ) )
    {
    /*
     * Shield parry
     */
    if ( skill_check( victim, gsn_parry, 50 )
      && !IS_SET(victim->fbits, FGT_PRIMARY) )
    {
        OBJ_DATA *vo;

	vo = get_eq_char( victim, WEAR_SHIELD );
        if ( vo == NULL )
        vo = get_eq_char( victim, WEAR_HOLD_1 );
        if ( vo == NULL )
        vo = get_eq_char( victim, WEAR_HOLD_2 );

        if ( vo != NULL && vo->item_type == ITEM_ARMOR )
        {
        MSG_weapon_parry( ch, obj, vo, victim );
	SET_BIT( victim->fbits, FGT_PARRY );
        return;
        }
    }

    /*
     * Weapon parry, lower chance
     */
    if ( skill_check( victim, gsn_parry, 30 )
      && !IS_SET(victim->fbits, FGT_PRIMARY) )
    {
        OBJ_DATA *vo;

        vo = get_eq_char( victim, WEAR_WIELD_1 );
        if ( vo == NULL )
        vo = get_eq_char( victim, WEAR_WIELD_2 );

        if ( vo != NULL )
        {
        MSG_weapon_parry( ch, obj, vo, victim );
	SET_BIT( victim->fbits, FGT_PARRY );
        SET_BIT( victim->fbits, FGT_PRIMARY );
        return;
        }
    }

    /*
     * Full dodge
     */
    if ( skill_check( victim, gsn_dodge, 50 ) )
    {
        MSG_weapon_dodge( ch, obj, victim );
	SET_BIT( victim->fbits, FGT_DODGE );
        return;
    }

    if ( skill_check( victim, gsn_dodge, 75 ) )
    {
        MSG_weapon_partial_dodge( ch, obj, victim );
        damage( ch, victim, dam / 2 );
	SET_BIT( victim->fbits, FGT_PARRY );
        SET_BIT( victim->fbits, FGT_DODGE );
        return;
    }
    }

    if ( attack_table[obj->value[3]].hit_fun != NULL
     && attack_table[obj->value[3]].hit_fun( ch, victim, dam, victim->hit) )
    return;

    oldhit = victim->hit;
    MSG_weapon_hit( ch, obj, victim, PERCENTAGE(dam,oldhit) );
    damage( ch, victim, dam );
    tail_chain( );
    return;
}



/*
 * Inflict damage from a hit.
 */
void damage( CHAR_DATA *ch, CHAR_DATA *victim, int dam )
{
/*
    if ( dam != 0 )
    {
        CHAR_DATA *rch;
        char buf[12];

        for( rch = ch->in_room->people; rch != NULL; rch = rch->next_in_room )
        {
            if ( IS_IMMORTAL(rch) )
            {
                sprintf( buf, "[%c%c%3d%c%c] ",
                         rch != ch ? STR(ch,name)[0] : '*',
                         rch != ch ? STR(ch,name)[1] : '*',
                         dam,
                         rch != victim ? STR(victim,name)[0] : '*',
                         rch != victim ? STR(victim,name)[1] : '*' );
                send_to_char( buf, rch );
            }
        }
    }
*/
    if ( IS_IMMORTAL(victim) && victim->hit < dam )
    return;

    if ( victim->position == POS_DEAD )
        return;

    /*
     * Stop up any residual loopholes.
     */
    if ( dam > 500 )
    {
    bug( "Damage: %d: more than 500 points!", dam );
    dam = 500;
    }

    if ( victim != ch )
    {
        /*
         * Certain attacks are forbidden.
         * Most other attacks are returned.
         */
        if ( is_safe( ch, victim ) )
            return;

        if ( victim->position > POS_STUNNED )
        {
            if ( victim->fighting == NULL )
                set_fighting( victim, ch );
            victim->position = POS_FIGHTING;
            if ( ch->fighting == NULL )
                set_fighting( ch, victim );
        }

        /*
         * More charm stuff.
         */
        if ( victim->master == ch )
            stop_follower( victim );

        /*
         * Inviso attacks ... not.
         */
        if ( IS_AFFECTED(ch, AFF_INVISIBLE) )
        {
            affect_strip( ch, gsn_invis );
            affect_strip( ch, gsn_mass_invis );
            REMOVE_BIT( ch->affected_by, AFF_INVISIBLE );
            act( "$n fades into existence.", ch, NULL, NULL, TO_ROOM );
        }

        /*
         * Damage modifiers.
         */
        if ( IS_AFFECTED(victim, AFF_SANCTUARY) )
            dam /= 2;

        if ( IS_AFFECTED(victim, AFF_PROTECT) )
            dam -= dam / 4;

        if ( dam < 0 )
            dam = 0;
    }
    if ( dam == 0 )
	return;

    /*
     * Hurt the victim.
     */
    if ( PERCENTAGE( victim->hit, MAXHIT(victim) ) <= 40 )
	SET_BIT(victim->fbits,FGT_WOUNDED);
    SET_BIT(victim->fbits,FGT_HIT);
    victim->hit -= dam;
    if ( IS_IMMORTAL(victim) && victim->hit < 1 ) victim->hit = 1;

    update_pos( victim );

    /*
     * These messages have to be here instead of violence_update()
     * because stop_fighting() is called below...
     */
    if ( victim->position == POS_STUNNED )
    {
        act( "$n is mortally wounded!", victim, NULL, NULL, TO_ROOM );
        send_to_char("You are badly wounded, but will probably recover.\n\r", victim );
    }
    else if ( victim->position == POS_DEAD )
    {
        send_to_char( "** You have been killed. **\n\r\n\r", victim );
        if ( IS_IMMORTAL(victim) )
        {
            send_to_char( "But you are immortal, so you arise again.\n\r", victim );
            victim->hit = 50;
            victim->position = POS_STANDING;
            return;
        }
    }

    /*
     * Sleep spells and extremely wounded folks.
     * If both ch and victim are PCs, stop the combat.
     */
    if ( !IS_AWAKE(victim) )
	stop_fighting( victim, !IS_NPC(ch) && !IS_NPC(victim) ? TRUE : FALSE );

    /*
     * Payoff for killing things.
     */
    if ( victim->position == POS_DEAD )
    {
        if ( !IS_NPC(victim) )
        {
        char  buf[MAX_STRING_LENGTH];

        sprintf( log_buf, "%s killed by %s at %d",
                NAME(victim),
                NAME(ch),
                victim->in_room->vnum );
            log_string( log_buf );

        sprintf( buf, "Notify> %s", log_buf );
        NOTIFY( buf, LEVEL_IMMORTAL, WIZ_NOTIFY_DEATH );
        }

        if ( victim->bounty == 0 )   ch->bounty += 500;      /* Crime */
        else
        if ( victim->bounty > 0 )
        {
            ch->owed   += victim->bounty;
            victim->bounty = 0;
        }

        script_update( ch, TYPE_MOB, TRIG_KILLS, victim, NULL, NULL, NULL );
        script_update( victim, TYPE_MOB, TRIG_DIES, ch, NULL, NULL, NULL );
        script_update( victim->in_room, TYPE_ROOM, TRIG_DIES, victim, NULL, NULL, NULL );
        trigger_list( victim->carrying, TRIG_DIES, victim, NULL, NULL, NULL );
        trigger_list( ch->carrying, TRIG_DIES, victim, NULL, NULL, NULL );
        trigger_list( ch->in_room->contents, TRIG_DIES, victim, NULL, NULL, NULL );
        raw_kill( victim );
        return;
    }

    if ( victim == ch )
        return;

    /*
     * Take care of link dead people.
     */
    if ( !IS_NPC(victim) && victim->desc == NULL )
	retreat( victim );
    tail_chain( );
    return;
}



bool is_safe( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( IS_NPC(ch) || IS_NPC(victim) )
        return FALSE;

    /* Thx Josh! */
    if ( victim->fighting == ch )
        return FALSE;

    if ( IS_SET( ch->in_room->room_flags, ROOM_SAFE ) )
    return TRUE;

    return FALSE;
}



/*
 * Set position of a victim.
 */
void update_pos( CHAR_DATA *victim )
{
    if ( victim->hit > 0 )
    {
	if ( victim->position <= POS_STUNNED ) victim->position = POS_STANDING;
	return;
    }

	victim->hit = 0;
	victim->position = POS_DEAD;
    return;
}



/*
 * Start fights.
 */
void set_fighting( CHAR_DATA *ch, CHAR_DATA *victim )
{
    TRIGGER_DATA *pTrig;

    if ( ch->fighting != NULL )
    {
        bug( "Set_fighting: already fighting", 0 );
        return;
    }

    /*
     * Halt non-combat scripts temporarily.
     * Can be triggered immediately after this.
     */
    for ( pTrig = ch->triggers; pTrig != NULL; pTrig = pTrig->next )
    {
        if ( pTrig->script->type != TRIG_COMBAT )
        {
            VARIABLE_DATA *pVar, *pVar_next;

            for ( pVar = pTrig->locals;  pVar != NULL; pVar = pVar_next )
            {
                pVar_next = pVar->next;
                free_variable( pVar );
            }

            pTrig->locals = NULL;
            pTrig->location = NULL;
        }
    }

    if ( IS_AFFECTED(ch, AFF_SLEEP) )
        affect_strip( ch, gsn_sleep );

    if ( ch->furniture ) set_furn( ch, NULL );
    ch->fighting = victim;
    ch->position = POS_FIGHTING;
    ch->fbits     = 0;
    return;
}



/*
 * Stop fights.
 */
void stop_fighting( CHAR_DATA *ch, bool fBoth )
{
    CHAR_DATA *fch;

    for ( fch = char_list; fch != NULL; fch = fch->next )
    {
        if ( fch == ch || ( fBoth && fch->fighting == ch ) )
        {
            if ( IS_NPC(fch) )
            {
                /* restart born scripts
                script_update( fch, TYPE_MOB, TRIG_BORN, NULL, NULL, NULL, NULL );
                */
                if ( fBoth )
                {
                    if ( number_fighting(fch) < 2 )
                    fch->wait = 0;
                    do_sheath( fch, "" );
                }
            }

            fch->fighting  = NULL;
            fch->position  = POS_STANDING;
            fch->fbits     = 0;
            update_pos( fch );
        }
    }

    return;
}


/*
 * Make a corpse out of character.
 */
void make_corpse( CHAR_DATA *ch )
{
    char buf[MAX_STRING_LENGTH];
    OBJ_DATA *corpse;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;
    char *name;

    if ( IS_NPC(ch) )
    {
        name          = NAME(ch);
        corpse        = create_object(get_obj_index(OBJ_VNUM_CORPSE_NPC), 0);
        corpse->timer = number_range( 20, 30 ) * 5;
    }
    else
    {
        name            = NAME(ch);
        corpse          = create_object(get_obj_index(OBJ_VNUM_CORPSE_PC), 0);
    }

    sprintf( buf, STR(corpse, short_descr), name );
    free_string( corpse->short_descr );
    corpse->short_descr = str_dup( buf );

    sprintf( buf, STR(corpse, description), name );
    free_string( corpse->description );
    corpse->description = str_dup( buf );

    for ( obj = ch->carrying; obj != NULL; obj = obj_next )
    {
        obj_next = obj->next_content;
        obj_from_char( obj );
        if ( IS_SET( obj->extra_flags, ITEM_INVENTORY ) )
            extract_obj( obj );
        else
            obj_to_obj( obj, corpse );
    }

    obj_to_room( corpse, ch->in_room );
    return;
}



/*
 * Improved Death_cry contributed by Diavolo.
 */
void death_cry( CHAR_DATA *ch )
{
    ROOM_INDEX_DATA *was_in_room;
    char *msg;
    int door;

    msg = "You hear $n's death cry.";
    switch ( number_range( 0, 12 ) )
    {
    case  0: msg  = "$n slumps onto the ground, dead.";         break;
    case  1: msg  = "$n collapses as the life flows from $m.";  break;
    case  2: msg  = "And then, suddenly, $n is dead.";          break;
    case  3: msg  = "The last ounce of life seeps from $n.";    break;
    case  4: msg  = "You fancy hearing the gods coming to claim $n's soul."; break;
    case  5:
        {
            OBJ_DATA *obj = ch->carrying;
            char buf[MAX_STRING_LENGTH];

            msg  = "As $n dies, you notice the blood everywhere.";
            if ( obj != NULL && obj->item_type != ITEM_MONEY )
            {
            sprintf( buf, "a bloodied %s",
                          smash_article(STR(obj, short_descr)) );
            free_string( obj->short_descr );
            obj->short_descr = str_dup( buf );
            }
        }
     break;
    }

    act( msg, ch, NULL, NULL, TO_ROOM );

    if ( IS_NPC(ch) )
    msg = "You hear the wail of something dying.";
    else
    msg = "You hear a shriek as someone dies.";

    was_in_room = ch->in_room;
    for ( door = 0; door < MAX_DIR; door++ )
    {
        EXIT_DATA *pexit;

        if ( ( pexit = was_in_room->exit[door] ) != NULL
        &&   pexit->to_room != NULL
        &&   pexit->to_room != was_in_room )
        {
            ch->in_room = pexit->to_room;
            act( msg, ch, NULL, NULL, TO_ROOM );
        }
    }
    ch->in_room = was_in_room;

    return;
}



void raw_kill( CHAR_DATA *victim )
{
    int i;

    stop_fighting( victim, TRUE );
    death_cry( victim );

    if ( victim->riding != NULL )
    {
        act( "As $n dies, you fall to the ground.",
             victim, NULL, victim->riding, TO_VICT );
        victim->riding->position   = POS_RESTING;
    }

    if ( victim->riding != NULL )
    act( "$n falls off you, dead.", victim, NULL, victim->riding, TO_VICT );
     
    dismount_char( victim );

    if ( IS_NPC(victim) )
    {
        make_corpse( victim );
        extract_char( victim, TRUE );
        return;
    }

    make_corpse( victim );
    extract_char( victim, FALSE );
    char_from_room( victim );

  /* If rather new.. */
    {
    int p = PC(victim,played);
    int ct = (int) (current_time - PC(victim,logon));

    i = (p + ct)/7200;
    }

    if ( ( i < 2 ) )
    {
        while ( victim->affected )
        affect_remove( victim, victim->affected );

        victim->affected_by = race_table[victim->race].affect_bits;
        victim->armor       = 100;
        victim->position    = POS_RESTING;
        victim->hit         = UMAX( 1, victim->hit  );
        victim->move        = UMAX( 1, victim->move );
        PC(victim,condition)[COND_DRUNK]   = 0;
        PC(victim,condition)[COND_FULL]    = 20;
        PC(victim,condition)[COND_THIRST]  = 20;
        char_to_room( victim,
                  get_room_index( race_table[victim->race].start_room ) );
        send_to_char( "For the first two hours, you will be given another chance.\n\r", victim );
    }
    else
    char_to_room( victim, get_room_index( ROOM_VNUM_DEATH ) );

    save_char_obj( victim );

    return;
}

/*
 * Syntax: kill [person]
 */
void do_kill( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;

    if ( IS_NPC(ch)
      && IS_SET(ch->act, ACT_PET) )  /* NOT */
       return;

    one_argument( argument, arg );

    if ( arg[0] == '\0' )
    {
        send_to_char( "Kill whom?\n\r", ch );
        return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
        send_to_char( "They aren't here.\n\r", ch );
        return;
    }

    if ( victim == ch )
    {
        send_to_char( "You hit yourself.\n\r", ch );
        oroc( ch, ch );
        return;
    }

    if ( is_safe( ch, victim ) )
    return;

    if ( IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim )
    {
        act( "$N is your beloved master.", ch, NULL, victim, TO_CHAR );
        return;
    }

    if ( ch->position == POS_FIGHTING )
    {
        send_to_char( "You do the best you can!\n\r", ch );
        return;
    }

	do_fight( ch, "" );
    WAIT_STATE( ch, 1 * PULSE_VIOLENCE );
    oroc( ch, victim  );
    return;
}

/*
 * Syntax: fight [fightmode]
 */
void do_fight( CHAR_DATA *ch, char *argument )
{
    char arg[ MAX_INPUT_LENGTH ];
    int fmode;

    if ( argument[0] != '\0' )
    {
	one_argument( argument, arg );
	if ( ( fmode = fight_mode_lookup( arg ) ) < 0 )
	{
	    send_to_char( "That is not a valid combat tactic.\n\r", ch );
	    return;
	}
	ch->fmode = fmode;
    }
    
    sprintf( arg, "Your combat tactic is %s.\n\r",
	fight_mode_table[ch->fmode].name );
    send_to_char( arg, ch );
    return;
}

void do_shoot( CHAR_DATA *ch, char *argument )
{
    return;
}


void do_reload( CHAR_DATA *ch, char *argument )
{
    return;
}

/*
 * Syntax:  backstab [person]
 */
void do_backstab( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    char w[ MAX_STRING_LENGTH ];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int dt;

    one_argument( argument, arg );

    if ( arg[0] == '\0' )
    {
    send_to_char( "Backstab whom?\n\r", ch );
    return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
    send_to_char( "They aren't here.\n\r", ch );
    return;
    }

    if ( victim == ch )
    {
    send_to_char( "How can you sneak up on yourself?\n\r", ch );
    return;
    }

    if ( is_safe( ch, victim ) )
      return;

    if ( ( obj = get_item_held( ch, ITEM_WEAPON ) ) == NULL )
    {
    send_to_char( "You need to wield a weapon.\n\r", ch );
    return;
    }

    dt = obj->value[3];

    if ( attack_table[dt].hit_type != TYPE_PIERCE )
    {
    send_to_char( "You need to wield a piercing weapon.\n\r", ch );
    return;
    }

    if ( victim->fighting != NULL )
    {
    send_to_char( "You can't backstab a fighting person.\n\r", ch );
    return;
    }

    if ( (victim->size - 4) > ch->size )
    {
	act( "You can't reach $N's back!", ch, NULL, victim, TO_CHAR );
	return;
    }
    if ( (victim->size + 4) < ch->size )
    {
	act( "$N is too small for you to backstab.", ch, NULL, victim, TO_CHAR );
	return;
    }

    if ( victim->hit < MAXHIT(victim)/2 && victim->position > POS_SLEEPING )
    {
    act( "$N is too alert with pain to sneak up.", ch, NULL, victim, TO_CHAR );
    return;
    }

    WAIT_STATE( ch, skill_table[gsn_backstab].beats );
    sprintf( w, "%s", smash_article(STR(obj,short_descr)) );

    if ( !IS_AWAKE(victim)
      || (IS_NPC(ch) && IS_NPC(victim))
      || skill_check( ch, gsn_backstab, 0 ) )
    {
	int deflected, dam;

	deflected = armor_deflect( victim, obj->value[3] );
	dam = number_range( UMIN(obj->value[1],obj->value[2]),
                            UMAX(obj->value[1],obj->value[2])  );

	if ( deflected == 0 ) deflected = 100;

	/* double damage from backstabbing? */
	act( "$n thrust$v $t into $N$Y back!", ch, w, victim, TO_ALL );
	dam = ((dam * deflected) / 100) * 2;
	damage( ch, victim, dam );
    }
    else
    {
	act( "$n tr$x to thrust $t into $N$Y back!", ch, w, victim, TO_ALL );
	damage( ch, victim, 0 );
    }

    return;
}


void do_sla( CHAR_DATA *ch, char *argument )
{
    send_to_char( "If you want to SLAY, spell it out.\n\r", ch );
    return;
}



/*
 * Syntax: slay [person]
 */
void do_slay( CHAR_DATA *ch, char *argument )
{
    CHAR_DATA *victim;
    char arg[MAX_INPUT_LENGTH];

    one_argument( argument, arg );
    if ( arg[0] == '\0' )
    {
        send_to_char( "Slay whom?\n\r", ch );
        return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
        send_to_char( "They aren't here.\n\r", ch );
        return;
    }

    if ( ch == victim )
    {
        send_to_char( "Suicide is a mortal sin.\n\r", ch );
        return;
    }

    if ( get_trust(victim) > get_trust(ch) )
    {
        send_to_char( "You failed.\n\r", ch );
        return;
    }

    script_update( victim, TYPE_MOB, TRIG_DIES, ch, NULL, NULL, NULL );
    act( "You grasp $S head and squeeze it until it explodes!",
                                            ch, NULL, victim, TO_CHAR    );
    act( "$n grasps your head and squeezes until it explodes!",
                                            ch, NULL, victim, TO_VICT    );
    act( "$n grasps $N's head and squeezes until it explodes!",
                                            ch, NULL, victim, TO_NOTVICT );
    raw_kill( victim );
    return;
}



void retreat( CHAR_DATA *ch )
{
    CHAR_DATA *vch;
    int chance = 0;

    if ( ch->fighting == NULL )
	return;

    chance += get_curr_dex(ch);
    chance -= get_curr_dex(ch->fighting) * 3 / 2;

    if ( skill_check( ch, gsn_flee, 50 + chance ) )
    {
	WAIT_STATE( ch, skill_table[gsn_flee].beats );
	for( vch = ch->in_room->people; vch; vch = vch->next_in_room )
	{
	    if ( vch->fighting == ch->fighting && vch != ch )
	    {
		act( "$n retreat$v from combat!", ch, NULL, NULL, TO_ALL );
		stop_fighting(ch,TRUE);
		act( "$N direct$v $S attack at $n!", vch, NULL, vch->fighting, TO_ALL );
		set_fighting(vch->fighting,vch);
		return;
	    }
	}
    }
    flee( ch );
    return;
}

/*
 * Syntax: retreat
 */
void do_retreat( CHAR_DATA *ch, char *argument )
{
    if ( ch->fighting == NULL )
    {
	send_to_char( "Calm down, you're not fighting!\n\r", ch );
	return;
    }
    if ( LEARNED(ch, gsn_flee) <= 0 )
    {
	send_to_char( "If you want out of combat, FLEE!\n\r", ch );
	return;
    }
    retreat(ch);
}

/*
 * Do flee checks.
 */
bool flee_check( CHAR_DATA *ch, int d )
{
    CHAR_DATA *fch, *fch_next;
    int penalty = 0, fdex, chdex, chstr, fstr;


    chdex = number_range( 0, get_curr_dex(ch) );
    chstr = number_range( 0, get_curr_str(ch) );

    for( fch = ch->in_room->people; fch != NULL; fch = fch_next )
    {
	/* Careful here, our victim may die while fleeing... */
	fch_next = fch->next_in_room;
	if ( fch->fighting != ch || fch->wait > 0 ) continue;
	fdex  = number_range( 0, get_curr_dex(fch) );
	fstr  = number_range( 0, get_curr_str(fch) );

	if ( chdex < fdex - penalty )
	{
	    act( "You attack $N as $E turns to run!", fch, NULL, ch, TO_CHAR );
	    act( "$n attacks you as you turn to run!", fch, NULL, ch, TO_VICT );
	    act( "$n attacks $N as $e turns to run!", fch, NULL, ch, TO_NOTVICT );
	    fch->fbits = 0;
	    oroc( fch, ch );
	    /* If the poor bugger died... */
	    if ( ch->position < POS_STANDING )
		return( FALSE );
	}
	else
	if ( chstr < fstr - penalty
	  || skill_check( ch, gsn_flee, 0 ) )
        {
            act( "$N block$V $n as $e tries to escape $t!",
                 ch, dir_name[d], fch, TO_ALL );
	    return( FALSE );
	}
        penalty++;
    }
    /* Whew... got away! */
    return( TRUE );
}

/*
 * Flee from combat
 */
bool flee( CHAR_DATA *ch )
{
    ROOM_INDEX_DATA *was_in;
    int tried[ MAX_DIR ];
    int attempt;

    for ( attempt = 0; attempt < MAX_DIR; attempt++ ) tried[attempt] = FALSE;
    act( "$n panic$v and tr$x to flee!\n\r", ch, NULL, NULL, TO_ALL );
    was_in = ch->in_room;
    for ( attempt = 0; attempt < MAX_DIR; attempt++ )
    {
        EXIT_DATA *pexit;
        int door;

        door = number_door();

	if ( tried[door]
        ||   ( pexit = was_in->exit[door] ) == NULL
	||   pexit->to_room == NULL
	||   IS_SET(pexit->exit_info, EX_CLOSED)
	||   IS_SET(pexit->exit_info, EX_WINDOW)
	|| ( IS_NPC(ch)
	  && IS_SET(pexit->to_room->room_flags, ROOM_NO_MOB) ) )
		continue;

        tried[door] = TRUE;
	if ( flee_check( ch, door ) ) 
	{
		stop_fighting( ch, TRUE );
		ch->position = POS_FIGHTING;
		move_char( ch, door );
		ch->position = POS_STANDING;
	}

	if ( ch->in_room != was_in ) return( TRUE );
    }

    return( FALSE );
}



/*
 * Syntax:  flee
 */
void do_flee( CHAR_DATA *ch, char *argument )
{

/*
    if ( ch->fighting == NULL )
    {
	send_to_char( "Calm down, you're not fighting!\n\r", ch );
	return;
    }

    if ( LEARNED(ch, gsn_flee) <= 0 )
    {
    send_to_char( "You aren't able to do that.\n\r", ch );
	return;
    }
 */

    flee( ch );    
    return;
}


/*
 * Syntax:  rescue [person]
 */
void do_rescue( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    CHAR_DATA *fch;

    one_argument( argument, arg );
    if ( arg[0] == '\0' )
    {
        send_to_char( "Rescue whom?\n\r", ch );
        return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
        send_to_char( "They aren't here.\n\r", ch );
        return;
    }

    if ( victim == ch )
    {
        send_to_char( "What about fleeing instead?\n\r", ch );
        return;
    }

    if ( !IS_NPC(ch) && IS_NPC(victim) )
    {
        send_to_char( "Doesn't need your help!\n\r", ch );
        return;
    }

    if ( ch->fighting == victim )
    {
        send_to_char( "Too late.\n\r", ch );
        return;
    }

    if ( ( fch = victim->fighting ) == NULL )
    {
        send_to_char( "That person is not fighting right now.\n\r", ch );
        return;
    }

    WAIT_STATE( ch, skill_table[gsn_rescue].beats );
    if ( skill_check( ch, gsn_rescue, 0 ) )
    {
        send_to_char( "You fail the rescue.\n\r", ch );
        return;
    }

    act( "You rescue $N!",  ch, NULL, victim, TO_CHAR    );
    act( "$n steps between you and $t!", ch, NAME(victim->fighting), victim, TO_VICT );
    act( "$n steps between $N and $t!", ch, NAME(victim->fighting), victim, TO_NOTVICT );

    stop_fighting( fch, FALSE );
    stop_fighting( ch, FALSE );
    stop_fighting( victim, FALSE );

    set_fighting( ch, fch );
    set_fighting( fch, ch );
    return;
}


bool hit_suck_disarm( CHAR_DATA *ch, CHAR_DATA *victim, int hit, int dam )
{
    OBJ_DATA *obj;

    if ( skill_check( ch, gsn_hth, 0 )
    && ( obj = get_item_held( victim, ITEM_WEAPON ) ) != NULL
    && hand_empty( ch ) != WEAR_NONE )
    {
        obj->wear_loc = hand_empty( ch );
        obj_from_char( obj );
        obj_to_char( obj, ch );
        act( "You suck $N's weapon right out of $s hand!",  ch, NULL, victim,  TO_CHAR );
        act( "$n sucks your weapon right out of your hand!", ch, NULL, victim, TO_VICT );
        act( "$n sucks $N's weapon right out of $s hand!",   ch, NULL, victim, TO_NOTVICT );

        return TRUE;
    }

    return FALSE;
}



bool hit_vorpal( CHAR_DATA *ch, CHAR_DATA *victim, int hit, int dam )
{
    char buf[MAX_STRING_LENGTH];
    OBJ_DATA *obj;
    char *name;

    if ( hit >= 17 )
    {
        act( "You slice $N's head clean off!",  ch, 0, victim, TO_CHAR );
        act( "$n slices your head clean off!",  ch, 0, victim, TO_VICT );
        act( "$n slices $N's head clean off!",  ch, 0, victim, TO_NOTVICT );
        act( "$n's severed head plops on the ground.", victim, 0, 0, TO_ROOM );
        stop_fighting( victim, TRUE );

        name        = NAME(ch);
        obj         = create_object( get_obj_index( OBJ_VNUM_SEVERED_HEAD ), 0 );
        obj->timer  = number_range( 4, 7 );

        sprintf( buf, STR(obj, short_descr), NAME(victim) );
        free_string( obj->short_descr );
        obj->short_descr = str_dup( buf );

        sprintf( buf, STR(obj, description), NAME(victim) );
        free_string( obj->description );
        obj->description = str_dup( buf );

        obj_to_room( obj, ch->in_room );

        raw_kill( victim );
        return TRUE;
    }

    return FALSE;
}

/*
 * Kick someone
 */
bool kick( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed )
{
    int chance = 0;

    if ( ch->fighting == NULL )
	return( FALSE );

    if (IS_AFFECTED(victim,AFF_FLYING) || IS_AFFECTED(ch,AFF_FLYING))
	return ( FALSE );

    chance  = fArmed ? 0 : 20;
    chance -= victim->size - ch->size;

    /* stats */
    chance += get_curr_dex(ch);
    chance -= get_curr_str(victim) * 4 / 3;

    if ( skill_check( ch, gsn_hth, 30+chance ) )
    {
	char *i = NULL;
	int deflected, dam;

	deflected = armor_deflect( victim, 0 );
	dam = number_range( 2, UMAX(8,chance) );
        dam = damage_adjust( ch, dam, deflected );
	switch( number_range(0,4) )
	{
	case 0: i = "$n send$v a flying kick into $N!";
		break;
	case 1: i = "$n kick$v $N painfully!";
		break;
	case 2: i = "$n$y kick connects $N with an audible thud!";
		break;
	case 3: i = "$n$y swirl$v and kick$v $N with ferocity!";
		break;
	case 4: i = "$N $K surprised by $n$y sudden kick!";
		break;
        }
	act( i, ch, NULL, victim, TO_ALL );
	damage( ch, victim, dam );
	WAIT_STATE( victim, PULSE_VIOLENCE * 2 );
    }
    else
    {
	act( "$n tr$x to kick $N but miss$w!", ch, NULL, victim, TO_ALL );
        WAIT_STATE( ch, PULSE_VIOLENCE * 2 );
    }
    return ( TRUE );
}

/*
 * Disarm a creature.
 */
bool disarm( CHAR_DATA *ch, CHAR_DATA *victim )
{
    OBJ_DATA *obj;


    if (victim->position < POS_FIGHTING)
	return ( FALSE );

    if ( ( obj = get_item_held( victim, ITEM_WEAPON ) ) == NULL )
        return ( FALSE );

    if ( get_item_held( ch, ITEM_WEAPON ) == NULL )
	return ( FALSE );

    if ( IS_SET( obj->extra_flags, ITEM_NOREMOVE ) && !IS_IMMORTAL(ch) )
    {
        act( "$B$n tries to disarm you, but $p won't come off your hand!$R",
             ch, obj, victim, TO_VICT );
        act( "You try to knock the weapon from $S hand, but it is useless!",
             ch, obj, victim,      TO_CHAR );
        return ( TRUE );
    }

    WAIT_STATE( ch, skill_table[gsn_hth].beats );
    if ( skill_check( ch, gsn_hth, 70 - (get_curr_dex(victim) * 3) ) )
    {
	const char *i = NULL;
	switch( number_range(0,3) )
	{
	case 0: i = "$BWith a skillful twist, $n disarm$v $N!$R"; 
		break;
	case 1: i = "$B$n disarm$v $N and send$v $S weapon flying!$R";
		break;
	case 2: i = "$B$n feint$v and suddenly yank$v $N$Y weapon away!$R";
		break;
	case 3: i = "$B$n send$v $N$Y weapon flying, disarming $M!$R";
		break;
	}
	act( i,  ch, NULL, victim, TO_ALL );

	obj_from_char( obj );
	obj_to_room( obj, victim->in_room );
    }
    else
    {
	act( "$n tr$x to disarm $N, but fail$v!", ch, NULL, victim, TO_ALL );
    }
    return ( TRUE );
}


/*
 * Bash a creature.
 */
bool bash( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed )
{
    int chance = 0;

    if ( get_item_held(ch, ITEM_ARMOR) == NULL )
	return ( FALSE );

    if (victim->position < POS_FIGHTING)
	return ( FALSE );

    chance  = fArmed ? 0 : 10;

    chance += ch->carry_weight / 25;
    chance -= victim->carry_weight / 25;

    chance -= (victim->size - ch->size);

    /* stats */
    chance += get_curr_str(ch);
    chance -= get_curr_dex(victim) * 4 / 3;

    /* now the attack */
    if ( skill_check( ch, gsn_hth, 40 + chance ) )
    {
	const char *i = NULL;
	switch(number_range(0,3))
	{
	case 0: i = "$B$n send$v $N sprawling with a powerful bash!$R";
		break;
	case 1: i = "$B$n slam$v into $N, toppling $M off balance!$R";
		break;
	case 2: i = "$B$n$y forceful bash send$v $N sprawling!$R";
		break;
	case 3: i = "$BWith a vicious blow, $n send$v $N sprawling!$R";
		break;
	}
	act( i, ch, NULL, victim, TO_ALL );
	WAIT_STATE( victim, 2 * PULSE_VIOLENCE);
	victim->position = POS_RESTING;
	damage( ch, victim, number_range(2, UMAX(8,chance)) );	
    }
    else
    {
	act( "$n tri$x to bash $N, but fall$v flat on $s face!",
		ch, NULL, victim, TO_ALL );
	WAIT_STATE( ch,  2 * PULSE_VIOLENCE); 
	ch->position = POS_RESTING;
    }
    return ( TRUE );
}

/*
 * Grapple
 */
bool grapple( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int chance = 0;

    if ( get_eq_char( ch, WEAR_HOLD_1 )
    ||   get_eq_char( ch, WEAR_HOLD_2 ) )
	return( FALSE );

    if (IS_AFFECTED(victim,AFF_FLYING) || IS_AFFECTED(ch,AFF_FLYING))
	return ( FALSE );

    chance -= victim->size - ch->size;

    chance += ch->carry_weight / 25;
    chance -= victim->carry_weight / 25;

    chance += get_curr_str(ch);
    chance -= get_curr_dex(victim) * 3 / 2;

    /* now the attack */
    if ( skill_check(ch, gsn_hth, 40 + chance) )
    {
	const char *i = NULL;
	switch(number_range(0,3))
	{
	case 0: i = "$B$n grab$v $N and throw$v $M to the ground!$R";
		break;
	case 1: i = "$B$n tackle$v $N, sending $M flat on $S face!$R";
		break;
	case 2: i = "$B$n seize$v a hold of $N and throw$v $M down!$R";
		break;
	case 3: i = "$B$n grasp$v $N and swing$v $M off balance!$R";
		break;
	}
	act( i, ch, NULL, victim, TO_ALL );
	WAIT_STATE(victim,3 * PULSE_VIOLENCE);
	victim->position = POS_RESTING;
	damage(ch,victim,number_range(2, 1 + 2 * victim->size / 20));
    }
    else
    {
	act( "$n tr$x to grapple $N, but fail$v.", ch, NULL, victim, TO_ALL );
	WAIT_STATE(ch, 2 * PULSE_VIOLENCE);
    } 
    return ( TRUE );

}

/*
 * Trip
 */
bool trip( CHAR_DATA *ch, CHAR_DATA *victim, bool fArmed )
{
    int chance=0;

    if (IS_AFFECTED(victim,AFF_FLYING) || IS_AFFECTED(ch,AFF_FLYING))
	return ( FALSE );

    chance  = fArmed ? 0 : 20;
    chance -= victim->size - ch->size;

    /* dex */
    chance += get_curr_dex(ch);
    chance -= get_curr_dex(victim) * 3 / 2;

    /* now the attack */
    if ( skill_check(ch, gsn_hth, 30 + chance) )
    {
	const char *i = NULL;
	switch(number_range(0,3))
	{
	case 0: i = "$B$N feint$V, but $n trip$v $M off balance!$R";
		break;
	case 1: i = "$B$n trip$v $N, sending $M flat on $S face!$R";
		break;
	case 2: i = "$B$N lunge$V forward, but $n tackle$v $M down!$R";
		break;
	case 3: i = "$B$n anticipate$v $N$Y move and trip$v $M!$R";
		break;
	}
	act( i, ch, NULL, victim, TO_ALL );
	WAIT_STATE(victim,3 * PULSE_VIOLENCE);
	victim->position = POS_RESTING;
	damage(ch,victim,number_range(2, 1 + 2 * victim->size / 20));
    }
    else
    {
	act( "$n tr$x to trip $N, but fail$v.", ch, NULL, victim, TO_ALL );
    } 
    return ( TRUE );
}

/*
 * Sweep
 */
bool sweep( CHAR_DATA *ch )
{
    CHAR_DATA *vch, *vch_next;
    OBJ_DATA *obj;
    char w[MAX_STRING_LENGTH];
    int count = 0;

    if ( ch->fighting == NULL )
	return ( FALSE );

    if ( ( obj = get_item_held( ch, ITEM_WEAPON ) ) == NULL )
	return ( FALSE );

    if ( attack_table[obj->value[3]].hit_type != TYPE_SLASH )
	return ( FALSE );


    sprintf( w, "%s", smash_article(STR(obj,short_descr)) );
    act( "$n swing$v $s $t around in a wide arc...", ch, w, NULL, TO_ALL);

    for ( vch = ch->in_room->people; vch != NULL; vch = vch_next )
    {
	vch_next = vch->next_in_room;
	if ( vch->fighting == ch )
	{
	    count++;
	    if ( skill_check(ch,gsn_hth,80/count) )
	       weapon_attack( ch, get_eq_char( ch, WEAR_WIELD_1 ), vch );
	}
    }
    if ( !count )
    {
	act( "$n$y sweep cuts only thin air!", ch, NULL, NULL, TO_ALL);
	count = number_range(1,3);
    }
    WAIT_STATE( ch, UMIN( 3, PULSE_VIOLENCE * count) ); 
    return ( TRUE );
}

int number_fighting( CHAR_DATA *ch )
{
    CHAR_DATA *vch;
    int count = 0;

    if ( ch->in_room == NULL )
	return( 0 );
    for ( vch = ch->in_room->people; vch; vch = vch->next_in_room )
	if ( vch->fighting == ch ) count++;
    return( count );
}

 
/* This func tries to determine when a fighting party attempts to flee */
void auto_flee( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int cond=0, wits = 0;

    /* We have no business here? */
    if ( victim->position != POS_FIGHTING
    ||   victim->wait > 0 )
	return;

    /* How smart am I? */
    cond = ch->fbits & combat_int[get_curr_int(victim)];
    wits = victim->fbits & combat_int[get_curr_int(victim)];

    /* Opponent stunned or I don't care if I've been hit */
    if ( !IS_SET(wits,FGT_HIT)
    ||    IS_SET(cond,FGT_STUNNED)
    ||   victim->fmode > fight_mode_lookup("normal") )
	return;


    /* Charmed folks won't go nowhere */
    if ( IS_AFFECTED(victim, AFF_CHARM)
    &&   victim->master != NULL
    &&   victim->master->in_room == victim->in_room )
    {
	return;
    }

    /* Normal state of wimpiness */
    if ( !IS_SET( wits, FGT_WOUNDED )
    &&   victim->fmode > fight_mode_lookup("elusive")
    &&   victim->hit > MAXHIT(victim)/5 )
    {
       return;
    }
    else
    /* Cautious approach */
    if ( victim->fmode > fight_mode_lookup("cautious") 
    &&   victim->hit > MAXHIT(victim)/4 )
    {
	return;
    }

    /* Otherwise try to leave the scene */
    /* If they're outnumbered, try retreating gracefully */
    retreat( victim );
    return;
}

void auto_fight( CHAR_DATA *ch, CHAR_DATA *victim )
{
    int cond=0;
    int us=0, them=0;

    /* Not fighting, so bail out */
    if ( ch->position != POS_FIGHTING
    ||   ch->wait > 0 )
    {
	return;
    }

    /* Charmed folks are none too smart */
    if ( IS_AFFECTED(ch, AFF_CHARM)
    &&   ch->master != NULL
    &&   ch->master->in_room == ch->in_room )
    {
	return;
    }

    /* If victim can't fight back, we'll pretend he's stunned */
    if ( victim->wait > 0 ) victim->fbits |= FGT_STUNNED;

    /* How smart we are? */
    cond = victim->fbits & combat_int[get_curr_int(ch)];

    /* Who's outnumbered? */
    us = number_fighting(victim);
    them = number_fighting(ch);

    /* If we're smart enough to know when to change tactic... */
    if ( IS_SET(combat_int[get_curr_int(ch)],FGT_CMP_HIT) )
    {

	/* If we're getting hurt and the opponent is not... */
    if ( (ch->hit < MAXHIT(ch)/3)
    &&  ((victim->hit > MAXHIT(victim)/2)
	  ||  us < them) )
	{
	    if ( ch->fmode > fight_mode_lookup("elusive") )
		--ch->fmode;
	    return;
	}
	else
	/* If the opponent is getting hurt and we're not... */
    if ( (ch->hit > MAXHIT(ch)/2)
    &&  ((victim->hit < MAXHIT(victim)/3)
	  ||  us > them) )
	{
	    if ( victim->fmode < fight_mode_lookup("berserk") )
		++ch->fmode;
	    return;
	}
    }

}

bool armed_attack( CHAR_DATA *ch, CHAR_DATA *victim )
{
    /* Self or victim not fighting, so bail out */
    if ( ch->position != POS_FIGHTING
    ||   ch->wait > 0 || victim->wait > 0 )
    {
	return( FALSE );
    }

    /* Attempt skill use */
    if ( number_bits(3) == 0 && LEARNED(ch,gsn_hth) > number_percent() )
    {
	int i;
	i = number_range(0,4);
	switch ( i )
	{
	case 0:	/* BASH */
	    return( bash(ch, victim, TRUE) );
	case 1: /* KICK */
	    return( kick(ch, victim, TRUE) );
	case 2: /* SWEEP */
	    if ( number_fighting(ch) > 1 )
		return( sweep(ch) );
	    break;
	case 3: /* TRIP */
	    return( trip(ch, victim, TRUE) );
	case 4: /* DISARM */
	    return( disarm(ch, victim) );
	default: break;
	}
	return( TRUE );
    }
    return( FALSE );
}

bool unarmed_attack( CHAR_DATA *ch, CHAR_DATA *victim )
{
    /* Self or victim not fighting, so bail out */
    if ( ch->position != POS_FIGHTING
    ||   ch->wait > 0 || victim->wait > 0 )
    {
	return( FALSE );
    }

    if ( number_bits(3) == 0 && LEARNED(ch, gsn_hth) > number_percent() )
    {
	
	switch ( number_range(0,3) )
	{
	case 0:	/* BASH */
	    return( bash(ch, victim, FALSE) );
	case 1: /* KICK */
	    return( kick(ch, victim, FALSE) );
	case 2: /* TRIP */
	    return( trip(ch, victim, FALSE) );
	case 3: /* GRAPPLE */
	    return( grapple(ch, victim) );
	default: break;
	}
	return( TRUE );
    }
    return( FALSE );
}

