/*
******************************************************************************
* Locke's   __ -based on merc v2.2-____        NIM Server Software           *
* ___ ___  (__)__    __ __   __ ___| G| v4.0   Version 4.0 GOLD EDITION      *
* |  /   \  __|  \__/  |  | |  |     O|        documentation release         *
* |       ||  |        |  \_|  | ()  L|        Hallow's Eve 1999             *
* |    |  ||  |  |__|  |       |     D|                                      *
* |____|__|___|__|  |__|\___/__|______|        http://www.nimud.org/nimud    *
*   n a m e l e s s i n c a r n a t e          dedicated to chris cool       *
******************************************************************************
 */



#if defined(macintosh)
#include <types.h>
#else
#include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "mud.h"
#include "skills.h"


/***************************************************************************
 *   File: graph.c                                       Part of CircleMUD *
 *  Usage: various graph algorithms                                        *
 *                                                                         *
 *  All rights reserved.  See license.doc for complete information.        *
 *                                                                         *
 *  Copyright (C) 1993 by the Trustees of the Johns Hopkins University     *
 *  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
 ***************************************************************************/


#define TRACK_THROUGH_DOORS

/*
 * You can define or not define TRACK_THROUGH_DOORS, above, depending on
 * whether or not you want track to find paths which lead through closed
 * or hidden doors.
 */

/*
 * BFS return values.
 */
#define BFS_ERROR           -1 
#define BFS_ALREADY_THERE   -2  
#define BFS_NO_PATH         -3


struct bfs_queue_struct 
{
   ROOM_INDEX_DATA *room;
   sh_int   dir;
   sh_int   depth;
   struct bfs_queue_struct *next;
};

struct room_list_struct 
{
   ROOM_INDEX_DATA *room;
   struct room_list_struct *next;
};

static struct bfs_queue_struct *queue_head = NULL, *queue_tail = NULL;

static struct room_list_struct *list_head = NULL, *list_tail = NULL;

/* Utility macros */
#define MARK( room )          ( SET_BIT( ( room )->room_flags, ROOM_MARK))
#define UNMARK( room )        ( REMOVE_BIT( ( room )->room_flags, ROOM_MARK))
#define IS_MARKED( room )     ( IS_SET( ( room )->room_flags, ROOM_MARK))
#define TOROOM( room, y )     ( ( room )->exit[ ( y ) ]->to_room )
#define IS_CLOSED( room, y )  ( IS_SET( ( room )->exit[( y )]->exit_info, \
                                EX_CLOSED))

#ifdef TRACK_THROUGH_DOORS
#define VALID_EDGE( room, y )                                              \
                              ( ( room )->exit[( y )] &&                   \
                              ( TOROOM(( room ), ( y ) ) != NULL) &&       \
                              ( !IS_MARKED( TOROOM( ( room ), ( y ) ) ) ) )
#else
#define VALID_EDGE( room , y )                                             \
                              ( ( room )->exit[ ( y ) ] &&                 \
                              ( TOROOM( ( room ), ( y ) ) != NULL ) &&     \
                              ( !IS_CLOSED( (room), ( y ) ) ) &&           \
                              ( !IS_MARKED( TOROOM( ( room ), ( y ) ) ) ) )
#endif

void list_enqueue( ROOM_INDEX_DATA *room )
{
   static struct room_list_struct *curr;

   curr = alloc_mem( sizeof( struct room_list_struct ) );
   curr->room = room;
   curr->next = NULL;

   if( list_tail != NULL )
   {
      list_tail->next = curr;
      list_tail = curr;
   }
   else
      list_head = list_tail = curr;

   return;
}

void bfs_enqueue( ROOM_INDEX_DATA *room, sh_int dir, sh_int depth )
{
   struct bfs_queue_struct *curr;

   curr = alloc_mem( sizeof( struct bfs_queue_struct ) );
   curr->room = room;
   curr->dir = dir;
   curr->depth = depth + 1;
   curr->next = NULL;

   if( queue_tail != NULL )
   {
      queue_tail->next = curr;
      queue_tail = curr;
   }
   else
      queue_head = queue_tail = curr;

   list_enqueue( room );

   return;
}

void bfs_dequeue(void)
{
   struct bfs_queue_struct *curr;

   curr = queue_head;

   if( (queue_head = queue_head->next) == NULL )
      queue_tail = NULL;

   free_mem( curr, sizeof( struct bfs_queue_struct ) );
   return;
}

void list_dequeue( void )
{
   struct room_list_struct *curr;

   curr = list_head;

   if( (list_head = list_head->next) == NULL )
      list_tail = NULL;

   UNMARK( curr->room );
   free_mem( curr, sizeof( struct room_list_struct ) );
   return;
}

void bfs_clear_queue(void) 
{
   while( queue_head != NULL )
      bfs_dequeue();
   return;
}

void list_clear_queue( void )
{
   while( list_head != NULL )
      list_dequeue();
}

/* find_first_step: given a source room and a target room, find the first
   step on the shortest path from the source to the target.

   Intended usage: in mobile_activity, give a mob a dir to go if they're
   tracking another mob or a PC.  Or, a 'track' skill for PCs.
*/


int find_first_step( ROOM_INDEX_DATA *src, ROOM_INDEX_DATA *target, int depth )
{
   int curr_dir;

   if ( src == NULL || target == NULL ) {
      bug("Illegal value passed to find_first_step (graph.c)", 0 );
      return BFS_ERROR;
   }

   if (src == target)
      return BFS_ALREADY_THERE;

   queue_head = queue_tail = NULL;
   list_head = list_tail = NULL;

   MARK( src );
   list_enqueue( src );

   /* first, enqueue the first steps, saving which direction we're going. */
   for (curr_dir = 0; curr_dir < MAX_DIR; curr_dir++)
      if ( VALID_EDGE( src, curr_dir ) ) 
      {
         MARK( TOROOM( src, curr_dir ) );
         bfs_enqueue( TOROOM( src, curr_dir ), curr_dir, 0 );
      }

   /* now, do the classic BFS. */
   while( queue_head )
   {
      if( queue_head->depth >= depth )
      {
         bfs_clear_queue();
         list_clear_queue();
         return BFS_NO_PATH;
      }
      if( queue_head->room == target ) 
      {
	 curr_dir = queue_head->dir;
	 bfs_clear_queue();
         list_clear_queue();
	 return curr_dir;
      }
      else
      {
         for( curr_dir = 0; curr_dir < MAX_DIR; curr_dir++ )
         {
            if (VALID_EDGE( queue_head->room, curr_dir ) ) 
            {
               MARK( TOROOM( queue_head->room, curr_dir ) );
	       bfs_enqueue( TOROOM( queue_head->room, curr_dir  ),
                queue_head->dir, queue_head->depth );
            }
         }
         bfs_dequeue();
      }
   }
   list_clear_queue();
   return BFS_NO_PATH;
}


/************************************************************************
 *  Functions and Commands which use the above fns                      *
 ************************************************************************/

/*
 * Syntax:  track [person]
 */
void do_track ( CHAR_DATA *ch, char *argument ) 
{
   char buf[MAX_STRING_LENGTH];
   char arg[MAX_INPUT_LENGTH];
   CHAR_DATA *vict;
   int dir;
   one_argument(argument, arg);

   if ( arg[0] == '\0' && !MTD(ch->hunting) )
   {
      send_to_char( "Track whom?\n\r", ch);
      return;
   }

   if ( arg[0] == '\0' && !MTD(ch->hunting) )
   {
      do_track( ch, "continue" );
      return;
   }

   if ( str_cmp( arg, "continue" ) )
   {
      if ( ( vict = get_char_world( ch, arg ) ) == NULL )
      {
         send_to_char( "There is no path that leads there.\n\r", ch );
         free_string( ch->hunting );
         ch->hunting = str_dup( "" );
         return;
      }

      if ( ch == vict && !MTD(ch->hunting) )
      {
         send_to_char( "You stop looking for signs of your quarry.\n\r", ch );
         free_string( ch->hunting );
         ch->hunting = str_dup( "" );
         return;
      }

      if ( ch != vict && !MTD(ch->hunting) )
      {
         send_to_char(
          "Type 'track self' to stop tracking your quarry.\n\r", ch );
         return;
      }
   }

   if ( MTD(ch->hunting) )
   {
      send_to_char( "You are not tracking anyone.\n\r", ch );
      free_string( ch->hunting );
      ch->hunting = str_dup( "" );
      return;
   }

   if( ( vict = get_char_world( ch, ch->hunting ) ) == NULL )
   {
      send_to_char("You lost track of your quarry.\n\r", ch);
      free_string( ch->hunting );
      ch->hunting = str_dup( "" );
      return;
   }

   dir = find_first_step( ch->in_room, vict->in_room, TRACK_DEPTH(ch) );

   switch( dir )
   {
      case BFS_ERROR:
         send_to_char( "BFS Error: Report to a god at once.\n\r", ch);
         bug( "do_track: BFS Error", 0 );
         ch->hunting = NULL;
         break;
      case BFS_ALREADY_THERE:
         if ( MTD(ch->hunting) )
             sprintf( buf, "The trail leads you right to %s.\n\r",
                           HIM_HER(vict) );
         else
          sprintf( buf, "You have tracked down %s.", NAME(vict) );
         send_to_char( buf, ch );
         free_string( ch->hunting );
         ch->hunting = str_dup( "" );
         break;
      case BFS_NO_PATH:
         sprintf( buf, "The trail to %s leads nowhere.\n\r", HIM_HER(vict) );
         send_to_char(buf, ch);
         free_string( ch->hunting );
         ch->hunting = str_dup( "" );
         break;
      default:
         /*
          * The next few lines make it give you a random direction if you
          * fail the random skill roll.
          */
         if( !skill_check( ch, gsn_track, 0 ) )
           dir = number_percent( ) % MAX_DIR;

         sprintf(buf, "The trail leads off to %s.\n\r",
                      dir_rev[rev_dir[dir]]);
         send_to_char(buf, ch);
         if( ch->hunting[0] == '\0' )
            ch->hunting = str_dup( arg );
         break;
   }
   return;
}


