/***************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,        *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.   *
 *                                                                         *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael          *
 *  Chastain, Michael Quan, and Mitchell Tse.                              *
 *                                                                         *
 *  In order to use any part of this Merc Diku Mud, you must comply with   *
 *  both the original Diku license in 'license.doc' as well the Merc       *
 *  license in 'license.txt'.  In particular, you may not remove either of *
 *  these copyright notices.                                               *
 *                                                                         *
 *  Much time and thought has gone into this software and you are          *
 *  benefitting.  We hope that you share your changes too.  What goes      *
 *  around, comes around.                                                  *
 ***************************************************************************/

#if defined(macintosh)
#include <types.h>
#else
#include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include "merc.h"



/*
 * Local functions.
 */
bool    check_dodge args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
bool	check_parry	args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
void	dam_message	args( ( CHAR_DATA *ch, CHAR_DATA *victim, int dam,
			    int dt ) );
void	death_cry	args( ( CHAR_DATA *ch ) );
void	group_gain	args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
int     xp_compute  args( ( CHAR_DATA *gch, CHAR_DATA *victim ) );
bool	is_safe		args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
void	make_corpse	args( ( CHAR_DATA *ch ) );
void    one_hit     args( ( CHAR_DATA *ch, CHAR_DATA *victim, int dt,
                            OBJ_DATA *obj ) );
void	raw_kill	args( ( CHAR_DATA *victim ) );
void	set_fighting	args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
void	disarm		args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );
void	trip		args( ( CHAR_DATA *ch, CHAR_DATA *victim ) );



/*
 * Control the fights going on.
 * Called periodically by update_handler.
 */
void violence_update( void )
{
    CHAR_DATA *ch;
    CHAR_DATA *ch_next;
    CHAR_DATA *victim;
    CHAR_DATA *rch;
    CHAR_DATA *rch_next;

    for ( ch = char_list; ch != NULL; ch = ch->next )
    {
	ch_next	= ch->next;

	if ( ( victim = ch->fighting ) == NULL || ch->in_room == NULL )
	    continue;

    script_update( ch, TRIG_COMBAT );

	if ( IS_AWAKE(ch) && ch->in_room == victim->in_room )
	    multi_hit( ch, victim, TYPE_UNDEFINED );
	else
	    stop_fighting( ch, FALSE );

	if ( ( victim = ch->fighting ) == NULL )
	    continue;

    /*
	 * Fun for the whole family!
	 */
	for ( rch = ch->in_room->people; rch != NULL; rch = rch_next )
	{
	    rch_next = rch->next_in_room;

	    if ( IS_AWAKE(rch) && rch->fighting == NULL )
	    {
		/*
		 * PC's auto-assist others in their group.
		 */
		if ( !IS_NPC(ch) || IS_AFFECTED(ch, AFF_CHARM) )
		{
		    if ( ( !IS_NPC(rch) || IS_AFFECTED(rch, AFF_CHARM) )
            && is_same_group(ch, rch) && IS_SET(rch->act2, PLR_ASSIST))
			multi_hit( rch, victim, TYPE_UNDEFINED );
		    continue;
		}

/* mounts assist riders if its possible */
          if ( IS_NPC(rch) 
            && IS_SET(rch->act, ACT_PET)
            && rch == ch->riding 
            && IS_SET(rch->act, ACT_FIGHTSCR) )
               {
               multi_hit( rch, victim, TYPE_UNDEFINED );
               continue;
               }

		/*
                 * NPC's assist NPC's of same type.
		 */
                if ( IS_NPC(rch) && !IS_AFFECTED(rch, AFF_CHARM) )
		{
                    if ( rch->pIndexData == ch->pIndexData )
		    {
			CHAR_DATA *vch;
			CHAR_DATA *target;
                        int number;

			target = NULL;
			number = 0;
			for ( vch = ch->in_room->people; vch; vch = vch->next )
			{
                        if ( can_see( rch, vch )
                        &&   is_same_group( vch, victim )
                        &&   number_range( 0, number ) == 0 )
                        {
                             target = vch;
                             number++;
                        }
                    }

			if ( target != NULL )
			    multi_hit( rch, target, TYPE_UNDEFINED );
		    }
		}
	    }
	}
    }

    return;
}



/*
 * Do one group of attacks.
 */
void multi_hit( CHAR_DATA *ch, CHAR_DATA *victim, int dt )
{
    int     chance;
    OBJ_DATA *wield;

    if ( dt == TYPE_UNDEFINED )
    {
        OBJ_DATA *wield1;
        OBJ_DATA *wield2;

        wield1 = get_eq_char( ch, WEAR_HOLD_1 );
        wield2 = get_eq_char( ch, WEAR_HOLD_2 );

        if ( wield1 != NULL && wield1->item_type == ITEM_WEAPON )
        {
            one_hit( ch, victim, dt, wield1 );
            if ( wield2 != NULL && wield2->item_type == ITEM_WEAPON &&
               ( (!IS_NPC(ch) &&
                ch->pcdata->learned[gsn_dual_wield] > number_range( 0, 100 ))
               || IS_NPC(ch)) )
            {
                one_hit( ch, victim, dt, wield2 );
            }
        }
        else
        if ( wield2 != NULL && wield2->item_type == ITEM_WEAPON )
        {
            one_hit( ch, victim, dt, wield2 );
        }
        else
        {
            one_hit( ch, victim, TYPE_HIT, NULL );
        }
    }
    else  one_hit( ch, victim, TYPE_HIT, NULL );
    if ( ch->fighting != victim || dt == gsn_backstab )
	return;

    wield = get_item_held( ch, ITEM_WEAPON );

    chance = IS_NPC(ch) ? ch->level : ch->pcdata->learned[gsn_second_attack]/2;
    if ( number_percent( ) < chance )
    {
    one_hit( ch, victim, dt, wield );
	if ( ch->fighting != victim )
	    return;
    }

    chance = IS_NPC(ch) ? ch->level : ch->pcdata->learned[gsn_third_attack]/4;
    if ( number_percent( ) < chance )
    {
    one_hit( ch, victim, dt, wield );
	if ( ch->fighting != victim )
	    return;
    }

    chance = IS_NPC(ch) ? ch->level / 2 : 0;
    if ( number_percent( ) < chance )
    one_hit( ch, victim, dt, wield );

    if ( IS_NPC(ch) )
    {
    for ( chance = 0;  chance < MAX_ATTACK_DATA;  chance++ )
    {
        if ( ch->pIndexData->attacks[chance] != NULL
          && number_range( 0, MAX_ATTACK_DATA ) <= 3 )
        {
            one_hit( ch, victim, TYPE_ATTACK_DATA - chance, NULL );
        }
    }
    }
    
    return;
}



/*
 * Hit one guy once.
 */
void one_hit( CHAR_DATA *ch, CHAR_DATA *victim, int dt, OBJ_DATA *obj )
{
    OBJ_DATA *wield;
    int victim_ac;
    int thac0;
    int thac0_00;
    int thac0_32;
    int dam;
    int diceroll;
    int hit_type = WP_HIT;

    /*
     * Can't beat a dead char!
     * Guard against weird room-leavings.
     */
    if ( victim->position == POS_DEAD || ch->in_room != victim->in_room )
	return;

    /*
     * Figure out the type of damage message.
     */
    wield = obj;
    if ( dt == TYPE_UNDEFINED )
    {
        dt = TYPE_HIT;
        if ( wield != NULL && wield->item_type == ITEM_WEAPON )
        {
        dt += wield->value[3];
        hit_type = attack_table[wield->value[3]].hit_type - 1000;
        }
    }
    else
    if ( dt <= TYPE_ATTACK_DATA && IS_NPC(ch) )
    {
    int dn = abs(dt)-2;

    dt = TYPE_HIT;
    if ( ch->pIndexData->attacks[dn] != NULL )
    {
    dt += ch->pIndexData->attacks[dn]->idx;
    hit_type = attack_table[ch->pIndexData->attacks[dn]->idx].hit_type - 1000;
    }
    }

    /*
     * Calculate to-hit-armor-class-0 versus armor.    (THAC0)
     */
    thac0_00 = IS_NPC(ch) ? 20 : class_table[ch->class].thac0_00;
    thac0_32 = IS_NPC(ch) ? 0  : class_table[ch->class].thac0_32;

    thac0     = interpolate( ch->level, thac0_00, thac0_32 ) - GET_HITROLL(ch);
    victim_ac = UMAX( -15, GET_AC(victim) / 10 );
    if ( !can_see( ch, victim ) )
    victim_ac -= 4;
    /*
     * Weapon proficiencies.  Careful, victim_ac aint in 100s
     */
    if ( wield != NULL && wield->item_type == ITEM_WEAPON )
    {
    if ( (IS_NPC(ch) ? ch->level : ch->pcdata->profs[hit_type]) >
         (IS_NPC(victim) ? victim->level : victim->pcdata->profs[hit_type]) )
         victim_ac += 2;
    }

    /*
     * The moment of excitement!
     */
    while ( ( diceroll = number_bits( 5 ) ) >= 20 )
	;

    if ( diceroll == 0
    || ( diceroll != 19 && diceroll < thac0 - victim_ac ) )
    {
	/* Miss. */
	damage( ch, victim, 0, dt );
	tail_chain( );
	return;
    }

    /*
     * Hit.
     * Calc damage.
     */

    if ( IS_NPC(ch) )
    {
	dam = number_range( ch->level / 2, ch->level * 3 / 2 );
	if ( wield != NULL )
	    dam += dam / 2;
    }
    else
    {
    if ( wield != NULL && wield->item_type == ITEM_WEAPON )
    {
        /*
         * Weapon Proficiencies.
         */
        dam = number_range( wield->value[1], wield->value[2] );
        if ( (ch->pcdata->profs[hit_type]+1 >= number_percent( )) )
            dam += dam * (ch->pcdata->profs[hit_type] / 100);

        if ( (ch->pcdata->profs[hit_type]+1 >= number_percent( )) )
        {
        ch->pcdata->profs[hit_type]++;
        if ( ch->pcdata->profs[hit_type] > 50 )
             ch->pcdata->profs[hit_type] = 50;
        }
    }
	else
	    dam = number_range( 1, 4 );
    }

    /*
     * Bonuses.
     */
    dam += GET_DAMROLL(ch);
    if ( !IS_NPC(ch) && ch->pcdata->learned[gsn_enhanced_damage] > 0 )
	dam += dam * ch->pcdata->learned[gsn_enhanced_damage] / 100;
    if ( !IS_AWAKE(victim) )
	dam *= 2;
    if ( dt == gsn_backstab )
	dam *= 2 + ch->level / 8;

    if ( dam <= 0 )
	dam = 1;

    if ( (dt-TYPE_HIT) > 0 && (dt-TYPE_HIT) < MAX_ATTACK )
    {
     if ( attack_table[dt-TYPE_HIT].hit_fun != NULL )
     {
     if ( (*attack_table[dt-TYPE_HIT].hit_fun) (ch, victim, diceroll, dam) )
         return;
     }
    }

    damage( ch, victim, dam, dt );

    /*
     * Damage armor.
     */
    if ( dam >= 5 )
      {
      OBJ_DATA *armor;
        
      armor = get_eq_char( victim, number_range( WEAR_NONE+1, MAX_WEAR ));
      if ( armor != NULL
          && armor->item_type == ITEM_ARMOR 
          && armor->value[1] )
	  {
      armor->value[2]--;
      if ( armor->value[2] <= 0 )
	  {
            act( "$p falls apart!", ch, armor, NULL, TO_CHAR );
            extract_obj( armor );
	  }
      }
      }

    tail_chain( );
    return;
}



/*
 * Inflict damage from a hit.
 */
void damage( CHAR_DATA *ch, CHAR_DATA *victim, int dam, int dt )
{

    if ( victim->position == POS_DEAD )
	return;

    /*
     * Stop up any residual loopholes.
     */
    if ( dam > 10000 )
    {
    bug( "Damage: %d: more than 10000 points!", dam );
    dam = 10000;
    }

    if ( victim != ch )
    {
	/*
	 * Certain attacks are forbidden.
	 * Most other attacks are returned.
	 */
	if ( is_safe( ch, victim ) )
	    return;

	if ( victim->position > POS_STUNNED )
	{
	    if ( victim->fighting == NULL )
		set_fighting( victim, ch );
	    victim->position = POS_FIGHTING;
	}

	if ( victim->position > POS_STUNNED )
	{
	    if ( ch->fighting == NULL )
		set_fighting( ch, victim );

	    /*
	     * If victim is charmed, ch might attack victim's master.
	     */
	    if ( IS_NPC(ch)
	    &&   IS_NPC(victim)
	    &&   IS_AFFECTED(victim, AFF_CHARM)
	    &&   victim->master != NULL
	    &&   victim->master->in_room == ch->in_room
	    &&   number_bits( 3 ) == 0 )
	    {
		stop_fighting( ch, FALSE );
		multi_hit( ch, victim->master, TYPE_UNDEFINED );
		return;
	    }
	}

	/*
	 * More charm stuff.
	 */
	if ( victim->master == ch )
	    stop_follower( victim );

	/*
	 * Inviso attacks ... not.
	 */
	if ( IS_AFFECTED(ch, AFF_INVISIBLE) )
	{
	    affect_strip( ch, gsn_invis );
	    affect_strip( ch, gsn_mass_invis );
	    REMOVE_BIT( ch->affected_by, AFF_INVISIBLE );
	    act( "$n fades into existence.", ch, NULL, NULL, TO_ROOM );
	}

	/*
	 * Damage modifiers.
	 */
	if ( IS_AFFECTED(victim, AFF_SANCTUARY) )
	    dam /= 2;

	if ( IS_AFFECTED(victim, AFF_PROTECT) && IS_EVIL(ch) )
	    dam -= dam / 4;

	if ( dam < 0 )
	    dam = 0;

	/*
	 * Check for disarm, trip, parry, and dodge.
	 */
	if ( dt >= TYPE_HIT )
	{
	    if ( IS_NPC(ch) && number_percent( ) < ch->level / 2 )
		disarm( ch, victim );
	    if ( IS_NPC(ch) && number_percent( ) < ch->level / 2 )
		trip( ch, victim );
	    if ( check_parry( ch, victim ) )
		return;
	    if ( check_dodge( ch, victim ) )
		return;
	}

	dam_message( ch, victim, dam, dt );
    }

    /*
     * Hurt the victim.
     * Inform the victim of his new state.
     */
    victim->hit -= dam;
    if ( !IS_NPC(victim)
    &&   victim->level >= LEVEL_IMMORTAL
    &&   victim->hit < 1 )
	victim->hit = 1;
    update_pos( victim );

    switch( victim->position )
    {
    case POS_MORTAL:
	act( "$n is mortally wounded, and will die soon, if not aided.",
	    victim, NULL, NULL, TO_ROOM );
	send_to_char( 
	    "You are mortally wounded, and will die soon, if not aided.\n\r",
	    victim );
	break;

    case POS_INCAP:
	act( "$n is incapacitated and will slowly die, if not aided.",
	    victim, NULL, NULL, TO_ROOM );
	send_to_char(
	    "You are incapacitated and will slowly die, if not aided.\n\r",
	    victim );
	break;

    case POS_STUNNED:
	act( "$n is stunned, but will probably recover.",
	    victim, NULL, NULL, TO_ROOM );
	send_to_char("You are stunned, but will probably recover.\n\r",
	    victim );
	break;

    case POS_DEAD:
	act( "$n is DEAD!!", victim, 0, 0, TO_ROOM );
        send_to_char( "You have been KILLED!!\n\r\n\r", victim );
	break;

    default:
	if ( dam > victim->max_hit / 4 )
	    send_to_char( "That really did HURT!\n\r", victim );
	if ( victim->hit < victim->max_hit / 4 )
	    send_to_char( "You sure are BLEEDING!\n\r", victim );
	break;
    }

    /*
     * Sleep spells and extremely wounded folks.
     */
    if ( !IS_AWAKE(victim) )
	stop_fighting( victim, FALSE );

    /*
     * Payoff for killing things.
     */
    if ( victim->position == POS_DEAD )
    {
	group_gain( ch, victim );

	if ( !IS_NPC(victim) )
	{
        char  buf[MAX_STRING_LENGTH];

	    sprintf( log_buf, "%s killed by %s at %d",
        NAME(victim),
        NAME(ch),
		victim->in_room->vnum );
	    log_string( log_buf );

        if ( victim->bounty == 0 )
            ch->bounty += 80;
       else {
               ch->owed   += victim->bounty;
               victim->bounty = 0;
	     }

        sprintf( buf, "Notify> %s", log_buf );
        NOTIFY( buf, LEVEL_IMMORTAL, WIZ_NOTIFY_DEATH );

	    /*
	     * Dying penalty:
	     * 1/2 way back to previous level.
	     */
        gain_exp( victim, -(victim->exp/2) );
	}

    if ( HAS_SCRIPT(victim) )
    {
        TRIGGER_DATA *pTrig;

        for ( pTrig = victim->triggers; pTrig != NULL; pTrig = pTrig->next )
        {
            if ( pTrig->trigger_type == TRIG_DIES
              && pTrig->script == NULL
              && !IS_SET(ch->act, ACT_HALT) )
            {
                pTrig->current = pTrig->script;
                act_trigger( victim, pTrig->name, NULL, NAME(ch), NULL );
                script_interpret( victim, pTrig );
                if ( IS_SET(pTrig->bits, SCRIPT_ADVANCE) )
                     pTrig->script = pTrig->script->next;
            }
        }
    }

    raw_kill( victim );

    if ( !IS_NPC(ch) && IS_NPC(victim) )
    {
        int old_gold = 0;

        char buf[MAX_STRING_LENGTH];

        old_gold = tally_coins( ch );

        if ( IS_SET(ch->act2, PLR_AUTOLOOT) )      do_get( ch, "all corpse" );
        else if ( IS_SET(ch->act2, PLR_AUTOGOLD) ) do_get( ch, "all.money corpse" );

        if ( IS_SET(ch->act2, PLR_AUTOSAC) )
		do_sacrifice( ch, "corpse" );

        if ( IS_SET( ch->act2, PLR_AUTOSPLIT )
          && tally_coins( ch ) - old_gold > 1 )
        {
            sprintf( buf, "%d",  tally_coins( ch ) - old_gold );
            do_split( ch, buf );
        }
	}

	return;
    }

    if ( victim == ch )
	return;

    /*
     * Take care of link dead people.
     */
    if ( !IS_NPC(victim) && victim->desc == NULL )
    {
	if ( number_range( 0, victim->wait ) == 0 )
	{
	    do_recall( victim, "" );
	    return;
	}
    }

    /*
     * Wimp out?
     */
    if ( IS_NPC(victim) && dam > 0 )
    {
	if ( ( IS_SET(victim->act, ACT_WIMPY) && number_bits( 1 ) == 0
	&&   victim->hit < victim->max_hit / 2 )
	||   ( IS_AFFECTED(victim, AFF_CHARM) && victim->master != NULL
	&&     victim->master->in_room != victim->in_room ) )
	    do_flee( victim, "" );
    }

    tail_chain( );
    return;
}



bool is_safe( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( IS_NPC(ch) || IS_NPC(victim) )
	return FALSE;

    /* Thx Josh! */
    if ( victim->fighting == ch )
	return FALSE;

    if ( IS_SET( ch->in_room->room_flags, ROOM_SAFE ) )
    return TRUE;

    return FALSE;
}



/*
 * See if an attack justifies a KILLER flag.
 */
/*
 * Check for parry.
 */
bool check_parry( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( !IS_AWAKE(victim) )
	return FALSE;

    if ( get_item_held( victim, ITEM_WEAPON ) == NULL )
	    return FALSE;

    if ( !skill_check( victim, gsn_parry, IS_NPC(victim) ? 0 : victim->pcdata->learned[gsn_parry]/2 ) )
	return FALSE;

    act( "You parry $n's attack.",  ch, NULL, victim, TO_VICT    );
    act( "$N parries your attack.", ch, NULL, victim, TO_CHAR    );
    return TRUE;
}



/*
 * Check for dodge.
 */
bool check_dodge( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( !IS_AWAKE(victim) )
	return FALSE;

    if ( !skill_check( victim, gsn_dodge, IS_NPC(victim) ? 0 : victim->pcdata->learned[gsn_parry]/2 ) )
        return FALSE;

    act( "You dodge $n's attack.", ch, NULL, victim, TO_VICT    );
    act( "$N dodges your attack.", ch, NULL, victim, TO_CHAR    );
    return TRUE;
}



/*
 * Set position of a victim.
 */
void update_pos( CHAR_DATA *victim )
{
    if ( victim->hit > 0 )
    {
    	if ( victim->position <= POS_STUNNED )
	    victim->position = POS_STANDING;
	return;
    }

    if ( IS_NPC(victim) || victim->hit <= -11 )
    {
	victim->position = POS_DEAD;
	return;
    }

         if ( victim->hit <= -6 ) victim->position = POS_MORTAL;
    else if ( victim->hit <= -3 ) victim->position = POS_INCAP;
    else                          victim->position = POS_STUNNED;

    return;
}



/*
 * Start fights.
 */
void set_fighting( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( ch->fighting != NULL )
    {
	bug( "Set_fighting: already fighting", 0 );
	return;
    }

    if ( IS_AFFECTED(ch, AFF_SLEEP) )
	affect_strip( ch, gsn_sleep );

    ch->fighting = victim;
    ch->position = POS_FIGHTING;

    return;
}



/*
 * Stop fights.
 */
void stop_fighting( CHAR_DATA *ch, bool fBoth )
{
    CHAR_DATA *fch;

    for ( fch = char_list; fch != NULL; fch = fch->next )
    {
	if ( fch == ch || ( fBoth && fch->fighting == ch ) )
	{
	    fch->fighting	= NULL;
        fch->position   = POS_STANDING;
        update_pos( fch );
    }
    }

    return;
}


/*
 * Make a corpse out of character.
 */
void make_corpse( CHAR_DATA *ch )
{
    char buf[MAX_STRING_LENGTH];
    OBJ_DATA *corpse;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;
    char *name;

    if ( IS_NPC(ch) )
    {
    name        = NAME(ch);
	corpse		= create_object(get_obj_index(OBJ_VNUM_CORPSE_NPC), 0);
    corpse->timer   = number_range( 2, 20 );
    }
    else
    {
    name        = NAME(ch);
	corpse		= create_object(get_obj_index(OBJ_VNUM_CORPSE_PC), 0);
	corpse->timer	= number_range( 25, 40 );
    }

    sprintf( buf, STR(corpse, short_descr), name );
    free_string( corpse->short_descr );
    corpse->short_descr = str_dup( buf );

    sprintf( buf, STR(corpse, description), name );
    free_string( corpse->description );
    corpse->description = str_dup( buf );

    for ( obj = ch->carrying; obj != NULL; obj = obj_next )
    {
	obj_next = obj->next_content;
	obj_from_char( obj );
	if ( IS_SET( obj->extra_flags, ITEM_INVENTORY ) )
	    extract_obj( obj );
	else
	    obj_to_obj( obj, corpse );
    }

    obj_to_room( corpse, ch->in_room );
    return;
}



/*
 * Improved Death_cry contributed by Diavolo.
 */
void death_cry( CHAR_DATA *ch )
{
    ROOM_INDEX_DATA *was_in_room;
    char *msg;
    int door;
    int vnum;

    vnum = 0;
    switch ( number_range( 0, 32 ) )
    {
    default: msg  = "You hear $n's death cry.";				break;
    case  0: msg  = "$n hits the ground ... DEAD.";			break;
    case  1: msg  = "$n splatters blood on your armor.";		break;
    case  2: msg  = "You smell $n's sphincter releasing in death.";
	     vnum = OBJ_VNUM_FINAL_TURD;				break;
    case  3: msg  = "$n's severed head plops on the ground.";
	     vnum = OBJ_VNUM_SEVERED_HEAD;				break;
    case  4: msg  = "$n's heart is torn from $s chest.";
	     vnum = OBJ_VNUM_TORN_HEART;				break;
    case  5: msg  = "$n's arm is sliced from $s dead body.";
	     vnum = OBJ_VNUM_SLICED_ARM;				break;
    case  6: msg  = "$n's leg is sliced from $s dead body.";
	     vnum = OBJ_VNUM_SLICED_LEG;				break;
    }

    act( msg, ch, NULL, NULL, TO_ROOM );

    if ( vnum != 0 )
    {
	char buf[MAX_STRING_LENGTH];
	OBJ_DATA *obj;
	char *name;

    name        = NAME(ch);
	obj		= create_object( get_obj_index( vnum ), 0 );
	obj->timer	= number_range( 4, 7 );

    sprintf( buf, STR(obj, short_descr), name );
	free_string( obj->short_descr );
	obj->short_descr = str_dup( buf );

    sprintf( buf, STR(obj, description), name );
	free_string( obj->description );
	obj->description = str_dup( buf );

	obj_to_room( obj, ch->in_room );
    }

    if ( IS_NPC(ch) )
	msg = "You hear something's death cry.";
    else
	msg = "You hear someone's death cry.";

    was_in_room = ch->in_room;
    for ( door = 0; door < MAX_DIR; door++ )
    {
	EXIT_DATA *pexit;

	if ( ( pexit = was_in_room->exit[door] ) != NULL
	&&   pexit->to_room != NULL
	&&   pexit->to_room != was_in_room )
	{
	    ch->in_room = pexit->to_room;
	    act( msg, ch, NULL, NULL, TO_ROOM );
	}
    }
    ch->in_room = was_in_room;

    return;
}



void raw_kill( CHAR_DATA *victim )
{
    stop_fighting( victim, TRUE );
    death_cry( victim );

    if ( victim->mounted_by != NULL )
    {
        act( "$n dies suddenly, and you fall to the ground.", 
             victim, NULL, victim->mounted_by, TO_VICT );
        victim->mounted_by->riding     = NULL;
        victim->mounted_by->position   = POS_RESTING;
        victim->mounted_by             = NULL;
    }

    if ( victim->riding != NULL )
    {
        act( "$n falls off you, dead.", 
             victim, NULL, victim->riding, TO_VICT );
        victim->riding->mounted_by = NULL;
        victim->riding             = NULL;
    }
     

    if ( IS_NPC(victim) )
    {
    make_corpse( victim );
	victim->pIndexData->killed++;
	kill_table[URANGE(0, victim->level, MAX_LEVEL-1)].killed++;
	extract_char( victim, TRUE );
	return;
    }

    if ( victim->level < 2 )
    {
      send_to_char( "The gods frown from the heavens.  A booming echo sounds\n\r", victim );
      send_to_char( "across the land.  One speaks:\n\r", victim );
      send_to_char( "\"Small one!  Have no fear, the gods are smiling upon you\n\r", victim );
      send_to_char( "today!  We cannot let you die so young!  We will save as\n\r", victim );
      send_to_char( "much as is within our powers.  But you will lose some of your\n\r", victim );
      send_to_char( "experience, yet your items will remain on you, and we will\n\r", victim );
      send_to_char( "resurrect what of you that we can in the entrance to the mud\n\r", victim );
      send_to_char( "school (in hopes you will travel its halls again to learn from\n\r", victim );
      send_to_char( "your most haenous mistake).\"\n\r", victim );

      victim->hit       = number_range( 1, victim->max_hit );
      victim->move      = victim->max_move;
      victim->position  = POS_SLEEPING;
      char_from_room( victim );
      char_to_room( victim, get_room_index( ROOM_VNUM_SCHOOL ) );
      return;
    }

    make_corpse( victim );
    extract_char( victim, FALSE );
    while ( victim->affected ) affect_remove( victim, victim->affected );
    victim->affected_by = race_table[victim->race].affect_bits;
    victim->armor       = 100;
    victim->position	= POS_RESTING;
    victim->hit		= UMAX( 1, victim->hit  );
    victim->move        = UMAX( 1, victim->move );
    victim->pcdata->condition[COND_DRUNK]   = 0;
    victim->pcdata->condition[COND_FULL]    = 20;
    victim->pcdata->condition[COND_THIRST]  = 20;
    save_char_obj( victim );
    return;
}



void group_gain( CHAR_DATA *ch, CHAR_DATA *victim )
{
    char buf[MAX_STRING_LENGTH];
    CHAR_DATA *gch;
    CHAR_DATA *lch;
    int xp;
    int members;

    /*
     * Monsters don't get kill xp's or alignment changes.
     * P-killing doesn't help either.
     * Dying of mortal wounds or poison doesn't give xp to anyone!
     */
    if ( IS_NPC(ch) || !IS_NPC(victim) || victim == ch )
	return;
    
    members = 0;
    for ( gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room )
    {
	if ( is_same_group( gch, ch ) )
	    members++;
    }

    if ( members == 0 )
    {
	bug( "Group_gain: members.", members );
	members = 1;
    }

    lch = (ch->leader != NULL) ? ch->leader : ch;

    for ( gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room )
    {
	OBJ_DATA *obj;
	OBJ_DATA *obj_next;

	if ( !is_same_group( gch, ch ) )
	    continue;

    if ( gch->level - lch->level >  5 )
	{
        send_to_char( "You are too high for this group, and gain no experience.\n\r", gch );
	    continue;
	}

    if ( gch->level - lch->level < -5 )
	{
        send_to_char( "You are too low for this group, and gain no experience.\n\r", gch );
	    continue;
    }

        xp = (victim->exp / members) + (victim->exp / 2);
        if (members > 1)
        sprintf( buf, "You receive %d of %d total experience points.\n\r",
                      xp, xp * members );
   else sprintf( buf, "You receive %d total experience points.\n\r", xp );

        if (xp == 0)  sprintf( buf, "You receive no experience.\n\r" );

    send_to_char( buf, gch );
	gain_exp( gch, xp );

	for ( obj = ch->carrying; obj != NULL; obj = obj_next )
	{
	    obj_next = obj->next_content;
	    if ( obj->wear_loc == WEAR_NONE )
		continue;

	    if ( ( IS_OBJ_STAT(obj, ITEM_ANTI_EVIL)    && IS_EVIL(ch)    )
	    ||   ( IS_OBJ_STAT(obj, ITEM_ANTI_GOOD)    && IS_GOOD(ch)    )
	    ||   ( IS_OBJ_STAT(obj, ITEM_ANTI_NEUTRAL) && IS_NEUTRAL(ch) ) )
	    {
		act( "You are zapped by $p.", ch, obj, NULL, TO_CHAR );
		act( "$n is zapped by $p.",   ch, obj, NULL, TO_ROOM );
		obj_from_char( obj );
		obj_to_room( obj, ch->in_room );
	    }
	}
    }

    return;
}



/*
 * Compute xp for a kill.
 * Also adjust alignment of killer.
 * Edit this function to change xp computations.
 */
int xp_compute( CHAR_DATA *gch, CHAR_DATA *victim )
{
    int align;
    int xp;
    int extra;
    int level;
    int number;

    xp    = 300 - URANGE( -3, gch->level - victim->level, 6 ) * 50;
    align = gch->alignment - victim->alignment;

    if ( align >  500 )
    {
	gch->alignment  = UMIN( gch->alignment + (align-500)/4,  1000 );
	xp = 5 * xp / 4;
    }
    else if ( align < -500 )
    {
	gch->alignment  = UMAX( gch->alignment + (align+500)/4, -1000 );
    }
    else
    {
	gch->alignment -= gch->alignment / 4;
	xp = 3 * xp / 4;
    }

    /*
     * Adjust for popularity of target:
     *   -1/8 for each target over  'par' (down to -100%)
     *   +1/8 for each target under 'par' (  up to + 25%)
     */
    level  = URANGE( 0, victim->level, MAX_LEVEL - 1 );
    number = UMAX( 1, kill_table[level].number );
    extra  = victim->pIndexData->killed - kill_table[level].killed / number;
    xp    -= xp * URANGE( -2, extra, 8 ) / 8;

    xp     = number_range( xp * 3 / 4, xp * 5 / 4 );
    xp     = UMAX( 0, xp );

    return xp/2;
}






/*
 * Disarm a creature.
 * Caller must check for successful attack.
 */
void disarm( CHAR_DATA *ch, CHAR_DATA *victim )
{
    OBJ_DATA *obj;

    if ( ( obj = get_item_held( victim, ITEM_WEAPON ) ) == NULL )
	return;

    if ( get_item_held( ch, ITEM_WEAPON ) == NULL )
	return;

    if ( IS_SET( obj->extra_flags, ITEM_NOREMOVE ) || IS_IMMORTAL(ch) )
    {
        act( "$B$n tries to disarm you, but $P won't come off your hand!$R",
             ch, NULL, victim, TO_VICT );
        act( "You try to knock the weapon from $S hand, but it is useless!",
             ch, NULL, victim,      TO_CHAR );
        return;
    }

    act( "$B$n disarms you and sends your weapon flying!$R", ch, NULL, victim,
                                                                   TO_VICT );
    act( "You have managed to disarm $N!",  ch, NULL, victim,      TO_CHAR );
    act( "$n disarms $N!",  ch, NULL, victim, TO_NOTVICT );

    obj_from_char( obj );
    obj_to_room( obj, victim->in_room );

    return;
}



/*
 * Trip a creature.
 * Caller must check for successful attack.
 */
void trip( CHAR_DATA *ch, CHAR_DATA *victim )
{

    if ( IS_AFFECTED( victim, AFF_FREEACTION )
      || IS_AFFECTED( victim, AFF_FLYING )
      || IS_IMMORTAL(ch) )
    {
    act( "$n tries to trip you, but you float over it!", ch, NULL, victim, TO_VICT );
    act( "You try to trip $N, but $N floats over it!", ch, NULL, victim, TO_CHAR );
    act( "$n tries to trip $N, but $N floats over it!", ch, NULL, victim, TO_NOTVICT );
    return;
    }

    if ( victim->wait == 0 )
    {
    act( "$B$n trips you and you go down!$R", ch, NULL, victim, TO_VICT    );
    act( "You trip $N and $E goes down!", ch, NULL, victim, TO_CHAR    );
    act( "$n trips $N!", ch, NULL, victim, TO_NOTVICT );

	WAIT_STATE( ch,     2 * PULSE_VIOLENCE );
	WAIT_STATE( victim, 2 * PULSE_VIOLENCE );
	victim->position = POS_RESTING;
    }

    return;
}



void do_kill( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;

    if ( IS_NPC(ch) && IS_SET(ch->act, ACT_PET) )  /* NOT */
       return;

    one_argument( argument, arg );

    if ( arg[0] == '\0' )
    {
	send_to_char( "Kill whom?\n\r", ch );
	return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
	send_to_char( "They aren't here.\n\r", ch );
	return;
    }

    if ( victim == ch )
    {
	send_to_char( "You hit yourself.  Ouch!\n\r", ch );
	multi_hit( ch, ch, TYPE_UNDEFINED );
	return;
    }

    if ( is_safe( ch, victim ) )
	return;

    if ( IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim )
    {
	act( "$N is your beloved master.", ch, NULL, victim, TO_CHAR );
	return;
    }

    if ( ch->position == POS_FIGHTING )
    {
	send_to_char( "You do the best you can!\n\r", ch );
	return;
    }

    WAIT_STATE( ch, 1 * PULSE_VIOLENCE );
    multi_hit( ch, victim, TYPE_UNDEFINED );
    return;
}



void do_murde( CHAR_DATA *ch, char *argument )
{
    send_to_char( "If you want to MURDER, spell it out.\n\r", ch );
    return;
}



void do_murder( CHAR_DATA *ch, char *argument )
{
    char buf[MAX_STRING_LENGTH];
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;

    one_argument( argument, arg );

    if ( arg[0] == '\0' )
    {
	send_to_char( "Murder whom?\n\r", ch );
	return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
	send_to_char( "They aren't here.\n\r", ch );
	return;
    }

    if ( victim == ch )
    {
	send_to_char( "Suicide is a mortal sin.\n\r", ch );
	return;
    }

    if ( is_safe( ch, victim ) )
	return;

    if ( IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim )
    {
	act( "$N is your beloved master.", ch, NULL, victim, TO_CHAR );
	return;
    }

    if ( ch->position == POS_FIGHTING )
    {
	send_to_char( "You do the best you can!\n\r", ch );
	return;
    }

    WAIT_STATE( ch, 1 * PULSE_VIOLENCE );
    sprintf( buf, "Help!  I am being attacked by %s!", NAME(ch) );
    do_shout( victim, buf );
    multi_hit( ch, victim, TYPE_UNDEFINED );
    return;
}



void do_backstab( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int dt;

    one_argument( argument, arg );

    if ( arg[0] == '\0' )
    {
	send_to_char( "Backstab whom?\n\r", ch );
	return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
	send_to_char( "They aren't here.\n\r", ch );
	return;
    }

    if ( victim == ch )
    {
	send_to_char( "How can you sneak up on yourself?\n\r", ch );
	return;
    }

    if ( is_safe( ch, victim ) )
      return;

    if ( ( obj = get_item_held( ch, ITEM_WEAPON ) ) == NULL )
    {
    send_to_char( "You need to wield a weapon.\n\r", ch );
    return;
    }

    dt = obj->value[3];

    if ( attack_table[dt].hit_type != TYPE_PIERCE )
    {
	send_to_char( "You need to wield a piercing weapon.\n\r", ch );
	return;
    }

    if ( victim->fighting != NULL )
    {
	send_to_char( "You can't backstab a fighting person.\n\r", ch );
	return;
    }

    if ( victim->hit < victim->max_hit/2 && victim->position != POS_SLEEPING )
    {
	act( "$N is hurt and suspicious ... you can't sneak up.",
	    ch, NULL, victim, TO_CHAR );
	return;
    }

    WAIT_STATE( ch, skill_table[gsn_backstab].beats );
    if ( !IS_AWAKE(victim)
    ||   IS_NPC(ch)
    ||   number_percent( ) < ch->pcdata->learned[gsn_backstab] )
	multi_hit( ch, victim, gsn_backstab );
    else
	damage( ch, victim, 0, gsn_backstab );

    return;
}



void do_flee( CHAR_DATA *ch, char *argument )
{
    ROOM_INDEX_DATA *was_in;
    ROOM_INDEX_DATA *now_in;
    CHAR_DATA *victim;
    int attempt;

    if ( ( victim = ch->fighting ) == NULL )
    {
	if ( ch->position == POS_FIGHTING )
	    ch->position = POS_STANDING;
	send_to_char( "You aren't fighting anyone.\n\r", ch );
	return;
    }

    was_in = ch->in_room;
    for ( attempt = 0; attempt < 6; attempt++ )
    {
	EXIT_DATA *pexit;
	int door;

	door = number_door( );
	if ( ( pexit = was_in->exit[door] ) == 0
	||   pexit->to_room == NULL
	||   IS_SET(pexit->exit_info, EX_CLOSED)
	|| ( IS_NPC(ch)
	&&   IS_SET(pexit->to_room->room_flags, ROOM_NO_MOB) ) )
	    continue;

	move_char( ch, door );
	if ( ( now_in = ch->in_room ) == was_in )
	    continue;

	ch->in_room = was_in;
	act( "$n has fled!", ch, NULL, NULL, TO_ROOM );
	ch->in_room = now_in;

	if ( !IS_NPC(ch) )
	{
	    send_to_char( "You flee from combat!  You lose 25 exps.\n\r", ch );
	    gain_exp( ch, -25 );
	}

	stop_fighting( ch, TRUE );
	return;
    }

    send_to_char( "You failed!  You lose 10 exps.\n\r", ch );
    gain_exp( ch, -10 );
    return;
}



void do_rescue( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    CHAR_DATA *fch;

    one_argument( argument, arg );
    if ( arg[0] == '\0' )
    {
	send_to_char( "Rescue whom?\n\r", ch );
	return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
	send_to_char( "They aren't here.\n\r", ch );
	return;
    }

    if ( victim == ch )
    {
	send_to_char( "What about fleeing instead?\n\r", ch );
	return;
    }

    if ( !IS_NPC(ch) && IS_NPC(victim) )
    {
	send_to_char( "Doesn't need your help!\n\r", ch );
	return;
    }

    if ( ch->fighting == victim )
    {
	send_to_char( "Too late.\n\r", ch );
	return;
    }

    if ( ( fch = victim->fighting ) == NULL )
    {
	send_to_char( "That person is not fighting right now.\n\r", ch );
	return;
    }

    WAIT_STATE( ch, skill_table[gsn_rescue].beats );
    if ( !IS_NPC(ch) && number_percent( ) > ch->pcdata->learned[gsn_rescue] )
    {
	send_to_char( "You fail the rescue.\n\r", ch );
	return;
    }

    act( "You rescue $N!",  ch, NULL, victim, TO_CHAR    );
    act( "$n rescues you!", ch, NULL, victim, TO_VICT    );
    act( "$n rescues $N!",  ch, NULL, victim, TO_NOTVICT );

    stop_fighting( fch, FALSE );
    stop_fighting( ch, FALSE );
    stop_fighting( victim, FALSE );

    set_fighting( ch, fch );
    set_fighting( fch, ch );
    return;
}



void do_kick( CHAR_DATA *ch, char *argument )
{
    CHAR_DATA *victim;

    if ( !IS_NPC(ch)
    &&   ch->level < skill_table[gsn_kick].skill_level[ch->class] )
    {
	send_to_char(
	    "You better leave the martial arts to fighters.\n\r", ch );
	return;
    }

    if ( ( victim = ch->fighting ) == NULL )
    {
	send_to_char( "You aren't fighting anyone.\n\r", ch );
	return;
    }

    WAIT_STATE( ch, skill_table[gsn_kick].beats );
    if ( IS_NPC(ch) || number_percent( ) < ch->pcdata->learned[gsn_kick] )
	damage( ch, victim, number_range( 1, ch->level ), gsn_kick );
    else
	damage( ch, victim, 0, gsn_kick );

    return;
}




void do_disarm( CHAR_DATA *ch, char *argument )
{
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int percent;

    if ( !IS_NPC(ch)
    &&   ch->level < skill_table[gsn_disarm].skill_level[ch->class] )
    {
	send_to_char( "You don't know how to disarm opponents.\n\r", ch );
	return;
    }

    if ( get_item_held( ch, ITEM_WEAPON ) == NULL )
    {
	send_to_char( "You must wield a weapon to disarm.\n\r", ch );
	return;
    }

    if ( ( victim = ch->fighting ) == NULL )
    {
	send_to_char( "You aren't fighting anyone.\n\r", ch );
	return;
    }

    if ( ( obj = get_item_held( victim, ITEM_WEAPON ) ) == NULL )
    {
	send_to_char( "Your opponent is not wielding a weapon.\n\r", ch );
	return;
    }

    WAIT_STATE( ch, skill_table[gsn_disarm].beats );
    percent = number_percent( ) + victim->level - ch->level;
    if ( skill_check( ch, gsn_disarm, 2 ) )
	disarm( ch, victim );
    else
	send_to_char( "You failed.\n\r", ch );
    return;
}



void do_sla( CHAR_DATA *ch, char *argument )
{
    send_to_char( "If you want to SLAY, spell it out.\n\r", ch );
    return;
}



void do_slay( CHAR_DATA *ch, char *argument )
{
    CHAR_DATA *victim;
    char arg[MAX_INPUT_LENGTH];

    one_argument( argument, arg );
    if ( arg[0] == '\0' )
    {
	send_to_char( "Slay whom?\n\r", ch );
	return;
    }

    if ( ( victim = get_char_room( ch, arg ) ) == NULL )
    {
	send_to_char( "They aren't here.\n\r", ch );
	return;
    }

    if ( ch == victim )
    {
	send_to_char( "Suicide is a mortal sin.\n\r", ch );
	return;
    }

    if ( !IS_NPC(victim) && victim->level >= ch->level )
    {
	send_to_char( "You failed.\n\r", ch );
	return;
    }

    act( "You grasp $S head and squeeze it until it explodes!",
                                            ch, NULL, victim, TO_CHAR    );
    act( "$n grasps your head and squeezes until it explodes!",
                                            ch, NULL, victim, TO_VICT    );
    act( "$n grasps $N's head and squeezes until it explodes!",
                                            ch, NULL, victim, TO_NOTVICT );
    raw_kill( victim );
    return;
}




void dam_message( CHAR_DATA *ch, CHAR_DATA *victim, int dam, int dt )
{
    char buf1[256], buf2[256], buf3[256];
    const char *vs;
    const char *vp;
    const char *attack;
    char punct;

         if ( dam ==   0 ) { vs =  "miss";           vp =  "misses";          }
    else if ( dam <=   4 ) { vs =  "scratch";        vp =  "scratches";       }
    else if ( dam <=   8 ) { vs =  "graze";          vp =  "grazes";          }
    else if ( dam <=  12 ) { vs =  "hit";            vp =  "hits";            }
    else if ( dam <=  16 ) { vs =  "injure";         vp =  "injures";         }
    else if ( dam <=  20 ) { vs =  "wound";          vp =  "wounds";          }
    else if ( dam <=  24 ) { vs =  "maul";           vp =  "mauls";           }
    else if ( dam <=  28 ) { vs =  "devastate";      vp =  "devastates";      }
    else if ( dam <=  32 ) { vs =  "maim";           vp =  "maims";           }
    else if ( dam <=  36 ) { vs =  "$7MUTILATE$R";   vp =  "$7MUTILATES$R";   }
    else if ( dam <=  40 ) { vs =  "$7DECIMATE$R";   vp =  "$7DECIMATES$R";   }
    else if ( dam <=  44 ) { vs =  "$BEVISCERATE$R"; vp =  "$BEVISCERATES$R"; }
    else if ( dam <=  48 ) { vs =  "$BMASSACRE$R";   vp =  "$BMASSACRES$R";   }
    else if ( dam <= 100 ) { vs =  "$4*** $BDEMOLISH$R$4 ***$R";
                             vp =  "$4*** $BDEMOLISHES$R$4 ***$R";            }
    else                   { vs =  "$4$B*** ANNIHILATE ***$R";
                             vp =  "$4$B*** ANNIHILATES ***$R";               }

    punct   = (dam <= 24) ? '.' : '!';

    if ( dt == gsn_backstab )
    {
    sprintf( buf1, "$n slips $s weapon into $N's back and %s $M%c", vp, punct );

    sprintf( buf2, "You slip your weapon into $N's back and %s $M%c", vs, punct 
);
    sprintf( buf3, "$n slips $s weapon into your back and %s you%c", vp, punct )
;
    }
    else
    if ( dt == TYPE_HIT || dt < 0 )
    {
        sprintf( buf1, "$n %s $N%c",  vp, punct );
        sprintf( buf2, "You %s $N%c", vs, punct );
        sprintf( buf3, "$n %s you%c", vp, punct );
    }
    else
    {
        if ( dt >= 0 && dt < MAX_SKILL )
            attack      = skill_table[dt].noun_damage;
    else
      if ( (dt - TYPE_HIT) < MAX_ATTACK )
      {
        attack  = attack_table[dt - TYPE_HIT].name;
      }
        else
        {
            bug( "Dam_message: bad dt %d.", dt );
            dt  = TYPE_HIT;
        attack  = attack_table[0].name;
        }

        sprintf( buf1, "$n's %s %s $N%c",  attack, vp, punct );
        sprintf( buf2, "Your %s %s $N%c",  attack, vp, punct );
        sprintf( buf3, "$n's %s %s you%c", attack, vp, punct );
    }

    act( buf1, ch, NULL, victim, TO_NOTVICT );
    act( buf2, ch, NULL, victim, TO_CHAR );
    act( buf3, ch, NULL, victim, TO_VICT );

    return;
}


void do_shoot( CHAR_DATA *ch, char *argument )
{
    OBJ_DATA *obj;
    OBJ_DATA *newobj;
    char arg1[MAX_INPUT_LENGTH];
    char arg2[MAX_INPUT_LENGTH];
    char arg3[MAX_INPUT_LENGTH];
    char buf[MAX_STRING_LENGTH];
    int range=3;
    int dir, n;
    OBJ_INDEX_DATA *pObjIndex;
    EXIT_DATA *pexit;
    ROOM_INDEX_DATA *troom;
    CHAR_DATA *mob;
    CHAR_DATA *mob_next;

    argument = one_argument( argument, arg1 );
    argument = one_argument( argument, arg2 );
    argument = one_argument( argument, arg3 );

    if ( IS_NPC(ch) && IS_SET(ch->act, ACT_PET) )
       return;

    if ( arg1[0] == '\0' || arg2[0] == '\0' )
    {
         send_to_char("Syntax:  shoot <item> <direction> [range]\n\r", ch);
         return;
    }

    if( (obj = get_obj_held(ch, arg1) ) == NULL )
    {
         if( (obj = get_obj_here(ch, arg1) ) == NULL || obj->value[3] != 2 )
         {
         send_to_char("I don't see that here.\n\r", ch);
         return;
         }
    }

    if ( obj->item_type != ITEM_RANGED_WEAPON )
    {
         send_to_char( "You can't do that.\n\r", ch);
         return;
    }

    if ( obj->value[0] == 0 )
    {
         send_to_char( "Its payload is empty.\n\r", ch);
         return;
    }

    if ( obj->value[1] < 0 || obj->value[2] < 0 )
    {
         send_to_char( "You can't do that.\n\r", ch);
         bug( "Invalid Ranged Weapon (vnum %d)", obj->pIndexData->vnum );
         return;
    }

    if ( obj->value[3] == 2 && arg3[0] )
       range = atoi(arg3);

    if ( range <= 0 && obj->value[3] == 2 )
    {
          send_to_char("Try shooting it away from you.\n\r", ch);
          return;
     }

     if ( range > obj->value[4] && obj->value[3] == 2 )
     {
          if(obj->value[4] > 0) range = obj->value[4];
          else                  range = 1;
     }

     dir = get_dir( arg2 );
     if ( dir >= MAX_DIR )
     {
          do_shoot( ch, "" );
          return;
     }

     if ( obj->value[3] == RNG_BOW )
     {
          sprintf(buf, "You pull back the string on %s and fire %s!\n\r",
                       STR(obj, short_descr), dir_name[dir]);
          send_to_char(buf, ch);
          sprintf(buf, "%s pulls back the string on %s and fires %s!",
                        NAME(ch), STR(obj, short_descr), dir_name[dir]);
          act(buf, ch, NULL, NULL, TO_ROOM);
     }
  

     if ( obj->value[3] == RNG_CROSSBOW )
     {
          sprintf(buf, "You fire a crossbow bolt %s!\n\r", dir_name[dir]);
          send_to_char(buf, ch);
          sprintf(buf, "%s fires a crossbow bolt %s!\n\r", NAME(ch), dir_name[dir]);
          act(buf, ch, NULL, NULL, TO_ROOM);
     }


     if ( obj->value[3] == RNG_CATAPULT )
     {
          sprintf(buf, "You crank back %s, and release it %sward!\n\r",
          STR(obj, short_descr), dir_name[dir]);
          send_to_char(buf, ch);
          sprintf(buf, "%s cranks back %s, and releases it %sward!\n\r",
          NAME(ch), STR(obj,short_descr), dir_name[dir]);
          act(buf, ch, NULL, NULL, TO_ROOM);
     }

    pObjIndex = get_obj_index( obj->value[0] );
    if ( pObjIndex == NULL )
    {
         bug( "Ranged weapon (vnum %d) has invalid ammunition.", 0 );
         return;
    }

    if ( ch->in_room->exit[dir] == NULL
      || IS_SET(ch->in_room->exit[dir]->exit_info, EX_CLOSED) )
    {
        newobj = create_object( get_obj_index(obj->value[0]), 0 );

        obj->value[0] = 0;
        if ( obj->value[3] == RNG_BOW )
           sprintf( buf,
              "%s impales itself into the ground.\n\r", STR(newobj, short_descr) );
   else if ( obj->value[3] == RNG_CROSSBOW )
           sprintf(buf,
              "%s falls to the ground.\n\r", STR(newobj, short_descr) );
   else if ( obj->value[3] == RNG_CATAPULT )
           sprintf(buf,
              "%s crashes to the ground.\n\r", STR(newobj, short_descr) );

        act( buf, ch, NULL, NULL, TO_ROOM );
        send_to_char( capitalize(buf), ch);
        obj_to_room( newobj, ch->in_room );
        return;
    }

    newobj = create_object( pObjIndex, 0 );
 
    pexit = ch->in_room->exit[dir];
    n = 1;
  
    obj->value[0] = 0;

    while(pexit != NULL && n++ < range)
    {
        if ( IS_SET(pexit->exit_info, EX_CLOSED)
          || IS_SET(pexit->exit_info, EX_JAMMED) )
        {
               if ( obj->value[3] == RNG_BOW )
                   sprintf(buf, "%s impales itself into the ground.\n\r",
                                STR(newobj, short_descr) );
          else if ( obj->value[3] == RNG_CROSSBOW )
                   sprintf(buf, "%s falls to the ground.\n\r",
                                STR(newobj, short_descr) );
          else if ( obj->value[3] == RNG_CATAPULT )
                   sprintf(buf, "%s crashes to the ground.\n\r",
                                STR(newobj, short_descr) );

               act( buf, pexit->to_room->people, NULL, NULL, TO_ROOM );
               send_to_char( buf, pexit->to_room->people );
               obj_to_room( newobj, pexit->to_room );
               act( buf, ch, NULL, NULL, TO_ROOM );
               return;
        }

        troom = pexit->to_room;
        if ( troom->people != NULL )
        {
        for( mob = troom->people; mob != NULL; mob = mob_next )
        {
        mob_next = mob->next_in_room;

        if ( mob != NULL
         && (mob->position == POS_STANDING || mob->position == POS_FIGHTING)
         && !skill_check( mob, gsn_dodge, 0 ) )
        {
             int dam;

             sprintf(buf, "%s streaks into %s from %s!\n\r",
                          STR(newobj, short_descr), NAME(mob), dir_rev[dir] );
             act(buf, mob, NULL, NULL, TO_ROOM);
             n = range;

             sprintf(buf, "%s streaks into you from %s!\n\r",
                          STR(newobj, short_descr), dir_rev[dir] );
             act( buf, mob, NULL, NULL, TO_CHAR );

             dam = number_range( obj->value[1], obj->value[2] );
             dam += newobj->value[1];

             /*
              * Weapon Proficiencies.
              */
             if ( !IS_NPC(ch) )
             {
             if ( (ch->pcdata->profs[WP_SHOT]+1 >= number_percent( )) )
                 dam += dam * (ch->pcdata->profs[WP_SHOT] / 100);

             if ( (ch->pcdata->profs[WP_SHOT]+1 >= number_percent( )) )
             {
             ch->pcdata->profs[WP_SHOT]++;
             if ( ch->pcdata->profs[WP_SHOT] > 50 )
                  ch->pcdata->profs[WP_SHOT] = 50;
             }
             }

             extract_obj( newobj );
             damage( ch, mob, dam, 1000 + 11 );
             stop_fighting(mob, FALSE);

             if ( mob->position > POS_STUNNED && obj->value[3] == RNG_CATAPULT )
              mob->position = POS_RESTING;

             return;
        }
        } 
        }

      if ( n <= range && troom->people != NULL )
      {
           sprintf(buf, "%s streaks through the air and continues %s.\n\r",
              STR(newobj, short_descr), dir_name[dir]);
        act(buf, troom->people, NULL, NULL, TO_ROOM);
        act(buf, troom->people, NULL, NULL, TO_CHAR);
      }
      else
      if (troom->people)
      {
           sprintf(buf, "%s zooms into the room and lands on the ground.\n\r",
                        STR(newobj, short_descr) );
           act(buf, troom->people, NULL, NULL, TO_ROOM);
           send_to_char(buf, troom->people);
      }
    pexit = troom->exit[dir];
  }

}

void do_reload( CHAR_DATA *ch, char *argument )
{
    OBJ_DATA *weapon;
    OBJ_DATA *ammo;
    char arg1[MAX_INPUT_LENGTH];
    char arg2[MAX_INPUT_LENGTH];
    char arg3[MAX_INPUT_LENGTH];
    char buf[MAX_STRING_LENGTH];

    argument = one_argument( argument, arg1);
    argument = one_argument( argument, arg2);
    argument = one_argument( argument, arg3);

    if ( IS_NPC(ch) && IS_SET(ch->act, ACT_PET) )
       return;

    if ( arg1[0] == '\0' || arg2[0] == '\0' )
    {
        send_to_char("Syntax:  reload <weapon> <ammo>\n\r", ch);
        return;
    }

    if ( (weapon = get_obj_here(ch, arg1)) == NULL )
    {
        send_to_char( "I don't see that weapon here.\n\r", ch );
        return;
    }

    if( (ammo = get_obj_held(ch, arg2)) == NULL )
    {
        send_to_char("You do not have that ammo in your hand.\n\r", ch);
        return;
    }

    if ( weapon->item_type != ITEM_RANGED_WEAPON
      || ammo->item_type != ITEM_AMMO)
    {
        send_to_char("You can't do that.\n\r", ch);
        return;
    }

    if ( weapon->value[0] != 0 )
    {
        sprintf( buf, "%s is already loaded.\n\r", STR(weapon, short_descr) );
        send_to_char(buf, ch);
        return;
    }

    if ( ammo->value[0] != weapon->value[3] )
    {
        send_to_char("The ammunition is not made for this weapon.\n\r", ch);
        return;
    }

    weapon->value[0] = ammo->pIndexData->vnum;

    act( "You load $p with $P.", ch, ammo, weapon, TO_CHAR );
    act( "$n loads $p with $P.", ch, ammo, weapon, TO_ROOM );

    extract_obj(ammo);
    return;
}
  


bool hit_suck_disarm( CHAR_DATA *ch, CHAR_DATA *victim, int hit, int dam )
{
    OBJ_DATA *obj;

    if ( number_range( 1, 100 ) >= ch->level
    && ( obj = get_item_held( victim, ITEM_WEAPON ) ) != NULL
    && hand_empty( ch ) != WEAR_NONE )
    {
        unequip_char( victim, obj );
        obj->wear_loc = hand_empty( ch );
        obj_from_char( obj );
        obj_to_char( obj, ch );
        act( "You sucks $N's weapon right out of $s hand!",  ch, NULL, victim, TO_CHAR );
        act( "$n sucks your weapon right out of your hand!", ch, NULL, victim, TO_VICT );
        act( "$n sucks $N's weapon right out of $s hand!",   ch, NULL, victim, TO_NOTVICT );

        return TRUE;
    }

    return FALSE;
}



bool hit_vorpal( CHAR_DATA *ch, CHAR_DATA *victim, int hit, int dam )
{
	char buf[MAX_STRING_LENGTH];
	OBJ_DATA *obj;
	char *name;

    if ( hit >= 17 )
    {
        act( "You slice $N's head clean off!",  ch, 0, victim, TO_CHAR );
        act( "$n slices your head clean off!",  ch, 0, victim, TO_VICT );
        act( "$n slices $N's head clean off!",  ch, 0, victim, TO_NOTVICT );
        act( "$n's severed head plops on the ground.", victim, 0, 0, TO_ROOM );
        stop_fighting( victim, TRUE );

        name        = NAME(ch);
        obj         = create_object( get_obj_index( OBJ_VNUM_SEVERED_HEAD ), 0 );
        obj->timer  = number_range( 4, 7 );

        sprintf( buf, STR(obj, short_descr), NAME(victim) );
        free_string( obj->short_descr );
        obj->short_descr = str_dup( buf );

        sprintf( buf, STR(obj, description), NAME(victim) );
        free_string( obj->description );
        obj->description = str_dup( buf );

        obj_to_room( obj, ch->in_room );
        make_corpse( victim );

        if ( IS_NPC(victim) )
        {
        char buf[MAX_STRING_LENGTH];

        sprintf( buf, "You receive %d experience points.\n\r", victim->exp );
        gain_exp( ch, victim->exp );
        send_to_char( buf, ch );    
        victim->pIndexData->killed++;
        kill_table[URANGE(0, victim->level, MAX_LEVEL-1)].killed++;
        extract_char( victim, TRUE );
        return TRUE;
        }

        extract_char( victim, FALSE );
        while ( victim->affected )
        affect_remove( victim, victim->affected );
        victim->affected_by = race_table[victim->race].affect_bits;
        victim->armor   = 100;
        victim->position    = POS_RESTING;
        victim->hit     = UMAX( 1, victim->hit  );
        victim->move    = UMAX( 1, victim->move );
        save_char_obj( victim );

        return TRUE;
    }

    return FALSE;
}
