/***************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,        *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.   *
 *                                                                         *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael          *
 *  Chastain, Michael Quan, and Mitchell Tse.                              *
 *                                                                         *
 *  In order to use any part of this Merc Diku Mud, you must comply with   *
 *  both the original Diku license in 'license.doc' as well the Merc       *
 *  license in 'license.txt'.  In particular, you may not remove either of *
 *  these copyright notices.                                               *
 *                                                                         *
 *  Much time and thought has gone into this software and you are          *
 *  benefitting.  We hope that you share your changes too.  What goes      *
 *  around, comes around.                                                  *
 ***************************************************************************/

#if defined(macintosh)
#include <types.h>
#else
#include <sys/types.h>
#endif
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "merc.h"



/*
 * Local functions.
 */
void	affect_modify	args( ( CHAR_DATA *ch, AFFECT_DATA *paf, bool fAdd ) );


/*
 * Retrieve a character's trusted level for permission checking.
 */
int get_trust( CHAR_DATA *ch )
{
    if ( ch->desc != NULL && ch->desc->original != NULL )
	ch = ch->desc->original;

    if ( ch->trust != 0 )
	return ch->trust;

    if ( IS_NPC(ch) && ch->level >= LEVEL_HERO )
	return LEVEL_HERO - 1;
    else
	return ch->level;
}



/*
 * Retrieve a character's age.
 */
int get_age( CHAR_DATA *ch )
{
    return 17 + (( ch->played + (int) (current_time - ch->logon) ) / 7200)/4;
}



/*
 * Retrieve character's current strength.
 */
int get_curr_str( CHAR_DATA *ch )
{
    int max;

    if ( IS_NPC(ch) )
    return 13+(ch->level/4);

	max = 25;

    return URANGE( 3, ch->pcdata->perm_str + ch->pcdata->mod_str, max );
}



/*
 * Retrieve character's current intelligence.
 */
int get_curr_int( CHAR_DATA *ch )
{
    int max;

    if ( IS_NPC(ch) )
    return 13+(ch->level/4);

	max = 25;

    return URANGE( 3, ch->pcdata->perm_int + ch->pcdata->mod_int, max );
}



/*
 * Retrieve character's current wisdom.
 */
int get_curr_wis( CHAR_DATA *ch )
{
    int max;

    if ( IS_NPC(ch) )
    return 13+(ch->level/4);

	max = 25;

    return URANGE( 3, ch->pcdata->perm_wis + ch->pcdata->mod_wis, max );
}



/*
 * Retrieve character's current dexterity.
 */
int get_curr_dex( CHAR_DATA *ch )
{
    int max;

    if ( IS_NPC(ch) )
    return 13+(ch->level/4);

	max = 25;

    return URANGE( 3, ch->pcdata->perm_dex + ch->pcdata->mod_dex, max );
}



/*
 * Retrieve character's current constitution.
 */
int get_curr_con( CHAR_DATA *ch )
{
    int max;

    if ( IS_NPC(ch) )
    return 13+(ch->level/4);

	max = 25;

    return URANGE( 3, ch->pcdata->perm_con + ch->pcdata->mod_con, max );
}



/*
 * Retrieve a character's carry capacity.
 */
int can_carry_n( CHAR_DATA *ch )
{
    if ( !IS_NPC(ch) && ch->level >= LEVEL_IMMORTAL )
	return 1000;

/*    if ( IS_NPC(ch) && IS_SET(ch->act, ACT_PET) )
    return 0;    */

    return MAX_WEAR + 2 * get_curr_dex( ch ) / 3;
}



/*
 * Retrieve a character's carry capacity.
 */
int can_carry_w( CHAR_DATA *ch )
{
    if ( !IS_NPC(ch) && ch->level >= LEVEL_IMMORTAL )
	return 1000000;

/*    if ( IS_NPC(ch) && IS_SET(ch->act, ACT_PET) )
    return 0;   */

    return str_app[get_curr_str(ch)].carry;
}



/*
 * See if a string is one of the names of an object.
 */
bool is_name( const char *str, char *namelist )
{
    char name[MAX_INPUT_LENGTH];

    for ( ; ; )
    {
	namelist = one_argument( namelist, name );
	if ( name[0] == '\0' )
	    return FALSE;
	if ( !str_cmp( str, name ) )
	    return TRUE;
    }
}



/*
 * Apply or remove an affect to a character.
 */
void affect_modify( CHAR_DATA *ch, AFFECT_DATA *paf, bool fAdd )
{
    OBJ_DATA *wield, *wield_next;
    int mod;
    static int depth;

    mod = paf->modifier;

    if ( fAdd )
    {
	SET_BIT( ch->affected_by, paf->bitvector );
    }
    else
    {
    REMOVE_BIT( ch->affected_by, paf->bitvector );
    if (!IS_SET(ch->affected_by, race_table[ch->race].affect_bits))
        SET_BIT(ch->affected_by, race_table[ch->race].affect_bits);
	mod = 0 - mod;
    }

    if ( IS_NPC(ch) )
	return;

    switch ( paf->location )
    {
    default:
	bug( "Affect_modify: unknown location %d.", paf->location );
    return;

    case APPLY_NONE:						break;
    case APPLY_STR:           ch->pcdata->mod_str	+= mod;	break;
    case APPLY_DEX:           ch->pcdata->mod_dex	+= mod;	break;
    case APPLY_INT:           ch->pcdata->mod_int	+= mod;	break;
    case APPLY_WIS:           ch->pcdata->mod_wis	+= mod;	break;
    case APPLY_CON:           ch->pcdata->mod_con	+= mod;	break;
    case APPLY_SEX:           ch->sex               += mod; break;
    case APPLY_CLASS:                                       break;
    case APPLY_LEVEL:                                       break;
    case APPLY_AGE:                                         break;
    case APPLY_HEIGHT:                                      break;
    case APPLY_WEIGHT:                                      break;
    case APPLY_HIT:           ch->max_hit           += mod; break;
    case APPLY_MOVE:          ch->max_move          += mod; break;
    case APPLY_GOLD:                                        break;
    case APPLY_EXP:                                         break;
    case APPLY_AC:            ch->armor             += mod; break;
    case APPLY_HITROLL:       ch->hitroll           += mod; break;
    case APPLY_DAMROLL:       ch->damroll           += mod; break;
    case APPLY_SAVING_THROW_A:
    case APPLY_SAVING_THROW_B:
    case APPLY_SAVING_THROW_C:
    case APPLY_SAVING_THROW_D:
    case APPLY_SAVING_THROW:  ch->saving_throw      += mod; break;
    }

    if ( depth == 0 )    /* obj_from/to_char recurses */
    {
    depth++;
    for ( wield = ch->carrying; 
          wield != NULL && can_carry_w( ch ) < ch->carry_weight; 
          wield = wield_next )
    {
    wield_next = wield->next_content;
    
    act( "You drop $p.", ch, wield, NULL, TO_CHAR );
    act( "$n drops $p.", ch, wield, NULL, TO_ROOM );
    obj_from_char( wield );
    obj_to_room( wield, ch->in_room );
    }
    depth--;
    }
    return;
}



/*
 * Give an affect to a char.
 */
void affect_to_char( CHAR_DATA *ch, AFFECT_DATA *paf )
{
    AFFECT_DATA *paf_new;

    if ( affect_free == NULL )
    {
	paf_new		= alloc_perm( sizeof(*paf_new) );
    }
    else
    {
	paf_new		= affect_free;
	affect_free	= affect_free->next;
    }

    *paf_new		= *paf;
    paf_new->next	= ch->affected;
    ch->affected	= paf_new;

    affect_modify( ch, paf_new, TRUE );
    return;
}



/*
 * Remove an affect from a char.
 */
void affect_remove( CHAR_DATA *ch, AFFECT_DATA *paf )
{
    if ( ch->affected == NULL )
    {
	bug( "Affect_remove: no affect.", 0 );
	return;
    }

    affect_modify( ch, paf, FALSE );

    if ( paf == ch->affected )
    {
	ch->affected	= paf->next;
    }
    else
    {
	AFFECT_DATA *prev;

	for ( prev = ch->affected; prev != NULL; prev = prev->next )
	{
	    if ( prev->next == paf )
	    {
		prev->next = paf->next;
		break;
	    }
	}

	if ( prev == NULL )
	{
	    bug( "Affect_remove: cannot find paf.", 0 );
	    return;
	}
    }

    paf->next	= affect_free;
    affect_free	= paf->next;
    return;
}



/*
 * Strip all affects of a given sn.
 */
void affect_strip( CHAR_DATA *ch, int sn )
{
    AFFECT_DATA *paf;
    AFFECT_DATA *paf_next;

    for ( paf = ch->affected; paf != NULL; paf = paf_next )
    {
	paf_next = paf->next;
	if ( paf->type == sn )
	    affect_remove( ch, paf );
    }

    return;
}



/*
 * Return true if a char is affected by a spell.
 */
bool is_affected( CHAR_DATA *ch, int sn )
{
    AFFECT_DATA *paf;

    for ( paf = ch->affected; paf != NULL; paf = paf->next )
    {
	if ( paf->type == sn )
	    return TRUE;
    }

    return FALSE;
}



/*
 * Add or enhance an affect.
 */
void affect_join( CHAR_DATA *ch, AFFECT_DATA *paf )
{
    AFFECT_DATA *paf_old;
    bool found;

    found = FALSE;
    for ( paf_old = ch->affected; paf_old != NULL; paf_old = paf_old->next )
    {
	if ( paf_old->type == paf->type )
	{
	    paf->duration += paf_old->duration;
	    paf->modifier += paf_old->modifier;
	    affect_remove( ch, paf_old );
	    break;
	}
    }

    affect_to_char( ch, paf );
    return;
}



/*
 * Move a char out of a room.
 */
void char_from_room( CHAR_DATA *ch )
{
    OBJ_DATA *obj;
    int iWear;

    if ( ch->in_room == NULL )
    {
	bug( "Char_from_room: NULL.", 0 );
	return;
    }

    if ( !IS_NPC(ch) )
	--ch->in_room->area->nplayer;

    for ( iWear = WEAR_NONE+1; iWear < MAX_WEAR; iWear++ )
    {

    if ( ( obj = get_eq_char( ch, iWear ) ) != NULL
    &&   obj->item_type == ITEM_LIGHT
    &&   IS_LIT(obj)
    &&   ch->in_room->light > 0 )
	--ch->in_room->light;
    }

    if ( ch == ch->in_room->people )
    {
	ch->in_room->people = ch->next_in_room;
    }
    else
    {
	CHAR_DATA *prev;

	for ( prev = ch->in_room->people; prev; prev = prev->next_in_room )
	{
	    if ( prev->next_in_room == ch )
	    {
		prev->next_in_room = ch->next_in_room;
		break;
	    }
	}

	if ( prev == NULL )
	    bug( "Char_from_room: ch not found.", 0 );
    }

    ch->in_room      = NULL;
    ch->next_in_room = NULL;
    return;
}



/*
 * Move a char into a room.
 */
void char_to_room( CHAR_DATA *ch, ROOM_INDEX_DATA *pRoomIndex )
{
    OBJ_DATA *obj;
    int iWear;

    if ( pRoomIndex == NULL )
    {
	bug( "Char_to_room: NULL.", 0 );
	return;
    }

    ch->in_room		= pRoomIndex;
    ch->next_in_room	= pRoomIndex->people;
    pRoomIndex->people	= ch;

    if ( !IS_NPC(ch) )
	++ch->in_room->area->nplayer;

    for ( iWear = WEAR_NONE+1; iWear < MAX_WEAR; iWear++ )
    {

    if ( ( obj = get_eq_char( ch, iWear ) ) != NULL
    &&   obj->item_type == ITEM_LIGHT
    &&   IS_LIT(obj) )
    ++ch->in_room->light;
    }

    return;
}



/*
 * Give an obj to a char.
 */
void obj_to_char( OBJ_DATA *obj, CHAR_DATA *ch )
{
    AFFECT_DATA *paf;

    obj->next_content	 = ch->carrying;
    ch->carrying         = obj;
    obj->carried_by	 = ch;
    obj->in_room	 = NULL;
    obj->in_obj		 = NULL;
    ch->carry_number	+= get_obj_number( obj );
    ch->carry_weight	+= get_obj_weight( obj );

    for ( paf = obj->pIndexData->affected; paf != NULL; paf = paf->next )
	affect_modify( ch, paf, TRUE );
    for ( paf = obj->affected; paf != NULL; paf = paf->next )
	affect_modify( ch, paf, TRUE );
}



/*
 * Take an obj from its character.
 */
void obj_from_char( OBJ_DATA *obj )
{
    CHAR_DATA *ch;
    AFFECT_DATA *paf;

    if ( ( ch = obj->carried_by ) == NULL )
    {
	bug( "Obj_from_char: null ch.", 0 );
	return;
    }

    for ( paf = obj->pIndexData->affected; paf != NULL; paf = paf->next )
    affect_modify( ch, paf, FALSE );
    for ( paf = obj->affected; paf != NULL; paf = paf->next )
    affect_modify( ch, paf, FALSE );

    if ( obj->wear_loc != WEAR_NONE
      && obj->wear_loc != WEAR_HOLD_1
      && obj->wear_loc != WEAR_HOLD_2 )        unequip_char( ch, obj );

    if ( ch->carrying == obj )
    {
	ch->carrying = obj->next_content;
    }
    else
    {
	OBJ_DATA *prev;

	for ( prev = ch->carrying; prev != NULL; prev = prev->next_content )
	{
	    if ( prev->next_content == obj )
	    {
		prev->next_content = obj->next_content;
		break;
	    }
	}

	if ( prev == NULL )
	    bug( "Obj_from_char: obj not in list.", 0 );
    }

    obj->carried_by	 = NULL;
    obj->next_content	 = NULL;
    ch->carry_number	-= get_obj_number( obj );
    ch->carry_weight	-= get_obj_weight( obj );
    return;
}



/*
 * Find the ac value of an obj, including position effect.
 */
int apply_ac( OBJ_DATA *obj, int iWear )
{
    if ( obj->item_type != ITEM_ARMOR )
	return 0;

    switch ( iWear )
    {
    case WEAR_BODY:     return 3 * obj->value[0];
    case WEAR_HEAD:     return 2 * obj->value[0];
    case WEAR_LEGS:     return 2 * obj->value[0];
    case WEAR_FEET:     return     obj->value[0];
    case WEAR_HANDS:	return     obj->value[0];
    case WEAR_ARMS:     return     obj->value[0];
    case WEAR_SHIELD:	return     obj->value[0];
    case WEAR_FINGER_L:	return     obj->value[0];
    case WEAR_FINGER_R: return     obj->value[0];
    case WEAR_NECK_1:	return     obj->value[0];
    case WEAR_NECK_2:	return     obj->value[0];
    case WEAR_ABOUT:	return 2 * obj->value[0];
    case WEAR_WAIST:	return     obj->value[0];
    case WEAR_WRIST_L:	return     obj->value[0];
    case WEAR_WRIST_R:	return     obj->value[0];
    case WEAR_HOLD_1:
    case WEAR_HOLD_2:   return     0;
    }

    return 0;
}


OBJ_DATA *get_item_held( CHAR_DATA *ch, int itype )
{
    OBJ_DATA *obj;

    if ( ( obj = get_eq_char( ch, WEAR_HOLD_1 ) ) != NULL
        && obj->item_type == itype )
        return obj;

    if ( ( obj = get_eq_char( ch, WEAR_HOLD_2 ) ) != NULL
        && obj->item_type == itype )
        return obj;

    return NULL;
}


/*
 * Find a piece of eq on a character.
 */
OBJ_DATA *get_eq_char( CHAR_DATA *ch, int iWear )
{
    OBJ_DATA *obj;

    for ( obj = ch->carrying; obj != NULL; obj = obj->next_content )
    {
	if ( obj->wear_loc == iWear )
	    return obj;
    }

    return NULL;
}



/*
 * Equip a char with an obj.  Note: Affect has been moved to the obj_*_char()s
 * Now it sorts it.
 */
void equip_char( CHAR_DATA *ch, OBJ_DATA *obj, int iWear )
{

    if ( get_eq_char( ch, iWear ) != NULL )
    {
	bug( "Equip_char: already equipped (%d).", iWear );
	return;
    }


    /*
     * Thanks to Morgenes for the bug fix here!
     */
    if ( ( IS_OBJ_STAT(obj, ITEM_ANTI_EVIL)    && IS_EVIL(ch)    )
    ||   ( IS_OBJ_STAT(obj, ITEM_ANTI_GOOD)    && IS_GOOD(ch)    )
    ||   ( IS_OBJ_STAT(obj, ITEM_ANTI_NEUTRAL) && IS_NEUTRAL(ch) ) )
    {
    if (!IS_IMMORTAL(ch))
    {
        act( "You are zapped by $p and drop it.", ch, obj, NULL, TO_CHAR );
        act( "$n is zapped by $p and drops it.",  ch, obj, NULL, TO_ROOM );
        obj_from_char( obj );
        obj_to_room( obj, ch->in_room );
        return;
    }
    }

    ch->armor      	-= apply_ac( obj, iWear );
    obj->wear_loc	 = iWear;

    if ( obj->item_type == ITEM_LIGHT
    &&   IS_LIT(obj)
    &&   ch->in_room != NULL )
	++ch->in_room->light;

    return;
}



/*
 *  Return if an object can be put in their hands (hands empty?)
 */
int hand_empty( CHAR_DATA *ch )
{
    OBJ_DATA *h1, *h2;

    h1 = get_eq_char( ch, WEAR_HOLD_1 );
    h2 = get_eq_char( ch, WEAR_HOLD_2 );

    if ( (h1 != NULL && IS_SET( h1->wear_flags, ITEM_TWO_HANDED ))
      || (h2 != NULL && IS_SET( h2->wear_flags, ITEM_TWO_HANDED )) )
            return WEAR_NONE;

    if ( h1 == NULL ) return WEAR_HOLD_1;
    if ( h2 == NULL ) return WEAR_HOLD_2;

    return WEAR_NONE;
}


/*
 *  Return if an object can be put in their belt (hands empty?)
 */
int belt_empty( CHAR_DATA *ch )
{
    if ( get_eq_char( ch, WEAR_WAIST )  == NULL )        return WEAR_NONE;

    if ( get_eq_char( ch, WEAR_BELT_1 ) == NULL )        return WEAR_BELT_1;
    if ( get_eq_char( ch, WEAR_BELT_2 ) == NULL )        return WEAR_BELT_2;
    if ( get_eq_char( ch, WEAR_BELT_3 ) == NULL )        return WEAR_BELT_3;
    if ( get_eq_char( ch, WEAR_BELT_4 ) == NULL )        return WEAR_BELT_4;
    if ( get_eq_char( ch, WEAR_BELT_5 ) == NULL )        return WEAR_BELT_5;

    return WEAR_NONE;
}


/*
 * Unequip a char with an obj.
 */
bool unequip_char( CHAR_DATA *ch, OBJ_DATA *obj )
{
    if ( obj->wear_loc == WEAR_NONE
      || obj->wear_loc == WEAR_HOLD_1
      || obj->wear_loc == WEAR_HOLD_2)
    {
    bug( "Unequip_char: already unequipped.", 0 );
    return TRUE;
    }


    ch->armor       += apply_ac( obj, obj->wear_loc );
    obj->wear_loc    = hand_empty( ch );

    if ( obj->item_type == ITEM_LIGHT
    &&   IS_LIT(obj)
    &&   ch->in_room != NULL
    &&   ch->in_room->light > 0 )
    --ch->in_room->light;

    return TRUE;
}

/*
 * Count occurrences of an obj in a list.
 */
int count_obj_list( OBJ_INDEX_DATA *pObjIndex, OBJ_DATA *list )
{
    OBJ_DATA *obj;
    int nMatch;

    nMatch = 0;
    for ( obj = list; obj != NULL; obj = obj->next_content )
    {
	if ( obj->pIndexData == pObjIndex )
	    nMatch++;
    }

    return nMatch;
}



/*
 * Move an obj out of a room.
 */
void obj_from_room( OBJ_DATA *obj )
{
    ROOM_INDEX_DATA *in_room;

    if ( ( in_room = obj->in_room ) == NULL )
    {
	bug( "obj_from_room: NULL.", 0 );
	return;
    }

    if ( obj == in_room->contents )
    {
	in_room->contents = obj->next_content;
    }
    else
    {
	OBJ_DATA *prev;

	for ( prev = in_room->contents; prev; prev = prev->next_content )
	{
	    if ( prev->next_content == obj )
	    {
		prev->next_content = obj->next_content;
		break;
	    }
	}

	if ( prev == NULL )
	{
	    bug( "Obj_from_room: obj not found.", 0 );
	    return;
	}
    }

    if ( obj->item_type == ITEM_LIGHT && IS_LIT(obj) )  ++obj->in_room->light;

    obj->in_room      = NULL;
    obj->next_content = NULL;
    return;
}



/*
 * Move an obj into a room.
 */
void obj_to_room( OBJ_DATA *obj, ROOM_INDEX_DATA *pRoomIndex )
{
    if ( obj->item_type == ITEM_LIGHT && IS_LIT(obj) )  ++pRoomIndex->light;

    obj->next_content		= pRoomIndex->contents;
    pRoomIndex->contents	= obj;
    obj->in_room		= pRoomIndex;
    obj->carried_by		= NULL;
    obj->in_obj			= NULL;
    obj->wear_loc       = WEAR_NONE;
    return;
}



/*
 * Move an object into an object.
 */
void obj_to_obj( OBJ_DATA *obj, OBJ_DATA *obj_to )
{
    obj->next_content		= obj_to->contains;
    obj_to->contains		= obj;
    obj->in_obj			= obj_to;
    obj->in_room		= NULL;
    obj->carried_by		= NULL;
    obj->wear_loc       = WEAR_NONE;

    for ( ; obj_to != NULL; obj_to = obj_to->in_obj )
    {
	if ( obj_to->carried_by != NULL )
	{
	    obj_to->carried_by->carry_number += get_obj_number( obj );
	    obj_to->carried_by->carry_weight += get_obj_weight( obj );
	}
    }

    return;
}



/*
 * Move an object out of an object.
 */
void obj_from_obj( OBJ_DATA *obj )
{
    OBJ_DATA *obj_from;

    if ( ( obj_from = obj->in_obj ) == NULL )
    {
	bug( "Obj_from_obj: null obj_from.", 0 );
	return;
    }

    if ( obj == obj_from->contains )
    {
	obj_from->contains = obj->next_content;
    }
    else
    {
	OBJ_DATA *prev;

	for ( prev = obj_from->contains; prev; prev = prev->next_content )
	{
	    if ( prev->next_content == obj )
	    {
		prev->next_content = obj->next_content;
		break;
	    }
	}

	if ( prev == NULL )
	{
	    bug( "Obj_from_obj: obj not found.", 0 );
	    return;
	}
    }

    obj->next_content = NULL;
    obj->in_obj       = NULL;

    for ( ; obj_from != NULL; obj_from = obj_from->in_obj )
    {
	if ( obj_from->carried_by != NULL )
	{
	    obj_from->carried_by->carry_number -= get_obj_number( obj );
	    obj_from->carried_by->carry_weight -= get_obj_weight( obj );
	}
    }

    return;
}



/*
 * Extract an obj from the world.
 */
void extract_obj( OBJ_DATA *obj )
{
    OBJ_DATA *obj_content;
    OBJ_DATA *obj_next;

    if ( obj->in_room != NULL )
	obj_from_room( obj );
    else if ( obj->carried_by != NULL )
	obj_from_char( obj );
    else if ( obj->in_obj != NULL )
	obj_from_obj( obj );

    for ( obj_content = obj->contains; obj_content; obj_content = obj_next )
    {
	obj_next = obj_content->next_content;
	extract_obj( obj->contains );
    }

    if ( object_list == obj )
    {
	object_list = obj->next;
    }
    else
    {
	OBJ_DATA *prev;

	for ( prev = object_list; prev != NULL; prev = prev->next )
	{
	    if ( prev->next == obj )
	    {
		prev->next = obj->next;
		break;
	    }
	}

	if ( prev == NULL )
	{
	    bug( "Extract_obj: obj %d not found.", obj->pIndexData->vnum );
	    return;
	}
    }

    {
	AFFECT_DATA *paf;
	AFFECT_DATA *paf_next;

	for ( paf = obj->affected; paf != NULL; paf = paf_next )
	{
	    paf_next    = paf->next;
	    paf->next   = affect_free;
	    affect_free = paf;
	}
    }

    {
	EXTRA_DESCR_DATA *ed;
	EXTRA_DESCR_DATA *ed_next;

	for ( ed = obj->extra_descr; ed != NULL; ed = ed_next )
	{
	    ed_next		= ed->next;
	    free_string( ed->description );
	    free_string( ed->keyword     );
        ed->next    = extra_descr_free;
        extra_descr_free    = ed;
	}
    }

    free_string( obj->name        );
    free_string( obj->description );
    free_string( obj->short_descr );
    free_string( obj->action_descr );
    --obj->pIndexData->count;
    obj->next	= obj_free;
    obj_free	= obj;
    return;
}



/*
 * Extract a char from the world.
 */
void extract_char( CHAR_DATA *ch, bool fPull )
{
    CHAR_DATA *wch;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;

    if ( ch->in_room == NULL )
    {
	bug( "Extract_char: NULL.", 0 );
	return;
    }

    if ( fPull )
    {
    CHAR_DATA *pet;
        for ( pet = char_list; pet != NULL; pet = pet->next )
	  {
          if ( ( pet->master == ch || pet->leader == ch )
          && IS_SET(pet->act, ACT_PET) )
                       break;
          }
 
        if ( pet != NULL && pet != ch )
         extract_char( pet, TRUE );
    }


    if ( fPull )
	die_follower( ch );

    stop_fighting( ch, TRUE );

    for ( obj = ch->carrying; obj != NULL; obj = obj_next )
    {
	obj_next = obj->next_content;
	extract_obj( obj );
    }
    
    char_from_room( ch );
    
    if ( ch->riding != NULL )
    {
       ch->riding->mounted_by = NULL;
       ch->riding             = NULL;
    }
    
    if ( ch->mounted_by != NULL )
    {
       ch->mounted_by->riding = NULL;
       ch->mounted_by         = NULL;
    }

    
    if ( !fPull )
    {
    char_to_room( ch, get_room_index( ROOM_VNUM_ALTAR ) );
	return;
    }

    if ( IS_NPC(ch) )
	--ch->pIndexData->count;

    if ( ch->desc != NULL && ch->desc->original != NULL )
	do_return( ch, "" );

    for ( wch = char_list; wch != NULL; wch = wch->next )
    {
	if ( wch->reply == ch )
	    wch->reply = NULL;
    }

    if ( ch == char_list )
    {
       char_list = ch->next;
    }
    else
    {
	CHAR_DATA *prev;

	for ( prev = char_list; prev != NULL; prev = prev->next )
	{
	    if ( prev->next == ch )
	    {
		prev->next = ch->next;
		break;
	    }
	}

	if ( prev == NULL )
	{
	    bug( "Extract_char: char not found.", 0 );
	    return;
	}
    }

    if ( ch->desc )
	ch->desc->character = NULL;
    free_char( ch );
    return;
}



/*
 * Find a char in the room.
 */
CHAR_DATA *get_char_room( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *rch;
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    if ( !str_cmp( arg, "self" ) )
	return ch;
    for ( rch = ch->in_room->people; rch != NULL; rch = rch->next_in_room )
    {
    if ( !can_see( ch, rch ) || !is_name( arg, STR(rch, name) ) )
	    continue;
	if ( ++count == number )
	    return rch;
    }

    return NULL;
}




/*
 * Find a char in the world.
 */
CHAR_DATA *get_char_world( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *wch;
    int number;
    int count;

    if ( ( wch = get_char_room( ch, argument ) ) != NULL )
	return wch;

    number = number_argument( argument, arg );
    count  = 0;
    for ( wch = char_list; wch != NULL ; wch = wch->next )
    {
    if ( !can_see( ch, wch ) || !is_name( arg, STR(wch, name) ) )
	    continue;
	if ( ++count == number )
	    return wch;
    }

    return NULL;
}



/*
 * Find some object with a given index data.
 * Used by area-reset 'P' command.
 */
OBJ_DATA *get_obj_type( OBJ_INDEX_DATA *pObjIndex )
{
    OBJ_DATA *obj;

    for ( obj = object_list; obj != NULL; obj = obj->next )
    {
	if ( obj->pIndexData == pObjIndex )
	    return obj;
    }

    return NULL;
}


/*
 * Find an obj in a list.
 */
OBJ_DATA *get_obj_list( CHAR_DATA *ch, char *argument, OBJ_DATA *list )
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = list; obj != NULL; obj = obj->next_content )
    {
    if ( can_see_obj( ch, obj ) && is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }

    return NULL;
}



/*
 * Find an obj in player's inventory.
 */
OBJ_DATA *get_obj_carry( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = ch->carrying; obj != NULL; obj = obj->next_content )
    {
    if ( (obj->wear_loc == WEAR_NONE
      ||  obj->wear_loc == WEAR_HOLD_1
      ||  obj->wear_loc == WEAR_HOLD_2)
    &&   can_see_obj( ch, obj )
    &&   is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }


    return NULL;
}


/*
 * Find an object on the hands!
 */
OBJ_DATA *get_obj_held( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = ch->carrying; obj != NULL; obj = obj->next_content )
    {
    if ( (obj->wear_loc == WEAR_HOLD_1
       || obj->wear_loc == WEAR_HOLD_2
       || obj->wear_loc == WEAR_NONE)
    &&   can_see_obj( ch, obj )
    &&   is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }

    return NULL;
}

/*
 * Find an obj in player's equipment.
 */
OBJ_DATA *get_obj_wear( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = ch->carrying; obj != NULL; obj = obj->next_content )
    {
	if ( obj->wear_loc != WEAR_NONE
    &&   obj->wear_loc != WEAR_HOLD_1
    &&   obj->wear_loc != WEAR_HOLD_2
	&&   can_see_obj( ch, obj )
    &&   is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }

    return NULL;
}


/*
 * Find an object in the character's inventory
 */
OBJ_DATA *get_obj_inv( CHAR_DATA *ch, char *argument )
{
    OBJ_DATA *obj;
    char arg[MAX_INPUT_LENGTH];
    int number;
    int count;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = ch->carrying; obj != NULL; obj = obj->next_content )
    {
    if ( can_see_obj( ch, obj )  &&   is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }

    return NULL;
}

/*
 * Find an obj in the room or in inventory.
 */
OBJ_DATA *get_obj_here( CHAR_DATA *ch, char *argument )
{
    OBJ_DATA *obj;

    if ( ( obj = get_obj_inv( ch, argument ) ) != NULL )
    return obj;

    obj = get_obj_list( ch, argument, ch->in_room->contents );

    if ( obj != NULL )
    return obj;

    return NULL;
}



/*
 * Find an obj in the world.
 */
OBJ_DATA *get_obj_world( CHAR_DATA *ch, char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    int number;
    int count;

    if ( ( obj = get_obj_here( ch, argument ) ) != NULL )
	return obj;

    number = number_argument( argument, arg );
    count  = 0;
    for ( obj = object_list; obj != NULL; obj = obj->next )
    {
    if ( can_see_obj( ch, obj ) && is_name( arg, STR(obj, name) ) )
	{
	    if ( ++count == number )
		return obj;
	}
    }

    return NULL;
}

void update_money( OBJ_DATA *obj )
{
    char buf[MAX_STRING_LENGTH];
    char buf1[MAX_STRING_LENGTH];
    char buf2[MAX_STRING_LENGTH];
    char buf3[MAX_STRING_LENGTH];

    if ( obj->item_type != ITEM_MONEY )
    {
        bug( "update_money: item is not of type money.", 0 );
        return;
    }

    if ( obj->value[0] > 1  )
    {
    obj->pIndexData = get_obj_index( OBJ_VNUM_MONEY_SOME );
    if ( obj->value[0] <= 10 )       strcpy( buf, "few"               ); else
    if ( obj->value[0] <= 100 )      strcpy( buf, "small pile of"     ); else
    if ( obj->value[0] <= 1000 )     strcpy( buf, "pile of"           ); else
    if ( obj->value[0] <= 10000 )    strcpy( buf, "large pile of"     ); else
    if ( obj->value[0] <= 100000 )   strcpy( buf, "bunch of"          ); else
    if ( obj->value[0] <= 1000000 )  strcpy( buf, "heaping mound of"  ); else
    if ( obj->value[0] <= 10000000 ) strcpy( buf, "mountain of"       );
                         else strcpy( buf, "whole shitload of" );

    sprintf( buf3, obj->pIndexData->name, coin_table[obj->value[1]].long_name );
    sprintf( buf1, obj->pIndexData->short_descr, obj->value[0],
                   coin_table[obj->value[1]].long_name );
    sprintf( buf2, obj->pIndexData->description, buf,
                   coin_table[obj->value[1]].long_name );
    free_string( obj->name );
    free_string( obj->description );
    free_string( obj->short_descr );
    obj->short_descr = str_dup( buf1 );
    obj->description = str_dup( buf2 );
    obj->name        = str_dup( buf3 );
    obj->weight         = obj->value[0] * coin_table[obj->value[1]].weight / 4;
    obj->cost           = obj->value[0] * coin_table[obj->value[1]].convert;
    }
    else
    if ( obj->value[0] == 1 )
    {
    obj->pIndexData = get_obj_index( OBJ_VNUM_MONEY_ONE );
    sprintf( buf1, obj->pIndexData->short_descr,
                   coin_table[obj->value[1]].long_name );
    sprintf( buf2, obj->pIndexData->description,
                   coin_table[obj->value[1]].long_name );
    free_string( obj->description );
    free_string( obj->short_descr );
    obj->short_descr = str_dup( buf1 );
    obj->description = str_dup( buf2 );
    obj->weight         = coin_table[obj->value[1]].weight;
    obj->cost           = coin_table[obj->value[1]].convert;
    }

    return;
}

OBJ_DATA *create_money( int amount, int type )
{
    OBJ_DATA *obj;

    if ( amount <= 0 )
    {
	bug( "Create_money: zero or negative money %d.", amount );
    amount = 1;
    }

    if ( amount == 1 )
    {
    obj = create_object( get_obj_index( OBJ_VNUM_MONEY_ONE ), 0 );
    obj->item_type      = ITEM_MONEY;
    obj->value[0]       = 1;
    obj->value[1]       = type;
    obj->wear_loc = -1;
    obj->weight         = coin_table[type].weight;
    obj->cost           = coin_table[type].convert;
    update_money( obj );
    }
    else
    {
	obj = create_object( get_obj_index( OBJ_VNUM_MONEY_SOME ), 0 );
    obj->value[0]       = amount;
    obj->value[1]       = type;
    obj->item_type      = ITEM_MONEY;
    obj->wear_loc = -1;
    obj->weight         = amount * coin_table[type].weight / 4;
    obj->cost           = amount * coin_table[type].convert;
    update_money( obj );
    }

    return obj;
}

/*
 * Conglomerate several money items.
 */
void merge_money( CHAR_DATA *ch )
{
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;
    int sub_totals[MAX_COIN];
    int type;

    for ( type = 0; type < MAX_COIN; type++ )
    {
        sub_totals[type] = tally_one_coin( ch, type );
    }

    for ( obj = ch->carrying;  obj != NULL;  obj = obj_next )
    {
        obj_next = obj->next_content;
        if ( obj->item_type == ITEM_MONEY )
        {
            obj_from_char( obj );
            extract_obj( obj );
        }
    }

    for ( type = 0; type < MAX_COIN; type++ )
    {
        if ( sub_totals[type] > 0 )
          obj_to_char( create_money( sub_totals[type], type ), ch );
    }
    return;
}

/*
 * Create a set of 'money' objects.
 */
void create_amount( int amount, CHAR_DATA *pMob, ROOM_INDEX_DATA *pRoom,
                    OBJ_DATA *pObj )
{
    OBJ_DATA *obj;
    int type;

    if ( amount <= 0 )
    {
    bug( "Create_amount: zero or negative money %d.", amount );
    return;
    }

    if ( amount == 1 )
    {
    obj = create_money( 1, MAX_COIN-1 );
    if ( pMob ) obj_to_char( obj, pMob );
    if ( pRoom ) obj_to_room( obj, pRoom );
    if ( pObj ) obj_to_obj( obj, pObj );
    }
    else
    {
        for ( type = 0; type < MAX_COIN; type++ )
        {

        if ( amount/coin_table[type].convert <= 0 )
            continue;

        obj = create_money( amount/coin_table[type].convert, type );
        amount %= coin_table[type].convert;

        if ( pMob ) obj_to_char( obj, pMob );
        if ( pRoom ) obj_to_room( obj, pRoom );
        if ( pObj ) obj_to_obj( obj, pObj );
        }
    }

    if ( pMob ) merge_money( pMob );
    return;
}


/*
 * Tally the number of coins a character has.
 */
int tally_coins( CHAR_DATA *mob )
{
    OBJ_DATA *obj;
    int total = 0;

    for ( obj = mob->carrying;  obj != NULL; obj = obj->next_content )
    {
        if ( obj->item_type == ITEM_MONEY
          && obj->value[1] >= 0
          && obj->value[1] < MAX_COIN )
        total += obj->value[0] * coin_table[obj->value[1]].convert;

        if ( ( obj->value[1] < 0 || obj->value[1] >= MAX_COIN ) 
          && obj->item_type == ITEM_MONEY )
        {
        bug( "Tally_coins: obj (vnum %d) bad coinage",
             obj->pIndexData->vnum );
        obj->value[1] = URANGE( 0, obj->value[1], MAX_COIN-1 );
        }
    }

    return total;
}

/*
 * Tally the number of A SINGLE TYPE OF coin a character has.
 */
int tally_one_coin( CHAR_DATA *mob, int type )
{
    OBJ_DATA *obj;
    int total = 0;

    for ( obj = mob->carrying;  obj != NULL; obj = obj->next_content )
    {
        if ( obj->item_type == ITEM_MONEY
          && obj->value[1] >= 0
          && obj->value[1] < MAX_COIN
          && obj->value[1] == type )
        total += obj->value[0];

        if ( ( obj->value[1] < 0 || obj->value[1] >= MAX_COIN ) 
          && obj->item_type == ITEM_MONEY )
        {
        bug( "Tally_one_coin: obj (vnum %d) bad coinage",
             obj->pIndexData->vnum );
        obj->value[1] = URANGE( 0, obj->value[1], MAX_COIN-1 );
        }
    }

    return total;
}


/*
 * Create a cost out of the blue.
 */
char *name_amount( int amount )
{
    char buf[MAX_STRING_LENGTH];
    char buf2[MAX_STRING_LENGTH];
    int type;

    buf[0] = '\0';

    for ( type = 0; ; type++ )
    {
    if (type >= MAX_COIN || amount == 0 )
        break;

    if ( amount/coin_table[type].convert <= 0 )
        continue;

    sprintf( buf2, "%s%d %s", buf[0] != '\0' ? ", " : "",
                              (amount/coin_table[type].convert),
                              coin_table[type].long_name );
    strcat( buf, buf2 );
    amount %= coin_table[type].convert;
    }
    if ( buf[0] == '\0' ) sprintf( buf, "nothing" );

    return buf;
}

/*
 * Nail unused money items.
 */
void strip_empty_money( CHAR_DATA *ch )
{
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;

    for ( obj = ch->carrying;  obj != NULL;  obj = obj_next )
    {
        obj_next = obj->next_content;
        if ( obj->item_type == ITEM_MONEY  && obj->value[0] <= 0 )
        {
            obj_from_char( obj );
            extract_obj( obj );
        }
    }
    return;
}
     

/*
 * Ok, here's the tricky part... managing two money totals at once,
 * this is the routine that makes the change if you give the dude
 * too much.  This is for shops mainly, but can be used elsewhere.
 */
char *sub_coins( int amount, CHAR_DATA *ch )
{
    OBJ_DATA *obj;
    int money = 0;

    if ( tally_coins( ch ) < amount )
    {
        bug( "Sub_coins: ch doesn't have enough money (%d)", amount );
        return "nothing";
    }

    for ( obj = ch->carrying;  obj != NULL;  obj = obj->next_content )
    {
        if ( money >= amount ) break;
        if ( obj->item_type == ITEM_MONEY )
        {
            while ( obj->value[0] > 0 && money < amount )
            {
                 money += coin_table[obj->value[1]].convert;
                 obj->value[0]--;
            }
            if ( obj->value[0] > 0 ) update_money( obj );
        }
    }

    strip_empty_money( ch );
    if ( money > amount ) create_amount( money - amount, ch, NULL, NULL );
    return name_amount( money - amount );
}
      
/*
 * Return # of objects which an object counts as.
 * Thanks to Tony Chamberlain for the correct recursive code here.
 */
int get_obj_number( OBJ_DATA *obj )
{
    int number;

    if ( obj->item_type == ITEM_CONTAINER )
	number = 0;
    else
	number = 1;

    for ( obj = obj->contains; obj != NULL; obj = obj->next_content )
	number += get_obj_number( obj );

    return number;
}



/*
 * Return weight of an object, including weight of contents.
 */
int get_obj_weight( OBJ_DATA *obj )
{
    int weight;

    weight = obj->weight;
    for ( obj = obj->contains; obj != NULL; obj = obj->next_content )
	weight += get_obj_weight( obj );

    return weight;
}



/*
 * True if room is dark.
 */
int room_is_dark( ROOM_INDEX_DATA *pRoomIndex )
{
    if ( pRoomIndex == NULL ) return FALSE;

    if ( pRoomIndex->light > 0 )
    return FALSE;

    if ( IS_SET(pRoomIndex->room_flags, ROOM_DARK) )
    return TRUE;

    if ( pRoomIndex->sector_type == SECT_INSIDE )
    return FALSE;

    if ( weather_info.sunlight == MOON_RISE
    && pRoomIndex->sector_type == SECT_FIELD )
    return FALSE;

    if ( weather_info.sunlight == SUN_RISE
      || weather_info.sunlight == SUN_LIGHT )
    return FALSE;

    if ( weather_info.sky > SKY_CLOUDLESS
      && pRoomIndex->sector_type == SECT_FOREST )
    return TRUE;

    return FALSE;
}


/*
 * True if room is private.
 */
bool room_is_private( ROOM_INDEX_DATA *pRoomIndex )
{
    CHAR_DATA *rch;
    int count;

    count = 0;
    for ( rch = pRoomIndex->people; rch != NULL; rch = rch->next_in_room )
	count++;

    if ( IS_SET(pRoomIndex->room_flags, ROOM_PRIVATE)  && count >= 2 )
	return TRUE;

    if ( IS_SET(pRoomIndex->room_flags, ROOM_SOLITARY) && count >= 1 )
	return TRUE;

    return FALSE;
}



/*
 * True if char can see victim.
 */
bool can_see( CHAR_DATA *ch, CHAR_DATA *victim )
{
    if ( ch == victim )
	return TRUE;
    
    if ( !IS_NPC(victim)
    &&   victim->wizinvis >= ch->level
    &&   get_trust( ch ) < get_trust( victim ) )
	return FALSE;

    if ( !IS_NPC(ch) && IS_SET(ch->act, PLR_HOLYLIGHT) )
	return TRUE;

    if ( IS_AFFECTED(ch, AFF_BLIND) )
	return FALSE;

    if ( room_is_dark( ch->in_room ) && !IS_AFFECTED(ch, AFF_INFRARED) )
	return FALSE;

    if ( IS_NPC(ch) && HAS_SCRIPT(ch) && !IS_SET(ch->act, ACT_AGGRESSIVE ) )
    return TRUE;

    if ( IS_AFFECTED(victim, AFF_INVISIBLE)
    &&   !IS_AFFECTED(ch, AFF_DETECT_INVIS) )
	return FALSE;

    if ( IS_AFFECTED(victim, AFF_HIDE)
    &&   !IS_AFFECTED(ch, AFF_DETECT_HIDDEN)
    &&   victim->fighting == NULL
    &&   ( IS_NPC(ch) ? !IS_NPC(victim) : IS_NPC(victim) ) )
	return FALSE;

    return TRUE;
}



/*
 * True if char can see obj.
 */
bool can_see_obj( CHAR_DATA *ch, OBJ_DATA *obj )
{
    if ( !IS_NPC(ch) && IS_SET(ch->act, PLR_HOLYLIGHT) )
	return TRUE;

    if ( IS_AFFECTED( ch, AFF_BLIND ) )
    return FALSE;

    if ( IS_SET( obj->extra_flags, ITEM_BURNING ) )
    return TRUE;

    if ( obj->item_type == ITEM_LIGHT && IS_LIT( obj ) )
	return TRUE;

    if ( room_is_dark( ch->in_room ) && !IS_AFFECTED(ch, AFF_INFRARED) )
	return FALSE;

    if ( IS_SET(obj->extra_flags, ITEM_INVIS)
    &&   !IS_AFFECTED(ch, AFF_DETECT_INVIS) )
	return FALSE;

    return TRUE;
}



/*
 * True if char can drop obj.
 */
bool can_drop_obj( CHAR_DATA *ch, OBJ_DATA *obj )
{
    if ( !IS_SET(obj->extra_flags, ITEM_NODROP) )
	return TRUE;

    if ( !IS_NPC(ch) && ch->level >= LEVEL_IMMORTAL )
	return TRUE;

    if ( NOT_WORN( obj ) )
    return TRUE;

    return FALSE;
}


/*
 * Get an extra description from a list.
 */
char *get_extra_descr( const char *name, EXTRA_DESCR_DATA *ed )
{
    for ( ; ed != NULL; ed = ed->next )
    {
	if ( is_name( name, ed->keyword ) )
        return ed->description;
    }
    return NULL;
}


bool use_tool( OBJ_DATA *obj, int bit )
{
    if ( obj == NULL
      || obj->item_type != ITEM_TOOL
      || !VAL_SET(obj, 0, bit) )
    return FALSE;

    if ( obj->value[1] <= 0 )
    return FALSE;

    obj->value[1]--;
    return TRUE;
}


int PERCENTAGE( int amount, int max )
{
    if (max > 0) return (amount*100)/max;

    return amount*100;
}


bool skill_check( CHAR_DATA *ch, int sn, int modifier )
{
    if (IS_NPC(ch) || ch->pcdata == NULL)
    return number_range( 0, 100 ) <= ch->level+(ch->level/4) ? TRUE : FALSE;

    return URANGE(0, ch->pcdata->learned[sn]-modifier, 100) > number_percent( )
           ? TRUE : FALSE;
}



