/***************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,        *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.   *
 *                                                                         *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael          *
 *  Chastain, Michael Quan, and Mitchell Tse.                              *
 *                                                                         *
 *  In order to use any part of this Merc Diku Mud, you must comply with   *
 *  both the original Diku license in 'license.doc' as well the Merc       *
 *  license in 'license.txt'.  In particular, you may not remove either of *
 *  these copyright notices.                                               *
 *                                                                         *
 *  Much time and thought has gone into this software and you are          *
 *  benefitting.  We hope that you share your changes too.  What goes      *
 *  around, comes around.                                                  *
 ***************************************************************************/

#if defined(macintosh)
#include <types.h>
#else
#include <sys/types.h>
#endif
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "merc.h"



/*
 * Local functions.
 */
int	hit_gain	args( ( CHAR_DATA *ch ) );
int	move_gain	args( ( CHAR_DATA *ch ) );
void	mobile_update	args( ( void ) );
void	weather_update	args( ( void ) );
void	char_update	args( ( void ) );
void	obj_update	args( ( void ) );
void	aggr_update	args( ( void ) );



/*
 * Advancement stuff.
 */
void advance_level( CHAR_DATA *ch )
{
    char buf[MAX_STRING_LENGTH];
    int add_hp;
    int add_move;
    int add_prac;
    int add_learn;

    add_hp	= con_app[get_curr_con(ch)].hitp + number_range(
		    class_table[ch->class].hp_min,
		    class_table[ch->class].hp_max );
    add_move	= number_range( 5, (get_curr_con(ch)+get_curr_dex(ch))/4 );
    add_prac	= wis_app[get_curr_wis(ch)].practice;
    add_learn   = add_prac / 2;

    add_hp      = UMAX(  1, add_hp   );
    add_move	= UMAX( 10, add_move );

    ch->hit         += add_hp;
    ch->max_hit 	+= add_hp;
    ch->move        += add_move;
    ch->max_move	+= add_move;
    ch->practice	+= add_prac;
    ch->learn       += UMAX( 1, add_learn );

    if ( !IS_NPC(ch) )
	REMOVE_BIT( ch->act, PLR_BOUGHT_PET );

    if ( ch->level >= LEVEL_HERO )
    {
         sprintf( buf,
         "You gain %d/%dhp, %d/%dmv %d/%d practices %d/%d learns.\n\r",
         add_hp,     ch->max_hit,
         add_move,   ch->max_move,
         add_prac,   ch->practice,
         add_learn,  ch->learn     );
         send_to_char( buf, ch );
    }
    else
    {
         sprintf( buf, "You gain %d practices and %d learns.\n\r", add_prac, add_learn );
         send_to_char( buf, ch );
    }
    return;
}   



void gain_exp( CHAR_DATA *ch, int gain )
{
    if ( IS_NPC(ch) || ch->level >= LEVEL_HERO )
	return;

    ch->exp += gain;
    if (ch->exp < 0) ch->exp = 0;

    while ( ch->level < LEVEL_HERO && ch->exp >= FIND_EXP(ch->level, ch->race) )
    {
    ansi_color( BOLD, ch );
    send_to_char( "YOU RAISE A LEVEL! ", ch );
    ch->exp += -FIND_EXP(ch->level, ch->race);
    ch->level++;
    advance_level( ch );
    ansi_color( NTEXT, ch );
    }


    return;
}



/*
 * Regeneration stuff.
 */
int hit_gain( CHAR_DATA *ch )
{
    int gain;

    if ( IS_NPC(ch) )
    {
	gain = ch->level * 3 / 2;
    }
    else
    {
	gain = UMIN( 5, ch->level );

	switch ( ch->position )
	{
	case POS_SLEEPING: gain += get_curr_con(ch);		break;
	case POS_RESTING:  gain += get_curr_con(ch) / 2;	break;
	}

	if ( ch->pcdata->condition[COND_FULL]   == 0 )
	    gain /= 2;

	if ( ch->pcdata->condition[COND_THIRST] == 0 )
	    gain /= 2;

    }

    if ( IS_AFFECTED(ch, AFF_POISON) )
	gain /= 4;

    return UMIN(gain, ch->max_hit - ch->hit);
}





int move_gain( CHAR_DATA *ch )
{
    int gain;

    if ( IS_NPC(ch) )
    {
	gain = ch->level;
    }
    else
    {
	gain = UMAX( 15, 2 * ch->level );

	switch ( ch->position )
	{
	case POS_SLEEPING: gain += get_curr_dex(ch);		break;
	case POS_RESTING:  gain += get_curr_dex(ch) / 2;	break;
	}

	if ( ch->pcdata->condition[COND_FULL]   == 0 )
	    gain /= 2;

	if ( ch->pcdata->condition[COND_THIRST] == 0 )
	    gain /= 2;
    }

    if ( IS_AFFECTED(ch, AFF_POISON) )
	gain /= 4;

    return UMIN(gain, ch->max_move - ch->move);
}



void gain_condition( CHAR_DATA *ch, int iCond, int value )
{
    int condition;

    if ( IS_NPC(ch) || ch->level >= LEVEL_HERO )
	return;

    condition				= ch->pcdata->condition[iCond];
    ch->pcdata->condition[iCond]    = URANGE( -10, condition + value, 48 );

    if ( iCond == COND_DRUNK && ch->pcdata->condition[iCond] < 0 )
        ch->pcdata->condition[iCond] = 0;

    if ( ch->pcdata->condition[iCond] < 10 
      && ch->pcdata->condition[iCond] >= 0 )
    {
	switch ( iCond )
	{
	case COND_FULL:
	    send_to_char( "You are hungry.\n\r",  ch );
	    break;

	case COND_THIRST:
	    send_to_char( "You are thirsty.\n\r", ch );
	    break;

	case COND_DRUNK:
	    if ( condition != 0 && ch->pcdata->condition[iCond] == 0 )
		send_to_char( "You are sober.\n\r", ch );
	    break;
	}
    }

    if ( ch->pcdata->condition[iCond] < 0 )
    {
	switch ( iCond )
	{
	case COND_FULL:
        send_to_char( "You are starving!\n\r",  ch );
        if ( ch->pcdata->condition[iCond] <= -10 )
        {
            send_to_char(  "You have died of starvation!", ch );
            act( "$n dies of starvation!", ch, NULL, NULL, TO_ROOM );
            raw_kill(ch);
        }
	    break;

	case COND_THIRST:
        send_to_char( "You are dehydrating!\n\r", ch );
        if ( ch->pcdata->condition[iCond] <= -10 )
        {
            send_to_char(  "You have died of dehydration!", ch );
            act( "$n dies of dehydration!", ch, NULL, NULL, TO_ROOM );
            raw_kill(ch);
        }
	    break;
	}
    }

    return;
}


/*
 * Mob autonomous action.
 * This function takes 25% to 35% of ALL Merc cpu time.
 * -- Furey
 */
void mobile_update( void )
{
    CHAR_DATA *ch;
    CHAR_DATA *ch_next;
    EXIT_DATA *pexit;
    int door;

    /* Examine all mobs. */
    for ( ch = char_list; ch != NULL; ch = ch_next )
    {
	ch_next = ch->next;

    if ( !IS_NPC(ch) || ch->in_room == NULL )
        continue;

     if ( HAS_SCRIPT(ch) )
     {
         script_update( ch, TRIG_EACH_PULSE );
     }

    if ( ch->timer > 0 )
    {
        if ( --ch->timer == 0 )
        {
        act( "$n has left.", ch, NULL, NULL, TO_ROOM );
        act( "Your time has run out.", ch, NULL, NULL, TO_CHAR );
        raw_kill( ch );
        continue;
        }
    }

    if ( IS_AFFECTED(ch, AFF_CHARM) )
	    continue;

	/* Examine call for special procedure */
	if ( ch->spec_fun != 0 )
	{
	    if ( (*ch->spec_fun) ( ch ) )
		continue;
	}

	/* That's all for sleeping / busy monster */
	if ( ch->position != POS_STANDING )
	    continue;


	/* Scavenge */
	if ( IS_SET(ch->act, ACT_SCAVENGER)
	&&   ch->in_room->contents != NULL
	&&   number_bits( 2 ) == 0 )
	{
	    OBJ_DATA *obj;
	    OBJ_DATA *obj_best;
	    int max;

	    max         = 1;
	    obj_best    = 0;
	    for ( obj = ch->in_room->contents; obj; obj = obj->next_content )
	    {
		if ( CAN_WEAR(obj, ITEM_TAKE) && obj->cost > max )
		{
		    obj_best    = obj;
		    max         = obj->cost;
		}
	    }

	    if ( obj_best )
	    {
		obj_from_room( obj_best );
		obj_to_char( obj_best, ch );
		act( "$n gets $p.", ch, obj_best, NULL, TO_ROOM );
	    }
	}

	/* Wander */
	if ( !IS_SET(ch->act, ACT_SENTINEL)
    && ( door = number_bits( MAX_DIR ) ) < MAX_DIR
	&& ( pexit = ch->in_room->exit[door] ) != NULL
	&&   pexit->to_room != NULL
	&&   !IS_SET(pexit->exit_info, EX_CLOSED)
	&&   !IS_SET(pexit->to_room->room_flags, ROOM_NO_MOB)
	&& ( !IS_SET(ch->act, ACT_STAY_AREA)
	||   pexit->to_room->area == ch->in_room->area ) )
	{
	    move_char( ch, door );
	    if ( ch->position < POS_STANDING )
	        continue;
	}

	/* Flee */
	if ( ch->hit < ch->max_hit / 2
    && ( door = number_bits( MAX_DIR ) ) < MAX_DIR
	&& ( pexit = ch->in_room->exit[door] ) != NULL
	&&   pexit->to_room != NULL
	&&   !IS_SET(pexit->exit_info, EX_CLOSED)
	&&   !IS_SET(pexit->to_room->room_flags, ROOM_NO_MOB) )
	{
	    CHAR_DATA *rch;
	    bool found;

	    found = FALSE;
	    for ( rch  = pexit->to_room->people;
		  rch != NULL;
		  rch  = rch->next_in_room )
	    {
		if ( !IS_NPC(rch) )
		{
		    found = TRUE;
		    break;
		}
	    }
	    if ( !found )
		move_char( ch, door );
	}

    }

    return;
}


/*
 * Update the weather.
 *
 * Its late.  I ask myself this question: what the hell is mmhg?  Pondering,
 * I can only hope but ask myself why?  There are no comments, no mentions of
 * this variable?  Yet I trudge on.     - Locke   (mmhg=Millimeters of Mercury)
 */
void weather_update( void )
{
    char buf[MAX_STRING_LENGTH];
    DESCRIPTOR_DATA *d;
    int diff;

    buf[0] = '\0';

    switch ( ++time_info.hour )
    {
    case  4:
    weather_info.sunlight = MOON_SET;
    strcat( buf, "The moon sets.\n\r" );
    break;

    case  5:
	weather_info.sunlight = SUN_LIGHT;
	strcat( buf, "The day has begun.\n\r" );
    if ( time_info.month <= 4 || time_info.month >= 15 )
    weather_info.temperature = number_fuzzy( 20 );
    else
    weather_info.temperature = number_fuzzy( 50 );
	break;

    case  6:
	weather_info.sunlight = SUN_RISE;
	strcat( buf, "The sun rises in the east.\n\r" );
    weather_info.temperature += number_fuzzy( 10 );
	break;

    case  12:
    strcat( buf, "It is noon.\n\r" );
    weather_info.temperature += number_fuzzy( 20 );
	break;

    case 19:
	weather_info.sunlight = SUN_SET;
	strcat( buf, "The sun slowly disappears in the west.\n\r" );
    weather_info.temperature -= number_fuzzy( 20 );
	break;

    case 20:
	weather_info.sunlight = SUN_DARK;
	strcat( buf, "The night has begun.\n\r" );
    weather_info.temperature -= number_fuzzy( 10 );
	break;

    case 24:
    weather_info.sunlight = MOON_RISE;
    strcat( buf, "The moon rises, casting a silver glow over the night.\n\r" );
    weather_info.temperature -= number_fuzzy( 10 );
	time_info.hour = 0;
    time_info.day++;                 /* New Day */
	break;
    }

    if ( time_info.day   >= 30 )     /* New Month */
    {
    log_string( "Weather_update: New Month <game diagnostic>" );
    fix_exits( );
    log_string( "Weather_update: New Month <end diagnostic>" );
	time_info.day = 0;
	time_info.month++;
    }

    if ( time_info.month >= 17 )     /* New Year  */
    {
	time_info.month = 0;
	time_info.year++;
    }

    /*
     * Weather change.
     */
    weather_info.winddir += number_range( 0, 2 )-1;

    if ( time_info.month >= 9 && time_info.month <= 16 )
	diff = weather_info.mmhg >  985 ? -2 : 2;
    else
	diff = weather_info.mmhg > 1015 ? -2 : 2;

    weather_info.change   += diff * dice(1, 4) + dice(2, 6) - dice(2, 6);
    weather_info.change    = UMAX(weather_info.change, -12);
    weather_info.change    = UMIN(weather_info.change,  12);

    weather_info.mmhg += weather_info.change;
    weather_info.mmhg  = UMAX(weather_info.mmhg,  960);
    weather_info.mmhg  = UMIN(weather_info.mmhg, 1040);

    switch ( weather_info.sky )
    {
    default: 
	bug( "Weather_update: bad sky %d.", weather_info.sky );
	weather_info.sky = SKY_CLOUDLESS;
	break;

    case SKY_CLOUDLESS:
	if ( weather_info.mmhg <  990
	|| ( weather_info.mmhg < 1010 && number_bits( 2 ) == 0 ) )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "A few flakes of snow are falling.\n\r" );
        weather_info.temperature -= 10;
    }
 else   strcat( buf, "The sky is getting cloudy.\n\r" );
	    weather_info.sky = SKY_CLOUDY;
        weather_info.windspeed += 10;
	}
	break;

    case SKY_CLOUDY:
	if ( weather_info.mmhg <  970
	|| ( weather_info.mmhg <  990 && number_bits( 2 ) == 0 ) )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "It starts to snow.\n\r" );
        weather_info.temperature -= 10;
    }
 else   strcat( buf, "It starts to rain.\n\r" );
	    weather_info.sky = SKY_RAINING;
        weather_info.windspeed += 10;
	}

	if ( weather_info.mmhg > 1030 && number_bits( 2 ) == 0 )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "The snow lets up.\n\r" );
        weather_info.temperature += 10;
    }
 else   strcat( buf, "The clouds disappear.\n\r" );
	    weather_info.sky = SKY_CLOUDLESS;
        weather_info.windspeed -= 10;
	}
	break;

    case SKY_RAINING:
	if ( weather_info.mmhg <  970 && number_bits( 2 ) == 0 )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "You are caught in a blizzard.\n\r" );
        weather_info.temperature -= 30;
    }
 else   strcat( buf, "Lightning flashes in the sky.\n\r" );
	    weather_info.sky = SKY_LIGHTNING;
        weather_info.windspeed += 10;
	}

	if ( weather_info.mmhg > 1030
	|| ( weather_info.mmhg > 1010 && number_bits( 2 ) == 0 ) )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "The snow is letting up.\n\r" );
        weather_info.temperature += 30;
    }
 else   strcat( buf, "The rain stopped.\n\r" );
	    weather_info.sky = SKY_CLOUDY;
        weather_info.windspeed -= 10;
	}
	break;

    case SKY_LIGHTNING:
	if ( weather_info.mmhg > 1010
	|| ( weather_info.mmhg >  990 && number_bits( 2 ) == 0 ) )
	{
    if ( time_info.month <= 4 || time_info.month >= 15 )
    {
        strcat( buf, "The blizzard subsides.\n\r" );
        weather_info.temperature += 10;
    }
 else   strcat( buf, "The lightning has stopped.\n\r" );
	    weather_info.sky = SKY_RAINING;
        weather_info.windspeed -= 10;
	    break;
	}
	break;
    }

    if ( buf[0] != '\0' )
    {
	for ( d = descriptor_list; d != NULL; d = d->next )
	{
	    if ( d->connected == CON_PLAYING
	    &&   IS_OUTSIDE(d->character)
	    &&   IS_AWAKE(d->character) )
		send_to_char( buf, d->character );
	}
    }

    return;
}



/*
 * Update all chars, including mobs.
 * This function is performance sensitive.
 */
void char_update( void )
{
    CHAR_DATA *ch;
    CHAR_DATA *ch_next;
    CHAR_DATA *ch_save;
    CHAR_DATA *ch_quit;
    time_t save_time;

    save_time	= current_time;
    ch_save	= NULL;
    ch_quit	= NULL;
    for ( ch = char_list; ch != NULL; ch = ch_next )
    {
	AFFECT_DATA *paf;
	AFFECT_DATA *paf_next;

	ch_next = ch->next;

	/*
	 * Find dude with oldest save time.
	 */
	if ( !IS_NPC(ch)
    && ( ch->desc == NULL || ch->desc->connected <= CON_PLAYING )
	&&   ch->level >= 2
	&&   ch->save_time < save_time )
	{
	    ch_save	= ch;
	    save_time	= ch->save_time;
	}

	if ( ch->position >= POS_STUNNED )
	{
	    if ( ch->hit  < ch->max_hit )
		ch->hit  += hit_gain(ch);

	    if ( ch->move < ch->max_move )
		ch->move += move_gain(ch);
	}

	if ( ch->position == POS_STUNNED )
	    update_pos( ch );

	if ( !IS_NPC(ch) && ch->level < LEVEL_IMMORTAL )
	{
	    if ( ++ch->timer >= 12 )
	    {
		if ( ch->was_in_room == NULL && ch->in_room != NULL )
		{
		    ch->was_in_room = ch->in_room;
            if ( ch->fighting != NULL ) stop_fighting( ch, TRUE );
            act( "$n disappears into the void.", ch, NULL, NULL, TO_ROOM );
		    send_to_char( "You disappear into the void.\n\r", ch );
		    save_char_obj( ch );
		    char_from_room( ch );
            char_to_room( ch, get_room_index( ROOM_VNUM_LIMBO ) );
		}
	    }

	    if ( ch->timer > 30 )
		ch_quit = ch;


        gain_condition( ch, COND_DRUNK,  -1 );
	    gain_condition( ch, COND_FULL,   -1 );
	    gain_condition( ch, COND_THIRST, -1 );
	}

	for ( paf = ch->affected; paf != NULL; paf = paf_next )
	{
	    paf_next	= paf->next;
	    if ( paf->duration > 0 )
		paf->duration--;
	    else if ( paf->duration < 0 )
		;
	    else
	    {
		if ( paf_next == NULL
		||   paf_next->type != paf->type
		||   paf_next->duration > 0 )
		{
		    if ( paf->type > 0 && skill_table[paf->type].msg_off )
		    {
			send_to_char( skill_table[paf->type].msg_off, ch );
			send_to_char( "\n\r", ch );
		    }
		}
	  
		affect_remove( ch, paf );
	    }
	}

	/*
	 * Careful with the damages here,
	 *   MUST NOT refer to ch after damage taken,
	 *   as it may be lethal damage (on NPC).
	 */
	if ( IS_AFFECTED(ch, AFF_POISON) )
	{
	    act( "$n shivers and suffers.", ch, NULL, NULL, TO_ROOM );
	    send_to_char( "You shiver and suffer.\n\r", ch );
	    damage( ch, ch, 2, gsn_poison );
	}
	else if ( ch->position == POS_INCAP )
	{
        damage( ch, ch, 1, TYPE_UNDEFINED );
	}
	else if ( ch->position == POS_MORTAL )
	{
	    damage( ch, ch, 2, TYPE_UNDEFINED );
	}
    }

    /*
     * Autosave and autoquit.
     * Check that these chars still exist.
     */
    if ( ch_save != NULL || ch_quit != NULL )
    {
	for ( ch = char_list; ch != NULL; ch = ch_next )
	{
	    ch_next = ch->next;
        if ( ch == ch_save )        save_char_obj( ch );
        if ( ch == ch_quit )        do_quit( ch, "" );
	}
    }

    return;
}



/*
 * Update all objs.
 * This function is performance sensitive.
 */
void obj_update( void )
{   
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;
    ROOM_INDEX_DATA *pRoom;

    for ( obj = object_list; obj != NULL; obj = obj_next )
    {
        CHAR_DATA *rch;
        CHAR_DATA *ch;
        char *message;

        obj_next = obj->next;
        ch = obj->carried_by;
    
        if ( obj->item_type == ITEM_LIGHT && IS_LIT(obj) )
	    {
        if ( --obj->value[0] == 0 && ch->in_room != NULL )
		{
            --ch->in_room->light;
            if ( ch != NULL )
            {
            act( "$p goes out.", ch, obj, NULL, TO_ROOM );
            act( "$p that you are carrying goes out.", ch, obj, NULL, TO_CHAR );
            }
            else if ( obj->in_room != NULL
            &&      ( rch = obj->in_room->people ) != NULL )
            {
            act( "$p goes out.", rch, obj, NULL, TO_ROOM );
            }
            REMOVE_BIT(obj->value[3], LIGHT_LIT);
            if (!IS_SET(obj->value[3], LIGHT_FILLABLE))    extract_obj( obj );
		}
        else
        if ( ch != NULL )
        {
            if (obj->value[0] > 0)
            {
                switch ( PERCENTAGE( obj->value[0], obj->value[1] )/10 )
                {
                    case 0:
                    act( "$p carried by $n flickers and sputters.", ch, obj, NULL, TO_ROOM );
                    act( "$p flickers and sputters.", ch, obj, NULL, TO_CHAR );
                    break;
                    case 1:
                    act( "$p carried by $n flickers.", ch, obj, NULL, TO_ROOM );
                    act( "$p flickers.", ch, obj, NULL, TO_CHAR );
                    break;
                    case 2:
                    act( "$p carried by $n flickers slightly.", ch, obj, NULL, TO_ROOM );
                    act( "$p flickers slightly.", ch, obj, NULL, TO_CHAR );
                    break;
                }
            }
        }
    }

    if ( (pRoom = obj->in_room) != NULL
      && (pRoom->sector_type == SECT_WATER_NOSWIM
       || pRoom->sector_type == SECT_WATER_SWIM
       || pRoom->sector_type == SECT_AIR
       || pRoom->sector_type == SECT_UNDERWATER) )
    {
        EXIT_DATA *pExit;
        char buf[MAX_STRING_LENGTH];

        sprintf( buf, "$p %s.",
                      pRoom->sector_type == SECT_WATER_NOSWIM ||
                      pRoom->sector_type == SECT_WATER_SWIM ?
                          "splashes into the water and disappears." :
                      pRoom->sector_type == SECT_AIR ?
                          "plummets down to the earth below." :
                          "floats down into the murky waters below." );

        rch = pRoom->people;
        if ( rch != NULL )
        {
        act( buf, rch, obj, NULL, TO_ROOM );
        act( buf, rch, obj, NULL, TO_CHAR );
        }

        if ( (pExit = pRoom->exit[DIR_DOWN]) != NULL
         &&   pExit->to_room != pRoom
         &&   pExit->to_room != NULL )
        {
            obj_from_room( obj );
            pRoom = pExit->to_room;
            sprintf( buf, "$p %s.",
                      pRoom->sector_type == SECT_WATER_NOSWIM ||
                      pRoom->sector_type == SECT_WATER_SWIM ?
                          "washes in from above." :
                      pRoom->sector_type == SECT_AIR ?
                          "plummets down from above." :
                          "floats down from above." );
            obj_to_room( obj, pExit->to_room );
            rch = pExit->to_room->people;
            if ( rch != NULL )
            {
            act( buf, rch, obj, NULL, TO_ROOM );
            act( buf, rch, obj, NULL, TO_CHAR );
            }
        }
        else
        {
            extract_obj( obj );
            continue;
        }

    }


	if ( obj->timer <= 0 || --obj->timer > 0 )
        continue;

	switch ( obj->item_type )
	{
	default:              message = "$p vanishes.";         break;
    case ITEM_POTION:     message = "$p fades away.";       break;
    case ITEM_WAND:       message = "$p fizzles, pops and implodes."; break;
    case ITEM_STAFF:      message = "$p folds up into oblivion."; break;
    case ITEM_SCROLL:     message = "$p becomes unreadible."; break;
    case ITEM_ARMOR:      message = "$p falls apart."; break;
    case ITEM_GEM:        message = "$p shatters into thousands of shards!"; break;
    case ITEM_PILL:       message = "$p crumbles into dust."; break;
	case ITEM_FOUNTAIN:   message = "$p dries up.";         break;
    case ITEM_CORPSE_NPC: message = "$p rots away to nothing."; break;
    case ITEM_CORPSE_PC:  message = "$p decays into dust."; break;
    case ITEM_FOOD:       message = "$p goes bad.";   break;
    case ITEM_DRINK_CON:  message = "$p disappears in a puff of smoke.";   break;
	}

	if ( obj->carried_by != NULL )
	{
	    act( message, obj->carried_by, obj, NULL, TO_CHAR );
	}
	else if ( obj->in_room != NULL
	&&      ( rch = obj->in_room->people ) != NULL )
	{
	    act( message, rch, obj, NULL, TO_ROOM );
	    act( message, rch, obj, NULL, TO_CHAR );
	}

	extract_obj( obj );
    }

    return;
}



/*
 * Aggress.
 *
 * for each mortal PC
 *     for each mob in room
 *         aggress on some random PC
 * also take care of scramblers
 *
 * This function takes 25% to 35% of ALL Merc cpu time.
 * Unfortunately, checking on each PC move is too tricky,
 *   because we don't the mob to just attack the first PC
 *   who leads the party into the room.
 *
 * -- Furey
 */
void aggr_update( void )
{
    CHAR_DATA *wch;
    CHAR_DATA *wch_next;
    CHAR_DATA *ch;
    CHAR_DATA *ch_next;
    CHAR_DATA *vch;
    CHAR_DATA *vch_next;
    CHAR_DATA *victim;
    EXIT_DATA *pexit;
    int door;

    for ( wch = char_list; wch != NULL; wch = wch_next )
    {
	wch_next = wch->next;

    /* Scramble */
    if ( IS_NPC(wch)
    && !IS_SET(wch->act, ACT_SENTINEL)
    && IS_SET(wch->act, ACT_SCRAMBLE)
    && ( door = number_bits( MAX_DIR ) ) < MAX_DIR
    && ( pexit = wch->in_room->exit[door] ) != NULL
	&&   pexit->to_room != NULL
	&&   !IS_SET(pexit->exit_info, EX_CLOSED)
	&&   !IS_SET(pexit->to_room->room_flags, ROOM_NO_MOB)
    && ( !IS_SET(wch->act, ACT_STAY_AREA)
    ||   pexit->to_room->area == wch->in_room->area ) )
	{
        move_char( wch, door );
	}

    if ( IS_NPC(wch)
    ||   wch->level >= LEVEL_IMMORTAL
    ||   wch->in_room == NULL )
        continue;

	for ( ch = wch->in_room->people; ch != NULL; ch = ch_next )
	{
	    int count;

	    ch_next	= ch->next_in_room;

	    if ( !IS_NPC(ch)
	    ||  ( !IS_SET(ch->act, ACT_AGGRESSIVE)
             &&   !IS_SET(ch->act, ACT_ANGRY) )
	    ||   ch->fighting != NULL
	    ||   IS_AFFECTED(ch, AFF_CHARM)
	    ||   !IS_AWAKE(ch)
	    ||   ( IS_SET(ch->act, ACT_WIMPY) && IS_AWAKE(wch) )
	    ||   !can_see( ch, wch ) )
		continue;

        if ( IS_SET(ch->act, ACT_ANGRY) && ( number_range( 0, 10 ) != 1 ) )
        continue;

	    /*
	     * Ok we have a 'wch' player character and a 'ch' npc aggressor.
	     * Now make the aggressor fight a RANDOM pc victim in the room,
	     *   giving each 'vch' an equal chance of selection.
	     */
	    count	= 0;
	    victim	= NULL;
	    for ( vch = wch->in_room->people; vch != NULL; vch = vch_next )
	    {
		vch_next = vch->next_in_room;

		if ( !IS_NPC(vch)
		&&   vch->level < LEVEL_IMMORTAL
		&&   ( !IS_SET(ch->act, ACT_WIMPY) || !IS_AWAKE(vch) )
		&&   can_see( ch, vch ) )
		{
		    if ( number_range( 0, count ) == 0 )
			victim = vch;
		    count++;
		}
	    }

	    if ( victim == NULL )
	    {
		bug( "Aggr_update: null victim.", count );
		continue;
	    }

	    multi_hit( ch, victim, TYPE_UNDEFINED );
	}
    }

    return;
}



/*
 * Reset one room.
 */
void reset_room( ROOM_INDEX_DATA *pRoom )
{
    RESET_DATA *pReset;                /* Current Reset                   */
    CHAR_DATA *pMob;                   /* Current Working Mobile          */
    OBJ_DATA *pObj;                    /* Current Working Object          */
    CHAR_DATA    *LastMob;             /* Last Mob Created                */
    OBJ_DATA     *LastObj;             /* Last Object Created             */
    int iExit;                         /* For exit resets                 */
    int olevel;                        /* for loading okie dokies on mobs */

    if ( pRoom == NULL ) return;
    if ( get_room_index( pRoom->vnum ) == NULL )
    {
          bug( "Reset_room: invalid room %d", pRoom->vnum);
          return;
    }

    pMob    = NULL;
    LastMob = NULL;
    LastObj = NULL;
    olevel  = 0;
    
    for ( iExit = 0;  iExit < MAX_DIR;  iExit++ )
    {
       EXIT_DATA *pExit;
       EXIT_DATA *tExit;
       
       pExit = pRoom->exit[iExit];
       if ( pExit == NULL ) continue;
       tExit = pExit->to_room != NULL ? pExit->to_room->exit[rev_dir[iExit]]
                                      : NULL;
       
       pExit->exit_info = pExit->rs_flags;      /* set the reset flags   */
       if ( tExit != NULL )
        tExit->exit_info = tExit->rs_flags;     /* nail the other side */
    }

    for ( pReset = pRoom->reset_first; pReset != NULL; pReset = pReset->next )
    {
	MOB_INDEX_DATA *pMobIndex;
	OBJ_INDEX_DATA *pObjIndex;

    if ( number_percent( ) > pReset->percent )
        continue;

	switch ( pReset->command )
	{
        default: bug( "Reset_room: bad command %c.", pReset->command );
	    break;

    case 'M':

        if ( ( pMobIndex = get_mob_index( pReset->rs_vnum ) ) == NULL )
        {
            bug( "Reset_room: 'M': bad vnum %d.", pReset->rs_vnum );
            break;
        }

        if ( pMobIndex->count >= pReset->loc )
            break;

        pMob = create_mobile( pMobIndex );

        char_to_room( pMob, pRoom );
        script_update( pMob, TRIG_BORN );

        LastObj = NULL;
        LastMob = pMob;
        olevel  = pMob->level;
     break;

	case 'O':
        if ( ( pObjIndex = get_obj_index( pReset->rs_vnum ) ) == NULL )
        {
            bug( "Reset_room: 'O': bad vnum %d.", pReset->rs_vnum );
            continue;
        }

        if ( LastMob != NULL
          && LastMob->pIndexData->pShop != NULL
          && pObjIndex->item_type == ITEM_LIST )
        pObj = create_object( pObjIndex, -1 );
   else pObj = create_object( pObjIndex, number_fuzzy( olevel ) );

        if ( pReset->loc == RESET_LOC_INOBJ && LastObj != NULL )
                                                obj_to_obj( pObj, LastObj );
   else if ( pReset->loc == RESET_LOC_PREVOBJ )
        {
            bug( "Reset_room: Obsolete form (%d).", pRoom->vnum );
            break;
        }
   else if ( pReset->loc == RESET_LOC_ONOBJ && LastObj != NULL )
/* change */                                    obj_to_obj( pObj, LastObj );
   else if ( pReset->loc == RESET_LOC_INROOM )
        {
        OBJ_DATA *obj2;
        for ( obj2 = pRoom->contents; obj2 != NULL; obj2 = obj2->next_content )
        {
            if ( obj2->pIndexData->vnum == pObj->pIndexData->vnum )
                break;
        }

        if ( obj2 == NULL )
            obj_to_room( pObj, pRoom );
        }
   else if ( LastMob != NULL )
        {
            obj_to_char( pObj, LastMob );
            pObj->wear_loc = URANGE( WEAR_NONE, pReset->loc, MAX_WEAR );
            /* For Shopkeepers */
            if ( LastMob->pIndexData->pShop != NULL
              && pObj->wear_loc == WEAR_NONE )
            {
                SET_BIT(pObj->extra_flags, ITEM_INVENTORY);
            }
        }
        
        if ( pObj->carried_by == NULL
          && pObj->in_room    == NULL
          && pObj->in_obj     == NULL )
        {
            extract_obj( pObj );
            pObj = NULL;
            continue;
        }

        if ( LastObj == NULL || pObj->item_type == ITEM_CONTAINER )
                LastObj = pObj;
        break;

	}
    }

    return;
}


/*
 * Repopulate rooms periodically (update 1/1024 of the rooms).
 */
void room_update( void )
{
    static int iHash;
    ROOM_INDEX_DATA *pRoom;

    if (iHash < 0) iHash = -1;
    if (++iHash > MAX_KEY_HASH )
        iHash = 0;

    for (pRoom = room_index_hash[iHash]; pRoom != NULL; pRoom = pRoom->next )
    {
        CHAR_DATA *pch;

        for ( pch = pRoom->people; pch != NULL; pch = pch->next_in_room )
            send_to_char( pRoom->area->repop, pch );

        reset_room( pRoom );
    }

    return;
}

/*
 * Handle all kinds of updates.
 * Called once per pulse from game loop.
 * Random times to defeat tick-timing clients and players.
 */
void update_handler( void )
{
    static  int     pulse_area;
    static  int     pulse_mobile;
    static  int     pulse_violence;
    static  int     pulse_point;
    static  int     autosave_counter;

    if ( --pulse_area     <= 0 )
    {
    pulse_area  = number_range( PULSE_AREA / 2, 3 * PULSE_AREA / 2 );
    char_update ( );
    obj_update  ( );
    }

    if ( autosave_counter == 999999 )
    {
        DESCRIPTOR_DATA *d;
        autosave_counter = (PULSE_PER_SECOND * 60) * 60;
        do_asave( NULL, "" );
        for ( d = descriptor_list; d != NULL; d = d->next )
        {
        if ( d->character == NULL ) continue;
        send_to_char( "Database saved.\n\r",  d->character );
        }
    }

    if ( --autosave_counter <= 0 )
    {
        DESCRIPTOR_DATA *d;
        autosave_counter = 999999;
        for ( d = descriptor_list; d != NULL; d = d->next )
        {
        if ( d->character == NULL ) continue;
        send_to_char( "Please wait while the database is being saved.\n\r",
                      d->character );
        }
    }

    if ( --pulse_mobile   <= 0 )
    {
        pulse_mobile    = PULSE_MOBILE;
        mobile_update   ( );
        room_update ( ); 
    }

    if ( --pulse_violence <= 0 )
    {
	pulse_violence	= PULSE_VIOLENCE;
	violence_update	( );
    }

    if ( --pulse_point    <= 0 )
    {
    CHAR_DATA *mob;
    for ( mob = char_list; mob != NULL; mob = mob->next )
    {
    script_update( mob, TRIG_TICK_PULSE );
    }

        pulse_point     = number_range( PULSE_TICK / 2, 3 * PULSE_TICK / 2 );
        weather_update  ( );
    }

    aggr_update( );
    tail_chain( );
    return;
}




