/******************************************************************************
 Copyright 2000-2001 Richard Woolcock.  All rights reserved.  This software may
 only be used, copied, modified, distributed or sub-licensed under the terms of
 the Glad license, which must always be included with any distributions of this
 software.  This copyright notice must remain unmodified at the top of any file
 containing any of the code found within this file.
 ******************************************************************************/

/******************************************************************************
 File Name        : combat_legs.c
 ******************************************************************************
 Description      : The "legs" combat table and associated commands.
 ******************************************************************************
 Revision History :

 Date/Author : DD-MMM-YYYY   <author's name>
 Description : <description of change>

 Date/Author : 15-Jan-2001   Richard Woolcock (aka KaVir).
 Description : Initial version for Glad 2.0a.
 ******************************************************************************/

/******************************************************************************
 Required library files.
 ******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include "sockets.h"
#include "combat.h"
#include "combat_legs.h"
#include "glad.h"

/******************************************************************************
 Required local variables.
 ******************************************************************************/

/* Create the combat table */
const cmbt_table_t kstCmbtTableLegs[] =
{ /* Requirements Function        AP   ATT  DEF  DAM  Block message */
   { " ",         CmbtNull,       1,   +0,  +0,  +0,  "misses" },
   { "-a",        CmbtRiseA,      6,   +0,  -20, +0,  "misses" },
   { "-d",        CmbtRiseD,      7,   +0,  -20, +0,  "misses" },
   { "-j",        CmbtRiseJ,      5,   +0,  -30, +0,  "misses" },
   { "-k",        CmbtRiseK,      5,   +0,  -40, +0,  "misses" },
   { "-r",        CmbtRiseR,      8,   +0,  -10, +0,  "misses" },
   { "-s",        CmbtRiseS,      6,   +0,  -30, +0,  "misses" },
   { "dsk",       CmbtSweep,      7,   +0,  -10, +0,  "misses" },
   { "dk",        CmbtSweep,      6,   -20, -5,  +0,  "misses" },
   { "dd",        CmbtCrouch,     1,   +0,  +10, +0,  "misses" },
   { "kkkkk",     CmbtTurningKick,6,   +0,  -15, +0,  "misses" },
   { "kkkk",      CmbtTurningKick,5,   +0,  -10, +0,  "misses" },
   { "kkk",       CmbtTurningKick,4,   +0,  -5,  +0,  "misses" },
   { "kks",       CmbtSpinKick,   5,   +5,  -10, +5,  "misses" },
   { "kk",        CmbtSideKick,   4,   +0,  -5,  +0,  "misses" },
   { "jja",       CmbtSummersault,5,   +0,  +25, +0,  "misses" },
   { "jjd",       CmbtLand,       1,   +0,  +0,  +0,  "misses" },
   { "jjj",       CmbtLand,       1,   +0,  +0,  +0,  "misses" },
   { "jjk",       CmbtJumpKick,   7,   +15, -10, +5,  "misses" },
   { "jjr",       CmbtBackflip,   3,   +0,  +50, +0,  "misses" },
   { "jjsk",      CmbtJumpSpnKick,3,   +25, +25, +10, "misses" },
   { "jjs",       CmbtJumpSpin,   3,   +0,  +25, +0,  "misses" },
   { "jj",        CmbtLeap,       3,   +0,  +25, +0,  "misses" },
   { "sksksks",   CmbtSpinKick,   5,   +5,  -20, +5,  "misses" },
   { "sksksk",    CmbtStepKick,   6,   +5,  -15, +0,  "misses" },
   { "sksks",     CmbtSpinKick,   5,   +5,  -15, +5,  "misses" },
   { "sksk",      CmbtStepKick,   6,   +5,  -10, +0,  "misses" },
   { "sks",       CmbtSpinKick,   5,   +5,  -10, +5,  "misses" },
   { "sk",        CmbtStepKick,   6,   +5,  -5,  +0,  "misses" },
   { "-",         CmbtFall,       5,   +0,  -50, +0,  "misses" },
   { "a",         CmbtAdvance,    5,   +5,  +0,  +0,  "misses" },
   { "d",         CmbtDuck,       1,   +0,  +5,  +0,  "misses" },
   { "j",         CmbtJump,       1,   +0,  +0,  +0,  "misses" },
   { "k",         CmbtFrontKick,  6,   +0,  -5,  +0,  "misses" },
   { "r",         CmbtRetreat,    5,   +0,  +5,  +0,  "misses" },
   { "s",         CmbtCircle,     3,   +0,  +5,  +0,  "misses" },

   /* Commands (an '@' followed by one letter) */
   { "@a",        CmbtCmdAdvance, 0,   +0,  +0,  +0,  "misses" },
   { "@d",        CmbtCmdDuck,    0,   +0,  +0,  +0,  "misses" },
   { "@j",        CmbtCmdJump,    0,   +0,  +0,  +0,  "misses" },
   { "@k",        CmbtCmdKick,    0,   +0,  +0,  +0,  "misses" },
   { "@r",        CmbtCmdRetreat, 0,   +0,  +0,  +0,  "misses" },
   { "@s",        CmbtCmdSpin,    0,   +0,  +0,  +0,  "misses" },

   { NULL }
};

/******************************************************************************
 Commands specific to this table.
 ******************************************************************************/

CMBT(CmdAdvance)
{
   const int kiCost = 50; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to advance, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 'a', TABLE_LEGS ); /* Call the combat function   */
   }
}

CMBT(CmdDuck)
{
   const int kiCost = 75; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to duck, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 'd', TABLE_LEGS ); /* Call the combat function   */
   }
}

CMBT(CmdJump)
{
   const int kiCost = 75; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to jump, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 'j', TABLE_LEGS ); /* Call the combat function   */
   }
}

CMBT(CmdKick)
{
   const int kiCost = 100; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to kick, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 'k', TABLE_LEGS ); /* Call the combat function   */
   }
}

CMBT(CmdRetreat)
{
   const int kiCost = 50; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to retreat, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 'r', TABLE_LEGS ); /* Call the combat function   */
   }
}

CMBT(CmdSpin)
{
   const int kiCost = 75; /* The cost of this particular fighting move */

   if ( pstBody->bBusy[AP_LEGS] )
   {
      SendToBody( pstBody, TO_USER, "Your legs are already in action!\n\r" );
   }
   else if ( pstBody->iActions[AP_LEGS] < kiCost )
   {
      /* Inform the player that they don't have enough action points */
      SendToBody( pstBody, TO_USER, "You need %d action points to spin, you have %d.\n\r", 
         kiCost, pstBody->iActions[AP_LEGS] );
   }
   else /* The character is able to make the attack */
   {
      /* Inform the user that the move is taking place */
      SendToBody( pstBody, TO_USER, "Ok.\n\r" );

      pstBody->iActions[AP_LEGS] -= kiCost;        /* Subtract the action points */
      pstBody->bBusy[AP_LEGS] = TRUE;              /* Set legs to "busy"         */
      Combat( pstBody, AP_LEGS, 's', TABLE_LEGS ); /* Call the combat function   */
   }
}

/******************************************************************************
 Combat techniques.
 ******************************************************************************/

CMBT(Land)
{
   SendToBody( pstBody, TO_USER, "You land back on your feet.\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} lands back on {his/her} feet.\n\r" );

   /* Once you land, you're back in a regular standing position */
   PushMove(pstBody->a_chTechniques[eLocation], ' ');
   pstBody->pfnAttack[eLocation] = CmbtNull;
   pstBody->iSpeed[eLocation] = 0;
   pstBody->bBusy[eLocation] = FALSE;
   pstBody->iCmbtIndex[eLocation] = 0;
}

CMBT(Advance)
{
   SendToBody( pstBody, TO_USER, "You advance towards {opponent}.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} advances towards you.\n\r" );
}

CMBT(Circle)
{
   SendToBody( pstBody, TO_USER, "You slowly circle {opponent}, moving warily.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} slowly circles you, moving warily.\n\r" );
}

CMBT(Duck)
{
   SendToBody( pstBody, TO_USER, "You duck.\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} ducks.\n\r" );
}

CMBT(Crouch)
{
   SendToBody( pstBody, TO_USER, "You crouch.\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} crouches.\n\r" );
}

CMBT(Jump)
{
   SendToBody( pstBody, TO_USER, "You coil your legs beneath you, ready to spring.\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} coils {his/her} legs beneath {him/her}, ready to spring.\n\r" );
}

CMBT(Leap)
{
   SendToBody( pstBody, TO_USER, "You leap up into the air.\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} leaps up into the air.\n\r" );

   /* If they don't do anything else, they land after a couple of seconds */
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(Summersault)
{
   SendToBody( pstBody, TO_USER, "You summersault over {opponent}'s head!\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} summersault over your head!\n\r" );

   /* If they don't do anything else, they land after a couple of seconds */
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(Backflip)
{
   SendToBody( pstBody, TO_USER, "You backflip away from {opponent}!\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} backflips away from you!\n\r" );

   /* If they don't do anything else, they land after a couple of seconds */
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(JumpSpin)
{
   SendToBody( pstBody, TO_USER, "You spin around in mid-air!\n\r" ); 
   SendToBody( pstBody, TO_ROOM, "{name} spins around in mid-air!\n\r" );

   /* If they don't do anything else, they land after a couple of seconds */
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(Retreat)
{
   SendToBody( pstBody, TO_USER, "You retreat away from {opponent}.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} retreats away from you.\n\r" );
}

CMBT(FrontKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "front kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a front kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a front kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }
}

CMBT(SideKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "side kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a side kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a side kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }
}

CMBT(TurningKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "turning kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a turning kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a turning kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }
}

CMBT(SpinKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "spinning side kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a spinning side kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a spinning side kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }
}

CMBT(StepKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "stepping side kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a stepping side kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a stepping side kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }
}

CMBT(JumpKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "jumping side kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a jumping side kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a jumping side kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }

   /* They HAVE to land after this one */
   pstBody->bBusy[eLocation] = TRUE;
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(JumpSpnKick)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iDamage = GetEnhancedDamage(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iDamage;
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;

   if ( !IsBlocked( pstBody, "jumping spinning side kick", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You strike {opponent} with a jumping spinning side kick!\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} strikes you with a jumping spinning side kick!\n\r" );

      /* Inflict damage */
      pstOpponent->iDam += RollDice(iDamage);
   }

   /* They HAVE to land after this one */
   pstBody->bBusy[eLocation] = TRUE;
   pstBody->pfnAttack[eLocation] = CmbtLand;
   pstBody->iSpeed[eLocation] = 2;
}

CMBT(Sweep)
{
   int iCmbtIndex = pstBody->iCmbtIndex[AP_LEGS];
   table_t eTable = pstBody->eTable[AP_LEGS];
   int iAttack = GetAttack(pstBody) + kstCmbtTable[eTable][iCmbtIndex].iAttack;
   body_t* pstOpponent = pstBody->pstOpponent;
   int i; /* Loop counter */

   if ( !IsBlocked( pstBody, "leg sweep", iAttack ) )
   {
      /* Send the message */
      SendToBody( pstBody, TO_USER, "You sweep {opponent} off {opponent.his/her} feet.\n\r" );
      SendToBody( pstOpponent, TO_USER, "{opponent} sweeps you off your feet.\n\r" );

      /* No damage, but screws up their moves */
      for ( i = AP_LEFT_HAND; i <= AP_RIGHT_HAND; i++ )
      {
         PushMove(pstOpponent->a_chTechniques[i], ' ');
         pstOpponent->iCmbtIndex[i] = 0;
         pstOpponent->iSpeed[i] = 0;
         pstOpponent->bBusy[i] = FALSE;
         pstOpponent->pfnAttack[i] = CmbtNull;
      }

      pstOpponent->bBusy[AP_LEGS] = TRUE;
      Combat( pstOpponent, AP_LEGS, '-', TABLE_LEGS );
   }
}

CMBT(RiseA)
{
   SendToBody( pstBody, TO_USER, "You roll forward, landing back on your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} rolls forward, landing back on {his/her} feet.\n\r" );
}

CMBT(RiseD)
{
   SendToBody( pstBody, TO_USER, "You clamber back up to your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} clambers back up to {his/her} feet.\n\r" );
}

CMBT(RiseJ)
{
   SendToBody( pstBody, TO_USER, "You flip back up to your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} flips back up to {his/her} feet.\n\r" );
}

CMBT(RiseK)
{
   SendToBody( pstBody, TO_USER, "You swing your legs in an arc, spinning back up to your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} swings {his/her} legs in an arc, spinning back up to {his/her} feet.\n\r" );
}

CMBT(RiseR)
{
   SendToBody( pstBody, TO_USER, "You roll backwards, landing back up on your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} rolls backwards, landing back up on {his/her} feet.\n\r" );
}

CMBT(RiseS)
{
   SendToBody( pstBody, TO_USER, "You roll over and clamber back up to your feet.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} rolls over and clambers back up to {his/her} feet.\n\r" );
}

CMBT(Fall)
{
   SendToBody( pstBody, TO_USER, "You crash to the floor.\n\r" );
   SendToBody( pstBody, TO_ROOM, "{name} crashes to the floor.\n\r" );
}

