#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include "defs.h"
#include "str.h"

#if defined (WIN32)
#	define vsnprintf	_vsnprintf
int strncasecmp (const char *s1, const char *s2, size_t n);
int strcasecmp (const char *s1, const char *s2);
#endif

char str_empty[1];

int str_count;
int str_real_count;

const char *str_dup(const char *str)
{
	if (str == NULL)
		return NULL;

	if (str[0] == '\0')
		return str_empty;

	return strdup(str);
}

const char *str_add(const char *str,...)
{
	va_list ap;
	size_t len;
	char *p;
	char *str_new;

	/* calculate length of sum */
	va_start(ap, str);
	len = strlen(str);
	while ((p = va_arg(ap, char*)) != NULL)
		len += strlen(p);
	va_end(ap);

	/* cat them */
	str_new = malloc(len + 1);
	strcpy(str_new, str);
	va_start(ap, str);
	while ((p = va_arg(ap, char*)) != NULL)
		strcat(str_new, p);
	va_end(ap);

	return str_new;
}

void free_string(const char *pstr)
{
	if (pstr == NULL || pstr == str_empty)
		return;
	free((void*) pstr);
}

/*
 * str_printf -- like sprintf, but prints into string.
 *		 the format is string itself
 */
const char *str_printf(const char* format,...)
{
	va_list ap;
	char buf[MAX_STRING_LENGTH];

	if (format == NULL)
		return NULL;

	va_start(ap, format);
	vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);

	return str_dup(buf);
}

/*
 * strnzcpy - copy from dest to src and always append terminating '\0'.
 *            len MUST BE > 0
 */
char *strnzcpy(char *dest, const char *src, size_t len)
{
	strncpy(dest, src, len);
	dest[len-1] = '\0';
	return dest;
}

char *strnzcat(char *dest, const char *src, size_t len)
{
	size_t old_len;

	old_len = strlen(dest);
	if (old_len >= len - 1)
		return dest;

	strncat(dest, src, len - old_len - 1);
	return dest;
}

char *strnzncat(char *dest, const char *src, size_t len, size_t count)
{
	size_t old_len;
	size_t ncopy;

	old_len = strlen(dest);
	if (old_len >= len - 1)
		return dest;

	ncopy = len - old_len - 1;
	if (count < ncopy)
		ncopy = count;

	strncat(dest, src, ncopy);
	return dest;
}

char *strlwr(const char *s)
{
	static char buf[MAX_STRING_LENGTH];
	char *p;

	if (s == NULL)
		return str_empty;

	for (p = buf; p-buf < sizeof(buf)-1 && *s; s++, p++)
		*p = LOWER(*s);
	*p = '\0';
	return buf;
}

/*
 * Compare strings, case insensitive.
 */
int str_cmp(const char *astr, const char *bstr)
{
	if (astr == NULL)
		return bstr == NULL ? 0 : -1;
	if (bstr == NULL)
		return 1;
	return strcasecmp(astr, bstr);
}

int str_ncmp(const char *astr, const char *bstr, size_t len)
{
	if (astr == NULL)
		return bstr == NULL ? 0 : -1;
	if (bstr == NULL)
		return 1;
	return strncasecmp(astr, bstr, len);
}

/*
 * Compare strings, case insensitive, for prefix matching.
 * Return TRUE if astr not a prefix of bstr
 *   (compatibility with historical functions).
 */
bool str_prefix(const char *astr, const char *bstr)
{
	if (astr == NULL)
		return TRUE;

	if (bstr == NULL)
		return TRUE;

	for (; *astr; astr++, bstr++) {
		if (LOWER(*astr) != LOWER(*bstr))
			return TRUE;
	}

	return FALSE;
}

/*
 * Compare strings, case insensitive, for match anywhere.
 * Returns TRUE is astr not part of bstr.
 *   (compatibility with historical functions).
 */
bool str_infix(const char *astr, const char *bstr)
{
	int sstr1;
	int sstr2;
	int ichar;
	char c0;

	if ((c0 = LOWER(astr[0])) == '\0')
		return FALSE;

	sstr1 = strlen(astr);
	sstr2 = strlen(bstr);

	for (ichar = 0; ichar <= sstr2 - sstr1; ichar++) {
		if (c0 == LOWER(bstr[ichar]) && !str_prefix(astr, bstr + ichar))
		    return FALSE;
	}

	return TRUE;
}

/*
 * Compare strings, case insensitive, for suffix matching.
 * Return TRUE if astr not a suffix of bstr
 *   (compatibility with historical functions).
 */
bool str_suffix(const char *astr, const char *bstr)
{
	int sstr1;
	int sstr2;

	sstr1 = strlen(astr);
	sstr2 = strlen(bstr);
	if (sstr1 <= sstr2 && !str_cmp(astr, bstr + sstr2 - sstr1))
		return FALSE;
	else
		return TRUE;
}

int cmpstr(const void *p1, const void *p2)
{
	return -str_cmp(*(char**) p1, *(char**) p2);
}


#if 0
char upp[] = "ABCDEFGHIGKLMNOPQRSTUVWXYZ";
char low[] = "abcdefghigklmnopqrstuvwxyzţ";

char to_lower(char c)
{
    int n;

    if ((n = strchr(upp, c)) == NULL)
	return c;
	
    n = n - (int)upp;
    return low[n];
}

char to_upper(char c)
{
    int n;

    if ((n = strchr(low, c)) == NULL)
	return c;
	
    n = n - (int)low;
    return upp[n];
}
#endif
