/* $Id: argument.c,v 1.666 2004/09/20 10:49:47 shrike Exp $ */


/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
       
#include <sys/types.h>
#include <stdio.h>
#include <string.h>

#include "merc.h"

static uint number = 0;
static int pc_id = 0;
static int mob_id = 0;
static int obj_id = 0;
static MOB_INDEX_DATA *pMobIndex = NULL;
static OBJ_INDEX_DATA *pObjIndex = NULL;
static ROOM_INDEX_DATA *pRoomIndex = NULL;

static void reset_static_data()
{
	pc_id = 0;
	mob_id = 0;
	obj_id = 0;
	pMobIndex = NULL;
	pObjIndex = NULL;
	pRoomIndex = NULL;
}

static bool set_static_data(CHAR_DATA *ch, const char *name, flag32_t flags)
{
	if (!IS_SET(flags, GET_F_CHAR_IS_CHAR))
		return FALSE;
	if (IS_SET(flags, GET_F_VICT_IS_CHAR))
	{
		if ((pc_id = get_remembered_pc_id(ch, name)) != 0)
			return TRUE;
		if ((pMobIndex = get_remembered_mob_index(ch, name)) != NULL)
			return TRUE;
	}
	if (IS_SET(flags, GET_F_VICT_IS_OBJ))
	{
		if ((pObjIndex = get_remembered_obj_index(ch, name)) != NULL)
			return TRUE;
	}
	return FALSE;
}

ROOM_INDEX_DATA *get_room(void *vo, flag32_t flags)
{
	CHAR_DATA *ch = (CHAR_DATA *) vo;
	OBJ_DATA *obj = (OBJ_DATA *) vo;
	ROOM_INDEX_DATA *room = NULL;

	if (IS_SET(flags, GET_F_CHAR_IS_OBJ))
	{
	   	if (obj == NULL)
			return NULL;
		while (obj->in_obj != NULL)
			obj = obj->in_obj;
		if ((obj->carried_by) != NULL)
			room = obj->carried_by->in_room;
		else
			room = obj->in_room;
	}
	else if (IS_SET(flags, GET_F_CHAR_IS_CHAR))
	{
		if (ch == NULL)
			return NULL;
		room = ch->in_room;
	}
	else if (IS_SET(flags, GET_F_CHAR_IS_ROOM))
		room = (ROOM_INDEX_DATA *) vo;
	
	return room;
}

CHAR_DATA *get_doppelganger(CHAR_DATA *ch, CHAR_DATA *rch, flag32_t flags)
{
	if (IS_SET(flags, GET_F_NO_DOPPELGANGER))
		return rch;
	if ((!IS_SET(flags, GET_F_CHAR_IS_CHAR)
		 || (IS_NPC(ch) || !IS_SET(ch->plr_flags, PLR_HOLYLIGHT)))
		&& is_affected(rch, gsn_doppelganger))
		return rch->doppel;
	return rch;
}

bool check_can_see(CHAR_DATA *ch, CHAR_DATA *rch, flag32_t flags)
{

	if (IS_SET(flags, GET_F_NO_CHECK_CAN_SEE))
		return TRUE;
	// Rooms and Objects haven't detects currently
	if (!IS_SET(flags, GET_F_CHAR_IS_CHAR))
		return TRUE;

	return can_see(ch, rch);
}

bool check_can_see_obj(CHAR_DATA *ch, OBJ_DATA *obj, flag32_t flags)
{

	if (IS_SET(flags, GET_F_NO_CHECK_CAN_SEE))
		return TRUE;
	// Rooms and Objects haven't detects currently
	if (!IS_SET(flags, GET_F_CHAR_IS_CHAR))
		return TRUE;

	return can_see_obj(ch, obj);
}

bool check_location(void *vo, flag32_t flags)
{
	ROOM_INDEX_DATA *room_char, *room_vict;
	room_char = pRoomIndex;
	room_vict = get_room(vo, flags >> 3);

	if (room_char == NULL || room_vict == NULL)
		return FALSE;

	if (IS_SET(flags, GET_F_IN_ROOM)
		&& room_char != room_vict)
		return FALSE;

	if (IS_SET(flags, GET_F_NOT_IN_ROOM)
		&& room_char == room_vict)
		return FALSE;

	if (IS_SET(flags, GET_F_IN_AREA)
		&& room_char->area != room_vict->area)
		return FALSE;

	return TRUE;
}

bool check_is_found(CHAR_DATA *ch, const char *name, CHAR_DATA *rch, flag32_t flags)
{
	CHAR_DATA *vch;
	
	if (!check_location(rch, flags))
		return FALSE;

	if (!check_can_see(ch, rch, flags))
		return FALSE;

	if (IS_NULLSTR(name))
	{
		if (IS_SET(flags, GET_F_NULLSTR_OK))
			return TRUE;
		else
			return FALSE;
	}

	if ((vch = get_doppelganger(ch, rch, flags)) == NULL)
		return FALSE;

	if (IS_SET(flags, GET_F_CHECK_FORM))
    {
        if (IS_SET(muddy_mode, MUDDY_FORM_SHOW_NEW))
        {
            if (IS_SET(vch->form, flag_value(form_brief_flags, name)))
                return TRUE;
        }
        else
        {
             if (IS_SET(vch->form, flag_value(form_flags, name)))
                return TRUE;
        }
    }

    if (IS_SET(flags, GET_F_NAME))
    {
        if (is_name_rus(vch, name))
            return TRUE;
    }

	if (IS_SET(flags, GET_F_REMEMBERED))
	{
		if (!IS_NPC(rch))
		{
			if(pc_id != 0 && rch->id == pc_id)
				return TRUE;
		}
		else
		{
			if (mob_id != 0 && rch->id == mob_id)
				return TRUE;
			if (pMobIndex != NULL && rch->pIndexData == pMobIndex)
				return TRUE;
		}
	}
	
	return FALSE;
}

bool check_is_found_obj(CHAR_DATA *ch, const char *name, OBJ_DATA *obj, flag32_t flags)
{
	if (IS_SET(flags, GET_F_WEAR_NONE)
		&& obj->wear_loc != WEAR_NONE)
		return FALSE;

	if (IS_SET(flags, GET_F_WEAR)
		&& obj->wear_loc == WEAR_NONE)
		return FALSE;

	if (!check_location(obj, flags))
		return FALSE;

	if (!check_can_see_obj(ch, obj, flags))
		return FALSE;

	if (IS_SET(flags, GET_F_NAME)
		&& is_name(name, obj->name))
		return TRUE;

	if (IS_SET(flags, GET_F_REMEMBERED))
	{
		if (obj_id != 0
			&& obj->id == obj_id)
			return TRUE;
		if (pObjIndex != NULL
			&& obj->pIndexData == pObjIndex)
			return TRUE;
	}

	return FALSE;
}

/*
 * Character finding procedures
 */

/*
 * Find a char in the room.
 */
CHAR_DATA *get_char_room_raw(CHAR_DATA *ch, const char *name, flag32_t flags)
{
	CHAR_DATA *rch;

	if (pRoomIndex == NULL)
		if ((pRoomIndex = get_room(ch, flags)) == NULL)
			return NULL;

	if (!number)
		return NULL;

	if (!str_cmp(name, "self") || !str_cmp(name, "") || !str_cmp(name, ""))
		return ch;

	flags = flags | GET_F_VICT_IS_CHAR | GET_F_IN_ROOM;

	for (rch = pRoomIndex->people; rch; rch = rch->next_in_room) 
    {
		
		if (!check_is_found(ch, name, rch, flags))
			continue;

		if (!--number)
			return rch;
	}

	return NULL;
}

CHAR_DATA *get_char_area_raw(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	CHAR_DATA *ach;

	flags = flags | GET_F_VICT_IS_CHAR;

	if ((ach = get_char_room_raw(ch, argument, flags)))
		return ach;

	if (argument[0] == '\0')
		return NULL;
	
	flags = flags | GET_F_NOT_IN_ROOM | GET_F_IN_AREA;
	
	for (ach = char_list; ach; ach = ach->next) { 
		
		if (!check_is_found(ch, argument, ach, flags))
			continue;

		if (!--number)
			return ach;
	}
	return NULL;
}

/*
 * Find a char in the world.
 */
CHAR_DATA *get_char_world_raw(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	CHAR_DATA *wch;

	flags = flags | GET_F_VICT_IS_CHAR;
 
	if ((wch = get_char_room_raw(ch, argument, flags)))
		return wch;

	if (argument[0] == '\0')
		return NULL;

	flags = flags | GET_F_NOT_IN_ROOM;
	
	for (wch = char_list; wch; wch = wch->next) {
	
		if (!check_is_found(ch, argument, wch, flags))
			continue;

		if (!--number)
			return wch;
	}

	return NULL;
}

CHAR_DATA *find_char_raw(CHAR_DATA *ch, const char *argument, int door, int range, flag32_t flags)
{
	EXIT_DATA *pExit, *bExit;
	ROOM_INDEX_DATA *back_room;
	CHAR_DATA *target;
	int opdoor;

	flags = flags | GET_F_VICT_IS_CHAR | GET_F_IN_ROOM;
	
	if ((target = get_char_room_raw(ch, argument, flags)))
		return target;

	if ((opdoor = opposite_door(door)) == -1) {
		bug("In find_char wrong door: %d", door);
		char_act("You don't see that there.", ch);
		return NULL;
	}

	while (range > 0) {
		range--;

		/* find target room */
		back_room = pRoomIndex;
		if ((pExit = pRoomIndex->exit[door]) == NULL
		||  (pRoomIndex = pExit->to_room.r) == NULL
		||  IS_SET(pExit->exit_info, EX_CLOSED))
			break;

		if ((bExit = pRoomIndex->exit[opdoor]) == NULL
		||  bExit->to_room.r != back_room) {
			char_act("The path you choose prevents your power to pass.", ch);
			return NULL;
		}
		if ((target = get_char_room_raw(ch, argument, flags))) 
			return target;
	}

	char_act("You don't see that there.", ch);
	return NULL;
}

/*
 * Find a char in the room.
 */
CHAR_DATA *get_char_room_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	CHAR_DATA *rch = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	
	if (!number)
		return NULL;
	
	reset_static_data();
	flags = flags | GET_F_VICT_IS_CHAR | GET_F_IN_ROOM;
	
	if (set_static_data(ch, arg, flags))
		rch = get_char_room_raw(ch, arg, flags | GET_F_REMEMBERED);

	if (rch)
		return rch;
	
	flags = flags 
			| GET_F_NAME
			| GET_F_CHECK_FORM
			| GET_F_NULLSTR_OK;
	
	return get_char_room_raw(ch, arg, flags);
	
}

CHAR_DATA *get_char_area_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	CHAR_DATA *ach = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	
	if (!number)
		return NULL;
	
	reset_static_data();
	flags = flags | GET_F_VICT_IS_CHAR;
	
	if (set_static_data(ch, arg, flags))
		ach = get_char_area_raw(ch, arg, flags | GET_F_REMEMBERED);

	if (ach)
		return ach;
	
	flags = flags | GET_F_NAME;
	
	return get_char_area_raw(ch, arg, flags);
	
}

CHAR_DATA *get_char_world_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	CHAR_DATA *wch = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	
	if (!number)
		return NULL;
	
	reset_static_data();
	flags = flags | GET_F_VICT_IS_CHAR;
	
	if (set_static_data(ch, arg, flags))
		wch = get_char_world_raw(ch, arg, flags | GET_F_REMEMBERED);

	if (wch)
		return wch;
	
	flags = flags | GET_F_NAME;
	
	return get_char_world_raw(ch, arg, flags);
	
}

CHAR_DATA *find_char_ext(CHAR_DATA *ch, const char *argument, int door, int range, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	CHAR_DATA *wch = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	
	if (!number)
		return NULL;
	
	reset_static_data();
	flags = flags | GET_F_VICT_IS_CHAR;
	
	if (set_static_data(ch, arg, flags))
		wch = find_char_raw(ch, arg, door, range, flags | GET_F_REMEMBERED);

	if (wch)
		return wch;

	reset_static_data();
	flags = flags | GET_F_NAME;
	
	return find_char_raw(ch, arg, door, range, flags);
	
}


/*
 * Find a char for range casting.
 * argument must specify target in form '[d.][n.]name' where
 * 'd' - direction
 * 'n' - number
 */
CHAR_DATA *get_char_spell_ext(CHAR_DATA *ch, const char *argument,
			  int *door, int range, flag32_t flags)
{
	char buf[MAX_INPUT_LENGTH];
	char *p;

	flags = flags | GET_F_VICT_IS_CHAR | GET_F_IN_ROOM;
	
	p = strchr(argument, '.');
	if (!p) {
		*door = -1;
		return get_char_room_ext(ch, argument, flags);
	}

	strnzncpy(buf, sizeof(buf), argument, p-argument);
	if ((*door = check_exit(buf)) < 0)
		return get_char_room_ext(ch, argument, flags);

	return find_char_ext(ch, p+1, *door, range, flags);
}

/*
 * Object finding procedures
 */

/*
 * Find an obj in a list.
 */
OBJ_DATA *get_obj_list_raw(CHAR_DATA *ch, const char *name, 
				OBJ_DATA *list, flag32_t flags)
{
	OBJ_DATA *obj;

	flags = flags | GET_F_VICT_IS_OBJ;
	pRoomIndex = get_room(ch, flags);
	
	for (obj = list; obj; obj = obj->next_content) {

		if (!check_is_found_obj(ch, name, obj, flags))
			continue;

		if (!--number)
			return obj;
	}

	return NULL;
}

/*
 * Find an obj in the room or in eq/inventory.
 */
OBJ_DATA *get_obj_here_raw(CHAR_DATA *ch, const char *name, flag32_t flags)
{
	OBJ_DATA *obj;

	if (pRoomIndex == NULL)
		if ((pRoomIndex = get_room(ch, flags)) == NULL)
			return NULL;

	flags = flags | GET_F_VICT_IS_OBJ;
	
	if (IS_SET(flags, GET_F_CHAR_IS_CHAR) && (ch != NULL))
	{
		/* search in player's inventory */
		obj = get_obj_list_raw(ch, name, ch->carrying, flags | GET_F_WEAR_NONE);
		if (obj)
			return obj;

		/* search in player's eq */
		obj = get_obj_list_raw(ch, name, ch->carrying, flags | GET_F_WEAR);
		if (obj)
			return obj;
	}

/* search in room contents */
	obj = get_obj_list_raw(ch, name, pRoomIndex->contents, flags);
	if (obj)
		return obj;

	return NULL;
}

OBJ_DATA *get_obj_room_raw(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	OBJ_DATA *obj;
	CHAR_DATA *vch;

	if (!number || argument[0] == '\0')
		return NULL;
	flags = flags | GET_F_VICT_IS_OBJ;

	if ((obj = get_obj_here_raw(ch, argument, flags)))
		return obj;

	if (pRoomIndex == NULL)
		return NULL;

	for (vch = pRoomIndex->people; vch; vch = vch->next_in_room) {
		/*
		 * search in the vch's inventory
		 */
		obj = get_obj_list_raw(ch, argument, vch->carrying, flags | GET_F_WEAR_NONE);
		if (obj)
			return obj;

		/*
		 * search in the vch's eq
		 */
		obj = get_obj_list_raw(ch, argument, vch->carrying, flags | GET_F_WEAR);
		if (obj)
			return obj;
	}

	return NULL;
}

/*
 * Find an obj in the world.
 */
OBJ_DATA *get_obj_world_raw(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	OBJ_DATA *obj;

	if (!number || argument[0] == '\0')
		return NULL;
	flags = flags | GET_F_VICT_IS_OBJ;

	if ((obj = get_obj_here_raw(ch, argument, flags)))
		return obj;

	for (obj = object_list; obj; obj = obj->next)
	{
		if (obj->carried_by == ch)
			continue;
		if (obj->in_room == pRoomIndex)
			continue;
		
		if (!check_is_found_obj(ch, argument, obj, flags))
			continue;
		
		if (!--number)
			return obj;
	}

	return NULL;
}

/*
 * Find an obj in a list.
 */
OBJ_DATA *get_obj_list_ext(CHAR_DATA *ch, const char *argument, OBJ_DATA *list, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_list_raw(ch, arg, list, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_list_raw(ch, arg, list, flags);
}

/*
 * Find an obj in player's inventory.
 */
OBJ_DATA *get_obj_carry_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ | GET_F_WEAR_NONE;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_list_raw(ch, arg, ch->carrying, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_list_raw(ch, arg, ch->carrying, flags);
}

/*
 * Find an obj in player's equipment.
 */
OBJ_DATA *get_obj_wear_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;
	
	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ | GET_F_WEAR;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_list_raw(ch, arg, ch->carrying, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_list_raw(ch, arg, ch->carrying, flags);

}

/*
 * Find an obj in the room or in inventory.
 */
OBJ_DATA *get_obj_here_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_here_raw(ch, arg, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_here_raw(ch, arg, flags);
}

OBJ_DATA *get_obj_room_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_room_raw(ch, arg, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_room_raw(ch, arg, flags);
}

OBJ_DATA *get_obj_world_ext(CHAR_DATA *ch, const char *argument, flag32_t flags)
{
	char arg[MAX_INPUT_LENGTH];
	OBJ_DATA *obj = NULL;

	number = number_argument(argument, arg, sizeof(arg));
	if (!number || arg[0] == '\0')
		return NULL;
	reset_static_data();
	flags = flags | GET_F_VICT_IS_OBJ;

	if (set_static_data(ch, arg, flags))
		obj = get_obj_world_raw(ch, arg, flags | GET_F_REMEMBERED);
	if (obj)
		return obj;
	flags = flags | GET_F_NAME;

	return get_obj_world_raw(ch, arg, flags);
} 

bool is_name_rus (CHAR_DATA *ch, const char *name)
{
    int i;

    if (is_name(name, ch->name))
        return TRUE;
    if (!IS_NPC(ch) && ch->pcdata)
        for (i = 0; i < 6; ++i)
        {
            if (!str_prefix(name, ch->pcdata->rusnames[i]))
                return TRUE;
        }
    return FALSE;
}
