/*
 * $Id: auction.c,v 1.666 2004/09/20 10:49:47 shrike Exp $
 */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
 /************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "merc.h"
#include "auction.h"

void    show_list_to_char       (OBJ_DATA *list, CHAR_DATA *ch,
                                 bool fShort, bool fShowNothing);
extern STAT_DATA stat_record;

AUCTION_DATA auction = { NULL };

/***************************************************************************
 *  This snippet was orginally written by Erwin S. Andreasen.              *
 *  erwin@pip.dknet.dk, http://pip.dknet.dk/~pip1773/                  *
 *  Adopted to Anatolia MUD by chronos.                                    *
 ***************************************************************************/

void talk_auction(const char *fmt, ...)
{
    va_list ap;
    char buf[MAX_STRING_LENGTH];
    DESCRIPTOR_DATA *d;
    CHAR_DATA *original;

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    for (d = descriptor_list; d != NULL; d = d->next) {
        /* if switched */
        original = d->original ? d->original : d->character;
        if (d->connected == CON_PLAYING
        &&  !IS_SET(original->comm, COMM_NOAUCTION))
                act_puts("{R{W: {x$t{x", original, buf, NULL, TO_CHAR, POS_DEAD);
    }
}


/*
  This function allows the following kinds of bets to be made:

  Absolute bet
  ============

  bet 14k, bet 50m66, bet 100k

  Relative bet
  ============

  These bets are calculated relative to the current bet. The '+' symbol adds
  a certain number of percent to the current bet. The default is 25, so
  with a current bet of 1000, bet + gives 1250, bet +50 gives 1500 etc.
  Please note that the number must follow exactly after the +, without any
  spaces!

  The '*' or 'x' bet multiplies the current bet by the number specified,
  defaulting to 2. If the current bet is 1000, bet x  gives 2000, bet x10
  gives 10,000 etc.

*/

int advatoi (const char *s)
/*
  util function, converts an 'advanced' ASCII-number-string into a number.
  Used by parsebet() but could also be used by do_give or do_wimpy.

  Advanced strings can contain 'k' (or 'K') and 'm' ('M') in them, not just
  numbers. The letters multiply whatever is left of them by 1,000 and
  1,000,000 respectively. Example:

  14k = 14 * 1,000 = 14,000
  23m = 23 * 1,000,0000 = 23,000,000

  If any digits follow the 'k' or 'm', the are also added, but the number
  which they are multiplied is divided by ten, each time we get one left. This
  is best illustrated in an example :)

  14k42 = 14 * 1000 + 14 * 100 + 2 * 10 = 14420

  Of course, it only pays off to use that notation when you can skip many 0's.
  There is not much point in writing 66k666 instead of 66666, except maybe
  when you want to make sure that you get 66,666.

  More than 3 (in case of 'k') or 6 ('m') digits after 'k'/'m' are automatically
  disregarded. Example:

  14k1234 = 14,123

  If the number contains any other characters than digits, 'k' or 'm', the
  function returns 0. It also returns 0 if 'k' or 'm' appear more than
  once.

*/

{

/* the pointer to buffer stuff is not really necessary, but originally I
   modified the buffer, so I had to make a copy of it. What the hell, it 
   works:) (read: it seems to work:)
*/

  char string[MAX_INPUT_LENGTH]; /* a buffer to hold a copy of the argument */
  char *stringptr = string; /* a pointer to the buffer so we can move around */
  char tempstring[2];       /* a small temp buffer to pass to atoi*/
  int number = 0;           /* number to be returned */
  int multiplier = 0;       /* multiplier used to get the extra digits right */


  strnzcpy(string, sizeof(string),s);        /* working copy */

  while (isdigit (*stringptr)) /* as long as the current character is a digit */
  {
      strncpy (tempstring,stringptr,1);           /* copy first digit */
      number = (number * 10) + atoi (tempstring); /* add to current number */
      stringptr++;                                /* advance */
  }

  switch (UPPER(*stringptr)) {
      case 'K'  : multiplier = 1000;    number *= multiplier; stringptr++; break;
      case 'M'  : multiplier = 1000000; number *= multiplier; stringptr++; break;
      case '\0' : break;
      default   : return 0; /* not k nor m nor NUL - return 0! */
  }

  while (isdigit (*stringptr) && (multiplier > 1)) /* if any digits follow k/m, add those too */
  {
      strncpy (tempstring,stringptr,1);           /* copy first digit */
      multiplier = multiplier / 10;  /* the further we get to right, the less are the digit 'worth' */
      number = number + (atoi (tempstring) * multiplier);
      stringptr++;
  }

  if (*stringptr != '\0' && !isdigit(*stringptr)) /* a non-digit character was found, other than NUL */
    return 0; /* If a digit is found, it means the multiplier is 1 - i.e. extra
                 digits that just have to be ignore, liked 14k4443 -> 3 is ignored */


  return (number);
}


int parsebet (const int currentbet, const char *argument)
{

  int newbet = 0;                /* a variable to temporarily hold the new bet */
  char string[MAX_INPUT_LENGTH]; /* a buffer to modify the bet string */
  char *stringptr = string;      /* a pointer we can move around */

  strnzcpy(string, sizeof(string),argument);      /* make a work copy of argument */


  if (*stringptr)               /* check for an empty string */
  {

    if (isdigit (*stringptr)) /* first char is a digit assume e.g. 433k */
      newbet = advatoi (stringptr); /* parse and set newbet to that value */

    else
      if (*stringptr == '+') /* add ?? percent */
      {
        if (strlen (stringptr) == 1) /* only + specified, assume default */
          newbet = (currentbet * 125) / 100; /* default: add 25% */
        else
          newbet = (currentbet * (100 + atoi (++stringptr))) / 100; /* cut off the first char */
      }
      else
        {
        if ((*stringptr == '*') || (*stringptr == 'x')) { /* multiply */
          if (strlen (stringptr) == 1) /* only x specified, assume default */
            newbet = currentbet * 2 ; /* default: twice */
          else /* user specified a number */
            newbet = currentbet * atoi (++stringptr);  /* cut off the first char */
            }
        }
  }

  return newbet;        /* return the calculated bet */
}


void auction_give_obj(CHAR_DATA* victim, OBJ_DATA *obj)
{
    act("    {C{x, "
        "   {x$p{x.", victim, obj, NULL, TO_CHAR);
    act("  $n  {C{x   $gn{} {x$p{x.",
        victim, obj, NULL, TO_ROOM);

    if (victim->carry_weight + get_obj_weight(obj) >
        can_carry_w(victim)) {
        act("    {x$p{x,      .",
            victim, obj, NULL, TO_CHAR);
        act("$n    ,     {x$p{x,  $gn{}  .",
            victim, obj, NULL, TO_ROOM);
        obj_to_room (obj, victim->in_room);
        return;
    }

    if (victim->carry_number + get_obj_number(obj) > can_carry_n(victim)) {
        act("    {x$p{x,   .",
            victim, obj, NULL, TO_CHAR);
        act("$n    ,     {x$p{x,  $gn{}  .",
            victim, obj, NULL, TO_ROOM);
        obj_to_room (obj, victim->in_room);
        return;
    }

    if (!can_see_obj(victim, obj)) {
        act("    {x$p{x,     {x$p{x.",
            victim, obj, NULL, TO_CHAR);
        act("$n   {x$p{x,  $gn{}  .",
            victim, obj, NULL, TO_ROOM);
        obj_to_room (obj, victim->in_room);
        return;
    }

    if (obj->pIndexData->limit != -1) {
        if ((IS_OBJ_STAT(obj, ITEM_ANTI_EVIL) && IS_EVIL(victim))
        ||  (IS_OBJ_STAT(obj, ITEM_ANTI_GOOD) && IS_GOOD(victim))
        ||  (IS_OBJ_STAT(obj, ITEM_ANTI_NEUTRAL) && IS_NEUTRAL(victim))) {
            act("     {x$p{x   {x$p{x.",
                victim, obj, NULL, TO_CHAR);
            act("$n    {x$p{x,  $gn{}  .",
                victim, obj, NULL, TO_ROOM);
            obj_to_room (obj, victim->in_room);
            return;
        }
    }
    obj_to_char (obj, victim);
}

void auction_update (void)
{
    if (auction.item == NULL)
        return;

    if (--auction.pulse > 0)
        return;

    auction.pulse = PULSE_AUCTION;
    switch (++auction.going) { /* increase the going state */
    case 1 : /* going once */
    case 2 : /* going twice */
            if (auction.bet > 0)
            /* XXX */
            talk_auction("%s{x - {C%s{x!    {G%d {Y{x!",
                fix_short(mlstr_mval(auction.item->short_descr)),
                        ((auction.going == 1) ? "" : ""),
                auction.bet);
            else 
            /* XXX */
                talk_auction("%s{x - {C%s{x!   {G%d {Y{x -  ?",
                     fix_short(mlstr_mval(auction.item->short_descr)),
                             ((auction.going == 1) ? "" : ""),
                     auction.starting);
            break;

     case 3 : /* SOLD! */
            if (auction.bet > 0) {
            int tax;
            int pay;

            /* XXX */
                talk_auction("%s{x  {W%s{x  {G%d {Y{x.",
                     fix_short(mlstr_mval(auction.item->short_descr)),
                     auction.buyer->invis_level >= LEVEL_HERO ?
                     "-" : auction.buyer->name,
                     auction.bet);

            auction_give_obj(auction.buyer, auction.item);

            tax = (auction.bet * 15) / 100;
            pay = auction.bet - tax;

             /* give him the money */
            act_puts3("      {G$j {Y$qj{}{x,       ,     {B$J {Y$qj{}{x.", auction.seller, (const void *) pay, NULL, (const void *) tax, TO_CHAR, POS_DEAD);
            if (auction.seller != NULL)
                auction.seller->pcdata->bank_g += pay;
        }
            else { /* not sold */
            /* XXX */
                talk_auction("%s{x -   , ",
                     fix_short(mlstr_mval(auction.item->short_descr)));
            talk_auction("   !");
            auction_give_obj(auction.seller, auction.item);
            }
        auction.item = NULL; /* reset item */
        }
}
 
#define get_name(s)     (s)?((s)->name):("noone")

DO_FUN(do_auction)
{
    int tax;
    OBJ_DATA *obj;
    char arg1[MAX_INPUT_LENGTH];
    char starting[MAX_INPUT_LENGTH];

    argument = one_argument(argument, arg1, sizeof(arg1));

    if (IS_NPC(ch))    /* NPC can't auction cos the can be extracted ! */
        return;

    if (IS_SET(ch->comm, COMM_NOAUCTION)) {
        if (!str_cmp(arg1, "on")) {
            char_act(" '{R{x'  .", ch);
            REMOVE_BIT(ch->comm,COMM_NOAUCTION);
            return;
        }
        else {
            char_act(" '{R{x'  .", ch);
            char_act("   '{R{x'.", ch);
            return;
        }
    }

    if (arg1[0] == '\0') {
        if (auction.item != NULL) {
            if (auction.bet > 0)
                act_puts(",       {G$j {Y$qj{}{x!      -   ?", ch, (const void *) auction.bet, NULL, TO_CHAR, POS_DEAD);
            else
                act_puts("      {G$j {Y$qj{}{x.\n      .", ch, (const void *) auction.starting, NULL, TO_CHAR, POS_DEAD);
            spell_research(0, 0, ch, auction.item, 0);


            if (auction.item->pIndexData->item_type == ITEM_CONTAINER) 
            {
                if (IS_SET(auction.item->value[1], CONT_CLOSED)) 
                {
                    char_act(str_empty, ch);
                    char_act(" .", ch);
                    if (IS_IMMORTAL(ch)) 
                    {
                        char_act(str_empty, ch);
                        act_puts("$p :", ch, auction.item, NULL, TO_CHAR, POS_DEAD);
                        show_list_to_char(auction.item->contains, ch, TRUE, TRUE);
                    }
                } else 
                {
                    char_act(str_empty, ch);
                    act_puts("$p :", ch, auction.item, NULL, TO_CHAR, POS_DEAD);
                    show_list_to_char(auction.item->contains, ch, TRUE, TRUE);
                }
            }
            if (auction.item->pIndexData->item_type == ITEM_SCABBARD) 
            {
                char_act(str_empty, ch);
                act_puts(" $p :", ch, auction.item, NULL, TO_CHAR, POS_DEAD);
                show_list_to_char(auction.item->contains, ch, TRUE, TRUE);
            }

            if (IS_IMMORTAL(ch)) 
            {
                char_act(str_empty, ch);
                act_puts("    : {W$N{x.", 
                    ch, NULL, auction.seller, TO_CHAR, POS_DEAD);
                if (auction.bet > 0) 
                    act_puts("    : {W$N{x.", 
                        ch, NULL, auction.buyer, TO_CHAR, POS_DEAD);
            }
            return;
        } else 
        {
            char_act("   ?", ch);
            return;
        }
    }

    if (!str_cmp(arg1, "off")) 
    {
        char_act(" '{R{x'  .", ch);
        SET_BIT(ch->comm,COMM_NOAUCTION);
        return;
    }

    if (IS_IMMORTAL(ch) && !str_cmp(arg1, "stop")) 
    {
        if (auction.item == NULL) 
        {
            char_act("    -  .",ch);
            return;
        }
        else { /* stop the auction */
            talk_auction(" %s{x   {C{x.",
                     fix_short(mlstr_mval(auction.item->short_descr)));
            auction_give_obj(auction.seller, auction.item);
            auction.item = NULL;

            /* return money to the buyer */
            if (auction.buyer != NULL) 
            {
                auction.buyer->pcdata->bank_g += auction.bet;
                char_act("      .", auction.buyer);
            }

            /* return money to the seller */
            if (auction.seller != NULL) 
            {
                auction.seller->pcdata->bank_g += (auction.starting * 20) / 100;
                char_act("      .", auction.seller);
            }
            return;
        }
    }

    if (!str_cmp(arg1, "bet")) {
        int newbet;

        if (auction.item == NULL) {
            char_act("       .", ch);
            return;
        }

        if (ch == auction.seller) {
            char_act("        !", ch);
            return;
        }

        if (IS_AFFECTED(ch, AFF_CHARM))
        {
            char_act("   !", ch);
            return;
        }

        /* make - perhaps - a bet now */
        if (argument[0] == '\0') {
            char_act(" ?", ch);
            return;
        }

        newbet = parsebet (auction.bet, argument);

        if (newbet > (ch->gold + ch->pcdata->bank_g)) {
            char_act("    !", ch);
            return;
        }

        if (auction.bet > 0) {
            int nbet;

            nbet = (auction.bet * 10) / 100;
            if (nbet < 5)
                nbet = 5;
            if (newbet < auction.bet + nbet) {
                act_puts("     ,   {G$j {Y$qj{}{x.", ch, (const void *) nbet, NULL, TO_CHAR, POS_DEAD);
                return;
            }
        }
        else {
            if (newbet < auction.starting) {
                char_act("     ,   .", ch);
                return;
            }
        }

        /* the actual bet is "Ok.\n"! */

        /* return the gold to the last buyer, if one exists */
        if (auction.buyer != NULL)
            auction.buyer->pcdata->bank_g += auction.bet;

        /* substract the gold - important :) */
        ch->pcdata->bank_g -= newbet;
        if (ch->pcdata->bank_g < 0) {
            ch->gold += ch->pcdata->bank_g;
            ch->pcdata->bank_g = 0;
        }

        auction.buyer = ch;
        auction.bet   = newbet;
        auction.going = 0;
        auction.pulse = PULSE_AUCTION; /* start the auction over again */

        wiznet ("$N  : $j", ch, (const void *) newbet, WIZ_AUCTION, 0, 0) ;
        talk_auction("  {G%d {Y{x  %s{x!  ?",
        newbet, fix_short(mlstr_mval(auction.item->short_descr)));
        return;
    }

    /* finally... */

    obj = get_obj_carry (ch, arg1); /* does char have the item ? */ 

    if (obj == NULL) {
        char_act("   .", ch);
        return;
    }

    if (obj->timer > 0) {
        char_act("       .", ch);
        return;
    }

    if (IS_OBJ_STAT(obj, ITEM_NOSELL))
    {
        act("      $p{x!", ch, obj, NULL, TO_CHAR);
        return;
    }

    if (!can_drop_obj(ch, obj)) {
        char_act("     .", ch);
        return;
    }

    if (auction.item != NULL) {
        act("  -     $p{x!", ch, auction.item, NULL, TO_CHAR);
        return;
    }

    argument = one_argument(argument, starting, sizeof(starting));
    if (starting[0] == '\0')
        auction.starting = MIN_START_PRICE;
    else if ((auction.starting = atoi(starting)) < MIN_START_PRICE) {
        act_puts(" $gn{}    ( {G$j {Y$qj{}{x).", ch, (const void *) MIN_START_PRICE, NULL, TO_CHAR, POS_DEAD);
        return;
    }
    if (obj->pIndexData->item_type == ITEM_MONEY
    || obj->pIndexData->item_type == ITEM_CORPSE_NPC
    || obj->pIndexData->item_type == ITEM_CORPSE_PC
    || obj->pIndexData->item_type == ITEM_RIDDLE
    || obj->pIndexData->item_type == ITEM_TATTOO)
    {
        act_puts("      $T.", ch, NULL, flag_string(item_types, obj->pIndexData->item_type), TO_CHAR, POS_SLEEPING);
        return;
    }

    tax = (auction.starting * 20) / 100;
    if ((ch->pcdata->bank_g + ch->gold) < tax) {
        act_puts("   ,    ({B$j {Y$qj{}{x).", ch, (const void *) tax, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    act_puts("     {B$j {Y$qj{}{x. ;)", ch, (const void *) tax, NULL, TO_CHAR, POS_DEAD);
    ch->pcdata->bank_g -= tax;
    if (ch->pcdata->bank_g < 0) {
        ch->gold += ch->pcdata->bank_g;
        ch->pcdata->bank_g = 0;
    }

    obj_from_char(obj);
    auction.item = obj;
    auction.bet = 0;    /* obj->cost / 100 */
    auction.buyer = NULL;
    auction.seller = ch;
    auction.pulse = PULSE_AUCTION;
    auction.going = 0;

    talk_auction("{Y  !    {W:{x %s{x.",
                 fix_short(mlstr_mval(obj->short_descr)));

    if (obj->pIndexData->limit > -1)
        ++stat_record.auctionned_limits;
    else
        ++stat_record.auctionned_items;
//        break;
//    } /* switch */

}
