/* $Id: clan.c,v 1.666 2004/09/20 10:49:47 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#if !defined (WIN32)
#include <sys/time.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined (WIN32)
#include <compat/compat.h>
#else
#include <dirent.h>
#endif

#include "merc.h"
#include "auction.h"

#define MAX_ITEM_TIME 5

extern STAT_DATA stat_record;
extern bool check_altar_negation (CHAR_DATA * ch);

varr clans = { sizeof (clan_t), 4 } ;

char *      fread_word             (FILE * fp);
void        fread_to_eol           (FILE * fp);
int         fread_clan             (FILE * fp);
int         fread_number           (FILE * fp);
OBJ_DATA *  get_clan_altar         (clan_t *);
OBJ_DATA *  get_clan_item          (clan_t *);
OBJ_DATA *  get_clan_trophy        (clan_t *);
void        do_clanbank            (CHAR_DATA * ch, const char * arg);
void        update_member_info     (CHAR_DATA * ch);

extern CHAR_DATA * gq_findchar(const char *name);
/*
 * create new clan
 */
clan_t * clan_new ()
{
    clan_t *clan ;
    int     i ;

    clan = varr_enew (&clans);

    clan->skills.nsize      = sizeof (clan_skill_t);
    clan->skills.nstep      = 4 ;
    clan->mark_vnum         = 0 ;
    clan->obj_vnum          = 0 ;
    clan->obj_ptr           = NULL ;
    clan->altar_vnum        = 0 ;
    clan->altar_trophy_vnum = 0 ;
    clan->altar_ptr         = NULL ;
    clan->altar_trp         = NULL ;
    clan->item_time         = 0 ;
    clan->clan_align        = 0 ;
    clan->bank_gold         = 0 ;
    clan->bank_questp       = 0 ;
    clan->bank_bonus        = 0 ;
    clan->bank_gems         = 0 ;
    clan->bank_rgems        = 0 ;
    clan->bank_bgems        = 0 ;
    clan->bank_ggems        = 0 ;
    clan->bank_mgems        = 0 ;
    clan->bank_wgems        = 0 ;
    clan->bank_ygems        = 0 ;
    clan->item_at           = 0 ;
    clan->msg_prays         = str_empty;
    clan->msg_vanishes      = str_empty;
    clan->energy            = 0;
    clan->conq_flags        = 0;
    clan->min_clan_level    = 30;
    clan->allow_alliance    = 1;

    for (i = 0 ; i < MAX_CLANS - 1 ; ++i) clan->diplomacy[i] = DIP_NEUTRAL ;
    return clan ;
}

/*
 * reset clan skills?
 */
void clan_free (clan_t * clan)
{
    varr_free (&clan->skills);
    free_string(clan->msg_prays);
    free_string(clan->msg_vanishes);
}

/*
 * lookup clan by name
 */
int cln_lookup (const char * name)
{
    int cln;

    if (IS_NULLSTR(name))
        return -1;

    for (cln = 0; cln < clans.nused; cln++)
        if (!str_cmp(name, CLAN(cln)->name))
            return cln;

    return -1;
}

/*
 * get clan name by index
 */
const char * clan_name (int cn)
{
    clan_t * clan = clan_lookup (cn);
    if (clan) return clan->name ;
    return "None" ;
}

/*
 * update clan member information for the specified character
 */
void update_member_info (CHAR_DATA * ch)
{
    clan_t * clan = CLAN (ch->clan);

    if (clan)
    {
        CLAN_MEMBER * member ;
        for (member = clan->member_list ; member ; member = member->next)
        {
            if (str_cmp (member->name, ch->name)) continue ;
            member->alignment = ch->alignment ;
            member->ethos     = ch->ethos ;
            member->level     = ch->level ;
            member->rank      = ch->pcdata->clan_status ;
            break ;
        }
    }

    return ;
}

void do_mark(CHAR_DATA *ch, const char *argument)
{
    OBJ_DATA *mark;
    clan_t *clan = NULL;

    if ((ch->clan == 0) || ((clan=clan_lookup(ch->clan)) == NULL))
    {
        char_act("You are not in clan.", ch);
        return;
    }
    if (!clan->mark_vnum)
    {
        char_act("Your clan does not have any mark.", ch);
        return;
    }
    if ((mark=get_eq_char(ch, WEAR_CLANMARK))!=NULL)
    {
        obj_from_char(mark);
        extract_obj(mark);
    }
    mark = create_obj(get_obj_index(clan->mark_vnum), 0);
    obj_to_char (mark, ch);
    equip_char (ch, mark, WEAR_CLANMARK);
}

// Allow Leader control of CLANBANK as Leader wish with some fee 
DO_FUN(do_clanreward)
{
    char arg1 [MAX_INPUT_LENGTH];
    char arg2 [MAX_INPUT_LENGTH];
    char arg3 [MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    clan_t      *ch_clan = NULL;
    clan_t      *victim_clan = NULL;
    int value = 0, fee = 20, amount = 0;
    char *reward=str_empty;

    if (IS_NPC(ch))
        return;

    if (ch->pcdata->clan_status != CLAN_LEADER)
    {
        char_act("Only Leader can use CLANREWARD command.", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        char_act("Back your WILL!", ch);
        return;
    }

    ch_clan = CLAN(ch->clan);

    // just to be sure
    if (ch_clan == NULL)
        return;

    argument = one_argument(argument, arg1, sizeof(arg1));
    argument = one_argument(argument, arg2, sizeof(arg2));
    one_argument(argument, arg3, sizeof(arg3));

    if (arg1[0] == '\0' || arg2[0] == '\0')
    {
        char_act("Syntax:", ch);
        char_act("  clanreward <name> <field> <value>", ch);
        char_act("  Field being one of:",            ch);
        char_act("    gold, questpoints, bonuspoints, mark, medal,", ch);
        return;
    }

    if ((victim = get_char_world(ch, arg1)) == NULL)
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (IS_NPC(victim))
    {
        char_act("Not on NPC!!!", ch);
        return;
    }

    victim_clan = CLAN(victim->clan);

    if (victim_clan == NULL)
    {
        fee = 20;
    } else
    {
        if (victim_clan == ch_clan)
            fee = 10;
        else
        if (!victim_clan->clan_align  || !ch_clan->clan_align)
            fee = 20;
        else
        if ((victim_clan->clan_align == -1000 && ch_clan->clan_align == 1000)
          ||(victim_clan->clan_align == 1000 && ch_clan->clan_align == -1000))
        {
            char_act("Not on ENEMY!!!", ch);
            return;
        }
        else
        if (victim_clan->clan_align == ch_clan->clan_align)
            fee = 10;
    }

    // Snarf the value (which must be numeric).
    // Add fun such as MEDAL and MARK.
    if (str_cmp(arg2, "mark") && str_cmp(arg2, "medal"))
    {
        if (arg3[0] == '\0')
            do_clanreward (ch, str_empty);

        if (!is_number(arg3))
        {
            char_act("Value must be numeric.", ch);
            return;
        }
        else
        {
            value = atoi(arg3);
        }

        if (value <= 0)
        {
            char_act("Value must be more than zero.", ch);
            return;
        }
    }

    // Let's set some values and get some fee

    // JUST FOR FUN!!!
    if (!str_cmp(arg2, "medal"))
    {
        reward = "+++ MEDAL +++";

        switch(number_bits(4))
        {
        case 0:
            reward = "+++ MEDAL +++";
            break;
        case 1:
            reward = "=== MEDAL ===";
            break;
        case 2:
            reward = "<<< MEDAL >>>";
            break;
        case 3:
            reward = ">=- MEDAL -=<";
            break;
        case 4:
            reward = ">>> MEDAL <<<";
            break;
        }
    }

    if (!str_cmp(arg2, "mark"))
    {
        OBJ_DATA *mark;
        if (!ch_clan->mark_vnum)
        {
            char_act("Your clan do not have any mark.", ch);
            return;
        }
        if (victim_clan == NULL || victim_clan != ch_clan)
        {
            char_act("Use this option only on your clan members.", ch);
            return;
        }
        if ((mark=get_eq_char(victim, WEAR_CLANMARK)) != NULL)
        {
            obj_from_char(mark);
            extract_obj(mark);
        }
        mark = create_obj(get_obj_index(ch_clan->mark_vnum), 0);
        obj_to_char (mark, victim);
        equip_char (victim, mark, WEAR_CLANMARK);
        reward = "clan mark";
    }

    if (!str_cmp(arg2, "gold"))
    {
        if (!IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
        {
            char_act("This option works only in clan bank!", ch);
            return;
        }

        if (ch == victim)
        {
            do_clanbank (ch, str_empty);
            return;
        }

        if (value < 100)
        {
            char_act("Minimum is 100.", ch);
            return;
        }

        amount = value + ((value / 100) * fee);

        if (get_carry_weight(victim) + value / 10 > can_carry_w(victim))
        {
            char_act("He/she can't carry that weight.", ch);
            return;
        }

        if (amount > ch_clan->bank_gold)
        {
            char_act("This more, than in clan bank!", ch);
            return;
        }

        victim->gold += value;
        ch_clan->bank_gold -= amount;

        save_clan(ch, ch_clan, FALSE);

        reward = "gold";
    }

    if (!str_cmp(arg2, "questpoints"))
    {
        if (!IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
        {
            char_act("This option works only in clan bank!", ch);
            return;
        }

        if (ch == victim)
        {
            do_clanbank (ch, str_empty);
            return;
        }

        if (value < 100)
        {
            char_act("Minimum is 100.", ch);
            return;
        }

        amount = value + ((value / 100) * fee);

        if (amount > ch_clan->bank_questp)
        {
            char_act("This more, than in clan bank!", ch);
            return;
        }

        victim->pcdata->questpoints += value;
        ch_clan->bank_questp -= amount;

        save_clan(ch, ch_clan, FALSE);

        reward = "quest points";
    }

    if (!str_cmp(arg2, "bonuspoints"))
    {
        if (!IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
        {
            char_act("This option works only in clan bank!", ch);
            return;
        }

        if (ch == victim)
        {
            do_clanbank (ch, str_empty);
            return;
        }

        if (value < 10)
        {
            char_act("Minimum is 10.", ch);
            return;
        }

        amount = value + ((value / 100) * fee);

        if (amount > ch_clan->bank_bonus)
        {
            char_act("This more, than in clan bank!", ch);
            return;
        }

        victim->pcdata->bonuspoints += value;
        ch_clan->bank_bonus -= amount;

        save_clan(ch, ch_clan, FALSE);

        reward = "bonus points";
    }

    // Generate usage message.
    if (reward == NULL)
    {
        do_clanreward (ch, str_empty);
    } else
    {
        if (value == 0)
        {
            char_printf(victim, "%s rewards you with %s.\n", ch->name, reward);
            char_printf(ch, "You reward %s with %s.\n", victim->name, reward);
        } else
        {
            char_printf(victim, "%s rewards you with %d %s.\n", ch->name, value, reward);
            char_printf(ch, "You reward %s with %d %s (fee is %d%%).\n", victim->name, value, reward, fee);
        }
        WAIT_STATE(ch, 4 * PULSE_VIOLENCE);
        ++stat_record.rewards;
    }
    return;
}

/*
 * clan petition command
 */
void do_petitio (CHAR_DATA * ch, const char * argument)
{
    char_act("  'petition' .", ch);
}

#define    MIN_CLAN_LEVEL  11

void do_petition (CHAR_DATA * ch, const char * argument)
{
    bool             accept ;
    int              cn = 0 ;
    clan_t          *clan = NULL  ;
    clan_t          *tclan = NULL ;
    CLAN_MEMBER     *member    ;
    OBJ_DATA        *mark      ;
    char             arg1 [MAX_STRING_LENGTH] ;


    if (IS_NPC (ch)) return;

    argument = one_argument(argument, arg1, sizeof(arg1));

    if (IS_NULLSTR(arg1))
    {
        if (IS_IMMORTAL(ch) || (ch->clan  && (ch->pcdata->clan_status == CLAN_LEADER || ch->pcdata->clan_status == CLAN_SECOND)))
        {
            char_printf (ch,": petition %s<accept | reject> "
                            "<char name>\n", IS_IMMORTAL(ch) ? "<clan name> " : str_empty);
        }

        if (IS_IMMORTAL(ch) || !ch->clan) char_act(": petition <clan name>", ch);
            return;
    }

    if (IS_IMMORTAL (ch))
    {
        cn = cln_lookup (arg1);

        if (cn <= 0)
        {
            char_printf (ch, "%s:  \n", arg1);
            do_petition (ch, str_empty);
            return;
        }

        argument = one_argument(argument, arg1, sizeof(arg1));

        if (IS_NULLSTR (arg1))
        {
            do_petition (ch, str_empty);
            return;
        }

        clan = CLAN (cn);
    }

    if ((accept = !str_prefix (arg1, "accept")) || !str_prefix (arg1, "reject"))
    {
        CHAR_DATA * victim = NULL;
        char arg2 [MAX_STRING_LENGTH];
        bool loaded = FALSE;

        if (!IS_IMMORTAL(ch))
        {
            if (ch->clan == 0 || (clan = clan_lookup (ch->clan)) == NULL)
            {
                do_petition (ch, str_empty);
                return;
            }
            cn = ch->clan;
        }

        argument = one_argument(argument, arg2, sizeof(arg2));

        if (IS_NULLSTR (arg2))
        {
            do_petition (ch, str_empty);
            return;
        }

        if (ch->pcdata->clan_status != CLAN_LEADER
        && ch->pcdata->clan_status != CLAN_SECOND
        && !IS_IMMORTAL(ch))
        {
            char_act("   ,  /  .", ch);
            return;
        }

        victim = gq_findchar(arg2);
        if (victim == NULL)
        {
            loaded = TRUE;
            victim = char_load_special(arg2);
        }
        if (victim == NULL)
        {
            act_puts("There's noone named $t.", ch, arg2, NULL, TO_CHAR, POS_DEAD);
            return;
        }

        if (IS_SET (victim->comm, PLR_PEACE) && !IS_IMMORTAL(ch))
        {
            char_act("       NO_PK.", ch);
            return;
        }

        // accept new clan member
        if (accept)
        {
            if (victim->pcdata->petition != cn)
            {
                char_act("    .", ch);
                return;
            }

            if (victim->clan)
            {
                char_act("     .", ch);
                return;
            }

            mark = get_eq_char (victim, WEAR_CLANMARK);

            if (mark != NULL)
            {
                char_act("      !.  - .", ch);
                return;
            }

            // special immortal warning
            if (clan->cur_clan_members > clan->max_clan_members)
                log_printf ("petition: too many clan members in clan %s", clan->name);

            // check whether max number of clan members is exceeded
            if (clan->cur_clan_members >= clan->max_clan_members)
            {
                char_act("     .", ch);
                return;
            }

            victim->clan                = cn;
            victim->pcdata->clan_status = CLAN_NEWBIE;

            // update member_list property of clan_data structure
            // to accomodate information about new member

            member = (CLAN_MEMBER *) calloc (1, sizeof (CLAN_MEMBER));

            member->name       = str_dup (victim->name);
            member->last_login = str_dup (victim->pcdata->last_login);
            member->rank       = victim->pcdata->clan_status ;
            member->level      = victim->level     ;
            member->alignment  = victim->alignment ;
            member->ethos      = victim->ethos     ;

            member->next = clan->member_list ;
            clan->member_list = member ;

            ++clan->cur_clan_members;

            update_skills (victim);

            if (IS_SET(clan->flags, CLAN_HIDDEN)
            && victim->pcdata->headprice > 0)
                victim->pcdata->headprice = 0;

            char_act("  !", ch);
            char_printf (victim, "   %s  .\n", clan->name);
            char_printf (victim, "    %s!\n",                  clan->name);

            if (clan->mark_vnum)
            {
                mark = create_obj (get_obj_index (clan->mark_vnum), 1);
                obj_to_char (mark, victim);
                equip_char  (victim, mark, WEAR_CLANMARK);
            }
            if (loaded)
                char_nuke(victim);
            return;
        }

        // reject
        if (victim == ch)
        {
            char_act("      .", ch);
            return;
        }

        if (victim->clan == cn)
        {
            CLAN_MEMBER * prev = NULL ;

            if (victim->pcdata->clan_status == CLAN_LEADER && !IS_IMMORTAL(ch))
            {
                char_act("     .", ch);
                return;
            }

            victim->clan             = CLAN_NONE ;
            victim->pcdata->petition = CLAN_NONE ;

            // remove player from the members list

            for (member = clan->member_list ; member ; prev = member, member = member->next)
            {
                if (str_cmp (member->name, victim->name)) continue ;

                // memory clean-up
                free_string (member->name);
                free_string (member->last_login);

                if (prev) prev->next   = member->next ;
                else clan->member_list = member->next ;

                --clan->cur_clan_members ;

                free (member);
                break;
            }

            update_skills (victim);

            char_printf (ch,     "      %s.\n",  clan->name);
            char_printf (victim, "      %s.\n", clan->name);

            mark = get_eq_char (victim, WEAR_CLANMARK);

            if (mark != NULL)
            {
                unequip_char (victim, mark);
                extract_obj  (mark);
            }
            if (loaded)
                char_nuke(victim);

            return;
        }

        if (victim->pcdata->petition == cn)
        {
            victim->pcdata->petition = CLAN_NONE ;
            char_act("   .", ch);
            char_printf (victim, "   %s   .\n", clan->name);
            if (loaded)
                char_nuke(victim);
            return;
        }

        char_act("    .", ch);
        if (loaded)
            char_nuke(victim);
        return;
    }

    // petition list
    if (IS_IMMORTAL(ch) || (ch->clan && (ch->pcdata->clan_status == CLAN_LEADER  || ch->pcdata->clan_status == CLAN_SECOND)))
    {
        DESCRIPTOR_DATA * d ;
        bool found = FALSE  ;

        if (IS_IMMORTAL(ch))
        {
            if ((cn = cln_lookup (arg1)) <= 0)
            {
                char_act("  .", ch);
                return;
            }
        } else cn = ch->clan ;

        for (d = descriptor_list; d; d = d->next)
        {
            CHAR_DATA * vch = d->original ? d->original : d->character ;

            if (!vch || vch->clan || vch->pcdata->petition != cn) continue ;

            if (!found)
            {
                found = TRUE ;
                char_act(" ,       :", ch);
            }

            act_puts("$N {x({W$t{x)", ch, vch->name, vch, TO_CHAR, POS_DEAD);
        }

        if (!found)
            char_act("        .", ch);
        return;
    }

    if ((cn = cln_lookup (arg1)) <= 0)
    {
        char_act("  .", ch);
        return;
    }

    if (ch->pcdata->petition)
    {
        char_printf(ch, "  (,)      %s.", clan_name(ch->pcdata->petition));
        return;
    }
    tclan = clan_lookup (cn);

    if (ch->level < tclan->min_clan_level)
    {
         char_printf(ch, "Only from %d level your can send petition to this clan.", tclan->min_clan_level);
         return;

    }

    // Welesh: Clan hunters has no align so players can't send petitions?
    //         to discuss.
//    if (!tclan->clan_align)
//    {
//        char_act("      .", ch);
//        return;
//    }

    if (((tclan->clan_align == 1000 || tclan->clan_align == 500) && IS_EVIL (ch))
    || ((tclan->clan_align == -1000 || tclan->clan_align == -500) && IS_GOOD(ch))
    || (tclan->clan_align == 1000 && !IS_GOOD(ch))
    || (tclan->clan_align == -1000 && !IS_EVIL(ch)))
    {
        char_act("     .", ch);
        return;
    }

    ch->pcdata->petition = cn ;
    char_act(" .", ch);
}

/*
 * clan promote command
 */
void do_promote (CHAR_DATA * ch, const char * argument)
{
    char arg1 [MAX_STRING_LENGTH] ;
    char arg2 [MAX_STRING_LENGTH] ;
    CHAR_DATA * victim ;
    clan_t * clan   ;
    bool loaded = FALSE;

    if (IS_NPC (ch) || (!IS_IMMORTAL (ch)
    && ch->pcdata->clan_status != CLAN_LEADER
    && ch->pcdata->clan_status != CLAN_SECOND))
    {
        char_act("    .", ch);
        return ;
    }

    argument = one_argument(argument, arg1, sizeof(arg1));
    argument = one_argument(argument, arg2, sizeof(arg2));

    if (!*arg1 || !*arg2)
    {
        char_act(": promote <char name> < newbie | commoner | veteran | elite | secondary >", ch);
        if (IS_IMMORTAL(ch)) char_act("    : promote <char name> <leader>", ch);
        return ;
    }

    // find character to promote
    victim = gq_findchar(arg1);
    if (victim == NULL)
    {
        loaded = TRUE;
        victim = char_load_special(arg1);
    }
    if (victim == NULL)
    {
        act_puts("There's noone named $t.", ch, arg1, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    if (victim->clan == 0 || (clan = clan_lookup (victim->clan)) == NULL
    || (victim->clan != ch->clan && !IS_IMMORTAL (ch)))
    {
        char_act("  .", ch);
        if (loaded)
            char_nuke(victim);
        return ;
    }

    if (!IS_IMMORTAL (ch) && victim->pcdata->clan_status == CLAN_LEADER)
    {
        char_act("        .", ch);
        if (loaded)
            char_nuke(victim);
        return ;
    }


    if ((ch->pcdata->clan_status == CLAN_SECOND)
    && ((!str_prefix (arg2, "elite")) || (!str_prefix (arg2, "secondary"))))
    {
        char_act("       .", ch);
        if (loaded)
            char_nuke(victim);
        return ;
    }

    if (!str_prefix (arg2, "leader") && IS_IMMORTAL (ch))
    {
        if (victim->pcdata->clan_status == CLAN_LEADER)
        {
            act_puts("$N   .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return ;
        }

        victim->pcdata->clan_status = CLAN_LEADER ;
        act_puts("$N    .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("    .", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        else
        {
            if (!IS_IMMORTAL(victim))
                info(victim, 0, "{r[{RINFO{r]{W : %s promoted to {GLeader{z.{x", victim->name);
        }
        return;
    }

    if (!str_prefix (arg2, "secondary"))
    {
        if (victim->pcdata->clan_status == CLAN_SECOND)
        {
            act_puts("$N      .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return;
        }

        victim->pcdata->clan_status = CLAN_SECOND ;
        act_puts("$N     .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("     .", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        else
            info(victim, 0, "{r[{RINFO{r]{W : %s promoted to {GSecondary{z.{x", victim->name);
        return;
    }

    if (!str_prefix (arg2, "commoner"))
    {
        if (victim->pcdata->clan_status == CLAN_COMMONER)
        {
            act_puts("$N     .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return ;
        }

        victim->pcdata->clan_status = CLAN_COMMONER ;
        act_puts("$N    .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("    .", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        return;
    }

    if (!str_prefix (arg2, "newbie"))
    {
        if (victim->pcdata->clan_status == CLAN_NEWBIE)
        {
            act_puts("$N   $gN{}  .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return ;
        }

        victim->pcdata->clan_status = CLAN_NEWBIE ;
        act_puts("$N  $gN{} .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("     .", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        return;
    }

    if (!str_prefix (arg2, "veteran"))
    {
        if (victim->pcdata->clan_status == CLAN_VETERAN)
        {
            act_puts("$N    .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return ;
        }

        victim->pcdata->clan_status = CLAN_VETERAN ;
        act_puts("$N   .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("   .", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        else
            info(victim, 0, "{r[{RINFO{r]{W : %s promoted to {YVeteran{z.{x", victim->name);
        return;
    }

    if (!str_prefix (arg2, "elite"))
    {
        if (victim->pcdata->clan_status == CLAN_ELITE)
        {
            act_puts("$N   .", ch, NULL, victim, TO_CHAR, POS_DEAD);
            if (loaded)
                char_nuke(victim);
            return ;
        }

        victim->pcdata->clan_status = CLAN_ELITE ;
        act_puts("$N  .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        char_act("  !.   flex    :)", victim);
        update_member_info (victim);
        if (loaded)
            char_nuke(victim);
        else
            info(victim, 0, "{r[{RINFO{r]{W : %s promoted to {RElite{z.{x", victim->name);
        return;
    }

    if (loaded)
        char_nuke(victim);

    do_promote (ch, str_empty);
}

/*
 * clan status name by index
 */
char * get_status_alias (int status)
{
    switch (status)
    {
    case CLAN_COMMONER: return "commoner"  ;
    case CLAN_SECOND:   return "secondary" ;
    case CLAN_LEADER:   return "leader"    ;
    case CLAN_NEWBIE:   return "newbie"    ;
    case CLAN_VETERAN:  return "veteran"   ;
    case CLAN_ELITE:    return "elite"     ;
    }

    return "commoner" ;
}


/*
 * initialize clan info
 */
void init_clan ()
{
    int cn ;
    for (cn = 0 ; cn < clans.nused ; ++cn)
    {
         CLAN (cn)->altar_ptr = get_clan_altar  (CLAN (cn));
         CLAN (cn)->altar_trp = get_clan_trophy (CLAN (cn));
         CLAN (cn)->obj_ptr   = get_clan_item   (CLAN (cn));
    }
}

/*
 * list clan members
 */
void do_clanlist (CHAR_DATA * ch, const char * argument)
{
    char arg1 [MAX_INPUT_LENGTH] ;

    int           cn, i    ;
    int           max, fee ;
    clan_t      * clan     ;
    CLAN_MEMBER * member   ;
    BUFFER      * buf      ;

    one_argument(argument, arg1, sizeof(arg1));

    if (is_number(arg1) && IS_IMMORTAL (ch))
    {
        max = atoi(arg1);
        clan = CLAN (ch->clan);
        if (!clan)
        {
            act("Clan?", ch, NULL, NULL, TO_CHAR);
            return;
        }
        clan->max_clan_members = max;
        save_clan(ch, clan, FALSE);
        act("Ok.", ch, NULL, NULL, TO_CHAR);
        return;
    }
    if (IS_IMMORTAL (ch) && arg1[0])
    {
        if ((cn = cln_lookup (arg1)) <= 0)
        {
            char_printf (ch, "%s: no such clan.\n", arg1);
            return ;
        }

        clan = CLAN (cn);
    }
    // take default clan using current clan of the player
    else clan = CLAN (ch->clan);

    if (!clan || (ch->clan == 0 && !str_cmp (clan->name, "None")))
    {
        char_act("    .", ch);
        return ;
    }

    if (is_name(arg1, "upgrade")
    && ch->pcdata->clan_status == CLAN_LEADER)
    {
        // max == 15
        if (clan->max_clan_members > 14)
        {
            act("Maximum clanlist limit is achieved!",
                ch, NULL, NULL, TO_CHAR);
            return;
        }
        fee = 10000;
        if (clan->max_clan_members < 10
            && clan->bank_questp > fee
            && clan->bank_gold > fee)
        {
            clan->max_clan_members = 10;
            clan->bank_questp -= fee;
            clan->bank_gold -= fee;
        } 
        else if (clan->max_clan_members < 11
                 && clan->bank_questp > fee
                 && clan->bank_gold > fee)
        {
            clan->max_clan_members = 11;
            clan->bank_questp -= fee;
            clan->bank_gold -= fee;
        } 
        else if (clan->bank_questp > fee * 2
                 && clan->bank_gold > fee * 2)
        {
            clan->max_clan_members++;
            clan->bank_questp -= fee * 2;
            clan->bank_gold -= fee * 2;
        } else
        {
            act("Not sufficiently gold or quest point in clanbank!",
                ch, NULL, NULL, TO_CHAR);
            return;
        }
        save_clan(ch, clan, FALSE);
        act("Ok.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    buf = buf_new (ch->lang);
    buf_printf (buf, "     {W=== {R> {Y  : %s{x\n", clan->name);

    i = 1 ;
    for (member = clan->member_list ; member ; member = member->next, ++i)
    {
        buf_printf (buf, "%2d. %-13s |", i, member->name);

        switch (member->rank)
        {
        case CLAN_COMMONER: buf_add (buf, " Commoner  |");     break ;
        case CLAN_SECOND:   buf_add (buf, "{R Secondary{z |"); break ;
        case CLAN_LEADER:   buf_add (buf, "{RClan Leader{z|"); break ;
        case CLAN_NEWBIE:   buf_add (buf, "{G  Newbie {z  |"); break ;
        case CLAN_VETERAN:  buf_add (buf, "  Veteran  |");     break ;
        case CLAN_ELITE:    buf_add (buf, "   Elite   |");     break ;
        default:            buf_add (buf, "  Commoner |");     break ;
        }

        if (!IS_IMMORTAL(ch) && ch->pcdata->clan_status != CLAN_LEADER &&
                                ch->pcdata->clan_status != CLAN_SECOND &&
                                ch->pcdata->clan_status != CLAN_ELITE)
        {
            buf_add (buf, "\n");
            continue ;
        }

        buf_printf (buf, " %3d %7s-%7s",
                         member->level,
                         flag_string (ethos_table, member->ethos),
                         member->alignment >=  350 ? "good" :
                         member->alignment <= -350 ? "evil" : "neutral");

        if (!IS_IMMORTAL(ch) && ch->pcdata->clan_status != CLAN_LEADER && ch->pcdata->clan_status != CLAN_SECOND)
        {
            buf_add (buf, "\n");
            continue ;
        }

        buf_printf (buf, " %.24s\n", member->last_login);
    }

    buf_printf   (buf, "     {W=== {R> {Y   : %d{x\n",
        clan->max_clan_members);
    page_to_char (buf_string (buf), ch);
    buf_free     (buf);
}

/*
 * get clan main altar object
 */
OBJ_DATA * get_clan_altar (clan_t * clan)
{
    OBJ_DATA * obj ;

    for (obj = object_list ; obj != NULL ; obj = obj->next)
    {
        if (!obj->pIndexData)
            continue;
        if ((obj->pIndexData->vnum      == clan->altar_vnum) &&
            (obj->pIndexData->item_type == ITEM_CONTAINER))
            return obj;
    }

    return NULL ;
}

/*
 * get clan trophy altar object
 */
OBJ_DATA * get_clan_trophy (clan_t * clan)
{
    OBJ_DATA * obj ;

    for (obj = object_list ; obj != NULL ; obj = obj->next)
    {
        if (!obj->pIndexData)
            continue;
        if ((obj->pIndexData->vnum == clan->altar_trophy_vnum) && (obj->pIndexData->item_type == ITEM_CONTAINER))
            return obj;
    }

    return NULL ;
}

/*
 * get clan item object
 */
OBJ_DATA * get_clan_item (clan_t * clan)
{
    OBJ_DATA * obj ;

    for (obj = object_list ; obj != NULL ; obj = obj->next)
    {
        if (!obj->pIndexData)
            continue;
        if ((obj->pIndexData->vnum == clan->obj_vnum))
            return obj;
    }

    return NULL ;
}

/*
 * check whether clan item is in the clan main altar
 */
bool is_clan_item_in_altar (clan_t * clan)
{
    OBJ_DATA * list ;

    // check whether clan has altar at all
    if (clan->altar_ptr == NULL) return FALSE ;

    for (list = clan->altar_ptr->contains ; list ; list = list->next_content)
        if (list == clan->obj_ptr) return TRUE ;

    return FALSE ;
}

/*
 * check whether clan item is in the enemy trophy altar
 */
bool is_clan_item_at_enemy (clan_t * clan)
{
    int cn ;
    clan_t * eclan ;
    OBJ_DATA  * list  ;

    for (cn = 0 ; cn < clans.nused ; ++cn)
    {
        if ((eclan = CLAN (cn)) == clan) continue ;

        // check whether enemy has trophy altar at all
        if (eclan->altar_trp == NULL) continue ;

        for (list = eclan->altar_trp->contains ; list ; list = list->next_content)
            if (list == clan->obj_ptr) return TRUE ;
    }

    return FALSE ;
}

/*
 * check whether clan item is ok
 */
bool is_clan_item_ok (CHAR_DATA * ch)
{
    clan_t * clan ;

    if (ch->clan == CLAN_NONE) return TRUE ;

    // check whether clan has any items at all
    clan = CLAN (ch->clan);
    if (clan->obj_vnum == 0) return TRUE ;

    // item is ok if it is NOT stored is someone else trophy altar
    //
    // note: should item be carried by someone or being destroyed
    // this function will return true
    return (!is_clan_item_at_enemy (clan));
}

/*
 * called from update.c
 */
void check_clans ()
{
    int cn ;
    clan_t * clan ;

    for (cn = 0 ; cn < clans.nused ; ++cn)
    {
        clan = CLAN (cn);
        check_clan_item (clan);
    }
}

/*
 * called from update.c -> check_clans () for each clan
 */
void check_clan_item (clan_t * clan)
{
    OBJ_DATA * tmp_altar = NULL ;
    int cn ;

    // try to restore altars and clan item pointers
    if (!clan->altar_ptr) clan->altar_ptr = get_clan_altar  (clan);
    if (!clan->altar_trp) clan->altar_trp = get_clan_trophy (clan);
    if (!clan->obj_ptr)   clan->obj_ptr   = get_clan_item   (clan);

    // if clan item is safe nothing more is done here
    if (is_clan_item_in_altar (clan)) return ;

    // clan item was sacrificed? demolished?
    if (clan->obj_ptr == NULL)
    {
        if (clan->obj_vnum == 0) // clan has no clan item
        {
//       ,     
//  ӣ  ,    . () GrayMage
//            if (clan != CLAN(cn_lookup("none")))
//                log_printf ("obj_vnum == 0");
            return ;
        }

        // recreate clan item
        clan->obj_ptr = create_obj (get_obj_index (clan->obj_vnum), 1);

        for (cn = 0 ; cn < clans.nused ; ++cn)
        {
            // check if clan item was last put into clan main or trophy altar
            if (clan->item_at == CLAN (cn)->altar_vnum)
            {
                tmp_altar = CLAN (cn)->altar_ptr ;
                break ;
            }

            if (clan->item_at == CLAN (cn)->altar_trophy_vnum)
            {
                tmp_altar = CLAN (cn)->altar_trp ;
                break ;
            }
        }

        // if clan item was in someone's altar put it there, otherwise
        // put it back to the clan's main altar
        if (tmp_altar) obj_to_obj (clan->obj_ptr, tmp_altar);
        else           obj_to_obj (clan->obj_ptr, clan->altar_ptr);

        // reset timer
        clan->item_time = 0 ;
        return ;
    }

    // clan item is at enemy trophy altar
    if (is_clan_item_at_enemy (clan))
    {
        // reset timer
        clan->item_time = 0 ;
        return ;
    }

    // if item is on auction wait
    if (auction.item == clan->obj_ptr) return ;

    // update timer
    ++clan->item_time ;

    // time has come
    if (clan->item_time > MAX_ITEM_TIME)
    {
        // if item is put into container - take it out
        if (clan->obj_ptr->in_obj) obj_from_obj (clan->obj_ptr); else

        // if item is carried by someone or is lying in the room
        // then get it out
        if (clan->obj_ptr->carried_by) obj_from_char (clan->obj_ptr); else
        if (clan->obj_ptr->in_room)    obj_from_room (clan->obj_ptr);

        for (cn = 0 ; cn < clans.nused ; ++cn)
        {
            if (clan->item_at == CLAN (cn)->altar_vnum)
            {
                tmp_altar = CLAN (cn)->altar_ptr ;
                break ;
            }

            if (clan->item_at == CLAN (cn)->altar_trophy_vnum)
            {
                tmp_altar = CLAN (cn)->altar_trp ;
                break ;
            }
        }

        if (tmp_altar) obj_to_obj (clan->obj_ptr, tmp_altar);
        else           obj_to_obj (clan->obj_ptr, clan->altar_ptr);

        // reset timer
        clan->item_time = 0 ;
        return ;
    }
}

/*
 * check whether given object is clan altar (main or trophy)
 */
clan_t * is_clan_altar (OBJ_DATA * obj)
{
    int cn ;
    clan_t * clan ;

    if (obj == NULL) return NULL ;
    for (cn = 0 ; cn < clans.nused ; ++cn)
    {
        clan = CLAN (cn);
        if (clan->altar_ptr == obj || clan->altar_trp == obj) return clan ;
    }

    return NULL ;
}

/*
 * check whether given object is clan item
 */
clan_t * is_clan_item (OBJ_DATA * obj)
{
    int cn ;
    clan_t * clan ;

    if (obj == NULL) return NULL ;

    for (cn = 0 ; cn < clans.nused ; ++cn)
    {
        clan = CLAN (cn);
        if (clan->obj_ptr == obj) return clan ;
    }

    return NULL ;
}

/*
 * get max percent for clan skill
 */
int get_clan_max_percent (CHAR_DATA * ch, int sn, int skill)
{
    // skills to work without clan items
    if (sn == gsn_resistance || sn == gsn_spellbane || sn == gsn_truesight)
        return skill ;

    // if clan item is not ok - skill doesn't work
    if (!is_clan_item_ok (ch))
        skill = 0 ;

    // if an altar of negation of non-ally is in area - skill doesn't work as well
    if (check_altar_negation (ch))
    {
        //char_act ("An altar of {mnegation{x blocks your clan power!", ch);
        skill = 0;
    }

    return skill ;
}

void clanbank_balance(CHAR_DATA *ch, int clan, const char *argument)
{
    clan_t *cn;

    cn = CLAN(clan);
    act_puts(" Current balance $t$T{z :", ch, IS_IMMORTAL(ch) ? "of clan {c" : "of your clan", IS_IMMORTAL(ch) ? cn->name : str_empty, TO_CHAR, POS_DEAD);
    act_puts("{YGold{z                = $j", ch, (const void *) cn->bank_gold, NULL, TO_CHAR, POS_DEAD);
    act_puts("{CQuest Points{z        = $j", ch, (const void *) cn->bank_questp, NULL, TO_CHAR, POS_DEAD);
    act_puts("{WBonus Points{z        = $j", ch, (const void *) cn->bank_bonus, NULL, TO_CHAR, POS_DEAD);
    char_act(" {G {z  :  ", ch);
    act_puts("{w {z(gems)      = $j", ch, (const void *) cn->bank_gems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{R {z(rgems)     = $j", ch, (const void *) cn->bank_rgems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{B {z(bgems)    = $j", ch, (const void *) cn->bank_bgems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{G {z(ggems)   = $j", ch, (const void *) cn->bank_ggems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{M {z(mgems)   = $j", ch, (const void *) cn->bank_mgems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{W {z(wgems) = $j", ch, (const void *) cn->bank_wgems, NULL, TO_CHAR, POS_DEAD);
    act_puts("{Y {z(ygems)     = $j", ch, (const void *) cn->bank_ygems, NULL, TO_CHAR, POS_DEAD);
    return;
}

void clanbank_help(CHAR_DATA *ch, const char *argument)
{
    char_printf(ch, ": clanbank %sdeposit  <value> <questpoints | gold | bonuspoints | gems>\n",
                IS_IMMORTAL(ch) ? "<clan> " : str_empty);
    char_printf(ch, "          %s <value> for deposit gems is <all>. \n", str_empty);
    char_printf(ch, "           clanbank %stransaction  <value> gold\n",
                IS_IMMORTAL(ch) ? "<clan> " : str_empty);
    if (IS_IMMORTAL(ch)
//    || ch->pcdata->clan_status == CLAN_SECOND
    || ch->pcdata->clan_status == CLAN_LEADER)
    {
        char_printf(ch, "           clanbank %swithdraw <value> <questpoints | gold | bonuspoints | gems>\n",
                IS_IMMORTAL(ch) ? "<clan> " : str_empty);
        char_printf(ch, "          %s withdraw gems is : gems, rgems, bgems, ggems, mgems, wgems, ygems.\n", str_empty);
    }
    if (IS_IMMORTAL(ch)
    || ch->pcdata->clan_status == CLAN_SECOND
    || ch->pcdata->clan_status == CLAN_ELITE
    || ch->pcdata->clan_status == CLAN_LEADER)
    {
        char_printf(ch, "           clanbank %sbalance\n",
                IS_IMMORTAL(ch) ? "<clan> " : str_empty);
    }
    return;
}

void clanbank_withdraw(CHAR_DATA *ch, int clan, const char *arg)
{
    OBJ_DATA *gems;
    int amount;
    char    arg1[MAX_INPUT_LENGTH];
    clan_t *cn;

    cn = CLAN(clan);

    if (!IS_IMMORTAL(ch)
    && !IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
    {
        char_act("     .", ch);
        return;
    }

    arg = one_argument(arg, arg1, sizeof(arg1));
    if (arg1[0] == '\0')
    {
        char_act(" ,  ,       ?", ch);
        return;
    }

    amount = atoi(arg1);
    if (amount <= 0)
    {
        char_act("?", ch);
        return;
    }

    if (ch->pcdata->clan_status == CLAN_ELITE
    && !is_name(arg, "gold"))
    {
        char_act("      !", ch);
        return;
    }

    if (is_name(arg, "gems")
    || is_name(arg, "rgems")
    || is_name(arg, "bgems")
    || is_name(arg, "ggems")
    || is_name(arg, "mgems")
    || is_name(arg, "wgems")
    || is_name(arg, "ygems"))
    {
        if (amount > 10 && !IS_IMMORTAL(ch))
        {
            char_act("  10-    .", ch);
            return;
        }

        if (ch->carry_number >= can_carry_n(ch))
        {
            char_act("You don't have free space for gems.", ch);
            return;
        }

        if (get_carry_weight(ch) >= can_carry_w(ch))
        {
            char_act("You can't get gems. Too gravely.", ch);
            return;
        }
    }
    if (is_name(arg, "gold"))
    {
        if (ch->pcdata->clan_status == CLAN_ELITE
        && amount > 50)
        {
            char_act("      50 !", ch);
            return;
        }

        if (get_carry_weight(ch) + amount / 10 > can_carry_w(ch))
        {
            char_act("     .", ch);
            return;
        }

        if (amount > cn->bank_gold)
        {
            char_act(" ,    !", ch);
            return;
        }

        ch->gold += amount;
        cn->bank_gold -= amount;

        char_printf(ch, "  %d   .\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "questpoints"))
    {
        if (amount > cn->bank_questp)
        {
            char_act(" ,    !", ch);
            return;
        }

        ch->pcdata->questpoints += amount;
        cn->bank_questp -= amount;

        char_printf(ch, "  %d   .\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "bonuspoints"))
    {
        if (amount > cn->bank_bonus)
        {
            char_act(" ,    !", ch);
            return;
        }

        ch->pcdata->bonuspoints += amount;
        cn->bank_bonus -= amount;

        char_printf(ch, "  %d   .\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "gems"))
    {
        if (amount > cn->bank_gems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_gems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "rgems"))
    {
        if (amount > cn->bank_rgems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_rgems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_R_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "bgems"))
    {
        if (amount > cn->bank_bgems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_bgems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_B_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "ggems"))
    {
        if (amount > cn->bank_ggems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_ggems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_G_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "mgems"))
    {
        if (amount > cn->bank_mgems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_mgems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_M_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "wgems"))
    {
        if (amount > cn->bank_wgems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_wgems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_W_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "ygems"))
    {
        if (amount > cn->bank_ygems)
        {
            char_act(" ,    !", ch);
            return;
        }

        cn->bank_ygems -= amount;
        save_clan(ch, cn, FALSE);

        while (amount--)
        {
            gems = create_obj(get_obj_index(OBJ_VNUM_Y_GEMS), 0);
            obj_to_char(gems, ch);
            char_printf(ch, "  %s  .\n",
                mlstr_cval (gems->short_descr, ch));
        }
    }

    if (is_name(arg, "gold")
    || is_name(arg, "questpoints")
    || is_name(arg, "bonuspoints")
    || is_name(arg, "gems")
    || is_name(arg, "rgems")
    || is_name(arg, "bgems")
    || is_name(arg, "ggems")
    || is_name(arg, "mgems")
    || is_name(arg, "wgems")
    || is_name(arg, "ygems"))
    {
        if (IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
            act("$n    .", ch, NULL, NULL, TO_ROOM);
    } else
    {
        clanbank_help(ch, str_empty);
    }
}

void clanbank_transaction(CHAR_DATA *ch, int clan, const char *arg)
{
    int amount;
    uint    fee;
    char    arg1[MAX_INPUT_LENGTH];
    clan_t *cn;

    cn = CLAN(clan);

    if (!IS_IMMORTAL(ch)
    &&  !IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
    {
        char_act("     .", ch);
        return;
    }

    arg = one_argument(arg, arg1, sizeof(arg1));
    if (arg1[0] == '\0')
    {
        char_act("   ?", ch);
        return;
    }

    amount = atoi(arg1);
    if (amount <= 0)
    {
        char_act("?", ch);
        return;
    }

    if (!is_name(arg, "gold") && arg[0] != '\0')
    {
        char_act("       .", ch);
        return;
    }

    fee = UMAX(1, amount * 2 / 100);

    if (ch->pcdata->bank_g < amount)
    {
        char_act("    .", ch);
        return;
    }

    cn->bank_gold += amount - fee;
    ch->pcdata->bank_g -= amount;

    char_printf(ch, "%d     ! ( %d  )\n", amount, fee);

    if (IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
        act("$n    .", ch, NULL, NULL, TO_ROOM);

    save_clan(ch, cn, FALSE);
}

void clanbank_deposit(CHAR_DATA *ch, int clan, const char *arg)
{
    int amount = 0;
    OBJ_DATA *gems, *gems_next;
    char    arg1[MAX_INPUT_LENGTH];
    clan_t *cn;

    cn = CLAN(clan);

    if (!IS_IMMORTAL(ch)
    && !IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
    {
        char_act("     .", ch);
        return;
    }

    arg = one_argument(arg, arg1, sizeof(arg1));
    if (arg1[0] == '\0')
    {
        char_act("   ?", ch);
        return;
    }

    if (is_number(arg1))
    {
        amount = atoi(arg1);
        if (amount <= 0)
        {
            char_act("?", ch);
            return;
        }
    }

    if (is_name(arg, "gold"))
    {
        if (amount > ch->gold)
        {
            char_act(" ,       .", ch);
            return;
        }

        cn->bank_gold += amount;
        ch->gold -= amount;

        char_printf(ch, "%d     !\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "questpoints"))
    {

        if (amount > ch->pcdata->questpoints)
        {
            char_act(" ,       .", ch);
            return;
        }

        cn->bank_questp += amount;
        ch->pcdata->questpoints -= amount;

        char_printf(ch, "%d     !\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "bonuspoints"))
    {

        if (amount > ch->pcdata->bonuspoints)
        {
            char_act(" ,       .", ch);
            return;
        }

        cn->bank_bonus += amount;
        ch->pcdata->bonuspoints -= amount;

        char_printf(ch, "%d     !\n", amount);

        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "gems"))
    {
        for (gems = ch->carrying; gems != NULL; gems = gems_next)
        {
            gems_next = gems->next_content;

            if (gems->pIndexData->vnum == OBJ_VNUM_GEMS)
            {
                cn->bank_gems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_R_GEMS)
            {
                cn->bank_rgems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_B_GEMS)
            {
                cn->bank_bgems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_G_GEMS)
            {
                cn->bank_ggems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_M_GEMS)
            {
                cn->bank_mgems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_W_GEMS)
            {
                cn->bank_wgems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            } else if (gems->pIndexData->vnum == OBJ_VNUM_Y_GEMS)
            {
                cn->bank_ygems += 1;
                char_printf(ch, "%s    !\n",
                    mlstr_cval (gems->short_descr, ch));
                obj_from_char(gems);
                continue;
            }
        }
        char_act("         .", ch);
        save_clan(ch, cn, FALSE);
    }

    if (is_name(arg, "gold")
    || is_name(arg, "questpoints")
    || is_name(arg, "bonuspoints")
    || is_name(arg, "gems"))
    {
        if (IS_SET(ch->in_room->room_flags, ROOM_CLANBANK))
            act("$n    .", ch, NULL, NULL, TO_ROOM);
    } else
    {
        clanbank_help(ch, str_empty);
    }
}

void do_clanbank(CHAR_DATA *ch, const char *arg)
{
    char arg1[MAX_INPUT_LENGTH];
    int clan;

    arg = one_argument(arg, arg1, sizeof(arg1));

    if (IS_IMMORTAL(ch))
    {
        if (arg1[0] == '\0')
        {
            clanbank_help(ch, str_empty);
            return;
        }

        if ((clan = cln_lookup(arg1)) > 0)
            arg = one_argument(arg, arg1, sizeof(arg1));
        else
        {
            clan = ch->clan;
            if (!clan)
            {
                char_act("     /   .", ch);
                return;
            }
        }
    }
    else
        clan = ch->clan;

    if (!clan && !ch->clan)
    {
        char_act("    .", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        char_act("  !", ch);
        return;
    }

    WAIT_STATE(ch, 4 * PULSE_VIOLENCE);

    if (arg1[0] == '\0')
    {
        clanbank_help(ch, str_empty);
        return;
    }

    if (is_name(arg1, "deposit"))
    {
        clanbank_deposit(ch, clan, arg);
        return;
    }
    if (is_name(arg1, "transaction"))
    {
        clanbank_transaction(ch, clan, arg);
        return;
    }

    if (IS_IMMORTAL(ch)
//    || ch->pcdata->clan_status == CLAN_ELITE
    || ch->pcdata->clan_status == CLAN_SECOND
    || ch->pcdata->clan_status == CLAN_LEADER)
    {
        if (is_name(arg1, "withdraw"))
        {
            clanbank_withdraw(ch, clan, arg);
            return;
        }
    }
    if (IS_IMMORTAL(ch)
    || ch->pcdata->clan_status == CLAN_ELITE
    || ch->pcdata->clan_status == CLAN_SECOND
    || ch->pcdata->clan_status == CLAN_LEADER)
    {
        if (is_name(arg1, "balance"))
        {
            clanbank_balance(ch, clan, str_empty);
            return;
        }
    }

    clanbank_help(ch, str_empty);
    return;
}

void show_diplomacy(CHAR_DATA *ch)
{
    int i, j;
    BUFFER *outb;

    outb = buf_new(-1);
    buf_printf(outb, "                 |");

    for (i = 1; i < clans.nused; i++)
        buf_printf(outb, "{M%2d{z |", i);
    buf_add(outb, "\n");

    for (i = 1; i < clans.nused; i++) {
        buf_printf(outb, "{M%2d{z. {c%-13s{z|", i, CLAN(i)->name);
        for (j = 1; j < clans.nused; j++)
            buf_printf(outb, "%s|", (i==j) ? " * " :
                (CLAN(i)->diplomacy[j] == DIP_PEACE) ? " {GP{z " :
                (CLAN(i)->diplomacy[j] == DIP_WAR) ? " {RW{z " :
                (CLAN(i)->diplomacy[j] == DIP_ALLY) ? " {WA{z " :
                (CLAN(i)->diplomacy[j] == DIP_MISTRUST) ? " {rM{z " :
                " - ");
        buf_add(outb, "\n");
    }

    buf_add(outb, "\n");
    buf_add(outb, ":\n");
    buf_add(outb, "{WA{z , {GP{z , {rM{z , {RW{z .\n");
    buf_add(outb, "| - |  .\n");
    buf_add(outb, "  {CHELP <>{z.\n");
    page_to_char(buf_string(outb), ch);
    buf_free(outb);

    return;
}

void show_clan_diplomacy(CHAR_DATA *ch, int cln)
{
    int i;
    BUFFER *outb;

    outb = buf_new(-1);
    if (ch->pcdata->clan_status == CLAN_LEADER
    || IS_IMMORTAL(ch))
        buf_printf(outb, "  {c%s (clan align : %d){z:\n",
            CLAN(cln)->name, CLAN(cln)->clan_align);
    else
        buf_printf(outb, "  {c%s{z:\n", CLAN(cln)->name);

    buf_add(outb, "+---------------+----------+----------+\n");
    buf_add(outb, "|           | :     |  :  |\n");
    buf_add(outb, "+---------------+----------+----------+\n");

    for (i = 1; i < clans.nused; i++)
    {
        if (cln == i)
            continue;
        buf_printf(outb, "| {c%-13s{z | %s | %s |\n", CLAN(i)->name,
        (CLAN(cln)->diplomacy[i] == DIP_PEACE) ? "  {Gpeace{z " :
        (CLAN(cln)->diplomacy[i] == DIP_WAR) ? "   {Rwar{z  " :
        (CLAN(cln)->diplomacy[i] == DIP_ALLY) ? "  {Wally{z  " :
        (CLAN(cln)->diplomacy[i] == DIP_MISTRUST) ? "{rmistrust{z" :
        " neutral",
        (CLAN(i)->diplomacy[cln] == DIP_PEACE) ? "  {Gpeace{z " :
        (CLAN(i)->diplomacy[cln] == DIP_WAR) ? "   {Rwar{z  " :
        (CLAN(i)->diplomacy[cln] == DIP_ALLY) ? "  {Wally{z  " :
        (CLAN(i)->diplomacy[cln] == DIP_MISTRUST) ? "{rmistrust{z" :
        " neutral");
    }
    buf_add(outb, "+---------------+----------+----------+\n");

    page_to_char(buf_string(outb), ch);
    buf_free(outb);

    return;
}

void do_diplomacy(CHAR_DATA *ch, const char *arg)
{
    char arg1[MAX_INPUT_LENGTH];
    char arg2[MAX_INPUT_LENGTH];
    clan_t  * ch_clan   = NULL ;
    clan_t  * vch_clan  = NULL ;
    int clan;

    arg = one_argument(arg, arg1, sizeof(arg1));

    if (arg1[0] == '\0')
    {
        act_puts(": diplomacy < >$t",
            ch, (ch->pcdata->clan_status == CLAN_LEADER || IS_IMMORTAL(ch)) ? " < war | mistrust | neutrality | peace | alliance >" : str_empty, NULL, TO_CHAR, POS_DEAD);
        char_act("               diplomacy table", ch);
        return;
    }

    if (is_name(arg1, "table"))
    {
        show_diplomacy(ch);
        return;
    }

    if ((clan = cln_lookup(arg1)) < 1)
    {
        char_act("   .", ch);
        return;
    }

    ch_clan = clan_lookup (ch->clan);
    vch_clan = clan_lookup (clan);

    if (ch_clan == NULL || vch_clan == NULL)
    {
        char_act(" .", ch);
        return;
    }

    arg = one_argument(arg, arg2, sizeof(arg2));

    if ((ch->pcdata->clan_status == CLAN_LEADER /* || ch->pcdata->clan_status == CLAN_SECOND*/  || IS_IMMORTAL(ch)) && arg2[0] != '\0')
    {
        if (CLAN(ch->clan) == CLAN(clan))
        {
            char_act("    ? , .", ch);
            return;
        }

        // Welesh: to discuss - why clans with clan_align 0 can't declare the war?
//        if (!vch_clan->clan_align  || !ch_clan->clan_align)
//        {
//            char_act("     .", ch);
//            return;
//        }

        if (IS_IMMORTAL(ch) && !char_security(ch,"SEC_CLAN_DIPLOMACY"))
        {
            char_act("Your security level doesn't allow to change clan diplomacy.", ch);
            return;
        }

        if (is_name(arg2, "alliance"))
        {
            if (vch_clan->clan_align != ch_clan->clan_align)
            {
                char_act("    .", ch);
                return;
            }

            if (ch_clan->allow_alliance < ALLIANCE_ALLOW)
            {
                  char_act("     .", ch);
                  return;
            }

            if (vch_clan->allow_alliance < ALLIANCE_ALLOW)
            {
                  char_act("     .", ch);
                  return;
            }

        } 
        else if (is_name(arg2, "war"))
        {
            if (vch_clan->clan_align == ch_clan->clan_align
            || vch_clan->clan_align + 500 == ch_clan->clan_align
            || vch_clan->clan_align - 500 == ch_clan->clan_align
            || vch_clan->clan_align == ch_clan->clan_align + 500
            || vch_clan->clan_align == ch_clan->clan_align - 500)
            {
                char_act("  ,  .", ch);
                return;
            }
        } 
        else if (is_name(arg2, "peace"))
        {
            if (vch_clan->clan_align != ch_clan->clan_align
            && vch_clan->clan_align + 500 != ch_clan->clan_align
            && vch_clan->clan_align - 500 != ch_clan->clan_align
            && vch_clan->clan_align != ch_clan->clan_align + 500
            && vch_clan->clan_align != ch_clan->clan_align - 500)
            {
                char_act("  ,  .", ch);
                return;
            }
        }

        // Extra check for HATE_MAGIC and LOVE_MAGIC flags
        // No peace, ally or neutrality options
        if (is_name(arg2, "peace")  || is_name(arg2, "alliance") || is_name(arg2, "neutrality"))
        {
            if ((IS_SET (vch_clan->flags, CLAN_HATE_MAGIC)  && IS_SET (ch_clan->flags, CLAN_LOVE_MAGIC))
              ||(IS_SET (ch_clan->flags, CLAN_HATE_MAGIC)   && IS_SET (vch_clan->flags, CLAN_LOVE_MAGIC)))
            {
                char_act("       .", ch);
                return;
            }
        }

        if (is_name(arg2, "war"))
        {
            if (CLAN(ch->clan)->diplomacy[clan] == DIP_WAR)
            {
                act_puts("        $t.",  ch, CLAN(clan)->name, NULL, TO_CHAR, POS_DEAD);
                return;
            }

            CLAN(ch->clan)->diplomacy[clan] = DIP_WAR;
            CLAN(clan)->diplomacy[ch->clan] = DIP_WAR;
            char_act("        !", ch);
            save_clan(ch, CLAN(ch->clan), FALSE);
            save_clan(ch, CLAN(clan), FALSE);
            return;

        } else if (is_name(arg2, "peace"))
        {
            if (CLAN(clan)->diplomacy[ch->clan] == DIP_WAR)
            {
                act_puts("     $t.", ch, CLAN(clan)->name, NULL, TO_CHAR, POS_DEAD);
            } else if (CLAN(clan)->diplomacy[ch->clan] == DIP_NEUTRAL)
            {
                act_puts("        $t,          .", ch, CLAN(clan)->name, NULL, TO_CHAR, POS_DEAD);
                CLAN(ch->clan)->diplomacy[clan] = DIP_PEACE;
            } else if (CLAN(clan)->diplomacy[ch->clan] == DIP_PEACE)
            {
                char_act("     .", ch);
                CLAN(ch->clan)->diplomacy[clan] = DIP_PEACE;
            }

            save_clan(ch, CLAN(ch->clan), FALSE);
            save_clan(ch, CLAN(clan), FALSE);
            return;
        } else if (is_name(arg2, "neutrality"))
        {
            if (CLAN(clan)->diplomacy[ch->clan] == DIP_WAR)
            {
                CLAN(ch->clan)->diplomacy[clan] = DIP_NEUTRAL;
                act_puts("       $t,          .", ch, CLAN(clan)->name, NULL, TO_CHAR, POS_DEAD);
            } else
            {
                CLAN(ch->clan)->diplomacy[clan] = DIP_NEUTRAL;
                CLAN(clan)->diplomacy[ch->clan] = DIP_NEUTRAL;
                char_act("       .", ch);
                save_clan(ch, CLAN(ch->clan), FALSE);
                save_clan(ch, CLAN(clan), FALSE);
                return;
            }
        } else if (is_name(arg2, "mistrust"))
        {
            char_act("     .", ch);
            CLAN(ch->clan)->diplomacy[clan] = DIP_MISTRUST;
            save_clan(ch, CLAN(ch->clan), FALSE);
            save_clan(ch, CLAN(clan), FALSE);
            return;
        } else if (is_name(arg2, "alliance"))
        {
            char_act("      .", ch);
            CLAN(ch->clan)->diplomacy[clan] = DIP_ALLY;
            save_clan(ch, CLAN(ch->clan), FALSE);
            save_clan(ch, CLAN(clan), FALSE);
            return;
        } else
            char_act("  ?", ch);
        return;
    }
    if (arg2[0] != '\0')
    {
        if (!ch->clan)
        {
            show_clan_diplomacy(ch, clan);
            return;
        } else
            char_act("   ,     .", ch);
        return;
    }
    show_clan_diplomacy(ch, clan);
    return;
}


static CLANBANK_CMD    clanbank_cmds [] =
{
    { "help",        clanbank_help,          },
//    { "deposit",     clanbank_deposit,       },
//    { "transaction", clanbank_transaction,   },
//    { "withdraw",    clanbank_withdraw,      },
//    { "balance",     clanbank_balance,       },

    { NULL }
};


void do_clanbank_new (CHAR_DATA* ch, const char* argument)
{
    CLANBANK_CMD     *cmd ;
    static char       command [MAX_INPUT_LENGTH] ;
    int               clan;
    char              arg1[MAX_INPUT_LENGTH];

    // for immortals
    if (IS_IMMORTAL(ch))
    {
        one_argument(argument, arg1, sizeof(arg1));

        if ((clan = cln_lookup(arg1)) > 0)
              argument = one_argument(argument, arg1, sizeof(arg1));
        else
              clan = ch->clan;

        if (!clan)
        {
            char_act("     /   .", ch);
            return;
        }
    }

    // for mortals
    clan = ch->clan;

    if (!clan && !ch->clan)
    {
        char_act("    .", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        char_act("  !", ch);
        return;
    }

    argument = one_argument (argument, command, sizeof (command)) ;

    for (cmd = clanbank_cmds ; cmd->name ; ++cmd)
    {
          if (str_prefix (command, cmd->name))
                continue ;

          (*cmd->fun) (ch, argument) ;
          return ;
    }

    (clanbank_help) (ch, argument);
}

