/* $Id: comm.c,v 1.666 2004/09/20 10:50:15 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
/************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/
 

/*
 * This file contains all of the OS-dependent stuff:
 *   startup, signals, BSD sockets for tcp/ip, i/o, timing.
 *
 * The data flow for input is:
 *    Game_loop ---> Read_from_descriptor ---> Read
 *    Game_loop ---> Read_from_buffer
 *
 * The data flow for output is:
 *    Game_loop ---> Process_Output ---> Write_to_descriptor -> Write
 *
 * The OS-dependent functions are Read_from_descriptor and Write_to_descriptor.
 * -- Furey  26 Jan 1993
 */

#include <sys/types.h>
#if !defined (WIN32)
#   include <sys/socket.h>
#   include <netinet/in.h>
#   include <arpa/telnet.h>
#   include <arpa/inet.h>
#   include <unistd.h>
#   include <netdb.h>
#   include <sys/wait.h>
#else
#   include <winsock.h>
#   include <sys/timeb.h>
#endif

#include <sys/time.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <locale.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "merc.h"
#include "quest.h"
#include "update.h"
#include "ban.h"
#include "charset.h"
#include "resolver.h"
#include "olc/olc.h"
#include "db/db.h"
#include "comm_info.h"
#include "comm_colors.h"
#include "db/dofun.h"
#include "db/lang.h"
#include "quest.h"
#include "gquest.h"
#include "crash.h"

char *crypt(const char *key, const char *salt);


DESCRIPTOR_DATA *   new_descriptor  (void);
void            free_descriptor (DESCRIPTOR_DATA *d);

/*
 * Malloc debugging stuff.
 */
#if defined(MALLOC_DEBUG)
#include <malloc.h>
extern  int malloc_debug    (int );
extern  int malloc_verify   (void);
#endif

bool class_ok(CHAR_DATA *ch , int class);

/*
 * Signal handling.
 * Apollo has a problem with __attribute(atomic) in signal.h,
 *   I dance around it.
 */
#if defined(apollo)
#define __attribute(x)
#endif

/* #include <signal.h> */

#if defined(apollo)
#undef __attribute
#endif

struct codepage {
    char* name;
    unsigned char* from;
    unsigned char* to;
};

struct codepage codepages[] = {
    { "koi8-r",             koi8_koi8,  koi8_koi8   },
    { "alt (cp866)",        alt_koi8,   koi8_alt    },
    { "win (cp1251)",       win_koi8,   koi8_win    },
    { "iso (ISO-8859-5)",   iso_koi8,   koi8_iso    },
    { "mac",                mac_koi8,   koi8_mac    },
    { "translit",           koi8_koi8,  koi8_vola   },
};
#define NCODEPAGES (sizeof(codepages) / sizeof(struct codepage))

/*
 * Socket and TCP/IP stuff.
 */

#if defined (WIN32)
#include <winsock.h>

void    gettimeofday    args( ( struct timeval *tp, void *tzp ) );

/*  Definitions for the TELNET protocol. Copied from telnet.h */

#define IAC     255 /* interpret as command: */
#define DONT    254 /* you are not to use option */
#define DO      253 /* please, you use option */
#define WONT    252 /* I won't use option */
#define WILL    251 /* I will use option */
#define SB      250 /* interpret as subnegotiation */
#define GA      249 /* you may reverse the line */
#define EL      248 /* erase the current line */
#define EC      247 /* erase the current character */
#define AYT     246 /* are you there */
#define AO      245 /* abort output--but let prog finish */
#define IP      244 /* interrupt process--permanently */
#define BREAK   243 /* break */
#define DM      242 /* data mark--for connect. cleaning */
#define NOP     241 /* nop */
#define SE      240 /* end sub negotiation */
#define EOR     239 /* end of record (transparent mode) */
#define SYNCH   242 /* for telfunc calls */

#define TELOPT_ECHO 1   /* echo */
#endif

const char go_ahead_str   [] = { IAC, GA, '\0' };
const char echo_off_str   [] = { IAC, WILL, TELOPT_ECHO,      '\0' };
const char echo_on_str    [] = { IAC, WONT, TELOPT_ECHO,      '\0' };
const char telopt_binary  [] = { IAC, DO,   TELOPT_BINARY,    '\0' };
const char compress_will  [] = { IAC, WILL, TELOPT_COMPRESS,  '\0' };
const char compress_do    [] = { IAC, DO,   TELOPT_COMPRESS,  '\0' };
const char compress_dont  [] = { IAC, DONT, TELOPT_COMPRESS,  '\0' };
const char compress_wont  [] = { IAC, WONT, TELOPT_COMPRESS,  '\0' };
const char compress2_will [] = { IAC, WILL, TELOPT_COMPRESS2, '\0' };
const char compress2_do   [] = { IAC, DO,   TELOPT_COMPRESS2, '\0' };
const char compress2_dont [] = { IAC, DONT, TELOPT_COMPRESS2, '\0' };
const char compress2_wont [] = { IAC, WONT, TELOPT_COMPRESS2, '\0' };

char *get_stat_alias        (CHAR_DATA* ch, int which);

/*
 * Global variables.
 */
DESCRIPTOR_DATA *   descriptor_list;    /* All open descriptors     */
DESCRIPTOR_DATA *   d_next;     /* Next descriptor in loop  */
FILE *          fpReserve;      /* Reserved file handle     */
bool            merc_down;      /* Shutdown         */
bool            wizlock;        /* Game is wizlocked        */
bool            newlock;        /* Game is newlocked        */
bool            exploring = TRUE;      /* Exploring on/off */
char            str_boot_time[26];
time_t          current_time;   /* time of this pulse */
int                 iNumPlayers = 0; /* The number of players on */
CONNECTIONS_CACHE_ITEM* connectionsCache;

/* Zarik comments */
/* const char *     info_trusted = "127.0.0.1"; */

extern int      max_on;

void    game_loop_unix      ();
int init_socket     (int port);
void    process_who     (int port);
void    init_descriptor     (int control);
void    close_descriptor    (DESCRIPTOR_DATA *d);
bool    read_from_descriptor    (DESCRIPTOR_DATA *d);
/* bool write_to_descriptor (int desc, char *txt, uint length); */
bool   write_to_descriptor      (DESCRIPTOR_DATA *d, char *txt, int length);
bool   write_to_descriptor_2    (int desc, char *txt, int length);
void    resolv_done     (void);

void free_mem(void *p, int sMem);
extern void    substitute_alias (DESCRIPTOR_DATA *d, const char *input);

/*
 * Other local functions (OS-independent).
 */

bool    check_parse_name    (const char *name);
void    force_bust_prompt   (CHAR_DATA *ch);
bool    check_reconnect     (DESCRIPTOR_DATA *d, const char *name,
                 bool fConn);
bool    check_playing       (DESCRIPTOR_DATA *d, const char *name);
bool    check_cache         (DESCRIPTOR_DATA *d, const char *name);

int main            (int argc, char **argv);
void    nanny           (DESCRIPTOR_DATA *d, const char *argument);
bool    process_output      (DESCRIPTOR_DATA *d, bool fPrompt);
bool    read_from_buffer    (DESCRIPTOR_DATA *d);
void    stop_idling     (CHAR_DATA *ch);
void    bust_a_prompt           (CHAR_DATA *ch);
int     log_area_popularity(void);


/* Zarik added NEW funcs */

varr    control_sockets = { sizeof(int), 2 };
varr    info_sockets = { sizeof(int), 2 };
varr    info_trusted = { sizeof(struct in_addr), 2 };


#define GETINT(v, i) (*(int*) VARR_GET(v, i))

static void usage(const char *name)
{
    fprintf(stderr, "Usage: %s [-p port...] [-i port...]\n"
            "Where:\n"
            "\t-p -- listen port\n"
            "\t-i -- info service port\n",
        get_filename(name));
    exit(1);
}

static void open_sockets(varr *v, const char *logm)
{
    int i, j;

    for (i = 0, j = 0; i < v->nused; i++) {
        int port = GETINT(v, i);
        int sock;
        if ((sock = init_socket(port)) < 0)
            continue;
        log_printf(logm, port);
        GETINT(v, j++) = sock;
    }
    v->nused = j;
}

void close_sockets(varr *v)
{
    int i;

    for (i = 0; i < v->nused; i++) {
        int fd = GETINT(v, i);
#if defined (WIN32)
        closesocket(fd);
#else
        close(fd);
#endif
    }
}
static void add_fds(varr *v, fd_set *in_set, int *maxdesc)
{
    int i;

    for (i = 0; i < v->nused; i++) {
        int fd = GETINT(v, i);
        FD_SET(fd, in_set);
        if (*maxdesc < fd) *maxdesc = fd;
    }
}

static void check_fds(varr *v, fd_set *in_set, void (*new_conn_cb)(int))
{
    int i;

    for (i = 0; i < v->nused; i++) {
        int fd = GETINT(v, i);
        if (FD_ISSET(fd, in_set))
            new_conn_cb(fd);
    }
}

/* END of Zarik NEW funcs */

void boot_db_ports()
{
    char filename[PATH_MAX];
    FILE *fp;

    control_sockets.nused = 0;
    info_sockets.nused = 0;

    snprintf(filename, sizeof(filename), "%s%c%s", ETC_PATH, PATH_SEPARATOR, PORTS_CONF);
    if ((fp = fopen(filename, "r")) == NULL) {
        perror(filename);
        exit(1);
    }
    while ( !feof(fp) )
    {
        char name[80];
        int val;
        int *p;

        fscanf(fp,"%s %d\n",name,&val);
        if ( strcmp(name,"listen") == 0 ) {
            p = varr_enew(&control_sockets);
            *p = val;
        } else if ( strcmp(name,"info") == 0 ) {
            p = varr_enew(&info_sockets);
            *p = val;
        } else {
            log_printf("%s: invalid format.", PORTS_CONF);

        }
    }
    fclose(fp);

}

int main(int argc, char **argv)
{
    struct timeval now_time;
    int ch;
    int check_info;

#if defined WIN32
    WORD    wVersionRequested = MAKEWORD(1, 1);
    WSADATA wsaData;
    int err;
#endif

    /*
     * Memory debugging if needed.
     */
#if defined(MALLOC_DEBUG)
    malloc_debug(2);
#endif

    setlocale(LC_ALL, str_empty);

    /*
     * Init time.
     */
    gettimeofday(&now_time, NULL);
    current_time    = (time_t) now_time.tv_sec;
    strnzcpy(str_boot_time, sizeof(str_boot_time), strtime(current_time));

    /*
     * Reserve one channel for our use.
     */
    if ((fpReserve = fopen(NULL_FILE, "r")) == NULL) {
        perror(NULL_FILE);
        exit(1);
    }

    // init garbage collector
    gc.char_enabled = FALSE;
    gc.room_enabled = FALSE;
    gc.pulse_char_default = PULSE_MOBILE;
    gc.pulse_room_default = PULSE_TRACK;
    gc.pulse_char = gc.pulse_char_default;
    gc.pulse_room = gc.pulse_room_default;

    /*
     * Run the game.
     */

#if defined (WIN32)
    srand((unsigned) time(NULL));
    err = WSAStartup(wVersionRequested, &wsaData);
    if (err) {
        perror("WINSOCK.DLL");
        exit(1);
    }
#else
    resolver_init();
#endif

    boot_db_ports();

    if (argc > 1) {
        /*
         * command line parameters override configuration settings
         */
    /*  control_sockets.nused = 0;
    */  info_sockets.nused = 0;

        opterr = 0;
        while ((ch = getopt(argc, argv, "p:i:")) != -1) {
            int *p;

            switch (ch) {
            case 'p':
                if (!is_number(optarg))
                    usage(argv[0]);
                p = varr_enew(&control_sockets);
                *p = atoi(optarg);
                break;

            case 'i':
                if (!is_number(optarg))
                    usage(argv[0]);
                p = varr_enew(&info_sockets);
                *p = atoi(optarg);
                break;

            default:
                usage(argv[0]);
            }
        }
        argc -= optind;
        argv += optind;
    }

    if (!control_sockets.nused) {
        log_printf("no control sockets defined");
        exit(1);
    }
    check_info = (!!info_sockets.nused);

      crash_install_handlers ();

    setlocale(LC_ALL,str_empty);
    setlocale(LC_NUMERIC, "C");

    boot_db();

    open_sockets(&control_sockets, "ready to rock on port %d");
    open_sockets(&info_sockets, "info service started on port %d");

    if (!control_sockets.nused) {
        log_printf("no control sockets could be opened.");
        exit(1);
    }

    if (check_info && !info_sockets.nused) {
        log_printf("no info service sockets could be opened.");
        exit(1);
    }
#ifdef IMC
    imc_startup(FALSE); // FALSE arg, so the autoconnect setting can govern it.
#endif
#ifdef I3
       /* Initialize and connect to Intermud-3 */
       I3_main(FALSE, 5967 /*XXX*/, FALSE);
#endif


    game_loop_unix();

    close_sockets(&control_sockets);
    close_sockets(&info_sockets);

#ifdef IMC
       imc_shutdown( FALSE );
#endif
#ifdef I3
       I3_shutdown(0);
#endif


#if defined (WIN32)
    WSACleanup();
#else
    resolver_done();
#endif

/* End of NEW code Zarik */

    log_area_popularity();

    /*
     * That's all, folks.
     */
    log("Normal termination of game.");
    return 0;
}

/* stuff for recycling descriptors */
DESCRIPTOR_DATA *descriptor_free;

DESCRIPTOR_DATA *new_descriptor(void)
{
    DESCRIPTOR_DATA *d;

    if (descriptor_free == NULL)
        d = malloc(sizeof(*d));
    else {
        d = descriptor_free;
        descriptor_free = descriptor_free->next;
    }

    memset(d, 0, sizeof(*d));
    return d;
}

void free_descriptor(DESCRIPTOR_DATA *d)
{
    if (!d)
        return;

    free_string(d->host);
    free(d->outbuf);
    d->next = descriptor_free;
    descriptor_free = d;
}

int init_socket(int port)
{
    static struct sockaddr_in sa_zero;
    struct sockaddr_in sa;
    struct linger ld;
    int x = 1;
    int fd;

#if defined (WIN32)
    if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
#else
    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
#endif
        log_printf("init_socket(%d): socket: %s",
               port, strerror(errno));
        return -1;
    }

    if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,
               (char *) &x, sizeof(x)) < 0) {
        log_printf("init_socket(%d): setsockopt: SO_REUSEADDR: %s",
               port, strerror(errno));
#if defined (WIN32)
        closesocket(fd);
#else
        close(fd);
#endif
        return -1;
    }

    ld.l_onoff  = 0;
    ld.l_linger = 1000;

    if (setsockopt(fd, SOL_SOCKET, SO_LINGER,
               (char *) &ld, sizeof(ld)) < 0) {
        log_printf("init_socket(%d): setsockopt: SO_LINGER: %s",
               port, strerror(errno));
#if defined (WIN32)
        closesocket(fd);
#else
        close(fd);
#endif
        return -1;
    }

    sa      = sa_zero;
#if !defined (WIN32)
    sa.sin_family   = AF_INET;
#else
    sa.sin_family   = PF_INET;
#endif
    sa.sin_port = htons(port);

    if (bind(fd, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
        log_printf("init_socket(%d): bind: %s", port, strerror(errno));
#if defined (WIN32)
        closesocket(fd);
#else
        close(fd);
#endif
        return -1;
    }

    if (listen(fd, 3) < 0) {
        log_printf("init_socket(%d): listen: %s",
               port, strerror(errno));
#if defined (WIN32)
        closesocket(fd);
#else
        close(fd);
#endif
        return -1;
    }

    return fd;
}

void game_loop_unix(void)
{
    static struct timeval null_time;
    struct timeval last_time;

    gettimeofday(&last_time, NULL);
    current_time = (time_t) last_time.tv_sec;
    // TODO:    
    //artefact_init_dummy();

    /* Main loop */
    while (!merc_down) {
        fd_set in_set;
        fd_set out_set;
        fd_set exc_set;
        DESCRIPTOR_DATA *d;
        INFO_DESC *id;
        INFO_DESC *id_next;
        int maxdesc;

#if defined(MALLOC_DEBUG)
        if (malloc_verify() != 1)
            abort();
#endif

        /*
         * Poll all active descriptors.
         */
        FD_ZERO(&in_set );
        FD_ZERO(&out_set);
        FD_ZERO(&exc_set);

        maxdesc = 0;
        add_fds(&control_sockets, &in_set, &maxdesc);
        add_fds(&info_sockets, &in_set, &maxdesc);

#if !defined (WIN32)
        FD_SET(fileno(rfin), &in_set);
        maxdesc = UMAX(maxdesc, fileno(rfin));
#endif

        for (d = descriptor_list; d; d = d->next) {
            maxdesc = UMAX(maxdesc, d->descriptor);
            FD_SET(d->descriptor, &in_set );
            FD_SET(d->descriptor, &out_set);
            FD_SET(d->descriptor, &exc_set);
        }

        for (id = id_list; id; id = id->next) {
            maxdesc = UMAX(maxdesc, id->fd);
            FD_SET(id->fd, &in_set);
        }

        if (select(maxdesc+1,
               &in_set, &out_set, &exc_set, &null_time) < 0) {
            log_printf("game_loop: select: %s", strerror(errno));
            exit(1);
        }

#if !defined (WIN32)
        if (FD_ISSET(fileno(rfin), &in_set))
            resolv_done();
#endif

        check_fds(&control_sockets, &in_set, init_descriptor);
        check_fds(&info_sockets, &in_set, info_newconn);

        for (id = id_list; id; id = id_next) {
            id_next = id->next;

            if (FD_ISSET(id->fd, &in_set))
                info_process_cmd(id);
        }

        /*
         * Kick out the freaky folks.
         */
        for (d = descriptor_list; d; d = d_next) {
            d_next = d->next;
            if (FD_ISSET(d->descriptor, &exc_set)) {
                FD_CLR(d->descriptor, &in_set );
                FD_CLR(d->descriptor, &out_set);
                if (d->character && d->character->level > 1)
                    save_char_obj(d->character, FALSE, FALSE);
                d->outtop = 0;
                close_descriptor(d);
            }
        }

        /*
         * Process input.
         */
        for (d = descriptor_list; d != NULL; d = d_next) {
            d_next      = d->next;
            d->fcommand = FALSE;

            if (FD_ISSET(d->descriptor, &in_set)) {
                if (d->character != NULL)
                    d->character->timer = 0;

                if (!read_from_descriptor(d)) {
                    FD_CLR(d->descriptor, &out_set);
                    if (d->character != NULL
                    &&  d->character->level > 1)
                        save_char_obj(d->character,
                                  FALSE, FALSE);
                    d->outtop = 0;
                    close_descriptor(d);
                    continue;
                }
            }

            if (d->character != NULL && d->character->daze > 0) {
                if (--d->character->daze == 0)
                    force_bust_prompt (d->character);
            }

            if (d->character != NULL && d->character->wait > 0) {
                if (--d->character->wait == 0)
                    force_bust_prompt (d->character);

                continue;
            }

            if (!read_from_buffer(d)) {
                FD_CLR(d->descriptor, &out_set);
                if (d->character != NULL
                &&  d->character->level > 1)
                    save_char_obj(d->character,
                              FALSE, FALSE);
                d->outtop = 0;
                close_descriptor(d);
                continue;
            }

            if (d->incomm[0] != '\0') {
                d->fcommand = TRUE;
                stop_idling(d->character);

                if (d->showstr_point)
                    show_string(d, d->incomm);
                else if (d->pString)
                    string_add(d->character, d->incomm);
                else if (d->connected == CON_PLAYING) {
                    if (!run_olc_editor(d))
                            substitute_alias(d, d->incomm);
                }
                else
                    nanny(d, d->incomm);

                if (d->connected != CON_RESOLV)
                    d->incomm[0]    = '\0';
            }
        }

#ifdef IMC
        imc_loop();
#endif
#ifdef I3
        I3_loop();
#endif


        /*
         * Autonomous game motion.
         */
        update_handler();

        /*
         * Output.
         */
        for (d = descriptor_list; d != NULL; d = d_next) {
            d_next = d->next;

//          if ((d->fcommand || d->outtop > 0)
            if ( ( d->fcommand || d->outtop > 0 || d->out_compress )
            &&  FD_ISSET(d->descriptor, &out_set)) {
//              if (!process_output(d, TRUE))
                {
                    bool ok = TRUE;

                    if ( d->fcommand || d->outtop > 0 )
                        ok = process_output( d, TRUE );

                    if (ok && d->out_compress)
                        ok = processCompressed(d);

                    if (!ok)
                    {
                        if (d->character != NULL
                            &&  d->character->level > 1)
                            save_char_obj(d->character, FALSE, TRUE);
                        d->outtop = 0;
                        close_descriptor(d);
                    }
                }
            }
        }

    /*
     * Synchronize to a clock.
     * Sleep(last_time + 1/PULSE_PER_SCD - now).
     * Careful here of signed versus unsigned arithmetic.
     */
#if !defined (WIN32)
     {
        struct timeval now_time;
        long secDelta;
        long usecDelta;

        gettimeofday(&now_time, NULL);
        usecDelta   = ((int) last_time.tv_usec) - ((int) now_time.tv_usec)
            + 1000000 / PULSE_PER_SCD;
        secDelta    = ((int) last_time.tv_sec) - ((int) now_time.tv_sec);
        while (usecDelta < 0) {
        usecDelta += 1000000;
        secDelta  -= 1;
        }

        while (usecDelta >= 1000000) {
        usecDelta -= 1000000;
        secDelta  += 1;
        }

        if (secDelta > 0 || (secDelta == 0 && usecDelta > 0)) {
        struct timeval stall_time;

        stall_time.tv_usec = usecDelta;
        stall_time.tv_sec  = secDelta;
        if (select(0, NULL, NULL, NULL, &stall_time) < 0) {
            log_printf("game_loop: select: stall: %s", strerror(errno));
            exit(1);
        }
        }
    }
#else
    {
        int times_up;
        int nappy_time;
        struct _timeb start_time;
        struct _timeb end_time;
        _ftime( &start_time );
        times_up = 0;

        while( times_up == 0 )
        {
            _ftime( &end_time );
            if ( ( nappy_time = (int) ( 1000 * (double) ( ( end_time.time - start_time.time ) +
                       ( (double) ( end_time.millitm - start_time.millitm ) /
                    1000.0 ) ) ) ) >= (double)( 1000 / PULSE_PER_SCD ) )
              times_up = 1;
        else
        {
            Sleep( (int) ( (double) ( 1000 / PULSE_PER_SECOND ) -
                  (double) nappy_time ) );
            times_up = 1;
        }
      }
    }
#endif
        gettimeofday(&last_time, NULL);
        current_time = (time_t) last_time.tv_sec;
    }
}


static void cp_print(DESCRIPTOR_DATA* d)
{
    char buf[MAX_STRING_LENGTH];
    int i;

    write_to_buffer(d, "\n\r", 0);
    for (i = 0; i < NCODEPAGES; i++) {
        snprintf(buf, sizeof(buf), "%s%d. %s",
             i ? " " : str_empty, i+1, codepages[i].name);
        write_to_buffer(d, buf, 0);
    }
    write_to_buffer(d, "\n\rPlease, choose the codepage (non-russian players should choose translit): ", 0);
}

extern const HELP_DATA *help_greeting;

#if !defined(FNDELAY)
#define FNDELAY O_NDELAY
#endif

void init_descriptor(int control)
{
    char buf[MAX_STRING_LENGTH];
    DESCRIPTOR_DATA *dnew;
    struct sockaddr_in sock;
    int desc;
    int size;

    size = sizeof(sock);
    getsockname(control, (struct sockaddr *) &sock, &size);
    if ((desc = accept(control, (struct sockaddr *) &sock, &size)) < 0) {
        log_printf("init_descriptor: accept: %s", strerror(errno));
        return;
    }

#if !defined (WIN32)
    if (fcntl(desc, F_SETFL, FNDELAY) < 0) {
        log_printf("init_descriptor: fcntl: FNDELAY: %s",
               strerror(errno));
        return;
    }
#endif

    /*
     * Cons a new descriptor.
     */
    dnew = new_descriptor();

    dnew->descriptor    = desc;
    dnew->connected     = CON_GET_CODEPAGE;
    dnew->showstr_head  = NULL;
    dnew->showstr_point = NULL;
    dnew->pString       = NULL;
    dnew->olced         = NULL;
    dnew->pEdit         = NULL;
    dnew->pEdit2        = NULL;
    dnew->outsize       = 2000;
    dnew->outbuf        = malloc(dnew->outsize);
    dnew->wait_for_se   = 0;
    dnew->codepage      = codepages;
    dnew->host      = NULL;

    size = sizeof(sock);
    if (getpeername(desc, (struct sockaddr *) &sock, &size) < 0) {
        log_printf("init_descriptor: getpeername: %s",
               strerror(errno));
        return;
    }
#if defined (WIN32)
    else {
        /* Copying from ROM 2.4b6 */
        int addr;
        struct hostent *from;

        addr = ntohl(sock.sin_addr.s_addr);
        from = gethostbyaddr((char *) &sock.sin_addr,
                     sizeof(sock.sin_addr), AF_INET);
        dnew->host = str_dup(from ? from->h_name : "unknown");
    }
#endif

    log_printf("sock.sinaddr: %s", inet_ntoa(sock.sin_addr));

    dnew->next      = descriptor_list;
    descriptor_list     = dnew;

    write_to_descriptor_2(desc, (char *)telopt_binary, 3);
//  write_to_descriptor_2(desc, (char *)compress_wont, 3);
    /*
     * Send the greeting.
         */
        if (help_greeting) {
            parse_colors(mlstr_mval(help_greeting->text), buf, sizeof(buf),
                         //FORMAT_DUMB);
                         FORMAT_ANSI);
            write_to_buffer(dnew, buf + (buf[0] == '.'), 0);
        }
        else
            bug("No greeting help", 0);
    cp_print(dnew);
}

void close_descriptor(DESCRIPTOR_DATA *dclose)
{
    CHAR_DATA *ch;
    DESCRIPTOR_DATA *d;

    if (dclose->outtop > 0)
        process_output(dclose, FALSE);

    if (dclose->snoop_by != NULL)
        write_to_buffer(dclose->snoop_by,
                "Your victim has left the game.\n\r", 0);

    for (d = descriptor_list; d != NULL; d = d->next)
        if (d->snoop_by == dclose)
            d->snoop_by = NULL;

    if ((ch = dclose->character) != NULL) {
        log_printf("Closing link to %s.", ch->name);
        if (dclose->connected == CON_PLAYING) {
            act("$n has lost $gn{his} link.", ch, NULL, NULL, TO_ROOM);
            wiznet("Net death has claimed $N.", ch, NULL,
                   WIZ_LINKS, 0, ch->level);
            ch->desc = NULL;
        }
        else
                free_char(dclose->character);
    }

    if (d_next == dclose)
        d_next = d_next->next;

    if (dclose == descriptor_list)
        descriptor_list = descriptor_list->next;
    else {
        DESCRIPTOR_DATA *d;

        for (d = descriptor_list; d && d->next != dclose; d = d->next)
            ;
        if (d != NULL)
            d->next = dclose->next;
        else
            bug("Close_socket: dclose not found.", 0);
    }

    if (dclose->out_compress) {
            deflateEnd(dclose->out_compress);
        free_mem(dclose->out_compress_buf, COMPRESS_BUF_SIZE);
        free_mem(dclose->out_compress, sizeof(z_stream));
    }

#if !defined( WIN32 )
    close(dclose->descriptor);
#else
    closesocket(dclose->descriptor);
#endif
    free_descriptor(dclose);
}

bool read_from_descriptor(DESCRIPTOR_DATA *d)
{
    int iOld;
    int iStart;
    unsigned char *p, *q;

    /*
     * Hold horses if pending command already
     */
    if (d->incomm[0] != '\0')
        return TRUE;

    /* Check for overflow. */
    iOld = iStart = strlen(d->inbuf);
    if (iStart >= sizeof(d->inbuf) - 10) {
        log_printf("%s input overflow!", d->host);
        write_to_descriptor(d,
                    "\n\r*** PUT A LID ON IT!!! ***\n\r", 0);
        return FALSE;
    }

    for (; ;) {
        int nRead;

#if !defined (WIN32)
    nRead = read( d->descriptor, d->inbuf + iStart,
             sizeof( d->inbuf ) - 10 - iStart );
#else
    nRead = recv( d->descriptor, d->inbuf + iStart,
             sizeof( d->inbuf ) - 10 - iStart, 0 );
#endif
        if (nRead > 0) {
            iStart += nRead;
            if (d->inbuf[iStart-1] == '\n'
            ||  d->inbuf[iStart-1] == '\r')
                break;
        }
        else if (nRead == 0) {
            log("EOF encountered on read.");
            return FALSE;
            break;
        }
#if !defined (WIN32)
        else if (errno == EWOULDBLOCK)
            break;
#else
        else if ( WSAGetLastError() == WSAEWOULDBLOCK)
        break;
#endif
        else {
            log_printf("read_from_descriptor: %s", strerror(errno));
            return FALSE;
        }
    }

    // -------------------------------------------------------------------
    // telnet parser stuff by Illinar
    // -------------------------------------------------------------------

    // return if no pending input
    d->inbuf[iStart] = '\0' ;
    if (iOld == iStart) return TRUE ;

    // loop over new data to see if we have telnet options there
    for (p = d->inbuf + iOld ; *p ;)
    {
        unsigned char * r ;
        bool ok = FALSE ;

        // IAC in the input can be viewed either as the intention to
        // initiate negotiation or as the plain data, in the latter
        // case it must be sent doubled according to the standard, however
        // some clients seem to send it in a single way, for those morons
        // we have the following strategy, if notelnet option is toggled
        // on then no parsing is done and data is sent as is, otherwise
        // we follow standard procedure outlined in RFC854

        // although we always want to see if to enable/disable compression
        // so check this independently on the settings of notelnet/noiac

        if (*p != IAC) { p++ ; continue ; } // skip plain characters

        // at this point we have *p = IAC
        // always check for compression

        // note! we are hoping that the whole command (3 bytes) is sent,
        // that is true for the most cases

        // negotiations
        if (d->wait_compression)
        {
            // check for version2 first
            if (!memcmp (p, compress2_do, strlen(compress2_do))) {
                // log ("starting compression2") ;
                compressStart (d, TRUE) ;
                d->wait_compression = FALSE ; ok = TRUE ;
            }
            else
            if (!memcmp (p, compress_do, strlen(compress_do))) {
                // log ("starting compression") ;
                compressStart (d, FALSE) ;
                d->wait_compression = FALSE ; ok = TRUE ;
            }
            else
            if (!memcmp (p, compress2_dont, strlen(compress2_dont))) {
                // log ("compression2 refused, offering compression1") ;

                write_to_descriptor_2 (d->descriptor,
                                       (char *) compress_will, 3) ;
                d->wait_compression = TRUE ; ok = TRUE ;
            }
            else
            if (!memcmp (p, compress_dont,  strlen(compress_dont))) {
                // log ("ending compression") ;
                compressEnd (d) ;
                d->wait_compression = FALSE ; ok = TRUE ;
            }
        }

        // see if we want parser to be turned off
        if (d->connected == CON_PLAYING && d->character &&
            IS_SET (d->character->comm, COMM_NOTELNET) && !ok)
        {
            p++ ; continue ;
        }

        // skip normal parsing of DO/DONT/WONT/WILL if inside subnegotiation
        // procedure
        if (d->wait_for_se) goto wse ;

        // at this point we want to parse the command
        // p[1] contains command code and p[2] contains option code
        switch (p[1])
        {
        case DONT: case DO: case WONT: case WILL:
            q = p + 3 ; // skip the command
            break ;

        wse:
        case SB: // subnegotiations
            // look for the end of the command
            if ((q = strchr (p, SE)) == NULL)
            {
                q = strchr (p, '\0')  ;
                d->wait_for_se = TRUE ;
            }
            else
            {
                q++ ;
                d->wait_for_se = FALSE ;
            }
            break ;

        case IAC: // double IAC -- no negotiations --> convert to single
            memmove (p, p + 1, strlen(p)) ; p++ ;
            continue ;

        default: // skip other stuff
            q = p + 2 ;
            break ;
        }

        // remove the control codes from the input
        if ((r = strchr (p, '\0')) < q) q = r ;
        memmove (p, q, strlen (q) + 1) ;
    }

    return TRUE ;
}

/*
 * Transfer one line from input buffer to input line.
 */
bool read_from_buffer(DESCRIPTOR_DATA *d)
{
    int i, j, k;

    /*
     * Hold horses if pending command already.
     */
    if (d->incomm[0] != '\0')
        return TRUE;

    /*
     * Look for at least one new line.
     */
    for (i = 0; d->inbuf[i] != '\n' && d->inbuf[i] != '\r'; i++)
        if (d->inbuf[i] == '\0')
            return TRUE;

    /*
     * Canonical input processing.
     */
    for (i = 0, k = 0; d->inbuf[i] != '\n' && d->inbuf[i] != '\r'; i++) {
        if (k >= MAX_INPUT_LENGTH - 2) {
            write_to_descriptor(d,
                        "Line too long.\n\r", 0);

            /* skip the rest of the line */
            for (; d->inbuf[i] != '\0'; i++)
                if (d->inbuf[i] == '\n' || d->inbuf[i] == '\r')
                    break;

            d->inbuf[i]   = '\n';
            d->inbuf[i+1] = '\0';
            break;
        }

        if (d->inbuf[i] == '\b' && k > 0)
            --k;
/*      else if (d->inbuf[i] == (signed char)IAC) {
                if (!memcmp(&d->inbuf[i], compress_do, strlen(compress_do))) {
                i += strlen(compress_do) - 1;
                compressStart(d);
            }
            else if (!memcmp(&d->inbuf[i], compress_dont, strlen(compress_dont))) {
                i += strlen(compress_dont) - 1;
                compressEnd(d);
            }
        } */
        else if ((unsigned)d->inbuf[i] >= ' ')
            d->incomm[k++] =
                d->codepage->from[(unsigned char) d->inbuf[i]];
    }

    /*
     * Finish off the line.
     */
    if (k == 0)
        d->incomm[k++] = ' ';
    d->incomm[k] = '\0';

    /*
     * Deal with bozos with #repeat 1000 ...
     */
    if (k > 1 || d->incomm[0] == '!') {
        if (d->incomm[0] != '!' && strcmp(d->incomm, d->inlast))
            d->repeat = 0;
        else {
            CHAR_DATA *ch = d->original ? d->original :
                              d->character;
            if (ch && ++d->repeat >= 100) {

                log_printf("%s input spamming!", d->host);

                wiznet_printf(d->character,NULL,WIZ_SPAM,0,d->character->level, "SPAM SPAM SPAM %s spamming, and OUT!",d->character->name);
                    wiznet_printf(d->character,NULL,WIZ_SPAM,0,d->character->level, "[%s]'s  Inlast:[%s] Incomm:[%s]!", d->character->name,d->inlast,d->incomm);

                write_to_descriptor(d, "\n\r*** PUT A LID ON IT!!! ***\n\r", 0);
                d->repeat = 0;
                if (d->showstr_point) {
                    if (d->showstr_head) {
                        free_string(d->showstr_head);
                        d->showstr_head = NULL;
                    }
                    d->showstr_point = NULL;
                }
                if (d->pString) {
                    free_string(*d->pString);
                    d->pString = NULL;
                }
                return FALSE;
            }
        }
    }

    /*
     * Do '!' substitution.
     */
    if (d->incomm[0] == '!')
        strnzcpy(d->incomm, sizeof(d->incomm), d->inlast);
    else
        strnzcpy(d->inlast, sizeof(d->inlast), d->incomm);

    /*
     * Shift the input buffer.
     */
    while (d->inbuf[i] == '\n' || d->inbuf[i] == '\r')
        i++;
    for (j = 0; (d->inbuf[j] = d->inbuf[i+j]) != '\0'; j++)
        ;
    return TRUE;
}

/*
 * Low level output function.
 */
void battle_prompt(CHAR_DATA *ch, CHAR_DATA *victim)
{
    int percent;
    char* msg;
    char buf[MAX_STRING_LENGTH];

        if (victim->max_hit > 0)
        percent = victim->hit * 100 / victim->max_hit;
        else
        percent = -1;

        if (percent >= 100)
        msg = "{C  ";
        else if (percent >= 90)
        msg = "{b  ";
        else if (percent >= 75)
        msg = "{B    ";
        else if (percent >= 50)
        msg = "{G   ";
        else if (percent >= 30)
        msg = "{Y ";
        else if (percent >= 15)
        msg = "{M  ";
        else if (percent >= 0)
        msg = "{R   ";
        else
        msg = "{R ";


    if (IS_SET(ch->plr_flags,PLR_AUTODAMAGE)) {
        snprintf(buf, sizeof(buf), "%s %s [%d]{x.\n",
             PERS(victim, ch), GETMSG(msg, ch->lang), victim->hit);
    }
    else {
        snprintf(buf, sizeof(buf), "%s %s{x.\n",
             PERS(victim, ch), GETMSG(msg, ch->lang));
    }
    buf[0] = UPPER(buf[0]);
    send_to_char(buf, ch);
}

void snoop_decode(DESCRIPTOR_DATA *d, struct codepage *cp, const char *txt,
                  uint length)
{
        int i;
        unsigned char *outbuf;

        if (length <= 0)
                length = strlen(txt);

        outbuf = malloc(length);

        for (i=0; i < length; i++) {
                outbuf[i] = cp->from[(unsigned char) *txt++];
        if ((unsigned char) *(txt-1) == IAC
        &&  (unsigned char) *txt   == IAC) {
            txt++;
            length--;
        }
    }

        write_to_buffer(d, outbuf, length);

        free(outbuf);

}

/*
 * Some specials added by KIO
 */
bool process_output(DESCRIPTOR_DATA *d, bool fPrompt)
{
    extern bool merc_down;

    /*
     * Bust a prompt.
     */
    if (!merc_down) {
        if (d->showstr_point)
            write_to_buffer(d, "[ ENTER  ]\n\r", 0);
        else if (fPrompt && d->pString && d->connected == CON_PLAYING)
            write_to_buffer(d, "  > ", 0);
        else if (fPrompt && d->connected == CON_PLAYING) {
            CHAR_DATA *ch;
            CHAR_DATA *victim;

            ch = d->character;

            /* battle prompt */
            if (ch != NULL
        && (victim = ch->fighting) != NULL
        && can_see(ch,victim))
        {
                battle_prompt(ch, victim);
        }

            ch = d->original ? d->original : d->character;

        if (ch)
        {
        if (!IS_SET(ch->comm, COMM_COMPACT))
            write_to_buffer(d, "\n\r", 2);

        if (IS_SET(ch->comm, COMM_PROMPT))
            bust_a_prompt(d->character);

        if (IS_SET(ch->comm,COMM_TELNET_GA))
            write_to_descriptor(d, (char *)go_ahead_str, 0);
        }
        }
    }

    /*
     * Short-circuit if nothing to write.
     */
    if (d->outtop == 0)
        return TRUE;

    /*
     * Snoop-o-rama.
     */
    if (d->snoop_by != NULL) {
        if (d->character != NULL)
            write_to_buffer(d->snoop_by, d->character->name, 0);
        write_to_buffer(d->snoop_by, "> ", 2);
                snoop_decode(d->snoop_by, d->codepage, d->outbuf, d->outtop);
//      write_to_buffer(d->snoop_by, d->outbuf, d->outtop);
    }

    /*
     * OS-dependent output.
     */
    if (!write_to_descriptor(d, d->outbuf, d->outtop)) {
        d->outtop = 0;
        return FALSE;
    }
    else {
        d->outtop = 0;
        return TRUE;
    }
}

void percent_hp(CHAR_DATA *ch, char buf[MAX_STRING_LENGTH])
{
    if (ch->hit >= 0)
        snprintf(buf, sizeof(buf), "%d%%",
             ((100 * ch->hit) / UMAX(1, ch->max_hit)));
    else
        strnzcpy(buf, sizeof(buf), "BAD!");
}

/*
 * Bust a prompt (player settable prompt)
 * coded by Morgenes for Aldara Mud
 * bust
 */
void bust_a_prompt(CHAR_DATA *ch)
{
    char                buf[MAX_STRING_LENGTH];
    char                buf2[MAX_STRING_LENGTH];
    char                buf3[MAX_STRING_LENGTH];
    const char         *str;
    const char         *i = NULL;
    char               *point;
    char                wound[20];
    int                 percent;
    float               expl_percent ;
    CHAR_DATA          *victim;
    CHAR_DATA          *gch;
    CHAR_DATA          *tank;
    EXIT_DATA          *pexit;
    bool                found;
    const char         *dir_name[] = {"N","E","S","W","U","D"};
    int                 door;
    CHAR_EXPLORED_DATA *curr_area ;
    ROOM_INDEX_DATA    *edit_room;
    MOB_INDEX_DATA     *edit_mob;
    OBJ_INDEX_DATA     *edit_obj;
    MPCODE             *edit_code;
    int                 edit_vnum;

    if (IS_SET(ch->comm, COMM_AFK))
    {
        act_puts("{c<AFK>{x $t", ch, ch->prefix, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    edit_vnum = -1;

    point = buf;
    str = ch->prompt;
    if (IS_NULLSTR(str))
        str = DEFAULT_PROMPT;

    while(*str != '\0')
    {
        if(*str != '%')
        {
            *point++ = *str++;
            continue;
        }

        switch(*++str)
        {
        default:
            i = str_empty;
            break;

        case 'A':
            if (!IS_NPC(ch))
            {
                if (IS_SET(ch->plr_flags, PLR_GHOST))
                    i = "@";
                else if (IS_PUMPED(ch))
                    i = "$";
                else
                    i = str_empty;
            } else
                i = str_empty;
            break;

        case 't':
            snprintf(buf2, sizeof(buf2),
                 "%d%s", (time_info.hour % 12 == 0) ?
                 12 : time_info.hour % 12,
                 time_info.hour >= 12 ? "pm" : "am");
            i = buf2;
            break;

        case 'T':
            snprintf(buf2,sizeof(buf2),"%d", time_info.hour);
            i = buf2;
            break;

        case 'e':
            found = FALSE;
            buf2[0] = '\0';

            if (IS_AFFECTED(ch, AFF_SLEEP) && !IS_IMMORTAL(ch))
            {
                strnzcat(buf2, sizeof(buf2), " ");
                i = buf2;
                break;
            }

            for (door = 0; door < 6; door++)
                if ((pexit = ch->in_room->exit[door]) != NULL
                && pexit ->to_room.r != NULL
                && can_see_room(ch, pexit->to_room.r)
                && check_blind_raw(ch)
                && !IS_SET(pexit->exit_info, EX_INVIS)
                && (!IS_SET(pexit->exit_info, EX_CLOSED)
                || IS_IMMORTAL(ch)))
                {
                    found = TRUE;
                    if (IS_SET(pexit->exit_info, EX_CLOSED))
                        strnzcat(buf2, sizeof(buf2), "[");
                    strnzcat(buf2, sizeof(buf2), dir_name[door]);
                    if (IS_SET(pexit->exit_info, EX_CLOSED))
                        strnzcat(buf2, sizeof(buf2), "]");
                }
            if (buf2[0])
                strnzcat(buf2, sizeof(buf2), " ");
            i = buf2;
            break;

        case 'E':

            for (curr_area = ch->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
            {
                if (!str_cmp (ch->in_room->area->name, curr_area->name)
                && ch->in_room->area->explorable_rooms > 0)
                {
                    expl_percent = ((float) curr_area->explored_room_count * 100) /
                        (float) ch->in_room->area->explorable_rooms ;
                    snprintf(buf2, sizeof(buf2), "%3.2f", expl_percent);
                    i = buf2;
                    break;
                }
            }
            i = str_empty;
            break;

        case 'i':
            buf2[0] = '\0';

            if (IS_AFFECTED (ch, AFF_INVIS))
                strnzcat(buf2, sizeof(buf2), "{yI");
            if (IS_AFFECTED (ch, AFF_HIDE))
                strnzcat(buf2, sizeof(buf2), "{DH");
            if (IS_AFFECTED (ch, AFF_TRAP))
                strnzcat(buf2, sizeof(buf2), "{DT");
            if (IS_AFFECTED (ch, AFF_IMP_INVIS))
                strnzcat(buf2, sizeof(buf2), "{bI");
            if (IS_AFFECTED (ch, AFF_CAMOUFLAGE))
                strnzcat(buf2, sizeof(buf2), "{yC");
            if (IS_AFFECTED (ch, AFF_FADE))
                strnzcat(buf2, sizeof(buf2), "{gF");
            if (IS_AFFECTED (ch, AFF_EARTHFADE))
                strnzcat(buf2, sizeof(buf2), "{rE");

            strnzcat(buf2, sizeof(buf2), "{x");
            i = buf2;
            break;

        case 'n':
            i = ch->name;
            break;

        case 'N':
            if ((victim = ch->fighting) != NULL)
            {
                if (can_see(ch, victim))
                {
//                    i = victim->name;
                    buf2[0]='\0';
                    buf3[0]='\0';
                    snprintf(buf2, sizeof(buf2), "%s", fmt_color_str(victim->name, 6));
                    strlcat(buf3, buf2, sizeof(buf3));
                    i = buf3;
                }
                else
                    i = "???";
            }
            else
                i = "-";
            break;

        case 'S':
            i = flag_string(sex_table, ch->sex);
            break;

        case 'y':
            percent_hp(ch, buf2);
            i = buf2;
            break;

        case 'o':
            if ((victim = ch->fighting) != NULL)
            {
                if (can_see(ch, victim))
                {
                    percent_hp(victim, buf2);
                    i = buf2;
                }
                else
                    i = "???";
            }
            else
                i = "-";
            break;

        case 'O':
            if ((victim = ch->fighting) != NULL)
            {
                if (can_see(ch, victim))
                {
                    snprintf(buf2, sizeof(buf2), "%d", victim->hit);
                    i = buf2;
                }
                else
                    i = "???";
            } else
                i = "None";
            break;

        case 'P':
            if (ch->prefix[0] != '\0')
            {
                snprintf(buf2, sizeof(buf2), "%s ", ch->prefix);
                i = buf2;
            } else
                i = str_empty;
            break;

        case 'h':
            snprintf(buf2, sizeof(buf2), "%d", ch->hit);
            i = buf2;
            break;

        case 'H':
            snprintf(buf2, sizeof(buf2), "%d", ch->max_hit);
            i = buf2;
            break;

        case 'm':
            snprintf(buf2, sizeof(buf2), "%d", ch->mana);
            i = buf2;
            break;

        case 'M':
            snprintf(buf2, sizeof(buf2), "%d", ch->max_mana);
            i = buf2;
            break;

        case 'v':
            snprintf(buf2, sizeof(buf2), "%d", ch->move);
            i = buf2;
            break;

        case 'V':
            snprintf(buf2, sizeof(buf2), "%d", ch->max_move);
            i = buf2;
            break;

        case 'x':
            snprintf(buf2, sizeof(buf2), "%d", ch->exp);
            i = buf2;
            break;

        case 'X':
            if (!IS_NPC(ch))
            {
                snprintf(buf2, sizeof(buf2), "%d", exp_to_level(ch));
                i = buf2;
            }
            else
                i = str_empty;
            break;

        case 'g':
            snprintf(buf2, sizeof(buf2), "%d", ch->gold);
            i = buf2;
            break;

        case 's':
            snprintf(buf2, sizeof(buf2), "%d", ch->silver);
            i = buf2;
            break;

        case 'a':
            snprintf(buf2, sizeof(buf2), "%s [%d]", IS_GOOD(ch) ? "good" : IS_EVIL(ch) ? "evil" : "neutral", ch->alignment);
            i = buf2;
            break;

        case 'r':
            if (ch->in_room != NULL)
            {
                if (IS_AFFECTED(ch, AFF_SLEEP) && !IS_IMMORTAL(ch))
                {
                    i = "darkness";
                } else
                {
                    i = (check_blind_raw(ch) && !room_is_dark(ch)) ?
                         mlstr_cval(ch->in_room->name, ch) :
                         "darkness";
                }
            } else
                i = str_empty;
            break;

    case 'R':
        if((IS_IMMORTAL(ch) || IS_BUILDER(ch, ch->in_room->area))
        && ch->in_room != NULL)
        {
            snprintf(buf2, sizeof(buf2), "%d", ch->in_room->vnum);
            i = buf2;
        }
        else
            i = str_empty;
        break;

        case 'z':
            if (IS_IMMORTAL(ch) && ch->in_room != NULL)
                i = ch->in_room->area->name;
            else
                i = str_empty;
            break;

        case '%':
            i = "%%";
            break;

        case 'B':
            i = OLCED(ch) ? OLCED(ch)->name : str_empty;
            if (!IS_NULLSTR(i))
            {   if (!str_cmp(i, "RoomEd"))
                {
                    edit_room = ch->desc->pEdit;
                    if( edit_room != NULL)
                       edit_vnum = edit_room->vnum;
                }

                if (!str_cmp(i, "MobEd"))
                {
                    edit_mob = ch->desc->pEdit;
                    if( edit_mob != NULL)
                       edit_vnum = edit_mob->vnum;
                }

                if (!str_cmp(i, "ObjEd"))
                {
                    edit_obj = ch->desc->pEdit;
                    if( edit_obj != NULL)
                       edit_vnum = edit_obj->vnum;
                }

                if (!str_cmp(i, "MPEd") || !str_cmp(i, "OPEd") || !str_cmp(i, "RPEd") )
                {
                    edit_code = ch->desc->pEdit;
                    if( edit_code != NULL)
                       edit_vnum = edit_code->vnum;
                }

                if (edit_vnum > -1)
                   snprintf(buf2, sizeof(buf2), "%s %d", i, edit_vnum);
                else
                   snprintf(buf2, sizeof(buf2), "%s ", i);

                i = buf2;
            }
            break;

        case 'W':
            if (ch->invis_level)
            {
                snprintf(buf2, sizeof(buf2), "%d", ch->invis_level);
                i = buf2;
            }
            else
                i = str_empty;
            break;

        case 'I':
            if (ch->incog_level)
            {
                snprintf(buf2, sizeof(buf2), "%d", ch->incog_level);
                i = buf2;
            }
            else
                i = str_empty;
            break;

        case 'd':
            if (ch->daze > 0)
            {
//                i = "D";
                snprintf(buf2, sizeof(buf2), "%dD", ch->daze);
                i = buf2;
            }
            else
                i = str_empty;
            break;

        case 'D':
            if (IS_AFFECTED(ch, AFF_BLIND))
            {
                i = "???";
                break;
            }

            buf2[0]='\0';
            buf3[0]='\0';
            for (gch = char_list; gch != NULL; gch = gch->next)
                if (is_same_group(gch, ch) && (gch != ch))
                {
                    snprintf(buf2, sizeof(buf2), "%s%d", fmt_color_str(gch->name, 3), gch->hit);
                    strlcat(buf3, buf2, sizeof(buf3));
                }
            i = buf3;
            break;

        case 'w':
            if (ch->wait > 0)
            {
//                i = "W";
                snprintf(buf2, sizeof(buf2), "%dW", ch->wait);
                i = buf2;
            }
            else
                i = str_empty;
            break;

        case 'q':
            if (!IS_NPC(ch))
            {
                if (ch->pcdata->questtime >= 0)
                {
                    snprintf(buf2, sizeof(buf2), "%d", ch->pcdata->questtime);
                    i = buf2;
                }
            else
                i = str_empty;
            }
            break;

        case 'Q':
            if (!IS_NPC(ch))
            {
                if (ch->pcdata->questtime < 0)
                {
                    snprintf(buf2, sizeof(buf2), "%d", -ch->pcdata->questtime);
                    i = buf2;
                }
                else
                i = str_empty;
            }
            break;

        case 'G':

            i = "U";

            if (ch->in_room->sector_type == SECT_FIELD)
                i = "P";
            else if (ch->in_room->sector_type == SECT_DESERT)
                i = "D";
            else if (ch->in_room->sector_type == SECT_INSIDE)
                i = "I";
            else if (ch->in_room->sector_type == SECT_CITY)
                i = "C";
            else if (ch->in_room->sector_type == SECT_FOREST)
                i = "F";
            else if (ch->in_room->sector_type == SECT_HILLS)
                i = "H";
            else if (ch->in_room->sector_type == SECT_MOUNTAIN)
                i = "M";
            else if (ch->in_room->sector_type == SECT_WATER_SWIM)
                i = "W";
            else if (ch->in_room->sector_type == SECT_WATER_NOSWIM)
                i = "W";
            else if (ch->in_room->sector_type == SECT_AIR)
                i = "A";
            else if (ch->in_room->sector_type == SECT_UNUSED)
                i = "U";
            else if (ch->in_room->sector_type == SECT_MAX)
                i = "U";

            break;

        case 'C': //climat

            i = "C";

            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_ARCTIC))
                i = "A";
            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_DESSERT))
                i = "D";
            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_TROPIC))
                i = "T";
            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_SEA))
                i = "S";
            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_MOUNTAIN))
                i = "M";
            if (IS_SET(ch->in_room->area->climat_type, CLIMAT_HIGHMOUNTAIN))
                i = "H";

            break;

        case 'c': //weather

            if  (IS_SET(ch->in_room->room_flags, ROOM_INDOORS) || IS_SET(ch->in_room->area->flags, AREA_NOWEATHER))
            {
                i = "U";
                break;
            }
            i = "N";

            if (ch->in_room->area->weather.weather_status >= WEATHER_FINE)
                i = "F";
            if (ch->in_room->area->weather.weather_status >= WEATHER_BAD)
                i = "B";
            if (ch->in_room->area->weather.weather_status >= WEATHER_RAIN)
                i = "R";
            if (ch->in_room->area->weather.weather_status >= WEATHER_DISGUSTING)
                i = "D";
            if (ch->in_room->area->weather.weather_status >= WEATHER_CATACLYSM)
                i = "C";

            break;

        case 'p': //planets

            i= flag_string(planet_flags, ch->in_room->area->planet);

            break;
        case 'l':
            if ((victim = ch->fighting) != NULL)
            {
                if (can_see(ch, victim))
                {
                    if (victim->max_hit > 0)
                        percent = victim->hit * 100 / victim->max_hit;
                    else
                        percent = -1;
                    if (percent > 80)
                        snprintf(wound, sizeof(wound),"{G>>{Y-{R--");
                    else if (percent >= 65)
                        snprintf(wound, sizeof(wound),"{G >{Y-{R--");
                    else if (percent >= 40)
                        snprintf(wound, sizeof(wound),"  {Y-{R--");
                    else if (percent >= 15)
                        snprintf(wound, sizeof(wound),"   {R--");
                    else
                        snprintf(wound, sizeof(wound),"    {R-");
                    snprintf(buf2, sizeof(buf2)," %s",wound);
                    i = buf2;
                } else
                    i = "{G ??{Y?{R??";
            }
            else
                i = "{G >>{Y-{R--";
        break;

        case 'L':
            if (((victim = ch->fighting) != NULL)
            && ((tank = victim->fighting) != NULL))
            {
                if (tank != ch)
                {
                    percent_hp(tank, buf2);
                    i = buf2;
                }
                else i = str_empty;
            }
            else
                i = str_empty;
            break;
        }

        ++str;
        while((*point = *i) != '\0')
        {
            ++point;
            ++i;
        }
    }

    *point = '\0';
    send_to_char(buf, ch);

    if (ch->prefix[0] != '\0')
        write_to_buffer(ch->desc, ch->prefix, 0);
}

/*
 * Append onto an output buffer.
 */
void write_to_buffer(DESCRIPTOR_DATA *d, const char *txt, uint length)
{
    uint size;
    int i;
    bool noiac = (d->connected == CON_PLAYING &&
              d->character != NULL &&
              IS_SET(d->character->comm, COMM_NOIAC));

    /*
     * Find length in case caller didn't.
     */
    if (length <= 0)
        length = strlen(txt);

    /*
     * Adjust size in case of IACs (they will be doubled)
     */
    size = length;
    if (!noiac)
        for (i = 0; i < length; i++)
            if (d->codepage->to[(unsigned char) txt[i]] == IAC)
                size++;

    /*
     * Initial \n\r if needed.
     */
    if (d->outtop == 0 && !d->fcommand) {
        d->outbuf[0]    = '\n';
        d->outbuf[1]    = '\r';
        d->outtop   = 2;
    }

    /*
     * Expand the buffer as needed.
     */
    while (d->outtop + size >= d->outsize) {
        char *outbuf;

        if (d->outsize >= 32000) {
            bug("Buffer overflow. Closing.\n\r",0);
            close_descriptor(d);
            return;
        }
        outbuf = malloc(2 * d->outsize);
        strncpy(outbuf, d->outbuf, d->outtop);
        free(d->outbuf);
        d->outbuf = outbuf;
        d->outsize *= 2;
    }

    /*
     * Copy.
     */
    while (length--) {
        unsigned char c;

        c = d->codepage->to[(unsigned char) *txt++];
        d->outbuf[d->outtop] = c;
        if (c == IAC)  {
            if (noiac)
                d->outbuf[d->outtop] = IAC_REPL;
            else
                d->outbuf[++d->outtop] = IAC;
        }
        d->outtop++;
    }
    return;
}

/*
 * Lowest level output function.
 * Write a block of text to the file descriptor.
 * If this gives errors on very long blocks (like 'ofind all'),
 *   try lowering the max block size.
 */
bool write_to_descriptor_2(int desc, char *txt, int length)
{
    uint iStart;
    uint nWrite;
    uint nBlock;

    if (!length)
        length = strlen(txt);

    for (iStart = 0; iStart < length; iStart += nWrite) {
        nBlock = UMIN(length - iStart, 4096);
#if !defined( WIN32 )
        if ((nWrite = write(desc, txt + iStart, nBlock)) < 0) {
#else
        if ((nWrite = send(desc, txt + iStart, nBlock, 0)) < 0) {
#endif
            log_printf("write_to_descriptor: %s", strerror(errno));
            return FALSE;
        }
    }
    return TRUE;
}

bool write_to_descriptor(DESCRIPTOR_DATA *d, char *txt, int length)
{
    if (d->out_compress)
        return writeCompressed(d, txt, length);
    else
        return write_to_descriptor_2(d->descriptor, txt, length);
}

int search_sockets(DESCRIPTOR_DATA *inp)
{
    DESCRIPTOR_DATA *d;

    if (IS_IMMORTAL(inp->character))
        return 0;

    for(d = descriptor_list; d; d = d->next) {
        if(!strcmp(inp->host, d->host)) {
            if (d->character && inp->character
            &&  !strcmp(inp->character->name, d->character->name))
                continue;
            return 1;
        }
    }
    return 0;
}

bool align_restrict(CHAR_DATA *ch);
bool align_ok(CHAR_DATA *ch, int align);
bool ethos_restrict(CHAR_DATA *ch);
bool ethos_ok(CHAR_DATA *ch, int ethos);

void advance(CHAR_DATA *victim, int level);

static void print_hometown(CHAR_DATA *ch)
{
    race_t *r;
    class_t *cl;
    int htn;

    if ((r = race_lookup(ORG_RACE(ch))) == NULL
    ||  !r->pcdata
    ||  (cl = class_lookup(ch->class)) == NULL) {
        char_act("You should create your character anew.", ch);
        close_descriptor(ch->desc);
        return;
    }

    if ((htn = hometown_permanent(ch)) >= 0) {
        ch->hometown = htn;
        char_act(str_empty, ch);
        act_puts("Your hometown is $t, permanently.", ch, hometown_name(htn), NULL, TO_CHAR, POS_DEAD);
        char_act("[Hit Return to continue]", ch);


/* XXX */
        ch->endur = 100;
        ch->desc->connected = CON_GET_ETHOS;
        return;
    }

    char_act(str_empty, ch);
    do_help(ch, "HOMETOWN");
    hometown_print_avail(ch);
    char_puts("? ", ch);
    ch->desc->connected = CON_PICK_HOMETOWN;
}

/*
 * Deal with sockets that haven't logged in yet.
 */
void nanny(DESCRIPTOR_DATA *d, const char *argument)
{
    DESCRIPTOR_DATA *d_old, *d_next;
    char buf[MAX_STRING_LENGTH];
    char buf1[MAX_STRING_LENGTH];
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *ch;
    char *pwdnew;
    const char *pwd_crypted;
    int iClass, race, align, i;
    int obj_count;
    int obj_count2;
    OBJ_DATA *obj;
// OBJ_DATA *inobj;
    int nextquest = 0;
    struct sockaddr_in sock;
    int size;
    race_t *r;

    while (isspace(*argument))
        argument++;

    ch = d->character;

    switch (d->connected) {
    default:
        bug("Nanny: bad d->connected %d.", d->connected);
        close_descriptor(d);
        return;

    case CON_GET_CODEPAGE: {
        int num;

        if (argument[0] == '\0') {
            close_descriptor(d);
            return;
        }

        if (argument[1] != '\0'
        || (num = argument[0] - '1') < 0
        || num >= NCODEPAGES) {
            cp_print(d);
            break;
        }

        d->codepage = codepages+num;
        log_printf("'%s' codepage selected", d->codepage->name);
        d->connected = CON_GET_NAME;
        write_to_buffer(d, "     ? ", 0);
        break;
    }

    case CON_GET_NAME:
        if (argument[0] == '\0') {
            close_descriptor(d);
            return;
        }

        if (!check_parse_name(argument)) {
            write_to_buffer(d, " !  .\n\r"
                       "[HINT]      .\n\r", 0);
            write_to_buffer(d, ": ",0);
            return;
        }

        load_char_obj(d, argument, TRUE);
        ch = d->character;

        if (d->host == NULL) {
            size = sizeof(sock);
            if (getpeername(d->descriptor,
                    (struct sockaddr *) &sock, &size) < 0)
                d->host = str_dup("(unknown)");
            else {
#if defined (WIN32)
                printf("%s@%s\n", ch->name, inet_ntoa(sock.sin_addr));
#else
                fprintf(rfout, "%s@%s\n",
                    ch->name, inet_ntoa(sock.sin_addr));
#endif
                d->connected = CON_RESOLV;
/* wait until sock.sin_addr gets resolved */
                break;
            }
        }

        /* FALLTHRU */

    case CON_RESOLV:
        if (d->host == NULL)
            break;

    /*
     * Swiftest: I added the following to ban sites. I don't
     * endorse banning of sites, but Copper has few descriptors now
     * and some people from certain sites keep abusing access by
     * using automated 'autodialers' and leaving connections hanging.
     *
     * Furey: added suffix check by request of Nickel of HiddenWorlds.
     */

    if (check_ban(d->host,BAN_ALL))
    {
        log_printf("BAN: Try connection from banned site: %s", d->host);
        write_to_buffer(d, "   .\n\r", 0);
        close_descriptor(d);
        return;
    }

    if (!IS_IMMORTAL(ch))
    {
        if (check_ban(d->host,BAN_PLAYER))
        {
            log_printf("BAN: Try connection player from banned site: %s", d->host);
            write_to_buffer(d,"   .\n\r",0);
            close_descriptor(d);
            return;
        }

#undef NO_PLAYING_TWICE
#ifdef NO_PLAYING_TWICE
        if(search_sockets(d))
        {
            write_to_buffer(d, "    ...\n\r", 0);
            close_descriptor(d);
            return;
        }
#endif
        if (iNumPlayers > MAX_OLDIES
        && !IS_SET(ch->plr_flags, PLR_NEW))
        {
            snprintf(buf, sizeof(buf),
            "\n\rThere are currently %i players mudding out of a maximum of %i.\n\rPlease try again soon.\n\r",iNumPlayers - 1, MAX_OLDIES);
             write_to_buffer(d, buf, 0);
             close_descriptor(d);
             return;
        }
        if (iNumPlayers > MAX_NEWBIES && IS_SET(ch->plr_flags, PLR_NEW))
        {
            snprintf(buf, sizeof(buf),
            "\n\r     %i .    \n\r.  .\n\r",
            iNumPlayers - 1);
//             iNumPlayers - 1, MAX_NEWBIES);
            write_to_buffer(d, buf, 0);
            close_descriptor(d);
            return;
         }
    }

    if (IS_SET(ch->plr_flags, PLR_DENY))
    {
        log_printf("Denying access to %s@%s.", argument, d->host);
        write_to_buffer(d, "You are denied access.\n\r", 0);
        close_descriptor(d);
        return;
    }

    if (check_reconnect(d, argument, FALSE))
        REMOVE_BIT(ch->plr_flags, PLR_NEW);
    else if (wizlock && !IS_IMMORTAL(ch))
    {
        write_to_buffer(d, "   .    .\n\r", 0);
        close_descriptor(d);
        return;
    }

    if (!IS_SET(ch->plr_flags, PLR_NEW))
    {
            /* Old player */
            write_to_descriptor_2(d->descriptor, (char *)echo_off_str, 0);
            write_to_buffer(d, ": ", 0);
            d->connected = CON_GET_OLD_PASSWORD;
            return;
    }
    else
    {
        if (!IS_SET(muddy_mode, MUDDY_NEWBIES))
        {
            write_to_buffer(d, "At this time the institution of new characters is forbidden.\n\r", 0);
            close_descriptor(d);
            return;
        }

        /* New player */
        if (newlock)
        {
            write_to_buffer(d, "   .    .\n\r", 0);
            close_descriptor(d);
            return;
        }

        if (check_ban(d->host, BAN_NEWBIES))
        {
            log_printf("BAN: Try new player from banned site: %s", d->host);
            write_to_buffer(d, "      .\n\r", 0);
            close_descriptor(d);
            return;
        }

        do_help(ch,"NAME");
        d->connected = CON_CONFIRM_NEW_NAME;
        return;
    }
    break;

/* RT code for breaking link */
    case CON_BREAK_CONNECT:
        switch(*argument) {
        case '' : case '' : case 'y' : case 'Y':
            for (d_old = descriptor_list; d_old; d_old = d_next) {
                d_next = d_old->next;
                if (d_old == d || d_old->character == NULL)
                    continue;

                if (str_cmp(ch->name,d_old->character->name))
                    continue;

                close_descriptor(d_old);
            }

            if (check_reconnect(d, ch->name, TRUE))
                return;

            write_to_buffer(d,"   .\n\r",0);

            /* FALLTHRU */

        case '' : case '' : case 'n' : case 'N':
            write_to_buffer(d,": ",0);
            if (d->character != NULL) {
                free_char(d->character);
                d->character = NULL;
            }
            d->connected = CON_GET_NAME;
            break;

        default:
            write_to_buffer(d, "  (Y)  (N)? ", 0);
            break;
        }
        break;

    case CON_CONFIRM_NEW_NAME:
        switch (*argument) {
        case '' : case '' : case 'y': case 'Y':
            snprintf(buf, sizeof(buf),
                        "        \n\r{CAstrum Metaphora{x!\n\r"
                 "   %s: ", ch->name);
            write_to_buffer(d, buf, 0);
            write_to_descriptor_2(d->descriptor, (char *)echo_off_str, 0);
            d->connected = CON_GET_NEW_PASSWORD;
            break;

        case '' : case '' : case 'n': case 'N':
            write_to_buffer(d, ",       ?", 0);
            free_char(d->character);
            d->character = NULL;
            d->connected = CON_GET_NAME;
            break;

        default:
            write_to_buffer(d, ",  Yes ()  No ()!", 0);
            break;
        }
        break;

    case CON_GET_NEW_PASSWORD:
#if defined(unix)
        write_to_buffer( d, "\n\r", 2 );
#endif

        if (strlen(argument) < 5) {
            write_to_buffer(d, "     5 .\n\r: ", 0);
            return;
        }

        pwdnew = str_crypt(argument);
        free_string(ch->pcdata->pwd);
        ch->pcdata->pwd = str_dup(pwdnew);
        write_to_buffer(d, "   ,  : ", 0);
        d->connected = CON_CONFIRM_NEW_PASSWORD;
        break;

    case CON_CONFIRM_NEW_PASSWORD:
#if defined(unix)
    write_to_buffer( d, "\n\r", 2 );
#endif

        if (strcmp(str_crypt(argument), ch->pcdata->pwd)) {
            write_to_buffer(d, "  .\n\r"
                       "   : ", 0);
            d->connected = CON_GET_NEW_PASSWORD;
            return;
        }

        write_to_descriptor(d, (char *) echo_on_str, 0);
        char_act(" {CAstrum Metaphora{x     :", ch);
        do_help(ch, "RACETABLE");
        d->connected = CON_GET_NEW_RACE;
        break;

    case CON_GET_NEW_RACE:
        one_argument(argument, arg, sizeof(arg));

        if (!str_cmp(arg, "help")) {
            argument = one_argument(argument, arg, sizeof(arg));
            if (argument[0] == '\0') {
                char_act(" {CAstrum Metaphora{x     :", ch);
                do_help(ch,"RACETABLE");
            }
            else {
                do_help(ch, argument);
                write_to_buffer(d, " 'help'  'help <race>'    .\n\r",0);
                write_to_buffer(d, "[HINT]   ,  'cyborg'.\n\r",0);
                write_to_buffer(d, "   ? ",0);

            }
            break;
        }

        race = rn_lookup(argument);
        r = RACE(race);

        if (race == -1 || !r->pcdata) 
        {
            write_to_buffer(d, "     .\n\r", 0);

/*
            write_to_buffer(d, "  :\n\r ", 0);
            for (race = 2; race < races.nused; race++) {
                r = RACE(race);
                    if (!r->pcdata)
                            break;
                if ((race-2) % 6)
                    write_to_buffer(d,"\n\r ",0);
                write_to_buffer(d,"(",0);
                write_to_buffer(d, r->name, 0);
                write_to_buffer(d,") ",0);
                }
            write_to_buffer(d, "\n\r", 0);
*/
            write_to_buffer(d, "  ! ( 'help <race>'   .) ", 0);
            break;
        }

        SET_ORG_RACE(ch, race);
        ch->race = race;
        for (i=0; i < MAX_STATS;i++)
            ch->mod_stat[i] = 0;

        /* Add race stat modifiers
        for (i = 0; i < MAX_STATS; i++)
            ch->mod_stat[i] += r->pcdata->stats[i]; */

        /* Add race modifiers */
        if (ch)
        {
              if (r->pcdata != NULL)
              {
                    ch->max_hit += r->pcdata->hp_bonus;
                    ch->max_mana += r->pcdata->mana_bonus;
                    ch->practice = r->pcdata->prac_bonus;
              }

              ch->max_hit += 10;
              ch->hit = ch->max_hit;

              ch->max_mana += 10;
              ch->mana = ch->max_mana;
      //        ch->max_move += 10;
      //  Don't touch psp here for now.


              ch->affected_by = ch->affected_by| r->aff;

              ch->immunes = ch->immunes | r->immunes;
              for (i = 0; i < MAX_DAM; i++)
                  ch->resists[i] += r->resists[i];

              ch->form = r->form;
              ch->parts = r->parts;

              /* add cost */
              if (r->pcdata != NULL ) ch->pcdata->points = r->pcdata->points;
              if (r->pcdata != NULL ) ch->size = r->pcdata->size;
        }

        write_to_buffer(d, "   - (M)ale  (F)emale? ", 0);
        d->connected = CON_GET_NEW_SEX;
        break;

    case CON_GET_NEW_SEX:

    switch (argument[0])
    {
        case '' : case '' : case 'm': case 'M':
            ch->sex = SEX_MALE;
            ch->pcdata->true_sex = SEX_MALE;
            break;
        case '' : case '' : case 'f': case 'F':
            ch->sex = SEX_FEMALE;
            ch->pcdata->true_sex = SEX_FEMALE;
            break;
        default:
            write_to_buffer(d, "  .\n\r   ? ", 0);
            return;
    }

    {
        race_t *r;
        r = race_lookup(ORG_RACE(ch));
        if (r && r->pcdata != NULL && r->pcdata->restrict_sex)
        {
            switch (r->pcdata->restrict_sex)
            {
                case R_SEX_NEUTRAL:
                    write_to_buffer(d, "You haven't any sex.\n\r", 0);
                    ch->pcdata->true_sex = SEX_NEUTRAL;
                    break;
                case R_SEX_MALE:
                    write_to_buffer(d, "You are a male.\n\r", 0);
                    ch->pcdata->true_sex = SEX_MALE;
                    break;
                case R_SEX_FEMALE:
                    write_to_buffer(d, "You are female.\n\r", 0);
                    ch->pcdata->true_sex = SEX_FEMALE;
                    break;
            }
        }
    }

    do_help(ch,"class help");

    strnzcpy(buf, sizeof(buf), " :\n\r[ ");
    snprintf(buf1, sizeof(buf1)," (ݣ:) ");
    for (iClass = 0; iClass < classes.nused; iClass++)
    {
      if (class_ok(ch,iClass))
        {
         if (iClass < 7)
          {
            strnzcat(buf, sizeof(buf), CLASS(iClass)->name);
            strnzcat(buf, sizeof(buf), " ");
          }
         else
          {
            strnzcat(buf1, sizeof(buf1), CLASS(iClass)->name);
            strnzcat(buf1, sizeof(buf1), " ");
          }
        }
    }
    strnzcat(buf, sizeof(buf), "\n\r");
    strnzcat(buf1, sizeof(buf1), "]:\n\r");
    write_to_buffer(d, buf, 0);
    write_to_buffer(d, buf1, 0);
        write_to_buffer(d, " 'help'  'help <class>'   .\n\r",0);
        write_to_buffer(d, "[HINT]   ,  'warrior'.\n\r",0);
        write_to_buffer(d, " : ",0);
        d->connected = CON_GET_NEW_CLASS;
        break;

    case CON_GET_NEW_CLASS:
    iClass = cn_lookup(argument);
    argument = one_argument(argument, arg, sizeof(arg));

    if (!str_cmp(arg,"help"))
      {
        if (argument[0] == '\0')
        do_help(ch,"class help");
        else
        do_help(ch,argument);
            write_to_buffer(d,
        "  ( 'help <class>'   )",0);
        return;
      }

    if (iClass == -1)
    {
        write_to_buffer(d,
        "   .\n\r  ? ", 0);
        return;
    }

    if (!class_ok(ch,iClass))
      {
        write_to_buffer(d,
        "        .\n\r : ",0);
        return;
      }

        ch->class = iClass;
        ch->pcdata->points += CLASS(iClass)->points;
        act("  $t.", ch, CLASS(iClass)->name, NULL, TO_CHAR);

        for (i = 0; i < MAX_STATS; i++)
            ch->perm_stat[i] = number_range(10, get_max_train(ch, i));

        snprintf(buf, sizeof(buf),
             "Str:%s Int:%s Wis:%s Dex:%s Con:%s Cha:%s Lck:%s\n\r? Y()  N()? ",
             get_stat_alias(ch, STAT_STR),
             get_stat_alias(ch, STAT_INT),
             get_stat_alias(ch, STAT_WIS),
             get_stat_alias(ch, STAT_DEX),
             get_stat_alias(ch, STAT_CON),
             get_stat_alias(ch, STAT_CHA),
             get_stat_alias(ch, STAT_LCK));


        do_help(ch, "stats");
        write_to_buffer(d, "\n\r    (10-20+). [] - [H]elp\n\r", 0);
        write_to_buffer(d, "      ,   .\n\r", 0);
        write_to_buffer(d, "\n\r[HINT]   'POOR'  'DIM',    N().\n\r",0);
        write_to_buffer(d, "[HINT]      [C]ontinue.\n\r\n\r",0);
        write_to_buffer(d, buf, 0);
        d->connected = CON_ACCEPT_STATS;
        break;

    case CON_ACCEPT_STATS:
    WAIT_STATE(ch, 1);
    switch(argument[0])
      {
      case '' : case '' : case 'H': case 'h': case '?':
        do_help(ch,"stats");
        break;
      case '' : case '' : case 'C': case 'c':
        for (i = 0; i < MAX_STATS; i++)
        {
            ch->perm_stat[i] = (get_max_train(ch, i) - 1);
            ch->mod_stat[i] = 0;
        }
        write_to_buffer(d, "\n\r", 2);
        if (!align_restrict(ch))
        {
            write_to_buffer(d, "   ,   .\n\r",0);
            write_to_buffer(d, "[HINT]   ,  'n'.\n\r",0);
            write_to_buffer(d, ",    (G/N/E)? ",0);
            d->connected = CON_GET_ALIGNMENT;
        }
        else
        {
            write_to_buffer(d, "[ ENTER  ]\n\r",0);
            ch->endur = 100;
            print_hometown(ch);
        }
        break;
      case '' : case '' : case 'y': case 'Y':
        for (i=0; i < MAX_STATS;i++)
            ch->mod_stat[i] = 0;
        write_to_buffer(d, "\n\r", 2);
        if (!align_restrict(ch))
        {
            write_to_buffer(d, "   ,   .\n\r",0);
            write_to_buffer(d, "[HINT]   ,  'n'.\n\r",0);
            write_to_buffer(d, ",    (G/N/E)? ",0);
            d->connected = CON_GET_ALIGNMENT;
        }
        else
        {
            write_to_buffer(d, "[ ENTER  ]\n\r",0);
            ch->endur = 100;
            print_hometown(ch);
        }
        break;

        case '' : case '' : case 'n': case 'N':
            write_to_buffer(d,"Ok\n\r",0);

        for (i = 0; i < MAX_STATS; i++)
            ch->perm_stat[i] = number_range(10, get_max_train(ch, i));

            snprintf(buf, sizeof(buf),
                 "Str:%s Int:%s Wis:%s Dex:%s Con:%s Cha:%s Lck:%s\n\r? (Y/N)? ",
                 get_stat_alias(ch, STAT_STR),
                 get_stat_alias(ch, STAT_INT),
                 get_stat_alias(ch, STAT_WIS),
                 get_stat_alias(ch, STAT_DEX),
                 get_stat_alias(ch, STAT_CON),
                 get_stat_alias(ch, STAT_CHA),
                 get_stat_alias(ch, STAT_LCK));

            write_to_buffer(d, buf,0);
            d->connected = CON_ACCEPT_STATS;
            break;

        default:
            write_to_buffer(d,",  Yes ()  No ()! ",0);
            break;
        }
        break;

    case CON_GET_ALIGNMENT:
        switch(argument[0])
        {
            case 'g' : case 'G' :
                align = 1000;
                snprintf(buf, sizeof(buf), "   .\n\r");
                break;
            case 'n' : case 'N' :
                align = 0;
                snprintf(buf, sizeof(buf), "   .\n\r");
                break;
            case 'e' : case 'E' :
                align = -1000;
                snprintf(buf, sizeof(buf), "   .\n\r");
                break;
            default:
                write_to_buffer(d, "    .\n\r", 0);
                write_to_buffer(d, ",     (Good/Neutral/Evil;)? ", 0);
                return;
        }
        ch->alignment = align;
        if (!align_ok(ch, RALIGN(ch)))
        {
            write_to_buffer(d, "         .\n\r", 0);
            write_to_buffer(d, ",     (Good/Neutral/Evil)? ", 0);
            return;
        }
        write_to_buffer(d, buf, 0);
          write_to_buffer(d, "\n\r[ ENTER  ]\n\r",0);
          ch->endur = 100;
          print_hometown(ch);
    break;

      case CON_PICK_HOMETOWN: {
        int htn;

        if (argument[0] == '\0'
        ||  (htn = htn_lookup(argument)) < 0
        ||  hometown_restrict(HOMETOWN(htn), ch)) {
            char_act("That's not a valid hometown.", ch);
            print_hometown(ch);
            return;
        }

        ch->hometown = htn;
        char_act(str_empty, ch);
        act_puts("Now your hometown is $t.", ch, hometown_name(htn), NULL, TO_CHAR, POS_DEAD);
        char_act("[Hit Return to continue]", ch);

        ch->endur = 100;
        d->connected = CON_GET_ETHOS;
        break;
      }

      case CON_GET_ETHOS:

        if (!ethos_restrict(ch))
        {
            if (!ch->endur)
            {
                int ethos;
                switch(argument[0])
                {
                    case 'H': case 'h': case '?':
                        do_help(ch, "alignment");
                        return;
                        break;
                    case 'L': case 'l':
                        ethos = ETHOS_LAWFUL;
                        break;
                    case 'N': case 'n':
                        ethos = ETHOS_NEUTRAL;
                        break;
                    case 'C': case 'c':
                        ethos = ETHOS_CHAOTIC;
                        break;
                    default:
                        write_to_buffer(d, "\n\r  .\n\r", 0);
                        write_to_buffer(d, "  , (L/N/C) < help ethos   >? ",0);
                        return;
                }
                if (!ethos_ok(ch, ethos))
                {
                    write_to_buffer(d, "         .\n\r", 0);
                    write_to_buffer(d, "  , (L/N/C) < help ethos   >? ",0);
                    return;
                }
                ch->ethos = ethos;
                if (ethos == ETHOS_LAWFUL)
                    snprintf(buf, sizeof(buf),"\n\r    %s.\n\r", IS_GOOD(ch) ? "  " : IS_EVIL(ch) ? "  " : "  ");
                else if (ethos == ETHOS_NEUTRAL)
                snprintf(buf, sizeof(buf),"\n\r   %s.\n\r", IS_GOOD(ch) ? "  " : IS_EVIL(ch) ? "  " : "   ");
                else
                    snprintf(buf, sizeof(buf),"\n\r    , %s.\n\r", IS_GOOD(ch) ? "     " : IS_EVIL(ch) ? "  " : "      ");
                write_to_buffer(d, buf, 0);
            }
            else
            {
                char_act("\n\r{R!{x.    {Cethos chaotic{x,       .", ch);
                write_to_buffer(d, " 'help ethos'   .\n\r",0);
                write_to_buffer(d, "[HINT]   ,  'n'.\n\r",0);
                write_to_buffer(d, "  , (L/N/C): ",0);
                ch->endur = 0;
                return;
            }
        }
        write_to_buffer(d, "\n\r[ ENTER  ]\n\r",0);
        d->connected = CON_CREATE_DONE;
        break;

    case CON_CREATE_DONE:
        log_printf("%s@%s -  .", ch->name, d->host);
        write_to_buffer(d, "\n\r", 2);
        do_help(ch, "motd");
        char_puts("[ ENTER  ]", ch);
        d->connected = CON_READ_MOTD;
        break;

    case CON_GET_OLD_PASSWORD:
        write_to_buffer(d, "\n\r", 2);

        pwd_crypted = str_dup(ch->pcdata->pwd);

                if (strcmp(str_crypt(argument), pwd_crypted)/* || argument[0] == '\0'*/) {
                free_string(pwd_crypted);
                write_to_buffer(d, " .\n\r", 0);
                log_printf("   %s@%s", ch->name, d->host);
                if (ch->endur == 2)
                close_descriptor(d);
                else
                {
                write_to_descriptor_2(d->descriptor, (char *)echo_off_str, 0);
                write_to_buffer(d, ": ", 0);
                d->connected = CON_GET_OLD_PASSWORD;
                        ch->endur++;
                }
        return;
    }
    free_string(pwd_crypted);

    if (ch->pcdata->pwd[0] == (int) NULL) {
        write_to_buffer(d, "Warning! Null password!\n\r",0);
        write_to_buffer(d, "Please report old password with bug.\n\r",0);
        write_to_buffer(d,
        "Type 'password null <new password>' to fix.\n\r",0);
        write_to_buffer(d, "  !\n\r",0);
        write_to_buffer(d, "   .\n\r",0);
        write_to_buffer(d,
        " ' null < >' ,   .\n\r",0);
    }

    write_to_descriptor(d, (char *) echo_on_str, 0);

    if (!IS_IMMORTAL(ch) && check_cache(d, ch->name))
        return;

    if (check_reconnect(d, ch->name, TRUE))
        return;

    if (check_playing(d, ch->name))
        return;


    /* Count objects in loaded player file */
    for (obj = ch->carrying,obj_count = 0; obj != NULL;
         obj = obj->next_content)
      obj_count += get_obj_realnumber(obj);

    strnzcpy(buf, sizeof(buf),ch->name);


    // this won't help us too much, but in_room now is not NULL :/
    // TODO: why do non-existing character exists in room? (aseroth)
    //ch->in_room = NULL; // it doesn't help anyways
    free_char(ch);

    load_char_obj(d, buf, FALSE);
    ch = d->character;

    if (IS_SET(ch->plr_flags, PLR_NEW)) {
      write_to_buffer(d,
              " ,    .\n\r",
              0);
      close_descriptor(d);
      return;
    }

    /* Count objects in refreshed player file */
    for (obj = ch->carrying,obj_count2 = 0; obj != NULL;
         obj = obj->next_content)
      obj_count2 += get_obj_realnumber(obj);


    log_printf("%s@%s has connected.", ch->name, d->host);
    d->connected = CON_READ_IMOTD;

    /* FALL THRU */

    case CON_READ_IMOTD:
        if (IS_IMMORTAL(ch))
        {
           do_help(ch, "imotd");
           write_to_buffer(d,"\n\r",2);
        }
        else do_help(ch, "motd");
        d->connected = CON_READ_MOTD;

        /* FALL THRU */

    case CON_READ_MOTD:

        // offer version2 for compression by Illinar
        write_to_descriptor_2 (d->descriptor, (char *) compress2_will, 3) ;
        d->wait_compression = TRUE ;

        update_skills(ch);
        if (ch->pcdata->last_login != NULL)
        {
            char_act(str_empty, ch);
            act_puts("      : $t", ch, ch->pcdata->last_login, NULL, TO_CHAR, POS_DEAD);
        }
        char_act(str_empty, ch);
        char_act("    {CAstrum Metaphora{x!", ch);
        char_act(str_empty, ch);

        // Let them see it.
        if (reboot_counter >= 0 && reboot_counter < 60)
        {
            if (reboot_counter == 0)
                char_act("{R    {W {R!{x", ch);
            else
                act_puts("{R    {W$j {R.{x", ch, (const void *) reboot_counter, NULL, TO_CHAR, POS_DEAD);
        }

        // show GQ
        gq_list (ch, "brief") ;

        ch->next = char_list;
        char_list = ch;
        d->connected = CON_PLAYING;
        ch->pcdata->host= str_dup(d->host);
        {
            int count;
            FILE *max_on_file;
            int tmp = 0;
            count = 0;
            for (d = descriptor_list; d != NULL; d = d->next)
                if (d->connected == CON_PLAYING)
                        count++;
            max_on = UMAX(count, max_on);
            if(!(max_on_file = dfopen(TMP_PATH, MAXON_FILE, "r")))
                log_printf("nanny: couldn't open %s for reading",
                       MAXON_FILE);
            else {
                fscanf(max_on_file, "%d", &tmp);
                fclose(max_on_file);
            }
            if (tmp < max_on) {
                if(!(max_on_file = dfopen(TMP_PATH, MAXON_FILE, "w")))
                    log_printf("nanny: couldn't open %s "
                           "for writing", MAXON_FILE);
                else {
                    fprintf(max_on_file, "%d", max_on);
                    log("Global max_on changed.");
                    fclose(max_on_file);
                }
            }
        }

        if (ch->level == 0) {
            OBJ_DATA *wield;
            OBJ_INDEX_DATA *map;

            ch->level = 1;
            ch->exp = base_exp(ch);
            ch->hit = ch->max_hit;
            ch->mana = ch->max_mana;
            ch->move = ch->max_move;
        ch->psp  = psp_max(ch);
            ch->train = 3;
            ch->practice += 5;
            ch->pcdata->death = 0;

            set_title(ch, title_lookup(ch));

            do_outfit(ch, str_empty);

            obj_to_char(create_obj(get_obj_index(OBJ_VNUM_MAP), 0), ch);
            obj_to_char(create_obj(get_obj_index(OBJ_VNUM_NMAP1), 0), ch);
            obj_to_char(create_obj(get_obj_index(OBJ_VNUM_NMAP2), 0), ch);

            if ((map = get_map(ch)) != NULL)
                obj_to_char(create_obj(map, 0), ch);

            if ((wield = get_eq_char(ch, WEAR_WIELD)))
                set_skill_raw(ch, get_weapon_sn(wield),
                          40, FALSE);

            char_to_room(ch, get_room_index(ROOM_VNUM_SCHOOL));
            char_act(str_empty, ch);
            do_help(ch, "NEWBIE INFO");
            char_act(str_empty, ch);
        }
        else {
            if (ch->in_room
            && (room_is_private(ch->in_room)
            || (ch->in_room->area->clan
            && ch->in_room->area->clan != ch->clan
            && CLAN(ch->clan)->diplomacy[ch->in_room->area->clan] != DIP_ALLY
            && ch->in_room->vnum != ROOM_VNUM_RULER_CELL)))
                ch->in_room = NULL;

            if (ch->in_room)
                char_to_room(ch, ch->in_room);
            else if (IS_IMMORTAL(ch))
                char_to_room(ch, get_room_index(ROOM_VNUM_CHAT));
            else
                char_to_room(ch, get_room_index(ROOM_VNUM_TEMPLE));
        }

        if (!IS_IMMORTAL(ch)) {
            act("$n $gn{}  .", ch, NULL, NULL, TO_ROOM);
            for (i = 2; exp_for_level(ch, i) < ch->exp; i++)
                ;

            if (i < ch->level) {
                int con;
                int wis;
                int inte;
                int dex;

                con = ch->perm_stat[STAT_CON];
                wis = ch->perm_stat[STAT_WIS];
                inte = ch->perm_stat[STAT_INT];
                dex = ch->perm_stat[STAT_DEX];
                ch->perm_stat[STAT_CON] = get_max_train(ch, STAT_CON);
                ch->perm_stat[STAT_WIS] = get_max_train(ch, STAT_WIS);
                ch->perm_stat[STAT_INT] = get_max_train(ch, STAT_INT);
                ch->perm_stat[STAT_DEX] = get_max_train(ch, STAT_DEX);
                do_remove(ch, "all");
                advance(ch, i-1);
                ch->perm_stat[STAT_CON] = con;
                ch->perm_stat[STAT_WIS] = wis;
                ch->perm_stat[STAT_INT] = inte;
                ch->perm_stat[STAT_DEX] = dex;
            }
        }

        reset_char(ch);

        /* quest code */
        nextquest = -abs(ch->pcdata->questtime);
        quest_cancel(ch);
        ch->pcdata->questtime = nextquest;
        /* !quest code */

        wiznet_printf(ch, NULL, WIZ_LOGINS, 0, ch->level,
                      "{W$N{x $gN{}   (%s).",
                      !IS_NULLSTR(ch->pcdata->host) ? ch->pcdata->host : "unknown");

        connectionsCache = AddCharToCache(connectionsCache, ch);

        for (i = 0; i < MAX_STATS; i++) {
            int max_stat = get_max_train(ch, i);

            if (ch->perm_stat[i] > max_stat) {
                ch->train += ch->perm_stat[i] - max_stat;
                ch->perm_stat[i] = max_stat;
            }
        }


        if (ch->gold > 6000 && !IS_IMMORTAL(ch))
        {
            act_puts("   $j $qj{}   .", ch, (const void *) ((ch->gold - 6000) / 2), NULL, TO_CHAR, POS_DEAD);
            ch->gold -= (ch->gold - 6000) / 2;
        }

        if (ch->pet != NULL) {
            char_to_room(ch->pet,ch->in_room);
            act("$n $gn{}  .", ch->pet, NULL, NULL, TO_ROOM);
        }

        if ((ch->class == CLASS_WARRIOR && ch->pcdata->specialization[2] == 0)
              || ((ch->class == CLASS_RANGER
                 || ch->class == CLASS_PALADIN
                 || ch->class == CLASS_ANTI_PALADIN)
                 && ch->pcdata->specialization[0] == 0))
        act("  '' (specialize),     .",
                                ch, NULL, NULL, TO_CHAR);
                else if(IS_SAMURAI(ch))
                        ch->pcdata->specialization[0] = SPEC_SWORD;
                else if(ch->class == CLASS_NINJA)
                    ch->pcdata->specialization[0] = SPEC_HAND;
                else if(ch->class == CLASS_THIEF)
                        ch->pcdata->specialization[0] = SPEC_DAGGER;

                if (!JUST_KILLED(ch)) {
            do_look(ch, "auto");
            do_unread(ch, "login");
        }

        break;
    }
}

/*
 * Parse a name for acceptability.
 */
bool check_parse_name(const char *name)
{
    const char *pc;
    bool fIll,adjcaps = FALSE,cleancaps = FALSE;
    int total_caps = 0;
    int i;

    /*
     * Reserved words.
     */
    if (is_name(name, "chronos all auto immortals self someone something"
              "the you demise balance circle loner honor "
              "none clan"))
        return FALSE;

    /*
     * Length restrictions.
     */

    if (strlen(name) < 2)
        return FALSE;

    if (strlen(name) > MAX_CHAR_NAME)
        return FALSE;

    /*
     * Alphanumerics only.
     * Lock out IllIll twits.
     */
    fIll = TRUE;
    for (pc = name; *pc != '\0'; pc++) {
        if (!isalpha(*pc))
            return FALSE;

    if (!((*pc >= 'a' && *pc <= 'z')||(*pc >= 'A' && 'Z'))) //only latin symbols
        return FALSE;

        if (isupper(*pc)) { /* ugly anti-caps hack */
            if (adjcaps)
                cleancaps = TRUE;
            total_caps++;
            adjcaps = TRUE;
        }
        else
            adjcaps = FALSE;

        if (LOWER(*pc) != 'i' && LOWER(*pc) != 'l')
            fIll = FALSE;
    }

    if (fIll)
        return FALSE;

    if (total_caps > strlen(name) / 2)
        return FALSE;

    /*
     * Prevent players from naming themselves after mobs.
     */
    {
        MOB_INDEX_DATA *pMobIndex;
        int iHash;

        for (iHash = 0; iHash < MAX_KEY_HASH; iHash++) {
            for (pMobIndex  = mob_index_hash[iHash];
                 pMobIndex != NULL; pMobIndex  = pMobIndex->next)
                if (is_name(name, pMobIndex->name))
                    return FALSE;
        }
    }

    for (i = 0; i < clans.nused; i++) {
        class_t *clan = VARR_GET(&clans, i);
        if (!str_cmp(name, clan->name))
            return FALSE;
    }

    return TRUE;
}

/*
 * Look for link-dead player to reconnect.
 */
bool check_reconnect(DESCRIPTOR_DATA *d, const char *name, bool fConn)
{
    CHAR_DATA *ch;
    DESCRIPTOR_DATA *d2;

    if (!fConn) {
        for (d2 = descriptor_list; d2; d2 = d2->next) {
            if (d2 == d)
                continue;
            ch = d2->original ? d2->original : d2->character;
            if (ch && !str_cmp(d->character->name, ch->name)) {
                free_string(d->character->pcdata->pwd);
                d->character->pcdata->pwd = str_dup(ch->pcdata->pwd);
                return TRUE;
            }
        }
    }

    for (ch = char_list; ch != NULL; ch = ch->next) {
        if (!IS_NPC(ch)
        &&  (!fConn || ch->desc == NULL)
        &&  !str_cmp(d->character->name, ch->name)) {
            if (!fConn) {
                free_string(d->character->pcdata->pwd);
                d->character->pcdata->pwd = str_dup(ch->pcdata->pwd);
            }
            else {
                free_char(d->character);
                d->character    = ch;
                ch->desc    = d;
                ch->timer   = 0;
                ch->pcdata->host= str_dup(d->host);

                // offer version2 for compression by Illinar
                write_to_descriptor_2 (d->descriptor, (char *) compress2_will, 3) ;
                d->wait_compression = TRUE ;

                char_act("Reconnecting.  replay    .", ch);
                act("$n $gn{}   .", ch, NULL, NULL, TO_ROOM);

                log_printf("%s@%s reconnected.", ch->name, d->host);
                wiznet("$N $gN{}   .",
                       ch, NULL, WIZ_LINKS, 0, ch->level);

                connectionsCache = AddCharToCache(connectionsCache, ch);
                d->connected = CON_PLAYING;
            }
            return TRUE;
        }
    }

    return FALSE;
}

/*
 * Check if already playing.
 */
bool check_playing(DESCRIPTOR_DATA *d, const char *name)
{
    DESCRIPTOR_DATA *dold;

    for (dold = descriptor_list; dold; dold = dold->next) {
        if (dold != d
        &&  dold->character != NULL
        &&  dold->connected != CON_GET_CODEPAGE
        &&  dold->connected != CON_GET_NAME
        &&  dold->connected != CON_RESOLV
        &&  dold->connected != CON_GET_OLD_PASSWORD
        &&  !str_cmp(name, dold->original ?  dold->original->name :
                             dold->character->name)) {
            write_to_buffer(d, "      .\n\r",0);
            write_to_buffer(d, "  ? (Y/N)?",0);
            d->connected = CON_BREAK_CONNECT;
            return TRUE;
        }
    }

    return FALSE;
}

bool check_cache(DESCRIPTOR_DATA *d, const char *name)
{
    CONNECTIONS_CACHE_ITEM* victim;

    for (victim = connectionsCache; victim != NULL; victim = victim->next)
    {
        if (!victim || victim->ignore_item)
            continue;

        if (!is_name_raw(victim->name,
                        d->character->pcdata->linked_list, str_cmp))
            continue;

        if (!str_cmp(victim->name, d->character->name))
            continue;

    //Checking for bug with advself command and reconnect.
    if (d->character->pcdata->real_level > 91)
        return FALSE;

         write_to_buffer(d, "         .\n\r", 0);
         write_to_buffer(d,": ",0);
         if (d->character != NULL) {
              free_char(d->character);
              d->character = NULL;
         }
         d->connected = CON_GET_NAME;

         return TRUE;
    }
    return FALSE;
}

void stop_idling(CHAR_DATA *ch)
{
    if (ch == NULL
    ||  ch->desc == NULL
    ||  ch->desc->connected != CON_PLAYING
    ||  !ch->was_in_room
    ||  ch->in_room->vnum != ROOM_VNUM_LIMBO)
        return;

    ch->timer = 0;
    char_from_room(ch);
    char_to_room(ch, ch->was_in_room);
    ch->was_in_room = NULL;
    act("$n has returned from the void.", ch, NULL, NULL, TO_ROOM);
}

void char_puts(const char *txt, CHAR_DATA *ch)
{
    send_to_char(GETMSG(txt, ch->lang), ch);
}

void char_printf(CHAR_DATA *ch, const char *format, ...)
{
    char buf[MAX_STRING_LENGTH];
    va_list ap;

    va_start(ap, format);
    vsnprintf(buf, sizeof(buf), GETMSG(format, ch->lang), ap);
    va_end(ap);
    send_to_char(buf, ch);
}

/*
 * Write to one char.
 */
void send_to_char(const char *txt, CHAR_DATA *ch)
{
    char buf[MAX_STRING_LENGTH*4];

    if (txt == NULL || ch->desc == NULL)
        return;

    parse_colors(txt, buf, sizeof(buf), OUTPUT_FORMAT(ch));
    write_to_buffer(ch->desc, buf, 0);
}

/*
 * Send a page to one char.
 */
void page_to_char(const char *txt, CHAR_DATA *ch)
{
    if (txt == NULL || ch->desc == NULL)
        return; /* ben yazdim ibrahim */

    if (ch->lines == 0) {
        char_puts(txt, ch);
        return;
    }

    ch->desc->showstr_head = str_dup(txt);
    ch->desc->showstr_point = ch->desc->showstr_head;
    show_string(ch->desc, str_empty);
}

/* string pager */
void show_string(struct descriptor_data *d, char *input)
{
    char buffer[4*MAX_STRING_LENGTH];
    char buf[MAX_INPUT_LENGTH];
    char *scan;
    int lines = 0;
    int show_lines;

    one_argument(input, buf, sizeof(buf));
    if (buf[0] != '\0') {
        if (d->showstr_head) {
            free_string(d->showstr_head);
            d->showstr_head = NULL;
        }
        d->showstr_point  = NULL;
        return;
    }

    if (d->character)
        show_lines = d->character->lines;
    else
        show_lines = 0;

    for (scan = buffer; scan - buffer < sizeof(buffer)-2;
                        scan++, d->showstr_point++) {
        /*
         * simple copy if not eos and not eol
         */
        if ((*scan = *d->showstr_point) && (*scan) != '\n')
            continue;

        /*
         * bamf out buffer if we reached eos or show_lines limit
         */
        if (!*scan || (show_lines > 0 && ++lines >= show_lines)) {
            const char *chk;

            if (*scan)
                *++scan = '\0';
            send_to_char(buffer, d->character);

            for (chk = d->showstr_point; isspace(*chk); chk++)
                ;
            if (!*chk) {
                if (d->showstr_head) {
                    free_string(d->showstr_head);
                    d->showstr_head = NULL;
                }
                d->showstr_point  = NULL;
            }
            return;
        }
    }
}

/*
 *  writes bug directly to user screen.
 */
void dump_to_scr(char *text)
{
#if !defined( WIN32 )
    write(1, text, strlen(text));
#else
    printf (text);
#endif

}

int log_area_popularity(void)
{
    FILE *fp;
    AREA_DATA *area;
    extern AREA_DATA *area_first;

    fp = dfopen(TMP_PATH, AREASTAT_FILE, "w");
    fprintf(fp,"\nBooted %sArea popularity statistics (in char * ticks)\n",
            str_boot_time);

    for (area = area_first; area != NULL; area = area->next)
        if (area->count >= 5000000)
            fprintf(fp,"%-60s overflow\n",area->name);
        else
            fprintf(fp,"%-60s %u\n",area->name,area->count);

    fclose(fp);

    return 1;
}

char *get_stat_alias(CHAR_DATA *ch, int where)
{
    char *stat;
    int istat;

    if (where == STAT_STR)  {
      istat=get_curr_stat(ch,STAT_STR);
      if      (istat >  22) stat = "Titanic";
      else if (istat >= 20) stat = "Herculian";
      else if (istat >= 18) stat = "Strong";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Poor";
      else                    stat = "Weak";
      return(stat);
    }

    if (where == STAT_WIS)  {
      istat=get_curr_stat(ch,STAT_WIS);
      if      (istat >  22) stat = "Excellent";
      else if (istat >= 20) stat = "Wise";
      else if (istat >= 18) stat = "Good";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Dim";
      else                    stat = "Fool";
      return(stat);
    }

    if (where == STAT_CON)  {
      istat=get_curr_stat(ch,STAT_CON);
      if      (istat >  22) stat = "Iron";
      else if (istat >= 20) stat = "Hearty";
      else if (istat >= 18) stat = "Healthy";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Poor";
      else                    stat = "Fragile";
      return(stat);
    }

    if (where == STAT_INT)  {
      istat=get_curr_stat(ch,STAT_INT);
      if      (istat >  22) stat = "Genious";
      else if (istat >= 20) stat = "Clever";
      else if (istat >= 18) stat = "Good";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Poor";
      else                    stat = "Hopeless";
      return(stat);
    }

    if (where == STAT_DEX)  {
      istat=get_curr_stat(ch,STAT_DEX);
      if      (istat >  22) stat = "Fast";
      else if (istat >= 20) stat = "Quick";
      else if (istat >= 18) stat = "Dextrous";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Clumsy";
      else                    stat = "Slow";
      return(stat);
    }

    if (where == STAT_CHA)  {
      istat=get_curr_stat(ch,STAT_CHA);
      if      (istat >  22) stat = "Charismatic";
      else if (istat >= 20) stat = "Familier";
      else if (istat >= 18) stat = "Good";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Poor";
      else                    stat = "Mongol";
      return(stat);
    }

    if (where == STAT_LCK)  {
      istat=get_curr_stat(ch,STAT_LCK);
      if      (istat >  22) stat = "Lucky";
      else if (istat >= 20) stat = "Happy";
      else if (istat >= 18) stat = "Good";
      else if (istat >= 14) stat = "Average";
      else if (istat >= 10) stat = "Poor";
      else                    stat = "Looser";
      return(stat);
    }

   bug("stat_alias: Bad stat number.", 0);
   return(NULL);

}

void force_bust_prompt (CHAR_DATA* ch)
{
    if (!IS_NPC(ch)
    &&  ch->desc != NULL && ch->desc->pString == NULL
    &&  ch->desc->showstr_point == NULL
    &&  IS_SET (ch->comm, COMM_BUST_PROMPT))
        char_puts(str_empty, ch);
}

bool class_ok(CHAR_DATA *ch, int class)
{
    race_t *r;
    class_t *cl;

    if ((cl = class_lookup(class)) == NULL
    ||  (r = race_lookup(ORG_RACE(ch))) == NULL
    ||  !r->pcdata)
        return FALSE;

    if (race_class_lookup(r, cl->name) == NULL)
        return FALSE;
    if (cl->restrict_sex)
    {
        int sex;
        switch (ch->sex)
        {
            case SEX_MALE:
                sex = R_SEX_MALE;
                break;
            case SEX_FEMALE:
                sex = R_SEX_FEMALE;
                break;
            case SEX_NEUTRAL:
                sex = R_SEX_NEUTRAL;
                break;
            default:
                sex = 0;
        }
        if (!IS_SET(cl->restrict_sex, sex))
        return FALSE;
    }

    if (!IS_IMMORTAL(ch) &&
        IS_SET(cl->flags, CLASS_HIDDEN))
    return FALSE;

    return TRUE;
}

bool align_restrict(CHAR_DATA *ch)
{
    DESCRIPTOR_DATA *d = ch->desc;
    race_t *r;

    if ((r = race_lookup(ORG_RACE(ch))) == NULL
    ||  !r->pcdata)
        return FALSE;

    if (r->pcdata->restrict_align == RA_GOOD
    ||  CLASS(ch->class)->restrict_align == RA_GOOD) {
        write_to_buffer(d, "Your character has good tendencies.\n\r",0);
        ch->alignment = 1000;
        return TRUE;
    }

    if (r->pcdata->restrict_align == RA_NEUTRAL
    ||  CLASS(ch->class)->restrict_align == RA_NEUTRAL) {
        write_to_buffer(d, "Your character has neutral tendencies.\n\r",0);
        ch->alignment = 0;
        return TRUE;
    }

    if (r->pcdata->restrict_align == RA_EVIL
    ||  CLASS(ch->class)->restrict_align == RA_EVIL) {
        write_to_buffer(d, "Your character has evil tendencies.\n\r",0);
        ch->alignment = -1000;
        return TRUE;
    }

    return FALSE;
}

bool align_ok(CHAR_DATA *ch, int align)
{
    race_t *r;
    int     race;

    race = -1;

    if (ch == NULL)
    {
          log_printf("*** BUG ***: comm.c align_ok: NULL ch");
          return FALSE;
    }

    if (IS_NPC(ch))
       return TRUE;

    if (IS_NPC(ch))
        race = ch->pIndexData->race;
    else
        if (ch->pcdata)
           race = ch->pcdata->race;

    if (race < 0)
    {
          log_printf("*** BUG ***: comm.c align_ok: race < 0");
          return FALSE;
    }

    r = race_lookup(race);

    if (r == NULL)
    {
          log_printf("*** BUG ***: comm.c align_ok: null race_t ");
          return FALSE;
    }

    if (!r->pcdata)
    {
          log_printf("*** BUG ***: comm.c align_ok: not such pcdate for race %s", r->name);
          return FALSE;
    }

    if (r->pcdata->restrict_align  && !IS_SET(r->pcdata->restrict_align, align))
        return FALSE;

    if (CLASS(ch->class)->restrict_align &&
        !IS_SET(CLASS(ch->class)->restrict_align, align))
        return FALSE;

    return TRUE;
}

bool ethos_restrict(CHAR_DATA *ch)
{
    DESCRIPTOR_DATA *d = ch->desc;
    race_t *r;

    if (((r = race_lookup(ORG_RACE(ch))) == NULL)
        || (r->pcdata == NULL)
        || (CLASS(ch->class) == NULL))
        return FALSE;

    if (r->pcdata->restrict_ethos == ETHOS_LAWFUL
    ||  CLASS(ch->class)->restrict_ethos == ETHOS_LAWFUL) {
        write_to_buffer(d, "Your character has lawful tendencies.\n\r",0);
        ch->ethos = ETHOS_LAWFUL;
        return TRUE;
    }

    if (r->pcdata->restrict_ethos == ETHOS_NEUTRAL
    ||  CLASS(ch->class)->restrict_ethos == ETHOS_NEUTRAL) {
        write_to_buffer(d, "Your character has neutral tendencies.\n\r",0);
        ch->ethos = ETHOS_NEUTRAL;
        return TRUE;
    }

    if (r->pcdata->restrict_ethos == ETHOS_CHAOTIC
    ||  CLASS(ch->class)->restrict_ethos == ETHOS_CHAOTIC) {
        write_to_buffer(d, "Your character has chaotic tendencies.\n\r",0);
        ch->ethos = ETHOS_CHAOTIC;
        return TRUE;
    }

    return FALSE;
}

bool ethos_ok(CHAR_DATA *ch, int ethos)
{
    race_t *r;

    if ((r = race_lookup(ORG_RACE(ch))) != NULL
        && r->pcdata
        && r->pcdata->restrict_ethos &&
        !IS_SET(r->pcdata->restrict_ethos, ethos))
        return FALSE;

    if (CLASS(ch->class)->restrict_ethos &&
        !IS_SET(CLASS(ch->class)->restrict_ethos, ethos))
        return FALSE;

    return TRUE;
}

void resolv_done()
{
#if !defined (WIN32)
    char *host;
    char buf[MAX_STRING_LENGTH];
    char *p;
    DESCRIPTOR_DATA *d;

    if (fgets(buf, sizeof(buf), rfin) == NULL)
        return;

    if ((p = strchr(buf, '\n')) == NULL) {
        log_printf("rfin: line too long, skipping to '\\n'");
        while(fgetc(rfin) != '\n')
            ;
        return;
    }
    *p = '\0';

    if ((host = strchr(buf, '@')) == NULL)
        return;
    *host++ = '\0';

    log_printf("resolv_done: %s@%s", buf, host);

    for (d = descriptor_list; d; d = d->next) {
        if (d->host
        ||  d->character == NULL
        ||  str_cmp(buf, d->character->name))
            continue;
        d->host = str_dup(host);
        return;
    }
#endif
}

/* Windows 95 and Windows NT support functions (copied from Envy) */
#if defined (WIN32)

void gettimeofday (struct timeval *tp, void *tzp)
{
    tp->tv_sec  = time( NULL );
    tp->tv_usec = 0;
}
#endif

CONNECTIONS_CACHE_ITEM* AddCharToCache(CONNECTIONS_CACHE_ITEM* cache, CHAR_DATA* ch)
{
    CONNECTIONS_CACHE_ITEM* ci;
    CONNECTIONS_CACHE_ITEM* newItem;
    char buf[MAX_STRING_LENGTH];


    if (ch->pcdata == NULL)
    {
        log("AddCharToCache: What da fuck? ch->pcdata is NULL");
        return cache;
    }
    if (ch->desc == NULL)
    {
        log("AddCharToCache: Shit! ch->desc is NULL");
        return cache;
    }

    for (ci = cache; ci != NULL; ci = ci->next)
    {
        if (!str_cmp(ci->name, ch->name))
        {
            // we'll catch that bad muttafuckas
            if (str_cmp(ch->desc->host, ci->host))
            {
                snprintf(buf, sizeof(buf), "%s reconnected from different IP: %s (%s)", ch->name, ch->desc->host, ci->host);
                wiznet(buf, NULL, NULL, WIZ_LINKS, 0, ch->level);
            }

            // gotta update this item
            cache = DeleteItemFromCache(cache, ci);
            break;
        }
    }


    newItem = malloc(sizeof(*newItem));
    newItem->name = str_dup(ch->name);
    newItem->password = str_dup(ch->pcdata->pwd);
    newItem->host = str_dup(ch->desc->host);
    newItem->ignore_item = IS_IMMORTAL(ch);

    newItem->ttl = CACHE_TTL;

    newItem->next = cache;
    cache = newItem;

    // previosly placed inside login/reconnect handlers
    CheckCharsByHost(cache, ch);
    CheckCharsByPassword(cache, ch);

    return cache;
}

CONNECTIONS_CACHE_ITEM* DeleteItemFromCache(CONNECTIONS_CACHE_ITEM* cache, CONNECTIONS_CACHE_ITEM* item)
{
    CONNECTIONS_CACHE_ITEM* ci;
    // initial preparations -- delete item from cache list
    if (item == cache)
    {
        cache = item->next;
    }
    else
    {
        // .    ģ,  .
        for (ci = cache; ci->next != item; ci = ci->next);

        ci->next = item->next;
    }

    free_string(item->name);
    free_string(item->host);
    free_string(item->password);

    //free_mem(item);
    free(item);

    return cache;
}

int CheckCharsByHost(CONNECTIONS_CACHE_ITEM* cache, CHAR_DATA* ch)
{
    CONNECTIONS_CACHE_ITEM* victim;
    char buf[MAX_STRING_LENGTH];
    int count = 0;
    int left = 0;


    if (ch->desc == NULL)
    {
        log("CheckCharsByHost: ch->desc is NULL");
        return 0;
    }

    snprintf(buf, sizeof(buf), "5-min rule violation: [%s]",
             ch->name);

    for (victim = connectionsCache; victim != NULL; victim = victim->next)
    {
        if (str_cmp(victim->host, ch->desc->host)
            || !str_cmp(ch->name, victim->name)
            || victim->ignore_item)
        {
            continue;
        }

        snprintf(buf, sizeof(buf), "%s %s", buf, victim->name);
        left = victim->ttl;
        count++;
     }
     if (count == 0) return count;
     if (count == 1) snprintf(buf, sizeof(buf), "%s, (%d min)", buf, left);

     snprintf(buf, sizeof(buf), "%s :%d total from (%s).", buf, count, ch->desc->host);
     wiznet(buf, NULL, NULL, WIZ_LINKS, 0, ch->level);


     return count;
}

int CheckCharsByPassword(CONNECTIONS_CACHE_ITEM* cache, CHAR_DATA* ch)
{
    CONNECTIONS_CACHE_ITEM* victim;
    char buf[MAX_STRING_LENGTH];
    int count = 0;

    if (ch->pcdata == NULL)
    {
        log("CheckCharsByPassword: ch->pcdata is NULL");
        return 0;
    }

    snprintf(buf, sizeof(buf), "matching passwords: [%s]",
             ch->name);

    for (victim = connectionsCache; victim != NULL; victim = victim->next)
    {
        if (str_cmp(victim->password, ch->pcdata->pwd)
            || !str_cmp(ch->name, victim->name)
            || victim->ignore_item)
        {
            continue;
        }

        snprintf(buf, sizeof(buf), "%s %s", buf, victim->name);
        count++;
     }
     if (count == 0) return count;

     snprintf(buf, sizeof(buf), "%s :%d.", buf, count);
     wiznet(buf, NULL, NULL, WIZ_SECURE, 0, ch->level);

     return count;
}
