/* $Id: comm_act.c,v 1.666 2004/09/20 10:50:15 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
/************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/
 


#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "merc.h"
#include "comm_colors.h"
#include "mob_prog.h"
#include "obj_prog.h"
#include "db/lang.h"

extern const char *garble(CHAR_DATA *ch, const char *i);

typedef struct actopt_t
{
    int to_lang;        /* target's interface language */
    int to_sex;         /* target's sex */
    int act_flags;      /* act flags to pass to act_puts3() */
} actopt_t;

/*
 * static functions declarations
 */
static char *       translate(CHAR_DATA *ch, CHAR_DATA *victim,
                              const char *i);
static CHAR_DATA *  act_args(CHAR_DATA *ch, CHAR_DATA *vch,
                             int flags, const char *format);
static bool     act_skip(CHAR_DATA *ch, CHAR_DATA *vch, CHAR_DATA *to,
                         int flags, int min_pos);
static void     act_raw(CHAR_DATA *ch, CHAR_DATA *to,
                        const void *arg1, 
                        const void *arg2,
                        const void *arg3,
                        const char *str, int flags);
void act_buf(const char *str, CHAR_DATA *ch, CHAR_DATA *to,
             const void *arg1, const void *arg2, const void *arg3,
             actopt_t *opt, char *buf, size_t buf_len);


void act_raw_global (CHAR_DATA *ch, CHAR_DATA *to,
                     const void *arg1, const void *arg2, 
                     const char *str, int flags)
{
    act_raw(ch, to, arg1, arg2, NULL, str, flags);
}


void buf_act(BUFFER *buffer, const char *format, CHAR_DATA *ch,
             const void *arg1, const void *arg2, const void *arg3,
              int flags)
{
    char buf[MAX_STRING_LENGTH];
    CHAR_DATA *to = NULL;
    CHAR_DATA *vch = (CHAR_DATA *) arg2;
    static actopt_t opt;
    
    opt.to_lang = buffer->lang;
    opt.act_flags = flags | ACT_USEOPT;
    if (IS_SET(flags, TO_CHAR) && ch != NULL)
    {
        to = ch;
        opt.to_sex = ch->sex;
    }
    else if (IS_SET(flags, TO_VICT) && vch != NULL)
    {
        to = vch;
        opt.to_sex = vch->sex;
    }
    else        
        opt.to_sex = SEX_NEUTRAL;
    
    act_buf(GETMSG(format, buffer->lang), ch, to, arg1, arg2, arg3, &opt, buf, sizeof(buf));
    buf_cat(buffer, buf);
}

void act_puts3(const char *format, CHAR_DATA *ch,
               const void *arg1, const void *arg2, const void *arg3,
               int flags, int min_pos)
{
    CHAR_DATA *to;
    CHAR_DATA *vch = (CHAR_DATA *) arg2;

    if ((to = act_args(ch, vch, flags, format)) == NULL)
        return;

    if (IS_SET(flags, TO_CHAR | TO_VICT))
    {
        if (!act_skip(ch, vch, to, flags, min_pos))
        {
            act_raw(ch, to, arg1, arg2, arg3,
                    GETMSG(format, to->lang), flags);
        }
        return;
    }

    for (; to; to = to->next_in_room)
    {
        if (act_skip(ch, vch, to, flags, min_pos))
            continue;
        act_raw(ch, to, arg1, arg2, arg3,
                GETMSG(format, to->lang), flags);
    }
}

void act_printf(CHAR_DATA *ch, const void *arg1, const void *arg2, int flags,
                int min_pos, const char* format, ...)
{
    CHAR_DATA *to;
    CHAR_DATA *vch = (CHAR_DATA *) arg2;
    va_list ap;

    if ((to = act_args(ch, vch, flags, format)) == NULL)
        return;

    va_start(ap, format);
    for (; to ; to = to->next_in_room)
    {
        char buf[MAX_STRING_LENGTH];
        if (act_skip(ch, vch, to, flags, min_pos))
            continue;
        vsnprintf(buf, sizeof(buf), GETMSG(format, to->lang), ap);
        act_raw(ch, to, arg1, arg2, NULL, buf, flags);
    }
    va_end(ap);
}

const char*
act_speech2(CHAR_DATA *ch, CHAR_DATA *vch, const char *text, const void *arg, const void *arg3)
{
    static char buf[MAX_STRING_LENGTH];
    actopt_t opt;
    buf[0] = '\0';
    opt.act_flags = ACT_SPEECH(ch) | ACT_NOLF | ACT_USEOPT;
    opt.to_lang = vch->lang;
    opt.to_sex  = vch->sex;
    if (IS_NPC(ch) && !IS_AFFECTED(ch, AFF_CHARM))
        ch->lang = vch->lang;
    act_buf(text, ch, ch, arg, vch, arg3, &opt, buf, sizeof(buf));
    return buf;
}

void act_yell(CHAR_DATA *ch, const char *text, const void *arg, const char *format)
{
    CHAR_DATA *vch;
    const char msg[] = "yells";

    if (ch->in_room == NULL)
        return;
    act_puts3("You yell: '{M$U{x'", ch, NULL, NULL,
        garble(ch, act_speech(ch, ch, text, arg)), TO_CHAR | ACT_SPEECH(ch), POS_RESTING);

    for (vch = char_list; vch != NULL; vch = vch->next)
    {
        if (vch == ch 
        || vch->in_room == NULL
        || vch->in_room->area != ch->in_room->area)
            continue;
        act_puts3("$n $t: '{M$U{x'", ch, format == NULL ? msg : format, vch,
            garble(ch, act_speech(ch, vch, text, arg)), TO_VICT | ACT_TRANS | ACT_SPEECH(ch), POS_RESTING);
    }

}

void act_clan(CHAR_DATA *ch, const char *text, const void *arg) 
{
    CHAR_DATA *vch;
    clan_t *clan;
    int flags;

    if (!ch->clan)
    {
        char_act("    .", ch);
        return;
    }

    clan = clan_lookup(ch->clan);
    if (clan == NULL)
    {
        char_act("  .", ch);
        return;
    }

    act_puts("{C[{W{C]{w $n: {M$t{x", 
        ch, garble(ch, act_speech(ch, ch, text, arg)), NULL, TO_CHAR, POS_DEAD);

    flags = TO_VICT | ACT_TOBUF | ACT_SPEECH(ch);
    for (vch = char_list; vch; vch = vch->next)
        if (vch->clan == ch->clan
        && vch != ch
        && !IS_SET(vch->comm, COMM_NOCLAN))
            act_puts("{C[{W{C]{w $n: {M$t{x", 
                ch, garble(ch, act_speech(ch, vch, text, arg)), vch, flags, POS_DEAD);
}

void act_allytalk(CHAR_DATA *ch, const char *text, const void *arg) 
{
    CHAR_DATA *vch;
    clan_t *clan;
    int flags;

    if (!ch->clan)
    {
        char_act("    .", ch);
        return;
    }

    clan = clan_lookup(ch->clan);
    if (clan == NULL)
    {
        char_act("  .", ch);
        return;
    }

    act_puts("{C[{WALLY{C]{w $n: {M$t{x", 
        ch, garble(ch, act_speech(ch, ch, text, arg)), NULL, TO_CHAR, POS_DEAD);

    flags = TO_VICT | ACT_TOBUF | ACT_SPEECH(ch);
    for (vch = char_list; vch; vch = vch->next)
        if ((vch->clan == ch->clan 
        || (CLAN(ch->clan)->diplomacy[vch->clan] == DIP_ALLY
        && CLAN(vch->clan)->diplomacy[ch->clan] == DIP_ALLY))
        && vch != ch
        && !IS_SET(vch->comm, COMM_NOCLAN))
            act_puts("{C[{WALLY{C]{w $n: {M$t{x", 
                ch, garble(ch, act_speech(ch, vch, text, arg)), vch, flags, POS_DEAD);
}

void act_say(CHAR_DATA *ch, const char *text, const void *arg)
{
    CHAR_DATA *vch;
    OBJ_DATA *char_obj;
    OBJ_DATA *char_obj_next;
    OBJ_DATA *obj, *obj_next;

    if (ch->in_room == NULL)
        return;

    act_puts(" : '{G$t{x'", ch, garble(ch, act_speech(ch, ch, text, arg)), NULL, TO_CHAR | ACT_NODEAF, POS_DEAD);

    for (vch = ch->in_room->people; vch != NULL; vch = vch->next_in_room)
        act("$n : '{G$t{x'", ch, garble(ch, act_speech(ch, vch, text, arg)), vch, TO_VICT | ACT_TOBUF | ACT_SPEECH(ch));

    if (!IS_NPC(ch))
    {
        CHAR_DATA *mob, *mob_next;
        for (mob = ch->in_room->people; mob != NULL; mob = mob_next)
        {
            mob_next = mob->next_in_room;
            if (IS_NPC(mob) && HAS_TRIGGER_MOB(mob, TRIG_SPEECH)
                &&  mob->position == mob->pIndexData->default_pos)
                p_act_trigger(arg, mob, NULL, NULL, ch, NULL, NULL,TRIG_SPEECH);
            for (obj = mob->carrying; obj; obj = obj_next)
            {
                obj_next = obj->next_content;
                if (HAS_TRIGGER_OBJ( obj, TRIG_SPEECH))
                    p_act_trigger(arg, NULL, obj, NULL, ch, NULL, NULL, TRIG_SPEECH);
            }
        }
    }

    for (char_obj = ch->carrying; char_obj != NULL;
        char_obj = char_obj_next)
    {
        char_obj_next = char_obj->next_content;
        oprog_call(OPROG_SPEECH, char_obj, ch, arg);
    }

    for (char_obj = ch->in_room->contents; char_obj != NULL;
        char_obj = char_obj_next)
    {
        char_obj_next = char_obj->next_content;
        oprog_call(OPROG_SPEECH, char_obj, ch, arg);
    }

    for ( obj = ch->in_room->contents; obj; obj = obj_next )
    {
        obj_next = obj->next_content;
        if ( HAS_TRIGGER_OBJ( obj, TRIG_SPEECH ) )
        p_act_trigger( arg, NULL, obj, NULL, ch, NULL, NULL, TRIG_SPEECH );
    }
    
    if ( HAS_TRIGGER_ROOM( ch->in_room, TRIG_SPEECH ) )
        p_act_trigger( arg, NULL, NULL, ch->in_room, ch, NULL, NULL, TRIG_SPEECH );
}

void act_tell2(const char *text, CHAR_DATA *ch, const void *arg, CHAR_DATA *victim, const void *arg3)
{
    ch->repeat = victim ;
    act_puts3("  $c2{$N}: '{G$t{x'", ch, garble(ch, act_speech(ch, ch, text, arg)), victim, arg3, TO_CHAR | ACT_NODEAF, POS_DEAD);
    act_puts3("$n  : '{G$t{x'", ch, garble(ch, act_speech(ch, victim, text, arg)), victim, arg3, TO_VICT | ACT_TOBUF | ACT_SPEECH(ch), POS_SLEEPING);

    if (IS_NPC(ch))
        return;

    if (IS_NPC(victim))
    {
        if (HAS_TRIGGER_MOB(victim, TRIG_SPEECH))
            p_act_trigger(text, victim, NULL, NULL, ch, NULL, NULL, TRIG_SPEECH);
    }
    else
    {
        if (!IS_IMMORTAL(victim)
            &&  !IS_IMMORTAL(ch)
            &&  is_name(ch->name, victim->pcdata->twitlist))
            return;

        if (victim->desc == NULL)
            act_puts("$N $gN{}   ,    ,  $gN{} .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        else if (IS_SET(victim->comm, COMM_AFK))
            act_puts("$gN{}   AFK,    ,  $gN{} .", ch, NULL, victim, TO_CHAR, POS_DEAD);

        if (!IS_NPC(ch))
            victim->reply  = ch;
    }

}

/*----------------------------------------------------------------------------
 * static functions
 */

/* common and slang should have the same size */
char common[] =
"aAbBcCdDeEfFgGhHiIjJkKlLmMnNoOpPqQrRsStTuUvVwWxXyYzZ"
"ų";
char slang[] =
"eEcCdDfFiIgGhHjJoOkKlLmMnNpPuUqQrRsStTvVyYwWxXzZaAbB"
"";

/* ch says, victim hears */
static char *translate(CHAR_DATA *ch, CHAR_DATA *victim, const char *i)
{
    static char trans[MAX_STRING_LENGTH];
    char *o;
    race_t *r;

    if (IS_NULLSTR(i) || (ch == NULL) || (victim == NULL))
    {
        strnzcpy(trans, sizeof(trans), i);
        return trans;
    }

    if (IS_NPC(ch) || IS_NPC(victim)
        ||  IS_IMMORTAL(ch) || IS_IMMORTAL(victim)
        ||  ch->slang == SLANG_COMMON
        ||  ((r = race_lookup(ORG_RACE(victim))) &&
             r->pcdata &&
             ch->slang == r->pcdata->slang))
    {
        if (IS_IMMORTAL(victim))
            snprintf(trans, sizeof(trans), "[%s] %s",
                     flag_string(slang_table, ch->slang), i);
        else
            strnzcpy(trans, sizeof(trans), i);
        return trans;
    }

    snprintf(trans, sizeof(trans), "[%s] ",
             flag_string(slang_table, ch->slang));
    o = strchr(trans, '\0');
    for (; *i && o-trans < sizeof(trans)-1; i++, o++)
    {
        char *p = strchr(common, *i);
        *o = p ? slang[p-common] : *i;
    }
    *o = '\0';

    return trans;
}

static CHAR_DATA *act_args(CHAR_DATA *ch, CHAR_DATA *vch,
                           int flags, const char *format)
{
    if (format == NULL)
        return NULL;

    if (IS_SET(flags, TO_CHAR))
        return ch;

    if (IS_SET(flags, TO_VICT))
        return vch;

    if (!ch || !ch->in_room)
        return NULL;

    return ch->in_room->people;
}

static bool act_skip(CHAR_DATA *ch, CHAR_DATA *vch, CHAR_DATA *to,
                     int flags, int min_pos)
{
    if (to == NULL)
        return TRUE;

    if (to->position < min_pos)
        return TRUE;

    if (IS_SET(flags, TO_CHAR) && to != ch)
        return TRUE;
    if (IS_SET(flags, TO_VICT) && (to != vch || to == ch))
        return TRUE;
    if (IS_SET(flags, TO_ROOM) && to == ch)
        return TRUE;
    if (IS_SET(flags, TO_NOTVICT) && (to == ch || to == vch))
        return TRUE;

    if (IS_NPC(to)
        &&  to->desc == NULL
        &&  !HAS_TRIGGER_MOB(to, TRIG_ACT))
        return TRUE;

    if (IS_SET(flags, ACT_NOMORTAL) && !IS_NPC(to) && !IS_IMMORTAL(to))
        return TRUE;

/* twitlist handling */
    if ((ch != NULL) && IS_SET(flags, ACT_NOTWIT)
        &&  !IS_NPC(to) && !IS_IMMORTAL(to)
        &&  !IS_NPC(ch) && !IS_IMMORTAL(ch)
        &&  is_name(ch->name, to->pcdata->twitlist))
        return TRUE;

/* check "deaf dumb blind" chars */
    if (IS_SET(flags, ACT_NODEAF) && is_affected(to, gsn_deafen))
        return TRUE;

/* skip verbose messages */
    if (IS_SET(flags, ACT_VERBOSE)
        &&  IS_SET(to->comm, COMM_NOVERBOSE))
        return TRUE;

    return FALSE;
}

static char * const he_she  [] = { "it", "he", "she", "it", "they"};
static char * const him_her [] = { "it", "him", "her", "it", "them"};
static char * const his_her [] = { "its", "his", "her", "its", "their"};

struct tdata
{
    char    type;
    int arg;
    char *  p;
};

#define TSTACK_SZ 4

static int SEX(CHAR_DATA *ch, CHAR_DATA *looker)
{
    if (ch == NULL)
        return SEX_NEUTRAL;
    if (is_affected(ch, gsn_doppelganger) && 
        (looker != NULL && 
         (IS_NPC(looker) || !IS_SET(looker->plr_flags, PLR_HOLYLIGHT))))
        ch = ch->doppel;
    switch (ch->sex)
    {
    case SEX_MALE:
    case SEX_FEMALE:
    case SEX_NEUTRAL:
    case SEX_PLURAL:
        return ch->sex;
        break;
    default:
        return SEX_NEUTRAL;
        break;
    }
}

static const char * 
act_format_text(const char *text, CHAR_DATA *ch, CHAR_DATA *to, int sp, flag32_t flags, int to_lang)
{
    static char buf[MAX_STRING_LENGTH];
    const char * stext;
    if (text == NULL)
        text = str_empty;
    if (text != str_empty)
    {
        if (IS_SET(flags, ACT_TRANS))
            text = GETMSG(text, to_lang);
        stext = (sp < 0 && !IS_SET(flags, ACT_NOFIXTEXT)) ? fix_short(text) : text;
        strnzcpy(buf, sizeof(buf), stext);
        if (IS_SET(flags, ACT_STRANS))
            text = translate(ch, to, buf);
    }
    return text;
}

static inline const char * 
act_format_obj(OBJ_DATA *obj, CHAR_DATA *to, int sp, flag32_t flags)
{
    const char *descr;
    if (obj == NULL)
        return GETMSG("nothing", to->lang);

    if (!can_see_obj(to, obj))
        return GETMSG("something", to->lang);

    if (sp < 0)
    {
        if (IS_SET(flags, ACT_FORMSH))
        {
            memento_t * memento;
            if ((memento = lookup_remembered_obj(to, obj->pIndexData)) != NULL)
                return format_short(obj->short_descr, memento->name, to, flags | ACT_FORMSH_ALWAYS);
            return format_short(obj->short_descr, obj->name, to, flags);
        }
        if (!IS_SET(flags, ACT_NOFIXSH))
            return fix_short(mlstr_cval(obj->short_descr, to));
    }

    return descr = mlstr_cval(obj->short_descr, to);
}

static inline const char * 
act_format_room(ROOM_INDEX_DATA *room, CHAR_DATA *to, int sp, flag32_t flags)
{
    const char *descr;

    if (room == NULL)
        return GETMSG("nowhere", to->lang);

    if (!can_see_room(to, room))
        return GETMSG("somewhere", to->lang);
    
    if (sp < 0)
    {
        if (IS_SET(flags, ACT_FORMSH))
        {
            memento_t * memento;
            if ((memento = lookup_remembered_room(to, room)) != NULL)
                return format_short(room->name, memento->name, to, flags | ACT_FORMSH_ALWAYS);
            return format_short(room->name, str_empty, to, flags);
        }
        if (!IS_SET(flags, ACT_NOFIXSH))
            return fix_short(mlstr_cval(room->name, to));
    }

    return descr = mlstr_cval(room->name, to);
}

/*
 * vch is (CHAR_DATA*) arg2
 * vch1 is (CHAR_DATA*) arg1
 * obj1 is (OBJ_DATA*) arg1
 * obj2 is (OBJ_DATA*) arg2
 *
 * Known act_xxx format codes are:
 *
 * a
 * A
 * b
 * B
 * c - $cn{...} - case number ``n''
 *      where n is one of [0-6] or [jJ]
 * C
 * d - door name (arg2)
 * D
 * e - he_she(ch)
 * E - he_she(vch)
 * f
 * F - $FLnn{...} - formats string to nn characters with left     align
 *     $FCnn{...} -             -//-                      center  align
 *     $FRnn{...} -             -//-                        right align
 * g - $gx{...} - gender form depending on sex/gender of $x, 
 *      where x is one of [nNiIpPo] or [0123jJ] 
 * G
 * h
 * H
 * i - name(vch1)
 * I - name(vch3)
 * j - num(arg1)
 * J - num(arg3)
 * k
 * K
 * l
 * L
 * m - him_her(ch)
 * M - him_her(vch)
 * n - name(ch)
 * N - name(vch)
 * o
 * O
 * p - name(obj1)
 * P - name(obj2)
 * q - $qx{...} - numeric form depending on ``x'' where x is:
 *  j - num(arg1)
 *  J - num(arg2)
 * [0-9A-E] - numbers 0-15
 * Q
 * r - room name (arg1)
 * R - room name (arg3)
 * s - his_her(ch)
 * S - his_her(vch)
 * t - text(arg1)
 * T - text(arg2)
 * u - text(arg1)
 * U - text(arg3)
 * v
 * V
 * w
 * W
 * x
 * X
 * y
 * Y
 * z
 * Z
 *
 */

void act_buf(const char *str, CHAR_DATA *ch, CHAR_DATA *to,
             const void *arg1, const void *arg2, const void *arg3,
             actopt_t *opt, char *buf, size_t buf_len)
{
    CHAR_DATA * vch = (CHAR_DATA*) arg2;
    CHAR_DATA * vch1 = (CHAR_DATA*) arg1;
    CHAR_DATA * vch3 = (CHAR_DATA*) arg3;
    int     num1 = (int) arg1;
    int     num3 = (int) arg3;
    ROOM_INDEX_DATA *room1 = (ROOM_INDEX_DATA*) arg1;
    ROOM_INDEX_DATA *room3 = (ROOM_INDEX_DATA*) arg3;
    OBJ_DATA *  obj1 = (OBJ_DATA*) arg1;
    OBJ_DATA *  obj2 = (OBJ_DATA*) arg2;
    char        tmp [MAX_STRING_LENGTH];

    char *      point = buf;
    const char *    s = str;

    struct tdata    tstack[TSTACK_SZ];
    int     sp = -1;
    int to_lang = 0, to_sex = SEX_PLURAL;
    CHAR_DATA * rch = NULL;

    if (opt != NULL && (to == NULL || IS_SET(opt->act_flags, ACT_USEOPT)))
    {
        to_lang = opt->to_lang;
        to_sex = opt->to_sex;
    }
    else
    {
        to_lang = to->lang;
        to_sex = to->sex;
    }

    while (*s)
    {
        char        code;
        char        subcode;
        const char *    i;

        switch (*s)
        {
        default:
            *point++ = *s++;
            break;

        case '}':
            if (sp < 0)
            {
                *point++ = *s++;
                continue;
            }

            if (sp < TSTACK_SZ)
            {
                int rulecl = 0;

                switch (tstack[sp].type)
                {
                case 'g':
                    rulecl = RULES_GENDER;
                    break;
                case 'c':
                    rulecl = RULES_CASE;
                    break;
                case 'q':
                    rulecl = RULES_QTY;
                    break;
                case 'L':
                    rulecl = ALIGN_LEFT;
                    break;
                case 'C':
                    rulecl = ALIGN_CENTER;
                    break;
                case 'R':
                    rulecl = ALIGN_RIGHT;
                    break;
                }

                *point = '\0';
                if (tstack[sp].type == 'L'
                    || tstack[sp].type == 'R'
                    || tstack[sp].type == 'C')
                    strnzcpy(tstack[sp].p, 
                             buf_len - (tstack[sp].p - buf),
                             fmt_color_string(tstack[sp].p, tstack[sp].arg, 
                                              rulecl));
                else
                {
                    strnzcpy(tstack[sp].p, 
                             buf_len - (tstack[sp].p - buf),
                             rch == NULL ? word_form(tstack[sp].p, tstack[sp].arg, to_lang, rulecl) 
                                         : get_rus_name(rch, tstack[sp].arg));
                }
                rch = NULL;
                point = strchr(tstack[sp].p, '\0');
            }

            sp--;
            s++;
            continue;

        case '{':
            if (*(s+1) == '}')
                s++;
            *point++ = *s++;
            continue;

        case '$':
            s++;

            switch (code = *s++)
            {
            default:  
                i = " <@@@> ";
                log_printf("act_raw: '%s': bad code $%c",
                           str, code);
                continue;
            case '$':   
                i = NULL;
                *point++ = '$';
                break;
            case '{':
                i = NULL;
                *point++ = '{';
                break;
/* text arguments */
            case 't': 
            case 'u':
                i = act_format_text(arg1, ch, to, sp, opt->act_flags, to_lang);
                break;

            case 'T':
                i = act_format_text(arg2, ch, to, sp, opt->act_flags, to_lang);
                break;

            case 'U':
                i = act_format_text(arg3, ch, to, sp, opt->act_flags, to_lang);
                break;

/* room arguments */
            case 'r':
                i = act_format_room(room1, to, sp, opt->act_flags);
                break;

            case 'R':
                i = act_format_room(room3, to, sp, opt->act_flags);
                break;

/* char arguments */
            case 'n':
                i = PERS2(ch, to,
                          (sp < 0) ? opt->act_flags : ACT_NOFIXSH);
                if (!IS_NPC(ch) && !IS_NPC(to) && IS_SET(to->comm2, COMM2_RUSNAMES) && sp > -1)
                    //tstack[sp].rch = ch;
                    rch = ch;
                break;

            case 'N':
                i = PERS2(vch, to,
                          (sp < 0) ? opt->act_flags : ACT_NOFIXSH);
                if (!IS_NPC(vch) && !IS_NPC(to) && IS_SET(to->comm2, COMM2_RUSNAMES) && sp > -1)
                    rch = vch;
                break;

            case 'i':
                i = PERS2(vch1, to,
                          (sp < 0) ? opt->act_flags : ACT_NOFIXSH);
                if (!IS_NPC(vch1) && !IS_NPC(to) && IS_SET(to->comm2, COMM2_RUSNAMES) && sp > -1)
                    rch = vch1;
                break;

            case 'I':
                i = PERS2(vch3, to,
                          (sp < 0) ? opt->act_flags : ACT_NOFIXSH);
                if (!IS_NPC(vch3) && !IS_NPC(to) && IS_SET(to->comm2, COMM2_RUSNAMES) && sp > -1)
                    rch = vch3;
                break;

/* numeric arguments */
            case 'j':
                snprintf(tmp, sizeof(tmp), "%d", num1);
                i = tmp;
                break;

            case 'J':
                snprintf(tmp, sizeof(tmp), "%d", num3);
                i = tmp;
                break;

/* him/her arguments. obsolete. $gx{...} should be used instead */
            case 'e':
                i = he_she[SEX(ch, to)];
                break;

            case 'E':
                i = he_she[SEX(vch, to)];
                break;

            case 'm':
                i = him_her[SEX(ch, to)];
                break;

            case 'M':
                i = him_her[SEX(vch, to)];
                break;

            case 's':
                i = his_her[SEX(ch, to)];
                break;

            case 'S':
                i = his_her[SEX(vch, to)];
                break;

/* obj arguments */
            case 'p':
                i = act_format_obj(obj1, to, sp, opt->act_flags);
                break;

            case 'P':
                i = act_format_obj(obj2, to, sp, opt->act_flags);
                break;

/* door arguments */
            case 'd':
                if (IS_NULLSTR(arg2))
                    i = GETMSG("door", to_lang);
                else
                {
                    one_argument(arg2, tmp, sizeof(tmp));
                    i = tmp;
                }
                break;

/* $gx{...}, $cx{...}, $qx{...} arguments */
            case 'g':
            case 'c':
            case 'q':
            case 'F':
                if (*(s+1) != '{' && 
                    !(code == 'F' && (*(s+2) == '{' || *(s+3) == '{')))
                {
                    log_printf("act_raw: '%s': "
                               "syntax error", str);
                    continue;
                }

                if (++sp >= TSTACK_SZ)
                {
                    log_printf("act_raw: '%s': "
                               "tstack overflow", str);
                    continue;
                }

                tstack[sp].p = point;
                tstack[sp].type = code;
                subcode = *s++;
                s++;

                switch (code)
                {
                case 'F':
                    tstack[sp].type = subcode;
                    tstack[sp].arg = *(s-1) - '0';
                    if (*s != '{')
                    {
                        tstack[sp].arg *= 10;
                        tstack[sp].arg += *s -'0';
                        s += 2;
                    }
                    else
                        s++;
                    break;

                case 'c':
                    if (subcode == 'j')
                        tstack[sp].arg = num1;
                    else if (subcode == 'J')
                        tstack[sp].arg = num3;
                    else
                        tstack[sp].arg = subcode - '0';
                    break;

                case 'g':
                    tstack[sp].arg = SEX_NEUTRAL;
                    switch (subcode)
                    {
                    case 'N':
                        if (vch != NULL)
                            tstack[sp].arg = vch->sex;
                        break;

                    case 'n':
                        if (ch != NULL)
                            tstack[sp].arg = ch->sex;
                        break;

                    case 'i':
                        if (vch1 != NULL)
                            tstack[sp].arg = vch1->sex;
                        break;

                    case 'I':
                        if (vch3 != NULL)
                            tstack[sp].arg = vch3->sex;
                        break;

                    case 'o':
                        tstack[sp].arg = to_sex;
                        break;

                    case 'p':
                        if (obj1 != NULL)
                            tstack[sp].arg = obj1->pIndexData->gender;
                        break;

                    case 'P':
                        if (obj2 != NULL)
                            tstack[sp].arg = obj2->pIndexData->gender;
                        break;
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                        tstack[sp].arg = subcode - '0';
                        break;
                    case 'j':
                        tstack[sp].arg = num1;
                        break;
                    case 'J':
                        tstack[sp].arg = num3;
                        break;
                    default:
                        log_printf("act_raw: '%s': "
                                   "bad subcode '%c'",
                                   str, subcode);
                        sp--;
                        break;
                    }
                    break;

                case 'q':
                    switch (subcode)
                    {
                    case 'j':
                        tstack[sp].arg = num1;
                        break;

                    case 'J':
                        tstack[sp].arg = num3;
                        break;
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                        tstack[sp].arg = subcode - '0';
                        break;
                    case 'A':
                    case 'B':
                    case 'C':
                    case 'D':
                    case 'E':
                    case 'F':
                        tstack[sp].arg = subcode - 'A' + 10;
                        break;
                    default:
                        log_printf("act_raw: '%s': "
                                   "bad subcode '%c'",
                                   str, subcode);
                        sp--;
                        break;
                    }
                    break;
                }
                continue;
            }

            if (i)
            {
                while ((*point++ = *i++));
                point--;
            }
            break;
        }
    }

    if (!IS_SET(opt->act_flags, ACT_NOLF))
        *point++ = '\n';
    *point = '\0';

    if (!IS_SET(opt->act_flags, ACT_NOUCASE))
    {
        point = (char*) cstrfirst(buf);
        *point = UPPER(*point);
    }

}

static void act_raw(CHAR_DATA *ch, CHAR_DATA *to,
                    const void *arg1, const void *arg2, const void *arg3,
                    const char *str, int flags)
{
    char        buf [MAX_STRING_LENGTH];
    char        tmp [MAX_STRING_LENGTH];
    static actopt_t opt;
    opt.act_flags = flags;
    opt.to_lang = to->lang;
    opt.to_sex = to->sex;

    act_buf(str, ch, to, arg1, arg2, arg3, &opt, buf, sizeof(buf));

    memset(tmp, 0, sizeof(tmp)); //  , ,   ?
    parse_colors(buf, tmp, sizeof(tmp), OUTPUT_FORMAT(to));

    if (!IS_NPC(to))
    {
        if ((IS_SET(to->comm, COMM_AFK) || to->desc == NULL) &&
            IS_SET(flags, ACT_TOBUF))
            buf_add(to->pcdata->buffer, tmp);
        else if (to->desc)
        {
            if (IS_SET(to->comm, COMM_QUIET_EDITOR)
                &&  to->desc->pString)
                buf_add(to->pcdata->buffer, tmp);
            else
                write_to_buffer(to->desc, tmp, 0);
        }
    }
    else
    {
        if (!IS_SET(flags, ACT_NOTRIG))
            p_act_trigger(tmp, to, NULL, NULL, ch, arg1, arg2, TRIG_ACT);
        if (to->desc)
            write_to_buffer(to->desc, tmp, 0);
    }
} 
