/* $Id: conquer.c,v 1.666 2004/09/20 10:49:47 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "merc.h"
#include "db/dofun.h"
#include "db/db.h"
#include "clan.h"
#include "conquer.h"
#include "olc/olc.h"
#include "quest.h"

extern AREA_DATA *area_first;

// NB!
// #define C_VALUE - default value
// static int VALUE - variable. Variable can be set using do_cconfig function
// all variable are saved to the file and default values are overwritten during
// system boot

// energy stuff
static int world_free_energy;               // global variable can be accessed via
                                            // get_world_free_energy and put_world_free_energy
#define C_ENERGY_OUTPUT_ROOM_MOD    40
static int ENERGY_OUTPUT_ROOM_MOD  = C_ENERGY_OUTPUT_ROOM_MOD;

#define FREE_ENERGY_LIMIT         2000000    // if new_value in put_world_free_energy
                                             // is greater than limit - truncate

#define CLAN_ENERGY_LIMIT         1000000    // each tick each clan's energy is checked
                                             // and is truncated if needed

static int pulse_conquered;               // save time counter
#define AREACONTROL_SAVE_LIMIT    15      // interval we must save conquered.lst (ticks)

// conquerable areas. Counted when booting and in
// conquer enable/disable functions
static int conq_area_count;

#define C_AREA_ATTACK_TIME_LIMIT      8       // ticks
static int AREA_ATTACK_TIME_LIMIT   = C_AREA_ATTACK_TIME_LIMIT;
#define C_AREA_DEFENCE_TIME_LIMIT     4
static int AREA_DEFENCE_TIME_LIMIT  = C_AREA_DEFENCE_TIME_LIMIT;
#define C_AREA_PREVLOCK_TIME_LIMIT    20
static int AREA_PREVLOCK_TIME_LIMIT = C_AREA_PREVLOCK_TIME_LIMIT;

#define C_VNUM_AREAGUARD_STANDARD   15101   // default standard area guard vnum
#define C_VNUM_AREAGUARD_VETERAN    15102   // default veteran area guard vnum
#define C_VNUM_AREAGUARD_ELITE      15103   // default elite area guard vnum
#define C_VNUM_PRIEST               15104   // default vnum for priest
#define C_VNUM_ADVISOR              15105   // default vnum for advisor
#define C_VNUM_SPY                  15106   // default vnum for spy
#define C_VNUM_TRANSPORTER          15107   // default vnum for transporter

int VNUM_AREAGUARD_STANDARD  = C_VNUM_AREAGUARD_STANDARD;
int VNUM_AREAGUARD_VETERAN   = C_VNUM_AREAGUARD_VETERAN;
int VNUM_AREAGUARD_ELITE     = C_VNUM_AREAGUARD_ELITE;
int VNUM_PRIEST              = C_VNUM_PRIEST;
int VNUM_ADVISOR             = C_VNUM_ADVISOR;
int VNUM_SPY                 = C_VNUM_SPY;
int VNUM_TRANSPORTER         = C_VNUM_TRANSPORTER;

// ---------------------------------------------------------------------------------------
// altars
// to add an altar:
//      1) increase MAX_ALTAR_COUNT in merc.h
//      2) inrease MAX_UPGRADES
//      3) define a vnum
//      4) add values to altar_vnums, altar_flags, altar_msg, check_msg arrays
//      5) add ALTAR_POS (position in array) to "conquer.h"
//      6) add appropriate code to 'manage_add_altar' function
//      7) add functionality. For example:
//            if (altar_works (ch, ALTAR_HASTE_POS))
//                     do_somthing ();
//      8) don't forget to create object index! It can be checked by 'conquer check' command
// ---------------------------------------------------------------------------------------

// todo: add structure altar_descriptions

#define MAX_UPGRADES                13    // 2 guard upgrades + 11 altars
#define MAX_UPGRADES_CLAN           6     // 6 altars
#define MAX_UPGRADES_BR             10    // 2 guard upgrades + 8 altars
#define MAX_UPGRADES_CLAN_BR        4     // 4 altars

#define C_VNUM_ALTAR_GREED          15102 // converts a part of silver into gold
                                          // when corpses are sacrificed by clan members
#define C_VNUM_ALTAR_HASTE          15103 // 0-2 extra attacks
#define C_VNUM_ALTAR_MAGIC          15104 // 1-2 points to spell level
#define C_VNUM_ALTAR_TRAVEL         15105 // mage-transporter can operate in area
#define C_VNUM_ALTAR_TRANQUILITY    15106 // aggro mobs don't attack clan members
#define C_VNUM_ALTAR_SUMMONING      15107 // summoned mobs have level + 2
#define C_VNUM_ALTAR_IMMOBILITY     15108 // only clan members can use transportation spells in area
#define C_VNUM_OUTPOST              15109 // spy can operate in area
#define C_VNUM_ALTAR_REGENERATION   15110 // 20% better hp/mana restoring rate and 4% chance to
                                          // restore all hp/mana
#define C_VNUM_ALTAR_EXPERIENCE     15111 // clan members gain +20% experience on kills
#define C_VNUM_ALTAR_NEGATION       15115 // prevents using of clan skills for opposite clans
                                          // todo: "far" skills e.g. eyes of intrigue or hunt

static int altar_vnums[MAX_ALTAR_COUNT] = {
    C_VNUM_ALTAR_GREED,
    C_VNUM_ALTAR_HASTE,
    C_VNUM_ALTAR_MAGIC,
    C_VNUM_ALTAR_TRAVEL,
    C_VNUM_ALTAR_TRANQUILITY,
    C_VNUM_ALTAR_SUMMONING,
    C_VNUM_ALTAR_IMMOBILITY,
    C_VNUM_OUTPOST,
    C_VNUM_ALTAR_REGENERATION,
    C_VNUM_ALTAR_EXPERIENCE,
      C_VNUM_ALTAR_NEGATION
};

static flag32_t altar_flags[MAX_ALTAR_COUNT] = {
    ALTAR_NO_CLANAREA,                  // greed
    0,                                  // haste
    ALTAR_NO_BR,                        // magic
    ALTAR_NO_BR,                        // travel
    ALTAR_NO_CLANAREA,                  // tranquility
    ALTAR_NO_BR | ALTAR_NO_CLANAREA,    // summoning
    ALTAR_NO_CLANAREA,                  // immobility
    0,                                  // outpost
    0,                                  // regeneration
    ALTAR_NO_CLANAREA,                  // experience
      0                                   // negation
};

// messages for manage check
static char altar_msg[MAX_ALTAR_COUNT][MAX_INPUT_LENGTH] = {
    "There is an Altar of {YGREED{x in this area.",
    "There is an Altar of {RHASTE{x in this area.",
    "There is an Altar of {MMAGIC{x in this area.",
    "There is an Altar of {CTRAVEL{x in this area.",
    "There is an Altar of {gTRANQUILITY{x in this area.",
    "There is an Altar of {ySUMMONING{x in this area.",
    "There is an Altar of {rIMMOBILITY{x in this area.",
    "There is an {DOUTPOST{x in this area.",
    "There is an Altar of {GREGENERATION{x in this area.",
    "There is an Altar of {CEXPERIENCE{x in this area.",
    "There is an Altar of {mNEGATION{x in this area."
};

// messages for conquer check
static char check_msg[MAX_ALTAR_COUNT][MAX_INPUT_LENGTH] = {
    "Altar of {YGREED{x doesn't exist.",
    "Altar of {RHASTE{x doesn't exist.",
    "Altar of {MMAGIC{x doesn't exist.",
    "Altar of {CTRAVEL{x doesn't exist.",
    "Altar of {gTRANQUILITY{x doesn't exist.",
    "Altar of {ySUMMONING{x doesn't exist.",
    "Altar of {rIMMOBILITY{x doesn't exist.",
    "{DOUTPOST{x doesn't exist.",
    "Altar of {GREGENERATION{x doesn't exist.",
    "Altar of {CEXPERIENCE{x doesn't exist.",
    "Altar of {mNEGATION{x doesn't exist."
};


// ticks between upgrades
#define C_TIME_TO_UPGRADE           60
static int TIME_TO_UPGRADE = C_TIME_TO_UPGRADE;

// local function return codes
#define RC_OK                     0 // all is OK
#define RC_ALTAR_ALREADY_IN_ROOM  1 // in target room already exists an altar
#define RC_ALTAR_ALREADY_EXISTS   2 // altar of this type already placed in area
#define RC_INVALID_PLACE          3 // safe room or something else in the room wrong
#define RC_BUG                    4 // BUG!

static char err_msg[][MAX_INPUT_LENGTH] = {
    "",
    "But an altar in this room already exists!",
    "But an altar of this type already exists in this area!",
    "You can't place an altar right here!",
    "{RInternal error! Please report to immortals!{x"
};

#define C_ATTACK_TIME_MSG 0
#define C_DEFENCE_TIME_MSG 1
#define C_LOCK_TIME_MSG 2
#define C_UPGRADE_TIME_MSG 3
#define C_UPKEEP_CLANMEMBER_MSG 4
#define C_UPKEEP_ALTAR_MSG 5
#define C_UPKEEP_ADVISOR_MSG 6
#define C_UPKEEP_SPY_MSG 7
#define C_UPKEEP_TRANSPORTER_MSG 8
#define C_UPKEEP_STANDARD_MSG 9
#define C_UPKEEP_VETERAN_MSG 10
#define C_UPKEEP_ELITE_MSG 11
#define C_ENERGY_OUTPUT_ROOM_MSG 12

static char value_show_msg[][MAX_INPUT_LENGTH] = {
    "Time needed for successful attack: {C%d{x ticks per guard.\n",
    "Time needed for successful defence: {C%d{x ticks per guard.\n",
    "Area can't be reconquered during {C%d{x ticks.\n",
    "Time between upgrades is {C%d{x ticks.\n",
    "Clanmember upkeep cost is {C%d{x energy.\n",
    "Altar upkeep cost is {C%d{x energy.\n",
    "Advisor upkeep cost is {C%d{x energy.\n",
    "Spy upkeep cost is {C%d{x energy.\n",
    "Mage-transporter upkeep cost is {C%d{x energy.\n",
    "Standard guard upkeep cost is {C%d{x energy.\n",
    "Veteran guard upkeep cost is {C%d{x energy.\n",
    "Elite guard upkeep cost is {C%d{x energy.\n",
    "Energy output is {C%d{x energy/tick per room.\n"
};

static char value_set_msg[][MAX_INPUT_LENGTH] = {
    "Time needed for successful attack set to {C%d{x ticks per guard.\n",
    "Time needed for successful defence set to {C%d{x ticks per guard.\n",
    "Now area can't be reconquered during {C%d{x ticks.\n",
    "Now time between upgrades is {C%d{x ticks.\n",
    "Now clanmember upkeep cost is {C%d{x energy.\n",
    "Now altar upkeep cost is {C%d{x energy.\n",
    "Now advisor upkeep cost is {C%d{x energy.\n",
    "Now spy upkeep cost is {C%d{x energy.\n",
    "Now mage-transporter upkeep cost is {C%d{x energy.\n",
    "Now standard guard upkeep cost is {C%d{x energy.\n",
    "Now veteran guard upkeep cost is {C%d{x energy.\n",
    "Now elite guard upkeep cost is {C%d{x energy.\n",
    "Now energy output is {C%d{x energy/tick per room.\n"
};

// local functions - conquer commands
static void conquer_area     ( CHAR_DATA *ch, const char *arg ); // conquer area in which char is (ch_area)
static void conquer_list     ( CHAR_DATA *ch, const char *arg ); // list of conquered areas
static void conquer_free     ( CHAR_DATA *ch, const char *arg ); // list of conquerable but free areas
static void conquer_defend   ( CHAR_DATA *ch, const char *arg ); // place guards when all of them are lost
static void conquer_liberate ( CHAR_DATA *ch, const char *arg ); // free area
static void conquer_info     ( CHAR_DATA *ch, const char *arg ); // full conquer info for area (for imms)
static void conquer_rating   ( CHAR_DATA *ch, const char *arg ); // clan rating
static void conquer_enable   ( CHAR_DATA *ch, const char *arg ); // enable conquering of an area
static void conquer_disable  ( CHAR_DATA *ch, const char *arg ); // disable conquering of an area
static void conquer_report   ( CHAR_DATA *ch, const char *arg ); // guards report their condition
static void conquer_check    ( CHAR_DATA *ch, const char *arg ); // check it all
static void conquer_reset    ( CHAR_DATA *ch, const char *arg ); // reset all conquest info
static void conquer_allow    ( CHAR_DATA *ch, const char *arg ); // allow clan to conquer areas
static void conquer_refuse   ( CHAR_DATA *ch, const char *arg ); // clan can't conquer areas no more

// local functions - advisor commands
static void advisor_check    ( CHAR_DATA *ch, const char *arg );
static void advisor_hire     ( CHAR_DATA *ch, const char *arg ); // hire an advisor for a clan
static void advisor_dismiss  ( CHAR_DATA *ch, const char *arg );
static void advisor_upkeep   ( CHAR_DATA *ch, const char *arg );
static void advisor_info     ( CHAR_DATA *ch, const char *arg );

// local functions - manage commands
static void manage_check     ( CHAR_DATA *ch, const char *arg );
static void manage_upgrade   ( CHAR_DATA *ch, const char *arg );
static void manage_downgrade ( CHAR_DATA *ch, const char *arg );
static void manage_add_altar ( CHAR_DATA *ch, const char *arg );
static void manage_rem_altar ( CHAR_DATA *ch, const char *arg );

// local functions - spy commands
static void spy_check        ( CHAR_DATA *ch, const char *arg );
static void spy_hire         ( CHAR_DATA *ch, const char *arg );
static void spy_dismiss      ( CHAR_DATA *ch, const char *arg );
static void spy_report       ( CHAR_DATA *ch, const char *arg );

// local functions - cconfig commands
static void cconfig_info     ( CHAR_DATA *ch, const char *arg );
static void cconfig_reset    ( CHAR_DATA *ch, const char *arg );
static void cconfig_save     ( CHAR_DATA *ch, const char *arg );
static void cconfig_load     ( CHAR_DATA *ch, const char *arg );
static void cconfig_attack   ( CHAR_DATA *ch, const char *arg );
static void cconfig_defence  ( CHAR_DATA *ch, const char *arg );
static void cconfig_lock     ( CHAR_DATA *ch, const char *arg );
static void cconfig_upgrade  ( CHAR_DATA *ch, const char *arg );
static void cconfig_upkeep   ( CHAR_DATA *ch, const char *arg );
static void cconfig_output   ( CHAR_DATA *ch, const char *arg );

// --------------------------------------------------------------------------------
// data structures
// --------------------------------------------------------------------------------

static CMD_DATA conq_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra
    { "info",     conquer_info,     POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "",     conquer_info,     POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE | CONQCMD_RUS},
    { "area",     conquer_area,     POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED},
    { "",     conquer_area,     POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED | CONQCMD_RUS},
    { "list",     conquer_list,     POS_DEAD,     1,              1,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "",   conquer_list,     POS_DEAD,     1,              1,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE | CONQCMD_RUS},
    { "free",     conquer_free,     POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "", conquer_free,     POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE | CONQCMD_RUS},
    { "defend",   conquer_defend,   POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED},
    { "",   conquer_defend,   POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED | CONQCMD_RUS},
    { "report",   conquer_report,   POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED},
    { "",    conquer_report,   POS_STANDING, 1,              0,  0,   CONQ_CLAN,      CONQCMD_IF_PERMITTED | CONQCMD_RUS},
    { "liberate", conquer_liberate, POS_STANDING, 1,              0,  0,   CONQ_ELITE,     CONQCMD_IF_PERMITTED},
    { "",conquer_liberate,POS_STANDING, 1,              0,  0,   CONQ_ELITE,     CONQCMD_IF_PERMITTED | CONQCMD_RUS},
    { "rating",   conquer_rating,   POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "",  conquer_rating,   POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE | CONQCMD_RUS},
    { "enable",   conquer_enable,   POS_DEAD,     LEVEL_IMMORTAL, 0,  0,   CONQ_ALL,       0},
    { "disable",  conquer_disable,  POS_DEAD,     LEVEL_IMMORTAL, 0,  0,   CONQ_ALL,       0},
    { "check",    conquer_check,    POS_DEAD,     LEVEL_IMMORTAL, 0,  0,   CONQ_ALL,       0},
    { "reset",    conquer_reset,    POS_DEAD,     LEVEL_IMMORTAL, 0,  0,   CONQ_ALL,       0},
    { "allow",    conquer_allow,    POS_DEAD,     LEVEL_IMMORTAL, 0,  0,   CONQ_ALL,       0},
    { "refuse",   conquer_refuse,   POS_DEAD,     1,              0,  0,   CONQ_LEADERS,   CONQCMD_IF_PERMITTED},
    { "",    conquer_refuse,   POS_DEAD,     1,              0,  0,   CONQ_LEADERS,   CONQCMD_IF_PERMITTED | CONQCMD_RUS},
    { NULL}
};

static CMD_DATA advisor_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra
    { "check",    advisor_check,    POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_KEEP_HIDE | CONQCMD_IF_PERMITTED},
    { "hire",     advisor_hire,     POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "dismiss",  advisor_dismiss,  POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "upkeep",   advisor_upkeep,   POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "info",     advisor_info,     POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { NULL}
};

static CMD_DATA manage_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra
    { "check",    manage_check,     POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_KEEP_HIDE | CONQCMD_IF_PERMITTED},
    { "upgrade",  manage_upgrade,   POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "downgrade",manage_downgrade, POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "addaltar", manage_add_altar, POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "remaltar", manage_rem_altar, POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "info",     conquer_info,     POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { NULL}
};

static CMD_DATA spy_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra
    { "check",    spy_check,        POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_KEEP_HIDE | CONQCMD_IF_PERMITTED},
    { "hire",     spy_hire,         POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "dismiss",  spy_dismiss,      POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { "report",   spy_report,       POS_DEAD,     1,              0,  0,   CONQ_CLAN,       CONQCMD_IF_PERMITTED},
    { NULL}
};

static CMD_DATA cconfig_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra
    { "info",     cconfig_info,     POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "reset",    cconfig_reset,    POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "save",     cconfig_save,     POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "load",     cconfig_load,     POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "attack",   cconfig_attack,   POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "defence",  cconfig_defence,  POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "lock",     cconfig_lock,     POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "upgrade",  cconfig_upgrade,  POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "upkeep",   cconfig_upkeep,   POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { "output",   cconfig_output,   POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_KEEP_HIDE},
    { NULL}
};

bool check_altar_negation (CHAR_DATA * ch);
bool check_altar_immobility (CHAR_DATA * ch, ROOM_INDEX_DATA * where);

// local functions
static void fread_conquered_area (FILE *fp);
static void fread_conquered_global_data (FILE *fp);
static void fread_conquered_info_clan (FILE *fp);
static void place_area_guards (AREA_DATA *curr_area, int clan);
static void nuke_area_guards (AREA_DATA *pArea);
static int  apply_count (AREA_DATA * pArea);
static int  add_altar_raw (int room_vnum, int clan, int alt_pos);
static int  rem_altar_raw (AREA_DATA *pArea, int altar_type);
static int  calculate_clan_upkeep (clan_t *clan);

static void set_numeric_value (CHAR_DATA *ch, const char *arg, int *var, int msg_num);

// --------------------------------------------------------------------------------

// upkeep costs
#define C_UPKEEP_SGUARD 150  // standard guard
#define C_UPKEEP_VGUARD 600  // veteran guard
#define C_UPKEEP_EGUARD 1200 // elite guard
static int upkeep_cost_guard[3] = {
    C_UPKEEP_SGUARD,
    C_UPKEEP_VGUARD,
    C_UPKEEP_EGUARD
};

#define C_UPKEEP_CLANMEMBER           500
static int upkeep_cost_clanmember   = C_UPKEEP_CLANMEMBER;
#define C_UPKEEP_ADVISOR              20000
static int upkeep_cost_advisor      = C_UPKEEP_ADVISOR;
#define C_UPKEEP_SPY                  10000
static int upkeep_cost_spy          = C_UPKEEP_SPY;
#define C_UPKEEP_TRANSPORTER          20000
static int upkeep_cost_transporter  = C_UPKEEP_TRANSPORTER;
#define C_UPKEEP_ALTAR                2000
static int upkeep_cost_altar        = C_UPKEEP_ALTAR;
//static int upkeep_cost_clanskill    = 4000;


// ********************************************************************************
//            conquest commands
// ********************************************************************************

// --------------------------------------------------------------------------------
// function called from interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_conquer)
{
    if (!parse_command(ch, argument, conq_cmd_table))
    {
        show_command_list (ch, conq_cmd_table);
        char_puts (".\nUse {CHELP CONQUER{x for more information.\n", ch);

        if (IS_IMMORTAL(ch))
            char_act ("\n{RWarning!!!{x Target area would be saved immediatly after 'conquer enable/disable' command.", ch);
    }
}

// --------------------------------------------------------------------------------
// take conquerable free area under control
// --------------------------------------------------------------------------------
static void conquer_area (CHAR_DATA *ch, const char* arg)
{
    clan_t              * clan    = NULL ;
    AREA_DATA           * curr_area = ch->in_room->area;
    CHAR_EXPLORED_DATA  * expl_area;
    int                   i, j;
    bool                  guardrooms_visited = FALSE;

    // only for PCs - just to be sure
    if (IS_NPC (ch))
        return;

    if (IS_IMMORTAL (ch))
    {
        char_act ("'Conquer area' command is for mortals only.", ch);
        return;
    }

    // conquer area in which ch now
    // only for clan players - just to be sure
    if (!ch->clan)
    {
        char_act ("Area can be conquered only by clan players.", ch);
        return;
    }

    clan = clan_lookup (ch->clan) ;

    // only for neutral areas without guards
    if (curr_area->clan_owner)
    {
        if (clan == curr_area->clan_owner)
            char_act ("This area is already conquered by your clan.\n", ch);
        else
        {
            char_printf (ch, "This area is already conquered by %s.\n",
                curr_area->clan_owner->name);
            char_act ("You have to eliminate area guards first.", ch);
        }
        return;
    }
    if (curr_area->guards_left)
    {
        char_act ("This area is still protected.", ch);
        char_act ("You have to eliminate area guards first.", ch);
        return;
    }

    // areas with max_guard_count = 0 can't be conqered at all
    if  ((!curr_area->conq_enabled) ||
        (!curr_area->max_guard_count)) // let it be - just to be sure
    {
        char_act ("This area can't be conquered.", ch);
        return;
    }

    // if the clan of this char just has lost control of this area
    if (curr_area->prev_lock_time && (clan == curr_area->prev_clan_owner))
    {
        char_act ("Your weak clan just has lost this area. It can't be reconquered now.", ch);
        return;
    }

    // check 'explore required' feature
    switch (curr_area->explore_required)
    {
    case EXPLORE_NOT_REQ:
        break;
    case EXPLORE_GUARDROOMS_REQ:
        // explore_data loop
        for (expl_area = ch->pcdata->explored_areas ; expl_area ; expl_area=expl_area->next)
            if (!str_cmp (expl_area->name, curr_area->name)) // area found!
            {
                // guardrooms loop
                for (j = 0; j < MAX_GUARD_COUNT; ++j)
                {
                    if (!curr_area->guard_room_vnums [j])
                        continue;

                    guardrooms_visited = FALSE;
                    // explored_rooms_in_target_area loop
                    for (i = 0 ; i < expl_area->explored_room_count ; ++i)
                        if (expl_area->explored_room_vnums[i] == curr_area->guard_room_vnums [j])
                        {
                            guardrooms_visited = TRUE;
                            break;
                        }
                    if (!guardrooms_visited)
                        break;
                }
                break;
            }

        if (!guardrooms_visited)
        {
            char_act("You can't choose the right position for the guards.", ch);
            return;
        }
        break;

    case EXPLORE_ALL_AREA_REQ:
        for (expl_area = ch->pcdata->explored_areas ; expl_area ; expl_area=expl_area->next)
        {
            if (!str_cmp (expl_area->name, curr_area->name))
            {
                // area doesn't completely explored
                if (expl_area->explored_room_count != curr_area->explorable_rooms)
                {
                    char_act("You must explore whole area before conquer it.", ch);
                    return ;
                }
                else // wow! area is completely explored
                {
                    break;
                }
            }
        }
        break;
    }

    // place area guards
    place_area_guards (curr_area, ch->clan);

    curr_area->clan_owner = clan;
    clan->conquered_areas += 1;
    char_printf (ch, "Now area %s is under your control!\n", curr_area->name);
    save_conquered_areas();
}

// --------------------------------------------------------------------------------
// show to char a list of conquered areas
// --------------------------------------------------------------------------------
static void conquer_list(CHAR_DATA *ch, const char* arg)
{
    char        cname [MAX_INPUT_LENGTH] ;
    AREA_DATA * pArea;
    BUFFER    * output;
    clan_t    * clan = NULL ;
    int         count = 0;

    one_argument (arg, cname, sizeof(cname)) ;

    // areas conquered by any clan
    if (cname[0] == '\0' )
    {
        output = buf_new(ch->lang);
        buf_printf (output, "%-20s %-16s\n\r", "  Area Name", "Clan");
        buf_printf (output, "---------------------------------\n");

        count = 0;
        for (pArea = area_first; pArea; pArea = pArea->next)
        {
            if (pArea->clan_owner)
            {
                ++count;
                buf_printf (output, "%s %-16s\n\r", fmt_color_str(pArea->name, 20),
                    pArea->clan_owner->name);
            }
        }
        if (count)
            page_to_char(buf_string(output), ch);
        else
            char_act("There are no conquered areas.", ch);
        buf_free(output);
    }
    // areas conquered by specified clan
    else
    {
        clan = clan_lookup (cln_lookup (cname));

        if (!clan)
        {
            char_printf (ch, "%s: no such clan.\n", cname) ;
            return ;
        }

        output = buf_new(ch->lang);
        buf_printf (output, "{x  Areas conquered by clan %s\n", capitalize (cname));
        buf_printf (output, "----------------------------------------\n");

        count = 0;
        for (pArea = area_first; pArea; pArea = pArea->next)
        {
            if (pArea->clan_owner == clan)
            {
                ++count;
                buf_printf (output, "%s\n", pArea->name);
            }
        }

        buf_printf (output, "----------------------------------------\n");
        buf_printf (output, "%d area(s) total.\n", count);

        if (count)
            page_to_char (buf_string (output), ch);
        else
            char_printf(ch, "There are no areas under control of %s clan.\n", clan->name);
        buf_free (output);
    }
}

// --------------------------------------------------------------------------------
// show to char a list of areas which can be conquered
// --------------------------------------------------------------------------------
static void conquer_free (CHAR_DATA *ch, const char* arg)
{
    AREA_DATA * pArea;
    BUFFER    * output;
    int         count = 0;

    output = buf_new(-1);
    buf_printf (output, "{x  Areas can be conquered\n");
    buf_printf (output, "--------------------------\n");

    for (pArea = area_first; pArea; pArea = pArea->next)
    {
        if (!(pArea->clan_owner) && (pArea->max_guard_count) && pArea->conq_enabled)
        {
            ++count;
            buf_printf (output, "%s\n", pArea->name);
        }
    }

    buf_printf (output, "--------------------------\n");
    buf_printf (output, "%d area(s) total.\n", count);

    if (count)
        page_to_char (buf_string (output), ch);
    else
        char_act("There are no free areas.", ch);
    buf_free (output);
    return;
}

// --------------------------------------------------------------------------------
// place area guards where all of they are lost during attack
// --------------------------------------------------------------------------------
static void conquer_defend (CHAR_DATA *ch, const char *arg)
{
    clan_t    * clan    = NULL ;
    AREA_DATA * curr_area = ch->in_room->area;

    if (IS_IMMORTAL (ch))
    {
        char_act ("'Conquer defend' command is for mortals only.", ch);
        return;
    }

    // only for clan players - just to be sure
    if (!ch->clan)
    {
        char_act ("Area can be defended only by clan players.", ch);
        return;
    }

    clan = clan_lookup (ch->clan) ;

    //area can be only defended by clan_owner
    if (clan != curr_area->clan_owner)
    {
        char_act ("Your clan doesn't own this area. You can't defend it!", ch);
        return;
    }

    //still guards in the area
    if (curr_area->guards_left)
    {
        char_act("This area is already protected!", ch);
        return;
    }

    if (curr_area->curr_attack_time < AREA_DEFENCE_TIME_LIMIT*curr_area->max_guard_count) {
        char_act ("You have no time to place the guards!", ch);
        return;
    }

    // clan_owner has 2 attempts for defence if he has elite area guards in this area
    if (curr_area->defence_attempts == 0)
    {
        if (curr_area->areaguard_rank == AREAGUARD_ELITE)
            curr_area->defence_attempts = 1;
        else
            curr_area->defence_attempts = 0;
    }
    else
        --curr_area->defence_attempts;

    place_area_guards (curr_area, ch->clan);
    curr_area->curr_defence_time = AREA_DEFENCE_TIME_LIMIT*curr_area->max_guard_count;
    char_act ("Now this area is protected again!", ch);
}

// --------------------------------------------------------------------------------
// remove all area guards and place neutral guards
// --------------------------------------------------------------------------------
static void conquer_liberate ( CHAR_DATA *ch, const char *arg )
{
    AREA_DATA * curr_area = ch->in_room->area;
    clan_t    * clan    = NULL ;
    int         i;

    if (IS_IMMORTAL (ch))
    {
        char_act ("'Conquer liberate' command is for mortals only.", ch);
        return;
    }

    // only conquered areas
    clan = clan_lookup (ch->clan) ;

    //area can be only defended by clan_owner
    if (clan != curr_area->clan_owner)
    {
        char_act ("Try to conquer an area before liberate it.", ch);
        return;
    }

    // not during attack
    if (curr_area->curr_attack_time)
    {
        char_act ("This area is under attack and can't be liberated yet.", ch);
        return;
    }

    // and finally - the liberation!
    nuke_area_guards (curr_area);
    curr_area->areaguard_rank = AREAGUARD_STANDARD;
    place_area_guards (curr_area, CLAN_NONE);

    curr_area->clan_owner = CLAN_NONE;
    curr_area->prev_clan_owner = clan;

    curr_area->curr_attack_time = 0;
    curr_area->curr_defence_time = 0;
    curr_area->prev_lock_time = AREA_PREVLOCK_TIME_LIMIT;
    curr_area->time_owned = 0;

    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        rem_altar_raw (curr_area, i);

    clan->conquered_areas -= 1;
    save_conquered_areas();
    char_printf(ch, "Now area '%s' is free for conquest.\n", curr_area->name);
}

// --------------------------------------------------------------------------------
// disable conquest of area in which char is now (Immortal)
// --------------------------------------------------------------------------------
static void conquer_disable  ( CHAR_DATA *ch, const char *arg )
{
    AREA_DATA * curr_area = ch->in_room->area;
    int         i;

    if (!curr_area->conq_enabled)
    {
        char_printf(ch, "Conquest of '%s' is already disabled.\n", curr_area->name);
        return;
    }

    curr_area->conq_enabled = FALSE;

    if (curr_area->clan_owner)
        curr_area->clan_owner->conquered_areas -= 1;
    curr_area->clan_owner = CLAN_NONE;
    curr_area->prev_clan_owner = CLAN_NONE;

    curr_area->curr_attack_time = 0;
    curr_area->curr_defence_time = 0;
    curr_area->prev_lock_time = 0;
    curr_area->time_owned = 0;

    save_area(curr_area);
    nuke_area_guards(curr_area);

    curr_area->areaguard_rank = AREAGUARD_STANDARD;

    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        rem_altar_raw (curr_area, i);

    save_conquered_areas();

    char_printf(ch, "\nClans now {Rcan't{x conquer area %s.\n", curr_area->name);
    char_act("Area was saved.", ch);
    --conq_area_count;
}

// --------------------------------------------------------------------------------
// enable conquest of area in which char is now (Immortal)
// --------------------------------------------------------------------------------
static void conquer_enable  ( CHAR_DATA *ch, const char *arg )
{
    AREA_DATA * curr_area = ch->in_room->area;

    if (curr_area->conq_enabled)
    {
        char_printf(ch, "Conquest of '%s' is already enabled.\n", curr_area->name);
        return;
    }

    if (!curr_area->max_guard_count)
    {
        char_printf(ch, "Too few places for area guards in '%s'.\n", curr_area->name);
        return;
    }

    curr_area->conq_enabled = TRUE;
    save_area(curr_area);
    place_area_guards (curr_area, CLAN_NONE);
    char_printf(ch, "\nClans now {Ccan{x conquer area '%s'.\n", curr_area->name);
    char_act("Area was saved.", ch);
    ++conq_area_count;
}

// --------------------------------------------------------------------------------
// Conquest info for an area
// --------------------------------------------------------------------------------
static void conquer_info ( CHAR_DATA *ch, const char *arg )
{
    AREA_DATA * curr_area = ch->in_room->area;
    int i;

    if (!curr_area->max_guard_count || !curr_area->conq_enabled)
    {
        if (IS_IMMORTAL(ch))
            char_printf(ch, "Area '%s' can't be conquered.\n", curr_area->name);
        else
            char_act("This area can't be conquered.", ch);
        return;
    }

    if (IS_IMMORTAL(ch))
        char_printf(ch, "Conquest info for area '%s'{x: \n", curr_area->name);
    else
        char_act("Conquest info:", ch);
    if (!curr_area->clan_owner)
    {
        char_printf(ch, "This area is neutral.\n");
        char_printf(ch, "Guards left %d from %d.\n", curr_area->guards_left, curr_area->max_guard_count);
    }
    else
    {
        if (IS_IMMORTAL(ch))
            char_printf(ch, "This area belongs to '%s' clan during %d ticks.\n", curr_area->clan_owner->name, curr_area->time_owned);
        else
            if (clan_lookup (ch->clan) == curr_area->clan_owner)
                char_act("This area belongs to your clan!", ch);
            else
                char_printf(ch, "This area belongs to '%s' clan.\n", curr_area->clan_owner->name);
        char_printf(ch, "Guards left %d from %d.\n", curr_area->guards_left, curr_area->max_guard_count);
        switch (curr_area->areaguard_rank)
        {
        case AREAGUARD_STANDARD:
            char_act("Area is protected by standard guards.", ch);
            break;
        case AREAGUARD_VETERAN:
            char_act("Area is protected by veteran guards.", ch);
            break;
        case AREAGUARD_ELITE:
            char_act("Area is protected by elite guards.", ch);
            break;
        }
    }

    if (curr_area->prev_clan_owner)
    {
        if (clan_lookup (ch->clan) == curr_area->prev_clan_owner)
            char_printf(ch, "This area was controlled by your clan and can't be reconquered by it during %d ticks.\n",
               curr_area->prev_lock_time);
        else
            char_printf(ch, "This area was controlled by '%s' clan and can't be reconquered by it during %d ticks.\n",
                curr_area->prev_clan_owner->name, curr_area->prev_lock_time);
    }

    if (curr_area->curr_attack_time)
        char_printf(ch, "Area is attacked and %d ticks left for success.\n", curr_area->curr_attack_time);
    if (curr_area->curr_defence_time)
        char_printf(ch, "Area is defended and %d ticks left for success.\n", curr_area->curr_defence_time);

    // Immortal conquest info
    if (IS_IMMORTAL(ch))
    {
        char_printf(ch, "Guards can be set in rooms: [ {x");
        for ( i=0; i<curr_area->max_guard_count; ++i)
            char_printf(ch, "%d ",curr_area->guard_room_vnums[i]);
        char_printf(ch, "]\n");
    }
}

// --------------------------------------------------------------------------------
// Guards report their condition to a char
// --------------------------------------------------------------------------------
static void conquer_report ( CHAR_DATA *ch, const char *arg )
{
    AREA_DATA * curr_area = ch->in_room->area;
    int         i;
    CHAR_DATA * curr_guard;
    int         percent;

    if  ((!curr_area->conq_enabled) ||
        (!curr_area->max_guard_count)) // let it be - just to be sure
    {
        char_act ("This area can't be conquered. There are no guards here.", ch);
        return;
    }

    if ((clan_lookup (ch->clan) != curr_area->clan_owner) &&
        (!IS_IMMORTAL(ch)))
    {
        char_act ("This area is not under your control.", ch);
        return;
    }

    for (i = 0; i < MAX_GUARD_COUNT; ++i)
    {
        if ((curr_guard = curr_area->areaguards[i]))
        {

            if (curr_guard->max_hit > 0)
                percent = curr_guard->hit * 100 / curr_guard->max_hit;
            else
                percent = -1;
            if (curr_guard->hit == curr_guard->max_hit)
                do_tell_raw(curr_guard, ch, GETMSG ("{CI'm in the best condition!{x", ch->lang));
            else if (percent >= 60)
                do_tell_raw(curr_guard, ch, GETMSG ("{GI'm slightly wounded.{x", ch->lang));
            else if (percent >= 15)
                do_tell_raw(curr_guard, ch, GETMSG ("{rI'm heavily wounded.{x", ch->lang));
            else
                do_tell_raw(curr_guard, ch, GETMSG ("{DDeath{R is near me...{x", ch->lang));
        }
    }
}

// --------------------------------------------------------------------------------
// Checks the world (conquer feature) : vnums existence, act flags etc
// Parameters : all(default)/area/clans
// --------------------------------------------------------------------------------
static void conquer_check ( CHAR_DATA *ch, const char *arg )
{
    BUFFER          * report;
    MOB_INDEX_DATA  * pGuardIndex;
    ROOM_INDEX_DATA * pRoomIndex;
    OBJ_INDEX_DATA  * altar_index;
    AREA_DATA       * curr_area;
    int               i;

    report = buf_new(-1);
    buf_printf(report, "World free energy: %d\n\n", get_world_free_energy());

    // we check all the world
    if (arg[0] == '\0'  || !str_prefix(arg, "all"))
    {
        // area check
        for (curr_area = area_first; curr_area; curr_area = curr_area->next)
        {
            // skip all non-conquerable
            if (!curr_area->max_guard_count || !curr_area->conq_enabled)
                continue;
            for (i = 0; i < curr_area->max_guard_count; ++i)
            {
                if ((curr_area->guard_room_vnums[i] < curr_area->min_vnum) ||
                    (curr_area->guard_room_vnums[i] > curr_area->max_vnum))
                    buf_printf(report, "Area %s: guardroom vnum %d is not within vnum range.\n",
                        curr_area->name, curr_area->guard_room_vnums[i]);
                if (!(pRoomIndex = get_room_index(curr_area->guard_room_vnums[i])))
                    buf_printf(report, "Area %s: guardroom vnum %d does not exist.\n",
                        curr_area->name, curr_area->guard_room_vnums[i]);

            }
        }

        // vnum and act check
        // default standard
        if ((pGuardIndex = get_mob_index(VNUM_AREAGUARD_STANDARD)) == NULL)
            buf_printf(report, "Default standard areaguard vnum does not exist: %d\n", VNUM_AREAGUARD_STANDARD);
        else
        {
            if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                buf_printf(report, "Default standard areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", VNUM_AREAGUARD_STANDARD);
            if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                buf_printf(report, "Default standard areaguard (vnum %d): ACT_NOPURGE does not set.\n", VNUM_AREAGUARD_STANDARD);
        }

        // default veteran
        if ((pGuardIndex = get_mob_index(VNUM_AREAGUARD_VETERAN)) == NULL)
            buf_printf(report, "Default veteran areaguard vnum does not exist: %d\n", VNUM_AREAGUARD_VETERAN);
        else
        {
            if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                buf_printf(report, "Default veteran areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", VNUM_AREAGUARD_VETERAN);
            if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                buf_printf(report, "Default veteran areaguard (vnum %d): ACT_NOPURGE does not set.\n", VNUM_AREAGUARD_VETERAN);
        }

        // default elite
        if ((pGuardIndex = get_mob_index(VNUM_AREAGUARD_ELITE)) == NULL)
            buf_printf(report, "Default elite areaguard vnum does not exist: %d\n", VNUM_AREAGUARD_ELITE);
        else
        {
            if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                buf_printf(report, "Default elite areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", VNUM_AREAGUARD_ELITE);
            if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                buf_printf(report, "Default elite areaguard (vnum %d): ACT_NOPURGE does not set.\n", VNUM_AREAGUARD_ELITE);
        }

        // advisor
        if ((pGuardIndex = get_mob_index(VNUM_ADVISOR)) == NULL)
            buf_printf(report, "Advisor vnum does not exist: %d\n", VNUM_ADVISOR);

        // spy
        if ((pGuardIndex = get_mob_index(VNUM_SPY)) == NULL)
            buf_printf(report, "Spy vnum does not exist: %d\n", VNUM_SPY);

        // transporter
        if ((pGuardIndex = get_mob_index(VNUM_TRANSPORTER)) == NULL)
            buf_printf(report, "Mage-transporter vnum does not exist: %d\n", VNUM_TRANSPORTER);

        // clans check
        for (i = 1; i < clans.nused; ++i)
        {
            if (CLAN(i)->standard_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->standard_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  standard areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->standard_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's standard areaguard (vnum %d): ACT_AREA_GUARD does not set.\n",
                            CLAN(i)->name, CLAN(i)->standard_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's standard areaguard (vnum %d): ACT_NOPURGE does not set.\n",
                            CLAN(i)->name, CLAN(i)->standard_areaguard_vnum);
                }
            }

            if (CLAN(i)->veteran_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->veteran_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  veteran areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's veteran areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's veteran areaguard (vnum %d): ACT_NOPURGE does not set.\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum);

                }
            }

            if (CLAN(i)->elite_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->elite_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  elite areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's elite areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's elite areaguard (vnum %d): ACT_NOPURGE does not set.\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum);
                }
            }
        }

        // altars check
        for (i = 0; i < MAX_ALTAR_COUNT; ++i)
            if ((altar_index = get_obj_index(altar_vnums[i])) == NULL)
            {
                 buf_printf(report, check_msg[i]);
                 buf_printf(report, " VNUM = %d.\n", altar_vnums[i]);
            }
        // priest check
        if ((pGuardIndex = get_mob_index(VNUM_PRIEST)) == NULL)
            buf_printf(report, "\nDefault priest vnum does not exist: %d\n", VNUM_PRIEST);
    }

    // we check an area in which char is now
    else if (!str_prefix(arg, "area"))
    {
        curr_area = ch->in_room->area;
        if (curr_area->max_guard_count && curr_area->conq_enabled)
            for (i = 0; i < curr_area->max_guard_count; ++i)
            {
                if ((curr_area->guard_room_vnums[i] < curr_area->min_vnum) ||
                    (curr_area->guard_room_vnums[i] > curr_area->max_vnum))
                    buf_printf(report, "Area %s: guardroom vnum %d is not within vnum range.\n",
                        curr_area->name, curr_area->guard_room_vnums[i]);
                if (!(pRoomIndex = get_room_index(curr_area->guard_room_vnums[i])))
                    buf_printf(report, "Area %s: guardroom vnum %d does not exist.\n",
                        curr_area->name, curr_area->guard_room_vnums[i]);

            }
    }

    // we check only the clans
    else if (!str_prefix(arg, "clans"))
    {
        for (i = 1; i < clans.nused; ++i)
        {
            if (CLAN(i)->standard_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->standard_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  standard areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->standard_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's standard areaguard (vnum %d): ACT_AREA_GUARD does not set.\n",
                            CLAN(i)->name, CLAN(i)->standard_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's standard areaguard (vnum %d): ACT_NOPURGE does not set.\n",
                            CLAN(i)->name, CLAN(i)->standard_areaguard_vnum);
                }
            }

            if (CLAN(i)->veteran_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->veteran_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  veteran areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's veteran areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's veteran areaguard (vnum %d): ACT_NOPURGE does not set.\n", CLAN(i)->name, CLAN(i)->veteran_areaguard_vnum);

                }
            }

            if (CLAN(i)->elite_areaguard_vnum)
            {
                if ((pGuardIndex = get_mob_index(CLAN(i)->elite_areaguard_vnum)) == NULL)
                    buf_printf(report, "Clan %s:  elite areaguard vnum does not exist: %d\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum);
                else
                {
                    if (!IS_SET(pGuardIndex->act, ACT_AREA_GUARD))
                        buf_printf(report, "Clan %s's elite areaguard (vnum %d): ACT_AREA_GUARD does not set.\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum );
                    if (!IS_SET(pGuardIndex->act, ACT_NOPURGE))
                        buf_printf(report, "Clan %s's elite areaguard (vnum %d): ACT_NOPURGE does not set.\n", CLAN(i)->name, CLAN(i)->elite_areaguard_vnum);
                }
            }
        }
    }
    else
        char_printf(ch, "Invalid option: %s.\n", arg);
    page_to_char(buf_string(report), ch);
    buf_free(report);
}

// --------------------------------------------------------------------------------
// Clan rating in the conquest domain
// --------------------------------------------------------------------------------
static void conquer_rating ( CHAR_DATA *ch, const char *arg )
{
    int      i;
    BUFFER * rating;

    char_act ("Sorry. Not sorted and not well-designed yet :).", ch);

    rating = buf_new(ch->lang);
    if (IS_IMMORTAL(ch))
    {
        buf_printf(rating, "        Clan         | Areas |  Rating  |  Energy\n");
        buf_printf(rating, "---------------------------------------------------\n");
    }
    else
    {
        buf_printf(rating, "        Clan         | Areas |  Rating\n");
        buf_printf(rating, "----------------------------------------\n");
    }

    for (i = 1; i < clans.nused; ++i)
        if (IS_IMMORTAL(ch))
        {
            buf_printf(rating, "%-20s | %-5d | %-8d | %-8d\n",
                CLAN(i)->name, CLAN(i)->conquered_areas, CLAN(i)->areacontrol_rating, CLAN(i)->energy);
        }
        else
        {
            buf_printf(rating, "%-20s | %-5d | %-8d\n",
                CLAN(i)->name, CLAN(i)->conquered_areas, CLAN(i)->areacontrol_rating);
        }
    page_to_char(buf_string(rating), ch);
    buf_free(rating);
}

// --------------------------------------------------------------------------------
// reset all conquest info
// --------------------------------------------------------------------------------
static void conquer_reset(CHAR_DATA *ch, const char* arg)
{
    int      i;
    AREA_DATA * curr_area;

    // free energy reset
    put_world_free_energy(0);
    char_act("World free energy was reset.\n", ch);
    // clan reset
    for (i = 1; i < clans.nused; ++i)
    {
        CLAN(i)->areacontrol_rating = 0;
        CLAN(i)->energy = 0;
        CLAN(i)->conquered_areas = 0;
    }
    char_act("Clan conquest info was reset.\n", ch);

    // area reset
    char_act("Area reset:", ch);

    for (curr_area = area_first; curr_area; curr_area = curr_area->next)
    {
        // skip all non-conquerable
        if (!curr_area->max_guard_count || !curr_area->conq_enabled)
            continue;
        // liberate area
        nuke_area_guards (curr_area);
        curr_area->areaguard_rank = AREAGUARD_STANDARD;
        place_area_guards (curr_area, CLAN_NONE);

        curr_area->clan_owner = CLAN_NONE;
        curr_area->prev_clan_owner = CLAN_NONE;

        curr_area->curr_attack_time = 0;
        curr_area->curr_defence_time = 0;
        curr_area->prev_lock_time = 0;
        curr_area->time_owned = 0;

        for (i = 0; i < MAX_ALTAR_COUNT; ++i)
            rem_altar_raw (curr_area, i);
        char_printf(ch, "   %s\n", curr_area->name);
    }
    // and finally save it all
    save_conquered_areas ();
    char_act("\nInfo saved.\nOk.", ch);
}

// --------------------------------------------------------------------------------
// Immortal's command: allow clan to conquer areas
// --------------------------------------------------------------------------------
static void conquer_allow    ( CHAR_DATA *ch, const char *arg )
{
    char        cname [MAX_INPUT_LENGTH] ;
    clan_t    * clan = NULL ;

    one_argument (arg, cname, sizeof(cname)) ;
    clan = clan_lookup (cln_lookup (cname));

    if (!clan)
    {
        char_printf (ch, "%s: no such clan.\n", cname) ;
        return ;
    }

    if (IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
    {
        char_act ("But this clan already can conquer areas.", ch);
        return;
    }

    SET_BIT (clan->conq_flags, CLAN_CAN_CONQUER);
    char_printf (ch, "Now clan %s can conquer areas.\n", capitalize (cname));
    save_conquered_areas ();
}

// --------------------------------------------------------------------------------
// Clan leaders' command: clan can't conquer areas no more
// --------------------------------------------------------------------------------
static void conquer_refuse   ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    if (!IS_IMMORTAL (ch)) // for clan leader
    {
        clan = clan_lookup (ch->clan);

        if (!clan ) // just to be sure
            return;
        if (!IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
        {
            char_act ("But your clan has already refused area conquest.", ch);
            return;
        }
        char_act ("Now your clan can't conquer areas.", ch);
    }
    else // for immortals
    {
        char cname [MAX_INPUT_LENGTH] ;

        one_argument (arg, cname, sizeof(cname)) ;
        clan = clan_lookup (cln_lookup (cname));

        if (!clan)
        {
            char_printf (ch, "%s: no such clan.\n", cname) ;
            return ;
        }
        if (!IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
        {
            char_printf (ch, "But clan %s already can't conquer areas.\n", capitalize (cname));
            return;
        }
        char_printf (ch, "Now clan %s can't conquer areas.\n", capitalize (cname));
    }

    REMOVE_BIT (clan->conq_flags, CLAN_CAN_CONQUER);
    save_conquered_areas ();
}

// ********************************************************************************
//            manage commands - manage conquered area (several commands)
// ********************************************************************************

// --------------------------------------------------------------------------------
// do_manage - function called from interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_manage)
{
    if (IS_NPC(ch))
        return;

    if (!ch->clan)
    {
        char_act("{CManage{x command is for clan players only!", ch);
        return;
    }

    if (!parse_command(ch, argument, manage_cmd_table))
    {
        show_command_list (ch, manage_cmd_table);
        char_puts (".\nUse {CHELP MANAGE{x for more information.\n", ch);
    }
}

// --------------------------------------------------------------------------------
// says to char if his clan can upgrade an area in which char is
// --------------------------------------------------------------------------------
static void manage_check ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;
    AREA_DATA * curr_area = ch->in_room->area;
    int upgrade_count, i, max_upgr_in_area;

    clan = clan_lookup (ch->clan);

    if (!clan)
        return;

    char_printf (ch, "Your clan has {C%d{x energy.\n", clan->energy);

    if (clan != ch->in_room->area->clan_owner)
    {
        char_act("This area does not belong to your clan.", ch);
        return;
    }

    switch (curr_area->areaguard_rank)
    {
    case AREAGUARD_STANDARD:
        char_act("Area is protected by standard guards.", ch);
        break;
    case AREAGUARD_VETERAN:
        char_act("Area is protected by veteran guards.", ch);
        break;
    case AREAGUARD_ELITE:
        char_act("Area is protected by elite guards.", ch);
        break;
    }

    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        if (curr_area->altars[i])
            char_act(altar_msg[i], ch);

    // calculate max_upgrades.
    if (IS_SET (clan->flags, CLAN_HATE_MAGIC))
    {
        if (ch->in_room->area->clan == ch->clan)
            max_upgr_in_area = MAX_UPGRADES_CLAN_BR;
        else
            max_upgr_in_area = MAX_UPGRADES_BR;
    }
    else
    {
        if (ch->in_room->area->clan == ch->clan)
            max_upgr_in_area = MAX_UPGRADES_CLAN;
        else
            max_upgr_in_area = MAX_UPGRADES;
    }

    if ((curr_area->time_owned / (apply_count (curr_area) + 1)) < TIME_TO_UPGRADE)
        char_act ("You can't upgrade this area yet.", ch);
    else
    {
        upgrade_count = (curr_area->time_owned / TIME_TO_UPGRADE) - apply_count (curr_area);
        char_printf (ch, "Right now your clan can do %d upgrade(s) in this area.\n",
            UMIN (upgrade_count, max_upgr_in_area - apply_count (curr_area)));

    }
}

// --------------------------------------------------------------------------------
// upgrade area guards STANDARD -> VETERAN -> ELITE
// --------------------------------------------------------------------------------
static void manage_upgrade ( CHAR_DATA *ch, const char *arg )
{
    clan_t    * clan;
    AREA_DATA * curr_area = ch->in_room->area;
    int         cost, i, guard_vnum, count, percent;
    CHAR_DATA * guard;
    MOB_INDEX_DATA * pGuardIndex;
    ROOM_INDEX_DATA * pRoomIndex;


    clan = clan_lookup (ch->clan);
    if (clan != ch->in_room->area->clan_owner)
    {
        char_act ("Your clan doesn't own this area. You can't manage it!", ch);
        return;
    }

    if (curr_area->areaguard_rank == AREAGUARD_ELITE)
    {
        char_act ("Guards in this area are already as strong as possible.", ch);
        return;
    }

    if ((curr_area->time_owned / (apply_count (curr_area) + 1)) < TIME_TO_UPGRADE)
    {
        char_act ("You can't upgrade this area yet.", ch);
        return;
    }

    for (i = 0; i < MAX_GUARD_COUNT; ++i)
    {
        if (curr_area->areaguards[i] && curr_area->areaguards[i]->fighting)
        {
            char_act ("At least one of the guards in this area is fighting now and can't leave.", ch);
            char_act ("You can't upgrade this area yet.", ch);
            return;
        }
    }

    // upgrade cost = 4 * guard count * new upkeep cost
    cost = 4 * upkeep_cost_guard[curr_area->areaguard_rank + 1] * curr_area->guards_left;
    if (clan->energy < cost)
    {
        char_printf (ch, "Upgrade costs %d energy, but your clan only has %d.\n", cost, clan->energy);
        return;
    }

    // all is OK. Upgrading!

    // firstly select the mob will be placed as areaguard
    // if vnum specified for clan, use it. Otherwise use default one
    switch (curr_area->areaguard_rank + 1)
    {
    case AREAGUARD_VETERAN:
        guard_vnum = clan->veteran_areaguard_vnum ? clan->veteran_areaguard_vnum : VNUM_AREAGUARD_VETERAN;
        break;
    case AREAGUARD_ELITE:
        guard_vnum = clan->elite_areaguard_vnum ? clan->elite_areaguard_vnum : VNUM_AREAGUARD_ELITE;
        break;
    default :
        log_printf ("[**BUG**] in manage upgrade: invalid guard rank %d.", curr_area->areaguard_rank + 1);
        return;
    }

    // place new area guards instead of old ones
    // hitpoint percentage is the same, but affects are new
    count = 0;
    if ((pGuardIndex = get_mob_index(guard_vnum)) == NULL)
    {
        log_printf ("[**BUG**] in manage upgrade: can't create mob %d.", guard_vnum);
        return;
    }

    for (i = 0; i < MAX_GUARD_COUNT; ++i)
    {
        if (!curr_area->areaguards[i])
            continue;

        guard = create_mob(pGuardIndex);
        guard->clan = ch->clan;

        pRoomIndex = curr_area->areaguards[i]->in_room;
        percent = ((100 * curr_area->areaguards[i]->hit) / UMAX(1, curr_area->areaguards[i]->max_hit));

        act("Obeying someone's command $n leaves.",
            curr_area->areaguards[i],NULL,NULL,TO_ROOM);
        extract_char(curr_area->areaguards[i], TRUE);

        guard->hit = (guard->max_hit * percent) / 100;
        curr_area->areaguards[i] = guard;

        char_to_room(guard, pRoomIndex);
        act("Instead him $n comes to protect this area!",guard,NULL,NULL,TO_ROOM);
        ++count;
    }

    curr_area->areaguard_rank += 1;
    clan->energy -= cost;
    save_conquered_areas ();
    char_act ("Now this area is protected by more powerful guards.", ch);
}

// --------------------------------------------------------------------------------
// downgrade area guards ELITE -> VETERAN -> STANDARD
// can be used to decrease upkeep manually or automatically
// upkeep * 2 per guard energy is added to the clan energy pool
// --------------------------------------------------------------------------------
static void manage_downgrade ( CHAR_DATA *ch, const char *arg )
{
    clan_t    * clan;
    AREA_DATA * curr_area = ch->in_room->area;
    int         cost, i, guard_vnum, count, percent;
    CHAR_DATA * guard;
    MOB_INDEX_DATA * pGuardIndex;
    ROOM_INDEX_DATA * pRoomIndex;


    clan = clan_lookup (ch->clan);
    if (clan != ch->in_room->area->clan_owner)
    {
        char_act ("Your clan doesn't own this area. You can't manage it!", ch);
        return;
    }

    if (curr_area->areaguard_rank == AREAGUARD_STANDARD)
    {
        char_act ("Guards in this area are just recriutred.", ch);
        return;
    }

    for (i = 0; i < MAX_GUARD_COUNT; ++i)
    {
        if (curr_area->areaguards[i] && curr_area->areaguards[i]->fighting)
        {
            char_act ("At least one of the guards in this area is fighting now and can't leave.", ch);
            char_act ("You can't do this yet.", ch);
            return;
        }
    }

    // all is OK. Downgrading!

    // firstly select the mob will be placed as areaguard
    // if vnum specified for clan, use it. Otherwise use default one
    switch (curr_area->areaguard_rank - 1)
    {
    case AREAGUARD_STANDARD:
        guard_vnum = clan->standard_areaguard_vnum ? clan->standard_areaguard_vnum : VNUM_AREAGUARD_STANDARD;
        break;
    case AREAGUARD_VETERAN:
        guard_vnum = clan->veteran_areaguard_vnum ? clan->veteran_areaguard_vnum : VNUM_AREAGUARD_VETERAN;
        break;
    default :
        log_printf ("[**BUG**] in manage upgrade: invalid guard rank %d.", curr_area->areaguard_rank + 1);
        return;
    }

    // place new area guards instead of old ones
    // hitpoint percentage is the same, but affects are new
    count = 0;
    if ((pGuardIndex = get_mob_index(guard_vnum)) == NULL)
    {
        log_printf ("[**BUG**] in manage upgrade: can't create mob %d.", guard_vnum);
        return;
    }

    for (i = 0; i < MAX_GUARD_COUNT; ++i)
    {
        if (!curr_area->areaguards[i])
            continue;

        guard = create_mob(pGuardIndex);
        guard->clan = ch->clan;

        pRoomIndex = curr_area->areaguards[i]->in_room;
        percent = ((100 * curr_area->areaguards[i]->hit) / UMAX(1, curr_area->areaguards[i]->max_hit));

        act("Obeying someone's command $n leaves.",
            curr_area->areaguards[i],NULL,NULL,TO_ROOM);
        extract_char(curr_area->areaguards[i], TRUE);

        guard->hit = (guard->max_hit * percent) / 100;
        curr_area->areaguards[i] = guard;

        char_to_room(guard, pRoomIndex);
        act("Instead him $n comes to protect this area!",guard,NULL,NULL,TO_ROOM);
        ++count;
    }
    cost = 2 * upkeep_cost_guard[curr_area->areaguard_rank + 1] * curr_area->guards_left;
    curr_area->areaguard_rank -= 1;
    clan->energy += cost;
    save_conquered_areas ();
    char_act ("Now this area is protected by less powerful guards.", ch);
}

// --------------------------------------------------------------------------------
// add an altar in the area (in room in which a char currently is, but only one
//                           altar in each room). Some rooms can't be choosen.
// see altar descriptions above
// --------------------------------------------------------------------------------
static void manage_add_altar ( CHAR_DATA *ch, const char *arg )
{
    clan_t    * clan;
    AREA_DATA * curr_area = ch->in_room->area;
    int         ret_code;
    int         pos = 0;

    clan = clan_lookup (ch->clan);

    if (!clan) // just to be sure
    {
        char_act ("Only clan characters can manage an area!", ch);
        return;
    }

    // clanarea can be managed any time!
    if (ch->clan != ch->in_room->area->clan)
    {
        if (clan != ch->in_room->area->clan_owner)
        {
            char_act ("Your clan doesn't own this area. You can't manage it!", ch);
            return;
        }

        if ((curr_area->time_owned / (apply_count (curr_area) + 1)) < TIME_TO_UPGRADE)
        {
            char_act ("You can't upgrade this area yet.", ch);
            return;
        }
    }

    // must specify type
    if (arg[0] == '\0')
    {
        char_act ("You can only place altars of {YGREED{x, {MMAGIC{x, {RHASTE{x, {CTRAVEL{x, {gTRANQUILITY{x,", ch);
        char_act ("{ySUMMONING{x, {rIMMOBILITY{x, {GREGENERATION{x, {CEXPERIENCE{x,{mNEGATION{x or {DOUTPOST{x.", ch);
        return;
    }

    // clan must have enough energy
    if (clan->energy < upkeep_cost_altar * 4)
    {
        char_printf(ch, "Your clan must have %d energy to do this but has only %d.\n",
            (upkeep_cost_altar * 4), clan->energy);
        char_act ("You can't establish an altar yet.", ch);
        return;
    }

    // choose altar type
    if (!str_prefix(arg, "greed"))
        pos = ALTAR_GREED_POS;
    else if (!str_prefix(arg, "haste"))
        pos = ALTAR_HASTE_POS;
    else if (!str_prefix(arg, "magic"))
        pos = ALTAR_MAGIC_POS;
    else if (!str_prefix(arg, "travel"))
        pos = ALTAR_TRAVEL_POS;
    else if (!str_prefix(arg, "tranquility"))
        pos = ALTAR_TRANQUILITY_POS;
    else if (!str_prefix(arg, "summoning"))
        pos = ALTAR_SUMMONING_POS;
    else if (!str_prefix(arg, "immobility"))
        pos = ALTAR_IMMOBILITY_POS;
    else if (!str_prefix(arg, "regeneration"))
        pos = ALTAR_REGENERATION_POS;
    else if (!str_prefix(arg, "experience"))
        pos = ALTAR_EXPERIENCE_POS;
    else if (!str_prefix(arg, "negation"))
        pos = ALTAR_NEGATION_POS;
    else if (!str_prefix(arg, "outpost"))
        pos = OUTPOST_POS;
    else
    {
        char_printf(ch, "You can't place here an altar of %s.\n", arg);
        char_act ("You can only place altars of {YGREED{x, {MMAGIC{x, {RHASTE{x, {CTRAVEL{x, {gTRANQUILITY{x,", ch);
        char_act ("{ySUMMONING{x, {rIMMOBILITY{x, {GREGENERATION{x, {CEXPERIENCE{x,{mNEGATION{x or {DOUTPOST{x.", ch);
        return;
    }

    // can we build such altar here?
    // BRs
    if (IS_SET (clan->flags, CLAN_HATE_MAGIC) && IS_SET (altar_flags[pos], ALTAR_NO_BR))
    {
        char_act ("No, no, no! You hate magic too much to establish this altar!", ch);
        return;
    }
    // clanarea
    if ((ch->clan == ch->in_room->area->clan) && IS_SET (altar_flags[pos], ALTAR_NO_CLANAREA))
    {
        char_act ("No, no, no! You can't build an altar of this type in your clanarea!", ch);
        return;
    }

    // all is OK
    ret_code = add_altar_raw(ch->in_room->vnum, ch->clan, pos);
    if (ret_code)
        char_act (err_msg[ret_code], ch);
    else
    {
        clan->energy -= upkeep_cost_altar * 4;
        save_conquered_areas ();
    }
}

// --------------------------------------------------------------------------------
// add an altar of specified type in specified room. Called from manage_add_altar
// --------------------------------------------------------------------------------
static int add_altar_raw (int room_vnum, int clan, int alt_type)
{
    int i;
    OBJ_INDEX_DATA  * new_altar_index;
    MOB_INDEX_DATA  * new_priest_index;
    ROOM_INDEX_DATA * room;

    if ((room = get_room_index(room_vnum)) == NULL)
    {
        log_printf ("[**BUG**] in manage_add_altar: invalid room vnum %d.\n", room_vnum);
        return RC_BUG;
    }

    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        if (room->area->altars[i] && (room->area->altars[i]->in_room == room))
            return RC_ALTAR_ALREADY_IN_ROOM;

    if (room->area->altars[alt_type])
        return RC_ALTAR_ALREADY_EXISTS;

    if (IS_SET(room->room_flags, ROOM_SAFE | ROOM_PEACE |
                                 ROOM_PRIVATE | ROOM_SOLITARY))

        return RC_INVALID_PLACE;

    // all is OK. (amount of energy was checked in manage_add_altar)
    if ((new_altar_index = get_obj_index(altar_vnums[alt_type])) == NULL)
    {
        log_printf ("[**BUG**] in manage_add_altar: can't create obj %d.\n", altar_vnums[alt_type]);
        return RC_BUG;
    }

    if ((new_priest_index = get_mob_index(VNUM_PRIEST)) == NULL)
    {
        log_printf ("[**BUG**] in manage_add_altar: can't create mob %d.\n", VNUM_PRIEST);
        return RC_BUG;
    }

    room->area->altars[alt_type]  = create_obj(new_altar_index, 100);
    room->area->priests[alt_type] = create_mob(new_priest_index);
    obj_to_room(room->area->altars[alt_type], room);
    char_to_room(room->area->priests[alt_type], room);
    act("Suddenly $n comes from nowhere!", room->area->priests[alt_type], NULL,NULL,TO_ROOM);
    act("He chants and a strange looking altar appears in the room.", room->area->priests[alt_type], NULL, NULL, TO_ROOM);
    room->area->priests[alt_type]->clan = clan;

    return RC_OK;
}

// --------------------------------------------------------------------------------
// remove an altar of specified type in specified room.
// Called from manage_rem_altar, areacontrol_update, conquer_disable, conquer_liberate
// --------------------------------------------------------------------------------
static int rem_altar_raw (AREA_DATA *pArea, int altar_type)
{
    if (!pArea->altars[altar_type])
        return RC_BUG;
    act ("$n slowly dissolves in the air and $p tumbles down.",
        pArea->priests[altar_type], pArea->altars[altar_type],
        NULL,TO_ROOM);
    extract_obj (pArea->altars[altar_type]);
    pArea->altars[altar_type] = NULL;
    extract_char (pArea->priests[altar_type], TRUE);
    pArea->altars[altar_type] = NULL;

    save_conquered_areas ();
    return RC_OK;
}

// --------------------------------------------------------------------------------
// removes an altar in the room in which char is
// char's clan receives some energy
// --------------------------------------------------------------------------------
static void manage_rem_altar ( CHAR_DATA *ch, const char *arg )
{
    clan_t    * clan;
    AREA_DATA * curr_area = ch->in_room->area;
    int         i, altar_type = 0;
    bool        found;

    clan = clan_lookup (ch->clan);

    if (!clan) // just to be sure
    {
        char_act ("Only clan characters can manage an area!", ch);
        return;
    }

    if (clan != ch->in_room->area->clan_owner)
    {
        char_act ("Your clan doesn't own this area. You can't manage it!", ch);
        return;
    }

    found = FALSE;
    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
    {
        if (!curr_area->altars[i])
            continue;

        if (curr_area->altars[i]->in_room == ch->in_room)
        {
            found = TRUE;
            altar_type = i;
            break;
        }
    }

    if (!found)
    {
        char_act ("But there are no altars in this room!", ch);
        return;
    }

    // finally demolish an altar
    act ("$n slowly dissolves in the air and $p tumbles down.",
        ch->in_room->area->priests[altar_type],
        ch->in_room->area->altars[altar_type],
        NULL,TO_ROOM);
    extract_obj (ch->in_room->area->altars[altar_type]);
    ch->in_room->area->altars[altar_type] = NULL;
    extract_char (ch->in_room->area->priests[altar_type], TRUE);
    ch->in_room->area->altars[altar_type] = NULL;

    curr_area->clan_owner->energy += 2 * upkeep_cost_altar;
    save_conquered_areas ();
}
// ********************************************************************************
//            advisor commands - clan advisor communications (several commands)
// ********************************************************************************

// --------------------------------------------------------------------------------
// do_advisor - function called from interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_advisor)
{
    if (!parse_command(ch, argument, advisor_cmd_table))
    {
        show_command_list (ch, advisor_cmd_table);
        char_puts (".\nUse {CHELP ADVISOR{x for more information.\n", ch);
    }
}

static void advisor_check ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    if (!IS_SET(clan->conq_flags, CLAN_WITH_ADVISOR))
        char_act("Your clan has not an advisor yet.", ch);
    else
        do_tell_raw(clan->advisor, ch, GETMSG ("What do you want?", ch->lang));
}

static void advisor_hire ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    // already hired
    if (IS_SET(clan->conq_flags, CLAN_WITH_ADVISOR))
    {
        char_act("Your clan has already hired an advisor.", ch);
        return;
    }

    // not enough energy
    if (clan->energy < upkeep_cost_advisor * 4)
    {
        char_printf(ch, "Your clan must pay %d energy to hire but posesses only %d!",
            upkeep_cost_advisor * 4, clan->energy);
        return;
    }

    // all is OK!
    SET_BIT(clan->conq_flags, CLAN_WITH_ADVISOR);
    clan->energy -= upkeep_cost_advisor * 4;
    do_clan (clan->advisor, "I'm ready to help!");
    save_conquered_areas();
}

static void advisor_dismiss ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    // not hired
    if (!IS_SET(clan->conq_flags, CLAN_WITH_ADVISOR))
    {
        char_act("But your clan has no advisor yet.", ch);
        return;
    }
    // all is OK!
    REMOVE_BIT(clan->conq_flags, CLAN_WITH_ADVISOR);
    clan->energy += upkeep_cost_advisor * 2;
    do_clan (clan->advisor, "I hope I've served you well... *wave!");
    save_conquered_areas();
}

static void advisor_upkeep ( CHAR_DATA *ch, const char *arg )
{
    DESCRIPTOR_DATA * d;
    CLAN_MEMBER     * member;
    BUFFER          * output ;
    int               upkeep = 0,
                      upkeep_altars = 0,
                      upkeep_guards = 0,
                      area_energy = 0,
                      total_sub = 0,      // subtracted from clan's energy
                      total_add = 0,      // added: from areas, clanarea and
                      i;
    clan_t          * clan;
    AREA_DATA       * area;

    clan = clan_lookup (ch->clan);
    if (!clan)
        return;

    output = buf_new (ch->lang) ;
    buf_add (output, "{x    Your clan's energy statistics:\n");

    // upkeep for each member (lower level - higher upkeep)
    for (member = clan->member_list ; member ; member = member->next)
        (member->level >= 91) ? (upkeep += upkeep_cost_clanmember) :
            (upkeep += (11 - member->level/10) * upkeep_cost_clanmember);

    buf_printf (output, "Clan members     : %d\n", upkeep);
    total_sub += upkeep;

    // upkeep for clan applies
    if (IS_SET(clan->conq_flags, CLAN_WITH_ADVISOR))
    {
        total_sub += upkeep_cost_advisor;
        buf_printf (output, "Advisor          : %d\n", upkeep_cost_advisor);
    }
    if (IS_SET(clan->conq_flags, CLAN_WITH_SPY))
    {
        total_sub += upkeep_cost_spy;
        buf_printf (output, "Spy              : %d\n", upkeep_cost_spy);
    }
    if (IS_SET(clan->conq_flags, CLAN_WITH_TRANSPORTER))
    {
        total_sub += upkeep_cost_transporter;
        buf_printf (output, "Mage             : %d\n", upkeep_cost_transporter);
    }

    for (area = area_first; area != NULL; area = area->next)
    {
        if (area->clan_owner == clan)
        {
            // areaguards
            for (i = 0; i < MAX_GUARD_COUNT; ++i)
                if (area->areaguards[i])
                    upkeep_guards += upkeep_cost_guard[area->areaguard_rank];
            // altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (area->altars[i])
                    upkeep_altars += upkeep_cost_altar;
            // added enegry (arithmetic mean)
            area_energy += area->explorable_rooms * ENERGY_OUTPUT_ROOM_MOD / 3;
        }
        // special case: clanarea
        if (clan_lookup (area->clan) == clan)
        {
            // only altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (area->altars[i])
                    upkeep_altars += upkeep_cost_altar;

        }
    }

    total_sub += (upkeep_altars + upkeep_guards);
    buf_printf (output, "Altars           : %d\n", upkeep_altars);
    buf_printf (output, "Guards           : %d\n", upkeep_guards);

    buf_add (output,    "---------------------------------\n");
    buf_printf (output, "Total spent      : %d\n\n", total_sub);

    // each clan has energy to support 5 HEROes, 2 level 80 members and 1 level 70
    // for free
    // upkeep -= (5 * upkeep_cost_clanmember +     // HERO
    //            2 * upkeep_cost_clanmember * 3 + // level 80
    //            1 * upkeep_cost_clanmember * 4); // level 70
    total_add += upkeep_cost_clanmember * 15;

    buf_printf (output, "Clan area        : %d\n", upkeep_cost_clanmember * 15);
    buf_printf (output, "Conquered areas  : %d (%d average)\n",
        clan->area_energy_this_tick, area_energy);
    buf_printf (output, "Earned           : %d\n", clan->earned_energy_this_tick);

    total_add += (clan->area_energy_this_tick + clan->earned_energy_this_tick);

    buf_add (output,    "---------------------------------\n");
    buf_printf (output, "Total earned     : %d\n\n", total_add);

    buf_add (output,    "=================================\n");
    buf_printf (output, "Balance this tick: %d\n\n", total_add - total_sub);

    // small bonus for active clanmembers
    upkeep = 0;
    for (d = descriptor_list; d != NULL; d = d->next)
    {
        CHAR_DATA *ch;
        clan_t * ch_clan;

        if (d->connected != CON_PLAYING)
            continue;

        ch = d->character;
        ch_clan = clan_lookup (ch->clan);
        if (ch_clan != clan)
            continue;
        upkeep += upkeep_cost_clanmember / 2;
    }

    char_act(buf_string (output), ch);
    buf_free (output);
}

static void advisor_info ( CHAR_DATA *ch, const char *arg )
{
    char_act("advisor_info: not implemented yet.", ch);
}

// ********************************************************************************
//            spy commands - clan spy communications (several commands)
// ********************************************************************************

// --------------------------------------------------------------------------------
// do_spy - function called from interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_spy)
{
    if (!parse_command(ch, argument, spy_cmd_table))
    {
        show_command_list (ch, spy_cmd_table);
        char_puts (".\nUse {CHELP SPY{x for more information.\n", ch);
    }
}

static void spy_check ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    if (!IS_SET(clan->conq_flags, CLAN_WITH_SPY))
        char_act("Your clan has not a spy yet.", ch);
    else
        do_tell_raw(clan->spy, ch, GETMSG ("What do you want?", ch->lang));
}

static void spy_hire ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    // already hired
    if (IS_SET(clan->conq_flags, CLAN_WITH_SPY))
    {
        char_act("Your clan has already hired a spy.", ch);
        return;
    }

    // not enough energy
    if (clan->energy < upkeep_cost_spy * 4)
    {
        char_printf(ch, "Your clan must pay %d energy to hire but posesses only %d!",
            upkeep_cost_spy * 4, clan->energy);
        return;
    }

    // all is OK!
    SET_BIT(clan->conq_flags, CLAN_WITH_SPY);
    clan->energy -= upkeep_cost_spy * 4;
    do_clan (clan->spy, "I'm ready to help!");
    save_conquered_areas();
}

static void spy_dismiss ( CHAR_DATA *ch, const char *arg )
{
    clan_t * clan;

    clan = clan_lookup(ch->clan);

    if (!clan)
        return;

    // not hired
    if (!IS_SET(clan->conq_flags, CLAN_WITH_SPY))
    {
        char_act("But your clan has no spy yet.", ch);
        return;
    }
    // all is OK!
    REMOVE_BIT(clan->conq_flags, CLAN_WITH_SPY);
    clan->energy += upkeep_cost_spy * 2;
    do_clan (clan->spy, "I hope I've served you well... *wave!");
    save_conquered_areas();
}

static void spy_report ( CHAR_DATA *ch, const char *arg )
{
    char_act("spy_report: not implemented yet.", ch);
}

// ********************************************************************************
//            cconfig commands - configuration (several commands)
// ********************************************************************************

// --------------------------------------------------------------------------------
// do_cconfig - function called from interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_cconfig)
{
    if (!parse_command(ch, argument, cconfig_cmd_table))
    {
        show_command_list (ch, cconfig_cmd_table);
        char_puts (".\nUse {CHELP CCONFIG{x for more information.\n", ch);
    }
}

// --------------------------------------------------------------------------------
// show all parameter values
// --------------------------------------------------------------------------------
static void cconfig_info ( CHAR_DATA *ch, const char *arg )
{
    BUFFER *output;

    output = buf_new (ch->lang);

    buf_printf(output, "{x    {GArea conquest parameters.{x\n");
    buf_printf(output, "-------------------------------------\n");
    buf_printf(output, "Time settings:\n");
    buf_printf(output, "{x    Between upgrades  : %d\n", TIME_TO_UPGRADE);
    buf_printf(output, "{x    Attack time       : %d (per guard)\n", AREA_ATTACK_TIME_LIMIT);
    buf_printf(output, "{x    Defence time      : %d (per guard)\n", AREA_DEFENCE_TIME_LIMIT);
    buf_printf(output, "{x    Looser lock time  : %d\n", AREA_PREVLOCK_TIME_LIMIT);

    buf_printf(output, "-------------------------------------\n");
    buf_printf(output, "Upkeep costs:\n");
    buf_printf(output, "{x    Standard guard    : %d\n", upkeep_cost_guard[0]);
    buf_printf(output, "{x    Veteran guard     : %d\n", upkeep_cost_guard[1]);
    buf_printf(output, "{x    Elite guard       : %d\n\n", upkeep_cost_guard[2]);
    buf_printf(output, "{x    Clanmember (HERO) : %d\n", upkeep_cost_clanmember);
    buf_printf(output, "{x    Altar             : %d\n\n", upkeep_cost_altar);
    buf_printf(output, "{x    Advisor           : %d\n", upkeep_cost_advisor);
    buf_printf(output, "{x    Spy               : %d\n", upkeep_cost_spy);
    buf_printf(output, "{x    Mage-transporter  : %d\n", upkeep_cost_transporter);

    buf_printf(output, "-------------------------------------\n");
    buf_printf(output, "Energy output/tick    : %d (per room)\n", ENERGY_OUTPUT_ROOM_MOD);

    page_to_char(buf_string(output), ch);
    buf_free(output);
}

// --------------------------------------------------------------------------------
// restore default parameters
// --------------------------------------------------------------------------------
static void cconfig_reset ( CHAR_DATA *ch, const char *arg )
{
    upkeep_cost_guard[0] = C_UPKEEP_SGUARD;
    upkeep_cost_guard[1] = C_UPKEEP_VGUARD;
    upkeep_cost_guard[2] = C_UPKEEP_EGUARD;

    upkeep_cost_clanmember   = C_UPKEEP_CLANMEMBER;
    upkeep_cost_advisor      = C_UPKEEP_ADVISOR;
    upkeep_cost_spy          = C_UPKEEP_SPY;
    upkeep_cost_transporter  = C_UPKEEP_TRANSPORTER;
    upkeep_cost_altar        = C_UPKEEP_ALTAR;

    AREA_ATTACK_TIME_LIMIT   = C_AREA_ATTACK_TIME_LIMIT;
    AREA_DEFENCE_TIME_LIMIT  = C_AREA_DEFENCE_TIME_LIMIT;
    AREA_PREVLOCK_TIME_LIMIT = C_AREA_PREVLOCK_TIME_LIMIT;
    TIME_TO_UPGRADE          = C_TIME_TO_UPGRADE;

    ENERGY_OUTPUT_ROOM_MOD   = C_ENERGY_OUTPUT_ROOM_MOD;
    // todo : reset all VNUMS! if nessessary

    char_act("All conquest parameters have been reset.", ch);
}

// --------------------------------------------------------------------------------
// save parameters to file ETC_PATH/CONQ_CONFIG_FILE
// --------------------------------------------------------------------------------
static void cconfig_save ( CHAR_DATA *ch, const char *arg )
{
    FILE        * fp;

    if (!(fp = dfopen(ETC_PATH, CONQ_CONFIG_FILE, "w")))
    {
        log_printf("ERROR!!! cconfig_save: can't open output file %s/%s", ETC_PATH, CONQ_CONFIG_FILE);
        if (ch)
            char_printf(ch, "Can't open output file %s/%s", ETC_PATH, CONQ_CONFIG_FILE);
        return;
    }

    fprintf(fp, "#CONFIG\n");
    fprintf(fp, "UpkeepStandard %d\n", upkeep_cost_guard[0]);
    fprintf(fp, "UpkeepVeteran %d\n",  upkeep_cost_guard[1]);
    fprintf(fp, "UpkeepElite %d\n",    upkeep_cost_guard[2]);
    fprintf(fp, "UpkeepClanmember %d\n", upkeep_cost_clanmember);
    fprintf(fp, "UpkeepAdvisor %d\n", upkeep_cost_advisor);
    fprintf(fp, "UpkeepSpy %d\n", upkeep_cost_spy);
    fprintf(fp, "UpkeepTransporter %d\n", upkeep_cost_transporter);
    fprintf(fp, "UpkeepAltar %d\n", upkeep_cost_altar);
    fprintf(fp, "TimeAttack %d\n", AREA_ATTACK_TIME_LIMIT);
    fprintf(fp, "TimeDefence %d\n", AREA_DEFENCE_TIME_LIMIT);
    fprintf(fp, "TimeLock %d\n", AREA_PREVLOCK_TIME_LIMIT);
    fprintf(fp, "TimeBetweenUpgrades %d\n", TIME_TO_UPGRADE);
    fprintf(fp, "EnergyOutput %d\n", ENERGY_OUTPUT_ROOM_MOD);
    fprintf(fp, "End\n\n");
    fprintf(fp, "#END\n");
    fclose(fp);
    if (ch)
        char_printf(ch, "Conquest parameters saved to file %s/%s\n", ETC_PATH, CONQ_CONFIG_FILE);
}

// --------------------------------------------------------------------------------
// load parameters from file ETC_PATH/CONQ_CONFIG_FILE
// --------------------------------------------------------------------------------
static void cconfig_load ( CHAR_DATA *ch, const char *arg )
{
    FILE *fp;
    if ((fp = dfopen(ETC_PATH, CONQ_CONFIG_FILE, "r")))
    {
        fclose (fp) ;
        conquer_read_config ();
    }
    else if (ch)
        char_printf(ch, "File %s/%s not found.\n", ETC_PATH, CONQ_CONFIG_FILE);

}

// --------------------------------------------------------------------------------
// load parameters from file ETC_PATH/CONQ_CONFIG_FILE
// called from db.c (boot_db) and cconfig_load
// --------------------------------------------------------------------------------
void conquer_read_config ( void )
{
    FILE *fp;
    bool         fMatch;

    if ((fp = dfopen(ETC_PATH, CONQ_CONFIG_FILE, "r")))
    {
        for (;;)
        {
            char   letter ;
            char * word   ;

            letter = fread_letter (fp) ;

            if (letter == '*')
            {
                fread_to_eol (fp) ;
                continue ;
            }

            if (letter != '#')
            {
                log_printf ("cconfig_load: # not found in %s.", CONQ_FILE_NAME) ;
                break ;
            }

            word = fread_word (fp) ;

            if (!str_cmp (word, "CONFIG"))
            {
                for (;;)
                {
                    word = feof(fp) ? "End" : fread_word(fp);
                    fMatch = FALSE;

                    switch (UPPER(word[0])) {
                    case '*':
                        fMatch = TRUE;
                        fread_to_eol(fp);
                        break;

                    case 'E':
                        KEY("EnergyOutput", ENERGY_OUTPUT_ROOM_MOD, fread_number(fp));
                        if (!str_cmp(word, "End"))
                            return;
                        break;

                    case 'T':
                        KEY("TimeAttack", AREA_ATTACK_TIME_LIMIT, fread_number(fp));
                        KEY("TimeDefence", AREA_DEFENCE_TIME_LIMIT, fread_number(fp));
                        KEY("TimeLock", AREA_PREVLOCK_TIME_LIMIT, fread_number(fp));
                        KEY("TimeBetweenUpgrades", TIME_TO_UPGRADE, fread_number(fp));
                        break;

                    case 'U':
                        KEY("UpkeepStandard", upkeep_cost_guard[0], fread_number(fp));
                        KEY("UpkeepVeteran", upkeep_cost_guard[1], fread_number(fp));
                        KEY("UpkeepElite", upkeep_cost_guard[2], fread_number(fp));
                        KEY("UpkeepClanmember", upkeep_cost_clanmember, fread_number(fp));
                        KEY("UpkeepAdvisor", upkeep_cost_advisor, fread_number(fp));
                        KEY("UpkeepSpy", upkeep_cost_spy, fread_number(fp));
                        KEY("UpkeepTransporter", upkeep_cost_transporter, fread_number(fp));
                        KEY("UpkeepAltar", upkeep_cost_altar, fread_number(fp));
                        break;
                    }

                    if (!fMatch)
                    {
                        bug("cconfig_load: no match.", 0);
                        fread_to_eol(fp);
                    }
                }
            }
            else if (!str_cmp (word, "END")) break ;
            else
            {
                bug ("cconfig_load: bad section.", 0) ;
                break ;
            }
        }
        fclose (fp) ;
    }
}

// --------------------------------------------------------------------------------
// sets attack time
// --------------------------------------------------------------------------------
static void cconfig_attack ( CHAR_DATA *ch, const char *arg )
{
    set_numeric_value(ch, arg, &AREA_ATTACK_TIME_LIMIT, C_ATTACK_TIME_MSG);
}

// --------------------------------------------------------------------------------
// sets defence time
// --------------------------------------------------------------------------------
static void cconfig_defence ( CHAR_DATA *ch, const char *arg )
{
    set_numeric_value(ch, arg, &AREA_DEFENCE_TIME_LIMIT, C_DEFENCE_TIME_MSG);
}

// --------------------------------------------------------------------------------
// sets previous owner lockdown time
// --------------------------------------------------------------------------------
static void cconfig_lock ( CHAR_DATA *ch, const char *arg )
{
    set_numeric_value(ch, arg, &AREA_PREVLOCK_TIME_LIMIT, C_LOCK_TIME_MSG);
}

// --------------------------------------------------------------------------------
// sets time between two upgrades
// --------------------------------------------------------------------------------
static void cconfig_upgrade ( CHAR_DATA *ch, const char *arg )
{
    set_numeric_value(ch, arg, &TIME_TO_UPGRADE, C_UPGRADE_TIME_MSG);
}

static void cconfig_output   ( CHAR_DATA *ch, const char *arg )
{
    set_numeric_value(ch, arg, &ENERGY_OUTPUT_ROOM_MOD, C_ENERGY_OUTPUT_ROOM_MSG);
}

// --------------------------------------------------------------------------------
// sets all upkeep costs. All subfunctions are incapsulated
// --------------------------------------------------------------------------------
static void cconfig_upkeep ( CHAR_DATA *ch, const char *arg )
{
    char           cmd [MAX_INPUT_LENGTH];

    arg = one_argument (arg, cmd,  sizeof(cmd));

    if (cmd[0] == '\0')
    {
        char_act("Usage: {Gupkeep clanmember/altar/standard/veteran/elite/advisor/spy/transporter{x", ch);
        return;
    }

    if (!str_prefix(cmd, "clanmember"))
        set_numeric_value(ch, arg, &upkeep_cost_clanmember, C_UPKEEP_CLANMEMBER_MSG);
    else if (!str_prefix(cmd, "altar"))
        set_numeric_value(ch, arg, &upkeep_cost_altar, C_UPKEEP_ALTAR_MSG);
    else if (!str_prefix(cmd, "standard"))
        set_numeric_value(ch, arg, &upkeep_cost_guard[0], C_UPKEEP_STANDARD_MSG);
    else if (!str_prefix(cmd, "veteran"))
        set_numeric_value(ch, arg, &upkeep_cost_guard[1], C_UPKEEP_VETERAN_MSG);
    else if (!str_prefix(cmd, "elite"))
        set_numeric_value(ch, arg, &upkeep_cost_guard[2], C_UPKEEP_ELITE_MSG);
    else if (!str_prefix(cmd, "advisor"))
        set_numeric_value(ch, arg, &upkeep_cost_advisor, C_UPKEEP_ADVISOR_MSG);
    else if (!str_prefix(cmd, "spy"))
        set_numeric_value(ch, arg, &upkeep_cost_spy, C_UPKEEP_SPY_MSG);
    else if (!str_prefix(cmd, "transporter"))
        set_numeric_value(ch, arg, &upkeep_cost_transporter, C_UPKEEP_TRANSPORTER_MSG);
    else
        char_act("Usage: {Gupkeep clanmember/altar/standard/veteran/elite/advisor/spy/transporter{x", ch);
}

static void set_numeric_value (CHAR_DATA *ch, const char *arg, int *var, int msg_num)
{
    int value;
    if (arg[0] == '\0')
        char_printf(ch, value_show_msg[msg_num], *var);
    else
    {
        value = atoi (arg);
        if (!value)
            char_act ("Value must be numeric.", ch);
        else
        {
            *var = value;
            char_printf (ch, value_set_msg[msg_num], *var);
            char_act("Use {Ccconfig save{x to save parameters.", ch);
        }
    }
}

// --------------------------------------------------------------------------------
// local functions
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// choose right function from cmd_table according to user command
// called from do_conquer, do_manage, do_spy, do_advisor etc
// --------------------------------------------------------------------------------
bool parse_command (CHAR_DATA *ch, const char *argument, CMD_DATA cmd_table[])
{
    char       cmd [MAX_INPUT_LENGTH];
    CMD_DATA * conqcmd;
    clan_t   * clan;

    /*
    if (IS_SET(cmd_table[0].extra, CONQCMD_REUSE_ARG))
        one_argument (argument, cmd,  sizeof(cmd));
    else
        argument = one_argument (argument, cmd,  sizeof(cmd));
    */

    one_argument (argument, cmd,  sizeof(cmd));

    if (IS_NPC (ch))
        return TRUE;

    for (conqcmd = cmd_table; conqcmd->name != NULL; conqcmd++)
        if (str_prefix(cmd, conqcmd->name) == 0)
        {
            if (ch->level < conqcmd->level)
                continue;

            if (ch->position < conqcmd->min_position && !IS_IMMORTAL (ch))
            {
                switch(ch->position)
                {
                    case POS_DEAD:
                        char_act("Lie still; You are DEAD.", ch);
                        break;

                    case POS_MORTAL:
                    case POS_INCAP:
                        char_act("You are hurt far too bad for that.", ch);
                        break;

                    case POS_STUNNED:
                        char_act("You are too stunned to do that.", ch);
                        break;

                    case POS_SLEEPING:
                        char_act("In your dreams, or what?", ch);
                        break;

                    case POS_RESTING:
                        char_act("Nah... You feel too relaxed...", ch);
                        break;

                    case POS_SITTING:
                        char_act("Better stand up first.", ch);
                        break;

                    case POS_FIGHTING:
                        char_act("No way!  You are still fighting!", ch);
                        break;
                }
                return TRUE;
            }

            switch (conqcmd->min_clan_status)
            {
            case CONQ_CLAN:
                if (!ch->clan && (!IS_IMMORTAL(ch)))
                    continue;
                break;
            case CONQ_ELITE:
                if ((ch->pcdata->clan_status != CLAN_ELITE) &&
                    (ch->pcdata->clan_status != CLAN_SECOND) &&
                    (ch->pcdata->clan_status != CLAN_LEADER) &&
                    (!IS_IMMORTAL(ch)))
                    continue;
                break;
            case CONQ_LEADERS:
                if ((ch->pcdata->clan_status != CLAN_SECOND) &&
                    (ch->pcdata->clan_status != CLAN_LEADER) &&
                    (!IS_IMMORTAL(ch)))
                    continue;
                break;
            }

            // if command is only for clans with CLAN_CAN_CONQUER flag
            if (IS_SET (conqcmd->extra, CONQCMD_IF_PERMITTED) && !IS_IMMORTAL(ch))
            {
                clan = clan_lookup (ch->clan) ;
                if (!clan)
                    continue;

                if (!IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
                {
                    char_act ("Area conquest is not enabled for your clan.", ch);
                    return TRUE;
                }
            }

            if ((ch->pcdata->questpoints < conqcmd->quest_price) &&
               (!IS_IMMORTAL(ch)))
            {
                char_printf (ch, "It costs %d questpoints. You don't have enough.", conqcmd->quest_price);
                return TRUE;
            }
            else if (!IS_IMMORTAL(ch))
                ch->pcdata->questpoints -= conqcmd->quest_price;

            if ((ch->gold < conqcmd->gold_price) &&
               (!IS_IMMORTAL(ch)))
            {
                char_printf (ch, "It costs %d gold. You don't have enough.", conqcmd->gold_price);
                return TRUE;
            }
            else if (!IS_IMMORTAL(ch))
                ch->gold -= conqcmd->gold_price;

            if ((!IS_SET (conqcmd->extra, CONQCMD_KEEP_HIDE))
                &&  IS_AFFECTED(ch, AFF_HIDE | AFF_FADE | AFF_TRAP) && !IS_IMMORTAL (ch))
            {
                REMOVE_BIT(ch->affected_by, AFF_HIDE | AFF_FADE | AFF_TRAP);
                char_act("You step out of shadows.", ch);
                act_puts("$n steps out of shadows.", ch, NULL, NULL, TO_ROOM, POS_RESTING);
            }

            if (!IS_SET (conqcmd->extra, CONQCMD_REUSE_ARG))
                argument = one_argument(argument, cmd,  sizeof(cmd));

            conqcmd->do_fn (ch, argument);
            return TRUE;
        }
    // command not found
    return FALSE;
}

// --------------------------------------------------------------------------------
// show all commands (called from do_conquer, do_spy etc
//                    if wrong command was received)
// --------------------------------------------------------------------------------
void show_command_list (CHAR_DATA *ch, CMD_DATA cmd_table[])
{
    CMD_DATA *cmd;

    char_puts ("{GAvailable commands{x: ", ch);
    for (cmd = cmd_table; cmd->name != NULL; cmd++)
    {
        switch (cmd->min_clan_status)
        {
        case CONQ_ALL:
            if (ch->level >= cmd->level)
            {
                if (IS_SET(cmd->extra, CONQCMD_RUS))
                    act_puts("($t) ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
                else
                    act_puts("$t ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
            }
            break;
        case CONQ_CLAN:
            if (!ch->clan && !IS_IMMORTAL(ch))
                continue;
            if (ch->level >= cmd->level)
            {
                if (IS_SET(cmd->extra, CONQCMD_RUS))
                    act_puts("($t) ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
                else
                    act_puts("$t ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
            }
            break;
        case CONQ_ELITE:
            if ((ch->pcdata->clan_status != CLAN_ELITE) &&
                (ch->pcdata->clan_status != CLAN_SECOND) &&
                (ch->pcdata->clan_status != CLAN_LEADER) &&
                (!IS_IMMORTAL(ch)))
                continue;
            if (ch->level >= cmd->level)
            {
                if (IS_SET(cmd->extra, CONQCMD_RUS))
                    act_puts("($t) ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
                else
                    act_puts("$t ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
            }
            break;
        case CONQ_LEADERS:
            if ((ch->pcdata->clan_status != CLAN_SECOND) &&
                (ch->pcdata->clan_status != CLAN_LEADER) &&
                (!IS_IMMORTAL(ch)))
                continue;
            if (ch->level >= cmd->level)
            {
                if (IS_SET(cmd->extra, CONQCMD_RUS))
                    act_puts("($t) ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
                else
                    act_puts("$t ", ch, cmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
            }
            break;
        }
    }
}

// --------------------------------------------------------------------------------
// handle area guards and priests death and energy gain for mob killing
// this function is called from figth.c (raw_kill_org function) after each kill
// so do_slay is considered
// NOTE: all area guards MUST be ACT_NOPURGE
// --------------------------------------------------------------------------------
void areacontrol_handle_death (CHAR_DATA *ch, CHAR_DATA *victim)
{
    bool is_areaguard = FALSE;
    bool is_priest = FALSE;
    int  i, add_energy, priest_pos = 0;

    if (!victim || !IS_NPC(victim))
        return;

    // add energy to killer's clan
    if (ch->clan && (victim->level - ch->level > -15) && !IS_IMMORTAL(ch))
    {
        add_energy = get_world_free_energy() / 1000;
        if (add_energy)
        {
            if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                char_printf (ch, "Your clan receives %d energy as reward!\n", add_energy);
            put_world_free_energy (get_world_free_energy() - add_energy);
            if ((clan_lookup(ch->clan)->energy + add_energy) < CLAN_ENERGY_LIMIT)
                clan_lookup(ch->clan)->energy += add_energy;
            else
                clan_lookup(ch->clan)->energy = CLAN_ENERGY_LIMIT;
        }
    }


    // priest processing
    // was there priest the mob was killed?
    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        if (victim == victim->in_room->area->priests[i])
        {
            is_priest = TRUE;
            priest_pos = i;
            break;
        }

    // we have to demolish an appropriate altar
    if (is_priest)
    {
        act ("Mortally wounded $n falls to earth!",victim->in_room->area->priests[priest_pos],NULL,NULL,TO_ROOM);
        act ("His {rblood{x floods $p and it tumbles down.",victim->in_room->area->priests[priest_pos],victim->in_room->area->altars[priest_pos],NULL,TO_ROOM);
        extract_obj (victim->in_room->area->altars[priest_pos]);
        victim->in_room->area->altars[priest_pos] = NULL;
        victim->in_room->area->priests[priest_pos] = NULL;
        save_conquered_areas ();
    }

    // area guard processing
    // was there area guard the mob who was killed?
    for (i = 0; i < MAX_GUARD_COUNT; ++i) // ???? max_guards?
        if (victim == victim->in_room->area->areaguards[i])
        {
            is_areaguard = TRUE;
            victim->in_room->area->areaguards[i] = NULL;
            break;
        }

    if (is_areaguard)
        // last area guard was killed?
        if (! (--ch->in_room->area->guards_left))
        {
            // during defence?
            if (ch->in_room->area->curr_defence_time)
            {
                // not last attempt
                if (ch->in_room->area->defence_attempts > 0)
                    ch->in_room->area->curr_defence_time = 0;
                // last defence attempt failed!
                // sorry, we have to make area neutral
                else
                {
                    ch->in_room->area->prev_clan_owner = ch->in_room->area->clan_owner;
                    ch->in_room->area->clan_owner = NULL;
                    ch->in_room->area->prev_lock_time = AREA_PREVLOCK_TIME_LIMIT;
                    ch->in_room->area->time_owned = 0;
                    ch->in_room->area->curr_attack_time = 0;
                    ch->in_room->area->curr_defence_time = 0;
                    ch->in_room->area->areaguard_rank = AREAGUARD_STANDARD;

                    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                        rem_altar_raw (ch->in_room->area, i);

                    // decrease rating for clan - ex-owner (100 * (rating per tick))
                    ch->in_room->area->prev_clan_owner->areacontrol_rating -= (ch->in_room->area->explorable_rooms >= 100 ? ch->in_room->area->explorable_rooms/100 : 1) *
                                                              (ch->in_room->area->area_rating_modifier + 1) * 100;
                    if (ch->in_room->area->prev_clan_owner->areacontrol_rating < 0)
                        ch->in_room->area->prev_clan_owner->areacontrol_rating = 0;
                    // decrease clan - ex-owner conquered area count
                    ch->in_room->area->prev_clan_owner->conquered_areas -= 1;
                }
            }
            // no! it is the beginning of an attack
            else if (ch->in_room->area->clan_owner)
                ch->in_room->area->curr_attack_time = AREA_ATTACK_TIME_LIMIT*ch->in_room->area->max_guard_count;
            else //neutral area -> now it is free!!!
                ch->in_room->area->curr_attack_time = 0;
        }
}

// --------------------------------------------------------------------------------
// this function is called from update.c (update_handler function) each CHAR_TICK
// --------------------------------------------------------------------------------
void areacontrol_update (void)
{
    AREA_DATA   * pArea;
    int           rem_energy;
    int           add_energy;
    int           upkeep;
    int           i;
    clan_t      * clan;

    // first, we clear tick statistics for each clan
    for (i = 0; i < clans.nused; ++i)
    {
        clan = CLAN(i);
        clan->area_energy_this_tick = 0;
        clan->earned_energy_this_tick = 0;
    }

    for (pArea = area_first; pArea; pArea = pArea->next)
    {
        if (pArea->prev_lock_time)
            --pArea->prev_lock_time;

        if (!(pArea->prev_lock_time) && pArea->prev_clan_owner)
            pArea->prev_clan_owner = NULL;

        // energy generation for area
        // to free_energy

        // skip non-conquerable areas
        if (!pArea->max_guard_count || !pArea->conq_enabled)
            continue;

        // add_energy -----> free_energy
        // rem_energy -----> clan_owner store

        add_energy = pArea->explorable_rooms * ENERGY_OUTPUT_ROOM_MOD;
        if (pArea->clan_owner)
        {
            rem_energy = pArea->explorable_rooms * number_range(ENERGY_OUTPUT_ROOM_MOD / 4, ENERGY_OUTPUT_ROOM_MOD / 2);
            pArea->clan_owner->energy += rem_energy;
            add_energy -= rem_energy;
            pArea->clan_owner->area_energy_this_tick += rem_energy;

            // update clan conquest rating
            pArea->clan_owner->areacontrol_rating += (pArea->explorable_rooms >= 100 ? pArea->explorable_rooms/100 : 1) *
                                                      (pArea->area_rating_modifier + 1);

            // update ownership counter
            ++pArea->time_owned;

            // upkeep for current area (after we added energy)
            upkeep = 0;
            // areaguards
            for (i = 0; i < MAX_GUARD_COUNT; ++i)
                if (pArea->areaguards[i])
                    upkeep += upkeep_cost_guard[pArea->areaguard_rank];
            // altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (pArea->altars[i])
                    upkeep += upkeep_cost_altar;

            pArea->clan_owner->energy -= upkeep;
            //log_printf("%s: guard upkeep = %d", pArea->name, upkeep);
        }

        // special case: clanarea
        if (pArea->clan)
        {
            clan = clan_lookup (pArea->clan);

            upkeep = 0;
            // altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (pArea->altars[i])
                    upkeep += upkeep_cost_altar;

            clan->energy -= upkeep;
        }

        // world_free_energy modification
        put_world_free_energy (get_world_free_energy() + add_energy);

        // skip non-attacked areas
        if (!pArea->curr_attack_time)
            continue;

        // defence
        if (pArea->curr_defence_time)
            if (!(--pArea->curr_defence_time))
            {
                pArea->curr_defence_time = 0;
                pArea->curr_attack_time = 0;
                continue;
            }

        // attack time has gone
        if (!(--pArea->curr_attack_time))
        {
            // guards still in the area
            if (pArea->guards_left)
                continue;

            // no guards - make area neutral
            pArea->prev_clan_owner = pArea->clan_owner;
            pArea->clan_owner = NULL;
            pArea->prev_lock_time = AREA_PREVLOCK_TIME_LIMIT;
            pArea->time_owned = 0;
            pArea->areaguard_rank = AREAGUARD_STANDARD;

            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                rem_altar_raw (pArea, i);

            // decrease rating for clan - ex-owner (100 * (rating per tick))
            pArea->prev_clan_owner->areacontrol_rating -= (pArea->explorable_rooms >= 100 ? pArea->explorable_rooms/100 : 1) *
                                                      (pArea->area_rating_modifier + 1) * 100;
            if (pArea->prev_clan_owner->areacontrol_rating < 0)
                pArea->prev_clan_owner->areacontrol_rating = 0;
            // decrease clan - ex-owner conquered area count
            pArea->prev_clan_owner->conquered_areas -= 1;
        }
    }

    // clan upkeep phase
    for (i = 0; i < clans.nused; ++i)
    {
        clan = CLAN(i);

        if (!IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
        {
            clan->energy = 0;
            continue;
        }

        upkeep = calculate_clan_upkeep(clan);

        clan->energy -= upkeep;
        clan->energy = URANGE (0, clan->energy, CLAN_ENERGY_LIMIT);

        // conquest rating falls down for clans without areas
        if (clan->conquered_areas == 0)
            clan->areacontrol_rating -= 3;
        if (clan->areacontrol_rating < 0)
            clan->areacontrol_rating = 0;
    }

    // saving all info into 'conquered.lst'
    if (--pulse_conquered <= 0)
    {
        save_conquered_areas ();
        pulse_conquered = AREACONTROL_SAVE_LIMIT;
    }
}

// --------------------------------------------------------------------------------
// place area guards in targed area. clan may be CLAN_NONE in this function
// to place neutral guards
// --------------------------------------------------------------------------------
void place_area_guards (AREA_DATA *curr_area, int clan)
{
    CHAR_DATA *guard;
    MOB_INDEX_DATA  * pGuardIndex;
    ROOM_INDEX_DATA * pRoomIndex;
    int i;
    int guard_vnum = VNUM_AREAGUARD_STANDARD;
    clan_t * cln;

    // select the mob will be placed as areaguard
    // if vnum specified for clan, use it. Otherwise use default one
    if (clan)
    {
        cln = clan_lookup(clan);
        if (!cln) // just to be sure
        {
            log_printf("*** BUG ***: place_area_guards - invalid clan_number %d", clan);
            return;
        }
        switch (curr_area->areaguard_rank)
        {
        case AREAGUARD_STANDARD:
            guard_vnum = cln->standard_areaguard_vnum ? cln->standard_areaguard_vnum : VNUM_AREAGUARD_STANDARD;
            break;
        case AREAGUARD_VETERAN:
            guard_vnum = cln->veteran_areaguard_vnum ? cln->veteran_areaguard_vnum : VNUM_AREAGUARD_VETERAN;
            break;
        case AREAGUARD_ELITE:
            guard_vnum = cln->elite_areaguard_vnum ? cln->elite_areaguard_vnum : VNUM_AREAGUARD_ELITE;
            break;
        }
    }

    // place area guards
    for (i = 0; i < curr_area->max_guard_count; ++i)
    {
        if ((pGuardIndex = get_mob_index(guard_vnum)) == NULL)
        {
            log_printf ("[**BUG**] in do_conquer: can't create mob %d.", guard_vnum);
            return;
        }

        guard = create_mob(pGuardIndex);
        guard->clan = clan;

        if ((pRoomIndex = get_room_index(curr_area->guard_room_vnums[i])) == NULL)
        {
            log_printf ("[**BUG**] in do_conquer: room %d does not exist.", curr_area->guard_room_vnums[i]);
            return;
        }

        curr_area->areaguards[i] = guard;
        char_to_room(guard, pRoomIndex);
        act("Suddenly $n appears to protect this area!",guard,NULL,NULL,TO_ROOM);
        // do_clan (guard, "Ready to protect!");
    }
    curr_area->guards_left = curr_area->max_guard_count;
}

// --------------------------------------------------------------------------------
// kill all area guards in target area
// this function is called from conquer_disable and conquer_liberate functions
// --------------------------------------------------------------------------------
static void nuke_area_guards (AREA_DATA *pArea)
{
    int i;

    for (i = 0; i < pArea->max_guard_count; ++i)
    {
        if (pArea->areaguards[i])
        {
            act("There was a clap of thunder and $n was consumed by roaring {Mv{mi{Mo{ml{Me{mt{x flame!",
                pArea->areaguards[i],NULL,NULL,TO_ROOM);
            extract_char(pArea->areaguards[i], TRUE);
            pArea->areaguards[i] = NULL;
        }
    }
    pArea->guards_left = 0;
}

// --------------------------------------------------------------------------------
// this function is called from db.c (boot_db function) once during boot
// read info from CONQ_FILE_NAME
// --------------------------------------------------------------------------------
void load_conquered_areas ()
{
    FILE *fp;
    AREA_DATA *pArea;
    MOB_INDEX_DATA  *pAdvIndex, *pSpyIndex, *pTransIndex;
    int i;

    if ((fp = dfopen(CLANS_PATH, CONQ_FILE_NAME, "r")))
    {
        for (;;)
        {
            char   letter ;
            char * word   ;

            letter = fread_letter (fp) ;

            if (letter == '*')
            {
                fread_to_eol (fp) ;
                continue ;
            }

            if (letter != '#')
            {
                log_printf ("Load_conquered_areas: # not found in %s.", CONQ_FILE_NAME) ;
                break ;
            }

            word = fread_word (fp) ;

            if (!str_cmp (word, "AREA")) fread_conquered_area (fp);
            else if (!str_cmp (word, "GLOBAL")) fread_conquered_global_data (fp);
            else if (!str_cmp (word, "CLAN")) fread_conquered_info_clan (fp);
            else if (!str_cmp (word, "END")) break ;
            else
            {
                bug ("load_conquered_areas: bad section.", 0) ;
                break ;
            }
        }
        fclose (fp) ;
    }

    // next step - place neutral guard in conquerable but non-conquered areas
    // and calculate conquerable area count
    for (pArea = area_first; pArea; pArea = pArea->next)
    {
        if (pArea->conq_enabled && pArea->max_guard_count)// max_guard_count - just to be sure
        {
            if (!pArea->clan_owner)
                place_area_guards(pArea, CLAN_NONE);
            ++conq_area_count;
        }
    }

    // create advisor, spy, transporter for each clan
    // advisor
    if ((pAdvIndex = get_mob_index(VNUM_ADVISOR)) == NULL)
        return;
    // spy
    if ((pSpyIndex = get_mob_index(VNUM_SPY)) == NULL)
        return;
    // transporter
    if ((pTransIndex = get_mob_index(VNUM_TRANSPORTER)) == NULL)
        return;
    for (i = 1; i < clans.nused; ++i)
    {
        CLAN(i)->advisor = create_mob(pAdvIndex);
        CLAN(i)->advisor->clan = i;
        CLAN(i)->spy = create_mob(pSpyIndex);
        CLAN(i)->spy->clan = i;
        CLAN(i)->transporter = create_mob(pTransIndex);
        CLAN(i)->transporter->clan = i;
    }
}

// read one #AREA section from file
void fread_conquered_area (FILE *fp)
{
    char       * word;
    bool         fMatch;
    AREA_DATA  * pArea;
    bool         area_found;
    const char * curr_area_name = NULL;
    const char * clan_owner_name = NULL;
    clan_t     * clan;
    int          cn, i;
    int          time = 0;
    int          guard_rank = 0;
    int          altar_type, altar_room_vnum;

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'A':
            if (!str_cmp(word, "Altar"))
            {
                altar_type = fread_number(fp);
                altar_room_vnum = fread_number(fp);
                add_altar_raw(altar_room_vnum, 0, altar_type);
            }
            break;

        case 'C':
            SKEY("ClanOwner", clan_owner_name);
            break;

        case 'E':
            if (!str_cmp(word, "End"))
            {
                cn = cln_lookup(clan_owner_name);
                if (! (clan = clan_lookup(cn)))
                {
                    log_printf ("ERROR!!! fread_conquered_area: unknown clan %s", word);
                    if (clan_owner_name)
                        free_string(clan_owner_name);
                    if (curr_area_name)
                        free_string(curr_area_name);
                    return;
                }
                area_found = FALSE;
                for (pArea = area_first; pArea; pArea = pArea->next)
                    if (!str_cmp (pArea->name, curr_area_name))
                    {
                        area_found = TRUE;
                        if (pArea->conq_enabled || pArea->clan)
                        {
                            if (!pArea->clan)
                            {
                                pArea->clan_owner = clan;
                                clan->conquered_areas += 1;
                                pArea->time_owned = time;
                                pArea->areaguard_rank = guard_rank;
                                place_area_guards(pArea, cn);
                            }
                            // we have to set priest->clan for each priest
                            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                                if (pArea->priests[i])
                                    pArea->priests[i]->clan = cn;
                        }
                        break;
                    }
                if (!area_found)
                    log_printf ("ERROR!!! fread_conquered_area: unknown area %s", curr_area_name);
                if (clan_owner_name)
                    free_string(clan_owner_name);
                if (curr_area_name)
                    free_string(curr_area_name);
                return;
            }
            break;

        case 'G':
            KEY("GuardRank", guard_rank, fread_number(fp));
            break;

        case 'N':
            SKEY("Name", curr_area_name);
            break;

        case 'T':
            KEY("TimeOwned", time, fread_number(fp));
            break;
        }

        if (!fMatch)
        {
            bug("Fread_conquered_area: no match.", 0);
            fread_to_eol(fp);
        }
    }
}

// read one #GLOBAL section from file
static void fread_conquered_global_data (FILE *fp)
{
    char       * word;
    bool         fMatch;
    int          free_energy = 0;

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'E':
            if (!str_cmp(word, "End"))
            {
                put_world_free_energy(free_energy);
                return;
            }
            break;

        case 'F':
            KEY("FreeEnergy", free_energy, fread_number(fp));
            break;
        }

        if (!fMatch)
        {
            bug("Fread_conquered_global_data: no match.", 0);
            fread_to_eol(fp);
        }
    }
}

// read one #CLAN section from file
static void fread_conquered_info_clan (FILE *fp)
{
    char       * word;
    bool         fMatch;
    int          energy = 0;
    int          rating = 0;
    const char * clan_name = NULL;
    clan_t     * clan;
    flag64_t     conq_flags = 0;

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'C':
            SKEY("ClanName", clan_name);
            break;

        case 'E':
            KEY("Energy", energy, fread_number(fp));
            if (!str_cmp(word, "End"))
            {
                clan = clan_lookup(cln_lookup(clan_name));
                if (!clan)
                {
                    if (clan_name)
                        free_string(clan_name);
                    log_printf ("ERROR!!! fread_conquered_info_clan: unknown clan %s", word);
                    return;
                }

                clan->energy = energy;
                clan->areacontrol_rating = rating;
                clan->conq_flags = conq_flags;

                if (clan_name)
                    free_string(clan_name);
                return;
            }
            break;

        case 'F':
            KEY("Flags", conq_flags, fread_flags(fp));
            break;

        case 'R':
            KEY("Rating", rating, fread_number(fp));
            break;
        }

        if (!fMatch)
        {
            bug("Fread_conquered_info_clan: no match.", 0);
            fread_to_eol(fp);
        }
    }
}

// --------------------------------------------------------------------------------
// saving list of conquered areas and some info about areacontrol system
// --------------------------------------------------------------------------------
void save_conquered_areas ()
{
    FILE        * fp;
    AREA_DATA   * pArea;
    clan_t      * clan;
    int           i;

    if (!(fp = dfopen(CLANS_PATH, CONQ_FILE_NAME, "w")))
    {
        log_printf("ERROR!!! save_conquered_areas: can't open output file %s/%s", CLANS_PATH, CONQ_FILE_NAME);
        return;
    }

    // #GLOBAL section
    fprintf (fp, "#GLOBAL\n");
    fprintf (fp, "FreeEnergy %d\n", get_world_free_energy());
    fprintf (fp, "End\n\n");

    // #AREA sections
    for (pArea = area_first; pArea; pArea = pArea->next)
    {
        if (pArea->clan_owner && pArea->conq_enabled)
        {
            fprintf (fp, "#AREA\n");
            fwrite_string(fp, "Name", pArea->name);
            fwrite_string(fp, "ClanOwner", pArea->clan_owner->name);
            fprintf(fp, "TimeOwned %d\n", pArea->time_owned);
            fprintf(fp, "GuardRank %d\n", pArea->areaguard_rank);
            // saving altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (pArea->altars[i])
                    fprintf(fp, "Altar %d %d\n", i, pArea->altars[i]->in_room->vnum);
            fprintf(fp, "End\n\n");
        }
        else if (pArea->clan)
        {
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (pArea->altars[i])
                    break;

            if (!pArea->altars[i])
                continue;

            fprintf (fp, "#AREA\n");
            fwrite_string(fp, "Name", pArea->name);
            // saving altars
            for (i = 0; i < MAX_ALTAR_COUNT; ++i)
                if (pArea->altars[i])
                    fprintf(fp, "Altar %d %d\n", i, pArea->altars[i]->in_room->vnum);
            fprintf(fp, "End\n\n");
        }
    }
    // #CLAN sections
    for (i = 0; i < clans.nused; ++i)
    {
        clan = CLAN(i);
        // skip clans without permission
        if (!IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
            continue;

        if (clan->areacontrol_rating || // there is some info
            clan->energy ||
            clan->conq_flags )
        {
            fprintf (fp, "#CLAN\n");
            fwrite_string(fp, "ClanName", clan->name);
            if (clan->areacontrol_rating)
                fprintf(fp, "Rating %d\n", clan->areacontrol_rating);
            if (clan->energy)
                fprintf(fp, "Energy %d\n", clan->energy);
            if (clan->conq_flags)
                fprintf(fp, "Flags %s\n", format_flags(clan->conq_flags));
            fprintf(fp, "End\n\n");
        }
    }

    fprintf(fp, "#END\n");
    fclose(fp);
}

// --------------------------------------------------------------------------------
// this function is called from:
//      update.c (hit_gain, mana_gain, move_gain functions)
//      gquest.c (gq_checkbonus)
//      quest.c (quest_complete)
// --------------------------------------------------------------------------------
int areacontrol_modify_gain (CHAR_DATA *ch, int gain, int type)
{
    clan_t  * clan       = NULL ;
    int       new_gain   = gain ;
    int       add_energy = 0;
    float     mod; // gain modification, max values must be balanced

    if (ch == NULL)
        return 0;

    // for PC's only
    // but... may be, for clanguards... ?
    if (IS_NPC(ch) || (!ch->clan) || IS_IMMORTAL(ch))
        return gain;

    clan = clan_lookup (ch->clan);

    if (!clan) // just to be sure
        return gain;

    // add energy from world_free_energy pool
    // as quest or global quest reward
    switch (type)
    {
        case GLOBAL_QUEST_REWARD:
            add_energy = get_world_free_energy() / 1000;
            break;
        case QUEST_REWARD_QP:
            add_energy = get_world_free_energy() / 400;
            break;
    }
    if (add_energy && IS_SET (clan->conq_flags, CLAN_CAN_CONQUER))
    {
        if (!IS_SET(ch->comm,COMM_NOVERBOSE))
            char_printf (ch, "Your clan receives %d energy as reward!\n", add_energy);
        if ((clan->energy + add_energy) < CLAN_ENERGY_LIMIT)
            clan->energy += add_energy;
        else
            clan->energy = CLAN_ENERGY_LIMIT;
        put_world_free_energy (get_world_free_energy() - add_energy);
        clan->earned_energy_this_tick += add_energy;
    }

    // there are many "if" because in general case formulas for
    // increasing/decreasing hp/mana/moves etc might be different
    if (ch->in_room->area->clan_owner)
    {
    switch (type)
    {
        case GAIN_HP:
            if (ch->in_room->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of your clan in this area and your health restores faster!", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 20);
                new_gain = gain * mod;
            }
            else if ((ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of friendly clan in this area and your health restores faster!", ch);
                mod = 1.5 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 15);
                new_gain = gain * mod;
            }
            else
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel nervous in this area and your health restores slower.", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                // no limit for decreasing
                new_gain = gain / mod;
            }
            break;

        case GAIN_MANA:
            if (ch->in_room->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of your clan in this area and your mana restores faster!", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 20);
                new_gain = gain * mod;
            }
            else if ((ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of friendly clan in this area and your mana restores faster!", ch);
                mod = 1.5 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 15);
                new_gain = gain * mod;
            }
            else
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel nervous in this area and your mana restores slower.", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                new_gain = gain / mod;
            }
            break;

        case GAIN_MOVES:
            if (ch->in_room->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of your clan in this area and your moves restores faster!", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 20);
                new_gain = gain * mod;
            }
            else if ((ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel support of friendly clan in this area and your moves restores faster!", ch);
                mod = 1.5 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 15);
                new_gain = gain * mod;
            }
            else
            {
                if (!IS_NPC(ch) && !IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You feel nervous in this area and your moves restores slower.", ch);
                mod = 2 + 0.1*(ch->in_room->area->time_owned/180 + 1);
                new_gain = gain / mod;
            }
            break;

        case GLOBAL_QUEST_REWARD:
            if (ch->in_room->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 1.2 + 0.05*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 5);
                new_gain = gain * mod;
            }
            else if ((ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->in_room->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 1.1 + 0.02*(ch->in_room->area->time_owned/180 + 1);
                mod = UMIN (mod, 3);
                new_gain = gain * mod;
            }
            break;

        case QUEST_REWARD_GOLD:
            if (!ch->pcdata->questroom)
                return gain;

            if (ch->pcdata->questroom->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 2 + 0.2*(ch->pcdata->questroom->area->time_owned/180 + 1);
                mod = UMIN (mod, 5);
                new_gain = gain * mod;
            }
            else if ((ch->pcdata->questroom->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->pcdata->questroom->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 1.5 + 0.1*(ch->pcdata->questroom->area->time_owned/180 + 1);
                mod = UMIN (mod, 4);
                new_gain = gain * mod;
            }
            break;

        case QUEST_REWARD_QP:
            if (!ch->pcdata->questroom)
                return gain;

            if (ch->pcdata->questroom->area->clan_owner == clan)
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 1.4 + 0.05*(ch->pcdata->questroom->area->time_owned/180 + 1);
                mod = UMIN (mod, 4);
                new_gain = gain * mod;
            }
            else if ((ch->pcdata->questroom->area->clan_owner->diplomacy[ch->clan] == DIP_ALLY) ||
                     (ch->pcdata->questroom->area->clan_owner->diplomacy[ch->clan] == DIP_PEACE))
            {
                if (!IS_SET(ch->comm,COMM_NOVERBOSE))
                    char_act ("You receive areacontrol bonus!", ch);
                mod = 1.2 + 0.02*(ch->pcdata->questroom->area->time_owned/180 + 1);
                mod = UMIN (mod, 3);
                new_gain = gain * mod;
            }
            break;
        }
    }
//    if (!IS_NPC(ch) && !IS_SET(ch->comm,COMM_NOVERBOSE))
//        char_printf (ch, "Areacontrol modification: old gain = %d, new gain = %d.\n",
//            gain, new_gain);
    return new_gain;
}

// --------------------------------------------------------------------------------
// these 2 functions are for access to world_free_energy variable
// --------------------------------------------------------------------------------
int  get_world_free_energy(void)
{
    return world_free_energy;
}

void put_world_free_energy(int new_value)
{

    // must not be < 0
    if (new_value < 0)
    {
        world_free_energy = 0;
        return;
    }

    // must not be greater than limit
    if (new_value > FREE_ENERGY_LIMIT)
    {
        switch (number_range(1, 3))
        {
        case (1):
            world_free_energy = FREE_ENERGY_LIMIT;
            return;
        case (2):
            world_free_energy = FREE_ENERGY_LIMIT * 2 / 3;
            return;
        case (3):
            world_free_energy = FREE_ENERGY_LIMIT / 2;
            return;
        }
    }

    // all is OK
    world_free_energy = new_value;
}

// --------------------------------------------------------------------------------
// how many applies are in area (areaguard upgrade, altars etc)
// --------------------------------------------------------------------------------
static int apply_count (AREA_DATA * pArea)
{
    int i, count = 0;

    count += pArea->areaguard_rank;
    for (i = 0; i < MAX_ALTAR_COUNT; ++i)
        if (pArea->altars[i])
            ++count;
    return count;
}

// --------------------------------------------------------------------------------
// upkeep for clanmembers, advisor etc
// used in areacontrol_update and advisor_upkeep
// Warning! In the case of any change all must be repeated in advisor_upkeep
// --------------------------------------------------------------------------------
static int calculate_clan_upkeep (clan_t * clan)
{
    int upkeep = 0;
    CLAN_MEMBER * member;
    DESCRIPTOR_DATA * d;

    // upkeep for each member (lower level - higher upkeep)
    for (member = clan->member_list ; member ; member = member->next)
        (member->level >= 91) ? (upkeep += upkeep_cost_clanmember) :
            (upkeep += (11 - member->level/10) * upkeep_cost_clanmember);

    // upkeep for clan applies
    if (IS_SET(clan->conq_flags, CLAN_WITH_ADVISOR))
        upkeep += upkeep_cost_advisor;
    if (IS_SET(clan->conq_flags, CLAN_WITH_SPY))
        upkeep += upkeep_cost_spy;
    if (IS_SET(clan->conq_flags, CLAN_WITH_TRANSPORTER))
        upkeep += upkeep_cost_transporter;

    // each clan has energy to support 5 HEROes, 2 level 80 members and 1 level 70
    // for free
    //upkeep -= (5 * upkeep_cost_clanmember +     // HERO
    //           2 * upkeep_cost_clanmember * 3 + // level 80
    //           1 * upkeep_cost_clanmember * 4); // level 70
    upkeep -= upkeep_cost_clanmember * 15;

    // small bonus for active clanmembers
    for (d = descriptor_list; d != NULL; d = d->next)
    {
        CHAR_DATA *ch;
        clan_t * ch_clan;

        if (d->connected != CON_PLAYING)
            continue;

        ch = d->character;
        ch_clan = clan_lookup (ch->clan);
        if (ch_clan != clan)
            continue;
        upkeep -= upkeep_cost_clanmember / 2;
    }

    // upkeep can be even > 0.. if there are many active clanmembers
    return upkeep;
}

// --------------------------------------------------------------------------------
// checks whether an altar of specified type works for specified char
// NB! Altars of negation and immobility is the special case
// (see functions check_altar_negation and check_altar_immobility)
// these two function are separate because check for OTHER CLAN's altars
// --------------------------------------------------------------------------------
bool altar_works (CHAR_DATA * ch, int type)
{
    if (!ch || !ch->in_room || !ch->in_room->area)
        return FALSE;

    if (type >= MAX_ALTAR_COUNT)
        return FALSE;

    return (((ch->in_room->area->clan_owner == clan_lookup (ch->clan))
             || (ch->in_room->area->clan == ch->clan))
           && ch->in_room->area->altars[type]);
}

// --------------------------------------------------------------------------------
// checks whether an altar of NEGATION works for specified char
// called from get_clan_max_percent (clan.c)
// --------------------------------------------------------------------------------
bool check_altar_negation (CHAR_DATA * ch)
{
    clan_t * clan;

    if (!ch || !ch->in_room || !ch->in_room->area)
        return FALSE;

    if (ch->in_room->area->clan_owner)
        clan = ch->in_room->area->clan_owner;
    else  if (ch->in_room->area->clan)
        clan = clan_lookup (ch->in_room->area->clan);
    else
        return FALSE;

    // owner
    if (clan == clan_lookup (ch->clan))
        return FALSE;

    // ally
    if ((clan->diplomacy[ch->clan] == DIP_ALLY)
         || (clan->diplomacy[ch->clan] == DIP_PEACE))
        return FALSE;

    // other: check for altar
    return (ch->in_room->area->altars[ALTAR_NEGATION_POS] != NULL);
}

// --------------------------------------------------------------------------------
// checks whether an altar of IMMOBILITY works for specified char and victim
// returns FALSE if altar affects 'where'
// called from can_gate (handler.c)
// --------------------------------------------------------------------------------
bool check_altar_immobility (CHAR_DATA * ch, ROOM_INDEX_DATA * where)
{
    clan_t * clan;

    if (!ch || !ch->in_room || !ch->in_room->area || !where)
        return FALSE;

    if (!ch->clan)
        return FALSE;

    if (where->area->clan_owner)
        clan = where->area->clan_owner;
    else  if (where->area->clan)
        clan = clan_lookup (where->area->clan);
    else
        return FALSE;

    // owner
    if (clan == clan_lookup (ch->clan))
        return FALSE;

    // ally
    if ((clan->diplomacy[ch->clan] == DIP_ALLY)
         || (clan->diplomacy[ch->clan] == DIP_PEACE))
        return FALSE;

    // other: check for altar
    return (where->area->altars[ALTAR_IMMOBILITY_POS] != NULL);
}
