/* $Id: db_clan.c,v 1.666 2004/09/20 10:50:19 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined (WIN32)
#   include <compat/compat.h>
#else
#   include <dirent.h>
#endif

#include "merc.h"
#include "db.h"
#include "clan.h"

DECLARE_DBLOAD_FUN(load_clan);

DBFUN db_load_clans[] =
{
    { "CLAN",   load_clan   },
    { NULL }
};

DBLOAD_FUN(load_clan)
{
    clan_t * clan;
    int     i;

    clan = clan_new();
    clan->file_name = get_filename(filename);

    for (;;) {
        char *word = feof(fp) ? "End" : fread_word(fp);
        bool fMatch = FALSE;

        switch (UPPER(word[0])) {
        case 'A':
            KEY("Altar", clan->altar_vnum, fread_number(fp));
            KEY("AltarTrophy", clan->altar_trophy_vnum,
                fread_number(fp));
            KEY("AreaGuardStandard", clan->standard_areaguard_vnum, fread_number(fp));
            KEY("AreaGuardVeteran", clan->veteran_areaguard_vnum, fread_number(fp));
            KEY("AreaGuardElite", clan->elite_areaguard_vnum, fread_number(fp));
            KEY("AllianceAllow", clan->allow_alliance, fread_fstring(aliance_allow_flag, fp));
            break;
        case 'B':
            KEY("BankGold", clan->bank_gold, fread_number(fp));
            KEY("BankQuestp", clan->bank_questp, fread_number(fp));
            KEY("BankBonus", clan->bank_bonus, fread_number(fp));
            KEY("BankGems", clan->bank_gems, fread_number(fp));
            KEY("BankRGems", clan->bank_rgems, fread_number(fp));
            KEY("BankBGems", clan->bank_bgems, fread_number(fp));
            KEY("BankGGems", clan->bank_ggems, fread_number(fp));
            KEY("BankMGems", clan->bank_mgems, fread_number(fp));
            KEY("BankWGems", clan->bank_wgems, fread_number(fp));
            KEY("BankYGems", clan->bank_ygems, fread_number(fp));
            break;
        case 'C':
            KEY("ClanMark", clan->mark_vnum, fread_number(fp));
            KEY("ClanAlign", clan->clan_align, fread_number(fp));
            break;
        case 'D':
            if (!str_cmp(word, "Diplomacy")) {
                for (i = 0; i < MAX_CLANS-1; i++)
                        clan->diplomacy[i] = fread_number(fp);
                    fMatch = TRUE;
                }
            break;
        case 'E':
            if (!str_cmp(word, "End")) {
                if (IS_NULLSTR(clan->name)) {
                    db_error("load_clan",
                         "clan name not defined");
                    clan_free(clan);
                    clans.nused--;
                }
                varr_qsort(&clan->skills, cmpint);
                return;
            }
            break;
        case 'F':
            KEY("Flags", clan->flags, fread_fstring(clan_flags, fp));
            break;
        case 'I':
            KEY("Item", clan->obj_vnum, fread_number(fp));
            break;
        case 'M':
            SKEY("MsgPrays", clan->msg_prays);
            SKEY("MsgVanishes", clan->msg_vanishes);
            KEY("MaxMembers", clan->max_clan_members, fread_number (fp));
            KEY("MinLevel", clan->min_clan_level, fread_number (fp));
            break;
        case 'N':
            SKEY("Name", clan->name);
            break;
        case 'R':
            KEY("Recall", clan->recall_vnum, fread_number(fp));
            break;
        case 'S':
            if (!str_cmp(word, "Skill")) {
                clan_skill_t *sk = varr_enew(&clan->skills);
                sk->sn = sn_lookup(fread_word(fp));
                sk->level = fread_number(fp);
                sk->percent = fread_number(fp);
                fMatch = TRUE;
            }
            break;
        case 'W':
            KEY("WhereItem", clan->item_at, fread_number(fp));
            break;
        }

        if (!fMatch)
            db_error("load_clan", "%s: Unknown keyword", word);
    }
}

void load_clan_members ()
{
    struct dirent      * dp    ;
    struct clan_member * clanm ;

    DIR  * dirp   ;
    FILE * pfile  ;
    char   letter ;
    char * word   ;
    int    cn = 0 ;

    if ((dirp = opendir (PLAYER_PATH)) == NULL)
    {
        bug ("Load_clan_members: unable to open player directory.", 0) ;
        exit(1) ;
    }

    // check each player
    for (dp = readdir (dirp) ; dp != NULL ; dp = readdir (dirp))
    {
        const char * clanname ;
        clan_t  * clan ;

        clanm = (struct clan_member *) calloc (1, sizeof (struct clan_member)) ;

        // only BSD implementation has d_type member of dirent structure
        #if defined (__FreeBSD__)
        if (dp->d_namlen < 3 || dp->d_type != DT_REG) continue ;
        #else
        if (strlen (dp->d_name) < 3) continue ;
        #endif

        // do not process HOLDER_LIST file in any
        if (!str_cmp (dp->d_name, HOLDER_LIST)) continue ;

        // open pfile
        if ((pfile = dfopen (PLAYER_PATH, dp->d_name, "r")) == NULL)
        {
            log_printf ("Load_clan_members: Can't open player file %s.",
                        dp->d_name) ;
            continue ;
        }

        // look through player data
        clanname = NULL ;
        cn       =    0 ;

        for (;;)
        {
            letter = fread_letter (pfile) ;

            if (letter == '*') { fread_to_eol (pfile) ; continue ; }

            if (letter != '#')
            {
                log_printf ("Load_clan_members: # not found in %s.",
                            dp->d_name) ;
                break ;
            }

            word = fread_word (pfile) ; // word - static buffer ptr

            // process player section
            if (!str_cmp (word, "PLAYER"))
            {
                for (;;)
                {
                    word = fread_word (pfile) ; // word - static buffer ptr

                    // until EOF or End
                    if (feof (pfile) || !word || !str_cmp (word, "End")) break;

                    // load player name
                    if (!str_cmp (word, "Name"))
                    {
                        clanm->name = fread_string (pfile) ; // allocate here
                    }
                    else
                    if (!str_cmp (word, "Clan"))
                    {
                        free_string (clanname) ;
                        clanname = fread_string (pfile) ; // allocate here
                    }
                    else
                    if (!str_cmp (word, "ClanStatus"))
                    {
                        clanm->rank = fread_number (pfile) ;
                    }
                    else
                    if (!str_cmp (word, "Alig"))
                    {
                        clanm->alignment = fread_number (pfile) ;
                    }
                    else
                    if (!str_cmp (word, "Ethos"))
                    {
                        clanm->ethos = fread_fword (ethos_table, pfile) ;
                    }
                    else
                    if (!str_cmp (word, "Login"))
                    {
                        clanm->last_login = fread_string (pfile) ;
                    }
                    else
                    if (!str_cmp (word, "Levl"))
                    {
                        clanm->level = fread_number (pfile) ;
                    }
                    else
                    if (!str_cmp (word, "Desc"))
                    {
                        // workaround to skip description
                        mlstring * desc = mlstr_fread (pfile) ;
                        mlstr_free (desc) ;
                    }
                    else fread_to_eol (pfile) ;
                }
            }
            else
            // process object section
            if (!str_cmp (word, "O") || !str_cmp (word, "OBJECT")) break ;
            else
            if (!str_cmp (word, "PET")) break ;
            /* golem and undead section */
            else
            if (!str_cmp (word, "GOLEM")) break ;
            else
            if (!str_cmp (word, "UNDEAD")) break ;
            /* end golem and undead section */
            else
            if (!str_cmp (word, "EXPLORED")) break ;
            else
            if (!str_cmp (word, "END")) break ;
            else
            {
                log_printf ("Load_clan_members: bad section in %s.",
                            dp->d_name) ;
                break ;
            }
        }

        // lookup clan data by name
        if (clanname && (cn = cln_lookup (clanname)) > 0 && clanm->level <= LEVEL_HERO)
        {
            clan = CLAN (cn) ;

            // add member to the beginning of the list
            clanm->next       = clan->member_list ;
            clan->member_list = clanm ;

            ++clan->cur_clan_members ;

            if (clan->cur_clan_members > clan->max_clan_members)
            {
                log_printf ("Load_clan_members: clan members limit exceeded in %s",
                            clan->name) ;
            }
        }
        else
        {
            free (clanm) ;

            if (clanm->level <= LEVEL_HERO && clanname != NULL)
                log_printf ("Load_clan_members: Unknown clan %s in player %s.",
                            clanname, dp->d_name) ;
        }

        free_string (clanname) ;
        fclose      (pfile)    ;
    }

    closedir (dirp) ;
}
