/* $Id: hometown.c,v 1.666 2004/09/20 10:50:19 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <string.h>

#include "merc.h"

static hometown_t * get_hometown    (int htn);

varr hometowns = { sizeof(hometown_t), 4 };

hometown_t *hometown_new(void)
{
    return varr_enew(&hometowns);
}

void hometown_free(hometown_t *hometown)
{
    free_string(hometown->area);
}

/*
 * lookup hometown number by name
 */
int htn_lookup(const char *name)
{
    int i;

    for (i = 0; i < hometowns.nused; i++) {
        hometown_t *h = VARR_GET(&hometowns, i);
        if (!str_prefix(name, h->area))
            return i;
    }

    return -1;
}

/*
 * lookup hometown name by htn
 */
const char* hometown_name(int htn)
{
    return get_hometown(htn)->area;
}

bool hometown_restrict(hometown_t *h, CHAR_DATA *ch)
{
    race_t *r;
    class_t *cl;

    if ((r = race_lookup(ORG_RACE(ch))) == NULL
    ||  !r->pcdata
    ||  (cl = class_lookup(ch->class)) == NULL)
        return TRUE;

    if (IS_SET(r->flags, RACE_NOCH))
        return !is_name(r->name, h->restrict_race);

    if (IS_SET(cl->flags, CLASS_NOCH))
        return !is_name(cl->name, h->restrict_class);

    if (h->restrict_race
    &&  !is_name(r->name, h->restrict_race))
        return TRUE;

    if (h->restrict_class
    &&  !is_name(cl->name, h->restrict_class))
        return TRUE;

    if (h->restrict_align
    &&  (RALIGN(ch) & h->restrict_align) == 0)
        return TRUE;

    return FALSE;
}

/*
 * just prints the list of available hometowns
 */
void hometown_print_avail(CHAR_DATA *ch)
{
    int i;
    int col = 0;

    for (i = 0; i < hometowns.nused; i++) {
        hometown_t *h = VARR_GET(&hometowns, i);

        if (hometown_restrict(h, ch))
            continue;

        if (col > 60) {
            char_act(str_empty, ch);
            col = 0;
        }

        if (col)
            char_puts(", ", ch);
        char_puts(h->area, ch);
        col += strlen(h->area) + 2;
    }
}

int hometown_permanent(CHAR_DATA *ch)
{
    int i;
    int htn_perm = -1;

    for (i = 0; i < hometowns.nused; i++) {
        hometown_t *h = VARR_GET(&hometowns, i);

        if (hometown_restrict(h, ch))
            continue;

        if (htn_perm < 0)
            htn_perm = i;
        else
            return -1;  /* more than one hometown */
    }

    return htn_perm;
}

/*
 * recall lookup by ch hometown and align (must always return non NULL)
 */
ROOM_INDEX_DATA *get_recall(CHAR_DATA *ch)
{
//    if (!IS_NPC(ch) && ch->pcdata->homepoint)
//        return ch->pcdata->homepoint;
    return get_hometown(ch->hometown)->recall[NALIGN(ch)];
}

OBJ_INDEX_DATA *get_map(CHAR_DATA *ch)
{
    return get_hometown(ch->hometown)->map[NALIGN(ch)];
}

ROOM_INDEX_DATA *get_random_recall(void)
{
    return get_hometown(number_range(1, hometowns.nused-1))->recall[number_range(0, MAX_ANUM-1)];
}

altar_t *get_altar(CHAR_DATA *ch)
{
    return get_hometown(ch->hometown)->altar + NALIGN(ch);
}

/*----------------------------------------------------------------------------
 * local functions
 */
static hometown_t* get_hometown(int htn)
{
    hometown_t *h = varr_get(&hometowns, htn);
    if (!h)
        h = VARR_GET(&hometowns, 0);
    return h;
}

