/* $Id: event.h,v 1.666 2004/09/20 10:49:48 shrike Exp $                             */

/************************************************************************************
 *  Contains information on the event system structures etc...                      *
 ************************************************************************************/

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

/************************************************************************************
 *  Original code by DalekenMUD 1.12 (C) 2000                                       *
 *  Adapted for current codebase by Shrike aka Sauron at 2004                       *
 ***********************************************************************************/


#ifndef _EVENT_H_
#define _EVENT_H_

typedef struct          plane_data              PLANE_DATA;
typedef struct          event_data              EVENT;
typedef unsigned char   uint_8;

#define MAX_EVENT_HASH  256   /* The hash bucket size.      */
#define MAX_EVENT_DATA  4     /* The amount of data required. */


/*
 * Target type identifier.
 */
#define ENV_TARGET_VOID       0
#define ENV_TARGET_CHAR       1
#define ENV_TARGET_OBJ        2
#define ENV_TARGET_ROOM       3
#define ENV_TARGET_AREA       4
#define ENV_TARGET_PLANE      5


/*
 * Special Event flags.
 * These flags are set/unset in the event table.  Individual events may
 * override this by setting them on the flags and the value will be toggled,
 * using an XOR (^).
 * E = only used when set on an event.
 * T = only used when set in the event table.
 */
#define EVENT_LOADING         1     /* Event has just been loaded.       (E) */
#define EVENT_TEMPORARY       2     /* Event doesn't need saving etc...      */
#define EVENT_DEATH_REMOVE    4     /* Event removed when target killed.     */
#define EVENT_NO_QUIT         8     /* Target cannot be removed.             */
#define EVENT_STACKABLE       16    /* Multiple events can be added.     (T) */

typedef struct multi_target_type    ENV_TARGET_TYPE;
typedef void   EVENT_CALLBACK       args(( EVENT *ev));

#define DECLARE_EVENT_CALLBACK( evf )     EVENT_CALLBACK evf


/*
 * This is where OO programming would REALLY come in handy.
 * Programming like this is a little ugly.
 * Still, this could be a useful structure, think about how act() works...
 */
struct multi_target_type
{
    uint_8  type;
    union
    {
      void        *typeless;
      OBJ_DATA    *obj;
      CHAR_DATA   *ch;
      ROOM_INDEX_DATA   *room;
      AREA_DATA   *area;
      PLANE_DATA  *plane;
    }       target;
};


/*
 * The information required for an event.
 * Some effort could be made to slightly reduce the size required.
 */
struct event_data
{
    EVENT                    *next_local;
    EVENT                    *next_global;
    ENV_TARGET_TYPE           actor;            /* Where the event was generated.    */
    time_t                    when;             /* Which game pulse it goes off on.  */
    int                       flags;            /* See above for flag descriptions.  */
    int                       type;             /* Determines the callback function. */
                                                /* Storage elements, used for miscellaneous
                                                   information, normally dependant on the event. */
    const char               *text;
    int                       data[ MAX_EVENT_DATA ];
    ENV_TARGET_TYPE           extra;
};


/*
 * The event table.
 */
struct event_table_type
{
    const char          *name;
    int                 *type;
    EVENT_CALLBACK      *callback;
    int                  flags;
};

extern struct event_table_type event_table [];


/*
 * Function declarations.
 */

void event_update             args((void));
EVENT *load_event             args((FILE *fp));
void save_event               args((FILE *fp, EVENT *e));
void inject_events            args((CHAR_DATA *ch));
EVENT *create_char_event      args((CHAR_DATA *ch, int type, int delay));
EVENT *create_obj_event       args((OBJ_DATA *obj, int type, int delay));
EVENT *create_room_event      args((ROOM_INDEX_DATA *room, int type, int delay));
EVENT *create_area_event      args((AREA_DATA *area, int type, int delay));
EVENT *create_plane_event     args((PLANE_DATA *plane, int type, int delay));
EVENT *create_typeless_event  args((void *vo, int type, int delay));
EVENT *duplicate_event        args((EVENT *e, int delay));
void event_remove_global      args((EVENT *e));
void event_remove_local       args((EVENT **list, EVENT *e));
void strip_events             args((EVENT **list, int type));
int get_time_left             args((EVENT *e, int type));
void set_timer                args((OBJ_DATA *obj, int pulses));
void set_timer_tick           args((OBJ_DATA *obj, int ticks));
void set_imp_timer            args((OBJ_DATA *obj));


EVENT *new_event              args(( void ));
void   free_event             args(( EVENT *e ));
const char *print_event       args(( EVENT *e ));


/* for mud programs */
/*
void add_mob_triggers         args((CHAR_DATA *mob));
void add_mob_fight_triggers   args((CHAR_DATA *mob));
void add_obj_triggers         args((OBJ_DATA *obj));
void add_room_triggers        args((ROOM_INDEX_DATA *room));
*/
/* callback.c */
/*
int event_lookup        args( ( const char *name ) );
struct event_table_type *event_table_lookup     args( ( const char *name ) );
*/


/*
 * Event callback types and functions.
 */
/* Player actions */
/*
extern int evn_picklock;
DECLARE_EVENT_CALLBACK( ecb_finish_pick         );
extern int evn_picktrap;
DECLARE_EVENT_CALLBACK( ecb_pick_trap           );
extern int evn_mercenary;
DECLARE_EVENT_CALLBACK( ecb_mercenary           );
extern int evn_gate_demon;
DECLARE_EVENT_CALLBACK( ecb_gate_demon          );
extern int evn_raised_undead;
DECLARE_EVENT_CALLBACK( ecb_raised_undead );
extern int evn_swan_song;
DECLARE_EVENT_CALLBACK( ecb_swan_song           );
extern int evn_char_fall;
DECLARE_EVENT_CALLBACK( ecb_char_fall           );
extern int evn_char_sink;
DECLARE_EVENT_CALLBACK( ecb_char_sink           );
extern int evn_char_drown;
DECLARE_EVENT_CALLBACK( ecb_char_drown          );
extern int evn_char_suffocate;
DECLARE_EVENT_CALLBACK( ecb_char_suffocate      );
extern int evn_web_struggle;
DECLARE_EVENT_CALLBACK( ecb_web_struggle  );
extern int evn_stand_up;
DECLARE_EVENT_CALLBACK( ecb_stand_up            );
extern int evn_regeneration;
DECLARE_EVENT_CALLBACK( ecb_regeneration  );

extern int evn_plant_grow;
DECLARE_EVENT_CALLBACK( ecb_plant_grow          );
extern int evn_plant_die;
DECLARE_EVENT_CALLBACK( ecb_plant_die           );
extern int evn_explode;
DECLARE_EVENT_CALLBACK( ecb_explode       );
*/
extern int evn_obj_decay;
extern int evn_imp_grab;
/*
DECLARE_EVENT_CALLBACK( ecb_obj_decay           );

DECLARE_EVENT_CALLBACK( ecb_imp_grab            );
extern int evn_obj_fall;
DECLARE_EVENT_CALLBACK( ecb_obj_fall            );
extern int evn_spec_fun;
DECLARE_EVENT_CALLBACK( ecb_spec_fun            );
extern int evn_class_action;
DECLARE_EVENT_CALLBACK( ecb_class_action  );
extern int evn_hunt_fleer;
DECLARE_EVENT_CALLBACK( ecb_hunt_fleer          );
extern int evn_scavenge;
DECLARE_EVENT_CALLBACK( ecb_scavenge            );
extern int evn_wander;
DECLARE_EVENT_CALLBACK( ecb_wander        );
extern int evn_cone_remove;
DECLARE_EVENT_CALLBACK( ecb_cone_remove         );
extern int evn_room_violence;
DECLARE_EVENT_CALLBACK( ecb_room_violence );
*/
extern int evn_prog_trigger;
/*
DECLARE_EVENT_CALLBACK( ecb_prog_trigger  );
extern int evn_prog_wait;
DECLARE_EVENT_CALLBACK( ecb_prog_wait           );

extern int evn_area_reset;
DECLARE_EVENT_CALLBACK( ecb_area_reset          );
*/

DECLARE_EVENT_CALLBACK(ecb_char_test);
extern int evn_char_test;

#endif
