/* $Id: fighters.c,v 1.666 2004/09/20 10:49:48 shrike Exp $                          */


/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

/* Part of this code is based on Solace code */

#include <sys/types.h>
#include <stdio.h>
#include "merc.h"
#include "fight.h"
#include "war.h"

extern void do_yell (CHAR_DATA *, const char *) ;
extern bool check_recovery (CHAR_DATA *ch, CHAR_DATA *victim) ;
extern bool check_mirror_image (CHAR_DATA *ch, CHAR_DATA *victim);

//
// this command is used to set the primary specialization
// in weapon
//
void do_specialization (CHAR_DATA * ch, const char * argument)
{
    int choice ;
    char arg [MAX_INPUT_LENGTH]  ;

    const char * specialization_table [8] =
    {
    "none"                  ,
    "sword"                 ,
    "dagger"                ,
    "axe"                   ,
    "whip/flail"            ,
    "spear/polearm/staff"   ,
    "hand to hand"          ,
    "mace"
    } ;

    if (IS_NPC (ch)) return ;

    //
    // classes that have predefined single specialization
    //

    if (IS_SAMURAI(ch))
    {
        char_act("You are Master of the sword!", ch);

        ch->pcdata->specialization[0] = SPEC_SWORD ; 
        ch->pcdata->specialization[1] = SPEC_SWORD ;
        ch->pcdata->specialization[2] = SPEC_SWORD ;
        return ;
    }

    if (ch->class == CLASS_NINJA)
    {
        char_act("You are Master of the fisticuffs!", ch);

        ch->pcdata->specialization[0] = SPEC_HAND ;
        ch->pcdata->specialization[1] = SPEC_HAND ;
        ch->pcdata->specialization[2] = SPEC_HAND ;
        return ;
    }

    if (ch->class == CLASS_THIEF)
    {
        char_act("You are Master of the dagger!", ch);

        ch->pcdata->specialization[0] = SPEC_DAGGER ;
        ch->pcdata->specialization[1] = SPEC_DAGGER ;
        ch->pcdata->specialization[2] = SPEC_DAGGER ;
        return ;
    }
  
    //
    // check for those classes that really have any specialization
    //

    if (ch->class != CLASS_WARRIOR &&
        ch->class != CLASS_RANGER  &&
        ch->class != CLASS_PALADIN &&
        ch->class != CLASS_ANTI_PALADIN)
    {
        char_act("Huh?", ch);
        return ;
    }

    one_argument(argument, arg, sizeof(arg)) ;

    // display help
    if (arg[0] == '\0')
    {
        if (ch->pcdata->specialization[0] == 0 &&
            ch->pcdata->specialization[1] == 0 &&
            ch->pcdata->specialization[2] == 0)
        {
            char_act("You do not specialize in any type of weapon.", ch) ;

            if (ch->class == CLASS_WARRIOR)
            {
                char_act("You may specialize in sword, axe, dagger, mace, whip, spear or hand to hand.", ch) ;
            }
            else
            if (ch->class == CLASS_RANGER)
            {
                char_act("You may specialize in sword, spear or flail.", ch) ;
            }
            else
            if(ch->class == CLASS_PALADIN)
            {
                char_act("You may specialize in sword, spear or mace.", ch) ;
            }
            else
            if (ch->class == CLASS_ANTI_PALADIN)
            {
                char_act("You may specialize in sword, axe or spear.", ch) ;
            }

            return ;
        }
        else
        if (ch->pcdata->specialization[1] == 0 &&
            ch->pcdata->specialization[2] == 0)
        {

            act("You specialize in $c5{$t}.",
                ch, 
                specialization_table[ch->pcdata->specialization[0]],
                NULL, TO_CHAR | ACT_TRANS) ; 
          
            if( ch->class == CLASS_WARRIOR)
                char_act("You may choose two additional specializations.", ch);
            return ;
        }
        else if (ch->pcdata->specialization[2] == 0)
        {

            act("You specialize in $c5{$t} and $c5{$T}.",
                    ch,
                    specialization_table[ch->pcdata->specialization[0]],
                    specialization_table[ch->pcdata->specialization[1]],
                    TO_CHAR | ACT_TRANS) ; 
        
            if (ch->class == CLASS_WARRIOR)
                char_act("You may choose one additional specialization.", ch);
            return ;
        }
        else
        {
             act_puts3("You specialize in $c5{$t}, $c5{$T} and $c5{$U}.",
                    ch,
                    specialization_table[ch->pcdata->specialization[0]],
                    specialization_table[ch->pcdata->specialization[1]],
                    specialization_table[ch->pcdata->specialization[2]],
                    TO_CHAR | ACT_TRANS, POS_RESTING) ; 
       
            char_act("You have chosen all your specializations.", ch);
            return ;
        }
    }

    // ranger, paladin or anti-paladin can have only one specialization
    if (ch->class != CLASS_WARRIOR && ch->pcdata->specialization[0] != 0)
    {
        act("You specialize in $c5{$t}.",
            ch, specialization_table[ch->pcdata->specialization[0]],
            NULL, TO_CHAR | ACT_TRANS) ; 
        return ;
    }

    if (ch->pcdata->specialization[0] != 0 &&
        ch->pcdata->specialization[1] != 0 &&
        ch->pcdata->specialization[2] != 0)
    {
          act_puts3("You specialize in $c5{$t}, $c5{$T} and $c5{$U}.",
                ch,
                specialization_table[ch->pcdata->specialization[0]],
                specialization_table[ch->pcdata->specialization[1]],
                specialization_table[ch->pcdata->specialization[2]],
                TO_CHAR | ACT_TRANS, POS_RESTING) ; 
        char_act("You have chosen all your specializations.", ch);
        return ;
    }

    if (!str_cmp (arg, "sword"))  choice = SPEC_SWORD  ; else
    if (!str_cmp (arg, "dagger")) choice = SPEC_DAGGER ; else
    if (!str_cmp (arg, "axe"))    choice = SPEC_AXE    ; else
    if (!str_cmp (arg, "flail")  ||
        !str_cmp (arg, "whip"))   choice = SPEC_WHIP   ; else
    if (!str_cmp (arg,"spear")   ||
        !str_cmp (arg,"staff")   ||
        !str_cmp (arg,"polearm") ||
        !str_cmp (arg,"lance"))   choice = SPEC_SPEAR  ; else
    if (!str_cmp (arg,"hand"))    choice = SPEC_HAND   ; else
    if (!str_cmp (arg,"mace"))    choice = SPEC_MACE   ; else
    {
        if (ch->class == CLASS_WARRIOR)
        {
            char_act("You may specialize in sword, axe, dagger, mace, whip, spear or hand to hand.", ch) ;
        }
        else
        if (ch->class == CLASS_RANGER)
        {
            char_act("You may specialize in sword, spear or flail.", ch) ;
        }
        else
        if(ch->class == CLASS_PALADIN)
        {
            char_act("You may specialize in sword, spear or mace.", ch) ;
        }
        else
        if (ch->class == CLASS_ANTI_PALADIN)
        {
            char_act("You may specialize in sword, axe or spear.", ch) ;
        }

        return ;
    }

    if (choice != SPEC_SWORD && choice != SPEC_SPEAR)
    {
        if (ch->class == CLASS_RANGER && choice != SPEC_WHIP)
        {
            char_act("You may specialize in sword, spear or flail.", ch) ;
            return;
        }

        if (ch->class == CLASS_PALADIN && choice != SPEC_MACE)
        {
            char_act("You may specialize in sword, spear or mace.", ch) ;
            return;
        }

        if (ch->class == CLASS_ANTI_PALADIN && choice != SPEC_AXE)
        {
            char_act("You may specialize in sword, axe or spear.", ch) ;
            return;
        }
    }

    if (ch->pcdata->specialization[0] == choice ||
        ch->pcdata->specialization[1] == choice ||
        ch->pcdata->specialization[2] == choice)
    {
        act("You already specialize in $c5{$t}.",
               ch, specialization_table[choice], NULL, TO_CHAR | ACT_TRANS) ;
        return ;
    }

    act("You specialize in $c5{$t}.",
           ch, specialization_table[choice], NULL, TO_CHAR | ACT_TRANS) ;

    if (ch->pcdata->specialization[0] == 0)
        ch->pcdata->specialization[0] = choice ;
    else
    if (ch->pcdata->specialization[1] == 0)
        ch->pcdata->specialization[1] = choice ;
    else
        ch->pcdata->specialization[2] = choice ;

    // add corresponding skills
    switch (choice)
    {
    case SPEC_SWORD:
        set_skill_raw (ch, gsn_flurry,      1, 0) ;
        set_skill_raw (ch, gsn_flourentine, 1, 0) ;
        set_skill_raw (ch, gsn_riposte,     1, 0) ;
        set_skill_raw (ch, gsn_cross_slice, 1, 0) ;

        char_act("You have learned the following skills flurry, flourentine, riposte and cross slice.", ch) ;
        break ;

    case SPEC_DAGGER:
        set_skill_raw (ch, gsn_undercut, 1, 0) ;
        set_skill_raw (ch, gsn_bleed,    1, 0) ;
        set_skill_raw (ch, gsn_restrike, 1, 0) ;

        char_act("You have learned the following skills undercut, bleed and restrike.", ch) ;
        break ;

    case SPEC_MACE:
        set_skill_raw (ch, gsn_backhand,       1, 0) ;
        set_skill_raw (ch, gsn_drumming_maces, 1, 0) ;
        set_skill_raw (ch, gsn_boneshatter,    1, 0) ;

        char_act("You have learned the following skills backhand, drumming maces and boneshatter.", ch) ;
        break ;

    case SPEC_AXE:
        set_skill_raw (ch, gsn_axedigging,    1, 0) ;
        set_skill_raw (ch, gsn_pincer,        1, 0) ;
        set_skill_raw (ch, gsn_weapon_cleave, 1, 0) ;
        set_skill_raw (ch, gsn_shield_cleave, 1, 0) ;
        set_skill_raw (ch, gsn_mastering_axe, 1, 0) ;

        char_act("You have learned the following skills axedigging, weapon cleave, shield cleave, pincer and mastering axe.", ch) ;
        break ;

    case SPEC_HAND:
        set_skill_raw (ch, gsn_kung_fu,         1, 0) ;
        set_skill_raw (ch, gsn_throw,           1, 0) ;
        set_skill_raw (ch, gsn_ground_control,  1, 0) ;
        set_skill_raw (ch, gsn_hand_block,      1, 0) ;

        char_act("You have learned the following skills mastering pound, throw, ground control and hand block.", ch) ;
        break ;

    case SPEC_WHIP:
        set_skill_raw (ch, gsn_strip_weapon, 1, 0) ;
        set_skill_raw (ch, gsn_choke,        1, 0) ;
        set_skill_raw (ch, gsn_eyejab,       1, 0) ;
        set_skill_raw (ch, gsn_lash,         1, 0) ;

        char_act("You have learned the following skills strip weapon, choke, lash and eyejab.", ch) ;
        break ;

    case SPEC_SPEAR:
        set_skill_raw (ch, gsn_charge_set, 1, 0) ;
        set_skill_raw (ch, gsn_leg_sweep,  1, 0) ;
        set_skill_raw (ch, gsn_overhead,   1, 0) ;
        set_skill_raw (ch, gsn_stand_off,  1, 0) ;
        set_skill_raw (ch, gsn_pugil,      1, 0) ;

        char_act("You have learned the following skills charge set, pugil, leg sweep, overhead and stand off.", ch) ;
        break ;
    }

    if (ch->class != CLASS_WARRIOR)
        char_act("You have chosen your specialization.", ch);
    else
    if (ch->pcdata->specialization[1] == 0)
        char_act("You may choose two additional specializations.", ch);
    else
    if (ch->pcdata->specialization[2] == 0)
        char_act("You may choose one additional specialization.", ch);
    else 
        char_act("You have chosen all your specializations.", ch);
  
    return ;
}

void do_stance(CHAR_DATA *ch,const char *argument) 
{
  char arg[MAX_INPUT_LENGTH];
  int new_stance = 10,sk;
  AFFECT_DATA af;

  if ((sk=get_skill(ch,gsn_stance)) == 0) {
    char_act("You don't know how to fight in different stances.", ch);
    return;
  }
  if (IS_NPC(ch) && IS_AFFECTED(ch,AFF_CHARM))
    return;
  if (MOUNTED(ch)) {
    char_act("You can't enter a combat stance while riding.", ch);
    return;
  }
  one_argument(argument, arg, sizeof(arg));
  new_stance = 10;

  if (arg[0] == '\0') {
    switch(ch->stance) {
    default: case STANCE_NORMAL:
      char_act("You are in a normal fighting stance.", ch);
      break;

    case STANCE_OFFENSE:
      char_act("You are concentrating on beating your opponent down.", ch);
      break;

    case STANCE_OFFENSE_II:
      char_act("You are totally ignoring self-preservation and blazing in.", ch);
      break;

    case STANCE_DEFENSE:
      char_act("You are taking your opponent's attacks into careful consideration.", ch);
      break;

    case STANCE_DEFENSE_II:
      char_act("You are fully devoted to protecting yourself against adversity.", ch);
      break;
    }
    return;
  }

  if (!str_prefix(arg,"normal") || !str_prefix(arg,"relax"))
    new_stance = STANCE_NORMAL;
  else if (!str_prefix(arg,"offense"))
    new_stance = STANCE_OFFENSE;
  else if (!str_prefix(arg,"defense"))
    new_stance = STANCE_DEFENSE;
 
  if (new_stance == STANCE_NORMAL) {
    if (ch->stance == STANCE_NORMAL)
      char_act("You are already in a relaxed fighting stance.", ch);
    else if (ch->stance == STANCE_DEFENSE || ch->stance == STANCE_DEFENSE_II)
      char_act("You relax your vigilance and assume a normal combat posture.", ch);
    else
      char_act("You relax your aggression and assume a normal combat posture.", ch);

    act("$n seems to relax and assumes a more trusting posture.",ch,0,0,TO_ROOM);
    ch->stance = STANCE_NORMAL;
    affect_strip(ch,gsn_stance);
    return;
  } else if (new_stance == STANCE_OFFENSE) {
    af.type = gsn_stance;
    af.where = TO_AFFECTS;
    af.level = LVL(ch);
    af.bitvector = 0;
    af.duration = -1;

    if (ch->stance == STANCE_OFFENSE_II) {
      char_act("You are already as aggressive as your can get.", ch);
      return;
    } else if (ch->stance == STANCE_OFFENSE) {
      if (ch->class != CLASS_WARRIOR) {
    char_act("Your professional training in fighting doesn't cover higher stances.", ch);
    return;
      }
      char_act("You prepare to throw yourself head first into battle.", ch);
      act("$n gets a crazed look of a berserker in $gn{his} eyes.",ch,0,0,TO_ROOM);

      affect_strip(ch,gsn_stance);
      af.modifier = LVL(ch)/5 + 1;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = LVL(ch)/5 + 1;
      af.location = APPLY_DAMROLL;
      affect_to_char(ch,&af);

      ch->stance = STANCE_OFFENSE_II;
      return;
    }
    
      if (ch->stance == STANCE_NORMAL) {
      char_act("You prepare yourself for the heat of battle.", ch);
      act("$n prepares to throw $gn{himself} into the heat of battle.",ch,0,0,TO_ROOM);

      affect_strip(ch,gsn_stance);
      af.modifier = LVL(ch)/7;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = LVL(ch)/7;
      af.location = APPLY_DAMROLL;
      affect_to_char(ch,&af);

      ch->stance = STANCE_OFFENSE;
        return;
      }
      if (ch->stance == STANCE_DEFENSE) {
      char_act("You stop defending and throw yourself into the heat of battle!", ch);
      act("$n prepares to throw $gn{himself} into the heat of battle.",ch,0,0,TO_ROOM);

      affect_strip(ch,gsn_stance);
      af.modifier = LVL(ch)/7;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = LVL(ch)/7;
      af.location = APPLY_DAMROLL;
      affect_to_char(ch,&af);

      ch->stance = STANCE_OFFENSE;
      return;
    } 
    else {
      char_act("You are too concerned with self-preservation right now, you'll have to relax first.", ch);
      return;
    }
  } else if (new_stance == STANCE_DEFENSE) {
    
    af.type = gsn_stance;
    af.where = TO_AFFECTS;
    af.level = LVL(ch);
    af.bitvector = 0;
    af.duration = -1;

    if (ch->stance == STANCE_DEFENSE_II) {
      char_act("You are already as defensive as you can get.", ch);
      return;
    } else if (ch->stance == STANCE_DEFENSE) {
      if (ch->class != CLASS_WARRIOR) {
    char_act("Your professional training in fighting doesn't cover higher stances.", ch);
    return;
      }

      char_act("You concentrate fully on defending yourself from harm.", ch);
      act("$n steps back and assumes a fully defensive posture.",ch,0,0,TO_ROOM);

      affect_strip(ch,gsn_stance);
      af.modifier = -LVL(ch)/3;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = -LVL(ch)*3;
      af.location = APPLY_AC;
      affect_to_char(ch,&af);

      ch->stance = STANCE_DEFENSE_II;
      return;
    } else if (ch->stance == STANCE_NORMAL) {
      char_act("You stand-down and take things into better consideration.", ch);
      act("$n stands down and begins taking things into better consideration.",ch,0,0,TO_ROOM);

      affect_strip(ch,gsn_stance);
      af.modifier = -LVL(ch)/6;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = -LVL(ch) -10;
      af.location = APPLY_AC;
      affect_to_char(ch,&af);

      ch->stance = STANCE_DEFENSE;
      return;
    } else if (ch->stance == STANCE_OFFENSE) {
      char_act("You stand down from your aggressive posture and defend yourself.", ch);
      act("$n stands down and begins taking things into better consideration.",ch,0,0,TO_ROOM);
      affect_strip(ch,gsn_stance);
      af.modifier = -LVL(ch)/6;
      af.location = APPLY_HITROLL;
      affect_to_char(ch,&af);
      af.modifier = -LVL(ch) -10;
      af.location = APPLY_AC;
      affect_to_char(ch,&af);

      ch->stance = STANCE_DEFENSE;
      return;
    } else {
      char_act("You are too hyped up with killing frenzy, you need to relax first.", ch);
      return;
    }
  } else
    char_act("Assume which combat stance?", ch);

  return;
}

void do_style(CHAR_DATA *ch, const char *argument) 
{
    char arg[MAX_INPUT_LENGTH];
    int sk;
    AFFECT_DATA af;

    if ((sk = get_skill(ch, gsn_style)) == 0) 
    {
        char_act("You don't know how to fight in different styles.", ch);
        return;
    }

    if (IS_NPC(ch) && IS_AFFECTED(ch, AFF_CHARM))
        return;

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0') 
    {
        switch(ch->style) 
        {
        default: case STYLE_NORMAL:
        char_act("You are in a normal fighting style.", ch);
        break;

        case STYLE_TIGER:
        char_act("You are fighting in tiger style.", ch);
        break;

        case STYLE_COBRA:
        char_act("You are fighting in cobra style.", ch);
        break;

        case STYLE_IRONBODY:
        char_act("You are fighting in iron body style.", ch);
        break;

        case STYLE_DRUNKMAN:
        char_act("You are fighting in drunk man style.", ch);
        break;
        }
        return;
    }

    if (!str_prefix(arg, "normal") || !str_prefix(arg, "relax"))
    {
        ch->style       = STYLE_NORMAL;
        affect_strip(ch, gsn_style);
        char_act("You return to your normal style.", ch);
        return;
    } 
    if (!str_prefix(arg, "tiger"))
    {
        affect_strip(ch, gsn_style);

        af.where        = TO_AFFECTS;
        af.type         = gsn_style;
        af.level        = LVL(ch);
        af.duration     = -1;

        af.location     = APPLY_STR;
        af.modifier     = 5;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_HITROLL;
        af.modifier     = LVL(ch) / 20 + 1;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        ch->style       = STYLE_TIGER;
        char_act("You start moving like a tiger.", ch);
        check_improve(ch, gsn_style, TRUE, 10);
        return;
    } 
    if (!str_prefix(arg, "cobra"))
    {
        affect_strip(ch, gsn_style);

        af.where        = TO_AFFECTS;
        af.type         = gsn_style;
        af.level        = LVL(ch);
        af.duration     = -1;

        af.location     = APPLY_DEX;
        af.modifier     = 5;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_HITROLL;
        af.modifier     = LVL(ch) / 20 + 1;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        ch->style       = STYLE_COBRA;
        char_act("You start moving like a cobra.", ch);
        check_improve(ch, gsn_style, TRUE, 10);
        return;
    } 
    if (!str_prefix(arg, "ironbody"))
    {
        if (IS_PUMPED(ch))
        {
            char_act("alm down!", ch);
            return;
        }
        affect_strip(ch, gsn_style);

        af.where        = TO_AFFECTS;
        af.type         = gsn_style;
        af.level        = LVL(ch);
        af.duration     = -1;

        af.location     = APPLY_DEX;
        af.modifier     = -5;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_STR;
        af.modifier     = 5;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_HITROLL;
        af.modifier     = -1 * (LVL(ch) / 10 + 1);
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_AC;
        af.modifier     = -1 * (LVL(ch) / 2 + 1);
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_HIT;
        af.modifier     = LVL(ch) * 2;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_DAMROLL;
        af.modifier     = LVL(ch) / 5 + 2;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        ch->style       = STYLE_IRONBODY;
        char_act("Your body covered with iron.", ch);
        check_improve(ch, gsn_style, TRUE, 10);
        return;
    } 
    if (!str_prefix(arg, "drunkman"))
    {
        if (ch->pcdata->condition[COND_DRUNK] < 1)
        {
            char_act("You must be drunk.", ch);
            return;
        }
        affect_strip(ch, gsn_style);

        af.where        = TO_AFFECTS;
        af.type         = gsn_style;
        af.level        = LVL(ch);
        af.duration     = -1;

        af.location     = APPLY_DEX;
        af.modifier     = 10;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_STR;
        af.modifier     = 10;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_INT;
        af.modifier     = -10;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_WIS;
        af.modifier     = -10;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_HITROLL;
        af.modifier     = -1 * (LVL(ch) / 2 + 5);
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_DAMROLL;
        af.modifier     = LVL(ch) / 2 + 5;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        ch->style       = STYLE_DRUNKMAN;
        char_act("You start moving like a drunk man.", ch);
        check_improve(ch, gsn_style, TRUE, 1);
        return;
    } 
    char_act("Assume which combat style?", ch);
    return;
}

void do_feint(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance;
    AFFECT_DATA af;

    if ((chance = get_skill(ch, gsn_feint)) == 0) 
    {
        char_act("You don't know how to feint opponents.", ch);
        return;
    }

    victim = ch->fighting;
    if (victim == NULL) 
    {
        char_act("But you aren't fighting anyone.", ch);
        return;
    }

    if (ch->stance == STANCE_OFFENSE || ch->stance == STANCE_OFFENSE_II) 
    {
        char_act("You can't feint while in an offensive stance.", ch);
        return;
    }

    if (is_affected(victim, gsn_feint)) 
    {
        char_act("They are already being mislead.", ch);
        return;
    }

    if (LVL(victim) > LVL(ch))
        chance += (LVL(ch) - LVL(victim)) * 4;
    else
        chance += (LVL(ch) - LVL(victim)) * 2;

    chance = URANGE(5, chance, 90);
    if (number_percent() > chance) 
    {
        act("You try to mislead $N with a feint, but $gN{he} sees through it.",
            ch, 0, victim, TO_CHAR);
        act("$n tries to mislead you with a feint, but you see through it.",
            ch, 0, victim, TO_VICT);
        act("$n tries to mislead $N with a feint, but $gN{he} sees through  it.",
            ch, 0, victim, TO_NOTVICT);
        check_improve(ch, gsn_feint, FALSE, 2);
        WAIT_STATE(ch, SKILL(gsn_feint)->beats);
        return;
    }

    act("You mislead $N with a cunning feint.", ch, 0, victim, TO_CHAR);
    act("$n misleads you with a cunning feint.", ch, 0, victim, TO_VICT);
    act("$n misleads $N with a cunning feint.", ch, 0, victim, TO_NOTVICT);

    af.where        = TO_AFFECTS;
    af.type         = gsn_feint;
    af.bitvector    = 0;
    af.modifier     = 0;
    af.duration     = 1;
    af.location     = 0;
    af.level        = LVL(ch);
    affect_to_char(victim, &af);

    check_improve(ch, gsn_feint, TRUE, 2);
    WAIT_STATE(ch, SKILL(gsn_feint)->beats);
    return;
}

void do_flurry(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *wield;
    OBJ_DATA *dwield;

    if ((chance = get_skill(ch,gsn_flurry) ) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
 
    victim = ch->fighting;
    if (victim == NULL) 
    {
        char_act("But you aren't fighting anyone!", ch);
        return; 
    }

    wield = get_eq_char(ch, WEAR_WIELD);
    dwield = get_eq_char(ch, WEAR_SECOND_WIELD);

    if (wield == NULL || dwield == NULL) 
    {
        char_act("You must be dual wielding swords to flurry.", ch);
        return;
    }

    if (wield->value[0] != WEAPON_SWORD || dwield->value[0] != WEAPON_SWORD) 
    {
        char_act("You must be dual wielding swords to flurry.", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_SLOW)) 
    {
        char_act("You are moving too slowly to flurry.", ch);
        return;
    }

    if (ch->stance != STANCE_OFFENSE && ch->stance != STANCE_OFFENSE_II) 
    {
        char_act("You must be in an offensive stance to flurry.", ch);
        return;
    }

    chance /= 2;
    chance += get_curr_stat(ch, STAT_STR) / 2;
    chance += get_curr_stat(ch, STAT_DEX) / 3;

    if (IS_AFFECTED(ch, AFF_HASTE))
        chance += 5;

    chance += LVL(ch) - LVL(victim);
    chance = URANGE(5, chance, 95);

    if (number_percent() >= chance) 
    {
        act("$n tries to do something fancy with $gn{his} swords but fails.", 
            ch, 0, 0, TO_ROOM);
        char_act("You try to flurry but fail to gain the momentum.", ch);
        WAIT_STATE(ch, SKILL(gsn_flurry)->beats / 2);
        check_improve(ch, gsn_flurry, FALSE, 1);
        return;
    }

    act("$n performs a complicated series of rapid slashes.", ch, 0, 0, TO_ROOM);
    char_act("You perform a series of rapid slashes.", ch);
  
    one_hit(ch, victim, gsn_flurry, WEAR_WIELD);
    if (dice(5,6) < get_curr_stat(ch, STAT_STR))
        one_hit(ch, victim, gsn_flurry, WEAR_SECOND_WIELD);
    if (dice(6,6) < get_curr_stat(ch, STAT_STR))
        one_hit(ch, victim, gsn_flurry, WEAR_WIELD);
    if (dice(6,6) < get_curr_stat(ch, STAT_STR))
        one_hit(ch, victim, gsn_flurry, WEAR_SECOND_WIELD);
    if (dice(7,6) < get_curr_stat(ch, STAT_STR))
        one_hit(ch, victim, gsn_flurry, WEAR_WIELD);
    if (dice(8,6) < get_curr_stat(ch, STAT_STR))
        one_hit(ch, victim, gsn_flurry,WEAR_SECOND_WIELD);
  
    check_improve(ch, gsn_flurry, TRUE, 1);
    WAIT_STATE(ch, SKILL(gsn_flurry)->beats);
    return;
}

void do_crossslice(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *wield;
    OBJ_DATA *dwield;
    AFFECT_DATA af;

    if ((chance = get_skill(ch,gsn_cross_slice)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
 
    victim = ch->fighting;
    if (victim == NULL) 
    {
        char_act("But you aren't fighting anyone!", ch);
        return; 
    }

    wield = get_eq_char(ch, WEAR_WIELD);
    dwield = get_eq_char(ch, WEAR_SECOND_WIELD);

    if (wield == NULL || dwield == NULL) 
    {
        char_act("You must be dual wielding swords to cross-slice.", ch);
        return;
    }

    if (wield->value[0] != WEAPON_SWORD || dwield->value[0] != WEAPON_SWORD) 
    {
        char_act("You must be dual wielding swords to cross-slice.", ch);
        return;
    }

    if (ch->stance == STANCE_DEFENSE || ch->stance == STANCE_DEFENSE_II) 
    {
        char_act("You are concentrating too much on defense to cross-slice.", ch);
        return;
    }

    if (MOUNTED(ch))     
        chance -= 15;
        
    chance /= 2;
    chance += get_curr_stat(ch, STAT_STR) / 2;
    chance += get_curr_stat(ch, STAT_DEX) / 3;

    if (IS_AFFECTED(ch, AFF_HASTE))
        chance += 5;
  
    chance += LVL(ch) / 4;
    chance = URANGE(5, chance, 95);

    if (number_percent() >= chance) 
    {
        act("$n tries to do something fancy with $gn{his} swords but fails.",
            ch, 0, 0, TO_ROOM);
        char_act("You mix up your swords and fail to cross them properly.", ch);

        WAIT_STATE(ch, SKILL(gsn_cross_slice)->beats / 2);
        check_improve(ch, gsn_cross_slice, FALSE, 2);
        return;
    }
    act("$n slices $gn{his} swords across $N in a smooth scissor motion.",
        ch, 0, victim, TO_NOTVICT);
    act("You slice your swords across $N in a smooth scissor motion.",
        ch, 0, victim, TO_CHAR);
    act("$n slices $gn{his} swords across you in a smooth scissor motion.",
        ch, 0, victim, TO_VICT);
  
    one_hit(ch, victim, gsn_cross_slice, WEAR_WIELD);
    one_hit(ch, victim, gsn_cross_slice, WEAR_SECOND_WIELD);

    if (!is_affected(victim, gsn_cross_slice)
    && !check_mirror_image(ch, victim)) 
    {
        if (number_percent() < chance / 2) 
        {
            act("A gaping wound is torn across $n's body!", victim, 0, 0, TO_ROOM);
            act("A gaping wound is torn across your body!", victim, 0, 0, TO_CHAR);
            af.where        = TO_AFFECTS;
            af.type         = gsn_cross_slice;
            af.duration     = 2;
            af.level        = LVL(ch);
            af.modifier     = -2;
            af.bitvector    = 0;
            af.location     = APPLY_STR;
            affect_to_char(victim,&af);
        }
    }
    
    check_improve(ch, gsn_cross_slice, TRUE, 2);
    WAIT_STATE(ch, SKILL(gsn_cross_slice)->beats);

    return;
}

void do_pincer(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *wield;
    OBJ_DATA *dwield;
    char arg[MAX_INPUT_LENGTH];
    
    if ((chance = get_skill(ch,gsn_pincer) ) == 0) {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
     
    if (arg[0] == '\0')
      {
       victim = ch->fighting;
       if (victim == NULL
         || victim->in_room != ch->in_room) {
             char_act("But you aren't fighting anyone.", ch);
         return;
         }
      }
     else if ((victim = get_char_room(ch, arg)) == NULL) {
         char_act("They aren't here.", ch);
         return;
         }

    if (victim == ch
    || is_safe(ch, victim)) {
    char_act("You can't do that.", ch);
    return;
    }

  wield = get_eq_char(ch,WEAR_WIELD);
  dwield = get_eq_char(ch,WEAR_SECOND_WIELD);

  if (wield == NULL || dwield == NULL) {
  char_act("You must be dual wielding axes to pincer.", ch);
  return;
  }

  if ( (wield->value[0] != WEAPON_AXE) || (dwield->value[0] != WEAPON_AXE) ) {
  char_act("You must be dual wielding axes to pincer.", ch);
  return;
  }

  if (ch->stance == STANCE_DEFENSE || ch->stance == STANCE_DEFENSE_II) {
    char_act("You are concentrating too much on defense to pincer.", ch);
    return;
  }

  chance /= 2;
  chance += get_curr_stat(ch,STAT_STR)/2;
  chance += get_curr_stat(ch,STAT_DEX)/3;

  if (IS_AFFECTED(ch,AFF_HASTE))
    chance += 5;

  chance += LVL(ch) / 2;
  chance = URANGE(5,chance,95);

  if (number_percent() >= chance) {
  act("$n tries to do something fancy with $gn{his} axes but fails.",ch,0,0,TO_ROOM);
  char_act("You mix up your axes and fail.", ch);

  WAIT_STATE(ch, SKILL(gsn_pincer)->beats / 2);
  check_improve(ch,gsn_pincer,FALSE,1);
  return;
  }
  act("$n catches $N between $gn{his} axes in a pincer.",ch,0,victim,TO_NOTVICT);
  act("$n catches you between $gn{his} axes in a pincer.",ch,0,victim,TO_VICT);
  act("You catch $N between $gn{his} axes in a pincer.",ch,0,victim,TO_CHAR);
 
  one_hit(ch,victim,gsn_pincer,WEAR_WIELD);
  one_hit(ch,victim,gsn_pincer,WEAR_SECOND_WIELD);

  check_improve(ch,gsn_pincer,TRUE,2);
  WAIT_STATE(ch, SKILL(gsn_pincer)->beats);
  WAIT_STATE(victim, SKILL(gsn_pincer)->beats / 2);
  return;
}

void do_bleed(CHAR_DATA *ch,const char *argument) 
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;  
    AFFECT_DATA af;
    one_argument( argument, arg , sizeof(arg ));

    if ((chance = get_skill(ch,gsn_bleed)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (arg[0] == '\0')
        victim = ch->fighting;
    else 
        victim = get_char_room(ch, arg);

    if (victim == NULL)
    {
        char_act("They aren't here.", ch);
        return;
    }
    
    if (victim == ch)
    {
        char_act("You can't do that.", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL)
    {
        char_act("You need to wield a dagger to bleed someone.", ch);
        return;
    }
     
    if ( obj->value[0] != WEAPON_DAGGER )
    {
        char_act("You must wield a dagger to bleed someone.", ch);
        return;
    }
    
    if (is_safe(ch, victim))
        return;

    if (MOUNTED(ch) && !MOUNTED(victim))
    {
        char_act("You can't bleed an unmounted victim if you are high on a mount.", ch);
        return;
    }

    chance /= 2;
    chance += LVL(ch);
    chance -= LVL(victim);

    chance = URANGE(2, chance, 85);

    if (IS_UNDEAD(victim) || IS_CYBORG(victim))
    chance = 0;
        
    WAIT_STATE(ch, SKILL(gsn_bleed)->beats);    

    if(check_mirror_image(ch, victim))
        return;

    if (number_percent() >= chance)
    {
        char_act("You try to sever an artery, but fail.", ch);
        act("$n slashes at you in an attempt to sever a vital artery.",
            ch, 0, victim, TO_VICT);
        check_improve(ch, gsn_bleed, FALSE, 1);
        one_hit(ch, victim, gsn_bleed, WEAR_WIELD);
        return;
    }

    act("You slice open a vital artery on $N's body!", ch, 0, victim, TO_CHAR);
    act("$n slices open a vital artery on your body!", ch, 0, victim, TO_VICT);
    check_improve(ch, gsn_bleed, TRUE, 2);

    one_hit(ch, victim, gsn_bleed, WEAR_WIELD);
    if (JUST_KILLED (victim))
        return;

    if (!is_affected(victim, gsn_bleed))
        one_hit(ch, victim, gsn_bleed, WEAR_WIELD);
    if (JUST_KILLED (victim))
        return;
    
    if (!is_affected(victim, gsn_bleed))
    {
        af.type         = gsn_bleed;
        af.where        = TO_AFFECTS;
        af.level        = ch->level;
        af.bitvector    = 0;
        af.duration     = 5;
        af.modifier     = -5;
        af.location     = APPLY_CON;
        affect_to_char(victim,&af);
        af.modifier     = -3;
        af.location     = APPLY_STR;
        affect_to_char(victim,&af);
        af.modifier     = -3;
        af.location     = APPLY_DEX;
        affect_to_char(victim,&af);
    }
    return;
}

DO_FUN(do_undercut) {
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;  
    one_argument( argument, arg , sizeof(arg ));

    if ((chance = get_skill(ch,gsn_undercut)) == 0) {
    char_act("Huh?", ch);
    return;
    }

    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room)
        {
            char_act("But you aren't fighting anyone.", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim == ch)
    {
        char_act("You can't do that.", ch);
        return;
    }

    if ( ( obj = get_eq_char( ch, WEAR_WIELD ) ) == NULL)
    {
        char_act("You need to wield a dagger to undercut.", ch );
        return;
    }
     
    if ( obj->value[0] != WEAPON_DAGGER )
    {
        char_act("You must wield a dagger to undercut.", ch );
        return;
    }
    
    if ( is_safe( ch, victim ) )
        return;

    if ( MOUNTED(victim) && !MOUNTED(ch) )
    {
        char_act("They are mounted too high for you to undercut.", ch);
        return;
    }

    chance -= LVL(victim);
    chance += LVL(ch);
    chance -= get_skill(victim,gsn_dodge)/2;
    chance += get_curr_stat(ch,STAT_DEX);
    chance -= get_curr_stat(victim,STAT_DEX)/2;

    chance = URANGE(5,chance,85);

    if (victim->hit <= victim->max_hit / 10)
        chance += 15;

    if (number_percent() >= chance) 
    {
        act("$n tries to dodge beneath your defences but fails.",ch,0,victim,TO_VICT);
        act("You try to dodge beneath $N's defences but fail.",ch,0,victim,TO_CHAR);
        check_improve(ch,gsn_undercut,FALSE,1);
        WAIT_STATE(ch,  SKILL(gsn_undercut)->beats * 2/3);
        return;
    }

    act("You slip behind $N's defences and deliver strikes to $gN{his} ribs.",ch,0,victim,TO_CHAR);
    act("$n slips behind your defences and delivers strikes to your ribs.",ch,0,victim,TO_VICT);

    chance = get_skill(ch,gsn_undercut);

    one_hit(ch,victim,gsn_undercut,WEAR_WIELD);

    if (number_percent() < chance)
        one_hit(ch,victim,gsn_undercut,WEAR_WIELD);

    if (number_percent() < chance/2)
        one_hit(ch,victim,gsn_undercut,WEAR_WIELD);

    if (number_percent() < chance/4)
        one_hit(ch,victim,gsn_undercut,WEAR_WIELD);

    if (number_percent() < chance/6)
        one_hit(ch,victim,gsn_undercut,WEAR_WIELD);

    check_improve(ch,gsn_undercut,TRUE,2);

    WAIT_STATE(ch, SKILL(gsn_undercut)->beats);

    return;
}

void do_choke(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    char arg[MAX_INPUT_LENGTH];
    char buf[MAX_INPUT_LENGTH];
    int chance;
    AFFECT_DATA af;
    OBJ_DATA *weapon;
    int wield;

    one_argument (argument, arg, sizeof(arg));
    if ( (chance = get_skill (ch,gsn_choke)) == 0) 
    {
        char_act ("Huh?", ch);
        return;
    }

    weapon = get_eq_char (ch,WEAR_WIELD);
    wield = WEAR_WIELD;
    if (weapon == NULL) 
    {
        char_act("You must be wielding a whip or flail to choke.", ch);
        return;
    }

    if (weapon->value[0] != WEAPON_FLAIL && weapon->value[0] != WEAPON_WHIP)
    {
        weapon = get_eq_char (ch,WEAR_SECOND_WIELD);
        wield = WEAR_SECOND_WIELD;
    }
    
    if (weapon == NULL )
        return;

    if (weapon->value[0] != WEAPON_FLAIL && weapon->value[0] != WEAPON_WHIP)
    {
        char_act("You must be wielding a whip or flail to choke.", ch);
        return;
    }

    if (arg[0] == '\0')
        victim = ch->fighting;
    else 
        victim = get_char_room(ch, arg);

    if (victim == NULL) 
    {
        char_act("Choke who?", ch);
        return;
    }

    if (victim == ch || IS_CLAN_GUARD(victim) || IS_IMMORTAL(victim)) 
    {
        char_act ("You can't do that.", ch);
        return;
    }

    if (is_safe(ch,victim))
        return;

    if (!IS_AWAKE(victim)) 
    {
        char_act("They are already lying on the ground unconcious.", ch);
        return;
    }

    if (is_affected (victim, gsn_insomnia))
    {
        act ("But $N can't sleep.", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (chance == 100)
        chance = 15;
    else if (chance > 90)
        chance /=8;
    else
        chance /=10;

    chance += (LVL(ch) - LVL(victim));
    chance = URANGE(5,chance,15);

    WAIT_STATE(ch, SKILL(gsn_choke)->beats);
    if (check_mirror_image (ch, victim))
        return;
        
    if (number_percent() > chance
        || (IS_UNDEAD(victim)) )
    {
        if (!IS_NPC(ch) && !IS_NPC(victim)
            && (ch->fighting == NULL || victim->fighting == NULL)) 
        {
            act_yell(victim, "Help! $i just tried to choke me!", ch, NULL);
        } 
        else if (RIDDEN(victim)
                 && !IS_NPC(victim->mount) && !IS_NPC(ch)
                 && (victim->mount->fighting == NULL || ch->fighting == NULL)) 
        {
            doprintf (do_yell, victim->mount, buf, "Help, %s is attacking my mount!", ch->name);
        }
        one_hit (ch, victim, gsn_choke, wield);
        check_improve (ch, gsn_choke, FALSE, 1);
        return;
    }

    act("$n chokes $N into unconciousness with $gn{his} $p.", ch, weapon, victim, TO_NOTVICT);
    act("$n chokes you into unconciousness with $gn{his} $p.", ch, weapon, victim, TO_VICT); 
    act("You choke $N into unconciousness with your $p.", ch, weapon, victim, TO_CHAR);

    stop_fighting(victim,TRUE);
    af.where = TO_AFFECTS;
    af.type = gsn_choke;
    af.level = LVL(ch);
    af.duration = 1;
    af.modifier = 0;
    af.location = 0;
    af.bitvector = AFF_SLEEP;
    affect_to_char(victim,&af);
    victim->position = POS_SLEEPING;

    check_improve(ch,gsn_choke,TRUE,2);
    return;
}

void do_strip_weapon(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    OBJ_DATA *obj, *obj2, *obj3, *obj4, *weapon;
    int chance, loc;
    char arg[MAX_INPUT_LENGTH];

    if ((chance = get_skill(ch,gsn_strip_weapon)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument (argument, arg, sizeof(arg));
    if (arg[0] == '\0')
        victim = ch->fighting;
    else
        victim = get_char_room(ch, arg);

    if (victim == NULL)
    {
        char_act("Weapon strip who?", ch);
        return;
    }
    weapon = get_eq_char(ch,WEAR_WIELD);
    if (weapon == NULL)
    {
        char_act("You must be wielding a whip or flail to strip.", ch);
        return;
    }
    if (weapon->value[0] != WEAPON_FLAIL && weapon->value[0] != WEAPON_WHIP)
        weapon = get_eq_char(ch,WEAR_SECOND_WIELD);

    if (weapon->value[0] != WEAPON_FLAIL && weapon->value[0] != WEAPON_WHIP)
    {
        char_act("You must be wielding a whip or flail to strip.", ch);
        return;
    }
    
    if(is_safe(ch,victim))
        return;
    loc = 0;
    obj  = get_eq_char(victim, WEAR_WIELD);
    obj2 = get_eq_char(victim, WEAR_SECOND_WIELD);
    obj3 = get_eq_char(victim, WEAR_SHIELD);
    obj4 = get_eq_char(victim, WEAR_HOLD);
    if (!obj && !obj2 && !obj3 && !obj4 )
    {
        char_act("They aren't wielding a weapon, shield or hold.", ch);
        return;
    }
    if ((!obj || IS_OBJ_STAT(obj, ITEM_NOREMOVE)) && obj2) 
        obj = obj2;
    if ((!obj || IS_OBJ_STAT(obj, ITEM_NOREMOVE)) && obj3)
    {
        obj = obj3;
        loc = 1;
    }
    if ((!obj || IS_OBJ_STAT(obj, ITEM_NOREMOVE)) && obj4)
    { 
        obj = obj4;
        loc = 2;
    }
    
    chance /= 2;
    chance += get_curr_stat(ch,STAT_DEX);

    if (IS_AWAKE(victim))
        chance -= get_curr_stat(victim,STAT_DEX);

    if (number_percent() > chance || IS_OBJ_STAT(obj,ITEM_NOREMOVE))
    {
        if (!IS_NPC(ch) && !IS_NPC(victim)
            && (victim->fighting == NULL || ch->fighting == NULL)) 
            doprintf(do_yell,victim,"Help! %s is trying to strip my %s!",
                PERS(ch,victim), loc ? ((loc-1)?"hold":"shield"):"weapon");
    
        act("$N manages to hold onto $gN{his} $p.", ch, obj, victim, TO_CHAR);
        act("$N tries to strip your $p away, but fails.", ch, obj, victim, TO_VICT);
        check_improve(ch, gsn_strip_weapon, FALSE, 1);
        WAIT_STATE(ch, SKILL(gsn_strip_weapon)->beats * 2/3);
        multi_hit (victim, ch, TYPE_UNDEFINED);
        return;
    }


    if (!IS_NPC(ch) && !IS_NPC(victim)
        && (victim->fighting == NULL || ch->fighting == NULL)
        && !check_mirror_image(ch,victim)) 
        doprintf(do_yell,victim,"Help! %s just stripped my %s!",
            PERS(ch,victim), loc ? ((loc-1)?"hold":"shield"):"weapon");
  
    act( "$n strips your $p and sends it flying!", ch, obj, victim, TO_VICT    );
    act( "You strip $N's $p and send it flying!",  ch, obj, victim, TO_CHAR );
    act( "$n strips $N's $p and sends it flying!", ch, obj, victim, TO_NOTVICT );

    obj_from_char (obj);

    if (IS_OBJ_STAT(obj,ITEM_NODROP)
    || IS_OBJ_STAT(obj,ITEM_INVENTORY)
    || (victim->in_war && victim->war_status == PS_ALIVE)
    || (victim->in_room
    && IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA)))
        obj_to_char( obj, victim );
    else 
    {
        if( number_percent() < chance )
        {
            act("$n caught $p with a swift snatch!", ch, obj, victim, TO_ROOM);
            act("You've caught $p with a swift snatch!", ch, obj, victim, TO_CHAR);
            obj_to_char( obj, ch );
        }
        else 
            obj_to_room( obj, victim->in_room );
    }    
    WAIT_STATE(ch, SKILL(gsn_strip_weapon)->beats);
    multi_hit(victim,ch,TYPE_UNDEFINED);

    if ((obj = get_eq_char(victim, WEAR_WIELD)) == NULL
    && (obj2 = get_eq_char(victim, WEAR_SECOND_WIELD)) != NULL)
    {
        act_puts("You take your weapon with the other hand!", ch, NULL, victim, TO_VICT, POS_FIGHTING);
        act_puts("$N takes the weapon with the other hand!", ch, NULL, victim, TO_CHAR, POS_FIGHTING);
        act_puts("$N takes the weapon with the other hand!", ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        unequip_char(victim, obj2);
        equip_char(victim, obj2, WEAR_WIELD);
    }
    return;
}

void do_eyejab(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *wield;
    AFFECT_DATA af;
    char arg[MAX_INPUT_LENGTH];
    
    if ((chance = get_skill(ch,gsn_eyejab)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
     
    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room) 
        {
            char_act("But you aren't fighting anyone.", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("You can't do that.", ch);
        return;
    }

    if (is_safe (ch, victim)) return ;

    wield = get_eq_char(ch,WEAR_WIELD);

    if (wield == NULL)
    {
        char_act("You must be wielding a whip or flail to eyejab.", ch);
        return;
    }

    if (wield->value[0] != WEAPON_WHIP
    && wield->value[0] != WEAPON_FLAIL)
        wield = get_eq_char(ch,WEAR_SECOND_WIELD);

    if (wield == NULL)
    {
        char_act("You must be wielding a whip or flail to eyejab.", ch);
        return;
    }

    if (wield->value[0] != WEAPON_WHIP
    && wield->value[0] != WEAPON_FLAIL)
    {
        char_act("You must be wielding a whip or flail to eyejab.", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_HASTE))
        chance += 10;
    if (IS_AFFECTED(ch, AFF_SLOW))
        chance -= 10;

    chance += get_curr_stat(ch,STAT_DEX);
    chance -= get_curr_stat(victim,STAT_DEX);

    chance = chance *3/5;
    WAIT_STATE(ch, SKILL(gsn_eyejab)->beats);

    if(check_mirror_image(ch, victim))
        return;
        
    if (number_percent() < chance) /* Hit in eyes, blinded */
    {
        act("You catch $N's eye with a sharp flick of $p!",
                    ch, wield, victim, TO_CHAR);
        act("$n flicks $p and manages to catch you in the eye!",
                    ch, wield, victim, TO_VICT);
        damage(ch, victim, number_range(LVL(ch)/2, LVL(ch)), gsn_eyejab, DAM_SLASH,TRUE);
        check_improve(victim, gsn_eyejab, TRUE, 2);

        if (IS_UNDEAD(victim)) 
        {
            act("You attempt to jab $N's eyes , but sign off his rotten body.",
                        ch, wield, victim, TO_CHAR);
            act("You are unaffected to $n's eyejab.",
                        ch, wield, victim, TO_VICT);
            return;
        }

        if (IS_AFFECTED(victim, AFF_BLIND))
        {
            act("$N' already blinded!",ch,0,victim,TO_CHAR);
            return;
        }

        af.where = TO_AFFECTS;
        af.type = gsn_eyejab;
        af.modifier = -number_fuzzy(LVL(ch)/5);
        af.location = APPLY_HITROLL;
        af.bitvector = AFF_BLIND;
        af.duration = 1;
        af.level = LVL(ch);
        affect_to_char(victim,&af);
        return;
    }

    chance *=2 ;
    check_improve(victim,gsn_eyejab,FALSE,2);
    if (number_percent() < chance) /* Missed, but got a damage */
    {
        act("You try to catch $N's eyes with $p but hit in the $N's head.",
                      ch, wield, victim, TO_CHAR);
        damage(ch, victim, number_range(LVL(ch)*2,LVL(ch)*3), gsn_eyejab, DAM_SLASH, TRUE);
    } else 
        act("You try to catch $N's eyes with $p but miss.",
                    ch, wield, victim, TO_CHAR);
    return;
}

void do_backhand(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    int chance, dam;
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *wield;
    dam = 0;
    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch,gsn_backhand)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
    if (arg[0] == '\0') 
    {
        victim = ch->fighting;
        if (victim == NULL) 
        {
            char_act("But you aren't fighting anyone!", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    } if (victim == ch) 
    {
        char_act("You can't backhand yourself!", ch);
        return;
    }
        
    wield = get_eq_char(ch, WEAR_WIELD);
    if ((wield == NULL) || ( wield->value[0] != WEAPON_MACE)) 
    {
        wield = get_eq_char(ch, WEAR_SECOND_WIELD);
        chance -= 20;
    }

    if ((wield == NULL) || ( wield->value[0] != WEAPON_MACE)) 
    { 
        char_act("You must be wielding a mace to backhand.", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;

    chance /= 2;
    chance += get_curr_stat(ch,STAT_STR)*3/2;         
    chance += (LVL(ch) - LVL(victim))*2;
    chance -= get_curr_stat(victim,STAT_DEX);
    chance += wield->level/3;
    act("$n makes a powerful backhand blow with $p.",
                    ch, wield, victim, TO_ROOM);
    act("You make a powerful backhand blow with $p.",
                    ch, wield, victim, TO_CHAR);
        
    if (number_percent() > chance) 
    {
        check_improve(ch, gsn_backhand, FALSE, 1);
        damage(ch, victim, 0, gsn_backhand, DAM_BASH, TRUE);
        WAIT_STATE(ch, SKILL(gsn_backhand)->beats);
        return;
    }

    dam = dice(wield->value[1],wield->value[2]) * 
                        get_skill(ch,gsn_mace)/100;
    chance = number_percent();
    if (chance < get_skill(ch,gsn_enhanced_damage))
        dam += chance * dam/100;
    dam += (wield->weight * wield->level)/100;
    chance = number_percent();
    if (chance < get_skill(ch, gsn_backhand)/10)
        dam *= 4;
    else if (chance < get_skill(ch, gsn_backhand)/2)
        dam *= 3;
    else
        dam = dam*5/2;

    dam += LVL(ch)*2;

    check_improve(ch, gsn_backhand, TRUE, 2);
    damage(ch, victim, dam, gsn_backhand, DAM_BASH, TRUE);
    WAIT_STATE(ch, SKILL(gsn_backhand)->beats);
    return;
}

void do_drumming_maces(CHAR_DATA *ch, const char *argument) 
{
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;   
    char arg[MAX_INPUT_LENGTH]; 
    
    if ((chance = get_skill(ch,gsn_drumming_maces)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
    one_argument(argument, arg, sizeof(arg));
     
    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room) 
        {
            char_act("But you aren't fighting anyone.", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("You can't do that.", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL) 
    {
        char_act("You need to dual wield maces to drum.", ch);
        return;
    }
     
    if (obj->value[0] != WEAPON_MACE) 
    {
        char_act("You must dual wield maces to drum.", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_SECOND_WIELD)) == NULL) 
    {
        char_act("You need to dual wield maces to drum.", ch);
        return;
    }
     
    if (obj->value[0] != WEAPON_MACE) 
    {
        char_act("You must dual wield maces to drum.", ch);
        return;
    }

    if (is_safe(ch, victim))  
        return;

    if (MOUNTED(victim) && !MOUNTED(ch)) 
    {
        act("$N is mounted too high for you to drum $gN{his} head.",
                        ch, NULL, victim, TO_CHAR);
        return;
    }
        
    chance += LVL(ch);
    chance -= LVL(victim)/2;
    chance -= get_skill(victim,gsn_dodge)/2;
    chance += get_curr_stat(ch,STAT_DEX)/2;
    chance += get_curr_stat(ch,STAT_STR);
    chance -= get_curr_stat(victim,STAT_DEX)/2;
    
    chance = URANGE(5,chance,85);
    
    if (number_percent() >= chance) 
    {
        act("$n swings $gn{his} maces wildly but fails to connect with $gn{his} target.",
                    ch, NULL, NULL, TO_ROOM);
        char_act("You swing your maces about wildly but fail to hit anything.", ch);
        check_improve(ch,gsn_drumming_maces,FALSE,1);
        WAIT_STATE(ch, SKILL(gsn_drumming_maces)->beats);
        return;
    }   
    
    act("$n does a rapid drumming against $N's head with $gn{his} maces.",
                    ch, NULL, victim, TO_NOTVICT);
    act("$n does a rapid drumming against your head with $gn{his} maces.",
                    ch, NULL, victim, TO_VICT);
    act("You do a rapid drumming against $N's head with your maces.",
                    ch, NULL, victim, TO_CHAR);

    chance = get_skill(ch,gsn_drumming_maces);

    one_hit(ch,victim,gsn_drumming_maces,WEAR_WIELD);
    one_hit(ch,victim,gsn_drumming_maces,WEAR_SECOND_WIELD);

    if (number_percent() < chance/2)
        one_hit(ch,victim,gsn_drumming_maces,WEAR_WIELD);

    if (number_percent() < chance/2)
        one_hit(ch,victim,gsn_drumming_maces,WEAR_SECOND_WIELD);

    if (number_percent() < chance/4)
        one_hit(ch,victim,gsn_drumming_maces,WEAR_WIELD);

    if (number_percent() > get_curr_stat(victim,STAT_CON) * 2 
    && !check_mirror_image(ch, victim)) 
    {
        act("$n looks dazed from the blows.", victim, NULL, NULL, TO_ROOM);
        char_act("You are dazed by the blows to your head.", victim);
        WAIT_STATE(victim, SKILL(gsn_drumming_maces)->beats);
        DAZE_STATE(victim, SKILL(gsn_drumming_maces)->beats);
    }

    check_improve(ch,gsn_drumming_maces,TRUE,1); 
    WAIT_STATE(ch, SKILL(gsn_drumming_maces)->beats); 
    return;
}   

void do_boneshatter(CHAR_DATA *ch,const char *argument) 
{
    CHAR_DATA *victim;
    OBJ_DATA *obj, *obj2;
    int chance, dam, mod;  
    AFFECT_DATA af, *paf;
    char arg[MAX_INPUT_LENGTH];
    
    if ((chance = get_skill(ch,gsn_boneshatter)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
     
    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room) 
        {
            char_act("But you aren't fighting anyone.", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("You can't do that.", ch);
        return;
    }
    
    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL) 
    {
        char_act("You need to wield a mace to boneshatter someone.", ch);
        return;
    }
     
    if (obj->value[0] != WEAPON_MACE) 
    {
        char_act("You must wield a mace to boneshatter someone.", ch);
        return;
    }
    mod = 0;
    if (is_safe(ch, victim)) 
        return;
    for(paf = victim->affected; paf != NULL; paf = paf->next)
    {
        if(paf->type != gsn_boneshatter || paf->location != APPLY_NONE)
            continue;
        if(paf->modifier >= 2) 
        {
            char_act("They already have shattered bones of both hands.", ch);
            return;
        }
        mod = 1;
        break;
    }
    
    chance /= 2;
    chance += (LVL(ch)-LVL(victim));
    chance -= get_curr_stat(victim, STAT_CON);
    chance =  URANGE(10,chance,85);
  
    WAIT_STATE(ch, SKILL(gsn_boneshatter)->beats);

    if (check_mirror_image(ch,victim))
        return;

    if (number_percent() >= chance) 
    {
        char_act("You try to land a quick crushing blow, but fail.", ch);
        act("$n attempts a quick crushing blow, but fails.",ch,0,victim,TO_VICT);
        check_improve(ch, gsn_boneshatter, FALSE, 1);
        one_hit(ch, victim, gsn_boneshatter, WEAR_WIELD);
        return;
    }

    act("You land a sudden crushing blow on $N's wrist!",ch,0,victim,TO_CHAR);
    act("$n lands a sudden crushing blow to your hand!",ch,0,victim,TO_VICT);
    check_improve(ch, gsn_boneshatter, TRUE, 2);

    dam = GET_DAMROLL(ch) + LVL(ch);

    damage(ch, victim, dam, gsn_boneshatter, DAM_BASH, TRUE);

    if (mod)
        paf->modifier = 2;
    else 
    {
        af.type = gsn_boneshatter;
        af.where = TO_AFFECTS;
        af.level = LVL(ch);
        af.bitvector = 0;
        af.duration = 5;
        af.modifier = -5;
        af.location = APPLY_STR;
        affect_to_char(victim,&af);
        af.modifier = -1;
        af.location = APPLY_DEX;
        affect_to_char(victim,&af);
        af.modifier = 1;
        af.location = APPLY_NONE;
        affect_to_char(victim,&af);
    }
    
    if ((obj2 = get_eq_char(victim, WEAR_SECOND_WIELD)) != NULL) 
    {
        act("You cannot hold $p with the broken arm.", 
                        victim, obj2, NULL, TO_CHAR);
        obj_from_char(obj2);
        obj_to_char(obj2, victim);
    }
    if (mod && (obj2 = get_eq_char(victim, WEAR_WIELD)) != NULL) 
    {
        act("You cannot hold $p with the broken arm.", 
                        victim, obj2, NULL, TO_CHAR);
        obj_from_char(obj2);
        obj_to_char(obj2, victim);
    }

    return;
}

void do_charge_set(CHAR_DATA *ch,const char *argument) 
{
    AFFECT_DATA af;
    OBJ_DATA *wield;

    if (get_skill(ch, gsn_charge_set) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
    if (is_affected(ch, gsn_charge_set)) 
    {
        char_act("You stop preparing to impale attackers on your weapon.", ch);
        affect_strip(ch, gsn_charge_set);
        return;
    }
    wield = get_eq_char(ch,WEAR_WIELD);

    if (wield == NULL
    || (wield->value[0] != WEAPON_SPEAR
    && wield->value[0] != WEAPON_LANCE
    && wield->value[0] != WEAPON_POLEARM)) 
    {
        char_act("You need to be wielding a spear, polearm or lance to charge set.", ch);
        return;
    }

    if (number_percent() >= get_skill(ch, gsn_charge_set)) 
    {
        char_act("You try to prepare yourself for attackers but can't quite get the position right.", ch);
        check_improve(ch, gsn_charge_set, FALSE, 1);
        WAIT_STATE(ch, SKILL(gsn_charge_set)->beats);
        return;
    }
  
    af.where        = TO_AFFECTS;
    af.type         = gsn_charge_set;
    af.modifier     = 0;
    af.duration     = 5;
    af.location     = 0;
    af.bitvector    = 0;
    af.level        = LVL(ch);
    affect_to_char(ch, &af);
  
    char_act("You prepare to impale attackers on your weapon.", ch);
    check_improve(ch, gsn_charge_set, TRUE, 2);
    WAIT_STATE(ch, SKILL(gsn_charge_set)->beats);

    return;
}

void do_leg_sweep(CHAR_DATA *ch, const char *argument) 
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *wield;
    int dam;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_leg_sweep)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
 
    wield = get_eq_char(ch,WEAR_WIELD);
    if (wield == NULL
    ||( wield->value[0] != WEAPON_SPEAR 
    && wield->value[0] != WEAPON_POLEARM)) 
    {
        char_act("You must wield a staff, spear or polearm to leg sweep.", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        victim = ch->fighting; 
        if (victim == NULL) 
        {
            char_act("But you aren't fighting anyone!", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }
 
    if (is_safe(ch, victim))
        return;
     
    if (IS_AFFECTED(victim, AFF_FLYING)) 
    {
        act("$gN{his} feet aren't on the ground.", ch, NULL, victim, TO_CHAR);
        return;
    }
     
    if (victim->position < POS_FIGHTING) 
    {
        act("$N is already down.", ch, NULL, victim, TO_CHAR);
        return;
    }
     
    if (victim == ch) 
    {
        char_act("You stumble over your own weapon!", ch);
        return;
    }
        
    if (IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim) 
    {
        act("$N is your beloved master.", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (MOUNTED(ch)) 
    {
        char_act("You can't leg sweep while mounted.", ch);
        return;
    }

    if (MOUNTED(victim)) 
    {
        char_act("They are mounted too high for you to leg sweep.", ch);
        return;
    }
        
    /* modifiers */
     
    /* size */
    if (ch->size < victim->size)
        chance += (ch->size - victim->size) * 10; 
        
    chance += get_curr_stat(ch,STAT_DEX);
    chance -= get_curr_stat(victim,STAT_DEX) * 3 / 2;
    if (IS_AFFECTED(ch, AFF_HASTE))        chance += 10;
    if (IS_AFFECTED(victim, AFF_HASTE))    chance -= 20;
    
    chance += (LVL(ch) - LVL(victim));
    if (RIDDEN(victim))
        chance -= chance / 4;

    if (number_percent() < chance) 
    {
        act("$n sweeps your legs out from under you!", ch, NULL, victim, TO_VICT);
        act("You sweep $N's legs out from under $gN{him}!", ch, NULL, victim, TO_CHAR);
        act("$n sweeps $N's legs out from under $gN{him}!", ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_leg_sweep, TRUE, 1);
        
        if (!IS_NPC(ch) && !IS_NPC(victim)
        && (ch->fighting == NULL || victim->fighting == NULL)) 
        {
            act_yell(victim, "Help! $i just knocked me down!", ch, NULL);
        } else if (RIDDEN(victim)
        && !IS_NPC(victim->mount) && !IS_NPC(ch)
        && (victim->mount->fighting == NULL || ch->fighting == NULL)) 
        {
            act_yell(victim, "Help, $i is attacking my mount!", ch, NULL);
        }

        victim->position = POS_RESTING;

        dam = dice(wield->value[1], wield->value[2]);
        if (LVL(ch) < 60)       dam += dam / 2;
        else if (LVL(ch) < 70)  dam *= 2;
        else if (LVL(ch) < 80)  dam += dam * 3 / 2;
        else                    dam *= 3;

        damage(ch, victim, dam, gsn_leg_sweep, DAM_BASH, TRUE);

        if (victim == NULL || ch == NULL)   
            return;
        if (check_recovery(victim, ch)) 
        {
            WAIT_STATE(ch, SKILL(gsn_leg_sweep)->beats);
            return;
        }
        if (RIDDEN(victim)) 
            if (number_percent() * 2 > (get_skill(victim->mount, gsn_riding)
                + get_skill(victim->mount, gsn_dragon_riding) 
                + get_skill(victim->mount, gsn_animal_riding)))
            {
                act("$n is thrown from $N.", victim->mount, 0, victim, TO_NOTVICT);
                act("$n is thrown from your back.", victim->mount, 0, victim, TO_VICT);
                act("You are thrown from $N.", victim->mount, 0, victim, TO_CHAR);
                WAIT_STATE(victim->mount, SKILL(gsn_leg_sweep)->beats * 2/3);
                check_improve(victim->mount, gsn_riding, FALSE, 1);
                damage(victim->mount, victim->mount, dice(5, 8), gsn_riding, DAM_BASH, TRUE);
                if (victim->mount != NULL) 
                {
                    victim->mount->mount = NULL;
                    victim->mount->position = POS_RESTING;
                    victim->mount = NULL;
                }
            }
            WAIT_STATE(ch, SKILL(gsn_leg_sweep)->beats);
            WAIT_STATE(victim, SKILL(gsn_leg_sweep)->beats * 2 / 3);
    } else 
    {   
        if (!IS_NPC(ch) && !IS_NPC(victim)
        && (ch->fighting == NULL || victim->fighting == NULL)) 
        {
            act_yell(victim, "Help! $i just tried to leg sweep me!", ch, NULL);
        } else if (RIDDEN(victim)
        && !IS_NPC(victim->mount) && !IS_NPC(ch)
        && (victim->mount->fighting == NULL || ch->fighting == NULL)) 
        {
            act_yell(victim, "Help, $i is attacking my mount!", ch, NULL);
        }

        damage(ch, victim, 0, gsn_leg_sweep, DAM_BASH, TRUE);
        WAIT_STATE(ch, SKILL(gsn_leg_sweep)->beats / 2);
        check_improve(ch, gsn_leg_sweep, FALSE, 1);
    }
    return;
}

void do_overhead(CHAR_DATA *ch,const char *argument)
{
    CHAR_DATA *victim;
    int chance, dam;
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *wield;

    dam = 0;
    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch,gsn_overhead)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL)
        {
            char_act("But you aren't fighting anyone!", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("They aren't here.", ch);
        return;
    }
    if (victim == ch)
    {
        char_act("You can't knock yourself on the head like that!", ch);
        return;
    }

    wield = get_eq_char(ch,WEAR_WIELD);
    if ((wield == NULL)
    || (wield->value[0] != WEAPON_SPEAR))
    {
        char_act("You must be wielding a staff to overhead.", ch);
        return;
    }

    if (is_safe(ch,victim))
        return;

    chance /= 2;
    chance += get_curr_stat(ch, STAT_STR)*3/2;
    chance += (LVL(ch) - LVL(victim))*2;
    chance -= get_curr_stat(victim, STAT_DEX);
    chance += (wield->value[1] * wield->value[2])/2;

    act("$n makes a powerful overhead strike with $gn{his} staff.", ch, 0, 0, TO_ROOM);
    char_act("You make a powerful overhead strike with your staff.", ch);

    WAIT_STATE(ch, SKILL(gsn_overhead)->beats);

    if(check_mirror_image(ch, victim))
        return;

    if (number_percent() > chance)
    {
        check_improve(ch, gsn_overhead, FALSE, 2);
        damage(ch, victim, 0, gsn_overhead, DAM_WEAPON, TRUE);
        return;
    }

    dam = dice(wield->value[1], wield->value[2]) * get_skill(ch, gsn_spear)/100;

    chance = number_percent();

    if (chance < get_skill(ch, gsn_enhanced_damage)) 
        dam += chance*dam/30;

    chance = number_percent();

    if (chance < get_skill(ch, gsn_overhead)/3)
        dam *= 4;
    else if (chance < get_skill(ch, gsn_overhead)/2)
        dam *= 3;
    else
        dam = dam*5/2;

    dam += LVL(ch)*2;

    check_improve(ch,gsn_overhead,TRUE,1);
    damage(ch, victim, dam, gsn_overhead, DAM_WEAPON, TRUE);

/* try to makes overhead-legsweep combo */
    if (ch->fighting != NULL )
    {
        chance = get_skill(ch,gsn_overhead);
        chance += get_skill(ch, gsn_leg_sweep);
        if (chance >= 196) 
            chance /= 8; /* have a 25% chance to make combo*/
        else
            chance = 0;

        if (chance > number_percent())
        {
            act("$n swing $gn{his} staff and makes a powerful combo!", ch, 0, 0, TO_ROOM);
            char_act("You swing your staff and make combo!", ch);
            do_leg_sweep(ch, "\0");
        }
    }
    return;
}

void check_ground_control(CHAR_DATA *ch, CHAR_DATA *victim, int chance, int dam) 
{
    if (victim == NULL || ch == NULL || JUST_KILLED(victim) || JUST_KILLED(ch))
        return;

    if (chance < 10) 
        return;

    if (dam == 0)
        return;

    chance += get_skill(ch, gsn_ground_control) / 2; 
    chance /= 2; 

    if (ch->style)
    {
        if (ch->style == STYLE_IRONBODY)
        {
            dam = dam * 3 / 2;
            chance -= 20;
        }
        if (ch->style == STYLE_COBRA || ch->style == STYLE_TIGER)
            chance += 20;
        if (ch->style == STYLE_DRUNKMAN)
            chance += 40;
    }

    if (get_skill(ch, gsn_ground_control) == 0)
        return; 

    chance = URANGE(5, chance, 70);

    if (ch->class == CLASS_NINJA)
    {
        chance += 20;
        dam += dice(ch->level, 4);
    }
    
    if (number_percent() > chance) 
    {
        check_improve(ch, gsn_ground_control, FALSE, 2); 
        return; 
    }

    // GM : 6% on aikido skill
    if (number_range(0, 600) < get_skill(ch, sn_lookup("aikido")))
    {
        act("With brutal skill you painfully grind $N against the ground with your weight.",
            ch, NULL, victim, TO_CHAR); 
        act("$n takes hold of $N and painfully grinds $gN{him} against the ground.",
            ch, NULL, victim, TO_NOTVICT); 
        act("$n painfully grinds you against the ground with brutal skill.",
            ch, NULL, victim, TO_VICT);
        dam += dam / 4;
        if (IS_NPC(victim))
            dam *= 2;
        check_improve(ch, sn_lookup("aikido"), TRUE, 1);
    } else
    {
        act("With brutal skill you grind $N against the ground with your weight.",
            ch, NULL, victim, TO_CHAR); 
        act("$n takes hold of $N and grinds $gN{him} against the ground.",
            ch, NULL, victim, TO_NOTVICT); 
        act("$n grinds you against the ground with brutal skill.",
            ch, NULL, victim, TO_VICT); 
        check_improve(ch, sn_lookup("aikido"), FALSE, 1);
        dam -= 10; 
    }

    check_improve(ch, gsn_ground_control, TRUE, 2); 
    damage(ch, victim, dam, gsn_ground_control, DAM_BASH, TRUE);
    return; 
}

void check_downstrike(CHAR_DATA *ch, CHAR_DATA *victim) 
{
    OBJ_DATA *wield;
    int dam;
    int chance;

    chance = get_skill(ch, gsn_downstrike);

    if (JUST_KILLED(victim))
        return;

    if (chance <= 10)   
        return;

    chance /= 3;
    chance += LVL(ch) - LVL(victim);
    chance -= get_curr_stat(victim,STAT_DEX)/3;
    chance += get_curr_stat(ch,STAT_DEX)/3;

    chance = URANGE(2, chance, 45);

    wield = get_eq_char(ch,WEAR_WIELD);
    if (wield == NULL || wield->value[0] != WEAPON_DAGGER)
        wield = get_eq_char(ch,WEAR_SECOND_WIELD);
    if (wield == NULL || wield->value[0] != WEAPON_DAGGER)
        return;

    if (number_percent() > chance) 
    {
        check_improve(ch, gsn_downstrike, FALSE, 1);
        return;
    }

    dam = dice(wield->value[1], wield->value[2]);
    dam += ch->damroll * 2;
    if (IS_NPC(victim))
        dam *= 2;
    act("You strike at $N with a downward thrust of your dagger!",
                    ch, NULL, victim, TO_CHAR);
    act("$n strikes at you with a downward thrust of $gn{his} dagger!",
                    ch, NULL, victim, TO_VICT);
    act("$n strikes at $N with a downward thrust of $gn{his} dagger!",
                    ch, NULL, victim, TO_NOTVICT);
    check_improve(victim, gsn_downstrike, TRUE, 1);
    damage(ch, victim, dam, gsn_downstrike, attack_table[wield->value[3]].damage, TRUE);
    return;
}

void do_lash(CHAR_DATA *ch,const char *argument) 
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    int chance;
    OBJ_DATA *weapon;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_lash)) == 0) 
    {
        char_act("You don't have the skill to lash people's legs.", ch);
        return;
    }

    if (IS_NPC(ch) && IS_AFFECTED(ch, AFF_CHARM))
        return;

    weapon = get_eq_char(ch, WEAR_WIELD);
    if (weapon == NULL || (weapon->value[0] != WEAPON_WHIP
    && weapon->value[0] != WEAPON_FLAIL))
    {
        chance -= 15;
        weapon = get_eq_char(ch, WEAR_SECOND_WIELD);
    }
    if (weapon == NULL) 
    {
        char_act("You aren't wielding any weapon to lash with.", ch);
        return;
    }
    if (weapon->value[0] != WEAPON_WHIP && weapon->value[0] != WEAPON_FLAIL) 
    {
        char_act("You need to be wielding a whip or flail to lash.", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        victim = ch->fighting;
        if (victim == NULL) 
        {
            char_act("But you aren't fighting anyone!", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim->position == POS_SLEEPING
    || victim->position == POS_RESTING) 
    {
        act("$N isn't on $gN{his} feet.",ch,NULL,victim,TO_CHAR);
        return;
    }

    if (victim == ch) 
    {
        char_act("You try to lash your feet and look clumsy doing it.", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;

    if (IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim) 
    {
        act("But $N is your friend!",ch,NULL,victim,TO_CHAR);
        return;
    }

    /* speed */
    if (IS_AFFECTED(ch, AFF_HASTE))
        chance += 5;
    if (IS_AFFECTED(victim, AFF_HASTE))
        chance -= 15;
    chance += get_curr_stat(ch, STAT_DEX)/2;
    chance -= get_curr_stat(victim, STAT_DEX)/2;
    if (IS_AFFECTED(victim, AFF_FLYING))
        chance -= dice(2,5);
    if (!IS_NPC(victim) && (victim->class == CLASS_WARRIOR))
            chance -= 15;
    if (!IS_NPC(victim) && (victim->class == CLASS_PALADIN))
            chance -= 10;
    if (!IS_NPC(victim) && (victim->class == CLASS_ANTI_PALADIN))
            chance -= 5;
    /* level */
    chance += (LVL(ch) - LVL(victim))*3;

    if (!IS_NPC(ch) && !IS_NPC(victim)
    && (victim->fighting == NULL || ch->fighting == NULL)) 
    {
        act_yell(victim, "Help! $i is lashing me!", ch, NULL);
    } else if (RIDDEN(victim) && !IS_NPC(victim->mount) && !IS_NPC(ch)
    && (victim->mount->fighting == NULL || ch->fighting == NULL)) 
    {
        act_yell(victim->mount, "Help, $i is attacking my mount!", ch, NULL);
    }

    if (number_percent() > (chance+40) && (ch->in_room->sector_type == SECT_FOREST)) 
    {
        act("$n lashes at $N's legs, but forest surrounding catch his weapon.",
                     ch, NULL, victim, TO_NOTVICT);
        act("$n lashes at your legs but his weapon was entangled by surroundings.",
                     ch, NULL, victim, TO_VICT);
        act("You lash at $N's legs and almost lost your whip in surroundings.",
                     ch, NULL, victim, TO_CHAR);
        check_improve(ch, gsn_lash, FALSE, 1);
        multi_hit(victim, ch, TYPE_UNDEFINED);
        WAIT_STATE(ch,SKILL(gsn_lash)->beats);
        return;
     }

    if (number_percent() > chance) 
    {
        act("$n lashes at $N's legs but misses.", ch, NULL, victim, TO_NOTVICT);
        act("$n lashes at your legs but misses.", ch, NULL, victim, TO_VICT);
        act("You lash at $N's legs but miss.", ch, NULL, victim, TO_CHAR);
        check_improve(ch,gsn_lash,FALSE,1);
        multi_hit(victim, ch, TYPE_UNDEFINED);
        WAIT_STATE(ch, SKILL(gsn_lash)->beats);
        return;
    }
    act("$n lashes $N's legs, sending $gN{him} crashing to the ground.",
        ch,0,victim,TO_NOTVICT);
    act("$n lashes your legs, sending you crashing to the ground.",
        ch,0,victim,TO_VICT);
    act("You lash $N's legs, sending $gN{him} crashing to the ground.",
        ch,0,victim,TO_CHAR);
    check_improve(ch,gsn_lash,TRUE,1);

    WAIT_STATE(ch, SKILL(gsn_lash)->beats);

    if (damage(ch,victim,dice(2,7),gsn_lash,DAM_BASH,TRUE))
    {
        victim->position = POS_RESTING;
        if (!check_recovery(victim, ch)) 
        {
            WAIT_STATE(victim, SKILL(gsn_lash)->beats/2);
            DAZE_STATE(victim, SKILL(gsn_lash)->beats/2);
        }
        return;
    } else
    {
        victim->position = POS_STANDING;
        WAIT_STATE(victim, 0);
    }
}

void do_pugil(CHAR_DATA *ch,const char *argument) 
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;
    int dam;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch,gsn_pugil)) == 0) 
    {
        char_act("You're not trained in the art of pugiling.", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        victim = ch->fighting;
        if (victim == NULL) 
        {
            char_act("But you aren't fighting anyone.", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (ch->fighting == NULL) 
    {
        char_act("You can't pugil someone like that.", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("That would be a bit stupid.", ch);
        return;
    }

    obj = get_eq_char(ch, WEAR_WIELD);
    if (obj == NULL || obj->value[0] != WEAPON_SPEAR) 
    {
        char_act("You must be wielding a staff to pugil.", ch);
        return;
    }

    chance += LVL(ch) - LVL(victim);
    chance = URANGE(5, chance, 90);

    WAIT_STATE(ch,SKILL(gsn_pugil)->beats);
    if(check_mirror_image(ch,victim))
        return;

    if (number_percent() < chance) 
    {
        act("You smash $N with a bone crushing pugil!",
                        ch, NULL, victim, TO_CHAR);
        act("$n smashes you with a bone crushing pugil!",
                        ch, NULL, victim, TO_VICT);
        act("$n pugils $N with a bone crushing pugil!",
                        ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_pugil, TRUE, 1);
        dam = dice(obj->value[1], obj->value[2]);
        dam += (get_skill(ch,gsn_enhanced_damage) * dam/100);

        if (ch->level <= 40) 
        {
            dam *= number_range(10,13);
            dam /= 10;
        } else if (ch->level <= 50) 
        {
            dam *= number_range(11,14);
            dam /= 10;
        } else if (ch->level <= 60) 
        {
            dam *= number_range(12,15);
            dam /= 10;
        } else if (ch->level <= 70) 
        {
            dam *= number_range(12,17);
            dam /= 10;
        } else if (ch->level <= 80) 
        {
            dam *= number_range(13,18);
            dam /= 10;
        } else 
        {
            dam *= number_range(14,20);
            dam /= 10;
        }

        damage(ch,victim,dam,gsn_pugil, attack_table[obj->value[3]].damage, TRUE);
    } else 
    {
        check_improve(ch,gsn_pugil,FALSE,1);
        damage(ch,victim,0,gsn_pugil,DAM_NONE,TRUE);
    }
    return;
}


