/* $Id: gquest.c,v 1.666 2004/09/20 10:49:48 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

/* Part of this code is based on WoM's (World of Merlin) code */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>

#include "merc.h"
#include "update.h"
#include "gquest.h"
#include "db/db.h"
#include "conquer.h"

extern STAT_DATA stat_record;

void save_gq (void);
void load_gq (void);

// ----------------------------------------------------------------------
// external function references for linker
// ----------------------------------------------------------------------
CHAR_DATA* questor_lookup (CHAR_DATA* ch) ;

// ----------------------------------------------------------------------
// static variables
// ----------------------------------------------------------------------
static GQ_MASTER s_gq_master ; // keep master information for GQ system
static GQ_CMD    s_gq_cmds [] =
{
    { "info",     0,              POS_DEAD,     gq_info     },
    { "join",     0,              POS_STANDING, gq_join     },
    { "list",     0,              POS_DEAD,     gq_list     },
    { "progress", 0,              POS_DEAD,     gq_progress },
    { "complete", GQCMD_RUNONLY,  POS_STANDING, gq_complete },
    { "cancel",   GQCMD_RUNONLY,  POS_DEAD,     gq_cancel   },
    { "talk",     GQCMD_RUNONLY,  POS_DEAD,     gq_talk     },
    { "start",    GQCMD_IMMONLY,  POS_DEAD,     gq_start    },
    { "status",   GQCMD_IMMONLY,  POS_DEAD,     gq_status   },
    { "stop",     GQCMD_IMMONLY,  POS_DEAD,     gq_stop     },
    { "enable",   GQCMD_IMMONLY,  POS_DEAD,     gq_enable   },
    { "disable",  GQCMD_IMMONLY,  POS_DEAD,     gq_disable  },
    { NULL }
} ;

// ----------------------------------------------------------------------
// initializer
// ----------------------------------------------------------------------
void gq_init (bool enable)
{
    s_gq_master.enable     = enable ;
    s_gq_master.quests     = NULL   ;
    s_gq_master.ticks_next = number_range (10, 60) ;
    load_gq ();
}

// ----------------------------------------------------------------------
// helper for grammatic forms
// ----------------------------------------------------------------------
char* get_gram_number (int number, char* form1, char* form2, char* form5)
{
    if (number > 10 && number < 21) return form5 ;

    switch (number % 10)
    {
    case 1: return form1 ;
    case 2:
    case 3:
    case 4: return form2 ;
    }
    return form5 ;
}

// ----------------------------------------------------------------------
// generates new GQ
//
// min   - min level
// max   - max level
// count - number of distinct mobiles to select
// ----------------------------------------------------------------------
GQ_QUESTDATA* gq_generate (int min, int max, int count)
{
    CHAR_DATA**   mobs ; // mobs choosen for the GQ
    CHAR_DATA*  victim ; // potential victim
    int  mob_count = 0 ; // number of mobiles selected for the quest
    int  min_level = 0 ; // minimal mobile level
    int  max_level = 0 ; // maximal mobile level
    int  i, factor = 0 ;
    int  pass          ;

    // act flags for mobs that must not be affected
    flag64_t act_flags = 0 ;

    // the result will be store in here
    GQ_QUESTDATA* ptr ;

    // check the data
    if (min   < 1)   min   = 1   ;
    if (max   > 91)  max   = 91  ;
    if (min   > max) max   = min ;
    if (count < 3)   count = 3   ;
    if (count > 64)  count = 64  ;

    // those mobiles must not be choosen for the GQ
    act_flags = ACT_TRAIN   | ACT_PRACTICE   | ACT_HEALER  | ACT_CHANGER |
                ACT_FORGER  | ACT_REPAIRMAN  | ACT_FORGER  | ACT_QUESTOR |
                ACT_NOQUEST | ACT_CLAN_GUARD | ACT_NOTRACK | ACT_PET ;

    // calculate ranges for mobile levels
    min_level = min - number_range (0, min / 5) * 2    ;

    if (max > 90)
    {
        max_level = 150 ;
        min_level = 90 ;
    }
    else
    if (max > 70) max_level = max + number_range (max / 10, max / 2) ; else
    if (max > 50) max_level = max + number_range (max / 10, max / 5) ; else 
    max_level = max ;

    // allocate space
    mobs = (CHAR_DATA**) calloc (count, sizeof (CHAR_DATA*)) ;

    // go over the mob list to get some mobiles for the quest
    for (pass   = 0         ; pass < 2 ; ++pass)
    for (victim = char_list ; victim   ; victim = victim->next)
    {
        if (!IS_NPC (victim)                                    ||
            victim->level < (pass == 0 ? min_level : 1)         ||
            victim->level > (pass == 0 ? max_level : max_level) ||
            victim->pIndexData->pShop                           || // shopkeepers
            victim->invis_level                                 || // wizinvis trigger mobs
            victim->pIndexData->vnum < 100                      || // limbo.are
            victim->in_room == NULL                             || // mobile in nowhere
            IS_AFFECTED (victim, AFF_CHARM)                     ||

            IS_SET (victim->pIndexData->act, act_flags)         ||
            IS_SET (victim->in_room->area->flags, AREA_NOQUEST) ||
            
            number_percent () < victim->pIndexData->count * 5   ||

            // mobile in private or solitary room that can't get out of there
            (IS_SET (victim->pIndexData->act, ACT_SENTINEL) &&
             IS_SET (victim->in_room->room_flags, ROOM_PRIVATE | ROOM_SOLITARY)))
            continue ;

        // initial probability to pick up a mob is 80%
        // unless modified by factor
        if (number_percent () > (80 - 20 * factor))
        {
            factor = factor == 0 ? 0 : factor - 1 ;
            continue ;
        }

        // check whether this mob is in the same area
        // as the one already selected
        for (i = 0 ; i < mob_count ; ++i)
        {
            if (mobs [i]->pIndexData->vnum == victim->pIndexData->vnum)
            {
                // to avoid selection of two mobiles with the same vnum
                // it some times happens however i can't tell why
                i = -1 ;
                break  ;
            }

            if (mobs [i]->in_room->area == victim->in_room->area)
            {
                // there is 1% prob. to select such a mob
                if (number_percent () > 1) i = -1 ;
                break ;
            }
        }

        // means mob was not selected
        if (i < 0) continue ;

        if (pass == 0 && mob_count >= (count / 2)) break ; else
        if (mob_count >= count) break ;

        mobs [mob_count++] = victim ;

        // factor means that for the next 2 selection the probability will
        // be less - this introduces wider samples
        factor += 2 ;
    }

    // check we have enough mobiles
    if (mob_count < count)
    {
        free (mobs) ;
        return NULL ;
    }

    // fill in GQ structure
    ptr = (GQ_QUESTDATA*) calloc (1, sizeof (GQ_QUESTDATA)) ;

    // allocate memory for internal lists
    ptr->mobs  = (int*) calloc (mob_count, sizeof (int)) ;
    ptr->rooms = (int*) calloc (mob_count, sizeof (int)) ;
    ptr->count = (int*) calloc (mob_count, sizeof (int)) ;

    for (i = 0 ; i < mob_count ; ++i)
    {
        ptr->mobs  [i] = mobs [i]->pIndexData->vnum ;
        ptr->rooms [i] = mobs [i]->in_room->vnum    ;
        ptr->count [i] = number_range (1, (1 + mobs [i]->pIndexData->count / 3)) ;

        // anti-transportation system
        mobs[i]->in_room->area->flags |= AREA_GQACTIVE ;

        // total mobiles counter
        ptr->mob_total += ptr->count [i] ;
    }

    ptr->mob_count = mob_count ;

    // levels
    ptr->level_min = min ;
    ptr->level_max = max ;

    // timing
    ptr->ticks_left  = number_range (ptr->mob_total * 3, ptr->mob_total * 6) ;
    ptr->ticks_total = ptr->ticks_left ;
    ptr->ticks_start = number_range (2, 4) ;

    // type
    if (number_percent () < 70) ptr->type = 0 ;
    else                        ptr->type = 1 ;

    free (mobs) ;
    return ptr  ;
}

// ----------------------------------------------------------------------
// create new quest player
// ----------------------------------------------------------------------
GQ_QUESTPLR* gq_newplayer (GQ_QUESTDATA* quest, CHAR_DATA* ch)
{
    GQ_QUESTPLR* ptr ;
    
    // allocate memory
    ptr = (GQ_QUESTPLR*) calloc (1, sizeof (GQ_QUESTPLR)) ;

    // allocate memory for internal lists
    ptr->killed_count = (int*) calloc (quest->mob_count, sizeof (int)) ;

    // other
    ptr->quest  = quest ;
    ptr->name   = (char*) calloc (strlen (ch->name) + 1, sizeof (char)) ;
    strnzcpy (ptr->name, strlen (ch->name) + 1, ch->name) ;

    // link to the list
    ptr->next  = quest->plr ;
    quest->plr = ptr ;

    // counter
    ++quest->plr_count ;
    ptr->complete = 0 ;

    return ptr ;
}

// ----------------------------------------------------------------------
// unlink and free GQ structure
// ----------------------------------------------------------------------
void gq_freequest (GQ_QUESTDATA* data)
{
    GQ_QUESTDATA* quest ;
    GQ_QUESTDATA* prev  ;
    GQ_QUESTPLR*  plr   ;
    GQ_QUESTPLR*  next  ;

    ROOM_INDEX_DATA* room ;
    int i ;

    // find quest in the list and unlink it
    prev = NULL ;
    for (quest = s_gq_master.quests ; quest ; prev = quest, quest = quest->next)
    {
        if (quest == data)
        {
            if (quest == s_gq_master.quests)
                 s_gq_master.quests = quest->next ;
            else prev->next         = quest->next ;
            break ;
        }
    }

    // free players
    for (plr = data->plr ; plr ; plr = next)
    {
        next = plr->next ;
        gq_freeplr (plr) ;
    }

    // enable transporation
    for (i = 0 ; i < data->mob_count ; ++i)
    {
        if ((room = get_room_index (data->rooms[i])) != NULL)
            REMOVE_BIT (room->area->flags, AREA_GQACTIVE) ;
    }

    free (data->mobs)  ;
    free (data->rooms) ;
    free (data->count) ;
    free (data) ;
}

// ----------------------------------------------------------------------
// resets and free player structure
// ----------------------------------------------------------------------
void gq_freeplr (GQ_QUESTPLR* data)
{
    free (data->name)         ;
    free (data->killed_count) ;
    free (data) ;
}

// ----------------------------------------------------------------------
// join global quest
// ----------------------------------------------------------------------
void gq_join (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest ;
    GQ_QUESTPLR*  plr   ;

    if (IS_SET(ch->comm, COMM_NOQUEST))
    {
        char_printf(ch, "     .\n");
        return;
    }

    if ((plr = gq_findplr (ch)))
    {
        if (plr->killed_total < 0)
             char_printf (ch, "      .\n") ;
        else char_printf (ch, "     .\n") ;
        return ;
    }

    // go over all the currently active GQ
    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
        if (ch->level >= quest->level_min && ch->level <= quest->level_max) break ;

    if (!quest)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    // create new player
    gq_newplayer (quest, ch) ;

    char_printf (ch,    "        .\n") ;
    gq_showinfo (quest, "   {W%s{x.", capitalize (ch->name)) ;

    // update char statistics (see merc.h for details)
    ++ch->pcdata->StatAllGlobalQuests ;
    save_gq ();
}

// ----------------------------------------------------------------------
// cancel participation in global quest
// ----------------------------------------------------------------------
void gq_cancel (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTPLR* plr ;

    if ((plr = gq_findplr (ch)) == NULL || plr->killed_total < 0)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    char_printf (ch, "   .\n") ;

    // indicator we have cancelled the quest
    plr->killed_total = -1 ;

    // counter
    --plr->quest->plr_count ;
    save_gq ();
}

// ----------------------------------------------------------------------
// list currently active GQ
// ----------------------------------------------------------------------
void gq_list (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest   ;
    BUFFER*       output  ;
    int           num = 0 ;

    if (s_gq_master.quests == NULL && str_cmp (argument, "brief") != 0)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    output = buf_new (-1) ;
    buf_printf (output, "  :\n") ;

    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
    {
        buf_printf (output, "{G%2d{x.    {W%d{x-{W%d{x",
                    ++num, quest->level_min, quest->level_max) ;

        if (quest->ticks_start > 0)
             buf_printf (output, " (  {W%d{x %s)\n",
                         quest->ticks_start,
                         get_gram_number (quest->ticks_start, "", "", "")) ;
        else buf_add (output, "\n") ;
    }

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}

// ----------------------------------------------------------------------
// request global quest information
// ----------------------------------------------------------------------
void gq_info (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest  ;
    GQ_QUESTPLR*  plr    ;
    BUFFER*       output ;
    int           num, i ;

    static char arg [MAX_INPUT_LENGTH] ;

    argument = one_argument (argument, arg, sizeof (arg)) ;
    num      = atoi (arg) ;

    if (arg[0] == '\0' || num <= 0)
    {
        // try the case when player already has a GQ
        if ((quest = gq_findquest (ch)) == NULL)
        {
            char_printf (ch, "  .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_gq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_printf (ch, "   .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }

    output = buf_new (-1) ;
    buf_printf (output, "{C * * *    {MGLOBAL QUEST    {C* * *{x\n") ;

    // pre-start information
    if (quest->ticks_start > 0)
    {
        buf_printf (output, " : {W%d{x - {W%d{x,   {W%d{x   {W%d{x %s.\n",
                    quest->level_min, quest->level_max, quest->mob_total, quest->ticks_total,
                    get_gram_number (quest->ticks_total, "", "", "")) ;

        buf_printf (output, "         {W%d{x %s.\n",
                    quest->plr_count, get_gram_number (quest->plr_count, "", "", "")) ;

        buf_printf (output, "   {W%d{x %s.\n",
                    quest->ticks_start, get_gram_number (quest->ticks_start, "", "", "")) ;
        
        char_act(buf_string (output), ch) ;
        buf_free (output) ;
        return ;
    }

    // normal information
    buf_printf (output, " : {W%d{w - {W%d{x,   {W%d{x .\n",
                quest->level_min, quest->level_max, quest->mob_total) ;

    buf_printf (output, " {G%d{x %s,  {W%d{x %s.\n",
                 quest->ticks_left, get_gram_number (quest->ticks_left, "", "", ""),
                 quest->plr_count,  get_gram_number (quest->plr_count, "", "", "")) ;

    if ((plr = gq_findplr (ch)) == NULL || plr->killed_total < 0 || quest != gq_findquest (ch))
    {
        char_act(buf_string (output), ch) ;
        buf_free (output) ;
        return ;
    }

    buf_printf (output, "   {R%d{x %s  {W%d{x.\n",
                plr->killed_total, get_gram_number (plr->killed_total, "", "", ""),
                quest->mob_total) ;

    buf_printf (output, "\n:\n") ;

    for (i = 0 ; i < quest->mob_count ; ++i)
    {
        if (quest->count [i] - plr->killed_count [i] <= 0) continue ;

        buf_printf (output, " %d {W%s (%s){x\n",
                    quest->count [i] - plr->killed_count [i],
                    mlstr_cval (get_mob_index (quest->mobs [i])->short_descr, ch),
                    (quest->type == 0 ?
                     mlstr_mval (get_room_index (quest->rooms [i])->name) :
                                 get_room_index (quest->rooms [i])->area->name)) ;
    }

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}

// ----------------------------------------------------------------------
// comparator for qsort used in progress
// ----------------------------------------------------------------------
__inline int compare_plr (const void* p, const void* q)
{
    return ((* (GQ_QUESTPLR**) q)->killed_total - (* (GQ_QUESTPLR**) p)->killed_total) ;
}

// ----------------------------------------------------------------------
// request progress information
// ----------------------------------------------------------------------
void gq_progress (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest  ;
    GQ_QUESTPLR*  plr    ;
    BUFFER*       output ;
    int           num, i ;
    GQ_QUESTPLR** sorted ;

    static char arg [MAX_INPUT_LENGTH] ;

    argument = one_argument (argument, arg, sizeof (arg)) ;
    num      = atoi (arg) ;

    if (arg[0] == '\0' || num <= 0)
    {
        // try the case when player already has a GQ
        if ((quest = gq_findquest (ch)) == NULL)
        {
            char_printf (ch, "  .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_gq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_printf (ch, "   .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }

    if (quest->plr_count == 0)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    output = buf_new (-1) ;
    buf_printf (output, "    :\n") ;
    
    // allocate data for sorted list
    sorted = (GQ_QUESTPLR**) calloc (quest->plr_count, sizeof (GQ_QUESTPLR*)) ;

    // copy data
    num = 0 ;
    for (plr = quest->plr ; plr ; plr = plr->next)
    {
        if (plr->killed_total < 0) continue ;
        sorted [num] = plr ;
        ++num ;
    }

    // sort data
    qsort (sorted, num, sizeof (GQ_QUESTPLR*), compare_plr) ;

    for (i = 0 ; i < num ; ++i)
        buf_printf (output, "{w%2d. {W%s{x\n", i + 1, capitalize (sorted [i]->name)) ;

    free (sorted) ;

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}


// ----------------------------------------------------------------------
// count of gquest
// ----------------------------------------------------------------------
int gq_count()
{
    GQ_QUESTDATA* quest   ;
    int           num = 0 ;

    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
        ++num ;

    return num;
}

// ----------------------------------------------------------------------
// report about quest completion
// ----------------------------------------------------------------------
void gq_complete (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest ;
    GQ_QUESTPLR*  plr   ;

    // just in case
    if (IS_NPC (ch) || !ch->in_room) return ;

    if ((plr = gq_findplr (ch)) == NULL || plr->killed_total < 0)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    quest = plr->quest ;

    if (plr->killed_total < quest->mob_total)
    {
        char_printf (ch, "    !\n") ;
        return ;
    }

    // 1 means player failed to report the quest
    if (plr->complete == 1)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    // must be near questor to report
    if (questor_lookup(ch) == NULL)
    {
        char_printf (ch, " ,      .\n") ;
        return ;
    }

    gq_showinfo (NULL, "   {W%d{x-{W%d{x   %s!\n",
                 quest->level_min, quest->level_max, ch->name) ;

    char_printf (ch,   "{C      {x.\n") ;

    // reward
    gq_givebonus (ch, TRUE) ;

    // free gq structure
    gq_freequest (quest) ;
    
    // update char statistics (see merc.h for details)
    ++ch->pcdata->StatWinGlobalQuests ;

    // update global statistics (see merc.h for details)
    ++stat_record.gq_completed;
    save_gq ();
}

// ----------------------------------------------------------------------
// GQ channel
// ----------------------------------------------------------------------
void gq_talk (CHAR_DATA* ch, const char* argument )
{
    GQ_QUESTDATA* quest = 0 ;

    if (IS_SET (ch->comm, COMM_NOCHANNELS))
    {
        act ("     .", 
                         ch, NULL, NULL, TO_CHAR) ;
        return ;
    }

    // note: this channel is for everyone currently doing GQ independent
    // on which quest they are doing

    if (!IS_IMMORTAL (ch) && (quest = gq_findquest (ch)) == NULL) return ;

    if (IS_IMMORTAL (ch))
         gq_showinfo ((GQ_QUESTDATA*) -1, "{W[%s] {Y%s{x", ch->name, argument) ;
    else gq_showinfo (quest,              "{W[%s] {Y%s{x", capitalize (ch->name), argument) ;
}

// ----------------------------------------------------------------------
// obtain status of the GQ system (for immortals)
// ----------------------------------------------------------------------
void gq_status (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest  ;
    GQ_QUESTPLR*  plr    ;
    BUFFER*       output ;
    int           num, i ;

    static char arg [MAX_INPUT_LENGTH] ;

    if (s_gq_master.quests == NULL)
    {
        char_printf (ch, "    .\n") ;
        if (s_gq_master.ticks_next == 0) return ;

        char_printf (ch, "\n  -  %d .\n",
                     s_gq_master.ticks_next) ;
        return ;
    }

    argument = one_argument (argument, arg, sizeof (arg)) ;
    num      = atoi (arg) ;

    if (arg[0] == '\0' || num <= 0)
    {
        char_printf (ch, "  .\n") ;
        gq_list (ch, str_empty) ;

        if (!IS_IMMORTAL (ch)) return ;
        char_printf (ch, "\n  -  %d .\n",
                     s_gq_master.ticks_next) ;
        return ;
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_gq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_printf (ch, "   .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }

    if (!IS_IMMORTAL (ch) && s_gq_master.enable == 0)
    {
        char_printf (ch, "     .\n") ;
        return ;
    }

    output = buf_new (-1) ;

    if (quest->ticks_start > 0)
         buf_printf (output, "  ,    {W%d{x .\n", quest->ticks_start) ;
    else buf_printf (output, "  ,    {W%d{x .\n",   quest->ticks_left)  ;

    buf_printf (output, " : {W%d{x-{W%d{x\n",
                quest->level_min, quest->level_max) ;

    buf_printf (output, "  ( {R%d{x):\n", quest->mob_total) ;

    for (i = 0 ; i < quest->mob_count ; ++i)
    {
        buf_printf (output, " %d {W%s (vnum %d) in %s|%s (vnum %d){x\n",
                    quest->count [i],
                    mlstr_mval (get_mob_index (quest->mobs [i])->short_descr),
                    quest->mobs [i],
                    get_room_index (quest->rooms [i])->area->name,
                    mlstr_mval (get_room_index (quest->rooms [i])->name),
                    quest->rooms [i]) ;
    }

    buf_printf (output, "\n ( {G%d{x):\n", quest->plr_count) ;

    for (plr = quest->plr ; plr ; plr = plr->next)
    {
        if (plr->killed_total < 0) continue ;
        buf_printf (output, "{W%-12s    {G%2d{w\n", capitalize (plr->name), plr->killed_total) ;
    }

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}

// ----------------------------------------------------------------------
// start the quest (for immortals and MUD)
// NOTE: in this function ch MAY BE NULL!
// ----------------------------------------------------------------------
void gq_start (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA*    quest ;
    DESCRIPTOR_DATA* d     ;
    CHAR_DATA*       wch   ;

    int min = 1, max = LEVEL_HERO + 9, num, attempt, count = 0 ;

    static char arg [MAX_INPUT_LENGTH] ;

    // do not start GQ before reboot
    if (reboot_counter <= 30 && reboot_counter >= 0) return ;

    // disabled
    if (!s_gq_master.enable)
    {
        if (ch) char_printf (ch, "     .\n") ;
        return ;
    }

    if (gq_count() > 4)
    {
        if (ch) char_printf (ch, "    .\n"); // TODO:    
        return ;
    }

    attempt = 0 ;
    if (ch && str_cmp (argument, "auto")) // manual start
    {
        argument = one_argument (argument, arg, sizeof (arg)) ;
        min      = atoi (arg) ;

        argument = one_argument (argument, arg, sizeof (arg)) ;
        max      = atoi (arg) ;

        argument = one_argument (argument, arg, sizeof (arg)) ;
        count    = atoi (arg) ;

        if (count <= 0)
        {
            char_printf (ch, ": gquest start {{<min> <max> <count>|auto}\n") ;
            return ;
        }

        // check whether there is already a quest that covers this level range
        for (quest = s_gq_master.quests ; quest ; quest = quest->next)
        {
            if (quest->level_min <= min && min <= quest->level_max) break ;
            if (quest->level_min <= max && max <= quest->level_max) break ;
            if (min <= quest->level_min && quest->level_max <= max) break ;
        }

        if (quest)
        {
            char_printf (ch, "    (  )  .\n") ;
            return ;
        }
    }
    else
    for (attempt = 0 ; attempt < 48 ; ++attempt)
    {
        int level = number_range (1, LEVEL_HERO + 9) ;

        // Welesh : increased chance to generate gq for newbies
        if ((level > 60) && (number_percent () > 70)) // 30% to skip high-level gq
            continue;

        if ((level > 30) && (number_percent () > 80)) // 20% to skip middle-level gq
            continue;

        if (level < 11)
        {
            min = UMAX (1,          level - level / 2) ;
            max = UMIN (LEVEL_HERO, level + level / 2) ;
        }
        if (level < 21)
        {
            min = UMAX (1,          level - level / 5) ;
            max = UMIN (LEVEL_HERO, level + level / 5) ;
        }
        else
        if (level < 41)
        {
            min = UMAX (1,          level - level / 8) ;
            max = UMIN (LEVEL_HERO, level + level / 8) ;
        }
        else
        {
            min = UMAX (1,          level - level / 10) ;
            max = UMIN (LEVEL_HERO, level + level / 10) ;
        }

        // check whether there is already a quest that covers this level range
        for (quest = s_gq_master.quests ; quest ; quest = quest->next)
        {
            if (quest->level_min <= min && min <= quest->level_max) break ;
            if (quest->level_min <= max && max <= quest->level_max) break ;
            if (min <= quest->level_min && quest->level_max <= max) break ;
        }

        if (quest) continue ;

        count = number_range (3, 21) ;
        num   = 0 ;

        // calculate number of players that we have within this range
        for (d = descriptor_list ; d ; d = d->next)
        {
            wch = d->original ? d->original : d->character ;
            if (wch && d->connected == CON_PLAYING && wch->level >= min && wch->level <= max)
                ++num ;
        }

        // run quest when there is at least 2 persons
        if (num >= 2) break ;
    }

    // no success at all
    if (attempt >= 16)
    {
        if (ch) char_printf (ch, "   --  .\n") ;
        return ;
    }

    // generate quest
    quest = gq_generate (min, max, count) ;

    if (quest == NULL)
    {
        if (ch) char_printf (ch, "    --  .\n") ;
        return ;
    }

    // link quest to the list
    quest->next        = s_gq_master.quests ;
    s_gq_master.quests = quest ;

    // update global statistics (see merc.h for details)
    ++stat_record.gq_started;
    save_gq ();

    gq_showinfo (NULL, "\n{GHH!{w    {W%d{x-{W%d{x .\n"
                 ",   ,   {Chelp global quest{x.\n",
                 quest->level_min, quest->level_max) ;
}

// ----------------------------------------------------------------------
// stop the running quest (for immortals)
// ----------------------------------------------------------------------
void gq_stop (CHAR_DATA* ch, const char* argument)
{
    GQ_QUESTDATA* quest ;
    int           num   ;

    static char arg [MAX_INPUT_LENGTH] ;

    argument = one_argument (argument, arg, sizeof (arg)) ;
    num      = atoi (arg) ;

    if (arg[0] == '\0' || num <= 0)
    {
        char_printf (ch, "  .\n") ;
        gq_list (ch, str_empty) ;
        return ;
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_gq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_printf (ch, "   .\n") ;
            gq_list (ch, str_empty) ;
            return ;
        }
    }

    gq_showinfo  (quest, "  !\n") ;
    gq_freequest (quest) ;
    save_gq ();
}

// ----------------------------------------------------------------------
// enable/disable GQ system (for immortals)
// ----------------------------------------------------------------------
void gq_enable (CHAR_DATA* ch, const char* argument)
{
    if (s_gq_master.enable)
    {
        char_printf (ch, "   .\n") ;
        return ;
    }

    s_gq_master.enable = 1 ;
    char_printf (ch, "\n{R    !{x\n") ;
    log_printf  ("%s    .", ch->name) ;
}

void gq_disable (CHAR_DATA* ch, const char* argument)
{
    if (!s_gq_master.enable)
    {
        char_printf (ch, "   .\n") ;
        return ;
    }

    s_gq_master.enable = 0 ;
    char_printf (ch, "\n{R    !{x\n") ;
    log_printf  ("%s    .", ch->name) ;
}

// ----------------------------------------------------------------------
// shows GQ information
// ----------------------------------------------------------------------
void gq_showinfo (GQ_QUESTDATA* quest, const char* fmt, ...)
{
    CHAR_DATA*       ch  ;
    DESCRIPTOR_DATA* d   ;
    GQ_QUESTPLR*     plr ;

    va_list args;
    static char buf[MAX_STRING_LENGTH];
    static char buf2[MAX_STRING_LENGTH];
    
    // form the string
    va_start  (args, fmt) ;
    vsnprintf (buf2, sizeof(buf2), fmt, args) ;
    va_end    (args) ;
    snprintf  (buf, sizeof(buf), "{MGLOBAL QUEST: {w%s\n", buf2) ;

    // message from immortals to all players currently involved in gq
    if (quest == (GQ_QUESTDATA*) -1)
    {
        for (d = descriptor_list ; d ; d = d->next)
            if (d->connected == CON_PLAYING && !IS_SET (d->character->comm, COMM_NOGQUEST) &&
                gq_findplr (d->character) != NULL)
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf);
                else
                    char_printf (d->character, "%s", buf) ;
            }
    }
    else
    // check who must get the notification
    if (quest)
    {
        for (plr = quest->plr ; plr ; plr = plr->next)
        {
            ch = gq_findchar  (plr->name) ;
            if (ch && !IS_SET (ch->comm, COMM_NOGQUEST)) 
            {
                if (IS_SET(ch->comm, COMM_QUIET_EDITOR)
                    &&  ch->desc->pString)
                        buf_add(ch->pcdata->buffer, buf);
                else
                    char_printf (ch, "%s", buf) ;
            }
        }
    }
    // message for everyone
    else
    {
        for (d = descriptor_list ; d ; d = d->next)
            if (d->connected == CON_PLAYING && !IS_SET (d->character->comm, COMM_NOGQUEST)) 
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf);
                else
                    char_printf (d->character, "%s", buf) ;
            }
        return ;
    }

    // immortals
    for (d = descriptor_list ; d ; d = d->next)
    {
        if (d->connected == CON_PLAYING && IS_IMMORTAL (d->character) &&
            !IS_SET (d->character->comm, COMM_NOGQUEST))
        {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf);
                else
                    char_printf (d->character, "%s", buf) ;
        }
    }
}

// ----------------------------------------------------------------------
// finders
// ----------------------------------------------------------------------
GQ_QUESTPLR* gq_findplr (CHAR_DATA* ch)
{
    GQ_QUESTDATA* quest ;
    GQ_QUESTPLR*  plr   ;

    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
    {
        // use case-insensistive internal function
        for (plr = quest->plr ; plr ; plr = plr->next)
            if (str_cmp (plr->name, ch->name) == 0) return plr ;
    }

    return NULL ;
}

GQ_QUESTDATA* gq_findquest (CHAR_DATA* ch)
{
    GQ_QUESTPLR* plr = gq_findplr (ch) ;

    if (plr) return plr->quest ;
    else     return NULL ;
}

CHAR_DATA* gq_findchar (const char* name)
{
    CHAR_DATA* victim ;

    for (victim = char_list ; victim ; victim = victim->next)
        if (IS_NPC  (victim)) continue ; else
        if (str_cmp (name, victim->name) == 0) return victim ;

    return NULL ;
}


// ----------------------------------------------------------------------
// general interface to be used within MUD
// ----------------------------------------------------------------------

// ----------------------------------------------------------------------
// check whether mobile is used in GQ
// ----------------------------------------------------------------------
bool is_gquest_mob (CHAR_DATA* ch)
{
    GQ_QUESTDATA* quest ;

    if (!IS_NPC (ch)) return FALSE ;
    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
    {
        int i ;
        for (i = 0 ; i < quest->mob_count ; ++i)
            if (quest->mobs [i] == ch->pIndexData->vnum) return TRUE ;
    }

    return FALSE ;
}

// ----------------------------------------------------------------------
// function called from the interpreter
// ----------------------------------------------------------------------
void do_gquest (CHAR_DATA* ch, const char* argument)
{
    GQ_CMD* cmd ;

    // retrieve command
    static char command [MAX_INPUT_LENGTH] ;
    argument = one_argument (argument, command, sizeof (command)) ;

    for (cmd = s_gq_cmds ; cmd->name ; ++cmd)
    {
        if (str_prefix (command, cmd->name)) continue ;
//        if (IS_SET (cmd->flags, GQCMD_IMMONLY) && !IS_IMMORTAL (ch)) continue ;
        if (IS_SET (cmd->flags, GQCMD_IMMONLY) && !char_security (ch,"GLOBAL_QUEST_ACCESS")) continue ;

        if (IS_SET (cmd->flags, GQCMD_RUNONLY) && !IS_IMMORTAL (ch) &&
            !gq_findplr (ch)) continue ;

        (*cmd->fun) (ch, argument) ;
        return ;
    }

    char_printf (ch, "  .\n : ") ;
    for (cmd = s_gq_cmds ; cmd->name ; ++cmd)
        if (!IS_SET (cmd->flags, GQCMD_IMMONLY) || IS_IMMORTAL (ch))
            char_printf (ch, "%s ", cmd->name) ;

    char_printf (ch, "\n") ;
}

// ----------------------------------------------------------------------
// handle mob death
// ----------------------------------------------------------------------
void gq_mobdeath (CHAR_DATA* ch, CHAR_DATA* victim)
{
    GQ_QUESTDATA* quest;
    GQ_QUESTPLR*  plr;
    int i;

    if (!IS_NPC (victim) || !victim->pIndexData) 
        return;

    if (IS_NPC (ch))
    {
        if (IS_AFFECTED (ch, AFF_CHARM) && ch->master != NULL) 
            ch = ch->master;
        else 
            return;
    }

    // find whether player participates in quest
    if ((plr = gq_findplr (ch)) == NULL || plr->killed_total < 0) 
        return;
    quest = plr->quest;

    for (i = 0 ; i < quest->mob_count ; ++i)
    {
        if (quest->mobs [i] != victim->pIndexData->vnum) 
            continue;

        // enough mobiles killed
        if (plr->killed_count [i] >= quest->count [i]) 
            return;

        ++plr->killed_count [i];
        ++plr->killed_total;

        // reward
        char_printf  (ch, "\n{C   !{x\n");
        gq_givebonus (ch, FALSE);

        if (plr->killed_total == quest->mob_total)
        {
            char_printf (ch, "\n{C   !{x\n");
            char_printf (ch, "   10 ,     !{x\n");

            plr->complete = 11;
        }

        return;
    }
}

// ----------------------------------------------------------------------
// update GQ system (called from handler.c)
// ----------------------------------------------------------------------
void gq_update ()
{
    GQ_QUESTDATA* quest ;
    GQ_QUESTDATA* next  ;
    GQ_QUESTPLR*  plr   ;
    CHAR_DATA*    ch    ;
    bool          need_save = FALSE;

    // check whether we have to start new quest
    if (s_gq_master.ticks_next > 0)
    {
        --s_gq_master.ticks_next ;
        if (s_gq_master.ticks_next == 0)
        {
            gq_start (NULL, "auto") ;
            s_gq_master.ticks_next = number_range (10, 60) ;
        }
    }

    // preparing and quest running
    for (quest = s_gq_master.quests ; quest ; quest = next)
    {
        next = quest->next ;

        if (quest->ticks_start > 0) --quest->ticks_start ;
        if (quest->ticks_left  > 0) --quest->ticks_left  ;

        // for single player
        if (quest->ticks_left  > 0 && quest->plr_count == 1) --quest->ticks_left ;
 
        if (quest->ticks_start == 0)
        {
            if (quest->plr_count < 1)
            {
                gq_showinfo  (NULL, "\n   {W%d{x-{W%d{x     -  .\n",
                              quest->level_min, quest->level_max) ;
                gq_freequest (quest) ;
                need_save = TRUE;
                continue ;
            }

            gq_showinfo (quest, "  {W%d{x-{W%d{x  .    - {W%d{x .\n",
                         quest->level_min, quest->level_max, quest->ticks_total) ;

            if (quest->plr_count == 1)
                gq_showinfo (quest, "     ,     2  .\n") ;

            quest->ticks_start = -1 ;
            continue ;
        }

        if (quest->ticks_left == 0)
        {
            gq_showinfo  (quest, ",     -    .\n") ;
            gq_freequest (quest) ;
            need_save = TRUE;
            continue ;
        }
        else
        if (quest->ticks_left < 6)
            gq_showinfo (quest, "    {Y%d{w !\n", quest->ticks_left) ;

        for (plr = quest->plr ; plr ; plr = plr->next)
        {
            if (plr->complete <= 1) continue ;

            // decrease timer
            --plr->complete ;

            // check player is in the game
            ch = gq_findchar (plr->name) ;
            if (ch == NULL) continue ;

            if (plr->complete == 1)
            {
                char_printf (ch, "\n      .    .{x\n") ;
                continue ;
            }

            char_printf (ch, "\n   {w%d{w ,     !{x\n",
                         plr->complete - 1) ;
        }
    }

    if (need_save)
        save_gq ();
}

// ----------------------------------------------------------------------
// quest bonuses
// ----------------------------------------------------------------------
int gq_checkbonus (CHAR_DATA* ch, int chance, int min, int max,
                   char* info, char* form1, char* form2, char* form5)
{
    int reminder, bonus = min ;
    char* select ;

    if (max > 0) bonus = number_range (min, max) ;
    if (chance < number_percent ()) return 0 ;

    reminder = bonus % 10 ;

    if (reminder == 1)                select = form1 ; else
    if (reminder > 1 && reminder < 5) select = form2 ; else
                                      select = form5 ;

    // we modify bonus if mob was KILLED in conquered area,
    // not BORN in it :) so there may be tactical decisions
    bonus = areacontrol_modify_gain (ch, bonus, GLOBAL_QUEST_REWARD);
    char_printf (ch, info, bonus, select) ;
    return bonus ; 
}

void gq_givebonus (CHAR_DATA* ch, bool end)
{
    int chance_train = end ? 10 : 2  ;
    int chance_pract = end ? 25 : 5  ;
    int chance_gold  = end ? 90 : 80 ;
    int chance_exp   = end ? 100: 90 ;
    int chance_bp    = end ? 70 : 2  ;


    int bonus_train  = end ? 2 : 1 ;
    int bonus_pract  = end ? number_range (2,    4)    : 1 ;
    int bonus_qp     = end ? number_range (20,   250)  : number_range (1,   20)  ;
    int bonus_exp    = end ? number_range (100,  300)  : number_range (50,  150) ;
    int bonus_gold   = end ? number_range (10,   30)   : number_range (1,   15)  ;
    int bonus_bp     = end ? 2 : 1 ;


    int j, i = 0  ;
    skill_t*   sk ;
    pcskill_t* ps ;

    int add_val;


    // modifications based on level
    if (ch->level < 30)
    {
        bonus_qp   *= 2 ;
        bonus_gold *= 2 ;
    }
    else if (ch->level >= 50 && ch->level < 70) bonus_exp  = bonus_exp * 3 / 2 ;
    else if (ch->level >= 70)                   bonus_exp *= 2 ;

    if (ch->pcdata)
    {
        if (IS_SET(ch->pcdata->wishes, WISH_EXPRATE2))
        {
            bonus_exp = bonus_exp * 1.4;
        }
        else
        {
            if (IS_SET(ch->pcdata->wishes, WISH_EXPRATE1))
            {
                bonus_exp = bonus_exp * 1.2;
            }
        }
    }

    for (j = 0 ; j < ch->pcdata->learned.nused ; j++)
    {
        ps = VARR_GET(&ch->pcdata->learned, j) ;

        if (ps->percent == 0 || (sk = skill_lookup (ps->sn)) == NULL ||
            skill_level (ch, ps->sn) > ch->level) continue ;

        if (end)
        {
            if (i == 0 && ps->percent > 1 && ps->percent < 90 &&
                number_range (0, 3) == 0)
            {
                i = 1 ;
                ps->percent += 2 ;
                char_printf (ch, "  2%%  {W%s{z, {W%s {z {W%d%%{z.\n",
                                 sk->name, sk->name, ps->percent) ;
                continue ;
            }
        }
        else
        {
            if (i == 0 && ps->percent > 1 && ps->percent < 80 &&
                number_range (0, 99) == 77)
            {
                i = 1 ;
                ps->percent += 1 ;
                char_printf (ch, "  1%%  {W%s{z, {W%s {z {W%d%%{z.\n",
                             sk->name, sk->name, ps->percent) ;
                continue ;
            }
        }
    }

    //ch->pcdata->questpoints += gq_checkbonus (ch, 101, bonus_qp, 0,
    add_val = gq_checkbonus (ch, 101, bonus_qp, 0,
        "{G  {W%d{w quest points!\n", str_empty, str_empty, str_empty) ;
    ch->pcdata->questpoints += add_val;
    ch->pcdata->qp_earned += add_val;

    ch->train += gq_checkbonus (ch, chance_train, bonus_train, 0,
        "{G  {W%d{w %s (train)!\n", "", "", "") ;

    ch->practice += gq_checkbonus (ch, chance_pract, bonus_pract, 0,
        "{G  {W%d{w %s (practice)!\n", "", "", str_empty) ;

    if (gq_checkbonus (ch, chance_exp, bonus_exp, 0,
        "{G  {W%d{w %s  (exp)!\n", "", "", ""))
        gain_exp (ch, bonus_exp) ;

    add_val = gq_checkbonus (ch, chance_gold, bonus_gold, 0,
        "{G  {Y%d{w %s!\n",
        " ", " ", " ") ;
    ch->gold += add_val;
    stat_record.gold_gq_reward += add_val;

    add_val = gq_checkbonus (ch, chance_bp, bonus_bp, 0,
        "{G  {W%d{w bonus point!\n\r", str_empty, str_empty, str_empty) ;
    ch->pcdata->bonuspoints += add_val;
    ch->pcdata->bp_earned += add_val;

    char_printf (ch, "\n") ;
}

// ----------------------------------------------------------------------
// gq save/load stuff
// ----------------------------------------------------------------------
#define GQ_FILE_NAME              "gquest.save"

static void fread_gq (FILE *fp);
static void fread_gq_plr (FILE *fp);
static void fwrite_gq (FILE * fp, GQ_QUESTDATA* quest);
static void fwrite_gq_plr (FILE * fp, GQ_QUESTDATA* quest, GQ_QUESTPLR* plr);

void save_gq (void)
{
    FILE        * fp;
    GQ_QUESTDATA* quest;
    GQ_QUESTPLR*  plr;

    if (!(fp = dfopen(TMP_PATH, GQ_FILE_NAME, "w")))
    {
        log_printf("ERROR!!! save_gq: can't open output file %s/%s", TMP_PATH, GQ_FILE_NAME);
        return;
    }

    for (quest = s_gq_master.quests ; quest ; quest = quest->next)
    {
        fwrite_gq (fp, quest);
        for (plr = quest->plr ; plr ; plr = plr->next)
            fwrite_gq_plr (fp, quest, plr);
    }

    fprintf(fp, "#END\n");
    fclose(fp);
}

static void fwrite_gq (FILE * fp, GQ_QUESTDATA* quest)
{
    int i; 

    fprintf (fp, "#GQ\n");

    fprintf (fp, "LevelMax %d\n",   quest->level_max);
    fprintf (fp, "LevelMin %d\n",   quest->level_min);
    fprintf (fp, "TicksLeft %d\n",  quest->ticks_left);
    fprintf (fp, "TicksTotal %d\n", quest->ticks_total);
    fprintf (fp, "TicksStart %d\n", quest->ticks_start);
    fprintf (fp, "Type %d\n",       quest->type);
    fprintf (fp, "MobCount %d\n",   quest->mob_count);
    fprintf (fp, "MobTotal %d\n",   quest->mob_total);
    fprintf (fp, "PlrCount %d\n",   quest->plr_count);

    // mobiles
    fprintf (fp, "Mobiles");
    for (i = 0 ; i < quest->mob_count ; ++i)
        fprintf (fp, " %d", quest->mobs [i]);
    fprintf (fp, "\n");

    fprintf (fp, "Rooms");
    for (i = 0 ; i < quest->mob_count ; ++i)
        fprintf (fp, " %d", quest->rooms [i]);
    fprintf (fp, "\n");

    fprintf (fp, "Counts");
    for (i = 0 ; i < quest->mob_count ; ++i)
        fprintf (fp, " %d", quest->count [i]);
    fprintf (fp, "\n");

    fprintf (fp, "End\n\n");
}

static void fwrite_gq_plr (FILE * fp, GQ_QUESTDATA* quest, GQ_QUESTPLR* plr)
{
    int i; 

    fprintf (fp, "#PLR\n");

    fprintf (fp, "Name %s\n", plr->name);
    fprintf (fp, "Complete %d\n", plr->complete);
    fprintf (fp, "KillTotal %d\n", plr->killed_total);

    // mobiles
    fprintf (fp, "Mobiles");
    for (i = 0 ; i < quest->mob_count ; ++i)
        fprintf (fp, " %d", plr->killed_count [i]);
    fprintf (fp, "\n");

    fprintf (fp, "End\n\n");
}

void load_gq (void)
{
    FILE *fp;

    if ((fp = dfopen(TMP_PATH, GQ_FILE_NAME, "r")))
    {
        for (;;)
        {
            char   letter ;
            char * word   ;

            letter = fread_letter (fp) ;
            if (letter == '*')
            {
                fread_to_eol (fp) ;
                continue ;
            }
            if (letter != '#')
            {
                log_printf ("Load_gq: # not found in %s.", GQ_FILE_NAME) ;
                break ;
            }
            word = fread_word (fp) ;
            if (!str_cmp (word, "GQ")) fread_gq (fp);
            else if (!str_cmp (word, "PLR")) fread_gq_plr (fp) ;
            else if (!str_cmp (word, "END")) break ;
            else
            {
                bug ("Load_gq: bad section.", 0) ;
                break ;
            }
        }
        fclose (fp) ;
    }
}

static void fread_gq (FILE *fp)
{
    char        * word;
    bool          fMatch;
    GQ_QUESTDATA* ptr;
    int           i;

    ptr = (GQ_QUESTDATA*) calloc (1, sizeof (GQ_QUESTDATA)) ;
    ptr->mob_count = 0;

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'C':
            if (!str_cmp(word, "Counts"))
            {
                fMatch = TRUE;
                for (i = 0 ; i < ptr->mob_count ; ++i)
                    ptr->count[i] = fread_number (fp);
            }
            break;

        case 'E':
            if (!str_cmp(word, "End"))
            {
                // anti-transportation system
                for (i = 0 ; i < ptr->mob_count ; ++i)
                    (get_room_index (ptr->rooms[i]))->area->flags |= AREA_GQACTIVE ;

                ptr->next          = s_gq_master.quests ;
                s_gq_master.quests = ptr ;
               
                return;
            }
            break;

        case 'L':
            KEY("LevelMax", ptr->level_max, fread_number(fp));
            KEY("LevelMin", ptr->level_min, fread_number(fp));
            break;

        case 'M':
            if (!str_cmp(word, "MobCount"))
            {
                fMatch = TRUE;
                ptr->mob_count = fread_number(fp);
                // allocate memory for internal lists
                // Warning! Potential crush-bug if file is corrupted!!!
                ptr->mobs  = (int*) calloc (ptr->mob_count, sizeof (int)) ;
                ptr->rooms = (int*) calloc (ptr->mob_count, sizeof (int)) ;
                ptr->count = (int*) calloc (ptr->mob_count, sizeof (int)) ;
            }
            if (!str_cmp(word, "Mobiles"))
            {
                fMatch = TRUE;
                for (i = 0 ; i < ptr->mob_count ; ++i)
                    ptr->mobs[i] = fread_number (fp);
            }
            KEY("MobTotal", ptr->mob_total, fread_number(fp));
            break;

        case 'P':
            KEY("PlrCount", ptr->plr_count, fread_number(fp));
            break;

        case 'R':
            if (!str_cmp(word, "Rooms"))
            {
                fMatch = TRUE;
                for (i = 0 ; i < ptr->mob_count ; ++i)
                    ptr->rooms[i] = fread_number (fp);
            }
            break;

        case 'T':
            KEY("TicksLeft", ptr->ticks_left, fread_number(fp));
            KEY("TicksTotal", ptr->ticks_total, fread_number(fp));
            KEY("TicksStart", ptr->ticks_start, fread_number(fp));
            KEY("Type", ptr->type, fread_number(fp));
            break;

        }

        if (!fMatch) 
        {
            bug("Fread_gq: no match.", 0);
            fread_to_eol(fp);
        }
    }
}

static void fread_gq_plr (FILE *fp)
{
    char       * word ;
    char       * name ;
    bool         fMatch ;
    GQ_QUESTPLR* ptr ;
    int          i ;
    
    if (!s_gq_master.quests)
        return;

    // allocate memory
    ptr = (GQ_QUESTPLR*) calloc (1, sizeof (GQ_QUESTPLR)) ;

    // allocate memory for internal lists
    ptr->killed_count = (int*) calloc (s_gq_master.quests->mob_count, sizeof (int)) ;

    // other
    ptr->quest = s_gq_master.quests ;
    ptr->name  = NULL ;

    // link to the list
    ptr->next  = s_gq_master.quests->plr ;
    s_gq_master.quests->plr = ptr ;

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'C':
            KEY("Complete", ptr->complete, fread_number(fp));
            break;

        case 'E':
            if (!str_cmp(word, "End"))
                return;
            break;

        case 'K':
            KEY("KillTotal", ptr->killed_total, fread_number(fp));
            break;

        case 'M':
            if (!str_cmp(word, "Mobiles"))
            {
                fMatch = TRUE;
                for (i = 0 ; i < s_gq_master.quests->mob_count ; ++i)
                    ptr->killed_count[i] = fread_number (fp);
            }
            break;

        case 'N':
            if (!str_cmp(word, "Name"))
            {
                fMatch = TRUE;
                name = fread_word(fp);
                ptr->name   = (char*) calloc (strlen (name) + 1, sizeof (char)) ;
                strnzcpy (ptr->name, strlen (name) + 1, name) ;
            }
            break;

        }

        if (!fMatch) 
        {
            bug("Fread_gq_plr: no match.", 0);
            fread_to_eol(fp);
        }
    }
}
