/* $Id: handler.c,v 1.666 2004/09/20 10:49:48 shrike Exp $                          */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
 /************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "merc.h"
#include "obj_prog.h"
#include "clan.h"
#include "raffects.h"
#include "fight.h"
#include "quest.h"
#include "gquest.h"
#include "rating.h"
#include "db/db.h"
#include "olc/olc.h"
#include "db/lang.h"
#include "war.h"
#include "wanderers.h"

DECLARE_DO_FUN(do_resoul    );
DECLARE_DO_FUN(do_raffects  );
DECLARE_DO_FUN(do_return    );
DECLARE_DO_FUN(do_say       );
DECLARE_DO_FUN(do_track     );

extern void remove_contact(CHAR_DATA *ch, CHAR_DATA *victim);
extern bool war_can_charm (CHAR_DATA * ch, CHAR_DATA * victim);
extern bool check_altar_immobility (CHAR_DATA * ch, ROOM_INDEX_DATA * where);
extern flag_to_int_t sect_flags[];

/*
 * Local functions.
 */
void    affect_modify   (CHAR_DATA *ch, AFFECT_DATA *paf, bool fAdd);
void show_russian_name (CHAR_DATA * ch, CHAR_DATA * victim);
int age_to_num  (int age);

int check_exit(const char *arg)
{
    int door = -1;

         if (!str_cmp(arg, "n") || !str_cmp(arg, "north") || !str_prefix(arg, "")) door = 0;
    else if (!str_cmp(arg, "e") || !str_cmp(arg, "east" ) || !str_prefix(arg, "")) door = 1;
    else if (!str_cmp(arg, "s") || !str_cmp(arg, "south") || !str_prefix(arg, "")) door = 2;
    else if (!str_cmp(arg, "w") || !str_cmp(arg, "west" ) || !str_prefix(arg, "")) door = 3;
    else if (!str_cmp(arg, "u") || !str_cmp(arg, "up"   ) || !str_prefix(arg, "") || !str_prefix(arg, "")) door = 4;
    else if (!str_cmp(arg, "d") || !str_cmp(arg, "down" ) || !str_prefix(arg, "") || !str_prefix(arg, "")) door = 5;

    return door;
}

int opposite_door(int door)
{
    int opdoor;

    switch (door) 
    {
    case 0: opdoor = 2;   break;
    case 1: opdoor = 3;   break;
    case 2: opdoor = 0;   break;
    case 3: opdoor = 1;   break;
    case 4: opdoor = 5;   break;
    case 5: opdoor = 4;   break;
    default: opdoor=-1;   break;
    }

    return opdoor;
}


/* friend stuff -- for NPC's mostly */
bool is_friend(CHAR_DATA *ch, CHAR_DATA *victim)
{
    flag64_t off;
    if (is_same_group(ch, victim))
        return TRUE;

    if (!IS_NPC(ch))
        return FALSE;

    off = ch->pIndexData->off_flags;
    if (!IS_NPC(victim))
    {
        if (IS_SET(off, ASSIST_PLAYERS))
            return TRUE;
        else
            return FALSE;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
        return FALSE;

    if (IS_SET(off, ASSIST_ALL))
        return TRUE;

    if (ch->group && ch->group == victim->group)
        return TRUE;

    if (IS_SET(off, ASSIST_VNUM)
        &&  ch->pIndexData == victim->pIndexData)
        return TRUE;

    if (IS_SET(off, ASSIST_RACE) && ch->race == victim->race)
        return TRUE;

    if (IS_SET(off, ASSIST_ALIGN)
        &&  !IS_SET(ch->pIndexData->act, ACT_NOALIGN)
        &&  !IS_SET(victim->pIndexData->act, ACT_NOALIGN)
        &&  ((IS_GOOD(ch) && IS_GOOD(victim)) ||
             (IS_EVIL(ch) && IS_EVIL(victim)) ||
             (IS_NEUTRAL(ch) && IS_NEUTRAL(victim))))
        return TRUE;

    return FALSE;
}

/*
 * Room record:
 * For less than 5 people in room create a new record.
 * Else use the oldest one.
 */

void room_record(const char *name,ROOM_INDEX_DATA *room,int door)
{
    ROOM_HISTORY_DATA *rec;
    int i;

    for (i=0,rec = room->history;i < 5 && rec != NULL;
        i++,rec = rec->next);

    if (i < 5)
    {
        rec = calloc(1, sizeof(*rec));
        rec->next = room->history;
        if (rec->next != NULL)
            rec->next->prev = rec;
        room->history = rec;
        rec->name = NULL;
    }
    else
    {
        rec = room->history->next->next->next->next;
        rec->prev->next = NULL;
        rec->next = room->history;
        rec->next->prev = rec;
        room->history = rec;
    }
    rec->prev = NULL;

    if (rec->name)
    {
        free_string(rec->name);
    }


    rec->name = str_dup(name);
    rec->went = door;
}

/* returns number of people on an object */
int count_users(OBJ_DATA *obj)
{
    CHAR_DATA *fch;
    int count = 0;

    if (obj->in_room == NULL)
        return 0;

    for (fch = obj->in_room->people; fch != NULL; fch = fch->next_in_room)
        if (fch->on == obj)
            count++;

    return count;
}



int floating_time(OBJ_DATA *obj)
{
    int  ftime;

    ftime = 0;
    switch (obj->pIndexData->item_type)
    {
    default: break;
    case ITEM_KEY   : ftime = 1;    break;
    case ITEM_ARMOR     : ftime = 2;    break;
    case ITEM_TREASURE  : ftime = 2;    break;
    case ITEM_PILL  : ftime = 2;    break;
    case ITEM_POTION    : ftime = 3;    break;
    case ITEM_TRASH     : ftime = 3;    break;
    case ITEM_FOOD  : ftime = 4;    break;
    case ITEM_CONTAINER : ftime = 5;    break;
    case ITEM_CORPSE_NPC: ftime = 10;   break;
    case ITEM_CORPSE_PC : ftime = 10;   break;
    }
    ftime = number_fuzzy(ftime) ;

    return(ftime < 0 ? 0 : ftime);
}

flag64_t    dam_to_imm(int dam_type)
{
    flag64_t flags;

    flags = 1 << dam_type;

    switch (dam_type)
    {
        case DAM_BASH:
        case DAM_SLASH:
        case DAM_PIERCE:
            flags = flags | IMM_WEAPON | IMM_PHYSICAL;
            break;
        case DAM_FIRE:
        case DAM_COLD:
        case DAM_LIGHTNING:
        case DAM_ENERGY:
        case DAM_ACID:
        case DAM_LIGHT:
        case DAM_SOUND:
            flags = flags | IMM_MAGIC | IMM_PHYSICAL;
            break;
        case DAM_NEGATIVE:
        case DAM_HOLY:
        case DAM_HARM:
        case DAM_CHARM:
        case DAM_LIGHT_V:
        case DAM_SUMMON:
            flags = flags | IMM_MAGIC | IMM_OTHER;
            break;
        default:
            flags = flags | IMM_OTHER;
            break;
    }
    flags = flags | IMM_NONE;

    return flags;
}

bool is_immune(CHAR_DATA *ch, int dam_type)
{
    return IS_SET(ch->immunes, dam_to_imm(dam_type));
}

/* for vulnerabiltiy and resistant
   the 'globals' (magic and weapons) may be overriden
   three other cases -- wood, silver, and iron -- are checked in fight.c */

int check_immune(CHAR_DATA *ch, int dam_type)
{
    sh_int res;
    flag64_t dam_flag;

    if (is_immune(ch, dam_type))
        return 100;

    dam_flag = dam_to_imm(dam_type);

    res = ch->resists[dam_type];

    if (dam_type == DAM_MAGIC
        || dam_type == DAM_WEAPON
        || dam_type == DAM_PHYSICAL
        || dam_type == DAM_OTHER
        || dam_type == DAM_NONE)
    return res;

    if (IS_SET(dam_flag, IMM_WEAPON))
        res += ch->resists[DAM_WEAPON];
    if (IS_SET(dam_flag, IMM_MAGIC))
        res += ch->resists[DAM_MAGIC];
    if (IS_SET(dam_flag, IMM_PHYSICAL))
        res += ch->resists[DAM_PHYSICAL];
    if (IS_SET(dam_flag, IMM_OTHER))
        res += ch->resists[DAM_OTHER];
    res += ch->resists[DAM_NONE];

    return res;
}


/*
 * This function modifies damage (or saves) with respect to resists.
 * I use "dam *= (1 + x/8)^8" -- something between dam += dam * x
 * and dam = dam * e^x.
 * Actually, I believe that the best resists_modify function
 * should depend on symetric polynomials of higher degree
 * (not only sum of all resists from objects, affects, race, but
 * a more general symetric mono-increasing function of these guys;
 * there's no reason why $\sum_{i=0}^N res_i$, where N is number of resists,
 * is better than $\max_{i=0}^N res_i$,
 * or $N (\sum_{i=0}^N res_i^2)^frac{1}{2}$,
 * or some function of these, for example.
 * So, we'll use $(1 + \frac{\sum_{i=0}^N res_i}{8})^8$ as the second
 * step to the Perfectness. (the first was $(1 + \sum_{i=0}^N res_i)$).
 * /sg
 */

void resists_modify(int *dam, int res, bool fInvert)
{
    int i, base;
    int number, Number;

    number = 3;

    if (res == 0)       /* just to avoid pointless arithmetics */
        return;
    if (fInvert)
        res = 0 - res;

    Number = 1;
    for (i = 0; i < number; i++)
        Number *= 2;
    base = 100 * Number;
    res = UMAX(0, 100 * Number + res);

    for (i = 0; i < number; i++)
        res = res * res / base;

    (*dam) = (*dam) * res / base;
}

void reset_affects(CHAR_DATA *ch, AFFECT_DATA *af)
{
    for (; af != NULL; af = af->next)
    {
        int mod = af->modifier;

        switch (af->location)
        {
        case APPLY_MANA:
            ch->max_mana    -= mod;
            break;
        case APPLY_HIT:
            ch->max_hit -= mod;
            break;
        case APPLY_MOVE:
            ch->max_move    -= mod;
            break;
        }
    }
}

void add_apply(CHAR_DATA *ch, AFFECT_DATA *list)
{
    AFFECT_DATA *af;
    int mod, i;

    for (af = list; af != NULL; af = af->next)
    {
        mod = af->modifier;
        if (af->where == TO_RESIST)
        {
            if (af->bitvector >= 0 && af->bitvector < MAX_DAM)
                ch->resists[af->bitvector] += mod;
            continue;
        }

        switch (af->location)
        {
            case APPLY_STR:     ch->mod_stat[STAT_STR]  += mod; break;
            case APPLY_DEX:     ch->mod_stat[STAT_DEX]  += mod; break;
            case APPLY_INT:     ch->mod_stat[STAT_INT]  += mod; break;
            case APPLY_WIS:     ch->mod_stat[STAT_WIS]  += mod; break;
            case APPLY_CON:     ch->mod_stat[STAT_CON]  += mod; break;
            case APPLY_CHA:     ch->mod_stat[STAT_CHA]  += mod; break;
            case APPLY_LCK:     ch->mod_stat[STAT_LCK]  += mod; break;
            case APPLY_MANA:    ch->max_mana        += mod; break;
            case APPLY_HIT:     ch->max_hit     += mod; break;
            case APPLY_MOVE:    ch->max_move        += mod; break;
            case APPLY_AGE:     ch->played += age_to_num(mod); break;
            case APPLY_AC:
                for (i = 0; i < 4; i ++)
                    ch->armor[i] += mod;
                break;
            case APPLY_HITROLL: ch->hitroll     += mod; break;
            case APPLY_DAMROLL: ch->damroll     += mod; break;
            case APPLY_SIZE:    ch->size        += mod; break;
            case APPLY_SAVES:       ch->saving_throw += mod; break;
            case APPLY_SAVING_ROD:      ch->saving_throw += mod; break;
            case APPLY_SAVING_PETRI:    ch->saving_throw += mod; break;
            case APPLY_SAVING_BREATH:   ch->saving_throw += mod; break;
            case APPLY_SAVING_SPELL:    ch->saving_throw += mod; break;
        }
    }
}

/* used to de-screw characters */
void reset_char(CHAR_DATA *ch)
{
    int loc, stat;
    OBJ_DATA *obj;
    race_t *race;
    int i;

    if (ORG_RACE(ch) == 0)
       SET_ORG_RACE(ch, rn_lookup("human"));
    if (ch->race == 0)
        ch->race = rn_lookup("human");

     race = RACE(ch->race);

    if (!IS_NPC(ch))
        ch->size = race->pcdata->size;

    ch->dam_type = 17;  /* punch */

    ch->affected_by = race->aff;
    ch->immunes = race->immunes;
    ch->form = race->form;
    ch->parts = race->parts;
    ch->pcnt = race->pcnt;

    affect_check(ch, -1, -1);

    for (i = 0; i < MAX_DAM; i++)
        ch->resists[i] = race->resists[i];

    if (ch->psi_def == 0)
    ch->psi_def = -1;
    if (ch->psi_att == 0)
    ch->psi_att = -1;

    if (IS_NPC(ch))
        return;

    if (    ch->pcdata->perm_hit == 0
        ||  ch->pcdata->perm_mana == 0
        ||  ch->pcdata->perm_move == 0
        ||  ch->pcdata->last_level == 0)
    {
        /* do a FULL reset */
        for (loc = 0; loc < MAX_WEAR; loc++)
        {
            obj = get_eq_char(ch,loc);
            if (obj == NULL)
                continue;
            if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
                reset_affects(ch, obj->pIndexData->affected);
            reset_affects(ch, obj->affected);
        }
        reset_affects(ch, ch->affected);

        /* now reset the permanent stats */
        ch->pcdata->perm_hit    = ch->max_hit;
        ch->pcdata->perm_mana   = ch->max_mana;
        ch->pcdata->perm_move   = ch->max_move;
        ch->pcdata->last_level  = ch->played/3600;
    }

    switch (ch->pcdata->true_sex)
    {
        case SEX_MALE:
        case SEX_FEMALE:
        case SEX_NEUTRAL:
        case SEX_PLURAL:
            break;
        default:
            ch->pcdata->true_sex = SEX_NEUTRAL;
    }

    /* now restore the character to his/her true condition */

    ch->sex         = ch->pcdata->true_sex;
    ch->max_hit     = ch->pcdata->perm_hit;
    ch->max_mana    = ch->pcdata->perm_mana;
    ch->max_move    = ch->pcdata->perm_move;

    for (stat = 0; stat < MAX_STATS; stat++)
        ch->mod_stat[stat] = 0;

    for (i = 0; i < 4; i++)
        ch->armor[i]    = 100;

    ch->hitroll     = 0;
    ch->damroll     = 0;
    ch->saving_throw    = 0;


    /* now start adding back the effects */
    for (loc = 0; loc < MAX_WEAR; loc++)
    {
        obj = get_eq_char(ch,loc);
        if (obj == NULL)
            continue;
        for (i = 0; i < 4; i++)
            ch->armor[i] -= apply_ac(obj, loc, i);


        if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
            add_apply(ch, obj->pIndexData->affected);
        add_apply(ch, obj->affected);
    }

    /* now add back spell effects */
    add_apply(ch, ch->affected);

    /* make sure sex is RIGHT! */
    switch (ch->sex)
    {
        case SEX_MALE:
        case SEX_FEMALE:
        case SEX_NEUTRAL:
        case SEX_PLURAL:
            break;
        default:
            ch->sex = ch->pcdata->true_sex;
        break;
    }
    update_skills(ch);
}

/*
 * Retrieve a character's age.
 */
int get_age(CHAR_DATA *ch)
{
    return 17 + (ch->played + (int) (current_time - ch->logon)) / 72000;
}

int age_to_num(int age)
{
    return  age * 72000;
}

/*
 * Retrieve a character's carry capacity.
 */
int can_carry_n(CHAR_DATA *ch)
{
    if (IS_IMMORTAL(ch))
        return 1000;

    if (IS_NPC(ch) && IS_SET(ch->pIndexData->act, ACT_PET))
        return 0;

    return MAX_WEAR + get_curr_stat(ch,STAT_DEX) - 10 + ch->size / 3;
}

/*
 * Retrieve a character's carry capacity.
 */
int can_carry_w(CHAR_DATA *ch)
{
    if (IS_IMMORTAL(ch))
        return 10000000;

    if (IS_NPC(ch) && IS_SET(ch->pIndexData->act, ACT_PET))
        return 0;

    return str_app[get_curr_stat(ch,STAT_STR)].carry * 10 + ch->level * 25;
}

/*
 * See if a string is one of the names of an object.
 */
bool is_name_raw(const char *str, const char *namelist,
                 int (*cmpfun)(const char*, const char*))
{
    char name[MAX_INPUT_LENGTH], part[MAX_INPUT_LENGTH];
    const char *list, *string;

    if (IS_NULLSTR(namelist) || IS_NULLSTR(str))
        return FALSE;

    if (!str_cmp(namelist, "all"))
        return TRUE;

    string = str;
    /* we need ALL parts of string to match part of namelist */
    for (; ;)
    { /* start parsing string */
        str = one_argument(str, part, sizeof(part));

        if (part[0] == '\0')
            return TRUE;

        /* check to see if this is part of namelist */
        list = namelist;
        for (; ;)
        { /* start parsing namelist */
            list = one_argument(list, name, sizeof(name));
            if (name[0] == '\0')  /* this name was not found */
                return FALSE;

            if (!cmpfun(string, name))
                return TRUE; /* full pattern match */

            if (!cmpfun(part, name))
                break;
        }
    }
}

bool is_name(const char *str, const char *namelist)
{
    return is_name_raw(str, namelist, str_prefix);
}

void cat_name(char *buf, const char *name, size_t len)
{
    bool have_spaces = strpbrk(name, " \t") != NULL;

    if (buf[0])
        strnzcat(buf, len, " ");
    if (have_spaces)
        strnzcat(buf, len, "'");
    strnzcat(buf, len, name);
    if (have_spaces)
        strnzcat(buf, len, "'");
}

#define NE_F_DELETE (A) /* delete name if found     */
#define NE_F_ADD    (B) /* add name if not found    */

void name_edit(CHAR_DATA *ch, const char *name,
               const char *editor_name, const char **nl, int flags)
{
    bool found;
    const char *p;
    char buf[MAX_STRING_LENGTH];

    if (!str_cmp(name, "all"))
    {
        free_string(*nl);
        *nl = str_dup(name);
        act_puts("$t: name list set to ALL.", ch, editor_name, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    if (!str_cmp(name, "none"))
    {
        free_string(*nl);
        *nl = str_empty;
        if (editor_name && ch != NULL)
            act_puts("$t: name list reset.", ch, editor_name, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    if (!str_cmp(*nl, "all"))
    {
        free_string(*nl);
        *nl = str_empty;
    }

    found = FALSE;
    p = *nl;
    buf[0] = '\0';
    for (;;)
    {
        char arg[MAX_STRING_LENGTH];

        p = one_argument(p, arg, sizeof(arg));

        if (arg[0] == '\0')
            break;

        if (!str_cmp(name, arg))
        {
            found = TRUE;
            if (IS_SET(flags, NE_F_DELETE))
                continue;
        }

        cat_name(buf, arg, sizeof(buf));
    }

    if (!found)
    {
        if (!IS_SET(flags, NE_F_ADD))
            return;

        if (strlen(buf) + strlen(name) + 4 > MAX_STRING_LENGTH)
        {
            if (editor_name && ch != NULL)
                act_puts("$t: name list too long", ch, editor_name, NULL, TO_CHAR, POS_DEAD);
            return;
        }
        cat_name(buf, name, sizeof(buf));
        if (editor_name && ch != NULL)
            act_puts("$t: $T: name added.", ch, editor_name, name, TO_CHAR, POS_DEAD);
    }
    else
    {
        if (!IS_SET(flags, NE_F_DELETE))
            return;

        if (editor_name && ch != NULL)
            act_puts("$t: $T: name removed.", ch, editor_name, name, TO_CHAR, POS_DEAD);
    }

    free_string(*nl);
    *nl = str_dup(buf);
}

void name_add(CHAR_DATA *ch, const char *name,
              const char *editor_name, const char **nl)
{
    name_edit(ch, name, editor_name, nl, NE_F_ADD);
}

void name_delete(CHAR_DATA *ch, const char *name,
                 const char *editor_name, const char **nl)
{
    name_edit(ch, name, editor_name, nl, NE_F_DELETE);
}

void name_toggle(CHAR_DATA *ch, const char *name,
                 const char *editor_name, const char **nl)
{
    name_edit(ch, name, editor_name, nl, NE_F_ADD | NE_F_DELETE);
}

/* enchanted stuff for eq */
void affect_enchant(OBJ_DATA *obj)
{
    /* okay, move all the old flags into new vectors if we have to */
    if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
    {
        AFFECT_DATA *paf, *af_new;
        SET_BIT(obj->extra_flags, ITEM_ENCHANTED);

        for (paf = obj->pIndexData->affected;
            paf != NULL; paf = paf->next)
        {
            af_new = aff_new();

            af_new->next = obj->affected;
            obj->affected = af_new;

            af_new->where   = paf->where;
            af_new->type        = UMAX(0,paf->type);
            af_new->level       = paf->level;
            af_new->duration    = paf->duration;
            af_new->location    = paf->location;
            af_new->modifier    = paf->modifier;
            af_new->bitvector   = paf->bitvector;
        }
    }
}

/*
 * Apply or remove an affect to a character.
 */
void affect_modify(CHAR_DATA *ch, AFFECT_DATA *paf, bool fAdd)
{
    OBJ_DATA *wield, *obj2;
    int mod, i;

    if (paf->where == TO_RESIST && paf->bitvector >= MAX_DAM)
    {
        bug("Affect_modify: Invalid bitvector %d in resistance affect.",
            paf->bitvector);
        return;
    }
/*    if (paf->where == TO_IMMUNE && paf->bitvector >= MAX_DAM)
    {
        bug("Affect_modify: Invalid bitvector %d in immunity affect.",
            paf->bitvector);
        return;
    }*/


    mod = paf->modifier;

    if (fAdd)
    {
        switch (paf->where)
        {
        case TO_AFFECTS:
            SET_BIT(ch->affected_by, paf->bitvector);
            break;
        case TO_RESIST:
            ch->resists[paf->bitvector] += mod;
            break;
        case TO_FORM:
            SET_BIT(ch->form, paf->bitvector);
            break;
        case TO_IMMUNE:
            SET_BIT(ch->immunes, paf->bitvector);
            break;
        }
    }
    else
    {
        switch (paf->where)
        {
        case TO_AFFECTS:
            REMOVE_BIT(ch->affected_by, paf->bitvector);
            break;
        case TO_RESIST:
            ch->resists[paf->bitvector] -= mod;
            break;
        case TO_FORM:
            REMOVE_BIT(ch->form, paf->bitvector);
            break;
        case TO_IMMUNE:
            REMOVE_BIT(ch->immunes, paf->bitvector);
            break;
        }
        mod = 0 - mod;
    }

    if (paf->where != TO_RESIST)
    {
        switch (paf->location)
        {
        default:
            bug("Affect_modify: unknown location %d", paf->location);
            if (ch->pIndexData)
                log_printf("mob vnum %d, name: %s.",
                           ch->pIndexData->vnum, ch->name);
            else
                log_printf("name %s", ch->name);
            return;

        case APPLY_NONE:                        break;
        case APPLY_CLASS:                       break;
        case APPLY_LEVEL:   ch->aff_level += mod;           break;
        case APPLY_HEIGHT:                      break;
        case APPLY_WEIGHT:                      break;
        case APPLY_GOLD:                        break;
        case APPLY_EXP:                     break;
        case APPLY_STR:           ch->mod_stat[STAT_STR]    += mod; break;
        case APPLY_DEX:           ch->mod_stat[STAT_DEX]    += mod; break;
        case APPLY_INT:           ch->mod_stat[STAT_INT]    += mod; break;
        case APPLY_WIS:           ch->mod_stat[STAT_WIS]    += mod; break;
        case APPLY_CON:           ch->mod_stat[STAT_CON]    += mod; break;
        case APPLY_CHA:       ch->mod_stat[STAT_CHA]    += mod; break;
        case APPLY_LCK:       ch->mod_stat[STAT_LCK]    += mod; break;
        case APPLY_AGE: ch->played += age_to_num(mod);  break;
        case APPLY_MANA:          ch->max_mana      += mod; break;
        case APPLY_HIT:           ch->max_hit       += mod; break;
        case APPLY_MOVE:          ch->max_move      += mod; break;
        case APPLY_AC:
            for (i = 0; i < 4; i ++)
                ch->armor[i] += mod;
            break;
        case APPLY_HITROLL:       ch->hitroll       += mod; break;
        case APPLY_DAMROLL:       ch->damroll       += mod; break;
        case APPLY_SIZE:        ch->size        += mod; break;
        case APPLY_SAVES:   ch->saving_throw        += mod; break;
        case APPLY_SAVING_ROD:    ch->saving_throw      += mod; break;
        case APPLY_SAVING_PETRI:  ch->saving_throw      += mod; break;
        case APPLY_SAVING_BREATH: ch->saving_throw      += mod; break;
        case APPLY_SAVING_SPELL:  ch->saving_throw      += mod; break;
        case APPLY_SPELL_AFFECT:                    break;
        case APPLY_RACE:
            {
                int from;
                int to;
                race_t *rto;
                race_t *rfrom;

                if (fAdd)
                {
                    from = ORG_RACE(ch);
                    to = ch->race = paf->modifier;
                }
                else
                {
                    from = ch->race;
                    to = ch->race = ORG_RACE(ch);
                }

                rfrom = race_lookup(from);
                rto = race_lookup(to);
                if (!rfrom || !rto || !rfrom->pcdata || !rto->pcdata)
                    return;

                REMOVE_BIT(ch->affected_by, rfrom->aff);
                SET_BIT(ch->affected_by, rto->aff);
                affect_check(ch, TO_AFFECTS, rfrom->aff);

                REMOVE_BIT(ch->form, rfrom->form);
                SET_BIT(ch->form, rto->form);
                affect_check(ch, TO_FORM, rfrom->form);

                REMOVE_BIT(ch->immunes, rfrom->immunes);
                SET_BIT(ch->immunes, rto->immunes);
                affect_check(ch, TO_IMMUNE, rfrom->immunes);

                ch->parts = rto->parts;

                for (i = 0; i < MAX_DAM; i++)
                    ch->resists[i] += rto->resists[i] - rfrom->resists[i];

                /*
                 *    ,     ,
                 *     ,    
                 *     "affect_check(ch, TO_AFFECTS, rfrom->aff, APPLY_SIZE)".
                 *     size    
                 *      .
                 *   -, -,      size,
                 *      . /sg
                 */

                // ch->size = rto->pcdata->size;

                update_skills(ch);
            }
            break;
        }
    }

    /*
     * Check for weapon wielding.
     * Guard against recursion (for weapons with affects).
     */
    if (!IS_NPC(ch) && (wield = get_eq_char(ch, WEAR_WIELD)) != NULL
        &&  get_obj_weight(wield) > (str_app[get_curr_stat(ch,STAT_STR)].wield*10))
    {
        static int depth;

        if (depth == 0
            && !IS_OBJ_STAT(wield, ITEM_NOREMOVE))
        {
            depth++;
            obj_from_char(wield);
            if (IS_OBJ_STAT(wield, ITEM_NODROP)
                || IS_OBJ_STAT(wield, ITEM_INVENTORY)
                || (ch->in_room
                    && IS_SET(ch->in_room->room_flags, ROOM_BATTLE_ARENA))
                || (ch->in_war && (ch->war_status == PS_ALIVE)))
            {
                obj_to_char(wield, ch);
                act("You remove $p.", ch, wield, NULL, TO_CHAR);
                act("$n removes $p.", ch, wield, NULL, TO_ROOM);
            }
            else
            {
                obj_to_room(wield, ch->in_room);
                act("You drop $p!", ch, wield, NULL, TO_CHAR);
                act("$n drops $p!", ch, wield, NULL, TO_ROOM);
            }

            if ((obj2 = get_eq_char(ch, WEAR_SECOND_WIELD)) != NULL)
            {
                if (!IS_OBJ_STAT(obj2, ITEM_NOREMOVE))
                {
                    act("You wield his second weapon as your first!",
                        ch, NULL, NULL, TO_CHAR);
                    act("$n wields his second weapon as first!",
                        ch, NULL, NULL, TO_ROOM);
                    unequip_char(ch, obj2);
                    equip_char(ch, obj2 , WEAR_WIELD);
                }
                else
                {
                    act("You remove $p!",
                        ch, obj2, NULL, TO_CHAR);
                    act("$n removes $p!",
                        ch, obj2, NULL, TO_ROOM);
                    unequip_char(ch, obj2);
                }
            }
            depth--;
        }
    }
}


/* find an effect in an affect list */
AFFECT_DATA  *affect_find(AFFECT_DATA *paf, int sn)
{
    AFFECT_DATA *paf_find;

    for (paf_find = paf; paf_find != NULL; paf_find = paf_find->next)
    {
        if (paf_find->type == sn)
            return paf_find;
    }

    return NULL;
}

void affect_check_list(CHAR_DATA *ch, AFFECT_DATA *paf,
                       int where, flag64_t vector)
{
    for (; paf; paf = paf->next)
        if ((where < 0 || paf->where == where)
            &&  (paf->bitvector & vector))
            switch (paf->where)
            {
            case TO_AFFECTS:
                SET_BIT(ch->affected_by, paf->bitvector);
                break;
            case TO_FORM:
                SET_BIT(ch->form, paf->bitvector);
                break;
            case TO_IMMUNE:
                SET_BIT(ch->immunes, paf->bitvector);
                break;
            }
}

/* fix object affects when removing one */
void affect_check(CHAR_DATA *ch, int where, flag64_t vector)
{
    OBJ_DATA *obj;

    if (where == TO_OBJECT || where == TO_WEAPON
        || where == TO_RESIST || vector == 0)
        return;

    affect_check_list(ch, ch->affected, where, vector);
    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
    {
        if (obj->wear_loc == WEAR_NONE || obj->wear_loc == WEAR_STUCK_IN)
            continue;
        affect_check_list(ch, obj->affected, where, vector);

        if (IS_SET(obj->extra_flags, ITEM_ENCHANTED))
            continue;

        affect_check_list(ch, obj->pIndexData->affected, where, vector);
    }
}

/*
 * Give an affect to a char.
 */
void affect_to_char(CHAR_DATA *ch, AFFECT_DATA *paf)
{
    AFFECT_DATA *paf_new;

    paf_new = aff_new();

    *paf_new    = *paf;
    paf_new->next   = ch->affected;
    ch->affected    = paf_new;

    affect_modify(ch, paf_new, TRUE);
}

/* give an affect to an object */
void affect_to_obj(OBJ_DATA *obj, AFFECT_DATA *paf)
{
    AFFECT_DATA *paf_new;

    paf_new = aff_new();
    *paf_new    = *paf;
    paf_new->next   = obj->affected;
    obj->affected   = paf_new;

    /* apply any affect vectors to the object's extra_flags */
    if (paf->bitvector)
        switch (paf->where)
        {
        case TO_OBJECT:
            SET_BIT(obj->extra_flags,paf->bitvector);
            break;
        case TO_WEAPON:
            if (obj->pIndexData->item_type == ITEM_WEAPON)
                SET_BIT(obj->value[4],paf->bitvector);
            break;
        }
}



/*
 * Remove an affect from a char.
 */
void affect_remove(CHAR_DATA *ch, AFFECT_DATA *paf)
{
    int where;
    int vector;

    if (ch->affected == NULL)
    {
        bug("Affect_remove: no affect.", 0);
        return;
    }

    affect_modify(ch, paf, FALSE);
    where = paf->where;
    vector = paf->bitvector;

    if (paf == ch->affected)
        ch->affected    = paf->next;
    else
    {
        AFFECT_DATA *prev;

        for (prev = ch->affected; prev; prev = prev->next)
        {
            if (prev->next == paf)
            {
                prev->next = paf->next;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Affect_remove: cannot find paf.", 0);
            return;
        }
    }

    aff_free(paf);
    affect_check(ch, where, vector);
}

void affect_remove_obj(OBJ_DATA *obj, AFFECT_DATA *paf)
{
    int where, vector;

    if (obj->affected == NULL)
        return;

    if (obj->carried_by != NULL && obj->wear_loc != -1)
        affect_modify(obj->carried_by, paf, FALSE);

    where = paf->where;
    vector = paf->bitvector;

    /* remove flags from the object if needed */
    if (paf->bitvector)
        switch (paf->where)
        {
        case TO_OBJECT:
            REMOVE_BIT(obj->extra_flags,paf->bitvector);
            break;
        case TO_WEAPON:
            if (obj->pIndexData->item_type == ITEM_WEAPON)
                REMOVE_BIT(obj->value[4],paf->bitvector);
            break;
        }

    if (paf == obj->affected)
        obj->affected    = paf->next;
    else
    {
        AFFECT_DATA *prev;

        for (prev = obj->affected; prev != NULL; prev = prev->next)
        {
            if (prev->next == paf)
            {
                prev->next = paf->next;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Affect_remove_object: cannot find paf.", 0);
            return;
        }
    }

    aff_free(paf);

    if (obj->carried_by != NULL && obj->wear_loc != -1)
        affect_check(obj->carried_by,where,vector);
}

/*
 * Strip all affects of a given sn.
 */
void affect_strip(CHAR_DATA *ch, int sn)
{
    AFFECT_DATA *paf;
    AFFECT_DATA *paf_next;

    for (paf = ch->affected; paf; paf = paf_next)
    {
        paf_next = paf->next;
        if (paf->type == sn)
            affect_remove(ch, paf);
    }
}

/*
 * strip all affects which affect given bitvector
 */
void affect_bit_strip(CHAR_DATA *ch, int where, flag64_t bits)
{
    AFFECT_DATA *paf, *paf_next;

    for (paf = ch->affected; paf; paf = paf_next)
    {
        paf_next = paf->next;
        if (paf->where == where && (paf->bitvector & bits))
            affect_remove(ch, paf);
    }
}

/*
 * Return true if a char is affected by a spell.
 */
bool is_affected(CHAR_DATA *ch, int sn)
{
    AFFECT_DATA *paf;

    if (ch == NULL)
        return FALSE;

    for (paf = ch->affected; paf; paf = paf->next)
        if (paf->type == sn)
            return TRUE;

    return FALSE;
}

/*
 * Return 0 if not affected by sn, otherwise sn->level
 */
int affect_level(CHAR_DATA *ch, int sn)
{
    AFFECT_DATA *paf;

    for (paf = ch->affected; paf; paf = paf->next)
        if (paf->type == sn)
            return(paf->level > 0 ? paf->level : 1);

    return 0;
}

bool is_bit_affected(CHAR_DATA *ch, int where, flag64_t bits)
{
    AFFECT_DATA *paf;

    for (paf = ch->affected; paf; paf = paf->next)
        if (paf->where == where && (paf->bitvector & bits))
            return TRUE;

    return FALSE;
}

bool has_obj_affect(CHAR_DATA *ch, int vector)
{
    OBJ_DATA *obj;

    for (obj = ch->carrying; obj; obj = obj->next_content)
    {
        AFFECT_DATA *paf;

        if (obj->wear_loc == -1 || obj->wear_loc == WEAR_STUCK_IN)
            continue;

        for (paf = obj->affected; paf; paf = paf->next)
            if (paf->bitvector & vector)
                return TRUE;

        if (IS_SET(obj->extra_flags, ITEM_ENCHANTED))
            continue;

        for (paf = obj->pIndexData->affected; paf; paf = paf->next)
            if (paf->bitvector & vector)
                return TRUE;
    }
    return FALSE;
}


/*
 * Add or enhance an affect.
 */
void affect_join(CHAR_DATA *ch, AFFECT_DATA *paf)
{
    AFFECT_DATA *paf_old;
    bool found;

    found = FALSE;
    for (paf_old = ch->affected; paf_old != NULL; paf_old = paf_old->next)
    {
        if (paf_old->type == paf->type)
        {
            paf->level = (paf->level += paf_old->level) / 2;
            paf->duration += paf_old->duration;
            paf->modifier += paf_old->modifier;
            affect_remove(ch, paf_old);
            break;
        }
    }

    affect_to_char(ch, paf);
    return;
}

/*
 * Move a char out of a room.
 */
void char_from_room(CHAR_DATA *ch)
{
    OBJ_DATA *obj;
    ROOM_INDEX_DATA *prev_room = ch->in_room;

    if (ch->in_room == NULL)
    {
        bug("Char_from_room: NULL.", 0);
        return;
    }

    if (!IS_NPC(ch))
        --ch->in_room->area->nplayer;

    if ((obj = get_eq_char(ch, WEAR_LIGHT)) != NULL
        &&   obj->pIndexData->item_type == ITEM_LIGHT
        &&   obj->value[2] != 0
        &&   ch->in_room->light > 0)
        --ch->in_room->light;

    if (ch == ch->in_room->people)
        ch->in_room->people = ch->next_in_room;
    else
    {
        CHAR_DATA *prev;

        for (prev = ch->in_room->people; prev; prev = prev->next_in_room)
        {
            if (prev->next_in_room == ch)
            {
                prev->next_in_room = ch->next_in_room;
                break;
            }
        }

        if (prev == NULL)
            bug("Char_from_room: ch not found.", 0);
    }

    ch->in_room      = NULL;
    ch->next_in_room = NULL;
    ch->on       = NULL;  /* sanity check! */

    if (MOUNTED (ch))
    {
        ch->mount->riding  = FALSE ;
        ch->riding         = FALSE ;
    }

    if (RIDDEN (ch))
    {
        ch->mount->riding  = FALSE ;
        ch->riding         = FALSE ;
    }

    if (prev_room && prev_room->affected_by)
        raffect_back_char (prev_room, ch) ;

    return ;
}

/*
 * Move a char into a room.
 */
void char_to_room(CHAR_DATA *ch, ROOM_INDEX_DATA *pRoomIndex)
{
    OBJ_DATA      * obj;
    extern bool     exploring;
    CHAR_DATA     * vch;
    flag_to_int_t * sect;
    race_t        * race;

    if (pRoomIndex == NULL)
    {
        ROOM_INDEX_DATA *room;

        bug("Char_to_room: NULL.", 0);

        if ((room = get_room_index(ROOM_VNUM_TEMPLE)) != NULL)
            char_to_room(ch,room);

        return;
    }

    ch->in_room         = pRoomIndex;
    ch->next_in_room    = pRoomIndex->people;
    pRoomIndex->people  = ch;

    if (!IS_NPC(ch))
    {
        if (ch->in_room->area->empty)
        {
            ch->in_room->area->empty = FALSE;
            ch->in_room->area->age = 0;
        }
        ++ch->in_room->area->nplayer;
    }

    if ((obj = get_eq_char(ch, WEAR_LIGHT)) != NULL
    && obj->pIndexData->item_type == ITEM_LIGHT
    && obj->value[2] != 0)
        ++ch->in_room->light;

    while (IS_AFFECTED(ch, AFF_PLAGUE))
    {
        AFFECT_DATA *af, plague;

        for (af = ch->affected; af != NULL; af = af->next)
        {
            if (af->type == gsn_plague)
                break;
        }

        if (af == NULL)
        {
            REMOVE_BIT(ch->affected_by, AFF_PLAGUE);
            break;
        }

        if (af->level == 1)
            break;

        plague.where        = TO_AFFECTS;
        plague.type         = gsn_plague;
        plague.level        = af->level - 1;
        plague.duration     = number_range(1,2 * plague.level);
        plague.location     = APPLY_STR;
        plague.modifier     = -5;
        plague.bitvector    = AFF_PLAGUE;

        for (vch = ch->in_room->people; vch != NULL; vch = vch->next_in_room)
        {
            if (!saves_spell(plague.level - 2,vch,DAM_DISEASE)
                &&  !IS_IMMORTAL(vch) &&
                !IS_AFFECTED(vch,AFF_PLAGUE) && number_bits(6) == 0)
            {
                char_act("    .", vch);
                act("$n     .",vch,NULL,NULL,TO_ROOM);
                affect_join(vch,&plague);
            }
        }
        break;
    }

    // affects
    if (ch->in_room->affected_by)
    {
        if (IS_IMMORTAL (ch)) do_raffects(ch, str_empty);
        else                  raffect_to_char(ch->in_room, ch);
    }

    // exploring enabled/disabled (c) Illinar
    if (exploring)
        explore_update_char (ch, pRoomIndex);

    if (ch->desc && OLCED(ch) && IS_EDIT(ch, ED_ROOM))
        roomed_edit_room(ch, pRoomIndex, TRUE);

    // add race_specific sector bonus
    // currently only to level
    if (IS_SET(muddy_mode, MUDDY_SECTOR_BONUS))
    {
        for (sect = sect_flags; sect->flag != FLAG_TO_INT_MAX; sect++)
        {
            if (ch->in_room->sector_type == sect->n_int)
                break;
        }
        if (sect->flag != FLAG_TO_INT_MAX) // to avoid bugs
        {
            race = RACE(ORG_RACE(ch));
            ch->sector_level = 0;
            if (race->loved_sectors && IS_SET(race->loved_sectors, sect->flag))
                ch->sector_level = 1;
            if (race->hated_sectors && IS_SET(race->hated_sectors, sect->flag))
                ch->sector_level = -1;
        }
    }

    // Welesh : mobs with 'BODYGUARD' extra ability
    // We need to choose right probability.
    // May be, it would be nice to use the code above as separate function
    // But! if we'll use such function we'll have to use it in 100 places,
    // almost each time we call char_to_room
    // (otherwise ch will see this message first, and only then room desc)
/*
    for (vch = ch->in_room->people; vch != NULL; vch = vch->next_in_room)
    {
        if (IS_SET (vch->abilities, EA_BODYGUARD)
            && vch->master != NULL
            && (vch->master->in_room == vch->in_room)
            && in_PK (ch, vch->master)
            && can_see (ch, vch)
            && !is_same_group (ch, vch)
            && !IS_NPC (ch)
            && vch->position == POS_STANDING
            && number_percent() > 90) // 10% - for test
        {
            act("$N blocks $n's way.", ch, NULL, vch, TO_ROOM);
            act("\n\n$N {Rblocks your way{x!\n\n", ch, NULL, vch, TO_CHAR);
            WAIT_STATE (ch, PULSE_VIOLENCE); // small wait state
            break;
        }
    }
*/
}

/*
 * Give an obj to a char.
 */
void obj_to_char(OBJ_DATA *obj, CHAR_DATA *ch)
{
    obj->next_content   = ch->carrying;
    ch->carrying        = obj;
    obj->carried_by     = ch;
    obj->in_room        = NULL;
    obj->in_obj     = NULL;
    ch->carry_number    += get_obj_number(obj);
    ch->carry_weight    += get_obj_weight(obj);
}

/*
 * Take an obj from its character.
 */
void obj_from_char(OBJ_DATA *obj)
{
    CHAR_DATA *ch;

    if ((ch = obj->carried_by) == NULL)
    {
        bug("Obj_from_char: null ch.", 0);
        return;
    }

    if (obj->wear_loc != WEAR_NONE)
        unequip_char(ch, obj);

    if (ch->carrying == obj)
        ch->carrying = obj->next_content;
    else
    {
        OBJ_DATA *prev;

        for (prev = ch->carrying; prev; prev = prev->next_content)
        {
            if (prev->next_content == obj)
            {
                prev->next_content = obj->next_content;
                break;
            }
        }

        if (prev == NULL)
            bug("Obj_from_char: obj not in list.", 0);
    }

    obj->carried_by     = NULL;
    obj->next_content   = NULL;
    ch->carry_number    -= get_obj_number(obj);
    ch->carry_weight    -= get_obj_weight(obj);

/* XXX */
/*
    if (!IS_NPC(ch)) {
        int vnum = obj->pIndexData->vnum;
        const char *p = mlstr_mval(obj->short_descr);

        if (p && strstr(p, ch->name)
        &&  ch->level < obj->pIndexData->level
        &&  IS_SET(obj->pIndexData->extra_flags, ITEM_QUEST)) {
            ch->pcdata->questpoints += 1000;
            qtrouble_set(ch, vnum, 4);
        }
    }
*/
}

/*
 * Find the ac value of an obj, including position effect.
 */
int apply_ac(OBJ_DATA *obj, int iWear, int type)
{
    if (obj->pIndexData->item_type != ITEM_ARMOR)
        return 0;

    switch (iWear)
    {
    case WEAR_BODY:     return 3 * obj->value[type];
    case WEAR_LEGS:
    case WEAR_ABOUT:    return 2 * obj->value[type];
    default:            return  obj->value[type];
    }

    return 0;
}

/*
 * Find a piece of eq on a character.
 */
OBJ_DATA *get_eq_char(CHAR_DATA *ch, int iWear)
{
    OBJ_DATA *obj;

    if (ch == NULL)
        return NULL;

    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
        if (obj->wear_loc == iWear)
            return obj;

    return NULL;
}

/*
 * Equip a char with an obj. Return obj on success. Otherwise returns NULL.
 */
OBJ_DATA * equip_char(CHAR_DATA *ch, OBJ_DATA *obj, int iWear)
{
    AFFECT_DATA *paf;
    int i;

    if (iWear == WEAR_STUCK_IN)
    {
        obj->wear_loc = iWear;
        return obj;
    }

    if (get_eq_char(ch, iWear))
    {
        bug("Equip_char: already equipped (%d).", iWear);
        if (IS_NPC(ch))
            log_printf("mob name: %s, vnum %d, in room %d",
                       ch->pIndexData->name, ch->pIndexData->vnum,
                       (ch->in_room != NULL) ? ch->in_room->vnum : 0);
        else
            log_printf("char name: %s", ch->name);
        return NULL;
    }

    if (!class_can_wear(ch, iWear) || !race_can_wear(ch, iWear))
        return NULL;

    if (!IS_IMMORTAL(ch)
        &&  (  (IS_OBJ_STAT(obj, ITEM_ANTI_EVIL)    && IS_EVIL(ch)   )
               || (IS_OBJ_STAT(obj, ITEM_ANTI_GOOD)    && IS_GOOD(ch)   )
               || (IS_OBJ_STAT(obj, ITEM_ANTI_NEUTRAL) && IS_NEUTRAL(ch))))
    {
        /*
         * Thanks to Morgenes for the bug fix here!
         */
        act("     $p.", ch, obj, NULL, TO_CHAR);
        act("$n     $p.",  ch, obj, NULL, TO_ROOM);
//      obj_from_char(obj);
//      obj_to_room(obj, ch->in_room);
        return NULL;
    }

    // size stuff: for non-imms and non-limited objects
    //if (!IS_NPC(ch) && !IS_IMMORTAL (ch) && obj->pIndexData->limit == -1 && obj->size != 0)
    if (!IS_NPC(ch) && obj->pIndexData->limit == -1 && obj->size != 0)
    {
        int diff = ch->size - obj->size;
        if (diff > 3) // very small obj => can't wear 
        {
            act("You can't wear $p, it's too small for you.", ch, obj, NULL, TO_CHAR);
            return NULL;
        }
        else if (diff > 1) // 2..3: small obj: can wear but eq is to be damaged
        {
            obj->condition -= 20;
            act("$p is too small for you and is damaged when you wear it.", ch, obj, NULL, TO_CHAR);
            if (obj->condition < 1) 
            {
                act("$p{g   !{x", ch, obj, NULL, TO_ALL);
                extract_obj(obj);
                return NULL;
            }
        }
        else if (diff > -2) // -1..1 : just wear
        {
            // fallthru
        }
        else  // too large object: let it fall down
        {
            act("You can't wear $p, it's too large for you. You drop it.", ch, obj, NULL, TO_CHAR);
            obj_from_char(obj);
            obj_to_room(obj, ch->in_room);
            return NULL;
        }
    }

    for (i = 0; i < 4; i++)
        ch->armor[i]        -= apply_ac(obj, iWear,i);
    obj->wear_loc    = iWear;

    if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
        for (paf = obj->pIndexData->affected; paf; paf = paf->next)
            if (paf->location != APPLY_SPELL_AFFECT)
                affect_modify(ch, paf, TRUE);
    for (paf = obj->affected; paf; paf = paf->next)
        if (paf->location == APPLY_SPELL_AFFECT)
            affect_to_char(ch, paf);
        else
            affect_modify(ch, paf, TRUE);

    if (obj->pIndexData->item_type == ITEM_LIGHT
        &&  obj->value[2] != 0
        &&  ch->in_room != NULL)
        ++ch->in_room->light;

    oprog_call(OPROG_WEAR, obj, ch, NULL);
    if (ch->fighting != NULL
        &&  iWear != WEAR_WIELD
        &&  iWear != WEAR_SECOND_WIELD
        &&  iWear != WEAR_HOLD
        &&  iWear != WEAR_SHIELD)
        WAIT_STATE(ch, 3 * PULSE_VIOLENCE);
    return obj;
}

void strip_obj_affects(CHAR_DATA *ch, OBJ_DATA *obj, AFFECT_DATA *paf)
{
    AFFECT_DATA *lpaf_next = NULL;
    AFFECT_DATA *lpaf = NULL;

    for (; paf != NULL; paf = paf->next)
    {
        if (paf->location == APPLY_SPELL_AFFECT)
        {
            for (lpaf = ch->affected; lpaf != NULL; lpaf = lpaf_next)
            {
                lpaf_next = lpaf->next;
                if ((lpaf->type == paf->type)
                    &&  (lpaf->level == paf->level)
                    &&  (lpaf->location == APPLY_SPELL_AFFECT))
                {
                    affect_remove(ch, lpaf);
                    lpaf_next = NULL;
                }
            }
        }
        else
        {
            affect_modify(ch, paf, FALSE);
            affect_check(ch,paf->where,paf->bitvector);
        }
    }
}


/*
 * Unequip a char with an obj.
 */
void unequip_char(CHAR_DATA *ch, OBJ_DATA *obj)
{
    int i;

    if (obj->wear_loc == WEAR_NONE)
    {
        bug("Unequip_char: already unequipped.", 0);
        return;
    }

    if (obj->wear_loc == WEAR_STUCK_IN)
    {
        obj->wear_loc = WEAR_NONE;
        return;
    }

    for (i = 0; i < 4; i++)
        ch->armor[i]    += apply_ac(obj, obj->wear_loc,i);
    obj->wear_loc    = WEAR_NONE;

    if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
        strip_obj_affects(ch, obj, obj->pIndexData->affected);
    strip_obj_affects(ch, obj, obj->affected);

    if (obj->pIndexData->item_type == ITEM_LIGHT
        &&   obj->value[2] != 0
        &&   ch->in_room != NULL
        &&   ch->in_room->light > 0)
        --ch->in_room->light;

    oprog_call(OPROG_REMOVE, obj, ch, NULL);
}

/*
 * Count occurrences of an obj in a list.
 */
int count_obj_list(OBJ_INDEX_DATA *pObjIndex, OBJ_DATA *list)
{
    OBJ_DATA *obj;
    int nMatch;

    nMatch = 0;
    for (obj = list; obj != NULL; obj = obj->next_content)
    {
        if (obj->pIndexData == pObjIndex)
            nMatch++;
    }

    return nMatch;
}

/*
 * Move an obj out of a room.
 */
void obj_from_room(OBJ_DATA *obj)
{
    ROOM_INDEX_DATA *in_room;
    CHAR_DATA *ch;

    if ((in_room = obj->in_room) == NULL)
    {
        bug("obj_from_room: NULL.", 0);
        return;
    }

    for (ch = in_room->people; ch != NULL; ch = ch->next_in_room)
        if (ch->on == obj)
            ch->on = NULL;

    if (obj == in_room->contents)
        in_room->contents = obj->next_content;
    else
    {
        OBJ_DATA *prev;

        for (prev = in_room->contents; prev; prev = prev->next_content)
        {
            if (prev->next_content == obj)
            {
                prev->next_content = obj->next_content;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Obj_from_room: obj not found.", 0);
            return;
        }
    }

    obj->in_room      = NULL;
    obj->next_content = NULL;
    return;
}

/*
 * Move an obj into a room.
 */
void obj_to_room(OBJ_DATA *obj, ROOM_INDEX_DATA *pRoomIndex)
{
    if (pRoomIndex == NULL)
    {
        log_printf("BUG: trying to put obj into NULL room");
        return;
    }
    obj->next_content   = pRoomIndex->contents;
    pRoomIndex->contents    = obj;
    obj->in_room        = pRoomIndex;
    obj->carried_by     = NULL;
    obj->in_obj     = NULL;

    if (IS_WATER(pRoomIndex))
    {
        if (may_float(obj))
            obj->water_float = -1;
        else
            obj->water_float = floating_time(obj);
    }
}

/*
 * Move an object into an object.
 */
void obj_to_obj(OBJ_DATA *obj, OBJ_DATA *obj_to)
{
    clan_t *clan;

    if (obj == NULL || obj_to == NULL)
    {
        int vnum_obj = 0;
        int vnum_obj_to = 0;

        if (obj != NULL)
           vnum_obj = obj->pIndexData->vnum;

        if (obj_to != NULL)
            vnum_obj_to = obj_to->pIndexData->vnum;

    	log_printf("[***** BUG] obj_to_obj: %d to %d", vnum_obj, vnum_obj_to);
    	return;
    }

    if (obj == obj_to)
    {
        log_printf("obj_to_obj: obj == obj_to (vnum %d)",
                   obj->pIndexData->vnum);
        return;
    }

    obj->next_content   = obj_to->contains;
    obj_to->contains    = obj;
    obj->in_obj     = obj_to;
    obj->in_room        = NULL;
    obj->carried_by     = NULL;
    if (IS_SET(obj_to->pIndexData->extra_flags, ITEM_PIT))
        obj->cost = 0;

// Save place of clan item
    if (is_clan_altar(obj_to) != NULL
        && (clan = is_clan_item(obj)) != NULL)
    {
        clan->item_at = obj_to->pIndexData->vnum;
        save_clan(NULL, clan, FALSE);
    }

    for (; obj_to != NULL; obj_to = obj_to->in_obj)
    {
        if (obj_to->carried_by != NULL)
        {
/*      obj_to->carried_by->carry_number += get_obj_number(obj); */
            obj_to->carried_by->carry_weight += get_obj_weight(obj)
                                                * WEIGHT_MULT(obj_to) / 100;
        }
    }
}

/*
 * Move an object out of an object.
 */
void obj_from_obj(OBJ_DATA *obj)
{
    OBJ_DATA *obj_from;

    if ((obj_from = obj->in_obj) == NULL)
    {
        bug("Obj_from_obj: null obj_from.", 0);
        return;
    }

    if (obj == obj_from->contains)
        obj_from->contains = obj->next_content;
    else
    {
        OBJ_DATA *prev;

        for (prev = obj_from->contains; prev; prev = prev->next_content)
        {
            if (prev->next_content == obj)
            {
                prev->next_content = obj->next_content;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Obj_from_obj: obj not found.", 0);
            return;
        }
    }

    obj->next_content = NULL;
    obj->in_obj       = NULL;

    for (; obj_from != NULL; obj_from = obj_from->in_obj)
    {
        if (obj_from->carried_by != NULL)
/*      obj_from->carried_by->carry_number -= get_obj_number(obj); */
            obj_from->carried_by->carry_weight -= get_obj_weight(obj) * WEIGHT_MULT(obj_from) / 100;
    }
}

/*
 * Extract an obj from the world.
 */
void extract_obj_raw(OBJ_DATA *obj, int flags)
{
    clan_t *clan;
    int cn;
    OBJ_DATA *obj_content;
    OBJ_DATA *obj_next;

    if (!obj)
        return; // special for lamers ! :)

    if (obj->extracted)
    {
        log_printf("extract_obj_raw: %s, vnum %d: already extracted",
                   obj->name, obj->pIndexData->vnum);
        return;
    }
    else
        obj->extracted = TRUE;

    for (cn = 0; cn < clans.nused; cn++)
    {
        clan = CLAN(cn);
        if (clan->obj_ptr == obj) clan->obj_ptr = NULL;
    }
    for (obj_content = obj->contains; obj_content; obj_content = obj_next)
    {
        obj_next = obj_content->next_content;

        if (!IS_SET(flags, X_F_NORECURSE))
        {
            extract_obj_raw(obj_content, flags);
            continue;
        }

        obj_from_obj(obj_content);
        if (obj->in_room)
            obj_to_room(obj_content, obj->in_room);
        else if (obj->carried_by)
            obj_to_char(obj_content, obj->carried_by);
        else if (obj->in_obj)
            obj_to_obj(obj_content, obj->in_obj);
        else
            extract_obj(obj_content);
    }

    if (obj->in_room)
        obj_from_room(obj);
    else if (obj->carried_by)
        obj_from_char(obj);
    else if (obj->in_obj)
        obj_from_obj(obj);

    if (obj->pIndexData->vnum == OBJ_VNUM_MAGIC_JAR)
    {
        CHAR_DATA *wch;

        for (wch = char_list; wch != NULL ; wch = wch->next)
        {
            if (IS_NPC(wch)) continue;
            if (is_name(wch->name, obj->name))
            {
                REMOVE_BIT(wch->plr_flags, PLR_NOSOUL);
                char_act("Now you catch your spirit.", wch);
                break;
            }
        }
    }

    if (object_list == obj)
        object_list = obj->next;
    else
    {
        OBJ_DATA *prev;

        for (prev = object_list; prev != NULL; prev = prev->next)
        {
            if (prev->next == obj)
            {
                prev->next = obj->next;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Extract_obj: obj %d not found.", obj->pIndexData->vnum);
            return;
        }
    }

    if (!IS_SET(flags, X_F_NOCOUNT))
        --obj->pIndexData->count;
    free_obj(obj);
}


void extract_char(CHAR_DATA *ch, bool fPull)
{
    extract_char_org(ch, fPull, TRUE);
    return;
}


void extract_char_nocount(CHAR_DATA *ch, bool fPull)
{
    extract_char_org(ch, fPull, FALSE);
    return;
}


/*
 * Extract a char from the world.
 */
void extract_char_org(CHAR_DATA *ch, bool fPull, bool Count)
{
    CHAR_DATA *wch;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;
    OBJ_DATA *wield;
    char buf[MAX_STRING_LENGTH];

    if (fPull) /* only for total extractions should it check */
    {
        if (ch->extracted)  /* if the char has already been extracted once */
        {
            snprintf(buf, sizeof(buf), "Warning! Extraction of %s.", ch->name);
            bug(buf, 0);
            return; /* if it's already been extracted, something bad is going on */
        }
        else
        {
            ch->extracted = TRUE;  /* if it hasn't been extracted yet, now
                                    * it's being extracted. */
        }
    }

    if (ch->in_room == NULL)
    {
        bug("Extract_char: NULL.", 0);
        return;
    }

    nuke_pets(ch);
    ch->pet = NULL; /* just in case */

    if (fPull)
        die_follower(ch);

    stop_fighting(ch, TRUE);

    if ((wield = get_eq_char(ch, WEAR_WIELD)) != NULL)
        unequip_char(ch, wield);

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;

        if (IS_OBJ_STAT(obj,ITEM_NOLOOT))
            continue;

        if (Count)
            extract_obj(obj);
        else
            extract_obj_nocount(obj);
    }

    char_from_room(ch);

    if (!fPull)
    {
        char_to_room(ch, get_altar(ch)->room);
        return;
    }

    if (IS_NPC(ch))
        --ch->pIndexData->count;

    if (ch->desc != NULL && ch->desc->original != NULL)
    {
        do_resoul(ch, str_empty);
        do_return(ch, str_empty);
        ch->desc = NULL;
    }

    for (wch = char_list; wch != NULL; wch = wch->next)
    {
        if (wch->reply == ch)
            wch->reply = NULL;
        if (wch->repeat == ch)
            wch->repeat = NULL;

        if (ch->mprog_target == wch)
            wch->mprog_target = NULL;
        remove_contact(wch, ch);
    }


    if (ch == char_list)
        char_list = ch->next;
    else
    {
        CHAR_DATA *prev;

        for (prev = char_list; prev != NULL; prev = prev->next)
        {
            if (prev->next == ch)
            {
                prev->next = ch->next;
                break;
            }
        }

        if (prev == NULL)
        {
            bug("Extract_char: char not found.", 0);
            return;
        }
    }

    if (ch->desc != NULL)
        ch->desc->character = NULL;
    free_char(ch);
}

/*
 * Find some object with a given index data.
 * Used by area-reset 'P' command.
 */
OBJ_DATA *get_obj_type(OBJ_INDEX_DATA *pObjIndex)
{
    OBJ_DATA *obj;

    for (obj = object_list; obj != NULL; obj = obj->next)
    {
        if (obj->pIndexData == pObjIndex)
            return obj;
    }

    return NULL;
}




/* deduct cost from a character */

void deduct_cost(CHAR_DATA *ch, uint cost)
{
    uint silver = 0, gold = 0;

    silver = UMIN(ch->silver, cost);

    if (silver < cost)
    {
        gold = ((cost - silver + 99) / 100);
        silver = cost - 100 * gold;
    }

    if (ch->gold < gold)
        ch->gold = gold;

    if (ch->silver < silver)
        ch->silver = silver;

    ch->gold -= gold;
    ch->silver -= silver;
}

/*
 * Create a 'money' obj.
 */
OBJ_DATA *create_money(int gold, int silver)
{
    OBJ_DATA *obj;

    if (gold < 0 || silver < 0 || (gold == 0 && silver == 0))
    {
        bug("Create_money: zero or negative money.",UMIN(gold,silver));
        gold = UMAX(1,gold);
        silver = UMAX(1,silver);
    }

    if (gold == 0 && silver == 1)
    {
        obj = create_obj(get_obj_index(OBJ_VNUM_SILVER_ONE), 0);
    }
    else if (gold == 1 && silver == 0)
    {
        obj = create_obj(get_obj_index(OBJ_VNUM_GOLD_ONE), 0);
    }
    else if (silver == 0)
    {
        obj = create_obj(get_obj_index(OBJ_VNUM_GOLD_SOME), 0);
        obj->short_descr = mlstr_printf(obj->pIndexData->short_descr, gold);
        obj->value[1]           = gold;
        obj->cost               = gold;
        obj->weight     = gold/5;
    }
    else if (gold == 0)
    {
        obj = create_obj(get_obj_index(OBJ_VNUM_SILVER_SOME), 0);
        obj->short_descr = mlstr_printf(obj->pIndexData->short_descr, silver);
        obj->value[0]           = silver;
        obj->cost               = silver;
        obj->weight     = silver/20;
    }

    else
    {
        obj = create_obj(get_obj_index(OBJ_VNUM_COINS), 0);
        obj->short_descr = mlstr_printf(obj->pIndexData->short_descr, silver, gold);
        obj->value[0]       = silver;
        obj->value[1]       = gold;
        obj->cost       = 100 * gold + silver;
        obj->weight     = gold / 5 + silver / 20;
    }

    return obj;
}


/*
 * Return # of objects which an object counts as.
 * Thanks to Tony Chamberlain for the correct recursive code here.
 */
int get_obj_number(OBJ_DATA *obj)
{
    int number;
/*
    if (obj->pIndexData->item_type == ITEM_CONTAINER || obj->pIndexData->item_type == ITEM_MONEY
    ||  obj->pIndexData->item_type == ITEM_GEM || obj->pIndexData->item_type == ITEM_JEWELRY)
        number = 0;
*/
    if (obj->pIndexData->item_type == ITEM_MONEY)
        number = 0;
    else
        number = 1;

/*
    for (obj = obj->contains; obj != NULL; obj = obj->next_content)
        number += get_obj_number(obj);
*/
    return number;
}

int get_obj_realnumber(OBJ_DATA *obj)
{
    int number = 1;

    for (obj = obj->contains; obj != NULL; obj = obj->next_content)
        number += get_obj_number(obj);

    return number;
}

/*
 * Return weight of an object, including weight of contents.
 */
int get_obj_weight(OBJ_DATA *obj)
{
    int weight;
    OBJ_DATA *tobj;

    weight = obj->weight;
    for (tobj = obj->contains; tobj != NULL; tobj = tobj->next_content)
        weight += get_obj_weight(tobj) * WEIGHT_MULT(obj) / 100;

    return weight;
}

int get_true_weight(OBJ_DATA *obj)
{
    int weight;

    weight = obj->weight;
    for (obj = obj->contains; obj != NULL; obj = obj->next_content)
        weight += get_obj_weight(obj);

    return weight;
}

/*
 * True if room is dark.
 */
bool room_is_dark(CHAR_DATA *ch)
{
    ROOM_INDEX_DATA * pRoomIndex = ch->in_room;

    if (!pRoomIndex)
        return TRUE;

    if (!IS_NPC(ch) && IS_SET(ch->plr_flags, PLR_HOLYLIGHT))
        return FALSE;

    if (IS_SET(ch->form, FORM_VAMPIRE | FORM_UNDEAD | FORM_DEMON))
        return FALSE;

    if (IS_AFFECTED(ch, AFF_DARK_VISION))
        return FALSE;

    return room_dark(pRoomIndex);
}

bool room_dark(ROOM_INDEX_DATA *pRoomIndex)
{
    if (pRoomIndex->light > 0)
        return FALSE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_DARK))
        return TRUE;

    if (pRoomIndex->sector_type == SECT_INSIDE
        ||   pRoomIndex->sector_type == SECT_CITY)
        return FALSE;

    if (weather_info.sunlight == SUN_SET
        || weather_info.sunlight == SUN_DARK)
        return TRUE;

    return FALSE;
}


bool is_room_owner(CHAR_DATA *ch, ROOM_INDEX_DATA *room)
{
    if (room->owner == NULL || room->owner[0] == '\0')
        return FALSE;

    return is_name(ch->name,room->owner);
}

/*
 * True if room is private.
 */
bool room_is_private(ROOM_INDEX_DATA *pRoomIndex)
{
    CHAR_DATA *rch;
    int count;

/*
    if (pRoomIndex->owner != NULL && pRoomIndex->owner[0] != '\0')
        return TRUE;
*/
    count = 0;
    for (rch = pRoomIndex->people; rch != NULL; rch = rch->next_in_room)
        count++;

    if (IS_SET(pRoomIndex->room_flags, ROOM_PRIVATE)  && count >= 2)
        return TRUE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_SOLITARY) && count >= 1)
        return TRUE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_IMP_ONLY))
        return TRUE;

    return FALSE;
}

/* visibility on a room -- for entering and exits */
bool can_see_room(CHAR_DATA *ch, ROOM_INDEX_DATA *pRoomIndex)
{
    if (IS_SET(pRoomIndex->room_flags, ROOM_IMP_ONLY)
        &&  !IS_TRUSTED(ch, IMPLEMENTOR))
        return FALSE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_GODS_ONLY)
        &&  !IS_TRUSTED(ch, GOD))
        return FALSE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_HEROES_ONLY)
        &&  !IS_TRUSTED(ch, HERO))
        return FALSE;

    if (IS_SET(pRoomIndex->room_flags, ROOM_NEWBIES_ONLY)
        &&  ch->level > LEVEL_NEWBIE && !IS_IMMORTAL(ch))
        return FALSE;

    return TRUE;
}

/*
 * True if char can see victim.
 */
bool can_see(CHAR_DATA *ch, CHAR_DATA *victim)
{
/* RT changed so that WIZ_INVIS has levels */

    clan_t *clan = NULL;

    if (ch == victim)
        return TRUE;

    if (ch == NULL || victim == NULL)
    {
        if (ch == NULL)
            log_printf("ch:NULL");
        else
            log_printf(ch->name);

        if (victim == NULL)
            log_printf("victim:NULL");
        else
            log_printf(victim->name);

        dump_to_scr(">>> >>> >>> CAN SEE ERROR <<< <<< <<<\n");
        return FALSE;
    }

    if (!IS_NPC(ch)
    && IS_SET(ch->plr_flags, PLR_HOLYLIGHT)
    && victim->invis_level < 93
    && victim->incog_level < 93)
        return TRUE;

    if (!IS_NPC(victim)
    && !IS_TRUSTED(ch, victim->invis_level))
        return FALSE;

    if (IS_CLAN_GUARD(ch)) return TRUE;

    if (!IS_NPC(ch) && !IS_NPC(victim))
    {
          if (ch->pcdata)
                if (ch->pcdata->marryed == str_dup(victim->name))
                    return TRUE;
    }

    if (!IS_TRUSTED(ch, victim->incog_level)
    && ch->in_room != victim->in_room)
        return FALSE;

    if (!IS_NPC(ch) && IS_SET(ch->plr_flags, PLR_HOLYLIGHT)
    && IS_TRUSTED(ch, victim->invis_level))
        return TRUE;

    if (IS_NPC(victim) && !IS_TRUSTED(ch, victim->invis_level))
    {
        AREA_DATA *pArea = area_vnum_lookup(victim->pIndexData->vnum);
        if (pArea == NULL
        || !IS_BUILDER(ch, pArea))
            return FALSE;
    }

    if (IS_AFFECTED(ch, AFF_BLIND) && !IS_UNDEAD(ch))
        return FALSE;

    // visibility for clanmembers...
    clan = CLAN(ch->clan);
    if (IS_SET(clan->flags, CLAN_CAN_SEE)
    && (ch->clan == victim->clan
    || (CLAN(ch->clan)->diplomacy[victim->clan] == DIP_ALLY
    && CLAN(victim->clan)->diplomacy[ch->clan] == DIP_ALLY)))
        return TRUE;

    if (IS_AFFECTED(victim, AFF_TRAP)
    && !IS_AFFECTED(ch, AFF_DETECT_TRAP))
        return FALSE;

    if (ch->in_room == NULL)
        return FALSE;

    if (!IS_UNDEAD(ch))
    {
        if (room_is_dark(ch)
        && !(IS_AFFECTED(ch, AFF_INFRARED)))
            return FALSE;

        if (IS_AFFECTED(victim, AFF_INVIS)
        && !IS_AFFECTED(ch, AFF_DETECT_INVIS))
            return FALSE;

        if (IS_AFFECTED(victim, AFF_IMP_INVIS)
        && !IS_AFFECTED(ch, AFF_DETECT_IMP_INVIS))
            return FALSE;

        if (IS_AFFECTED(victim,AFF_CAMOUFLAGE)
        && !IS_AFFECTED(ch,AFF_ACUTE_VISION))
            return FALSE;

        if (IS_AFFECTED(victim, AFF_HIDE)
        && !IS_AFFECTED(ch, AFF_DETECT_HIDDEN)
        && victim->fighting == NULL)
            return FALSE;
    }

    if (IS_AFFECTED(victim, AFF_FADE)
    && !IS_AFFECTED(ch, AFF_DETECT_FADE)
    && victim->fighting == NULL)
        return FALSE;

    if (IS_AFFECTED(victim, AFF_EARTHFADE))
        return FALSE;

    return TRUE;
}

/*
 * True if char can see obj.
 */
bool can_see_obj(CHAR_DATA *ch, OBJ_DATA *obj)
{
    CHAR_DATA *victim;
    if (!IS_NPC(ch) && IS_SET(ch->plr_flags, PLR_HOLYLIGHT))
        return TRUE;

    if (IS_SET(obj->extra_flags, ITEM_VIS_DEATH))
        return FALSE;

    if (IS_UNDEAD(ch))
        return TRUE;

    if (IS_AFFECTED(ch, AFF_BLIND) &&
        obj->pIndexData && obj->pIndexData->item_type != ITEM_POTION)
        return FALSE;

    if (obj->pIndexData && obj->pIndexData->item_type == ITEM_LIGHT &&
        obj->value[2] != 0)
        return TRUE;

    if (IS_SET(obj->extra_flags, ITEM_INVIS)
        &&   !IS_AFFECTED(ch, AFF_DETECT_INVIS))
        return FALSE;

    if (IS_OBJ_STAT(obj, ITEM_GLOW))
        return TRUE;

    if (room_is_dark(ch) && !(IS_AFFECTED(ch, AFF_INFRARED)))
        return FALSE;

    if ((victim = obj->carried_by) != NULL)
    {
       if (obj->wear_loc == WEAR_CLANMARK)
       {
           clan_t *clan;
           if ((clan = clan_lookup(victim->clan)) != NULL)
       {
           if (IS_SET(clan->flags, CLAN_HIDDEN)
                   && ch->clan != victim->clan)
          return FALSE;
       }
       }
       if (obj->wear_loc == WEAR_TATTOO)
       {
           religion_t *religion;
           if ((religion = religion_lookup(victim->religion)) != NULL)
       {
           if (IS_SET(religion->flags, RELIGION_HIDDEN)
                   && ch->religion != victim->religion)
          return FALSE;
       }
       }
    }

    return TRUE;
}

/*
 * True if char can drop obj.
 */
bool can_drop_obj(CHAR_DATA *ch, OBJ_DATA *obj)
{
    if (!IS_SET(obj->extra_flags, ITEM_NODROP))
        return TRUE;

    if (IS_IMMORTAL(ch))
        return TRUE;

    return FALSE;
}

int isn_dark_safe(CHAR_DATA *ch)
{
    CHAR_DATA *rch;
    OBJ_DATA *light;
    int light_exist;

    if (!IS_SET(ch->form, FORM_VAMPIRE)
        ||  IS_SET(ch->in_room->room_flags, ROOM_DARK))
        return 0;

    if (weather_info.sunlight == SUN_LIGHT
        ||  weather_info.sunlight == SUN_RISE)
        return 2;

    light_exist = 0;
    for (rch = ch->in_room->people; rch; rch = rch->next_in_room)
    {
        if ((light = get_eq_char(rch, WEAR_LIGHT))
            && (light->value[2] != 0))
        {
            light_exist = 1;
            break;
        }
    }

    return light_exist;
}

int count_charmed(CHAR_DATA *ch)
{
    CHAR_DATA *gch;
    int count = 0;

    if (!war_can_charm (ch, NULL))
    {
        return 666;
    }

    for (gch = char_list; gch != NULL; gch = gch->next)
    {
        if (IS_AFFECTED(gch, AFF_CHARM) && gch->master == ch)
            count++;
    }

    if (count >= MAX_CHARM(ch) + get_skill(ch, gsn_mastering_charm) / 100)
    {
        char_act("You are already controlling as many charmed mobs as you can!", ch);
        return count;
    }
    return 0;
}

/*
 * add_mind - remember 'str' in mind buffer of 'ch'
 *        remember the place to return in mind buffer
 */
void add_mind(CHAR_DATA *ch, const char *str)
{
    if (!IS_NPC(ch) || ch->in_room == NULL)
        return;

    if (ch->in_mind == NULL)
        /* remember a place to return */
        ch->in_mind = str_printf("%d", ch->in_room->vnum);

    if (!is_name(str, ch->in_mind))
    {
        const char *p = ch->in_mind;
        ch->in_mind = str_printf("%s %s", ch->in_mind, str);
        free_string(p);
    }
}

/*
 * remove_mind - remove 'str' from mind buffer of 'ch'
 *       if it was the last revenge - return home
 */
void remove_mind(CHAR_DATA *ch, const char *str)
{
    char buf[MAX_STRING_LENGTH];
    char buff[MAX_STRING_LENGTH];
    char arg[MAX_INPUT_LENGTH];
    const char *mind = ch->in_mind;

    if (!IS_NPC(ch) || ch->in_room == NULL
        ||  mind == NULL || !is_name(str, mind)) return;

    buf[0] = '\0';
    do
    {
        mind = one_argument(mind, arg, sizeof(arg));
        if (!is_name(str,arg))
        {
            if (buf[0] == '\0')
                strnzcpy(buff, sizeof(buff), arg);
            else
                snprintf(buff, sizeof(buff), "%s %s", buf, arg);
            strnzcpy(buf, sizeof(buf),buff);
        }
    }
    while (mind[0] != '\0');

    free_string(ch->in_mind);
    if (is_number(buf))
    {
        do_say(ch, " ,  - !");
        back_home(ch);
        ch->in_mind = NULL;
    }
    else
        ch->in_mind = str_dup(buf);
}

void back_home(CHAR_DATA *ch)
{
    ROOM_INDEX_DATA *location;
    char arg[MAX_INPUT_LENGTH];

    if (!IS_NPC(ch) || ch->in_mind == NULL)
        return;

    one_argument(ch->in_mind, arg, sizeof(arg));
    if ((location = find_location(ch, arg)) == NULL)
    {
        log("back_home: reset place not found");
        return;
    }

    if (ch->fighting == NULL && location != ch->in_room)
    {
        act("$n   .",ch,NULL,NULL,TO_ROOM);
        char_from_room(ch);
        char_to_room(ch, location);
        act("$n   .",ch,NULL,NULL,TO_ROOM);
    }
}



void path_to_track(CHAR_DATA *ch, CHAR_DATA *victim, int door)
{
    ROOM_INDEX_DATA *temp;
    EXIT_DATA *pExit;
    int opdoor;
    int range = 0;

    SET_FIGHT_TIME(ch);
    if (!IS_NPC(victim))
        SET_FIGHT_TIME(victim);

    if (IS_NPC(victim) && victim->position != POS_DEAD)
    {
        victim->last_fought = ch;

        if ((opdoor = opposite_door(door)) == -1)
        {
            bug("In path_to_track wrong door: %d",door);
            return;
        }
        temp = ch->in_room;
        while (1)
        {
            range++;
            if (victim->in_room == temp) break;
            if ((pExit = temp->exit[ door ]) == NULL
                || (temp = pExit->to_room.r) == NULL)
            {
                bug("In path_to_track: couldn't calculate range %d",range);
                return;
            }
            if (range > 100)
            {
                bug("In path_to_track: range exceeded 100",0);
                return;
            }
        }

        temp = victim->in_room;
        while (--range > 0)
        {
            room_record(ch->name,temp, opdoor);
            if ((pExit = temp->exit[opdoor]) == NULL
                || (temp = pExit->to_room.r) == NULL)
            {
                log_printf("[*****] Path to track: Range: %d Room: %d opdoor:%d",
                           range,temp->vnum,opdoor);
                return;
            }
        }
        do_track(victim,str_empty);
    }
}

__inline int pk_range (int level)
{
    return UMAX (4, level/10 + 2) ;
}

bool in_PK (CHAR_DATA * ch, CHAR_DATA* victim)
{
    int is_wanted = (IS_SET (victim->plr_flags, PLR_WANTED)) ? 2 : 1 ;

    if ((IS_NPC (ch) && ch->desc == NULL) || IS_NPC (victim)) return TRUE ;

    if (ch->in_war && victim->in_war && (ch->in_war == victim->in_war)
       && (ch->war_status == PS_ALIVE))
        return TRUE;

    // switched?
    if (IS_NPC (ch) && ch->desc != NULL)
    {
        if ((ch->desc->original->level >= (victim->level + is_wanted * pk_range (ch->desc->original->level)) ||
             ch->desc->original->level <= (victim->level - pk_range (ch->desc->original->level)))
            &&
            (victim->level >= (ch->desc->original->level + pk_range (victim->level)) ||
             victim->level <= (ch->desc->original->level - is_wanted * pk_range (victim->level)))) return FALSE ;

        return TRUE;
    }

    // NOPK
    if (victim->level < PK_MIN_LEVEL || ch->level < PK_MIN_LEVEL ||
        IS_SET (ch->comm, PLR_PEACE)  || IS_SET (victim->comm, PLR_PEACE)) return FALSE ;

    // victim with wanted flag adjustment
    if (ch != victim && !IS_IMMORTAL(ch)
        &&
        (ch->level >= (victim->level + is_wanted * pk_range (ch->level)) ||
         ch->level <= (victim->level - pk_range (ch->level)))
        &&
        (victim->level >= (ch->level + pk_range (victim->level)) ||
         victim->level <= (ch->level - is_wanted * pk_range (victim->level)))) return FALSE ;

    return TRUE ;
}

bool in_PK_level (int level, CHAR_DATA* victim)
{
    int is_wanted = (IS_SET (victim->plr_flags, PLR_WANTED)) ? 2 : 1 ;

    if (IS_NPC (victim)) return TRUE ;

    // NOPK
    if (victim->level < PK_MIN_LEVEL || level < PK_MIN_LEVEL ||
        IS_SET(victim->comm, PLR_PEACE)) return FALSE ;

    // victim with wanted flag adjustment
    if ((level >= (victim->level + is_wanted * pk_range (level)) ||
         level <= (victim->level - pk_range (level)))
        &&
        (victim->level >= (level + pk_range (victim->level)) ||
         victim->level <= (level - is_wanted * pk_range (victim->level)))) return FALSE ;

    return TRUE ;
}

bool can_gate_room(CHAR_DATA* ch, ROOM_INDEX_DATA *room)
{
    if (room == NULL
    || ch->fighting != NULL
    || !can_see_room (ch, room)
    || IS_SET (ch->in_room->room_flags, ROOM_SAFE | ROOM_NORECALL | ROOM_PEACE | ROOM_NOSUMMON)
    || IS_SET (room->room_flags, ROOM_SAFE | ROOM_NORECALL | ROOM_PEACE | ROOM_NOSUMMON)
    || IS_SET (room->area->flags, AREA_UNDER_CONSTRUCTION)
    || room_is_private (room)
    || IS_RAFFECTED (ch->in_room, RAFF_ESPIRIT)
    || IS_RAFFECTED(ch->in_room, RAFF_PREVENT)
    || IS_RAFFECTED(room, RAFF_PREVENT))
        return FALSE ;

    // GQ stuff
    if (IS_SET (room->area->flags, AREA_GQACTIVE))
    {
        GQ_QUESTPLR* plr = gq_findplr (ch) ;
        if (plr != NULL && plr->killed_total >= 0)
            return FALSE ;
    }

    if (guild_check(ch, room) < 0)
        return FALSE ;

    return TRUE ;
}

bool can_gate(CHAR_DATA* ch, CHAR_DATA* victim)
{
    if (victim == NULL
    || !can_gate_room(ch, victim->in_room)
    || is_immune(victim, DAM_SUMMON))
        return FALSE;

    if (!check_war_move (ch, victim->in_room, FALSE))
        return FALSE;

    if (ch->in_room == victim->in_room)
    {
        char_act ("You are in the same place with your target. Just look.", ch);
        return FALSE;
    }

	if (check_altar_immobility (ch, ch->in_room) || check_altar_immobility (ch, victim->in_room))
    {
        char_act ("Altar of {rimmobility{x blocks your transportation.", ch);
        return FALSE;
    }

    if (IS_NPC (victim))
        return TRUE;

    if (((!in_PK(ch, victim)
    || ch->in_room->area != victim->in_room->area)
    && IS_SET(victim->plr_flags, PLR_NOSUMMON)))
        return FALSE;

    if (victim->level > LEVEL_HERO)
        return FALSE;

    return TRUE;
}

const char *PERS2(CHAR_DATA *ch, CHAR_DATA *looker, flag32_t flags)
{
    int lang;
    memento_t * memento;

    lang = looker == NULL ? 0 : looker->lang;

    if (ch == NULL)
        return GETMSG("nobody", lang);

    if (looker == NULL)
        return ch->name;

    if (looker && (IS_NPC(looker) || !IS_SET(looker->plr_flags, PLR_HOLYLIGHT))
        && is_affected(ch, gsn_doppelganger))
        ch = ch->doppel;

    if (looker == NULL || can_see(looker, ch))
    {
        if (IS_NPC(ch))
        {
            const char *descr;

            if (IS_SET(flags, ACT_FORMSH))
            {
                if ((memento = lookup_remembered_mob(looker, ch->pIndexData)) != NULL)
                    return format_short(ch->short_descr, memento->name, looker, flags | ACT_FORMSH_ALWAYS);

                return format_short(ch->short_descr, ch->name, looker, flags);
            }

            descr = mlstr_val(ch->short_descr, lang);
            if (!IS_SET(flags, ACT_NOFIXSH))
                return fix_short(descr);

            return descr;
        }
        else if (looker != NULL && IS_IMMORTAL(looker))
        {
        }
        else if (IS_SET(ch->form, FORM_VAMPIRE))
            return word_form(GETMSG("an ugly creature", lang),
                             ch->sex, lang, RULES_GENDER);
        else if (IS_SET(ch->form, FORM_MECH))
            return word_form(GETMSG("an enormous mech", lang),
                             ch->sex, lang, RULES_GENDER);
        else if (IS_SET(ch->form, FORM_DEMON_FIRE))
            return word_form(GETMSG("a demon of fire", lang),
                             ch->sex, lang, RULES_GENDER);
        else if (IS_SET(ch->form, FORM_DEMON_WATER))
            return word_form(GETMSG("a demon of water", lang),
                             ch->sex, lang, RULES_GENDER);
        else if (IS_SET(ch->form, FORM_DEMON_EARTH))
            return word_form(GETMSG("a demon of earth", lang),
                             ch->sex, lang, RULES_GENDER);
        else if (IS_SET(ch->form, FORM_DEMON_AIR))
            return word_form(GETMSG("a demon of air", lang),
                             ch->sex, lang, RULES_GENDER);
        if (ch->imm_name)
            return ch->imm_name;
        if ((memento = lookup_remembered_pc(looker, ch->id)) != NULL)
            return memento->name;
        // Welesh : for rusnames
        if (IS_SET(looker->comm2, COMM2_RUSNAMES) && !IS_NPC(ch) && ch->pcdata != NULL)
        {
            if (ch->pcdata->rusnames[0] != NULL)
                return ch->pcdata->rusnames[0];
            else 
                return ch->name;
        }
        else
            return ch->name;
    }

    if (IS_IMMORTAL(ch))
        return word_form(GETMSG("an immortal", lang),
                         ch->sex, lang, RULES_GENDER);

    return GETMSG("someone", lang);
}

void format_obj(BUFFER *output, OBJ_DATA *obj, bool russpells)
{
    char buf[MAX_STRING_LENGTH];

    if (obj->size != 0)
        buf_printf(output,
                   "Object '%s' is type %s.\n"
                   "Extra flags %s.\n"
                   "Weight is %d, value is %d, level is %d.\n"
                   "Size is %d.\n"
                   "Wear flags %s.\n",
                   obj->name,
                   flag_string(item_types, obj->pIndexData->item_type),
                   flag_string(extra_flags, obj->extra_flags & ~ITEM_ENCHANTED),
                   obj->weight / 10,
                   obj->cost,
                   obj->level,
                   obj->size,
                   flag_string(wear_flags, obj->wear_flags & ~ITEM_WEAR_NOSAC));
    else
        buf_printf(output,
                   "Object '%s' is type %s.\n"
                   "Extra flags %s.\n"
                   "Weight is %d, value is %d, level is %d.\n"
                   "Wear flags %s.\n",
                   obj->name,
                   flag_string(item_types, obj->pIndexData->item_type),
                   flag_string(extra_flags, obj->extra_flags & ~ITEM_ENCHANTED),
                   obj->weight / 10,
                   obj->cost,
                   obj->level,
                   flag_string(wear_flags, obj->wear_flags & ~ITEM_WEAR_NOSAC));

    if (obj->pIndexData->limit != -1)
        buf_printf(output, "This equipment has been LIMITED by number %d.\n", obj->pIndexData->limit);

    switch (obj->pIndexData->item_type)
    {
    case ITEM_SCROLL:
    case ITEM_POTION:
    case ITEM_PILL:
        buf_printf(output, "Level %d spells of:", obj->value[0]);

        if (obj->value[1] >= 0) 
        {
            if (russpells)
                antisubstitute_skill_alias(skill_name(obj->value[1]), buf,  sizeof(buf));
            else
                strnzcpy(buf, sizeof(buf), skill_name(obj->value[1]));
            buf_printf(output, " '%s'", buf);
        }
        if (obj->value[2] >= 0) 
        {
            if (russpells)
                antisubstitute_skill_alias(skill_name(obj->value[2]), buf,  sizeof(buf));
            else
                strnzcpy(buf, sizeof(buf), skill_name(obj->value[2]));
            buf_printf(output, " '%s'", buf);
        }
        if (obj->value[3] >= 0) 
        {
            if (russpells)
                antisubstitute_skill_alias(skill_name(obj->value[3]), buf,  sizeof(buf));
            else
                strnzcpy(buf, sizeof(buf), skill_name(obj->value[3]));
            buf_printf(output, " '%s'", buf);
        }
        if (obj->value[4] >= 0) 
        {
            if (russpells)
                antisubstitute_skill_alias(skill_name(obj->value[4]), buf,  sizeof(buf));
            else
                strnzcpy(buf, sizeof(buf), skill_name(obj->value[4]));
            buf_printf(output, " '%s'", buf);
        }

        buf_add(output, ".\n");
        break;

    case ITEM_WAND:
    case ITEM_STAFF:
        buf_printf(output, "Has %d charges of level %d",
                   obj->value[2], obj->value[0]);

        if (obj->value[3] >= 0) 
        {
            if (russpells)
                antisubstitute_skill_alias(skill_name(obj->value[3]), buf,  sizeof(buf));
            else
                strnzcpy(buf, sizeof(buf), skill_name(obj->value[3]));
            buf_printf(output, " '%s'", buf);
        }

        buf_add(output, ".\n");
        break;

    case ITEM_DRINK_CON:
        if (obj->value[1] <= 0 && obj->value[0] != -1)
              buf_printf(output, "It empty.\n");
        else
              buf_printf(output, "It holds %s-colored %s.\n",
                   //liq_table[obj->value[2]].liq_color,
                   //liq_table[obj->value[2]].liq_name);
                        liquid_color(obj->value[2]),
                        liquid_name(obj->value[2]));

        break;

    case ITEM_CONTAINER:
        buf_printf(output, "Capacity: %d#  Maximum weight: %d#  flags: %s\n",
                   obj->value[0],
                   obj->value[3],
                   flag_string(cont_flags, obj->value[1]));
        if (obj->value[4] != 100)
            buf_printf(output, "Weight multiplier: %d%%\n", obj->value[4]);
        break;

    case ITEM_WEAPON:
        buf_printf(output, "Weapon type is %s.\n",
                   flag_string(weapon_class, obj->value[0]));
        buf_printf(output, "Damage is %dd%d (average %d).\n",
                   obj->value[1],obj->value[2],
                   (1 + obj->value[2]) * obj->value[1] / 2);
        if (obj->value[4])
            buf_printf(output, "Weapons flags: %s\n",
                       flag_string(weapon_type2, obj->value[4]));
        break;

    case ITEM_ARMOR:
        buf_printf(output, "Armor class is %d pierce, "
                   "%d bash, %d slash, and %d vs. exotic.\n",
                   obj->value[0], obj->value[1],
                   obj->value[2], obj->value[3]);
        break;

    case ITEM_PAINT:
        buf_printf(output, "Color: %s\n", flag_string(paint_color_int_flags, obj->value[0]));
        if (obj->value[1] != -1)
            buf_printf(output, "Portions: %d\n", obj->value[1]);
        else
            buf_printf(output, "Portions: infinite\n", obj->value[1]);
        buf_printf(output, "Power: %d\n", obj->value[2]);
        break;

    case ITEM_LEARN_BOOK:
        buf_printf(output, "From this book you can learn %s for %d%% up to %d%%.\n", 
            skill_name(obj->value[2]), obj->value[0], obj->value[1]);
        break;
    }
}

void format_obj_affects(BUFFER *output, AFFECT_DATA *paf, int flags)
{
    for (; paf; paf = paf->next)
    {
        where_t *w;

        if (paf->location != APPLY_NONE && paf->modifier)
        {
            buf_printf(output, "Affects %s by %d",
                       flag_string(apply_flags, paf->location),
                       paf->modifier);
            if (!IS_SET(flags, FOA_F_NODURATION)
                &&  paf->duration > -1)
                buf_printf(output, " for %d hours",
                           paf->duration);
            buf_add(output, ".\n");
        }

        if (IS_SET(flags, FOA_F_NOAFFECTS))
            continue;

        if ((w = where_lookup(paf->where)) && paf->bitvector)
        {
            buf_add(output, "Adds ");
            buf_printf(output, w->format,
                       flag_string(w->table, paf->bitvector));
            if (w->where == TO_RESIST)
                buf_printf(output, " by %d%%", paf->modifier);
            buf_add(output, ".\n");
        }
    }
}

int get_wear_level(CHAR_DATA *ch, OBJ_DATA *obj)
{
    int wear_level = ch->level;
    class_t *cl;

    if ((cl = class_lookup(ch->class)) == NULL)
        return wear_level;

    switch (obj->pIndexData->item_type)
    {
    case ITEM_POTION:
    case ITEM_PILL:
    case ITEM_WAND:
    case ITEM_STAFF:
    case ITEM_SCROLL:
        return wear_level;
    }

    if (!IS_SET(obj->pIndexData->extra_flags, ITEM_QUEST)
        &&  (obj->pIndexData->limit < 0 || obj->pIndexData->limit > 1))
        wear_level += pk_range(wear_level);


    if (obj->pIndexData->limit != 1)
    {
        if (is_class_mage(ch))
        {
            if (obj->pIndexData->item_type == ITEM_ARMOR)
                wear_level += 3;
        }
    if (is_class_warrior(ch))
    {
            if (obj->pIndexData->item_type == ITEM_WEAPON)
                wear_level += 3;
    }
    }

    return wear_level;
}

/*
 * Compute a saving throw.
 * Negative apply's make saving throw better.
 */

bool saves_spell(int level, CHAR_DATA* victim, int dam_type)
{
    int save;

    if (HAS_SKILL(victim, gsn_spellbane))
        save = get_skill(victim, gsn_spellbane) + (LVL(victim) - level) * 6;
    else
        save = (LVL(victim) - level) * 4 - victim->saving_throw;

    if (IS_AFFECTED(victim, AFF_BERSERK))
        save += 10;

    if (HAS_SKILL(victim, gsn_adv_magic_evasion))
    {
        save += (get_skill(victim, gsn_adv_magic_evasion) / 3);
        check_improve(victim, gsn_adv_magic_evasion, TRUE, 4);
    } 
    else
    {
        save += (get_skill(victim, gsn_magic_evasion) / 8);
        check_improve(victim, gsn_magic_evasion, TRUE, 4);
    }

    if (is_immune(victim, dam_type))
        return TRUE;

    resists_modify(&save, check_immune(victim, dam_type), FALSE);

    if (HAS_SKILL(victim, gsn_spellbane))
        save = URANGE(2, save, 80);
    else
        save = URANGE(2, save, 98);

    return number_percent() < save;
}

/* RT configuration smashed */

bool saves_dispel(int dis_level, int spell_level, int duration)
{
    int save;

    /* impossible to dispel permanent effects */
    if (duration == -2) return 1;
    if (duration == -1) spell_level += 5;

    save = 50 + (spell_level - dis_level) * 5;
    save = URANGE(5, save, 95);
    return number_percent() < save;
}

/* co-routine for dispel magic and cancellation */
void dispel_message(CHAR_DATA *victim, int sn, bool is_dispel)
{
    skill_t *sk;
    if ((sk = skill_lookup(sn)))
    {
        if (!IS_NULLSTR(sk->msg_off))
            char_act(sk->msg_off, victim);
        if (!IS_NULLSTR(sk->msg_off_other))
            act(sk->msg_off_other, victim, NULL, NULL, TO_ROOM);
        if (is_dispel)
            act("$n blinks as $gn{his} '$t' was dispelled.", victim, sk->name, NULL, TO_ROOM);
    }
}

bool check_dispel(int dis_level, CHAR_DATA *victim, int sn)
{
    AFFECT_DATA *af;

    if (is_affected(victim, sn))
    {
        for (af = victim->affected; af != NULL; af = af->next)
        {
            if (af->type == sn)
            {
                if (!saves_dispel(dis_level,af->level,af->duration))
                {
                    dispel_message(victim, sn, TRUE);
                    affect_strip(victim,sn);
                    return TRUE;
                }
                else
                    af->level--;
            }
        }
    }
    return FALSE;
}

bool check_blind_raw(CHAR_DATA *ch)
{
    if (!IS_NPC(ch) && IS_SET(ch->plr_flags, PLR_HOLYLIGHT))
        return TRUE;

    if (IS_AFFECTED(ch, AFF_BLIND))
        return FALSE;

    return TRUE;
}

bool check_blind(CHAR_DATA *ch)
{
    bool can_see = check_blind_raw(ch);

    if (!can_see)
        char_act("   !", ch);

    return can_see;
}

/*----------------------------------------------------------------------------
 * show affects stuff
 */

void show_name(CHAR_DATA *ch, BUFFER *output,
               AFFECT_DATA *paf, AFFECT_DATA *paf_last)
{
    skill_t * skill;
    char      buf[MAX_STRING_LENGTH];

    if (paf_last && paf->type == paf_last->type)
    {
        if (!ch || ch->level >= 20)
            buf_add(output, "                      ");
        return;
    }
    skill = skill_lookup(paf->type);

    if (skill != NULL)
    {
        if (ch && IS_SET(ch->comm2, COMM2_RUSSKILLS))
            antisubstitute_skill_alias(skill->name, buf, 15);
        else
            strnzcpy(buf, 15, skill->name);
    }

    buf_act(output, "$FL11{$t:} {c$FR15{$T}{x", ch, skill ? flag_string(skill_types, skill->type) : "none", skill ? buf : "none", NULL, TO_CHAR | ACT_TRANS | ACT_NOLF);
}

void show_duration(BUFFER *output, AFFECT_DATA *paf)
{
    if (paf->duration == -1 || paf->duration == -2)
        buf_act(output, ".", NULL, NULL, NULL, NULL, TO_CHAR);
    else
        buf_act(output, " {c$j{x $qj{}.", NULL, (const void *) paf->duration, NULL, NULL, TO_CHAR);
}

void show_loc_affect(CHAR_DATA *ch, BUFFER *output,
                     AFFECT_DATA *paf, AFFECT_DATA **ppaf)
{
    if (ch->level < 20)
    {
        show_name(ch, output, paf, *ppaf);
        if (*ppaf && (*ppaf)->type == paf->type)
            return;
        buf_add(output, "\n");
        *ppaf = paf;
        return;
    }

    if (paf->location > 0)
    {
        show_name(ch, output, paf, *ppaf);
        buf_printf(output, ": modifies {c%s{x by {c%d{x ",
                   flag_string(apply_flags, paf->location),
                   paf->modifier);
        show_duration(output, paf);
        *ppaf = paf;
    }
    else if (!paf->bitvector)
    {
        show_name(ch, output, paf, *ppaf);
        buf_add(output, ": ");
        show_duration(output, paf);
        *ppaf = paf;
    }
}

void show_bit_affect(BUFFER *output, AFFECT_DATA *paf, AFFECT_DATA **ppaf,
                     flag32_t where)
{
    char buf[MAX_STRING_LENGTH];
    where_t *w;

//  if (paf->where != where
    if ((w = where_lookup(paf->where)) == NULL
        ||  !paf->bitvector)
        return;

    show_name(NULL, output, paf, *ppaf);
    snprintf(buf, sizeof(buf), ":  %s ", w->format);
    buf_printf(output, buf, flag_string(w->table, paf->bitvector));
    show_duration(output, paf);
    *ppaf = paf;
}

void show_obj_affects(CHAR_DATA *ch, BUFFER *output, AFFECT_DATA *paf)
{
    AFFECT_DATA *paf_last = NULL;

    for (; paf; paf = paf->next)
        if (paf->location != APPLY_SPELL_AFFECT)
            show_bit_affect(output, paf, &paf_last, TO_AFFECTS);
}

void show_affects(CHAR_DATA *ch, BUFFER *output)
{
    OBJ_DATA *obj;
    AFFECT_DATA *paf, *paf_last = NULL;
    int chance = 0;

    chance = (get_skill(ch, gsn_computers)
    + get_skill(ch, gsn_spell_craft)
    + get_skill(ch, gsn_psionics)
    + get_skill(ch, gsn_faith)
    + get_skill(ch, gsn_singing));

    if (IS_AFFECTED(ch, AFF_DETECT_MAGIC))
        chance += 100;

    chance += LVL(ch);

    buf_printf(output, "%s is affected by :\n", ch->name);
    for (paf = ch->affected; paf; paf = paf->next)
    {
        show_loc_affect(ch, output, paf, &paf_last);
        if (number_percent() > chance + 10)
            continue;
        show_bit_affect(output, paf, &paf_last, TO_AFFECTS);
    }

    if (number_percent() > chance + 10)
        return;

    for (obj = ch->carrying; obj; obj = obj->next_content)
        if (obj->wear_loc != WEAR_NONE)
        {
            if (!IS_SET(obj->extra_flags, ITEM_ENCHANTED))
                show_obj_affects(ch, output, obj->pIndexData->affected);
            show_obj_affects(ch, output, obj->affected);
        }
}

#ifdef WIN32
void SET_ORG_RACE(CHAR_DATA *ch, int race)
{
    if (IS_NPC(ch))
        ch->pIndexData->race = race;
    else
        ch->pcdata->race = race;
}
#endif

/* Materials stuff */

void eq_damage(CHAR_DATA *ch, CHAR_DATA *victim, OBJ_DATA *obj, int chance, int value)
{
    if (!obj)
        return;
    chance+=chance*(100-obj->condition)/100;
    if (IS_AFFECTED(ch,AFF_BERSERK))
    {
        chance*=2;
        value*=2;
    }
    if (chance>number_range(1,100))
        if (value<obj->condition && value>0)
        {
            obj->condition-=value;
            act( "$p   $N.", victim, obj, ch, TO_CHAR );
            act( "$p $N   .", ch, obj, victim, TO_CHAR );
        }
        else
        {
            act( "$p   $N.", victim, obj, ch, TO_CHAR );
            act( "$p $N   .", ch, obj, victim, TO_CHAR );
            unequip_char( victim, obj );
            obj->condition=0;
        }
    /* ,     */
    else
    {
        act( "$N   $p.", victim, obj, ch, TO_CHAR );
        act( "   $p $N.", ch, obj, victim, TO_CHAR );
    }
}

int part2wear(int part)
{
    switch (part)
    {
    case PART_HEAD: return WEAR_HEAD;
    case PART_ARMS: return WEAR_ARMS;
    case PART_LEGS: return WEAR_LEGS;
    case PART_HEART: return WEAR_BODY;
    case PART_BRAINS: return WEAR_HEAD;
    case PART_GUTS: return WEAR_BODY;
    case PART_HANDS: return WEAR_HANDS;
    case PART_FEET: return WEAR_FEET;
    case PART_FINGERS: return WEAR_FINGER_L+number_range(0,1);
    case PART_EAR: return WEAR_HEAD;
    case PART_EYE: return WEAR_HEAD;
    case PART_LONG_TONGUE: return WEAR_HEAD;
    case PART_EYESTALKS: return WEAR_BODY;
    case PART_TENTACLES: return WEAR_BODY;
    case PART_FINS: return WEAR_BODY;
    case PART_WINGS: return WEAR_BODY;
    case PART_TAIL: return WEAR_TAIL;
    case PART_CLAWS: return WEAR_BODY;
    case PART_FANGS: return WEAR_BODY;
    case PART_HORNS: return WEAR_BODY;
    case PART_SCALES: return WEAR_BODY;
    case PART_TUSKS: return WEAR_BODY;
    default: return WEAR_BODY;
    }
}


int count_used_space(ROOM_INDEX_DATA *room)
{
    CHAR_DATA *ch;
    int size = 0;

    for (ch = room->people; ch != NULL; ch = ch->next_in_room)
        size += ch->size;

    return size;
}

/*****************************************************************************
 * some formatting stuff
 *
 */

/*
 * smash '~'
 * for IMC2-compatibility
 */
void smash_tilde(char *s)
{
    char *c = s;
    while ((c = strchr(c, '~')))
        *c = '-';
}

/*
 * smash '~'
 */
const char *fix_short(const char *s)
{
    char *p;
    static char buf[MAX_STRING_LENGTH];

    if (!strchr(s, '~'))
        return s;

    for (p = buf; *s && p-buf < sizeof(buf)-1; s++)
    {
        if (*s == '~')
            continue;
        *p++ = *s;
    }

    *p = '\0';
    return buf;
}

const char *format_short(mlstring *mlshort, const char *name, CHAR_DATA *looker, int act_flags)
{
    static char buf[MAX_STRING_LENGTH];
    const char *sshort;
    int lang;
    lang = looker == NULL ? 0 : looker->lang;

    sshort = IS_SET(act_flags, ACT_NOFIXSH) ?
             mlstr_val(mlshort, lang) :
             fix_short(mlstr_val(mlshort, lang));

    strnzcpy(buf, sizeof(buf), sshort);

    if (IS_NULLSTR(name))
        return buf;

    if (IS_SET(act_flags, ACT_FORMSH_ALWAYS)
        || (looker != NULL && !IS_SET(looker->comm, COMM_NOENG)
            &&  sshort != mlstr_mval(mlshort)))
    {
        char buf2[MAX_STRING_LENGTH];
        char buf3[MAX_STRING_LENGTH];

        one_argument(name, buf3, sizeof(buf3));
        snprintf(buf2, sizeof(buf2), " (%s)", buf3);
        strnzcat(buf, sizeof(buf), buf2);
    }
    return buf;
}

/*
 * format description (long descr for mobs, description for objs)
 *
 * eng name expected to be in form " (foo)" and is stripped
 * if COMM_NOENG is set
 */
const char *format_descr(mlstring *ml, CHAR_DATA *looker)
{
    const char *s;
    const char *p, *q;
    static char buf[MAX_STRING_LENGTH];

    s = mlstr_cval(ml, looker);
    if (IS_NULLSTR(s)
        ||  !IS_SET(looker->comm, COMM_NOENG)
        ||  (p = strchr(s, '(')) == NULL
        ||  (q = strchr(p+1, ')')) == NULL)
        return s;

    if (p != s && *(p-1) == ' ')
        p--;

    strnzncpy(buf, sizeof(buf), s, p-s);
    strnzcat(buf, sizeof(buf), q+1);
    return buf;
}

DO_FUN (do_name)
{
    char arg[MAX_STRING_LENGTH];
    CHAR_DATA * victim;
    int num;

    // char name
    argument = one_argument (argument, arg, sizeof(arg));

    if (!IS_IMMORTAL(ch))
    {
        if (arg[0] == '\0')
            victim = ch;
        else
        {
            if ((victim = get_char_world(ch, arg)) == NULL)
            {
                char_act("No chars with this name found.", ch) ;
                return;
            }
            if (IS_NPC(victim))
            {
                char_act("Not on NPCs.", ch) ;
                return;
            }
        }
        show_russian_name (ch, victim);
    }
    else
    {
        if (arg[0] == '\0')
            victim = ch;
        else
        {
            if ((victim = get_char_world(ch, arg)) == NULL)
            {
                char_act("No chars with this name found.", ch) ;
                return;
            }
            if (IS_NPC(victim))
            {
                char_act("Not on NPCs.", ch) ;
                return;
            }
        }

        argument = one_argument (argument, arg, sizeof(arg));
        if (arg[0] == '\0')
        {
            show_russian_name (ch, victim);
            return;
        }
        num = atoi (arg);
        if (num < 0 || num > 5)
        {
            char_act ("Case must be between 0 and 5.", ch);
            return;
        }

        one_argument (argument, arg, sizeof(arg));
        free_string(victim->pcdata->rusnames[num]);
        victim->pcdata->rusnames[num] = str_dup (capitalize(arg));
        char_act ("Ok.", ch);
    }
}

void show_russian_name (CHAR_DATA * ch, CHAR_DATA * victim)
{
    if ((victim == NULL) || (IS_NPC(victim)) || (victim->pcdata == NULL))
        return;

    char_printf (ch, "   %s:\n", victim->name);
    char_printf (ch, "{x    (?):          [%s]\n", victim->pcdata->rusnames[0] ? victim->pcdata->rusnames[0] : " ");
    char_printf (ch, "{x    ( ?):  [%s]\n", victim->pcdata->rusnames[1] ? victim->pcdata->rusnames[1] : " ");
    char_printf (ch, "{x    ( ?):       [%s]\n", victim->pcdata->rusnames[2] ? victim->pcdata->rusnames[2] : " ");
    char_printf (ch, "{x    ( ?): [%s]\n", victim->pcdata->rusnames[3] ? victim->pcdata->rusnames[3] : " ");
    char_printf (ch, "{x    ( ?):  [%s]\n", victim->pcdata->rusnames[4] ? victim->pcdata->rusnames[4] : " ");
    char_printf (ch, "{x    (  ?):   [%s]\n", victim->pcdata->rusnames[5] ? victim->pcdata->rusnames[5] : " ");
}

const char * get_rus_name (CHAR_DATA * ch, int fnum)
{
    if (ch == NULL)
        return "nobody";
    if (IS_NPC (ch) || ch->pcdata == NULL)
        return ch->name;
    if (fnum == 6)
        fnum = 3;
    if (fnum < 0 || fnum > 5)
        fnum = 0;
    return  (ch->pcdata->rusnames[fnum] == NULL ? ch->name : ch->pcdata->rusnames[fnum]);
}
