/* $Id: marry.c,v 1.666 2004/09/20 10:49:50 shrike Exp $                             */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

/* Part of this code is based on WoM's (World of Merlin) code */

#ifndef WIN32
#include <sys/time.h>
#endif

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <time.h>
#include "merc.h"
#include "db/dofun.h"
#include "tables.h"
#include "olc/olc.h"
#include "db/db.h"

/*
 *  Create room in 'castle of love' or find existing non-used
 */
int generate_marry_room ( CHAR_DATA *ch, CHAR_DATA *victim )
{
    static char buf[MAX_STRING_LENGTH];
    AREA_DATA *pArea;
    ROOM_INDEX_DATA *pRoom;
    int vnum;

    pRoom=NULL;
    
    /* Find area */
    for ( pArea = area_first; pArea; pArea = pArea->next )
        if ( !str_cmp ( pArea->name, "MLC" ) )
            break;

    if ( !pArea )
    {
        log_printf("generate_marry_room: area 'MLC!");
        return 0;
    }
        

    /* Find free room */
    for ( vnum = pArea->min_vnum; vnum < pArea->max_vnum; vnum++ )
    {
        pRoom = get_room_index (vnum);
        if ( !pRoom || mlstr_null(pRoom->name) )
            break;
    }

    if ( vnum == pArea->max_vnum || vnum == pArea->min_vnum)
    {
        log_printf("generate_marry_room: no free rooms!");
        return 0;
    }

    /* Create room if needed, make exit to ROOM_VNUM_TEMPLE */
    if ( !pRoom )
    {
        ROOM_INDEX_DATA *toRoom;
        int iHash;
                
        pRoom = new_room_index();
        pRoom->area = pArea;
        pRoom->vnum = vnum;
        pRoom->space=255;

        if ( vnum > top_vnum_room )
            top_vnum_room = vnum;

        iHash = vnum % MAX_KEY_HASH;
        pRoom->next = room_index_hash[iHash];
        room_index_hash[iHash] = pRoom;

        toRoom = get_room_index ( ROOM_VNUM_TEMPLE );

        if ( !toRoom )
        {
            log_printf("generate_marry_room: get_room_index (ROOM_VNUM_TEMPLE)");
            return 0;
        }
        
        if ( !pRoom->exit[DIR_SOUTH] )
            pRoom->exit[DIR_SOUTH] = new_exit();

        pRoom->exit[DIR_SOUTH]->to_room.r = toRoom;    /* ROM OLC */
        pRoom->exit[DIR_SOUTH]->orig_door = DIR_SOUTH;
        pRoom->exit[DIR_SOUTH]->size=25;

        SET_BIT(pRoom->exit[DIR_SOUTH]->rs_flags, EX_NOSCAN);
    }

    /* Replace room name and desc, set room flags */
    mlstr_free ( pRoom->name );
    snprintf(buf, sizeof(buf), "{G %s  %s{x", ch->name, victim->name );
    pRoom->name = mlstr_new ( buf );

    mlstr_free ( pRoom->description );
    pRoom->description  = mlstr_new ( "     .\n" );

    pRoom->room_flags = ROOM_PRIVATE|ROOM_SAFE|ROOM_NOWHERE|ROOM_NOSUMMON|ROOM_NOTIMER;
    pRoom->sector_type = SECT_INSIDE;

    save_area(pArea);
    return vnum;
}

/*
 *  Freed room in 'castle of love'
 */
void free_marry_room ( CHAR_DATA *ch )
{
    AREA_DATA *pArea;
    ROOM_INDEX_DATA *pRoom;
    int vnum;

    pRoom=NULL;
    /* Find area */
    for ( pArea = area_first; pArea; pArea = pArea->next )
        if ( !str_cmp ( pArea->name, "MLC" ) )
            break;

    if ( !pArea )
        return;

    /* Find room */
    for ( vnum = pArea->min_vnum; vnum < pArea->max_vnum; vnum++ )
    {
        pRoom = get_room_index (vnum);
        if (ch->pcdata->marry_room==vnum && pRoom && is_name ( ch->name, mlstr_val(pRoom->name,0)))
            break;
    }

    if ( vnum == pArea->max_vnum )
        return;

    /* Clear room name */
    mlstr_free ( pRoom->name );
    pRoom->name = mlstr_new(str_empty);
}

void divorce_char ( CHAR_DATA *ch )
{
    CHAR_DATA *victim=NULL;
    
    if ( !IS_MARRYED (ch) )
        return;

    victim = get_char_world(ch, ch->pcdata->marryed);

    if (!victim)
    {
        char_act("    .", ch);
        return;
    }
    
    free_string (victim->pcdata->marryed);
    victim->pcdata->marryed = str_empty;
    victim->pcdata->marry_room = 0;
    act_puts("$N   .", victim, 0, ch, TO_CHAR, POS_DEAD);

    free_marry_room ( ch );
    free_string (ch->pcdata->marryed);
    ch->pcdata->marryed = str_empty;
    ch->pcdata->marry_room = 0;

    act_puts(" ", ch, 0, victim, TO_CHAR, POS_DEAD);
    ch->pcdata->last_divorce = current_time;
    victim->pcdata->last_divorce = current_time;

    ch->pcdata->last_marry = -1;
    victim->pcdata->last_marry = -1;

    save_char_obj(ch, FALSE, FALSE);
    save_char_obj(victim, FALSE, FALSE);
}


/*
 *  Is <victim> can be marry with <anybody>? If no, errors will be shown to <ch>
 */
bool can_be_married ( CHAR_DATA *ch, CHAR_DATA *victim )
{
    double   time_diff;

    /* Now we'll checks */
    if ( IS_NPC(victim) )
    {
        char_act("  ?    ????", ch);
        return FALSE;
    }
       
    if ( IS_MARRYED ( victim ) )
    {
        if ( ch != victim )
        act_puts(" $N   -.", ch, 0, victim, TO_CHAR, POS_DEAD);
        else
        act_puts("    -.", ch, 0, 0, TO_CHAR, POS_DEAD);

        return FALSE;
    }

    if ( victim->level < 25 )
    {
        if ( ch != victim )
        act_puts("$N   .", ch, 0, victim, TO_CHAR, POS_DEAD);
        else
        act_puts("    ?.", ch, 0, 0, TO_CHAR, POS_DEAD);

        return FALSE;
    }

    if ( victim->gold < victim->level * 100 )
    {
        if ( ch != victim )
        act_puts(" $N    .", ch, 0, victim, TO_CHAR, POS_DEAD);
        else
          char_printf ( ch, "     ( %d ).\n", victim->level * 100 );

        return FALSE;
    }

    if ( !IS_NPC (victim) && victim->pcdata->questpoints < victim->level * 2 )
    {
        if ( ch != victim )
        act_puts(" $N   .", ch, 0, victim, TO_CHAR, POS_DEAD);
        else
          char_printf( ch, "      ( %d).\n",
                victim->level * 2 ); 

        return FALSE;
    }

    if ( victim->pcdata->last_divorce != -1 && victim->pcdata->last_divorce != 0 )
    {
        time_diff = difftime ( current_time, victim->pcdata->last_divorce );

  
        if ( time_diff < ( 60*60*24*30 ) )
        {
            if ( ch != victim )

            act_puts(" $N       .", ch, 0, victim, TO_CHAR, POS_DEAD);
            else
              char_printf( ch, "        "
                    "( %d ).\n", 30-(int) time_diff / (60*60*24) );

            return FALSE;
        }
    }

    return TRUE;
}


bool marry_chars ( CHAR_DATA *ch, CHAR_DATA *victim, CHAR_DATA *cleric )
{
    static char buf[MAX_STRING_LENGTH];
    OBJ_DATA *ring;
    ED_DATA *ed;
    const char* name;

    if (victim == ch)
        return FALSE;

    /* Get fee */
    ch->gold -= ch->level * 100;
    ch->pcdata->questpoints -= ch->level * 2;
    victim->gold -= victim->level * 100;
    victim->pcdata->questpoints -= victim->level * 2;

    /* Prepare rings */
    name=str_dup(capitalize(ch->name));
    ring = create_obj( get_obj_index (OBJ_VNUM_MARRY_RING), 1);

    if ( ring )
    {
        snprintf(buf, sizeof(buf),
        "   :\n\r {G, {W%s{G,      \n\r  {W%s{G   %s          %s \n\r   ,      -     {x.\n\r",
        name, capitalize (victim->name), victim->sex == SEX_FEMALE ? "" : "",  victim->sex == SEX_FEMALE ? "ţ" : "" );
        ed = ed_new();
        ed->keyword     = str_dup( "letters" );
        ed->description = mlstr_new( buf );
        ed->next        = ring->ed;
        ring->ed = ed;
        ring->owner=str_dup(victim->name);
        obj_to_char ( ring, victim );
    }

    ring = create_obj ( get_obj_index (OBJ_VNUM_MARRY_RING), 1);
    if ( ring )
    {
        snprintf(buf, sizeof(buf),
        "   :\n\r {G , {W%s{G,      \n\r  {W%s{G   %s          %s \n\r   ,      -     {x.\n\r",
        capitalize (victim->name), name, ch->sex == SEX_FEMALE ? "" : "",  ch->sex == SEX_FEMALE ? "ţ" : "" );

        ed = ed_new();
        ed->keyword     = str_dup( "letters" );
        ed->description = mlstr_new( buf );
        ed->next        = ring->ed;
        ring->ed = ed;
        ring->owner=str_dup(ch->name);
        obj_to_char ( ring, ch );
    }
    
    free_string(name);
    name=NULL;

    /* Automatic ceremonial */
    if ( !cleric )
    {
        act( "  $p $n.", ch, ring, NULL, TO_ROOM );
        act( "   $p.", ch, ring, NULL, TO_CHAR );
        act( "  $p $n.", victim, ring, NULL, TO_ROOM );
        act( "   $p.", victim, ring, NULL, TO_CHAR );
    }
    else /* manual */
    {
        act( "  $p $N.", cleric, ring, ch, TO_CHAR );
        act( "$n  $p $N.", cleric, ring, ch, TO_NOTVICT );
        act( "$n   $p.", cleric, ring, ch, TO_VICT );
        act( "  $p $N.", cleric, ring, victim, TO_CHAR );
        act( "$n  $p $N.", cleric, ring, victim, TO_NOTVICT );
        act( "$n   $p.", cleric, ring, victim, TO_VICT );
        
        do_say ( cleric, ",   ,      !");
    }

    act("   $N!", ch, NULL, victim, TO_CHAR);
    act("   $N!", victim, NULL, ch, TO_CHAR);

    info(NULL, 1, "{r[{RINFO{r]{W : {R%s{z and {R%s{z just married!{x\n", ch->name, victim->name);

    ch->pcdata->marryed = str_dup(victim->name);
    victim->pcdata->marryed = str_dup(ch->name);

    ch->pcdata->last_divorce = -1;
    victim->pcdata->last_divorce = -1;

    ch->pcdata->last_marry = current_time;
    victim->pcdata->last_marry = current_time;
        
    save_char_obj(ch, FALSE, FALSE);
    save_char_obj(victim, FALSE, FALSE);

    return TRUE;
}


DO_FUN ( do_proposal )
{
    static char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;

    argument = one_argument(argument, arg, sizeof(arg));

    if ( !str_cmp(arg, "clean") && IS_IMMORTAL(ch) )
    {
        victim = get_char_world(ch, argument);
        if ( victim == NULL)
            return;

        if(victim->pcdata == NULL)
            return;

        victim->pcdata->last_divorce = -1;
        return;
    }
    if ( !str_cmp (arg, "accept") )
    {
        victim = get_char_world(ch, argument);
        if (victim == NULL)
            return;

        if(victim->pcdata == NULL) 
            return;

        if ( !victim->pcdata->proposal || str_cmp (victim->pcdata->proposal, ch->name) )
        {
            act( "H $N       !", ch, NULL, victim, TO_CHAR );
            return;
        }

        if (ch->in_room->vnum!=ROOM_VNUM_TEMPLE)
        {
            char_act("    .", ch);
            return;
        }

        if ( victim->in_room != ch->in_room )
        {
            char_act("   , ?", ch);
            return;
        }

        if ( !can_be_married ( ch, ch ) || !can_be_married ( ch, victim ))
            return;

        marry_chars ( ch, victim, NULL );
        return;
    }

    if ( !str_cmp (arg, "reject") )
    {
        victim = get_char_world(ch, argument);
        if ( victim  == NULL)
            return;

        if(victim->pcdata == NULL)
            return;
        

        if ( !victim->pcdata->proposal || str_cmp (victim->pcdata->proposal, ch->name) )
        {
            act( "H $N       !", ch, NULL, victim, TO_CHAR );
            return;
        }

        act( "$N       !", victim, NULL, ch, TO_CHAR );
        act( "  $N  !", ch, NULL, victim, TO_CHAR );

        free_string ( victim->pcdata->proposal );
        victim->pcdata->proposal = str_empty;
        return;
    }

    if ( !str_cmp (arg, "list") )
    {
        
        DESCRIPTOR_DATA *d;
        bool found = FALSE;

        for (d = descriptor_list; d; d = d->next) 
        {
            CHAR_DATA *vch = d->original ? d->original : d->character;

            if(victim->pcdata == NULL)
                continue;

            if (!vch || !vch->pcdata->proposal  || str_cmp (vch->pcdata->proposal, ch->name))
                continue;

            if (!found) 
            {
                found = TRUE;
                char_act("    :", ch);
            }
            char_printf(ch, "%s\n", vch->name);
        }

        if (!found) 
            char_act(",    .", ch);
        return;
    }

    if ( IS_NULLSTR (arg) )
    {
        char_printf ( ch, ": proposal accept/reject/list [name]  none\n" );
        return;
    }

    if ( !str_cmp ( arg, "none" ) )
    {
        if ( !ch->pcdata->proposal )
        {
            char_printf ( ch, "      (proposal).\n" );
            return;
        }

        char_printf ( ch, "   .\n" );

        victim = get_char_world(ch, ch->pcdata->proposal);

	if (victim == NULL)
	    return;

        if ( victim )
            act( "$N      .", victim, NULL, ch, TO_CHAR );

        free_string ( ch->pcdata->proposal );
        ch->pcdata->proposal = NULL;
        return;
    }

    victim = get_char_world(ch, arg);
    if ( !victim )
    {
    char_printf ( ch, "     .\n" ); 
        return;
    }

    if ( victim == ch )
    {
        char_printf ( ch, " .\n" );
        return;
    }

    if(victim->pcdata == NULL)
        return;

    if ( !can_be_married ( ch, ch ) || !can_be_married ( ch, victim ))
        return;

    free_string(ch->pcdata->proposal);
    ch->pcdata->proposal = str_dup(victim->name);
    act( "$N      !", victim, NULL, ch, TO_CHAR );
    act( "  $N    !", ch, NULL, victim, TO_CHAR );
}

DO_FUN ( do_marry )
{
    static char arg1[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH];
    CHAR_DATA *victim, *victim2;

    argument = one_argument( argument, arg1 , sizeof(arg1 ));
    argument = one_argument( argument, arg2 , sizeof(arg2 ));

    if ( !IS_IMMORTAL (ch) )
    {
        char_act("  ...    (proposal)!", ch);
    }
    
    if ( IS_NULLSTR (arg1) || IS_NULLSTR (arg2) )
    {
        char_act("Syntax: marry <char1> <char2>", ch);
        return;
    }

    victim = get_char_room(ch, arg1);
    victim2 = get_char_room(ch, arg2);

    if ( victim == NULL || victim2 == NULL )
        return;

    if ( !can_be_married ( ch, victim2 ) || !can_be_married ( ch, victim ) )
        return;

    marry_chars ( victim, victim2, ch );
}

DO_FUN ( do_divorce )
{
    CHAR_DATA *victim;
    DESCRIPTOR_DATA d ;
    bool fCharLoaded=FALSE;

    if ( IS_NULLSTR (argument) )
    {
        char_act(": divorce < >", ch);
        return;
    }

    if ( !IS_MARRYED (ch) )
    {
        act( "H  !", ch, NULL, NULL, TO_CHAR );
        return;
    }

    victim = get_char_world(ch, argument);
    if ( victim == NULL )
    {
        load_char_obj (&d, argument, FALSE) ;

        d.character->desc = NULL ;
        d.character->next = char_list ;
        char_list         = d.character ;
        d.connected       = CON_PLAYING ;
        reset_char (d.character) ;

       // bring player to imm
        if (d.character->in_room != NULL)
        {  
             d.character->was_in_room = d.character->in_room ;
             char_to_room (d.character, get_room_index (ROOM_VNUM_LIMBO)) ;
        }

        act("$n has pulled $N from the pattern!",
                        ch, NULL, d.character, TO_ROOM) ;

        if (d.character->pet != NULL)
        {  
            char_to_room (d.character->pet, 
                            get_room_index (ROOM_VNUM_LIMBO));
            act("$n has entered the game.", 
                            d.character->pet, NULL, NULL, TO_ROOM) ;
        }
        victim = get_char_world(ch, argument);
        if (victim == NULL)
        {
            char_act("   .", ch);
            return;
        }
        fCharLoaded=TRUE;
    }

    if ( IS_IMMORTAL (ch) )
    {
        if ( !IS_MARRYED (victim) )
        {
            char_act("   .", ch);
                save_char_obj (victim, FALSE, TRUE) ;
            extract_char_nocount (victim, TRUE) ;
            return;
        }

        divorce_char ( victim );
        char_printf ( ch, ".\n" );
        if (fCharLoaded)
            {            
                 save_char_obj (victim, FALSE, TRUE) ;
                 extract_char_nocount (victim, TRUE) ;
            }
           return;
    }

    if ( str_cmp ( ch->pcdata->marryed, victim->name ) )
    {
    act( "H   c $N!", ch, NULL, victim, TO_CHAR );
    if (fCharLoaded)
    {
            save_char_obj (victim,FALSE, TRUE) ;
            extract_char_nocount (victim, TRUE) ;
        }
        return;
    }

    divorce_char ( ch );
    if (fCharLoaded)
    {
        save_char_obj (victim, FALSE, TRUE) ;
        extract_char_nocount (victim, TRUE) ;
    }
}

DO_FUN(do_sptalk)
{
    CHAR_DATA *victim;

    if (!IS_MARRYED(ch))
    {
        act_puts("H   $t!", ch, 
                 ch->sex == SEX_FEMALE ? "" : "", NULL,
                 TO_CHAR, POS_DEAD);
        return;
    }

    if (IS_NULLSTR(argument))
    {
        act_puts("    $t?", ch,
                 ch->sex == SEX_FEMALE ? " " : " ", NULL,
                 TO_CHAR, POS_DEAD);
        return;
    }

    victim = get_char_world(ch, ch->pcdata->marryed);

    if (!victim)
    {
        act_puts("$t    .", ch,
                ch->sex == SEX_FEMALE ? " " : " ", NULL,
                TO_CHAR, POS_DEAD);
        return;
    }

    act_puts("  $N: '{G$t{x'", ch, argument, victim, TO_CHAR | ACT_NODEAF, POS_DEAD);
    act_puts("$n  : '{G$t{x'", ch, argument, victim, TO_VICT | ACT_TOBUF | ACT_NODEAF, POS_SLEEPING);
}

DO_FUN(do_sprecall)
{
    ROOM_INDEX_DATA *location;

    if (!IS_MARRYED(ch))
    {
        char_printf ( ch, "H   %s!\n\r", ch->sex==SEX_FEMALE ?
            "" : "" );
        return;
    }

    location = get_room_index( ch->pcdata->marry_room);
    if ( location == NULL )
    {
         char_printf (ch, "  ,   .\n\r" );
         return;
    }

    if ( ch->in_room == location )
        return;

    act("$n    !", ch, NULL, NULL, TO_ROOM);

    if (IS_PUMPED(ch)) 
    {
        char_act("You are too pumped to pray now.", ch);
        return;
    }
    
    if (!check_war_move (ch, location, FALSE))
        return;

    if ( IS_SET(ch->in_room->room_flags, ROOM_NORECALL)
    || IS_AFFECTED(ch, AFF_CURSE)
    || IS_AFFECTED(ch, AFF_SLEEP)
    || IS_AFFECTED(ch, AFF_CHARM)
    || IS_AFFECTED(ch, AFF_BLIND)
    ||  IS_RAFFECTED(ch->in_room, RAFF_ESPIRIT)
    ||  IS_RAFFECTED(ch->in_room, RAFF_CURSE))
    {
        char_printf (ch, "     .\n\r" );
        return;
    }
    
    if ( ch->fighting != NULL )
    {
        char_printf (ch, "   -  ?\n\r" );
        return;
    }

    if (ch->in_room->vnum != ROOM_VNUM_TEMPLE)
    {
        act("You must be deep inside Great Temple.",
            ch, NULL, NULL, TO_CHAR);
        return;
    }

    act("$n  .", ch, NULL, NULL, TO_ROOM);

    char_from_room( ch );
    char_to_room( ch, location );
    act( "$n  ,   .", ch, NULL, NULL, TO_ROOM );
    do_look( ch, "auto" );
}

DO_FUN ( do_weddingroom )
{
    CHAR_DATA *victim=NULL;
    int vRoom;

    if (!IS_MARRYED(ch))
    {
        char_printf ( ch, "H   %s!\n\r", ch->sex==SEX_FEMALE ?
            "" : "" );
        return;
    }
    if (ch->pcdata->marry_room>0)
    {
        char_printf ( ch, "   ! n\r");
        return;
    }

    victim = get_char_world(ch, ch->pcdata->marryed);
    if (!victim)
    {
        char_act("     .", ch);
        return;
    }

    if (ch->pcdata->bonuspoints<10)
    {
        char_act(" 10     .", ch);
        return;
    }
    
    vRoom = generate_marry_room ( ch, victim );
    if ( !vRoom )
        return;

    ch->pcdata->bonuspoints-=10;
    victim->pcdata->marry_room = vRoom;
    ch->pcdata->marry_room = vRoom; 

    char_act("  .", ch);
    char_act("  .", victim);
}

DO_FUN ( do_reunion )
{
    CHAR_DATA *victim;

    if ( !IS_MARRYED(ch) )
    {
        char_printf ( ch, "H   %s!\n\r", ch->sex==SEX_FEMALE ?
            "" : "" );
        return;
    }

    victim = get_char_world (ch, ch->pcdata->marryed );

    if ( !victim )
    {
        char_printf ( ch, "%s    .\n\r", ch->sex==SEX_FEMALE ?
            " " : " " );
        return;
    }

    act("$n      !", ch, NULL, NULL, TO_ROOM);

    if ( victim->in_room == NULL
    || victim->in_room->vnum == ROOM_VNUM_LIMBO
    || IS_SET(ch->in_room->room_flags, ROOM_NORECALL)
    || IS_AFFECTED(ch, AFF_CURSE))
    {
        char_printf (ch, "     .\n\r" );
        return;
    }

    if ( IS_SET(ch->in_room->room_flags, ROOM_NORECALL)
    || IS_AFFECTED(ch, AFF_CURSE)
    ||  IS_RAFFECTED(ch->in_room, RAFF_ESPIRIT)
    ||  IS_RAFFECTED(ch->in_room, RAFF_CURSE)
    ||  IS_RAFFECTED(victim->in_room, RAFF_ESPIRIT)
    ||  IS_RAFFECTED(victim->in_room, RAFF_CURSE))
    {
        char_printf (ch, "     .\n\r" );
        return;
    }

    if ( IS_AFFECTED(ch, AFF_CHARM)
    || IS_AFFECTED(ch, AFF_SLEEP)
    || IS_AFFECTED(ch, AFF_BLIND)
    || IS_AFFECTED(victim, AFF_SLEEP)
    || IS_AFFECTED(victim, AFF_BLIND)
    || IS_AFFECTED(victim, AFF_CHARM))
    {
        char_printf (ch, "    .\n\r" );
        return;
    }
 
    if ( ch->fighting != NULL )
    {
        char_printf (ch, "   -  ?\n\r" );
        return;
    }

    if (!check_war_move (ch, victim->in_room, FALSE))
        return;

    if (ch->pcdata->questpoints < 10)
    {
        char_printf ( ch, "  10    " );
        return;
    }
    
    ch->pcdata->questpoints-=10;
    
    act("$n   .", ch, NULL, NULL, TO_ROOM);

    char_from_room( ch );
    char_to_room( ch, victim->in_room );
    act("$n  ,   .", ch, NULL, NULL, TO_ROOM);
    do_look( ch, "auto" );
}
