/* $Id: martial_art.c,v 1.666 2004/09/20 10:49:50 shrike Exp $                      */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
 /************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/


#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <stdarg.h>

#if !defined (WIN32)
#include <unistd.h>
#include <sys/time.h>
#endif
#include <ctype.h>

#ifdef SUNOS
#include <stdarg.h>
#include "compat/compat.h"
#endif
#include "war.h"
#include "merc.h"
#include "fight.h"
#include "wanderers.h"

DECLARE_DO_FUN (do_yell);
DECLARE_DO_FUN (do_sleep);
DECLARE_DO_FUN (do_sit);
DECLARE_DO_FUN (do_bash_door);
DECLARE_DO_FUN (do_look);
DECLARE_DO_FUN (set_spellbane);
DECLARE_DO_FUN (do_visible);
DECLARE_DO_FUN (do_throw_spear);
DECLARE_DO_FUN (do_throw_shuriken);
DECLARE_DO_FUN (do_throw_knife);
DECLARE_DO_FUN (do_throw_grenade);

extern bool check_trust(CHAR_DATA *ch, CHAR_DATA *victim);
extern STAT_DATA stat_record;

void check_downstrike     (CHAR_DATA *, CHAR_DATA *);
void check_ground_control (CHAR_DATA *, CHAR_DATA *, int,int);
void agent_net_printf     (CHAR_DATA *, CHAR_DATA *, int);
void one_hit              (CHAR_DATA *, CHAR_DATA *, int, int);
void set_fighting         (CHAR_DATA *, CHAR_DATA *);
bool recharge_battery     (CHAR_DATA *, int);
bool check_mirror_image   (CHAR_DATA *ch, CHAR_DATA *victim);


static inline bool check_yell (CHAR_DATA *, CHAR_DATA *, bool);

//
// rolling is auto-skill of thieves that enables them to roll out
// of bash, throw, trip attacks
//
bool check_roll (CHAR_DATA * ch)
{
    int chance ;

    if ((chance = get_skill (ch, gsn_roll)) <= 1) return FALSE ;
    if (ch->move < 15 && !IS_NPC (ch))            return FALSE ;

    chance  = chance / 2 ;
    chance += get_curr_stat (ch, STAT_DEX) / 2 ;
    chance -= 2 * (ch->size - 6);

    if (IS_AFFECTED (ch, AFF_HASTE)) chance += 10 ;
    if (IS_AFFECTED (ch, AFF_SLOW))  chance -= 10 ;

    chance -= number_range (0, 15);

    if (number_percent () > chance)
    {
        check_improve (ch, gsn_roll, FALSE, 1);
        return FALSE ;
    } 

    act("$n  - .",  ch, NULL, NULL, TO_ROOM);
    act("  - .", ch, NULL, NULL, TO_CHAR);

    check_improve (ch, gsn_roll, TRUE, 1);

    // reset character position
    RESET_WAIT_STATE (ch);
    ch->position = POS_STANDING ;
    return TRUE ;
}

// rolling is auto-skill of warriors that enables them to recovery
// of trip, throw attacks
bool check_recovery (CHAR_DATA * ch, CHAR_DATA * victim)
{
    int chance;

    if ((chance = get_skill (ch, gsn_recovery)) <= 1) return FALSE;
    if (ch->move < 15 && !IS_NPC (ch))                return FALSE;
                                              
    chance  = chance / 2;
    chance += get_curr_stat (ch, STAT_DEX);
  
    if (IS_AFFECTED (ch, AFF_HASTE)) chance += 10;
    if (IS_AFFECTED (ch, AFF_SLOW))  chance -= 10;

    chance -= number_range (0, 10);

    // samurai's recovery is slightly worse
    if (IS_SAMURAI(ch)) chance -= 10;

    if (number_percent() > chance)
    {
        check_improve (ch, gsn_recovery, FALSE, 1);
        return FALSE;
    }

    // GM : aikido skill
    if (number_range(0, 300) < get_skill(ch, sn_lookup("aikido")))
    {
        act("$n   ,   .", ch, NULL, NULL, TO_ROOM);
        act("   ,   .", ch, NULL, NULL, TO_CHAR);
        if (victim)
            DAZE_STATE(victim, PULSE_VIOLENCE * 2);
        check_improve(ch, sn_lookup("aikido"), TRUE, 1);
    } else
    {
        act("$n      .", ch, NULL, NULL, TO_ROOM);
        act("      .", ch, NULL, NULL, TO_CHAR);
        check_improve(ch, sn_lookup("aikido"), FALSE, 1);
    }

    // reset character position
    RESET_WAIT_STATE (ch);
    ch->position = POS_STANDING;
    return TRUE;
}

// jammer is clan skill that enables to jamm
// bash, throw attacks
bool check_jammer (CHAR_DATA * ch)
{
    int chance ;

    if ((chance = get_skill (ch, gsn_jammer)) <= 1) return FALSE ;
    if (ch->move < 15 && !IS_NPC (ch))              return FALSE ;

    chance = chance * 6 / 10 ;

    if (number_percent () < 50) chance += number_range (1, 10);
    else                        chance -= number_range (1, 10);

    if (number_percent () < chance)
    {
        check_improve (ch, gsn_jammer, TRUE, 1);
        return TRUE ;
    } 

    check_improve (ch, gsn_jammer, FALSE, 1);
    return FALSE ;
}

// check if perception skill is working to detect any danger
bool check_perception (CHAR_DATA * ch, CHAR_DATA * victim)
{
    int chance = 0 ;
    chance = get_skill (ch, gsn_perception) / 2;

    if (chance == 0)
    {
        chance = get_skill (ch, gsn_awareness) * (2 / 3);
        if (is_affected(ch, sn_lookup("force sense")))
            chance += get_curr_stat(ch, STAT_LCK);
    } else
    {
        chance += get_skill (ch, gsn_awareness) / 4;
        if (is_affected(ch, sn_lookup("force sense")))
            chance += get_curr_stat(ch, STAT_LCK) / 4;
    }

    if (chance == 0) return FALSE ;

    chance = chance / 4 + 4 * (get_curr_stat (ch,     STAT_LCK) -
                               get_curr_stat (victim, STAT_LCK));

    if (number_percent () < chance)
    {
        check_improve (ch, gsn_perception, 2, TRUE);
        check_improve (ch, gsn_awareness,  2, TRUE);
        return TRUE ;
    }

    check_improve (ch, gsn_perception, 2, FALSE);
    check_improve (ch, gsn_awareness,  2, FALSE);
    return FALSE ; 
}

//
// those who are mastered in swords can riposte the attack
// and redirect it to the opponent
//
void check_riposte (CHAR_DATA * ch, CHAR_DATA * victim, int dam)
{
    int        chance ;
    OBJ_DATA * wield  ;

    if ((chance = get_skill (ch, gsn_riposte)) <= 1) return ;

    // must wield a primary weapon to riposte
    // and this weapon must be sword

    if ((wield = get_eq_char (ch, WEAR_WIELD)) == NULL) return ;
    if (wield->value[0] != WEAPON_SWORD)                return ;

    // chance to success
    chance  = chance * 2 / 10 ;
    chance += get_curr_stat (ch, STAT_STR);
    chance -= get_curr_stat (victim, STAT_STR) / 2 ;
    
    if (number_percent () < chance)
    { 
        act("   $N    !",
             ch, NULL, victim, TO_CHAR | ACT_VERBOSE);

        act("$n       !",
             ch, NULL, victim, TO_VICT | ACT_VERBOSE);

        check_improve (ch, gsn_riposte, TRUE, 3);

        damage (ch, victim, dam, gsn_riposte, TYPE_UNDEFINED, TRUE);
        return ;
    }       

    check_improve (ch, gsn_riposte, FALSE, 4);
    return ;
}

//
// those who are mastered in daggers can rapidly restrike
// when parrying the attack
//
void check_restrike (CHAR_DATA * ch, CHAR_DATA * victim)
{
    int        chance ;
    OBJ_DATA * wield  ;

    if ((chance = get_skill (ch, gsn_restrike)) <= 1) return ;

    // must wield a primary weapon to restrike
    // and this weapon must be dagger

    if ((wield = get_eq_char (ch, WEAR_WIELD)) == NULL) return ;
    if (wield->value[0] != WEAPON_DAGGER)               return ;

    // chance to success
    chance = chance * 1 / 10 ;

    if (IS_AFFECTED (ch, AFF_HASTE)) chance += 5 ;
    if (IS_AFFECTED (ch, AFF_SLOW))  chance -= 5 ;

    chance += get_curr_stat (ch, STAT_DEX) / 4 ;

    if (number_percent () < chance)
    {
        act("       $N.",
             ch, NULL, victim, TO_CHAR | ACT_VERBOSE);

        act("$n       .",
             ch, NULL, victim, TO_VICT | ACT_VERBOSE);

        check_improve (ch, gsn_restrike, TRUE, 1);

        one_hit (ch, victim, gsn_restrike, WEAR_WIELD);
        return ;
    }

    check_improve (ch, gsn_restrike, FALSE, 1);
    return ;
}

//
// warriors can bash opponent to the ground disabling him
//
void do_bash (CHAR_DATA * ch, const char * argument)
{
    char arg [MAX_INPUT_LENGTH] ;
    int chance, wait, damage_bash ;

    CHAR_DATA * victim ;

    if (MOUNTED (ch))
    {
        char_act("      !", ch);
        return ;
    }

    argument = one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill (ch, gsn_bash)) == 0)
    {
        char_act("?  ?", ch);
        return ;
    }

    // special skill to bash doors
    if (arg[0] != '\0' && !str_cmp (arg, "door"))
    {
        do_bash_door (ch, argument);
        return ;
    }

    if (ch->move < 40 && !IS_NPC (ch))
    {
        char_act("  .", ch);
        return ;
    }

    // find opponent
    if (arg[0] == '\0')
    {
        victim = ch->fighting ;

        if (victim == NULL ||
            victim->in_room != ch->in_room)
        {
            char_act("      !", ch);
            return ;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("  .", ch);
        return ;
    }
    
    if (IS_AFFECTED (ch, AFF_FEAR))
    {
        char_act(" ?    -   !", ch);
        return ;
    }

    if (is_safe (ch, victim)) return ;

    // wait state to character that performs bashing
    WAIT_STATE (ch, SKILL(gsn_bash)->beats);

    if (victim->position < POS_FIGHTING)
    {
        act(" $gN{}   .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (victim == ch)
    {
        char_act("    , ,  , .", ch);
        return ;
    }

    if (MOUNTED (victim))
    {
        char_act("    !", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // 70% probability that protective shield would prevent the attack
    if (is_affected (victim, gsn_protective_shield)
    && number_percent () < 70)
    {
        act_puts("   $N  ,     $gN{}  .", 
            ch, NULL, victim, TO_CHAR, POS_FIGHTING);
        act_puts("$n        ,      .", 
            ch, NULL, victim, TO_VICT, POS_FIGHTING);
        act_puts("$n     $N,    $gN{}  .", 
            ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        return;
    }

    // modifiers
    if (IS_NPC (ch) && IS_SET (ch->pIndexData->off_flags, OFF_FAST))
        chance += 10 ;

    if (IS_NPC (victim) && IS_SET (victim->pIndexData->off_flags, OFF_FAST))
        chance -= 20 ;

    chance += (ch->size - victim->size) * 3 ; // size bonus
    chance += (LVL (ch) - LVL (victim)) * 2 ; // level bonus

    chance += get_curr_stat (ch, STAT_STR);
    chance -= get_curr_stat (victim, STAT_DEX) * 3 / 2 ;

    if (IS_AFFECTED (ch, AFF_FLYING))     chance -= 10 ;
    if (IS_AFFECTED (victim, AFF_FLYING)) chance -= 10 ;
    if (IS_AFFECTED (victim, gsn_shield)) chance -= 10 ;

    RESET_WAIT_STATE (ch);

    if(check_mirror_image(ch,victim))
        return;
    
    // now the attack
    if (number_percent () < chance)
    {
        AFFECT_DATA af ;

        af.where     = TO_AFFECTS ;
        af.type      = gsn_bash   ;
        af.level     = LVL (ch)   ;
        af.duration  = 1 ;
        af.modifier  = 0 ;
        af.location  = APPLY_NONE ;
        af.bitvector = 0 ;

        act("$n      !",
             ch, NULL, victim, TO_VICT);

        act("   $N  $gN{} !",
             ch, NULL, victim, TO_CHAR);

        act("$n   $N  $gN{} .",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_bash, TRUE, 1);
        
        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (2);

        victim->position = POS_RESTING ;

        damage_bash = (ch->damroll / 2) +
                      number_range (4, 4 + ch->size + chance / 10);

        damage (ch, victim, damage_bash, gsn_bash, DAM_BASH, TRUE);

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_bash);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)  ;
        WAIT_STATE (ch,     SKILL(gsn_bash)->beats);

        if (wait > 0) affect_to_char (victim, &af);
    }
    else
    {
        damage (ch, victim, 0, gsn_bash, DAM_BASH, TRUE);

        act(" ,    !",
             ch, NULL, victim, TO_CHAR);

        act("$n ,    !",
             ch, NULL, victim, TO_NOTVICT);

        act("      $n ,    !",
             ch, NULL, victim, TO_VICT);

        check_improve (ch, gsn_bash, FALSE, 1);

//        ch->position = POS_RESTING ;
        WAIT_STATE (ch, SKILL(gsn_bash)->beats);
    }

    if (check_yell (ch, victim, (ch->fighting != NULL)))
    {
        act_yell(victim, " ! $i  !", ch, NULL);

        agent_net_printf (victim, ch, ANET_ATTACK);
    }
}

//
// trip non-flying opponent to put him onto the ground
//
void do_trip (CHAR_DATA * ch, const char * argument)
{
    char arg [MAX_INPUT_LENGTH] ;
    int chance, wait, damage_trip ;
    bool avoided = FALSE;

    CHAR_DATA * victim ;

    if (MOUNTED(ch))
    {
        char_act("        !", ch);
        return ;
    }

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill (ch, gsn_trip)) == 0)
    {
        char_act("?    ?", ch);
        return ;
    }

    if (ch->move < 20 && !IS_NPC (ch))
    {
        char_act("  .", ch);
        return ;
    }

    // find opponent
    if (arg[0] == '\0')
    {
        victim = ch->fighting ;

        if (victim == NULL 
        || victim->in_room != ch->in_room)
        {
            char_act("      !", ch);
            return ;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("  .", ch);
        return ;
    }

    if (is_safe (ch, victim)) 
        return;

    if (IS_AFFECTED (ch, AFF_FEAR))
    {
        char_act(" ?    ?!", ch);
        return ;
    }

    if (victim->position < POS_FIGHTING)
    {
        act("$N   .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (victim == ch)
    {
        act("$n    !", ch, NULL, NULL, TO_ROOM);
        return ;
    }

    if (MOUNTED (victim))
    {
        char_act("     !", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (IS_AFFECTED (victim, AFF_FLYING))
    {
        act("$gN{}    .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // modifiers

    if (IS_NPC (ch) && IS_SET (ch->pIndexData->off_flags, OFF_FAST))
        chance += 10 ;

    if (IS_NPC (victim) && IS_SET (victim->pIndexData->off_flags, OFF_FAST))
        chance -= 20 ;

    chance += (ch->size - victim->size) * 2 ; // size bonus
    chance += (LVL (ch) - LVL (victim)) * 2 ; // level bonus

    chance += get_curr_stat (ch, STAT_DEX) -
              get_curr_stat (victim, STAT_DEX);

    if (IS_AFFECTED (ch, AFF_FLYING)) chance -= 10 ;

    RESET_WAIT_STATE (ch);

    if (check_mirror_image(ch,victim))
        return;
    
    // now the attack
    if (number_percent () < chance)
    {
        AFFECT_DATA af ;

        af.where     = TO_AFFECTS   ;
        af.type      = gsn_trip     ;
        af.level     = LVL (ch)     ;
        af.duration  = 1            ;
        af.modifier  = LVL (victim);
        af.location  = APPLY_AC     ;
        af.bitvector = 0            ;

        act("$n      !",
             ch, NULL, victim, TO_VICT);

        act("   $N  $gN{} !",
              ch, NULL, victim, TO_CHAR);

        act("$n   $N,  $gn{}  .",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_trip, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim))
        {
            WAIT_STATE (ch, SKILL(gsn_trip)->beats);
            return;
        }

        wait = number_fuzzy (2);

        victim->position = POS_RESTING ;

        if ((avoided = (check_recovery(victim, ch) || check_jammer (victim))))
            wait = wait / 2;
        
        damage_trip = (ch->damroll / 2) +
                      number_range (4, 4 + ch->size + chance / 10);

        damage (ch, victim, damage_trip, gsn_trip, DAM_BASH, TRUE);

        if (!avoided) 
            check_downstrike (ch, victim);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)  ;
        WAIT_STATE (ch,     SKILL(gsn_trip)->beats);

        if (wait > 0) affect_to_char (victim, &af);
    } else
    {
        damage (ch, victim, 0, gsn_trip, DAM_BASH, TRUE);

        WAIT_STATE (ch, SKILL(gsn_trip)->beats);

        check_improve (ch, gsn_trip, FALSE, 1);
    }

    if (JUST_KILLED(victim))
        return;

    if (check_yell (ch, victim, (ch->fighting != NULL)))
    {
        act_yell(victim, " ! $i   !", ch, NULL);
        agent_net_printf (victim, ch, ANET_ATTACK);
    }
}

//
// throws opponent to the ground damaging and disabling him
// or throw item (shuriken, spear, knife, grenade)
//
void do_throw (CHAR_DATA * ch, const char * argument)
{
    char arg [MAX_INPUT_LENGTH] ;
    int chance, wait, damage_throw, hand ;

    CHAR_DATA * victim ;
    
    if (MOUNTED (ch))
    {
        char_act("      !", ch);
        return ;
    }
    
    if (IS_AFFECTED (ch, AFF_FEAR))
    {
        char_act("  !    ?", ch);
        return ;
    }

    argument = one_argument(argument, arg, sizeof(arg));

    // special throwings
    if (!str_cmp (arg, "spear"))
    {
        do_throw_spear (ch, argument);
        return ;
    }

    if (!str_cmp (arg, "shuriken"))
    {
        do_throw_shuriken (ch, argument);
        return ;
    }

    if (!str_cmp (arg, "knife"))
    {
        do_throw_knife (ch, argument);
        return ;
    }

    if (!str_cmp (arg, "grenade"))
    {
        do_throw_grenade (ch, argument);
        return ;
    }

    if ((chance = get_skill (ch, gsn_throw)) == 0)
    {
        char_act("       .", ch);
        return ;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
        char_act("  .", ch);
        return ;
    }

    hand = (get_eq_char (ch, WEAR_WIELD)        != NULL) +
           (get_eq_char (ch, WEAR_SECOND_WIELD) != NULL) +
           (get_eq_char (ch, WEAR_SHIELD)       != NULL) +
           (get_eq_char (ch, WEAR_HOLD)         != NULL);

    if (hand >= 2)
    {
        char_act("      ,  ", ch);
        return ;
    }

    if (hand > 0) chance = chance * 3 / 5 ;
    

    if ((victim = ch->fighting) == NULL || victim->in_room != ch->in_room)
    {
        char_act("     .", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_STUN) || IS_AFFECTED (ch, AFF_WEAK_STUN))
    {
        act("   $N     $gn{}!\n", 
                        ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (is_safe (ch, victim)) return ;

    WAIT_STATE (ch, SKILL(gsn_throw)->beats);

    if (IS_AFFECTED (ch, AFF_FLYING))
    {
        char_act("      .", ch);
        return; 
    }

    if (IS_AFFECTED (ch,AFF_CHARM) && ch->master == victim)
    {
        act("$N  !", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // 50% probability that protective shield would prevent the attack
    if (is_affected (victim, gsn_protective_shield) &&
        number_percent () < 50)
    {
        act_puts("    $gn{}    .", ch, NULL, victim, TO_CHAR, POS_FIGHTING);

        act_puts(" $n      .", ch, NULL, victim, TO_VICT, POS_FIGHTING);

        act_puts(" $n      $N.", ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        return ;
    }

    // modifiers

    if (IS_NPC (ch) && IS_SET (ch->pIndexData->off_flags, OFF_FAST))
        chance += 10 ;

    if (IS_NPC (victim) && IS_SET (victim->pIndexData->off_flags, OFF_FAST))
        chance -= 20 ;

    chance += (ch->size - victim->size) * 2 ; // size bonus
    chance += (LVL (ch) - LVL (victim)) * 2 ; // level bonus

    chance += get_curr_stat (ch, STAT_STR);
    chance -= get_curr_stat (victim, STAT_DEX) * 2 ;

    if (IS_AFFECTED (ch, AFF_FLYING))     chance -= 10 ;
    if (IS_AFFECTED (victim, AFF_FLYING)) chance -= 10 ;

    RESET_WAIT_STATE (ch);

    if(check_mirror_image(ch,victim))
        return;
    
    // now the attack
    if (number_percent () < chance)
    {
        AFFECT_DATA af ;
             
        af.where     = TO_AFFECTS ;
        af.type      = gsn_throw  ;
        af.level     = LVL (ch)   ;
        af.duration  = 1          ;
        af.modifier  = 0          ;
        af.location  = APPLY_NONE ;
        af.bitvector = 0          ;

        act("     $N  .",
             ch, NULL, victim, TO_CHAR);

        act("$n       !",
             ch, NULL, victim, TO_VICT);

        act("$n     $N  .",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_throw, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (2);

        victim->position = POS_RESTING ;

        damage_throw = LVL (ch) + get_curr_stat (ch, STAT_STR);

        damage (ch, victim, damage_throw, gsn_throw, DAM_BASH, TRUE);

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_throw);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)   ;
        WAIT_STATE (ch,     SKILL(gsn_throw)->beats);

        if (wait > 0) affect_to_char (victim, &af);
    }
    else
    {
        act("      .",
             ch, NULL, NULL, TO_CHAR);

        act("$N   ,      $c1{$gN{}} .",
             victim, NULL, ch, TO_CHAR);

        act("$n   $N.",
             ch, NULL, victim, TO_NOTVICT);

        WAIT_STATE (ch, SKILL(gsn_throw)->beats * 3 / 2);

        check_improve (ch, gsn_throw, FALSE, 1);
    }
}

//
// Lizards can trip opponent with their tail
// 
void do_tail (CHAR_DATA * ch, const char * argument)
{
    char arg [MAX_INPUT_LENGTH] ;
    int chance, wait, damage_tail, damage_bonus = 0 ;
    OBJ_DATA    * tail_ring  ;
    AFFECT_DATA * af ;

    CHAR_DATA * victim ;

    if (MOUNTED (ch))
    {
        char_act("    ,    !", ch);
        return ;
    }

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill (ch, gsn_tail)) == 0)
    {
        char_act("Huh?", ch);
        return ;
    }

    if (ch->move < 20 && !IS_NPC (ch))
    {
        char_act("  .", ch);
        return ;
    }

    // find opponent
    if (arg[0] == '\0')
    {
        victim = ch->fighting ;

        if (victim == NULL 
        || victim->in_room != ch->in_room)
        {
            char_act("      !", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("  .", ch);
        return ;
    }

    if (is_safe (ch, victim)) return ;

    WAIT_STATE (ch, SKILL(gsn_tail)->beats);

    if (is_affected(ch, gsn_polymorph))
    {
        char_act("      !", ch);
        return ;
    }

    if (victim->position < POS_FIGHTING)
    {
        act(" $gN{}   .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (victim == ch)
    {
        char_act("      ,   !", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  !", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // 50% probability that protective shield would prevent the attack
    if (is_affected (victim, gsn_protective_shield) 
    && number_percent () < 50)
    {
        act_puts("     ,  $N.", 
            ch, NULL, victim, TO_CHAR, POS_FIGHTING);
        act_puts(" $n    ,  .", 
            ch, NULL, victim, TO_VICT, POS_FIGHTING);
        act_puts(" $n    ,  $N.", 
            ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        return ;
    }

    // modifiers

    if (IS_NPC (ch) && IS_SET (ch->pIndexData->off_flags, OFF_FAST))
        chance += 10 ;

    if (IS_NPC (victim) && IS_SET (victim->pIndexData->off_flags, OFF_FAST))
        chance -= 20 ;

    // size, level and weight
    if (ch->carry_weight > victim->carry_weight)
        chance += 5;
    else
        chance -= 5;

    if (ch->size < victim->size) chance += (ch->size - victim->size) * 3 ;
    else                         chance += (ch->size - victim->size) * 7 / 5 ;

    chance += (LVL (ch) - LVL (victim)) * 2 ;

    // stats
    chance += get_curr_stat (ch, STAT_STR) + get_curr_stat (ch, STAT_DEX);
    chance -= get_curr_stat (victim, STAT_DEX) * 2 ;

    if (IS_AFFECTED (ch, AFF_FLYING))     chance -= 10 ;
    if (IS_AFFECTED (victim, AFF_FLYING)) chance -= 10 ;
    if (MOUNTED     (victim))             chance -= 20 ;

    RESET_WAIT_STATE (ch);

    if (check_mirror_image(ch, victim))
        return;
    
    if ((tail_ring = get_eq_char (ch, WEAR_TAIL)) != NULL)
    {
        damage_bonus  = tail_ring->weight > 150 ? 
                    (50 + (tail_ring->weight - 150) / 5) : tail_ring->weight / 3 ;

        for (af = tail_ring->affected ; af ; af = af->next)
        {
            if (af->where != TO_AFFECTS) continue ;

            switch (af->location)
            {
            case APPLY_STR:     damage_bonus += af->modifier * 25 ; break ;
            case APPLY_DEX:     chance       += af->modifier * 10 ; break ;
            case APPLY_HITROLL: chance       += af->modifier * 4  ; break ;
            case APPLY_DAMROLL: damage_bonus += af->modifier * 10 ; break ;
            }
        }
    }
    // now the attack
    if (number_percent() < (chance * 2 / 3))
    {
        act("$n      !",
             ch, NULL, victim, TO_VICT);
        act("   $N   !",
             ch, NULL, victim, TO_CHAR);
        act("$n   $N   .",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_tail, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (3);

        victim->position = POS_RESTING ;

        damage_tail = ch->damroll +
                      (2 * number_range(4, 4 + 2 * ch->size + chance / 10));

        damage_tail += damage_bonus;

        damage (ch, victim, damage_tail, gsn_tail, DAM_BASH, TRUE);

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_tail);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)  ;
        WAIT_STATE (ch,     SKILL(gsn_tail)->beats);
    } else
    {
        damage (ch, victim, 0, gsn_tail, DAM_BASH, TRUE);

        act("    !",
             ch, NULL, victim, TO_CHAR);
        act("$n    !.",
             ch, NULL, victim, TO_NOTVICT);
        act("    $n, "
             " $gn{}    .",
             ch, NULL, victim, TO_VICT);

        check_improve (ch, gsn_tail, FALSE, 1);

        ch->position = POS_RESTING ;
        WAIT_STATE (ch, SKILL(gsn_tail)->beats * 3 / 2);
    }

    if (check_yell (ch, victim, (ch->fighting != NULL)))
    {
        act_yell(victim, " ! $i    !", ch, NULL);
        agent_net_printf (victim, ch, ANET_ATTACK);
    }
}

//
// Large warriors can crush opponent with their boots
//
void do_crush_boots (CHAR_DATA * ch, const char * argument)
{
    int chance, wait, damage_crush ;

    CHAR_DATA   * victim ;
    OBJ_DATA    * boots  ;
    AFFECT_DATA * af ;

    if (IS_NPC (ch)) return ;

    if (MOUNTED (ch))
    {
        char_act("     ,    !", ch);
        return ;
    }
    
    if ((chance = get_skill (ch, gsn_crush_boots)) == 0)
    {
        char_act("     .", ch);
        return ;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
         char_act("  .", ch);
         return ;
    }

    if ((victim = ch->fighting) == NULL || victim->in_room != ch->in_room)
    {
        char_act("     .", ch);
        return ;
    }

    if (is_safe (ch, victim)) return;

    WAIT_STATE (ch, SKILL(gsn_crush_boots)->beats);

    if (victim->position < POS_FIGHTING)
    {
        act(" $gN{}   .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  !", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if ((boots = get_eq_char (ch, WEAR_FEET)) == NULL)
    {
         char_act("  ,    .", ch);
         return ;
    }

    if (victim->size * 3 / 2 > ch->size)
    {
         char_act("         ,   .", ch);
         return ;
    }

    // 50% probability that protective shield would prevent the attack
    if (is_affected (victim, gsn_protective_shield) 
    && number_percent () < 50)
    {
         act_puts("   $N    ,  $N.", 
            ch, NULL, victim, TO_CHAR, POS_FIGHTING);
         act_puts(" $n      .", 
            ch, NULL, victim, TO_VICT, POS_FIGHTING);
         act_puts(" $n  $N    ,  $N.", 
            ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
         return;
    }

    // modifiers
    // size, level  and weight
    if (ch->carry_weight > victim->carry_weight)
        chance += 4;
    if (ch->carry_weight < victim->carry_weight)
        chance -= 4;
    
    chance += (ch->size - victim->size) * 2 ;
    chance += (LVL (ch) - LVL (victim));
    
    // stats
    chance += (get_curr_stat (ch, STAT_STR) - 18) * 4 ;
    chance += (get_curr_stat (ch, STAT_DEX) - 18) * 4 ;
    chance -= (get_curr_stat (victim, STAT_STR) - 18) * 4 ;
    chance -= (get_curr_stat (victim, STAT_DEX) - 18) * 4 ;

    if (IS_AFFECTED (ch, AFF_FLYING))     chance -= 10 ;
    if (IS_AFFECTED (victim, AFF_FLYING)) chance -= 20 ;
    if (MOUNTED     (victim))             chance -= 20 ;
    
    damage_crush  = boots->weight > 150 ? 
                    (50 + (boots->weight - 150) / 5) : boots->weight / 3 ;

    damage_crush  = UMIN(damage_crush, 200);
    damage_crush += LVL (ch) + ch->damroll ;

    // boots
    for (af = boots->affected ; af ; af = af->next)
    {
        if (af->where != TO_AFFECTS) continue ;

        switch (af->location)
        {
        case APPLY_STR:     damage_crush += af->modifier * 25 ; break ;
        case APPLY_DEX:     chance       += af->modifier * 10 ; break ;
        case APPLY_HITROLL: chance       += af->modifier * 4  ; break ;
        case APPLY_DAMROLL: damage_crush += af->modifier * 10 ; break ;
        }
    }

    if (IS_NPC(victim))
    {
        chance += 20;
        damage_crush *= 2;
        if (IS_AFFECTED (victim, AFF_CHARM))
        {
            chance += 20;
        }
    }
    
    RESET_WAIT_STATE (ch);

    if (check_mirror_image(ch, victim))
            return;
    
    // now the attack
    if (number_percent () < chance)
    {
        act("$n    $p!", ch, boots, victim, TO_VICT);
        act("  $N  $p!",  ch, boots, victim, TO_CHAR);
        act("$n  $N  $p.",   ch, boots, victim, TO_NOTVICT);

        check_improve (ch, gsn_crush_boots, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (2);

        victim->position = POS_RESTING;

        damage (ch, victim, damage_crush, gsn_crush_boots, DAM_BASH, TRUE);

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_crush);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)   ;
        WAIT_STATE (ch,     SKILL(gsn_crush_boots)->beats);
    } else
    {
        damage (ch, victim, 0, gsn_crush_boots, DAM_BASH, TRUE);

//        act(" !", ch, boots, victim, TO_CHAR);
        act("$n   $N  $p,  .",
             ch, boots, victim, TO_NOTVICT);
        act("$n     $p,   !",
             ch, boots, victim, TO_VICT);

        check_improve (ch, gsn_crush_boots, FALSE, 1);

        WAIT_STATE (ch, SKILL(gsn_crush_boots)->beats);
    }
}

//
// Crushes an opponent inflicting damage and disabling him
//
void do_crush (CHAR_DATA * ch, const char * argument)
{
    int chance, wait, damage_crush ;

    CHAR_DATA * victim ;

    if (MOUNTED (ch))
    {
        char_act("     ,    !", ch);
        return ;
    }

    if ((chance = get_skill (ch, gsn_crush)) == 0)
    {
        char_act("     .", ch);
        return ;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
         char_act("  .", ch);
         return ;
    }

    if ((victim = ch->fighting) == NULL || victim->in_room != ch->in_room)
    {
        char_act("     .", ch);
        return ;
    }

    if (is_safe (ch, victim)) return ;

    WAIT_STATE (ch, SKILL(gsn_crush)->beats);

    if (victim->position < POS_FIGHTING)
    {
        act(" $gN{}   .", ch, NULL, victim, TO_CHAR);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  !", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // 50% probability that protective shield would prevent the attack
    if (is_affected (victim, gsn_protective_shield) 
    && number_percent () < 50)
    {
        act_puts("      ,  $N.", 
            ch, NULL, victim, TO_CHAR, POS_FIGHTING);
        act_puts("  $n      .", 
            ch, NULL, victim, TO_VICT, POS_FIGHTING);
        act_puts("  $n    ,  $N.", 
            ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        return;
    }

    // modifiers

    if (IS_NPC (ch) && IS_SET (ch->pIndexData->off_flags, OFF_FAST))
        chance += 10 ;

    if (IS_NPC (victim) && IS_SET (victim->pIndexData->off_flags, OFF_FAST))
        chance -= 20 ;

    // size, level and weight
    if (ch->carry_weight > victim->carry_weight)
        chance += 5;
    else
        chance -= 5;

    chance += (LVL (ch) - LVL (victim)) * 2;

    if (ch->size < victim->size) chance += (ch->size - victim->size) * 3 ;
    else                         chance += (ch->size - victim->size) * 7 / 5 ;

    // stats
    chance += get_curr_stat (ch, STAT_STR);
    chance -= get_curr_stat (victim, STAT_DEX) * 4 / 3 ;

    if (IS_AFFECTED (ch, AFF_FLYING))     chance -= 10 ;
    if (IS_AFFECTED (victim, AFF_FLYING)) chance -= 10 ;
    if (MOUNTED     (victim))             chance -= 20 ;

    RESET_WAIT_STATE (ch);
    
    if (check_mirror_image(ch, victim))
        return;

    // now the attack
    if (number_percent () < chance)
    {
        act("$n   ,   !",
             ch, NULL, victim, TO_VICT);
        act("  $N  ,  $gN{} !",
             ch, NULL, victim, TO_CHAR);
        act("$n  $N  ,  $gN{} .",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_crush, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (2);

        victim->position = POS_RESTING;

        damage_crush = (ch->damroll / 2) +
                       number_range(4, 4 + ch->size + chance / 10);

        damage (ch, victim, damage_crush, gsn_crush, DAM_BASH, TRUE);

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_crush);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)   ;
        WAIT_STATE (ch,     SKILL(gsn_crush)->beats);
    } else
    {
        damage (ch, victim, 0, gsn_crush, DAM_BASH, TRUE);

        act("    !", ch, NULL, victim, TO_CHAR);
        act("$n    .",  ch, NULL, victim, TO_NOTVICT);
        act("    $n, "
             " $gn{}    !",
             ch, NULL, victim, TO_VICT);

        check_improve (ch, gsn_crush, FALSE, 1);

        ch->position = POS_RESTING ;
        WAIT_STATE (ch, SKILL(gsn_crush)->beats * 3 / 2);
    }
}

//
// Clan Demons skill, that allows to prick opponent with horns and
// crush him to the ground
//
void do_horn_prick (CHAR_DATA * ch, const char * argument)
{
    char arg [MAX_INPUT_LENGTH] ;
    int chance = 0, wait, damage_horn ;

    CHAR_DATA * victim ;

    if (MOUNTED(ch))
    {
        char_act("  ?  .", ch);
    }
    
    if (!IS_NPC(ch))
    {
        if ((chance = get_skill (ch, gsn_horn_prick)) == 0)
        {
            char_act("Huh?", ch);
            return ;
        }   
    }

    if (ch->move < 30 && !IS_NPC (ch))
    {
        char_act("  .", ch);
        return ;
    }

    argument = one_argument(argument, arg, sizeof(arg));

    // find opponent
    if (arg[0] == '\0')
    {
        victim = ch->fighting ;

        if (victim == NULL || victim->in_room != ch->in_room)
        {
            char_act("      !", ch);
            return ;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("  .", ch);
        return ;
    }
   
    if (!IS_NPC(ch))
    {
        if (!IS_DEMON (ch))
        {
            char_act("    .", ch);
            return ;
        }
    }

    if (is_safe (ch, victim)) 
        return;

    if (IS_AFFECTED (ch, AFF_CHARM) && ch->master == victim)
    {
        act("$N  !", ch, NULL, victim, TO_CHAR);
        return ;
    }

    // modifiers

    chance -= (ch->size - victim->size) * 3 ;
    chance += (LVL (ch) - LVL (victim)) * 2 ;

    chance += get_curr_stat (ch, STAT_STR);
    chance -= get_curr_stat (victim, STAT_DEX) * 2 ;

    if (is_affected (victim, gsn_protective_shield)) 
        chance /= 2;

    // instant kill on NPC
    if (!IS_NPC(ch))
    {
        if (number_range(1, 200) == 1 && IS_NPC (victim))
        {
            act(" {R******{x   $N  {R {x.", 
                ch, NULL, victim, TO_CHAR);
            act("$n {R******{x $N    {R {x.", 
                ch, NULL, victim, TO_NOTVICT);
            act("$n {R******{x     {R {x.", 
                ch, NULL, victim, TO_VICT);
            char_act(" {R{x   .", victim);
    
            victim->position = POS_DEAD ;
            handle_death (ch, victim);
            return ;
        }
    } else
    if (number_range(1, 200) == 1 )
    {
        act(" {R******{x   $N  {R {x.", 
            ch, NULL, victim, TO_CHAR);
        act("$n {R******{x $N    {R {x.", 
            ch, NULL, victim, TO_NOTVICT);
        act("$n {R******{x     {R {x.", 
            ch, NULL, victim, TO_VICT);
        char_act(" {R{x   .", victim);

        victim->position = POS_DEAD ;
        handle_death (ch, victim);
        return ;
    }

    RESET_WAIT_STATE (ch);
    
    if (check_mirror_image(ch, victim))
        return;

    // now the attack
    if (number_percent() < chance)
    {
        AFFECT_DATA af ;

        af.where     = TO_AFFECTS ;
        af.type      = gsn_throw  ;
        af.level     = LVL (ch)   ;
        af.duration  = 1          ;
        af.modifier  = 0          ;
        af.location  = APPLY_NONE ;
        af.bitvector = 0          ;

        act("$n        !", 
            ch, NULL, victim, TO_VICT);
        act("  $N         !", 
            ch, NULL, victim, TO_CHAR);
        act("$n  $N      !", 
            ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_horn_prick, TRUE, 1);

        // check for rolling out the attack
        if (check_roll (victim)) return ;

        wait = number_fuzzy (2);

        damage_horn = number_range (LVL (ch) * 2, LVL (ch) * 10);
        if (is_affected (victim, gsn_protective_shield)) damage_horn /= 2 ;
        
        if (number_percent () > 90)
        {
            act("{R     !{x", ch, NULL, victim, TO_CHAR);
            act("{R! !       !{x", ch, NULL, victim, TO_VICT);
            
            damage (ch, victim, damage_horn * 2, gsn_throw, DAM_BASH, TRUE);
            check_ground_control (ch, victim, chance, damage_horn);
        }
        
        damage (ch, victim, damage_horn, gsn_throw, DAM_BASH, TRUE);
            
        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, damage_horn);

        wait = UMAX (wait, 1);
    
        // daze/wait states
        DAZE_STATE (victim, PULSE_VIOLENCE * wait)        ;
        WAIT_STATE (ch,     SKILL(gsn_horn_prick)->beats);

        if (wait > 0) affect_to_char (victim, &af);
    }
    else
    {
        act("       ", 
            ch, NULL, victim, TO_CHAR);
        act("$n        .", 
            ch, NULL, victim, TO_NOTVICT);
        act("      "
             "$n    .", ch, NULL, victim, TO_VICT);

        check_improve (ch, gsn_horn_prick, FALSE, 1);

        ch->position = POS_RESTING ;
        WAIT_STATE (ch, SKILL(gsn_horn_prick)->beats * 2);
    }   
    
    if (check_yell (ch, victim, (ch->fighting != NULL)))
    {
        act_yell(victim, " ! $i   !", ch, NULL);
        agent_net_printf (victim, ch, ANET_ATTACK);
    }
}

/*
 * Disarm a creature.
 * Caller must check for successful attack.
 */
void disarm(CHAR_DATA *ch, CHAR_DATA *victim, OBJ_DATA *obj)
{
    OBJ_DATA *obj2;
    int skill;

    if (IS_OBJ_STAT(obj, ITEM_NOREMOVE)) {
        act("$gN{}   !", ch, NULL, victim, TO_CHAR);
        act("$n   ,     !",
            ch, NULL, victim, TO_VICT);
        act("$n   $N,   .",
            ch, NULL, victim, TO_NOTVICT);
        return;
    }

    if ((skill = get_skill(victim, gsn_grip))) {
        skill += (get_curr_stat(victim, STAT_STR) -
              get_curr_stat(ch, STAT_STR)) * 5;
        if (number_percent() < skill) {
            act("$N        $gN{} !",
                ch, NULL, victim, TO_CHAR);
            act("$n   ,    !",
                ch, NULL, victim, TO_VICT);
            act("$n   $N,   .",
                ch, NULL, victim, TO_NOTVICT);
            check_improve(victim, gsn_grip, TRUE, 1);
            return;
        }
        else
            check_improve(victim,gsn_grip,FALSE,1);
    }

    act_puts("$n     !", ch, NULL, victim, TO_VICT, POS_FIGHTING);
    act_puts("  $N!", ch, NULL, victim, TO_CHAR, POS_FIGHTING);
    act_puts("$n  $N!", ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);

    obj_from_char(obj);
    if (IS_OBJ_STAT(obj, ITEM_NODROP) 
    || IS_OBJ_STAT(obj,ITEM_INVENTORY)
    || (victim->in_war && victim->war_status == PS_ALIVE)
    || (victim->in_room
    && IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA)))
        obj_to_char(obj, victim);
    else 
    {
        obj_to_room(obj, victim->in_room);
        if (IS_NPC(victim)
        && victim->wait == 0
        && can_see_obj(victim,obj)
        && !can_get_obj(victim, obj, TRUE)) 
        {
            get_obj(victim, obj, NULL, TRUE);
            wear_obj(victim, obj, TRUE);
        }
    }

    if ((obj2 = get_eq_char(victim, WEAR_SECOND_WIELD)) != NULL) 
    {
        act_puts("       !", 
            ch, NULL, victim, TO_VICT, POS_FIGHTING);
        act_puts("$N        !", 
            ch, NULL, victim, TO_CHAR, POS_FIGHTING);
        act_puts("$N        !", 
            ch, NULL, victim, TO_NOTVICT, POS_FIGHTING);
        unequip_char(victim, obj2);
        equip_char(victim, obj2, WEAR_WIELD);
    }
}

DO_FUN(do_berserk)
{
    int chance, hp_percent;
    int need, chars;

    if ((chance = get_skill(ch, gsn_berserk)) == 0) 
    {
        char_act("  {r{x  ,    .", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_BERSERK)
    || is_affected(ch, gsn_berserk))
    {
        char_act("   !", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CALM)) 
    {
        char_act("        .", ch);
        return;
    }

    need = SKILL(gsn_berserk)->min_mana;
    chars = IS_SET (muddy_mode, MUDDY_WARR_MOVES) ? ch->move : ch->mana;

    if (chars < need && !IS_NPC(ch)) 
    {
        char_act("   .", ch);
        return;
    }

    /* modifiers */

    /* fighting */
    if (ch->position == POS_FIGHTING)
        chance += 25;

    /* damage -- below 50% of hp helps, above hurts */
    hp_percent = 100 * ch->hit / ch->max_hit;
    chance += 50 - hp_percent / 2;

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        WAIT_STATE(ch, SKILL(gsn_berserk)->beats);
        if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
            ch->move -= need;
        else
            ch->mana -= need;

        /* heal a little damage */
        ch->hit += (ch->level + chance) * 2;
        ch->hit = UMIN(ch->hit, ch->max_hit);

        char_act("      !", ch);
        act_puts("  $c1{$n}  !", ch, NULL, NULL, TO_ROOM, POS_FIGHTING);
        check_improve(ch, gsn_berserk, TRUE, 2);

        af.where        = TO_AFFECTS;
        af.type         = gsn_berserk;
        af.level        = LVL(ch);
        af.duration     = number_fuzzy(LVL(ch) / 8);
        af.modifier     = UMAX(1, LVL(ch) / 5);
        af.bitvector    = 0;
        af.location     = APPLY_HITROLL;
        affect_to_char(ch,&af);

        af.location     = APPLY_DAMROLL;
        affect_to_char(ch,&af);

        af.bitvector    = AFF_BERSERK;
        af.modifier     = 50;
//        af.modifier     = -1 * UMAX(10, 10 * (LVL(ch) / 5));
        af.location     = APPLY_AC;
        affect_to_char(ch,&af);
    } else 
    {
        WAIT_STATE(ch, (SKILL(gsn_berserk)->beats) * 2);
        if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
            ch->move -= need/2;
        else
            ch->mana -= need/2;
        char_act("  ,    .", ch);
        check_improve(ch, gsn_berserk, FALSE, 2);
    }
}

DO_FUN(do_arrest)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    clan_t *clan = NULL;
    int chance;

    clan = clan_lookup(ch->clan);

    if (((chance = get_skill(ch, gsn_arrest)) == 0) && !IS_NPC(ch)) 
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == 0) 
    {
        char_act(" .", ch);
        return;
    }
    victim = get_char_room(ch, arg);

    if (!IS_NPC(ch)) WAIT_STATE(ch, SKILL(gsn_arrest)->beats);
    
    if (victim == NULL) 
    {
        char_act("  .", ch);
        return;
    }
    
    if (IS_NPC(victim)) 
    {
        char_act("   ", ch);
        return;
    }
    
    if (ch == victim) 
    {
        char_act(" ,       - .", ch);
        return;
    }

    if (victim->fighting) 
    {
        char_act("    .", ch);
        return;
    }

    if (ch->fighting) 
    {
        char_act("  !", ch);
        return;
    }

    if (!IS_SET(victim->plr_flags, PLR_WANTED)) 
    {
        char_act(" ?     !", ch);
        return;
    }

    if (is_affected(victim, gsn_arrest)) 
    {
        act("   !.", ch, NULL, NULL, TO_CHAR);
        return;
    }


    if (IS_SET(ch->plr_flags, PLR_WANTED)) 
    {
        char_act("    !", ch);
        return;
    }

    if (IS_IMMORTAL(victim)) 
    {
        char_act("   ,      ", ch);
        return;
    }
    
    if (IS_AFFECTED(ch,AFF_FEAR))
    {
        char_act(" ?      ?!", ch);
        return;
    }

    chance += can_see(victim, ch) ? -20 : 20;
    chance += URANGE(-10, (get_curr_stat(ch, STAT_DEX)-20)*2, 10);
    if (IS_NPC(ch)) chance = 20;
    chance += (LVL(ch) - LVL(victim)) / 10;
    if (current_time - victim->last_fight_time < 300) chance /= 2;

    if(check_mirror_image(ch,victim))
        return;

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;
        ROOM_INDEX_DATA* pRoomIndex;
        pcskill_t *ps;

        af.where = TO_AFFECTS;
        af.type = gsn_arrest;
        af.level = LVL(ch);
        af.duration = LVL(victim) * 2;
        af.bitvector = 0;
        af.location = APPLY_MOVE;
        af.modifier = -1 * (victim->max_move - 30);

        TOGGLE_BIT(victim->plr_flags, PLR_WANTED);

        affect_to_char(victim, &af);

        act_puts("$n           .", ch, NULL, victim, TO_VICT, POS_RESTING);
        act_puts("    $N      $gN{} .", ch, NULL, victim, TO_CHAR, POS_RESTING);
        act_puts("$n    $N      $gN{} .", ch, NULL, victim, TO_NOTVICT, POS_RESTING);

        act_yell(ch, "!   $gi{}!", victim, NULL);

        act_puts("      $N  .\n  $gN{}      .", ch, NULL, victim, TO_NOTVICT, POS_RESTING);
        act_puts("      $N  .\n  $gN{}      .", ch, NULL, victim, TO_CHAR, POS_RESTING);
        char_act("       .\n        .", victim);

        pRoomIndex = get_room_index(ROOM_VNUM_RULER_CELL);
        char_from_room(victim);
        char_to_room(victim, pRoomIndex);

        ++stat_record.arrested;

        if (IS_AWAKE(victim))
             do_look(victim, "auto");

        if (!IS_NPC(ch)) 
        {
            ps = pc_skill_lookup(ch, gsn_arrest);
            ps->percent++;
            ps->percent = UMIN(ps->percent, 100);

    /* Gold 4 Hunters */ 
            if (IS_SET(clan->flags, CLAN_HIDDEN))
            {
                  ch->pcdata->bank_g += (victim->level * 2);
                  char_act("         .", ch);
            }

        }
        agent_net_printf(ch, victim, ANET_ARREST);
        return;
    }

    act("   $N,     ", ch, NULL, victim, TO_CHAR);
    act("$n   ,  $gn{}   ", ch, NULL, victim, TO_VICT);
    act("$n   $N,   $gn{}   ", ch, NULL, victim, TO_NOTVICT);

    if (!IS_NPC(ch)) check_improve(victim, gsn_arrest, FALSE, 1);
//  if (ch->level > victim->level) if (is_safe(ch, victim)) return;
    multi_hit(victim,ch,TYPE_UNDEFINED);
}

DO_FUN(do_dirt)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    bool fighting;
    int chance;

    if (MOUNTED(ch)) 
    {
        char_act("       !", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_dirt)) == 0) 
    {
        char_act("    .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_dirt)->beats);

    if (arg[0] == '\0') 
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room) 
        {
            char_act("   !", ch);
            return;
        }
    }
    else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_FLYING)) 
    {
         char_act(" ?", ch);
         return;
    }

    if (IS_AFFECTED(victim, AFF_BLIND)) 
    {
        act("$gN{}    .", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (victim == ch) 
    {
        char_act(".", ch);
        return;
    }

    if (MOUNTED(victim)) 
    {
        char_act("       !", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim) 
    {
        act(" $N   !", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (IS_AFFECTED(ch, AFF_FEAR)) 
    {
        char_act("You too scare to do this!    ?!", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;

    /* modifiers */

    /* dexterity */
    chance += get_curr_stat(ch, STAT_DEX);
    chance -= 2 * get_curr_stat(victim, STAT_DEX);

    /* speed  */
    if ((IS_NPC(ch) && IS_SET(ch->pIndexData->off_flags, OFF_FAST))
    ||  IS_AFFECTED(ch, AFF_HASTE))
        chance += 10;
    if ((IS_NPC(victim) && IS_SET(victim->pIndexData->off_flags, OFF_FAST))
    ||  IS_AFFECTED(victim, AFF_HASTE))
        chance -= 25;

    if (is_affected(ch, gsn_protective_shield))
        chance -= 10;
    if (is_affected(victim, gsn_protective_shield))
        chance -= 20;

    /* level */
    chance += (LVL(ch) - LVL(victim)) * 2;

    if (chance % 5 == 0)
        chance += 1;

    /* terrain */

    switch(ch->in_room->sector_type) 
    {
    case(SECT_INSIDE):          chance -= 20;   break;
    case(SECT_CITY):            chance -= 10;   break;
    case(SECT_FIELD):           chance +=  5;   break;
    case(SECT_FOREST):                          break;
    case(SECT_HILLS):                           break;
    case(SECT_MOUNTAIN):        chance -= 10;   break;
    case(SECT_WATER_SWIM):      chance  =  0;   break;
    case(SECT_WATER_NOSWIM):    chance  =  0;   break;
    case(SECT_AIR):             chance  =  0;   break;
    case(SECT_DESERT):          chance += 10;   break;
    }

    if (chance == 0) 
    {
        char_act("  ,     .", ch);
        return;
    }

    fighting = (ch->fighting != NULL);
    
    if(check_mirror_image(ch,victim))
        return;

    /* now the attack */
    if (number_percent() < chance) 
    {
        AFFECT_DATA af;
        check_improve(ch, gsn_dirt, TRUE, 2);
        if(IS_UNDEAD(victim)) 
        {
            act("$n     ,   !",
                                        victim, NULL, NULL, TO_ROOM);
            act("$N     -  .",
                            victim, NULL, ch, TO_CHAR);
            return;
        }       
        act("$n   ,   !",
            victim, NULL, NULL, TO_ROOM);
        act("   ,   !", 
            victim, NULL, NULL, TO_CHAR);


        af.where        = TO_AFFECTS;
        af.type         = gsn_dirt;
        af.level        = LVL(ch) + LVL(ch) / 10;
        af.duration     = 0;
        af.location     = APPLY_HITROLL;
        af.modifier     = 0 - number_fuzzy(LVL(ch)/5);
        af.bitvector    = AFF_BLIND;

        affect_to_char(victim, &af);
        damage(ch, victim, number_range(2, 5),
               gsn_dirt, DAM_NONE, FALSE);
    }
    else {
        damage(ch, victim, 0, gsn_dirt, DAM_NONE, TRUE);
        check_improve(ch, gsn_dirt, FALSE, 2);
    }

    if (check_yell(ch, victim, fighting)) {
        act_yell(victim, " ! $i      !", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
    return;
}

DO_FUN(do_soul)
{
    int chance;
    CHAR_DATA *soul;
    CHAR_DATA *victim;
    
    if ((chance = get_skill(ch, gsn_soul)) == 0) {
        char_act(" ? ..", ch);
        return;
    }
                
    if ((victim = ch->fighting) != NULL) {
    char_act("      .", ch);
    return;
    }
    if (!IS_DEMON(ch)) {
        char_act("     .", ch);
        return;
    }
    if (ch->mana < 50) {
        char_act("       .", ch);
        return;
    }
    else {
    ch->mana -= 50;
    }
    if (number_percent() < chance) {
        soul = create_mob(get_mob_index(MOB_VNUM_SOUL));
        char_to_room(soul,ch->in_room);
        if (soul->desc != NULL) {
            char_act("      !!.", ch);
            return;
        }
        ch->desc->character = soul;
        ch->desc->original = ch;
        soul->desc = ch->desc;
        ch->desc = NULL;
        soul->comm = ch->comm;
        soul->lines = ch->lines;
        /*TOGGLE_BIT(soul->plr_flags, PLR_GHOST);*/
        char_act("       ..", soul);
        act(" $n    !", ch, NULL, NULL, TO_ROOM);

        check_improve(ch,gsn_soul,TRUE, 5);
        }
        else
        {
            char_act("      ...", ch);
            check_improve(ch,gsn_soul,FALSE,1);
        }
}
    
bool backstab_ok(CHAR_DATA *ch, CHAR_DATA *victim)
{
    if (victim->fighting) 
    {
        if (ch)
            char_act("       .", ch);
        return FALSE;
    }

    if (victim->hit < 7 * victim->max_hit / 10 && IS_AWAKE(victim)) 
    {
        if (ch)
            act("$N    ... "
                "    .",
                ch, NULL, victim, TO_CHAR);
        return FALSE;
    }

    //if (current_time - victim->last_fight_time < 300 && !IS_AFFECTED(victim, AFF_SLEEP)) 
    if (current_time - victim->last_fight_time < 300 && IS_AWAKE(victim)) 
    {
        if (ch)
            act("$N  ...     .",
                ch, NULL, victim, TO_CHAR);
        return FALSE;
    }

    if (victim && ch && !IS_NPC(victim))
    if (can_see(victim, ch) && IS_AWAKE(victim)) 
    {
        act("$N  .       ",
            ch, NULL, victim, TO_CHAR);
        return FALSE;
    }
    
    if (ch && IS_AFFECTED(ch, AFF_FEAR))
    {
        act("  !     $N?!", ch, NULL, victim, TO_CHAR);
        return FALSE;
    }

    return TRUE;
}

void backstab(CHAR_DATA *ch, CHAR_DATA *victim, int chance)
{
    if(check_mirror_image(ch,victim))
        return;     
    if (number_percent() < chance) 
    {
        check_improve(ch, gsn_backstab, TRUE, 1);
        one_hit(ch, victim, gsn_backstab, WEAR_WIELD);
        if (number_percent() <
                get_skill(ch, gsn_dual_backstab) * 8 / 10 &&
                get_eq_char(ch, WEAR_SECOND_WIELD)) 
        {
            check_improve(ch, gsn_dual_backstab, TRUE, 1);
            one_hit(ch, victim, gsn_dual_backstab, WEAR_SECOND_WIELD);
        }
        else 
        {
            check_improve(ch, gsn_dual_backstab, FALSE, 1);
        }
    }
    else 
    {
        check_improve(ch, gsn_backstab, FALSE, 1);
        damage(ch, victim, 0, gsn_backstab, DAM_NONE, TRUE);
    }

    if (!IS_NPC(victim) && victim->position == POS_FIGHTING) 
    {
        act_yell(victim, ", $i! ,  !", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
}

DO_FUN(do_backstab)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;

    one_argument(argument, arg, sizeof(arg));

    if (MOUNTED(ch)) 
    {
        char_act("         !", ch);
        return;
    }

    if ((chance = get_skill(ch, gsn_backstab)) == 0) 
    {
        char_act("     .", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL) 
    {
        char_act("  ,    .", ch);
        return;
    }

    if (attack_table[obj->value[3]].damage != DAM_PIERCE) 
    {
        char_act("         .", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        char_act("   ?", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_backstab)->beats);

    if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("      ?", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;

    if (!backstab_ok(ch, victim))
        return;

    backstab(ch, victim, chance);
}

DO_FUN(do_cleave)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    OBJ_DATA *obj;
    int chance;

    one_argument(argument, arg, sizeof(arg));

    if (ch->master != NULL && IS_NPC(ch))
        return;

    if ((chance = get_skill(ch, gsn_cleave)) == 0) 
    {
        char_act("    .", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        char_act(" ?", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_cleave)->beats);

    if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("      ?", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL) 
    {
        char_act("    .", ch);
        return;
    }

    if ((obj->value[0] != WEAPON_AXE) && (obj->value[0] != WEAPON_SWORD)) 
    {
        char_act("      .", ch);
        return;
    }

    if (victim->fighting != NULL) 
    {
        char_act("     .", ch);
        return;
    }

    if ((victim->hit < (0.9 * victim->max_hit))
        &&  (IS_AWAKE(victim))) 
    {
        act("$N    ...     .",
            ch, NULL, victim, TO_CHAR);
        return;
    }
    
    if (IS_AFFECTED(ch,AFF_FEAR))
    {
        char_act("  ?    ?!", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;
    
    if(check_mirror_image(ch,victim))
        return;

    if (!IS_AWAKE(victim)
        ||  IS_NPC(ch)
        ||  number_percent() < chance) 
    {
        check_improve(ch, gsn_cleave, TRUE, 1);
        one_hit(ch, victim, gsn_cleave,WEAR_WIELD);
    }
    else 
    {
        check_improve(ch, gsn_cleave, FALSE, 1);
        damage(ch, victim, 0, gsn_cleave, DAM_NONE, TRUE);
    }
    if (!IS_NPC(victim) && victim->position==POS_FIGHTING) 
    {
        act_yell(victim, ", $i,  !", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
}

DO_FUN(do_ambush)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    int chance;

    if (MOUNTED(ch)) {
        char_act("       !", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_ambush)) == 0) {
        char_act("     .", ch);
        return;
    }

    if (arg[0] == '\0') {
        char_act("   ?", ch);
        return;
    }
    
    if (IS_AFFECTED(ch,AFF_FEAR)){
        char_act("    !", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch)) {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_ambush)->beats);

    if ((victim = get_char_room(ch, arg)) == NULL) {
        char_act("  .", ch);
        return;
    }

    if (victim == ch) {
        char_act("     ?", ch);
        return;
    }

    if (victim->hit < victim->max_hit
    &&  can_see(victim, ch) && IS_AWAKE(victim)) {
        act("$N    ...     .",
            ch, NULL, victim, TO_CHAR);
        return;
    }

    if (is_safe(ch, victim))
        return;

    if(check_mirror_image(ch,victim))
            return;
    
    if (!IS_AWAKE(victim)
    ||  IS_NPC(ch)
    ||  number_percent() < chance) {
        check_improve(ch, gsn_ambush, TRUE, 1);
        one_hit(ch, victim, gsn_ambush,WEAR_WIELD);
    }
    else {
        check_improve(ch, gsn_ambush, FALSE, 1);
        damage(ch, victim, 0, gsn_ambush, DAM_NONE, TRUE);
    }
    if (!IS_NPC(victim) && victim->position==POS_FIGHTING) {
        act_yell(victim, "!      $i!", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
}

DO_FUN(do_rescue)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    CHAR_DATA *fch;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0') 
    {
        char_act(" ?", ch);
        return;
    }

    if (ch->move < 20 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_rescue)->beats);

    if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (victim == ch) 
    {
        char_act("  ?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_FEAR)) 
    {
        char_act(" ,  ?", ch);
        return;
    }

    if (!IS_NPC(ch) && IS_NPC(victim)) 
    {
        char_act("    !", ch);
        return;
    }

    if (ch->fighting == victim) 
    {
        char_act(" .", ch);
        return;
    }

    if ((fch = victim->fighting) == NULL) 
    {
        char_act("    .", ch);
        return;
    }

    if (!can_see(ch, fch)) 
    {
        char_act("    ,       .", ch);
        return;
    }

    if (IS_NPC(ch) && ch->master != NULL && IS_NPC(victim))
        return;

    if (ch->master != NULL && is_safe(ch->master, fch))
        return;

    if (is_affected(victim, gsn_mirror_image))
    {
        act("  $N,   .", 
                        ch, NULL, victim, TO_CHAR);
        return;
    }
    
    if (IS_SAMURAI(victim) && !IS_SAMURAI(ch))
    {
        act("$N     .", 
                        ch, NULL, victim, TO_CHAR);
        act("      $n.", 
                        ch, NULL, victim, TO_VICT);
        act("$N     $n.", 
                        ch, NULL, victim, TO_NOTVICT);
        return;
    }
            
    if (is_safe(ch, fch))
        return;

    if (!check_trust(ch, victim))
    {
        act_puts("$N     .", 
            ch, NULL, victim, TO_CHAR, POS_DEAD);
        return;
    }

    if (number_percent() > get_skill(ch,gsn_rescue)) 
    {
        char_act("   .", ch);
        check_improve(ch, gsn_rescue, FALSE, 1);
        return;
    }   

    act("  $N!",  ch, NULL, victim, TO_CHAR   );
    act("$n  !", ch, NULL, victim, TO_VICT   );
    act("$n  $N!",  ch, NULL, victim, TO_NOTVICT);
    check_improve(ch, gsn_rescue, TRUE, 1);

    stop_fighting(fch, FALSE);
    stop_fighting(victim, FALSE);

    set_fighting(ch, fch);
    set_fighting(fch, ch);
}

DO_FUN(do_kick)
{
    CHAR_DATA *victim;
    int kick_dam;
    int chance;
    int i;

    if (MOUNTED(ch)) {
        char_act("      !", ch);
        return;
    }

    if ((chance = get_skill(ch, gsn_kick)) == 0) {
        char_act("    .", ch);
        return;
    }

    if ((victim = ch->fighting) == NULL
    ||   victim->in_room != ch->in_room) {
        char_act("     .", ch);
        return;
    }

    if (ch->move < 10 && !IS_NPC(ch)) {
        char_act("  .", ch);
        return;
    }
    
    if (IS_AFFECTED(ch, AFF_FLYING))
        chance = chance * 11 / 10;

    WAIT_STATE(ch, SKILL(gsn_kick)->beats);

    if(check_mirror_image(ch,victim))
        return;

    if (IS_NPC(ch) || number_percent() < chance) {
        kick_dam = number_range(1, LVL(ch));
        if (IS_SAMURAI(ch) && (get_eq_char(ch, WEAR_FEET) == NULL))
            kick_dam *= 2;
        kick_dam += ch->damroll / 2;
        damage(ch, victim, kick_dam, gsn_kick, DAM_BASH, TRUE);
        check_improve(ch, gsn_kick, TRUE, 1);
/*            ((get_eq_char(ch, WEAR_FEET) == NULL)
            || (get_curr_stat(ch, STAT_DEX) > 23))) */
        if ((chance = get_skill(ch, gsn_master_kick)) !=0)
        {
            for (i = 1; i <= LVL(ch) / 20; i++)
                if(number_percent() < (chance / i))
                {
                    kick_dam = number_range(1, LVL(ch));
                    kick_dam += ch->damroll / 2 * i;
                    if (IS_NPC(victim))
                        kick_dam *= 2;
                    damage(ch, victim, kick_dam, gsn_kick, DAM_BASH, TRUE);
                    check_improve(ch, gsn_master_kick, TRUE, i);
                }
                else
                {
                    check_improve(ch, gsn_master_kick, FALSE, 1);
                    break; /* for */
                }
        }
    }
    else 
    {
        if ((IS_AFFECTED(victim, AFF_FLYING)&& !IS_AFFECTED(ch, AFF_FLYING)) 
            && ch->class == CLASS_THIEF)
        {
            chance = get_skill(ch, gsn_kick)*2/3 
                     + (get_carry_weight(ch) - get_carry_weight(ch))/100
                     - UMAX(get_curr_stat(victim,STAT_CON) - get_curr_stat(ch,STAT_CON), 0)*5
                     + (ch->size-victim->size)*5
                     - (is_affected (ch, gsn_protective_shield)? 15:0);
            if (number_percent() < chance)
            {
                char_act("   .", ch);
                DAZE_STATE(victim, SKILL(gsn_kick)->beats*3/4);          
            }
            //char_act("     .", ch);
        }
        damage(ch, victim, 0, gsn_kick, DAM_BASH, TRUE);
        check_improve(ch, gsn_kick, FALSE, 1);
    }
}

DO_FUN(do_circle)
{
    CHAR_DATA *victim;
    CHAR_DATA *rch;
    int chance;
    OBJ_DATA *obj;

    if (MOUNTED(ch)) {
        char_act("        !", ch);
        return;
    }

    if ((chance = get_skill(ch, gsn_circle)) == 0) {
        char_act("         .", ch);
        return;
    }

    if ((victim = ch->fighting) == NULL) {
        char_act("     .", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL
    ||  attack_table[obj->value[3]].damage != DAM_PIERCE) {
         char_act("       .", ch);
         return;
    }

    if (is_safe(ch, victim))
        return;

    if (ch->move < 20 && !IS_NPC(ch)) {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_circle)->beats);

    for (rch = ch->in_room->people; rch; rch = rch->next_in_room) {
        if (rch->fighting == ch) {
            char_act("     ,   .", ch);
            return;
        }
    }

    if(check_mirror_image(ch,victim))
        return;

    if (number_percent() < chance) {
        one_hit(ch, victim, gsn_circle, WEAR_WIELD);
        check_improve(ch, gsn_circle, TRUE, 1);
    }
    else {
        damage(ch, victim, 0, gsn_circle, TYPE_UNDEFINED, TRUE);
        check_improve(ch, gsn_circle, FALSE, 1);
    }
}

DO_FUN(do_disarm)
{
    CHAR_DATA *victim;
    OBJ_DATA *wield;
    OBJ_DATA *vwield;
    int chance, ch_weapon, vict_weapon;
    int loc = WEAR_WIELD;
    char arg[MAX_INPUT_LENGTH];
    int hth = 0;

    if (ch->master != NULL && IS_NPC(ch))
        return;

    if ((chance = get_skill(ch, gsn_disarm)) == 0) 
    {
        char_act("  ,   .", ch);
        return;
    }

    if ((wield = get_eq_char(ch, WEAR_WIELD)) == NULL
    && (hth = get_skill(ch, gsn_hand_to_hand)) == 0) 
    {
        char_act("  .", ch);
        return;
    }

    if ((victim = ch->fighting) == NULL) 
    {
        char_act("      !", ch);
        return;
    }

    if (ch->move < 30 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }
    
    argument = one_argument(argument, arg, sizeof(arg));
    if (arg[0] && !str_prefix(arg, "second"))
        loc = WEAR_SECOND_WIELD;

    if ((vwield = get_eq_char(victim, loc)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    /* find weapon skills */
    ch_weapon = get_weapon_skill(ch, get_weapon_sn(wield));
    vict_weapon = get_weapon_skill(victim, get_weapon_sn(vwield));

    /* modifiers */

    /* skill */
    if (wield == NULL)
        chance = chance * hth/150;
    else
        chance = chance * ch_weapon/100;

    chance += (ch_weapon/2 - vict_weapon) / 2;

    /* dex vs. strength */
    chance += get_curr_stat(ch, STAT_DEX);
    chance -= 2 * get_curr_stat(victim, STAT_STR);

    /* level */
    chance += (LVL(ch) - LVL(victim)) * 2;

    /* and now the attack */
    WAIT_STATE(ch, SKILL(gsn_disarm)->beats);

    if(check_mirror_image(ch,victim))
        return;
    
    if (IS_NPC(ch)) chance = chance * 2 / 3;

    if (number_percent() < chance) 
    {
        disarm(ch, victim, vwield);
        check_improve(ch, gsn_disarm, TRUE, 1);
    }
    else {
        act("     $c3{$N}.", ch, NULL, victim, TO_CHAR);
        act("$n     .",
            ch, NULL, victim, TO_VICT);
        act("$n   $c3{$N},   .",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_disarm, FALSE, 1);
    }
}

DO_FUN(do_nerve)
{
    CHAR_DATA *victim;
    char arg[MAX_INPUT_LENGTH];
    int chance;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_nerve)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("    ?    ?!", ch);
        return;
    }

    if (MOUNTED(ch)) {
        char_act(" ,        .", ch);
        return;
    }

    if (!(FREE_HAND1(ch) || FREE_HAND2(ch))) {
        char_act("      .", ch);
        return;
    }

    if (arg[0] == '\0') {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room) {
            char_act("      !", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL) {
        char_act("  .", ch);
        return;
    }
    
    if (ch == victim) {
        char_act("   !", ch);
        return;
    }

       if (is_affected(victim,gsn_nerve)) {
        char_act("       .", ch);
        return;
    }

        if (is_safe(ch, victim))
        return;

    if (ch->fighting)
        chance /= 3;

    if (ch->move < 30 && !IS_NPC(ch)) {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_nerve)->beats);

    if(check_mirror_image(ch,victim))
        return;

    if (ch->style && ch->style == STYLE_COBRA)
        chance *= 2;

    if (IS_NPC(ch)
    ||  number_percent() <  4 * chance / 5 + LVL(ch) - LVL(victim) 
                    + get_curr_stat(ch,STAT_DEX)
                    - get_curr_stat(victim,STAT_DEX))  {

        AFFECT_DATA af;
        af.where    = TO_AFFECTS;
        af.type     = gsn_nerve;
        af.level    = LVL(ch);
        af.duration = 10;
        af.location = APPLY_STR;
        af.modifier = (15)*(-1);
        af.bitvector    = 0;

        affect_join(victim, &af);
        act("  $N,      $gn{} .",
            ch, NULL, victim, TO_CHAR);
        act("$n  ,       .",
                    ch, NULL, victim, TO_VICT);
        act("$n  $N,      $gn{} .",
            ch, NULL, victim, TO_NOTVICT);
        char_act(" ,    ...", victim);
        check_improve(ch, gsn_nerve, TRUE, 1);
    }
    else {
        char_act("     .", ch);
        act("$n   ,         .",
            ch, NULL, victim, TO_VICT);
        act("$n   $N,      $gn{} ,  .",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_nerve, FALSE, 1);
    }

    if (IS_AWAKE(victim))
        multi_hit(victim,ch,TYPE_UNDEFINED);
}


DO_FUN(do_tame)
{
    CHAR_DATA *victim;
    char arg[MAX_INPUT_LENGTH];
    int chance;

    if ((chance = get_skill(ch, gsn_tame)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        if (ch->fighting != NULL)
            victim = ch->fighting;
        else {
            char_act("You are beyond taming.\n", ch);
            act("$n tries to tame $gn{himself} but fails miserably.",
                ch, NULL, NULL, TO_ROOM);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("They're not here.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_tame)->beats);

    if (IS_NPC(ch))
    {
        char_act("Why don't you tame yourself first?", ch);
        return;
    }

    if (!IS_NPC(victim))
    {
        act("$N is beyond taming.", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (!IS_SET(victim->pIndexData->act, ACT_AGGRESSIVE)
    && (victim->fighting == NULL))
    {
        act("$N is not usually aggressive.", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (!IS_SET(victim->pIndexData->act, ACT_AGGRESSIVE))
        chance /= 4;
    if (number_percent() < chance + 15 + 4*(LVL(ch) - LVL(victim)))
    {
        SET_BIT(victim->affected_by, AFF_CALM);
        char_act("You calm down.", victim);
        act("You calm $N down.", ch, NULL, victim, TO_CHAR);
        act("$n calms $N down.", ch, NULL, victim, TO_NOTVICT);
        stop_fighting(victim, TRUE);
        check_improve(ch, gsn_tame, TRUE, 1);
    } else
    {
        char_act("You failed.",ch);
        act("$n tries to calm down $N but fails.", ch, NULL, victim, TO_NOTVICT);
        act("$n tries to calm you down but fails.", ch, NULL, victim, TO_VICT);
        check_improve(ch, gsn_tame, FALSE, 1);
    }
}

DO_FUN(do_assassinate)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    int chance;

    if (MOUNTED(ch))
    {
        char_act("You can't assassinate while riding!", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if (ch->master != NULL && IS_NPC(ch))
        return;

    if ((chance = get_skill(ch, gsn_assassinate)) == 0)
    {
        char_act("You don't know how to assassinate.", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        char_act("You don't want to kill your beloved master.", ch);
        return;
    }

    if (arg[0] == '\0')
    {
        char_act("Assassinate whom?", ch);
        return;
    }
    
    if (IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("  ?    ?!", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
        char_act("You are too tired.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_assassinate)->beats);

    if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("They aren't here.", ch);
        return;
    }

    if (victim == ch)
    {
        char_act("Suicide is against your way.", ch);
        return;
    }

    if ((IS_IMMORTAL(victim) && !IS_NPC(victim))
        || IS_SET(victim->abilities, EA_IMMUN_INSTANT))
    {
        char_act("Your hands pass through.", ch);
        return;
    }

    if (victim->fighting != NULL)
    {
        char_act("You can't assassinate a fighting person.", ch);
        return;
    }

    if (!FREE_HAND1(ch) || !FREE_HAND2(ch))
    {
        char_act("You need both hands free to assassinate somebody.", ch);
        return;
    }

    if (victim->hit < victim->max_hit
    &&  can_see(victim, ch) && IS_AWAKE(victim))
    {
        act("$N is hurt and suspicious ... you can't sneak up.",
            ch, NULL, victim, TO_CHAR);
        return;
    }

    if (IS_SET(victim->immunes, IMM_BASH))
    {
        act("$N seems immune to your assassination attempt.",
            ch, NULL, victim, TO_CHAR);
        act("$N seems immune to $n's assassination attempt.",
            ch, NULL, victim, TO_ROOM);
        return;
    }

    if (is_safe(ch, victim))
        return;

    if(check_mirror_image(ch,victim))
        return;

    if (number_percent() < chance
    && !IS_CLAN_GUARD(victim)
    && !IS_IMMORTAL(victim))
    {
        one_hit(ch, victim, gsn_assassinate,WEAR_WIELD);
    } else
    {
        check_improve(ch, gsn_assassinate, FALSE, 1);
        damage(ch, victim, 0, gsn_assassinate, DAM_NONE, TRUE);
    }
    if (!IS_NPC(victim) && victim->position == POS_FIGHTING)
    {
        act_yell(victim, "Help! $i tries to assasinate me!", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
}

DO_FUN(do_caltrops)
{
    CHAR_DATA *victim = ch->fighting;
    int chance;

    if ((chance = get_skill(ch, gsn_caltrops)) == 0)
    {
        char_act("Caltrops? Is that a dance step?", ch);
        return;
    }

    if (victim == NULL
    || victim->in_room != ch->in_room)
    {
        char_act("You must be in combat.", ch);
        return;
    }

    if (is_safe(ch,victim))
        return;

    if (IS_AFFECTED(victim, AFF_FLYING))
    {
        act(" $N   .", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (ch->move < 20 && !IS_NPC(ch))
    {
        char_act("You are too tired.", ch);
        return;
    }

    act("You throw a handful of sharp spikes at the feet of $N.",
        ch, NULL, victim, TO_CHAR);
    act("$n throws a handful of sharp spikes at your feet!",
        ch, NULL, victim, TO_VICT);

    WAIT_STATE(ch, SKILL(gsn_caltrops)->beats);

    if (!IS_NPC(ch) && number_percent() >= chance)
    {
        damage(ch, victim, 0, gsn_caltrops, DAM_PIERCE, TRUE);
        check_improve(ch, gsn_caltrops, FALSE, 1);
        return;
    }

    if(check_mirror_image(ch,victim))
            return;
    
    damage(ch, victim, LVL(ch), gsn_caltrops, DAM_PIERCE, TRUE);
    if (JUST_KILLED(victim))
        return;

    if (!is_affected(victim, gsn_caltrops))
    {
        AFFECT_DATA tohit, todam, todex;

        tohit.where     = TO_AFFECTS;
        tohit.type      = gsn_caltrops;
        tohit.level     = LVL(ch);
        tohit.duration  = -1;
        tohit.location  = APPLY_HITROLL;
        tohit.modifier  = -5;
        tohit.bitvector = 0;
        affect_to_char(victim, &tohit);

        todam.where = TO_AFFECTS;
        todam.type = gsn_caltrops;
        todam.level = LVL(ch);
        todam.duration = -1;
        todam.location = APPLY_DAMROLL;
        todam.modifier = -5;
        todam.bitvector = 0;
        affect_to_char(victim, &todam);

        todex.type = gsn_caltrops;
        todex.level = LVL(ch);
        todex.duration = -1;
        todex.location = APPLY_DEX;
        todex.modifier = -5;
        todex.bitvector = 0;
        affect_to_char(victim, &todex);

        act("$N starts limping.", ch, NULL, victim, TO_CHAR);
        act("You start to limp.", ch, NULL, victim, TO_VICT);
        check_improve(ch, gsn_caltrops, TRUE, 1);
    }
}

void do_strangle (CHAR_DATA* ch, const char* argument)
{
    CHAR_DATA* victim ;
    AFFECT_DATA af ;
    int chance, dt ;

    if (MOUNTED (ch))
    {
        char_act("    ,    !", ch);
        return ;
    }

    if ((chance = get_skill(ch, gsn_strangle)) == 0)
    {
        char_act("      .", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM))
    {
        char_act("      !", ch);
        return ;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
        char_act("  .", ch);
        return ;
    }

    WAIT_STATE (ch, MISSING_TARGET_DELAY);

    if ((victim = get_char_room(ch, argument)) == NULL)
    {
        char_act("  .", ch);
        return ;
    }

    if (ch == victim)
    {
        char_act("  ,    !", ch);
        return ;
    }

    if (victim->fighting)
    {
        char_act("You can't strangle a fighting person.", ch);
        return ;
    }

    if (MOUNTED (victim))
    {
        char_act("    !", ch);
        return ;
    }

    if (!FREE_HAND1(ch) || !FREE_HAND2(ch))
    {
        char_act("    ,   .", ch);
        return ;
    }
    
    if (IS_AFFECTED (ch, AFF_FEAR))
    {
        char_act("You too scare to do this!    ?", ch);
        return ;
    }   

    if (is_affected (victim, gsn_insomnia))
    {
        act(" $N   .", ch, NULL, victim, TO_CHAR);
        return ;
    }   

    WAIT_STATE (ch, SKILL(gsn_strangle)->beats);

    if (is_affected (victim, gsn_strangle)) return ;
    if (is_safe (ch, victim))               return ;

    chance  = chance / 2 ;
    chance += URANGE (0, (get_curr_stat(ch, STAT_DEX) - 20) * 2, 10);
    chance += can_see (victim, ch) ? 0 : 33 ;

    if ((dt = current_time - victim->last_fight_time) < 200)
        chance -= dt < 200 ? (200 - dt) / 5 : 0 ;

    if (current_time - ch->last_fight_time < 200) 
        chance -= 5 ;

    if (IS_NPC (victim) && victim->pIndexData->pShop != NULL)
        chance -= 40 ;

    chance += (ch->size - victim->size) * 3 ;
    chance += (LVL (ch) - LVL (victim)) * 4 ;
    chance += (get_curr_stat(ch, STAT_STR) - get_curr_stat(victim, STAT_STR)) * 2 ;
    chance += (get_curr_stat(ch, STAT_DEX) - get_curr_stat(victim, STAT_DEX)) * 2 ;

    if (is_affected (ch, gsn_protective_shield)) chance -= 15 ; else
    if (is_affected (ch, gsn_shield))            chance -= 5 ;

    if (IS_NPC (victim)) chance += 10 ;

    if (check_perception (victim, ch)) chance = 0 ;
    
    SET_FIGHT_TIME (victim) ;
    SET_FIGHT_TIME (ch)     ;
   
    if (check_mirror_image (ch, victim))
    { 
       //The characters begin battle and must be visible
       do_visible (ch, str_empty)     ;
       do_visible (victim, str_empty) ;
       return ;
    }
     
    if (number_percent () < chance 
    && !IS_CLAN_GUARD (victim) 
    && !IS_IMMORTAL (victim) 
    && !IS_UNDEAD (victim))
    {
        act("  $gN{}   $N  .",
             ch, NULL, victim, TO_CHAR);

        act("$n       .",
             ch, NULL, victim, TO_VICT);

        act("$n  $N.",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_strangle, TRUE, 1);
        
        af.type      = gsn_strangle ;
        af.where     = TO_AFFECTS ;
        af.level     = LVL(ch);
        // af.duration  = LVL(ch) / 15 + 1 ;
        af.duration  = IS_SET (muddy_mode, MUDDY_RANDOM_SLEEP) ? 
                     number_range (0, LVL(ch) / 15 + 1) :  // new value
                     LVL(ch) / 15 + 1;                     // old value
        af.location  = APPLY_NONE ;
        af.modifier  = 0 ;
        af.bitvector = AFF_SLEEP ;
        affect_join (victim, &af);

        if (IS_AWAKE (victim)) victim->position = POS_SLEEPING ;
       
        //The characters begin battle and must be visible
        do_visible (ch, str_empty)     ;
        do_visible (victim, str_empty) ;       
    }
    else
    {
        damage (ch, victim, 0, gsn_strangle, DAM_NONE, TRUE);
        check_improve (ch, gsn_strangle, FALSE, 1);

        if (!IS_NPC (victim))
        {
            act_yell(victim, "! $i  !", ch, NULL);
            agent_net_printf (victim, ch, ANET_ATTACK);
        }
    }
}

void do_blackjack (CHAR_DATA* ch, const char* argument)
{
    CHAR_DATA* victim ;
    AFFECT_DATA af ;
    int chance, dt ;

    if (MOUNTED (ch))
    {
        char_act("You can't blackjack while riding!", ch);
        return ;
    }

    if ((chance = get_skill (ch, gsn_blackjack)) == 0)
    {
        char_act("Huh?", ch);
        return ;
    }

    if (ch->move < 40 && !IS_NPC(ch))
    {
        char_act("You're too tired.", ch);
        return ;
    }

    if ((victim = get_char_room(ch, argument)) == NULL)
    {
        char_act("You do not see that person here.", ch);
        WAIT_STATE (ch, MISSING_TARGET_DELAY);
        return ;
    }

    if (MOUNTED (victim))
    {
        char_act("     ,    !", ch);
        return ;
    }

    if (victim->fighting)
    {
        char_act("You can't blackjack a fighting person.", ch);
        return ;
    }

    if (ch == victim)
    {
        char_act("You idiot?! Blackjack your self?!", ch);
        return ;
    }

    if (IS_AFFECTED (ch, AFF_CHARM)) 
    {
        char_act("You don't want to hit your beloved masters' head with a full filled jack.", ch);
        return ;
    }

    if (IS_AFFECTED (victim, AFF_SLEEP))
    {
        act("$gN{}   .", ch, NULL, victim, TO_CHAR);
        return ;
    }
    
    if (IS_AFFECTED (ch, AFF_FEAR))
    {
        char_act("You too scare to do this!    ! ", ch);
        return ;
    }

    if (is_affected (victim, gsn_insomnia))
    {
        act("But $N could not sleep.", ch, NULL, victim, TO_CHAR);
        return ;
    }   
 
    if (IS_UNDEAD(victim))
    {
          act("$N is undead, it's pointless to struck his head!",
                ch,NULL,victim, TO_CHAR);
          return;
    }

    
    WAIT_STATE (ch, SKILL(gsn_blackjack)->beats);

    if (is_safe (ch,victim)) return ;

    chance  = chance /2 ;
    chance += can_see (victim, ch) ? 0 : 33 ;
    
    if ((dt = current_time - victim->last_fight_time) < 200)
        chance -= dt < 200 ? (200 - dt) / 5 : 0 ;

    //    . ӣ     .

    if ((current_time - ch->last_fight_time)     > 200 &&
        (current_time - victim->last_fight_time) > 200) chance *= 2 ;

    //  .   ,   -   .

    if (current_time - ch->last_fight_time < 200) chance -= 5 ;
    if (IS_NPC(victim) && victim->pIndexData->pShop != NULL) chance -= 40 ;

    chance += (ch->size - victim->size) * 3 ;
    chance += (LVL (ch) - LVL (victim)) * 4 ;
    chance += (get_curr_stat(ch, STAT_DEX) - get_curr_stat(victim, STAT_DEX)) * 2 ;

    if (is_affected (ch, gsn_protective_shield)) chance -= 15 ; else
    if (is_affected (ch, gsn_shield))            chance -= 5 ;

    if (IS_NPC(victim)) chance += 10 ;

    if (check_perception (victim, ch)) chance = 0 ;
    
    SET_FIGHT_TIME (victim);
    SET_FIGHT_TIME (ch);
   
    if (check_mirror_image (ch, victim))
     {  
        //The characters begin battle and must be visible
        do_visible (ch, str_empty)     ;
        do_visible (victim, str_empty) ;
        return ;
     }
       
    if (number_percent () < chance &&
        !IS_CLAN_GUARD (victim)    &&
        !IS_IMMORTAL   (victim)    &&
        !IS_UNDEAD     (victim))
    {
        act("  $N    .",
             ch, NULL, victim, TO_CHAR);

            act("     !",
             ch, NULL, victim, TO_VICT);

        act("$n  $N    !  *!*",
             ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_blackjack, TRUE, 1);
        
        af.type      = gsn_blackjack ;
        af.where     = TO_AFFECTS ;
        af.level     = LVL(ch);
        //af.duration  = LVL(ch) / 15 + 1 ;
        af.duration  = IS_SET (muddy_mode, MUDDY_RANDOM_SLEEP) ? 
                     number_range (0, LVL(ch) / 15 + 1) :  // new value
                     LVL(ch) / 15 + 1;                     // old value
        af.location  = APPLY_NONE ;
        af.modifier  = 0 ;
        af.bitvector = AFF_SLEEP ;
        affect_join (victim, &af);

        if (IS_AWAKE (victim)) victim->position = POS_SLEEPING ;
       
        //The characters begin battle and must be visible
        do_visible (ch, str_empty)     ;
        do_visible (victim, str_empty) ;
    }
    else
    {
        damage (ch, victim, 0, gsn_blackjack, DAM_NONE, TRUE);
        check_improve (ch, gsn_blackjack, FALSE, 1);

        if (!IS_NPC (victim))
        {
            act_yell(victim, "! $i   !", ch, NULL);
            agent_net_printf (victim, ch, ANET_ATTACK);
        }
    }
}

DO_FUN(do_bloodthirst)
{
    int chance, hp_percent;

    if ((chance = get_skill(ch, gsn_bloodthirst)) == 0)
    {
        char_act("You're not that thirsty.", ch);
        return;
    }
    if (IS_AFFECTED(ch, AFF_BLOODTHIRST))
    {
        char_act("Your thirst for blood continues.", ch);
        return;
    }
    if (IS_AFFECTED(ch, AFF_CALM))
    {
        char_act("You're feeling to mellow to be bloodthirsty.", ch);
        return;
    }
    if (ch->fighting == NULL)
    {
        char_act("You need to be fighting.", ch);
        return;
    }

    /* modifiers */

    hp_percent = 100 * ch->hit/ch->max_hit;
    chance += 25 - hp_percent/2;

    if (number_percent() < chance)
    {
        AFFECT_DATA af;

        WAIT_STATE(ch, PULSE_VIOLENCE);

        char_act("You hunger for blood!", ch);
        act("$n gets a bloodthirsty look in $gn{his} eyes.",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_bloodthirst, TRUE, 2);

        af.where    = TO_AFFECTS;
        af.type     = gsn_bloodthirst;
        af.level    = LVL(ch);
        af.duration = 2 + LVL(ch) / 18;
        af.modifier = 6 + LVL(ch) / 4;
        af.bitvector    = AFF_BLOODTHIRST;

        af.location = APPLY_HITROLL;
        affect_to_char(ch, &af);

        af.location = APPLY_DAMROLL;
        affect_to_char(ch, &af);

        af.modifier = - UMIN(LVL(ch) - 5,35);
        af.location = APPLY_AC;
        affect_to_char(ch, &af);
    } else
    {
        WAIT_STATE(ch,3 * PULSE_VIOLENCE);
        char_act("You feel bloodthirsty for a moment, but it passes.", ch);
        check_improve(ch, gsn_bloodthirst, FALSE, 2);
    }
}

DO_FUN(do_spellbane)
{
    int chance;
    char arg[MAX_INPUT_LENGTH];

    one_argument(argument, arg, sizeof(arg));
    if (ch->level < LEVEL_IMMORTAL)
    {
        char_act("Huh?", ch);
        return;
    }
    if ((chance = get_skill(ch, gsn_spellbane)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_SPELLBANE))
    {   
        if ((arg[0] != 0) && !str_prefix(arg, "remove")) 
            affect_bit_strip(ch, TO_AFFECTS, AFF_SPELLBANE); 
        else 
            char_act("You are already deflecting spells.", ch); 
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_spellbane)->beats); 
    set_spellbane(ch, str_empty); 
}

DO_FUN(set_spellbane)
{
    AFFECT_DATA af;

    if (IS_AFFECTED(ch, AFF_SPELLBANE)) return;

    af.where        = TO_AFFECTS;
    af.type         = gsn_spellbane;
    af.level        = LVL(ch);
    af.duration     = -1;
    af.location     = APPLY_HITROLL;
    af.modifier     = 20;
    af.bitvector    = AFF_SPELLBANE;

    affect_to_char(ch, &af);

    act("Your hatred of magic surrounds you.", 
                    ch, NULL, NULL, TO_CHAR);
    act("$n fills the air with $gn{his} hatred of magic.",
                    ch, NULL, NULL, TO_ROOM);
}

DO_FUN(do_resistance)
{
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_resistance)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_RESISTANCE))
    {
        char_act("You are as resistant as you will get.", ch);
        return;
    }

    mana = SKILL(gsn_resistance)->min_mana;
    if (ch->mana < mana)
    {
        char_act("You cannot muster up the energy.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_resistance)->beats);

    if (number_percent() < chance)
    {
        AFFECT_DATA af;

        af.where        = TO_AFFECTS;
        af.type         = gsn_resistance;
        af.level        = LVL(ch);
        af.duration     = LVL(ch)/4;
        af.location     = APPLY_DAMROLL;
        af.modifier     = 20;
        af.bitvector    = AFF_RESISTANCE;

        affect_to_char(ch, &af);
        ch->mana -= mana;

        act("You feel tough!", ch, NULL, NULL, TO_CHAR);
        act("$n looks tougher.", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_resistance, TRUE, 1);
    } else
    {
        ch->mana -= mana/2;

        char_act("You flex your muscles, but you don't feel tougher.", ch);
        act("$n flexes $gn{his} muscles, trying to look tough.",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_resistance, FALSE, 1);
    }
}

DO_FUN(do_demon)
{
    int chance;
    int mana;
    int demon_gsn;
    char arg[MAX_INPUT_LENGTH];

    argument = one_argument(argument, arg, sizeof(arg));
    demon_gsn = 0;

    if (!str_cmp(arg, "fire"))  demon_gsn = gsn_demon_fire;
    if (!str_cmp(arg, "water")) demon_gsn = gsn_demon_water;
    if (!str_cmp(arg, "earth")) demon_gsn = gsn_demon_earth;
    if (!str_cmp(arg, "air"))   demon_gsn = gsn_demon_air;

    if (demon_gsn == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if ((chance = get_skill(ch, demon_gsn)) == 0)
    {
        char_act("?", ch);
        return;
    }

    if (IS_DEMON(ch))
    {
        char_act("  .", ch);
        return;
    }

    if (is_affected(ch, gsn_demon_time))
    {
        char_act("   .", ch);
        return;
    }

    mana = SKILL(demon_gsn)->min_mana;
    if (ch->mana < mana)
    {
        char_act("     .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(demon_gsn)->beats);

    if (number_percent() < chance)
    {
        AFFECT_DATA af;

        af.where    = TO_RESIST;
        af.type     = demon_gsn;
        af.level    = LVL(ch);
        af.duration = LVL(ch)/4;
        af.location = APPLY_NONE;
        af.modifier = 0;

        if (demon_gsn == gsn_demon_fire) 
        {
            af.bitvector    = DAM_COLD;
            af.modifier     = -60;
            affect_to_char(ch, &af);
            af.bitvector    = DAM_FIRE;
            af.modifier     = 40;
            affect_to_char(ch, &af);
/*
            af.where        = TO_AFFECTS;
            af.bitvector    = AFF_INFRARED | AFF_FAERIE_FIRE;
            affect_to_char(ch, &af);
*/
            af.bitvector = FORM_DEMON_FIRE;         
        }
        if (demon_gsn == gsn_demon_water) 
        {
            af.bitvector    = DAM_FIRE;
            af.modifier     = -60;
            affect_to_char(ch, &af);
            af.bitvector    = DAM_SLASH;
            af.modifier     = 40;
            affect_to_char(ch, &af);
/*
            af.where        = TO_AFFECTS;
            af.bitvector    = AFF_SWIM;
            affect_to_char(ch, &af);
*/
            af.bitvector = FORM_DEMON_WATER;
        }
        if (demon_gsn == gsn_demon_earth) 
        {
            af.bitvector    = DAM_ACID;
            af.modifier     = -60;
            affect_to_char(ch, &af);
            af.bitvector    = DAM_BASH;
            af.modifier     = 40;
            affect_to_char(ch, &af);

            af.bitvector = FORM_DEMON_EARTH;
        }
        if (demon_gsn == gsn_demon_air) 
        {
            af.bitvector    = DAM_LIGHTNING;
            af.modifier     = -60;
            affect_to_char(ch, &af);
            af.bitvector    = DAM_PIERCE;
            af.modifier     = 40;
            affect_to_char(ch, &af);
/*
            af.where        = TO_AFFECTS;
            af.bitvector    = AFF_FLYING | AFF_PASS_DOOR;
            affect_to_char(ch, &af);
*/
            af.bitvector = FORM_DEMON_AIR;
         }
        af.where = TO_FORM;
        affect_to_char(ch, &af);

        af.where        = TO_AFFECTS;
        af.type         = demon_gsn;
        af.level        = LVL(ch);
        af.duration     = LVL(ch)/4;
//        af.modifier     = UMAX(1, LVL(ch) / 2);
        af.modifier     = 0;

        af.bitvector    = AFF_DETECT_GOOD;
        af.location     = APPLY_HITROLL;
        affect_to_char(ch,&af);

        af.bitvector    = AFF_DETECT_EVIL;
        af.location     = APPLY_DAMROLL;
        affect_to_char(ch,&af);
/*
        af.location = APPLY_SIZE;
        af.modifier = number_range(1,2);
        af.bitvector    = 0;
        affect_to_char(ch,&af);
*/
        ch->mana -= mana;

        act("   !", ch, NULL, NULL, TO_CHAR);
        act("$n   !", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, demon_gsn, TRUE, 1);
    }
    else {
        ch->mana -= mana/2;

        char_act("    ,       .", ch);
        act("$n    ,       .", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, demon_gsn, FALSE, 1);
    }
}

DO_FUN(do_trophy)
{
    int trophy_vnum;
    OBJ_DATA *trophy;
    AFFECT_DATA af;
    OBJ_DATA *part;
    char arg[MAX_INPUT_LENGTH];
    int level;
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_trophy)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    WAIT_STATE(ch, MISSING_TARGET_DELAY);

    if (is_affected(ch, gsn_trophy))
    {
        char_act("But you've already got one trophy!", ch);
        return;
    }

    mana = SKILL(gsn_trophy)->min_mana;
    if (ch->mana < mana)
    {
        char_act("You feel too weak to concentrate on a trophy.", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        char_act("Make a trophy of what?", ch);
        return;
    }

    if ((part = get_obj_carry(ch, arg)) == NULL)
    {
        char_act("You do not have that body part.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_trophy)->beats);

    if (part->pIndexData->vnum == OBJ_VNUM_SLICED_ARM
    ||  part->pIndexData->vnum == OBJ_VNUM_SLICED_LEG
    ||  part->pIndexData->vnum == OBJ_VNUM_SEVERED_HEAD
    ||  part->pIndexData->vnum == OBJ_VNUM_TORN_HEART
    ||  part->pIndexData->vnum == OBJ_VNUM_GUTS)
        trophy_vnum = OBJ_VNUM_BATTLE_PONCHO;
    else if (part->pIndexData->vnum == OBJ_VNUM_BRAINS)
    {
        char_act("Why don't you just eat those instead?", ch);
        return;
    } else
    {
        char_act("You can't make a trophy out of that!", ch);
        return;
    }

    if (IS_NULLSTR(part->owner) || part->timer == 0)
    {
        char_act("Invalid body part.", ch);
        return;
    }

    if (number_percent() < chance * 2 / 3)
    {
        char_act("You failed and destroyed it.", ch);
        extract_obj(part);
        return;
    }

    if (!IS_NPC(ch) && number_percent() < chance) 
    {
        af.where     = TO_AFFECTS;
        af.type      = gsn_trophy;
        af.level     = LVL(ch);
        af.duration  = LVL(ch)/2;
        af.modifier  = 0;
        af.bitvector = 0;
        af.location  = 0;

        affect_to_char(ch,&af);

        if (trophy_vnum != 0) 
        {
            level = UMIN(part->level + 5, MAX_LEVEL - 1);

            trophy = create_named_obj(get_obj_index(trophy_vnum), level, part->owner);
            trophy->timer   = 0;
            trophy->cost    = 0;
            trophy->level   = ch->level;
            trophy->owner   = str_dup(ch->name);

            ch->mana -= mana;
/*
            af.where        = TO_OBJECT;
            af.type         = gsn_trophy;
            af.level        = level;
            af.duration     = -1;
            af.location     = APPLY_DAMROLL;
            af.modifier     = LVL(ch) / 5;
            af.bitvector    = 0;
            affect_to_obj(trophy, &af);

            af.location     = APPLY_HITROLL;
            af.modifier     = LVL(ch) / 5;
            af.bitvector    = 0;
            affect_to_obj(trophy, &af);

            af.location     = APPLY_INT;
            af.modifier     = level > 20 ? -2 : -1;
            affect_to_obj(trophy, &af);

            af.location     = APPLY_STR;
            af.modifier     = level > 20 ? 2 : 1;
            affect_to_obj(trophy, &af);
*/
            trophy->value[3] = LVL(ch) * 2 + 18;
            trophy->value[4] = LVL(ch) * 2 + 18;

            obj_to_char(trophy, ch);
            check_improve(ch, gsn_trophy, TRUE, 1);

            act("You make a poncho from $p!",
                ch, part, NULL, TO_CHAR);
            act("$n makes a poncho from $p!",
                ch, part, NULL, TO_ROOM);

            extract_obj(part);
            return;
        }
    } else 
    {
        char_act("You destroyed it.", ch);
        extract_obj(part);
        ch->mana -= mana/2;
        check_improve(ch, gsn_trophy, FALSE, 1);
    }
}

DO_FUN(do_truesight)
{
    int chance;

    if ((chance = get_skill(ch, gsn_truesight)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_truesight))
    {
        char_act("Your eyes are as sharp as they can get.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_truesight)->beats);

    if (number_percent() < chance)
    {
        AFFECT_DATA af;

        af.where    = TO_AFFECTS;
        af.type     = gsn_truesight;
        af.level    = LVL(ch);
        if (!is_clan_item_ok(ch))
            af.duration = number_range(1, 4);
        else
            af.duration = LVL(ch) / 2 + 5;
        af.location = APPLY_NONE;
        af.modifier = 0;

        af.bitvector = AFF_DETECT_INVIS;
        affect_to_char(ch, &af);

        if (!is_clan_item_ok(ch))
            af.duration = number_range(8, 16);

        af.bitvector = AFF_DETECT_MAGIC;
        affect_to_char(ch,&af);

        if (is_clan_item_ok(ch))
        {
            af.bitvector = AFF_DETECT_HIDDEN;
            affect_to_char(ch, &af);

            af.bitvector = AFF_DETECT_IMP_INVIS;
            affect_to_char(ch,&af);

            af.bitvector = AFF_ACUTE_VISION;
            affect_to_char(ch,&af);
        }

        act("You look around sharply!", ch, NULL, NULL, TO_CHAR);
        act("$n looks more enlightened.", ch, NULL, NULL, TO_ROOM);
        check_improve(ch,gsn_truesight,TRUE,1);
    } else
    {
        char_act("You look about sharply, but you don't see anything new.",ch);
        act("$n looks around sharply but doesn't seem enlightened.",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_truesight, FALSE, 1);
    }
}

DO_FUN(do_warcry)
{
    AFFECT_DATA af;
    int chance;
    int need, chars;

    if ((chance = get_skill(ch, gsn_warcry)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_warcry)) 
    {
        char_act("    .", ch);
        return;
    }

    need = SKILL(gsn_warcry)->min_mana;
    chars = IS_SET (muddy_mode, MUDDY_WARR_MOVES) ? ch->move : ch->mana;

    if (chars < need && !IS_NPC(ch)) 
    {
        char_act("You don't have enough energy.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_warcry)->beats);

    if (number_percent() > chance) 
    {
        char_act("You grunt softly.", ch);
        act("$n makes some soft grunting noises.", ch, NULL, NULL, TO_ROOM);
        if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
            ch->move -= need/2;
        else
            ch->mana -= need/2;
        check_improve(ch, gsn_warcry, FALSE, 1);
        return;
    }

    if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
        ch->move -= need;
    else
        ch->mana -= need;

    check_improve(ch, gsn_warcry, TRUE, 1);

    af.where        = TO_AFFECTS;
    af.type         = gsn_warcry;
    af.level        = LVL(ch);
    af.duration     = 3 + LVL(ch) / 2;
    af.location     = APPLY_HITROLL;
    af.modifier     = LVL(ch) / 8;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    af.location     = APPLY_SAVING_SPELL;
    af.modifier     = - 10;
    affect_to_char(ch, &af);

    char_act("You feel righteous as you yell out your warcry.", ch);
}

DO_FUN(do_guard)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    int chance;

    if ((chance = get_skill(ch, gsn_guard)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0') 
    {
        char_act(" ?", ch);
        return;
    }

    if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (IS_NPC(victim)) 
    {
        act("$N     !", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (!str_cmp(arg, "none") || !str_cmp(arg, "self") || victim == ch || !str_cmp(arg, "") || !str_cmp(arg, "")) 
    {
        if (ch->guarding == NULL) 
        {
            char_act("    !", ch);
            return;
        }
        act("   $N.", ch, NULL, ch->guarding, TO_CHAR);
        act("$n   .", ch, NULL, ch->guarding, TO_VICT);
        act("$n   $N.", ch, NULL, ch->guarding, TO_NOTVICT);
        ch->guarding->guarded_by = NULL;
        ch->guarding             = NULL;
        return;
    }

    if (ch->guarding == victim) 
    {
        act("   $N!", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (ch->guarding != NULL) 
    {
        char_act("    - !", ch);
        return;
    }

    if (victim->guarding != NULL) 
    {
        char_act("   - !", ch);
        return;
    }

    if (victim->guarded_by != NULL) 
    {
        act("$N  - .", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (victim->guarding == ch) 
    {
        act(" $N  !", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (!is_same_group(victim, ch)) 
    {
        act("      $N.", ch, NULL, victim, TO_CHAR);
        return;
    }

    if (IS_AFFECTED(ch,AFF_CHARM)) 
    {
        act("          $N!",
            ch, NULL, victim, TO_VICT);
        return;
    }

    if (victim->fighting != NULL) 
    {
        char_act("        ?", ch);
        return;
    }

    if (ch->fighting != NULL) 
    {
        char_act("   ,        .", ch);
        return;
    }

    act("   $N.", ch, NULL, victim, TO_CHAR);
    act("$n   .", ch, NULL, victim, TO_VICT);
    act("$n   $N.", ch, NULL, victim, TO_NOTVICT);

    ch->guarding = victim;
    victim->guarded_by = ch;
}

CHAR_DATA *check_guard(CHAR_DATA *ch, CHAR_DATA *mob)
{
    int chance;

    if (ch->guarded_by == NULL 
    || get_char_room(ch, ch->guarded_by->name) == NULL)
        return ch;
    else 
    {
        chance = (get_skill(ch->guarded_by,gsn_guard) -
            (1.5 * (ch->level - mob->level)));
        if (number_percent() < chance) 
        {
            act("$n   $N!",
                ch->guarded_by, NULL, ch, TO_NOTVICT);
            act("$n   !",
                ch->guarded_by, NULL, ch, TO_VICT);
            act("   $N!",
                ch->guarded_by, NULL, ch, TO_CHAR);
            check_improve(ch->guarded_by, gsn_guard, TRUE, 3);
            return ch->guarded_by;
        } else 
        {
            check_improve(ch->guarded_by, gsn_guard, FALSE, 3);
            return ch;
        }
    }
}

DO_FUN(do_explode)
{
    CHAR_DATA *victim = ch->fighting;
    CHAR_DATA *vch, *vch_next;
    int dam=0,hp_dam,dice_dam,mana;
    int hpch,level= LVL(ch);
    char arg[MAX_INPUT_LENGTH];
    int chance;

    if ((chance = get_skill(ch, gsn_explode)) == 0)
    {
        char_act("?  ?", ch);
        return;
    }

    if (IS_AFFECTED(ch,AFF_FEAR))
    {
        char_act(" !    ?", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_explode)->beats);

    if (victim == NULL)
    {
        one_argument(argument, arg, sizeof(arg));
        if (arg == NULL)
        {
            char_act("    !", ch);
            return;
        }
        if ((victim = get_char_room(ch, arg)) == NULL)
        {
            char_act("  .", ch);
            return;
        }
    }

    mana = SKILL(gsn_explode)->min_mana;
    if (ch->mana < mana
    || (ch->move < 10
    && !IS_NPC(ch)))
    {
        char_act("       .", ch);
        return;
    }
    ch->mana -= mana;

    act("$n  -.", ch, NULL, victim, TO_NOTVICT);
    act("$n  -     ...   !",
        ch, NULL, victim, TO_VICT);
    act("  !!!", ch, NULL, NULL, TO_CHAR);

    if (is_safe(ch, victim))
        return;

    if (number_percent() >= chance)
    {
        damage(ch, victim, 0, gsn_explode, DAM_FIRE, TRUE);
        check_improve(ch, gsn_explode, FALSE, 1);
        return;
    }

    hpch = UMAX(10, ch->hit);
    hp_dam  = number_range(hpch/9+1, hpch/5);
    dice_dam = dice(level,20);

    dam = UMAX(hp_dam + dice_dam /10, dice_dam + hp_dam / 10);
    fire_effect(victim->in_room, level, dam/2, TARGET_ROOM);
    check_improve(ch, gsn_explode, TRUE, 1);

    for (vch = victim->in_room->people; vch != NULL; vch = vch_next)
    {
        vch_next = vch->next_in_room;

        if (is_safe_spell(ch,vch,TRUE)
        || (IS_NPC(vch) && IS_NPC(ch)
        && (ch->fighting != vch
        || vch->fighting != ch)))
              continue;

        if (vch == victim)
        {
            /* full damage */
            fire_effect(vch, level, dam, TARGET_CHAR);
            damage(ch, vch, dam, gsn_explode, DAM_FIRE, TRUE);
        } else 
        {
            /* partial damage */
            fire_effect(vch, level/2, dam/4, TARGET_CHAR);
            damage(ch, vch, dam/2, gsn_explode, DAM_FIRE,TRUE);
        }
    }

    if (number_percent() >= chance)
    {
        fire_effect(ch, level/4, dam/10, TARGET_CHAR);
        damage(ch, ch, (ch->hit / 10), gsn_explode, DAM_FIRE, TRUE);
    }
}

DO_FUN(do_target)
{
    CHAR_DATA *victim;
    int chance;

    if ((chance = get_skill(ch, gsn_target)) == 0) 
    {
        char_act("           c .",ch);
        return;
    }

    if (ch->fighting == NULL) 
    {
        char_act("   .", ch);
        return;
    }

    if (argument[0] == '\0') 
    {
        char_act("   ?", ch);
        return;
    }

    if ((victim = get_char_room(ch, argument)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }
    
    if (victim == ch->fighting)
    {
        char_act("You do the best you can!", ch);
        return;
    }

    /* check victim is fighting with him */
    if (victim->fighting != ch) 
    {
        char_act("One battle at a time, please.", ch);
        return;
    }

    if (ch->move < 20 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_target)->beats);

    act("You start aiming at $N.", ch, NULL, victim, TO_CHAR);

    if (number_percent() < chance * 2/3) 
    {
        check_improve(ch, gsn_target, TRUE, 1);
        
        ch->fighting = victim;

        act("$n     $N!",
            ch, NULL, victim, TO_NOTVICT);
        act("     $N!",
            ch, NULL, victim, TO_CHAR);
        act("$n     !", ch, NULL, victim, TO_VICT);
        return;
    }

    char_act(" ,  .", ch);
    if (IS_SAMURAI(ch)) 
        char_act("    ,  !", ch);
    check_improve(ch, gsn_target, FALSE, 1);
}

DO_FUN(do_deliverance)
{
    CHAR_DATA *victim;
    CHAR_DATA *master;
    int chance;

    if ((chance = get_skill(ch, gsn_deliverance)) == 0)
    {
        char_act("    !",ch);
        return;
    }

    if (ch->fighting == NULL)
    {
        char_act("   .", ch);
        return;
    }

    victim = ch->fighting;

    if (ch->move < 20 && !IS_NPC(ch))
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_deliverance)->beats);

    if (!IS_NPC(victim)
    || !IS_AFFECTED(victim, AFF_CHARM)
    || ch->in_room != victim->in_room)
        chance = 0;

//    if (number_percent() < chance / 3) GM : MAX
    if (number_percent() <= chance / 4)
    {
        check_improve(ch, gsn_deliverance, TRUE, 1);

        act("$n  $N!", ch, NULL, victim, TO_NOTVICT);
        act("  $N!", ch, NULL, victim, TO_CHAR);
        act("$n  !", ch, NULL, victim, TO_VICT);

        master = victim->master;

        if (!master)
            master = victim->leader;

        stop_follower(victim);
        victim->leader = NULL;

        if (master && master->in_room
        && master->in_room == victim->in_room)
        {
            if (number_percent() < get_curr_stat(master, STAT_LCK) * 2)
            {
                raw_kill(ch, victim);
                ch->fighting = master;
            } else
            {
                act("  $N    !",
                    master, NULL, victim, TO_CHAR);
                act(", $N      $n.",
                    master, NULL, victim, TO_ROOM);
                act(" '' $n   !",
                    master, NULL, victim, TO_VICT);
                ch->fighting = master;

//                stop_fighting(victim, FALSE);

                // check for possible problems
                if (!master || !victim || JUST_KILLED (master) || JUST_KILLED (victim))
                    return;
                multi_hit(victim, master, TYPE_UNDEFINED);
//                set_fighting(victim, master);
                victim->fighting = master;
            }
        } else
            raw_kill(ch, victim);
        return;
    }

    char_act("   .", ch);
    check_improve(ch, gsn_deliverance, FALSE, 1);
}

DO_FUN(do_taunt)
{
    CHAR_DATA *vch;
    CHAR_DATA *vch_next;
    CHAR_DATA *gch;
    CHAR_DATA *gch_next;
    int chance;

    if ((chance = get_skill(ch, gsn_taunt)) == 0)
    {
        char_act("Huh?",ch);
        return;
    }
/*     ,     ...   . // GM
    for (gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room) 
    {
        if (is_same_group(ch, gch)) 
        {
        }
    }
*/
    for (gch = ch->in_room->people; gch; gch = gch_next)
    {
        gch_next = gch->next_in_room;

        if (IS_NPC(gch) && IS_AFFECTED(gch,AFF_CHARM) && gch->master == ch
        && (gch->pIndexData->vnum == MOB_VNUM_FOREST_GIANT))
            break;
    }

    if (gch == NULL) gch = ch;

    if (IS_AFFECTED(gch, AFF_SLEEP))
    {
        char_act(" .", gch);
        if (gch != ch) char_act(" .", ch);
        return;
    }
    if (IS_AFFECTED(gch,AFF_FEAR))
    {
        char_act("You too scare to do this!", gch);
        if (gch != ch) char_act(" .", ch);
        return;
    }
    if (gch->move < 20 && !IS_NPC(gch))
    {
        char_act("  .", gch);
        return;
    }
    WAIT_STATE(ch, SKILL(gsn_taunt)->beats);

    if (number_percent() < chance)
    {
        act("$n     $gn{}!", gch, NULL, NULL, TO_ROOM);
        act("     !", gch, NULL, NULL, TO_CHAR);

        for (vch = ch->in_room->people; vch; vch = vch_next)
        {
            vch_next = vch->next_in_room;

            if (is_safe_spell(ch, vch, TRUE))
                continue;

            if (vch != gch && vch != ch)
            {
                stop_fighting(vch, FALSE);
                multi_hit(gch, vch, TYPE_UNDEFINED);
                set_fighting(vch, gch);
                vch->fighting = gch;
            }

            act("   $n!", gch, NULL, vch, TO_VICT);
            act("$N   $n!", gch, NULL, vch, TO_ROOM);
            check_improve(ch, gsn_taunt, TRUE, 10);
        }
    } else
    {
        char_act("      !", gch);
        check_improve(ch, gsn_taunt, FALSE, 1);
    }
}

DO_FUN(do_tiger)
{
    int chance, hp_percent;
    int mana;

    if ((chance = get_skill(ch, gsn_tiger_power)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }
    act("$n    !", ch, NULL, NULL, TO_ROOM);

    if (IS_AFFECTED(ch, AFF_BERSERK) || is_affected(ch, gsn_berserk)
    || is_affected(ch, gsn_tiger_power) || is_affected(ch, gsn_frenzy)) 
    {
        char_act("  .", ch);
        return;
    }

    if (IS_AFFECTED(ch,AFF_CALM)) 
    {
        char_act("         .", 
            ch);
        return;
    }

    if (ch->in_room->sector_type != SECT_FIELD
    && ch->in_room->sector_type != SECT_FOREST
    && ch->in_room->sector_type != SECT_MOUNTAIN
    && ch->in_room->sector_type != SECT_HILLS) 
    {
        char_act("   .", ch);
        return;
    }

    mana = SKILL(gsn_tiger_power)->min_mana;
    if (ch->mana < mana) 
    {
        char_act("   .", ch);
        return;
    }

    /* modifiers */

    /* fighting */
    if (ch->position == POS_FIGHTING)
        chance += 10;

    hp_percent = 100 * ch->hit/ch->max_hit;
    chance += 25 - hp_percent/2;
    ch->move /= 2;

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        WAIT_STATE(ch, SKILL(gsn_tiger_power)->beats);
        ch->mana -= mana;

        /* heal a little damage */
        ch->hit += LVL(ch) * 2;
        ch->hit = UMIN(ch->hit, ch->max_hit);

        char_act("10     !", ch);
        act("10     $n,   $gn{}  .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_tiger_power, TRUE, 2);

        af.where        = TO_AFFECTS;
        af.type         = gsn_tiger_power;
        af.level        = LVL(ch);
        af.duration     = number_fuzzy(LVL(ch) / 2);
        af.modifier     = UMAX(1, LVL(ch) / 5);
        af.bitvector    = AFF_BERSERK;

        af.location     = APPLY_HITROLL;
        affect_to_char(ch,&af);

        af.location     = APPLY_DAMROLL;
        affect_to_char(ch,&af);

        af.modifier     = UMAX(10, 10 * (LVL(ch) / 5));
        af.location     = APPLY_AC;
        affect_to_char(ch,&af);
    } else 
    {
        WAIT_STATE(ch, 2 * SKILL(gsn_tiger_power)->beats);
        ch->mana -= mana/2;
        char_act("          .", ch);
        check_improve(ch, gsn_tiger_power, FALSE, 2);
    }
}

DO_FUN(do_hara)
{
    int chance, max_hp = 0;
    AFFECT_DATA  af;
    char arg[MAX_INPUT_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    if (((chance = get_skill(ch, gsn_hara_kiri)) == 0) || IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("   ,       .", ch);
        return;
    }

    if (arg[0] != '\0')
    {
        max_hp = atoi(arg);
        if (max_hp == 0)
        {
            act("HP control reseted.", ch, NULL, NULL, TO_CHAR);
            ch->wimpy = 0;
            return;
        }
        if (max_hp < 0 || max_hp > ch->max_hit)
        {
            act("Can't set HP control.", ch, NULL, NULL, TO_CHAR);
            return;
        }
        if (max_hp < ch->max_hit / 2 + ch->level)
        {
            char_printf(ch, "Minimum value for HP control is %d hit points.\n", 
                ch->max_hit / 2 + ch->level);
            return;
        }
        ch->wimpy = max_hp;
        char_printf(ch, "HP control set to %d hit points. Set it to Zero for reset.\n", 
            max_hp);
        return;
    }

    if (MOUNTED(ch)) 
    {
        char_act("    ,    !", ch);
        return;
    }

    if (is_affected(ch, gsn_hara_kiri)) 
    {
        char_act("    .", ch);
        return;
    }

    /* fighting */
    if (ch->position == POS_FIGHTING) 
    {
        char_act("      .", ch);
        return;
    }

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        WAIT_STATE(ch, SKILL(gsn_hara_kiri)->beats);

        ch->hit = 1;
        ch->mana = 1;
        ch->move = 1;

        if (ch->pcdata->condition[COND_HUNGER] < 40)
            ch->pcdata->condition[COND_HUNGER] = 40;
        if (ch->pcdata->condition[COND_THIRST] < 40)
            ch->pcdata->condition[COND_THIRST] = 40;

        char_act("     ,    .", ch);
        act_puts("$n      .", 
            ch, NULL, NULL, TO_ROOM, POS_FIGHTING);
        check_improve(ch, gsn_hara_kiri, TRUE, 2);
        affect_strip(ch, gsn_dishonor);
        do_sleep(ch, str_empty);
        SET_BIT(ch->plr_flags,PLR_HARA_KIRI);

        af.where     = TO_AFFECTS;
        af.type      = gsn_hara_kiri;
        af.level     = LVL(ch);
        af.duration  = 10;
        af.location  = APPLY_NONE;
        af.modifier  = 0;
        af.bitvector = 0;
        affect_to_char(ch, &af);
    } else 
    {
        WAIT_STATE(ch, 2 * SKILL(gsn_hara_kiri)->beats);

        af.where     = TO_AFFECTS;
        af.type      = gsn_hara_kiri;
        af.level     = LVL(ch);
        af.duration  = 0;
        af.location  = APPLY_NONE;
        af.modifier  = 0;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        char_act("     .       .", ch);
        check_improve(ch, gsn_hara_kiri, FALSE, 2);
    }
}

//
// check for damage from critical strike
//
int critical_strike (CHAR_DATA * ch, CHAR_DATA * victim, int dam)
{
    int chance, sk, wait ;
    AFFECT_DATA baf ;
    
    // no damage
    if (dam == 0) return 0 ;

    // skill
    sk = get_skill (ch, gsn_critical);

    // not skilled in critical strike
    if (sk <= 1) return 0 ;

    // must wield at most one weapon
    if (get_eq_char (ch, WEAR_WIELD)        != NULL 
        && get_eq_char (ch, WEAR_SECOND_WIELD) != NULL
        && number_percent() < 50)
        
        return 0 ;

    // just in case
    if (ch->max_hit == 0 || victim->max_hit == 0) return 0 ;

    // chance to fail depends on victim and attacker hp
    chance  = 100 - ((ch->hit * sk) / ch->max_hit);
    chance += ((victim->hit * 100) / victim->max_hit) / 2 ;

    // make chance to fail to be at least 70
    chance  = URANGE (70, chance, 95);
    chance += LVL (victim) - LVL (ch);

    if (number_percent () < chance) return 0 ;
   
    check_improve (ch, gsn_critical, TRUE, 2);

    // increase damage (1.5...2.5 times)
    dam = (dam * number_range (15, 25)) / 10 ;
      
    // check for extra damage and effects
    chance = number_percent ();

    if (ch->style && ch->style == STYLE_DRUNKMAN)
        chance *= 2;

    // 60% to stop here
    if (chance <= 60) return dam ;

    if (chance <= 80) // 20% probability of throw
    {
        act_puts("  $N    !", 
            ch, NULL, victim, TO_CHAR, POS_DEAD);
        act("$n       !", 
            ch, NULL, victim, TO_VICT);
        act("$n  $c6{$N}     !", 
            ch, NULL, victim, TO_NOTVICT);

        check_improve (ch, gsn_critical, TRUE, 3);

        wait = number_range (1, 3);

        // either we recover from the throw or check for
        // possible ground control action

        if (check_roll (victim)) return 0 ;

        if (check_recovery (victim, ch)) wait /= 2 ; else
        if (check_jammer   (victim)) wait /= 2 ; else
        check_ground_control (ch, victim, chance, dam / 4);

        // check for possible problems
        if (!ch || !victim || JUST_KILLED (ch) || JUST_KILLED (victim))
            return dam ;

        // set daze state to victim
        DAZE_STATE (victim, PULSE_VIOLENCE * wait);

        // small extra damage
        return (dam + chance);
    }

    if (chance <= (90 - IS_NPC(ch) * 5)) // 10% we will blind the victim
    {
        check_improve (ch, gsn_critical, TRUE, 4);
        if (!IS_UNDEAD(victim)) 
        {
            act_puts("  $N  !", 
                ch, NULL, victim, TO_CHAR, POS_DEAD);
            act("    $n!", 
                ch, NULL, victim, TO_VICT);
            act("$N    $n!", 
                ch, NULL, victim, TO_NOTVICT);

            // add blind affect
            if (!IS_AFFECTED (victim, AFF_BLIND))
            {
                baf.where     = TO_AFFECTS;
                baf.type      = gsn_critical;
                baf.level     = LVL(ch) + LVL(ch) / 10;
                baf.location  = APPLY_HITROLL;
                baf.modifier  = -number_fuzzy (LVL(ch) / 5);
                baf.duration  = number_range (1, 3);
                baf.bitvector = AFF_BLIND;
    
                affect_to_char (victim, &baf);
            }
        }
        // small extra damage
        return (dam + chance);
    }

    if (chance >= 95) // 5% of extermely high damage
    {
        act_puts("  {r{x $N!", 
            ch, NULL, victim, TO_CHAR , POS_RESTING);
        act_puts("$n   {r{x! ,  !", 
            ch, NULL, victim, TO_VICT, POS_RESTING);
        act_puts("$n  {r{x $N!", 
            ch, NULL, victim, TO_NOTVICT, POS_RESTING);

        check_improve (ch, gsn_critical, TRUE, 5);

        // extra damage
        return (dam * 3 / 2);
    }

    return dam ;
}

DO_FUN(do_shield)
{
    CHAR_DATA *victim;
    int chance,ch_weapon,vict_shield;
    OBJ_DATA *shield,*axe;

    if (IS_NPC(ch))
        return;

    if ((victim = ch->fighting) == NULL
        || victim->in_room != ch->in_room) 
    {
        char_act("     .", ch);
        return;
    }

    if ((axe = get_eq_char(ch,WEAR_WIELD)) == NULL) 
    {
        char_act(" .", ch);
        return;
    }

    if ((chance = get_skill(ch, gsn_shield_cleave)) == 0) 
    {
        char_act("      .", ch);
        return;
    }

    if ((shield = get_eq_char(victim, WEAR_SHIELD)) == NULL) 
    {
        char_act("    .", ch);
        return;
    }

    if (check_material(shield,"titanium")
    || shield->pIndexData->limit != -1
    || IS_OBJ_STAT(shield, ITEM_INDESTRUCTABLE))
        return;

    if (axe->value[0] == WEAPON_AXE)
        chance *= 1.2;
    else if (axe->value[0] != WEAPON_SWORD) 
    {
        char_act("      .", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    /* find weapon skills */
    ch_weapon = get_weapon_skill(ch, get_weapon_sn(axe));
    vict_shield = get_skill(ch, gsn_shield_block);
    /* modifiers */

    /* skill */
    chance = chance * ch_weapon / 200;
    if (vict_shield)
        chance = chance * 100 / vict_shield;

    /* dex vs. strength */
    chance += get_curr_stat(ch, STAT_DEX);
    chance -= 2 * get_curr_stat(victim, STAT_STR);

    /* level */
/*  chance += (LVL(ch) - LVL(victim)) * 2; */
    chance += LVL(ch) - LVL(victim);
    chance += axe->level - shield->level;

    /* and now the attack */
    WAIT_STATE(ch, SKILL(gsn_shield_cleave)->beats);

    if(check_mirror_image(ch,victim))
        return;

    if (number_percent() < chance) 
    {
        act("   $N.",
            ch, NULL, victim, TO_CHAR);
        act("$n   .",
            ch, NULL, victim, TO_VICT);
        act("$n   $N.",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_shield_cleave, TRUE, 1);
        extract_obj(get_eq_char(victim, WEAR_SHIELD));
    } else 
    {
        act("      $N.",
            ch, NULL, victim, TO_CHAR);
        act("$n      .",
            ch, NULL, victim, TO_VICT);
        act("$n     $N.",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_shield_cleave, FALSE, 1);
    }
}

DO_FUN(do_weapon)
{
    CHAR_DATA *victim;
    OBJ_DATA *wield,*axe;
    int chance,ch_weapon,vict_weapon;

    if (IS_NPC(ch))
        return;

    if ((victim = ch->fighting) == NULL
    || victim->in_room != ch->in_room) 
    {
        char_act("     .", ch);
        return;
    }

    if ((axe = get_eq_char(ch, WEAR_WIELD)) == NULL) 
    {
        char_act(" .", ch);
        return;
    }

    if ((chance = get_skill(ch, gsn_weapon_cleave)) == 0) 
    {
        char_act("      .", ch);
        return;
    }

    if ((wield = get_eq_char(victim, WEAR_WIELD)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (axe->value[0] == WEAPON_AXE)
        chance *= 1.2;
    else if (axe->value[0] != WEAPON_SWORD) 
    {
        char_act("      .", ch);
        return;
    }

    if (ch->move < 40 && !IS_NPC(ch)) 
    {
        char_act("  .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_weapon_cleave)->beats);

    if (wield->pIndexData->limit != -1
    || IS_OBJ_STAT(wield, ITEM_INDESTRUCTABLE))
    {
        act("    $N,  $gN{}   .",
            ch, NULL, victim, TO_CHAR);
        act("$n   ϣ ,    .",
            ch, NULL, victim, TO_VICT);
        act("$n    $N,  $gN{}   .",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_weapon_cleave, FALSE, 1);
        return;
    }

    /* find weapon skills */
    ch_weapon = get_weapon_skill(ch, get_weapon_sn(axe));
    vict_weapon = get_weapon_skill(victim, get_weapon_sn(wield));
    /* modifiers */

    /* skill */
    chance = chance * ch_weapon / 200;
    chance = chance * 100 / vict_weapon;

    /* dex vs. strength */
    chance += get_curr_stat(ch, STAT_DEX) + get_curr_stat(ch, STAT_STR);
    chance -= get_curr_stat(victim, STAT_STR) +
                2 * get_curr_stat(victim, STAT_DEX);

    chance += LVL(ch) - LVL(victim);
    chance += axe->level - wield->level;

    if(check_mirror_image(ch,victim))
        return;

    /* and now the attack */
    if (number_percent() < chance)
    {
        act("   $N   .",
            ch, NULL, victim, TO_CHAR);
        act("$n      .",
            ch, NULL, victim, TO_VICT);
        act("$n   $N   .",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_weapon_cleave, TRUE, 1);
        extract_obj(get_eq_char(victim, WEAR_WIELD));
    } else 
    {
        act("      $N.",
            ch, NULL, victim, TO_CHAR);
        act("$n    ,   .",
            ch, NULL, victim, TO_VICT);
        act("$n     $N,  .",
            ch, NULL, victim, TO_NOTVICT);
        check_improve(ch, gsn_weapon_cleave, FALSE, 1);
    }
}


DO_FUN(do_bandage)
{
    int     heal = 200;
    int     chance;
    int     mana;
    int     nsn;
    int     level = LVL(ch) + LVL(ch) / 10;
    char        arg[MAX_INPUT_LENGTH];
    CHAR_DATA   *victim;
    clan_t   *clan;

    one_argument(argument, arg, sizeof(arg));

    if (IS_NPC(ch))
        chance = 101;
    else if ((chance = get_skill(ch, gsn_bandage)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (arg[0] == '\0')
        victim = ch;
    else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (is_affected(victim, gsn_bandage)) 
    {
        if (victim == ch) 
            char_act ("Your wounds are already covered by bandage.", ch); 
        else 
            act_puts("H $N     .", ch, NULL, victim, TO_CHAR, POS_RESTING);
        return;
    }

    clan = CLAN(victim->clan);
    if (victim != ch && IS_SET(clan->flags, CLAN_LOVE_MAGIC)) 
    {
        char_act("     ?!", ch);
        return;
    }

    mana = SKILL(gsn_bandage)->min_mana;
    if (ch->mana < mana && !IS_NPC(ch)) 
    {
        char_act("   .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_bandage)->beats);

    if (IS_UNDEAD(victim))
        chance = 0;

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        if (victim == ch) 
        {
            char_act("      !", ch);
            act("$n      .", ch, NULL, NULL, TO_ROOM);
        } else 
        {
            act("    p  $N.", ch, NULL, victim, TO_CHAR);
            act("$n      $N.", ch, NULL, victim, TO_NOTVICT);
            act("$n      .", ch, NULL, victim, TO_VICT);
        }
        check_improve(ch, gsn_bandage, TRUE, 2);

        heal += dice(4, 8) + LVL(ch);
        victim->hit = UMIN(victim->hit + heal, victim->max_hit);
        update_pos(victim);
        char_act("   !", victim);

        if (!IS_NPC(ch))
        {
            ch->mana -= mana;

            if (IS_AFFECTED(victim, AFF_BLIND))
            {
                nsn = sn_lookup("cure blindness");
                spell_cure_blindness(nsn, level, ch, (void *)victim, TARGET_CHAR);
            }
            if (IS_AFFECTED(victim, AFF_POISON))
            {
                spell_cure_poison(gsn_cure_poison, level, ch, (void *)victim, TARGET_CHAR);
            }
            if (IS_AFFECTED(victim, AFF_PLAGUE))
            {
                nsn = sn_lookup("cure disease");
                spell_cure_disease(nsn, level, ch, (void *)victim, TARGET_CHAR);
            }

            if (!is_affected(victim, gsn_bandage))
            {
                af.where        = TO_AFFECTS;
                af.type         = gsn_bandage;
                af.level        = LVL(ch);
                af.duration     = LVL(ch) / 10;
                af.modifier     = UMIN(1, LVL(ch) / 15);
                af.bitvector    = AFF_REGENERATION;
                af.location     = 0;
                affect_to_char(victim, &af);
            }
        }
    } else 
    {
        ch->mana -= mana/2;
        char_act("     .", ch);
        check_improve(ch, gsn_bandage, FALSE, 2);
    }
}

DO_FUN(do_needle_prick)
{
    int     heal;
    int     chance;
    int     nsn;
    int     level = LVL(ch) + LVL(ch)/10;
    char        arg[MAX_INPUT_LENGTH];
    CHAR_DATA   *victim;
    AFFECT_DATA af;

    one_argument(argument, arg, sizeof(arg));


    if ((chance = get_skill(ch,gsn_needle_prick)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (arg[0] == '\0')
        victim = ch;
    else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
            char_act("  .", ch);
            return;
    }

    if (!check_trust(ch, victim))
    {
        act_puts("$N     .", ch, NULL, victim, TO_CHAR, POS_DEAD);
        return;
    }

    if (is_affected(victim, gsn_needle_prick)) 
    {
        if (victim == ch) 
        {
            char_act("    .", ch);
            return;
        } else 
        {
            act_puts("H $N     ϣ .", 
                ch, NULL, victim, TO_CHAR, POS_RESTING);
            return;
        }
    }

    WAIT_STATE(ch, SKILL(gsn_needle_prick)->beats);

    if (IS_UNDEAD(victim))
        chance = 0;

    if (chance == 0)
    {
        char_act("     !", victim);
        return;
    }

    if ((ch != victim) && check_mirror_image(ch, victim))
        return;
    
    if (ch->fighting != NULL
    && number_percent() > get_curr_stat(ch, STAT_DEX))
        chance /= 2;

    if (IS_PUMPED(victim))
        chance /= 2;

    if (ch->style && ch->style == STYLE_COBRA)
        chance *= 2;

    if (number_percent() < chance)
    {
        if (IS_AFFECTED(victim, AFF_BLIND))
        {
            nsn = sn_lookup("cure blindness");
            spell_cure_blindness(nsn, level, ch, (void *)victim, TARGET_CHAR);
        }
        if (IS_AFFECTED(victim, AFF_POISON))
        {
            spell_cure_poison(gsn_cure_poison, level, ch, (void *)victim, TARGET_CHAR);
        }
        if (IS_AFFECTED(victim, AFF_PLAGUE))
        {
            nsn = sn_lookup("cure disease");
            spell_cure_disease(nsn, level, ch, (void *)victim, TARGET_CHAR);
        }

        if (victim == ch) 
        {
            char_act("   !", ch);
            act("$n   .", ch, NULL, NULL, TO_ROOM);
        } else
        {
            act("  $N .",
                ch, NULL, victim, TO_CHAR);
            act("$n  $N .",
                ch, NULL, victim, TO_NOTVICT);
            act("$n   .",
                ch, NULL, victim, TO_VICT);
        }
        check_improve(ch, gsn_needle_prick, TRUE, 2);

        heal = dice(4, 8) + LVL(ch);

        if (IS_CYBORG(victim))
            heal /= 2;

        victim->hit = UMIN(victim->hit + heal, victim->max_hit);
        update_pos(victim);
        char_act("   !", victim);

        af.where        = TO_AFFECTS;
        af.type         = gsn_needle_prick;
        af.level        = LVL(ch);
        af.duration     = number_fuzzy (1);
        af.modifier     = UMIN(15, LVL(ch) / 2);
        af.bitvector    = AFF_REGENERATION;
        af.location     = 0;
        affect_to_char(victim, &af); 
    } else 
    {
        char_act("     .", ch);
        if (victim == ch) 
        {
            char_act("   !", ch);
            act("$n   .", ch, NULL, NULL, TO_ROOM);
        } else
        {
            act("  $N .",
                ch, NULL, victim, TO_CHAR);
            act("$n  $N .",
                ch, NULL, victim, TO_NOTVICT);
            act("$n   .",
                ch, NULL, victim, TO_VICT);
        }
        check_improve(ch, gsn_needle_prick, FALSE, 2);

        af.where        = TO_AFFECTS;
        af.type         = gsn_needle_prick;
        af.level        = LVL(ch);
        af.duration     = 0;
        af.modifier     = 0;
        af.bitvector    = 0;
        af.location     = 0;
        affect_to_char(victim, &af); 
    }
}

DO_FUN(do_katana)
{
    OBJ_DATA *katana;
    AFFECT_DATA af;
    OBJ_DATA *part;
    char arg[MAX_INPUT_LENGTH];
    int chance;
    int mana;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_katana)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_katana)) 
    {
        char_act("      !", ch);
        return;
    }

    mana = SKILL(gsn_katana)->min_mana;
    if (ch->mana < mana) 
    {
        char_act("        .", ch);
        return;
    }

    if (arg[0] == '\0') 
    {
        char_act("   ?", ch);
        return;
    }

    if ((part = get_obj_carry(ch, arg)) == NULL) 
    {
        char_act("   .", ch);
        return;
    }

    if (part->pIndexData->vnum != OBJ_VNUM_CHUNK_IRON) 
    {
        char_act("    .", ch);
        return;
    }

    if (number_percent() < chance / 3 * 2) 
    {
        char_act("     .", ch);
        extract_obj(part);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_katana)->beats);

    if (number_percent() < chance) 
    {
        af.where    = TO_AFFECTS;
        af.type     = gsn_katana;
        af.level    = LVL(ch);
        af.duration = LVL(ch);
        af.modifier = 0;
        af.bitvector    = 0;
        af.location = 0;
        affect_to_char(ch,&af);

        katana = create_obj(get_obj_index(OBJ_VNUM_KATANA_SWORD), ch->level);
        katana->cost  = 0;
        katana->level = LVL(ch);
        ch->mana -= mana;

        af.where        = TO_OBJECT;
        af.type         = gsn_katana;
        af.level        = LVL(ch);
        af.duration     = -1;
        af.location     = APPLY_DAMROLL;
        af.modifier     = LVL(ch) / 10;
        af.bitvector    = 0;
        affect_to_obj(katana, &af);

        af.location     = APPLY_HITROLL;
        affect_to_obj(katana, &af);

        katana->value[2] = LVL(ch) / 10;
        katana->ed = ed_new2(katana->pIndexData->ed, ch->name);

        obj_to_char(katana, ch);
        check_improve(ch, gsn_katana, TRUE, 1);

        act("   $p !",ch,part,NULL,TO_CHAR);
        act("$n   $p !",ch,part,NULL,TO_ROOM);

        extract_obj(part);
        return;
    } else 
    {
        char_act("! !", ch);
        extract_obj(part);
        ch->mana -= mana/2;
        check_improve(ch, gsn_katana, FALSE, 1);
    }
}
DO_FUN(do_sense)
{
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_sense_life)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_sense_life)) 
    {
        char_act("     .", ch);
        return;
    }

    mana = SKILL(gsn_sense_life)->min_mana;
    if (ch->mana < mana) 
    {
        char_act("   .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_sense_life)->beats);

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        af.where    = TO_AFFECTS;
        af.type     = gsn_sense_life;
        af.level    = LVL(ch);
        af.duration = LVL(ch);
        af.location = APPLY_NONE;
        af.modifier = 0;
        af.bitvector    = AFF_DETECT_LIFE;
        affect_to_char(ch, &af);

        ch->mana -= mana;

        act("      !",
            ch, NULL, NULL, TO_CHAR);
        act(" $n .", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_sense_life, TRUE, 1);
    } else 
    {
        ch->mana -= mana/2;
        char_act("   .", ch);
        check_improve(ch, gsn_sense_life, FALSE, 1);
    }
}

DO_FUN(do_poison_smoke)
{
    int chance;
    int mana;
    CHAR_DATA *vch;
    CHAR_DATA *vch_next;

    if ((chance = get_skill(ch, gsn_poison_smoke)) == 0) {
        char_act("Huh?", ch);
        return;
    }
    
    if (IS_AFFECTED(ch,AFF_FEAR)){
        char_act("   !    - ?!", ch);
        return;
    }

    mana = SKILL(gsn_poison_smoke)->min_mana;
    if (ch->mana < mana) {
        char_act("   .", ch);
        return;
    }
    ch->mana -= mana;
    WAIT_STATE(ch, SKILL(gsn_poison_smoke)->beats);

    if (number_percent() > chance) {
        char_act("   .", ch);
        check_improve(ch, gsn_poison_smoke, FALSE, 1);
        return;
    }

    char_act("    .", ch);
    act("    .", ch, NULL, NULL, TO_ROOM);

    check_improve(ch, gsn_poison_smoke, TRUE, 1);

    for (vch = ch->in_room->people; vch; vch = vch_next) {
        vch_next = vch->next_in_room;

        if (is_safe_spell(ch, vch, TRUE))
            continue;

        spell_poison(gsn_poison, LVL(ch), ch, vch, TARGET_CHAR);
        if (vch != ch)
            multi_hit(vch, ch, TYPE_UNDEFINED);
    }
}

DO_FUN(do_blindness_dust)
{
    CHAR_DATA *vch;
    CHAR_DATA *vch_next;
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_blindness_dust)) == 0) {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch,AFF_FEAR)){
        char_act("You too scare to do this!    - ?", ch);
        return;
    }

    mana = SKILL(gsn_blindness_dust)->min_mana;
    if (ch->mana < mana) {
        char_act("   .", ch);
        return;
    }

    ch->mana -= mana;
    WAIT_STATE(ch, SKILL(gsn_blindness_dust)->beats);
    if (number_percent() > chance) {
        char_act("   .", ch);
        check_improve(ch,gsn_blindness_dust,FALSE,1);
        return;
    }

    char_act("   .", ch);
    act("   .", ch, NULL, NULL, TO_ROOM);

    check_improve(ch, gsn_blindness_dust, TRUE, 1);

    for (vch = ch->in_room->people; vch; vch = vch_next) {
        vch_next = vch->next_in_room;

        if (is_safe_spell(ch, vch, TRUE))
            continue;

        spell_blindness(gsn_blindness, LVL(ch), ch, vch, TARGET_CHAR);
        if (vch != ch)
            multi_hit(vch, ch, TYPE_UNDEFINED);
    }
}

bool check_yell(CHAR_DATA *ch, CHAR_DATA *victim, bool fighting)
{
    return (!IS_NPC(ch) && !IS_NPC(victim) &&
        victim->position > POS_STUNNED && !fighting);
}

DO_FUN(do_claws)
{
    char arg[MAX_INPUT_LENGTH];
    CHAR_DATA *victim;
    bool fighting, hand[1];
    int chance, i, dam, bonus_dam = 0;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch,gsn_claws)) == 0)
    {
        char_act("     ,   .", ch);
        return;
    }

    if (arg[0] == '\0')
    {
        victim = ch->fighting;
        if (victim == NULL
        || victim->in_room != ch->in_room)
        {
            char_act("      !", ch);
            return;
        }
    } else if ((victim = get_char_room(ch, arg)) == NULL)
    {
        char_act("  .", ch);
        WAIT_STATE(ch, SKILL(gsn_claws)->beats / 4);
        return;
    }

    hand[0] = FREE_HAND1(ch);
    hand[1] = FREE_HAND2(ch);

    if (number_percent() > get_skill(ch ,gsn_mastering_claws) + LVL(ch))
    {
        check_improve(ch, gsn_mastering_claws, TRUE, 4);
        if (!(hand[0] || hand[1]))
        {
            char_act("      .", ch);
            return;
        }
    } else
        check_improve(ch, gsn_mastering_claws, FALSE, 1);

    if (IS_AFFECTED(ch, AFF_CHARM) && ch->master == victim)
    {
        char_act("      ?", ch);
        return;
    }

    if (is_safe(ch, victim))
        return;

    if (ch->move < 20 && !IS_NPC(ch))
    {
         char_act("  .", ch);
         return;
    }

    /* modifiers */
    /* dex */
    chance += get_curr_stat(ch, STAT_STR) + get_curr_stat(ch, STAT_DEX);
    chance -= get_curr_stat(victim, STAT_DEX) * 2;

    if (IS_NPC(victim))
        chance *= 2;

    /* speed */
    if ((IS_NPC(ch) && IS_SET(ch->pIndexData->off_flags, OFF_FAST))
    || IS_AFFECTED(ch, AFF_HASTE))
        chance += 10;
    if ((IS_NPC(victim) && IS_SET(victim->pIndexData->off_flags, OFF_FAST))
    ||  IS_AFFECTED(victim, AFF_HASTE))
        chance -= 20;

    /* level */
    if (!IS_NPC(victim))
        chance += (LVL(ch) - LVL(victim)) * 2;

    RESET_WAIT_STATE(ch);
    fighting = (ch->fighting != NULL);

    if (check_mirror_image(ch, victim))
    {
        WAIT_STATE(ch, SKILL(gsn_claws)->beats / 2);
        return;
    }
    
    // in wolf form damage is smaller, but faster
    if (IS_SET(ch->form, FORM_WOLF))
    {
        WAIT_STATE(ch, SKILL(gsn_claws)->beats / 4);
        check_improve(ch, gsn_wolfform, TRUE, 4);
    }
    else
        WAIT_STATE(ch, SKILL(gsn_claws)->beats);

    if (LVL(ch) < 50)
        bonus_dam = LVL(ch) * ch->size / 8;
    else if (LVL(ch) < 80)
        bonus_dam = LVL(ch) * ch->size / 4;
    else
        bonus_dam = LVL(ch) * ch->size;

    bonus_dam += ch->damroll;
    bonus_dam += number_range(ch->damroll / 4, ch->damroll / 2);

    // in bear form more damage, less speed.
    if (IS_SET(ch->form, FORM_BEAR))
        check_improve(ch, gsn_bearform, TRUE, 4);
    else if (IS_SET(ch->form, FORM_WOLF))
        bonus_dam /= 4;
    else
        bonus_dam /= 2;

    /* now the attack */
    for (i = 0; i < 2; i++)
    {
        if (number_percent() > get_skill(ch ,gsn_mastering_claws) + LVL(ch))
        {
            check_improve(ch, gsn_mastering_claws, TRUE, 4);
            if (!hand[i]) continue;
        } else
            check_improve(ch, gsn_mastering_claws, FALSE, 1);

        if (number_percent() < get_skill(ch ,gsn_poisoning_claws) / 20
            + get_skill(ch ,gsn_mastering_claws) / 10)
        {
            check_improve(ch, gsn_poisoning_claws, TRUE, 4);
            if (!IS_UNDEAD(victim) && !IS_CYBORG(victim)
            && !is_affected(victim, gsn_bleed))
            {
                AFFECT_DATA af;

                af.type         = gsn_bleed;
                af.where        = TO_AFFECTS;
                af.level        = ch->level;
                af.bitvector    = 0;
                af.duration     = 5;
                af.modifier     = -5;
                af.location     = APPLY_CON;
                affect_to_char(victim,&af);

                af.modifier     = -3;
                af.location     = APPLY_STR;
                affect_to_char(victim,&af);

                af.modifier     = -3;
                af.location     = APPLY_DEX;
                affect_to_char(victim,&af);
                act("You slice open a vital artery on $N's body!",
                    ch, 0, victim, TO_CHAR);
                act("$n slices open a vital artery on your body!",
                    ch, 0, victim, TO_VICT);
            }
       } else
            check_improve(ch, gsn_poisoning_claws, FALSE, 1);

        ch->move = ch->move - ((ch->level * 2 + 10) / 10);
        if (number_percent() < chance)
        {
            // Daze Chance...
            if (number_percent() < ((get_skill(ch ,gsn_mastering_claws) / 10)
                + (LVL(ch) - LVL(victim) * 2)))
            {
                if (number_percent () < get_curr_stat(ch, STAT_LCK))
                {
                    DAZE_STATE (victim, PULSE_VIOLENCE * 2);
                    act("$n        !",
                        ch, NULL, victim, TO_VICT);
                    act("       $N!",
                        ch, NULL, victim, TO_CHAR);
                    act("$n        $N.",
                        ch, NULL, victim, TO_NOTVICT);
                } else
                {
                    DAZE_STATE (victim, PULSE_VIOLENCE * 1);
                    act("$n       !",
                        ch, NULL, victim, TO_VICT);
                    act("      $N!",
                        ch, NULL, victim, TO_CHAR);
                    act("$n       $N.",
                        ch, NULL, victim, TO_NOTVICT);
                }
            } else
            {
                act("$n      !", 
                    ch, NULL, victim, TO_VICT);
                act("     $N!", 
                    ch, NULL, victim, TO_CHAR);
                act("$n      $N.", 
                    ch, NULL, victim, TO_NOTVICT);
            }
            check_improve(ch, gsn_claws, TRUE, 4);
            
            dam = bonus_dam;

            // w weapon less damage on PC
            if (!IS_NPC(victim) && !hand[i])    dam = dam / 3 * 2;
            if (!IS_NPC(victim))                dam = dam / 3 * 2;
                    
            if (number_percent () < get_skill (ch, gsn_roar) / 2)
            {
                check_improve (ch, gsn_roar, TRUE, 4);

                if (IS_SET(ch->form, FORM_WOLF)
                && number_percent () < get_curr_stat(ch, STAT_LCK))
                {
                    act("$n     !",
                        ch, NULL, victim, TO_VICT | ACT_VERBOSE);
                    act("     !",
                        ch, NULL, victim, TO_CHAR | ACT_VERBOSE);
                    act("$n     !",
                        ch, NULL, victim, TO_NOTVICT | ACT_VERBOSE);
                    dam += dam / 4 ;
                } else if (IS_SET(ch->form, FORM_BEAR)
                && number_percent () < get_curr_stat(ch, STAT_LCK))
                {
                    act("$n     !",
                        ch, NULL, victim, TO_VICT | ACT_VERBOSE);
                    act("     !",
                        ch, NULL, victim, TO_CHAR | ACT_VERBOSE);
                    act("$n     !",
                        ch, NULL, victim, TO_NOTVICT | ACT_VERBOSE);
                    dam += dam / 2 ;
                } else
                {
                    act("$n   !",
                        ch, NULL, victim, TO_VICT | ACT_VERBOSE);
                    act("   $N!",
                        ch, NULL, victim, TO_CHAR | ACT_VERBOSE);
                    act("$n   $N!",
                        ch, NULL, victim, TO_NOTVICT | ACT_VERBOSE);
                    dam += dam / 8 ;
                    check_improve (ch, gsn_roar, FALSE, 1);
                }
            }            
            if (victim != NULL)
                damage(ch, victim, dam, gsn_claws, DAM_PIERCE, TRUE); 
            if (JUST_KILLED(victim))
                return;
        } else
        {
            damage(ch, victim, 0, gsn_claws, DAM_PIERCE, TRUE);
            check_improve(ch, gsn_claws, FALSE, 1);
        }
    }

    if (check_yell(ch, victim, fighting) && ch != victim)
    {
        act_yell(victim, " ! $i  !", ch, NULL);
        agent_net_printf(victim, ch, ANET_ATTACK);
    }
}

DO_FUN(do_autorepair)
{
    int heal, chance, mana;

    if ((chance = get_skill(ch,gsn_autorepair)) == 0) {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_AUTOREPAIR)) {
        char_act("    .", ch);
        return;
    }

    mana = SKILL(gsn_autorepair)->min_mana;
    if (ch->mana < mana) {
        char_act("   .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_autorepair)->beats);
    if (number_percent() < chance) {
        AFFECT_DATA af;

        char_act("   .  : [********]!", ch);
        act("$n        - .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_autorepair, TRUE, 2);

        heal = dice(4, 8) + LVL(ch);
        ch->hit = UMIN(ch->hit + heal, ch->max_hit);
        update_pos(ch);
        char_act("   !", ch);

        af.where    = TO_AFFECTS;
        af.type     = gsn_autorepair;
        af.level    = LVL(ch);
        af.duration = LVL(ch) / 4;
        af.modifier = UMIN(15,LVL(ch)/2);
        af.bitvector    = AFF_AUTOREPAIR;
        af.location = 0;
        affect_to_char(ch,&af);

        ch->mana -= mana;
    }
    else {
        char_act("        .\n    .", ch);
        check_improve(ch, gsn_autorepair, FALSE, 2);
        ch->mana -= mana / 2;
    }
}

DO_FUN(do_pwipe)
{
    CHAR_DATA *victim;
    AFFECT_DATA af;
    char arg[MAX_INPUT_LENGTH];

    if (get_skill(ch, gsn_pwipe) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0') {
        char_act("whom?", ch);
        return;
    }
    else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (IS_NPC(victim))
        return;

    af.where        = TO_AFFECTS;
    af.type         = gsn_pwipe;
    af.level        = ch->level;
    af.duration     = 10;
    af.location     = APPLY_NONE;
    af.bitvector    = 0;
    af.modifier     = 0;
    affect_to_char(victim, &af);

    return;
}

DO_FUN(do_lion_call)
{
        CHAR_DATA *     gch;
        CHAR_DATA *     lion;
        CHAR_DATA *     lion2;
        AFFECT_DATA     af;
        int             i;
        int             chance;
        int             sn;
        int             mana;

        if ((sn = sn_lookup("lion call")) < 0
        ||  (chance = get_skill(ch, sn)) == 0) {
                char_act("Huh?", ch);
                return;
        }

        char_act("     .", ch);
        act("$n    .",ch,NULL,NULL,TO_ROOM);

        if (is_affected(ch, sn)) {
                char_act("       .", ch);
                return;
        }

        for (gch = char_list; gch != NULL; gch = gch->next) {
                if (IS_NPC(gch) && IS_AFFECTED(gch,AFF_CHARM)
                &&  gch->master == ch
                && gch->pIndexData->vnum == MOB_VNUM_LION) {
                    char_act("    ,   ?", ch);
                    return;
                }
        }

        if (ch->in_room != NULL
        && IS_SET(ch->in_room->room_flags, ROOM_NOMOB)) {
                char_act("   .", ch);
                return;
        }

        if (IS_SET(ch->in_room->room_flags, ROOM_SAFE | ROOM_PEACE |
                                            ROOM_PRIVATE | ROOM_SOLITARY)
        ||  (ch->in_room->exit[0] == NULL && ch->in_room->exit[1] == NULL
        &&   ch->in_room->exit[2] == NULL && ch->in_room->exit[3] == NULL
        &&   ch->in_room->exit[4] == NULL && ch->in_room->exit[5] == NULL)
        ||  (ch->in_room->sector_type != SECT_FIELD
        &&   ch->in_room->sector_type != SECT_FOREST
        &&   ch->in_room->sector_type != SECT_MOUNTAIN
        &&   ch->in_room->sector_type != SECT_HILLS)) {
                char_act("    .", ch);
                return;
        }

        mana = SKILL(sn)->min_mana;
        if (ch->mana < mana) {
                char_act("      .", ch);
                return;
        }
        ch->mana -= mana;

        if (number_percent() > chance) {
                check_improve(ch, sn, FALSE, 1);
                char_act("   .", ch);
                return;
        }

        check_improve(ch, sn, TRUE, 1);
        lion = create_mob(get_mob_index(MOB_VNUM_LION));

        for (i=0;i < MAX_STATS; i++)
                lion->perm_stat[i] = UMIN(25,2 * ch->perm_stat[i]);

        lion->max_hit = IS_NPC(ch) ? ch->max_hit : ch->pcdata->perm_hit;
        lion->hit = lion->max_hit;
        lion->max_mana = IS_NPC(ch) ? ch->max_mana : ch->pcdata->perm_mana;
        lion->mana = lion->max_mana;
        lion->alignment = ch->alignment;
        lion->level = UMIN(100,1 * ch->level-2);
        for (i=0; i < 3; i++)
                lion->armor[i] = interpolate(lion->level,100,-100);
        lion->armor[3] = interpolate(lion->level,100,0);
        lion->sex = ch->sex;
        lion->gold = 0;

    lion->damage[DICE_NUMBER] = number_range(9, 11);
    lion->damage[DICE_TYPE]   = number_range(lion->level/10, lion->level/8);
    lion->damage[DICE_BONUS]  = 0;

    lion->hitroll = lion->level/2;
    lion->damroll = lion->level/2;

        lion2 = create_mob(lion->pIndexData);
        clone_mob(lion,lion2);

        SET_BIT(lion->affected_by, AFF_CHARM);
        SET_BIT(lion2->affected_by, AFF_CHARM);
        lion->master = lion2->master = ch;
        lion->leader = lion2->leader = ch;

        char_to_room(lion,ch->in_room);
        char_to_room(lion2,ch->in_room);
        char_act("     !", ch);
        act("  ,   $n!",ch,NULL,NULL,TO_ROOM);

        af.where              = TO_AFFECTS;
        af.type               = sn;
        af.level              = ch->level;
        af.duration           = SKILL(sn)->beats;
        af.bitvector          = 0;
        af.modifier           = 0;
        af.location           = APPLY_NONE;
        affect_to_char(ch, &af);
}

DO_FUN(do_mech)
{
    CHAR_DATA *mob;
    int chance;
    int mana;

    for (mob = ch->in_room->people; mob; mob = mob->next_in_room)
        if (IS_NPC(mob) && IS_SET(mob->pIndexData->act, ACT_REPAIRMAN))
            break;

    if (mob == NULL)
    {
        char_act("    ,     .", ch);
        return;
    }

    if (IS_NPC(ch) || !IS_CYBORG(ch) || (chance = get_skill(ch, gsn_mech)) == 0) 
    {
        act("     .", ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (IS_SET(ch->form, FORM_MECH)) 
    {
        act("  .", ch, NULL, NULL, TO_CHAR);
        return;
    }

    mana = SKILL(gsn_mech)->min_mana;
    if (ch->mana < mana) 
    {
        char_act("     .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_mech)->beats);

    if (number_percent() < chance) 
    {
        AFFECT_DATA af;

        if (is_affected(ch, sn_lookup("clan aura")))
            affect_strip(ch, sn_lookup("clan aura"));

        af.where     = TO_AFFECTS;
        af.type      = gsn_mech;
        af.level     = LVL(ch);
        af.duration  = 1 + LVL(ch);

        af.modifier  = LVL(ch) / 4;
        af.location  = APPLY_HITROLL;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.modifier  = LVL(ch) / 9;
        af.location  = APPLY_DAMROLL;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_FORM;
        af.location  = APPLY_AC;
        af.modifier  = - LVL(ch);
        af.bitvector = FORM_MECH;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = -10;
        af.location  = APPLY_SAVING_SPELL;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = LVL(ch);
        af.location  = APPLY_HIT;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = LVL(ch) * 2;
        af.location  = APPLY_MANA;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = LVL(ch);
        af.location  = APPLY_MOVE;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = 2;
        af.location  = APPLY_STR;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = 2;
        af.location  = APPLY_DEX;
        af.bitvector = 0;
        affect_to_char(ch, &af);

        af.where     = TO_AFFECTS;
        af.modifier  = 2;
        af.location  = APPLY_SIZE;
        af.bitvector = 0;
        affect_to_char(ch, &af);
/*
        af.where     = TO_RESIST;
        af.modifier  = 40;
        af.location  = APPLY_NONE;
        af.bitvector = DAM_ENERGY;
        affect_to_char(ch, &af);
*/
        ch->mana -= mana;

        act("   !", ch, NULL, NULL, TO_CHAR);
        act("$n   !", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_mech, TRUE, 1);
    } else 
    {
        ch->mana -= mana/2;

        char_act("    ,     .", ch);
        act("$n    ,     .", 
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_mech, FALSE, 1);
    }
}

DO_FUN(do_deathgrip)
{
    int chance, mana;
    OBJ_DATA *obj;
    AFFECT_DATA af;

    if ((chance = get_skill(ch, gsn_deathgrip)) == 0) {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_deathgrip)) {
        char_act("You already have a grip of death.", ch);
        return;
    }

    mana = SKILL(gsn_deathgrip)->min_mana;
    if (ch->mana < mana) {
        char_act("     .", ch);
        return;
    }

    if (number_percent() > chance) {
        char_act("You failed to create a grip of death.", ch);
        ch->mana -= mana/2;
        check_improve(ch, gsn_deathgrip, FALSE, 1);
        return;
    }

        obj = get_eq_char(ch, WEAR_WIELD);

        if ((obj != NULL) && !IS_OBJ_STAT(obj,ITEM_DEATH)) {
                SET_BIT( obj->extra_flags, ITEM_DEATH);
                act("$p flickers with dark power.",ch,obj,NULL,TO_ALL);
        }

        /* Now for adding the affect to the player */

        af.where        = TO_AFFECTS;
        af.type         = gsn_deathgrip;          
        af.level        = ch->level;
        af.duration     = ch->level / 3;
        af.location     = APPLY_DAMROLL;
        af.modifier     = ch->level / 8;
//        af.bitvector    = gsn_deathgrip;
        af.bitvector    = 0;

        affect_to_char(ch, &af);

        act("$n's hands are shrouded with a black mist.",ch,NULL,NULL,TO_ROOM);
        char_act("Your hands are shrouded with a black mist.", ch);

    ch->mana -= mana;
        check_improve(ch,gsn_deathgrip,TRUE,1);
}


DO_FUN(do_weaponsong)
{
    int chance, need, chars;
    OBJ_DATA *obj;
    AFFECT_DATA af;

    if ((chance = get_skill(ch, gsn_weaponsong)) == 0) {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_deafen)) 
        return;

    if (is_affected(ch, gsn_garble))
    { 
        char_act("    !", ch);
        return; 
    }

    if (is_affected(ch, gsn_weaponsong)) 
    {
        char_act("   ϣ   ϣ !.", ch);
        return;
    }

    need = SKILL(gsn_weaponsong)->min_mana;
    chars = IS_SET (muddy_mode, MUDDY_WARR_MOVES) ? ch->move : ch->mana;

    if (chars < need) 
    {
        char_act(" $gn{} .", ch);
        return;
    }

    if (number_percent() > chance) 
    {
        char_act(" $gn{}    .", ch);
        if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
            ch->move -= need/2;
        else
            ch->mana -= need/2;
        check_improve(ch, gsn_weaponsong, FALSE, 1);
        return;
    }

    /* Now for adding the affect to the player */

    af.where        = TO_AFFECTS;
    af.type         = gsn_weaponsong;          
    af.level        = ch->level;
    af.duration     = ch->level / 3;
    af.location     = APPLY_DAMROLL;
    af.modifier     = ch->level / 8;
//  af.bitvector    = gsn_deathgrip;
    af.bitvector    = 0;
    // affect to char
    affect_to_char(ch, &af);

    af.location     = APPLY_HITROLL;
    af.modifier     = ch->level/8;
    af.where    = TO_AFFECTS;
    af.bitvector    = 0;    

    affect_to_char(ch, &af);

    af.where = TO_RESIST;
    af.location = APPLY_NONE;
    af.modifier = 20;
    af.bitvector = DAM_SOUND;
    affect_to_char(ch, &af);

    act("$n    ϣ .", ch, NULL, NULL, TO_ROOM);
    act("    ϣ $c5{$t} .", ch, flag_string(align_names, NALIGN(ch)), NULL, TO_CHAR | ACT_TRANS);

    obj = get_eq_char(ch, WEAR_WIELD);

    if ((obj != NULL) && !IS_OBJ_STAT(obj,ITEM_SINGING)) {
        SET_BIT(obj->extra_flags, ITEM_SINGING);
        act("$p  .", ch, obj, NULL, TO_CHAR);
        act("$p  $n.", ch, obj, ch, TO_ROOM);
    }

    if (IS_SET (muddy_mode, MUDDY_WARR_MOVES))
        ch->move -= need;
    else
        ch->mana -= need;
    check_improve(ch, gsn_weaponsong, TRUE, 1);
}

DO_FUN(do_sharpen)
{
    char arg[MAX_INPUT_LENGTH];
    OBJ_DATA *obj;
    OBJ_DATA *stone;
    AFFECT_DATA af;
    int       chance = 0;

    one_argument( argument, arg , sizeof(arg ));

    if (arg[0] == '\0')
    {
        char_act("What are you trying to sharpen?", ch);
        return;
        }
    
    if (ch->fighting)
    {
        char_act("While you are fighting? Yea right!.", ch);
        return;
    }

    obj = get_obj_carry(ch, arg);
//  obj = get_eq_char(ch, WEAR_WIELD);

    if (obj == 0)
    {
        char_act("What are you trying to sharpen?", ch);
        return;
    }

    if (check_material(obj, "unique"))
    {
        act_puts("You can't sharpen $p.", ch, obj, NULL, TO_CHAR, POS_DEAD);
        return;
    }

    if ((obj->timer > 0) 
    || (obj->pIndexData->limit != -1))
    {
        char_act("You can't do this!", ch);
        return;
    }

    if ((obj->value[0] == WEAPON_MACE) 
    || (obj->value[0] == WEAPON_FLAIL)
    || (obj->value[0] == WEAPON_WHIP)
    || (obj->value[0] == WEAPON_BOW))
    {
        char_act("This weapon unsharpened!", ch);
        return;
    }

    if (obj->pIndexData->item_type != ITEM_WEAPON)
    {
        char_act("You can sharpen only weapon!", ch);
        return;
    }

    if (IS_WEAPON_STAT(obj,WEAPON_SHARP))
    {
        char_act("That weapon is already sharpened.", ch);
        return;
    }

/* Ok we have a sharpenable weapon but do we have the stone */ 
    stone = get_eq_char(ch,WEAR_HOLD);

    if (stone == NULL 
    || stone->pIndexData->vnum != OBJ_VNUM_SHARPEN_STONE)
    {
        char_act("You do not hold a sharpening stone.", ch);
        return;
    }

    extract_obj(stone);

    chance += get_skill(ch, gsn_sharpen);
    chance /= 2;
    chance += (get_curr_stat(ch, STAT_DEX) + get_curr_stat(ch, STAT_STR)) / 2;
    chance += get_curr_stat(ch, STAT_LCK);

    if (!is_class_warrior(ch))
        chance /= 2;

    if (number_percent() > chance)
    {
        char_act("Your hurt yourself. Cry... Cry...", ch);
        damage(ch, ch, ch->level, gsn_sharpen, DAM_NONE, FALSE);
        check_improve(ch, gsn_sharpen, FALSE, 1);
        return;
    }

    af.where         = TO_WEAPON;
    af.type          = gsn_sharpen;
    af.level         = ch->level;
    af.duration      = -1;
    af.location      = 0;
    af.modifier      = 0;
    af.bitvector     = WEAPON_SHARP;
    affect_to_obj(obj,&af);

    act("You sharpen $p.",ch,obj,NULL,TO_CHAR);
    act("$n pulls out a piece of stone and begins sharpening $p.",ch,obj,NULL,TO_ROOM);

    check_improve(ch, gsn_sharpen, TRUE, 1);
    return;
}

DO_FUN(do_axedigging)
{
    CHAR_DATA *vch;
    CHAR_DATA *vch_next;
    OBJ_DATA *obj;
    int chance;

    if ((chance = get_skill(ch, gsn_axedigging)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        char_act("  ?  !!!", ch);
        return;
    }

    if (ch->position == POS_RESTING
    || ch->position ==  POS_SLEEPING
    || ch->position ==  POS_SITTING)
        return;

    if (IS_AFFECTED(ch, AFF_SLEEP))
    {
        char_act(" ,   .", ch);
        return;
    }

    if (IS_AFFECTED(ch,AFF_FEAR))
    {
        char_act("You too scare to do this!     ?", ch);
        return;
    }

    if ((obj = get_eq_char(ch, WEAR_WIELD)) == NULL)
    {
        char_act("  ,  .", ch);
        return;
    }

    if (obj->value[0] != WEAPON_AXE)
    {
        char_act("  ,  .", ch);
        return;
    }

    if (MOUNTED(ch))
    {
        char_act("   ,   !", ch);
        return;
    }
/* GM :    ,    -   ? ? ??? !
    if (IS_AFFECTED(ch, AFF_FLYING)) {
         char_act(" ? ...", ch);
         return;
    }
*/
    if (ch->move < 40 && !IS_NPC(ch))
    {
         char_act("  .", ch);
         return;
    }

    WAIT_STATE(ch, SKILL(gsn_axedigging)->beats);
    if (number_percent() > chance)
    {
        char_act("   .", ch);
        check_improve(ch,gsn_axedigging,FALSE,1);
        return;
    }

    /* modifiers */
    chance = chance * 4/5;

    /* terrain */

    switch(ch->in_room->sector_type) {
    case(SECT_INSIDE):          chance -= 20;   break;
    case(SECT_CITY):            chance -= 10;   break;
    case(SECT_FIELD):           chance +=  5;   break;
    case(SECT_FOREST):                          break;
    case(SECT_HILLS):                           break;
    case(SECT_MOUNTAIN):        chance -= 10;   break;
    case(SECT_WATER_SWIM):      chance  =  0;   break;
    case(SECT_WATER_NOSWIM):    chance  =  0;   break;
    case(SECT_AIR):             chance  =  0;   break;
    case(SECT_DESERT):          chance += 10;   break;
    }

    if (chance == 0)
    {
        char_act("  .", ch);
        return;
    }

    char_act("    .", ch);
    act("$n    !", ch, NULL, NULL, TO_ROOM);

    check_improve(ch, gsn_axedigging, TRUE, 1);
    
    for (vch = ch->in_room->people; vch; vch = vch_next)
    {
        vch_next = vch->next_in_room;

        if (is_safe_spell(ch, vch, TRUE))
            continue;
                
        /* dexterity */
        chance += get_curr_stat(ch, STAT_DEX);
        chance -= 2 * get_curr_stat(vch, STAT_DEX);

        /* speed  */
        if ((IS_NPC(ch) && IS_SET(ch->pIndexData->off_flags, OFF_FAST))
        ||  IS_AFFECTED(ch, AFF_HASTE))
            chance += 10;
        if ((IS_NPC(vch) && IS_SET(vch->pIndexData->off_flags, OFF_FAST))
        ||  IS_AFFECTED(vch, AFF_HASTE))
            chance -= 25;

        /* level */
        chance += (LVL(ch) - LVL(vch)) * 2;

        if (chance % 5 == 0)
            chance += 1;

        if (vch == ch)
            chance /= 2;

        if (number_percent() < chance 
        && !is_affected(vch,gsn_axedigging) && !IS_UNDEAD(vch))
        {
            AFFECT_DATA af;
            act("$n   ,   !", vch, NULL, NULL, TO_ROOM);
            char_act("   ,   !", vch);

            af.where    = TO_AFFECTS;
            af.type     = gsn_axedigging;
            af.level    = LVL(ch);
            af.duration = number_fuzzy(1);
            af.location = APPLY_HITROLL;
            af.modifier = 0 - number_fuzzy(LVL(vch)/5);
            af.bitvector    = AFF_BLIND;

            affect_to_char(vch, &af);
        }
        if (vch != ch)
            multi_hit(vch, ch, TYPE_UNDEFINED);
    }
}

DO_FUN(do_stimpack)
{
    int chance, hp_percent, level, duration;

    if ((chance = get_skill(ch, gsn_stimpack)) == 0) {
        char_act("   ,  .", ch);
        return;
    }

    if (is_affected(ch, gsn_stimpack)) {
        char_act("ݣ   أ !", ch);
        return;
    }

    /* modifiers */

    hp_percent = 100 * ch->hit/ch->max_hit;
    chance -= hp_percent/2;
    chance = URANGE(5, chance, 95);

    if (number_percent() < chance) {
        AFFECT_DATA af;

        WAIT_STATE(ch, PULSE_VIOLENCE);

        char_act("   -!", ch);
        act("$n   -!",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_stimpack, TRUE, 2);


    level = LVL(ch);
    duration = level / 10 + 5;

    af.type      = gsn_stimpack;
    af.level     = level;
    af.duration  = duration;

/* haste */
    af.where     = TO_AFFECTS;
    af.location  = APPLY_DEX;
    af.modifier  = 1 + (level / 20);
    af.bitvector = AFF_HASTE;
    affect_to_char(ch, &af);

/* damroll */
    af.where     = TO_AFFECTS;
    af.location  = APPLY_DAMROLL;
    af.modifier  = level/2;
    af.bitvector = AFF_BERSERK;
    affect_to_char(ch, &af);

/* hitroll */
    af.where     = TO_AFFECTS;
    af.location  = APPLY_HITROLL;
    af.modifier  = level/2;
    af.bitvector = AFF_PASS_DOOR;
    affect_to_char(ch, &af);

    }
    else {
        WAIT_STATE(ch,2 * PULSE_VIOLENCE);
        char_act("-  .  ݣ .", ch);
        check_improve(ch, gsn_stimpack, FALSE, 2);
    }
}

DO_FUN(do_stealth)
{
    AFFECT_DATA af;
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_stealth)) == 0) 
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_stealth)) 
    {
        char_act("  .", ch);
        return;
    }

    mana = SKILL(gsn_stealth)->min_mana;
    if (ch->mana < mana) 
    {
        char_act("     .", ch);
        return;
    }

    if (IS_PUMPED(ch))
    {
        char_act("You can't stealth while pumped.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_stealth)->beats);

    if (number_percent() > chance) 
    {
        char_act("     .", ch);
        act("$n     .",
            ch, NULL, NULL, TO_ROOM);
        ch->mana -= mana/2;
        check_improve(ch, gsn_stealth, FALSE, 1);
        return;
    }

    ch->mana -= mana;
    check_improve(ch, gsn_stealth, TRUE, 1);

    af.where     = TO_AFFECTS;
    af.type      = gsn_stealth;
    af.level     = LVL(ch);
    af.duration  = LVL(ch)*4;
    af.location  = APPLY_SAVES;
    af.modifier  = -10;
    af.bitvector = AFF_SNEAK;
    affect_to_char(ch, &af);

    char_act("   .  !", ch);
}

DO_FUN(do_medical_help)
{
    int heal, chance, mana, nsn;
    int level = LVL(ch) + 1;

    if ((chance = get_skill(ch,gsn_medical_help)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_medical_help))
    {
        char_act("   .", ch);
        return;
    }

    mana = SKILL(gsn_medical_help)->min_mana;
    if (ch->mana < mana) {
        char_act("   .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_medical_help)->beats);
    if (number_percent() < chance) {
        AFFECT_DATA af;

        if (IS_AFFECTED(ch,AFF_BLIND))
        {
            nsn = sn_lookup("cure blindness");
            spell_cure_blindness(nsn,level,ch,(void *)ch,TARGET_CHAR);
        }
        if (IS_AFFECTED(ch,AFF_POISON))
        {
            spell_cure_poison(gsn_cure_poison,level,ch,(void *)ch,TARGET_CHAR);
        }
        if (IS_AFFECTED(ch,AFF_PLAGUE))
        {
            nsn = sn_lookup("cure disease");
            spell_cure_disease(nsn,level,ch,(void *)ch,TARGET_CHAR);
        }

        char_act("     .", ch);
        act("$n     .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_medical_help, TRUE, 2);

        heal = dice(4, 8) + LVL(ch);
        ch->hit = UMIN(ch->hit + heal, ch->max_hit);
        update_pos(ch);
        char_act("   !", ch);

        af.where        = TO_AFFECTS;
        af.type         = gsn_medical_help;
        af.level        = LVL(ch);
        af.duration     = LVL(ch) / 4;
        af.modifier     = UMIN(15,LVL(ch)/2);
        af.bitvector    = AFF_PROTECTION;
        af.location     = 0;
        affect_to_char(ch,&af);

        ch->mana -= mana;
    }
    else {
        char_act("    .", ch);
        check_improve(ch, gsn_medical_help, FALSE, 2);
        ch->mana -= mana / 2;
    }
}

DO_FUN(do_beret)
{
    OBJ_DATA *beret;
    AFFECT_DATA af;
    OBJ_DATA *part;
    char arg[MAX_INPUT_LENGTH];
    int level;
    int chance;
    int mana;
    int w = 1;

    if ((chance = get_skill(ch, gsn_beret)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    WAIT_STATE(ch, MISSING_TARGET_DELAY);

    if (is_affected(ch, gsn_beret))
    {
        char_act("      !", ch);
        return;
    }

    mana = SKILL(gsn_beret)->min_mana;
    if (ch->mana < mana)
    {
        char_act("      ,   .", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        char_act("   ?", ch);
        return;
    }

    if ((part = get_obj_here(ch, arg)) == NULL)
    {
        char_act("    .", ch);
        return;
    }
    if (part->pIndexData->item_type != ITEM_CORPSE_PC
    && part->pIndexData->item_type != ITEM_CORPSE_NPC)
    {
        char_act("      .", ch);
        return;
    }
    if (part->carried_by != NULL)
    {
        char_act("  .", ch);
        return;
    }

    if  (!can_loot(ch,part))
    {
        char_act("    .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_beret)->beats);

    if (number_percent() < chance * 2 / 3)
    {
        char_act("    .", ch);
        return;
    }

    if (!IS_NPC(ch) && number_percent() < chance)
    {
        af.where        = TO_AFFECTS;
        af.type         = gsn_beret;
        af.level        = LVL(ch);
        af.duration     = (48 / w);
        af.modifier     = 0;
        af.bitvector    = 0;
        af.location     = 0;

        affect_to_char(ch,&af);

        if (part->pIndexData->item_type == ITEM_CORPSE_PC)
            w = 2;

        level = UMIN(part->level, MAX_LEVEL-1);

        beret = create_named_obj(get_obj_index(OBJ_VNUM_BERET),
                        level, mlstr_mval(part->short_descr));

        beret->timer = LVL(ch) * 2;
        beret->cost  = 0;
        beret->level = ch->level;

        ch->mana -= mana;

        af.where        = TO_OBJECT;
        af.type         = gsn_beret;
        af.level        = level;
        af.duration     = -1;
        af.location     = APPLY_DAMROLL;
        af.modifier     = ((LVL(ch) / 10) * w);
        af.bitvector    = 0;
        affect_to_obj(beret, &af);

        af.location     = APPLY_HITROLL;
        af.modifier     = ((LVL(ch) / 10) * w);
        af.bitvector    = 0;
        affect_to_obj(beret, &af);

        af.location     = APPLY_SAVING_SPELL;
        af.modifier     = - 5;
        af.bitvector    = 0;
        affect_to_obj(beret, &af);

        af.location     = APPLY_HIT;
        af.modifier     = (LVL(ch) * (w+1));
        af.bitvector    = 0;
        affect_to_obj(beret, &af);

        af.location     = APPLY_MANA;
        af.modifier     = (LVL(ch) * (w+2));
        af.bitvector    = 0;
        affect_to_obj(beret, &af);

        obj_to_char(beret, ch);
        check_improve(ch, gsn_beret, TRUE, 1);

        act("      $p!", ch, part, NULL, TO_CHAR);
        act("$n      $p!", ch, part, NULL, TO_ROOM);

        return;
    }
    else {
        char_act("     .", ch);
        ch->mana -= mana/2;
        check_improve(ch, gsn_beret, FALSE, 1);
    }
}

DO_FUN(do_rejuvenation)
{
    int heal, chance, mana, nsn;
    int level = LVL(ch);
    CHAR_DATA   *victim;
    char        arg[MAX_INPUT_LENGTH];
    AFFECT_DATA *af;
    AFFECT_DATA af2;

    one_argument(argument, arg, sizeof(arg));

    if ((chance = get_skill(ch, gsn_rejuvenation)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    mana = SKILL(gsn_rejuvenation)->min_mana;
    if (ch->mana < mana)
    {
        char_act("You don't have enough mana.", ch);
        return;
    }
    ch->mana -= mana;

    if (arg[0] == '\0')
        victim = ch;
    else if ((victim = get_char_room(ch, arg)) == NULL) 
    {
        char_act("  .", ch);
        return;
    }

    if (number_percent () > chance)
    {
        char_act ("You failed.", ch);
        WAIT_STATE(ch, (SKILL(gsn_rejuvenation)->beats)/2);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_rejuvenation)->beats);

    if ((ch!=victim) && check_mirror_image(ch, victim))
        return;

    if (is_affected(victim, gsn_rejuvenation))
        chance /= 2;

    if (IS_UNDEAD(victim))
        chance = 0;

    if (chance == 0)
    {
        act("$n can't rejuvenate you!", ch, NULL, victim, TO_VICT);
        char_act("You can't rejuvenate this!", ch);
        return;
    }

    level += chance / 10;


    if (IS_AFFECTED(victim, AFF_BLIND))
    {
        nsn = sn_lookup("cure blindness");
        spell_cure_blindness(nsn, level, ch, victim, TARGET_CHAR);
    }
    if (IS_AFFECTED(victim, AFF_POISON))
    {
        spell_cure_poison(gsn_cure_poison, level, ch, victim, TARGET_CHAR);
    }
    if (IS_AFFECTED(victim, AFF_PLAGUE))
    {
        nsn = sn_lookup("cure disease");
        spell_cure_disease(nsn, level, ch, victim, TARGET_CHAR);
    }

    if (victim == ch) 
    {
        char_act("You start rejuvenation process.", ch);
        act("$n starting rejuvenation process.", ch, NULL, NULL, TO_ROOM);
    } 
    else
    {
        act("You rejuvenate $N.", ch, NULL, victim, TO_CHAR);
        act("$n rejuvenates $N.", ch, NULL, victim, TO_NOTVICT);
        act("$n rejuvenates you.", ch, NULL, victim, TO_VICT);
    }

    heal = dice(10, 10) + UMAX(50, level * 2);
    victim->hit = UMIN(victim->hit + heal, victim->max_hit);
    update_pos(victim);

    for (af = victim->affected; af; af = af->next)
    {
        if (af->type == gsn_cause_critical)
        {
            if (number_percent() < get_curr_stat(victim, STAT_LCK))
            {
                affect_remove(victim, af);
                char_act("Your wounds stopped bleeding so critically.", victim);
            }
        }
        if (af->type == gsn_cause_serious)
        {
            if (number_percent() < get_curr_stat(victim, STAT_LCK))
            {
                affect_remove(victim, af);
                char_act("Your wounds stopped bleeding so seriously.", victim);
            }
        }
        if (af->type == gsn_cause_light)
        {
            if (number_percent() < get_curr_stat(victim, STAT_LCK))
            {
                affect_remove(victim, af);
                char_act("Your wounds stopped bleeding.", victim);
            }
        }
    }
    if (!is_affected(victim, gsn_rejuvenation))
    {
        af2.where        = TO_AFFECTS;
        af2.type         = gsn_rejuvenation;
        af2.level        = ch->level;
        af2.duration     = LVL(ch) / 2;
        af2.modifier     = UMIN(15, LVL(ch) / 2);
        af2.bitvector    = AFF_REGENERATION;
        af2.location     = 0;
        affect_to_char(victim, &af2);
        check_improve(ch, gsn_rejuvenation, TRUE, 2);
    } 
    else
        check_improve(ch, gsn_rejuvenation, FALSE, 2);
}

DO_FUN(do_bearform)
{
    int chance;
    int mana;

    if (IS_NPC(ch)
    || (chance = get_skill(ch, gsn_bearform)) == 0)
    {
        act("      .", 
            ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (IS_SET(ch->form, FORM_BEAR | FORM_WOLF))
    {
        act("  !", ch, NULL, NULL, TO_CHAR);
        return;
    }

    mana = SKILL(gsn_bearform)->min_mana;
    if (ch->mana < mana)
    {
        char_act("     .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_bearform)->beats);

    if (number_percent() < chance)
    {
    AFFECT_DATA af;

    af.where    = TO_AFFECTS;
    af.type     = gsn_bearform;
    af.level    = LVL(ch);
    af.duration = LVL(ch) * 2;

    af.location = APPLY_DEX;
    af.modifier = -5;
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_HITROLL;
    af.modifier = ((-1)*(LVL(ch)/4));
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_DAMROLL;
    af.modifier = LVL(ch)/3;
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_SIZE;
    af.modifier = 3;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    af.location = APPLY_STR;
    af.modifier = 25;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    af.location = APPLY_HIT;
    af.modifier = LVL(ch) * 4;
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_AC;
    af.modifier = - UMAX(50, LVL(ch) * 5/2);
    af.where     = TO_FORM;
    af.bitvector = FORM_BEAR;
    affect_to_char( ch, &af );

    ch->mana -= mana;

    act("    !", ch, NULL, NULL, TO_CHAR);
    act("$n    !", ch, NULL, NULL, TO_ROOM);
    check_improve(ch, gsn_bearform, TRUE, 1);
    }
    else {
        ch->mana -= mana/2;

        char_act("     ,     .", ch);
        act("$n     ,     .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_bearform, FALSE, 1);
    }
}

DO_FUN(do_wolfform)
{
    int chance;
    int mana;

    if (IS_NPC(ch)
    || (chance = get_skill(ch, gsn_wolfform)) == 0)
    {
        act("      .", 
            ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (IS_SET(ch->form, FORM_BEAR | FORM_WOLF))
    {
        act("  !", ch, NULL, NULL, TO_CHAR);
        return;
    }

    mana = SKILL(gsn_wolfform)->min_mana;
    if (ch->mana < mana)
    {
        char_act("     .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_wolfform)->beats);

    if (number_percent() < chance)
    {
    AFFECT_DATA af;

    af.where    = TO_AFFECTS;
    af.type     = gsn_wolfform;
    af.level    = LVL(ch);
    af.duration = LVL(ch) * 2;

    af.location = APPLY_DEX;
    af.modifier = 25;
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_HITROLL;
    af.modifier = LVL(ch);
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_DAMROLL;
    af.modifier = ((-1)*(LVL(ch)/8));
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location = APPLY_SIZE;
    af.modifier = 1;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    af.location = APPLY_STR;
    af.modifier = -5;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    af.location = APPLY_HIT;
    af.modifier = LVL(ch) * 2;
    af.bitvector    = 0;
    affect_to_char(ch,&af);

    af.location  = APPLY_AC;
    af.modifier  = - UMAX(50, LVL(ch) * 5/2);
    af.where     = TO_FORM;
    af.bitvector = FORM_WOLF;
    affect_to_char( ch, &af );

    ch->mana -= mana;

    act("    !", ch, NULL, NULL, TO_CHAR);
    act("$n    !", ch, NULL, NULL, TO_ROOM);
    check_improve(ch, gsn_wolfform, TRUE, 1);
    }
    else {
        ch->mana -= mana/2;

        char_act("     ,     .", ch);
        act("$n     ,     .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_wolfform, FALSE, 1);
    }
}

DO_FUN(do_metamorphosis)
{
    AFFECT_DATA af;
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_metamorphosis)) == 0) {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_metamorphosis))
    {
        char_act("    .", ch);
        return;
    }

    mana = SKILL(gsn_metamorphosis)->min_mana;
    if (ch->mana < mana)
    {
        char_act("   .", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_metamorphosis)->beats);

    if (number_percent() > chance)
    {
        char_act("     .", ch);
        act("  $n   .",
            ch, NULL, NULL, TO_ROOM);
        ch->mana -= mana / 2;
        check_improve(ch, gsn_metamorphosis, FALSE, 1);
        return;
    }

    ch->mana -= mana;
    check_improve(ch, gsn_metamorphosis, TRUE, 1);

    af.where     = TO_AFFECTS;
    af.type      = gsn_metamorphosis;
    af.level     = LVL(ch);
    af.duration  = LVL(ch) / 2;
    af.location  = APPLY_HIT;
    af.modifier  = 500;
    af.bitvector = 0;
    affect_to_char(ch, &af);

    af.location  = APPLY_SAVING_SPELL;
    af.modifier  = - 10;
    affect_to_char(ch, &af);

    char_act("   ", ch);
    act("$n   .",
        ch, NULL, NULL, TO_ROOM);
    // GM : Some small FUN
    if (number_percent () < get_curr_stat(ch, STAT_LCK))
    {
        ch->hit += 500;
        ch->hit = UMIN(ch->hit, ch->max_hit);
        char_act("You feel better now.", ch);
    }
}

DO_FUN(do_thornsaura)
{
    AFFECT_DATA af;
    int chance;
    flag32_t     sector;

    if ((chance = get_skill(ch, gsn_thornsaura)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_thornsaura))
    {
        char_act("     .", ch);
        return;
    }

    sector = ch->in_room->sector_type;
    if (sector != SECT_FOREST
    &&  sector != SECT_HILLS
    &&  sector != SECT_MOUNTAIN)
    {
        char_act("   .", ch);
        act("$n     .",
            ch, NULL, NULL, TO_ROOM);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_thornsaura)->beats);

    if (number_percent() > chance)
    {
        char_act("   .", ch);
        act(" $n    .",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_thornsaura, FALSE, 1);
        return;
    }

    check_improve(ch, gsn_thornsaura, TRUE, 1);

    af.where        = TO_AFFECTS;
    af.type         = gsn_thornsaura;
    af.level        = LVL(ch);
    af.duration     = LVL(ch);
    af.modifier     = 0;
    af.bitvector    = 0;
    af.location     = 0;

    affect_to_char(ch,&af);

    char_act("          .", ch);
    act("$n          .",
        ch, NULL, NULL, TO_ROOM);
}

DO_FUN(do_immolation)
{
    AFFECT_DATA af;
    int chance;
    int mana;

    if ((chance = get_skill(ch, gsn_immolation)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_immolation))
    {
        affect_strip(ch, gsn_immolation);
        char_act("Immolation aura leaves you.", ch);
        return;
    }

    mana = ch->level;
    if (ch->mana < mana)
    {
        char_act("You need more mana for immolation.", ch);
        return;
    }

    WAIT_STATE(ch, SKILL(gsn_immolation)->beats);

    if (number_percent() > chance)
    {
        char_act( "Immolation can't fill you.", ch);
        act("Immolation can't fill $n.",
            ch, NULL, NULL, TO_ROOM);
        ch->mana -= mana / 2;
        check_improve(ch, gsn_immolation, FALSE, 1);
        return;
    }

    ch->mana -= mana;
    check_improve(ch, gsn_immolation, TRUE, 1);

    af.where        = TO_AFFECTS;
    af.type         = gsn_immolation;
    af.level        = LVL(ch);
    af.duration     = -1;
    af.modifier     = 0;
    af.bitvector    = 0;
    af.location     = 0;

    affect_to_char(ch,&af);

    char_act("Immolation fills you!", ch);
    act("Immolation fills $n!",
        ch, NULL, NULL, TO_ROOM);
}

DO_FUN(do_fortitude)
{
    int heal, chance, mana, hp_add;

    if ((chance = get_skill(ch, gsn_fortitude)) == 0)
    {
        char_act("Huh?", ch);
        return;
    }

    if (is_affected(ch, gsn_fortitude))
    {
        char_act("Your fortify at maximum!", ch);
        return;
    }

    mana = SKILL(gsn_fortitude)->min_mana;
    if (ch->mana < mana)
    {
        char_act("   .", ch);
        return;
    }

    //WAIT_STATE(ch, SKILL(gsn_medical_help)->beats);
    WAIT_STATE(ch, SKILL(gsn_fortitude)->beats);
    if (number_percent() < chance)
    {
        AFFECT_DATA af;

        char_act("Your fortify status greatly updated.", ch);
        act("$n look's more fortify.",
            ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_fortitude, TRUE, 2);

        if (LVL(ch) < 50)
            hp_add = LVL(ch);
        else if (LVL(ch) < 80)
            hp_add = LVL(ch) * 2;
        else
            hp_add = LVL(ch) * 3;

        af.where        = TO_AFFECTS;
        af.type         = gsn_fortitude;
        af.level        = LVL(ch);
        af.duration     = LVL(ch) / 2;
        af.modifier     = 0;
        af.bitvector    = AFF_PROTECTION;
        af.location     = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_AC;
        af.modifier     = - LVL(ch) * 3;
        af.bitvector    = 0;
        affect_to_char(ch, &af);

        af.location     = APPLY_HIT;
        af.modifier     = hp_add;
        af.bitvector    = 0;
        affect_to_char(ch,&af);

        af.location     = APPLY_SAVING_SPELL;
        af.bitvector    = 0;
        af.modifier     = - 10;
        affect_to_char(ch, &af);

        ch->mana -= mana;

        heal = dice(8, 10) + LVL(ch);
        ch->hit = UMIN(ch->hit + heal, ch->max_hit);
        update_pos(ch);
        char_act("You feel's better, now!", ch);
    }
    else {
        char_act("Your fortification attempt failed.", ch);
        check_improve(ch, gsn_fortitude, FALSE, 2);
        ch->mana -= mana / 2;
    }
}

DO_FUN(do_endure)
{
    do_concentrate(ch, "on endure");
}
