/* $Id: memento.c,v 1.666 2004/09/20 10:49:50 shrike Exp $                             */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>

#include "merc.h"

bool memento_set(CHAR_DATA *ch, const char *name, int vnum, int type)
{
	varr* list;
	vo_t vo;
	memento_t *memento;

	if (ch == NULL)
		return FALSE;

	switch (type)
	{
		case TYPE_ROOM:
			list = &ch->remembered_rooms;
			if ((vo.r = get_room_index(vnum)) == NULL)
				return FALSE;;
			break;
		case TYPE_OBJ:
			list = &ch->remembered_objects;
			if ((vo.o = get_obj_index(vnum)) == NULL)
				return FALSE;
			break;
		case TYPE_MOB:
			list = &ch->remembered_mobs;
			if ((vo.m = get_mob_index(vnum)) == NULL)
				return FALSE;
			break;
		case TYPE_PC:
			list = &ch->remembered_people;
			if ((vo.vnum = vnum) == 0)
				return FALSE;
			break;
		default:
			return FALSE;
			break;
	}
	memento = varr_enew(list);
    memento->name = str_dup(name);
    memento->id = vo;
    varr_qsort(list, cmpstr);
	return TRUE;
}

memento_t * lookup_remembered_room 	(CHAR_DATA *ch, ROOM_INDEX_DATA *room)
{
	int i;
	memento_t *memento;

	if (ch == NULL)
		return NULL;
	
	for (i = 0; i <  ch->remembered_rooms.nused; i++)
	{
		memento = VARR_GET(&ch->remembered_rooms, i);
		if (memento->id.r == room)
			return memento;
	}
	return NULL;
}

memento_t * lookup_remembered_obj 	(CHAR_DATA *ch, OBJ_INDEX_DATA *obj)
{	
	int i;
	memento_t *memento;

	if (ch == NULL)
		return NULL;
	
	for (i = 0; i <  ch->remembered_objects.nused; i++)
	{
		memento = VARR_GET(&ch->remembered_objects, i);
		if (memento->id.o == obj)
			return memento;
	}
	return NULL;
}

memento_t * lookup_remembered_mob 	(CHAR_DATA *ch, MOB_INDEX_DATA *mob)
{	
	int i;
	memento_t *memento;

	if (ch == NULL)
		return NULL;
	
	for (i = 0; i <  ch->remembered_mobs.nused; i++)
	{
		memento = VARR_GET(&ch->remembered_mobs, i);
		if (memento->id.m == mob)
			return memento;
	}
	return NULL;
}

memento_t * lookup_remembered_pc 	(CHAR_DATA *ch, int id)
{	
	int i;
	memento_t *memento;

	if (ch == NULL)
		return NULL;
	
	for (i = 0; i <  ch->remembered_people.nused; i++)
	{
		memento = VARR_GET(&ch->remembered_people, i);
		if (memento->id.vnum == id)
			return memento;
	}
	return NULL;
}

int memento_type(CHAR_DATA *ch, const char *name)
{
	memento_t *m;

	if (ch == NULL)
		return TYPE_NONE;
	
	if ((m = memento_lookup(ch->remembered_rooms, name)) != NULL)
		return TYPE_ROOM;
	if ((m = memento_lookup(ch->remembered_mobs, name)) != NULL)
		return TYPE_MOB;
	if ((m = memento_lookup(ch->remembered_objects, name)) != NULL)
		return TYPE_OBJ;
	if ((m = memento_lookup(ch->remembered_people, name)) != NULL)
		return TYPE_PC;
	return TYPE_NONE;
}

ROOM_INDEX_DATA	* get_remembered_room_index(CHAR_DATA *ch, const char *name)
{
	memento_t *m;

	if (ch == NULL)
		return NULL;
	
	if ((m = memento_lookup(ch->remembered_rooms, name)) == NULL)
		return NULL;
	return m->id.r;
}

MOB_INDEX_DATA	* get_remembered_mob_index(CHAR_DATA *ch, const char *name)
{
	memento_t *m;

	if (ch == NULL)
		return NULL;
	
	if ((m = memento_lookup(ch->remembered_mobs, name)) == NULL)
		return NULL;
	return m->id.m;
}

OBJ_INDEX_DATA	* get_remembered_obj_index(CHAR_DATA *ch, const char *name)
{
	memento_t *m;

	if (ch == NULL)
		return NULL;
	
	if ((m = memento_lookup(ch->remembered_objects, name)) == NULL)
		return NULL;
	return m->id.o;
}

int get_remembered_pc_id(CHAR_DATA *ch, const char *name)
{
	memento_t *m;
	
	if (ch == NULL)
		return 0;
	
	if ((m = memento_lookup(ch->remembered_people, name)) == NULL)
		return 0;
	return m->id.vnum;
}

void show_remembered_rooms(CHAR_DATA *ch)
{
	int i;
	memento_t *m;
	ROOM_INDEX_DATA *room;
	if (ch->remembered_rooms.nused == 0)
	{
		char_act("You haven't remembered any places.", ch);
		return;
	}
	char_act("You remembered the following places:", ch);
	for (i = 0; i < ch->remembered_rooms.nused; i++)
	{
		m = VARR_GET(&ch->remembered_rooms, i);
		room = m->id.r;
		if (room == NULL)
			act("$t...um, you forgot where is this.", ch, m->name, NULL, TO_CHAR);
		else
			act("$FL20{$T} $t", ch, mlstr_cval(room->name, ch), m->name, TO_CHAR);
	}
	char_act(str_empty, ch);
}

void show_remembered_objects(CHAR_DATA *ch)
{
	int i;
	memento_t *m;
	OBJ_INDEX_DATA *object;
	if (ch->remembered_objects.nused == 0)
	{
		char_act("You haven't remembered any objects.", ch);
		return;
	}
	char_act("You remembered the following objects:", ch);
	for (i = 0; i < ch->remembered_objects.nused; i++)
	{
		m = VARR_GET(&ch->remembered_objects, i);
		object = m->id.o;
		if (object == NULL)
			act("$t...um, you forgot what is this.", ch, m->name, NULL, TO_CHAR);
		else
			act("$FL20{$T} $t", ch, format_short(object->short_descr, object->name, ch, 0), m->name, TO_CHAR);
	}
	char_act(str_empty, ch);
}

void show_remembered_people(CHAR_DATA *ch)
{
	int i;
	memento_t *m;
	MOB_INDEX_DATA *mob;

	if (ch->remembered_mobs.nused == 0
		&& ch->remembered_people.nused == 0)
	{
		char_act("You haven't remembered any people or creatures.", ch);
		return;
	}
	char_act("You remembered the following people and creatures:", ch);
	for (i = 0; i < ch->remembered_mobs.nused; i++)
	{
		m = VARR_GET(&ch->remembered_mobs, i);
		mob = m->id.m;
		if (mob == NULL)
			act("$t...um, you forgot who is this.", ch, m->name, NULL, TO_CHAR);
		else
			act("$FL20{$T} $t", ch, format_short(mob->short_descr, mob->name, ch, 0), m->name, TO_CHAR);
	}
	for (i = 0; i < ch->remembered_people.nused; i++)
	{
		m = VARR_GET(&ch->remembered_people, i);
		act("$t", ch, m->name, NULL, TO_CHAR);
	}
	char_act(str_empty, ch);

}

void remember_room(CHAR_DATA *ch, ROOM_INDEX_DATA *room, const char *name)
{
	memento_t *memento;
    memento = varr_enew(&ch->remembered_rooms);
    memento->name = str_dup(name);
    memento->id.r = room;
    varr_qsort(&ch->remembered_rooms, cmpstr);
}

void remember_object(CHAR_DATA *ch, OBJ_INDEX_DATA *obj, const char *name)
{
	memento_t *memento;
    memento = varr_enew(&ch->remembered_objects);
    memento->name = str_dup(name);
    memento->id.o = obj;
    varr_qsort(&ch->remembered_objects, cmpstr);
	
}

void remember_mob(CHAR_DATA *ch, MOB_INDEX_DATA *mob, const char *name)
{
	memento_t *memento;
    memento = varr_enew(&ch->remembered_mobs);
    memento->name = str_dup(name);
    memento->id.m = mob;
    varr_qsort(&ch->remembered_mobs, cmpstr);
}

void remember_pc(CHAR_DATA *ch, int id, const char *name)
{
	memento_t *memento;
    memento = varr_enew(&ch->remembered_people);
    memento->name = str_dup(name);
    memento->id.vnum = id;
    varr_qsort(&ch->remembered_people, cmpstr);
}

DO_FUN(do_remember)
{
	char arg[MAX_STRING_LENGTH];
	char tmp[MAX_STRING_LENGTH];
	OBJ_DATA *obj;
	CHAR_DATA *mob;
	int type;

	argument = one_argument(argument, arg, sizeof(arg));
	one_argument(argument, tmp, sizeof(arg));
	if (!str_cmp(tmp, "as"))
		argument = one_argument(argument, tmp, sizeof(arg));

	if (IS_NULLSTR(arg))
	{
		show_remembered_rooms(ch);
		show_remembered_people(ch);
		show_remembered_objects(ch);
		return;
	}
	if (!str_cmp(arg, "places"))
	{
		show_remembered_rooms(ch);
		return;
	}
	if (!str_cmp(arg, "objects"))
	{
		show_remembered_objects(ch);
		return;
	}
	if (!str_cmp(arg, "people"))
	{
		show_remembered_people(ch);
		return;
	}
	if ((type = memento_type(ch, argument)) != TYPE_NONE)
	{
		act("You already remembered a $t by this name.", ch, (type == TYPE_ROOM) ? "room" : (type == TYPE_OBJ) ? "object" : "creature", NULL, TO_CHAR);
		return;
	}

	if (!str_cmp(arg, "here"))
	{
		remember_room(ch, ch->in_room, argument);
		act("You remembered $r as $T.", ch, ch->in_room, argument, TO_CHAR);
		return;
	}
	if ((mob = get_char_room(ch, arg)) != NULL)
	{
		if (IS_NPC(mob))
			remember_mob(ch, mob->pIndexData, argument);
		else
			remember_pc(ch, mob->id, argument);

		act("You remembered $i as $T.", ch, mob, argument, TO_CHAR);
		return;
	}
	if ((obj = get_obj_room(ch, arg)) != NULL)
	{
		remember_object(ch, obj->pIndexData, argument);
		act("You remembered $p as $T.", ch, obj, argument, TO_CHAR);
		return;
	}
	char_act("You don't see that here.", ch);
	return;
}

DO_FUN(do_forget)
{
	memento_t *m;
	const char *name = argument;
	
	if ((m = memento_lookup(ch->remembered_rooms, name)) != NULL)
	{
		char_act("You forgot about a place important to you before.", ch);
		varr_del(&ch->remembered_rooms, (void *) m);
		return;
	}
	if ((m = memento_lookup(ch->remembered_mobs, name)) != NULL)
	{

		char_act("You forgot about someone important to you before.", ch);
		varr_del(&ch->remembered_mobs, (void *) m);
		return;
	}
	if ((m = memento_lookup(ch->remembered_objects, name)) != NULL)
	{
		char_act("You forgot about someone important to you before.", ch);
		varr_del(&ch->remembered_objects, (void *) m);
		return;
	}
	if ((m = memento_lookup(ch->remembered_people, name)) != NULL)
	{
		char_act("You forgot about something important to you before.", ch);
		varr_del(&ch->remembered_people, (void *) m);
		return;
	}
	char_act("Seems, you already forgot this.", ch);
	return;

}
