/* $Id: mgrep.c,v 1.666 2004/09/20 10:49:51 shrike Exp $                              */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <sys/types.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>

#include "merc.h"
#include "db/db.h"
#include "grep.h"

static CMD_DATA mgrep_cmd_table[] =
{
    { "help",       mgrep_help,       POS_DEAD,      92,              0,  0,     0,              0},
    { "level",      mgrep_level,      POS_DEAD,      92,              0,  0,     0,              0},
    { "area",       mgrep_area,       POS_DEAD,      92,              0,  0,     0,              0},
    { "name",       mgrep_name,       POS_DEAD,      92,              0,  0,     0,              0},
    { "act",        mgrep_act,        POS_DEAD,      92,              0,  0,     0,              0},
    { "off",        mgrep_off,        POS_DEAD,      92,              0,  0,     0,              0},
    { "affected",   mgrep_affected,   POS_DEAD,      92,              0,  0,     0,              0},
    { "immunes",    mgrep_immunes,    POS_DEAD,      92,              0,  0,     0,              0},
    { "shop",       mgrep_shop,       POS_DEAD,      92,              0,  0,     0,              0},
    { "race",       mgrep_race,       POS_DEAD,      92,              0,  0,     0,              0},
    { NULL }
} ;


DO_FUN (do_mgrep)
{
    if (!parse_command(ch, argument, mgrep_cmd_table))
    {
        show_command_list (ch, mgrep_cmd_table);
        char_puts (".\nUse {CHELP MGREP{x for more information.\n", ch);
    }
}

const char * print_mob_dice(mgrep_mob_t *mob)
{
    const char   *output;

    output = str_printf("{D[{W%3dd%-3d+%5d{D] [{W%3dd%-3d+%5d{D] [{W%3dd%-3d+%5d]",
               mob->hit[DICE_NUMBER],
               mob->hit[DICE_TYPE],
               mob->hit[DICE_BONUS],
               mob->damage[DICE_NUMBER],
               mob->damage[DICE_TYPE],
               mob->damage[DICE_BONUS],
               mob->mana[DICE_NUMBER],
               mob->mana[DICE_TYPE],
               mob->mana[DICE_BONUS]
               );

    return str_dup(output);
}

const char * print_mob_ac(mgrep_mob_t *mob)
{
    const char   *output;

    output = str_printf("{D[p:{W%6d {Db:{W%6d {Ds:{W%6d {Dm:{W%6d{D]",

               mob->ac[AC_PIERCE],
               mob->ac[AC_BASH],
               mob->ac[AC_SLASH],
               mob->ac[AC_EXOTIC]
               );

    return str_dup(output);
}

const char * print_mob(mgrep_mob_t *mob)
{
    const char   *output;
    // vnum name level hit_dice dam_dice mana_dice
    output = str_printf("{D[{x%6d{D]{x %-20s{x {w%4d{x",

               mob->vnum,
               mob->name,
               mob->level
               );
    return str_dup(output);
}

void mgrep_help(CHAR_DATA *ch, const char *argument)
{
      do_help(ch, "mgrep");
}

//
// Compare function
//     -    -  
//

/*
static int cmp_weapon_ac(const void *p1, const void *p2)
{
     return ((ogrep_weapon_t*) p1)->ac - ((ogrep_weapon_t*) p2)->ac;
}

*/

//
//   - 
//

static int cmp_mob_level(const void *p1, const void *p2)
{
     return ((mgrep_mob_t*) p2)->level - ((mgrep_mob_t*) p1)->level;
}

void fill_mob_data(mgrep_mob_t *mgrep_mob, MOB_INDEX_DATA *pMobIndex)
{
      MPTRIG          *mptrig;

      mgrep_mob->name        = str_dup(fmt_color_str(mlstr_mval(pMobIndex->short_descr),20));
      mgrep_mob->vnum        = pMobIndex->vnum;
      mgrep_mob->act         = pMobIndex->act;
      mgrep_mob->affected_by = pMobIndex->affected_by;
      mgrep_mob->alignment   = pMobIndex->alignment;
      mgrep_mob->clan        = pMobIndex->clan;
      mgrep_mob->dam_type    = pMobIndex->dam_type;

      mgrep_mob->hitroll     = pMobIndex->hitroll;
      mgrep_mob->immunes     = pMobIndex->immunes;
      mgrep_mob->invis_level = pMobIndex->invis_level;
      mgrep_mob->level       = pMobIndex->level;

      mgrep_mob->off_flags   = pMobIndex->off_flags;
      mgrep_mob->pcnt        = pMobIndex->pcnt;
      mgrep_mob->practicer   = pMobIndex->practicer;
      mgrep_mob->race        = pMobIndex->race;
//    mgrep_mob->resists     = pMobIndex->resists;
      mgrep_mob->sex         = pMobIndex->sex;
      mgrep_mob->size        = pMobIndex->size;
      mgrep_mob->wealth      = pMobIndex->wealth;

      mgrep_mob->form        = pMobIndex->form;
      mgrep_mob->parts       = pMobIndex->parts;

      mgrep_mob->hit[DICE_NUMBER]     = pMobIndex->hit[DICE_NUMBER];
      mgrep_mob->hit[DICE_TYPE]       = pMobIndex->hit[DICE_TYPE];
      mgrep_mob->hit[DICE_BONUS]      = pMobIndex->hit[DICE_BONUS];

      mgrep_mob->damage[DICE_NUMBER]  = pMobIndex->damage[DICE_NUMBER];
      mgrep_mob->damage[DICE_TYPE]    = pMobIndex->damage[DICE_TYPE];
      mgrep_mob->damage[DICE_BONUS]   = pMobIndex->damage[DICE_BONUS];

      mgrep_mob->mana[DICE_NUMBER]    = pMobIndex->mana[DICE_NUMBER];
      mgrep_mob->mana[DICE_TYPE]      = pMobIndex->mana[DICE_TYPE];
      mgrep_mob->mana[DICE_BONUS]     = pMobIndex->mana[DICE_BONUS];

      mgrep_mob->ac[AC_PIERCE]        = pMobIndex->ac[AC_PIERCE];
      mgrep_mob->ac[AC_BASH]          = pMobIndex->ac[AC_BASH];
      mgrep_mob->ac[AC_SLASH]         = pMobIndex->ac[AC_SLASH];
      mgrep_mob->ac[AC_EXOTIC]        = pMobIndex->ac[AC_EXOTIC];

      if (pMobIndex->mptrig_list)
      {
            for (mptrig = pMobIndex->mptrig_list; mptrig; mptrig = mptrig->next)
            {
                  mgrep_mob->mptrig++;
            }

      }
}


void mgrep_level (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;

     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     char             arg2[MAX_INPUT_LENGTH];
     char             arg3[MAX_INPUT_LENGTH];

     int              total;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow;
     bool             bDiceShow;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     bAcShow   = FALSE;
     bDiceShow = FALSE;

     sort_mode = SORT_NONE;

     argument = one_argument(argument, arg1, sizeof(arg1));
     argument = one_argument(argument, arg2, sizeof(arg2));

     if (!is_number(arg1) || !is_number(arg2))
     {
          char_act("Mgrep: invalid argument. Valid syntax: level min_level max_level", ch);
          return;
     }

     min_level = atoi(arg1);
     max_level = atoi(arg2);

     if (min_level < 0 || max_level < min_level)
     {
          char_act("Mgrep: invalid argument. Valid syntax: level min_level max_level"
                   "wher min_level must be 0 or greater and max_level must be greater then min_level", ch);
          return;
     }

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg3, sizeof(arg3));

           if (!str_cmp(arg3, "sort"  ))
           {
                 argument = one_argument(argument, arg3, sizeof(arg3));

                 sort_mode = get_sort_mode(arg3, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg3, "affect"))             bAShow    = TRUE;
           else if (!str_cmp(arg3, "act"))                bActShow  = TRUE;
           else if (!str_cmp(arg3, "off"))                bOffShow  = TRUE;
           else if (!str_cmp(arg3, "immunes"))            bImmShow  = TRUE;
           else if (!str_cmp(arg3, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg3, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg3);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1)
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));
                 

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_area (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     AREA_DATA       *area;
     int              min_area_vnum;
     int              max_area_vnum;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];

     int              total;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;
     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (!is_number(arg1) || atoi(arg1) < 0 )
     {
           char_act("Mgrep: argument must be a number ant greater then 0.", ch);
           return;
     }

     area = area_lookup(atoi(arg1));

     if (area == NULL)
     {
           char_act("Mgrep: not such area.", ch);
           return;
     }

     min_area_vnum = area->min_vnum;
     max_area_vnum = area->max_vnum;

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && pMobIndex->vnum <= max_area_vnum
                    && pMobIndex->vnum >= min_area_vnum
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_name (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     const char      *name;

     int              total;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (arg1[0] == '\0')
     {
           char_act("Mgrep: argument cannot be empty.", ch);
           return;
     }

     name = str_dup(arg1);

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {

                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && (strstr(pMobIndex->name, name))
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_act (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *flag;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (flag_lookup(act_flags, arg1) == NULL)
     {
            char_act("Mgrep: invalid act flag.", ch);
            show_flags(ch, act_flags);
            return;
     }

     flag = str_dup(arg1);

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && IS_SET(pMobIndex->act, flag_value(act_flags, flag))
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_off (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *flag;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (flag_lookup(off_flags, arg1) == NULL)
     {
            char_act("Mgrep: invalid off flag.", ch);
            show_flags(ch, off_flags);
            return;
     }

     flag = str_dup(arg1);

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && IS_SET(pMobIndex->off_flags, flag_value(off_flags, flag))
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_affected (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *flag;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (flag_lookup(affect_flags, arg1) == NULL)
     {
            char_act("Mgrep: invalid affect flag.", ch);
            show_flags(ch, affect_flags);
            return;
     }

     flag = str_dup(arg1);

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && IS_SET(pMobIndex->affected_by, flag_value(affect_flags, flag))
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_immunes (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *flag;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     argument = one_argument(argument, arg1, sizeof(arg1));

     if (flag_lookup(imm_flags, arg1) == NULL)
     {
            char_act("Mgrep: invalid immunes flag.", ch);
            show_flags(ch, imm_flags);
            return;
     }

     flag = str_dup(arg1);

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && IS_SET(pMobIndex->immunes, flag_value(imm_flags, flag))
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_shop (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;

     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && pMobIndex->pShop != NULL
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void mgrep_race (CHAR_DATA *ch, const char * argument )
{
     MOB_INDEX_DATA  *pMobIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              race;
     int              total;


     mgrep_mob_t     *mgrep_mob;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bAShow;
     bool             bActShow;
     bool             bOffShow;
     bool             bImmShow;
     bool             bAcShow   = FALSE;
     bool             bDiceShow = FALSE;

     int              min_level;
     int              max_level;

     int              i;
     int              max_alevel = 0;

     varr mgrep_mobs  = {sizeof (mgrep_mob_t), 4};

     total = 0;

     min_level = 0;
     max_level = 1000;

     bSort     = FALSE;
     bAShow    = FALSE;
     bActShow  = FALSE;
     bOffShow  = FALSE;
     bImmShow  = FALSE;
     sort_mode = SORT_NONE;


     if ( argument[0] == '\0' )
     {
           char_act("Mgrep: You must specify race name.", ch);
           return;
     }

     argument = one_argument(argument, arg1, sizeof(arg1));


     if ((race = rn_lookup(arg1)) <  1 )
     {
           char_act("Mgrep: Invalid race.", ch);
           return;
     }

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_MOB);

                 if (sort_mode < 0)
                 {
                       char_act("Mgrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_MOB);
                       return;
                 }

                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Mgrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);

                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Mgrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);
           }
           else if (!str_cmp(arg1, "affect"))             bAShow = TRUE;
           else if (!str_cmp(arg1, "act"))                bActShow = TRUE;
           else if (!str_cmp(arg1, "off"))                bOffShow = TRUE;
           else if (!str_cmp(arg1, "immunes"))            bImmShow = TRUE;
           else if (!str_cmp(arg1, "dice"))               bDiceShow = TRUE;
           else if (!str_cmp(arg1, "ac"))                 bAcShow   = TRUE;
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_mob_index; vnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {
               total++;

               if (    pMobIndex->level > min_level - 1
                    && pMobIndex->level < max_level + 1
                    && pMobIndex->race == race
                   )
               {
                     mgrep_mob              = varr_enew(&mgrep_mobs);
                     fill_mob_data(mgrep_mob, pMobIndex);
               }
          }

     // Checking for max parameter
     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           if (max_alevel < mob->level)                 max_alevel = mob->level;
     }


     if (bSort)
     switch (sort_mode)
     {

          case SORT_LEVEL:       varr_qsort(&mgrep_mobs, cmp_mob_level);                   break;
          default:                                                                         break;
     }

     for (i = 0; i < mgrep_mobs.nused; i++)
     {
           mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);

           buf_printf(output,"%s",print_mob(mob));
           buf_printf(output," %15s",race_name(mob->race));

           if (bDiceShow)
                 buf_printf(output, " %s", print_mob_dice(mob));

           if (bAcShow)
                 buf_printf(output, " %s", print_mob_ac(mob));

           if (bAShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(affect_flags, mob->affected_by));

           if (bActShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(act_flags, mob->act));

           if (bOffShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(off_flags, mob->off_flags));

           if (bImmShow)
                 buf_printf(output, " {D[%s{D]{x", flag_dump(imm_flags, mob->immunes));

           buf_printf(output,"\n");
     }

     for (i = mgrep_mobs.nused - 1; 0; i-- )
     {
        mgrep_mob_t * mob = varr_get(&mgrep_mobs, i);
        free_string(mob->name);
        varr_del(&mgrep_mobs, (void*)mob);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}
