/*  $Id: obj_cmds.c,v 1.666 2004/09/20 10:49:51 shrike Exp $  */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/



#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "merc.h"
#include "typedef.h"
#include "obj_cmds.h"
#include "mob_cmds.h"
#include "mob_prog.h"
#include "fight.h"
#include "race.h"
#include "class.h"


DECLARE_DO_FUN(do_look      );
DECLARE_DO_FUN(do_say       );

/*
 * OBJcommand table.
 */
const   struct  obj_cmd_type obj_cmd_table  [] =
{
	{   "gecho",        do_opgecho},
	{   "zecho",        do_opzecho},
	{   "echo",         do_opecho},
	{   "echoaround",   do_opechoaround},
	{   "echoat",       do_opechoat},
	{   "mload",        do_opmload},
	{   "oload",        do_opoload},
	{   "purge",        do_oppurge},
	{   "goto",         do_opgoto},
	{   "transfer",     do_optransfer},
	{   "gtransfer",    do_opgtransfer},
	{   "otransfer",    do_opotransfer},
	{   "force",        do_opforce},
	{   "gforce",       do_opgforce},
	{   "vforce",       do_opvforce},
	{   "damage",       do_opdamage},
	{   "remember",     do_opremember},
	{   "forget",       do_opforget},
	{   "delay",        do_opdelay},
	{   "cancel",       do_opcancel},
	{   "call",         do_opcall},
	{   "remove",       do_opremove},
	{   str_empty,      0}
};


/*
 * OBJprog section
 */
/*
 * Obj command interpreter. Implemented separately for security and speed
 * reasons. A trivial hack of interpret()
 */

void obj_interpret( OBJ_DATA *obj, const char *argument )
{
	char buf[MAX_STRING_LENGTH], command[MAX_INPUT_LENGTH];
	int cmd;

	argument = one_argument( argument, command, sizeof(command));

	/*
	 * Look for command in command table.
	 */
	for (cmd = 0; obj_cmd_table[cmd].name[0] != '\0'; cmd++)
	{
		if (command[0] == obj_cmd_table[cmd].name[0]
			&&   !str_prefix( command, obj_cmd_table[cmd].name ))
		{
			(*obj_cmd_table[cmd].obj_fun) ( obj, argument );
			tail_chain( );
			return;
		}
	}
	sprintf( buf, "Obj_interpret: invalid cmd from obj %d: '%s'",
			 obj->pIndexData->vnum, command );
	bug( buf, 0 );
}

void do_obj( OBJ_DATA *obj, const char *argument )
{
	/*
	 * Security check!
	 */
	if (obj->level < MAX_LEVEL)
		return;
	obj_interpret( obj, argument );
}

/*
 * Prints the argument to all active players in the game
 *
 * Syntax: obj gecho [string]
 */
void do_opgecho( OBJ_DATA *obj, const char *argument )
{
	DESCRIPTOR_DATA *d;

	if (argument[0] == '\0')
	{
		bug( "OpGEcho: missing argument from vnum %d",
			 obj->pIndexData->vnum );
		return;
	}

	for (d = descriptor_list; d; d = d->next)
	{
		if (d->connected == CON_PLAYING)
		{
			if (IS_IMMORTAL(d->character))
				send_to_char( "Obj echo> ", d->character );
			send_to_char( argument, d->character );
			send_to_char( "\n\r", d->character );
		}
	}
}

/*
 * Prints the argument to all players in the same area as the mob
 *
 * Syntax: obj zecho [string]
 */
void do_opzecho( OBJ_DATA *obj, const char *argument )
{
	DESCRIPTOR_DATA *d;

	if (argument[0] == '\0')
	{
		bug( "OpZEcho: missing argument from vnum %d",
			 obj->pIndexData->vnum );
		return;
	}

	if (obj->in_room == NULL && (obj->carried_by == NULL || obj->carried_by->in_room == NULL))
		return;

	for (d = descriptor_list; d; d = d->next)
	{
		if (d->connected == CON_PLAYING 
			&&   d->character->in_room != NULL 
			&&   ( (obj->in_room && d->character->in_room->area == obj->in_room->area)
				   || (obj->carried_by && d->character->in_room->area == obj->carried_by->in_room->area) ))
		{
			if (IS_IMMORTAL(d->character))
				send_to_char( "Obj echo> ", d->character );
			send_to_char( argument, d->character );
			send_to_char( "\n\r", d->character );
		}
	}
}

/*
 * Prints the message to everyone in the room other than the mob and victim
 *
 * Syntax: obj echoaround [victim] [string]
 */

void do_opechoaround( OBJ_DATA *obj, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim, *vch;

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0')
		return;

	if (( victim=get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
		return;

	if (obj->in_room && obj->in_room->people)
		vch = obj->in_room->people;
	else if (obj->carried_by && obj->carried_by->in_room)
		vch = obj->carried_by->in_room->people;
	else
		vch	= NULL;

	for (; vch; vch = vch->next_in_room)
	{
		if (vch == victim)
			continue;
		send_to_char( argument, vch );
	}
}

/*
 * Prints the message to only the victim
 *
 * Syntax: obj echoat [victim] [string]
 */
void do_opechoat( OBJ_DATA *obj, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
		return;

	if (( victim = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
		return;

	act( argument, obj->carried_by?obj->carried_by:obj->in_room->people, obj, victim, TO_VICT );
}

/*
 * Prints the message to the room at large
 *
 * Syntax: obj echo [string]
 */
void do_opecho( OBJ_DATA *obj, const char *argument )
{
	if (argument[0] == '\0')
		return;

	if (!obj->carried_by && !obj->in_room->people)
		return;

	act( argument, obj->carried_by?obj->carried_by:obj->in_room->people, NULL, NULL, TO_ROOM );
	act( argument, obj->carried_by?obj->carried_by:obj->in_room->people, NULL, NULL, TO_CHAR );
}

/*
 * Lets the object load a mobile.
 *
 * Syntax: obj mload [vnum]
 */
void do_opmload( OBJ_DATA *obj, const char *argument )
{
	char            arg[ MAX_INPUT_LENGTH ];
	MOB_INDEX_DATA *pMobIndex;
	CHAR_DATA      *victim;
	int vnum;

	one_argument(argument, arg, sizeof(arg));

	if ((obj->in_room == NULL
		 && (obj->carried_by == NULL || obj->carried_by->in_room == NULL))
		|| arg[0] == '\0' || !is_number(arg))
		return;

	vnum = atoi(arg);
	if (( pMobIndex = get_mob_index( vnum ) ) == NULL)
	{
		sprintf( arg, "Opmload: bad mob index (%d) from obj %d",
				 vnum, obj->pIndexData->vnum );
		bug( arg, 0 );
		return;
	}
	victim = create_mob( pMobIndex );
	char_to_room( victim, obj->in_room?obj->in_room:obj->carried_by->in_room );
	return;
}

/*
 * Lets the object load another object
 *
 * Syntax: obj oload [vnum] [level]
 */
void do_opoload( OBJ_DATA *obj, const char *argument )
{
	char arg1[ MAX_INPUT_LENGTH ];
	char arg2[ MAX_INPUT_LENGTH ];
	OBJ_INDEX_DATA *pObjIndex;
	OBJ_DATA       *nobj;
	int             level;

	argument = one_argument( argument, arg1, sizeof(arg1));
	argument = one_argument( argument, arg2, sizeof(arg2));

	if (arg1[0] == '\0' || !is_number( arg1 ))
	{
		bug( "Opoload - Bad syntax from vnum %d.",
			 obj->pIndexData->vnum );
		return;
	}

	if (arg2[0] == '\0')
	{
		level = obj->level;
	}
	else
	{
		/*
		 * New feature from Alander.
		 */
		if (!is_number( arg2 ))
		{
			bug( "Opoload - Bad syntax from vnum %d.", 
				 obj->pIndexData->vnum );
			return;
		}
		level = atoi( arg2 );
		if (level < 0 || level > obj->level)
		{
			bug( "Opoload - Bad level from vnum %d.", 
				 obj->pIndexData->vnum );
			return;
		}
	}

	if (( pObjIndex = get_obj_index( atoi( arg1 ) ) ) == NULL)
	{
		bug( "Opoload - Bad vnum arg from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	nobj = create_obj( pObjIndex, level );
	obj_to_room( nobj, obj->in_room?obj->in_room:obj->carried_by->in_room );

	return;
}

/*
 * Lets the object purge all other objects and npcs in the room,
 * or purge a specified object or mob in the room. The object cannot
 * purge itself for safety reasons.
 *
 * syntax obj purge {target}
 */
void do_oppurge( OBJ_DATA *obj, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;
	OBJ_DATA  *vobj;

	one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0')
	{
		/* 'purge' */
		CHAR_DATA *vnext;
		OBJ_DATA  *obj_next;

		if (obj->in_room && obj->in_room->people)
			victim = obj->in_room->people;
		else if (obj->carried_by && obj->carried_by->in_room)
			victim = obj->carried_by->in_room->people;
		else
			victim = NULL;

		for (; victim != NULL; victim = vnext)
		{
			vnext = victim->next_in_room;
			if (IS_NPC( victim )
				&&   !IS_SET(victim->pIndexData->act, ACT_NOPURGE))
				extract_char( victim, TRUE );
		}

		if (obj->in_room)
			vobj = obj->in_room->contents;
		else
			vobj = obj->carried_by->in_room->contents;

		for (; vobj != NULL; vobj = obj_next)
		{
			obj_next = vobj->next_content;
			if (!IS_SET(vobj->extra_flags, ITEM_NOPURGE) && vobj != obj)
				extract_obj( vobj );
		}

		return;
	}

	if (( victim = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
	{
		if ((vobj = get_obj_here_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ)))
		{
			extract_obj( vobj );
		}
		else if (obj->carried_by && (vobj = get_obj_carry( obj->carried_by, arg)) != NULL)
		{
			extract_obj( vobj );
		}
		else if (obj->carried_by && (vobj = get_obj_wear_ext(obj->carried_by, arg, GET_F_NO_CHECK_CAN_SEE)) != NULL)
		{
			unequip_char( vobj->carried_by, vobj );
			extract_obj( vobj );
		}
		else
		{
			bug( "Oppurge - Bad argument from vnum %d.",
				 obj->pIndexData->vnum );
		}
		return;
	}

	if (!IS_NPC( victim ))
	{
		bug( "Oppurge - Purging a PC from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}
	extract_char( victim, TRUE );
	return;
}


/*
 * Lets the object goto any location it wishes that is not private.
 *
 * Syntax: obj goto [location]
 */
void do_opgoto( OBJ_DATA *obj, const char *argument )
{
	char             arg[ MAX_INPUT_LENGTH ];
	ROOM_INDEX_DATA *location;
	CHAR_DATA *victim;
	OBJ_DATA *dobj;

	dobj = NULL;

        one_argument( argument, arg, sizeof(arg));
	if (arg[0] == '\0')
	{
		bug( "Opgoto - No argument from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if (is_number(arg))
		location = get_room_index( atoi( arg ) );
	else if ((victim = get_char_world_ext((CHAR_DATA *)dobj, arg, GET_F_CHAR_IS_OBJ )) != NULL)
		location = victim->in_room;
	else if (( dobj = get_obj_world_ext((CHAR_DATA *) dobj, arg, GET_F_CHAR_IS_OBJ )) != NULL)
		location = dobj->in_room;
	else
	{
		bug( "Opgoto - No such location from vnum %d.",  obj->pIndexData->vnum );
		return;
	}

	if (obj->in_room != NULL)
		obj_from_room( obj );
	else if (obj->carried_by != NULL)
		obj_from_char( obj );
	obj_to_room( obj, location );

	return;
}

/*
 * Lets the object transfer people.  The 'all' argument transfers
 *  everyone in the current room to the specified location
 *
 * Syntax: obj transfer [target|'all'] [location]
 */
void do_optransfer( OBJ_DATA *obj, const char *argument )
{
	char             arg1[ MAX_INPUT_LENGTH ];
	char             arg2[ MAX_INPUT_LENGTH ];
	char         buf[MAX_STRING_LENGTH];
	ROOM_INDEX_DATA *location;
	CHAR_DATA       *victim;
	OBJ_DATA        *dobj;

	dobj = NULL;

        argument = one_argument( argument, arg1, sizeof(arg1));
	argument = one_argument( argument, arg2, sizeof(arg2));

	if (arg1[0] == '\0')
	{
		bug( "Optransfer - Bad syntax from vnum %d.", obj->pIndexData->vnum );
		return;
	}

	if (!str_cmp( arg1, "all" ))
	{
		CHAR_DATA *victim_next;

		if (obj->in_room && obj->in_room->people)
			victim = obj->in_room->people;
		else if (obj->carried_by)
			victim = obj->carried_by->in_room->people;
		else
			victim = NULL;
		for (; victim != NULL; victim = victim_next)
		{
			victim_next = victim->next_in_room;
			if (!IS_NPC(victim))
			{
				sprintf( buf, "%s %s", victim->name, arg2 );
				do_optransfer( obj, buf );
			}
		}
		return;
	}

	/*
	 * Thanks to Grodyn for the optional location parameter.
	 */

	if (arg2[0] == '\0')
	{
		location = obj->in_room?obj->in_room:obj->carried_by->in_room;
	}
	else
	{
		if (is_number(arg2))
			location = get_room_index( atoi(arg2) );
		else if ((victim = get_char_world_ext((CHAR_DATA *) dobj, arg2, GET_F_CHAR_IS_OBJ )) != NULL)
			location = victim->in_room;
		else if (( dobj = get_obj_world_ext((CHAR_DATA *) dobj, arg2, GET_F_CHAR_IS_OBJ)) != NULL)
			location = dobj->in_room;
		else
		{
			bug( "Optransfer - No such location from vnum %d.",
				 obj->pIndexData->vnum );
			return;
		}

		if (room_is_private( location ))
			return;
	}

	if (( victim = get_char_world_ext((CHAR_DATA *) dobj, arg1,GET_F_CHAR_IS_OBJ )) == NULL)
		return;

	if (victim->in_room == NULL)
		return;

	if (victim->fighting != NULL)
		stop_fighting( victim, TRUE );
	char_from_room( victim );
	char_to_room( victim, location );
	if (!JUST_KILLED(victim))
		do_look( victim, "auto" );

	return;
}

/*
 * Lets the object transfer all chars in same group as the victim.
 *
 * Syntax: obj gtransfer [victim] [location]
 */
void do_opgtransfer( OBJ_DATA *obj, const char *argument )
{
	char             arg1[ MAX_INPUT_LENGTH ];
	char             arg2[ MAX_INPUT_LENGTH ];
	char         buf[MAX_STRING_LENGTH];
	CHAR_DATA       *who, *victim, *victim_next;

	argument = one_argument( argument, arg1, sizeof(arg1));
	argument = one_argument( argument, arg2, sizeof(arg2));

	if (arg1[0] == '\0')
	{
		bug( "Opgtransfer - Bad syntax from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if ((who = get_char_room_ext((CHAR_DATA *) obj, arg1, GET_F_CHAR_IS_OBJ)) == NULL)
		return;

	if (obj->in_room && obj->in_room->people)
		victim = obj->in_room->people;
	else if (obj->carried_by && obj->carried_by->in_room)
		victim = obj->carried_by->in_room->people;
	else
		victim = NULL;

	for (; victim; victim = victim_next)
	{
		victim_next = victim->next_in_room;
		if (is_same_group( who,victim ))
		{
			sprintf( buf, "%s %s", victim->name, arg2 );
			do_optransfer( obj, buf );
		}
	}
	return;
}

/*
 * Lets the object force someone to do something. Must be mortal level
 * and the all argument only affects those in the room with the object.
 *
 * Syntax: obj force [victim] [commands]
 */
void do_opforce( OBJ_DATA *obj, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "Opforce - Bad syntax from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if (!obj->in_room && !obj->carried_by)
		return;
	if (obj->in_room && !obj->in_room->people)
		return;
	if (obj->carried_by && !obj->carried_by->in_room)
		return;

	if (!str_cmp( arg, "all" ))
	{
		CHAR_DATA *vch;
		CHAR_DATA *vch_next;

		for (vch = char_list; vch != NULL; vch = vch_next)
		{
			vch_next = vch->next;

			if (((obj->in_room && vch->in_room == obj->in_room)
				 || (obj->carried_by && vch->in_room == obj->carried_by->in_room)) &&  IS_TRUSTED(vch, obj->level))
			{
				interpret( vch, argument );
			}
		}
	}
	else
	{
		CHAR_DATA *victim;

		if (( victim = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
			return;

		interpret( victim, argument );
	}

	return;
}

/*
 * Lets the object force a group something. Must be mortal level.
 *
 * Syntax: obj gforce [victim] [commands]
 */
void do_opgforce( OBJ_DATA *obj, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim, *vch, *vch_next;

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "OpGforce - Bad syntax from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if (( victim = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
		return;

	for (vch = victim->in_room->people; vch != NULL; vch = vch_next)
	{
		vch_next = vch->next_in_room;

		if (is_same_group(victim,vch))
		{
			interpret( vch, argument );
		}
	}
	return;
}

/*
 * Forces all mobiles of certain vnum to do something
 *
 * Syntax: obj vforce [vnum] [commands]
 */
void do_opvforce( OBJ_DATA *obj, const char *argument )
{
	CHAR_DATA *victim, *victim_next;
	char arg[ MAX_INPUT_LENGTH ];
	int vnum;

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "OpVforce - Bad syntax from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if (!is_number( arg ))
	{
		bug( "OpVforce - Non-number argument vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	vnum = atoi( arg );

	for (victim = char_list; victim; victim = victim_next)
	{
		victim_next = victim->next;
		if (IS_NPC(victim) && victim->pIndexData->vnum == vnum
			&& victim->fighting == NULL)
			interpret( victim, argument );
	}
	return;
}

/*
 * Lets obj cause unconditional damage to someone. Nasty, use with caution.
 * Also, this is silent, you must show your own damage message...
 *
 * Syntax: obj damage [victim] [min] [max] {kill}
 */
void do_opdamage( OBJ_DATA *obj, const char *argument )
{
	CHAR_DATA *victim = NULL, *victim_next;
	char target[ MAX_INPUT_LENGTH ],
	min[ MAX_INPUT_LENGTH ],
	max[ MAX_INPUT_LENGTH ];
	int low, high;
	bool fAll = FALSE, fKill = FALSE;

	argument = one_argument( argument, target, sizeof(target));
	argument = one_argument( argument, min, sizeof(min));
	argument = one_argument( argument, max, sizeof(max));

	if (target[0] == '\0')
	{
		bug( "OpDamage - Bad syntax from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}
	if (!str_cmp( target, "all" ))
		fAll = TRUE;
	else if ((victim = get_char_room_ext((CHAR_DATA *) obj, target, GET_F_CHAR_IS_OBJ)) == NULL)
		return;

	if (is_number( min ))
		low = atoi( min );
	else
	{
		bug( "OpDamage - Bad damage min vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}
	if (is_number( max ))
		high = atoi( max );
	else
	{
		bug( "OpDamage - Bad damage max vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}
	one_argument( argument, target, sizeof(target));

	/*
	 * If kill parameter is omitted, this command is "safe" and will not
	 * kill the victim.
	 */

	if (target[0] != '\0')
		fKill = TRUE;
	if (fAll)
	{
		if (obj->in_room && obj->in_room->people)
			victim = obj->in_room->people;
		else if (obj->carried_by)
			victim = obj->carried_by->in_room->people;
		else
			victim = NULL;

		for (; victim; victim = victim_next)
		{
			victim_next = victim->next_in_room;
			if (obj->carried_by && victim != obj->carried_by)
				damage( victim, victim, 
						fKill ? 
						number_range(low,high) : UMIN(victim->hit,number_range(low,high)),
						TYPE_UNDEFINED, DAM_NONE, FALSE );
		}
	}
	else
		damage( victim, victim, 
				fKill ? 
				number_range(low,high) : UMIN(victim->hit,number_range(low,high)),
				TYPE_UNDEFINED, DAM_NONE, FALSE );
	return;
}

/*
 * Lets the object to remember a target. The target can be referred to
 * with $q and $Q codes in OBJprograms. See also "obj forget".
 *
 * Syntax: obj remember [victim]
 */
void do_opremember( OBJ_DATA *obj, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];
	one_argument( argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		obj->oprog_target = get_char_world_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ );
	else
		bug( "OpRemember: missing argument from vnum %d.", 
			 obj->pIndexData->vnum );
}

/*
 * Reverse of "obj remember".
 *
 * Syntax: obj forget
 */
void do_opforget( OBJ_DATA *obj, const char *argument )
{
	obj->oprog_target = NULL;
}

/*
 * Sets a delay for OBJprogram execution. When the delay time expires,
 * the object is checked for a OBJprogram with DELAY trigger, and if
 * one is found, it is executed. Delay is counted in PULSE_TICK
 *
 * Syntax: obj delay [pulses]
 */
void do_opdelay( OBJ_DATA *obj, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];

	one_argument( argument, arg, sizeof(arg));
	if (!is_number( arg ))
	{
		bug( "OpDelay: invalid arg from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}
	obj->oprog_delay = atoi( arg );
}

/*
 * Reverse of "obj delay", deactivates the timer.
 *
 * Syntax: obj cancel
 */
void do_opcancel( OBJ_DATA *obj, const char *argument )
{
	obj->oprog_delay = -1;
}


/*
 * Lets the object to call another OBJprogram withing a OBJprogram.
 * This is a crude way to implement subroutines/functions. Beware of
 * nested loops and unwanted triggerings... Stack usage might be a problem.
 * Characters and objects referred to must be in the same room with the
 * mobile.
 *
 * Syntax: obj call [vnum] [victim|'null'] [object1|'null'] [object2|'null']
 *
 */

void do_opcall( OBJ_DATA *obj, const char *argument )
{   
	char arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *vch;
	OBJ_DATA *obj1, *obj2;
	MPCODE *prg;
	extern void    program_flow(int pvnum, CHAR_DATA *mob, OBJ_DATA *obj, ROOM_INDEX_DATA *room,
								CHAR_DATA *ch, const void *arg1,  const void *arg2);
	extern MPCODE *get_prog_index( int vnum, int type );

	argument = one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0')
	{
		bug( "OpCall: missing arguments from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if (( prg = get_prog_index( atoi(arg), PRG_OPROG ) ) == NULL)
	{
		bug( "OpCall: invalid prog from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	vch = NULL;
	obj1 = obj2 = NULL;
	argument = one_argument(argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		vch = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ);
	argument = one_argument(argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		obj1 = get_obj_here_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ);
	argument = one_argument(argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		obj2 = get_obj_here_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ);

	program_flow( prg->vnum, NULL, obj, NULL, vch, (void *)obj1, (void *)obj2 ); 
}

/*
 * Lets the object to transfer an object. The object must be in the same
 * room with the object.
 *
 * Syntax: obj otransfer [item name] [location]
 */
void do_opotransfer( OBJ_DATA *obj, const char *argument )
{
	OBJ_DATA *obj1, *dobj;
	ROOM_INDEX_DATA *location;
	char arg[ MAX_INPUT_LENGTH ];
	char buf[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;

	dobj=NULL;

        argument = one_argument(argument, arg, sizeof(arg));
	if (arg[0] == '\0')
	{
		bug( "OpOTransfer - Missing argument from vnum %d.",obj->pIndexData->vnum );
		return;
	}
	one_argument( argument, buf, sizeof(buf));
	if (is_number( buf ))
		location = get_room_index( atoi(buf) );
	else if ((victim = get_char_world_ext((CHAR_DATA *) dobj, buf, GET_F_CHAR_IS_OBJ )) != NULL)
		location = victim->in_room;
	else if (( dobj = get_obj_world_ext((CHAR_DATA *) dobj, arg, GET_F_CHAR_IS_OBJ )) != NULL)
		location = dobj->in_room;
	else
	{
		bug( "OpOTransfer - No such location from vnum %d.", 
			 obj->pIndexData->vnum );
		return;
	}

	if ((obj1 = get_obj_here_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ)) == NULL)
		return;
	if (obj1->carried_by == NULL)
		obj_from_room( obj1 );
	else
	{
		if (obj1->wear_loc != WEAR_NONE)
			unequip_char( obj1->carried_by, obj1 );
		obj_from_char( obj1 );
	}
	obj_to_room( obj1, location );
}

/*
 * Lets the object to strip an object or all objects from the victim.
 * Useful for removing e.g. quest objects from a character.
 *
 * Syntax: obj remove [victim] [object vnum|'all']
 */
void do_opremove( OBJ_DATA *obj, const char *argument )
{
	CHAR_DATA *victim;
	OBJ_DATA *obj1, *obj_next;
	sh_int vnum = 0;
	bool fAll = FALSE;
	char arg[ MAX_INPUT_LENGTH ];

	argument = one_argument(argument, arg, sizeof(arg));
	if (( victim = get_char_room_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ) ) == NULL)
		return;

	one_argument( argument, arg, sizeof(arg));
	if (!str_cmp( arg, "all" ))
		fAll = TRUE;
	else if (!is_number( arg ))
	{
		bug ( "OpRemove: Invalid object from vnum %d.", 
			  obj->pIndexData->vnum );
		return;
	}
	else
		vnum = atoi( arg );

	for (obj1 = victim->carrying; obj1; obj1 = obj_next)
	{
		obj_next = obj->next_content;
		if (fAll || obj1->pIndexData->vnum == vnum)
		{
			unequip_char( victim, obj1 );
			obj_from_char( obj1 );
			extract_obj( obj1 );
		}
	}
}



