/* $Id: ogrep.c,v 1.666 2004/09/20 10:49:51 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <sys/types.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>

#include "limited.h"
#include "merc.h"
#include "update.h"
#include "quest.h"
#include "obj_prog.h"
#include "fight.h"
#include "quest.h"
#include "db/cmd.h"
#include "db/db.h"
#include "db/dofun.h"
#define _OBJ_GREP
#include "grep.h"
#undef _OBJ_GREP
#include "olc/olc.h"

static CMD_DATA ogrep_cmd_table[] =
{
    { "help",       ogrep_help,       POS_DEAD,      92,              0,  0,     0,              0},
    { "weapon",     ogrep_weapons,    POS_DEAD,      92,              0,  0,     0,              0},
    { "armor",      ogrep_armor,      POS_DEAD,      92,              0,  0,     0,              0},
    { "pill",       ogrep_pill,       POS_DEAD,      92,              0,  0,     0,              0},
    { "wear",       ogrep_wear,       POS_DEAD,      92,              0,  0,     0,              0},
    { "portal",     ogrep_portal,     POS_DEAD,      92,              0,  0,     0,              0},
    { "windows",    ogrep_window,     POS_DEAD,      92,              0,  0,     0,              0},
    { "potion",     ogrep_potion,     POS_DEAD,      92,              0,  0,     0,              0},
    { "scroll",     ogrep_scroll,     POS_DEAD,      92,              0,  0,     0,              0},
    { "material",   ogrep_material,   POS_DEAD,      92,              0,  0,     0,              0},
    { NULL }
} ;


int get_sort_mode (const char* argument, int mode)
{
    GREP_SORTMODE*  smode;
    int             output;
    
    output = SORT_INVALID;
    
    for (smode = grep_sortmode; smode->name != NULL ; smode++)
        if (!str_prefix(argument, smode->name) && IS_SET(smode->sort_mode, mode))
                output =  smode->sort_value;
     
     return output;
}

void sort_show(CHAR_DATA *ch, int mode)
{
     GREP_SORTMODE*  smode;

     char_printf(ch, "Avialable mode fo sort:\n");

     for (smode = grep_sortmode; smode->name ; ++smode)
        if (IS_SET(smode->sort_mode, mode))
              char_printf (ch, "%-12s{x\n", smode->name);

     char_printf (ch, "\n") ;
}

void ogrep_free(ogrep_obj_t * ogrep_obj)
{
      free_string(ogrep_obj->name);
      free_string(ogrep_obj->material);
}

char * percent_color(int value, int max_value)
{
     int          number;
     char       * code;
     
     if (max_value == 0)
     {
          code = "{D";
          return (code);
     }
     
     number = value * 100 / max_value;
     
     if (number == 100)     code = "{G";         else
     if (number >= 80)      code = "{C";         else
     if (number >= 60)      code = "{M";         else
     if (number >= 40)      code = "{R";         else
     if (number >= 20)      code = "{r";         else
     if (number >=  0)      code = "{D";         else
                            code = "{m";     
     return (code);
}

void ogrep_help(CHAR_DATA *ch, const char *argument)
{
      do_help(ch, "ogrep");
}

// Comare function

static int cmp_obj_int(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_int - ((ogrep_obj_t*) p1)->stat_int;
}

static int cmp_obj_str(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_str - ((ogrep_obj_t*) p1)->stat_str;
}

static int cmp_obj_con(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_con - ((ogrep_obj_t*) p1)->stat_con;
}

static int cmp_obj_dex(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_dex - ((ogrep_obj_t*) p1)->stat_dex;
}

static int cmp_obj_cha(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_cha - ((ogrep_obj_t*) p1)->stat_cha;
}

static int cmp_obj_wis(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_wis - ((ogrep_obj_t*) p1)->stat_wis;
}

static int cmp_obj_lck(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->stat_lck - ((ogrep_obj_t*) p1)->stat_lck;
}

static int cmp_obj_ave(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->ave - ((ogrep_obj_t*) p1)->ave; 
}

static int cmp_obj_saf(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->saf - ((ogrep_obj_t*) p1)->saf;
}

static int cmp_obj_svs(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->svs - ((ogrep_obj_t*) p2)->svs;
}

static int cmp_obj_ac(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->ac - ((ogrep_obj_t*) p2)->ac;
}

static int cmp_obj_hr(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->hr - ((ogrep_obj_t*) p1)->hr;
}

static int cmp_obj_dr(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->dr - ((ogrep_obj_t*) p1)->dr;
}

static int cmp_obj_hp(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->hp - ((ogrep_obj_t*) p1)->hp;
}

static int cmp_obj_mana(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->mana - ((ogrep_obj_t*) p1)->mana;
}

static int cmp_obj_move(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->move - ((ogrep_obj_t*) p1)->move;
}

static int cmp_obj_level(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->level - ((ogrep_obj_t*) p1)->level;
}

static int cmp_obj_alevel(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->apply_level - ((ogrep_obj_t*) p1)->apply_level;
}

static int cmp_obj_v0(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->v0 - ((ogrep_obj_t*) p1)->v0;
}

static int cmp_obj_v1(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->v1 - ((ogrep_obj_t*) p1)->v1;
}

static int cmp_obj_v2(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->v2 - ((ogrep_obj_t*) p1)->v2;
}

static int cmp_obj_v3(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->v3 - ((ogrep_obj_t*) p1)->v3;
}

static int cmp_obj_v4(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p2)->v4 - ((ogrep_obj_t*) p1)->v4;
}

static int cmp_obj_v0_r(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->v0 - ((ogrep_obj_t*) p2)->v0;
}

static int cmp_obj_v1_r(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->v1 - ((ogrep_obj_t*) p2)->v1;
}

static int cmp_obj_v2_r(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->v2 - ((ogrep_obj_t*) p2)->v2;
}

static int cmp_obj_v3_r(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->v3 - ((ogrep_obj_t*) p2)->v3;
}

static int cmp_obj_v4_r(const void *p1, const void *p2)
{
     return ((ogrep_obj_t*) p1)->v4 - ((ogrep_obj_t*) p2)->v4;
}

void sort_ogrep(varr ogrep_objs, int sort_mode)
{
      switch (sort_mode)
      {
            case SORT_INT:         varr_qsort(&ogrep_objs, cmp_obj_int);               break;
            case SORT_STR:         varr_qsort(&ogrep_objs, cmp_obj_str);               break;
            case SORT_DEX:         varr_qsort(&ogrep_objs, cmp_obj_dex);               break;
            case SORT_WIS:         varr_qsort(&ogrep_objs, cmp_obj_wis);               break;
            case SORT_CON:         varr_qsort(&ogrep_objs, cmp_obj_con);               break;
            case SORT_CHA:         varr_qsort(&ogrep_objs, cmp_obj_cha);               break;
            case SORT_LCK:         varr_qsort(&ogrep_objs, cmp_obj_lck);               break;
            case SORT_HP:          varr_qsort(&ogrep_objs, cmp_obj_hp);                break;
            case SORT_MANA:        varr_qsort(&ogrep_objs, cmp_obj_mana);              break;
            case SORT_MOVE:        varr_qsort(&ogrep_objs, cmp_obj_move);              break;
            case SORT_SVS:         varr_qsort(&ogrep_objs, cmp_obj_svs);               break;
            case SORT_SAF:         varr_qsort(&ogrep_objs, cmp_obj_saf);               break;
            case SORT_HR:          varr_qsort(&ogrep_objs, cmp_obj_hr);                break;
            case SORT_DR:          varr_qsort(&ogrep_objs, cmp_obj_dr);                break;
            case SORT_AVE:         varr_qsort(&ogrep_objs, cmp_obj_ave);               break;
            case SORT_LEVEL:       varr_qsort(&ogrep_objs, cmp_obj_level);             break;   
            case SORT_ALEVEL:      varr_qsort(&ogrep_objs, cmp_obj_alevel);            break;   
            case SORT_AC:          varr_qsort(&ogrep_objs, cmp_obj_ac);                break;   
            case SORT_V0:          varr_qsort(&ogrep_objs, cmp_obj_v0);                break;
            case SORT_V1:          varr_qsort(&ogrep_objs, cmp_obj_v1);                break;
            case SORT_V2:          varr_qsort(&ogrep_objs, cmp_obj_v2);                break;
            case SORT_V3:          varr_qsort(&ogrep_objs, cmp_obj_v3);                break;
            case SORT_V4:          varr_qsort(&ogrep_objs, cmp_obj_v4);                break;
            // reverse function
            case RSORT_V0:         varr_qsort(&ogrep_objs, cmp_obj_v0_r);              break;
            case RSORT_V1:         varr_qsort(&ogrep_objs, cmp_obj_v1_r);              break;
            case RSORT_V2:         varr_qsort(&ogrep_objs, cmp_obj_v2_r);              break;
            case RSORT_V3:         varr_qsort(&ogrep_objs, cmp_obj_v3_r);              break;
            case RSORT_V4:         varr_qsort(&ogrep_objs, cmp_obj_v4_r);              break;
            
            default:                                                                   break;
      }
}
    
void fill_obj_data(ogrep_obj_t *ogrep_weapon, OBJ_INDEX_DATA *pObjIndex)
{
      AFFECT_DATA     *paf;
      MPTRIG          *optrig;
      int              cnt;
      
      ogrep_weapon->name          = str_dup(fmt_color_str(mlstr_mval(pObjIndex->short_descr),10));
      ogrep_weapon->vnum          = pObjIndex->vnum;
      ogrep_weapon->level         = pObjIndex->level;
      ogrep_weapon->limit         = pObjIndex->limit;
      ogrep_weapon->dice_number   = pObjIndex->value[1];
      ogrep_weapon->dice_type     = pObjIndex->value[2];
      ogrep_weapon->ave           = 1 + pObjIndex->value[2] * pObjIndex->value[1] / 2;
      ogrep_weapon->v0            = pObjIndex->value[0];
      ogrep_weapon->v1            = pObjIndex->value[1];
      ogrep_weapon->v2            = pObjIndex->value[2];
      ogrep_weapon->v3            = pObjIndex->value[3];
      ogrep_weapon->v4            = pObjIndex->value[4];
      ogrep_weapon->item_type     = pObjIndex->item_type;
      ogrep_weapon->wear_flags    = pObjIndex->wear_flags;
      ogrep_weapon->material      = str_dup(pObjIndex->material);
      
      
      
      for (cnt = 0, paf = pObjIndex->affected; paf; paf = paf->next) 
      {
                  
            switch(paf->location)
            {
                  case APPLY_INT:            ogrep_weapon->stat_int    = ogrep_weapon->stat_int + paf->modifier;           break;
                  case APPLY_CON:            ogrep_weapon->stat_con    = ogrep_weapon->stat_con + paf->modifier;           break;
                  case APPLY_STR:            ogrep_weapon->stat_str    = ogrep_weapon->stat_str + paf->modifier;           break;
                  case APPLY_WIS:            ogrep_weapon->stat_wis    = ogrep_weapon->stat_wis + paf->modifier;           break;
                  case APPLY_DEX:            ogrep_weapon->stat_dex    = ogrep_weapon->stat_dex + paf->modifier;           break;
                  case APPLY_CHA:            ogrep_weapon->stat_cha    = ogrep_weapon->stat_cha + paf->modifier;           break;
                  case APPLY_LCK:            ogrep_weapon->stat_lck    = ogrep_weapon->stat_lck + paf->modifier;           break;
                  case APPLY_HIT:            ogrep_weapon->hp          = ogrep_weapon->hp       + paf->modifier;           break;
                  case APPLY_MANA:           ogrep_weapon->mana        = ogrep_weapon->mana     + paf->modifier;           break;
                  case APPLY_MOVE:           ogrep_weapon->move        = ogrep_weapon->move     + paf->modifier;           break;
                  case APPLY_HITROLL:        ogrep_weapon->hr          = ogrep_weapon->hr       + paf->modifier;           break;
                  case APPLY_DAMROLL:        ogrep_weapon->dr          = ogrep_weapon->dr       + paf->modifier;           break;
                  case APPLY_SAVING_SPELL:   
                  case APPLY_SAVING_ROD:
                  case APPLY_SAVING_PETRI:
                  case APPLY_SAVING_BREATH:  ogrep_weapon->svs         = ogrep_weapon->svs      + paf->modifier;           break;
                  case APPLY_LEVEL:          ogrep_weapon->apply_level = ogrep_weapon->apply_level + paf->modifier;        break;
                  case APPLY_AC:             ogrep_weapon->ac          = ogrep_weapon->ac       + paf->modifier;           break;
                  case APPLY_SPELL_AFFECT:   ogrep_weapon->saf         = ogrep_weapon->saf      + paf->modifier;           break;
                  default:                                                                                                 break;
            }
            
            
            cnt++;
      }
      
      if (pObjIndex->oprogs)
      {
            for (optrig = pObjIndex->oprogs; optrig; optrig = optrig->next)
            {
                  ogrep_weapon->opcods++;
            }
            
      }
      
      if (pObjIndex->oprogs_old)
      {
            for (cnt = 0; cnt < OPROG_MAX; cnt++)
                  if (pObjIndex->oprogs_old[cnt] != NULL)
                        ogrep_weapon->objprogs++;
      }
}


DO_FUN (do_ogrep)
{
    if (!parse_command(ch, argument, ogrep_cmd_table))
    {
        show_command_list (ch, ogrep_cmd_table);
        char_puts (".\nUse {CHELP OGREP{x for more information.\n", ch);
    }
}

void ogrep_weapons (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     char             wclass[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t  *ogrep_weapon;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_weapons  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     
     argument = one_argument(argument, wclass, sizeof(wclass));
     
     if (flag_lookup(weapon_class, wclass) == NULL)
     {
          char_act("Ogrep: invalid argument. Valid syntax: ogrep <ITEM_TYPE>", ch);
          char_act("Ogrep: weapon class:", ch);
          show_flags(ch, weapon_class);
          return;
     }
     
     min_level = 0;
     max_level = 100000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 
                 sort_mode = get_sort_mode(arg1, SORT_WEAPON);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.", ch);
                       sort_show(ch,  SORT_WEAPON);
                       return;
                 }
                 
                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Ogrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);
                 
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Ogrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);        
           }
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     
     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_WEAPON) 
                    && (pObjIndex->value[0] == flag_value(weapon_class, wclass))
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_weapon = varr_enew (&ogrep_weapons);
                     fill_obj_data(ogrep_weapon, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_weapons.nused; i++)
     {
           ogrep_obj_t * weapons = varr_get(&ogrep_weapons, i);
           
           // Invert ....
           if (max_svs    > weapons->svs)                     max_svs  = weapons->svs;
           if (max_ac     > weapons->ac)                      max_ac   = weapons->ac;
           // Normal
           if (max_int    < weapons->stat_int)                max_int    = weapons->stat_int;
           if (max_str    < weapons->stat_str)                max_str    = weapons->stat_str;
           if (max_wis    < weapons->stat_wis)                max_wis    = weapons->stat_wis;
           if (max_con    < weapons->stat_con)                max_con    = weapons->stat_con;
           if (max_dex    < weapons->stat_dex)                max_dex    = weapons->stat_dex;
           if (max_cha    < weapons->stat_cha)                max_cha    = weapons->stat_cha;
           if (max_lck    < weapons->stat_lck)                max_lck    = weapons->stat_lck;
           if (max_hr     < weapons->hr)                      max_hr     = weapons->hr;
           if (max_dr     < weapons->dr)                      max_dr     = weapons->dr;
           if (max_hp     < weapons->hp)                      max_hp     = weapons->hp;
           if (max_mana   < weapons->mana)                    max_mana   = weapons->mana;
           if (max_move   < weapons->move)                    max_move   = weapons->move;
           if (max_ave    < weapons->ave)                     max_ave    = weapons->ave;
           if (max_alevel < weapons->apply_level)             max_alevel = weapons->apply_level;
           if (max_saf    < weapons->saf)                     max_saf    = weapons->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_weapons, sort_mode);
  
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {D[   Flags   {D] {D[{W%7s - %4s{D]{W %3s %2s %3s {D[{W%3s %3s{D] {D[{W%2s %2s %2s %2s %2s %2s %2s{D]{W {D[{W%4s %4s %4s{D]{W O P {x\n",
           "No", "Vnum", "Name", "Lvl", "Dice", "Ave", "SVS", "Sa", " Ac", "Hr", "Dr", 
           "St", "In", "Dx", "Cn", "Ws", "Ch", "Lk",
           " Hp "," Mn ", " Mv ");  
                   
     for (i = 0; i < ogrep_weapons.nused; i++)
     {
           ogrep_obj_t * weapons = varr_get(&ogrep_weapons, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D[%s{D] [{x%3d{Dd{x%-3d{D - %s%4d{D] %s%3d{x %s%2d{x %s%3d {D[{x%s%3d %s%3d{D] [%s%2d %s%2d %s%2d %s%2d %s%2d %s%2d %s%2d{D] [%s%4d %s%4d %s%4d{D] %s %s{x\n", 
                             i + 1,
                             weapons->vnum, 
                             weapons->name, 
                             weapons->level,
                             weapons->apply_level > 0 ? "{R" : "{D",
                             weapons->apply_level,
                             flag_dump(weapon_type2, weapons->v4),
                             weapons->dice_number, 
                             weapons->dice_type,
                             percent_color(weapons->ave, max_ave),
                             weapons->ave,
                             percent_color(weapons->svs, max_svs),
                             weapons->svs,
                             percent_color(weapons->saf, max_saf),
                             weapons->saf,
                             percent_color(weapons->ac, max_ac),
                             weapons->ac,
                             percent_color(weapons->hr, max_hr),
                             weapons->hr,
                             percent_color(weapons->hr, max_dr),
                             weapons->dr,
                             percent_color(weapons->stat_str, max_str),
                             weapons->stat_str,
                             percent_color(weapons->stat_int, max_int),
                             weapons->stat_int,
                             percent_color(weapons->stat_dex, max_dex),
                             weapons->stat_dex,
                             percent_color(weapons->stat_con, max_con),
                             weapons->stat_con,
                             percent_color(weapons->stat_wis, max_wis),
                             weapons->stat_wis,
                             percent_color(weapons->stat_cha, max_cha),
                             weapons->stat_cha,
                             percent_color(weapons->stat_lck, max_lck),
                             weapons->stat_lck,
                             percent_color(weapons->hp, max_hp),
                             weapons->hp,
                             percent_color(weapons->mana, max_mana),
                             weapons->mana,
                             percent_color(weapons->move, max_move),
                             weapons->move,
                             weapons->objprogs > 0 ? "{R*":"{D-",
                             weapons->opcods > 0 ? "{R*":"{D-"
                             );

     }
     
     for (i = ogrep_weapons.nused - 1; 0; i-- )
     {
        ogrep_obj_t * weapons = varr_get(&ogrep_weapons, i);
        ogrep_free(weapons);
        varr_del(&ogrep_weapons, (void*)weapons);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_pill (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_pill;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_pills  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_PILL) 
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_pill                = varr_enew (&ogrep_pills);
                     fill_obj_data(ogrep_pill, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_pills.nused; i++)
     {
           ogrep_obj_t * pills = varr_get(&ogrep_pills, i);
           
           // Invert ....
           if (max_svs    > pills->svs)                     max_svs  = pills->svs;
           if (max_ac     > pills->ac)                      max_ac   = pills->ac;
           // Normal
           if (max_int    < pills->stat_int)                max_int    = pills->stat_int;
           if (max_str    < pills->stat_str)                max_str    = pills->stat_str;
           if (max_wis    < pills->stat_wis)                max_wis    = pills->stat_wis;
           if (max_con    < pills->stat_con)                max_con    = pills->stat_con;
           if (max_dex    < pills->stat_dex)                max_dex    = pills->stat_dex;
           if (max_cha    < pills->stat_cha)                max_cha    = pills->stat_cha;
           if (max_lck    < pills->stat_lck)                max_lck    = pills->stat_lck;
           if (max_hr     < pills->hr)                      max_hr     = pills->hr;
           if (max_dr     < pills->dr)                      max_dr     = pills->dr;
           if (max_hp     < pills->hp)                      max_hp     = pills->hp;
           if (max_mana   < pills->mana)                    max_mana   = pills->mana;
           if (max_move   < pills->move)                    max_move   = pills->move;
           if (max_ave    < pills->ave)                     max_ave    = pills->ave;
           if (max_alevel < pills->apply_level)             max_alevel = pills->apply_level;
           if (max_saf    < pills->saf)                     max_saf    = pills->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_pills, sort_mode);
           
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {W O P  SLvl %-18s %-18s %-18s %-18s{x\n", "No", "Vnum", "Name", "Lvl",
     "Spell 1","Spell 2","Spell 3","Spell 4");  
                   
     for (i = 0; i < ogrep_pills.nused; i++)
     {
           ogrep_obj_t * pills = varr_get(&ogrep_pills, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D%s %s{x", 
                             i + 1,
                             pills->vnum, 
                             pills->name, 
                             pills->level,
                             pills->apply_level > 0 ? "{R" : "{D",
                             pills->apply_level,
                             pills->objprogs > 0 ? "{R*":"{D-",
                             pills->opcods > 0 ? "{R*":"{D-"
                             );
           
           buf_printf(output, " {D[{R%4d{D]{x %-18s %-18s %-18s %-18s ", pills->v0, skill_name(pills->v1), skill_name(pills->v2), skill_name(pills->v3), skill_name(pills->v4));
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_pills.nused - 1; 0; i-- )
     {
        ogrep_obj_t * pills = varr_get(&ogrep_pills, i);
        ogrep_free(pills);
        varr_del(&ogrep_pills, (void*)pills);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_armor (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
 
     int              total;
     int              wear_location;
     
     ogrep_obj_t     *ogrep_armor;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     
     int              max_v0     = 0;
     int              max_v1     = 0;
     int              max_v2     = 0;
     int              max_v3     = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_armors  = {sizeof (ogrep_obj_t), 4};
     
     total         = 0;
     min_level     = 0;
     max_level     = 100000;
     bSort         = FALSE;
     sort_mode     = SORT_NONE;
     wear_location = -1;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_ARMOR);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_ARMOR);
                       return;
                 }
                 
                 bSort = TRUE;
           }
           else
           if (!str_cmp(arg1, "wear"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 if (flag_lookup(wear_flags, arg1) == NULL)
                 {
                       char_act("Ogrep: invalid wear location.", ch);
                       show_flags(ch, wear_flags);
                       return;
                 }
                 
                 wear_location = flag_value(wear_flags, arg1);
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Ogrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);
                 
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Ogrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);        
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     
     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_ARMOR) 
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1
                    && ((wear_location >= 0) ? IS_SET(pObjIndex->wear_flags, wear_location) : TRUE)
                   ) 
               {
                     
                     ogrep_armor = varr_enew (&ogrep_armors);
                     fill_obj_data(ogrep_armor, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_armors.nused; i++)
     {
           ogrep_obj_t * armors = varr_get(&ogrep_armors, i);
           
           // Invert ....
           if (max_svs    > armors->svs)                     max_svs  = armors->svs;
           if (max_ac     > armors->ac)                      max_ac   = armors->ac;
           // Normal
           if (max_int    < armors->stat_int)                max_int    = armors->stat_int;
           if (max_str    < armors->stat_str)                max_str    = armors->stat_str;
           if (max_wis    < armors->stat_wis)                max_wis    = armors->stat_wis;
           if (max_con    < armors->stat_con)                max_con    = armors->stat_con;
           if (max_dex    < armors->stat_dex)                max_dex    = armors->stat_dex;
           if (max_cha    < armors->stat_cha)                max_cha    = armors->stat_cha;
           if (max_lck    < armors->stat_lck)                max_lck    = armors->stat_lck;
           if (max_hr     < armors->hr)                      max_hr     = armors->hr;
           if (max_dr     < armors->dr)                      max_dr     = armors->dr;
           if (max_hp     < armors->hp)                      max_hp     = armors->hp;
           if (max_mana   < armors->mana)                    max_mana   = armors->mana;
           if (max_move   < armors->move)                    max_move   = armors->move;
           
           if (max_v0     < armors->v0)                      max_v0     = armors->v0;
           if (max_v1     < armors->v1)                      max_v1     = armors->v1;
           if (max_v2     < armors->v2)                      max_v2     = armors->v2;
           if (max_v3     < armors->v3)                      max_v3     = armors->v3;
           
           if (max_alevel < armors->apply_level)             max_alevel = armors->apply_level;
           if (max_saf    < armors->saf)                     max_saf    = armors->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_armors, sort_mode);
  
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {D[{WPrc Bsh Slh Ext{D]{W %3s %2s %3s {D[{W%3s %3s{D] {D[{W%2s %2s %2s %2s %2s %2s %2s{D]{W {D[{W%4s %4s %4s{D]{W O P {x\n",
           "No", "Vnum", "Name", "Lvl", "SVS", "Sa", " Ac", "Hr", "Dr", 
           "St", "In", "Dx", "Cn", "Ws", "Ch", "Lk",
           " Hp "," Mn ", " Mv ");  
                   
     for (i = 0; i < ogrep_armors.nused; i++)
     {
           ogrep_obj_t * armors = varr_get(&ogrep_armors, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d [%s%-3d %s%-3d %s%-3d %s%-3d{D] %s%3d{x %s%2d{x %s%3d {D[{x%s%3d %s%3d{D] [%s%2d %s%2d %s%2d %s%2d %s%2d %s%2d %s%2d{D] [%s%4d %s%4d %s%4d{D] %s %s{x\n", 
                             i + 1,
                             armors->vnum, 
                             armors->name, 
                             armors->level,
                             armors->apply_level > 0 ? "{R" : "{D",
                             armors->apply_level,
                             
                             
                             percent_color(armors->v0, max_v0),
                             armors->v0,
                             percent_color(armors->v1, max_v1),
                             armors->v1,
                             percent_color(armors->v2, max_v2),
                             armors->v2,
                             percent_color(armors->v3, max_v3),
                             armors->v3,
                                                                                    
                             percent_color(armors->svs, max_svs),
                             armors->svs,
                             percent_color(armors->saf, max_saf),
                             armors->saf,
                             percent_color(armors->ac, max_ac),
                             armors->ac,
                             percent_color(armors->hr, max_hr),
                             armors->hr,
                             percent_color(armors->hr, max_dr),
                             armors->dr,
                             percent_color(armors->stat_str, max_str),
                             armors->stat_str,
                             percent_color(armors->stat_int, max_int),
                             armors->stat_int,
                             percent_color(armors->stat_dex, max_dex),
                             armors->stat_dex,
                             percent_color(armors->stat_con, max_con),
                             armors->stat_con,
                             percent_color(armors->stat_wis, max_wis),
                             armors->stat_wis,
                             percent_color(armors->stat_cha, max_cha),
                             armors->stat_cha,
                             percent_color(armors->stat_lck, max_lck),
                             armors->stat_lck,
                             percent_color(armors->hp, max_hp),
                             armors->hp,
                             percent_color(armors->mana, max_mana),
                             armors->mana,
                             percent_color(armors->move, max_move),
                             armors->move,
                             armors->objprogs > 0 ? "{R*":"{D-",
                             armors->opcods > 0 ? "{R*":"{D-"
                             );

     }
     
     for (i = ogrep_armors.nused - 1; 0; i-- )
     {
        ogrep_obj_t * armors = varr_get(&ogrep_armors, i);
        ogrep_free(armors);
        varr_del(&ogrep_armors, (void*)armors);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}


void ogrep_wear (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              wear_location;
     int              total;
     
     ogrep_obj_t     *ogrep_wear;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_wears  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     
     argument = one_argument(argument, arg1, sizeof(arg1));
     
     if (flag_lookup(wear_flags, arg1) == NULL)
     {
          char_act("Ogrep: invalid argument. Valid syntax: ogrep <ITEM_TYPE>", ch);
          char_act("Ogrep: wear class:", ch);
          show_flags(ch, wear_flags);
          return;
     }
     
     wear_location = flag_value(wear_flags, arg1);
     
     min_level = 0;
     max_level = 100000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 
                 sort_mode = get_sort_mode(arg1, SORT_WEAPON);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.", ch);
                       sort_show(ch,  SORT_WEAPON);
                       return;
                 }
                 
                 bSort = TRUE;
           }
           else if (!str_cmp(arg1, "level"))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < 0)
                 {
                       char_act("Ogrep: argument must be number and greater then 0.", ch);
                       return;
                 }
                 else
                       min_level = atoi(arg1);
                 
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 if (!is_number(arg1) || atoi(arg1) < min_level)
                 {
                       char_act("Ogrep: argument must be number and greater then min_level.", ch);
                       return;
                 }
                 else
                       max_level = atoi(arg1);        
           }
           else
           {
                 char_printf(ch,"Mgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     
     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (     IS_SET(pObjIndex->wear_flags, wear_location)
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_wear = varr_enew (&ogrep_wears);
                     fill_obj_data(ogrep_wear, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_wears.nused; i++)
     {
           ogrep_obj_t * wears = varr_get(&ogrep_wears, i);
           
           // Invert ....
           if (max_svs    > wears->svs)                     max_svs  = wears->svs;
           if (max_ac     > wears->ac)                      max_ac   = wears->ac;
           // Normal
           if (max_int    < wears->stat_int)                max_int    = wears->stat_int;
           if (max_str    < wears->stat_str)                max_str    = wears->stat_str;
           if (max_wis    < wears->stat_wis)                max_wis    = wears->stat_wis;
           if (max_con    < wears->stat_con)                max_con    = wears->stat_con;
           if (max_dex    < wears->stat_dex)                max_dex    = wears->stat_dex;
           if (max_cha    < wears->stat_cha)                max_cha    = wears->stat_cha;
           if (max_lck    < wears->stat_lck)                max_lck    = wears->stat_lck;
           if (max_hr     < wears->hr)                      max_hr     = wears->hr;
           if (max_dr     < wears->dr)                      max_dr     = wears->dr;
           if (max_hp     < wears->hp)                      max_hp     = wears->hp;
           if (max_mana   < wears->mana)                    max_mana   = wears->mana;
           if (max_move   < wears->move)                    max_move   = wears->move;
           if (max_ave    < wears->ave)                     max_ave    = wears->ave;
           if (max_alevel < wears->apply_level)             max_alevel = wears->apply_level;
           if (max_saf    < wears->saf)                     max_saf    = wears->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_wears, sort_mode);
  
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {D[   Flags   {D] {D[{W%7s - %4s{D]{W %3s %2s %3s {D[{W%3s %3s{D] {D[{W%2s %2s %2s %2s %2s %2s %2s{D]{W {D[{W%4s %4s %4s{D]{W O P {x\n",
           "No", "Vnum", "Name", "Lvl", "Dice", "Ave", "SVS", "Sa", " Ac", "Hr", "Dr", 
           "St", "In", "Dx", "Cn", "Ws", "Ch", "Lk",
           " Hp "," Mn ", " Mv ");  
                   
     for (i = 0; i < ogrep_wears.nused; i++)
     {
           ogrep_obj_t * wears = varr_get(&ogrep_wears, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D[%11s{D] [{x%3d{Dd{x%-3d{D - %s%4d{D] %s%3d{x %s%2d{x %s%3d {D[{x%s%3d %s%3d{D] [%s%2d %s%2d %s%2d %s%2d %s%2d %s%2d %s%2d{D] [%s%4d %s%4d %s%4d{D] %s %s{x\n", 
                             i + 1,
                             wears->vnum, 
                             wears->name, 
                             wears->level,
                             wears->apply_level > 0 ? "{R" : "{D",
                             wears->apply_level,
                             flag_string(item_types, wears->item_type),
                             wears->dice_number, 
                             wears->dice_type,
                             percent_color(wears->ave, max_ave),
                             wears->ave,
                             percent_color(wears->svs, max_svs),
                             wears->svs,
                             percent_color(wears->saf, max_saf),
                             wears->saf,
                             percent_color(wears->ac, max_ac),
                             wears->ac,
                             percent_color(wears->hr, max_hr),
                             wears->hr,
                             percent_color(wears->hr, max_dr),
                             wears->dr,
                             percent_color(wears->stat_str, max_str),
                             wears->stat_str,
                             percent_color(wears->stat_int, max_int),
                             wears->stat_int,
                             percent_color(wears->stat_dex, max_dex),
                             wears->stat_dex,
                             percent_color(wears->stat_con, max_con),
                             wears->stat_con,
                             percent_color(wears->stat_wis, max_wis),
                             wears->stat_wis,
                             percent_color(wears->stat_cha, max_cha),
                             wears->stat_cha,
                             percent_color(wears->stat_lck, max_lck),
                             wears->stat_lck,
                             percent_color(wears->hp, max_hp),
                             wears->hp,
                             percent_color(wears->mana, max_mana),
                             wears->mana,
                             percent_color(wears->move, max_move),
                             wears->move,
                             wears->objprogs > 0 ? "{R*":"{D-",
                             wears->opcods > 0 ? "{R*":"{D-"
                             );

     }
     
     for (i = ogrep_wears.nused - 1; 0; i-- )
     {
        ogrep_obj_t * wears = varr_get(&ogrep_wears, i);
        ogrep_free(wears);
        varr_del(&ogrep_wears, (void*)wears);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_portal (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     ROOM_INDEX_DATA *room;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_portal;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_portals  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if ( pObjIndex->item_type == ITEM_PORTAL) 
               {
                     
                     ogrep_portal                = varr_enew (&ogrep_portals);
                     fill_obj_data(ogrep_portal, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_portals.nused; i++)
     {
           ogrep_obj_t * portals = varr_get(&ogrep_portals, i);
           
           // Invert ....
           if (max_svs    > portals->svs)                     max_svs  = portals->svs;
           if (max_ac     > portals->ac)                      max_ac   = portals->ac;
           // Normal
           if (max_int    < portals->stat_int)                max_int    = portals->stat_int;
           if (max_str    < portals->stat_str)                max_str    = portals->stat_str;
           if (max_wis    < portals->stat_wis)                max_wis    = portals->stat_wis;
           if (max_con    < portals->stat_con)                max_con    = portals->stat_con;
           if (max_dex    < portals->stat_dex)                max_dex    = portals->stat_dex;
           if (max_cha    < portals->stat_cha)                max_cha    = portals->stat_cha;
           if (max_lck    < portals->stat_lck)                max_lck    = portals->stat_lck;
           if (max_hr     < portals->hr)                      max_hr     = portals->hr;
           if (max_dr     < portals->dr)                      max_dr     = portals->dr;
           if (max_hp     < portals->hp)                      max_hp     = portals->hp;
           if (max_mana   < portals->mana)                    max_mana   = portals->mana;
           if (max_move   < portals->move)                    max_move   = portals->move;
           if (max_ave    < portals->ave)                     max_ave    = portals->ave;
           if (max_alevel < portals->apply_level)             max_alevel = portals->apply_level;
           if (max_saf    < portals->saf)                     max_saf    = portals->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_portals, sort_mode);
           
//                                                               
     buf_printf(output, "{WNo  {D[{W Vnum {D]{W    Name      Lvl {W O P  M Chr        ExitFlag    PortalF  Room{x\n");  
                   
     for (i = 0; i < ogrep_portals.nused; i++)
     {
           ogrep_obj_t * portals = varr_get(&ogrep_portals, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d  {D%s %s  %s %6d{x", 
                             i + 1,
                             portals->vnum, 
                             portals->name, 
                             portals->level,
                             portals->apply_level > 0 ? "{R" : "{D",
                             portals->apply_level,
                             portals->objprogs > 0 ? "{R*":"{D-",
                             portals->opcods > 0 ? "{R*":"{D-",
                             IS_SET(portals->wear_flags, ITEM_TAKE) ? "{RM{x" : "{D-{x",
                             portals->v0
                             );
           
           buf_printf(output,"  [%s]", flag_dump(exit_flags, portals->v1));
           buf_printf(output,"  [%s]", flag_dump(portal_flags, portals->v2));
           
           room = get_room_index(portals->v3);
           
           if (room != NULL)
                      buf_printf(output,"  {D[{C%-6d{D]{x",portals->v3);
           else
                      buf_printf(output,"  {D[{R%-6d{D]{x",portals->v3);
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_portals.nused - 1; 0; i-- )
     {
        ogrep_obj_t * portals = varr_get(&ogrep_portals, i);
        ogrep_free(portals);
        varr_del(&ogrep_portals, (void*)portals);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_window (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     ROOM_INDEX_DATA *room;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_window;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_windows  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if ( pObjIndex->item_type == ITEM_WINDOW) 
               {
                     
                     ogrep_window                = varr_enew (&ogrep_windows);
                     fill_obj_data(ogrep_window, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_windows.nused; i++)
     {
           ogrep_obj_t * windows = varr_get(&ogrep_windows, i);
           
           // Invert ....
           if (max_svs    > windows->svs)                     max_svs  = windows->svs;
           if (max_ac     > windows->ac)                      max_ac   = windows->ac;
           // Normal
           if (max_int    < windows->stat_int)                max_int    = windows->stat_int;
           if (max_str    < windows->stat_str)                max_str    = windows->stat_str;
           if (max_wis    < windows->stat_wis)                max_wis    = windows->stat_wis;
           if (max_con    < windows->stat_con)                max_con    = windows->stat_con;
           if (max_dex    < windows->stat_dex)                max_dex    = windows->stat_dex;
           if (max_cha    < windows->stat_cha)                max_cha    = windows->stat_cha;
           if (max_lck    < windows->stat_lck)                max_lck    = windows->stat_lck;
           if (max_hr     < windows->hr)                      max_hr     = windows->hr;
           if (max_dr     < windows->dr)                      max_dr     = windows->dr;
           if (max_hp     < windows->hp)                      max_hp     = windows->hp;
           if (max_mana   < windows->mana)                    max_mana   = windows->mana;
           if (max_move   < windows->move)                    max_move   = windows->move;
           if (max_ave    < windows->ave)                     max_ave    = windows->ave;
           if (max_alevel < windows->apply_level)             max_alevel = windows->apply_level;
           if (max_saf    < windows->saf)                     max_saf    = windows->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_windows, sort_mode);
           
//                                                               
     buf_printf(output, "{WNo  {D[{W Vnum {D]{W    Name      Lvl {W O P  M   Chr        Room{x\n");  
                   
     for (i = 0; i < ogrep_windows.nused; i++)
     {
           ogrep_obj_t * windows = varr_get(&ogrep_windows, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d  {D%s %s  %s %6d{x", 
                             i + 1,
                             windows->vnum, 
                             windows->name, 
                             windows->level,
                             windows->apply_level > 0 ? "{R" : "{D",
                             windows->apply_level,
                             windows->objprogs > 0 ? "{R*":"{D-",
                             windows->opcods > 0 ? "{R*":"{D-",
                             IS_SET(windows->wear_flags, ITEM_TAKE) ? "{RM{x" : "{D-{x",
                             windows->v0
                             );
           
           room = get_room_index(windows->v0);
           
           if (room != NULL)
                      buf_printf(output,"  {D[{C%-6d{D]{x",windows->v0);
           else
                      buf_printf(output,"  {D[{R%-6d{D]{x",windows->v0);
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_windows.nused - 1; 0; i-- )
     {
        ogrep_obj_t * windows = varr_get(&ogrep_windows, i);
        ogrep_free(windows);
        varr_del(&ogrep_windows, (void*)windows);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_potion (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_potion;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_potions  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_POTION) 
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_potion                = varr_enew (&ogrep_potions);
                     fill_obj_data(ogrep_potion, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_potions.nused; i++)
     {
           ogrep_obj_t * potions = varr_get(&ogrep_potions, i);
           
           // Invert ....
           if (max_svs    > potions->svs)                     max_svs  = potions->svs;
           if (max_ac     > potions->ac)                      max_ac   = potions->ac;
           // Normal
           if (max_int    < potions->stat_int)                max_int    = potions->stat_int;
           if (max_str    < potions->stat_str)                max_str    = potions->stat_str;
           if (max_wis    < potions->stat_wis)                max_wis    = potions->stat_wis;
           if (max_con    < potions->stat_con)                max_con    = potions->stat_con;
           if (max_dex    < potions->stat_dex)                max_dex    = potions->stat_dex;
           if (max_cha    < potions->stat_cha)                max_cha    = potions->stat_cha;
           if (max_lck    < potions->stat_lck)                max_lck    = potions->stat_lck;
           if (max_hr     < potions->hr)                      max_hr     = potions->hr;
           if (max_dr     < potions->dr)                      max_dr     = potions->dr;
           if (max_hp     < potions->hp)                      max_hp     = potions->hp;
           if (max_mana   < potions->mana)                    max_mana   = potions->mana;
           if (max_move   < potions->move)                    max_move   = potions->move;
           if (max_ave    < potions->ave)                     max_ave    = potions->ave;
           if (max_alevel < potions->apply_level)             max_alevel = potions->apply_level;
           if (max_saf    < potions->saf)                     max_saf    = potions->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_potions, sort_mode);
           
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {W O P  SLvl %-18s %-18s %-18s %-18s{x\n", "No", "Vnum", "Name", "Lvl",
     "Spell 1","Spell 2","Spell 3","Spell 4");  
                   
     for (i = 0; i < ogrep_potions.nused; i++)
     {
           ogrep_obj_t * potions = varr_get(&ogrep_potions, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D%s %s{x", 
                             i + 1,
                             potions->vnum, 
                             potions->name, 
                             potions->level,
                             potions->apply_level > 0 ? "{R" : "{D",
                             potions->apply_level,
                             potions->objprogs > 0 ? "{R*":"{D-",
                             potions->opcods > 0 ? "{R*":"{D-"
                             );
           
           buf_printf(output, " {D[{R%4d{D]{x %-18s %-18s %-18s %-18s ", potions->v0, skill_name(potions->v1), skill_name(potions->v2), skill_name(potions->v3), skill_name(potions->v4));
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_potions.nused - 1; 0; i-- )
     {
        ogrep_obj_t * potions = varr_get(&ogrep_potions, i);
        ogrep_free(potions);
        varr_del(&ogrep_potions, (void*)potions);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void ogrep_scroll (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_scroll;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
          
     varr ogrep_scrolls  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_SCROLL) 
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_scroll                = varr_enew (&ogrep_scrolls);
                     fill_obj_data(ogrep_scroll, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_scrolls.nused; i++)
     {
           ogrep_obj_t * scrolls = varr_get(&ogrep_scrolls, i);
           
           // Invert ....
           if (max_svs    > scrolls->svs)                     max_svs  = scrolls->svs;
           if (max_ac     > scrolls->ac)                      max_ac   = scrolls->ac;
           // Normal
           if (max_int    < scrolls->stat_int)                max_int    = scrolls->stat_int;
           if (max_str    < scrolls->stat_str)                max_str    = scrolls->stat_str;
           if (max_wis    < scrolls->stat_wis)                max_wis    = scrolls->stat_wis;
           if (max_con    < scrolls->stat_con)                max_con    = scrolls->stat_con;
           if (max_dex    < scrolls->stat_dex)                max_dex    = scrolls->stat_dex;
           if (max_cha    < scrolls->stat_cha)                max_cha    = scrolls->stat_cha;
           if (max_lck    < scrolls->stat_lck)                max_lck    = scrolls->stat_lck;
           if (max_hr     < scrolls->hr)                      max_hr     = scrolls->hr;
           if (max_dr     < scrolls->dr)                      max_dr     = scrolls->dr;
           if (max_hp     < scrolls->hp)                      max_hp     = scrolls->hp;
           if (max_mana   < scrolls->mana)                    max_mana   = scrolls->mana;
           if (max_move   < scrolls->move)                    max_move   = scrolls->move;
           if (max_ave    < scrolls->ave)                     max_ave    = scrolls->ave;
           if (max_alevel < scrolls->apply_level)             max_alevel = scrolls->apply_level;
           if (max_saf    < scrolls->saf)                     max_saf    = scrolls->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_scrolls, sort_mode);
           
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {W O P  SLvl %-18s %-18s %-18s %-18s{x\n", "No", "Vnum", "Name", "Lvl",
     "Spell 1","Spell 2","Spell 3","Spell 4");  
                   
     for (i = 0; i < ogrep_scrolls.nused; i++)
     {
           ogrep_obj_t * scrolls = varr_get(&ogrep_scrolls, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D%s %s{x", 
                             i + 1,
                             scrolls->vnum, 
                             scrolls->name, 
                             scrolls->level,
                             scrolls->apply_level > 0 ? "{R" : "{D",
                             scrolls->apply_level,
                             scrolls->objprogs > 0 ? "{R*":"{D-",
                             scrolls->opcods > 0 ? "{R*":"{D-"
                             );
           
           buf_printf(output, " {D[{R%4d{D]{x %-18s %-18s %-18s %-18s ", scrolls->v0, skill_name(scrolls->v1), skill_name(scrolls->v2), skill_name(scrolls->v3), skill_name(scrolls->v4));
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_scrolls.nused - 1; 0; i-- )
     {
        ogrep_obj_t * scrolls = varr_get(&ogrep_scrolls, i);
        ogrep_free(scrolls);
        varr_del(&ogrep_scrolls, (void*)scrolls);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}
/*
void ogrep_shop (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     ogrep_obj_t     *ogrep_shop;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;
     
     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;
     
     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;
     
     int              i;
     
// mob section     
     int              mobvnum;
     MOB_INDEX_DATA  *pMobIndex;
     int              keeper     = -1;
               
     varr ogrep_shops  = {sizeof (ogrep_obj_t), 4};
     
     total = 0;
     min_level = 0;
     max_level = 1000;
     
     bSort = FALSE;
     sort_mode = SORT_NONE;
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
           
           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));
                 
                 sort_mode = get_sort_mode(arg1, SORT_PILL);
                 
                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }
                                  
                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);
//      
     for (mobvnum = 0; total < top_mob_index; mobvnum++)
          if ((pMobIndex = get_mob_index(vnum)) != NULL)
          {          
               total++;
               
               if (pMobIndex->pShop != NULL)
               {
                     
                     
               }
          }

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {          
               total++;
                                       
               if (    (pObjIndex->item_type == ITEM_SCROLL) 
                    &&  pObjIndex->level > min_level -1 
                    &&  pObjIndex->level < max_level + 1)
               {
                     
                     ogrep_shop                = varr_enew (&ogrep_shops);
                     fill_obj_data(ogrep_shop, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_shops.nused; i++)
     {
           ogrep_obj_t * shops = varr_get(&ogrep_shops, i);
           
           // Invert ....
           if (max_svs    > shops->svs)                     max_svs  = shops->svs;
           if (max_ac     > shops->ac)                      max_ac   = shops->ac;
           // Normal
           if (max_int    < shops->stat_int)                max_int    = shops->stat_int;
           if (max_str    < shops->stat_str)                max_str    = shops->stat_str;
           if (max_wis    < shops->stat_wis)                max_wis    = shops->stat_wis;
           if (max_con    < shops->stat_con)                max_con    = shops->stat_con;
           if (max_dex    < shops->stat_dex)                max_dex    = shops->stat_dex;
           if (max_cha    < shops->stat_cha)                max_cha    = shops->stat_cha;
           if (max_lck    < shops->stat_lck)                max_lck    = shops->stat_lck;
           if (max_hr     < shops->hr)                      max_hr     = shops->hr;
           if (max_dr     < shops->dr)                      max_dr     = shops->dr;
           if (max_hp     < shops->hp)                      max_hp     = shops->hp;
           if (max_mana   < shops->mana)                    max_mana   = shops->mana;
           if (max_move   < shops->move)                    max_move   = shops->move;
           if (max_ave    < shops->ave)                     max_ave    = shops->ave;
           if (max_alevel < shops->apply_level)             max_alevel = shops->apply_level;
           if (max_saf    < shops->saf)                     max_saf    = shops->saf;
     }
     
     if (bSort)
           sort_ogrep(ogrep_shops, sort_mode);
           
//                                                               
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {W O P  SLvl %-18s %-18s %-18s %-18s{x\n", "No", "Vnum", "Name", "Lvl",
     "Spell 1","Spell 2","Spell 3","Spell 4");  
                   
     for (i = 0; i < ogrep_shops.nused; i++)
     {
           ogrep_obj_t * shops = varr_get(&ogrep_shops, i);
           
           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d {D%s %s{x", 
                             i + 1,
                             shops->vnum, 
                             shops->name, 
                             shops->level,
                             shops->apply_level > 0 ? "{R" : "{D",
                             shops->apply_level,
                             shops->objprogs > 0 ? "{R*":"{D-",
                             shops->opcods > 0 ? "{R*":"{D-"
                             );
           
           buf_printf(output, " {D[{R%4d{D]{x %-18s %-18s %-18s %-18s ", shops->v0, skill_name(shops->v1), skill_name(shops->v2), skill_name(shops->v3), skill_name(shops->v4));
                             
           buf_printf(output, "\n");

     }
     
     
     for (i = ogrep_shops.nused - 1; 0; i-- )
     {
        ogrep_obj_t * shops = varr_get(&ogrep_shops, i);
        ogrep_free(shops);
        varr_del(&ogrep_shops, (void*)shops);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}
*/

void ogrep_material (CHAR_DATA *ch, const char * argument )
{
     OBJ_INDEX_DATA  *pObjIndex;
     int              vnum;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     char            material_name[MAX_INPUT_LENGTH];

     ogrep_obj_t     *ogrep_material;
     flag32_t         sort_mode;
     bool             bSort;
     int              min_level;
     int              max_level;

     int              max_hr     = 0;
     int              max_dr     = 0;
     int              max_hp     = 0;
     int              max_mana   = 0;
     int              max_move   = 0;
     int              max_svs    = 0;
     int              max_ave    = 0;

     int              max_int    = 0;
     int              max_str    = 0;
     int              max_con    = 0;
     int              max_dex    = 0;
     int              max_wis    = 0;
     int              max_cha    = 0;
     int              max_lck    = 0;
     int              max_saf    = 0;
     int              max_ac     = 0;
     int              max_alevel = 0;

     int              i;

     varr ogrep_materials  = {sizeof (ogrep_obj_t), 4};

     total = 0;
     min_level = 0;
     max_level = 1000;

     bSort = FALSE;
     sort_mode = SORT_NONE;

     if (argument[0] == '\0')
     {
           char_act("Ogrep: You must specify material.", ch);
           return;
      }

      argument = one_argument(argument, material_name, sizeof(material_name));

     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));

           if (!str_cmp(arg1, "sort"  ))
           {
                 argument = one_argument(argument, arg1, sizeof(arg1));

                 sort_mode = get_sort_mode(arg1, SORT_PILL);

                 if (sort_mode < 0 )
                 {
                       char_act("Ogrep: invalid sort mode.\nValid mode:", ch);
                       sort_show(ch, SORT_PILL);
                       return;
                 }

                 bSort = TRUE;
           }
           else
           {
                 char_printf(ch,"Ogrep: invalid command: %s.\n",arg1);
                 return;
           }
     }

     output = buf_new(-1);

     for (vnum = 0; total < top_obj_index; vnum++)
          if ((pObjIndex = get_obj_index(vnum)) != NULL)
          {
               total++;

               if (    (!str_cmp(pObjIndex->material, material_name))
                    &&  pObjIndex->level > min_level -1
                    &&  pObjIndex->level < max_level + 1)
               {

                     ogrep_material                = varr_enew (&ogrep_materials);
                     fill_obj_data(ogrep_material, pObjIndex);
                }
          }

     // Checking for max parameter
     for (i = 0; i < ogrep_materials.nused; i++)
     {
           ogrep_obj_t * materials = varr_get(&ogrep_materials, i);

           // Invert ....
           if (max_svs    > materials->svs)                     max_svs  = materials->svs;
           if (max_ac     > materials->ac)                      max_ac   = materials->ac;
           // Normal
           if (max_int    < materials->stat_int)                max_int    = materials->stat_int;
           if (max_str    < materials->stat_str)                max_str    = materials->stat_str;
           if (max_wis    < materials->stat_wis)                max_wis    = materials->stat_wis;
           if (max_con    < materials->stat_con)                max_con    = materials->stat_con;
           if (max_dex    < materials->stat_dex)                max_dex    = materials->stat_dex;
           if (max_cha    < materials->stat_cha)                max_cha    = materials->stat_cha;
           if (max_lck    < materials->stat_lck)                max_lck    = materials->stat_lck;
           if (max_hr     < materials->hr)                      max_hr     = materials->hr;
           if (max_dr     < materials->dr)                      max_dr     = materials->dr;
           if (max_hp     < materials->hp)                      max_hp     = materials->hp;
           if (max_mana   < materials->mana)                    max_mana   = materials->mana;
           if (max_move   < materials->move)                    max_move   = materials->move;
           if (max_ave    < materials->ave)                     max_ave    = materials->ave;
           if (max_alevel < materials->apply_level)             max_alevel = materials->apply_level;
           if (max_saf    < materials->saf)                     max_saf    = materials->saf;
     }

     if (bSort)
           sort_ogrep(ogrep_materials, sort_mode);

//
     buf_printf(output, "{W%3s {D[{W%6s{D]{W %-10s   %3s {W O P  Material{x\n", "No", "Vnum", "Name", "Lvl");

     for (i = 0; i < ogrep_materials.nused; i++)
     {
           ogrep_obj_t * materials = varr_get(&ogrep_materials, i);

           buf_printf(output, "{D%3d{x {D[{w%6d{D]{x %-10s {W%3d %s%1d  {D%s %s{x",
                             i + 1,
                             materials->vnum,
                             materials->name,
                             materials->level,
                             materials->apply_level > 0 ? "{R" : "{D",
                             materials->apply_level,
                             materials->objprogs > 0 ? "{R*":"{D-",
                             materials->opcods > 0 ? "{R*":"{D-"
                             );

           buf_printf(output, " {D[{W%-20s{D]", materials->material);

           buf_printf(output, "\n");

     }


     for (i = ogrep_materials.nused - 1; 0; i-- )
     {
        ogrep_obj_t * materials = varr_get(&ogrep_materials, i);
        ogrep_free(materials);
        varr_del(&ogrep_materials, (void*)materials);
     }

     page_to_char(buf_string(output), ch);
     buf_free(output);

}
