/*  $Id: olc_alias.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <string.h>

#include "merc.h"
#include "olc.h"
#include "db/cmd.h"
#include "db/alias.h"
#include "../skills.h"

#define EDIT_ALIAS(ch, alias) (alias = (alias_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(aliased_create        );
DECLARE_OLC_FUN(aliased_edit      );
DECLARE_OLC_FUN(aliased_show      );
DECLARE_OLC_FUN(aliased_list      );

DECLARE_OLC_FUN(aliased_name      );
DECLARE_OLC_FUN(aliased_command   );
DECLARE_OLC_FUN(aliased_prefix    );
DECLARE_OLC_FUN(aliased_suffix        );

static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_alias[] =
{
    { "create",         aliased_create,      2                     },
    { "edit",           aliased_edit,        1                     },
    { "touch",          olced_dummy,         0                     },
    { "show",           aliased_show,        0                     },
    { "list",           aliased_list,        0                     },

    { "name",           aliased_name,        0,      validate_name },
    { "substitute",     aliased_command,     0,                    },
    { "prefix",         aliased_prefix,      0,                    },
    { "suffix",         aliased_suffix,      0,                    },

    { "commands",       show_commands,       0                     },
    { NULL}
};

#define EDIT_SPELL_ALIAS(ch, alias) (alias = (skill_alias_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(saliased_create        );
DECLARE_OLC_FUN(saliased_edit      );
DECLARE_OLC_FUN(saliased_show      );
DECLARE_OLC_FUN(saliased_list      );

DECLARE_OLC_FUN(saliased_name      );
DECLARE_OLC_FUN(saliased_skill     );
DECLARE_OLC_FUN(saliased_prefix    );
DECLARE_OLC_FUN(saliased_suffix        );

olc_cmd_t olc_cmds_skill_alias[] =
{
    { "create",         saliased_create,      2                     },
    { "edit",           saliased_edit,        1                     },
    { "touch",          olced_dummy,          0                     },
    { "show",           saliased_show,        0                     },
    { "list",           saliased_list,        0                     },

    { "name",           saliased_name,        0,                    },
    { "substitute",     saliased_skill,       0,                    },
    { "prefix",         saliased_prefix,      0,                    },
    { "suffix",         saliased_suffix,      0,                    },

    { "commands",       show_commands,        0                     },
    { NULL}
};

OLC_FUN(aliased_create)
{
    alias_t *alias;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_ALIASES)
    {
        char_puts("AliasEd: Insufficient security for creating aliases\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if ((alias = alias_lookup(arg, str_cmp)) != NULL)
    {
        char_printf(ch, "AliasEd: %s: already exists.\n", alias->name);
        return FALSE;
    }

    alias     = alias_new();
    alias->name   = str_dup(arg);

    ch->desc->pEdit = (void *) alias;
    OLCED(ch)   = olced_lookup(ED_ALIAS);
    char_puts("Alias created.\n",ch);
    return FALSE;
}

OLC_FUN(aliased_edit)
{
    alias_t *alias;
    char     arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_ALIASES)
    {
        char_puts("AliasEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if ((alias = alias_lookup(arg, str_cmp)) == NULL)
    {
        char_printf(ch, "AliasEd: %s: No such alias.\n", arg);
        return FALSE;
    }

    ch->desc->pEdit = alias;
    OLCED(ch)   = olced_lookup(ED_ALIAS);
    return FALSE;
}

OLC_FUN(aliased_show)
{
    char     arg[MAX_STRING_LENGTH];
    BUFFER  *output;
    alias_t *alias;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_ALIAS))
            EDIT_ALIAS(ch, alias);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if ((alias = alias_lookup(arg, str_cmp)) == NULL)
        {
            char_printf(ch, "AliasEd: %s: No such alias.\n", arg);
            return FALSE;
        }
    }

    output = buf_new(-1);

                                     buf_printf(output, "Name:          [%s]\n",     alias->name);
    if (alias->command != NULL)      buf_printf(output, "Command:       [%s]\n",     alias->command->name);
    if (!IS_NULLSTR(alias->prefix))  buf_printf(output, "Prefix:        [%s]\n",     alias->prefix);
    if (!IS_NULLSTR(alias->suffix))  buf_printf(output, "Suffix:        [%s]\n",     alias->suffix);

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(aliased_list)
{
    int  i;
    int  col = 0;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    for (i = 0; i < aliases.nused; i++)
    {
        alias_t *alias = (alias_t*) VARR_GET(&aliases, i);

        if (arg[0] && str_prefix(arg, alias->name))
            continue;

        char_printf(ch, "%-12s", alias->name);
        if (++col % 6 == 0)
            char_puts("\n", ch);
    }

    if (col % 6)
        char_puts("\n", ch);

    return FALSE;
}

OLC_FUN(aliased_name)
{
    alias_t *alias;

    EDIT_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->name);
}

OLC_FUN(aliased_command)
{
    alias_t   *alias;
    command_t *command;
    char       arg[MAX_STRING_LENGTH];

    EDIT_ALIAS(ch, alias);
    one_argument(argument, arg, sizeof(arg));

    if ((command = command_lookup(argument, str_cmp)) == NULL)
    {
        char_printf(ch, "AliasEd: %s: non-existant command.\n", argument);
        return FALSE;
    }

     alias->command = command;
     char_puts("Ok.\n", ch);
     return TRUE;
}

OLC_FUN(aliased_prefix)
{
    alias_t *alias;

    EDIT_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->prefix);
}

OLC_FUN(aliased_suffix)
{
    alias_t *alias;

    EDIT_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->suffix);
}

static VALIDATE_FUN(validate_name)
{
    const char *name = (const char*) arg;
    alias_t    *alias, *alias2;

    EDIT_ALIAS(ch, alias);

    if (strpbrk(name, " \t"))
    {
        char_printf(ch, "AliasEd: %s: illegal character in alias name.\n", arg);
        return FALSE;
    }

    if ((alias2 = alias_lookup(name, str_cmp)) != NULL
        &&  alias2 != alias)
    {
        char_printf(ch, "AliasEd: %s: duplicate alias name.\n", arg);
        return FALSE;
    }

    return TRUE;
}

// ------------------------------------------------------------------------------------------------------
// spell aliases
// ------------------------------------------------------------------------------------------------------

OLC_FUN(saliased_create)
{
    skill_alias_t *alias;

    if (ch->pcdata->security < SECURITY_ALIASES)
    {
        char_puts("SAliasEd: Insufficient security for creating aliases\n", ch);
        return FALSE;
    }

    if (argument[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if ((alias = skill_alias_lookup(argument, str_cmp)) != NULL)
    {
        char_printf(ch, "SAliasEd: %s: already exists.\n", alias->name);
        return FALSE;
    }

    alias = skill_alias_new();
    alias->name = str_dup(argument);

    ch->desc->pEdit = (void *) alias;
    OLCED(ch)   = olced_lookup(ED_SKILL_ALIAS);
    char_puts("Skill alias created.\n",ch);
    return FALSE;
}

OLC_FUN(saliased_edit)
{
    skill_alias_t *alias;
    char     arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_ALIASES)
    {
        char_puts("SAliasEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if ((alias = skill_alias_lookup(arg, str_cmp)) == NULL)
    {
        char_printf(ch, "SAliasEd: %s: No such alias.\n", arg);
        return FALSE;
    }

    ch->desc->pEdit = alias;
    OLCED(ch)   = olced_lookup(ED_SKILL_ALIAS);
    return FALSE;
}

OLC_FUN(saliased_show)
{
    char     arg[MAX_STRING_LENGTH];
    BUFFER  *output;
    skill_alias_t *alias;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_SKILL_ALIAS))
            EDIT_SPELL_ALIAS(ch, alias);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if ((alias = skill_alias_lookup(arg, str_cmp)) == NULL)
        {
            char_printf(ch, "SAliasEd: %s: No such alias.\n", arg);
            return FALSE;
        }
    }

    output = buf_new(ch->lang);

                                     buf_printf(output, "Name:          [%s]\n",     alias->name);
    if (alias->skill != NULL)        buf_printf(output, "Skill:         [%s]\n",     alias->skill->name);
    if (!IS_NULLSTR(alias->prefix))  buf_printf(output, "Prefix:        [%s]\n",     alias->prefix);
    if (!IS_NULLSTR(alias->suffix))  buf_printf(output, "Suffix:        [%s]\n",     alias->suffix);

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(saliased_list)
{
    int  i;
    int  col = 0;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    for (i = 0; i < skill_aliases.nused; i++)
    {
        skill_alias_t *alias = (skill_alias_t*) VARR_GET(&skill_aliases, i);

        if (arg[0] && str_prefix(arg, alias->name))
            continue;

        char_printf(ch, "%-12s", alias->name);
        if (++col % 6 == 0)
            char_puts("\n", ch);
    }

    if (col % 6)
        char_puts("\n", ch);

    return FALSE;
}

OLC_FUN(saliased_name)
{
    skill_alias_t *alias;

    EDIT_SPELL_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->name);
}

OLC_FUN(saliased_skill)
{
    skill_alias_t * alias;
    skill_t       * skill;
    char            arg[MAX_STRING_LENGTH];

    EDIT_SPELL_ALIAS(ch, alias);
    one_argument(argument, arg, sizeof(arg));

    if ((skill = skill_lookup(sn_lookup(argument))) == NULL)
    {
        char_printf(ch, "SAliasEd: %s: non-existant command.\n", argument);
        return FALSE;
    }

     alias->skill = skill;
     char_puts("Ok.\n", ch);
     return TRUE;
}

OLC_FUN(saliased_prefix)
{
    skill_alias_t *alias;

    EDIT_SPELL_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->prefix);
}

OLC_FUN(saliased_suffix)
{
    skill_alias_t *alias;

    EDIT_SPELL_ALIAS(ch, alias);
    return olced_str(ch, argument, cmd, &alias->suffix);
}
