/* $Id: olc_clan.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "merc.h"
#include "olc.h"

extern ROOM_INDEX_DATA *room_index_hash [MAX_KEY_HASH];

#define EDIT_CLAN(ch, clan) (clan = (clan_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(claned_create                 );
DECLARE_OLC_FUN(claned_edit                   );
DECLARE_OLC_FUN(claned_touch                  );
DECLARE_OLC_FUN(claned_show                   );
DECLARE_OLC_FUN(claned_list                   );

DECLARE_OLC_FUN(claned_name                   );
DECLARE_OLC_FUN(claned_filename               );
DECLARE_OLC_FUN(claned_recall                 );
DECLARE_OLC_FUN(claned_msg_prays              );
DECLARE_OLC_FUN(claned_msg_vanishes           );
DECLARE_OLC_FUN(claned_flags                  );
DECLARE_OLC_FUN(claned_item                   );
DECLARE_OLC_FUN(claned_mark                   );
DECLARE_OLC_FUN(claned_clanalign              );
DECLARE_OLC_FUN(claned_altar                  );
DECLARE_OLC_FUN(claned_altar_trophy           );
//DECLARE_OLC_FUN(claned_plist                );
DECLARE_OLC_FUN(claned_standard_areaguard     );
DECLARE_OLC_FUN(claned_veteran_areaguard      );
DECLARE_OLC_FUN(claned_elite_areaguard        );
DECLARE_OLC_FUN(claned_minlevel               );
DECLARE_OLC_FUN(claned_aliance_allow          );
DECLARE_OLC_FUN(claned_members                );

DECLARE_OLC_FUN(claned_skill                  );
DECLARE_OLC_FUN(claned_skill_add              );
DECLARE_OLC_FUN(claned_skill_del              );

static DECLARE_VALIDATE_FUN(validate_name     );
static DECLARE_VALIDATE_FUN(validate_altar    );
static DECLARE_VALIDATE_FUN(validate_clanmark );
static DECLARE_VALIDATE_FUN(validate_minlevel );

olc_cmd_t olc_cmds_clan[] =
{
    { "create",     claned_create,              5                       },
    { "edit",       claned_edit,                5                       },
    { "touch",      claned_touch,               5                       },
    { "show",       claned_show,                0                       },
    { "list",       claned_list,                0                       },

    { "name",       claned_name,                5,  validate_name       },
    { "filename",   claned_filename,            5,  validate_filename   },
    { "recall",     claned_recall,              0,  validate_room_vnum  },
    { "msgp",       claned_msg_prays,           0                       },
    { "msgv",       claned_msg_vanishes,        0                       },
    { "flags",      claned_flags,               5,  clan_flags          },
    { "skill",      claned_skill,               0                       },
    { "item",       claned_item,                5,  validate_obj_vnum   },
    { "mark",       claned_mark,                5,  validate_clanmark   },
    { "align",      claned_clanalign,           5                       },
    { "altar",      claned_altar,               5,  validate_altar      },
    { "trophy",     claned_altar_trophy,        5,  validate_altar      },
    { "level",      claned_minlevel,            5,  validate_minlevel   },
    { "allow_aliance", claned_aliance_allow,    5,  aliance_allow_flag  },
//  { "plist",      claned_plist,               5                       },
    { "agstandard", claned_standard_areaguard,  5                       },
    { "agveteran",  claned_veteran_areaguard,   5                       },
    { "agelite",    claned_elite_areaguard,     5                       },
    { "members",    claned_members,             5                       },

    { "commands",   show_commands,              0                       },

    { NULL }
};

OLC_FUN(claned_create)
{
    int cln;
    clan_t *clan;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_CLANS)
    {
        char_puts("ClanEd: Insufficient security for editing clans\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if ((cln = cln_lookup(arg)) >= 0)
    {
        char_printf(ch, "ClanEd: %s: already exists.\n", CLAN(cln)->name);
        return FALSE;
    }

    clan                = clan_new();
    clan->name          = str_dup(arg);
    clan->file_name     = str_printf("clan%02d.clan", clans.nused-1);

    ch->desc->pEdit     = (void *)clan;
    OLCED(ch)           = olced_lookup(ED_CLAN);

    touch_clan(clan);
    char_puts("Clan created.\n",ch);
    return FALSE;
}

OLC_FUN(claned_edit)
{
    int cln;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_CLANS)
    {
        char_puts("ClanEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if (is_number(arg) && atoi(arg) < clans.nused && atoi(arg) >= 0)
        cln = atoi(arg);
    else
    {
        if ((cln = cln_lookup(arg)) < 0)
        {
              char_printf(ch, "ClanEd: %s: No such clan.\n", argument);
              return FALSE;
        }
    }

    ch->desc->pEdit     = CLAN(cln);
    OLCED(ch)           = olced_lookup(ED_CLAN);
    return FALSE;
}

OLC_FUN(claned_touch)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return touch_clan(clan);
}

OLC_FUN(claned_show)
{
    char arg[MAX_STRING_LENGTH];
    int i;
    BUFFER *output;
    clan_t *clan;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_CLAN))
            EDIT_CLAN(ch, clan);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if ((i = cln_lookup(arg)) < 0)
        {
            char_printf(ch, "ClanEd: %s: No such clan.\n", arg);
            return FALSE;
        }
        clan = CLAN(i);
    }

    output = buf_new(-1);
    buf_printf(output,
           "Name:        [%s]\n"
           "Filename:    [%s]\n",
           clan->name,
           clan->file_name);
    if (clan->flags)
        buf_printf(output, "Flags:           [%s]\n", flag_string(clan_flags, clan->flags));
    if (clan->clan_align)
        buf_printf(output, "ClanAlign:       [%d]\n", clan->clan_align);
        buf_printf(output, "Aliance allow:   [%s]\n", flag_string(aliance_allow_flag, clan->allow_alliance));
    if (clan->recall_vnum)
        buf_printf(output, "Recall:          [%d]\n", clan->recall_vnum);
    if (clan->min_clan_level)
        buf_printf(output, "Minimal level:   [%d]\n", clan->min_clan_level);
    if (clan->obj_vnum)
        buf_printf(output, "Item:            [%d]",   clan->obj_vnum);
    else
        buf_printf(output, "Item:            [nothing]");
    if (clan->item_at)
        buf_printf(output, " Item_at:        [%d]\n", clan->item_at);
    else
        buf_printf(output, " Item_at:        [nowhere]\n");
    if (clan->mark_vnum)
        buf_printf(output, "Mark:            [%d]\n", clan->mark_vnum);
    if (clan->altar_vnum)
        buf_printf(output, "Altar:           [%d]\n", clan->altar_vnum);
    if (clan->altar_trophy_vnum)
        buf_printf(output, "AltarTr          [%d]\n", clan->altar_trophy_vnum);
    if (clan->bank_gold)
        buf_printf(output, "Gold:            [%d]\n", clan->bank_gold);
    if (clan->bank_questp)
        buf_printf(output, "Questp:          [%d]\n", clan->bank_questp);
    if (clan->bank_bonus)
        buf_printf(output, "Bonus:           [%d]\n", clan->bank_bonus);
    if (clan->bank_gems)
        buf_printf(output, "Gems:            [%d]\n", clan->bank_gems);
    if (clan->bank_rgems)
        buf_printf(output, "RGems:           [%d]\n", clan->bank_rgems);
    if (clan->bank_bgems)
        buf_printf(output, "BGems:           [%d]\n", clan->bank_bgems);
    if (clan->bank_ggems)
        buf_printf(output, "GGems:           [%d]\n", clan->bank_ggems);
    if (clan->bank_mgems)
        buf_printf(output, "MGems:           [%d]\n", clan->bank_mgems);
    if (clan->bank_wgems)
        buf_printf(output, "WGems:           [%d]\n", clan->bank_wgems);
    if (clan->bank_ygems)
        buf_printf(output, "YGems:           [%d]\n", clan->bank_ygems);
    if (!IS_NULLSTR(clan->msg_prays))
        buf_printf(output, "MsgPrays:        [%s]\n", clan->msg_prays);
    if (!IS_NULLSTR(clan->msg_vanishes))
        buf_printf(output, "MsgVanishes:     [%s]\n", clan->msg_vanishes);

    buf_printf(output, "Areaguards' vnums:\n");
    if (clan->standard_areaguard_vnum)
        buf_printf(output, "Standard:        [%d]\n", clan->standard_areaguard_vnum);
    else
        buf_printf(output, "Standard:        [does not set]\n");
    if (clan->veteran_areaguard_vnum)
        buf_printf(output, "Veteran:         [%d]\n", clan->veteran_areaguard_vnum);
    else
        buf_printf(output, "Veteran:         [does not set]\n");
    if (clan->elite_areaguard_vnum)
        buf_printf(output, "Elite:           [%d]\n", clan->elite_areaguard_vnum);
    else
        buf_printf(output, "Elite:           [does not set]\n");

    for (i = 0; i < clan->skills.nused; i++)
    {
        clan_skill_t *cs = VARR_GET(&clan->skills, i);
        skill_t *sk;

        if (cs->sn <= 0 ||  (sk = skill_lookup(cs->sn)) == NULL)
            continue;
        buf_printf(output, "Skill:       '%s' (level %d, %d%%)\n", sk->name, cs->level, cs->percent);
    }

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(claned_list)
{
    int i;

    for (i = 0; i < clans.nused; i++)
        char_printf(ch, "[%d] %s\n", i, CLAN(i)->name);
    return FALSE;
}

OLC_FUN(claned_name)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_str(ch, argument, cmd, &clan->name);
}

OLC_FUN(claned_filename)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_str(ch, argument, cmd, &clan->file_name);
}

OLC_FUN(claned_recall)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->recall_vnum);
}

OLC_FUN(claned_item)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->obj_vnum);
}

OLC_FUN(claned_mark)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->mark_vnum);
}

OLC_FUN(claned_clanalign)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->clan_align);
}

OLC_FUN(claned_altar)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->altar_vnum);
}

OLC_FUN(claned_minlevel)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->altar_vnum);
}

OLC_FUN(claned_altar_trophy)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->altar_trophy_vnum);
}

OLC_FUN(claned_members)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->max_clan_members);
}

OLC_FUN(claned_msg_prays)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_str(ch, argument, cmd, &clan->msg_prays);
}

OLC_FUN(claned_msg_vanishes)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_str(ch, argument, cmd, &clan->msg_vanishes);
}

OLC_FUN(claned_flags)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_flag64(ch, argument, cmd, &clan->flags);
}

OLC_FUN(claned_aliance_allow)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_flag64(ch, argument, cmd, &clan->allow_alliance);
}

OLC_FUN(claned_skill)
{
    char arg[MAX_STRING_LENGTH];

    argument = one_argument(argument, arg, sizeof(arg));
    if (!str_prefix(arg, "add"))
        return claned_skill_add(ch, argument, cmd);
    else if (!str_prefix(arg, "delete"))
        return claned_skill_del(ch, argument, cmd);

    do_help(ch, "'OLC CLAN SKILL'");
    return FALSE;
}

OLC_FUN(claned_skill_add)
{
    int sn;
    int percent;
    clan_skill_t *clsk;
    char    arg1[MAX_STRING_LENGTH];
    char    arg2[MAX_STRING_LENGTH];
    char    arg3[MAX_STRING_LENGTH];
    clan_t *clan;
    EDIT_CLAN(ch, clan);

    argument = one_argument(argument, arg1, sizeof(arg1));
    argument = one_argument(argument, arg2, sizeof(arg2));

    one_argument(argument, arg3, sizeof(arg3));

    if (arg1[0] == '\0' || arg2[0] == '\0' /*|| arg3[0] == '\0'*/)
    {
        do_help(ch, "'OLC CLAN SKILL'");
        return FALSE;
    }

    if ((sn = sn_lookup(arg1)) <= 0)
    {
        char_printf(ch, "ClanEd: %s: unknown skill.\n", arg1);
        return FALSE;
    }

    if (!IS_SET(SKILL(sn)->flags, SKILL_CLAN))
    {
        char_printf(ch, "ClanEd: %s: not a clan skill.\n", SKILL(sn)->name);
        return FALSE;
    }

    if ((clsk = clan_skill_lookup(clan, sn)))
    {
        char_printf(ch, "ClanEd: %s: already there.\n", SKILL(sn)->name);
        return FALSE;
    }

    percent = atoi(arg3);
    if (percent < 1 || percent > 100)
    {
        char_puts("ClanEd: percent value must be in range 1..100.\n", ch);
        return FALSE;
    }

    clsk = varr_enew(&clan->skills);
    clsk->sn = sn;
    clsk->level = atoi(arg2);
    clsk->percent = percent;
    varr_qsort(&clan->skills, cmpint);

    return TRUE;
}

OLC_FUN(claned_skill_del)
{
    char           arg[MAX_STRING_LENGTH];
    clan_skill_t  *clsk;
    clan_t        *clan;

    EDIT_CLAN(ch, clan);

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CLAN SKILL'");
        return FALSE;
    }

    if ((clsk = skill_vlookup(&clan->skills, arg)) == NULL)
    {
        char_printf(ch, "ClanEd: %s: not found in clan skill list.\n",arg);
        return FALSE;
    }
    varr_del(&clan->skills, (void*)clsk);
    varr_qsort(&clan->skills, cmpint);
    return TRUE;
}

bool touch_clan(clan_t *clan)
{
    SET_BIT(clan->flags, CLAN_CHANGED);
    return FALSE;
}

OLC_FUN(claned_standard_areaguard)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->standard_areaguard_vnum);
}

OLC_FUN(claned_veteran_areaguard)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->veteran_areaguard_vnum);
}

OLC_FUN(claned_elite_areaguard)
{
    clan_t *clan;
    EDIT_CLAN(ch, clan);
    return olced_number(ch, argument, cmd, &clan->elite_areaguard_vnum);
}

static VALIDATE_FUN(validate_name)
{
    int i;
    clan_t *clan;
    EDIT_CLAN(ch, clan);

    for (i = 0; i < clans.nused; i++)
        if (CLAN(i) != clan
        &&  !str_cmp(CLAN(i)->name, arg))
        {
            char_printf(ch, "ClanEd: %s: duplicate clan name.\n",  arg);
            return FALSE;
        }

    return TRUE;
}

static VALIDATE_FUN(validate_altar)
{
    OBJ_INDEX_DATA   *obj;
    ROOM_INDEX_DATA  *pRoom;
    RESET_DATA       *pReset;
    int               iHash;
    int               resets_count;

    int vnum = *(int*) arg;


    if (vnum && get_obj_index(vnum) == NULL)
    {
        char_printf(ch, "ClanEd: %d: no such obj.\n", vnum);
        return FALSE;
    }

    obj = get_obj_index(vnum);

    if (vnum && obj->item_type != ITEM_CONTAINER)
    {
        char_printf(ch, "ClanEd: %d: not container. It's can't be altar.\n", vnum);
        return FALSE;
    }

//                  :)

    resets_count = 0;

    for( iHash = 0; iHash < MAX_KEY_HASH; iHash++ )
    {
          for( pRoom = room_index_hash[iHash]; pRoom; pRoom = pRoom->next )
          {
               for( pReset = pRoom->reset_first; pReset; pReset = pReset->next )
               {
                    switch( pReset->command )
                    {
                         case 'O':
                               if(pReset->arg1 == vnum)
                                     resets_count++;
                         break;
                         case 'E':
                         case 'P':
                         case 'G':
                               if(pReset->arg1 == vnum)
                               {
                                     char_printf(ch, "ClanEd: Altar %d resets settings not in room. Aborted.\n", vnum);
                                     return FALSE;
                               }
                         break;
                    }
               }
          }
     }

    if (resets_count > 0)
    {
       if (resets_count == 1)
       {
             return TRUE;
       }
       else
       {
            char_printf(ch, "ClanEd: Found more then 1 resets for altar %d. Aborted.\n", vnum);
            return FALSE;
       }
    }

    char_printf(ch, "ClanEd: Not found resets for altar %d. Aborted.\n", vnum);
    return FALSE;
}

VALIDATE_FUN(validate_clanmark)
{
    int vnum = *(int*) arg;
    OBJ_INDEX_DATA * clanmark;

    clanmark =  get_obj_index(vnum);

    if (clanmark == NULL)
    {
        char_printf(ch, "ClanEd: %d: no such obj.\n", vnum);
        return FALSE;
    }

    if (!IS_SET(clanmark->wear_flags, WEAR_CLANMARK))
    {
        char_printf(ch, "ClanEd: object %d not clanmark type.\n", vnum);
        return FALSE;
    }

    return TRUE;
}
//TODO:       -     -  ....

VALIDATE_FUN(validate_minlevel)
{
    int level = *(int*) arg;

    if (level < MIN_CLAN_LEVEL)
    {
        char_printf(ch, "ClanEd: global min level for clan set %d.\n", MIN_CLAN_LEVEL);
        return FALSE;
    }

    return TRUE;
}
