/* $Id: olc_class.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "merc.h"
#include "class.h"
#include "olc.h"

#define EDIT_CLASS(ch, cls) (cls = (class_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(classed_create   );
DECLARE_OLC_FUN(classed_edit     );
DECLARE_OLC_FUN(classed_touch    );
DECLARE_OLC_FUN(classed_show     );
DECLARE_OLC_FUN(classed_list     );

DECLARE_OLC_FUN(classed_name             );
DECLARE_OLC_FUN(classed_filename     );
DECLARE_OLC_FUN(classed_flags            );
DECLARE_OLC_FUN(classed_skill        );

DECLARE_OLC_FUN(classed_skill_add    );
DECLARE_OLC_FUN(classed_skill_del    );

DECLARE_OLC_FUN(classed_ethos_restrict);
DECLARE_OLC_FUN(classed_align_restrict);
DECLARE_OLC_FUN(classed_sex_restrict  );
DECLARE_OLC_FUN(classed_weapon        );
DECLARE_OLC_FUN(classed_adept         );
DECLARE_OLC_FUN(classed_thac0         );
DECLARE_OLC_FUN(classed_thac32        );
DECLARE_OLC_FUN(classed_hprate        );
DECLARE_OLC_FUN(classed_manarate      );
DECLARE_OLC_FUN(classed_points        );
DECLARE_OLC_FUN(classed_flags         );
DECLARE_OLC_FUN(classed_dodge         );
DECLARE_OLC_FUN(classed_showskills    );
DECLARE_OLC_FUN(classed_short         );

static DECLARE_VALIDATE_FUN(validate_name  );

bool touch_class(class_t *cls);

olc_cmd_t olc_cmds_class[] =
    {
        { "create",     classed_create,          5                      },
        { "edit",       classed_edit,            5                      },
        { "touch",      classed_touch,           5                      },
        { "show",       classed_show,            0                      },
        { "list",       classed_list,            0                      },

        { "flags",      classed_flags,           0, class_flags         },
        { "ethos",      classed_ethos_restrict,  0, ethos_table         },
        { "align",      classed_align_restrict,  0, ralign_names        },
        { "sex",        classed_sex_restrict,    0, restrict_sex_table  },
        { "weapon",     classed_weapon,          0                      },
        { "adept",      classed_adept,           0                      },
        { "thac0",      classed_thac0,           0                      },
        { "thac32",     classed_thac32,          0                      },
        { "hprate",     classed_hprate,          0                      },
        { "manarate",   classed_manarate,        0                      },
        { "points",     classed_points,          0                      },
        { "dodge",      classed_dodge,           0                      },
        { "short",      classed_short,           0                      },

        { "name",       classed_name,            5, validate_name       },
        { "filename",   classed_filename,        5, validate_filename   },
        { "skill",      classed_skill,           0                      },
        { "showskills", classed_showskills,      0                      },

        { "commands",   show_commands,           0                      },
        { NULL }
    };

OLC_FUN(classed_create)
{
    int cln;
    class_t *cls;
    char arg[MAX_STRING_LENGTH];
    int i;

    if (ch->pcdata->security < SECURITY_CLASSES)
    {
        char_puts("ClassEd: Insufficient security for editing classes\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if ((cln = cn_lookup(arg)) >= 0)
    {
        char_printf(ch, "ClassEd: %s: already exists.\n",
                    CLASS(cln)->name);
        return FALSE;
    }

    cls                = class_new();
    cls->name          = str_dup(arg);
    cls->file_name     = str_printf("%s.class", arg);

    for (i = 0; i < MAX_WEAR; i++)
        cls->wear_loc[i] = 1;

    ch->desc->pEdit     = (void *)cls;
    OLCED(ch)   = olced_lookup(ED_CLASS);
    touch_class(cls);
    char_puts("Class created.\n",ch);
    return FALSE;
}

OLC_FUN(classed_edit)
{
    int cn;
    char arg[MAX_STRING_LENGTH];
    if (ch->pcdata->security < SECURITY_CLASSES)
    {
        char_puts("ClassEd: Insufficient security.\n", ch);
        return FALSE;
    }
    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }
    if ((cn = cn_lookup(arg)) < 0)
    {
        char_printf(ch, "ClassEd: %s: No such class.\n", argument);
        return FALSE;
    }
    ch->desc->pEdit     = CLASS(cn);
    OLCED(ch)   = olced_lookup(ED_CLASS);
    return FALSE;
}

OLC_FUN(classed_touch)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return touch_class(cls);
}

OLC_FUN(classed_show)
{
    char arg[MAX_STRING_LENGTH];
    int i,j;
    BUFFER *output;
    class_t *cls;

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_CLASS))
            EDIT_CLASS(ch, cls);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if ((i = cn_lookup(arg)) < 0)
        {
            char_printf(ch, "ClassEd: %s: No such class.\n", argument);
            return FALSE;
        }
        cls = CLASS(i);
    }
    output = buf_new(ch->lang);
    buf_printf(output,
               "Name:           [%s]\n"
               "Filename:       [%s]\n"
               "Display as:     [%s]\n"
               "School weapon:  [%d]\n"
               "THAC0 (level 0) [%d] THAC0 (level 32) [%d]\n"
               "HP rate:        [%d%%] Mana rate:      [%d%%]\n"
               "Points:         [%d]\n"
               "Adept:          [%d%%]\n"
               "Dodge:          [%d%]\n",

               cls->name,
               cls->file_name,
               cls->who_name,
               cls->weapon,
               cls->thac0_00, cls->thac0_32,
               cls->hp_rate, cls->mana_rate,
               cls->points,
               cls->skill_adept,
               cls->dodge);

    buf_printf(output,"Stats mod:      [");
    for (i = 0; i < MAX_STATS; i++)
    {
        buf_printf(output,"%s:%3d ", stat_names[i+1].name, cls->stats[i]);
    }
    buf_printf(output,"]\n");

    j = 0;

    buf_printf(output,"Wear loc:       [");
    for (i = 0; i < MAX_WEAR; i++)
    {
        /*//if (j == 0 || j == 3 || j=6 || j=9 || j=12 || j=15)
            buf_printf(output,"\n                  %-10s ", wear_flags[i+1].name);
                else
                        buf_printf(output," %-10s ", wear_flags[i+1].name);
                j++;*/
    }
    buf_printf(output,"]\n");

    buf_printf(output, "Flags:          [%s]\n",   flag_string(class_flags, cls->flags));

    buf_printf(output, "Align restrict: [%s]\n",   flag_string(ralign_names, cls->restrict_align));
    buf_printf(output, "Ethos restrict: [%s]\n",   flag_string(ethos_table, cls->restrict_ethos));
    buf_printf(output, "Sex restrict:   [%s]\n\r", flag_string(restrict_sex_table, cls->restrict_sex));

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}


OLC_FUN(classed_showskills)
{
    bool         sorted;
    int          i;
    BUFFER      *output;
    class_t     *cls;
    char         arg[MAX_STRING_LENGTH];
    int          col;

    one_argument(argument, arg, sizeof(arg));

    col = 0;

    if (!str_cmp(arg, "sorted"))
    {
        sorted = TRUE;
        argument = one_argument(argument, arg, sizeof(arg));
    }
    else
        sorted = FALSE;

    if (argument[0] == '\0')
    {
        if (IS_EDIT(ch, ED_CLASS))
            EDIT_CLASS(ch, cls);
        else
        {
            do_help(ch, "'OLC SHOWSKILLS'");
            return FALSE;
        }
    }
    else
    {
        if ((i = cn_lookup(argument)) < 0)
        {
            char_printf(ch, "ClassEd: %s: No such class.\n", argument);
            return FALSE;
        }
        cls = CLASS(i);
    }
    output = buf_new(-1);

    if (!sorted)
    {
        for (i = 0; i < cls->skills.nused; i++)
        {
            class_skill_t *cs = VARR_GET(&cls->skills, i);
            skill_t *skill;

            if (cs->sn <= 0  ||  (skill = skill_lookup(cs->sn)) == NULL)
                continue;
            buf_printf(output, "{D[{x%-20s{D]{x lvl: %-3d, %3d%% rt: %1d ",  skill->name, cs->level, cs->max_percent, cs->rating);

            if (++col % 3 == 0)
                       buf_add(output, "\n");
        }
    }
    else
    {
    //
    //TODO:   
    /*
        int level;
        char  skills_at_level[MAX_LEVEL][MAX_STRING_LENGTH];

        for (i=0; i < MAX_LEVEL; i++)
            skills_at_level[i][0] = '\0';

        for (i = 0; i < cls->skills.nused; i++)
        {
            class_skill_t *cs = VARR_GET(&cls->skills, i);
            skill_t *skill;

            if (cs->sn <= 0  ||  (skill = skill_lookup(cs->sn)) == NULL)
                continue;
            if (cs->level < 1 || cs->level > MAX_LEVEL)
                continue;
            snprintf(skills_at_level[cs->level -1], MAX_STRING_LENGTH, "%s  (%d%%)\n", skill->name, cs->max_percent);
        }

        for (level = 0; level < MAX_LEVEL; level++)
        {
            if (skills_at_level[i][0] == '\0')
                continue;
            buf_printf(output, "Level %d:\n", level+1);
            buf_printf(output, "%s", skills_at_level[i]);
        }
        */

          buf_printf(output, "Disabled now.\n");
    }

    buf_add(output, "\n");
    page_to_char(buf_string(output), ch);
    buf_free(output);

    return TRUE;
}

OLC_FUN(classed_list)
{
    int i;
    for (i = 0; i < classes.nused; i++)
        char_printf(ch, "[%d] %s\n", i, CLASS(i)->name);
    return FALSE;
}

OLC_FUN(classed_name)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_str(ch, argument, cmd, &cls->name);
}

OLC_FUN(classed_short)
{
    class_t * cls;
    EDIT_CLASS (ch, cls);
    strnzcpy (cls->who_name, sizeof(cls->who_name), argument);
    return TRUE;
}

OLC_FUN(classed_filename)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_str(ch, argument, cmd, &cls->file_name);
}

OLC_FUN(classed_flags)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_flag64(ch, argument, cmd, &cls->flags);
}

OLC_FUN(classed_skill)
{
    char arg[MAX_STRING_LENGTH];
    argument = one_argument(argument, arg, sizeof(arg));
    if (!str_prefix(arg, "add"))
        return classed_skill_add(ch, argument, cmd);
    else if (!str_prefix(arg, "delete"))
        return classed_skill_del(ch, argument, cmd);
    do_help(ch, "'OLC CLASS SKILL'");
    return FALSE;
}

OLC_FUN(classed_skill_add)
{
    int            sn;
    class_skill_t *cls_skill;
    char           arg1[MAX_STRING_LENGTH];
    char           arg2[MAX_STRING_LENGTH];
    int            level;
    int            rating;
    int            max_percent;
    class_t       *cls;


    EDIT_CLASS(ch, cls);

    rating      = 1;
    max_percent = 100;

    argument = one_argument(argument, arg1, sizeof(arg1));
    argument = one_argument(argument, arg2, sizeof(arg2));

    if (arg1[0] == '\0' || arg2[0] == '\0')
    {
        do_help(ch, "'OLC CLASS SKILL'");
        return FALSE;
    }
    if ((sn = sn_lookup(arg1)) <= 0)
    {
        char_printf(ch, "ClassEd: %s: unknown skill.\n", arg1);
        return FALSE;
    }

    if ((cls_skill = class_skill_lookup(cls, sn)))
    {
        char_printf(ch, "Cedit: %s: already there.\n", SKILL(sn)->name);
        return FALSE;
    }

    level = atoi(arg2);

    while( argument[0] != '\0' )
    {
          argument = one_argument(argument, arg2, sizeof(arg2));
          if (!str_cmp(arg2, "rating"))
          {
                argument = one_argument(argument, arg2, sizeof(arg2));
                if (!is_number(arg2) || atoi(arg2) < 0)
                {
                      char_printf(ch, "Cedit: argument for rating must be numeric and greater then 0.\n");
                      return FALSE;
                }
                else
                      rating = atoi(arg2);
          }
          else
          if (!str_cmp(arg2, "max"))
          {
                argument = one_argument(argument, arg2, sizeof(arg2));
                if (!is_number(arg2) || atoi(arg2) < 1 || atoi(arg2) > 100)
                {
                      char_printf(ch, "Cedit: argument for rating must be numeric and betwen 1 and 100.\n");
                      return FALSE;
                }
                else
                      max_percent = atoi(arg2);
          }
          else
          {
                char_printf(ch, "Cedit: unknown commands %s.\n", arg2);
                return FALSE;
          }
    }

    cls_skill               = varr_enew(&cls->skills);
    cls_skill->sn           = sn;
    cls_skill->level        = level;
    cls_skill->rating       = rating;
    cls_skill->max_percent  = max_percent;
    
    varr_qsort(&cls->skills, cmpint);
    
    return TRUE;
}

OLC_FUN(classed_skill_del)
{
    char             arg[MAX_STRING_LENGTH];
    class_skill_t   *cls_skill;
    class_t         *cls;

    EDIT_CLASS(ch, cls);

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CLASS SKILL'");
        return FALSE;
    }

    if ((cls_skill = skill_vlookup(&cls->skills, arg)) == NULL)
    {
        char_printf(ch, "ClassEd: %s: not found in class skill list.\n", arg);
        return FALSE;
    }


    varr_del(&cls->skills, (void*)cls_skill);
    varr_qsort(&cls->skills, cmpint);

    return TRUE;
}

bool touch_class(class_t *cls)
{
    SET_BIT(cls->flags, CLASS_CHANGED);
    return FALSE;
}

static VALIDATE_FUN(validate_name)
{
    int i;
    class_t *cls;
    EDIT_CLASS(ch, cls);

    for (i = 0; i < classes.nused; i++)
        if (CLASS(i) != cls  &&  !str_cmp(CLASS(i)->name, arg))
        {
            char_printf(ch, "ClassEd: %s: duplicate class name.\n", arg);
            return FALSE;
        }
    return TRUE;
}

OLC_FUN(classed_ethos_restrict)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_flag64(ch, argument, cmd, &cls->restrict_ethos);
}

OLC_FUN(classed_sex_restrict)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_flag64(ch, argument, cmd, &cls->restrict_sex);
}

OLC_FUN(classed_align_restrict)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_flag64(ch, argument, cmd, &cls->restrict_align);
}

OLC_FUN(classed_weapon)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->weapon);
}

OLC_FUN(classed_adept)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->skill_adept);
}

OLC_FUN(classed_thac0)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->thac0_00);
}

OLC_FUN(classed_thac32)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->thac0_32);
}

OLC_FUN(classed_dodge)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);

    return olced_number(ch, argument, cmd, &cls->dodge);
}

OLC_FUN(classed_hprate)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->hp_rate);
}

OLC_FUN(classed_manarate)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->mana_rate);
}

OLC_FUN(classed_points)
{
    class_t *cls;
    EDIT_CLASS(ch, cls);
    return olced_number(ch, argument, cmd, &cls->points);
}
