/*  $Id: olc_cmd.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <string.h>

#include "merc.h"
#include "olc.h"
#include "db/cmd.h"

#define EDIT_COMMAND(ch, command)   (command = (command_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(commanded_create        );
DECLARE_OLC_FUN(commanded_edit      );
DECLARE_OLC_FUN(commanded_show      );
DECLARE_OLC_FUN(commanded_list      );

DECLARE_OLC_FUN(commanded_name      );
DECLARE_OLC_FUN(commanded_min_pos   );
DECLARE_OLC_FUN(commanded_level           );
DECLARE_OLC_FUN(commanded_log       );
DECLARE_OLC_FUN(commanded_dofun     );
DECLARE_OLC_FUN(commanded_flags     );

static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_command[] =
{
    { "create",         commanded_create,       5                     },
    { "edit",           commanded_edit,         2                     },
    { "touch",          olced_dummy,            0                     },
    { "show",           commanded_show,         0                     },
    { "list",           commanded_list,         0                     },

    { "name",           commanded_name,         0,      validate_name },
    { "min_pos",        commanded_min_pos,      0,      position_table},
    { "level",          commanded_level,        0,                    },
    { "log",            commanded_log,          0,      log_flags     },
    { "flags",          commanded_flags,        0,      command_flags },
    { "function",       commanded_dofun,        0,      dofun_table   },

    { "commands",       show_commands,          0                     },
    { NULL}
};

OLC_FUN(commanded_create)
{
    command_t *command;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_COMMANDS)
    {
        char_puts("CommandEd: Insufficient security for creating commands\n",  ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if ((command = command_lookup(arg, str_cmp)) != NULL)
    {
        char_printf(ch, "CommandEd: %s: already exists.\n", command->name);
        return FALSE;
    }

    command         = command_new();
    command->name   = str_dup(arg);

    ch->desc->pEdit = (void *) command;

    OLCED(ch)   = olced_lookup(ED_COMMAND);
    char_puts("Command created.\n",ch);
    return FALSE;
}

OLC_FUN(commanded_edit)
{
    command_t *command;
    char       arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_COMMANDS)
    {
        char_puts("CommandEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if ((command = command_lookup(arg, str_cmp)) == NULL)
    {
        char_printf(ch, "CommandEd: %s: No such command.\n", arg);
        return FALSE;
    }

    ch->desc->pEdit = command;
    OLCED(ch)   = olced_lookup(ED_COMMAND);
    return FALSE;
}

OLC_FUN(commanded_show)
{
    char arg[MAX_STRING_LENGTH];
    BUFFER *output;
    command_t *command;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_COMMAND))
            EDIT_COMMAND(ch, command);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if ((command = command_lookup(arg, str_cmp)) == NULL)
        {
            char_printf(ch, "CommandEd: %s: No such command.\n", arg);
            return FALSE;
        }
    }

    output = buf_new(-1);

                                buf_printf(output, "Name:          [%s]\n",      command->name);
    if (command->position)      buf_printf(output, "MinPos:        [%s]\n",      flag_string(position_table, command->position));
    if (command->do_fun)        buf_printf(output, "DoFun:         [%s]\n",      namedp_name(dofun_table, command->do_fun));
    if (command->level)         buf_printf(output, "Level:         [%d]\n",      command->level);
    if (command->log)           buf_printf(output, "Log:           [%s]\n",      flag_string(log_flags, command->log));
    if (command->flags)         buf_printf(output, "Flags:         [%s]\n",      flag_string(command_flags, command->flags));

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(commanded_list)
{
    int i;
    int col = 0;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    for (i = 0; i < commands.nused; i++)
    {
        command_t *command = (command_t*) VARR_GET(&commands, i);

        if (arg[0] && str_prefix(arg, command->name))
            continue;

        char_printf(ch, "%-12s", command->name);
        if (++col % 6 == 0)
            char_puts("\n", ch);
    }

    if (col % 6)
        char_puts("\n", ch);

    return FALSE;
}

OLC_FUN(commanded_name)
{
    command_t *command;
    EDIT_COMMAND(ch, command);
    return olced_str(ch, argument, cmd, &command->name);
}

OLC_FUN(commanded_min_pos)
{
    command_t *command;
    EDIT_COMMAND(ch, command);
    return olced_flag32(ch, argument, cmd, &command->position);
}

OLC_FUN(commanded_log)
{
    command_t *command;
    EDIT_COMMAND(ch, command);
    return olced_flag32(ch, argument, cmd, &command->log);
}

OLC_FUN(commanded_flags)
{
    command_t *command;
    EDIT_COMMAND(ch, command);
    return olced_flag32(ch, argument, cmd, &command->flags);
}

OLC_FUN(commanded_level)
{
    command_t *command;
    EDIT_COMMAND(ch, command);
    return olced_number(ch, argument, cmd, &command->level);
}

OLC_FUN(commanded_dofun)
{
      bool res;
      command_t *command;
      void *p;

     EDIT_COMMAND(ch, command);

     p = (void *) command->do_fun;
     res = olced_named_pointer(ch, argument, cmd, &p);
     command->do_fun =  (DO_FUN *) p;
     return res;
}

static VALIDATE_FUN(validate_name)
{
    const char *name = (const char*) arg;
    command_t *command, *command2;
    EDIT_COMMAND(ch, command);

    if (strpbrk(name, " \t"))
    {
        char_printf(ch, "CommandEd: %s: illegal character in command name.\n",  arg);
        return FALSE;
    }

    if ((command2 = command_lookup(name, str_cmp)) != NULL   &&  command2 != command)
    {
        char_printf(ch, "CommandEd: %s: duplicate command name.\n", arg);
        return FALSE;
    }

    return TRUE;
}
