/* $Id: olc_immortal.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"

#define EDIT_IMMORTAL(ch, immortal)   (immortal = (immortal_t*) ch->desc->pEdit)
extern varr     immortals;
extern void save_immortal(CHAR_DATA *ch);

DECLARE_OLC_FUN(immortaled_create           );
DECLARE_OLC_FUN(immortaled_edit             );
DECLARE_OLC_FUN(immortaled_touch            );
DECLARE_OLC_FUN(immortaled_show             );
DECLARE_OLC_FUN(immortaled_list             );

DECLARE_OLC_FUN(immortaled_name             );
DECLARE_OLC_FUN(immortaled_realname         );
DECLARE_OLC_FUN(immortaled_level            );
DECLARE_OLC_FUN(immortaled_comment          );
DECLARE_OLC_FUN(immortaled_phone            );
DECLARE_OLC_FUN(immortaled_phone_sec_level  );
DECLARE_OLC_FUN(immortaled_accesstime       );
DECLARE_OLC_FUN(immortaled_email            );
DECLARE_OLC_FUN(immortaled_email_sec_level  );
DECLARE_OLC_FUN(immortaled_icq              );
DECLARE_OLC_FUN(immortaled_icq_sec_level    );
DECLARE_OLC_FUN(immortaled_birthday         );

static DECLARE_VALIDATE_FUN(validate_security);
static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_immortal[] =
{
    { "create",        immortaled_create,            5                            },
    { "edit",          immortaled_edit,              5                            },
    { "touch",         olced_dummy,                  5                            },
    { "show",          immortaled_show,              0                            },
    { "list",          immortaled_list,              0                            },

    { "name",          immortaled_name,              5,  validate_name            },
    { "realname",      immortaled_realname,          5                            },
    { "level",         immortaled_level,             5,  validate_security        },
    { "comment",       immortaled_comment,           5                            },
    { "phone",         immortaled_phone,             5,  validate_security        },
    { "access",        immortaled_accesstime,        5                            },
    { "phonesec",      immortaled_phone_sec_level,   5,  validate_security        },
    { "email",         immortaled_email,             5                            },
    { "emailsec",      immortaled_email_sec_level,   5,  validate_security        },
    { "icq",           immortaled_icq,               5                            },
    { "icqsec",        immortaled_icq_sec_level,     5,  validate_security        },
    { "birthday",      immortaled_birthday,          5                            },

    { "commands",      show_commands,                0                            },
    { NULL}
};

OLC_FUN(immortaled_create)
{
    int immrn;
    immortal_t *immortal;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_IMMORTAL"))
    {
        char_puts("ImmEd: Insufficient security for editing immortal.\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC IMMORTAL'");
        return FALSE;
    }

    if ((immrn = immnumber_lookup(arg)) >= 0)
    {
        char_printf(ch, "ImmEd: %s: already exists.\n", IMM(immrn)->name);
        return FALSE;
    }

    immortal                = immortal_new();
    immortal->name          = str_dup(arg);


    ch->desc->pEdit     = (void *)immortal;
    OLCED(ch)   = olced_lookup(ED_IMMORTAL);

    char_puts("Immortal created.\n",ch);
    return FALSE;
}

OLC_FUN(immortaled_edit)
{
    int immrn;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_IMMORTAL"))
    {
        char_puts("ImmEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC IMMORTAL'");
        return FALSE;
    }

    immrn = -1;

    if (is_number(arg))
    {
        if (!str_cmp(immortal_name(atoi(arg)),"None"))
        {
            char_printf(ch, "ImmEd: %s: No such immortal.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit     = IMM(atoi(arg));
    }
    else
    {
        if ((immrn = immnumber_lookup(arg)) < 0)
        {
            char_printf(ch, "ImmEd: %s: No such immortal.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit     = IMM(immrn);
    }

    OLCED(ch)   = olced_lookup(ED_IMMORTAL);
    return FALSE;
}

OLC_FUN(immortaled_list)
{
    int i;

    for (i = 0; i < immortals.nused; i++)
        char_printf(ch, "[%d] %s\n", i, IMM(i)->name);
    return FALSE;
}

OLC_FUN(immortaled_name)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->name);
}

OLC_FUN(immortaled_birthday)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->birthday);
}

OLC_FUN(immortaled_realname)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->realname);
}

OLC_FUN(immortaled_comment)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->comment);
}

OLC_FUN(immortaled_phone)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->phone);
}

OLC_FUN(immortaled_accesstime)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->phone_time_access);
}

OLC_FUN(immortaled_email)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->email);
}

OLC_FUN(immortaled_icq)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_str(ch, argument, cmd, &immortal->icq);
}

OLC_FUN(immortaled_level)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_number(ch, argument, cmd, &immortal->level);
}

OLC_FUN(immortaled_phone_sec_level)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_number(ch, argument, cmd, &immortal->phone_sec_level);
}

OLC_FUN(immortaled_email_sec_level)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_number(ch, argument, cmd, &immortal->email_sec_level);
}

OLC_FUN(immortaled_icq_sec_level)
{
    immortal_t *immortal;

    EDIT_IMMORTAL(ch, immortal);
    return olced_number(ch, argument, cmd, &immortal->icq_sec_level);
}

OLC_FUN(immortaled_show)
{
    char       arg[MAX_STRING_LENGTH];
    int        i;
    BUFFER     *output;
    immortal_t *immortal;

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_IMMORTAL))
            EDIT_IMMORTAL(ch, immortal);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (!str_cmp(immortal_name(atoi(arg)),"None"))
            {
                char_printf(ch, "ImmEd: %s: No such immortal.\n", arg);
                return FALSE;
            }
            else
                immortal = IMM(atoi(arg));
        }
        else
        {
            i = immnumber_lookup(argument);
            if ( i < 0 )
            {
                 char_printf(ch, "ImmEd: Must specify immortal name.\n", arg);
                 return FALSE;
            }

        immortal = IMM(i);
        }
    }

    output = buf_new(-1);

    buf_printf(output, "Name:               [%s]\n", immortal->name);
    buf_printf(output, "RealName:           [%s]\n", immortal->realname);
    buf_printf(output, "Level:              [%d]\n", immortal->level);
    buf_printf(output, "BirthDay:           [%s]\n", immortal->birthday);

    if (ch->level >= immortal->phone_sec_level)
    {
    buf_printf(output, "Phone:              [%s]\n", immortal->phone);
    buf_printf(output, "AccessTime:         [%s]\n", immortal->phone_time_access);
    buf_printf(output, "PhoneSecLevel:      [%d]\n", immortal->phone_sec_level);
    }

    if (ch->level >= immortal->email_sec_level)
    {
    buf_printf(output, "Emai:               [%s]\n", immortal->email);
    buf_printf(output, "EmaiSecLevel:       [%d]\n", immortal->email_sec_level);
    }

    if (ch->level >= immortal->icq_sec_level)
    {
    buf_printf(output, "ICQ:                [%s]\n", immortal->icq);
    buf_printf(output, "ICQSecLevel:        [%d]\n", immortal->icq_sec_level);
    }

    buf_printf(output, "Comment:            [%s]\n", immortal->comment);

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

static VALIDATE_FUN(validate_security)
{
      immortal_t *immortal;

  EDIT_IMMORTAL(ch, immortal);

  if (ch->level < immortal->level)
  {
      char_printf(ch, "ImmEd: You level less than edited immortal level.\n");
     return FALSE;
  }
      return TRUE;
}

static VALIDATE_FUN(validate_name)
{
  int         immrn;
  immortal_t *immortal;
  const char *name = (const char*) arg;

  EDIT_IMMORTAL(ch, immortal);

  if (strpbrk(name, " \t"))
  {
      char_printf(ch, "ImmEd: %s: illegal character in Immortal name.\n", arg);
      return FALSE;
  }

  if ((immrn = immnumber_lookup(arg)) >= 0)
  {
      char_printf(ch, "ImmEd: %s: already exists.\n", IMM(immrn)->name);
      return FALSE;
  }

      return TRUE;
}

bool delete_immortal(CHAR_DATA *ch, const char *argument)
{
      immortal_t *immortal;
      int         immnumber;
      char        name[MAX_INPUT_LENGTH];

      if (IS_NULLSTR(argument))
      {
        do_help(ch, "'OLC IMMORTAL'");
        return FALSE;
      }

      one_argument(argument, name, sizeof(name));

      immnumber = immnumber_lookup(name);

      if (immnumber < 0)
      {
            char_printf(ch, "Immortal %s not in database", name);
            return FALSE;
      }

        immortal = IMM(immnumber);
        if (immortal != NULL)
        {
              varr_del(&immortals, (void*)immortal);
              char_printf(ch, "ex-immortal %s deleted from database", name);
              save_immortal(ch);
              return TRUE;
        }
        else
              log_printf("** BUG ** olc_immortal:delete_immortal: immortall = NULL");

       return FALSE;
}
