 /* $Id: olc_liquid.c,v 1.666 2004/09/20 10:50:30 shrike Exp $  */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/ 
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"
#include "db/liquid.h"

#define EDIT_LIQUID(ch, liquid)        (liquid = (liquid_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(liquided_create       );
DECLARE_OLC_FUN(liquided_edit         );
DECLARE_OLC_FUN(liquided_touch        );
DECLARE_OLC_FUN(liquided_show         );
DECLARE_OLC_FUN(liquided_list         );

DECLARE_OLC_FUN(liquided_name         );
DECLARE_OLC_FUN(liquided_color        );
DECLARE_OLC_FUN(liquided_flag         );
DECLARE_OLC_FUN(liquided_proof        );
DECLARE_OLC_FUN(liquided_full         );
DECLARE_OLC_FUN(liquided_thirst       );
DECLARE_OLC_FUN(liquided_food         );
DECLARE_OLC_FUN(liquided_ssize        );


olc_cmd_t olc_cmds_liquid[] =
{
    { "create",        liquided_create,       5               },
    { "edit",          liquided_edit,         5               },
    { "touch",         olced_dummy,           5               },
    { "show",          liquided_show,         0               },
    { "list",          liquided_list,         0               },

    { "name",          liquided_name,         5               },
    { "color",         liquided_color,        5               },
    { "flag",          liquided_flag,         5, liquid_flags },
    { "proof",         liquided_proof,        5               },
    { "full",          liquided_full,         5               },
    { "thirst",        liquided_thirst,       5               },
    { "food",          liquided_food,         5               },
    { "ssize",         liquided_ssize,        5               },
    
    { "commands",      show_commands,         0               },
    { NULL}
};

OLC_FUN(liquided_create)
{
    int       liq;
    liquid_t *liquid;
    char      arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_LIQUID"))
    {
        char_puts("LiquidEd: Insufficient security for editing liquids.\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }
    
    if (is_number(arg))
    {
        char_printf(ch, "LiquidEd: Argument must be name.\n");
        return FALSE;
    }
    
    if ((liq = liquid_lookup(arg) >= 0))
    {
        char_printf(ch, "LiquidEd: %s: already exists.\n", LIQUID(liq)->name);
        return FALSE;
    }

    liquid                  = liquid_new();
    liquid->name            = str_dup(arg);
       

    ch->desc->pEdit     = (void *)liquid;
    OLCED(ch)   = olced_lookup(ED_LIQUID);                              
    
    char_puts("LiquidEd: liquid created.\n",ch);
    return FALSE;
}

OLC_FUN(liquided_edit)
{
    int  liq;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_LIQUID"))
    {
        char_puts("LiquidEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    liq = -1;

    if (is_number(arg))
    {
        if (atoi(arg) < 0 || atoi(arg) >= liquids.nused)
        {
            char_printf(ch, "LiquidEd: %s: No such liquid.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit     = LIQUID(atoi(arg));
    }
    else
    {
        if ((liq = liquid_lookup(arg)) < 0)
        {
            char_printf(ch, "LiquidEd: %s: No such liquid.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit     = LIQUID(liq);
    }

    OLCED(ch)   = olced_lookup(ED_LIQUID);
    return FALSE;
}

OLC_FUN(liquided_list)
{
    int i;

    for (i = 0; i < liquids.nused; i++)
        char_printf(ch, "[%3d] %-15s\n", i, LIQUID(i)->name);
    return FALSE;
}

OLC_FUN(liquided_name)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_str(ch, argument, cmd, &liquid->name);
}

OLC_FUN(liquided_color)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_str(ch, argument, cmd, &liquid->color);
}

// TODO: color need check for valid????

OLC_FUN(liquided_proof)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_number(ch, argument, cmd, &liquid->proof);
}

OLC_FUN(liquided_full)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_number(ch, argument, cmd, &liquid->full);
}

OLC_FUN(liquided_food)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_number(ch, argument, cmd, &liquid->food);
}

OLC_FUN(liquided_thirst)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_number(ch, argument, cmd, &liquid->thirst);
}

OLC_FUN(liquided_ssize)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_number(ch, argument, cmd, &liquid->ssize);
}

OLC_FUN(liquided_flag)
{
    liquid_t *liquid;
    
    EDIT_LIQUID(ch, liquid);
    return olced_flag64(ch, argument, cmd, &liquid->flag);
}

OLC_FUN(liquided_show)
{
    char        arg[MAX_STRING_LENGTH];
    BUFFER     *output;
    liquid_t   *liquid;
    
    one_argument(argument, arg, sizeof(arg));
    
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_LIQUID))
            EDIT_LIQUID(ch, liquid);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (atoi(arg) < 0 || atoi(arg) > liquids.nused)
            {
                char_printf(ch, "LiquidEd: %s: No such liquid.\n", arg);
                return FALSE;
            }
            else
                liquid = LIQUID(atoi(arg));
        }
        else
        {
            if (liquid_lookup(arg) < 0)
            {
               char_printf(ch, "LiquidEd: Not such liquid %s.\n", arg);
               return FALSE;
            }
            else
               liquid = LIQUID(liquid_lookup(arg));
        }
    }

    output = buf_new(-1);

    buf_printf(output, "Name:        [%s]\n", liquid->name);
    buf_printf(output, "Color:       [%s]\n", liquid->color);
    buf_printf(output, "Proof:       [%d]\n", liquid->proof);
    buf_printf(output, "Full:        [%d]\n", liquid->full);
    buf_printf(output, "Thirst:      [%d]\n", liquid->thirst);
    buf_printf(output, "Food:        [%d]\n", liquid->food);
    buf_printf(output, "Ssize:       [%d]\n", liquid->ssize);
    
    buf_printf(output, "Flags:       [%s]\n\n", flag_string(liquid_flags, liquid->flag));
    
    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

