 /* $Id: olc_material.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"
#include "material.h"

#define EDIT_MATERIAL(ch, material)        (material = (material_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(materialed_create            );
DECLARE_OLC_FUN(materialed_edit              );
DECLARE_OLC_FUN(materialed_touch             );
DECLARE_OLC_FUN(materialed_show              );
DECLARE_OLC_FUN(materialed_list              );

DECLARE_OLC_FUN(materialed_ename             );
DECLARE_OLC_FUN(materialed_rname             );
DECLARE_OLC_FUN(materialed_flag              );
DECLARE_OLC_FUN(materialed_rigidity          );
DECLARE_OLC_FUN(materialed_fragility         );
DECLARE_OLC_FUN(materialed_density           );

DECLARE_OLC_FUN(materialed_race              );
DECLARE_OLC_FUN(materialed_race_add          );
DECLARE_OLC_FUN(materialed_race_del          );

DECLARE_OLC_FUN(materialed_repairman         );
DECLARE_OLC_FUN(materialed_repairman_add     );
DECLARE_OLC_FUN(materialed_repairman_del     );

olc_cmd_t olc_cmds_material[] =
{
    { "create",        materialed_create,       5},
    { "edit",          materialed_edit,         5},
    { "touch",         olced_dummy,             5},
    { "show",          materialed_show,         0},
    { "list",          materialed_list,         0},

    { "ename",         materialed_ename,        5},
    { "rname",         materialed_rname,        5},
    { "flag",          materialed_flag,         5, material_flags},
    { "rigidity",      materialed_rigidity,     5},
    { "fragility",     materialed_fragility,    5},
    { "density",       materialed_density,      5},

    { "race",          materialed_race,         5},
    { "repairman",     materialed_repairman,    5},

    { "commands",      show_commands,           0},
    { NULL}
};

OLC_FUN(materialed_create)
{
    int mn;
    material_t *material;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_MATERIAL"))
    {
        char_puts("MaterialEd: Insufficient security for editing materials.\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

    if (is_number(arg))
    {
        char_printf(ch, "MaterialEd: Argument must be name.\n");
        return FALSE;
    }

    if ((mn = material_n_lookup(arg) >= 0))
    {
        char_printf(ch, "MaterialEd: %s: already exists.\n", MATERIAL(mn)->ename);
        return FALSE;
    }

    material                  = material_new();
    material->ename           = str_dup(arg);


    ch->desc->pEdit     = (void *)material;
    OLCED(ch)   = olced_lookup(ED_MATERIAL);

    char_puts("Material created.\n",ch);
    return FALSE;
}

OLC_FUN(materialed_edit)
{
    int mn;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_MATERIAL"))
    {
        char_puts("MaterialEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    mn = -1;

    if (is_number(arg))
    {
        if (atoi(arg) < 0 || atoi(arg) > materials.nused)
        {
            char_printf(ch, "MaterialEd: %s: No such material.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit     = MATERIAL(atoi(arg));
    }
    else
    {
        if ((mn = material_n_lookup(arg)) < 0)
        {
            char_printf(ch, "MaterialEd: %s: No such material.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit     = MATERIAL(mn);
    }

    OLCED(ch)   = olced_lookup(ED_MATERIAL);
    return FALSE;
}

OLC_FUN(materialed_list)
{
    int i;

    for (i = 0; i < materials.nused; i++)
        char_printf(ch, "[%3d] %-15s %-15s\n", i, MATERIAL(i)->ename, MATERIAL(i)->rname);
    return FALSE;
}

OLC_FUN(materialed_ename)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_str(ch, argument, cmd, &material->ename);
}

OLC_FUN(materialed_rname)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_str(ch, argument, cmd, &material->rname);
}

OLC_FUN(materialed_rigidity)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_number(ch, argument, cmd, &material->rigidity);
}

OLC_FUN(materialed_fragility)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_number(ch, argument, cmd, &material->fragility);
}

OLC_FUN(materialed_flag)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_flag64(ch, argument, cmd, &material->flag);
}

OLC_FUN(materialed_density)
{
    material_t *material;

    EDIT_MATERIAL(ch, material);
    return olced_number(ch, argument, cmd, &material->density);
}

OLC_FUN(materialed_show)
{
    char              arg[MAX_STRING_LENGTH];
    BUFFER           *output;
    material_t       *material;
    int               i,j;
    MOB_INDEX_DATA   *pMobIndex;

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_MATERIAL))
            EDIT_MATERIAL(ch, material);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (atoi(arg) < 0 || atoi(arg) > materials.nused)
            {
                char_printf(ch, "MaterialEd: %s: No such material.\n", arg);
                return FALSE;
            }
            else
                material = MATERIAL(atoi(arg));
        }
        else
        {
            if (material_n_lookup(arg) < 0)
            {
               char_printf(ch, "MaterialEd: Not such material %s.\n", arg);
               return FALSE;
            }
            else
               material = MATERIAL(material_n_lookup(arg));
        }
    }

    output = buf_new(-1);

    buf_printf(output, "English Name:       [%s]\n", material->ename);
    buf_printf(output, "Russian Name:       [%s]\n", material->rname);
    buf_printf(output, "Flags:              [%s]\n", flag_string(material_flags, material->flag));
    buf_printf(output, "Rigidity:           [%d]\n", material->rigidity);
    buf_printf(output, "Fragility:          [%d]\n", material->fragility);
    buf_printf(output, "Density:            [%d]\n", material->density);
    buf_printf(output, "Races:              [");

    j = 0;

    for (i = 0; i < material->races.nused; i++)
    {
        mtrace_t *race = varr_get(&material->races, i);
        if (race == NULL)
        {
            continue;
        }
        if (j > 0)
        {
            buf_printf(output, "\n                     %-8s %s ", race->name, flag_string(status_table, race->status));
        }
        else
            buf_printf(output, "  %-8s %s ", race->name, flag_string(material_wear_status, race->status));

        j++;
    }
    buf_printf(output, "]\n");

    buf_printf(output, "Repair:             [");

    j =0;
    for (i = 0; i < material->repairer.nused; i++)
    {
        mrepair_t *repairer = varr_get(&material->repairer, i);

        if (repairer == NULL)
        {
            continue;
        }

        pMobIndex = get_mob_index(repairer->vnum);

        if (j > 0)
        {
            buf_printf(output, "\n                      %-6d  Repair ratio: %4d  - %s", repairer->vnum, repairer->repair_rate, pMobIndex != NULL ? mlstr_mval(pMobIndex->short_descr) : "= Unknown =");
        }
        else
            buf_printf(output, " %-6d  Repair ratio: %4d  - %s ", repairer->vnum, repairer->repair_rate, pMobIndex != NULL ? mlstr_mval(pMobIndex->short_descr) : "= Unknown =");

        j++;
    }

    buf_printf(output, "]\n\r\n");

    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}



OLC_FUN(materialed_race)
{
    char arg[MAX_STRING_LENGTH];

    argument = one_argument(argument, arg, sizeof(arg));
    if (!str_prefix(arg, "add"))
        return materialed_race_add(ch, argument, cmd);
    else if (!str_prefix(arg, "delete"))
        return materialed_race_del(ch, argument, cmd);

    do_help(ch, "'OLC MATERIAL RACE'");
    return FALSE;
}

OLC_FUN(materialed_race_add)
{
    int           race;
    int           status;
    mtrace_t     *mtrace;
    char          name[MAX_STRING_LENGTH];
    const char   *tmp;
    material_t   *material;

    EDIT_MATERIAL(ch, material);

    argument = one_argument(argument, name, sizeof(name));

    if (IS_NULLSTR(name) || IS_NULLSTR(argument))
    {
        do_help(ch, "'OLC MATERIAL RACE'");
        return FALSE;
    }

    if ((race = rn_lookup(name)) <= 0)
    {
        char_printf(ch, "MaterialEd: %s: unknown race.\n", name);
        return FALSE;
    }

    if ((status = flag_value(material_wear_status, argument)) == 0)
    {
        char_printf(ch, "ReligionEd: %s: unknown flag.\n", argument);
        return FALSE;
    }

    tmp = str_dup(race_name(race));

    if ((mtrace = material_race_lookup (material, tmp)) != NULL)
    {
        char_printf(ch, "MaterialEd: %s: already there.\n", race_name(race));
        free_string(tmp);

        return FALSE;
    }

    free_string(tmp);

    mtrace         = varr_enew(&material->races);
    mtrace->name   = str_dup(race_name(race));
    mtrace->status = status;

    varr_qsort(&material->races, cmpstr);

    return TRUE;
}

OLC_FUN(materialed_race_del)
{
     const char *name;
     material_t *material;
     mtrace_t   *mtrace;
     int         race;

     EDIT_MATERIAL(ch, material);

    if (IS_NULLSTR(argument))
    {
        do_help(ch, "'OLC MATERIAL RACE'");
        return FALSE;
    }

    if ((race = rn_lookup(argument)) <= 0)
    {
        char_printf(ch, "MaterialEd: %s: unknown race.\n", name);
        return FALSE;
    }

    name = str_dup(race_name(race));

    if ((mtrace = material_race_lookup(material, name)) == NULL)
    {
        char_printf(ch, "MaterialEd: %s: it's currently deleted.\n", name);
        return FALSE;
    }

    varr_del(&material->races, (void*)mtrace);

    varr_qsort(&material->races, cmpstr);
    return TRUE;

}

OLC_FUN(materialed_repairman)
{
    char arg[MAX_STRING_LENGTH];

    argument = one_argument(argument, arg, sizeof(arg));
    if (!str_prefix(arg, "add"))
        return materialed_repairman_add(ch, argument, cmd);
    else if (!str_prefix(arg, "delete"))
        return materialed_repairman_del(ch, argument, cmd);

    do_help(ch, "'OLC MATERIAL REPAIR'");
    return FALSE;
}

OLC_FUN(materialed_repairman_add)
{
    char            arg1[MAX_STRING_LENGTH];
    material_t     *material;
    mrepair_t      *repairer;
    int             vnum;
    MOB_INDEX_DATA *pMobIndex;


    EDIT_MATERIAL(ch, material);

    argument = one_argument(argument, arg1, sizeof(arg1));

    if (IS_NULLSTR(arg1) || IS_NULLSTR(argument) || !is_number(arg1) || !is_number(argument))
    {
        do_help(ch, "'OLC MATERIAL REPAIR'");
        return FALSE;
    }

    vnum = atoi(arg1);

    if ((pMobIndex = get_mob_index(vnum)) == NULL)
    {
        char_printf(ch, "MaterialEd: Not such mob %d.\n", vnum);
        return FALSE;
    }

    if ((repairer = material_repair_lookup (material, vnum)) != NULL)
    {
        char_printf(ch, "MaterialEd: Mob %d: already there.\n", vnum);
        return FALSE;
    }

    repairer              = varr_enew(&material->repairer);
    repairer->vnum        = vnum;
    repairer->repair_rate = atoi(argument);

    varr_qsort(&material->repairer, cmpint);

    return TRUE;
}

OLC_FUN(materialed_repairman_del)
{
     int         vnum;
     material_t *material;
     mrepair_t  *repairer;

     EDIT_MATERIAL(ch, material);

    if (IS_NULLSTR(argument) || !is_number(argument))
    {
        do_help(ch, "'OLC MATERIAL REPAIR'");
        return FALSE;
    }

    vnum = atoi(argument);

    if ((repairer = material_repair_lookup(material, vnum)) == NULL)
    {
        char_printf(ch, "MaterialEd: %d: it's currently deleted.\n", vnum);
        return FALSE;
    }

    varr_del(&material->repairer, (void*)repairer);

    varr_qsort(&material->repairer, cmpint);
    return TRUE;

}
