/* $Id: olc_meteor.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"

#define EDIT_METEOR(ch, meteor)        (meteor = (meteor_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(meteored_create       );
DECLARE_OLC_FUN(meteored_edit         );
DECLARE_OLC_FUN(meteored_touch        );
DECLARE_OLC_FUN(meteored_show         );
DECLARE_OLC_FUN(meteored_list         );

DECLARE_OLC_FUN(meteored_name         );
DECLARE_OLC_FUN(meteored_autostart    );
DECLARE_OLC_FUN(meteored_damage       );
DECLARE_OLC_FUN(meteored_nopk         );
DECLARE_OLC_FUN(meteored_room_affect  );
DECLARE_OLC_FUN(meteored_now_active   );
DECLARE_OLC_FUN(meteored_size         );
DECLARE_OLC_FUN(meteored_waittick     );
DECLARE_OLC_FUN(meteored_obj          );
DECLARE_OLC_FUN(meteored_obj_add      );
DECLARE_OLC_FUN(meteored_obj_del      );

olc_cmd_t olc_cmds_meteor[] =
{
    { "create",        meteored_create,       5},
    { "edit",          meteored_edit,         5},
    { "touch",         olced_dummy,           5},
    { "show",          meteored_show,         0},
    { "list",          meteored_list,         0},

    { "name",          meteored_name,         5},
    { "autostart",     meteored_autostart,    5,   meteor_bool_flags},
    { "damage",        meteored_damage,       5},
    { "nopk",          meteored_nopk,         5,   meteor_bool_flags},
    { "room_affect",   meteored_room_affect,  5,   meteor_bool_flags},
    { "size",          meteored_size,         5},
    { "waittick",      meteored_waittick,     5},
    { "active",        meteored_now_active,   5,   meteor_bool_flags},
    { "obj",           meteored_obj,          5},

    { "commands",      show_commands,         0},
    { NULL}
};

OLC_FUN(meteored_create)
{
    int mn;
    meteor_t *meteor;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_METEOR"))
//    if (ch->pcdata->security < SECURITY_OLC_METEOR) 
    {
        char_puts("MeteorEd: Insufficient security for editing meteors\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }
//    -     
    if ((mn = mrn_lookup(arg)) >= 0)
    {
        char_printf(ch, "MeteorEd: %s: already exists.\n", METEOR(mn)->name);
        return FALSE;
    }

    meteor                  = meteor_new();
    meteor->name            = str_dup(arg);
    

    ch->desc->pEdit     = (void *)meteor;
    OLCED(ch)   = olced_lookup(ED_METEOR);                              
    //touch_meteor(meteor);
    char_puts("Meteor created.\n",ch);
    return FALSE;
}

OLC_FUN(meteored_edit)
{
    int mrn;
    char arg[MAX_STRING_LENGTH];

    if (!char_security(ch,"SECURITY_OLC_METEOR"))
    //if (ch->pcdata->security < SECURITY_METEORS)
    {
        char_puts("MeteorEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    mrn = -1;

    if (is_number(arg))
    {
        if (!str_cmp(meteor_name(atoi(arg)),"None"))
        {
            char_printf(ch, "MeteorEd: %s: No such meteor.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit     = METEOR(atoi(arg));
    }
    else
    {
        if ((mrn = mrn_lookup(arg)) < 0)
        {
            char_printf(ch, "MeteorEd: %s: No such meteor.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit     = METEOR(mrn);
    }

    OLCED(ch)   = olced_lookup(ED_METEOR);
    return FALSE;
}

OLC_FUN(meteored_list)
{
    int i;

    for (i = 0; i < meteors.nused; i++)
        char_printf(ch, "[%d] %s\n", i, METEOR(i)->name);
    return FALSE;
}

OLC_FUN(meteored_name)
{
    meteor_t *meteor;
    
    EDIT_METEOR(ch, meteor);
    return olced_str(ch, argument, cmd, &meteor->name);
}

OLC_FUN(meteored_damage)
{
    meteor_t *meteor;
    
    EDIT_METEOR(ch, meteor);
    return olced_number(ch, argument, cmd, &meteor->damage);
}

OLC_FUN(meteored_size)
{
    meteor_t *meteor;
    
    EDIT_METEOR(ch, meteor);
    return olced_number(ch, argument, cmd, &meteor->size);
}

OLC_FUN(meteored_waittick)
{
    meteor_t *meteor;

    EDIT_METEOR(ch, meteor);
    return olced_number(ch, argument, cmd, &meteor->wait_tick);
}

OLC_FUN(meteored_show)
{
    char arg[MAX_STRING_LENGTH];
    int i;
    int j;
    BUFFER *output;
    meteor_t *meteor;
    AREA_DATA *area;
    OBJ_INDEX_DATA *obj;

    area = NULL;
    obj = NULL;
 
    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_METEOR))
            EDIT_METEOR(ch, meteor);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (!str_cmp(meteor_name(atoi(arg)),"None"))
            {
                char_printf(ch, "MeteorEd: %s: No such meteor.\n", arg);
                return FALSE;
            }
            else
                meteor = METEOR(atoi(arg));
        }
        else
        {
            char_printf(ch, "MeteorEd: Must specify meteor id.\n", arg);
            return FALSE;
        }
    }

    output = buf_new(-1);

    // buf_printf(output, "Id:                [%d]\n", meteors.);
    buf_printf(output, "Name:               [%s]\n", meteor->name);
    buf_printf(output, "Size:               [%d]\n", meteor->size);
    buf_printf(output, "Tick between start: [%d]\n", meteor->wait_tick);
    buf_printf(output, "Damage:             [%d]\n", meteor->damage);
    buf_printf(output, "NOPK Damage:        [%s]\n", flag_string(meteor_bool_flags, meteor->nopk_damage));
    buf_printf(output, "Room affect:        [%s]\n", flag_string(meteor_bool_flags, meteor->room_affect));
    buf_printf(output, "Autostart:          [%s]\n\r\n", flag_string(meteor_bool_flags, meteor->auto_start));
    
    
    if (meteor->area_vnum > -1 )
    {
        area = area_lookup(meteor->area_vnum);
        if (area != NULL)
        {
            buf_printf(output, "Area:               [%d   %s]\n", meteor->area_vnum,fmt_color_str(area->name,30));
        }
    }
    else
       buf_printf(output, "Area:               [%d   ]\n", meteor->area_vnum);

    buf_printf(output, "Now active:         [%s]\n", flag_string(meteor_bool_flags, meteor->active));
    buf_printf(output, "Current tick:       [%d]\n", meteor->curr_tick);
    
    j=0;
    
    buf_printf(output, "Objects for load:   [");
    for (i = 0; i < meteor->objects.nused; i++)
    {
        mobject_t *mobject = varr_get(&meteor->objects, i);
        if (mobject == NULL)
        {
            continue;
        }

        obj = get_obj_index(mobject->obj_vnum);
        
        if (j > 0)
        {
            buf_printf(output, "\n                      Vnum: %6d  Qty: %3d  Chance: %3d  (%s) ", 
                                               mobject->obj_vnum,
                                               mobject->obj_count,
                                               mobject->chanse,
                                               fmt_color_str(obj->name,30));
        }
        else
            buf_printf(output, " Vnum: %6d  Qty: %3d  Chance: %3d  (%s) ", 
                                               mobject->obj_vnum,
                                               mobject->obj_count,
                                               mobject->chanse,
                                               fmt_color_str(obj->name,30));

        j++;
    }
    buf_printf(output, "]\n\r");
    buf_printf(output, "{RDamage on 100 lvl{x:  [%d]\n", 100*10 + meteor->size * meteor->damage);
    
    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(meteored_autostart)
{
    meteor_t *meteor;

    EDIT_METEOR(ch, meteor);
    return olced_flag64(ch, argument, cmd, &meteor->auto_start);
}

OLC_FUN(meteored_nopk)
{
    meteor_t *meteor;

    EDIT_METEOR(ch, meteor);
    return olced_flag64(ch, argument, cmd, &meteor->nopk_damage);
}

OLC_FUN(meteored_room_affect)
{
    meteor_t *meteor;

    EDIT_METEOR(ch, meteor);
    return olced_flag64(ch, argument, cmd, &meteor->room_affect);
}

OLC_FUN(meteored_obj)
{
    char arg[MAX_STRING_LENGTH];

    argument = one_argument(argument, arg, sizeof(arg));
    if (!str_prefix(arg, "add"))
        return meteored_obj_add(ch, argument, cmd);
    else if (!str_prefix(arg, "delete"))
        return meteored_obj_del(ch, argument, cmd);

    do_help(ch, "'OLC METEOR OBJ'");
    return FALSE;
}

OLC_FUN(meteored_obj_add)
{
    mobject_t *mobj;
    meteor_t  *meteor;
    int       obj_vnum;

    char    arg1[MAX_STRING_LENGTH];
    char    arg2[MAX_STRING_LENGTH];
    char    arg3[MAX_STRING_LENGTH]; 
    
    EDIT_METEOR(ch, meteor);

    argument = one_argument(argument, arg1, sizeof(arg1));
    argument = one_argument(argument, arg2, sizeof(arg2));
    one_argument(argument, arg3, sizeof(arg3));

    if (arg1[0] == '\0' || arg2[0] == '\0' /*|| arg3[0] == '\0'*/)
    {
        do_help(ch, "'OLC METEOR OBJ'");
        return FALSE;
    }

    if (!is_number(arg1))
    {
        char_printf(ch, "MeteorEd: first argument must be obj vnum.\n");
        return FALSE;
    }

    obj_vnum = atoi(arg1);
/*
    if (!validate_obj_vnum(obj_vnum))
    {
        return FALSE;
    }
*/
    if (!is_number(arg2))
    {
        char_printf(ch, "MeteorEd: second argument must be number between 1 and 50.\n");
        return FALSE;
    }

    if (atoi(arg2) < 1 || atoi(arg2) > 50 )
    {
        char_printf(ch, "MeteorEd: second argument must be number between 1 and 50.\n");
        return FALSE;
    }

    if (!is_number(arg3))
    {
        char_printf(ch, "MeteorEd: 3th argument must be number between 1 and 100.\n");
        return FALSE;
    }

    if (atoi(arg3) < 1 || atoi(arg3) > 100 )
    {
        char_printf(ch, "MeteorEd: 3th argument must be number between 1 and 100.\n");
        return FALSE;
    }

    if ((mobj = meteor_obj_lookup(meteor,obj_vnum )))
    {
        char_printf(ch, "MeteorEd: Vnum %s already there.\n", arg1);
        return FALSE;
    }

    mobj = varr_enew(&meteor->objects);
    mobj->obj_vnum  = atoi(arg1);
    mobj->obj_count = atoi(arg2);
    mobj->chanse    = atoi(arg3);
    varr_qsort(&meteor->objects, cmpint);

    char_printf(ch, "MeteorEd: Obj %d added.\n", obj_vnum); 

    return TRUE;
}

OLC_FUN(meteored_obj_del)
{
    meteor_t    *meteor;
    int         i;
    int         size;
    char        *mem;
    mobject_t  *array;
    int         obj_vnum;
    mobject_t   *mobj;
        
    EDIT_METEOR(ch, meteor);
    
    if (IS_NULLSTR(argument))
    {
        do_help(ch, "'OLC METEOR OBJ'");
        return FALSE;
    }

    if(!is_number(argument))
    {
       char_printf(ch, "MeteorEd: Argument must be object vnum.\n");
       return FALSE;
    }

    obj_vnum = atoi(argument);

    if (!(mobj = meteor_obj_lookup(meteor,obj_vnum )))
    {
        char_printf(ch, "MeteorEd: Obj vnum %d already not there.\n", obj_vnum);
        return FALSE;
    }

    array = (mobject_t*)meteor->objects.p;

    for (i = 0; i < meteor->objects.nused; i++)
    {
        if (obj_vnum == array[i].obj_vnum)
        {
            size = sizeof(mobject_t);
            mem = (char*)meteor->objects.p;
                        memcpy(mem+i*size,mem+(i+1)*size, (meteor->objects.nused-i)*size  );
                        meteor->objects.nused--;

                        varr_qsort(&meteor->objects, cmpint);
                        char_printf(ch, "ReligionEd: delete obj %d.\n", obj_vnum);
            return TRUE;
        }
    }

    varr_qsort(&meteor->objects, cmpint);
    return TRUE;
  
}

OLC_FUN(meteored_now_active)
{
    meteor_t *meteor;

    EDIT_METEOR(ch, meteor);
    return olced_flag64(ch, argument, cmd, &meteor->active);
}

