/* $Id: olc_price.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "merc.h"
#include "olc.h"
#include "price.h"

#define EDIT_PRICE(ch, price)	(price = (price_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(priceed_create       );
DECLARE_OLC_FUN(priceed_edit         );
DECLARE_OLC_FUN(priceed_touch        );
DECLARE_OLC_FUN(priceed_show         );
DECLARE_OLC_FUN(priceed_list         );
DECLARE_OLC_FUN(priceed_name         );
DECLARE_OLC_FUN(priceed_descriptor   );
DECLARE_OLC_FUN(priceed_amount       );
DECLARE_OLC_FUN(priceed_amount_type  );

static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_price[] =
{
    { "create",        priceed_create,         5                },
    { "edit",          priceed_edit,           5                },
    { "touch",         olced_dummy,            5                },
    { "show",          priceed_show,           0                },
    { "list",          priceed_list,           0                },

    { "name",          priceed_name,           0, validate_name },
    { "descriptor",    priceed_descriptor,     0                },
    { "amount",        priceed_amount,         0                },
	{ "type",          priceed_amount_type,    0, price_type    },

    { "commands",      show_commands,          0},
    { NULL}
};


OLC_FUN(priceed_create)
{
    price_t *price;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_PRICES)
    {
        char_puts("PriceEd: Insufficient security for editing religions\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);

    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

	price = price_lookup(arg, str_cmp);

    if (price != NULL)
    {
        char_printf(ch, "PriceEd: %s: already exists.\n", price->name);
        return FALSE;
    }

    price                = price_new();
    price->name          = str_dup(arg);
	price->descriptor    = str_empty;


    ch->desc->pEdit     = (void *)price;
    OLCED(ch)   = olced_lookup(ED_PRICE);  //need added to olc.h                            
    //touch_price(price);
    char_puts("PriceEd: price created.\n",ch);
    return FALSE;
}

OLC_FUN(priceed_edit)
{
    price_t *price;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_PRICES)
    {
        char_puts("PriceEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if (is_number(arg))
    {
        if (atoi(arg) < 0 || atoi(arg) > prices.nused)
        {
            char_puts("PriceEd: not such price.\n", ch);
            return FALSE;
        }
        else
            price = VARR_GET(&prices, atoi(arg));
    }
    else
        if ((price = price_lookup(arg, str_cmp)) == NULL)
    {
        char_printf(ch, "PriceEd: %s: No such price.\n", arg);
        return FALSE;
    }

    ch->desc->pEdit = price;
    OLCED(ch)   = olced_lookup(ED_PRICE);
    return FALSE;
}

OLC_FUN(priceed_show)
{
    char arg[MAX_STRING_LENGTH];
    BUFFER *output;
    price_t *price;

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_PRICE))
            EDIT_PRICE(ch, price);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (atoi(arg) < 0 || atoi(arg) > prices.nused)
            {
                char_puts("PriceEd: not such price.\n", ch);
                return FALSE;
            }
            else
                price = VARR_GET(&prices, atoi(arg));
        }
        else
            if ((price = price_lookup(arg, str_cmp)) == NULL)
        {
            char_printf(ch, "PriceEd: %s: No such price.\n", arg);
            return FALSE;
        }

    }

    output = buf_new(-1);

    buf_printf(output, "Name:          [%s]\n", price->name);
    buf_printf(output, "Description:   [%s]\n", price->descriptor);
    buf_printf(output, "Amount:        [%-6d %s]\n", price->amount, flag_string(price_type, price->amount_type));
	
    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(priceed_list)
{
    int i;
    BUFFER *output;

    output = buf_new(-1);

    for (i = 0; i < prices.nused; i++)
	{
        price_t *price = (price_t*) VARR_GET(&prices, i);
        buf_printf(output, "[%3d] %-15s %-30s [%5d] %d\n", i, price->name, price->descriptor, price->amount, price->amount_type);
	}

    page_to_char(buf_string(output), ch);
    buf_free(output);
    return FALSE;
}


OLC_FUN(priceed_name)
{
    price_t *price;
    
    EDIT_PRICE(ch, price);
    return olced_str(ch, argument, cmd, &price->name);
}

OLC_FUN(priceed_descriptor)
{
    price_t *price;

    EDIT_PRICE(ch, price);
    return olced_str(ch, argument, cmd, &price->descriptor);
}


OLC_FUN(priceed_amount)
{
    price_t *price;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    EDIT_PRICE(ch, price);
    return olced_number(ch, argument, cmd, &price->amount);
}

static VALIDATE_FUN(validate_name)
{
    price_t *price;
    EDIT_PRICE(ch, price);

	price = price_lookup(arg, str_cmp);

	if (price != NULL)
    {
        char_printf(ch, "PriceEd: %s: already exists.\n", price->name);
        return FALSE;
    }

    return TRUE;
}

OLC_FUN(priceed_amount_type)
{
    price_t *price;
    EDIT_PRICE(ch, price);
    return olced_flag64(ch, argument, cmd, &price->amount_type);
}

