/* $Id: olc_ritual.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"

#define EDIT_RITUAL(ch, ritual)   (ritual = (ritual_t*) ch->desc->pEdit)

extern varr rituals;

extern void save_ritual (CHAR_DATA *ch);

DECLARE_OLC_FUN(ritualed_create           );
DECLARE_OLC_FUN(ritualed_edit             );
DECLARE_OLC_FUN(ritualed_touch            );
DECLARE_OLC_FUN(ritualed_show             );
DECLARE_OLC_FUN(ritualed_list             );

DECLARE_OLC_FUN(ritualed_name             );
DECLARE_OLC_FUN(ritualed_components       );
DECLARE_OLC_FUN(ritualed_sector           );
DECLARE_OLC_FUN(ritualed_time             );
DECLARE_OLC_FUN(ritualed_extra            );
DECLARE_OLC_FUN(ritualed_minskill         );
DECLARE_OLC_FUN(ritualed_duration         );

static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_ritual[] =
{
    { "create",             ritualed_create,            5                            },
    { "edit",               ritualed_edit,              5                            },
    { "touch",              olced_dummy,                5                            },
    { "show",               ritualed_show,              0                            },
    { "list",               ritualed_list,              0                            },

    { "name",               ritualed_name,              5, validate_name             },
    { "components",         ritualed_components,        0                            },
    { "sector",             ritualed_sector,            0, sector_flag_types         },
    { "period",             ritualed_time,              0, time_flag_types           },
    { "extra",              ritualed_extra,             0, ritual_extra_flags        },
    { "minskill",           ritualed_minskill,          0                            },
    { "duration",           ritualed_duration,          0                            },

    { "commands",           show_commands,              0                            },
    { NULL}
};

OLC_FUN(ritualed_create)
{
    int        tn;
    ritual_t * ritual;

    if (argument[0] == '\0')
    {
        do_help(ch, "'OLC RITUAL'");
        return FALSE;
    }

    if ((tn = ritualnumber_lookup(argument)) >= 0)
    {
        char_printf(ch, "ritualEd: %s ritual already exists.\n", RITUAL(tn)->name);
        return FALSE;
    }

    ritual                = ritual_new();
    ritual->name          = str_dup(argument);

    ch->desc->pEdit     = (void *)ritual;
    OLCED(ch)   = olced_lookup(ED_RITUAL);

    char_puts("Ritual created.\n",ch);
    return FALSE;
}

OLC_FUN(ritualed_edit)
{
    int  tn;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC RITUAL'");
        return FALSE;
    }

    tn = -1;

    if (is_number(arg))
    {
        if (!str_cmp(ritual_name(atoi(arg)),"None"))
        {
            char_printf(ch, "ritualEd: %s: No such ritual.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit = RITUAL(atoi(arg));
    }
    else
    {
        if ((tn = ritualnumber_lookup(arg)) < 0)
        {
            char_printf(ch, "ritualEd: %s: No such ritual.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit = RITUAL(tn);
    }

    OLCED(ch)   = olced_lookup(ED_RITUAL);
    return FALSE;
}

OLC_FUN(ritualed_list)
{
    int i;

    for (i = 0; i < rituals.nused; i++)
        char_printf(ch, "[%d] %s\n", i, RITUAL(i)->name);
    return FALSE;
}

OLC_FUN(ritualed_name)
{
    ritual_t *ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_str(ch, argument, cmd, &ritual->name);
}

OLC_FUN(ritualed_minskill)
{
    ritual_t * ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_number(ch, argument, cmd, &ritual->min_skill);
}

OLC_FUN(ritualed_duration)
{
    ritual_t * ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_number(ch, argument, cmd, &ritual->duration);
}

OLC_FUN(ritualed_time)
{
    ritual_t * ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_flag32(ch, argument, cmd, &ritual->time);
}

OLC_FUN(ritualed_extra)
{
    ritual_t * ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_flag32(ch, argument, cmd, &ritual->extra);
}

OLC_FUN(ritualed_sector)
{
    ritual_t * ritual;

    EDIT_RITUAL(ch, ritual);
    return olced_flag32(ch, argument, cmd, &ritual->sector);
}

OLC_FUN(ritualed_components)
{
    ritual_t * ritual;
    char arg[MAX_STRING_LENGTH];
    int num, vnum;

    EDIT_RITUAL(ch, ritual);

    // number
    argument = one_argument (argument, arg, sizeof(arg));
    num = atoi (arg);
    if (num < 1 || num > MAX_RITUAL_COMP)
    {
        char_act ("Syntax: comp_paint <num> <vnum>", ch);
        char_printf (ch, "Num must be between 1 and %d.\n", MAX_RITUAL_COMP);
        return FALSE;
    }

    // vnum 
    argument = one_argument (argument, arg, sizeof(arg));
    vnum = atoi (arg);
    vnum = UMAX (0, vnum);
    // validate vnum
    if (vnum != 0 && get_obj_index (vnum) == NULL)
    {
        char_printf (ch, "Obj vnum %d does not exist.\n", vnum);
        return FALSE;
    }

    // fill the array
    ritual->components[num - 1] = vnum;
    char_act ("Ok.", ch);
    return TRUE;
}

OLC_FUN(ritualed_show)
{
    char       arg[MAX_STRING_LENGTH];
    int        i;
    BUFFER   * output;
    ritual_t * ritual;
    bool       found;
    OBJ_INDEX_DATA * obj;

    if (rituals.nused == 0)
    {
        char_act ("No rituals at all.", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_RITUAL))
            EDIT_RITUAL(ch, ritual);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (!str_cmp(ritual_name(atoi(arg)),"None"))
            {
                char_printf(ch, "RitualEd: %s: No such ritual.\n", arg);
                return FALSE;
            }
            else
                ritual = RITUAL(atoi(arg));
        }
        else
        {
            i = ritualnumber_lookup(argument);
            if ( i < 0 )
            {
                 char_printf(ch, "RitualEd: Must specify ritual name.\n");
                 return FALSE;
            }

        ritual = RITUAL(i);
        }
    }

    output = buf_new(ch->lang);

    buf_printf (output, "Name:                  [%s]\n", ritual->name);
    buf_printf (output, "Duration:              [%d]\n", ritual->duration);
    buf_printf (output, "Min. skill percent:    [%d]\n", ritual->min_skill);
    buf_printf (output, "Sector:                [%s]\n", flag_string(sector_flag_types, ritual->sector));
    buf_printf (output, "Time period:           [%s]\n", flag_string(time_flag_types, ritual->time));
    buf_printf (output, "Extra flags:           [%s]\n", flag_string(ritual_extra_flags, ritual->extra));

    found = FALSE;
    buf_add (output, "Components:         {x");
    for (i = 0; i < MAX_RITUAL_COMP; ++i)
    {
        if (ritual->components[i] != 0)
        {
            obj = get_obj_index (ritual->components[i]);
            buf_printf (output, "\n{G%2d{x. [%7d] %s", 
                i + 1,
                ritual->components[i], 
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, "   [none]\n");

    page_to_char (buf_string (output), ch);
    buf_free (output);

    return FALSE;
}

static VALIDATE_FUN(validate_name)
{
    int          tn;
    ritual_t   * ritual;

    EDIT_RITUAL(ch, ritual);

    if ((tn = ritualnumber_lookup(arg)) >= 0)
    {
      char_printf(ch, "RitualEd: %s: already exists.\n", RITUAL(tn)->name);
      return FALSE;
    }

    return TRUE;
}

bool delete_ritual (CHAR_DATA *ch, const char *argument)
{
    ritual_t * ritual;
    int        tnumber;
    char       name[MAX_INPUT_LENGTH];

    if (IS_NULLSTR(argument))
    {
        do_help (ch, "'OLC RITUAL'");
        return FALSE;
    }

    one_argument(argument, name, sizeof(name));

    tnumber = ritualnumber_lookup(name);

    if (tnumber < 0)
    {
        char_printf(ch, "Ritual %s not in database", name);
        return FALSE;
    }

    ritual = RITUAL(tnumber);
    if (ritual != NULL)
    {
        varr_del(&rituals, (void*)ritual);
        char_printf (ch, "Ex-ritual %s deleted from database", name);
        save_ritual(ch);
        return TRUE;
    }
    else
        log_printf("** BUG ** olc_ritual:delete_ritual: ritual = NULL");

    return FALSE;
}
