/* $Id: olc_security.c,v 1.666 2004/09/20 10:50:30 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "merc.h"
#include "olc.h"

#define EDIT_SEC(ch, security)	(security = (security_t*) ch->desc->pEdit)

DECLARE_OLC_FUN(seced_create       );
DECLARE_OLC_FUN(seced_edit         );
DECLARE_OLC_FUN(seced_touch        );
DECLARE_OLC_FUN(seced_show         );
DECLARE_OLC_FUN(seced_list         );
DECLARE_OLC_FUN(seced_name         );
DECLARE_OLC_FUN(seced_level        );


static DECLARE_VALIDATE_FUN(validate_sec_name);

olc_cmd_t olc_cmds_sec[] =
{
    { "create",        seced_create,         5                    },
    { "edit",          seced_edit,           5                    },
    { "touch",         olced_dummy,          5                    },
    { "show",          seced_show,           0                    },
    { "list",          seced_list,           0                    },

    { "name",          seced_name,           0, validate_sec_name },
    { "level",         seced_level,          0                    },
	

    { "commands",      show_commands,        0                    },
    { NULL}
};


OLC_FUN(seced_create)
{
    security_t *security;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_SECS)
    {
        char_puts("SecEd: Insufficient security for editing security\n", ch);
        return FALSE;
    }

    first_arg(argument, arg, sizeof(arg), FALSE);

    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC CREATE'");
        return FALSE;
    }

	security = security_lookup(arg, str_cmp);

    if (security != NULL)
    {
        char_printf(ch, "SecEd: %s: already exists.\n", security->name);
        return FALSE;
    }

    security                = security_new();
    security->name          = str_dup(arg);
	security->level         = 9;


    ch->desc->pEdit     = (void *)security;
    OLCED(ch)   = olced_lookup(ED_SEC);  //need added to olc.h                            
    char_puts("SecEd: security created.\n",ch);
    return FALSE;
}

OLC_FUN(seced_edit)
{
    security_t *security;
    char arg[MAX_STRING_LENGTH];

    if (ch->pcdata->security < SECURITY_SECS)
    {
        char_puts("SecEd: Insufficient security.\n", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC EDIT'");
        return FALSE;
    }

    if (is_number(arg))
    {
        if (atoi(arg) < 0 || atoi(arg) > securitys.nused)
        {
            char_printf(ch, "SecEd: %s not such security.\n", arg);
            return FALSE;
        }
        else
            security = VARR_GET(&securitys, atoi(arg));
    }
    else
        if ((security = security_lookup(arg, str_cmp)) == NULL)
    {
        char_printf(ch, "SecEd: %s: No such security.\n", arg);
        return FALSE;
    }

    ch->desc->pEdit = security;
    OLCED(ch)   = olced_lookup(ED_SEC);
    return FALSE;
}

OLC_FUN(seced_show)
{
    char arg[MAX_STRING_LENGTH];
    BUFFER *output;
    security_t *security;

    one_argument(argument, arg, sizeof(arg));

    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_SEC))
            EDIT_SEC(ch, security);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (atoi(arg) < 0 || atoi(arg) > securitys.nused)
            {
                char_printf(ch, "SecEd: %s not such security.\n", arg);
                return FALSE;
            }
            else
                security = VARR_GET(&securitys, atoi(arg));
        }
        else
            if ((security = security_lookup(arg, str_cmp)) == NULL)
        {
            char_printf(ch, "PriceEd: %s: No such security.\n", arg);
            return FALSE;
        }

    }

    output = buf_new(-1);

    buf_printf(output, "Name:          [%s]\n", security->name);
    buf_printf(output, "Level:         [%d]\n", security->level);
    
    page_to_char(buf_string(output), ch);
    buf_free(output);

    return FALSE;
}

OLC_FUN(seced_list)
{
    int i;
    BUFFER *output;

    output = buf_new(-1);

    for (i = 0; i < securitys.nused; i++)
	{
        security_t *security = (security_t*) VARR_GET(&securitys, i);
        buf_printf(output, "[%3d] %-25s [%d]\n", i, security->name, security->level);
	}

    page_to_char(buf_string(output), ch);
    buf_free(output);
    return FALSE;
}


OLC_FUN(seced_name)
{
    security_t *security;
 
    EDIT_SEC(ch, security);
    return olced_str(ch, argument, cmd, &security->name);
}

OLC_FUN(seced_level)
{
    security_t *security;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));

    if (!is_number(arg))
    {
        char_printf(ch, "SecEd: Level argument must be a number between 0 and 10.\n", arg);
        return FALSE;
    }

    if (atoi(arg) <0 || atoi(arg) > 10)
    {
        char_printf(ch, "SecEd: Level argument must be a number between 0 and 10.\n", arg);
        return FALSE;
    }

    EDIT_SEC(ch, security);
    return olced_number(ch, argument, cmd, &security->level);
}

static VALIDATE_FUN(validate_sec_name)
{
    security_t *security;
    
    EDIT_SEC(ch, security);

    security = security_lookup(arg, str_cmp);

    if (security != NULL)
    {
        char_printf(ch, "SecEd: %s: already exists.\n", security->name);
        return FALSE;
    }

    return TRUE;
}

