/* $Id: olc_tattoo.c,v 1.666 2004/09/20 10:50:31 shrike Exp $ */
 
/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "merc.h"
#include "olc.h"

#define EDIT_TATTOO(ch, tattoo)   (tattoo = (tattoo_t*) ch->desc->pEdit)

extern varr tattooes;

extern void save_tattoo (CHAR_DATA *ch);

DECLARE_OLC_FUN(tattooed_create           );
DECLARE_OLC_FUN(tattooed_edit             );
DECLARE_OLC_FUN(tattooed_touch            );
DECLARE_OLC_FUN(tattooed_show             );
DECLARE_OLC_FUN(tattooed_list             );

DECLARE_OLC_FUN(tattooed_name             );
DECLARE_OLC_FUN(tattooed_desc             );
DECLARE_OLC_FUN(tattooed_time_activation  );
DECLARE_OLC_FUN(tattooed_time_painting    );
DECLARE_OLC_FUN(tattooed_wear_loc         );
DECLARE_OLC_FUN(tattooed_color            );
DECLARE_OLC_FUN(tattooed_extra            );
DECLARE_OLC_FUN(tattooed_minskill         );
DECLARE_OLC_FUN(tattooed_maxactivations   );
DECLARE_OLC_FUN(tattooed_comp_paint       );
DECLARE_OLC_FUN(tattooed_comp_activate    );

static DECLARE_VALIDATE_FUN(validate_name);

olc_cmd_t olc_cmds_tattoo[] =
{
    { "create",             tattooed_create,            5                            },
    { "edit",               tattooed_edit,              5                            },
    { "touch",              olced_dummy,                5                            },
    { "show",               tattooed_show,              0                            },
    { "list",               tattooed_list,              0                            },

    { "name",               tattooed_name,              5,  validate_name            },
    { "desc",               tattooed_desc,              5,                           },
    { "time_activate",      tattooed_time_activation,   5,                           },
    { "time_paint",         tattooed_time_painting,     5,                           },
    { "wear_loc",           tattooed_wear_loc,          5, tattoo_wear_flags         },
    { "color",              tattooed_color,             5, paint_color_flags         },
    { "extra",              tattooed_extra,             5, tattoo_extra_flags        },
    { "minskill",           tattooed_minskill,          5, 0                         },
    { "maxactivations",     tattooed_maxactivations,    5, 0                         },
    { "comp_paint",         tattooed_comp_paint,        5, 0                         },
    { "comp_activate",      tattooed_comp_activate,     5, 0                         },

    { "commands",           show_commands,              0                            },
    { NULL}
};

#define EDIT_PAINT(ch, paint)   (paint = (paint_t*) ch->desc->pEdit)

extern varr paints;

extern void save_paint (CHAR_DATA *ch);

DECLARE_OLC_FUN(painted_create           );
DECLARE_OLC_FUN(painted_edit             );
DECLARE_OLC_FUN(painted_touch            );
DECLARE_OLC_FUN(painted_show             );
DECLARE_OLC_FUN(painted_list             );

DECLARE_OLC_FUN(painted_name             );
DECLARE_OLC_FUN(painted_objname          );
DECLARE_OLC_FUN(painted_short            );
DECLARE_OLC_FUN(painted_long             );
DECLARE_OLC_FUN(painted_components       );
DECLARE_OLC_FUN(painted_color            );
DECLARE_OLC_FUN(painted_minskill         );
DECLARE_OLC_FUN(painted_mana             );
DECLARE_OLC_FUN(painted_maxportions      );
DECLARE_OLC_FUN(painted_power            );

static DECLARE_VALIDATE_FUN(validate_pname);

olc_cmd_t olc_cmds_paint[] =
{
    { "create",             painted_create,            5                            },
    { "edit",               painted_edit,              5                            },
    { "touch",              olced_dummy,               5                            },
    { "show",               painted_show,              0                            },
    { "list",               painted_list,              0                            },

    { "name",               painted_name,              5,  validate_pname           },
    { "objname",            painted_objname,           5,                           },
    { "short",              painted_short,             5,                           },
    { "long",               painted_long,              5,                           },
    { "components",         painted_components,        5,                           },
    { "color",              painted_color,             5, paint_color_int_flags     },
    { "minskill",           painted_minskill,          5,                           },
    { "mana",               painted_mana,              5,                           },
    { "maxportions",        painted_maxportions,       5,                           },
    { "power",              painted_power,             5,                           },

    { "commands",           show_commands,             0                            },
    { NULL}
};


// -------------------------------------------------------------------------------------
// tattooes
// -------------------------------------------------------------------------------------

OLC_FUN(tattooed_create)
{
    int        tn;
    tattoo_t * tattoo;

    if (argument[0] == '\0')
    {
        do_help(ch, "'OLC TATTOO'");
        return FALSE;
    }

    if ((tn = tattoonumber_lookup(argument)) >= 0)
    {
        char_printf(ch, "TattooEd: %s tattoo already exists.\n", TATTOO(tn)->name);
        return FALSE;
    }

    tattoo                = tattoo_new();
    tattoo->name          = str_dup(argument);

    ch->desc->pEdit     = (void *)tattoo;
    OLCED(ch)   = olced_lookup(ED_TATTOO);

    char_puts("Tattoo created.\n",ch);
    return FALSE;
}

OLC_FUN(tattooed_edit)
{
    int  tn;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC TATTOO'");
        return FALSE;
    }

    tn = -1;

    if (is_number(arg))
    {
        if (!str_cmp(tattoo_name(atoi(arg)),"None"))
        {
            char_printf(ch, "TattooEd: %s: No such tattoo.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit = TATTOO(atoi(arg));
    }
    else
    {
        if ((tn = tattoonumber_lookup(arg)) < 0)
        {
            char_printf(ch, "TattooEd: %s: No such tattoo.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit = TATTOO(tn);
    }

    OLCED(ch)   = olced_lookup(ED_TATTOO);
    return FALSE;
}

OLC_FUN(tattooed_list)
{
    int i;

    for (i = 0; i < tattooes.nused; i++)
        char_printf(ch, "[%d] %s\n", i, TATTOO(i)->name);
    return FALSE;
}

OLC_FUN(tattooed_name)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_str(ch, argument, cmd, &tattoo->name);
}

OLC_FUN(tattooed_time_activation)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_number(ch, argument, cmd, &tattoo->wait_act);
}

OLC_FUN(tattooed_minskill)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_number(ch, argument, cmd, &tattoo->min_skill);
}

OLC_FUN(tattooed_maxactivations)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_number(ch, argument, cmd, &tattoo->max_activations);
}

OLC_FUN(tattooed_time_painting)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_number(ch, argument, cmd, &tattoo->wait_paint);
}

OLC_FUN(tattooed_wear_loc)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_flag32(ch, argument, cmd, &tattoo->wear_loc);
}

OLC_FUN(tattooed_color)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_flag32(ch, argument, cmd, &tattoo->paints);
}

OLC_FUN(tattooed_extra)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_flag32(ch, argument, cmd, &tattoo->extra);
}

OLC_FUN(tattooed_desc)
{
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);
    return olced_mlstr(ch, argument, cmd, &tattoo->desc);
}

OLC_FUN(tattooed_comp_paint)
{
    char arg[MAX_STRING_LENGTH];
    int num, vnum;
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);

    // number
    argument = one_argument (argument, arg, sizeof(arg));
    num = atoi (arg);
    if (num < 1 || num > MAX_TATTOO_COMP)
    {
        char_act ("Syntax: comp_paint <num> <vnum>", ch);
        char_printf (ch, "Num must be between 1 and %d.\n", MAX_TATTOO_COMP);
        return FALSE;
    }

    // vnum 
    argument = one_argument (argument, arg, sizeof(arg));
    vnum = atoi (arg);
    vnum = UMAX (0, vnum);
    // validate vnum
    if (vnum != 0 && get_obj_index (vnum) == NULL)
    {
        char_printf (ch, "Obj vnum %d does not exist.\n", vnum);
        return FALSE;
    }

    // fill the array
    tattoo->paint_comp[num - 1] = vnum;
    char_act ("Ok.", ch);
    return TRUE;
}

OLC_FUN(tattooed_comp_activate)
{
    char arg[MAX_STRING_LENGTH];
    int num, vnum;
    tattoo_t *tattoo;

    EDIT_TATTOO(ch, tattoo);

    // number
    argument = one_argument (argument, arg, sizeof(arg));
    num = atoi (arg);
    if (num < 1 || num > MAX_TATTOO_COMP)
    {
        char_act ("Syntax: comp_paint <num> <vnum>", ch);
        char_printf (ch, "Num must be between 1 and %d.\n", MAX_TATTOO_COMP);
        return FALSE;
    }

    // vnum 
    argument = one_argument (argument, arg, sizeof(arg));
    vnum = atoi (arg);
    vnum = UMAX (0, vnum);
    // validate vnum
    if (vnum != 0 && get_obj_index (vnum) == NULL)
    {
        char_printf (ch, "Obj vnum %d does not exist.\n", vnum);
        return FALSE;
    }

    // fill the array
    tattoo->act_comp[num - 1] = vnum;
    char_act ("Ok.", ch);
    return TRUE;
}

OLC_FUN(tattooed_show)
{
    char       arg[MAX_STRING_LENGTH];
    int        i;
    BUFFER   * output;
    tattoo_t * tattoo;
    bool       found;
    OBJ_INDEX_DATA * obj;

    if (tattooes.nused == 0)
    {
        char_act ("No tattooes at all.", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_TATTOO))
            EDIT_TATTOO(ch, tattoo);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (!str_cmp(tattoo_name(atoi(arg)),"None"))
            {
                char_printf(ch, "TattooEd: %s: No such tattoo.\n", arg);
                return FALSE;
            }
            else
                tattoo = TATTOO(atoi(arg));
        }
        else
        {
            i = tattoonumber_lookup(argument);
            if ( i < 0 )
            {
                 char_printf(ch, "TattooEd: Must specify tattoo name.\n");
                 return FALSE;
            }

        tattoo = TATTOO(i);
        }
    }

    output = buf_new(ch->lang);

    buf_printf (output, "Name:                  [%s]\n", tattoo->name);
    mlstr_dump (output, "Desc:\n ", tattoo->desc) ;
    buf_printf (output, "Activaition time:      [%d]\n", tattoo->wait_act);
    buf_printf (output, "Painting time:         [%d]\n", tattoo->wait_paint);
    buf_printf (output, "Min. skill percent:    [%d]\n", tattoo->min_skill);
    if (tattoo->max_activations > -1)
        buf_printf (output, "Max. activations:      [%d]\n", tattoo->max_activations);
    else
        buf_add (output, "Max. activations:      [infinite]\n");
    buf_printf (output, "Can be painted on:     [%s]\n", flag_string(tattoo_wear_flags, tattoo->wear_loc));
    buf_printf (output, "Paints needed:         [%s]\n", flag_string(paint_color_flags, tattoo->paints));
    buf_printf (output, "Extra flags:           [%s]\n", flag_string(tattoo_extra_flags, tattoo->extra));

    found = FALSE;
    buf_add (output, "Painting components:");
    for (i = 0; i < MAX_TATTOO_COMP; ++i)
    {
        if (tattoo->paint_comp[i] != 0)
        {
            obj = get_obj_index (tattoo->paint_comp[i]);
            buf_printf (output, "\n{G%2d{x. [%7d] %s", 
                i + 1,
                tattoo->paint_comp[i], 
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, "   [none]\n");

    found = FALSE;
    buf_add (output, "Activation components:");
    for (i = 0; i < MAX_TATTOO_COMP; ++i)
    {
        if (tattoo->act_comp[i] != 0)
        {
            obj = get_obj_index (tattoo->act_comp[i]);
            buf_printf (output, "\n{G%2d{x. [%7d] %s", 
                i + 1,
                tattoo->act_comp[i], 
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, " [none]\n");

    page_to_char (buf_string (output), ch);
    buf_free (output);

    return FALSE;
}

static VALIDATE_FUN(validate_name)
{
    int          tn;
    tattoo_t   * tattoo;

    EDIT_TATTOO(ch, tattoo);

    if ((tn = tattoonumber_lookup(arg)) >= 0)
    {
      char_printf(ch, "TattooEd: %s: already exists.\n", TATTOO(tn)->name);
      return FALSE;
    }

    return TRUE;
}

bool delete_tattoo (CHAR_DATA *ch, const char *argument)
{
    tattoo_t * tattoo;
    int        tnumber;
    char       name[MAX_INPUT_LENGTH];

    if (IS_NULLSTR(argument))
    {
        do_help (ch, "'OLC TATTOO'");
        return FALSE;
    }

    one_argument(argument, name, sizeof(name));

    tnumber = tattoonumber_lookup(name);

    if (tnumber < 0)
    {
        char_printf(ch, "tattoo %s not in database", name);
        return FALSE;
    }

    tattoo = TATTOO(tnumber);
    if (tattoo != NULL)
    {
        varr_del(&tattooes, (void*)tattoo);
        char_printf (ch, "Ex-tattoo %s deleted from database", name);
        save_tattoo(ch);
        return TRUE;
    }
    else
        log_printf("** BUG ** olc_tattoo:delete_tattoo: tattoo = NULL");

    return FALSE;
}

// -------------------------------------------------------------------------------
// paints
// -------------------------------------------------------------------------------

OLC_FUN(painted_create)
{
    int        tn;
    paint_t * paint;

    if (argument[0] == '\0')
    {
        do_help(ch, "'OLC paint'");
        return FALSE;
    }

    if ((tn = paintnumber_lookup(argument)) >= 0)
    {
        char_printf(ch, "paintEd: %s paint already exists.\n", PAINT(tn)->name);
        return FALSE;
    }

    paint                = paint_new();
    paint->name          = str_dup(argument);

    ch->desc->pEdit     = (void *)paint;
    OLCED(ch)   = olced_lookup(ED_PAINT);

    char_puts("paint created.\n",ch);
    return FALSE;
}

OLC_FUN(painted_edit)
{
    int  tn;
    char arg[MAX_STRING_LENGTH];

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        do_help(ch, "'OLC paint'");
        return FALSE;
    }

    tn = -1;

    if (is_number(arg))
    {
        if (!str_cmp(paint_name(atoi(arg)),"None"))
        {
            char_printf(ch, "paintEd: %s: No such paint.\n", arg);
            return FALSE;
        }
        else
            ch->desc->pEdit = PAINT(atoi(arg));
    }
    else
    {
        if ((tn = paintnumber_lookup(arg)) < 0)
        {
            char_printf(ch, "paintEd: %s: No such paint.\n", argument);
            return FALSE;
        }
        else
            ch->desc->pEdit = PAINT(tn);
    }

    OLCED(ch)   = olced_lookup(ED_PAINT);
    return FALSE;
}

OLC_FUN(painted_list)
{
    int i;

    for (i = 0; i < paints.nused; i++)
        char_printf(ch, "[%d] %s\n", i, PAINT(i)->name);
    return FALSE;
}

OLC_FUN(painted_name)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_str(ch, argument, cmd, &paint->name);
}

OLC_FUN(painted_objname)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_str(ch, argument, cmd, &paint->obj_name);
}

OLC_FUN(painted_short)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_mlstr(ch, argument, cmd, &paint->short_desc);
}

OLC_FUN(painted_long)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_mlstr(ch, argument, cmd, &paint->desc);
}

OLC_FUN(painted_minskill)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_number(ch, argument, cmd, &paint->min_skill);
}

OLC_FUN(painted_mana)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_number(ch, argument, cmd, &paint->mana_cost);
}

OLC_FUN(painted_maxportions)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_number(ch, argument, cmd, &paint->max_portions);
}

OLC_FUN(painted_power)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_number(ch, argument, cmd, &paint->power);
}

OLC_FUN(painted_color)
{
    paint_t *paint;

    EDIT_PAINT(ch, paint);
    return olced_flag32(ch, argument, cmd, &paint->color);
}

OLC_FUN(painted_components)
{
    paint_t *paint;
    char arg[MAX_STRING_LENGTH];
    int num, vnum;

    EDIT_PAINT(ch, paint);

    // number
    argument = one_argument (argument, arg, sizeof(arg));
    num = atoi (arg);
    if (num < 1 || num > MAX_PAINT_COMP)
    {
        char_act ("Syntax: component <num> <vnum>", ch);
        char_printf (ch, "Num must be between 1 and %d.\n", MAX_PAINT_COMP);
        return FALSE;
    }

    // vnum 
    argument = one_argument (argument, arg, sizeof(arg));
    vnum = atoi (arg);
    vnum = UMAX (0, vnum);
    // validate vnum
    if (vnum != 0 && get_obj_index (vnum) == NULL)
    {
        char_printf (ch, "Obj vnum %d does not exist.\n", vnum);
        return FALSE;
    }

    // fill the array
    paint->components[num - 1] = vnum;
    char_act ("Ok.", ch);
    return TRUE;
}

OLC_FUN(painted_show)
{
    char       arg[MAX_STRING_LENGTH];
    int        i;
    BUFFER   * output;
    paint_t * paint;
    bool       found;
    OBJ_INDEX_DATA * obj;

    if (paints.nused == 0)
    {
        char_act ("No paints at all.", ch);
        return FALSE;
    }

    one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        if (IS_EDIT(ch, ED_PAINT))
            EDIT_PAINT(ch, paint);
        else
        {
            do_help(ch, "'OLC ASHOW'");
            return FALSE;
        }
    }
    else
    {
        if (is_number(arg))
        {
            if (!str_cmp(paint_name(atoi(arg)),"None"))
            {
                char_printf(ch, "PaintEd: %s: No such paint.\n", arg);
                return FALSE;
            }
            else
                paint = PAINT(atoi(arg));
        }
        else
        {
            i = paintnumber_lookup(argument);
            if ( i < 0 )
            {
                 char_printf(ch, "PaintEd: Must specify paint name.\n", arg);
                 return FALSE;
            }

        paint = PAINT(i);
        }
    }

    output = buf_new(ch->lang);

    buf_printf (output, "Name:                  [%s]\n", paint->name);
    buf_printf (output, "Created obj name:      [%s]\n", paint->obj_name);
    mlstr_dump (output, "Short desc:\n ",                paint->short_desc);
    mlstr_dump (output, "Long desc:\n ",                 paint->desc);
    buf_printf (output, "Color:                 [%s]\n", flag_string(paint_color_int_flags, paint->color));
    buf_printf (output, "Min. skill percent:    [%d]\n", paint->min_skill);
    buf_printf (output, "Max. portions:         [%d]\n", paint->max_portions);
    buf_printf (output, "Power (default):       [%d]\n", paint->power);
    buf_printf (output, "Mana:                  [%d]\n", paint->mana_cost);

    found = FALSE;
    buf_add (output, "Components:         {x");
    for (i = 0; i < MAX_PAINT_COMP; ++i)
    {
        if (paint->components[i] != 0)
        {
            obj = get_obj_index (paint->components[i]);
            buf_printf (output, "\n{G%2d{x. [%7d] %s", 
                i + 1,
                paint->components[i], 
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, "   [none]\n");

    page_to_char (buf_string (output), ch);
    buf_free (output);

    return FALSE;
}

static VALIDATE_FUN(validate_pname)
{
    int          tn;
    paint_t   * paint;

    EDIT_PAINT(ch, paint);

    if ((tn = paintnumber_lookup(arg)) >= 0)
    {
      char_printf(ch, "paintEd: %s: already exists.\n", PAINT(tn)->name);
      return FALSE;
    }

    return TRUE;
}

bool delete_paint (CHAR_DATA *ch, const char *argument)
{
    paint_t * paint;
    int        tnumber;
    char       name[MAX_INPUT_LENGTH];

    if (IS_NULLSTR(argument))
    {
        do_help (ch, "'OLC PAINT'");
        return FALSE;
    }

    one_argument(argument, name, sizeof(name));

    tnumber = paintnumber_lookup(name);

    if (tnumber < 0)
    {
        char_printf(ch, "paint %s not in database", name);
        return FALSE;
    }

    paint = PAINT(tnumber);
    if (paint != NULL)
    {
        varr_del(&paints, (void*)paint);
        char_printf (ch, "Ex-paint %s deleted from database", name);
        save_paint(ch);
        return TRUE;
    }
    else
        log_printf("** BUG ** olc_paint:delete_paint: paint = NULL");

    return FALSE;
}
