/* $Id: quest.c,v 1.666 2004/09/20 10:49:52 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
/************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>

#include "merc.h"
#include "quest.h"
#include "update.h"
#include "conquer.h"
#include "mob_prog.h"

#ifdef SUNOS
#   include <stdarg.h>
#   include "compat/compat.h"
#endif

extern STAT_DATA stat_record;

DECLARE_DO_FUN(do_say);

static void quest_tell2(CHAR_DATA *ch, CHAR_DATA *questor, const char *fmt, const void *arg, const void *arg3);
#define quest_tell(ch, questor, fmt, arg) \
        quest_tell2((ch), (questor), (fmt), (arg), NULL)

CHAR_DATA *questor_lookup(CHAR_DATA *ch);
qtrouble_t *qtrouble_lookup(CHAR_DATA *ch, int vnum);

static void quest_points (CHAR_DATA *ch, char *argument);
static void quest_info (CHAR_DATA *ch, char *argument);
static void quest_time (CHAR_DATA *ch, char *argument);
static void quest_list (CHAR_DATA *ch, char *argument);
static void quest_buy (CHAR_DATA *ch, char *argument);
static void quest_request (CHAR_DATA *ch, char *argument);
static void quest_complete (CHAR_DATA *ch, char *argument);
static void quest_trouble (CHAR_DATA *ch, char *argument);
static void quest_nocanc (CHAR_DATA *ch, char *argument);
static void quest_canc (CHAR_DATA *ch, char *argument);
static void quest_gained (CHAR_DATA *ch, char *argument);
static void uniq_list (CHAR_DATA *ch, char *argument);

static bool quest_give_item(CHAR_DATA *ch, CHAR_DATA *questor,
                int item_vnum, int count_max);

static bool buy_gold(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_prac(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_tattoo(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_death(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_katana(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_vampire(CHAR_DATA *ch, CHAR_DATA *questor);
static bool buy_unique(CHAR_DATA *ch, CHAR_DATA *questor, char *argument);
static bool make_uniq_weapon(CHAR_DATA *ch, CHAR_DATA *questor, int item_vnum, int level);
static bool make_uniq_object(CHAR_DATA *ch, CHAR_DATA *questor, int item_vnum, int level);
static void quest_unique_trouble(CHAR_DATA *ch, CHAR_DATA *questor, char *argument);
static void quest_unique_upgrade(CHAR_DATA *ch, char *argument);

static int quest_hp_max (CHAR_DATA *ch);
static int quest_mana_max (CHAR_DATA *ch);

void free_mem(void *p, int sMem);
void *alloc_mem(int sMem);

extern bool align_ok(CHAR_DATA *ch, int align);
extern bool ethos_ok(CHAR_DATA *ch, int ethos);


enum qitem_type {
    TYPE_ITEM,
    TYPE_OTHER
};

typedef struct qitem_t qitem_t;
struct qitem_t {
    char    *name;
    int price;
    int vnum;
    bool    (*do_buy)(CHAR_DATA *ch, CHAR_DATA *questor);
};

qitem_t qitem_table[] = {
/*
    { "GIRTH ({W  {x)",  1000, 
       QUEST_VNUM_GIRTH, NULL               },

    { "MASK ({W  {x)", 1000, 
       QUEST_VNUM_MASK, NULL                },
*/
    { "SONG ({W  {x)",  5000, 
       QUEST_VNUM_SONG, NULL                },
/*
    { "RUG {W {x",        750, 
       QUEST_VNUM_RUG, NULL                 },
*/
    { "GOLD {R5000 {Y {x",    1500, 
       0, buy_gold                      },

    { "PRAC {R60 {W {x",   500, 
       0, buy_prac                      },

    { "TATTOO {W  {x",     200, 
       0, buy_tattoo                    },

    { "DEATH {W - {x",   50, 
       0, buy_death                     },

    { "KATANA {W-{x",             1500, 
       0, buy_katana                    },

    { "VAMPIRE {R{x",        50, 
       0, buy_vampire                   },
/*
    { "CANTEEN {W {x  {W{x",   1500, 
       VNUM_CANTEEN, NULL                   },

    { "SCROLL {W  {x'{B54 {x'", 10, 
        OBJ_VNUM_SCROLL_SCRIBING,   NULL    },
*/
    { "UNIQUE {W {x", 0,  -1, NULL},

    { NULL }
};

typedef struct uniq_t uniq_t;
struct uniq_t {
    char    *name;
    int price;
    bool    is_weap;
    int level;
    int vnum;
};

uniq_t uniq_table[] = {
    {"Axe",         1000, TRUE,  10, UNIQUE_AXE},
    {"Bow",         1000, TRUE,  10, UNIQUE_BOW},
    {"Mace",        1000, TRUE,  10, UNIQUE_MACE},
    {"Whip",        1000, TRUE,  10, UNIQUE_WHIP},
    {"Flail",       1000, TRUE,  10, UNIQUE_FLAIL},
    {"Sword",       1000, TRUE,  10, UNIQUE_SWORD},
    {"Dagger",      1000, TRUE,  10, UNIQUE_DAGGER},
    {"Weapon",      1000, TRUE,  10, UNIQUE_WEAPON},
    {"Polearm",     1000, TRUE,  10, UNIQUE_POLEARM},
    {"Spear",       1000, TRUE,  10, UNIQUE_SPEAR},
    {"Lance",       1000, TRUE,  10, UNIQUE_LANCE},

    {"Shield",      2000, FALSE, 10, UNIQUE_SHIELD},
    {"Ring Of Regeneration", 1500, FALSE, 10, UNIQUE_R_REGEN},
/*    {"Boots Of Silence",    1500, FALSE, 10, UNIQUE_BOOTS},
    {"Wings Of Winds",  1000, FALSE, 10, UNIQUE_WINGS},
    {"Ring Of Invisibility",1000, FALSE, 10, UNIQUE_R_INVIS},
    {"Helm Of True Sight",  1500, FALSE, 120, UNIQUE_HELM},
    {"Leggins Of Speed",    1500, FALSE, 10, UNIQUE_LEGGINS},
*/
    {NULL}
};

struct qcmd_t 
{
    char *name;
    void (*do_fn)(CHAR_DATA *ch, char* argument);
    int min_position;
    int extra;
};
typedef struct qcmd_t qcmd_t;

#define COMMAND_RUS (B)

qcmd_t qcmd_table[] = {
    { "points",     quest_points,   POS_DEAD,       COMMAND_KEEP_HIDE},
    { "",       quest_points,   POS_DEAD,       COMMAND_KEEP_HIDE | COMMAND_RUS},
    { "info",       quest_info,     POS_DEAD,       COMMAND_KEEP_HIDE},
    { "",       quest_info,     POS_DEAD,       COMMAND_KEEP_HIDE | COMMAND_RUS},
    { "time",       quest_time,     POS_DEAD,       COMMAND_KEEP_HIDE},
    { "",      quest_time,     POS_DEAD,       COMMAND_KEEP_HIDE | COMMAND_RUS},
    { "list",       quest_list,     POS_RESTING,    0},
    { "",     quest_list,     POS_RESTING,    COMMAND_RUS},
    { "buy",        quest_buy,      POS_RESTING,    0},
    { "",     quest_buy,      POS_RESTING,    COMMAND_RUS},
    { "request",    quest_request,  POS_RESTING,    0},
    { "",      quest_request,  POS_RESTING,    COMMAND_RUS},
    { "complete",   quest_complete, POS_RESTING,    0},
    { "",   quest_complete, POS_RESTING,    COMMAND_RUS},
    { "trouble",    quest_trouble,  POS_RESTING,    0},
    { "",   quest_trouble,  POS_RESTING,    COMMAND_RUS},
    { "cance",      quest_nocanc,   POS_RESTING,    COMMAND_KEEP_HIDE},
    { "",    quest_nocanc,   POS_RESTING,    COMMAND_KEEP_HIDE | COMMAND_RUS},
    { "cancel",     quest_canc,     POS_RESTING,    COMMAND_KEEP_HIDE},
    { "",   quest_canc,     POS_RESTING,    COMMAND_KEEP_HIDE | COMMAND_RUS},
    { "unique",     uniq_list,      POS_RESTING,    0},
    { "",       uniq_list,      POS_RESTING,    COMMAND_RUS},
    { "gained",     quest_gained,   POS_RESTING,    0},
    { "", quest_gained,   POS_RESTING,    COMMAND_RUS},
    { "upgrade",    quest_unique_upgrade,POS_RESTING,    0},
    { "",   quest_unique_upgrade,POS_RESTING,    COMMAND_RUS},
    { NULL}
};

char arg1[MAX_INPUT_LENGTH];

/*
 * The main quest function
 */
DO_FUN(do_quest)
{
    char cmd[MAX_INPUT_LENGTH];
    char arg[MAX_INPUT_LENGTH];
    qcmd_t *qcmd;

    argument = one_argument(argument, cmd, sizeof(cmd));
    argument = one_argument(argument, arg, sizeof(arg));
    argument = one_argument(argument, arg1, sizeof(arg1));

    if (IS_NPC(ch))
        return;

    for (qcmd = qcmd_table; qcmd->name != NULL; qcmd++)
        if (str_prefix(cmd, qcmd->name) == 0) 
        {
            if (ch->position < qcmd->min_position) 
            {
                char_act("In your dreams, or what?", ch);
                return;
            }
            if (!IS_SET(qcmd->extra, COMMAND_KEEP_HIDE)
            &&  IS_SET(ch->affected_by, AFF_HIDE | AFF_FADE | AFF_TRAP)) 
            {
                char_act("You step out of shadows.", ch);
                REMOVE_BIT(ch->affected_by, AFF_HIDE | AFF_FADE | AFF_TRAP);
                act("$n steps out of shadows.", ch, NULL, NULL, TO_ROOM);
            }
            qcmd->do_fn(ch, arg);
            return;
        }

    char_act("{GQUEST COMMANDS{x:", ch);

    for (qcmd = qcmd_table; qcmd->name != NULL; qcmd++)
        if (IS_SET(qcmd->extra, COMMAND_RUS))
            act_puts("($t) ", ch, qcmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);
        else
            act_puts("$t ", ch, qcmd->name, NULL, TO_CHAR | ACT_NOLF | ACT_NOUCASE, POS_DEAD);

    char_act(".\n   : {Chelp quest{x.", ch);
}

void quest_handle_death(CHAR_DATA *ch, CHAR_DATA *victim)
{
    if (IS_AFFECTED(ch, AFF_CHARM) 
    && ch->master != NULL
    && ch->master == victim->hunter) 
        ch = ch->master;

    if (victim->hunter != NULL) {
        if (victim->hunter == ch) {
            char_act("You have almost completed your QUEST!", ch); //T
            char_act("Return to Questor before time is out!", ch); //T
            ch->pcdata->questmob = -1;
        }
        else {
            char_act("You have completed someone's quest.", ch);//T

            ch = victim->hunter;
            char_act("Someone's completed your quest.", ch); //T
            quest_cancel(ch);
            ch->pcdata->questtime = -number_range(5, 10);
        }
    }
}

void quest_cancel(CHAR_DATA *ch)
{
    CHAR_DATA *fch;

    /*
     * remove mob->hunter
     */
    for (fch = char_list; fch; fch = fch->next)
        if (fch->hunter == ch) {
            fch->hunter = NULL;
            break;
        }

    if (IS_NPC(ch)) {
        bug("quest_cancel: called for NPC", 0);
        return;
    }

    ch->pcdata->questtime = 0;
    ch->pcdata->questgiver = 0;
    ch->pcdata->questmob = 0;
    ch->pcdata->questobj = 0;
    ch->pcdata->questroom = NULL;
}

/*
 * Called from update_handler() by pulse_area
 */
void quest_update(void)
{
    CHAR_DATA *ch, *ch_next;

    for (ch = char_list; ch != NULL; ch = ch_next) {
        ch_next = ch->next;

        if (IS_NPC(ch))
            continue;
        if (ch->pcdata->questtime < 0) {
            if (++ch->pcdata->questtime == 0) {
                char_act("{*     .", ch);
                return;
            }
        } else if (IS_ON_QUEST(ch)) {
            if (--ch->pcdata->questtime == 0) {
                char_act(",      !", ch);
                quest_cancel(ch);
                ch->pcdata->questtime = -number_range(5, 10);
            } else if (ch->pcdata->questtime < 6) {
                char_act(" ,       !", ch);
                return;
            }
        }
    }
}

void qtrouble_set(CHAR_DATA *ch, int vnum, int count)
{
    qtrouble_t *qt;

    if ((qt = qtrouble_lookup(ch, vnum)) != NULL)
        qt->count = count;
    else {
        qt = malloc(sizeof(*qt));
        qt->vnum = vnum;
        qt->count = count;
        qt->next = ch->pcdata->qtrouble;
        ch->pcdata->qtrouble = qt;
    }
}

/*
 * local functions
 */

static void quest_tell2(CHAR_DATA *ch, CHAR_DATA *questor, const char *fmt, const void *arg, const void *arg3)
{
    char buf[MAX_STRING_LENGTH];
    
    strnzcpy(buf, sizeof(buf), act_speech2(questor, ch, GETMSG(fmt, ch->lang), arg, arg3));

    do_tell_raw(questor, ch, buf);
}

CHAR_DATA* questor_lookup(CHAR_DATA *ch)
{
    CHAR_DATA *vch;
    CHAR_DATA *questor = NULL;

    for (vch = ch->in_room->people; vch != NULL; vch = vch->next_in_room) {
        if (!IS_NPC(vch))
            continue;
        if (IS_SET(vch->pIndexData->act, ACT_QUESTOR)) {
            questor = vch;
            break;
        }
    }

    if (questor == NULL)
    {
        char_act("You must find questor for this.", ch); //T
        return NULL;
    }

    if (questor->fighting != NULL)
    {
        char_act("Wait until the fighting stops.", ch); //T
        return NULL;
    }

    return questor;
}

qtrouble_t *qtrouble_lookup(CHAR_DATA *ch, int vnum)
{
    qtrouble_t *qt;

    for (qt = ch->pcdata->qtrouble; qt != NULL; qt = qt->next)
        if (qt->vnum == vnum)
            return qt;

    return NULL;
}

/*
 * quest do functions
 */

static void quest_points(CHAR_DATA *ch, char* argument)
{
    char_printf(ch, "   {W%d{x  .\n",
            ch->pcdata->questpoints);
}

static void quest_info(CHAR_DATA *ch, char* argument)
{
    if (!IS_ON_QUEST(ch)) {
        char_act("You have no quest right now.", ch); //T
        return;
    }

    if (ch->pcdata->questmob == -1) {
        char_act("   !\n      !", ch);
        return;
    }

    if (ch->pcdata->questobj > 0) {
        OBJ_INDEX_DATA *qinfoobj;

        qinfoobj = get_obj_index(ch->pcdata->questobj);
        if (qinfoobj != NULL) {
            char_printf(ch, "   -   {W%s{x!\n",
                    qinfoobj->name);
            if (ch->pcdata->questroom)
                char_printf(ch, "      {W%s{x  {W%s{x.\n",
                        ch->pcdata->questroom->area->name,
                        mlstr_mval(ch->pcdata->questroom->name));
        }
        else
            char_act("  .", ch);
        return;
    }

    if (ch->pcdata->questmob > 0) {
        MOB_INDEX_DATA *questinfo;

        questinfo = get_mob_index(ch->pcdata->questmob);
        if (questinfo != NULL) {
            char_printf(ch, " :  {W%s{x!\n",
                    mlstr_mval(questinfo->short_descr));
            if (ch->pcdata->questroom)
                char_printf(ch, "  -   {W%s{x  {W%s{x.\n",
                        ch->pcdata->questroom->area->name,
                        mlstr_mval(ch->pcdata->questroom->name));
        } else
            char_act("    .", ch);
        return;
    }
}

static void quest_time(CHAR_DATA *ch, char* argument)
{
    if (!IS_ON_QUEST(ch)) {
        char_act("    .", ch);
        if (ch->pcdata->questtime < -1)
            act_puts(" {W$j{x $qj{}        .", ch, (const void *) (-ch->pcdata->questtime), NULL, TO_CHAR, POS_DEAD);
            else if (ch->pcdata->questtime == -1)
            char_act("    ,      .", ch);
    }
    else
        act_puts(",    : {W$j{x $qj{}.", ch, (const void *) ch->pcdata->questtime, NULL, TO_CHAR, POS_DEAD);
}

static void quest_list(CHAR_DATA *ch, char *argument)
{
    CHAR_DATA *questor;
    qitem_t *qitem;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    act("$n   $c1{$N}   .", ch, NULL, questor, TO_ROOM);
    act_puts("   $c1{$N}   .", ch, NULL, questor, TO_CHAR, POS_DEAD);

    char_act("{R     {x:", ch);
    for (qitem = qitem_table; qitem->name; qitem++) {

        if (argument[0] != '\0' && !is_name(argument, qitem->name))
            continue;
        if (qitem->price == 0)
            char_printf(ch, " **** qp{Y...........{x%s\n",
            qitem->name);
        else
            char_printf(ch, "%5d qp{Y...........{x%s\n",
                qitem->price, qitem->name);
    }
    char_act("  ,  '{CQUEST BUY <item>{x'.", ch);
    char_act("  UNIQUE,  '{CQUEST BUY UNIQUE <item>{x'.", ch);
    char_act("{YQUEST UNIQUE{x    .", ch);
}

static void quest_buy(CHAR_DATA *ch, char *argument)
{

    CHAR_DATA *questor;
    qitem_t *qitem;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    if (IS_AFFECTED(ch, AFF_CHARM))
    { 
        char_act("  : '{G    ,      !'", ch);
        return;
    }


    if (argument[0] == '\0') {
        char_act("  ,  '{CQUEST BUY <item>{x'.", ch);
        return;
    }

    for (qitem = qitem_table; qitem->name; qitem++)
        if (is_name(argument, qitem->name)) {
            bool buy_ok = FALSE;


            if (ch->pcdata->questpoints < qitem->price) {
                quest_tell(ch, questor, "Sorry, {W$i{z, but you don't have enough quest points for that.", ch); //T
                return;
            }

            if (qitem->vnum == -1)
                buy_ok = buy_unique(ch, questor, arg1);
            else
                if (qitem->vnum == 0)
                    buy_ok = qitem->do_buy(ch, questor);
                else
                    buy_ok = quest_give_item(ch, questor,
                        qitem->vnum, 0);

            if (buy_ok)
                ch->pcdata->questpoints -= qitem->price;
            return;
        }

    quest_tell(ch, questor, "   , $i.", ch);
}

#define MAX_QMOB_COUNT 512

static void quest_request(CHAR_DATA *ch, char *argument)
{
    CHAR_DATA *mobs[MAX_QMOB_COUNT];
    size_t mob_count;
    CHAR_DATA *victim = NULL;
    CHAR_DATA *questor;
    CHAR_DATA *vch;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    act("$n asks $N for a quest.", ch, NULL, questor, TO_ROOM); //T
    act_puts("You ask $N for a quest.", ch, NULL, questor, TO_CHAR, POS_DEAD); //T

    if (IS_ON_QUEST(ch)) {
            quest_tell(ch, questor, "But you are already on a quest!", NULL); //T
            return;
    }

    if (ch->pcdata->questtime < 0) {
        quest_tell(ch, questor, "You're very brave, {W$i{z, but let someone else have a chance.", ch); //T
        quest_tell(ch, questor, "Come back later.", NULL); //T
        return;
    }

    if (IS_SET(ch->comm, COMM_NOQUEST))
    {
        char_printf(ch, "     .\n");
        return;
    }

    quest_tell(ch, questor, "Thank you, brave {W$i{z!", ch); //T

    /*
     * find MAX_QMOB_COUNT quest mobs and store their vnums in mob_buf
     */
    mob_count = 0;
    for (victim = char_list; victim; victim = victim->next) {
        int diff = victim->level - ch->level;

        if (!IS_NPC(victim)
        ||  (ch->level < 51 && (diff > 4 || diff < -1))
        ||  (ch->level > 50 && (diff > 6 || diff < 0))
        ||  (ch->level > 91 && (diff < 0))
        ||  victim->pIndexData->pShop
        ||  victim->race == ch->race
        ||  victim->invis_level
        ||  IS_AFFECTED(victim, AFF_CHARM)
        ||  (IS_EVIL(victim) && IS_EVIL(ch))
        ||  (IS_GOOD(victim) && IS_GOOD(ch))
        ||  victim->pIndexData->vnum < 100
        ||  IS_SET(victim->pIndexData->act,
               ACT_TRAIN | ACT_PRACTICE | ACT_HEALER |
               ACT_NOTRACK | ACT_PET | ACT_NOQUEST)
        ||  IS_SET(victim->pIndexData->immunes, IMM_SUMMON)
        ||  questor->pIndexData == victim->pIndexData
        ||  victim->in_room == NULL
        ||  (IS_SET(victim->pIndexData->act, ACT_SENTINEL) &&
             IS_SET(victim->in_room->room_flags,
                ROOM_PRIVATE | ROOM_SOLITARY))
        ||  !str_cmp(victim->in_room->area->name,
                 hometown_name(ch->hometown))
        ||  IS_SET(victim->in_room->area->flags,
               AREA_HOMETOWN | AREA_NOQUEST))
            continue;
        mobs[mob_count++] = victim;
        if (mob_count >= MAX_QMOB_COUNT)
            break;
    }

    if (mob_count == 0) {
        quest_tell(ch, questor, ",        .", NULL);
        quest_tell(ch, questor, "Come back later.", NULL); //T
        ch->pcdata->questtime = -5;
        return;
    }

    for (;;)
    {
        bool target = FALSE;

        victim = mobs[number_range(0, mob_count-1)];
        for (vch = char_list; vch; vch = vch->next)
        if (!IS_NPC(vch) && vch->desc
            && vch->pcdata->questmob == victim->pIndexData->vnum)
        {
            target = TRUE;
        }

        if (!target)
        break;
    }

    ch->pcdata->questroom = victim->in_room;

    if (chance(40)) { /* Quest to find an obj */
        OBJ_DATA *eyed;
        int obj_vnum;

        obj_vnum = number_range(QUEST_OBJ_FIRST, QUEST_OBJ_LAST);
        eyed = create_named_obj(get_obj_index(obj_vnum), ch->level,
                       ch->name);
        eyed->owner = str_dup(ch->name);
        eyed->altar = get_altar(ch);
        eyed->level = ch->level;
        eyed->ed = ed_new2(eyed->pIndexData->ed, ch->name);
        eyed->cost = 0;
        eyed->timer = 30;

        obj_to_room(eyed, victim->in_room);
        ch->pcdata->questobj = eyed->pIndexData->vnum;

        quest_tell(ch, questor, "    {W$p{z   !", eyed);
        quest_tell(ch, questor, " ,    ,    .", NULL);
        quest_tell2(ch, questor, " -   {W$t{z  {W$R{z!", victim->in_room->area->name, victim->in_room);
    } else 
    {  /* Quest to kill a mob */
        if (IS_GOOD(ch)) 
        {
            quest_tell(ch, questor, 
                "Rune's most heinous criminal, {W$t{z, has escaped from the dungeon.", 
                    victim->name);
            quest_tell2(ch, questor, 
                "Since the escape, {W$t{z has murdered {W$J{z $qJ{civilians}!", 
                    victim->name, (const void *) (number_range(2, 20)));
            quest_tell(ch, questor, 
                "The penalty for this crime is death, and you are to deliver the sentence!", 
                    NULL);
        }  else 
        {
            quest_tell(ch, questor, 
                "  , {W$t{G,   .", victim->name);
            quest_tell(ch, questor, "     !", NULL);
        }

        quest_tell2(ch, questor, "  {W$t{G  {W$R{z!",
            victim->name, victim->in_room);
        quest_tell(ch, questor, "  -   {W$t{z.",
            victim->in_room->area->name);

        ch->pcdata->questmob = victim->pIndexData->vnum;
        victim->hunter = ch;
    }

    ch->pcdata->questgiver = questor->pIndexData->vnum;
    ch->pcdata->questtime = number_range(10, 20) + ch->level/10;
    ch->pcdata->questnumber++;
    quest_tell(ch, questor, 
        "  {W$j{z $qj{}   .", 
            (const void *) ch->pcdata->questtime);
    quest_tell(ch, questor, "    {W{z!", NULL);
    ++stat_record.quest_requested;
}

static void quest_complete(CHAR_DATA *ch, char *argument)
{
    bool complete = FALSE;
    CHAR_DATA *questor;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;

    int gold_reward = 0;
    int qp_reward = 0;
    int prac_reward = 0;
    int exp_reward = 0;
    int hp_reward = 0;
    int mana_reward = 0;
    int bp_reward = 0;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    act("$n  $c2{$N}   .", ch, NULL, questor, TO_ROOM);
    act_puts("  $c2{$N}   .", ch, NULL, questor, TO_CHAR, POS_DEAD);

    if (!IS_ON_QUEST(ch)) 
    {
        quest_tell(ch, questor, "  , {W$i{z.", ch);
        return;
    }

    if (ch->pcdata->questgiver != questor->pIndexData->vnum) 
    {
        quest_tell(ch, questor, "I never sent you on a quest! Perhaps you're thinking of someone else.", NULL); //T
        return;
    }

    if (ch->pcdata->questobj > 0)
        for (obj = ch->carrying; obj != NULL; obj = obj_next) 
        {
            obj_next = obj->next_content;

            if (obj != NULL
            &&  obj->pIndexData->vnum == ch->pcdata->questobj
                        &&  obj->ed != NULL
            &&  strstr(mlstr_mval(obj->ed->description),
                            ch->name) != NULL) {
                act_puts("You pass the {W$p{x into $N's hands.", ch, obj, questor, TO_CHAR, POS_DEAD); //T
                act("$n pass {W$p{x into $N's hands.", ch, obj, questor, TO_ROOM); //T
                extract_obj(obj);

                if (chance(10))
                    prac_reward = number_range(1, 4);
                qp_reward = number_range(5, 15) + ch->pcdata->questtime;
                gold_reward = number_range(1, 10);

                complete = TRUE;
                break;
            }
        }
    else if (ch->pcdata->questmob == -1) 
    {
        if (chance(2))
            prac_reward = number_range(1, 6);
        qp_reward = number_range(10, 20) + ch->pcdata->questtime;
        gold_reward = number_range(1, 10);
        gold_reward = URANGE(0, gold_reward, ch->level/9);
        complete = TRUE;
    }

    if (!complete) 
    {
        quest_tell(ch, questor, "You haven't completed the quest yet, but there is still time!", NULL); //T
        return;
    }

    gold_reward = areacontrol_modify_gain (ch, gold_reward, QUEST_REWARD_GOLD);
    qp_reward   = areacontrol_modify_gain (ch, qp_reward,   QUEST_REWARD_QP  );

    ch->gold += gold_reward;
    ch->pcdata->questpoints += qp_reward;
    ch->pcdata->qp_earned += qp_reward;
    ch->pcdata->questcomplete++;


    quest_tell(ch, questor,  "Congratulations on completing your quest!", NULL); //T
    quest_tell2(ch, questor, "As reward you receive {R$j {G$qj{quest points} and {R$J {Y$qJ{gold coins}{x.", //T
            (const void *) qp_reward, (const void *) gold_reward);

    if (prac_reward) 
    {
        ch->practice += prac_reward;
        quest_tell(ch, questor, "  {R$j{x $qj{} !", (const void *) prac_reward);
    }

    if (IS_SET (muddy_mode, MUDDY_QUEST_NEW))
    {
        // gain bp
        if (chance (5))
        {
            bp_reward = 1; //number_range (1, 2);
            ch->pcdata->bonuspoints += bp_reward;
            ch->pcdata->bp_earned += bp_reward;
            quest_tell(ch, questor, "{GYou receive {R$j{G $qj{bonus points}!", (const void *) bp_reward);
        }

        // gain experience
        if (chance (90))
        {
            exp_reward = number_range (LVL (ch) * 2, LVL (ch) * 4);
            gain_exp (ch, exp_reward) ;
            quest_tell(ch, questor, "{GYou receive {R$j{G $qj{experience points}!", (const void *) exp_reward);
        }

        // gain hp
        if (chance (8) && ch->pcdata->quest_hp_gained < quest_hp_max (ch))
        {
            hp_reward = UMIN (number_range (1, 3),  quest_hp_max (ch) - ch->pcdata->quest_hp_gained);            
            ch->pcdata->perm_hit += hp_reward;
            ch->max_hit += hp_reward;
            ch->hit += hp_reward;
            ch->pcdata->quest_hp_gained += hp_reward;
            quest_tell(ch, questor, "{GYou receive {R$j{G hp!", (const void *) hp_reward);
        }

        // gain mana
        if (chance (8) && ch->pcdata->quest_mana_gained < quest_mana_max (ch))
        {
            mana_reward = UMIN (number_range (1, 3),  quest_mana_max (ch) - ch->pcdata->quest_mana_gained);            
            ch->pcdata->perm_mana += mana_reward;
            ch->max_mana += mana_reward;
            ch->mana += mana_reward;
            ch->pcdata->quest_mana_gained += mana_reward;
            quest_tell(ch, questor, "{GYou receive {R$j{G mana!", (const void *) mana_reward);
        }
    }

    quest_cancel(ch);
    ch->pcdata->questtime = -number_range(4, 10);
    ++stat_record.quest_completed;
    stat_record.gold_quest_reward += gold_reward;
}

static void quest_trouble(CHAR_DATA *ch, char *argument)
{
    CHAR_DATA *questor;
    qitem_t *qitem;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    if (argument[0] == '\0') 
    {
        char_act("     , : 'quest trouble <award>'.", ch);
        return;
    }

    if (is_name(argument, "unique")) 
    {
        quest_unique_trouble(ch, questor, arg1);
        return;
    }

    for (qitem = qitem_table; qitem->name; qitem++) 
    {
        if (qitem->vnum == QUEST_VNUM_RUG
        || qitem->vnum == VNUM_CANTEEN
        || qitem->vnum == OBJ_VNUM_SCROLL_SCRIBING)
            continue;

        if (qitem->vnum && is_name(argument, qitem->name)) 
        {
            quest_give_item(ch, questor, qitem->vnum, TROUBLE_MAX);
            return;
        }
    }

    quest_tell(ch, questor, ", {W$i{z,       .", ch);
}

static void quest_nocanc(CHAR_DATA *ch, char *argument)
{
    if (!IS_ON_QUEST(ch))
        char_act("    .", ch);
    else
        char_act("     cancel.", ch);
}

static void quest_canc(CHAR_DATA *ch, char *argument)
{
    int nextquest = 0;
    CHAR_DATA *questor;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    if (!IS_ON_QUEST(ch))
        char_act("    .", ch);
    else 
    {
        act("$n   $N  .", ch, NULL, questor, TO_ROOM);
        act_puts("   $N  .", ch, NULL, questor, TO_CHAR, POS_DEAD);
        quest_tell(ch, questor, ",      , {W$i{x.", ch);
        nextquest = -abs(ch->pcdata->questtime);
        quest_cancel(ch);
        ch->pcdata->questtime = nextquest;
    }
}

/*
 * quest buy functions
 */

static bool quest_give_item(CHAR_DATA *ch, CHAR_DATA *questor,
                int item_vnum, int count_max)
{
    OBJ_DATA *reward;
    qtrouble_t *qt;
    OBJ_INDEX_DATA *pObjIndex = get_obj_index(item_vnum);

    /*
     * check quest trouble data
     */

    qt = qtrouble_lookup(ch, item_vnum);

    if (count_max) 
    {
        /*
         * 'quest trouble'
         */

        if (qt == NULL) 
        {
            /* ch has never bought this item, but requested it */
            quest_tell(ch, questor, ", {W$i{z,       .", ch);
            return FALSE;
        }
        else if (qt->count > count_max 
        || !IS_SET(pObjIndex->extra_flags, ITEM_QUEST)) 
        {
            /* ch requested this item too many times    *
             * or the item is not quest         */
             quest_tell(ch, questor, "    .", NULL);
            return FALSE;
        }
    }
    else {
        /*
         * 'quest buy'
         */
        if (item_vnum != QUEST_VNUM_RUG
        && item_vnum != VNUM_CANTEEN
        && item_vnum != OBJ_VNUM_SCROLL_SCRIBING)
        {
            if (qt && qt->count <= TROUBLE_MAX)
            {
                quest_tell(ch, questor, "   $gn{}  .", NULL);
                return FALSE;
            }
        }
    }

    reward = create_obj(pObjIndex, 0);

    if (ch->level < reward->level) 
    {
        quest_tell(ch, questor, "     .", NULL);
        extract_obj(reward);
        return FALSE;
    }

    /* update quest trouble data */

    if (qt != NULL && count_max) 
    {
        OBJ_DATA *obj;
        OBJ_DATA *obj_next;

        /* `quest trouble' */
        for (obj = object_list; obj != NULL; obj = obj_next) 
        {
            obj_next = obj->next;
            if (obj->pIndexData->vnum == item_vnum
            && !str_cmp(obj->owner, ch->name)) 
            {
                extract_obj(obj);
                break;
            }
        }

        quest_tell(ch, questor, "  $j     .", 
            (const void *) qt->count);
        if (qt->count > count_max)
            quest_tell(ch, questor, "        .", NULL);
    }

    if (qt == NULL) 
    {
        qt = malloc(sizeof(*qt));
        qt->vnum = item_vnum;
        qt->count = 0;
        qt->next = ch->pcdata->qtrouble;
        ch->pcdata->qtrouble = qt;
    }

    if (item_vnum != QUEST_VNUM_RUG
    || item_vnum != VNUM_CANTEEN
    || item_vnum != OBJ_VNUM_SCROLL_SCRIBING)
    {
        if (count_max)
            qt->count++;
        else
            qt->count = 1;
    }
    /* ok, give him requested item */

    if (IS_SET(pObjIndex->extra_flags, ITEM_QUEST)) 
    {
        reward->owner = str_dup(ch->name);
        mlstr_free(reward->short_descr);
        reward->short_descr =
            mlstr_printf(reward->pIndexData->short_descr,
                     IS_GOOD(ch) ?  "holy" :
                     IS_NEUTRAL(ch) ?   "blue-green" :
                            "evil",
                     ch->name);
    }

    obj_to_char(reward, ch);

    act("$N  $n {W$p{x.", ch, reward, questor, TO_ROOM);
    act_puts("$N   {W$p{x.", ch, reward, questor, TO_CHAR, POS_DEAD);

    return TRUE;
}

static bool buy_gold(CHAR_DATA *ch, CHAR_DATA *questor)
{
    ch->pcdata->bank_g += 5000;
    act("$n   $N {R5000{x  .", 
        ch, NULL, questor, TO_ROOM);
    act("$N  {R5000{x       .\n",
        ch, NULL, questor, TO_CHAR);
    return TRUE;
}

static bool buy_prac(CHAR_DATA *ch, CHAR_DATA *questor)
{
    ch->practice += 60;
    act("$N  $n {R60{x {W {x.", 
        ch, NULL, questor, TO_ROOM);
    act_puts("$N   {R60{x {W {x.", 
        ch, NULL, questor, TO_CHAR, POS_DEAD);
    return TRUE;
}

static bool buy_tattoo(CHAR_DATA *ch, CHAR_DATA *questor)
{
    OBJ_DATA *tattoo;
    religion_t *religion;

    religion = religion_lookup(ch->religion);

    if (!ch->religion || religion == NULL) 
    {
        char_act("       .", ch);
        return FALSE;
    }

    tattoo = get_eq_char(ch, WEAR_TATTOO);
    if (tattoo != NULL) 
    {
        char_act("     !", ch);
        return FALSE;
    }

    if (!IS_NPC(questor))
    {
        char_act("PC-questor is a bug. Report to immortals.", ch);
        return FALSE;
    }

    if (religion->tattoo_master_vnum != 0
    && questor->pIndexData->vnum != religion->tattoo_master_vnum) 
    {
        act_say(questor, "Sorry, I know nothing about tattoo of $t.", 
            religion_name(ch->religion));
        return FALSE;
    }

    tattoo = create_obj(get_obj_index(RELIGION(ch->religion)->tattoo_vnum), 0);

    obj_to_char(tattoo, ch);
    equip_char(ch, tattoo, WEAR_TATTOO);
    act("$N    $n  {W$p{x!", ch, tattoo, questor, TO_ROOM);
    act_puts("$N      {W$p{x!", ch, tattoo, questor, TO_CHAR, POS_DEAD);
    return TRUE;
}

static bool buy_death(CHAR_DATA *ch, CHAR_DATA *questor)
{
    if (ch->pcdata->death < 1) 
    {
        quest_tell(ch, questor, "Sorry, {W$i{z, but you haven't got any deaths yet.", ch);
        return FALSE;
    }

    ch->pcdata->death -= 1;
    return TRUE;
}

static bool buy_katana(CHAR_DATA *ch, CHAR_DATA *questor)
{
    AFFECT_DATA af;
    OBJ_DATA *katana;

    if (!HAS_SKILL(ch, gsn_katana))
    {
        quest_tell(ch, questor, "Sorry, {W$i{z, but you're not the samurai.", ch);
        return FALSE;
    }

    for (katana = ch->carrying; katana; katana = katana->next_content)
        if (is_name("katana", katana->name)
        && !IS_WEAPON_STAT(katana, WEAPON_KATANA))
            break;

    if (katana == NULL) 
    {
        quest_tell(ch, questor, ", {W$i{z,        .", ch);
        return FALSE;
    }

    af.where        = TO_WEAPON;
    af.type         = gsn_katana;
    af.level        = 100;
    af.duration     = -1;
    af.modifier     = ch->level/10;
    af.bitvector    = WEAPON_KATANA;
    af.location     = APPLY_NONE;
    affect_to_obj(katana, &af);
    katana->owner = str_dup(ch->name);
    quest_tell(ch, questor, "   ,      !", NULL);
    return TRUE;
}

static bool buy_vampire(CHAR_DATA *ch, CHAR_DATA *questor)
{
    if (ch->class != CLASS_VAMPIRE)
    {
        quest_tell(ch, questor, ", {W$i{z,       .", ch);
        return FALSE;
    }
    
    set_skill(ch, gsn_vampire, 100);
    act("$N   $c1{$n},  $gn{} .", 
        ch, NULL, questor, TO_ROOM);
    act_puts("$N    ,  .", 
        ch, NULL, questor, TO_CHAR, POS_DEAD);
    act("   .", 
        ch, NULL, NULL, TO_ALL);
    return TRUE;
}

static void uniq_list(CHAR_DATA *ch, char *argument)
{
    CHAR_DATA *questor;
    uniq_t *uitem;

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    act("$n   $N   .", ch, NULL, questor, TO_ROOM);
    act_puts("   $N   .", ch, NULL, questor, TO_CHAR, POS_DEAD);

    char_act("{R     {x:", ch);
    for (uitem = uniq_table; uitem->name; uitem++) 
    {
        if (uitem->level > ch->level)
            continue;

            if (argument[0] != '\0' && !is_name(argument, uitem->name))
                if (!is_name(argument, "list"))
                    continue;

        char_printf(ch, "%5d qp{Y...........{BUnique {W%s{x\n",
        uitem->price, uitem->name);
    }
    char_act("  ,  '{CQUEST BUY UNIQUE <item>{x'.", ch);
}


/* Selling Unique Weapon */

static bool buy_unique(CHAR_DATA *ch, CHAR_DATA *questor, char *argument)
{
    uniq_t *uniq;
    qtrouble_t *qt;
    
    if (argument[0] == '\0') 
    {
        char_act("  Unique,  '{CQUEST BUY UNIQUE <item>{x'.", ch);
        return FALSE;
    }

    for (uniq = uniq_table; uniq->name; uniq++)
        if (is_name(argument, uniq->name)) 
        {
            if (uniq->level > ch->level) 
            {
                quest_tell(ch, questor, "    {W$j{z .", (const void *) uniq->level);
                return FALSE;
            }

            if (ch->pcdata->questpoints < uniq->price) 
            {
                quest_tell(ch, questor, "Sorry, {W$i{z, but you don't have enough quest points for that.", ch);
                return FALSE;
            }
            
            qt = qtrouble_lookup(ch, uniq->vnum);

            if (qt != NULL) 
            {
                quest_tell(ch, questor, "     .", NULL);
                return FALSE;
            }

            if (uniq->is_weap)
            {
                if (make_uniq_weapon(ch, questor, uniq->vnum, ch->level)) 
                {
                    ch->pcdata->questpoints -= uniq->price;
                    qtrouble_set(ch, uniq->vnum, ch->level);
                    /* save level on which we give uniq */
                    return TRUE;
                } else
                {
                    return FALSE;
                }
            }
            else
            {
                if (make_uniq_object(ch, questor, uniq->vnum, ch->level)) 
                {
                    ch->pcdata->questpoints -= uniq->price;
                    qtrouble_set(ch, uniq->vnum, ch->level);
                    return TRUE;
                } else
                {
                    return FALSE;
                }
            }
        }
    quest_tell(ch, questor, "   , {W$i{z.", ch);
    return FALSE;
}

/* Making unique weapon and applying stats depending on char level */

static bool make_uniq_weapon(CHAR_DATA *ch, CHAR_DATA *questor, int item_vnum, int level)
{
    OBJ_DATA *weapon;
    AFFECT_DATA af;

    if (!get_obj_index(item_vnum)) 
    {
        char_act("   .", ch);
        log_printf("warning unique weapon VNum doesn't exist.\n");
        return FALSE;
    }

    weapon = create_obj(get_obj_index(item_vnum), ch->level);

    if (weapon->pIndexData->item_type != ITEM_WEAPON) 
    {
        char_act("   .", ch);
        log_printf("warning unique weapon wrong type.\n");
        extract_obj(weapon);
        return FALSE;
    }

    af.where        = TO_OBJECT;
    af.type         = gsn_unique;
    af.level        = level;
    af.duration     = -1;
    af.location     = APPLY_HITROLL;
    af.modifier     = level / 4;
    af.bitvector    = 0;
    affect_to_obj(weapon,&af);

    af.location     = APPLY_DAMROLL;
    af.modifier     = level / 5;
    affect_to_obj(weapon,&af);

    weapon->level       = ch->level;
    weapon->value[1]    = level / 6;
    weapon->value[2]    = 9;
    weapon->owner       = str_dup(ch->name);
    mlstr_free(weapon->short_descr);
    weapon->short_descr = mlstr_printf(weapon->pIndexData->short_descr, ch->name);

    obj_to_char(weapon, ch);

    act("$N  $n {W$p{x.", ch, weapon, questor, TO_ROOM);
    act_puts("$N   {W$p{x.", ch, weapon, questor, TO_CHAR, POS_DEAD);

    return TRUE;
}

/* Making unique obj and applying stats depending on char level */

static bool make_uniq_object(CHAR_DATA *ch, CHAR_DATA *questor, int item_vnum, int level)
{
    OBJ_DATA *object;
        AFFECT_DATA af;
    int i;

    if (!get_obj_index(item_vnum)) 
    {
        char_act("   .", ch);
        log_printf("warning unique object VNum doesn't exist.\n");
        return FALSE;
    }

    object = create_obj(get_obj_index(item_vnum), ch->level);

    if (object->pIndexData->item_type != ITEM_ARMOR) 
    {
        char_act("   .", ch);
        log_printf("warning unique object wrong type.\n");
        extract_obj(object);
        return FALSE;
    }

    af.where        = TO_OBJECT;
    af.type         = gsn_unique;
    af.level        = level;
    af.duration     = -1;
    af.bitvector    = 0;

    if (item_vnum == UNIQUE_SHIELD) 
    {
        af.location = APPLY_HITROLL;
        af.modifier = level / 4;
        affect_to_obj(object,&af);

        af.location = APPLY_DAMROLL;
        af.modifier = level / 5;
        affect_to_obj(object,&af);

        af.modifier = -10;
    }
    else
        af.modifier = -5;
    af.location  = APPLY_SAVING_SPELL;
    affect_to_obj(object,&af);

    af.location = APPLY_HIT;
    af.modifier = level;
    affect_to_obj(object,&af);

    af.location = APPLY_MANA;
    affect_to_obj(object,&af);

    af.location = APPLY_MOVE;
    affect_to_obj(object,&af);

    object->level = ch->level;

    for( i=0; i<4; i++ )
        object->value[i] = level;

    object->owner = str_dup(ch->name);
    mlstr_free(object->short_descr);
    object->short_descr = mlstr_printf(object->pIndexData->short_descr, ch->name);

    obj_to_char(object, ch);
    act("$N  $n {W$p{x.", ch, object, questor, TO_ROOM);
    act_puts("$N   {W$p{x.", ch, object, questor, TO_CHAR, POS_DEAD);

    return TRUE;

}
static void quest_unique_trouble(CHAR_DATA *ch, CHAR_DATA *questor, char *argument)
{
    uniq_t *uniq;
    qtrouble_t *qt;
    int item_vnum = 0, level;
    bool weap = FALSE;
    OBJ_DATA *obj;
    OBJ_DATA *obj_next;

    if (argument[0] == '\0')
    {
        char_act("     , : 'quest trouble unique <award>'.", ch);
        return;
    }

    for (uniq = uniq_table; uniq->name; uniq++)
        if (is_name(argument, uniq->name)) 
        {
            item_vnum = uniq->vnum;
            weap = uniq->is_weap;
            break;
        }

    qt = qtrouble_lookup(ch, item_vnum);

    if (qt == NULL) 
    {
        quest_tell(ch, questor, ", {W$i{z,       .", ch);
        return;
    }

    level = qt->count; /*restore level on which we give uniq*/

    if (level < uniq->level) 
    {
            quest_tell(ch, questor, ", {W$i{z,      .", ch);
            return;
    }

    if (ch->pcdata->questpoints < uniq->price / 10) 
    {
        quest_tell(ch, questor, "Sorry, {W$i{z, but you don't have enough quest points for that.",  ch);
        return;
    }

    for (obj = object_list; obj != NULL; obj = obj_next) 
    {
        obj_next = obj->next;
        if (obj->pIndexData->vnum == item_vnum
        && !str_cmp(obj->owner, ch->name)) 
        {
            quest_tell(ch, questor, "{W$i{z,     .", ch);
            return;
        }
    }

    ch->pcdata->questpoints -= uniq->price / 10;

    if (weap)
        make_uniq_weapon(ch, questor, uniq->vnum, ch->level);
    else
        make_uniq_object(ch, questor, uniq->vnum, ch->level);
    return;
}

static void quest_gained (CHAR_DATA *ch, char *argument)
{
    CHAR_DATA * victim;
    bool loaded = FALSE;

    if (!IS_IMMORTAL (ch) || argument[0] == '\0')
    {
        char_act ("As reward for your quests you gained:", ch);
        victim = ch;
    }
    else
    {
        if ((victim = gq_findchar(argument)) == NULL)
        {
            loaded = TRUE;
            victim = char_load_special(argument);
        }
        if (victim == NULL)
        {
            char_act("No chars with such name.", ch) ;
            return;
        }
        if (IS_NPC(victim))
        {
            char_act("Mobiles don't do quests.", ch) ;
            return ;
        }
        char_printf (ch, "As reward for quests %s gained:\n", capitalize (argument));
    }

    char_printf (ch, "{x    {C%d{x of {C%d{x hp,\n", victim->pcdata->quest_hp_gained, quest_hp_max (victim));
    char_printf (ch, "{x    {C%d{x of {C%d{x mana.\n", victim->pcdata->quest_mana_gained, quest_mana_max (victim));

    if (loaded)
        char_nuke(victim);
}

static int quest_hp_max (CHAR_DATA *ch)
{
    race_t  * r;
    class_t * cl;
    int result;

    if (IS_NPC (ch))
        return 0;

    r = RACE(ORG_RACE(ch));

    if ((cl = class_lookup(ch->class)) == NULL)
    {
        log_printf("quest_hp_max: %s has unknown class", ch->name);
        return 0;
    }

    result = (get_curr_stat(ch,STAT_CON) - 20 + r->pcdata->hp_bonus + cl->hp_rate) * ch->level;
    result = URANGE (350 - LEVEL_HERO*4 + ch->level*4, result, 1000);
    return result;
}

static int quest_mana_max (CHAR_DATA *ch)
{
    race_t  * r;
    class_t * cl;
    int result;
    
    if (IS_NPC (ch))
        return 0;

    r = RACE(ORG_RACE(ch));

    if ((cl = class_lookup(ch->class)) == NULL)
    {
        log_printf("quest_hp_max: %s has unknown class", ch->name);
        return 0;
    }

    result = (get_curr_stat(ch,STAT_INT) - 23 + get_curr_stat(ch,STAT_WIS) - 22  + 
            r->pcdata->mana_bonus + cl->mana_rate) * ch->level;
    result = URANGE (400  - LEVEL_HERO*4 + ch->level*4, result, 1100);
    return result;
}

static void quest_unique_upgrade(CHAR_DATA *ch, char *argument)
{
        uniq_t *uniq;
        qtrouble_t *qt;
        int level;
        OBJ_DATA *obj;
        CHAR_DATA *questor;

        if ((questor = questor_lookup(ch)) == NULL)
                return;

        if (argument[0] == '\0') {
                char_act("o  a apa, a: 'quest upgrade <award>'.", ch);
                return;
        }

        uniq = uniq_table;
        while(uniq->name != NULL) {
                if (is_name(argument, uniq->name))
                        break;
                uniq++;
        }

        if (uniq->name == NULL
        || (qt = qtrouble_lookup(ch, uniq->vnum)) == NULL) {
                quest_tell(ch, questor, ", {W$i{z, o  e e o apa.", ch);
                return;
        }

        level = qt->count;

        if (level >= ch->level) {
                quest_tell(ch, questor, "Ho a apaa  cooece oe po.", ch);
                return;
        }

        if (ch->pcdata->questpoints < uniq->price / 10) {
                quest_tell(ch, questor, "Sorry, {W$i{z, but you don't have enough quest points for that.", ch);
                return;
        }

        for(obj = ch->carrying; obj != NULL; obj = obj->next_content)
                if ( obj->pIndexData->vnum == uniq->vnum
                &&   !str_cmp(obj->owner, ch->name) )
                        break;

        if (obj == NULL) {
                quest_tell(ch, questor, ", {W$i{z, o  e $gi{} c coo  .", ch);
                return;
        }

        if (obj->condition < 100) {
                act("$N .", NULL, NULL, questor, TO_ROOM);
                do_say(questor, "       ?!");
        }


        act("$N   $n $p.", ch, obj, questor, TO_ROOM);
        act_puts("$N    $p.", ch, obj, questor, TO_CHAR, POS_DEAD);

        extract_obj(obj);
        ch->pcdata->questpoints -= uniq->price / 10;
        qtrouble_set(ch, uniq->vnum, ch->level);

        if (uniq->is_weap)
                make_uniq_weapon(ch, questor, uniq->vnum, ch->level);
        else
                make_uniq_object(ch, questor, uniq->vnum, ch->level);
        return;
}

/*****************************************************************/
/* wishes */
typedef struct wish_data WISH_DATA;
struct wish_data {
    char *name; // a name of a wish
    char *expl; // long string, which explanes the mean
    int price;  // price in qp
    int flag;   // flag in ch->pcdata->wishes
    int pre_req;// pre-required wish
};

struct wish_data wish_table[] = {
    { "showscry", "You can sense outer powers used towards you.", 20000, WISH_SHOWSCRY, 0},
    { "exprate1", "You receive 20% more exp on kills.", 6000, WISH_EXPRATE1, 0},
    { "exprate2", "You receive 40% more exp on kills.", 12000, WISH_EXPRATE2, WISH_EXPRATE1},
    { "spellup1", "You can cast defensive magic faster.", 3000, WISH_SPELLUP1, 0},
    { "spellup2", "You can cast defensive magic in the fastest way.", 6000, WISH_SPELLUP2, WISH_SPELLUP1},
    { "group   ", "You can ask where your pets and charmises are.", 500, WISH_GROUP, 0},
//  { "duration", "Spell duration 50% higher.", 5000, WISH_DURATION, 0},
//  { "nohunger", "You have no hunger or thirst.", 5000, WISH_NOHUNGER, 0},
    { "learnup1", "You can learn skills faster.", 500, WISH_LEARNUP1, 0},
    { "learnup2", "You can learn skills in the fastest way.", 3000, WISH_LEARNUP2, WISH_LEARNUP1},
    { "saffects", "Allow your use SAFFECTS cheat-command.", 5000, WISH_SAFFECTS, 0},
    { "mindblank", "Learn basic defence from psionic attacks.", 500, WISH_MINDBLANK, 0},
    { NULL }
};

DO_FUN(do_wish)
{
    char arg[MAX_INPUT_LENGTH];
    char arg1[MAX_INPUT_LENGTH];
    WISH_DATA *wish;
    WISH_DATA *pre_wish;
    CHAR_DATA *questor;
    CHAR_DATA *victim;
    int new_align = 0;
    int cost = 0;

    if (IS_NPC(ch))
        return;

    if (IS_AFFECTED(ch, AFF_CHARM)
    || ch->fighting
    || current_time - ch->last_fight_time < 200
    || IS_AFFECTED(ch, AFF_SLEEP)
    || IS_PUMPED(ch))
    {
        char_act("You must be free from charm, not pumped, not sleeping and big last fight time.", ch);
        return;
    }
    // before actual wishes, some shit from PRICE /sg
    if (!str_cmp(argument, "to lose weapon specializations"))
    {
        cost = 500;
        if (cost > ch->pcdata->questpoints)
        {
            char_act("Your do not have enough QP for this services.", ch);
            return;
        }
        ch->pcdata->questpoints -= cost;

        ch->pcdata->specialization[0] = 0;
        ch->pcdata->specialization[1] = 0;
        ch->pcdata->specialization[2] = 0;
        update_special_skills(ch, FALSE);
        ch->pcdata->element = 0;
        char_act("Your weapon specializations reseted.", ch);
        return;
    }
    if (!str_cmp(argument, "to lose my religion"))
    {
        OBJ_DATA *tattoo;
    
        if (ch->religion == 0)
        {
            char_act("You already lost it.", ch);
            return;
        }
        cost = 2000;
        if (cost > ch->pcdata->questpoints)
        {
            char_act("Your do not have enough QP for this services.", ch);
            return;
        }
        ch->pcdata->questpoints -= cost;
        tattoo = get_eq_char(ch, WEAR_TATTOO);
        if (tattoo != NULL)
          extract_obj(tattoo);
        ch->religion = 0;
        char_act("You lost your faith in Gods.", ch);
        return;
    }
    if (!str_cmp(argument, "to escape"))
    {
        ROOM_INDEX_DATA *location;
        location = get_room_index(3030);
        if (location == NULL)
        {
            char_act("You are completely lost.", ch);
            return;
        }
        if (ch->in_room == location)
            return;

        if (!check_war_move (ch, location, FALSE))
            return;

        cost = 600;
        if (cost > ch->pcdata->questpoints)
        {
            char_act("Your do not have enough QP for this services.", ch);
            return;
        }
        ch->pcdata->questpoints -= cost;
        char_from_room(ch);
        char_to_room(ch, location);
        return;
    }
    if ( (!str_cmp(argument, "to become good aligned") ? (new_align = 3) : 0)
    || (!str_cmp(argument, "to become neutral aligned") ? (new_align = 2) : 0)
    || (!str_cmp(argument, "to become evil aligned") ? (new_align = 1) : 0) )
    {
        int price, old_align, ralign;
    
        if (IS_GOOD(ch))
            old_align = 3;
        else if (IS_EVIL(ch))
            old_align = 1;
        else
            old_align = 2;
        if (new_align == old_align)
            return;
        switch (new_align)
        {
            case 1: ralign = RA_EVIL;break;
            case 2: ralign = RA_NEUTRAL;break;
            case 3: ralign = RA_GOOD;break;
            default: ralign = 0; return; break;
        }
        if (!align_ok(ch, ralign))
        {
            char_act("You can't switch to that alignment.", ch);
            return;
        }
        price = 2000 * (new_align - old_align) * 
            ((new_align > old_align) ? 2 : -1);
        if (price > ch->pcdata->questpoints)
        {
            char_act("Your do not have enough QP for this services.", ch);
            return;
        }
        ch->pcdata->questpoints -= price;
        ch->alignment = 1000 * (new_align - 2);
        return;
    } 

    if ((!str_cmp(argument, "to have lawful tendencies") ? (new_align = 3) : 0)
    || (!str_cmp(argument, "to have neutral tendencies") ? (new_align = 2) : 0)
    || (!str_cmp(argument, "to have chaotic tendencies") ? (new_align = 1) : 0))
    {       
        int price, old_align, ralign;
    
        if (ch->ethos == ETHOS_LAWFUL)
            old_align = 3;
        else if (ch->ethos == ETHOS_CHAOTIC)
            old_align = 1;
        else
            old_align = 2;
        if (new_align == old_align)
            return;
        switch (new_align)
        {
            case 1: ralign = ETHOS_CHAOTIC;break;
            case 2: ralign = ETHOS_NEUTRAL;break;
            case 3: ralign = ETHOS_LAWFUL;break;
            default: ralign = 0; return; break;
        }
        if (!ethos_ok(ch, ralign))
        {
            char_act("You can't switch to that ethos.", ch);
            return;
        }
        price = 1000 * (new_align - old_align) * 
            ((new_align > old_align) ? 2 : -1);
        if (price > ch->pcdata->questpoints)
        {
            char_act("Your do not have enough QP for this services.", ch);
            return;
        }
        ch->pcdata->questpoints -= price;
        ch->ethos = ralign;
        return;
    }

    //"wish" command without argument shows a list of 
    // wishes, which a char already has
    if (argument[0] == '\0')
    {
        char_act("{C           You've already wished:{x", ch);
        for (wish = wish_table; wish->name != NULL; wish++)
            if (IS_SET(ch->pcdata->wishes, wish->flag)) {
                char_printf(ch, "{G%-.8s{x: %s\n", wish->name, wish->expl);
        }
        return;
    }

    //"wish" command with argument "info" shows a list of 
    // all available wishes with their prices
    argument = one_argument(argument, arg, sizeof(arg));
    if (!strcmp(arg,"info"))
    {
        char_act("{C           Available wishes{x", ch);
        for (wish = wish_table; wish->name != NULL; wish++)
            char_printf(ch, "{G%-.8s{x : %s (%d qp)\n", wish->name, wish->expl, wish->price);
        return;
    }

    //"wish" command with argument "remove" removes ALL
    // of the wishes from char (second arg = char name)
    if ((!strcmp(arg,"remove")) && (IS_IMMORTAL(ch)))
    {
        argument = one_argument(argument, arg1, sizeof(arg1));
        if (!(victim = get_char_world(ch, arg1)))
        {
            char_printf(ch, "You cannot find your victim!\n");
            return;
        }
        if (IS_NPC(victim))
            return;
        log_printf ("%s clears wishlist of %s", ch->name, victim->name);
        //char_printf (ch, "You clear wishlist of %s", victim->name);
        for (wish = wish_table; wish->name != NULL; wish++)
            if (IS_SET(victim->pcdata->wishes, wish->flag))
            {
                char_printf(victim, "An {CImmortal{x just removed {G%s{x wish from you!\n",
                    wish->name);
                REMOVE_BIT(victim->pcdata->wishes, wish->flag);
            }
        return;
    }

    //trying to find requested wish in wish_table
    for (wish = wish_table; wish->name != NULL; wish++)
        if (str_prefix(arg, wish->name) == 0) {         // wish found

            if (IS_SET(ch->pcdata->wishes, wish->flag)) // char already has this wish
                                                        // completed
            {
                char_printf(ch, "But this wish is already completed!\n");
                return;
            }

            if ((wish->pre_req) &&                             // char do not have 
                (!IS_SET(ch->pcdata->wishes, wish->pre_req)))  // pre-required wish
            {
                for (pre_wish = wish_table; pre_wish->name != NULL; pre_wish++)
                    if (pre_wish->flag == wish->pre_req)
                    {
                        char_printf(ch, "You do not have pre-required wish {G%s{x!\n",
                            pre_wish->name);
                        return;
                    }
                bug("Pre-required wish does not exist in the wish_table!", 0);
                return;
            }

            if ((questor = questor_lookup(ch)) == NULL) // char must be near the questor
                return;

            if (ch->pcdata->questpoints < wish->price) // not enough qp
            {
                char_printf(ch, "This requires %d quest points, and you only have %d!\n",
                    wish->price, ch->pcdata->questpoints);
                return;
            }
            // and at last questor can complete the wish!!!
            ch->pcdata->questpoints -= wish->price;
            SET_BIT(ch->pcdata->wishes, wish->flag);
            char_printf(ch, "{CCongratulations!{x %s\n", wish->expl);
            return;
        }

    // No such wish in the wish_table. 
    // We must show to char short list of available wishes
    char_act("Available wishes:", ch);
    for (wish = wish_table; wish->name != NULL; wish++)
        char_printf(ch, " %s", wish->name);
    char_act("Use {Cwish info{x for more information.", ch);
}
const char * wish_list (CHAR_DATA *ch, int len)
{
    WISH_DATA * wish;
    static char result[MAX_INPUT_LENGTH];
    int         strings = 1;

    sprintf (result, "{c| {GWish: {c[{C");
    for (wish = wish_table; wish->name != NULL; wish++)
        if (IS_SET(ch->pcdata->wishes, wish->flag))
        {
            if ((strlen(result) + strlen(wish->name) + 3) > (len * strings))
            {
                strcat(result, "\n{c|{C        ");
                ++strings;
            }
            else
            {
                strcat(result, wish->name);
                strcat(result, " ");
            }
        }
    strcat(result, "{c]{x\n");
    return result;
}


