/* $Id: rating.c,v 1.666 2004/09/20 10:49:52 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
/************************************************************************************
 *     ANATOLIA 2.1 is copyright 1996-1997 Serdar BULUT, Ibrahim CANPUNAR           *
 *     ANATOLIA has been brought to you by ANATOLIA consortium                      *
 *       Serdar BULUT {Chronos}         bulut@rorqual.cc.metu.edu.tr                *
 *       Ibrahim Canpunar  {Asena}      canpunar@rorqual.cc.metu.edu.tr             *
 *       Murat BICER  {KIO}             mbicer@rorqual.cc.metu.edu.tr               *
 *       D.Baris ACAR {Powerman}        dbacar@rorqual.cc.metu.edu.tr               *
 *     By using this code, you have agreed to follow the terms of the               *
 *     ANATOLIA license, in the file Anatolia/anatolia.licence                      *
 ***********************************************************************************/

/************************************************************************************
 *  Original Diku Mud copyright (C) 1990, 1991 by Sebastian Hammer,                 *
 *  Michael Seifert, Hans Henrik St{rfeldt, Tom Madsen, and Katja Nyboe.            *
 *                                                                                  *
 *  Merc Diku Mud improvments copyright (C) 1992, 1993 by Michael                   *
 *  Chastain, Michael Quan, and Mitchell Tse.                                       *
 *                                                                                  *
 *  In order to use any part of this Merc Diku Mud, you must comply with            *
 *  both the original Diku license in 'license.doc' as well the Merc                *
 *  license in 'license.txt'.  In particular, you may not remove either of          *
 *  these copyright notices.                                                        *
 *                                                                                  *
 *  Much time and thought has gone into this software and you are                   *
 *  benefitting.  We hope that you share your changes too.  What goes               *
 *  around, comes around.                                                           *
 ************************************************************************************/

/************************************************************************************
*       ROM 2.4 is copyright 1993-1995 Russ Taylor                                  *
*       ROM has been brought to you by the ROM consortium                           *
*           Russ Taylor (rtaylor@pacinfo.com)                                       *
*           Gabrielle Taylor (gtaylor@pacinfo.com)                                  *
*           Brian Moore (rom@rom.efn.org)                                           *
*       By using this code, you have agreed to follow the terms of the              *
*       ROM license, in the file Rom24/doc/rom.license                              *
*************************************************************************************/

/************************************************************************************
 * Copyright (c) 1998 fjoe <fjoe@iclub.nsu.ru>                                      *
 * All rights reserved.                                                             *
 *                                                                                  *
 * Redistribution and use in source and binary forms, with or without               *
 * modification, are permitted provided that the following conditions               *
 * are met:                                                                         *
 * 1. Redistributions of source code must retain the above copyright                *
 *    notice, this list of conditions and the following disclaimer.                 *
 * 2. Redistributions in binary form must reproduce the above copyright             *
 *    notice, this list of conditions and the following disclaimer in the           *
 *    documentation and/or other materials provided with the distribution.          *
 *                                                                                  *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND           *
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE            *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE       *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE          *
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL       *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS          *
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)            *
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT       *
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY        *
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF           *
 * SUCH DAMAGE.                                                                     *
 ************************************************************************************/

#if !defined (WIN32)
#include <sys/time.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "merc.h"
#include "rating.h"
#include "typedef.h"
#include "update.h"
#include "wanderers.h"
#include "db/db.h"
#include "db/dofun.h"

extern void do_yell (CHAR_DATA *, const char *);
extern void interpret_social(social_t *soc, CHAR_DATA *ch, const char *argument); 
extern void save_gq (void);
extern void multi_hit(CHAR_DATA *ch, CHAR_DATA *victim, int dt);

STAT_DATA stat_record;

// ------------------------------------------------------------------------
// rating
// ------------------------------------------------------------------------

typedef struct rating_data RATING_DATA ;
struct rating_data
{
    const char * name ;
    int     pc_killed ;
} ;

RATING_DATA rating_table [RATING_TABLE_SIZE] ;
bool table_updated ;

//
// update player's rating
// this functions is called every time death is encountered
//
void rating_update (CHAR_DATA * ch, CHAR_DATA * victim)
{
    if (IS_NPC (ch) || IS_NPC (victim) || ch == victim || IS_IMMORTAL (ch))
        return ;

    ++ch->pcdata->pc_killed ;
    ++victim->pcdata->pc_died ;

    rating_add (ch->name, ch->pcdata->pc_killed) ;
}

//
// add (or replace) name|pc_killed to (in) rating_table
//
void rating_add (const char * name, int pc_killed)
{
    int i ;
    RATING_DATA * p = rating_table ;

    // find the minimal entry in rating_table
    for (i = 0 ; i < RATING_TABLE_SIZE ; ++i)
    {
        //  there is already entry for this player
        if (rating_table[i].name != NULL &&
            !str_cmp (name, rating_table[i].name))
        {
            p = rating_table + i ;

            if (p->pc_killed < pc_killed)
            {
                p->pc_killed  = pc_killed ;
                table_updated = TRUE ;
            }

            return ;
        }

        if (rating_table[i].pc_killed < p->pc_killed) p = rating_table + i ;
    }

    if (p->pc_killed < pc_killed)
    {
        if (p->name != NULL) free_string (p->name) ;
        p->name       = str_dup (name) ;
        p->pc_killed  = pc_killed ;
        table_updated = TRUE ;
    }
}

//
// comparator functor for quick sort
//
static int rating_data_cmp (const void * a, const void * b)
{
    return ((RATING_DATA *) b)->pc_killed - ((RATING_DATA *) a)->pc_killed ;
}

//
// show statistics
//
void do_rating (CHAR_DATA * ch, const char * argument)
{
    int i ;

    if (table_updated)
    {
        qsort (rating_table, RATING_TABLE_SIZE, sizeof (RATING_DATA), rating_data_cmp) ;
        table_updated = FALSE ;
    }

    char_act("Name                    | PC's killed", ch);
    char_act("------------------------+------------", ch);

    for (i = 0 ; i < RATING_TABLE_SIZE ; i++)
    {
        if (rating_table[i].name == NULL) continue ;

        char_printf (ch, "%-2d. %-20s| %d\n", i + 1,
                     rating_table [i].name,
                     rating_table [i].pc_killed) ;
    }
}

// ------------------------------------------------------------------------
// exploring
// ------------------------------------------------------------------------

#define BASE_EXP 10
static int explore_rooms_counter = 0 ;

//
// count number of rooms characters are able to explore
// this function is called within db.c bootstrap process
//
void explore_count_rooms (bool log)
{
    AREA_DATA * curr_area ;
    ROOM_INDEX_DATA * room;
    int j;

    explore_rooms_counter = 0;

    for (curr_area = area_first ; curr_area ; curr_area = curr_area->next)
    {
        curr_area->explorable_rooms = 0;

        if (IS_SET (curr_area->flags, AREA_UNDER_CONSTRUCTION))
        {
            if (log) log_printf ("skipped: %s", curr_area->name) ;
            continue ;
        }
        
        for (j = curr_area->min_vnum; j <= curr_area->max_vnum; ++j)
            {
                if (!(room = get_room_index(j)))
                    continue; 

                if (IS_SET (room->area->flags, AREA_UNDER_CONSTRUCTION) ||
                    IS_SET (room->room_flags, ROOM_GODS_ONLY | ROOM_IMP_ONLY | ROOM_NOEXPLORE) ||
                    !room->exit) 
                    continue ;

                ++ curr_area->explorable_rooms ;
            }

        explore_rooms_counter += curr_area->explorable_rooms ;
        if (log) log_printf ("Area: %s, total explorable rooms: %d", curr_area->name, curr_area->explorable_rooms);
    }
}

//
// return the total number of rooms to explore 
//
int explore_total_rooms (void)
{
    return explore_rooms_counter ;
}

//
// update character information about exploring
// this function is called within char_to_room() each time character
// visits a location
//
void explore_update_char (CHAR_DATA * ch, ROOM_INDEX_DATA * pRoomIndex)
{
    CHAR_EXPLORED_DATA * curr_area ; // current area from explored list
    int i ;
    int gain;


    if (ch->pcdata)
    {
        if (IS_SET (pRoomIndex->area->flags, AREA_UNDER_CONSTRUCTION) ||
            IS_SET (pRoomIndex->room_flags, ROOM_GODS_ONLY | ROOM_IMP_ONLY | ROOM_NOEXPLORE) ||
            !pRoomIndex->exit) return ;

        // count exp gain
        gain = BASE_EXP + ch->level / 6;
        if (ch->pcdata->TotalExploredRooms > 3000)
            gain = gain * (1.1 + 0.1 * ((ch->pcdata->TotalExploredRooms - 3000) / 500));

        if (ch->pcdata->last_visited)
        {
            if (!str_cmp (ch->pcdata->last_visited->name, pRoomIndex->area->name))
            {
                for (i = 0 ; i < ch->pcdata->last_visited->explored_room_count ; ++i)
                {
                    if (ch->pcdata->last_visited->explored_room_vnums [i] == pRoomIndex->vnum)
                        return ; // already explored room
                }
            }
        }

        for (curr_area = ch->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
        {
            if (!str_cmp (curr_area->name, pRoomIndex->area->name))
            {
                if (curr_area->explored_room_count == pRoomIndex->area->explorable_rooms)
                    return ; // all rooms in this area are explored

                for (i = 0 ; i < curr_area->explored_room_count ; i++)
                {
                    if (curr_area->explored_room_vnums[i] == pRoomIndex->vnum)
                    {
                        ch->pcdata->last_visited = curr_area ;
                        return ; // already explored room
                    }
                }   

                curr_area->explored_room_vnums [curr_area->explored_room_count++] =
                    pRoomIndex->vnum ;

                ch->pcdata->TotalExploredRooms++ ;
                ch->pcdata->last_visited = curr_area ;

                //gain_exp (ch, BASE_EXP + ch->level / 6) ;
                gain_exp (ch, gain) ;

                if (curr_area->explored_room_count == pRoomIndex->area->explorable_rooms)
                {
                    char_printf (ch, "{C       %d qp!{x\n",
                                 pRoomIndex->area->explorable_rooms * 2) ;
                    
                    ch->pcdata->questpoints += pRoomIndex->area->explorable_rooms * 2 ;
                    ch->pcdata->qp_earned += pRoomIndex->area->explorable_rooms * 2 ;
                }
                
                return ; // not explored room in the same area
            }
        }

        // add new area to explored list
        ch->pcdata->TotalExploredRooms++ ;
        curr_area                         = new_char_explored_data () ;
        curr_area->name                   = str_dup (pRoomIndex->area->name) ;
        curr_area->explored_room_vnums[0] = pRoomIndex->vnum ;
        curr_area->explored_room_count    = 1 ;

        //gain_exp (ch, BASE_EXP + ch->level / 6) ;
        gain_exp (ch, gain) ;


        if (curr_area->explored_room_count == pRoomIndex->area->explorable_rooms)
        {
            char_printf (ch, "{C       %d qp!{x\n",
                         pRoomIndex->area->explorable_rooms * 2) ;

            // if only one room in the area
            ch->pcdata->questpoints += pRoomIndex->area->explorable_rooms * 2 ;
            ch->pcdata->qp_earned += pRoomIndex->area->explorable_rooms * 2 ;
        }

        if (ch->pcdata->explored_areas) curr_area->next = ch->pcdata->explored_areas ;
        ch->pcdata->explored_areas = curr_area ;
        ch->pcdata->last_visited   = curr_area ;
    }
}

//
// show statistics
//
void do_explored (CHAR_DATA * ch, const char * argument)
{
    CHAR_EXPLORED_DATA * curr_area ; // current area from explored list
    float percent ;
    AREA_DATA          * in_area = ch->in_room->area;
    BUFFER             * output ;
    int                  i, j;
    bool                 room_visited, area_visited; 
    ROOM_INDEX_DATA    * room; 

    int                  room_count = 0, 
                         area_count = 0;
    char                 arg[MAX_INPUT_LENGTH];
    CHAR_DATA          * victim;
    CHAR_EXPLORED_DATA * prev_explored_area ; 

    int                  total_bribes = 0,
                         total_donations = 0;


    if (IS_NPC (ch)) return ;

    if (!ch->in_room || !ch->in_room->area)
        return; 

    argument = one_argument(argument, arg, sizeof(arg));

    // explored list for chars with security > 0
    if ((!str_cmp(arg, "list")) && ch->pcdata && ch->pcdata->security)
    {
        if (argument[0] == '\0')
            victim = ch;
        else
        {
            if ((victim = get_char_world(ch, argument)) == NULL)
            {
                char_act("No such creature.", ch);
                return;
            }

            if (IS_NPC(victim))
            {
                char_act("That is not a PC.", ch);
                return;
            }

            if (!victim->pcdata)
            {
                char_act("Error: pcdata is NULL.", ch);
                return;
            }
            in_area = victim->in_room->area;
        }
            
        output = buf_new(ch->lang);

        buf_printf (output, "Unexplored rooms in area %s:\n", in_area->name);

        area_visited = FALSE;
        for (curr_area = victim->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
            if (!str_cmp (curr_area->name, in_area->name)) // area found!
            {
                for (j = in_area->min_vnum; j <= in_area->max_vnum; ++j)
                {
                    if (!(room = get_room_index(j)))
                        continue; 

                    if (IS_SET (room->area->flags, AREA_UNDER_CONSTRUCTION) ||
                        IS_SET (room->room_flags, ROOM_GODS_ONLY | ROOM_IMP_ONLY | ROOM_NOEXPLORE) ||
                        !room->exit) 
                        continue ;

                    // explored_rooms_in_target_area loop
                    room_visited = FALSE;
                    for (i = 0 ; i < curr_area->explored_room_count ; ++i)
                        if (curr_area->explored_room_vnums[i] == j)
                        {
                            room_visited = TRUE;
                            break;
                        }
                    if (!room_visited)
                    {
                        buf_printf (output, "%d ", j);
                        ++room_count; 
                    }
                }
                area_visited = TRUE;
                break; 
            }
        if (area_visited)
        {
            page_to_char(buf_string(output), ch);
            char_printf(ch, "\n%d room(s) total.", room_count);
        }
        else
            char_printf (ch, "You have not explored this area at all!\n");

        buf_free(output);
        return;
    }

    // "explored full" command: shows to a char list of fully explored areas
    if ((!str_cmp(arg, "full")) && ch->pcdata)
    {
        output = buf_new(ch->lang);
        buf_printf (output, "Fully explored areas:\n");
        area_count = 0;

        for (in_area = area_first; in_area; in_area = in_area->next) 
        {
            if (in_area->explorable_rooms == 0)
                continue;

            for (curr_area = ch->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
                if (!str_cmp (curr_area->name, in_area->name)) // area found!
                    if (curr_area->explored_room_count == in_area->explorable_rooms)
                    {
                        buf_printf(output, "{x   %s (%d rooms)\n", in_area->name, in_area->explorable_rooms);
                        ++area_count;
                        break;
                    }
        }

        if (area_count)
        {
            buf_printf(output, "\n%d area(s) total.\n", area_count);
            page_to_char(buf_string(output), ch);
        }
        else
            char_printf (ch, "You have no fully explored areas!\n");

        buf_free(output);
        return;
    }

    // "explored none" command: shows to a char list of non-explored areas
    if ((!str_cmp(arg, "none")) && ch->pcdata)
    {
        bool area_found;
        
        output = buf_new(ch->lang);
        buf_printf (output, "Unexplored areas:\n");
        area_count = 0;

        for (in_area = area_first; in_area; in_area = in_area->next) 
        {
            if (in_area->explorable_rooms == 0)
                continue;
            
            area_found = FALSE;
            
            for (curr_area = ch->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
                if (!str_cmp (curr_area->name, in_area->name)) // area found!
                {
                    area_found = TRUE;
                    if (curr_area->explored_room_count < in_area->explorable_rooms)
                    {
                        percent = (float) (curr_area->explored_room_count * 100) / (float) in_area->explorable_rooms ;
                        buf_printf(output, "{x   %s (%d/%d rooms - %3.2f%%)\n", 
                            in_area->name, curr_area->explored_room_count, 
                            in_area->explorable_rooms, percent);
                        ++area_count;
                    }
                    break;
                }
            if (!area_found)
            {
                buf_printf(output, "{x   %s (%d/%d rooms - %3.2f%%)\n", 
                    in_area->name, 0, in_area->explorable_rooms, 0);

                ++area_count;
            }
        }
        if (area_count)
        {
            buf_printf(output, "\n%d area(s) total.\n", area_count);
            page_to_char(buf_string(output), ch);
        }
        else
            char_printf (ch, "You have no unexplored areas!\n");

        buf_free(output);
        return;
    }

    // "explored forget <victim>" command (imm only): removes explore info for an area
    // in which victim is now
    if ((!str_cmp(arg, "forget")) && IS_IMMORTAL(ch))
    {
        if (argument[0] == '\0')
            victim = ch;
        else
        {
            if ((victim = get_char_world(ch, argument)) == NULL)
            {
                char_act("No such creature.", ch);
                return;
            }
            if (IS_NPC(victim) || !victim->pcdata)
            {
                char_act("That is not a PC.", ch);
                return;
            }
            if (!victim->in_room || !victim->in_room->area)
            {
                char_act("Error: victim->in_room is NULL!", ch);
                return;
            }
        }

        prev_explored_area = NULL;
        for (curr_area = victim->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
        {
            if (!str_cmp (curr_area->name, victim->in_room->area->name)) // area found!
            {
                if (ch != victim)
                {
                    char_printf (ch, "You force %s to forget all that $gn{he} knows about %s.\n",
                                     victim->name, curr_area->name);
                }
                char_printf (victim, "You've just forgot all that you know about %s.\n",
                                     curr_area->name);

                victim->pcdata->TotalExploredRooms -= curr_area->explored_room_count;

                if (prev_explored_area)
                    prev_explored_area->next = curr_area->next;
                else
                    victim->pcdata->explored_areas = curr_area->next;

                free_string(curr_area->name);
                free(curr_area);
                victim->pcdata->last_visited = NULL;
                return;
            }
            prev_explored_area = curr_area;
        }
        char_printf (ch, "$gn{He} has no knowlege about %s...\n", victim->in_room->area->name);
        return;
    }

    // "explored stat <victim>" command (imm only): detail statistics
    if ((!str_cmp(arg, "stat")) && IS_IMMORTAL(ch))
    {
        if (argument[0] == '\0')
            victim = ch;
        else
        {
            if ((victim = get_char_world(ch, argument)) == NULL)
            {
                char_act("No such creature.", ch);
                return;
            }

            if (IS_NPC(victim))
            {
                char_act("That is not a PC.", ch);
                return;
            }

            if (!victim->pcdata)
            {
                char_act("Error: pcdata is NULL.", ch);
                return;
            }
        }

        output = buf_new(ch->lang);
        buf_printf (output, "{x     {CExploring statictics for %s{x\n", victim->name);
        buf_printf (output, "----------------------------------------------\n");
       
        for (in_area = area_first; in_area; in_area = in_area->next) 
        {
            for (curr_area = victim->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
                if (!str_cmp (curr_area->name, in_area->name)) // area found!
                {
                    if (in_area->explorable_rooms > 0)
                        percent = (float) curr_area->explored_room_count * 100 / (float) in_area->explorable_rooms ;
                    else
                        percent = 0;

                    buf_printf (output, "  %-20s: %-3d / %-3d (%3.2f%%)", 
                                       fmt_color_str(in_area->name, 20), 
                                       curr_area->explored_room_count,
                                       in_area->explorable_rooms, percent);

                    if ((curr_area->attitude_good    != 0) || 
                        (curr_area->attitude_neutral != 0) || 
                        (curr_area->attitude_evil    != 0))
                            buf_printf (output, ", Att: [{YG{x:%d; N:%d; {RE{x:%d]", 
                                curr_area->attitude_good, curr_area->attitude_neutral, curr_area->attitude_evil);

                    if ((curr_area->bribes > 0) || 
                        (curr_area->donations > 0))
                            buf_printf (output, ", Bribes: [%d (%d/%d)], Donations: [%d (%d/%d)]",
                                curr_area->bribes, curr_area->bribe_gems, curr_area->bribe_gold,
                                curr_area->donations, curr_area->donate_gems, curr_area->donate_gold);

                    total_donations += curr_area->donations;
                    total_bribes += curr_area->bribes;

                    buf_add (output, "\n");
                    break;
                }

        }

        buf_printf (output, "--------------------------------------------\n");
        percent = (float) ch->pcdata->TotalExploredRooms * 100 / (float) explore_total_rooms () ;
        buf_printf (output, "Total rooms explored  : %d of %d (%3.2f%%)\n",
                            ch->pcdata->TotalExploredRooms, explore_total_rooms (), percent) ;

        if (total_donations)
            buf_printf (output, "Total donations: %d\n", total_donations);
        if (total_bribes)
            buf_printf (output, "Total bribes: %d\n", total_bribes);

        page_to_char(buf_string(output), ch);
        buf_free(output);
        return;
    }

    if ((!str_cmp(arg, "refresh")) && IS_IMMORTAL(ch))
    {
        explore_count_rooms (FALSE);
        char_act ("Explorable rooms count recalculated.", ch);
        return;
    }

    percent = (float) ch->pcdata->TotalExploredRooms * 100 / (float) explore_total_rooms () ;

    char_printf (ch,
                "   {CAstrum Metaphora{x %d .    %d   (%3.2f%%).\n",
                explore_total_rooms (), ch->pcdata->TotalExploredRooms, percent) ;
    
    for (curr_area = ch->pcdata->explored_areas ; curr_area ; curr_area=curr_area->next)
    {
        if (!str_cmp (ch->in_room->area->name, curr_area->name))
        {
            if (ch->in_room->area->explorable_rooms)
            {
                percent = ((float) curr_area->explored_room_count * 100) / 
                           (float) ch->in_room->area->explorable_rooms ;
                char_printf (ch, "   %d .    %d   (%3.2f%%).\n\n\n",
                             ch->in_room->area->explorable_rooms, curr_area->explored_room_count,
                             percent) ;
            }

            // it was temporary written code... but I'm too lazy to rewrite
            //      /Welesh

            // good mobs
            if (curr_area->attitude_good < REL_HATE)
                char_act ("Good population in this area {RREALLY hates{x you!", ch);
            else if (curr_area->attitude_good < REL_DISLIKE)
                char_act ("Good population in this area {rdislikes{x you!", ch);
            else if (curr_area->attitude_good < REL_UNLIKE)
                char_act ("Good population in this area doesn't like your actions!", ch);
            else if (curr_area->attitude_good < REL_INDIFF)
                char_act ("Good population in this area looks at you with indifference.", ch);
            else if (curr_area->attitude_good < REL_LIKE)
                char_act ("Good population in this area {glikes{x you!.", ch);
            else if (curr_area->attitude_good < REL_VERY_LIKE)
                char_act ("Good population in this area {gvery likes{x you!.", ch);
            else 
                char_act ("Good population in this area {CREALLY loves{x you.", ch);

            // neutral mobs
            if (curr_area->attitude_neutral < REL_HATE)
                char_act ("Neutral population in this area {RREALLY hates{x you!", ch);
            else if (curr_area->attitude_neutral < REL_DISLIKE)
                char_act ("Neutral population in this area {rdislikes{x you!", ch);
            else if (curr_area->attitude_neutral < REL_UNLIKE)
                char_act ("Neutral population in this area doesn't like your actions!", ch);
            else if (curr_area->attitude_neutral < REL_INDIFF)
                char_act ("Neutral population in this area looks at you with indifference.", ch);
            else if (curr_area->attitude_neutral < REL_LIKE)
                char_act ("Neutral population in this area {glikes{x you!.", ch);
            else if (curr_area->attitude_neutral < REL_VERY_LIKE)
                char_act ("Neutral population in this area {gvery likes{x you!.", ch);
            else 
                char_act ("Neutral population in this area {CREALLY loves{x you.", ch);

            // evil mobs
            if (curr_area->attitude_evil < REL_HATE)
                char_act ("Evil population in this area {RREALLY hates{x you!", ch);
            else if (curr_area->attitude_evil < REL_DISLIKE)
                char_act ("Evil population in this area {rdislikes{x you!", ch);
            else if (curr_area->attitude_evil < REL_UNLIKE)
                char_act ("Evil population in this area doesn't like your actions!", ch);
            else if (curr_area->attitude_evil < REL_INDIFF)
                char_act ("Evil population in this area looks at you with indifference.", ch);
            else if (curr_area->attitude_evil < REL_LIKE)
                char_act ("Evil population in this area {glikes{x you!.", ch);
            else if (curr_area->attitude_evil < REL_VERY_LIKE)
                char_act ("Evil population in this area {gvery likes{x you!.", ch);
            else 
                char_act ("Evil population in this area {CREALLY loves{x you.", ch);

            if (IS_IMMORTAL(ch))
                char_printf(ch, "Exact attitude: good = %d, neutral = %d, evil = %d\n",
                    curr_area->attitude_good, curr_area->attitude_neutral, curr_area->attitude_evil);

            char_act ("", ch);
            if (curr_area->bribes)
                char_printf (ch, "You've tried to bribe people {g%d{x times. {g%d{x gold and {g%d{x gems were spent.\n", 
                    curr_area->bribes, curr_area->bribe_gold, curr_area->bribe_gems);
            if (curr_area->donations)
                char_printf (ch, "You've done {g%d{x donations. {g%d{x gold and {g%d{x gems were spent.\n", 
                    curr_area->donations, curr_area->donate_gold, curr_area->donate_gems);

            return ;
        }
    }
}

//
// exploring on/off
//
void do_explock (CHAR_DATA * ch, const char * argument)
{
    extern bool exploring  ;
    exploring = !exploring ;

    if (!exploring)
    {
        wiznet ("$N has disabled the exploring feature.",ch,NULL,0,0,0);
        char_act("Exploring feature is OFF.", ch);
    }
    else
    {
        wiznet ("$N has enabled the exploring feature.",ch,NULL,0,0,0);
        char_act("Exploring feature is ON.", ch);
    }
}

// ------------------------------------------------------------------------
// ranking
// ------------------------------------------------------------------------

#define MAX_POSITIONS   10
#define RANK_TIMER_SIZE 180
#define RANK_DESC_FILE  "ranking.conf"

typedef struct rank_player RANK_PLAYER ;
struct rank_player
{
    const char * pname ;
    int          value ;
};

typedef struct rank_config RANK_CONFIG ;
struct rank_config
{
    const char * name          ;
    const char * pfile_field   ; 
    const char * stat_filename ; 
    int          min_value     ;
    int          player_count  ;
} ;

static RANK_CONFIG   ranking_file [MAX_POSITIONS] ;
static RANK_PLAYER * ranking_data [MAX_POSITIONS] ;

static int ranking_positions = 0 ;
static int ranking_update_timer = 0 ;

//
// read configuration about ranking statistics files
//
void ranking_read_config (void)
{
    FILE * dfile ;
    int i = 0 ;

    if ((dfile = dfopen (ETC_PATH, RANK_DESC_FILE, "r")) == NULL)
    {
        log_printf ("ranking_read_config: Can't open file %s.", RANK_DESC_FILE) ;
        return ;
    }

    for (;;)
    {
        ranking_file[i].name          = fread_string (dfile) ;
        ranking_file[i].pfile_field   = fread_string (dfile) ;
        ranking_file[i].min_value     = fread_number (dfile) ;
        ranking_file[i].stat_filename = fread_string (dfile) ;
        ranking_file[i].player_count  = 0 ;

        ++i ;
        if (feof (dfile) || i > MAX_POSITIONS) break ;
    }

    fclose (dfile) ;
    ranking_positions = i ;
}

//
// read ranking statistics about players
//
void ranking_read_stats (void)
{
    int i, curr_rank ;
    FILE * sfile ;
    char * pname ;

    for (curr_rank = 0 ; curr_rank < ranking_positions ; ++curr_rank)
    {
        // open file with ranking statistics
        if ((sfile = dfopen (TMP_PATH, ranking_file [curr_rank].stat_filename, "r")) == NULL)
        {
            log_printf ("ranking_read_stats: Can't open file %s.",
                        ranking_file [curr_rank].stat_filename) ;
            return ;
        }

        // free memory if something was already allocated
        if (ranking_data [curr_rank])
        {
            for (i = 0 ; i < ranking_file [curr_rank].player_count ; ++ i)
                free_string (ranking_data [curr_rank][i].pname) ;

            free (ranking_data [curr_rank]) ;
        }

        // allocate in chunks of 32 data structures
        ranking_data [curr_rank] = malloc (sizeof (RANK_PLAYER) * 32) ;
        ranking_file [curr_rank].player_count = 0 ;

        // read in data
        for (i = 0 ;; ++(ranking_file [curr_rank].player_count)) 
        {
            pname = fread_word (sfile) ; // word - static buffer ptr
            if (feof (sfile) || !pname) break ;

            ranking_data [curr_rank][i].pname = str_dup (pname) ;
            ranking_data [curr_rank][i].value = fread_number (sfile) ;

            if ((++i % 32) == 0) // fill in chunk -> reallocate
            {
                ranking_data [curr_rank] = realloc (ranking_data [curr_rank],
                                                    sizeof (RANK_PLAYER) * (i + 32)) ;
            }
        }

        // shrink
        ranking_data [curr_rank] = realloc (ranking_data [curr_rank],
                                            sizeof (RANK_PLAYER) * i) ;
    }
}

//
// update handler
//
void ranking_update (void)
{
    if (ranking_update_timer == 0)
    {
        CHAR_DATA * ch ;

        // reread statistics
        ranking_read_stats () ;

        for (ch = char_list ; ch ; ch = ch->next)
            if (IS_NPC (ch)) continue ;
            else  char_printf (ch, "{RPlayer rankings updated.{x\n") ;

        ranking_update_timer = RANK_TIMER_SIZE ;
    }

    --ranking_update_timer ;
}

//
// show statistics
//
void do_ranking (CHAR_DATA * ch, const char * argument)
{
    int i, number ;
    char arg [MAX_INPUT_LENGTH] ;
    BUFFER * output ;

    if (IS_NPC (ch)) return ;

    one_argument(argument, arg, sizeof(arg)) ;

    if (argument[0] == '\0')
    {
        char_printf (ch, "Syntax: {CRanking <type>{x\n") ;
        char_printf (ch, "Types:\n") ;

        for (i = 0 ; i < ranking_positions ; ++i)
            char_printf (ch, "{x      %d - %s (%d)\n", i + 1, ranking_file [i].name, ranking_file [i].min_value ) ;
    }
    else
    if (is_number (argument))
    {
        number = atoi (argument) - 1 ;
        if (number < 0 || number >= ranking_positions) return ;

        output = buf_new (-1) ;

        buf_printf (output, "  {CAstrum Metaphora{x\n") ;
        buf_printf (output, "      {c%s{x\n", ranking_file [number].name)  ;
        buf_printf (output, "                    \n") ; 
        buf_printf (output, " ----- -------------------- --------\n") ;

        for (i = 0 ; i < ranking_file [number].player_count ; ++i)
        {
            buf_printf (output, " %-5d %-20s %-5d\n", i + 1,
                        ranking_data [number][i].pname,
                        ranking_data [number][i].value) ;
        }

        buf_add (output, "\n") ;
        page_to_char (buf_string (output), ch) ;
        buf_free (output) ;
    }
    else return ;
}

//
// self-statistics
//
void do_myrank (CHAR_DATA * ch, const char * argument)
{
    int i, curr_rank ;
    BUFFER * output ;
    char arg [MAX_INPUT_LENGTH] ;
    bool found ;

    if (IS_NPC (ch)) return ;

    one_argument(argument, arg, sizeof(arg)) ;

    if (argument[0] == '\0')
    {
        output = buf_new (ch->lang) ;
        buf_printf (output, "  {CAstrum Metaphora{x  %s\n", ch->name) ;

        for (curr_rank = 0 ; curr_rank < ranking_positions ; ++curr_rank)
        {
            found = FALSE ;

            for (i = 0 ; i < ranking_file [curr_rank].player_count ; ++i)
            {
                if (!str_cmp(ranking_data [curr_rank][i].pname, ch->name))
                {
                    found = TRUE ; break ;
                }
            }

            if (found) buf_printf (output, "{G %d{x. %-35s: %-5d  %-5d\n",
                                   curr_rank + 1,
                                   ranking_file [curr_rank].name, i + 1,
                                   ranking_file [curr_rank].player_count) ;
            else       buf_printf (output, "{G %d{x. %-35s:  {rN/A{x   %-5d\n",
                                   curr_rank + 1, 
                                   ranking_file [curr_rank].name, 
                                   ranking_file [curr_rank].player_count) ;
        }

        buf_add (output, "\n") ;
        page_to_char (buf_string (output), ch) ;
        buf_free (output) ;
    }
    else
    if (is_number (argument))
    {
        output = buf_new (ch->lang) ;

        buf_printf (output, "  {CAstrum Metaphora{x  %s\n", ch->name) ;
        buf_printf (output, "not implemented yet\n") ;

        buf_add (output, "\n") ;
        page_to_char (buf_string(output), ch) ;
        buf_free (output) ;
    }
    else return ;
}

// ------------------------------------------------------------------------
// memorials
// ------------------------------------------------------------------------
#define OBJ_VNUM_MEMORIAL 15150
#define OBJ_VNUM_INCOG_MEMORIAL 15151
#define CREATE_NAMED    (B) // duplicated from db.c

void create_memorial(CHAR_DATA *ch, CHAR_DATA *victim)
{
    OBJ_DATA *memorial;
    OBJ_INDEX_DATA * memorialIndex;

    if (IS_NPC(victim) || IS_NPC(ch))
        return;

    if (victim == ch) 
        return;
    
    // if both char and victim don't want memorials or there is a duel - return
    if ((ch->in_room && IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA)) 
        || (!IS_SET(ch->comm, COMM_MEMORIAL) && !IS_SET(victim->comm, COMM_MEMORIAL)))
        return;

    // char wants to set a memorial - so we place on it both names of char and victim
    if (IS_SET(ch->comm, COMM_MEMORIAL))
    {
        memorialIndex = get_obj_index(OBJ_VNUM_MEMORIAL);
        if (!memorialIndex)
        {
            bug("can't create memorial", 0);
            return;
        }

        // when we create named obj, we doesn't fill name, short and long
        memorial = create_obj_org(memorialIndex, 1, CREATE_NAMED);
        if (!memorial)
        {
            bug("can't create memorial", 0);
            return;
        }
        memorial->name        = str_dup (memorialIndex->name);
        memorial->short_descr = mlstr_printf(memorialIndex->short_descr, ch->name, victim->name);
        memorial->description = mlstr_printf(memorialIndex->description, ch->name, victim->name);
        memorial->timer       = 360; // 6 hours
        if (victim->in_room)
            obj_to_room(memorial, victim->in_room);
    }
    // victim wants, char doesn't want
    else // create memorial to victim without char
    {
        memorialIndex = get_obj_index(OBJ_VNUM_INCOG_MEMORIAL);
        if (!memorialIndex)
        {
            bug("can't create memorial", 0);
            return;
        }
    
        memorial = create_obj_org(memorialIndex, 1, CREATE_NAMED);
        if (!memorial)
        {
            bug("can't create memorial", 0);
            return;
        }
        memorial->name        = str_dup (memorialIndex->name);
        memorial->short_descr = mlstr_printf(memorialIndex->short_descr, victim->name);
        memorial->description = mlstr_printf(memorialIndex->description, victim->name);
        memorial->timer       = 360; // 6 hours
        if (victim->in_room)
            obj_to_room(memorial, victim->in_room);
    }
}

// ------------------------------------------------------------------------
// relations
// ------------------------------------------------------------------------
void action_rel_hate (CHAR_DATA * ch, CHAR_DATA * mob);
void action_rel_dislike (CHAR_DATA * ch, CHAR_DATA * mob);
void action_rel_unlike (CHAR_DATA * ch, CHAR_DATA * mob);
void action_rel_like (CHAR_DATA * ch, CHAR_DATA * mob);
void action_rel_very_like (CHAR_DATA * ch, CHAR_DATA * mob);
void action_rel_love (CHAR_DATA * ch, CHAR_DATA * mob);

extern void spec_cast(CHAR_DATA *ch, const char *spell_name, CHAR_DATA *victim);

static MOB_SPELL mob_spells_protective[] = 
{
    { "sanctuary",         spell_sanctuary,         60, 0, MS_IF_NOT_AFFECTED | MS_GOOD_NEUTRAL, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "black shroud",      spell_black_shroud,      60, 0, MS_IF_NOT_AFFECTED | MS_EVIL_ONLY, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "shield",            spell_shield,            30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "armor",             spell_armor,             5,  0, MS_IF_NOT_AFFECTED, 0 },
    { "enhanced armor",    spell_enhanced_armor,    20, 0, MS_IF_NOT_AFFECTED, 0},
    { "fly",               spell_fly,               30, 0, MS_IF_NOT_AFFECTED, AFF_FLYING },
    { "giant strength",    spell_giant_strength,    5,  0, MS_IF_NOT_AFFECTED, 0 },
    { "bless",             spell_bless,             10, 0, MS_IF_NOT_AFFECTED, 0 },
    { "positive luck",     spell_positive_luck,     30, 0, MS_IF_NOT_AFFECTED, 0 },
    { NULL }
};

static MOB_SPELL mob_spells_healing[] = 
{
    { "cure light",        spell_cure_light,        4,  0, 0, 0 },
    { "cure serious",      spell_cure_serious,      8,  0, 0, 0 },
    { "cure critical",     spell_cure_critical,     18, 0, 0, 0 },
    { "heal",              spell_heal,              35, 0, 0, 0 },
    { "superior heal",     spell_superior_heal,     50, 0, 0, 0 },
    { "master healing",    spell_master_healing,    65, 0, 0, 0 },
    { NULL }
};

static MOB_SPELL mob_spells_curative[] = 
{
    { "cure blindness",    spell_cure_blindness,    20, 0, MS_IF_AFFECTED, AFF_BLIND },
    { "cure poison",       spell_cure_poison,       25, 0, MS_IF_AFFECTED, AFF_POISON },
    { "cure disease",      spell_cure_disease,      30, 0, MS_IF_AFFECTED, AFF_PLAGUE },
    { NULL }
};

// called from do_look (ch, "auto")
void relation_effect(CHAR_DATA * ch)
{
    CHAR_DATA *vch, * wimpy_mob;
    CHAR_EXPLORED_DATA *curr_expl_area;
    bool area_found;
    int curr_rel = 0;
    // act flags for mobs that must not be affected
    flag64_t act_flags = 0;
    EXIT_DATA *pexit;
    int door;
    ROOM_INDEX_DATA *was_in;
    ROOM_INDEX_DATA *now_in;
    int attempt, max_att = 6;
    static char yell_msg[MAX_INPUT_LENGTH];
    CHAR_DATA * vch_next;

    if (IS_NPC(ch))
        return;

    area_found = FALSE;
    for (curr_expl_area = ch->pcdata->explored_areas ; curr_expl_area ; curr_expl_area=curr_expl_area->next)
        if (!str_cmp (ch->in_room->area->name, curr_expl_area->name))
        {
            area_found = TRUE;
            break;
        }

    if (!area_found)
        return;

    act_flags = ACT_TRAIN   | ACT_PRACTICE   | ACT_HEALER  | ACT_CHANGER |
                ACT_FORGER  | ACT_REPAIRMAN  | ACT_FORGER  | ACT_QUESTOR |
                ACT_NOQUEST | ACT_CLAN_GUARD | ACT_NOTRACK | ACT_PET ;

    wimpy_mob = NULL;
    for (vch = ch->in_room->people; vch; vch = vch_next)
    {
        vch_next = vch->next_in_room;
        // skip some mobs
        if (!IS_NPC(vch) || IS_AFFECTED(vch,AFF_CHARM) 
            || vch->in_mind || vch->fighting || !can_see(vch,ch) 
            || IS_SET(vch->pIndexData->act, act_flags) 
            || vch->fighting)
            continue;

        if (RIDDEN(vch))
            continue;

        if (IS_EVIL(vch))
            curr_rel = curr_expl_area->attitude_evil;
        else if (IS_GOOD(vch))
            curr_rel = curr_expl_area->attitude_good;
        else if (IS_NEUTRAL(vch))
            curr_rel = curr_expl_area->attitude_neutral;

        if (curr_rel < REL_HATE)
        {
            action_rel_hate (ch, vch);
            // 5% chance that low level mob will flee from room
            if ((vch->level < ch->level - ch->level/10 + 2) && (number_percent() > 95))
                wimpy_mob = vch;
        }
        else if (curr_rel < REL_DISLIKE)
        {
            action_rel_dislike (ch, vch);
            // one chance from 100 that low level mob will flee from room
            if ((vch->level < ch->level - ch->level/10 + 2) && (number_percent() > 99))
                wimpy_mob = vch;
        }
        else if (curr_rel < REL_UNLIKE)
            action_rel_unlike (ch, vch);
        else if (curr_rel < REL_INDIFF)
            continue;
        else if (curr_rel < REL_LIKE)
            action_rel_like (ch, vch);
        else if (curr_rel < REL_VERY_LIKE)
            action_rel_very_like (ch, vch);
        else 
            action_rel_love (ch, vch);
    }
    // special case - one mob can flee from room 
    if (wimpy_mob)
    {
        was_in = wimpy_mob->in_room;
        for (attempt = 0; attempt < max_att; attempt++)
        {
            door = number_door();
            if ((pexit = was_in->exit[door]) == 0
                || pexit->to_room.r == NULL
                || (IS_SET(pexit->exit_info, EX_CLOSED)
                && (!IS_AFFECTED(wimpy_mob, AFF_PASS_DOOR)
                || IS_SET(pexit->exit_info,EX_NOPASS))
                && !IS_TRUSTED(wimpy_mob,ANGEL))
                || (IS_SET(pexit->exit_info , EX_NOFLEE))
                || (IS_NPC(wimpy_mob)
                && IS_SET(pexit->to_room.r->room_flags, ROOM_NOMOB)))
                continue;

            move_char(wimpy_mob, door, FALSE);
            if ((now_in = wimpy_mob->in_room) == was_in)
                //continue;
                break;

            wimpy_mob->in_room = was_in;
            if (number_percent() > 75)
            {
                sprintf(yell_msg, "%s is here! We'll die! Fleeeee!", ch->name);
                do_yell(wimpy_mob, yell_msg);
            }
            act("$n {Dflees{x in panic!", wimpy_mob, NULL, NULL, TO_ROOM);
            wimpy_mob->in_room = now_in;
            break;
        }
    }
}

void action_rel_hate (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char     * char_social = NULL;
    social_t * soc = NULL;
    bool       check_level;
    bool       check_chance;

    // chance to aggr: greater if ch follows Astern
    check_chance = (ch->religion == RELIGION_ASTERN) ? 
                        (number_percent() > 50) : 
                        (number_percent() > 77);

    // level check - for HEROES only low-level mob will not aggr
    if (ch->level >= LEVEL_HERO)
        check_level = (mob->level > (ch->level - (ch->level/15)));
    else
        check_level = ((mob->level > (ch->level - (ch->level/15))) &&
                       (mob->level < (ch->level + (ch->level/15))));

    // todo: add check_luck???

    if (check_chance && check_level && !IS_SET (ch->plr_flags, PLR_GHOST))
    {
        char_act (str_empty, ch);
        char_social = "screams in {Ranger{x and ATTACKS.\n";
        do_emote (mob, char_social);

        multi_hit(mob, ch, TYPE_UNDEFINED);

        ++stat_record.aggressions;
        return;
    }

    if (number_percent() > 90)
    {
        switch (number_range(0,4))
        {
            case 0 : 
                char_social = "growl";
                break;
            case 1 : 
                char_social = "fume";
                break;
            case 2 : 
                char_social = "grimace";
                break;
            case 3 : 
                char_social = "pissed";
                break;
            case 4 : 
                char_social = "curse";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

void action_rel_dislike (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char*           char_social = NULL;
    social_t *soc = NULL;
    
    if (number_percent() > 95)
    {
        switch (number_range(0,2))
        {
            case 0 : 
                char_social = "grin";
                break;
            case 1 : 
                char_social = "glare";
                break;
            case 2 : 
                char_social = "snort";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

void action_rel_unlike (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char*           char_social = NULL;
    social_t *soc = NULL;
    
    if (number_percent() > 96)
    {
        switch (number_range(0,2))
        {
            case 0 : 
                char_social = "accuse";
                break;
            case 1 : 
                char_social = "roll";
                break;
            case 2 : 
                char_social = "peer";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

void action_rel_like (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char*           char_social = NULL;
    social_t *soc = NULL;
    
    if (number_percent() > 96)
    {
        switch (number_range(0,1))
        {
            case 0 : 
                char_social = "tip";
                break;
            case 1 : 
                if (mob->sex == SEX_MALE)
                    char_social = "shake";
                else
                    char_social = "curtsey";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

void action_rel_very_like (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char*           char_social = NULL;
    social_t *soc = NULL;
    
    if (number_percent() > 95)
    {
        switch (number_range(0,3))
        {
            case 0 : 
                char_social = "tip";
                break;
            case 1 : 
                if (mob->sex != ch->sex)
                    char_social = "cuddle";
                break;
            case 2 : 
                if (mob->sex != ch->sex)
                    char_social = "ruffle";
                break;
            case 3 : 
                    char_social = "beam";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

void action_rel_love (CHAR_DATA * ch, CHAR_DATA * mob)
{
    char     * char_social = NULL;
    social_t * soc = NULL;
    clan_t   * clan;
    char     * spell_name = NULL;

    clan = clan_lookup(ch->clan);

    if (!(clan && IS_SET (clan->flags, CLAN_HATE_MAGIC)))
    {
        // can cast spell

        // 1. blind, poison, plague - highest priority
        /*if (IS_AFFECTED (ch, AFF_BLIND))
            spell_name = "cure blindness";
        else if (IS_AFFECTED (ch, AFF_POISON))
            spell_name = "cure poison";
        else if (IS_AFFECTED (ch, AFF_PLAGUE))
            spell_name = "cure disease";*/
        if ((spell_name != NULL) && (number_percent() > 90))
        {
            act("$n tries to cure $N.", mob, NULL, ch, TO_ROOM | TO_NOTVICT);
            act("$n tries to cure you.", mob, NULL, ch, TO_VICT);
            if (wanderer_spec_cast (mob, ch, mob_spells_curative, SO_FIRST|SO_NO_MANA, UMIN(LVL(ch)+2, mob->level+2), TARGET_CHAR))
            {
                spec_cast (mob, spell_name, ch);
                ++stat_record.good_spells;
                return;
            }
        }

        // 2. different heal spells  - according to mob level - high priority
        if ((ch->hit < ch->max_hit) && (number_percent() > 90))
        {
            if (!IS_UNDEAD(ch))
            {
                if (wanderer_spec_cast (mob, ch, mob_spells_healing, SO_RANDOM|SO_NO_MANA, UMIN(LVL(ch)+2, mob->level+2), TARGET_CHAR))
                {
                    act("$n tries to heal $N.", mob, NULL, ch, TO_ROOM | TO_NOTVICT);
                    act("$n tries to heal you.", mob, NULL, ch, TO_VICT);
                    ++stat_record.good_spells;
                    return;
                }
            }
        }

        // 3. todo - restore mana and refresh

        // 4. shield/haste/inner fire/bless (cyber bless)/frenzy/fly (?)/
        //           armor/enh. armor/positive luck/sanc (black shroud) - SMALL chances...
        // (now all except haste, frenzy, inner fire, cyber bless
        if (number_percent() > 95)
        {
            if (wanderer_spec_cast (mob, ch, mob_spells_protective, SO_RANDOM|SO_NO_MANA, UMIN(LVL(ch)+2, mob->level+2), TARGET_CHAR))
            {
                act("$n tries to help $N.", mob, NULL, ch, TO_ROOM | TO_NOTVICT);
                act("$n tries to help you.", mob, NULL, ch, TO_VICT);
                ++stat_record.good_spells;
                return;
            }
        }
    }
    else 
    {
        // for BRs - needle prick and bandage only. Later :) (todo)
    }
    
    if (number_percent() > 93)
    {
        switch (number_range(0,4))
        {
            case 0 :
                if (mob->sex != ch->sex)
                    char_social = "squeeze";
                break;
            case 1 : 
                if (mob->sex != ch->sex)
                    char_social = "love";
                break;
            case 2 : 
                if (mob->sex != ch->sex)
                    char_social = "stare";
                break;
            case 3 : 
                if (mob->sex != ch->sex)
                    char_social = "french";
                break;
            case 4 : 
                if (mob->sex != ch->sex)
                    char_social = "peck";
                break;
        }
        if ((soc = social_lookup(char_social, str_prefix)) == NULL)
            bug("relation: invalid social!",0);
        if (soc) 
            interpret_social(soc, mob, ch->name);
        return;
    }
}

CHAR_EXPLORED_DATA * get_curr_explored_data (CHAR_DATA * ch)
{
    CHAR_EXPLORED_DATA * curr_expl_area;
    
    if (!ch || IS_NPC(ch) || !ch->pcdata || !ch->in_room || !ch->in_room->area) 
        return NULL;

    if (ch->pcdata->last_visited &&
        !str_cmp (ch->in_room->area->name, ch->pcdata->last_visited->name))
        return ch->pcdata->last_visited;

    for (curr_expl_area = ch->pcdata->explored_areas ; curr_expl_area ; curr_expl_area=curr_expl_area->next)
        if (!str_cmp (ch->in_room->area->name, curr_expl_area->name))
        {
            ch->pcdata->last_visited = curr_expl_area;
            return curr_expl_area;
        }
    
    return NULL;
}

DO_FUN (do_bribe)
{    
    CHAR_EXPLORED_DATA * curr_expl_area;
    char arg[MAX_INPUT_LENGTH];
    int amount = 0, final_amount, mod;
    OBJ_DATA * gem;


    curr_expl_area = get_curr_explored_data (ch);
    if (!curr_expl_area)
    {
        char_act ("People in this area will not accept your bribe.", ch);
        return;
    }

    argument = one_argument(argument, arg, sizeof(arg));

    if (is_number(arg) && !str_cmp (argument, "gold"))
    {
        // bribe gold
        amount = atoi(arg);
        if (ch->gold < amount)
        {
            char_act ("You don't have enough gold.", ch);
            return;
        }
        curr_expl_area->bribe_gold += amount;
        ch->gold -= amount;
    }
    else
    {
        // bribe gem
        gem = get_obj_carry(ch, arg);
        if (!gem) // no such thing in the inventory
        {
            char_act("You're not carrying it.", ch);
            return;
        }

        // wrong thing
        if (gem->pIndexData->vnum != OBJ_VNUM_R_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_B_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_G_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_M_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_W_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_Y_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_GEMS)
        {
            char_act("People in this area will not accept such thing.\nTry to offer gold or gems.", ch);
            return;
        } 

        // Good!!! It's right gem!!!
        if (gem->pIndexData->vnum == OBJ_VNUM_GEMS)
            amount = 700;
        else
            amount = 1100;

        curr_expl_area->bribe_gems++;
        obj_from_char(gem);
    }

    mod = 1 + curr_expl_area->bribes / 20;
    curr_expl_area->bribes++;

    final_amount = UMAX(1,(amount/3) / mod);
    curr_expl_area->attitude_evil += final_amount ;

    final_amount = UMAX(1,(amount/6) / mod);
    curr_expl_area->attitude_neutral += final_amount;

    final_amount = UMAX(1,(amount/200) * mod);
    curr_expl_area->attitude_good -= final_amount;

    char_act ("Now evil and neutral population is more tolerant towards you.", ch);
    char_act ("But good population doesn't like such actions...", ch);

    ++stat_record.bribes;
}

DO_FUN (do_donate)
{
    CHAR_EXPLORED_DATA * curr_expl_area;
    char arg[MAX_INPUT_LENGTH];
    int amount = 0, final_amount, mod;
    OBJ_DATA * gem;


    curr_expl_area = get_curr_explored_data (ch);
    if (!curr_expl_area)
    {
        char_act ("People in this area will not accept your donation.", ch);
        return;
    }

    argument = one_argument(argument, arg, sizeof(arg));

    if (is_number(arg) && !str_cmp (argument, "gold"))
    {
        // donate gold
        amount = atoi(arg);
        if (ch->gold < amount)
        {
            char_act ("You don't have enough gold.", ch);
            return;
        }
        curr_expl_area->donate_gold += amount;
        ch->gold -= amount;
    }
    else
    {
        // donate gem
        gem = get_obj_carry(ch, arg);
        if (!gem) // no such thing in the inventory
        {
            char_act("You're not carrying it.", ch);
            return;
        }

        // wrong thing
        if (gem->pIndexData->vnum != OBJ_VNUM_R_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_B_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_G_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_M_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_W_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_Y_GEMS
            && gem->pIndexData->vnum != OBJ_VNUM_GEMS)
        {
            char_act("People in this area will not accept such thing.\nTry to offer gold or gems.", ch);
            return;
        } 

        // Good!!! It's right gem!!!
        if (gem->pIndexData->vnum == OBJ_VNUM_GEMS)
            amount = 700;
        else
            amount = 1100;

        curr_expl_area->donate_gems++;
        obj_from_char(gem);
    }

    mod = 1 + curr_expl_area->donations / 20;
    curr_expl_area->donations++;

    final_amount = UMAX(1,(amount/3) / mod);
    curr_expl_area->attitude_good += final_amount ;

    final_amount = UMAX(1,(amount/5) / mod);
    curr_expl_area->attitude_neutral += final_amount;

    final_amount = UMAX(1,(amount/200) * mod);
    curr_expl_area->attitude_evil -= final_amount;

    char_act ("Now good and neutral population is more tolerant towards you.", ch);
    char_act ("But evil population doesn't like such actions...", ch);

    ++stat_record.donation;
}

//------------------------------------------------------------------------
// history stuff - some statistics (see struct in merc.h)
//------------------------------------------------------------------------
static int pulse_stat_record;               // save time counter for stat_record
#define PULSE_STAT_RECORD           15      // interval we must save stat_record (ticks)
static int pulse_save_gq;                   // save time counter for global guests
#define PULSE_SAVE_GQ               15      // interval we must save gq (ticks)
#define STAT_FILE_NAME              "stat.txt"

void load_stat_record (void);
void save_stat_record (void);
void service_data_update (void);

static void clear_history (char * arg);
static void fread_history (FILE *fp);

DO_FUN (do_history)
{
    BUFFER*       output ;

    output = buf_new (ch->lang) ;

    if ((IS_IMMORTAL (ch) && argument[0] == '\0') ||
        !IS_IMMORTAL (ch))
    {
        buf_add(output,    "\n    {G/~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\\{x\n");
        buf_add(output,    "{x    {G|                       {CHistory of our realm                       {G|{x\n");
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Quests
        buf_add(output,    "{x    {G|    {WQuests:                                                       {G|{x\n");
        buf_printf(output, "{x    {G|        {WRequested{x :  {G[{W%8d{G]{x      {WCompleted{x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.quest_requested,stat_record.quest_completed);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Global quests
        buf_add(output,    "{x    {G|    {WGlobal quests:                                                {G|{x\n");
        buf_printf(output, "{x    {G|        {WStarted  {x :  {G[{W%8d{G]{x      {WCompleted{x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.gq_started,stat_record.gq_completed);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Win quests
        buf_add(output,    "{x    {G|    {WPersonal WinQuests:                                           {G|{x\n");
        buf_printf(output, "{x    {G|        {WStarted  {x :  {G[{W%8d{G]{x      {WCompleted{x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.wq_pers_started,stat_record.wq_pers_win);
        buf_add(output,    "{x    {G|    {WMultiplayer WinQuests:                                        {G|{x\n");
        buf_printf(output, "{x    {G|        {WStarted  {x :  {G[{W%8d{G]{x      {WCompleted{x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.wq_mult_started,stat_record.wq_mult_win);

        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");
        // Wars
        buf_add(output,    "{x    {G|    {WWars:                                                         {G|{x\n");
        buf_printf(output, "{x    {G|        {WStarted  {x :  {G[{W%8d{G]{x      {WTimed out{x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.wars_started, stat_record.wars_timed_out);
        buf_printf(output, "{x    {G|        {WWon      {x :  {G[{W%8d{G]{x      {WSurrenders{x: {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.wars_completed, stat_record.wars_surrenders);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Duels
        buf_add(output,    "{x    {G|    {WDuels:                                                        {G|{x\n");
        buf_printf(output, "{x    {G|        {WProvoked {x :  {G[{W%8d{G]{x      {WStarted  {x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.duels_provoked,stat_record.duels_started);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Kills
        buf_add(output,    "{x    {G|    {WKills:                                                        {G|{x\n");
        buf_printf(output, "{x    {G|        {WPK       {x :  {G[{W%8d{G]{x      {WMobs     {x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.PK_count, stat_record.mobs_killed);
        buf_printf(output, "{x    {G|        {WPK on CR {x :  {G[{W%8d{G]{x      {WBy mobs   {x: {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.PK_on_CR, stat_record.pc_killed_by_mobs);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Auction
        buf_add(output,    "{x    {G|    {WAuctionned items:                                             {G|{x\n");
        buf_printf(output, "{x    {G|        {WStandard {x :  {G[{W%8d{G]{x      {WLimited  {x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.auctionned_items,stat_record.auctionned_limits);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Rulers
        buf_add(output,    "{x    {G|    {WRulers' activity:                                             {G|{x\n");
        buf_printf(output, "{x    {G|        {WWanted set{x:  {G[{W%8d{G]{x      {WArrests  {x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.wanted_set,stat_record.arrested);
        buf_printf(output, "{x    {G|                {WCriminals killed:  {G[{W%8d{G]{x                     {G|{x\n", 
            stat_record.criminals_killed);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");
        
        // Crafting
        buf_add(output,    "{x    {G|    {WForged items:                                                 {G|{x\n");
        buf_printf(output, "{x    {G|        {WBy {r[{RPK{r]  {x :  {G[{W%8d{G]{x      {WBy {GNO PK{x  : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.crafted_items_forged - stat_record.crafted_by_no_pk,stat_record.crafted_by_no_pk);
        buf_printf(output, "{x    {G|                   {WTotal forged:  {G[{W%8d{G]{x                      {G|{x\n", 
            stat_record.crafted_items_forged);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Relations
        buf_add(output,    "{x    {G|    {WRelations:                                                    {G|{x\n");
        buf_printf(output, "{x    {G|        {WAggro     {x:  {G[{W%8d{G]{x      {WGood spell{x: {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.aggressions, stat_record.good_spells);
        buf_printf(output, "{x    {G|        {WDonations {x:  {G[{W%8d{G]{x      {WBribes    {x: {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.donation, stat_record.bribes);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

        // Gold
        buf_add(output,    "{x    {G|    {WGold:                                                         {G|{x\n");
        buf_printf(output, "{x    {G|        {WMade     {x :  {G[{W%8d{G]{x      {WStolen   {x : {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.gold_made, stat_record.gold_stolen);
        buf_printf(output, "{x    {G|        {WQuests   {x :  {G[{W%8d{G]{x      {WGlobals   {x: {G[{W%8d{G]{x       {G|{x\n", 
            stat_record.gold_quest_reward, stat_record.gold_gq_reward);
        buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");
        
        // Reward stuff
        buf_add(output,    "{x    {G|    {WRewards:                                                      {G|{x\n");
        buf_printf(output, "{x    {G|        {WTotal    {x :  {G[{W%8d{G]{x                                   {G|{x\n", 
            stat_record.rewards);

        buf_add(output,    "{x    {G|__________________________________________________________________|{x\n");

        page_to_char (buf_string (output), ch) ;
        buf_free (output) ;

    }
    else if (!strcmp (argument, "clear") && IS_IMMORTAL (ch))
    {
        clear_history ("all");
        char_act ("History cleared.", ch);
    }
    else 
    {
        char_act ("Invalid argument. Use {CHELP HISTORY{x to know about this stuff.", ch);
    }
}

void load_stat_record (void)
{
    FILE *fp;

    if ((fp = dfopen(TMP_PATH, STAT_FILE_NAME, "r")))
    {
        for (;;)
        {
            char   letter ;
            char * word   ;

            letter = fread_letter (fp) ;
            if (letter == '*')
            {
                fread_to_eol (fp) ;
                continue ;
            }
            if (letter != '#')
            {
                log_printf ("Load_stat_record: # not found in %s.", STAT_FILE_NAME) ;
                break ;
            }
            word = fread_word (fp) ;
            if (!str_cmp (word, "HISTORY")) fread_history (fp);
            else if (!str_cmp (word, "END")) break ;
            else
            {
                bug ("Load_stat_record: bad section.", 0) ;
                break ;
            }
        }
        fclose (fp) ;
    }
}

static void fread_history (FILE *fp)
{
    char       * word;
    bool         fMatch;

    clear_history ("all");

    for (;;)
    {
        word = feof(fp) ? "End" : fread_word(fp);
        fMatch = FALSE;

        switch (UPPER(word[0])) {
        case '*':
            fMatch = TRUE;
            fread_to_eol(fp);
            break;

        case 'A':
            KEY("Aggressions", stat_record.aggressions, fread_number(fp));
            KEY("Arrested", stat_record.arrested, fread_number(fp));
            KEY("AucItems", stat_record.auctionned_items, fread_number(fp));
            KEY("AucLimits", stat_record.auctionned_limits, fread_number(fp));
            break;

        case 'B':
            KEY("Bribes", stat_record.bribes, fread_number(fp));
            break;

        case 'C':
            KEY("CrimKilled", stat_record.criminals_killed, fread_number(fp));
            break;

        case 'D':
            KEY("Donations", stat_record.donation, fread_number(fp));
            KEY("DuelsProv", stat_record.duels_provoked, fread_number(fp));
            KEY("DuelsStart", stat_record.duels_started, fread_number(fp));
            break;

        case 'E':
            if (!str_cmp(word, "End"))
                return;
            break;

        case 'F':
            KEY("Forged", stat_record.crafted_items_forged, fread_number(fp));
            KEY("ForgedNOPK", stat_record.crafted_by_no_pk, fread_number(fp));
            break;

        case 'G':
            KEY("GoldMade", stat_record.gold_made, fread_number(fp));
            KEY("GoldStolen", stat_record.gold_stolen, fread_number(fp));
            KEY("GoldQuest", stat_record.gold_quest_reward, fread_number(fp));
            KEY("GoldGQ", stat_record.gold_gq_reward, fread_number(fp));
            KEY("GoodSpells", stat_record.good_spells, fread_number(fp));
            KEY("GQComp", stat_record.gq_completed, fread_number(fp));
            KEY("GQStart", stat_record.gq_started, fread_number(fp));
            break;

        case 'M':
            KEY("MobsKilled", stat_record.mobs_killed, fread_number(fp));
            break;

        case 'P':
            KEY("PCKilledM", stat_record.pc_killed_by_mobs, fread_number(fp));
            KEY("PK", stat_record.PK_count, fread_number(fp));
            KEY("PKonCR", stat_record.PK_on_CR, fread_number(fp));
            break;

        case 'Q':
            KEY("QuestComp", stat_record.quest_completed, fread_number(fp));
            KEY("QuestReq", stat_record.quest_requested, fread_number(fp));
            break;

        case 'R':
            KEY("Rewards", stat_record.rewards, fread_number(fp));
            break;

        case 'W':
            KEY("Wanted", stat_record.wanted_set, fread_number(fp));
            KEY("WarsComp", stat_record.wars_completed, fread_number(fp));
            KEY("WarsStart", stat_record.wars_started, fread_number(fp));
            KEY("WarsSurr", stat_record.wars_surrenders, fread_number(fp));
            KEY("WarsTimedOut", stat_record.wars_timed_out, fread_number(fp));
            KEY("WQMultStart", stat_record.wq_mult_started, fread_number(fp));
            KEY("WQMultWin", stat_record.wq_mult_win, fread_number(fp));
            KEY("WQPersStart", stat_record.wq_pers_started, fread_number(fp));
            KEY("WQPersWin", stat_record.wq_pers_win, fread_number(fp));
            break;
        }

        if (!fMatch) 
        {
            bug("Fread_history: no match.", 0);
            fread_to_eol(fp);
        }
    }
}

void save_stat_record (void)
{
    FILE        * fp;

    if (!(fp = dfopen(TMP_PATH, STAT_FILE_NAME, "w")))
    {
        log_printf("ERROR!!! save_stat_record: can't open output file %s/%s", TMP_PATH, STAT_FILE_NAME);
        return;
    }

    // #HISTORY section
    fprintf (fp, "#HISTORY\n");

    fprintf (fp, "Aggressions %d\n", stat_record.aggressions);
    fprintf (fp, "Arrested %d\n",    stat_record.arrested);
    fprintf (fp, "AucItems %d\n",    stat_record.auctionned_items);
    fprintf (fp, "AucLimits %d\n",   stat_record.auctionned_limits);

    fprintf (fp, "Bribes %d\n",      stat_record.bribes);

    fprintf (fp, "CrimKilled %d\n",  stat_record.criminals_killed);

    fprintf (fp, "Donations %d\n",   stat_record.donation);
    fprintf (fp, "DuelsProv %d\n",   stat_record.duels_provoked);
    fprintf (fp, "DuelsStart %d\n",  stat_record.duels_started);

    fprintf (fp, "Forged %d\n",      stat_record.crafted_items_forged);
    fprintf (fp, "ForgedNOPK %d\n",  stat_record.crafted_by_no_pk);

    fprintf (fp, "GoldMade %d\n",    stat_record.gold_made);
    fprintf (fp, "GoldStolen %d\n",  stat_record.gold_stolen);
    fprintf (fp, "GoldQuest %d\n",   stat_record.gold_quest_reward);
    fprintf (fp, "GoldGQ %d\n",      stat_record.gold_gq_reward);
    fprintf (fp, "GoodSpells %d\n",  stat_record.good_spells);
    fprintf (fp, "GQComp %d\n",      stat_record.gq_completed);
    fprintf (fp, "GQStart %d\n",     stat_record.gq_started);

    fprintf (fp, "MobsKilled %d\n",  stat_record.mobs_killed);

    fprintf (fp, "PCKilledM %d\n",   stat_record.pc_killed_by_mobs);
    fprintf (fp, "PK %d\n",          stat_record.PK_count);
    fprintf (fp, "PKonCR %d\n",      stat_record.PK_on_CR);

    fprintf (fp, "QuestComp %d\n",   stat_record.quest_completed);
    fprintf (fp, "QuestReq %d\n",    stat_record.quest_requested);

    fprintf (fp, "Rewards %d\n",     stat_record.rewards);

    fprintf (fp, "Wanted %d\n",      stat_record.wanted_set);
    fprintf (fp, "WarsComp %d\n",    stat_record.wars_completed);
    fprintf (fp, "WarsStart %d\n",   stat_record.wars_started);
    fprintf (fp, "WarsSurr %d\n",    stat_record.wars_surrenders);
    fprintf (fp, "WarsTimedOut %d\n",stat_record.wars_timed_out);

    fprintf (fp, "WQMultStart %d\n", stat_record.wq_mult_started);
    fprintf (fp, "WQMultWin %d\n",   stat_record.wq_mult_win);
    fprintf (fp, "WQPersStart %d\n", stat_record.wq_pers_started);
    fprintf (fp, "WQPersWin %d\n",   stat_record.wq_pers_win);

    fprintf (fp, "End\n\n");

    fprintf(fp, "#END\n");
    fclose(fp);
}

// --------------------------------------------------------------------------------
// this function is called from update.c (update_handler function) each CHAR_TICK
// --------------------------------------------------------------------------------
void service_data_update (void)
{
    // stat record
    if (--pulse_stat_record <= 0)
    {
        save_stat_record ();
        pulse_stat_record = PULSE_STAT_RECORD;
    }

    if (--pulse_save_gq <= 0)
    {
        save_gq ();
        pulse_save_gq = PULSE_SAVE_GQ;
    }
}

static void clear_history (char * arg)
{
    if ((arg[0] == '\0') || !str_cmp (arg, "all"))
    {
        stat_record.arrested             = 0;
        stat_record.auctionned_items     = 0;
        stat_record.auctionned_limits    = 0;
        stat_record.crafted_items_forged = 0;
        stat_record.crafted_by_no_pk     = 0;
        stat_record.criminals_killed     = 0;
        stat_record.gq_completed         = 0;
        stat_record.gq_started           = 0;
        stat_record.mobs_killed          = 0;
        stat_record.pc_killed_by_mobs    = 0;
        stat_record.PK_count             = 0;
        stat_record.PK_on_CR             = 0;
        stat_record.quest_completed      = 0;
        stat_record.quest_requested      = 0;
        stat_record.wanted_set           = 0;
        stat_record.wars_completed       = 0;
        stat_record.wars_started         = 0;
        stat_record.aggressions          = 0;
        stat_record.donation             = 0;
        stat_record.bribes               = 0;
        stat_record.good_spells          = 0;
        stat_record.wars_timed_out       = 0;
        stat_record.wars_surrenders      = 0;
        stat_record.duels_provoked       = 0;
        stat_record.duels_started        = 0;
        stat_record.gold_gq_reward       = 0;
        stat_record.gold_made            = 0;
        stat_record.gold_quest_reward    = 0;
        stat_record.gold_stolen          = 0;
        stat_record.rewards              = 0;
        stat_record.wq_mult_started      = 0;
        stat_record.wq_mult_win          = 0;
        stat_record.wq_pers_started      = 0;
        stat_record.wq_pers_win          = 0;
    }
}

DO_FUN (do_clanhistory)
{
}
