/* $Id: rgrep.c,v 1.666 2004/09/20 10:49:52 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/
 #include <sys/types.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>

#include "merc.h"
#include "db/db.h"
#include "grep.h"


extern char * percent_color(int value, int max_value);

static CMD_DATA rgrep_cmd_table[] =
{
//    name        do_fn             min_pos       min_level       qp  gold   min_clanstatus  extra
    { "help",     rgrep_help,      POS_DEAD,      92,              0,  0,     0,              0},
    { "sector",   rgrep_sector,    POS_DEAD,      92,              0,  0,     0,              0},
    { "flag",     rgrep_flag,      POS_DEAD,      92,              0,  0,     0,              0},
    { "reset",    rgrep_reset,     POS_DEAD,      92,              0,  0,     0,              0},

    { NULL}
};

static int cmp_room_vnum(const void *p1, const void *p2)
{
     return ((rgrep_room_t*) p1)->vnum - ((rgrep_room_t*) p2)->vnum;
}

const char * print_room_flags(flag64_t flag)
{
    const char   *room_flags;
    
    room_flags = str_printf("{D[%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s{D]{x",
                
        IS_SET(flag, ROOM_DARK             )   ? "{DD" : "{D-",
        IS_SET(flag, ROOM_NOMOB            )   ? "{DB" : "{D-",
        IS_SET(flag, ROOM_INDOORS          )   ? "{CI" : "{D-",
        IS_SET(flag, ROOM_PEACE            )   ? "{GP" : "{D-",
        IS_SET(flag, ROOM_PRIVATE          )   ? "{CP" : "{D-",
        IS_SET(flag, ROOM_SAFE             )   ? "{CS" : "{D-",
        IS_SET(flag, ROOM_SOLITARY         )   ? "{cS" : "{D-",
        IS_SET(flag, ROOM_PET_SHOP         )   ? "{YS" : "{D-",
        IS_SET(flag, ROOM_NORECALL         )   ? "{DR" : "{D-",
        IS_SET(flag, ROOM_IMP_ONLY         )   ? "{CI" : "{D-",
        IS_SET(flag, ROOM_GODS_ONLY        )   ? "{CG" : "{D-",
        IS_SET(flag, ROOM_HEROES_ONLY      )   ? "{CH" : "{D-",
        IS_SET(flag, ROOM_NEWBIES_ONLY     )   ? "{CN" : "{D-",
        IS_SET(flag, ROOM_LAW              )   ? "{CL" : "{D-",
        IS_SET(flag, ROOM_NOWHERE          )   ? "{DW" : "{D-",
        IS_SET(flag, ROOM_BANK             )   ? "{cB" : "{D-",
        IS_SET(flag, ROOM_NOMAGIC          )   ? "{DM" : "{D-",
        IS_SET(flag, ROOM_NOSUMMON         )   ? "{DS" : "{D-",
        IS_SET(flag, ROOM_BATTLE_ARENA     )   ? "{RA" : "{D-",
        IS_SET(flag, ROOM_REGISTRY         )   ? "{cR" : "{D-",
        IS_SET(flag, ROOM_CLANBANK         )   ? "{cC" : "{D-",
        IS_SET(flag, ROOM_FULL_PK          )   ? "{rP" : "{D-",
        IS_SET(flag, ROOM_GAMBLE           )   ? "{cG" : "{D-",
        IS_SET(flag, ROOM_SALTWATER_FISH   )   ? "{BS" : "{D-",
        IS_SET(flag, ROOM_FRESHWATER_FISH  )   ? "{BF" : "{D-",
        IS_SET(flag, ROOM_MAGIC            )   ? "{RM" : "{D-",
        IS_SET(flag, ROOM_NOTIMER          )   ? "{DT" : "{D-",
        IS_SET(flag, ROOM_NOEXPLORE        )   ? "{DE" : "{D-",
        IS_SET(flag, ROOM_NOHP             )   ? "{DH" : "{D-",
        IS_SET(flag, ROOM_NOMANA           )   ? "{DA" : "{D-",
        IS_SET(flag, ROOM_NOMOVE           )   ? "{DV" : "{D-",
        IS_SET(flag, ROOM_ROAD             )   ? "{cR" : "{D-",
        IS_SET(flag, ROOM_EXIT             )   ? "{cE" : "{D-"
        );
      
    return str_dup(room_flags);
} 

const char * printf_room(rgrep_room_t * room)
{
    const char * room_char;
    
    room_char = str_printf("{D[{w%-6d {D]{x  %s{x  %-3d {Y%-3d {D[{B%-3d %-3d{D]",
          
          room->vnum,
          room->name,
          room->space,
          room->light,
          room->heal_rate,
          room->mana_rate
          );
    
    return str_dup(room_char);
}

void fill_room_data(rgrep_room_t *room, ROOM_INDEX_DATA *pRoom)
{
          
    room->name           = str_dup(fmt_color_str(mlstr_mval(pRoom->name),20));       // *
//    room->owner          = pRoom->owner;      
    room->clan           = pRoom->clan;
    room->vnum           = pRoom->vnum;                                              // *
    room->room_flags     = pRoom->room_flags;
    room->light          = pRoom->light;                                             // *
    room->sector_type    = pRoom->sector_type;
    room->heal_rate      = pRoom->heal_rate;                                         // *
    room->mana_rate      = pRoom->mana_rate;                                         // *
//    room->affected       = pRoom->affected;
//    room->affected_by    = pRoom->affected_by;
    room->space          = pRoom->space;
//    room->rprogs         = pRoom->rprogs;     
//    room->rprog_flags    = pRoom->rprog_flags;
//    room->rprog_delay    = pRoom->rprog_delay;
}


DO_FUN (do_rgrep)
{
    if (!parse_command(ch, argument, rgrep_cmd_table))
    {
        show_command_list (ch, rgrep_cmd_table);
        char_puts (".\nUse {CHELP RGREP{x for more information.\n", ch);
    }
}

void rgrep_help    ( CHAR_DATA *ch, const char *arg )
{
}
 
void rgrep_sector (CHAR_DATA *ch, const char * argument )
{
     ROOM_INDEX_DATA *pRoomIndex;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *sector;
     
     rgrep_room_t    *rgrep_room;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bFShow;
     bool             bSShow;
     bool             bCShow;
     int              i;
     
     varr rgrep_rooms  = {sizeof (rgrep_room_t), 4};
     
     total = 0;
     
          
     bSort     = FALSE;
     bFShow    = FALSE;
     bSShow    = FALSE;
     bCShow    = FALSE;
     
     sort_mode = SORT_VNUM;
    
     
     argument = one_argument(argument, arg1, sizeof(arg1));
     
     if (flag_lookup(sector_types, arg1) == NULL)
     {
            char_act("Rgrep: invalid sector type.", ch);
            show_flags(ch, sector_types);
            return;
     }
     
     sector = str_dup(arg1);
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
          

           if      (!str_cmp(arg1, "sector"))                bSShow   = TRUE;
           else if (!str_cmp(arg1, "flags"))                 bFShow   = TRUE;
           else if (!str_cmp(arg1, "clans" ))                bCShow   = TRUE;
           else
           {
                 char_printf(ch,"Rgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     output = buf_new(-1);
     
     // fill room data
     for (i = 0; i < MAX_KEY_HASH; i++)
        for (pRoomIndex = room_index_hash[i]; pRoomIndex; pRoomIndex = pRoomIndex->next)
           //if (IS_SET(pRoomIndex->sector_type, flag_value(sector_types, arg1)))
           if (pRoomIndex->sector_type == flag_value(sector_types, sector))
           {
                 rgrep_room = varr_enew(&rgrep_rooms);
                 fill_room_data(rgrep_room, pRoomIndex);
           }
               
     // Checking for max parameter     
     
     // sort data                                  
          
     switch (sort_mode)
     {
          case SORT_VNUM:        varr_qsort(&rgrep_rooms, cmp_room_vnum);                   break;   
          default:                                                                          break;
     }
     
     
     for (i = 0; i < rgrep_rooms.nused; i++)
     {
           rgrep_room_t * room = varr_get(&rgrep_rooms, i);
           buf_printf(output,"%s",
                      
                      printf_room(room)
                      
                      ); 
           
           if (bCShow)
                 buf_printf(output, " %-10s", clan_name(room->clan));
                 
           if (bFShow)           
                 buf_printf(output," %s",print_room_flags(room->room_flags));
           
           if (bSShow)
                 buf_printf(output," %-10s", flag_string(sector_types, room->sector_type));
           
           buf_printf(output,"{x\n");           
     }
     
     
     // free data
     
     for (i = rgrep_rooms.nused - 1; 0; i-- )
     {
        rgrep_room_t * room = varr_get(&rgrep_rooms, i);
        free_string(room->name);
        free_string(room->owner);
        varr_del(&rgrep_rooms, (void*)room);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void rgrep_flag (CHAR_DATA *ch, const char * argument )
{
     ROOM_INDEX_DATA *pRoomIndex;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     const char      *flag;
     
     rgrep_room_t    *rgrep_room;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bFShow;
     bool             bSShow;
     bool             bCShow;
     int              i;
     
     varr rgrep_rooms  = {sizeof (rgrep_room_t), 4};
     
     total = 0;
     
          
     bSort     = FALSE;
     bFShow    = FALSE;
     bSShow    = FALSE;
     bCShow    = FALSE;
     
     sort_mode = SORT_VNUM;
    
     
     argument = one_argument(argument, arg1, sizeof(arg1));
     
     if (flag_lookup(room_flags, arg1) == NULL)
     {
            char_act("Rgrep: invalid room flags.", ch);
            show_flags(ch, room_flags);
            return;
     }
     
     flag = str_dup(arg1);
     
     while( argument[0] != '\0' )
     {
           argument = one_argument(argument, arg1, sizeof(arg1));
          
           if      (!str_cmp(arg1, "sector"))                bSShow   = TRUE;
           else if (!str_cmp(arg1, "flags"))                 bFShow   = TRUE;
           else if (!str_cmp(arg1, "clans"))                 bCShow   = TRUE;
           else
           {
                 char_printf(ch,"Rgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     output = buf_new(-1);
     
     // fill room data
     for (i = 0; i < MAX_KEY_HASH; i++)
        for (pRoomIndex = room_index_hash[i]; pRoomIndex; pRoomIndex = pRoomIndex->next)
           if (IS_SET(pRoomIndex->room_flags, flag_value(room_flags, flag)))
           {
                 rgrep_room = varr_enew(&rgrep_rooms);
                 fill_room_data(rgrep_room, pRoomIndex);
           }
               
     // Checking for max parameter     
     
     
     // sort data                                  
          
     switch (sort_mode)
     {
          case SORT_VNUM:        varr_qsort(&rgrep_rooms, cmp_room_vnum);                   break;   
          default:                                                                          break;
     }
     
     
     for (i = 0; i < rgrep_rooms.nused; i++)
     {
           rgrep_room_t * room = varr_get(&rgrep_rooms, i);
           
           buf_printf(output,"%s", printf_room(room)); 

           if (bCShow)
                 buf_printf(output, " %-10s", clan_name(room->clan));

           if (bFShow)      
                 buf_printf(output," %s",print_room_flags(room->room_flags));

           if (bSShow)
                 buf_printf(output," %-10s", flag_string(sector_types, room->sector_type));

           buf_printf(output,"{x\n");           
       }
     
     // free data
     
     for (i = rgrep_rooms.nused - 1; 0; i-- )
     {
        rgrep_room_t * room = varr_get(&rgrep_rooms, i);
        free_string(room->name);
        free_string(room->owner);
        varr_del(&rgrep_rooms, (void*)room);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}

void rgrep_reset (CHAR_DATA *ch, const char * argument )
{
     ROOM_INDEX_DATA *pRoomIndex;
     RESET_DATA      *pReset;
     BUFFER          *output;
     char             arg1[MAX_INPUT_LENGTH];
     int              total;
     
     rgrep_room_t    *rgrep_room;
     flag32_t         sort_mode;
     bool             bSort;
     bool             bFShow;
     bool             bSShow;
     bool             bCShow;
     int              i;
     int              reset;
     int              rvnum;
     
     varr rgrep_rooms  = {sizeof (rgrep_room_t), 4};
     
     total = 0;
     
          
     bSort     = FALSE;
     bFShow    = FALSE;
     bSShow    = FALSE;
     bCShow    = FALSE;
     
     sort_mode = SORT_VNUM;
    
     reset = -1;
     
     argument = one_argument(argument, arg1, sizeof(arg1));

 /*
 * Reset commands:
 *   '*': comment
 *   'M': read a mobile
 *   'O': read an object
 *   'P': put object in object
 *   'G': give object to mobile
 *   'E': equip object to mobile
 *   'D': set state of door
 *   'R': randomize room exits
 *   'S': stop (end of list)
 */
      
     if      (!str_cmp(arg1, "mob"))               reset = 1;
     else if (!str_cmp(arg1, "obj"))               reset = 0;
     else
     {
            char_printf(ch,"Rgrep: unknown reset type: %s.\n",arg1);
            return;
     }
     
     argument = one_argument(argument, arg1, sizeof(arg1));
     
     if (!is_number(arg1) || atoi(arg1) < 1)
     {
            char_printf(ch,"Rgrep: argument must be number and greater then 0.\n");
            return;
     }
     
     rvnum = atoi(arg1);
     
     while( argument[0] != '\0' )
     {
           log_printf("%s", argument);
           argument = one_argument(argument, arg1, sizeof(arg1));
          
           if      (!str_cmp(arg1, "sector"))                bSShow   = TRUE;
           else if (!str_cmp(arg1, "flags"))                 bFShow   = TRUE;
           else if (!str_cmp(arg1, "clans"))                 bCShow   = TRUE;
           else
           {
                 char_printf(ch,"Rgrep: invalid command: %s.\n",arg1);
                 return;
           }
     }
     
     output = buf_new(-1);
     
     // fill room data
     for (i = 0; i < MAX_KEY_HASH; i++)
        for (pRoomIndex = room_index_hash[i]; pRoomIndex; pRoomIndex = pRoomIndex->next)
            for( pReset = pRoomIndex->reset_first; pReset; pReset = pReset->next )
            {
                  if (reset == 1)
                  {
                        if (pReset->command == 'M' && pReset->arg1 == rvnum)
                        {
                              rgrep_room = varr_enew(&rgrep_rooms);
                              fill_room_data(rgrep_room, pRoomIndex);
                        }
                  }
                  else if (reset == 0)
                  {
                        switch( pReset->command )
                        {
                         case 'O':
                         case 'E':
                         case 'P':
                         case 'G':
                               if(pReset->arg1 == rvnum)
                               {
                                    rgrep_room = varr_enew(&rgrep_rooms);
                                    fill_room_data(rgrep_room, pRoomIndex);
                               }
                         break;
                         default: break;
                       }
                  }
            }     
                           
     // Checking for max parameter     
     
     
     // sort data                                  
     switch (sort_mode)
     {
          case SORT_VNUM:        varr_qsort(&rgrep_rooms, cmp_room_vnum);                   break;   
          default:                                                                          break;
     }
     
     
     for (i = 0; i < rgrep_rooms.nused; i++)
     {
           rgrep_room_t * room = varr_get(&rgrep_rooms, i);
           
           buf_printf(output,"%s", printf_room(room)); 

           if (bCShow)
                 buf_printf(output, " %-10s", clan_name(room->clan));

           if (bFShow)      
                 buf_printf(output," %s",print_room_flags(room->room_flags));

           if (bSShow)
                 buf_printf(output," %-10s", flag_string(sector_types, room->sector_type));

           buf_printf(output,"{x\n");           
       }
     
     // free data
     
     for (i = rgrep_rooms.nused - 1; 0; i-- )
     {
        rgrep_room_t * room = varr_get(&rgrep_rooms, i);
        free_string(room->name);
        free_string(room->owner);
        varr_del(&rgrep_rooms, (void*)room);
     }
     
     page_to_char(buf_string(output), ch);
     buf_free(output);

}
