/* $Id: room_cmds.c,v 1.666 2004/09/20 10:49:52 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "merc.h"
#include "argument.h"
#include "mob_cmds.h"
#include "room_cmds.h"
#include "mob_prog.h"

DECLARE_DO_FUN(do_look      );

/*
 * ROOMcommand table.
 */
const   struct  room_cmd_type   room_cmd_table  [] =
{
	{   "asound",       do_rpasound},
	{   "gecho",        do_rpgecho},
	{   "zecho",        do_rpzecho},
	{   "echo",         do_rpecho},
	{   "echoaround",   do_rpechoaround},
	{   "echoat",       do_rpechoat},
	{   "mload",        do_rpmload},
	{   "oload",        do_rpoload},
	{   "purge",        do_rppurge},
	{   "transfer",     do_rptransfer},
	{   "gtransfer",    do_rpgtransfer},
	{   "otransfer",    do_rpotransfer},
	{   "force",        do_rpforce},
	{   "gforce",       do_rpgforce},
	{   "vforce",       do_rpvforce},
	{   "damage",       do_rpdamage},
	{   "remember",     do_rpremember},
	{   "forget",       do_rpforget},
	{   "delay",        do_rpdelay},
	{   "cancel",       do_rpcancel},
	{   "call",         do_rpcall},
	{   "remove",       do_rpremove},
	{   str_empty,      0},
};

/*
 * ROOMprog section
 */

/*
 * Room command interpreter. Implemented separately for security and speed
 * reasons. A trivial hack of interpret()
 */
void room_interpret( ROOM_INDEX_DATA *room, const char *argument )
{
	char buf[MAX_STRING_LENGTH], command[MAX_INPUT_LENGTH];
	int cmd;

	argument = one_argument( argument, command, sizeof(command));

	/*
	 * Look for command in command table.
	 */
	for (cmd = 0; room_cmd_table[cmd].name[0] != '\0'; cmd++)
	{
		if (command[0] == room_cmd_table[cmd].name[0]
			&&   !str_prefix( command, room_cmd_table[cmd].name ))
		{
			(*room_cmd_table[cmd].room_fun) ( room, argument );
			tail_chain( );
			return;
		}
	}
	sprintf( buf, "Room_interpret: invalid cmd from room %d: '%s'", room->vnum, command );
	bug( buf, 0 );
}

void do_room( ROOM_INDEX_DATA *room, const char *argument )
{
	room_interpret( room, argument );
}


/*
 * Prints the argument to all active players in the game
 *
 * Syntax: room gecho [string]
 */
void do_rpgecho( ROOM_INDEX_DATA *room, const char *argument )
{
	DESCRIPTOR_DATA *d;

	if (argument[0] == '\0')
	{
		bug( "RpGEcho: missing argument from vnum %d",
			 room->vnum );
		return;
	}

	for (d = descriptor_list; d; d = d->next)
	{
		if (d->connected == CON_PLAYING)
		{
			if (IS_IMMORTAL(d->character))
				send_to_char( "Room echo> ", d->character );
			send_to_char( argument, d->character );
			send_to_char( "\n\r", d->character );
		}
	}
}

/*
 * Prints the argument to all players in the same area as the room
 *
 * Syntax: room zecho [string]
 */
void do_rpzecho( ROOM_INDEX_DATA *room, const char *argument )
{
	DESCRIPTOR_DATA *d;

	if (argument[0] == '\0')
	{
		bug( "RpZEcho: missing argument from vnum %d",
			 room->vnum );
		return;
	}

	for (d = descriptor_list; d; d = d->next)
	{
		if (d->connected == CON_PLAYING 
			&&   d->character->in_room != NULL 
			&&   d->character->in_room->area == room->area)
		{
			if (IS_IMMORTAL(d->character))
				send_to_char( "Room echo> ", d->character );
			send_to_char( argument, d->character );
			send_to_char( "\n\r", d->character );
		}
	}
}

/*
 * Prints the argument to all the rooms aroud the room
 *
 * Syntax: room asound [string]
 */
void do_rpasound( ROOM_INDEX_DATA *room, const char *argument )
{
	int              door;

	if (argument[0] == '\0')
		return;

	for (door = 0; door < 6; door++)
	{
		EXIT_DATA       *pexit;

		if (( pexit = room->exit[door] ) != NULL
			&&   pexit->to_room.r != NULL
			&&   pexit->to_room.r != room
			&&   pexit->to_room.r->people != NULL)
		{
			act( argument, pexit->to_room.r->people, NULL, NULL, TO_ROOM );
			act( argument, pexit->to_room.r->people, NULL, NULL, TO_CHAR );
		}
	}
	return;

}

/*
 * Prints the message to everyone in the room other than the victim
 *
 * Syntax: room echoaround [victim] [string]
 */

void do_rpechoaround( ROOM_INDEX_DATA *room, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;

	argument = one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0')
		return;

	if (( victim=get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
		return;

	act( argument, victim, NULL, victim, TO_NOTVICT );
}

/*
 * Prints the message to only the victim
 *
 * Syntax: room echoat [victim] [string]
 */
void do_rpechoat( ROOM_INDEX_DATA *room, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;

	argument = one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
		return;

	if (( victim = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
		return;

	act( argument, victim, NULL, NULL, TO_CHAR );
}

/*
 * Prints the message to the room at large
 *
 * Syntax: rpecho [string]
 */
void do_rpecho( ROOM_INDEX_DATA *room, const char *argument )
{
	if (argument[0] == '\0')
		return;

	if (!room->people)
		return;

	act( argument, room->people, NULL, NULL, TO_ROOM );
	act( argument, room->people, NULL, NULL, TO_CHAR );
}

/*
 * Lets the room load a mobile.
 *
 * Syntax: room mload [vnum]
 */
void do_rpmload( ROOM_INDEX_DATA *room, const char *argument )
{
	char            arg[ MAX_INPUT_LENGTH ];
	MOB_INDEX_DATA *pMobIndex;
	CHAR_DATA      *victim;
	int vnum;

	one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0' || !is_number(arg))
		return;

	vnum = atoi(arg);
	if (( pMobIndex = get_mob_index( vnum ) ) == NULL)
	{
		sprintf( arg, "Rpmload: bad mob index (%d) from room %d",
				 vnum, room->vnum );
		bug( arg, 0 );
		return;
	}
	victim = create_mob( pMobIndex );
	char_to_room( victim, room );
	return;
}

/*
 * Lets the room load an object
 *
 * Syntax: room oload [vnum] [level]
 */
void do_rpoload( ROOM_INDEX_DATA *room, const char *argument )
{
	char arg1[ MAX_INPUT_LENGTH ];
	char arg2[ MAX_INPUT_LENGTH ];
	char arg3[ MAX_INPUT_LENGTH ];
	OBJ_INDEX_DATA *pObjIndex;
	OBJ_DATA       *obj;
	int             level;

	argument = one_argument(argument, arg1, sizeof(arg1));
	argument = one_argument(argument, arg2, sizeof(arg2));
	one_argument( argument, arg3, sizeof(arg3));

	if (arg1[0] == '\0' || !is_number( arg1 ) || arg2[0] == '\0' 
		|| !is_number( arg2 ))
	{
		bug( "Rpoload - Bad syntax from vnum %d.",
			 room->vnum );
		return;
	}

	level = atoi( arg2 );
	if (level < 0 || level > LEVEL_IMMORTAL)
	{
		bug( "Rpoload - Bad level from vnum %d.", room->vnum );
		return;
	}

	if (( pObjIndex = get_obj_index( atoi( arg1 ) ) ) == NULL)
	{
		bug( "Rpoload - Bad vnum arg from vnum %d.", room->vnum );
		return;
	}

	obj = create_obj( pObjIndex, level );
	obj_to_room( obj, room );

	return;
}

/*
 * Lets the room purge all objects npcs in the room,
 * or purge a specified object or mob in the room.
 *
 * syntax room purge {target}
 */
void do_rppurge( ROOM_INDEX_DATA *room, const char *argument )
{
	char       arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;
	OBJ_DATA  *obj;

	one_argument(argument, arg, sizeof(arg));

	if (arg[0] == '\0')
	{
		/* 'purge' */
		CHAR_DATA *vnext;
		OBJ_DATA  *obj_next;

		for (victim = room->people; victim != NULL; victim = vnext)
		{
			vnext = victim->next_in_room;
			if (IS_NPC( victim ) 
				&&   !IS_SET(victim->pIndexData->act, ACT_NOPURGE))
				extract_char( victim, TRUE );
		}

		for (obj = room->contents; obj != NULL; obj = obj_next)
		{
			obj_next = obj->next_content;
			if (!IS_SET(obj->extra_flags, ITEM_NOPURGE))
				extract_obj( obj );
		}

		return;
	}

	if (( victim = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
	{
		if (( obj = get_obj_here_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ))
		{
			extract_obj( obj );
		}
		else
		{
			bug( "Rppurge - Bad argument from vnum %d.", room->vnum );
		}
		return;
	}

	if (!IS_NPC( victim ))
	{
		bug( "Rppurge - Purging a PC from vnum %d.", room->vnum );
		return;
	}
	extract_char( victim, TRUE );
	return;
}

/*
 * Lets the room transfer people.  The 'all' argument transfers
 *  everyone in the room to the specified location
 *
 * Syntax: room transfer [target|'all'] [location]
 */
void do_rptransfer( ROOM_INDEX_DATA *room, const char *argument )
{
	char             arg1[ MAX_INPUT_LENGTH ];
	char             arg2[ MAX_INPUT_LENGTH ];
	char         buf[MAX_STRING_LENGTH];
	ROOM_INDEX_DATA *location;
	CHAR_DATA       *victim;
	OBJ_DATA        *tobj;

	argument = one_argument(argument, arg1, sizeof(arg1));
	argument = one_argument(argument, arg2, sizeof(arg2));

        tobj=NULL;

	if (arg1[0] == '\0')
	{
		bug( "Rptransfer - Bad syntax from vnum %d.", room->vnum );
		return;
	}

	if (!str_cmp( arg1, "all" ))
	{
		CHAR_DATA *victim_next;

		for (victim = room->people; victim != NULL; victim = victim_next)
		{
			victim_next = victim->next_in_room;
			if (!IS_NPC(victim))
			{
				sprintf( buf, "%s %s", victim->name, arg2 );
				do_rptransfer( room, buf );
			}
		}
		return;
	}

	/*
	 * Thanks to Grodyn for the optional location parameter.
	 */
	if (arg2[0] == '\0')
	{
		location = room;
	}
	else
	{
		if (is_number(arg2))
			location = get_room_index( atoi(arg2) );
		else if ((victim = get_char_world_ext((CHAR_DATA *) room, arg2, GET_F_CHAR_IS_ROOM )) != NULL)
			location = victim->in_room;
		else if (( tobj = get_obj_world_ext((CHAR_DATA *) tobj, arg2, GET_F_CHAR_IS_OBJ )) != NULL)
			location = tobj->in_room;
		else
		{
			bug( "Rptransfer - No such location from vnum %d.", room->vnum );
			return;
		}

		if (room_is_private( location ))
			return;
	}

	if (( victim = get_char_world_ext((CHAR_DATA *) room, arg1, GET_F_CHAR_IS_ROOM)) == NULL)
		return;

	if (victim->in_room == NULL)
		return;

	if (victim->fighting != NULL)
		stop_fighting( victim, TRUE );
	char_from_room( victim );
	char_to_room( victim, location );
	do_look( victim, "auto" );

	return;
}

/*
 * Lets the room transfer all chars in same group as the victim.
 *
 * Syntax: room gtransfer [victim] [location]
 */
void do_rpgtransfer( ROOM_INDEX_DATA *room, const char *argument )
{
	char             arg1[ MAX_INPUT_LENGTH ];
	char             arg2[ MAX_INPUT_LENGTH ];
	char         buf[MAX_STRING_LENGTH];
	CHAR_DATA       *who, *victim, *victim_next;

	argument = one_argument(argument, arg1, sizeof(arg1));
	argument = one_argument(argument, arg2, sizeof(arg2));

	if (arg1[0] == '\0')
	{
		bug( "Rpgtransfer - Bad syntax from vnum %d.", room->vnum );
		return;
	}

	if ((who = get_char_room_ext((CHAR_DATA *) room, arg1, GET_F_CHAR_IS_ROOM)) == NULL)
		return;

	for (victim = room->people; victim; victim = victim_next)
	{
		victim_next = victim->next_in_room;
		if (is_same_group( who,victim ))
		{
			sprintf( buf, "%s %s", victim->name, arg2 );
			do_rptransfer( room, buf );
		}
	}
	return;
}

/*
 * Lets the room force someone to do something. Must be mortal level
 * and the all argument only affects those in the room.
 *
 * Syntax: room force [victim] [commands]
 */
void do_rpforce( ROOM_INDEX_DATA *room, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];

	argument = one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "Rpforce - Bad syntax from vnum %d.", room->vnum );
		return;
	}

	if (!str_cmp( arg, "all" ))
	{
		CHAR_DATA *vch;
		CHAR_DATA *vch_next;

		for (vch = char_list; vch != NULL; vch = vch_next)
		{
			vch_next = vch->next;

			if (vch->in_room == room && !IS_IMMORTAL( vch ))
				interpret( vch, argument );
		}
	}
	else
	{
		CHAR_DATA *victim;

		if (( victim = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
			return;

		interpret( victim, argument );
	}

	return;
}

/*
 * Lets the room force a group something. Must be mortal level.
 *
 * Syntax: room gforce [victim] [commands]
 */
void do_rpgforce( ROOM_INDEX_DATA *room, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim, *vch, *vch_next;

	argument = one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "RpGforce - Bad syntax from vnum %d.", room->vnum );
		return;
	}

	if (( victim = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
		return;

	for (vch = victim->in_room->people; vch != NULL; vch = vch_next)
	{
		vch_next = vch->next_in_room;

		if (is_same_group(victim,vch))
			interpret( vch, argument );
	}
	return;
}

/*
 * Forces all mobiles of certain vnum to do something
 *
 * Syntax: room vforce [vnum] [commands]
 */
void do_rpvforce( ROOM_INDEX_DATA *room, const char *argument )
{
	CHAR_DATA *victim, *victim_next;
	char arg[ MAX_INPUT_LENGTH ];
	int vnum;

	argument = one_argument( argument, arg, sizeof(arg));

	if (arg[0] == '\0' || argument[0] == '\0')
	{
		bug( "RpVforce - Bad syntax from vnum %d.", room->vnum );
		return;
	}

	if (!is_number( arg ))
	{
		bug( "RpVforce - Non-number argument vnum %d.", room->vnum );
		return;
	}

	vnum = atoi( arg );

	for (victim = char_list; victim; victim = victim_next)
	{
		victim_next = victim->next;
		if (IS_NPC(victim) && victim->pIndexData->vnum == vnum
			&& victim->fighting == NULL)
			interpret( victim, argument );
	}
	return;
}

/*
 * Lets room cause unconditional damage to someone. Nasty, use with caution.
 * Also, this is silent, you must show your own damage message...
 *
 * Syntax: room damage [victim] [min] [max] {kill}
 */
void do_rpdamage( ROOM_INDEX_DATA *room, const char *argument )
{
	CHAR_DATA *victim = NULL, *victim_next;
	char target[ MAX_INPUT_LENGTH ],
	min[ MAX_INPUT_LENGTH ],
	max[ MAX_INPUT_LENGTH ];
	int low, high;
	bool fAll = FALSE, fKill = FALSE;

	argument = one_argument(argument, target, sizeof(target));
	argument = one_argument( argument, min, sizeof(min));
	argument = one_argument( argument, max, sizeof(max));

	if (target[0] == '\0')
	{
		bug( "RpDamage - Bad syntax from vnum %d.", room->vnum );
		return;
	}
	if (!str_cmp( target, "all" ))
		fAll = TRUE;
	else if ((victim = get_char_room_ext((CHAR_DATA *) room, target, GET_F_CHAR_IS_ROOM)) == NULL)
		return;

	if (is_number( min ))
		low = atoi( min );
	else
	{
		bug( "RpDamage - Bad damage min vnum %d.", room->vnum );
		return;
	}
	if (is_number( max ))
		high = atoi( max );
	else
	{
		bug( "RpDamage - Bad damage max vnum %d.", room->vnum );
		return;
	}
	one_argument(argument, target, sizeof(target));

	/*
	 * If kill parameter is omitted, this command is "safe" and will not
	 * kill the victim.
	 */

	if (target[0] != '\0')
		fKill = TRUE;
	if (fAll)
	{
		for (victim = room->people; victim; victim = victim_next)
		{
			victim_next = victim->next_in_room;
			damage( victim, victim, fKill ? number_range(low,high) : UMIN(victim->hit,number_range(low,high)),
					TYPE_UNDEFINED, DAM_NONE, FALSE );
		}
	}
	else
		damage( victim, victim, 
				fKill ? 
				number_range(low,high) : UMIN(victim->hit,number_range(low,high)),
				TYPE_UNDEFINED, DAM_NONE, FALSE );
	return;
}

/*
 * Lets the room remember a target. The target can be referred to
 * with $q and $Q codes in ROOMprograms. See also "room forget".
 *
 * Syntax: room remember [victim]
 */
void do_rpremember( ROOM_INDEX_DATA *room, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];
	one_argument(argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		room->rprog_target = get_char_world( NULL, arg );
	else
		bug( "RpRemember: missing argument from vnum %d.", room->vnum );
}

/*
 * Reverse of "room remember".
 *
 * Syntax: room forget
 */
void do_rpforget( ROOM_INDEX_DATA *room, const char *argument )
{
	room->rprog_target = NULL;
}

/*
 * Sets a delay for ROOMprogram execution. When the delay time expires,
 * the room is checked for a ROOMprogram with DELAY trigger, and if
 * one is found, it is executed. Delay is counted in PULSE_AREA
 *
 * Syntax: room delay [pulses]
 */
void do_rpdelay( ROOM_INDEX_DATA *room, const char *argument )
{
	char arg[ MAX_INPUT_LENGTH ];

	one_argument(argument, arg, sizeof(arg));
	if (!is_number( arg ))
	{
		bug( "RpDelay: invalid arg from vnum %d.", room->vnum );
		return;
	}
	room->rprog_delay = atoi( arg );
}

/*
 * Reverse of "room delay", deactivates the timer.
 *
 * Syntax: room cancel
 */
void do_rpcancel( ROOM_INDEX_DATA *room, const char *argument )
{
	room->rprog_delay = -1;
}
/*
 * Lets the room call another ROOMprogram within a ROOMprogram.
 * This is a crude way to implement subroutines/functions. Beware of
 * nested loops and unwanted triggerings... Stack usage might be a problem.
 * Characters and objects referred to must be in the room.
 *
 * Syntax: room call [vnum] [victim|'null'] [object1|'null'] [object2|'null']
 *
 */
void do_rpcall( ROOM_INDEX_DATA *room, const char *argument )
{   
	char arg[ MAX_INPUT_LENGTH ];
	CHAR_DATA *vch;
	OBJ_DATA *obj1, *obj2;
	MPCODE *prg;
	extern void    program_flow      (int pvnum, CHAR_DATA *mob, OBJ_DATA *obj, ROOM_INDEX_DATA *room,
									  CHAR_DATA *ch, const void *arg1,  const void *arg2);
	extern MPCODE *get_prog_index    ( int vnum, int type );

	argument = one_argument( argument, arg, sizeof(arg));
	if (arg[0] == '\0')
	{
		bug( "RpCall: missing arguments from vnum %d.", room->vnum );
		return;
	}
	if (( prg = get_prog_index( atoi(arg), PRG_RPROG ) ) == NULL)
	{
		bug( "RpCall: invalid prog from vnum %d.", room->vnum );
		return;
	}
	vch = NULL;
	obj1 = obj2 = NULL;
	argument = one_argument( argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		vch = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM);
	argument = one_argument( argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		obj1 = get_obj_here_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM);
	argument = one_argument( argument, arg, sizeof(arg));
	if (arg[0] != '\0')
		obj2 = get_obj_here_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM);
	program_flow( prg->vnum, NULL, NULL, room, vch, (void *)obj1, (void *)obj2 );
	
}

/*
 * Lets the room transfer an object. The object must be in the room.
 *
 * Syntax: room otransfer [item name] [location]
 */
void do_rpotransfer( ROOM_INDEX_DATA *room, const char *argument )
{
	OBJ_DATA *obj, *tobj;
	ROOM_INDEX_DATA *location;
	char arg[ MAX_INPUT_LENGTH ];
	char buf[ MAX_INPUT_LENGTH ];
	CHAR_DATA *victim;

	argument = one_argument( argument, arg, sizeof(arg));

        obj = NULL;

	if (arg[0] == '\0')
	{
		bug( "RpOTransfer - Missing argument from vnum %d.", room->vnum );
		return;
	}
	one_argument( argument, buf, sizeof(buf));

	if (is_number( buf ))
		location = get_room_index( atoi(buf) );
	else if ((victim = get_char_world_ext((CHAR_DATA *) room, buf, GET_F_CHAR_IS_ROOM)) != NULL)
		location = victim->in_room;
        else if (( tobj = get_obj_world_ext((CHAR_DATA *) obj, arg, GET_F_CHAR_IS_OBJ )) != NULL)
		location = tobj->in_room;
	else
	{
		bug( "RpOTransfer - No such location from vnum %d.", room->vnum );
		return;
	}

	if ((obj = get_obj_here_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM)) == NULL)
		return;

	if (obj->carried_by == NULL)
		obj_from_room( obj );
	else
	{
		if (obj->wear_loc != WEAR_NONE)
			unequip_char( obj->carried_by, obj );
		obj_from_char( obj );
	}
	obj_to_room( obj, location );
}

/*
 * Lets the room strip an object or all objects from the victim.
 * Useful for removing e.g. quest objects from a character.
 *
 * Syntax: room remove [victim] [object vnum|'all']
 */
void do_rpremove( ROOM_INDEX_DATA *room, const char *argument )
{
	CHAR_DATA *victim;
	OBJ_DATA *obj, *obj_next;
	sh_int vnum = 0;
	bool fAll = FALSE;
	char arg[ MAX_INPUT_LENGTH ];

	argument = one_argument( argument, arg, sizeof(arg));
	if (( victim = get_char_room_ext((CHAR_DATA *) room, arg, GET_F_CHAR_IS_ROOM) ) == NULL)
		return;

	one_argument(argument, arg, sizeof(arg));
	if (!str_cmp( arg, "all" ))
		fAll = TRUE;
	else if (!is_number( arg ))
	{
		bug ( "RpRemove: Invalid object from vnum %d.", room->vnum );
		return;
	}
	else
		vnum = atoi( arg );

	for (obj = victim->carrying; obj; obj = obj_next)
	{
		obj_next = obj->next_content;
		if (fAll || obj->pIndexData->vnum == vnum)
		{
			unequip_char( victim, obj );
			obj_from_char( obj );
			extract_obj( obj );
		}
	}
}

