/* $Id: shamans.c,v 1.666 2004/09/20 10:49:53 shrike Exp $ */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

// --------------------------------------------------------------------------------------
// Notes:
// 1. In each tattoo_xxx (spell function) should be section of tattoo initialization, in
//    which tattoo_obj is supplied with hp/mana/ac/saves etc modificators
//
// 2. How to add new paint color: see "shamans.h"
//
// 3. To add a tattoo spell_function: 
//    - add function in this file using the following template:
// SPELL_FUN(tattoo_template)
// {
//     bool improved = check_violet_stars (ch);
//     
//     if (target == TAR_CREATING_TATTOO)
//     {
//         char_act ("Creating tattoo.", ch);
//         return;
//     }
//     char_act ("Activating tattoo", ch);
// }
//    - add refercences into db/spellfn.c, db/spellfn.h
//
// 4. Object with type TYPE_SHAMAN_TATTOO has following values:
//     value[0]: tattoo's power (spell_level for activation)
//     value[1]: activations left (-1 means infinite)
//     value[2]: sn
// 
// 5. Object with type TYPE_PAINT has following values:
//     value[0]: color (see flag_t paint_color_int_flags[] in tables.c)
//     value[1]: portions left (-1 means infinite)
//     value[2]: power (not used yet)
//
// --------------------------------------------------------------------------------------

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "merc.h"
#include "shamans.h"
#include "war.h"
#include "conquer.h"
#include "db/dofun.h"
#include "db/spellfn.h"

extern bool target_ioga(CHAR_DATA *ch, int sn, int *ptarget, void **pvo, int *poff, int *pdoor);
extern bool can_cast(CHAR_DATA *ch, CHAR_DATA *victim);
extern bool check_components(CHAR_DATA *ch, int sn, int chance);
extern void multi_hit(CHAR_DATA *ch, CHAR_DATA *victim, int dt);
extern void update_pos(CHAR_DATA *victim);
extern bool is_safe(CHAR_DATA *ch, CHAR_DATA *victim);
extern bool is_safe_nomessage(CHAR_DATA *ch, CHAR_DATA *victim);
extern bool damage (CHAR_DATA * ch, CHAR_DATA * victim, int dam, int dt, int dam_type, int dam_flags);
extern void do_skills_org(CHAR_DATA *ch, const char *argument, int type);
extern void interpret_social(social_t *soc, CHAR_DATA *ch, const char *argument);
extern DECLARE_DO_FUN(do_yell);

extern const char * target_name;
extern const char * extra_arguments;

#define TAR_CREATING_TATTOO    765

static void create_affect (void * obj, int sn, int level, int location, 
                    int mod, flag64_t bitvector, bool improved);
static void create_resist (void * obj, int sn, int level, int location, 
                    int mod, flag64_t bitvector, bool improved);

typedef struct tcolor_t tcolor_t;
struct tcolor_t
{
    int          color;
    const char * name;
};

static tcolor_t tattoo_colors[TATTOO_COLOR_MAX] = 
{
    { F_TATTOO_COLOR_RED,      "red"},
    { F_TATTOO_COLOR_BLUE,     "blue"},
    { F_TATTOO_COLOR_GREEN,    "green"},
    { F_TATTOO_COLOR_MAGENTA,  "magenta"},
    { F_TATTOO_COLOR_WHITE,    "white"},
    { F_TATTOO_COLOR_BLACK,    "black"},
    { F_TATTOO_COLOR_GOLD,     "gold"},
    { F_TATTOO_COLOR_SILVER,   "silver"},
    { F_TATTOO_COLOR_BROWN,    "brown"}
};

typedef struct appear_table_t appear_table_t;
struct appear_table_t
{
    const char * name;
    flag64_t     loc;
    flag64_t     can_wear;
};

static appear_table_t tattoo_wear_pos[] =
{
    { "head",           WEAR_HEAD,     ITEM_WEAR_HEAD    },
    { "neck",           WEAR_NECK_1,   ITEM_WEAR_NECK    },
    { "wrist_left",     WEAR_WRIST_L,  ITEM_WEAR_WRIST   },
    { "wrist_right",    WEAR_WRIST_R,  ITEM_WEAR_WRIST   },
    { "arms",           WEAR_ARMS,     ITEM_WEAR_ARMS    },
    { "body",           WEAR_BODY,     ITEM_WEAR_BODY    },
    { "waist",          WEAR_WAIST,    ITEM_WEAR_WAIST   },
    { "hands",          WEAR_HANDS,    ITEM_WEAR_HANDS   },
    { "legs",           WEAR_LEGS,     ITEM_WEAR_LEGS    },
    { "feet",           WEAR_FEET,     ITEM_WEAR_FEET    },
    { "face",           WEAR_FACE,     ITEM_WEAR_FACE    },
    { NULL }
};

flag_to_int_t sect_flags[] =
{
    { F_SECT_INSIDE,        SECT_INSIDE },
    { F_SECT_CITY,          SECT_CITY },
    { F_SECT_FIELD,         SECT_FIELD },
    { F_SECT_FOREST,        SECT_FOREST },
    { F_SECT_HILLS,         SECT_HILLS },
    { F_SECT_MOUNTAIN,      SECT_MOUNTAIN },
    { F_SECT_WATER_SWIM,    SECT_WATER_SWIM },
    { F_SECT_WATER_NOSWIM,  SECT_WATER_NOSWIM },
    { F_SECT_UNUSED,        SECT_UNUSED },
    { F_SECT_AIR,           SECT_AIR },
    { F_SECT_DESERT,        SECT_DESERT },
    { FLAG_TO_INT_MAX }
};

// ---------------------------------------------------------------------------------------
// functions for tattooes
// ---------------------------------------------------------------------------------------
varr tattooes  = {sizeof (tattoo_t), 4};

tattoo_t * tattoo_new()
{
    tattoo_t * tattoo;

    tattoo        = varr_enew (&tattooes);
    tattoo->name  = str_empty;
    tattoo->desc  = NULL;
    tattoo->max_activations = -1;
    
    return tattoo ;
}

void tattoo_free (tattoo_t * tattoo)
{
    free_string (tattoo->name);
    mlstr_free (tattoo->desc);
    free (tattoo);
}

int tattoonumber_lookup (const char * name)
{
    int tn;

    if (IS_NULLSTR(name))
                return -1;

    for (tn = 0; tn < tattooes.nused; tn++)
        if (!str_prefix(name, TATTOO(tn)->name))
                return tn;

    return -1;
}

const char * tattoo_name (int tn)
{
    if ((tn < 0) || (tn > tattooes.nused -1 ))
        return "None";
    else
        return TATTOO(tn)->name; 
}

// ---------------------------------------------------------------------------------------
// functions for paints
// ---------------------------------------------------------------------------------------
varr paints  = {sizeof (paint_t), 4};

paint_t * paint_new()
{
    paint_t * paint;

    paint               = varr_enew (&paints);
    paint->name         = str_empty;
    paint->obj_name     = str_empty;
    paint->desc         = NULL;
    paint->short_desc   = NULL;
    paint->max_portions = -1;
    
    return paint;
}

void paint_free (paint_t * paint)
{
    free_string (paint->name);
    free_string (paint->obj_name);
    mlstr_free (paint->desc);
    mlstr_free (paint->short_desc);
    free (paint);
}

int paintnumber_lookup (const char * name)
{
    int pn;

    if (IS_NULLSTR(name))
        return -1;

    for (pn = 0; pn < paints.nused; pn++)
        if (!str_prefix(name, PAINT(pn)->name))
            return pn;

    return -1;
}

const char * paint_name (int pn)
{
    if ((pn < 0) || (pn > paints.nused - 1 ))
        return "None";
    else
        return PAINT(pn)->name; 
}

// ---------------------------------------------------------------------------------------
// functions for rituals
// ---------------------------------------------------------------------------------------
varr rituals  = {sizeof (ritual_t), 4};

ritual_t * ritual_new()
{
    ritual_t * ritual;

    ritual        = varr_enew (&rituals);
    ritual->name  = str_empty;
    
    return ritual ;
}

void ritual_free (ritual_t * ritual)
{
    free_string (ritual->name);
    free (ritual);
}

int ritualnumber_lookup (const char * name)
{
    int rn;

    if (IS_NULLSTR(name))
        return -1;

    for (rn = 0; rn < rituals.nused; rn++)
        if (!str_prefix(name, RITUAL(rn)->name))
            return rn;

    return -1;
}

const char * ritual_name (int rn)
{
    if ((rn < 0) || (rn > rituals.nused - 1 ))
        return "None";
    else
        return RITUAL(rn)->name; 
}

// ---------------------------------------------------------------------------------------
// do_make_paint called from do_make (act_info.c)
// ---------------------------------------------------------------------------------------
DO_FUN(do_make_paint)
{
    char       arg[MAX_INPUT_LENGTH];
    int        ingr[MAX_PAINT_COMP];
    OBJ_DATA * ingr_obj[MAX_PAINT_COMP];
    int        p, i, j, count;
    paint_t  * paint = NULL;
    int        matches = 0;
    bool       found = FALSE;
    OBJ_INDEX_DATA * ind;
    OBJ_DATA       * paint_obj;
    int              skill;

    // just to be sure
    for (i = 0; i < MAX_PAINT_COMP; ++i)
    {
        ingr[i] = 0;
        ingr_obj[i] = NULL;
    }

    if (argument[0] == '\0')
    {
        char_act("You try to make a paint from clear air but failed.", ch);
        return;
    }

    // now we have to select ingridients from char's inventory
    for (i = 0; i < MAX_PAINT_COMP && argument[0] != '\0'; ++i)
    {
        argument = one_argument (argument, arg, sizeof(arg));
        if ((ingr_obj[i] = get_obj_carry (ch, arg)) == NULL)
        {
            act("You have no $t.", ch, arg, NULL, TO_CHAR);
            return;
        }
        for (j = 0; j < i; ++j)
            if (ingr_obj[j] == ingr_obj[i])
            {
                act("Do you really want to use $t twice?", ch, arg, NULL, TO_CHAR);
                return;
            }
        ingr[i] = ingr_obj[i]->pIndexData->vnum;
    }
    count = i;
    // now we have to choose an appropriate paint 
    for (p = 0; p < paints.nused; ++p)
    {
        paint = PAINT(p);

        for (i = 0; i < MAX_PAINT_COMP; ++i)
            for (j = 0; j < MAX_PAINT_COMP; ++j)
                if ((paint->components[i] != 0) && (paint->components[i] == ingr[j]))
                {
                    ++matches;
                    ingr[j] = 0;
                    break;
                }
        if (matches == count)
        {
            found = TRUE;
            break;
        }
    }
    if (!found)
    {
        char_act("You try to mix it all but can't make anything useful.", ch);
        return;
    }

    // check for skill level
    skill = get_skill (ch, gsn_paint_craft);
    if (skill < paint->min_skill)
    {
        char_act ("You aren't skilled enough in paint craft to do this.", ch);
        return;
    }

    if (ch->mana < paint->mana_cost)
    {
        char_act("You don't have enough mana.", ch);
        return;
    }

    WAIT_STATE (ch, 3*PULSE_VIOLENCE);
    if (number_percent() > skill + 10) // those who aren't skilled in paint_craft can also make paints
    {
        char_act("You try to make a perfect paint but fail.", ch);
        ch->mana -= paint->mana_cost/2;
        check_improve(ch, gsn_paint_craft, FALSE, 1);
        return;
    }

    // all is Ok and we can create paint
    ind = get_obj_index (C_EMPTY_PAINT_VNUM);
    if (ind == NULL)
    {
        char_act("You try to make a paint but something beyond your knowledge prevents you.", ch);
        log_printf("C_EMPTY_PAINT_VNUM (%d) does not exist.", C_EMPTY_PAINT_VNUM);
        return;
    }

    // expense of components
    for (i = 0; i < count; ++i)
    {
        if (ingr_obj[i] != NULL)
        {
            act("Your $p disappears.", ch, ingr_obj[i], NULL, TO_CHAR);
            extract_obj(ingr_obj[i]);
        }
    }

    paint_obj = create_obj(ind, LVL(ch));
    free_string(paint_obj->name);
    paint_obj->name = str_dup (paint->obj_name);
    mlstr_free (paint_obj->short_descr);
    paint_obj->short_descr = mlstr_dup (paint->short_desc); 
    mlstr_free (paint_obj->description);
    paint_obj->description = mlstr_dup (paint->desc);
    paint_obj->level = LVL(ch);
    paint_obj->value[0] = paint->color; // color
    paint_obj->value[1] = paint->max_portions; // portions
    paint_obj->value[2] = paint->power == 0 ? LVL(ch) : paint->power; // power

    act("You have successfully made $p.", ch, paint_obj, NULL, TO_CHAR);
    act("$n has successfully made $p.", ch, paint_obj, NULL, TO_ROOM);
    obj_to_char(paint_obj, ch);
}

// ---------------------------------------------------------------------------------------
// do_make_tattoo called from do_make (act_info.c)
// NB! Only self-tattooing so far!
// ---------------------------------------------------------------------------------------
DO_FUN(do_make_tattoo)
{
    const char     * victim_name;
    char             where[MAX_INPUT_LENGTH];
    char             arg[MAX_INPUT_LENGTH];
    //CHAR_DATA      * victim;
    void           * vo;
    int              sn = -1, chance = 0, tn, i, mana;
    skill_t        * tattoo_skill;
    tattoo_t       * tattoo_desc;
    pcskill_t      * ps;
    OBJ_DATA       * obj,
                   * tattoo_obj,
                   * obj_next;
    appear_table_t * app;
    OBJ_INDEX_DATA * ind, 
                   * lack_ind;
    int              lack_comp[MAX_TATTOO_COMP];
    int              lack;
    flag32_t         paints;

    if (IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("You too scare to do this!", ch);
        return;
    }
    
    argument = one_argument(argument, arg, sizeof(arg));  // to remove word 'tattoo'
    victim_name = one_argument(argument, where, sizeof(where));

    if ((ps = skill_vlookup(&ch->pcdata->learned, arg))
        &&  skill_level(ch, sn = ps->sn) <= ch->level)
        chance = ps->percent;

    if (sn < 0 || (tattoo_skill = SKILL(sn)) == NULL)
    {
        act("You don't know how to make this tattoo.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    tn = tattoonumber_lookup (tattoo_skill->name);
    if (tn < 0)
    {
        act("You try to make a tattoo but something beyond your knowledge prevents you.", ch, NULL, NULL, TO_CHAR);
        log_printf("Tattoo %s does not exist.", tattoo_skill->name);
        return;
    }
    else 
        tattoo_desc = TATTOO(tn);

    ind = get_obj_index (C_SHAMAN_TATTOO_VNUM);
    if (ind == NULL)
    {
        act("You try to make a tattoo but something beyond your knowledge prevents you.", ch, NULL, NULL, TO_CHAR);
        log_printf("C_SHAMAN_TATTOO_VNUM (%d) does not exist.", C_SHAMAN_TATTOO_VNUM);
        return;
    }

    if (IS_SET(tattoo_desc->extra, TE_ONE_COPY) && check_tattoo(ch, sn))
    {
        char_act("But this tattoo has been already painted on your skin.", ch);
        return;
    }

    for (app = tattoo_wear_pos; app->name != NULL; app++)
        if (where[0] != '\0' && !str_prefix (where, app->name))
            break;

    if (app->name == NULL)
    {
        act("Where do you want to make a tattoo?", ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (!IS_SET (tattoo_desc->wear_loc, app->can_wear))
    {
        act("You can only make this tattoo on: {x", ch, NULL, NULL, TO_CHAR | ACT_NOLF);
        for (app = tattoo_wear_pos; app->name != NULL; app++)
            if (IS_SET (tattoo_desc->wear_loc, app->can_wear))
                char_printf(ch, "%s ", app->name);
        char_act (".", ch);
        return;
    }

    if (get_eq_char(ch, app->loc) != NULL)
    {
        act("But you have something worn there.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    // check for paints and components
    paints = 0;
    for (i = 0, lack = 0; i < MAX_TATTOO_COMP; ++i)
        if (tattoo_desc->paint_comp[i] != 0)
        {
            lack_comp[i] = tattoo_desc->paint_comp[i];
            ++lack;
        }

    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
    {
        if (obj->pIndexData->item_type == ITEM_PAINT)
        {
            if (IS_SET(tattoo_desc->paints, tattoo_colors[obj->value[0]].color))
                SET_BIT(paints, tattoo_colors[obj->value[0]].color);
        }
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                lack_comp[i] = 0;
                --lack;
                break;
            }
    }
    if (paints != tattoo_desc->paints)
    {
        char_printf (ch, "You have to find paints: {x");
        for (i = 0; i < TATTOO_COLOR_MAX; ++i)
        {
            if (IS_SET(tattoo_desc->paints, tattoo_colors[i].color) 
                && !IS_SET(paints, tattoo_colors[i].color))
                char_printf (ch, "%s ", tattoo_colors[i].name);
        }
        char_act (str_empty, ch);
        return;
    }
    if (lack > 0)
    {
        char_act ("For this tattoo you should find:", ch);
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] != 0)
            {
                lack_ind = get_obj_index (lack_comp[i]);
                if (lack_ind == NULL)
                    continue;
                char_printf (ch, "%s\n", mlstr_cval (lack_ind->short_descr, ch));
            }
        return;
    }

    // check for skill level
    if (get_skill (ch, gsn_tattoo) < tattoo_desc->min_skill)
    {
        char_act ("You aren't skilled enough in tattoo to do this.", ch);
        return;
    }

    mana = mana_cost(ch, sn) * 2;
    // check whether ch has enough mana for 1 tattoo
    if (ch->mana < mana) 
    {
        char_act("You don't have enough mana.", ch);
        return;
    }

    // todo: needle (?)

    if (number_percent () > chance)
    {
        WAIT_STATE(ch, tattoo_desc->wait_paint/2);
        ch->mana -= mana/2;
        act("You try to make a tattoo but fail.", ch, NULL, NULL, TO_CHAR);
        act("$n tries to make a tattoo but fails.", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, sn, FALSE, 1);
        check_improve(ch, gsn_tattoo, FALSE, 1);
        return;
    }

    // after all those checks - make tattoo
    tattoo_obj = create_obj (ind, LVL(ch));
    mlstr_free (tattoo_obj->short_descr);
    tattoo_obj->short_descr = mlstr_dup (tattoo_desc->desc);
    mlstr_free (tattoo_obj->description);
    tattoo_obj->description = mlstr_dup (tattoo_desc->desc);
    tattoo_obj->value[0] = LVL(ch);
    tattoo_obj->value[1] = tattoo_desc->max_activations; // default value from tattoo desc, 
                               // may be overwritten in tattoo_skill->spell_fun (init section)
    tattoo_obj->value[2] = sn;

    // expense of paints
    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;
        if (obj->pIndexData->item_type != ITEM_PAINT)
            continue;
        if (IS_SET(paints, tattoo_colors[obj->value[0]].color))
        {
            REMOVE_BIT(paints, tattoo_colors[obj->value[0]].color); // only one paint of each type
            if (obj->value[1] != -1)
                if (--obj->value[1] <= 0)
                    extract_obj (obj);
        }
    }
    // expense of components
    for (i = 0; i < MAX_TATTOO_COMP; ++i)
        if (tattoo_desc->paint_comp[i] != 0)
            lack_comp[i] = tattoo_desc->paint_comp[i];

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                extract_obj (obj);
                lack_comp[i] = 0;
                break;
            }
    }

    vo = (void *) tattoo_obj;
    if (tattoo_skill->spell_fun != NULL)
    {
        tattoo_skill->spell_fun (sn, LVL(ch), ch, vo, TAR_CREATING_TATTOO);
        check_improve (ch, sn, TRUE, 1);
    } 

    obj_to_char (tattoo_obj, ch);
    equip_char (ch, tattoo_obj, app->loc);

    char_printf (ch, "You have successfully made %s!\n", mlstr_cval (tattoo_desc->desc, ch));
    act("$n has successfully made $t.", ch, (void *) mlstr_mval(tattoo_desc->desc), NULL, TO_ROOM|ACT_TRANS);
    check_improve(ch, sn, TRUE, 4);
    check_improve(ch, gsn_tattoo, TRUE, 4);

    if (ch->pcdata && IS_SET(tattoo_desc->extra, TE_SPELLUP))       
    {
        if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP2))
            WAIT_STATE(ch, tattoo_desc->wait_paint/4);
        else if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP1))
            WAIT_STATE(ch, tattoo_desc->wait_paint/2);
        else
            WAIT_STATE(ch, tattoo_desc->wait_paint);
    } 
    else
        WAIT_STATE(ch, tattoo_desc->wait_paint);

    ch->mana -= mana;
}

// ---------------------------------------------------------------------------------------
// do_remove_tattoo called from do_remove (act_obj.c)
// ---------------------------------------------------------------------------------------
DO_FUN(do_remove_tattoo)
{
    appear_table_t * app;
    char             where[MAX_INPUT_LENGTH];
    OBJ_DATA       * obj;

    if (ch->fighting != NULL)
    {
        char_act ("But you are fighting!", ch);
        return;
    }

    argument = one_argument(argument, where, sizeof(where));
    argument = one_argument(argument, where, sizeof(where));

    for (app = tattoo_wear_pos; app->name != NULL; app++)
        if (where[0] != '\0' && !str_prefix (where, app->name))
            break;

    if (app->name == NULL)
    {
        act("Remove tattoo from... where???", ch, NULL, NULL, TO_CHAR);
        return;
    }
    obj = get_eq_char(ch, app->loc);
    if (obj == NULL)
    {
        act("But you have no tattoo there.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
    {
        act("But you have no tattoo there.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    damage (ch, ch, number_range(LVL(ch)*2, LVL(ch)*4), gsn_tattoo, DAM_MAGIC, TRUE);
    if (JUST_KILLED (ch))
        return;

    WAIT_STATE(ch, 2*PULSE_VIOLENCE);
    if (number_percent() > get_skill (ch, gsn_tattoo))
    {
        act("You try to remove a tattoo but fail.", ch, NULL, NULL, TO_CHAR);
        act("$n tries to remove a tattoo but fails.", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_tattoo, FALSE, 1);
        return;
    }

    // All is OK. Removing
    // todo: more informative messages, e.g. "You remove fire dragon tattoo from your wrist."
    act("You have successfully removed your $p.", ch, obj, NULL, TO_CHAR);
    act("$n has successfully removed $p from $gn{his} skin.", ch, obj, NULL, TO_ROOM);
    check_improve(ch, gsn_tattoo, TRUE, 1);
    extract_obj(obj);
}

// ---------------------------------------------------------------------------------------
// do_examine_tattoo called from do_examine (act_info.c)
// ---------------------------------------------------------------------------------------
DO_FUN(do_examine_tattoo)
{
    appear_table_t * app;
    char             where[MAX_INPUT_LENGTH];
    OBJ_DATA       * obj;

    if (ch->fighting != NULL)
    {
        char_act ("But you are fighting!", ch);
        return;
    }

    argument = one_argument(argument, where, sizeof(where));
    argument = one_argument(argument, where, sizeof(where));

    for (app = tattoo_wear_pos; app->name != NULL; app++)
        if (where[0] != '\0' && !str_prefix (where, app->name))
            break;

    if (app->name == NULL)
    {
        act("Examine tattoo painted... where???", ch, NULL, NULL, TO_CHAR);
        return;
    }
    obj = get_eq_char(ch, app->loc);
    if (obj == NULL)
    {
        act("But you have no tattoo there.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    if (obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
    {
        act("But you have no tattoo there.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    WAIT_STATE(ch, 2 * PULSE_VIOLENCE);
    if (number_percent() > get_skill (ch, gsn_tattoo))
    {
        act("You try to examine a tattoo but can't understand its condition.", ch, NULL, NULL, TO_CHAR);
        act("$n looks at $gn{his} tattoo but looks somewhat puzzled.", ch, NULL, NULL, TO_ROOM);
        check_improve(ch, gsn_tattoo, FALSE, 1);
        return;
    }

    // All is OK. Show info
    act("You carefully examine your $p.", ch, obj, NULL, TO_CHAR);
    if (obj->value[1] > -1)
        char_printf (ch, "This tattoo has %d activations and grants you level %d spell.\n", obj->value[1], obj->value[0]);
    else
        char_printf (ch, "This tattoo has very many activations and grants you level %d spell.\n", obj->value[0]);

    act("$n carefully examines $gn{his} skin.", ch, NULL, NULL, TO_ROOM);
    check_improve(ch, gsn_tattoo, TRUE, 1);
}

// ---------------------------------------------------------------------------------------
// perform ritual
// do_perform called from interpreter and from do_cast_org (do_universal_cast)
// when skill->type == TYPE_RITUAL
// ---------------------------------------------------------------------------------------
DO_FUN(do_perform)
{
    char        arg[MAX_INPUT_LENGTH];
    char        arg2[MAX_INPUT_LENGTH];
    CHAR_DATA * victim;
    void      * vo;
    int         mana, sn = -1, target, door, slevel, chance = 0, rn, i;
    bool        cast_far = FALSE, offensive = FALSE;
    skill_t   * ritual;
    pcskill_t * ps;
    OBJ_DATA  * obj;
    OBJ_DATA  * obj_next;
    bool        found;
    ritual_t  * ritual_desc;
    OBJ_INDEX_DATA * lack_ind;
    int              lack_comp[MAX_TATTOO_COMP];
    int              lack;
    flag_to_int_t  * sect;

    // NPCs can't use rituals yet
    if (IS_NPC (ch))
        return;

    if (IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("You too scare to do this!", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
    if (!str_cmp(arg,"ritual") || !str_cmp(arg,""))
        argument = one_argument(argument, arg, sizeof(arg));

    target_name = one_argument(argument, arg2, sizeof(arg2));

    if (arg2[0] == '\0') 
    {
        char_act("To make that, how, where?", ch);
        return;
    }

    substitute_skill_alias(arg2, arg, sizeof(arg));

    if ((ps = skill_vlookup(&ch->pcdata->learned, arg))
        &&  skill_level(ch, sn = ps->sn) <= ch->level)
        chance = ps->percent;

    if (sn < 0 || (ritual = SKILL(sn)) == NULL)
    {
        act("No such ritual.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    rn = ritualnumber_lookup (ritual->name);
    if (rn < 0)
    {
        act("You try to perform a ritual but something beyond your knowledge prevents you.", ch, NULL, NULL, TO_CHAR);
        return;
    }
    else 
        ritual_desc = RITUAL(rn);

    if (!check_components(ch, sn, chance))
        return;

    // check for ritual components
    for (i = 0, lack = 0; i < MAX_RITUAL_COMP; ++i)
        if (ritual_desc->components[i] != 0)
        {
            lack_comp[i] = ritual_desc->components[i];
            ++lack;
        }

    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
    {
        for (i = 0; i < MAX_RITUAL_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                lack_comp[i] = 0;
                --lack;
                break;
            }
    }
    if (lack > 0)
    {
        char_act ("To perform this ritual you should find:", ch);
        for (i = 0; i < MAX_RITUAL_COMP; ++i)
            if (lack_comp[i] != 0)
            {
                lack_ind = get_obj_index (lack_comp[i]);
                if (lack_ind == NULL)
                    continue;
                char_printf (ch, "%s\n", mlstr_cval (lack_ind->short_descr, ch));
            }
        return;
    }

    // check for extra flags
    if (IS_SET (ritual_desc->extra, RE_NO_ADRENALIN) && IS_PUMPED (ch))
    {
        char_act ("Your adrenalin is gushing. You can't concentrate enough.", ch);
        return;
    }
    if (IS_SET (ritual_desc->extra, RE_FULL_HP) && ch->hit < ch->max_hit)
    {
        char_act ("Current condition of your health doesn't allow you to perform this ritual.", ch);
        return;
    }

    // check for sector
    for (sect = sect_flags; sect->flag != FLAG_TO_INT_MAX; sect++)
    {
        if (ch->in_room->sector_type == sect->n_int)
            break;
    }
    if (sect->flag == FLAG_TO_INT_MAX)
    {
        act("You try to perform a ritual but something beyond your knowledge prevents you.", ch, NULL, NULL, TO_CHAR);
        return;
    }
    if (ritual_desc->sector && !IS_SET(ritual_desc->sector, sect->flag))
    {
        act("You can't perform such ritual in this sector.", ch, NULL, NULL, TO_CHAR);
        return;
    }
    // todo: check for time of the day

    // check for skill level
    if (get_skill (ch, gsn_ceremonialism) < ritual_desc->min_skill)
    {
        char_act ("You aren't skilled enough in rituals to do this.", ch);
        return;
    }

    if (!target_ioga(ch, sn, &target, &vo, &offensive, &door))
    {
        WAIT_STATE(ch, MISSING_TARGET_DELAY);           
        return;
    }

    victim = NULL;
    if (target == TARGET_CHAR)
    {
        victim = (CHAR_DATA *) vo;
        cast_far = victim->in_room != ch->in_room;
        if (victim->in_room 
        && (IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA) 
        || (victim->in_war && victim->war_status == PS_ALIVE ))
        && IS_SET(ritual->flags, SKILL_NOARENA))
        {
            char_act("You can't use it versus people at the battle arena.", ch);
            return;
        }
    }

    if (victim)
    {
        if (!can_cast(ch, victim))
            return;
    }

    found = FALSE;
    mana = mana_cost(ch, sn);
    // check whether ch has enough mana for 1 tattoo
    if (ch->mana < mana) 
    {
        char_act("You don't have enough mana.", ch);
        return;
    }

    if (ch->daze >= 16) 
       chance = URANGE(0, chance, 10); 
    else if (ch->daze >=  8) 
       chance = URANGE(0, chance, 25); 
    else if (ch->daze >=  4) 
       chance = URANGE(0, chance, 50);

    // chance
    if (number_percent () > chance)
    {
        char_act ("You try to perform this ritual but fail.", ch);
        check_improve(ch, sn, FALSE, 2);
        check_improve(ch, gsn_ceremonialism, TRUE, 2);
		ch->mana -= mana/2;
        WAIT_STATE(ch, ritual_desc->duration/2);
        return;
    }

    // expense of components
    for (i = 0; i < MAX_RITUAL_COMP; ++i)
        if (ritual_desc->components[i] != 0)
            lack_comp[i] = ritual_desc->components[i];

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;
        for (i = 0; i < MAX_RITUAL_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                extract_obj (obj);
                lack_comp[i] = 0;
                break;
            }
    }

    ch->mana -= mana;
    slevel = LVL(ch); // temporary

    if (ch->pcdata && IS_SET(ritual_desc->extra, RE_SPELLUP))       
    {
        if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP2))
            WAIT_STATE(ch, ritual_desc->duration/4);
        else if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP1))
            WAIT_STATE(ch, ritual_desc->duration/2);
        else
            WAIT_STATE(ch, ritual_desc->duration);
    } 
    else
        WAIT_STATE(ch, ritual_desc->duration);

    if (ritual->spell_fun != NULL)
    {
        ritual->spell_fun (sn, slevel, ch, vo, target);
        check_improve (ch, sn, TRUE, 1);
        check_improve(ch, gsn_ceremonialism, TRUE, 4);
    } 
    else if (ritual->do_fun != NULL)
    {
        ritual->do_fun (ch, target_name);
    }

    if (target != TARGET_CHAR || JUST_KILLED(ch) || JUST_KILLED(victim))
        return;

    if (cast_far && door != -1)
    {
        path_to_track(ch, victim, door);
        return;
    }

    if (offensive && victim != ch && victim->master != ch
    && victim->fighting == NULL && IS_AWAKE(victim)
    && victim->in_room == ch->in_room)
        multi_hit(victim, ch, TYPE_UNDEFINED);}

// ---------------------------------------------------------------------------------------
// activate tattoo(es)
// do_activate called from interpreter and from do_cast_org (do_universal_cast)
// when skill->type == TYPE_TATTOO
// ---------------------------------------------------------------------------------------
DO_FUN(do_activate)
{
    char        arg[MAX_INPUT_LENGTH];
    char        arg2[MAX_INPUT_LENGTH];
    CHAR_DATA * victim;
    void      * vo;
    int         mana, sn = -1, target, door, slevel, chance = 0, tn, i;
    bool        cast_far = FALSE, offensive = FALSE;
    skill_t   * tattoo;
    pcskill_t * ps;
    OBJ_DATA  * obj;
    OBJ_DATA  * obj_next;
    bool        found, one_tattoo = FALSE;
    tattoo_t  * tattoo_desc;
    OBJ_INDEX_DATA * lack_ind;
    int              lack_comp[MAX_TATTOO_COMP];
    int              lack;

    // NPCs can't use tattooes
    if (IS_NPC (ch))
        return;

    if (IS_AFFECTED(ch, AFF_FEAR))
    {
        char_act("You too scare to do this!", ch);
        return;
    }

    one_argument(argument, arg, sizeof(arg));
    if (!str_cmp(arg,"one") || !str_cmp(arg,""))
    {
        one_tattoo = TRUE;
        argument = one_argument(argument, arg, sizeof(arg));
    }

    one_argument(argument, arg, sizeof(arg));
    if (!str_cmp(arg,"tattoo") || !str_cmp(arg,""))
        argument = one_argument(argument, arg, sizeof(arg));

    target_name = one_argument(argument, arg2, sizeof(arg2));

    if (arg2[0] == '\0') 
    {
        char_act("To make that, how, where?", ch);
        return;
    }

    substitute_skill_alias(arg2, arg, sizeof(arg));

    if ((ps = skill_vlookup(&ch->pcdata->learned, arg))
        &&  skill_level(ch, sn = ps->sn) <= ch->level)
        chance = ps->percent;

    if (sn < 0 || (tattoo = SKILL(sn)) == NULL)
    {
        act("No such tattoo.", ch, NULL, NULL, TO_CHAR);
        return;
    }

    tn = tattoonumber_lookup (tattoo->name);
    if (tn < 0)
    {
        act("You try to activate a tattoo but something beyond your knowledge prevents you.", ch, NULL, NULL, TO_CHAR);
        return;
    }
    else 
        tattoo_desc = TATTOO(tn);

    if (!check_components(ch, sn, chance))
        return;

    // check for activation components
    for (i = 0, lack = 0; i < MAX_TATTOO_COMP; ++i)
        if (tattoo_desc->act_comp[i] != 0)
        {
            lack_comp[i] = tattoo_desc->act_comp[i];
            ++lack;
        }

    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
    {
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                lack_comp[i] = 0;
                --lack;
                break;
            }
    }
    if (lack > 0)
    {
        char_act ("To activate this tattoo you should find:", ch);
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] != 0)
            {
                lack_ind = get_obj_index (lack_comp[i]);
                if (lack_ind == NULL)
                    continue;
                char_printf (ch, "%s\n", mlstr_cval (lack_ind->short_descr, ch));
            }
        return;
    }

    if (!target_ioga(ch, sn, &target, &vo, &offensive, &door))
    {
        WAIT_STATE(ch, MISSING_TARGET_DELAY);           
        return;
    }

    victim = NULL;
    if (target == TARGET_CHAR)
    {
        victim = (CHAR_DATA *) vo;
        cast_far = victim->in_room != ch->in_room;
        if (victim->in_room 
        && (IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA) 
        || (victim->in_war && victim->war_status == PS_ALIVE ))
        && IS_SET(tattoo->flags, SKILL_NOARENA))
        {
            char_act("You can't use it versus people at the battle arena.", ch);
            return;
        }
    }

    if (victim)
    {
        if (!can_cast(ch, victim))
            return;
    }

    found = FALSE;
    mana = mana_cost(ch, sn);
    
    // check whether ch has enough mana for 1 tattoo
    if (ch->mana < mana) 
    {
        char_act("You don't have enough mana.", ch);
        return;
    }

    if (ch->daze >= 16) 
       chance = URANGE(0, chance, 10); 
    else if (ch->daze >=  8) 
       chance = URANGE(0, chance, 25); 
    else if (ch->daze >=  4) 
       chance = URANGE(0, chance, 50);

    // chance
    if (number_percent () > chance && check_tattoo(ch, sn))
    {
        char_act ("You try to activate your tattoo but fail.", ch);
        check_improve(ch, sn, FALSE, 4);
		check_improve(ch, gsn_tattoo, FALSE, 4);
		ch->mana -= mana/2;
        WAIT_STATE(ch, tattoo_desc->wait_act/2);
        return;
    }

    // expense of components
    for (i = 0; i < MAX_TATTOO_COMP; ++i)
        if (tattoo_desc->act_comp[i] != 0)
            lack_comp[i] = tattoo_desc->act_comp[i];

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;
        for (i = 0; i < MAX_TATTOO_COMP; ++i)
            if (lack_comp[i] == obj->pIndexData->vnum)
            {
                extract_obj (obj);
                lack_comp[i] = 0;
                break;
            }
    }

    //todo: modify target_ioga (don't yell there)
/*    if (tattoo->target == TAR_CHAR_OFFENSIVE && victim != NULL 
        && !IS_NPC(victim) && victim != ch)
    {
        if (!can_see(victim, ch))
            do_yell(victim, "Help! Someone is attacking me!");
        else
            act_yell(victim, "Die, $i, you swamp rat!", ch, NULL); 
    }*/

    if (ch->pcdata && IS_SET(tattoo_desc->extra, TE_SPELLUP))       
    {
        if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP2))
            WAIT_STATE(ch, tattoo_desc->wait_act/4);
        else if (IS_SET(ch->pcdata->wishes, WISH_SPELLUP1))
            WAIT_STATE(ch, tattoo_desc->wait_act/2);
        else
            WAIT_STATE(ch, tattoo_desc->wait_act);
    } 
    else
        WAIT_STATE(ch, tattoo_desc->wait_act);

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;

        if (obj->wear_loc == WEAR_NONE || obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
            continue;

        // it's a tattoo.. but what a tattoo?
        if (obj->value[2] != sn)
            continue;

        found = TRUE;
        if (ch->mana < mana) 
        {
            char_act("You don't have enough mana.", ch);
            break;
        }

        ch->mana -= mana;
        slevel = obj->value[0];
        if (tattoo->spell_fun != NULL)
        {
            tattoo->spell_fun (sn, slevel, ch, vo, target);
            check_improve (ch, sn, TRUE, 1);
            check_improve(ch, gsn_tattoo, TRUE, 4);
        } 
        else if (tattoo->do_fun != NULL)
        {
            tattoo->do_fun (ch, target_name);
        }
        
        if (obj->value[1] != -1)
        {
            if (--obj->value[1] <= 0)
            {
                act("Your $p pales and disappears.", ch, obj, NULL, TO_CHAR); 
                act("$n's $p pales and disappears.", ch, obj, NULL, TO_ROOM);
                extract_obj (obj);
            }
        }

        if (one_tattoo)
            break;

        if (JUST_KILLED(ch))
            break;
        if (victim && JUST_KILLED(victim))
            break;
        mana *= 2;
    }

    if (!found)
    {
        act ("You should make $t tattoo first.", ch, GETMSG (tattoo->name, ch->lang), NULL, TO_CHAR);
        return;
    }

    if (target != TARGET_CHAR || JUST_KILLED(ch) || JUST_KILLED(victim))
        return;

    if (cast_far && door != -1)
    {
        path_to_track(ch, victim, door);
        return;
    }

    if (offensive && victim != ch && victim->master != ch
    && victim->fighting == NULL && IS_AWAKE(victim)
    && victim->in_room == ch->in_room)
        multi_hit(victim, ch, TYPE_UNDEFINED);
}

// ---------------------------------------------------------------------------------------
// functions used to see lists and details of rituals
// ---------------------------------------------------------------------------------------
static void ritual_list ( CHAR_DATA *ch, const char *argument );
static void ritual_info ( CHAR_DATA *ch, const char *argument );

static CMD_DATA rituals_cmd_table[] = 
{
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra      
    { "list",     ritual_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",   ritual_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "info",     ritual_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",     ritual_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { NULL }
};


DO_FUN(do_rituals)
{
    if (!parse_command(ch, argument, rituals_cmd_table))
    {
        show_command_list (ch, rituals_cmd_table);
        char_act (".", ch);
        char_act ("Use {CHELP RITUALS{x for more information.", ch);
    }
}

static void ritual_list ( CHAR_DATA *ch, const char *argument )
{
    do_skills_org(ch, argument, TYPE_RITUAL);
}

static void ritual_info ( CHAR_DATA *ch, const char *argument )
{
    int        i, count;
    BUFFER   * output;
    ritual_t * ritual;
    bool       found;
    OBJ_INDEX_DATA * obj;

    i = ritualnumber_lookup(argument);
    if ( i < 0 )
    {
         char_printf(ch, "Must specify ritual name.\n");
         return;
    }

    ritual = RITUAL(i);

    output = buf_new(ch->lang);

    buf_printf (output, "Name:                  [%s]\n", ritual->name);
    buf_printf (output, "Duration:              [%d]\n", ritual->duration);
    if (ritual->min_skill > 0)
        buf_printf (output, "Min. skill percent:    [%d]\n", ritual->min_skill);
    if (ritual->sector)
        buf_printf (output, "Sector:                [%s]\n", flag_string(sector_flag_types, ritual->sector));
    //buf_printf (output, "Time period:           [%s]\n", flag_string(time_flag_types, ritual->time));

    found = FALSE;
    buf_add (output, "Components:         {x");
    for (i = 0, count = 0; i < MAX_RITUAL_COMP; ++i)
    {
        if (ritual->components[i] != 0)
        {
            obj = get_obj_index (ritual->components[i]);
            buf_printf (output, "\n{G%2d{x. %s", 
                ++count,
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, "   [none]\n");

    page_to_char (buf_string (output), ch);
    buf_free (output);    
}

// ---------------------------------------------------------------------------------------
// functions used to see lists and details of tattooes
// ---------------------------------------------------------------------------------------
static void tattoo_list ( CHAR_DATA *ch, const char *argument );
static void tattoo_info ( CHAR_DATA *ch, const char *argument );

static CMD_DATA tattoo_cmd_table[] = 
{
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra      
    { "list",     tattoo_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",   tattoo_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "info",     tattoo_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",     tattoo_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { NULL }
};

DO_FUN(do_tattooes)
{
    if (!parse_command(ch, argument, tattoo_cmd_table))
    {
        show_command_list (ch, tattoo_cmd_table);
        char_act (".", ch);
        char_act ("Use {CHELP TATTOO{x for more information.", ch);
    }
}

static void tattoo_list ( CHAR_DATA *ch, const char *argument )
{
    flag_t * wear;
    int i, count;

    if (argument[0] == '\0')
        do_skills_org(ch, argument, TYPE_TATTOO);
    else // show all tattooes for specified body part
    {
        for (wear = tattoo_wear_flags; wear->name != NULL; wear++)
            if (argument[0] != '\0' && !str_prefix (argument, wear->name))
                break;

        // invalid wear_loc
        if (wear->name == NULL)
        {
            char_printf(ch, "Tattooes can be only painted on: {x");
            for (wear = tattoo_wear_flags; wear->name != NULL; wear++)
                char_printf (ch, "%s ", wear->name);
            char_act(str_empty, ch);
            return;
        }
        
        char_act ("Here you can paint:", ch);
        for (i = 0, count = 0; i < tattooes.nused; ++i)
        {
            tattoo_t * tattoo = TATTOO(i);

            if (IS_SET(tattoo->wear_loc, wear->bit))
                char_printf (ch, "{G %2d{x. %s\n", ++count, tattoo->name);
        }
    }
}

static void tattoo_info ( CHAR_DATA *ch, const char *argument )
{
    int        i, count; 
    BUFFER   * output;
    tattoo_t * tattoo;
    bool       found;
    OBJ_INDEX_DATA * obj;


    i = tattoonumber_lookup(argument);
    if ( i < 0 )
    {
         char_printf(ch, "No such tattoo.\n", argument);
         return;
    }

    tattoo = TATTOO(i);

    output = buf_new(ch->lang);

    buf_printf (output, "{x     Tattoo %s\n", tattoo->name);
    buf_printf (output, "Activaition time:      [%d]\n", tattoo->wait_act);
    buf_printf (output, "Painting time:         [%d]\n", tattoo->wait_paint);
    if (tattoo->min_skill > 0)
        buf_printf (output, "Min. skill percent:    [%d]\n", tattoo->min_skill);
    buf_printf (output, "You can paint it on:   [%s]\n", flag_string(tattoo_wear_flags, tattoo->wear_loc));
    buf_printf (output, "Paints needed:         [%s]\n\n", flag_string(paint_color_flags, tattoo->paints));
    if (IS_SET (tattoo->extra, TE_SPELLUP))
        buf_printf (output, "Wish SPELLUP will work with this tattoo.\n", flag_string(paint_color_flags, tattoo->paints));
    if (IS_SET (tattoo->extra, TE_ONE_COPY))
        buf_printf (output, "You can make only one copy of this tattoo on your skin.\n", flag_string(paint_color_flags, tattoo->paints));
    if (IS_SET (tattoo->extra, TE_ONE_COPY))
        buf_printf (output, "You can paint this tattoo on others.\n", flag_string(paint_color_flags, tattoo->paints));


    found = FALSE;
    buf_add (output, "To paint this tattoo you must use:");
    for (i = 0, count = 0; i < MAX_TATTOO_COMP; ++i)
    {
        if (tattoo->paint_comp[i] != 0)
        {
            obj = get_obj_index (tattoo->paint_comp[i]);
            buf_printf (output, "\n{G%2d{x. %s", 
                ++count,
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, "    [none]\n");

    found = FALSE;
    buf_add (output, "To activate this tattoo you must use:");
    for (i = 0, count = 0; i < MAX_TATTOO_COMP; ++i)
    {
        if (tattoo->act_comp[i] != 0)
        {
            obj = get_obj_index (tattoo->act_comp[i]);
            buf_printf (output, "\n{G%2d{x. %s", 
                ++count,
                obj == NULL ? "non-existent object" : mlstr_cval (obj->short_descr, ch));
            found = TRUE;
        }
    }
    if (found)
        buf_add (output, "\n");
    else
        buf_add (output, " [none]\n");

    page_to_char (buf_string (output), ch);
    buf_free (output);
}

// ---------------------------------------------------------------------------------------
// functions used to create target affect or resist on object
// called from tattoo_xxx in init section
// improved used to apply effect of violet stars (+20%)
// ---------------------------------------------------------------------------------------
static void create_affect (void * obj, int sn, int level, int location, 
                    int mod, flag64_t bitvector, bool improved)
{
    AFFECT_DATA af;

    af.where        = TO_OBJECT;
    af.type         = sn;
    af.level        = level;
    af.duration     = -1;
    af.location     = location;
    af.modifier     = improved ? mod + mod/5 : mod;
    af.bitvector    = bitvector;
    affect_to_obj((OBJ_DATA*) obj, &af);
}

static void create_resist (void * obj, int sn, int level, int location, 
                    int mod, flag64_t bitvector, bool improved)
{
    AFFECT_DATA af;

    af.where        = TO_RESIST;
    af.type         = sn;
    af.level        = level;
    af.duration     = -1;
    af.location     = location;
    af.modifier     = improved ? mod + mod/10 : mod;
    af.bitvector    = bitvector;
    affect_to_obj((OBJ_DATA*) obj, &af);
}

// ---------------------------------------------------------------------------------------
// checks whether ch has a tattoo with specified sn on his body
// check_violet_stars - this code is to be used in all tattoo_xxx functions
// ---------------------------------------------------------------------------------------
int check_tattoo (CHAR_DATA * ch, int sn)
{
    OBJ_DATA * obj;

    for (obj = ch->carrying; obj != NULL; obj = obj->next_content)
    {
        if (obj->wear_loc == WEAR_NONE || obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
            continue;
        // it's a tattoo.. but what a tattoo?
        if (obj->value[2] == sn)
            return TRUE;
    }

    return FALSE;
}

bool check_violet_stars (CHAR_DATA * ch)
{
    int vsn = sn_lookup("violet stars"); 
    if (check_tattoo (ch, vsn))
    {
        if (number_percent () < get_skill(ch, vsn) * 9 / 10)
        {
            act ("{MV{miolet {MS{mtars{x on your skin blaze brightly.", ch, NULL, NULL, TO_CHAR | ACT_VERBOSE);
            act ("{MV{miolet {MS{mtars{x on $n's skin blaze brightly.", ch, NULL, NULL, TO_ROOM | ACT_VERBOSE);
            check_improve (ch, vsn, TRUE, 4);
            return TRUE;
        }
        check_improve (ch, vsn, FALSE, 4);
        return FALSE;
    }
    return FALSE;
}

bool check_tigers_snout (CHAR_DATA * ch, CHAR_DATA * victim)
{
    int vsn = sn_lookup("tigers snout"); 
    int dam;

    if (check_tattoo (ch, vsn))
    {
        if (number_percent () < get_skill(ch, vsn) / 10 && IS_NPC(victim) && IS_AFFECTED(victim, AFF_CHARM))
        {
            dam = number_range(LVL(ch) * 2, LVL(ch) * 7);
            damage (ch, victim, dam, vsn, TYPE_UNDEFINED, TRUE);
            check_improve (ch, vsn, TRUE, 4);
            return TRUE;
        }
        check_improve (ch, vsn, FALSE, 4);
        return FALSE;
    }
    return FALSE;
}

// --------------------------------------------------------------------------------
// TATTOO Spell functions
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// simple healing tattoo
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_glowing_heart)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    CHAR_DATA * gch;
    bool        improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, 2 * LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -8, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -20 - LVL(ch)/3, 0, improved);
        return;
    }

    if (victim == NULL)
        victim = ch;

    if (improved)
        level += number_range (2,5);

    if (ch == victim && check_tattoo (ch, sn_lookup("white cloud")))
    {
        char_act ("{WWhite{x {Ccloud{x on your skin moves a little.", ch);
        act("{WWhite{x {Ccloud{x on $n's skin moves a little.", ch, NULL, NULL, TO_ROOM);
        act("Your tattoo spreads warm waves.", ch, NULL, NULL, TO_CHAR);
        for (gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room) 
        {
            if (is_same_group(ch, gch)) 
            {
                if (CAN_HEAL(ch, gch)) 
                {
                    if (IS_CYBORG(gch) && ch != gch) 
                    {
                        char_act ("It is difficult to cure cyborg.", ch);
                        gch->hit = (UMIN(gch->hit + level/5 + dice (2, level), gch->max_hit));
                        update_pos(gch);
                    }
                    else 
                    {
                        gch->hit = UMIN(gch->hit + level / 2 + dice (3, level), gch->max_hit);
                        update_pos(gch);
                    }
                    if (ch != gch)
                        act("$n's tattoo spreads warm wave to you.", ch, NULL, gch, TO_VICT);
                }
                else 
                    act("You feel warm wave spreading by $n's tattoo.", ch, NULL, gch, TO_VICT);
            }
            else
                act("You feel warm wave spreading by $n's tattoo.", ch, NULL, gch, TO_VICT);
        }
    }
    else
    {
        if (CAN_HEAL(ch, victim)) 
        {
            if (IS_CYBORG(victim) && ch != victim) 
            {
                char_act ("It is difficult to cure cyborg.", ch);
                victim->hit = (UMIN(victim->hit + level/5 + dice (2, level), victim->max_hit));
                update_pos(victim);
            }
            else 
            {
                victim->hit = UMIN(victim->hit + level / 2 + dice (3, level), victim->max_hit);
                update_pos(victim);
            }
            act("You feel warm wave spreading by $n's tattoo.", ch, NULL, victim, TO_NOTVICT);
            if (ch != victim)
            {
                act("$n's tattoo spreads warm wave to you.", ch, NULL, victim, TO_VICT);
                act("Your tattoo spreads warm wave to $N.", ch, NULL, victim, TO_CHAR);
            }
            else
            {
                act("Warm wave from your tattoo covers your body.", ch, NULL, NULL, TO_CHAR);
            }
        }
        else if (!is_safe(ch,victim))
            damage (ch, victim, 100+level/10, sn, DAM_LIGHT, TRUE);

    }
}
// --------------------------------------------------------------------------------
// casts fly
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_winged_horse)
{
    CHAR_DATA  * victim = (CHAR_DATA *) vo;
    AFFECT_DATA  af;
    bool         improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch) * 3 / 2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -20 - LVL(ch)/3, 0, improved);
        return;
    }

    if (IS_AFFECTED(victim, AFF_FLYING)
        ||  is_affected(victim, sn))
    {
        if (victim == ch)
          char_act("But you are already in flight.", ch);
        else
          act("But $N is already flying!",ch,NULL,victim,TO_CHAR);
        return;
    }

    if (improved)
        level += number_range (2,5);

    af.where     = TO_AFFECTS;
    af.type      = sn;
    af.level     = level;
    af.duration  = level + 3;
    af.location  = 0;
    af.modifier  = 0;

    af.bitvector = AFF_FLYING;
    affect_to_char(victim, &af);

    if (ch != victim)
    {
        act("You feel a capful of wind from $n's tattoo to $N.", ch, NULL, victim, TO_NOTVICT);
        act("You feel a capful of wind from $n's tattoo to you.", ch, NULL, victim, TO_VICT);
        act("You feel a capful of wind from your tattoo to $N.", ch, NULL, victim, TO_CHAR);
    }
    else
    {
        act("You feel a capful of wind from $n's tattoo.", ch, NULL, NULL, TO_ROOM);
        act("You feel a capful of wind from your tattoo.", ch, NULL, NULL, TO_CHAR);
    }
    char_act("You are slowly flying up into the sky.", victim);
    act("$n slowly flying up.", victim, NULL, NULL, TO_ROOM);
}

// --------------------------------------------------------------------------------
// dam_bash, chance to daze or deafen victim 
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_hammer_of_heaven)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;
    bool        improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/8, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 5, DAM_BASH, improved);
        return;
    }

    if (improved)
        level += number_range (0,2);

    // damage
    dam = number_range(level*7, level*11);
    if (improved)
        dam += number_range (dam/10, dam/5);

    if (saves_spell(level, victim, DAM_MAGIC))
        dam /= 2;

    act("{CBright blue{x hammer on $n's tattoo flashes suddenly.", ch, NULL, NULL, TO_ROOM);
    act("{CBright blue{x hammer on your tattoo flashes suddenly.", ch, NULL, NULL, TO_CHAR);

    damage(ch, victim, dam, sn, DAM_BASH, TRUE);

    if (JUST_KILLED (ch) || JUST_KILLED (victim))
        return;

    // chance to daze
    if (chance(25))
    {
        act("$n looks dazed a little.", victim, NULL, NULL, TO_ROOM);
        act("$n's hammer of heaven dazes you.", ch, NULL, victim, TO_VICT);

        DAZE_STATE (victim, PULSE_VIOLENCE);
        return;
    }
    // or small chance to deafen
    if (chance(5) && !is_affected(victim, gsn_deafen))
    {
        spell_deafen(gsn_deafen, level + level/15, ch, victim, TARGET_CHAR);
        return;
    }
}

// --------------------------------------------------------------------------------
// kills one randomly selected victim from ch->in_room
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_laughing_skull)
{
    CHAR_DATA  ** victims = NULL;
    int           vict_count = 0;
    CHAR_DATA   * victim;
    int           chance;
    bool          improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/8, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DEX, -1 - LVL(ch) / 20, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);
        return;
    }

    if (IS_SET(ch->plr_flags, PLR_GHOST))
    {
        char_act("You return to your normal form.", ch);
        REMOVE_BIT(ch->plr_flags, PLR_GHOST);
    }

    if (improved)
        level += number_range (2,4);

    for (victim = ch->in_room->people, vict_count = 0; victim; victim = victim->next_in_room)
    {
        if (!can_cast (ch, victim) || is_safe(ch, victim) || IS_IMMORTAL (victim))
            continue;

        // chance to avoid death
        chance = number_range (5,15); // base chance
        chance += (LVL(victim) - LVL (ch)) * 2;
        if (saves_spell (level, victim, DAM_NEGATIVE))
            chance += number_range (15, 25);
        chance = URANGE(5, chance, 30);
        if (victim == ch)
            chance = 50; 
        if (number_percent () < chance)
            continue;

        // potential victim
        if (victims == NULL)
            victims = (CHAR_DATA**) calloc (1, sizeof (CHAR_DATA*) * (++vict_count));
        else 
            victims = (CHAR_DATA**) realloc (victims, sizeof (CHAR_DATA*) * (++vict_count));
        victims[vict_count-1] = victim;
    }
    if (vict_count > 0)
    {
        victim = victims[number_range (0, vict_count - 1)];
        if (ch != victim)
        {
            act("Laughing skull appears in the air from $n's tattoo and touches $N.", ch, NULL, victim, TO_NOTVICT);
            act("Laughing skull appears in the air from $n's tattoo and touches you.", ch, NULL, victim, TO_VICT);
            act("Laughing skull appears in the air from your tattoo and touches $N.", ch, NULL, victim, TO_CHAR);
        }
        else
        {
            act("Laughing skull appears in the air from $n's tattoo and consumes $gn{himself}.", ch, NULL, NULL, TO_ROOM);
            act("Laughing skull appears in the air from your tattoo and consumes you.", ch, NULL, NULL, TO_CHAR);
        }
        handle_death (ch, victim);
    }
    else // no potential victims..
    {
        act("Laughing skull appears in the air from $n's tattoo but seems puzzled a little.", ch, NULL, NULL, TO_ROOM);
        act("Laughing skull appears in the air from your tattoo but seems puzzled a little.", ch, NULL, NULL, TO_CHAR);
    }

    free (victims);
}

// --------------------------------------------------------------------------------
// casts shield, protective shield and armor
// in melee combat can block an attack
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_tower_shield)
{
    bool improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*3/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -20 - LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)*2, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 5, DAM_WEAPON, improved);
        return;
    }

    act ("Tower shield on $n's tattoo glows bright white.", ch, NULL, NULL, TO_ROOM);
    act ("Tower shield on your tattoo glows bright white.", ch, NULL, NULL, TO_CHAR);

    if (improved)
        level += number_range (2,5);

    if (!is_affected (ch, gsn_shield))
    {
        spell_shield (gsn_shield, level - level/15, ch, (void *) ch, TARGET_CHAR);
    }

    if (!is_affected (ch, gsn_protective_shield))
    {
        spell_protective_shield (gsn_protective_shield, level - level/15, ch, (void *) ch, TARGET_CHAR);
    }

    if (!is_affected (ch, sn_lookup("armor")))
    {
        spell_armor (sn_lookup("armor"), level - level/15, ch, (void *) ch, TARGET_CHAR);
    }
}

// --------------------------------------------------------------------------------
// casts shield, protective shield and bark skin
// todo: rewrite it :)
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_knight_in_full_armor)
{
    bool improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -30 - LVL(ch), 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 5, DAM_WEAPON, improved);
        create_resist (vo, sn, LVL(ch), APPLY_NONE, 5, DAM_MAGIC, improved);
        return;
    }
    
    act ("Knight on $n's tattoo moves sharply.", ch, NULL, NULL, TO_ROOM);
    act ("Knight on your tattoo moves sharply.", ch, NULL, NULL, TO_CHAR);
    char_act (str_empty, ch);

    if (improved)
        level += number_range (2,5);

    if (!is_affected (ch, gsn_shield))
    {
        spell_shield (gsn_shield, level - level/15, ch, (void *) ch, TARGET_CHAR);
    }

    if (!is_affected (ch, gsn_protective_shield))
    {
        spell_protective_shield (gsn_protective_shield, level - level/15, ch, (void *) ch, TARGET_CHAR);
    }

    if (!IS_AFFECTED (ch, AFF_PROTECTION|AFF_SANCTUARY|AFF_BLACK_SHROUD))
    {
        spell_bark_skin (sn_lookup("bark skin"), level - level/15, ch, (void *) ch, TARGET_CHAR);
    }
}

// --------------------------------------------------------------------------------
// something like priests' guardian angel
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_nine_green_spiders)
{
    AFFECT_DATA * paf, af;
    bool          improved = check_violet_stars (ch);

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -55 - LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)*2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_CON, 1 + LVL(ch)/45, 0, improved);
        return;
    }

    act ("It seems that {Ggreen {Dsp{Gi{Ders{x on $n's tattoo budge.", ch, NULL, NULL, TO_ROOM);
    act ("It seems that {Ggreen {Dsp{Gi{Ders{x on your tattoo budge.", ch, NULL, NULL, TO_CHAR);

    if (improved)
        level += number_range (2,5);

    paf = affect_find(ch->affected, sn);
    if (paf != NULL)
    {
        paf->duration = UMIN (LVL(ch)/4, paf->duration + level/8);
    }
    else
    {
        af.where        = TO_AFFECTS;
        af.type         = sn;
        af.level        = level;
        af.duration     = level/10;
        af.location     = APPLY_NONE;
        af.modifier     = 0;
        af.bitvector    = 0;
        affect_to_char(ch, &af);
    }
}

// --------------------------------------------------------------------------------
// forces all the char's opponents in combat to flee (randomly)
// small chance to inflict fear to victim
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_war_paint)
{
    CHAR_DATA   * victim,
                * v_next;
    bool          improved = check_violet_stars (ch);
    
    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -5, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);
        return;
    }

    if (improved)
        level += number_range (1,4);

    char_act ("Terrible grimace warps your face.\n", ch);
    act ("Terrible grimace warps $n's face.\n", ch, NULL, NULL, TO_ROOM);
    for (victim = ch->in_room->people; victim; victim = v_next)
    {
        v_next = victim->next_in_room;
        if (victim->fighting == ch) 
        {
            if (number_percent () < 70 && !(saves_spell(level, victim, DAM_MAGIC)))
            {
                // small chance to scare an enemy (fear affect)
                spell_fear (gsn_fear, level + level/15, ch, victim, TARGET_CHAR);
                if (!IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA))
                    do_flee(victim, str_empty);
            }
            else // brave enemy... receive a small (?) amount of damage
            {
                int dam = number_range(level*2, level*5);

                if (saves_spell(level, victim, DAM_MAGIC))
                    dam /= 2;

                damage(ch, victim, dam, sn, DAM_MAGIC, TRUE);
            }
        }
    }
}

// --------------------------------------------------------------------------------
// dam_fire, works worse against the dragons
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_fire_dragon)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;
    race_t    * race;
    bool        improved = check_violet_stars (ch);
    
    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*3/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -7, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -25 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 20, DAM_FIRE, improved);
        return;
    }

    if (improved)
        level += number_range (1,3);

    act("{RR{re{Rd dr{ra{Rgon{x on $n's tattoo suddenly comes to life...", ch, NULL, NULL, TO_ROOM);
    act("{RR{re{Rd dr{ra{Rgon{x on your tattoo suddenly comes to life...", ch, NULL, NULL, TO_CHAR);

    race = RACE(victim->race);
    if (strstr(race->name, "dragon") != NULL && (number_percent() > get_skill (ch, sn)*3/4))
    {
        act("{x   ... but it seems it can't harm your victim.", ch, NULL, NULL, TO_CHAR);
        act("{x   ... but seems powerless.", ch, NULL, NULL, TO_ROOM);
        return;
    }

    act("{x   ... and breathes flame to $N.", ch, NULL, victim, TO_NOTVICT);
    act("{x   ... and breathes flame to you.", ch, NULL, victim, TO_VICT);

    // damage
    dam = number_range(level*6, level*10);
    if (improved)
        dam += number_range (dam/10, dam/6);

    if (saves_spell(level, victim, DAM_FIRE))
        dam /= 2;

    damage(ch, victim, dam, sn, DAM_FIRE, TRUE);

    if (JUST_KILLED (ch) || JUST_KILLED (victim))
        return;
    fire_effect (victim, level, dam, TARGET_CHAR);
}

// --------------------------------------------------------------------------------
// dam_fire, works worse against the dragons
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_fire_drake)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;
    race_t    * race;
    bool        improved = check_violet_stars (ch);
    
    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -5, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 12, DAM_FIRE, improved);
        return;
    }

    if (improved)
        level += number_range (1,2);

    act("{RR{re{Rd dr{ra{Rke{x on $n's tattoo suddenly comes to life...", ch, NULL, NULL, TO_ROOM);
    act("{RR{re{Rd dr{ra{Rke{x on your tattoo suddenly comes to life...", ch, NULL, NULL, TO_CHAR);

    race = RACE(victim->race);
    if (strstr(race->name, "dragon") != NULL && (number_percent() > get_skill (ch, sn)*3/4))
    {
        act("{x   ... but it seems it can't harm your victim.", ch, NULL, NULL, TO_CHAR);
        act("{x   ... but seems powerless.", ch, NULL, NULL, TO_ROOM);
        return;
    }

    act("{x   ... and breathes flame to $N.", ch, NULL, victim, TO_NOTVICT);
    act("{x   ... and breathes flame to you.", ch, NULL, victim, TO_VICT);

    // damage
    dam = number_range(level*4, level*6);
    if (improved)
        dam += number_range (dam/10, dam/5);

    if (saves_spell(level, victim, DAM_FIRE))
        dam /= 2;

    damage(ch, victim, dam, sn, DAM_FIRE, TRUE);

    if (JUST_KILLED (ch) || JUST_KILLED (victim))
        return;
    fire_effect (victim, level - level/10, dam, TARGET_CHAR);
}

// --------------------------------------------------------------------------------
// tattoo_ancient_dragon, works worse against the dragons
// dam (fire, cold, acid, lightning) - randomly choosen
// --------------------------------------------------------------------------------
#define AD_MAX_DT 4
int dam_types [AD_MAX_DT] = {DAM_FIRE, DAM_COLD, DAM_ACID, DAM_LIGHTNING};

SPELL_FUN(tattoo_ancient_dragon)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;
    race_t    * race;
    int         dam_type;
    bool        improved = check_violet_stars (ch);
    
    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -9, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -35 - LVL(ch)/2, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 30, dam_types[number_range(0, AD_MAX_DT)], improved);
        return;
    }

    if (improved)
        level += number_range (2,5);

    act("{DAnci{We{Dnt dr{Wa{Dgon{x on $n's tattoo suddenly comes to life...", ch, NULL, NULL, TO_ROOM);
    act("{DAnci{We{Dnt dr{Wa{Dgon{x on your tattoo suddenly comes to life...", ch, NULL, NULL, TO_CHAR);

    race = RACE(victim->race);
    if (strstr(race->name, "dragon") != NULL && (number_percent() > get_skill (ch, sn)*3/4))
    {
        act("{x   ... but it seems it can't harm your victim.", ch, NULL, NULL, TO_CHAR);
        act("{x   ... but seems powerless.", ch, NULL, NULL, TO_ROOM);
        return;
    }

    switch (number_range(0,3))
    {
    case 0: 
        act("{x   ... and breathes flame to $N.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and breathes flame to you.", ch, NULL, victim, TO_VICT);
        dam_type = DAM_FIRE;
        break;
    case 1: 
        act("{x   ... and breathes a cone of ice to $N.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and breathes a cone of ice to you.", ch, NULL, victim, TO_VICT);
        dam_type = DAM_COLD;
        break;
    case 2: 
        act("{x   ... and breathes a stream of acid to $N.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and breathes a stream of acid to you.", ch, NULL, victim, TO_VICT);
        dam_type = DAM_ACID;
        break;
    case 3: 
        act("{x   ... and breathes lightning to $N.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and breathes lightning to you.", ch, NULL, victim, TO_VICT);
        dam_type = DAM_LIGHTNING;
        break;
    default: 
        act("{x   ... and breathes flame to $N.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and breathes flame to you.", ch, NULL, victim, TO_VICT);
        dam_type = DAM_FIRE;
        break;
    }

    // damage
    dam = number_range(level*10, level*13);
    if (improved)
        dam += number_range (dam/10, dam/5);

    if (saves_spell(level, victim, dam_type))
        dam /= 2;

    damage(ch, victim, dam, sn, dam_type, TRUE);

    if (JUST_KILLED (ch) || JUST_KILLED (victim))
        return;

    switch (dam_type)
    {
    case DAM_FIRE: 
        fire_effect (victim, level - level/10, dam, TARGET_CHAR);
        break;
    case DAM_COLD: 
        cold_effect (victim, level - level/10, dam, TARGET_CHAR);
        break;
    case DAM_ACID: 
        acid_effect (victim, level - level/10, dam, TARGET_CHAR);
        break;
    case DAM_LIGHTNING: 
        shock_effect (victim, level - level/10, dam, TARGET_CHAR);
        break;
    }
}

// --------------------------------------------------------------------------------
// cures poison and blindness
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_green_cross)
{
    bool        improved = check_violet_stars (ch);
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    CHAR_DATA * gch;

    if (victim == NULL)
        victim = ch;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -15 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 40, DAM_POISON, improved);
        return;
    }

    if (improved)
        level += number_range (1,3);

    act("$n's {ggreen cross{x glows brightly.", ch, NULL, NULL, TO_ROOM);
    act("Your {ggreen cross{x glows brightly.", ch, NULL, NULL, TO_CHAR);

    if (ch == victim && check_tattoo (ch, sn_lookup("white cloud")))
    {
        for (gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room) 
        {
            if (is_same_group(ch, gch)) 
            {
                if (IS_AFFECTED(gch, AFF_BLIND))
                    spell_cure_blindness(sn_lookup("cure blindness"), level, ch, gch, TARGET_CHAR);
                if (IS_AFFECTED(gch, AFF_POISON))
                    spell_cure_poison(sn_lookup("cure poison"), level, ch, gch, TARGET_CHAR);
            }
        }
    }
    else
    {
        if (IS_AFFECTED(victim, AFF_BLIND))
            spell_cure_blindness(sn_lookup("cure blindness"), level, ch, victim, TARGET_CHAR);
        if (IS_AFFECTED(victim, AFF_POISON))
            spell_cure_poison(sn_lookup("cure poison"), level, ch, victim, TARGET_CHAR);
    }
}

// --------------------------------------------------------------------------------
// cures poison, blindness, plague, heals some hp
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_ornate_green_cross)
{
    bool        improved = check_violet_stars (ch);
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    CHAR_DATA * gch;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*3/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -40 - LVL(ch)/2, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 40, DAM_POISON, improved);
        create_resist (vo, sn, LVL(ch), APPLY_NONE, 40, DAM_DISEASE, improved);
        return;
    }

    if (victim == NULL)
        victim = ch;

    if (improved)
        level += number_range (2,5);

    act("$n's ornate {ggreen cross{x glows brightly.", ch, NULL, NULL, TO_ROOM);
    act("Your ornate {ggreen cross{x glows brightly.", ch, NULL, NULL, TO_CHAR);

    if (ch == victim && check_tattoo (ch, sn_lookup("white cloud")))
    {
        for (gch = ch->in_room->people; gch != NULL; gch = gch->next_in_room) 
        {
            if (is_same_group(ch, gch)) 
            {
                if (CAN_HEAL(ch, gch)) 
                {
                    if (IS_CYBORG(gch) && ch != gch) 
                    {
                        char_act ("It is difficult to cure cyborg.", ch);
                        gch->hit = (UMIN(gch->hit + level/5 + dice (3, level/2), gch->max_hit));
                        update_pos(gch);
                    }
                    else 
                    {
                        gch->hit = UMIN(gch->hit + level / 2 + dice (4, level*2/3), gch->max_hit);
                        update_pos(gch);
                    }
                    if (IS_AFFECTED(gch, AFF_BLIND))
                        spell_cure_blindness(sn_lookup("cure blindness"), level+level/10, ch, gch, TARGET_CHAR);
                    if (IS_AFFECTED(gch, AFF_POISON))
                        spell_cure_poison(sn_lookup("cure poison"), level+level/10, ch, gch, TARGET_CHAR);
                    if (IS_AFFECTED(gch, AFF_PLAGUE))
                        spell_cure_disease(sn_lookup("cure disease"), level+level/10, ch, gch, TARGET_CHAR);
                }
            }
        }
    }
    else
    {
        if (CAN_HEAL(ch, victim)) 
        {
            if (IS_CYBORG(victim) && ch != victim) 
            {
                char_act ("It is difficult to cure cyborg.", ch);
                victim->hit = (UMIN(victim->hit + level/5 + dice (3, level/2), victim->max_hit));
                update_pos(victim);
            }
            else 
            {
                victim->hit = UMIN(victim->hit + level / 2 + dice (4, level*2/3), victim->max_hit);
                update_pos(victim);
            }
            if (IS_AFFECTED(victim, AFF_BLIND))
                spell_cure_blindness(sn_lookup("cure blindness"), level+level/10, ch, victim, TARGET_CHAR);
            if (IS_AFFECTED(victim, AFF_POISON))
                spell_cure_poison(sn_lookup("cure poison"), level+level/10, ch, victim, TARGET_CHAR);
            if (IS_AFFECTED(victim, AFF_PLAGUE))
                spell_cure_disease(sn_lookup("cure disease"), level+level/10, ch, victim, TARGET_CHAR);
        }
        else if (!is_safe(ch,victim))
            damage (ch, victim, 100+level/10, sn, DAM_LIGHT, TRUE);
    }
}

// --------------------------------------------------------------------------------
// dam_cold, slow
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_snowflake)
{
    bool        improved = check_violet_stars (ch);
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;


    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)*3/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -7, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 15, DAM_COLD, improved);
        return;
    }

    if (improved)
        level += number_range (2,5);

    act("Huge snoflake on $n's skin glows blue.", ch, NULL, NULL, TO_ROOM);
    act("Huge snoflake on your skin glows blue.", ch, NULL, NULL, TO_CHAR);

    // damage
    dam = number_range(level*5, level*8);
    if (improved)
        dam += number_range (dam/8, dam/6);

    if (saves_spell(level, victim, DAM_COLD))
        dam /= 2;

    damage(ch, victim, dam, sn, DAM_COLD, TRUE);

    if (JUST_KILLED (ch) || JUST_KILLED (victim))
        return;

    if (number_percent () < 30)
        spell_slow (gsn_slow, level, ch, victim, TARGET_CHAR);
    if (number_percent () < 20)
        cold_effect (victim, level - level/10, dam, TARGET_CHAR);
}

// --------------------------------------------------------------------------------
// dam_light, range
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_blazing_sun)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    bool        improved = check_violet_stars (ch);
    int         dam;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -8, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/15, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -5 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 15, DAM_LIGHT, improved);
        return;
    }

    if (victim != ch)
    {
        act ("Blazing arrow darts from your tattoo and strikes $N.", ch, NULL, victim, TO_CHAR);
        act ("Blazing arrow darts from $n's tattoo and strikes $N.", ch, NULL, victim, TO_ROOM);
        act ("Blazing arrow darts from $n's tattoo and strikes you.", ch, NULL, victim, TO_VICT);
    }
    else
    {
        act ("Blazing arrow darts from $n's tattoo and strikes $gn{him}.", ch, NULL, NULL, TO_ROOM);
        act ("Blazing arrow darts from your tattoo and strikes you.", ch, NULL, NULL, TO_CHAR);
    }

    if (improved)
        level += number_range (2,5);

    // damage
    dam = number_range(level*4, level*8);
    if (improved)
        dam += number_range (dam/10, dam/5);

    if (saves_spell(level, victim, DAM_LIGHT))
        dam /= 2;
    else
        spell_blindness(sn, 2*level/3, ch, (void *) victim, TARGET_CHAR);

    damage(ch, victim, dam, sn, DAM_LIGHT, TRUE);
}

// --------------------------------------------------------------------------------
// slow on victim, immunity to slow for char
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_sandglass)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    bool        improved = check_violet_stars (ch); 

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -7, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -15 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 5, DAM_MAGIC, improved);
        return;
    }

    if (improved)
        level += number_range (2,4);

    act ("The sand in sandglass on $n's skin moves!", ch, NULL, NULL, TO_ROOM);
    act ("The sand in sandglass on your skin moves!", ch, NULL, NULL, TO_CHAR);

    spell_slow (gsn_slow, level, ch, victim, TARGET_CHAR);
}

// --------------------------------------------------------------------------------
// strengthens all other tattooes
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_violet_stars)
{
    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)*2, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -10, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/10, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_AC, - 30 - LVL(ch)/2, 0, FALSE);
        return;
    }
    char_act ("{MV{miolet {MS{mtars{x on your skin becomes warm but you can't notice any other visible effect.", ch);
}

// --------------------------------------------------------------------------------
// increases melee damage when painted
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_curved_dagger)
{
    bool improved = check_violet_stars (ch);
    int  dam;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -7, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, - 15 - LVL(ch)/2, 0, FALSE);
        return;
    }

    // activate tattoo.. No, no! Just receive some damage
    act("{DCurv{We{Dd{x dagger from $n's tattoo pricks $gn{him}.", ch, NULL, NULL, TO_ROOM);
    act("{DCurv{We{Dd{x dagger from your tattoo pricks you.", ch, NULL, NULL, TO_CHAR);
    dam = number_range (level * 2, level * 4);
    damage(ch, ch, dam, sn, DAM_PIERCE, TRUE);
}

// --------------------------------------------------------------------------------
// web or dam_bash to victim
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_octopus)
{
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    bool        improved = check_violet_stars (ch); 
    int         dam;
    AFFECT_DATA af;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/6, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, - 15 - LVL(ch)/3, 0, FALSE);
        return;
    }

    if (improved)
        level += number_range (1,3);

    act("An enormous octopus on $n's tattoo moves...", ch, NULL, NULL, TO_ROOM);
    act("An enormous octopus on your tattoo moves...", ch, NULL, NULL, TO_CHAR);

    // web
    if (!IS_AFFECTED (victim, AFF_WEB))
    {
        if (saves_spell (level, victim, DAM_MAGIC))
        {
            act("{x   ... but failed to muffle $N.", ch, NULL, victim, TO_NOTVICT);
            act("{x   ... but failed to muffle $N.", ch, NULL, victim, TO_CHAR);
            act("{x   ... but failed to muffle you.", ch, NULL, victim, TO_VICT);
            return;
        }

        af.type      = sn;
        af.level     = level;
        af.duration  = 1;
        af.location  = APPLY_HITROLL;
        af.modifier  = -1 * (level / 6);
        af.where     = TO_AFFECTS;
        af.bitvector = AFF_WEB;
        affect_to_char(victim, &af);

        af.location  = APPLY_DEX;
        af.modifier  = -2;
        affect_to_char(victim, &af);

        af.location  = APPLY_DAMROLL;
        af.modifier  = -1 * (level / 6);
        affect_to_char(victim, &af);

        char_act("{x   ... and muffles you with his tentacles.", victim);
        act("{x   ... and muffles $N with his tentacles.", ch, NULL, victim, TO_NOTVICT);
        act("{x   ... and muffles $N with his tentacles.", ch, NULL, victim, TO_CHAR);
        return;
    }
    // damage
    dam = number_range (level * 2, level * 4);
    damage(ch, ch, dam, sn, DAM_BASH, TRUE);
}

// --------------------------------------------------------------------------------
// steal a spell from victim
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_black_sun)
{
    bool improved = check_violet_stars (ch); 

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        return;
    }
    char_act ("Activating tattoo", ch);
}

// --------------------------------------------------------------------------------
// spell poison, if victim is already poisoned - DAM_POSION. 
// Damage is bigger when victim is weakened or plagued
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_queen_cobra)
{
    bool        improved = check_violet_stars (ch); 
    CHAR_DATA * victim = (CHAR_DATA *) vo;
    int         dam;
    AFFECT_DATA af;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch)/2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -15 - LVL(ch)/3, 0, improved);

        create_resist (vo, sn, LVL(ch), APPLY_NONE, 15, DAM_POISON, improved);
        return;
    }

    act("It seems that large cobra on $n's skin comes to life and starts to hiss.", ch, NULL, NULL, TO_ROOM);
    act("It seems that large cobra on your skin comes to life and starts to hiss.", ch, NULL, NULL, TO_CHAR);

    // victim is not poisoned - poison it
    if (!IS_AFFECTED(victim, AFF_POISON))
    {
        af.where     = TO_AFFECTS;
        af.type      = sn;
        af.level     = level;
        af.duration  = (10 + level / 10);
        af.location  = APPLY_STR;
        af.modifier  = -5;
        af.bitvector = AFF_POISON;
        affect_join(victim, &af);
        char_act("    !", victim);
        act(" $N   .",ch,NULL,victim,TO_CHAR);
    } 
    else 
    {
        // damage to poisoned victim
        dam = number_range (level * 5, level * 7);
        if (IS_AFFECTED(victim, AFF_PLAGUE))
            dam *= 3/2;
        if (is_affected(victim, gsn_weaken))
            dam *= 3/2;
        damage(ch, victim, dam, sn, DAM_POISON, TRUE);
    }
}

// --------------------------------------------------------------------------------
// melee attacks drain mana and hp from victim
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_yellow_bat)
{
    bool improved = check_violet_stars (ch); 

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)*2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -15 - LVL(ch)/2, 0, improved);
        return;
    }
    char_act ("{YYellow{x {Dbat{x on your tattoo flaps its wings.", ch);
}

// --------------------------------------------------------------------------------
// some tattooes affect shaman's group
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_white_cloud)
{
    bool improved = check_violet_stars (ch); 

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_SAVING_SPELL, -12, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch)*2, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, improved);
        create_affect (vo, sn, LVL(ch), APPLY_AC, -15 - LVL(ch)/2, 0, improved);
        return;
    }
    char_act ("{WWhite{x {Ccloud{x on your skin moves a little.", ch);
}

// --------------------------------------------------------------------------------
// Tattoo for inquisitors. Something like war_paint
// --------------------------------------------------------------------------------
SPELL_FUN(tattoo_tigers_snout)
{
    CHAR_DATA * victim, * v_next;

    if (target == TAR_CREATING_TATTOO)
    {
        create_affect (vo, sn, LVL(ch), APPLY_HIT, LVL(ch), 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_MANA, LVL(ch), 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_SAVES, -5, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_HITROLL, LVL(ch)/10, 0, FALSE);
        create_affect (vo, sn, LVL(ch), APPLY_DAMROLL, LVL(ch)/10, 0, FALSE);
        return;
    }

    char_act ("Snout of tiger on your tattoo grins evilly.\n", ch);
    act ("Snout of tiger on $n's tattoo grins evilly.\n", ch, NULL, NULL, TO_ROOM);
    for (victim = ch->in_room->people; victim; victim = v_next)
    {
        v_next = victim->next_in_room;
        if (victim->fighting == ch && IS_NPC(victim) && IS_AFFECTED(victim, AFF_CHARM)) 
        {
            if (number_percent () < 20 && !(saves_spell(level, victim, DAM_MAGIC)))
            {
                // small chance to scare an enemy and cause him to flee
                if (!IS_SET(victim->in_room->room_flags, ROOM_BATTLE_ARENA))
                    do_flee(victim, str_empty);
            }
            else // brave enemy... receive a small (?) amount of damage
            {
                int dam = number_range(level*2, level*5);

                if (saves_spell(level, victim, DAM_MAGIC))
                    dam /= 2;

                damage(ch, victim, dam, sn, DAM_MAGIC, TRUE);
            }
        }
    }
}

// --------------------------------------------------------------------------------
// RITUAL Spell functions
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// after this ritual all mobs (and PC in PK) in the area lose half of their HP
// caster will die with 85% probability (at 100% skill)
// todo: see ritual_fire_wind
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_dark_moon)
{
    CHAR_DATA * wch;
    AFFECT_DATA af;
    int         count = 0, 
                total = 0;
    flag64_t    act_flags = 0;

    if (ch->in_room == NULL)
        return;

    if (is_affected (ch,sn))
    {
        char_act("You can't perform this ritual yet.", ch);
        return;
    }

    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 15;
    af.bitvector    = 0;
    af.modifier     = 0;
    af.location     = APPLY_NONE;
    affect_to_char (ch, &af);

    act_flags = ACT_HEALER | ACT_QUESTOR    | ACT_CLAN_GUARD | ACT_NOTRACK | 
                ACT_FORGER | ACT_REPAIRMAN  | ACT_AREA_GUARD;

    char_act ("You raise your hands and the {Ddarkness{x floods all around you.", ch);
    act ("$n raises $gn{his} hands and the {Ddarkness{x floods all around $gn{him}.", ch, NULL, NULL, TO_ROOM);
    for (wch = char_list; wch != NULL; wch = wch->next)
    {
        // skip the caster. He will receive his own damage after it all
        if (wch == ch)
            continue;

        // skip "nowhere" chars
        if (wch->in_room == NULL)
            continue;
        // skip chars from other areas
        if (wch->in_room->area != ch->in_room->area)
            continue;

        ++total;
        char_act ("Suddenly {Ddarkness{x floods the room...", wch);
        // skip 'special' mobiles
        if (IS_NPC(wch) && IS_SET(wch->pIndexData->act, act_flags))
            continue;
        // skip chars not in PK
        if (is_safe_nomessage(ch, wch))
            continue;

        // saves give 40% chance to avoid harm
        if (saves_spell(level, wch, DAM_MAGIC) && number_percent () < 40)
            continue;

        // all is OK. Harm victim.
        ++count;
        char_act ("You feel insufferable pain!", wch);
        act ("$n screams in agony as $gn{his} energy drained.", wch, NULL, NULL, TO_ROOM);
        wch->hit /= 2;
    }

    // deal with the caster
    if (IS_IMMORTAL(ch))
    {
        char_printf (ch, "Victims harmed: %d of %d\n", count, total);
        return;
    }

    if (number_percent () > (get_skill (ch, sn) - 85))
    {
        char_act ("The flood of {Ddarkness{x drains all your energy.", ch);
        act ("It seems the flood of {Ddarkness{x drains all $n's energy... $gn{He} is dead.", ch, NULL, NULL, TO_ROOM);
        handle_death (ch, ch);
        return;
    }
}

// --------------------------------------------------------------------------------
// summon powerful treant to serve the caster
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_call_treant)
{
    CHAR_DATA      * gch;
    CHAR_DATA      * treant;
    MOB_INDEX_DATA * ind;
    AFFECT_DATA      af;
    int              i = 0;

    if (is_affected (ch,sn) || count_charmed(ch))
    {
        char_act("You are too tired to call a treant.", ch);
        return;
    }

    char_act("You call a treant.", ch);
    act("$n calls a treant.", ch, NULL, NULL, TO_ROOM);

    for (gch = char_list; gch != NULL; gch = gch->next)
    {
        if (IS_NPC(gch) && IS_AFFECTED(gch,AFF_CHARM) && gch->master == ch
            && (gch->pIndexData->vnum == MOB_VNUM_TREANT))
        {
            do_tell_raw (gch, ch, GETMSG("Master, don't leave me!", ch->lang));
            return;
        }
    }

    ind = get_mob_index (MOB_VNUM_TREANT);
    if (ind == NULL)
    {
        char_act ("{RInternal error! Please report it to Immortals.{x", ch);
        return;
    }

    treant = create_mob (ind);

    for (i = 0; i < MAX_STATS; i ++)
       treant->perm_stat[i] = UMIN(25,15 + LVL(ch)/10);

    treant->perm_stat[STAT_STR] += 3;
    treant->perm_stat[STAT_INT] -= 1;
    treant->perm_stat[STAT_CON] += 2;

    treant->max_hit = IS_NPC(ch)? URANGE(ch->max_hit,1 * ch->max_hit,30000)
        : UMIN((2 * ch->pcdata->perm_hit) + 4000, 30000);
    treant->alignment = ch->alignment;
    treant->hit = treant->max_hit;
    treant->max_mana = IS_NPC(ch)? ch->max_mana : ch->pcdata->perm_mana;
    treant->mana = treant->max_mana;
    treant->level = LVL(ch);
    for (i=0; i < 3; i++)
        treant->armor[i] = interpolate(treant->level,100,-100);
    treant->armor[3] = interpolate(treant->level,100,0);
    treant->gold = 0;
    treant->timer = 0;
    treant->damage[DICE_NUMBER] = 13;
    treant->damage[DICE_TYPE] = 9;
    treant->damage[DICE_BONUS] = level / 2 + 10;

    char_to_room (treant,ch->in_room);

    char_act("Powerful treant came to your call!", ch);
    act("Powerful treant came to $n's call!", ch, NULL, NULL, TO_ROOM);

    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 45;
    af.bitvector    = 0;
    af.modifier     = 0;
    af.location     = APPLY_NONE;
    affect_to_char (ch, &af);

    SET_BIT(treant->affected_by, AFF_CHARM);
    treant->master = treant->leader = ch;
}

// --------------------------------------------------------------------------------
// RAFF: noone can walk through this circle, but it weakens for each 
// attempt and each second
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_circle_of_protection)
{
    AFFECT_DATA af,af2;

    if (ch->in_room == NULL)
        return;

    if (is_affected_room(ch->in_room, sn))
    {
        char_act("This room is already as safe as possible!", ch);
        return;
    }

    if (is_affected(ch, sn))
    {
        char_act("You can't protect another room yet!", ch);
        return;
    }

    act("$n starts to chant ancient incantation.", ch, NULL, NULL, TO_ROOM);

    af.where      = TO_ROOM_AFFECTS;
    af.type       = sn;
    af.level      = level;
    af.duration   = 6;
    af.location   = APPLY_NONE;
    af.modifier   = level/5;
    af.bitvector  = RAFF_PROTECTION;
    affect_to_room (ch->in_room, &af);

    af2.where     = TO_AFFECTS;
    af2.type      = sn;
    af2.level     = level;
    af2.duration  = 15;
    af2.modifier  = 0;
    af2.location  = APPLY_NONE;
    af2.bitvector = 0;
    affect_to_char (ch, &af2);
    act("The room is now protected from enemies!", ch, NULL, NULL, TO_ALL);
    return;
}

// --------------------------------------------------------------------------------
// all members of caster's group gain full hp/mana/moves
// caster has 1 hp/mana/move
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_selfsacrifice)
{
    AFFECT_DATA af;
    CHAR_DATA * gch; 

    if (is_affected (ch,sn))
    {
        char_act("You can't perform this ritual yet.", ch);
        return;
    }

    act ("You sprinkle all around you with your blood.", ch, NULL, NULL, TO_CHAR);
    act ("$n sprinkles all around $gn{him} with your blood.", ch, NULL, NULL, TO_ROOM);

    for (gch = ch->in_room->people; gch; gch = gch->next_in_room) 
    {
        if (!is_same_group(ch, gch))
            continue;
        gch->hit = gch->max_hit;
        gch->mana = gch->max_mana;
        gch->move = gch->max_move;
        act ("$n's sacrificial blood restores you!", ch, NULL, gch, TO_VICT);
    }

    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 20;
    af.bitvector    = 0;
    af.modifier     = 0;
    af.location     = APPLY_NONE;
    affect_to_char (ch, &af);

    if (number_percent () < 10)
    {
        act ("You lost too much blood... You are dead.", ch, NULL, NULL, TO_CHAR);
        act ("$n lost too much blood... $gn{He} is dead.", ch, NULL, NULL, TO_ROOM);
        handle_death (ch, ch);
    }
    else
    {
        act ("You feel SO bad!", ch, NULL, NULL, TO_CHAR);
        ch->hit = 1;
        ch->mana = 0;
        ch->move = 0;
    }
}

// --------------------------------------------------------------------------------
// all persons in the area receive fire_effect and faerie fire spell
// todo: generate event for it, than if char is damaged - cancel event
//       all following code must be moved into callback function
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_fire_wind)
{
    CHAR_DATA * wch;
    AFFECT_DATA af;
    int         count = 0, 
                total = 0, 
                dam;
    flag64_t    act_flags = 0;
    

    if (ch->in_room == NULL)
        return;

    if (is_affected (ch,sn))
    {
        char_act("You can't perform this ritual yet.", ch);
        return;
    }

    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 0;
    af.bitvector    = 0;
    af.modifier     = 0;
    af.location     = APPLY_NONE;
    affect_to_char (ch, &af);

    act_flags = ACT_HEALER | ACT_QUESTOR    | ACT_CLAN_GUARD | ACT_NOTRACK | 
                ACT_FORGER | ACT_REPAIRMAN  | ACT_AREA_GUARD;

    char_act ("You raise your hands and send a stream of heat in the air.", ch);
    act ("$n raises $gn{his} hands and sends a stream of heat in the air.", ch, NULL, NULL, TO_ROOM);
    for (wch = char_list; wch != NULL; wch = wch->next)
    {
        // skip the caster. He will receive his own damage after it all
        if (wch == ch)
            continue;

        // skip "nowhere" chars
        if (wch->in_room == NULL)
            continue;
        // skip chars from other areas
        if (wch->in_room->area != ch->in_room->area)
            continue;

        ++ total;

        char_act ("You feel a capful of extremly {Rhot{x wind.{x", wch);
        // skip 'special' mobiles
        if (IS_NPC(wch) && IS_SET(wch->pIndexData->act, act_flags))
            continue;
        // skip chars not in PK
        if (is_safe_nomessage(ch, wch))
            continue;
        // saves give 20% chance to avoid ritual effect
        if (saves_spell(level, wch, DAM_FIRE) && number_percent () < 20)
            continue;

        // make char visible
        do_visible (wch, str_empty);

        // saves give 40% chance to avoid harm
        if (saves_spell(level, wch, DAM_FIRE) && number_percent () < 40)
            continue;

        // all is OK. Harm victim.
        ++count;
        if (saves_spell(level, wch, DAM_FIRE) && number_percent () < 50)
        {
            fire_effect (wch, level/2, 100, TARGET_CHAR);
        }
        else
        {
            fire_effect (wch, level + level/20, 100, TARGET_CHAR);
            if (!IS_AFFECTED(wch, AFF_FAERIE_FIRE))
            {
                af.where     = TO_AFFECTS;
                af.type      = gsn_faerie_fire;
                af.level     = level;
                af.duration  = level / 5;
                af.location  = APPLY_AC;
                af.modifier  = 2 * level;
                af.bitvector = AFF_FAERIE_FIRE;
                affect_to_char(wch, &af);
                char_act("You are surrounded by {Mpink aura{x.", wch);
                act("$n is surrounded by {Mpink aura{x.", wch, NULL, NULL, TO_ROOM);
            }
        }
    }

    // deal with the caster
    if (IS_IMMORTAL(ch))
    {
        char_printf (ch, "Victims harmed: %d of %d\n", count, total);
        return;
    }

    if (number_percent () > (get_skill (ch, sn) - 30) && !saves_spell(level, ch, DAM_FIRE))
    {
        char_act ("Fire wind broke from under your control and burnt you.", ch);
        act ("Fire wind broke from under $n's control and burnt $gn{him}.", ch, NULL, NULL, TO_ROOM);
        handle_death (ch, ch);
        return;
    }
    else
    {
        char_act ("You are burnt by the fire wind.", ch);
        act ("$n is burnt by the fire wind.", ch, NULL, NULL, TO_ROOM);
        if (number_percent () < 10)
            fire_effect (wch, level - level/5, 100, TARGET_CHAR);
        dam = number_range (level * 3, level * 5);
        damage (ch, ch, dam, sn, DAM_FIRE, TRUE);
    }
}

// --------------------------------------------------------------------------------
// ritual 'eye of Torrog'
// allows shaman to use pills/potions/staves/wands/scrolls 
// with various detects (effect is combined)
// --------------------------------------------------------------------------------
#define MAX_INVIS 6 // detect invis, improved detect, detect hide, 
                    // acute vision, detect fade, detect magic
struct detect_t 
{
	int sn;
    int duration;
    int level;
    flag64_t bitvector;
    int (*calc_duration) (int level);
    const char * msg;
};

static int dur_detect_invis (int level)
{
    return (5 + level / 2);
}
static int dur_impr_detect (int level)
{
    return (5 + level / 3);
}
static int dur_detect_hide (int level)
{
    return (5 + level / 3);
}
static int dur_acute_vision (int level)
{
    return (5 + level / 3);
}
static int dur_detect_fade (int level)
{
    return 5;
}
static int dur_detect_magic (int level)
{
    return (5 + level / 3);
}

static struct detect_t detects[MAX_INVIS] = 
{
    // detect invis
    {0, 0, 0, AFF_DETECT_INVIS,     dur_detect_invis, "    ."},
    // improved detect
    {0, 0, 0, AFF_DETECT_IMP_INVIS, dur_impr_detect,  "     ."},
    // detect hidden
    {0, 0, 0, AFF_DETECT_HIDDEN,    dur_detect_hide,  "  ."},
    // acute vision
    {0, 0, 0, AFF_ACUTE_VISION,     dur_acute_vision, "    !"},
    // detect fade
    {0, 0, 0, AFF_DETECT_FADE,      dur_detect_fade,  "  ."},
    // detect magic
    {0, 0, 0, AFF_DETECT_MAGIC,     dur_detect_magic, "   ."},
};

SPELL_FUN(ritual_eye_of_torrog)
{
    OBJ_DATA    * obj,
                * obj_next;
    int           i, j;
    AFFECT_DATA   af; 
    AFFECT_DATA * paf;
    bool          found, invis_found;
    int           chance_blind;

    if (is_affected (ch, sn))
    {
        char_act ("Torrog doesn't want to hear you!", ch);
        return;
    }

    act("You call to Torrog and ask him to grant you keen vision...", ch, NULL, NULL, TO_CHAR);
    act("$n makes some strange looking motions...", ch, NULL, NULL, TO_ROOM);
    chance_blind = get_skill (ch, sn) * 95 / 100;
    if (number_percent() > chance_blind) // Torrog can blind unlucky worshipper
    {
        af.where     = TO_AFFECTS;
        af.type      = sn;
        af.level     = level;
        af.location  = APPLY_HITROLL;
        af.modifier  = 0 - number_fuzzy(level/3);
        af.duration  = 0;
        af.bitvector = AFF_BLIND;
        affect_to_char(ch, &af);
        act("{x   ... but angry deity inflicts blindness to you.", ch, NULL, NULL, TO_CHAR);
        act("{x   ... but as far as you can see nothing happens.", ch, NULL, NULL, TO_ROOM);
        return;
    }

    act("{x   ... and misty frog-like figure appears before you.", ch, NULL, NULL, TO_ALL);

    // init table
    detects[0].sn = sn_lookup ("detect invis");
    detects[1].sn = sn_lookup ("improved detect");
    detects[2].sn = sn_lookup ("detect hidden");
    detects[3].sn = sn_lookup ("acute vision");
    detects[4].sn = sn_lookup ("detect fade");
    detects[5].sn = sn_lookup ("detect magic");
    for (i = 0; i < MAX_INVIS; ++i)
    {
        detects[i].duration = 0;
        detects[i].level = 0;
    }

    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 1;
    af.location     = APPLY_NONE;
    af.modifier     = 0;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    for (obj = ch->carrying; obj != NULL; obj = obj_next)
    {
        obj_next = obj->next_content;
        found = FALSE;
        switch (obj->pIndexData->item_type)
        {
        case ITEM_STAFF:
        case ITEM_WAND:
            for (i = 0; i < MAX_INVIS; ++i)
                if (obj->value[3] == detects[i].sn)
                {
                    detects[i].level += obj->value[0];
                    detects[i].duration += detects[i].calc_duration (obj->value[0]);
                    found = TRUE;
                    break;
                }
            break;
        case ITEM_PILL:
        case ITEM_POTION:
        case ITEM_SCROLL:
            for (i = 0; i < MAX_INVIS; ++i)
                for (j = 1; j < 5; ++j)
                    if (obj->value[j] == detects[i].sn)
                    {
                        detects[i].level += obj->value[0];
                        detects[i].duration += detects[i].calc_duration (obj->value[0]);
                        found = TRUE;
                        break;
                    }
            break;
        }
        if (found)
        {
            act("Your $p disappears.", ch, obj, NULL, TO_CHAR);
            extract_obj (obj);
        }
    }
    af.where     = TO_AFFECTS;
    af.modifier  = 0;
    af.location  = APPLY_NONE;

    char_act (str_empty, ch);

    invis_found = FALSE;
    for (i = 0; i < MAX_INVIS; ++i)
    {
        paf = affect_find(ch->affected, detects[i].sn);
        if (paf != NULL)
        {
            if (detects[i].duration == 0)
                detects[i].duration = 2 + LVL(ch) / 9;
            paf->duration = UMIN (paf->duration + detects[i].duration, LVL(ch) * 4);
            paf->level = UMIN (paf->level + detects[i].level, LVL(ch) + LVL(ch) /10);
            if (i == 0)
                invis_found = TRUE;
        }
        else
        {
            if (detects[i].duration != 0) // skip empty slots
            {
                af.type      = detects[i].sn;
                af.level     = UMIN (detects[i].level, LVL(ch) + LVL(ch) /10);
                af.bitvector = detects[i].bitvector;
                af.duration  = detects[i].duration;
                affect_join (ch, &af);
                char_act (detects[i].msg, ch);
            }
        }
    }
    // in all cases: detect invis
    if (!invis_found)
    {
        af.type      = detects[0].sn;
        af.level     = 2;
        af.duration  = 2 + LVL(ch) / 45;
        af.bitvector = detects[0].bitvector;
        affect_join (ch, &af);
        if (!is_affected (ch, detects[0].sn))
            char_act (detects[0].msg, ch);
    }

    char_act ("Torrog grants you keen vision!", ch);
}

// --------------------------------------------------------------------------------
// all persons in the area receive slow, curse and weaken spells
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_stone_curse)
{
    char_act ("ritual_stone_curse", ch);
}

// --------------------------------------------------------------------------------
// a sort of gate spell
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_call_to_naata)
{
    char_act ("ritual_call_to_naata", ch);
}

// --------------------------------------------------------------------------------
// trap
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_roots_and_branches)
{
    char_act ("ritual_roots_and_branches", ch);
}

// --------------------------------------------------------------------------------
// a sort of summon spell
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_forest_call)
{
    char_act ("ritual_forest_call", ch);
}

// --------------------------------------------------------------------------------
// destroy all objects with flag 'MAGIC' on the ground 
// and give some mana to the shaman
// --------------------------------------------------------------------------------
SPELL_FUN(ritual_flame_of_torrog)
{
    OBJ_DATA    * obj;
    OBJ_DATA    * r_next_cont;
    int           add_mana = 0;
    AFFECT_DATA   af; 
    int           chance;

    if (is_affected (ch, sn))
    {
        char_act ("Torrog doesn't want to hear you!", ch);
        return;
    }

    act("You call to Torrog and ask him to grant you magical energy...", ch, NULL, NULL, TO_CHAR);
    act("$n chants loudly...", ch, NULL, NULL, TO_ROOM);
    chance = get_skill (ch, sn) * 95 / 100;
    if (number_percent() > chance) // Torrog can curse unlucky worshipper
    {
        af.where     = TO_AFFECTS;
        af.type      = sn;
        af.level     = level;
        af.location  = APPLY_HITROLL;
        af.modifier  = 0 - number_fuzzy(level/3);
        af.duration  = 0;
        af.bitvector = AFF_CURSE;
        affect_to_char(ch, &af);
        act("{x   ... but angry deity curses you.", ch, NULL, NULL, TO_CHAR);
        act("{x   ... but as far as you can see nothing happens.", ch, NULL, NULL, TO_ROOM);
        return;
    }

    act("{x   ... and misty frog-like figure appears before you.", ch, NULL, NULL, TO_ALL);
    
    af.where        = TO_AFFECTS;
    af.type         = sn;
    af.level        = level;
    af.duration     = 2;
    af.location     = APPLY_NONE;
    af.modifier     = 0;
    af.bitvector    = 0;
    affect_to_char(ch, &af);

    for (obj = ch->in_room->contents; obj; obj = r_next_cont)
    {
        r_next_cont = obj->next_content;
        if (!IS_OBJ_STAT(obj, ITEM_MAGIC))
            continue;

        if (IS_OBJ_STAT(obj, ITEM_NOPURGE)
        || IS_OBJ_STAT(obj, ITEM_BURN_PROOF)
        || IS_OBJ_STAT(obj, ITEM_QUEST)
        || IS_OBJ_STAT(obj, ITEM_INDESTRUCTABLE)
        || obj->level > level * 3 / 2
        || obj->pIndexData->limit != -1
        || check_material(obj, "unique"))
            continue;

        // all is ok.. convert item to mana
        add_mana += obj->level * get_skill(ch, sn) / 20;
        act("$p is consumed by Torrog.", ch, obj, NULL, TO_ALL);
        extract_obj(obj);
    }
    if (add_mana > 0)
    {
        ch->mana = UMIN(ch->mana + add_mana, ch->max_mana); 
        char_act ("You feel a surge of magic energy.", ch);
    }
}

// --------------------------------------------------------------------------------
// OTHER Spell functions
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// Removes ALL tattooes from caster's skin or destroys one tattoo 
// from enemy's skin
// For each removed tattoo caster receives tattoo_level/5 mana
// --------------------------------------------------------------------------------
SPELL_FUN(spell_burn_skin)
{
    OBJ_DATA  ** objects = NULL;
    int          obj_count = 0;
    OBJ_DATA   * obj, 
               * obj_next;
    CHAR_DATA  * victim = (CHAR_DATA *) vo;
    bool         found = FALSE;
    int          dam;

    // Cast on self
    if (victim == ch || victim == NULL)
    {
        for (obj = ch->carrying; obj != NULL; obj = obj_next)
        {
            obj_next = obj->next_content;

            // skip non-tattooes
            if (obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
                continue;
            ch->mana += number_range (obj->value[0], obj->value[0] * 2);

            extract_obj (obj);
            found = TRUE;
        }
        if (found)
        {
            char_act ("Your skin has been burnt so all of your tattooes were destroyed.", ch);
            char_act ("You feel a surge of magic energy.", ch);
        }
        else
            char_act ("Your skin feels heat but nothing happens.", ch);
    }
    // cast on other
    else
    {
        // select potential "victims"
        for (obj = victim->carrying; obj != NULL; obj = obj->next_content)
        {
            if (obj->pIndexData->item_type != ITEM_SHAMAN_TATTOO)
                continue;

            if (objects == NULL)
                objects = (OBJ_DATA**) calloc (1, sizeof (OBJ_DATA*) * (++obj_count));
            else 
                objects = (OBJ_DATA**) realloc (objects, sizeof (OBJ_DATA*) * (++obj_count));
            objects[obj_count-1] = obj;
        }

        // try to destroy tattoo
        if (obj_count > 0 && !saves_spell(level, victim, DAM_MAGIC))
        {
            obj = objects[number_range (0, obj_count - 1)];
            act ("$p on $n's skin is destroyed.\n", victim, obj, NULL, TO_ROOM);
            act ("$p on your skin is destroyed.\n", ch, obj, victim, TO_VICT);

            // ch receives some mana, victim - some damage
            ch->mana += number_range (obj->value[0], obj->value[0] * 2);

            dam = number_range (obj->value[0], obj->value[0] * 2);
            damage(ch, victim, dam, sn, DAM_FIRE, TRUE);

            extract_obj (obj);
        }
        else // no object to destroy or saves check successlully passed
        {
            act ("You feel heat from $N's skin but nothing happens.", ch, NULL, victim, TO_NOTVICT);
            char_act ("Your skin feels heat but nothing happens.", victim);
            act ("Your try to burn $N's skin but nothing happens.", ch, NULL, victim, TO_CHAR);
        }
        free (objects);
    }
}

// --------------------------------------------------------------------------------
// VOODOO magic
// --------------------------------------------------------------------------------

// --------------------------------------------------------------------------------
// make a voodoo doll from body part.. can only be done from PC's body part
// --------------------------------------------------------------------------------
SPELL_FUN(spell_voodoo_doll)
{
}

// --------------------------------------------------------------------------------
// command called from the interpreter
// --------------------------------------------------------------------------------
DO_FUN (do_voodoo)
{
    int         chance; 
    social_t  * soc;
    //CHAR_DATA * victim;
    //char        arg[MAX_INPUT_LENGTH];    

    if ((chance = get_skill (ch, gsn_voodoo)) == 0) // ch has no skill - interpret social
    {
        soc = social_lookup ("voodoo", str_cmp);
        interpret_social (soc, ch, str_empty);
    }
}
