/* $Id: str.c,v 1.666 2004/09/20 10:49:53 shrike Exp $                               */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
// Welesh : compat
#if	!defined (WIN32)
#include <unistd.h>
#endif

#include "typedef.h"
#include "const.h"
#include "str.h"

#if defined (WIN32)
#   define vsnprintf    _vsnprintf
int strncasecmp (const char *s1, const char *s2, size_t n);
int strcasecmp (const char *s1, const char *s2);
#endif

char str_empty[1];

int str_count;
int str_real_count;

#define strhash(s)  (hashstr(s, 64, MAX_STRING_HASH))

typedef struct str str;
struct str {
    int     ref;
    str *		next;
    char    	p[0];
};

str *hash_str[MAX_STRING_HASH];

static str *str_lookup(const char* p, int *hash);
static str *str_alloc(const char*, int hash);

const char *str_dup(const char *p)
{
    int hash;
    str *s;

    if (IS_NULLSTR(p))
        return str_empty;

    str_count++;
    if ((s = str_lookup(p, &hash)) == NULL)
        s = str_alloc(p, hash);
    s->ref++;
    return s->p;
}

const char *str_qdup(const char *p)
{
    str *s;

    if (IS_NULLSTR(p))
        return str_empty;

    str_count++;
	s = (str *)(p - sizeof(str));
    s->ref++;
    return p;
}

void free_string(const char *p)
{
    str *s, *q;
    int hash;

    if (p == NULL || p == str_empty)
        return;

    str_count--;
    hash = strhash(p);
    for (q = NULL, s = hash_str[hash]; s; s = s->next) {
        if (!strcmp(s->p, p))
            break;
        q = s;
    }

    if (!s || --s->ref)
        return;

    if (q)
        q->next = s->next;
    else
        hash_str[hash] = hash_str[hash]->next;
    str_real_count--;
	free(s);
}
    
/*
 * str_printf -- snprintf to string
 */
const char *str_printf(const char* format,...)
{
    va_list ap;
    char buf[MAX_STRING_LENGTH];

    if (format == NULL)
        return NULL;

    va_start(ap, format);
    vsnprintf(buf, sizeof(buf), format, ap);
    va_end(ap);

    return str_dup(buf);
}

/*
 * strnzcpy - copy from dest to src and always append terminating '\0'.
 *            len MUST BE > 0
 */
char *strnzcpy(char *dest, size_t len, const char *src)
{
    strncpy(dest, src, len);
    dest[len-1] = '\0';
    return dest;
}

char *strnzcat(char *dest, size_t len, const char *src)
{
    size_t old_len;

    old_len = strlen(dest);
    if (old_len >= len - 1)
        return dest;

    strncat(dest, src, len - old_len - 1);
    return dest;
}

char *strnzncat(char *dest, size_t len, const char *src, size_t count)
{
    size_t old_len;

    old_len = strlen(dest);
    if (old_len >= len - 1)
        return dest;

    strncat(dest, src, UMIN(len - old_len - 1, count));
    return dest;
}

char *strlwr(const char *s)
{
    static char buf[MAX_STRING_LENGTH];
    char *p;

    if (s == NULL)
        return str_empty;

    for (p = buf; p-buf < sizeof(buf)-1 && *s; s++, p++)
        *p = LOWER(*s);
    *p = '\0';
    return buf;
}

/*
 * Compare strings, case insensitive.
 */
int str_cmp(const char *astr, const char *bstr)
{
    if (astr == NULL)
        return bstr == NULL ? 0 : -1;
    if (bstr == NULL)
        return 1;
    return strcasecmp(astr, bstr);
}

int str_ncmp(const char *astr, const char *bstr, size_t len)
{
    if (astr == NULL)
        return bstr == NULL ? 0 : -1;
    if (bstr == NULL)
        return 1;
    return strncasecmp(astr, bstr, len);
}

/*
 * Compare strings, case insensitive, for prefix matching.
 * Return TRUE if astr not a prefix of bstr
 *   (compatibility with historical functions).
 */
bool str_prefix(const char *astr, const char *bstr)
{
    if (astr == NULL || astr[0] == '\0')
        return TRUE;

    if (bstr == NULL || bstr[0] == '\0')
        return TRUE;

    for (; *astr; astr++, bstr++) {
        if (LOWER(*astr) != LOWER(*bstr))
            return TRUE;
    }

    return FALSE;
}

/*
 * Compare strings, case insensitive, for match anywhere.
 * Returns TRUE is astr not part of bstr.
 *   (compatibility with historical functions).
 */
bool str_infix(const char *astr, const char *bstr)
{
    int sstr1;
    int sstr2;
    int ichar;
    char c0;

    if ((c0 = LOWER(astr[0])) == '\0')
        return FALSE;

    sstr1 = strlen(astr);
    sstr2 = strlen(bstr);

    for (ichar = 0; ichar <= sstr2 - sstr1; ichar++) {
        if (c0 == LOWER(bstr[ichar]) && !str_prefix(astr, bstr + ichar))
            return FALSE;
    }

    return TRUE;
}

/*
 * Compare strings, case insensitive, for suffix matching.
 * Return TRUE if astr not a suffix of bstr
 *   (compatibility with historical functions).
 */
bool str_suffix(const char *astr, const char *bstr)
{
    int sstr1;
    int sstr2;

    sstr1 = strlen(astr);
    sstr2 = strlen(bstr);
    if (sstr1 <= sstr2 && !str_cmp(astr, bstr + sstr2 - sstr1))
        return FALSE;
    else
        return TRUE;
}

/*
** A simple and fast generic string hasher based on Peter K. Pearson's
** article in CACM 33-6, pp. 677.
*/

static int TT[] = {
    1, 87, 49, 12, 176, 178, 102, 166, 121, 193, 6, 84, 249, 230, 44, 163,
    14, 197, 213, 181, 161, 85, 218, 80, 64, 239, 24, 226, 236, 142, 38, 200,
    110, 177, 104, 103, 141, 253, 255, 50, 77, 101, 81, 18, 45, 96, 31, 222,
    25, 107, 190, 70, 86, 237, 240, 34, 72, 242, 20, 214, 244, 227, 149, 235,
    97, 234, 57, 22, 60, 250, 82, 175, 208, 5, 127, 199, 111, 62, 135, 248,
    174, 169, 211, 58, 66, 154, 106, 195, 245, 171, 17, 187, 182, 179, 0, 243,
    132, 56, 148, 75, 128, 133, 158, 100, 130, 126, 91, 13, 153, 246, 216, 219,
    119, 68, 223, 78, 83, 88, 201, 99, 122, 11, 92, 32, 136, 114, 52, 10,
    138, 30, 48, 183, 156, 35, 61, 26, 143, 74, 251, 94, 129, 162, 63, 152,
    170, 7, 115, 167, 241, 206, 3, 150, 55, 59, 151, 220, 90, 53, 23, 131, 
    125, 173, 15, 238, 79, 95, 89, 16, 105, 137, 225, 224, 217, 160, 37, 123,
    118, 73, 2, 157, 46, 116, 9, 145, 134, 228, 207, 212, 202, 215, 69, 229,
    27, 188, 67, 124, 168, 252, 42, 4, 29, 108, 21, 247, 19, 205, 39, 203,
    233, 40, 186, 147, 198, 192, 155, 33, 164, 191, 98, 204, 165, 180, 117, 76,
    140, 36, 210, 172, 41, 54, 159, 8, 185, 232, 113, 196, 231, 47, 146, 120,
    51, 65, 28, 144, 254, 221, 93, 189, 194, 139, 112, 43, 71, 109, 184, 209,
};

int
hashstr(s, maxn, hashs)
const char *s;          /* string to hash */
int maxn;           /* maximum number of chars to consider */
int hashs;          /* hash table size. */
{
    register int h;
    register u_char *p;
    register int i;

    for(h = 0, i = 0, p = (u_char *)s; *p && i < maxn; i++, p++)
    h = TT[h ^ *p];
    if (hashs > 256 && *s) {
    int oh = h;
    for(i = 1, p = (u_char *)s, h = (*p++ + 1)&0xff; *p && i < maxn; i++, p++)
        h = TT[h ^ *p];
    h += (oh << 8);
    }
    return h % hashs;       /* With 16 bit ints h has to be made positive first! */
}

/*
 * case insensitive version of hashstr, hashs must not be greater than 256
 */
int
hashistr(s, maxn, hashs)
const char *s;          /* string to hash */
int maxn;           /* maximum number of chars to consider */
int hashs;          /* hash table size. */
{
    register int h;
    register u_char *p;
    register int i;

    for(h = 0, i = 0, p = (u_char *)s; *p && i < maxn; i++, p++)
    h = TT[h ^ LOWER(*p)];
    return h % hashs;
}

int cmpstr(const void *p1, const void *p2)
{
    return -str_cmp(*(char**) p1, *(char**) p2);
}

/*----------------------------------------------------------------------------
 * static functions
 */

static str *str_alloc(const char *p, int hash)
{
    str *s;
	str_real_count++;
    s = malloc(sizeof(*s) + strlen(p) + 1);
    strcpy(s->p, p);
    s->ref = 0; 
    s->next = hash_str[hash];
    return hash_str[hash] = s;
}

static str *str_lookup(const char *p, int *hash)
{
    str *s;
    for (s = hash_str[*hash = strhash(p)]; s; s = s->next)
        if (!strcmp(s->p, p))
            return s;
    return NULL;
}

/* New functions */
const char *fmt_color_string(const char *txt, int max, int align)
{
	int i, count, len, pre_spaces = 0, post_spaces = 0;
	static char buf[MAX_STRING_LENGTH];
	const char *p;
	bool last = FALSE;

	buf[0] = '\0';
	len = strlen(txt);
	count = 0;
	p = txt;
	for (i = 0; i < len && i < max + count; i++)
	{
		if (*p++ == '{')
		{
			if (last)
			{
				count--;
				last = FALSE;
			}
			else
			{
				if (*p == '\0')
					continue;
				if (*p == '{')
					last = TRUE;
				count += 2;
			}
		}
	}
	
	if (align == ALIGN_LEFT)
		post_spaces = max + count - len;
	else if (align == ALIGN_RIGHT)
		pre_spaces = max + count - len;
	else
	{
		pre_spaces = (max + count - len + 1)/2;
		post_spaces = (max + count - len)/2;
	}

	for (i = 0; i < pre_spaces; i++)
		strnzcat(buf, sizeof(buf), " ");

	strnzncat(buf, sizeof(buf), txt, max + count);

	for (i = 0; i < post_spaces; i++)
		strnzcat(buf, sizeof(buf), " ");

	return buf;
}

const char * fmt_wrap_str(const char * txt, int len, const char * pad)
{
    static char buf[MAX_STRING_LENGTH];
    const char  *ch, *prev, *last;

    if (len < 5) return txt;

    memset (buf, 0, MAX_STRING_LENGTH);
	ch = last = prev = txt;
    for (; *ch; ch++)
    {
        if (*ch == ' ')
	    last = ch;
        if (ch - prev >= len)
        {
            strnzncat(buf, sizeof(buf), prev, last - prev);
            strnzcat(buf, sizeof(buf), "\n");
            strnzcat(buf, sizeof(buf), pad);
            prev = last + 1;
        }
    }

    // tail
    strnzncat(buf, sizeof(buf), prev, ch - prev);
    return buf;
}

char *crypt(const char *key, const char *salt);
char *str_crypt(const char *s)
{
  char *str_crypted;
#if defined (WIN32)
  str_crypted = s;
#else
  str_crypted = crypt(s, CRYPT_SALT);
#endif
    return str_crypted;
}

#if 0
char upp[] = "ABCDEFGHIGKLMNOPQRSTUVWXYZ";
char low[] = "abcdefghigklmnopqrstuvwxyzţ";

char to_lower(char c)
{
    int n;

    if ((n = strchr(upp, c)) == NULL)
    return c;

    n = n - (int)upp;
    return low[n];
}

char to_upper(char c)
{
    int n;

    if ((n = strchr(low, c)) == NULL)
    return c;

    n = n - (int)low;
    return upp[n];
}
#endif

/*
 * format string alignment functions
 */
#define ALIGN_NONE			0
#define ALIGN_LEFT			1
#define ALIGN_RIGHT			2
#define ALIGN_CENTER		        3

char * format_str_len(char * string, int length, int align) {
    char buf[MAX_STRING_LENGTH];
    char buf2[MAX_STRING_LENGTH];
    char *new_string;
    char *count_string;
    char temp;
    int count = 0, nCount = 0;
    int pos = 0;

    new_string = buf;
	count_string = buf2;
	strcpy(buf2, string);

	/* get count for alignment */
	while( *count_string && nCount != length )
    {
        temp = *count_string++;

		if (temp == '{' )
        {
			temp = *count_string++;
			if (temp == '{')
                nCount++;
            continue;
        }
        nCount++;
    }

	/* force alignment of text to the right */
	if(align == ALIGN_RIGHT)
	{
		count = (length - ++nCount);
		while(nCount++ <= length)
		{
	        buf[pos++] = ' ';
		}
	}

	/* force alignment of text to the center */
	if(align == ALIGN_CENTER)
	{
		nCount = (length - nCount) / 2;
		count = nCount;
		while(nCount-- > 0)
		{
	        buf[pos++] = ' ';
		}
	}

	/* add to buffer */
	while( *string && count != length )
    {
        temp = *string++;
        buf[pos++] = temp;

		if (temp == '{' )
        {
            buf[pos] = *string++;

			if (buf[pos] == '{')
                count++;

			pos++;
            continue;
        }
        count++;
    }

	/* pad remaining space with blanks */
	while (count++ < length)
        buf[pos++] = ' ';

	buf[pos] = '\0';

	return (new_string);
}


// Borrowed strlcpy and strlcat from OpenSSH, copyright notice and license is in STRLCPY-STRLCAT-LICENSE file
#ifndef HAVE_STRLCPY
/*
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 * Returns strlen(src); if retval >= siz, truncation occurred.
 */
size_t strlcpy(dst, src, siz)
	char *dst;
	const char *src;
	size_t siz;
{
	register char *d = dst;
	register const char *s = src;
	register size_t n = siz;

	/* Copy as many bytes as will fit */
	if (n != 0 && --n != 0) {
		do {
			if ((*d++ = *s++) == 0)
				break;
		} while (--n != 0);
	}

	/* Not enough room in dst, add NUL and traverse rest of src */
	if (n == 0) {
		if (siz != 0)
			*d = '\0';		/* NUL-terminate dst */
		while (*s++)
			;
	}

	return(s - src - 1);	/* count does not include NUL */
}

#endif /* !HAVE_STRLCPY */

#ifndef HAVE_STRLCAT
/*
 * Appends src to string dst of size siz (unlike strncat, siz is the
 * full size of dst, not space left).  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz <= strlen(dst)).
 * Returns strlen(initial dst) + strlen(src); if retval >= siz,
 * truncation occurred.
 */
size_t strlcat(dst, src, siz)
	char *dst;
	const char *src;
	size_t siz;
{
	register char *d = dst;
	register const char *s = src;
	register size_t n = siz;
	size_t dlen;

	/* Find the end of dst and adjust bytes left but don't go past end */
	while (n-- != 0 && *d != '\0')
		d++;
	dlen = d - dst;
	n = siz - dlen;

	if (n == 0)
		return(dlen + strlen(s));
	while (*s != '\0') {
		if (n != 1) {
			*d++ = *s;
			n--;
		}
		s++;
	}
	*d = '\0';

	return(dlen + (s - src));	/* count does not include NUL */
}

#endif 

unsigned char lower_case[] = "abcdefghijklmnopqrstuvwxyzţ";
unsigned char upper_case[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

int toupper1 (int c) 
{
    unsigned char * p = strchr(lower_case, c);
    return p ? upper_case[p - lower_case] : c;
}

int tolower1 (int c) 
{
    unsigned char * p = strchr(upper_case, c);
    return p ? lower_case[p - upper_case] : c;
}

int isupper1 (int c) 
{
    return strchr(upper_case, c) == NULL ? FALSE : TRUE;
}

int islower1 (int c) 
{
    return strchr(lower_case, c) == NULL ? FALSE : TRUE;
}

