/* $Id: wanderers.c,v 1.666 2004/09/20 10:49:54 shrike Exp $                         */
 
/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "merc.h"
#include "fight.h"
#include "wanderers.h"
#include "conquer.h"

// some external functions
extern DECLARE_DO_FUN(do_yell      );
extern int sn_lookup (const char *name);
extern void spec_cast(CHAR_DATA *ch, const char *spell_name, CHAR_DATA *victim);
extern __inline int pk_range (int level);
extern bool check_trust(CHAR_DATA *ch, CHAR_DATA *victim);

bool does_not_affect_spell_level (CHAR_DATA * mob);
bool does_not_affect_exp (CHAR_DATA * mob);

// local functions
int mana_cost_for_mob ( int sn, CHAR_DATA * mob );
static bool mob_cant_cast (CHAR_DATA * ch, CHAR_DATA * victim, MOB_SPELL * spell);
static void select_extra_ability (CHAR_DATA *mob, MOB_EXTRA table[], bool silent);
static int get_max_extra_attacks (CHAR_DATA * mob);
static int calc_extra_tnl (CHAR_DATA * mob);
static MOB_DESC * mob_desc_lookup (CHAR_DATA * mob);

// mob extra tables
// we should create MORE tables :) spec_function <-> extra_ability_table?
// created_golem_extra will be default table
MOB_EXTRA created_golem_extra[] = 
{
    { "Auto rescue master",          EA_AUTO_RESC,         10,  80 },
    { "Auto find master",            EA_AUTO_FIND,         8,  80 },
    { "Curative spells on self",     EA_CURE_SELF,         20,  80 },
    //{ "Curative spells on master",   EA_CURE_MASTER,       5,  80 },
    { "Healing spells on self",      EA_HEAL_SELF,         7,  80 },
    //{ "Healing spells on master",    EA_HEAL_MASTER,       5,  80 },
    { "Protective spells on self",   EA_PROTECT_SELF,      7,  80 },
    //{ "Protective spells on master", EA_PROTECT_MASTER,    5,  80 },
    { "Fast regeneration",           EA_FAST_REGEN,        3,  80 },
    { "Flash regeneration",          EA_FLASH_REGEN,       5,  80 },
    { "Mana link",                   EA_MANA_LINK,         5,  80 },
    { "Immunity to vorpal",          EA_IMMUN_VORPAL,      5,  80 },
    { "Gates with master",           EA_GATE_WITH,         15, 80 },
    { "Doesn't consume experience",  EA_NO_EXP_CONSUME,    12, 80 },
    { "Doesn't consume spell level", EA_NO_SLEVEL_CONSUME, 20, 70 },
    //{ "Guards master",               EA_BODYGUARD,         10, 80 },
    { "Can blink attacks",           EA_BLINK,             10, 70 },
    { "Immunity to instant death",   EA_IMMUN_INSTANT,     25, 80 },
    { "Sneak",                       EA_SNEAK,             1,  90 },
    { NULL }
};

// mob extra_tnl calculation
#define EXTRA_ATTACK_TNL     10

MOB_EXTRA_TNL extra_tnl[] = 
{
//    name (for show feature)        flag                   where          extra_tnl
    { "Auto rescue master",          EA_AUTO_RESC,          IN_ABILITIES,  10 },
    { "Auto find master",            EA_AUTO_FIND,          IN_ABILITIES,  10 },
    { "Healing spells on self",      EA_HEAL_SELF,          IN_ABILITIES,  10 },
    { "Protective spells on self",   EA_PROTECT_SELF,       IN_ABILITIES,  10 },
    { "Fast regeneration",           EA_FAST_REGEN,         IN_ABILITIES,  10 },
    { "Flash regeneration",          EA_FLASH_REGEN,        IN_ABILITIES,  10 },
    { "Mana link",                   EA_MANA_LINK,          IN_ABILITIES,  10 },
    { "Immunity to vorpal",          EA_IMMUN_VORPAL,       IN_ABILITIES,  10 },
    { "Gates with master",           EA_GATE_WITH,          IN_ABILITIES,  10 },
    { "Doesn't consume experience",  EA_NO_EXP_CONSUME,     IN_ABILITIES,  10 },
    { "Doesn't consume spell level", EA_NO_SLEVEL_CONSUME,  IN_ABILITIES,  10 },
    { "Guards master",               EA_BODYGUARD,          IN_ABILITIES,  10 },
    { "Can blink attacks",           EA_BLINK,              IN_ABILITIES,  10 },
    { "Sneak",                       EA_SNEAK,              IN_ABILITIES,  10 },
    { "Immunity to instant death",   EA_IMMUN_INSTANT,      IN_ABILITIES,  10 },
    { "Act: warroir",                ACT_WARRIOR,           IN_ACT,        10 },
    { "Act: thief",                  ACT_THIEF,             IN_ACT,        10 },
    { "Off: berserk",                OFF_BERSERK,           IN_OFF,        10 },
    { "Off: bash",                   OFF_BASH,              IN_OFF,        10 },
    { "Off: trip",                   OFF_TRIP,              IN_OFF,        10 },
    { "Off: dirt kicking",           OFF_DIRT_KICK,         IN_OFF,        10 },
    { "Off: kick",                   OFF_KICK,              IN_OFF,        10 },
    { "Off: crush",                  OFF_CRUSH,             IN_OFF,        10 },
    { NULL }
};

//---------------------------------------------------------------------------------------
// spell tables
//---------------------------------------------------------------------------------------    
// mages and neromancers are the same
static MOB_SPELL mob_spells_protective_mage[] = 
{
    { "sanctuary",         spell_sanctuary,         30, 0, MS_IF_NOT_AFFECTED | MS_GOOD_NEUTRAL, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "black shroud",      spell_black_shroud,      30, 0, MS_IF_NOT_AFFECTED | MS_EVIL_ONLY, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "protective shield", spell_protective_shield, 30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, 0 },
    { "fly",               spell_fly,               30, 0, MS_IF_NOT_AFFECTED, AFF_FLYING },
    { "haste",             spell_haste,             35, 0, MS_IF_NOT_AFFECTED, AFF_HASTE },
    { "shield",            spell_shield,            20, 0, MS_IF_NOT_AFFECTED, 0 },
    { "armor",             spell_armor,             5,  0, MS_IF_NOT_AFFECTED, 0 },
    { "protection good",   spell_protection_good,   30, 0, MS_IF_NOT_AFFECTED | MS_EVIL_ONLY | MS_TAR_SELF, AFF_PROTECT_EVIL | AFF_PROTECT_GOOD },
    { "protection evil",   spell_protection_evil,   30, 0, MS_IF_NOT_AFFECTED | MS_GOOD_NEUTRAL | MS_TAR_SELF, AFF_PROTECT_EVIL | AFF_PROTECT_GOOD },
    { "pass door",         spell_pass_door,         30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, AFF_PASS_DOOR },
    { "detect invis",      spell_detect_invis,      30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, AFF_DETECT_INVIS },
    { "improved detect",   spell_improved_detect,   50, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, AFF_DETECT_IMP_INVIS},
    { "magic resistance",  spell_magic_resistance,  80, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, 0},
    { "protection heat",   spell_protection_heat,   50, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, 0},
    { NULL }
};

static MOB_SPELL mob_spells_maladictions[] = 
{
    { "slow",              spell_slow,              30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "curse",             spell_curse,             30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "web",               spell_web,               30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "weaken",            spell_weaken,            30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "blindness",         spell_blindness,         30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "poison",            spell_poison,            30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "plague",            spell_plague,            30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "deafen",            spell_deafen,            30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "faerie fire",       spell_faerie_fire,       30, 0, MS_IF_NOT_AFFECTED, 0 },
    { "corruption",        spell_corruption,        30, 0, MS_IF_NOT_AFFECTED, 0 },
    { NULL }
};

static MOB_SPELL mob_spells_attack[] = 
{
    { "acid blast",        spell_acid_blast,        30, 0, 0, 0 },
    { "chill touch",       spell_chill_touch,       30, 0, 0, 0 },
    { "energy drain",      spell_energy_drain,      30, 0, 0, 0 },
    { "colour spray",      spell_colour_spray,      30, 0, 0, 0 },
    { "hand of undead",    spell_hand_of_undead,    30, 0, 0, 0 },
    { NULL }
};

static MOB_SPELL mob_spells_room_maladictions[] = 
{
    { "black death",       spell_black_death,       30, 0, MS_IF_NOT_AFFECTED | MS_TAR_ROOM | MS_EVIL_ONLY },
    { "lethargic mist",    spell_lethargic_mist,    30, 0, MS_IF_NOT_AFFECTED | MS_TAR_ROOM | MS_EVIL_ONLY },
    { "mysterious dream",  spell_mysterious_dream,  30, 0, MS_IF_NOT_AFFECTED | MS_TAR_ROOM | MS_EVIL_ONLY },
    { "deadly venom",      spell_deadly_venom,      30, 0, MS_IF_NOT_AFFECTED | MS_TAR_ROOM | MS_EVIL_ONLY },
    { "cursed lands",      spell_cursed_lands,      30, 0, MS_IF_NOT_AFFECTED | MS_TAR_ROOM | MS_EVIL_ONLY },
    { NULL }
};

static MOB_SPELL mob_spells_curative[] = 
{
    { "cure blindness",    spell_cure_blindness,    20, 0, MS_IF_AFFECTED, AFF_BLIND },
    { "cure poison",       spell_cure_poison,       25, 0, MS_IF_AFFECTED, AFF_POISON },
    { "cure disease",      spell_cure_disease,      30, 0, MS_IF_AFFECTED, AFF_PLAGUE },
    { NULL }
};

static MOB_SPELL mob_spells_protective_golem[] = 
{
    { "sanctuary",         spell_sanctuary,         30, 0, MS_IF_NOT_AFFECTED | MS_GOOD_NEUTRAL, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "black shroud",      spell_black_shroud,      30, 0, MS_IF_NOT_AFFECTED | MS_EVIL_ONLY, AFF_PROTECTION | AFF_SANCTUARY | AFF_BLACK_SHROUD },
    { "protective shield", spell_protective_shield, 30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, 0 },
    { "fly",               spell_fly,               30, 0, MS_IF_NOT_AFFECTED, AFF_FLYING },
    { "haste",             spell_haste,             35, 0, MS_IF_NOT_AFFECTED, AFF_HASTE },
    { "shield",            spell_shield,            20, 0, MS_IF_NOT_AFFECTED, 0 },
    { "armor",             spell_armor,             5,  0, MS_IF_NOT_AFFECTED, 0 },
    { "protection good",   spell_protection_good,   30, 0, MS_IF_NOT_AFFECTED | MS_EVIL_ONLY | MS_TAR_SELF, AFF_PROTECT_EVIL | AFF_PROTECT_GOOD },
    { "protection evil",   spell_protection_evil,   30, 0, MS_IF_NOT_AFFECTED | MS_GOOD_NEUTRAL | MS_TAR_SELF, AFF_PROTECT_EVIL | AFF_PROTECT_GOOD },
    { "pass door",         spell_pass_door,         30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, AFF_PASS_DOOR },
    { "detect invis",      spell_detect_invis,      30, 0, MS_IF_NOT_AFFECTED | MS_TAR_SELF, AFF_DETECT_INVIS },
    { NULL }
};

static MOB_SPELL mob_spells_healing[] = 
{
    { "cure light",        spell_cure_light,        4,  0, 0, 0 },
    { "cure serious",      spell_cure_serious,      8,  0, 0, 0 },
    { "cure critical",     spell_cure_critical,     18, 0, 0, 0 },
    { "heal",              spell_heal,              35, 0, 0, 0 },
    { "superior heal",     spell_superior_heal,     50, 0, 0, 0 },
    { "master healing",    spell_master_healing,    65, 0, 0, 0 },
    { NULL }
};
//-------------------------------------------------------------------------------------
// mob descs
// summoned shadow
#define VNUM_SHAD_10 15021 // Wandering Spectre
#define VNUM_SHAD_20 15022 // Horrible Apparition
#define VNUM_SHAD_35 15023 // Agonal Terror
#define VNUM_SHAD_50 15024 // Sudden /+\ Death

// devil
#define VNUM_DEV_10  15006 // SoulReaver
#define VNUM_DEV_20  15007 // Demon Overseer
#define VNUM_DEV_35  15008 // Hell Overlord
#define VNUM_DEV_50  15009 // ^=Arch-devil=^

// angel
#define VNUM_ANG_10  15013 // Solaris
#define VNUM_ANG_20  15014 // LightBringer
#define VNUM_ANG_35  15015 // Redeemer
#define VNUM_ANG_50  15016 // -+ ArchAngel +-

// mechanical golem
#define VNUM_MECH_10 15017 // Mechanical Spider
#define VNUM_MECH_20 15018 // Mechanical Gryphon
#define VNUM_MECH_35 15019 // Mechanical Hydra
#define VNUM_MECH_50 15020 // >Mechanical< >Dragon<

// lesser golem
#define VNUM_LESS_10 15025 // Clay Statue
#define VNUM_LESS_20 15026 // Marble Sphinx
#define VNUM_LESS_35 15027 // Granite Goliath
#define VNUM_LESS_50 15028 // _|ThE CoIosSuS|_

#define VNUM_SQ_10   15034 // Militia
#define VNUM_SQ_20   15035 // Heroes of baron
#define VNUM_SQ_35   15036 // Army of the Seven Kingdoms
#define VNUM_SQ_50   15037 // _\ Godly Wrath /_ 

#define VNUM_ANIM_10 15029 // Inductor
#define VNUM_ANIM_20 15031 // Pain Deflector 
#define VNUM_ANIM_35 15032 // osmo  
#define VNUM_ANIM_50 15033 // \*/ Energizer \*

static MOB_DESC supermobs[] = {
//    vnum                     vnum_10       vnum_20       vnum_35       vnum_50        tnl
    { MOB_VNUM_CREATED_GOLEM,  VNUM_MECH_10, VNUM_MECH_20, VNUM_MECH_35, VNUM_MECH_50,  300 },
    { MOB_VNUM_ANGEL,          VNUM_ANG_10,  VNUM_ANG_20,  VNUM_ANG_35,  VNUM_ANG_50,   100 },         
    { MOB_VNUM_DEVIL,          VNUM_DEV_10,  VNUM_DEV_20,  VNUM_DEV_35,  VNUM_DEV_50,   100 },          
    { MOB_VNUM_SQUIRE,         VNUM_SQ_10,   VNUM_SQ_20,   VNUM_SQ_35,   VNUM_SQ_50,    200 },
    { MOB_VNUM_SUM_SHADOW,     VNUM_SHAD_10, VNUM_SHAD_20, VNUM_SHAD_35, VNUM_SHAD_50,  0   },
    { MOB_VNUM_LESSER_GOLEM,   VNUM_LESS_10, VNUM_LESS_20, VNUM_LESS_35, VNUM_LESS_50,  200 },
    { MOB_VNUM_ANIMATED_OTHER, VNUM_ANIM_10, VNUM_ANIM_20, VNUM_ANIM_35, VNUM_ANIM_50,  200 },
    { 0 }
};
//-------------------------------------------------------------------------------------

SPEC_FUN(spec_wandering_necr)
{
    CHAR_DATA * vch, * vch_next;
    bool in_pk;
    
    // fair mob! :)
    if (ch->wait)
        return TRUE;

    // two aspects: in the battle and when wandering

    // wandering
    if (!ch->fighting)
    {
        // a chance to cast good spell upon self
        if (number_range (1, 100) > 10)
        {
            if (wanderer_spec_cast (ch, ch, mob_spells_protective_mage, SO_FIRST, ch->level+ch->level/20, TARGET_CHAR))
                return TRUE;
        }

        // hunt enemy!
        if (number_range (1,100) > 10 && !IS_AFFECTED (ch, AFF_CHARM))
        {
            if (ch->hunting && ch->hunting->in_room)
            {
                int dir; 

                dir = find_path(ch->in_room->vnum, ch->hunting->in_room->vnum,
                                ch, -40000, FALSE);
                
                if (dir >= 0 && dir < MAX_DIR)
                {
                    act ("$n stares intently at the ground.", ch, NULL, NULL, TO_ROOM);
                    move_char(ch, dir, FALSE);
                    return TRUE;
                }
            }
        }

        // attack enemy
        if (number_range (1,100) > 30)
        {
            for (vch = ch->in_room->people; vch; vch = vch_next) 
            {
                vch_next = vch->next_in_room;

                if (!IS_NPC (vch) &&
                     ((ch->level >= (vch->level + pk_range (ch->level))) ||
                               (ch->level <= (vch->level - pk_range (ch->level))))
                            &&
                              ((vch->level >= (ch->level + pk_range (vch->level))) ||
                               (vch->level <= (ch->level - pk_range (vch->level)))) )
                    in_pk = FALSE;
                else
                    in_pk = TRUE;

                if (!IS_IMMORTAL (vch) && vch != ch && 
                    ((IS_NPC (vch) && !IS_SET(vch->pIndexData->act, ACT_WANDERER)) || !IS_NPC(vch))
                    && !is_same_group (ch, vch) && !IS_SET (vch->plr_flags, PLR_GHOST)
                    && (number_range (1,100) > 10) && in_pk)
                {
                    spec_cast (ch, "power word kill", vch);
                    return TRUE;
                }
            }
        }

        // move char
        if (number_range (1, 100) > 70)
        {
            int        attempt, door;
            EXIT_DATA * pexit;

            for (attempt = 0; attempt < 6; ++attempt)
            {
                door = number_door ();
                if (((pexit = ch->in_room->exit[door]) != NULL)
                    && (pexit->to_room.r != NULL)
//                    &&  !IS_SET(pexit->exit_info, EX_CLOSED)
                    &&  !IS_SET(pexit->to_room.r->room_flags, ROOM_NOMOB)
                    && (ch->position >= POS_STANDING) && !IS_AFFECTED(ch, AFF_CHARM))
                {
                    move_char(ch, door, FALSE);
                    return TRUE;
                }
            }
        }

        // room maladiction - one chance from 500
        if (number_range (1, 500) > 499)
        {
            if (!IS_SET(ch->in_room->room_flags, ROOM_LAW))
                if (wanderer_spec_cast (ch, ch, mob_spells_room_maladictions, SO_RANDOM, ch->level, TARGET_ROOM))
                    return TRUE;
        }
        return TRUE;
    }

    // battle
    else
    {
        // don't forget first victim
        if (!ch->hunting)
            ch->hunting = ch->fighting;

        if (number_range (0,1) == 0)
            wanderer_spec_cast (ch, ch->fighting, mob_spells_maladictions, SO_FIRST, ch->level+ch->level/10, TARGET_CHAR);
        else
            wanderer_spec_cast (ch, ch->fighting, mob_spells_attack, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR);
        return TRUE;
    }
    return TRUE;
}

int mana_cost_for_mob ( int sn, CHAR_DATA * mob )
{
    skill_t *sk;
    int mana;

    if ((sk = skill_lookup(sn)) == NULL)
        return 0;

    if (!mob || !IS_NPC (mob))
        return 0;

    if((IS_SKILL(sn) || IS_AUTOSKILL(sn)) && (sk->min_mana == 0))
        return 0;
    else
    {
        mana = sk->min_mana;
        if (IS_SET (mob->pIndexData->act, ACT_WARRIOR))
            mana *=4;
        else if (IS_SET (mob->pIndexData->act, ACT_THIEF))
            mana *=3;
        return mana;
    }
}

bool wanderer_spec_cast (CHAR_DATA *ch, CHAR_DATA * victim, MOB_SPELL spell_table[], 
                         int option, int level, int target)
{
    MOB_SPELL * curr_spell;
    int         mana;
    int         attempts, count = 0, num, i;

    if (!IS_NPC (ch))
         return FALSE;
    
    if (IS_SET (option, SO_RANDOM))
    {
        for (curr_spell = spell_table; curr_spell->name != NULL; curr_spell++, count++);

        for (attempts = 0; attempts < 20; attempts++)
        {
            // select random spell
            num = number_range (0, count - 1);
            for (curr_spell = spell_table, i = 0; i < num; curr_spell++, i++);

            // spell selected
            curr_spell->sn = sn_lookup (curr_spell->name); 
            mana = IS_SET(option, SO_NO_MANA) ? 0 : mana_cost_for_mob (curr_spell->sn, ch);
            
            if (!curr_spell->sn || mob_cant_cast (ch, victim, curr_spell) || ch->mana < mana)
            {
                curr_spell = NULL;
                continue; // next attempt
            }

            if (curr_spell) 
                ch->mana -= mana;
            break;
        }
    }
    else
    {
        for (curr_spell = spell_table; curr_spell->name != NULL; curr_spell++)
        {
            curr_spell->sn = sn_lookup (curr_spell->name);
            if (curr_spell->sn) 
            {
                if (mob_cant_cast (ch, victim, curr_spell))
                    continue; 

                mana = IS_SET(option, SO_NO_MANA) ? 0 : mana_cost_for_mob (curr_spell->sn, ch);
                if (ch->mana < mana)
                {
                    if (IS_SET (option, SO_SUITABLE))
                        continue;
                    if (IS_SET (option, SO_FIRST))
                        return FALSE;
                }
                ch->mana -= mana;
                break;
            }
        }
    }

    if (curr_spell && curr_spell->name != NULL)
    {
        // choose level
        //spec_cast (ch, curr_spell->name, victim);
        curr_spell->spell_fn(curr_spell->sn, level, ch, victim, target);
        // wait state
        if (IS_AFFECTED(ch, AFF_HASTE))
        {
            WAIT_STATE(ch, SKILL(curr_spell->sn)->beats / 2);
        } 
        else if (IS_AFFECTED(ch, AFF_SLOW))
        {
            WAIT_STATE(ch, SKILL(curr_spell->sn)->beats * 2);
        } 
        else
            WAIT_STATE(ch, SKILL(curr_spell->sn)->beats);

        return TRUE;
    }
    else
        return FALSE;
}

bool mob_cant_cast (CHAR_DATA * ch, CHAR_DATA * victim, MOB_SPELL * spell)
{
    // wrong position
    if (ch->position < SKILL(spell->sn)->min_pos)
        return TRUE;

    // skip if unsufficient level
    if (ch->level < spell->min_level)
        return TRUE;

    // skip if already affected and appropriate flag is set
    if (IS_SET(spell->extra, MS_IF_NOT_AFFECTED))
    {
        if (IS_SET (spell->extra, MS_TAR_ROOM))
        {
            if (ch->in_room && is_affected_room(ch->in_room, spell->sn))
                return TRUE;
        }
        else
        {
            if (is_affected(victim, spell->sn) || IS_AFFECTED (victim, spell->aff))
                return TRUE;
        }
    }

    if (IS_SET(spell->extra, MS_IF_AFFECTED))
    {
        if (is_affected(victim, spell->sn) || IS_AFFECTED (victim, spell->aff))
            return TRUE;
    }

    // skip if victim does not trust ch enough
    if (IS_SET(SKILL(spell->sn)->flags, SKILL_QUESTIONABLE)
           &&  !check_trust(ch, victim)) 
        return TRUE;

    // skip due to align restrictions
    if (IS_SET(spell->extra, MS_EVIL_ONLY) && !(IS_EVIL (victim) && IS_EVIL (ch)))
        return TRUE;
    if (IS_SET(spell->extra, MS_GOOD_ONLY) && !(IS_GOOD (victim) && IS_GOOD (ch)))
        return TRUE;
    if (IS_SET(spell->extra, MS_GOOD_NEUTRAL) && (IS_EVIL (victim) || IS_EVIL (ch)))
        return TRUE;
    if (IS_SET(spell->extra, MS_EVIL_NEUTRAL) && (IS_GOOD (victim) || IS_GOOD (ch)))
        return TRUE;

    // skip target_self spells on other players
    if (IS_SET(spell->extra, MS_TAR_SELF) && (ch != victim))
        return TRUE;

    // all is OK
    return FALSE;
}

// gain exp stuff - will be suitable for tough charmies :)
void upgrade_npc (CHAR_DATA * mob)
{
    MOB_INDEX_DATA * new_mob_index = NULL;
    MOB_DESC       * desc;
    
    if (!mob || !IS_NPC (mob))
        return;

    // special case - wanderers
    if (IS_SET(mob->pIndexData->act, ACT_WANDERER))
    {
    }
    // charmies
    else
    {
        if (mob->master && !IS_NPC(mob->master) 
            //&& (IS_SET(mob->master->comm, PLR_PEACE) || IS_IMMORTAL (mob->master)))
            && (mob->master->level >= LEVEL_HERO - 10))
        {
            ++mob->mob_rank;
            mob->mob_tnl = mob->level * 15 + 500 * mob->mob_rank + calc_extra_tnl (mob);
            //mob->mob_tnl = 0;
            char_printf (mob->master, "%s has now rank %d!\n", 
                fix_short(mlstr_cval (mob->short_descr, mob->master)), 
                mob->mob_rank);
            
            // now we have to "redraw" mob if an appropriate rank is reached
            desc = mob_desc_lookup (mob);
            if (desc != NULL)
            {
                // NB. if desc->vnum_xx == 0, get_mob_index will return NULL
                if (mob->mob_rank == 10)
                    new_mob_index = get_mob_index (desc->vnum_10);
                else if (mob->mob_rank == 20)
                    new_mob_index = get_mob_index (desc->vnum_20);
                else if (mob->mob_rank == 35)
                    new_mob_index = get_mob_index (desc->vnum_35);
                else if (mob->mob_rank == 50)
                    new_mob_index = get_mob_index (desc->vnum_50);
                else 
                    new_mob_index = NULL;

                if (new_mob_index != NULL) // yes, we've found it :)
                {
                    mlstr_free (mob->short_descr);
                    mob->short_descr = mlstr_dup(new_mob_index->short_descr);

                    mlstr_free (mob->long_descr);
                    mob->long_descr = mlstr_dup(new_mob_index->long_descr);

                    mlstr_free (mob->description);
                    mob->description = mlstr_dup(new_mob_index->description);

                    mob->sex = new_mob_index->sex;
                    mob->alignment = new_mob_index->alignment;

                    free_string (mob->name);
                    mob->name = str_dup (new_mob_index->name);

                    mob->race = new_mob_index->race;
                    mob->form = mob->form | new_mob_index->form;

                    char_act ("\nNow your pet has NEW image.\n", mob->master);
                }
            }

            // --------------------------------------------------------------------------
            // mandatory part:

            // + level each 5 ranks
            if (mob->mob_rank % 5 == 0 
                && (mob->level < (mob->master->level + mob->master->level / 12)))
            {
                char_printf (mob->master, "%s received a level!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master)));
                mob->level++;
            }

            if (mob->mob_rank < 11)
            {
                mob->max_hit += mob->level * 4 + mob->mob_rank * 10;
                mob->max_mana += mob->level * 2 + mob->mob_rank * 2;
                mob->damroll += mob->level/10 + number_range (0,4);
                mob->hitroll += mob->level/10 + number_range (0,4);
            }
            else if (mob->mob_rank < 17)
            {
                mob->max_hit += mob->level * 4 + mob->mob_rank * 10;
                mob->max_mana += mob->level * 2 + mob->mob_rank * 2;
                mob->damroll += mob->level/15 + number_range (0,2);
                mob->hitroll += mob->level/15 + number_range (0,2);
            } 
            else if (mob->mob_rank < 36)
            {
                mob->max_hit += mob->level * 4 + mob->mob_rank * 10;
                mob->max_mana += mob->level * 2 + mob->mob_rank * 2;
                mob->damroll += mob->level/20 + number_range (0,3);
                mob->hitroll += mob->level/20 + number_range (0,3);
            }
            // svs. otherwise high-level mobs can be easily killed by pwk etc
            if (mob->saving_throw > -140)
                mob->saving_throw -= 5;

            //---------------------------------------------------------------------------
            // random part
            // todo : may be dex/str/int/wis/lck?
            // 60% to gain an extra ability
            if (number_percent () > 40)
            {
                // so far only one table. todo: more tables
                select_extra_ability (mob, created_golem_extra, FALSE);

                // sneak
                if (IS_SET (mob->abilities, EA_SNEAK))
                    SET_BIT (mob->affected_by, AFF_SNEAK);

                if (number_percent() > 60) // 40% to gain another advantage
                    return;
            }

            // 10% to gain level
            if (number_percent () > 90 && 
                (mob->level < (mob->master->level + mob->master->level / 12)))
            {
                ++mob->level;
                char_printf (mob->master, "%s received a level!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master)));
                if (number_percent() > 80) // 20% to gain another advantage
                    return;
            }
            // 40% to gain hp
            if (number_percent () > 60)
            {
                mob->max_hit += mob->level * 2 + mob->mob_rank * 10;
                char_printf (mob->master, "%s became more healthy!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master))); 
                if (number_percent() > 60) // 40% to gain another advantage
                    return;
            }
            // 30% to gain damroll
            if (number_percent () > 70)
            {
                mob->damroll += mob->level/15 + number_range (0,4);
                char_printf (mob->master, "%s became more dangerous!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master))); 
                if (number_percent() > 60) // 40% to gain another advantage
                    return;
            }
            // 40% to gain hitroll
            if (number_percent () > 60)
            {
                mob->hitroll += mob->level/15 + number_range (0,4);
                char_printf (mob->master, "%s became more neat!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master))); 
                if (number_percent() > 60) // 40% to gain another advantage
                    return;
                return;
            }
            // 50% to gain mana
            if (number_percent () > 50)
            {
                mob->max_mana += mob->level * 2 + mob->mob_rank * 2;
                char_printf (mob->master, "%s now has more mana!\n", 
                    fix_short(mlstr_cval (mob->short_descr, mob->master))); 
                if (number_percent() > 50) // 50% to gain another advantage
                    return;
            }
        }
    }
}

// created golems
SPEC_FUN(spec_created_golem)
{
    int i, wait = 0; 

    if (ch->wait > 0)
        return TRUE;

    if (!ch->master || !IS_AFFECTED (ch, AFF_CHARM))
        return TRUE;

    if (ch->in_room == ch->master->in_room)
        REMOVE_BIT (ch->actions, CA_AUTO_FIND);

    if (IS_SET(ch->abilities, EA_HEAL_SELF))
    {
        if (ch->hit < ch->max_hit/2) 
            SET_BIT (ch->actions, CA_HEAL_SELF);
        else if (ch->hit > ch->max_hit * 8 / 10) 
            REMOVE_BIT (ch->actions, CA_HEAL_SELF);
    }

    // as usual two aspects: in the battle and when wandering

    // not fighting
    if (!ch->fighting)
    {
        if (ch->master->in_room 
            && IS_SET (ch->actions, CA_AUTO_FIND) 
            && ch->position == POS_STANDING)
        {
            int dir; 

            dir = find_path(ch->in_room->vnum, ch->master->in_room->vnum,
                            ch, -40000, FALSE);
            if (dir >= 0 && dir < MAX_DIR)
            {
                act ("$n stares intently at the ground.", ch, NULL, NULL, TO_ROOM);
                move_char(ch, dir, FALSE);
                WAIT_STATE (ch, PULSE_PER_SECOND); // one second wait
                return TRUE;
            }
            else
            {
                do_gtell (ch, GETMSG ("Master, I can't find a way to you!", ch->master->lang));
                REMOVE_BIT (ch->actions, CA_AUTO_FIND);
            }
        }
        
        if (IS_SET(ch->abilities, EA_PROTECT_SELF) && number_percent() < 80)
            if (wanderer_spec_cast (ch, ch, mob_spells_protective_golem, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR))
                return TRUE;

        if (IS_SET(ch->actions, CA_HEAL_SELF) && number_percent() < 30)
            if (wanderer_spec_cast (ch, ch, mob_spells_healing, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR))
                return TRUE;

        if (IS_SET(ch->abilities, EA_CURE_SELF) && number_percent() < 30)
            if (wanderer_spec_cast (ch, ch, mob_spells_curative, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR))
                return TRUE;
    }
    // fighting
    else
    {
        if (ch->in_room != ch->master->in_room)
            do_flee (ch, str_empty);

        for (i = 0; i < ch->extra_attacks; ++i)
        {
            if (number_percent() < 50)
            {
                one_hit (ch, ch->fighting, TYPE_UNDEFINED, WEAR_WIELD);
                wait = PULSE_VIOLENCE;
            }
        }

        if (IS_SET(ch->abilities, EA_AUTO_RESC) 
            && IS_SET(ch->actions, CA_CAN_RESCUE) 
            && number_percent() < 30)
        {
            do_rescue (ch, ch->master->name);
        }

        // autoheal feature - mob starts to heal when he has < 50% hp
        // and ends when he has > 80%
        if (IS_SET(ch->actions, CA_HEAL_SELF) && number_percent() < 20)
            if (wanderer_spec_cast (ch, ch, mob_spells_healing, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR))
                return TRUE;

        if (IS_SET(ch->abilities, EA_CURE_SELF) && number_percent() < 30)
            if (wanderer_spec_cast (ch, ch, mob_spells_curative, SO_RANDOM, ch->level+ch->level/10, TARGET_CHAR))
                return TRUE;

        if (!ch->wait)
            WAIT_STATE (ch, wait);
    }
    return TRUE;
}

static void select_extra_ability (CHAR_DATA *mob, MOB_EXTRA table[], bool silent)
{
    MOB_EXTRA * curr;
    int count, attempt, num;

    // chance to receive extra attack
    if (number_percent() > 65 && mob->extra_attacks < get_max_extra_attacks(mob)) 
    {
        ++mob->extra_attacks;
        char_printf (mob->master, "%s now has %d extra attacks!\n",
            fix_short(mlstr_cval (mob->short_descr, mob->master)), mob->extra_attacks);
        return;
    }
    // calculate abilities count in the table
    for (curr = table, count = 0; curr->name != NULL; curr++, count++);

    for (attempt = 0; attempt < 32; ++attempt)
    {
        num = number_range (0, count - 1);

        if (!IS_SET (mob->abilities, table[num].flag) 
            && (mob->mob_rank >= table[num].min_rank)
            && (number_percent() <= table[num].percent))
        {
            // special cases
            if ((table[num].flag == EA_SNEAK) && IS_AFFECTED (mob, AFF_SNEAK))
                continue;
            if ((table[num].flag == EA_NO_EXP_CONSUME) && does_not_affect_exp (mob))
                continue;
            if ((table[num].flag == EA_NO_SLEVEL_CONSUME) && does_not_affect_spell_level(mob))
                continue;

            SET_BIT (mob->abilities, table[num].flag);
            if (!silent && mob->master)
                char_printf (mob->master, "%s gained an extra ability: %s!\n",
                    fix_short(mlstr_cval (mob->short_descr, mob->master)), 
                    GETMSG (table[num].name, mob->master->lang));
            return;
        }
    }

    // no abilities choosen -> extra attack
    if (mob->extra_attacks < get_max_extra_attacks(mob)) 
    {
        ++mob->extra_attacks;
        char_printf (mob->master, "%s now has %d extra attacks!\n",
            fix_short(mlstr_cval (mob->short_descr, mob->master)), mob->extra_attacks);
    }
    return;
}

// attacks: 
#define MAX_EXTRA_ATTACKS_WARRIOR 2
#define MAX_EXTRA_ATTACKS_THIEF   3
#define MAX_EXTRA_ATTACKS_OTHER   4

static int get_max_extra_attacks (CHAR_DATA * mob)
{
    if (!IS_NPC (mob))
        return 0;

    if (IS_SET (mob->pIndexData->act, ACT_WARRIOR))
        return MAX_EXTRA_ATTACKS_WARRIOR;

    if (IS_SET (mob->pIndexData->act, ACT_THIEF))
        return MAX_EXTRA_ATTACKS_THIEF;

    return MAX_EXTRA_ATTACKS_OTHER;
}

int ea_modify_gain (CHAR_DATA * ch, int gain, int type)
{
    if (IS_SET (ch->abilities, EA_FLASH_REGEN)                 // has an ability
        && (ch->position == POS_SLEEPING                       // sleeping non-undead
           || (IS_UNDEAD (ch) && ch->position == POS_RESTING)) // or resting undead      
        && number_percent() < 10 )                             // and probability, of course
    {
        switch (type) 
        {
            case GAIN_HP:
                act ("Bright {ggreen{x aura surrounds $n for a moment.",ch,NULL,NULL,TO_ROOM);
                return (ch->max_hit - ch->hit);
            case GAIN_MANA:
                act ("Air around $n flickers and you feel a stream of magic.",ch,NULL,NULL,TO_ROOM);
                return (ch->max_mana - ch->mana);
            case GAIN_MOVES:
                return (ch->max_move - ch->move);
        }
    }

    // fast regeneration - increased rate (10 times)
    if (IS_SET (ch->abilities, EA_FAST_REGEN))
        return (gain * 10);

    return gain;
}

static int calc_extra_tnl (CHAR_DATA * mob)
{
    MOB_EXTRA_TNL * curr;
    int             tnl = 0;
    MOB_DESC      * mob_desc;

    if (!IS_NPC (mob))
        return 0;

    for (curr = extra_tnl; curr->name != NULL; curr++)
    {
        switch (curr->where)
        {
        case IN_ABILITIES:
            if (IS_SET (mob->abilities, curr->flag))
                tnl += curr->extra_tnl;
            break;
        case IN_OFF:
            if (IS_SET (mob->pIndexData->off_flags, curr->flag))
                tnl += curr->extra_tnl;
            break;
        case IN_ACT:
            if (IS_SET (mob->pIndexData->act, curr->flag))
                tnl += curr->extra_tnl;
            break;
        }
    }

    tnl += mob->extra_attacks * EXTRA_ATTACK_TNL;

    mob_desc = mob_desc_lookup (mob);
    if (mob_desc != NULL)
        tnl += mob_desc->extra_tnl;

    tnl *= mob->mob_rank;
    return tnl;
}

bool does_not_affect_exp (CHAR_DATA * mob)
{
    return (IS_NPC(mob)
            && (mob->pIndexData->vnum == MOB_VNUM_LESSER_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_STONE_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_IRON_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_TREANT
                || mob->pIndexData->vnum == MOB_VNUM_ADAMANTITE_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_DIAMOND_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_CREATED_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_FOREST_GIANT 
                || IS_SET (mob->abilities, EA_NO_EXP_CONSUME)));
}

bool does_not_affect_spell_level (CHAR_DATA * mob)
{
    return(IS_NPC(mob)
            && (mob->pIndexData->vnum == MOB_VNUM_LESSER_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_STONE_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_IRON_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_ADAMANTITE_GOLEM
                || mob->pIndexData->vnum == MOB_VNUM_DIAMOND_GOLEM
                || IS_SET (mob->abilities, EA_NO_SLEVEL_CONSUME)));
}

// count number of pets who can level-up (for ch)
int levelup_pet_count (CHAR_DATA * ch)
{
    CHAR_DATA * gch;
    int count = 0;

    for (gch = char_list; gch != NULL; gch = gch->next)
    {
        if (IS_NPC(gch) && IS_AFFECTED(gch,AFF_CHARM) && gch->master == ch 
        && (IS_SET (gch->pIndexData->act, ACT_RECEIVE_LEVEL)))
        {
            ++ count;
        }
    }
    return count;
}

// we try to find mob desc
// used to calculate tnl and "redraw" the mob when appropriate rank is reached
static MOB_DESC * mob_desc_lookup (CHAR_DATA * mob)
{
    MOB_DESC * desc;
    int        vnum;

    if (!IS_NPC (mob))
        return NULL;

    vnum = mob->pIndexData->vnum;

    for (desc = supermobs; desc->vnum != 0; desc++)
    {
        if ((desc->vnum == vnum) 
            || (desc->vnum_10 == vnum) 
            || (desc->vnum_20 == vnum)
            || (desc->vnum_35 == vnum)
            || (desc->vnum_50 == vnum))
            break;
    }

    if (desc->vnum == 0)
        return NULL;

    return desc;
}

// --------------------------------------------------------------------------------
// Adventures
// --------------------------------------------------------------------------------
DO_FUN (do_adventure)
{
}
