/* $Id: war.c,v 1.666 2004/09/20 10:49:54 shrike Exp $                               */ 
 
/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "merc.h"
#include "db/dofun.h"
#include "conquer.h"
#include "war.h"
#include "obj_prog.h"

extern AREA_DATA *area_first;

// some statistics
// stat_record.wars_started is updated in new_war_data
// stat_record.wars_completed in reward_winner
extern STAT_DATA stat_record; 

extern void stop_fighting(CHAR_DATA *ch, bool fBoth);
extern void save_area(AREA_DATA *pArea);
extern __inline int pk_range (int level); 
extern int sn_lookup (const char *name);

// rooms
#define WAR_PREPARATION_ROOM    15015 // room where fighters can prepare to war
#define ROOM_VNUM_CR            3502  // room in which defeated and victorious ones 
                                      // will be transferred (cross_roads)
#define VNUM_PRISON_CELL1       15001
#define VNUM_PRISON_CELL2       15002
#define VNUM_PRISON_CELL3       15004

// time
#define WAR_PREPARATION_TIME    2 // 4
#define WAR_SETPARM_TIME        1 // 2
#define WAR_AUTO_INTERVAL       30

#define DUEL_WAIT_TIME          8
#define DUEL_PREPARATION_TIME   2
#define DUEL_INTERVAL           12 // time interval between to duels for one char
#define DUEL_DURATION           number_range (15,20)

// flag definition for war->flags ( (!) both options and status flags)
// see also war.h
// war options flags
#define WO_USE_PETS             (A)
#define WO_USE_TRANSPORTATION   (B)
#define WO_USE_PILLS            (C)

#define WF_AUTOGENERATE         (A)

#define WAR_MAX_AREA_COUNT      120

bool war_can_charm (CHAR_DATA * ch, CHAR_DATA * victim);
void clear_bad_affects (CHAR_DATA * ch);

static ROOM_INDEX_DATA * room_CR;
static ROOM_INDEX_DATA * room_preparation;
static bool war_enabled;
static int time_to_next_war;

#define RND_TIME_BETWEEN_WARS   number_range (15,25)
#define RND_TIME_DURATION       number_range (20,40)

// check_same_team_fn - checks whether 2 players belong to the same team
// called from check_winner
// end_fn - called from reward_winner (reward distribution procedure can be 
// different for each type of war; and proper work with msgdb needs separate 
// procedures)
// special_end_fn called from check_winner. M.b. for clan wars it would be
// nice to keep some statictics etc.
// special_end_fn: message about war finish
// extra - additional flags
// can_start_fn - function for autostart stuff, checks whether there are enough 
// potential parpicipant in the world
struct war_service_data {
    char * type_str;
    bool (* check_same_team_fn) ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
    int  min_bp_reward;
    int  max_bp_reward;
    int  min_qp_reward;
    int  max_qp_reward;
    int  min_gold_reward;
    int  max_gold_reward;
    void (* end_fn ) ( CHAR_DATA * ch ); 
    void (* special_end_fn ) ( WAR_DATA * war );
    int  extra;
    bool (* can_start_fn ) ( int minlevel, int maxlevel ); 
};

typedef struct war_service_data WAR_SERVICE_DATA;

// check functions return 0 as "YES" and 1 as "NO"
// kill/death statistics is to be updated in war_handle_death function
static bool check_same_team_genocid  ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_clan     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_race     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_religion ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_class    ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_team     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );
static bool check_same_team_align    ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 );

static void end_genocid   ( CHAR_DATA * ch );
static void end_clan_war  ( CHAR_DATA * ch );
static void end_race_war  ( CHAR_DATA * ch );
static void end_crusade   ( CHAR_DATA * ch );
static void end_class_war ( CHAR_DATA * ch );
static void end_team_war  ( CHAR_DATA * ch );
static void end_align_war ( CHAR_DATA * ch );

static void sp_end_genocid ( WAR_DATA * war );
static void sp_end_clan_war ( WAR_DATA * war );
static void sp_end_race_war ( WAR_DATA * war );
static void sp_end_crusade ( WAR_DATA * war );
static void sp_end_class_war ( WAR_DATA * war );
static void sp_end_team_war ( WAR_DATA * war );
static void sp_end_align_war ( WAR_DATA * war );

static WAR_SERVICE_DATA war_desc[] = { 
//    type_str     check_same_team_fn              
    { "Genocid",   check_same_team_genocid,  0,    3,   40,   150, 50,   200, 
                   end_genocid,   sp_end_genocid,   WF_AUTOGENERATE, NULL },
    { "Clan war",  check_same_team_clan,     0,    2,   30,   90,  20,   150, 
                   end_clan_war,  sp_end_clan_war,  WF_AUTOGENERATE, NULL },
    { "Race war",  check_same_team_race,     0,    3,   40,   90,  40,   180, 
                   end_race_war,  sp_end_race_war,  WF_AUTOGENERATE, NULL },
    { "Crusade",   check_same_team_religion, 0,    2,   40,   90,  40,   180, 
                   end_crusade,   sp_end_crusade,   WF_AUTOGENERATE, NULL },
    { "Class war", check_same_team_class,    0,    2,   40,   90,  50, 180, 
                   end_class_war, sp_end_class_war, WF_AUTOGENERATE, NULL },
    { "Team war",  check_same_team_team,     0,    2,   40,   90,  50,   180, 
                   end_team_war,  sp_end_team_war,  0, NULL },
    { "Align war", check_same_team_align,    0,    2,   40,   90,  50,   180, 
                   end_align_war, sp_end_align_war, WF_AUTOGENERATE, NULL },
    { NULL }
};

// local functions for do_warfare command
static void war_help      ( CHAR_DATA *ch, const char *argument ); 
static void war_info      ( CHAR_DATA *ch, const char *argument ); 
static void war_start     ( CHAR_DATA *ch, const char *argument ); 
static void war_cancel    ( CHAR_DATA *ch, const char *argument ); 
static void war_join      ( CHAR_DATA *ch, const char *argument ); 
static void war_list      ( CHAR_DATA *ch, const char *argument ); 
static void war_surrender ( CHAR_DATA *ch, const char *argument ); 
static void war_enable    ( CHAR_DATA *ch, const char *argument ); 
static void war_disable   ( CHAR_DATA *ch, const char *argument ); 
static void war_allowed   ( CHAR_DATA *ch, const char *argument ); 
static void war_lock      ( CHAR_DATA *ch, const char *argument ); 
static void war_unlock    ( CHAR_DATA *ch, const char *argument );
static void war_talk      ( CHAR_DATA *ch, const char *argument ); 
static void war_team      ( CHAR_DATA *ch, const char *argument );
static void war_teamtalk  ( CHAR_DATA *ch, const char *argument );  

static ROOM_INDEX_DATA * get_initial_room (AREA_DATA * pArea);

static CMD_DATA war_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra      
    { "help",     war_help,         POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "info",     war_info,         POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "start",    war_start,        POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "cancel",   war_cancel,       POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "join",     war_join,         POS_STANDING, 1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "list",     war_list,         POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "surrender",war_surrender,    POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "enable",   war_enable,       POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "disable",  war_disable,      POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "allowed",  war_allowed,      POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "lock",     war_lock,         POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "unlock",   war_unlock,       POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "talk",     war_talk,         POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "team",     war_team,         POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "teamtalk", war_teamtalk,     POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { NULL }
};

// local functions
static WAR_DATA * new_war_data ( void );
static void free_war_data ( WAR_DATA * war );
static WAR_PLAYER * new_war_player ( void );
static void free_war_player ( WAR_PLAYER * player );
static bool check_winner ( WAR_DATA * war, bool give_reward);
static void transfer_pets ( CHAR_DATA * ch, ROOM_INDEX_DATA * to_room );
static void transfer_pets_from_same_area ( CHAR_DATA * ch, ROOM_INDEX_DATA * to_room );
static void war_showinfo (WAR_DATA *, const char *, ...);
static void give_reward_obj (CHAR_DATA * ch);
static void end_duel ( WAR_DATA * war );
static void transfer_mount (CHAR_DATA * ch, CHAR_DATA * mount);
static bool is_in_prison (CHAR_DATA * ch);
static bool can_join_war (CHAR_DATA * ch, bool not_silent);


//-------------------------------------------------------------------------------
// function called from interpreter
//-------------------------------------------------------------------------------
DO_FUN (do_warfare)
{
    // this stuff is from conquer.c
    if (!parse_command(ch, argument, war_cmd_table))
    {
        show_command_list (ch, war_cmd_table);
        char_act (".", ch);
        char_act ("Use {CHELP WARFARE{x for more information.", ch);
    }
}

//-------------------------------------------------------------------------------
// if no args were passed - show help 'warfare'
//-------------------------------------------------------------------------------
static void war_help      ( CHAR_DATA *ch, const char *argument )
{
    do_help (ch, "'1.WARFARE'");
}

//-------------------------------------------------------------------------------
// detailed information about war # (if no wars found - war_list)
//-------------------------------------------------------------------------------
static void war_info      ( CHAR_DATA *ch, const char *argument )
{
    static char arg [MAX_INPUT_LENGTH] ;
    BUFFER*       output ;
    int           num ;
    WAR_DATA *    war;
    WAR_PLAYER *  player;
    OBJ_INDEX_DATA * obj;

    if (!war_root)
    {
        char_act("No active wars.", ch);
        if (IS_IMMORTAL (ch))
            char_printf (ch, "\n%d tick(s) to next auto war.\n", time_to_next_war);
        return;
    }

    if (!str_cmp (argument, "auto"))
    {
        war = ch->in_war;
        if (!war)
        {
            char_act ("Hmmm. You have to join the war first.", ch);
            return;
        }
    }
    else
    {
        argument = one_argument (argument, arg, sizeof (arg)) ;
        num      = atoi (arg) ;

        if (arg[0] == '\0' || num <= 0)
        // all wars
        {
            char_act ("You must specify war number.", ch);
            war_list (ch, str_empty) ;
            return ;
        }
        else
        {
            int skip = num - 1 ;
            for (war = war_root ; war && skip > 0 ;
                 war = war->next, --skip) ;

            if (war == NULL)
            {
                char_act ("No such war.", ch) ;
                war_list (ch, str_empty) ;
                return ;
            }

        }
    }

    if (IS_SET (war->flags, WS_SET_PARMS) && !IS_IMMORTAL (ch))
    {
        char_act ("No information available for this war yet.", ch);
        return;
    }

    output = buf_new (ch->lang) ;
    buf_add (output, "{C * * *    {RWAR    {C* * *{x\n");
    buf_printf (output, "Levels        : %d - %d.\n", war->minlevel, war->maxlevel);
    buf_printf (output, "Area          : %s.\n",      war->war_area->name);
    if (war->type == WAR_TYPE_DUEL)
        buf_printf (output, "Type          : %s.\n", GETMSG ("Duel", ch->lang));
    else
        buf_printf (output, "Type          : %s.\n", GETMSG (war_desc[war->type].type_str, ch->lang));

    buf_add (output, "Options       : ");
    if (IS_SET (war->flags, WO_USE_PETS))
        buf_add (output, "pets are {Gallowed{x, ");
    else
        buf_add (output, "pets are {Rnot allowed{x, ");
    if (IS_SET (war->flags, WO_USE_PILLS))
        buf_add (output, "potions/pills/wands/staves/scrolls are {Gallowed{x.\n");
    else
        buf_add (output, "potions/pills/wands/staves/scrolls are {Rnot allowed{x.\n");

    buf_printf (output, "Time elapsed  : %d.\n",      war->time_lasts);

    if (war->time_limit > -1)
        buf_printf (output, "Time left     : %d.\n", war->time_limit - war->time_lasts);
    else
        buf_add (output, "Time left     : unlimited.\n");


    if (IS_IMMORTAL (ch))
    {
        if (war->bp_reward || war->qp_reward || war->gold_reward)
            buf_printf (output, "Reward        : %d bp, %d qp, %d gold.\n",
                war->bp_reward, war->qp_reward, war->gold_reward);

        if (war->obj_reward_vnum)
        {
            obj = get_obj_index (war->obj_reward_vnum);
            if (obj)
                buf_printf (output, "Object        : %s.\n", mlstr_cval (obj->short_descr, ch));
        }
    }

    if (war->bp_join_fee || war->qp_join_fee || war->gold_join_fee)
        buf_printf (output, "Join fee      : %d bp, %d qp, %d gold.\n",
            war->bp_join_fee, war->qp_join_fee, war->gold_join_fee);

    buf_add (output, "Fighters ");
    if (war->max_fighters > 0)
        buf_printf (output, "(%d of %d)", war->curr_fighters, war->max_fighters);
    buf_add (output, ":\n    ");

    if (!war->player)
    {
        buf_add (output, "None.");
        buf_add (output, "\n");
    }
    else
    {
        for (player = war->player; player; player = player->next)
            if (player->next)
                buf_printf (output, "%s, ", player->war_char->name);
            else
                buf_printf (output, "%s\n", player->war_char->name);
    }

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}

//-------------------------------------------------------------------------------
// shows a list of active wars
//-------------------------------------------------------------------------------
static void war_list     ( CHAR_DATA *ch, const char *argument )
{
    WAR_DATA *    war;
    BUFFER*       output;
    int           num = 0 ;

    if (war_root == NULL)
    {
        char_act("No active wars.", ch);
        if (IS_IMMORTAL (ch))
            char_printf (ch, "\n%d tick(s) to next auto war.\n", time_to_next_war);
        return;
    }

    output = buf_new (ch->lang);

    if (!str_cmp (argument, "duel"))
        buf_add (output, "Duels:\n");
    else
        buf_add (output, "Wars:\n");

    for (war = war_root ; war ; war = war->next)
    {
        if (war->type == WAR_TYPE_DUEL) // special case - duels
        {
            if (IS_SET (war->flags, WS_PREPARING))
            {
                if (war->player->next)
                    buf_printf (output, "{G%2d{x. Area %s: duel %s vs %s is preparing.\n",
                        ++num, war->war_area->name, war->player->war_char->name, war->player->next->war_char->name);
                else
                    buf_printf (output, "{G%2d{x. Area %s: duel, provoked by %s - no opponents.\n",
                        ++num, war->war_area->name, war->player->war_char->name);
            }
            else // it MUST be 2 opponents!
                buf_printf (output, "{G%2d{x. Area %s: duel %s vs %s.\n",
                    ++num, war->war_area->name, war->player->war_char->name, war->player->next->war_char->name);

        }
        else
        {
            if (!str_cmp (argument, "duel"))
                continue;

            if (IS_SET (war->flags, WS_PREPARING))
                buf_printf (output, "{G%2d{x. Area %s: war for levels {W%d{x-{W%d{x will start in %d ticks.\n",
                    ++num, war->war_area->name, war->minlevel, war->maxlevel, war->time_to_begin) ;
            else if (IS_SET (war->flags, WS_SET_PARMS))
                buf_printf (output, "{G%2d{x. Area %s: tension grows up.\n",
                    ++num, war->war_area->name);
            else
                buf_printf (output, "{G%2d{x. Area %s: war for levels {W%d{x-{W%d{x is in progress!\n",
                    ++num, war->war_area->name, war->minlevel, war->maxlevel) ;
        }
    }

    if (IS_IMMORTAL (ch))
        buf_printf (output, "\n%d tick(s) to next auto war.\n", time_to_next_war);

    char_act(buf_string (output), ch) ;
    buf_free (output) ;
}

//-------------------------------------------------------------------------------
// start a war in ch->in_room->area or automatically in any free area
//-------------------------------------------------------------------------------
static void war_start     ( CHAR_DATA *ch, const char *argument )
{
    int min, max;
    WAR_DATA * war;
    static char arg [MAX_INPUT_LENGTH] ;
    AREA_DATA * curr_area;

    // do not start wars before reboot
    if (reboot_counter <= 30 && reboot_counter >= 0) 
        return ;

    if (!war_enabled)
    {
        if (ch)
            char_act ("Wars are currently disabled.", ch);
        return;
    }

    if (ch && str_cmp (argument, "auto")) // manual start
    {
        if (!ch->in_room || !ch->in_room->area)
            return;

         curr_area = ch->in_room->area;
        
        if (!IS_SET(curr_area->flags, AREA_WAR_ENABLED))
        {
            char_act ("But the war can't start in this area!", ch);
            return;
        }

        for (war = war_root; war; war = war->next)
            if (war->war_area == ch->in_room->area)
                {
                    char_act ("But the war in this area is already active!", ch);
                    return;
                }

        argument = one_argument (argument, arg, sizeof (arg)) ;
        min      = atoi (arg) ;

        argument = one_argument (argument, arg, sizeof (arg)) ;
        max      = atoi (arg) ;

        if (min <= 10)
            min = 10;
        if (max == 0 || max > 91) 
            max = 91;
        if (max < min) max = min;

        war = new_war_data();

        war->next = war_root;
        war_root = war;

        war->minlevel = min;
        war->maxlevel = max;
        war->time_to_begin = WAR_SETPARM_TIME;
        war->war_area = curr_area;
        war->type = WAR_TYPE_GENOCID;

        // todo: generate reward more carefully - fix halyava :)
        // m.b. it would be nice modify reward according the number of fighter
        war->bp_reward = number_range (war_desc[war->type].min_bp_reward, war_desc[war->type].max_bp_reward);
        war->qp_reward = number_range (war_desc[war->type].min_qp_reward, war_desc[war->type].max_qp_reward);
        war->gold_reward = number_range (war_desc[war->type].min_gold_reward, war_desc[war->type].max_gold_reward);

        SET_BIT(war->flags, WS_SET_PARMS);

        char_printf(ch, "You've just started level %d-%d war in %s.\n", min, max, curr_area->name);
    }
    else
    {
        int               level,        // "base" level for war
                          max_attempt,  // attempts to find proper player range
                          plr_count,    // players found in current level PK range
                          area_count,   // count of free areas found
                          i,
                          type;         // war type
        AREA_DATA *       free_areas[WAR_MAX_AREA_COUNT]; // list of "free" areas with WAR_ENABLED flag
        bool              can_start;
        DESCRIPTOR_DATA * d;

        // find free area
        area_count = 0;
        for (curr_area = area_first; curr_area; curr_area = curr_area->next)
        {
            if (IS_SET(curr_area->flags, AREA_WAR_ENABLED) 
                && !IS_SET(curr_area->flags, AREA_GQACTIVE))
            {
                can_start = TRUE;
                for (war = war_root; war; war = war->next)
                    if (war->war_area == curr_area)
                    {
                        can_start = FALSE; 
                        break;
                    }
                if (can_start)
                    free_areas[area_count++] = curr_area;
            }
        }

        if (area_count == 0) 
        {
            if (ch)
                char_act ("There are no free areas to start the war.", ch);
            return;
        }
        curr_area = free_areas[number_range(0, area_count-1)]; // so: area found

        // find levels
        max_attempt = 30;

        for ( i = 0; i < max_attempt; ++i )
        {
            level = number_range (PK_MIN_LEVEL, LEVEL_HERO);
            plr_count = 0;
            min = UMAX (PK_MIN_LEVEL, level - pk_range (level)/2);
            max = UMIN (LEVEL_HERO,   level + pk_range (level)/2);

            for (d = descriptor_list ; d ; d = d->next)
            {
                if ((d->connected == CON_PLAYING) &&
                    (d->character->level >= min) &&
                    (d->character->level <= max))
                    ++plr_count;
            }

            if (plr_count > 1) // Yesss! We've found it!
                break;
        }

        if (plr_count < 2)
        {
            if (ch)
                char_act ("Not enough adventurers found. Can't calculate level range.", ch);
            return;
        }

        // choose war_type
        // todo: choose war_type before level selection and try to choose level
        // more carefully
        max_attempt = 10;
        for ( i = 0; i < max_attempt; ++i )
        {
            type = number_range (0, WAR_MAX_TYPE - 1);
            if (IS_SET (war_desc[type].extra, WF_AUTOGENERATE))
                break;
        }

        // finally! All is OK!
        war = new_war_data();

        war->next = war_root;
        war_root = war;

        war->minlevel = min;
        war->maxlevel = max;
        war->war_area = curr_area;
        war->type = type;
        if (number_range (0,1) == 0)
            war->time_limit = RND_TIME_DURATION;

        war->bp_reward = number_range (war_desc[war->type].min_bp_reward, war_desc[war->type].max_bp_reward);
        war->qp_reward = number_range (war_desc[war->type].min_qp_reward, war_desc[war->type].max_qp_reward);
        war->gold_reward = number_range (war_desc[war->type].min_gold_reward, war_desc[war->type].max_gold_reward);

        SET_BIT(war->flags, WS_PREPARING);
        war->time_to_begin = WAR_PREPARATION_TIME;

        war_showinfo (NULL, "Warfare for levels %d-%d declared in area %s!", 
             war->minlevel, war->maxlevel, war->war_area->name);
    }
}

//-------------------------------------------------------------------------------
// cancel war (s):
//     war cancel <number> - just sets flag WS_NEED_CANCEL 
//                           and calls 'war cancel auto'
//     war cancel auto - cancels ALL wars with flag WS_NEED_CANCEL set
//
// !!! WARNING!!! when we use 'war cancel auto' ch can be NULL!!!
//
//-------------------------------------------------------------------------------
static void war_cancel ( CHAR_DATA *ch, const char *argument )
{
    // if war is active - transfer all players to CR
    // free all memory
    // remove area flag
    int           num ;
    WAR_DATA *    war, * tmp;

    if (!war_root)
    {
        if (ch) char_act("No active wars.", ch);
        return;
    }

    if (!str_cmp(argument, "auto")) // auto cancel
    {
        if (!war_root)
            return;

        // 1) delete first items to be deleted
        for (;;)
        {
            if (!war_root)
                break;
            if (!IS_SET (war_root->flags, WS_NEED_CANCEL))
                break;
            else
            {
                tmp = war_root->next;
                free_war_data (war_root);
                war_root = tmp;
            }
        }
        // 2) delete all items to be deleted from the middle
        tmp = war_root;
        for (war = war_root; war; )
        {
            if (IS_SET (war->flags, WS_NEED_CANCEL))
            {
                tmp->next = war->next;
                free_war_data (war);
                war = tmp->next;
            }
            else
            {
                tmp = war;
                war = war->next;
            }
        }
    }
    else // manual cancel
    {
        num      = atoi (argument) ;

        if (argument[0] == '\0' || num <= 0)
        {
            if (ch)
            { 
                char_act ("You must specify war number.", ch);
                war_list (ch, str_empty) ;
            }
            return ;
        }
        else
        {
            int skip = num - 1 ;
            for (war = war_root ; war && skip > 0 ;
                 war = war->next, --skip) ;

            if (war == NULL)
            {
                if (ch)
                {
                    char_act ("No such war.", ch) ;
                    war_list (ch, str_empty) ;
                }
                return ;
            }
            SET_BIT (war->flags, WS_NEED_CANCEL);

            if (ch)
                char_printf (ch, "You've just cancelled war in %s.\n", war->war_area->name);

            if (war->type == WAR_TYPE_DUEL)
                war_showinfo (NULL, "Duel in the area %s was cancelled by the {CImmortals{x.",
                    war->war_area->name);
            else if (!IS_SET(war->flags, WS_SET_PARMS))
                war_showinfo (NULL, "War in the area %s was cancelled by the {CImmortals{x.",
                    war->war_area->name);

            war_cancel (NULL, "auto");
        }
    }
}

//-------------------------------------------------------------------------------
// join active war which is preparing
// ch is transferred into preparation "peace" room
//-------------------------------------------------------------------------------
static void war_join ( CHAR_DATA *ch, const char *argument )
{
    int          war_number;
    WAR_DATA   * war;
    WAR_PLAYER * war_player;
    OBJ_DATA   * tattoo;
    CHAR_DATA  * mount;

    if (IS_NPC (ch))
        return;

    if (!can_join_war (ch, TRUE))
        return; 

    if (ch->in_war)
    {
        char_act ("But you are already involved into warfare!", ch);
        return;
    }

    war_number = atoi (argument);
    if (war_number) // number specified
    {
        int skip = war_number - 1 ;
        for (war = war_root ; war && skip > 0 ;
             war = war->next, --skip) ;
        if (!war)
        {
            char_act ("No such war.", ch);
            war_list (ch, str_empty);
            return;
        }

        if (ch->level < war->minlevel)
        {
            char_act ("Sorry, you are too small for this war.", ch);
            return;
        }
        if (ch->level > war->maxlevel)
        {
            char_act ("Sorry, you are too tough for this war.", ch);
            return;
        }
        if (war->type == WAR_TYPE_DUEL)
        {
            char_act ("It's a duel. Use 'duel accept <name>' instead of 'warfare join'.", ch);
            return;
        }
    }
    else
    {
        for (war = war_root; war; war = war->next)
            if (ch->level >= war->minlevel && 
                ch->level <= war->maxlevel && 
                war->type != WAR_TYPE_DUEL)
                break;
        if (!war)
        {
            char_act ("No wars suitable for you found.", ch);
            return;
        }
    }

    if (ch->pcdata->bonuspoints < war->bp_join_fee)
    {
        char_act ("You have not enough bonus points to pay join fee.", ch);
        return;
    }

    if (ch->pcdata->questpoints < war->qp_join_fee)
    {
        char_act ("You have not enough quest points to pay join fee.", ch);
        return;
    }

    if (ch->gold < war->gold_join_fee)
    {
        char_act ("You have not enough gold to pay join fee.", ch);
        return;
    }

    // todo: another flag
    if (IS_SET(war->flags, WS_SET_PARMS))
    {
        char_act ("The clouds are gathering over that area but there is no war yet.", ch);
        return;
    }
    else if (!IS_SET(war->flags, WS_PREPARING))
    {
        char_act ("You are too late for this war.", ch);
        return;
    }

    if (war->type == WAR_TYPE_CLAN && !ch->clan)
    {
        char_act ("This war is for clan members only.", ch);
        return;
    }

    if (war->type == WAR_TYPE_RELIGION)
    {
        if (!ch->religion)
        {
            char_act ("You are atheist. These religious wars rest you calm.", ch);
            return;
        }
        tattoo = get_eq_char(ch, WEAR_TATTOO);
        if (!tattoo) 
        {
            char_act("You have to wear the tattoo of your religion!", ch);
            return;
        }
    }

    if ((war->max_fighters > 0) && (war->curr_fighters == war->max_fighters))
    {
        char_act ("All ranks are full. You are too late.", ch);
        return;
    }

    // all is ok - create war_player and transfer to preparation room

    war_player = new_war_player ();
    if (!war_player) return;

    war_player->war_char = ch;
    war_player->next = war->player;
    war->player = war_player;
    ch->war_status = PS_ALIVE;

    if (IS_SET(war->flags, WO_USE_PETS)) // transport all charmies to preparation room
        transfer_pets (ch, room_preparation);
    
    if (ch->fighting != NULL) // ch must not be pumped, but... just to be sure
        stop_fighting(ch, TRUE);

    mount = MOUNTED(ch);

    act("$n leaves in {Rred{x flash of light!",ch,NULL,NULL,TO_ROOM);
    char_from_room (ch);
    char_to_room (ch, room_preparation);
    do_look (ch, "auto");
    act("$n arrives in {Rred{x flash of light!",ch,NULL,NULL,TO_ROOM);

    // may be transfer if only WO_USE_PETS option is set?
    transfer_mount (ch, mount);

    char_act ("\nThe {CImmortals{x allow you to fight in this war.\n", ch);

    if (war->bp_join_fee || war->qp_join_fee || war->gold_join_fee)
    {
        char_printf (ch, "You've paid join fee : %d bp, %d qp, %d gold.\n",
            war->bp_join_fee, war->qp_join_fee, war->gold_join_fee);
        ch->pcdata->bonuspoints -= war->bp_join_fee;
        ch->pcdata->questpoints -= war->qp_join_fee;
        ch->gold -= war->gold_join_fee;
    }

    ch->in_war = war;

    war_info (ch, "auto");

    war_showinfo (NULL, "%s joins to war in area %s! Good luck!", 
        ch->name, war->war_area->name);

    war->curr_fighters++;
}

//-------------------------------------------------------------------------------
// stop participation in current war (needed for do_quit)
// when he surrenders, ch looses the possibility to win even if his team will win
//-------------------------------------------------------------------------------
static void war_surrender ( CHAR_DATA *ch, const char *argument )
{
    WAR_PLAYER * player, *tmp;
    WAR_DATA * war = ch->in_war;
    bool live = FALSE;

    if (!war)
    {
        char_act ("You should join a war before you can surrender.", ch);
        return;
    }

    if (IS_SET (war->flags, WS_PREPARING | WS_SET_PARMS))
    {
        char_act ("You can't surrender before the war starts.", ch);
        return;
    }

    if (IS_AFFECTED(ch, AFF_SLEEP))
    {
        char_act ("You must be awake to surrender.", ch);
        return;
    }

    for (player = war->player; player; player = player->next)
        if (player->war_char == ch)
            break;

    if (!player)
    {
        char_act ("BUG! Please report to immortals", ch);
        return;
    }

    if (ch->war_status == PS_ALIVE)
    {
        // transfer ch to CR with appropriate message, free war_player
        live = TRUE;
        char_act ("\nYou surrender.", ch);
        ch->pcdata->questpoints -= 50; // to minimize abuses
        char_act ("\nSurrender costs you 50 qp.", ch);
        act("$n surrenders!",player->war_char,NULL,NULL,TO_ROOM);

        if (war->type == WAR_TYPE_DUEL) // special case
        {
            war_showinfo (NULL, "%s cowardly surrenders... What a shame!", 
                ch->name);
            ++stat_record.wars_surrenders;
            // NB! we free memory in 'end-duel'
            end_duel (war);
            return;
        }
    }

    // free memory
    if (player == war->player)
    {
        war->player = player->next;
        free_war_player (player);
    }
    else
    {
        tmp = war->player;
        for (player = war->player; player; )
        {
            if (player->war_char == ch)
            {
                tmp->next = player->next;
                free_war_player (player);
                break;
            }
            else
            {
                tmp = player;
                player = player->next;
            }
        }
    }

    if (!IS_SET (war->flags, WS_PREPARING))
    {
        // check for winner and update info
        if (live)
            war_showinfo (NULL, "%s cowardly surrenders... What a shame!", 
                ch->name);
        ++stat_record.wars_surrenders;
        check_winner (war, TRUE); // here we will cancel war if needed
    }
}

//-------------------------------------------------------------------------------
// enable wars in ch->in_room->area
// m.b. save area after this?
//-------------------------------------------------------------------------------
static void war_enable     ( CHAR_DATA *ch, const char *argument )
{
    if (!ch->in_room || !ch->in_room->area)
    {
        char_act ("You should choose proper area.", ch);
        return;
    }

    if (IS_SET(ch->in_room->area->flags, AREA_WAR_ENABLED))
        char_printf(ch, "Wars in area %s are already enabled.\n", ch->in_room->area->name);
    else
    {
        SET_BIT (ch->in_room->area->flags, AREA_WAR_ENABLED);
        char_printf(ch, "Now wars in area %s are enabled.\n", ch->in_room->area->name);
        save_area(ch->in_room->area);
        char_act("Area was saved.", ch);
    }
}

//-------------------------------------------------------------------------------
// disable wars in ch->in_room->area
// m.b. save area after this?
//-------------------------------------------------------------------------------
static void war_disable     ( CHAR_DATA *ch, const char *argument )
{
    WAR_DATA * war;

    if (!ch->in_room || !ch->in_room->area)
    {
        char_act ("You should choose proper area.", ch);
        return;
    }

    if (!IS_SET(ch->in_room->area->flags, AREA_WAR_ENABLED))
        char_printf(ch, "Wars in area %s are already disabled.\n", ch->in_room->area->name);
    else
    {
        // cancel war in target area if any
        for (war = war_root; war; war = war->next)
            if (war->war_area == ch->in_room->area)
            {
                SET_BIT (war->flags, WS_NEED_CANCEL);
                war_cancel (ch, "auto");
                break;
            }

        REMOVE_BIT (ch->in_room->area->flags, AREA_WAR_ENABLED);
        char_printf(ch, "Now wars in area %s are disabled.\n", ch->in_room->area->name);
        save_area(ch->in_room->area);
        char_act("Area was saved.", ch);
    }
}

//-------------------------------------------------------------------------------
// show to char a list of areas where wars can be started
//-------------------------------------------------------------------------------
static void war_allowed   ( CHAR_DATA *ch, const char *argument )
{
    AREA_DATA * pArea;
        BUFFER    * output;
    int         count = 0;

    output = buf_new(ch->lang);
    buf_printf (output, "{x  Areas where wars are enabled\n");
    buf_printf (output, "-----------------------------------\n");

    for (pArea = area_first; pArea; pArea = pArea->next) 
    {
        if (IS_SET (pArea->flags, AREA_WAR_ENABLED))
        {
                    ++count;
            if (IS_SET (pArea->flags, AREA_WAR_ACTIVE))
                buf_printf (output, "%s (war is active)\n", fmt_color_str(pArea->name, 25));
            else
                buf_printf (output, "%s\n", pArea->name);
        }
        }

    buf_printf (output, "-----------------------------------\n");
    buf_printf (output, "%d area(s) total.\n", count);

    if (count)
        page_to_char (buf_string (output), ch);
    else
        char_act("Wars are disabled everywhere.", ch);
    buf_free (output);
    return;
}

//-------------------------------------------------------------------------------
// disable all wars; active wars will cancel immediately
//-------------------------------------------------------------------------------
static void war_lock      ( CHAR_DATA *ch, const char *argument )
{
    WAR_DATA * war;

    // cancel all wars
    for (war = war_root; war; war = war->next)
    {
        SET_BIT (war->flags, WS_NEED_CANCEL);
    }

    war_cancel (ch, "auto");

    war_enabled = FALSE;
    char_act ("You've just disabled the wars all over the world!", ch);
}

//-------------------------------------------------------------------------------
// enable wars
//-------------------------------------------------------------------------------
static void war_unlock    ( CHAR_DATA *ch, const char *argument )
{
    war_enabled = TRUE;
    char_act ("You've just enabled the wars all over the world!", ch);
}

//-------------------------------------------------------------------------------
// talk something to all fighters in the same war
//-------------------------------------------------------------------------------
static void war_talk      ( CHAR_DATA *ch, const char *argument ) 
{
    if (IS_NPC(ch))
        return;

    if (!ch->in_war && !IS_IMMORTAL(ch))
    {
        char_act ("Hmmm. You have to join the war first.", ch);
        return;
    }

    if (IS_SET (ch->comm, COMM_NOCHANNELS))
    {
        char_act ("You can not use channels.", ch);
        return ;
    }

    if (IS_IMMORTAL (ch))
        war_showinfo ((WAR_DATA*) -1, "{W[%s] {Y%s{x", ch->name, argument) ;
    else 
        war_showinfo (ch->in_war, "{W[%s] {Y%s{x", capitalize (ch->name), argument) ;
}

//-------------------------------------------------------------------------------
// say something to all fighters of your team
//-------------------------------------------------------------------------------
static void war_teamtalk  ( CHAR_DATA *ch, const char *argument )  
{
    WAR_PLAYER * player, * ch_player;
    
    if (IS_NPC(ch))
        return;

    if (!ch->in_war)
    {
        char_act ("Hmmm. You have to join the war first.", ch);
        return;
    }

    if (ch->in_war->type == WAR_TYPE_GENOCID)
    {
        char_act ("You have no team in this war...", ch);
        return;
    }

    if (ch->in_war->type == WAR_TYPE_DUEL)
    {
        char_act ("It's duel! You are alone.", ch);
        return;
    }

    char_printf (ch, "\n{r[***{RWAR{r***]{x You tell to team: {G%s{x\n", argument);

    for (ch_player = ch->in_war->player; ch_player; ch_player = ch_player->next)
        if (ch_player->war_char == ch)
            break;

    for (player = ch->in_war->player; player; player = player->next)
    {
        if (war_desc[ch->in_war->type].check_same_team_fn (ch_player, player)
            && player != ch_player)
        {
            char_printf (player->war_char, "\n{r[***{RWAR{r***]{x {W%s{x tells to team: {G%s{x\n", 
                ch->name, argument);
        }
    }
}

//-------------------------------------------------------------------------------
// talk something to all fighters of your team
//-------------------------------------------------------------------------------
static void war_team      ( CHAR_DATA *ch, const char *argument )
{
    WAR_PLAYER * player, * ch_player;

    if (IS_NPC(ch))
        return;

    if (!ch->in_war)
    {
        char_act ("Hmmm. You have to join the war first.", ch);
        return;
    }

    if (ch->in_war->type == WAR_TYPE_DUEL)
    {
        char_act ("It's duel! You are alone.", ch);
        return;
    }

    char_act ("Your friends in this war:", ch);
    for (ch_player = ch->in_war->player; ch_player; ch_player = ch_player->next)
        if (ch_player->war_char == ch)
            break;

    for (player = ch->in_war->player; player; player = player->next)
    {
        if (war_desc[ch->in_war->type].check_same_team_fn (ch_player, player))
        {
            if (player->war_char->war_status == PS_ALIVE)
                char_printf (ch, "  %s\n", player->war_char->name);
            else
                char_printf (ch, "{x  %s {r(killed){x\n", player->war_char->name);
        }
    }
    char_act ("", ch);
}

//-------------------------------------------------------------------------------
// recycling
//-------------------------------------------------------------------------------
static WAR_DATA * new_war_data    (void)
{
    WAR_DATA * war;

        war = calloc(1, sizeof(*war));
    if (war) // just to be sure
    {
        war->next = NULL;
        war->player = NULL;
        war->war_area = NULL;
        war->max_fighters = 0;
        war->curr_fighters = 0;
        war->time_limit = -1;
        SET_BIT (war->flags, WO_USE_PETS | WO_USE_PILLS);
        ++stat_record.wars_started;
    }
    return war;
}

static void free_war_data (WAR_DATA * war)
{
    WAR_PLAYER * curr, *next;

    // first, free all war_players
    for (curr = war->player; curr; )
    {
        next = curr->next;
        free_war_player (curr);
        curr = next;
    }
    // second, remove area flag
    REMOVE_BIT (war->war_area->flags, AREA_WAR_ACTIVE);
    // finally, free
    free (war);
}

static WAR_PLAYER * new_war_player (void)
{
    WAR_PLAYER * player;

    player = calloc(1, sizeof(*player));
    if (player)
    {
        player->war_char = NULL;
        player->next = NULL;
    }
    return player;
}

static void free_war_player (WAR_PLAYER * player)
{
    CHAR_DATA  * mount;
    
    if (player->war_char)
    {
        if (player->war_char->war_status == PS_ALIVE || player->war_char->war_status == PS_LIVE_WINNER)
        {
            if (player->war_char->fighting != NULL)
                stop_fighting(player->war_char, TRUE);

            if (IS_SET (player->war_char->in_war->flags, WS_PREPARING | WS_SET_PARMS))
                transfer_pets (player->war_char, room_CR);
            else
                transfer_pets_from_same_area (player->war_char, room_CR);

            RESET_FIGHT_TIME (player->war_char);
            player->war_char->position = POS_STANDING;

            player->war_char->war_status = PS_NONE;

            mount = MOUNTED(player->war_char);

            act("$n leaves in dim {rred{x flash of light!",player->war_char,NULL,NULL,TO_ROOM);
            char_from_room (player->war_char);
            char_to_room (player->war_char, room_CR);
            act("$n arrives in dim {rred{x flash of light!",player->war_char,NULL,NULL,TO_ROOM);
            char_act ("\nThe war is over... You can leave now.", player->war_char);

            transfer_mount (player->war_char, mount);
        }
        player->war_char->in_war->curr_fighters--;
        player->war_char->in_war = NULL;
    }

    free (player);
}

//-------------------------------------------------------------------------------
// update all war records (timers). clear war area before beginning
// called from update_handler (update.c)
//-------------------------------------------------------------------------------
void war_update (void)
{
    WAR_DATA        * war;
    CHAR_DATA       * wch;
    WAR_PLAYER      * player;
    ROOM_INDEX_DATA * room;
    CHAR_DATA       * mount;


    for (war = war_root; war; war = war->next)
    {
        if (war->time_to_begin > 0)
            --war->time_to_begin;

        if (IS_SET(war->flags, WS_SET_PARMS))
        {
            if (war->time_to_begin <= 0)
            {
                REMOVE_BIT (war->flags, WS_SET_PARMS);
                SET_BIT(war->flags, WS_PREPARING);
                war->time_to_begin = WAR_PREPARATION_TIME;
                war_showinfo (NULL, "Warfare for levels %d-%d declared in area %s!", 
                     war->minlevel, war->maxlevel, war->war_area->name);
            }
        }
        else if (IS_SET(war->flags, WS_PREPARING))
        {
            if (war->time_to_begin <= 0)
            // need to start this war
            {
                if (war->type == WAR_TYPE_DUEL)
                {
                    if (!war->player->next)
                    {
                        war_showinfo (NULL, "%s couldn't find any opponent for duel...", war->player->war_char->name);
                        war->player->war_char->last_duel_time = DUEL_INTERVAL;
                        SET_BIT (war->flags,WS_NEED_CANCEL); 
                        continue;
                    }
                }
                else if (check_winner (war, FALSE))
                {
                    war_showinfo (NULL, "Warfare in area %s ends before the fight starts... Not evough brave adventurers.", war->war_area->name);
                    SET_BIT (war->flags,WS_NEED_CANCEL); 
                    continue;
                }

                for (player = war->player; player; player = player->next)
                {
                    room = get_initial_room (war->war_area);

                    if (!room)
                    {
                        SET_BIT (war->flags,WS_NEED_CANCEL);
                        break;
                    }

                    if (IS_SET(war->flags, WO_USE_PETS)) // transport all charmies to preparation room
                        transfer_pets (player->war_char, room);

                    mount = MOUNTED(player->war_char);

                    if (player->war_char->fighting != NULL) // ch must not be pumped, but... just to be sure
                        stop_fighting(player->war_char, TRUE);
                    act("$n leaves in {Rred{x flash of light!",player->war_char,NULL,NULL,TO_ROOM);
                    char_from_room (player->war_char);
                    char_to_room (player->war_char, room);

                    player->war_char->hit = player->war_char->max_hit;
                    player->war_char->mana = player->war_char->max_mana;
                    player->war_char->move = player->war_char->max_move;
                    clear_bad_affects (player->war_char);
                    char_act ("\nYou are fully restored.\n", player->war_char);

                    do_look (player->war_char, "auto");
                    char_act ("\nLet the war begin!\n", player->war_char);

                    transfer_mount (player->war_char, mount);

                    war_team (player->war_char, str_empty);

                    act("$n arrives in {Rred{x flash of light!",player->war_char,NULL,NULL,TO_ROOM);

                    player->war_char->stat_wars_all++;
                    player->war_char->stat_type_wars_all[war->type]++;
                }

                // now we have to move all 
                for (wch = char_list ; wch ; wch = wch->next)
                {
                    if (!wch->in_room || !wch->in_room->area)
                        continue;
                    if (!IS_NPC (wch) && !IS_IMMORTAL (wch) 
                        && (wch->in_room->area == war->war_area) 
                        && (wch->in_war != war))
                    {
                        if (wch != NULL) 
                            stop_fighting(wch, TRUE);

                        mount = MOUNTED (wch);

                        act("$n leaves in {Mv{miolet{x flash of light!",wch,NULL,NULL,TO_ROOM);
                        char_from_room (wch);
                        char_to_room (wch, room_CR);
                        do_look (wch, "auto");
                        transfer_mount (wch, mount);
                        char_act ("\nHere the flames of war won't burn you!\n", wch);
                        act("$n arrives in {Mv{miolet{x flash of light!",wch,NULL,NULL,TO_ROOM);
                    }
                }

                war_showinfo (NULL, "Warfare in area %s now begins!", war->war_area->name);

                REMOVE_BIT (war->flags, WS_PREPARING);
                SET_BIT(war->war_area->flags,AREA_WAR_ACTIVE);
                war->time_lasts = 1;
            }
            // need to give some info to chars in target area
            else
            {
                if (war->type != WAR_TYPE_DUEL)
                    war_showinfo (NULL, "Warfare in area %s for levels %d-%d will start in %d ticks!", 
                        war->war_area->name, war->minlevel, war->maxlevel, war->time_to_begin);

                for (wch = char_list ; wch ; wch = wch->next)
                {
                    if (!wch->in_room || !wch->in_room->area)
                        continue;
                    if (wch->in_room->area == war->war_area && !IS_IMMORTAL (wch))
                        char_printf (wch, "\nThe {Rwar{x is approaching! You have only %d ticks to leave!\n\n", war->time_to_begin);
                }
            }
        }
        else 
        {
            war->time_lasts++;
            if (war->time_limit > -1) 
            {
                if (war->time_lasts > war->time_limit)
                {
                    // cancel war if time is over.
                    war_showinfo (NULL, "Warfare in area %s is over... There was not enough time to win.", 
                        war->war_area->name, war->minlevel, war->maxlevel, war->time_to_begin);

                    if (war->type == WAR_TYPE_DUEL)
                    {
                        war->player->war_char->last_duel_time = DUEL_INTERVAL;
                        war->player->next->war_char->last_duel_time = DUEL_INTERVAL;
                    }

                    ++stat_record.wars_timed_out;

                    SET_BIT (war->flags,WS_NEED_CANCEL);
                }
                else if ((war->time_limit - war->time_lasts < 3) && (war->type != WAR_TYPE_DUEL))
                    war_showinfo (NULL, "Warfare in area %s will end in %d ticks.", 
                        war->war_area->name, war->time_limit - war->time_lasts + 1);
            }
        }
    }
    war_cancel (NULL, "auto");
    // try to autogenerate war
    if (!--time_to_next_war)
    {
        war_start (NULL, "auto");
        time_to_next_war = RND_TIME_BETWEEN_WARS;
    }
}

//-------------------------------------------------------------------------------
// transfer killed fighter to CR and stop war if one side wins
// called from handle_death (fight.c)
//-------------------------------------------------------------------------------
bool war_handle_death (CHAR_DATA * ch, CHAR_DATA * victim)
{
    CHAR_DATA  * mount;
    
    if (! victim || ! victim->in_war)
        return FALSE;

    if (victim->war_status != PS_ALIVE)
        return FALSE;

    // objects with OPROG_DEATH (e.g. guardian angel) check
    {
        OBJ_DATA* obj;
        OBJ_DATA* obj_next;

        for (obj = victim->carrying;obj != NULL;obj = obj_next)
        {
            obj_next = obj->next_content;
            if (obj->wear_loc != WEAR_NONE
            && oprog_call(OPROG_DEATH, obj, victim, ch))
            {
                victim->position = POS_STANDING;
                return TRUE;
            }
        }
    }
    
    if (victim->fighting != NULL) // fight must be stopped, but... just to be sure
        stop_fighting(victim, TRUE);
    
    RESET_FIGHT_TIME(victim);
    victim->last_death_time = current_time;
    victim->hit = victim->max_hit / 2;
    victim->mana = victim->max_mana / 2;
    victim->move = victim->max_move / 2;

    clear_bad_affects (victim);

    if (ch != victim)
        ch->total_war_kills++;

    victim->total_war_deaths++;
    
    victim->war_status = PS_KILLED;

    transfer_pets_from_same_area (victim, room_CR);

    if (victim->in_war->type != WAR_TYPE_DUEL)
        war_showinfo (NULL, "%s has beed killed in the war in area %s! {DR.I.P.{x", 
            victim->name, victim->in_war->war_area->name);

    act("$n was slain and dissolves in the dim {yyellow{x flash of light!",
        victim,NULL,NULL,TO_ROOM);
    SET_BIT(victim->plr_flags, PLR_GHOST);
    mount = MOUNTED (victim);
    char_from_room (victim);
    char_to_room (victim, room_CR);
    transfer_mount (victim, mount);

    if (victim->in_war->type == WAR_TYPE_DUEL)
        char_act ("\nYou have been {rsl{Ra{rin{x in this duel!\n", victim);
    else
        char_act ("\nYou have been {rsl{Ra{rin{x in this war!\n", victim);

    act("$n arrives in in the dim {yyellow{x flash of light!",
        victim,NULL,NULL,TO_ROOM);

    // check for last kill
    if (victim->in_war->type != WAR_TYPE_DUEL)
        check_winner (victim->in_war, TRUE); // here we will cancel war if needed
    else
        end_duel (victim->in_war);

    return TRUE;
}

//-------------------------------------------------------------------------------
// check_same_team_functions: specified in war_desc table 
//-------------------------------------------------------------------------------
static bool check_same_team_genocid  ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    return (plr1 == plr2);
}

static bool check_same_team_clan     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2 || !plr1->war_char || !plr2->war_char)
        return FALSE;
    else
        return (plr1->war_char->clan == plr2->war_char->clan);
}

static bool check_same_team_race     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2 || !plr1->war_char || !plr2->war_char)
        return FALSE;
    else
        return (ORG_RACE(plr1->war_char) == ORG_RACE(plr2->war_char));
}

static bool check_same_team_religion ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2 || !plr1->war_char || !plr2->war_char)
        return FALSE;
    else
        return (plr1->war_char->religion == plr2->war_char->religion);
}

static bool check_same_team_class    ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2 || !plr1->war_char || !plr2->war_char)
        return FALSE;
    else
        return (plr1->war_char->class == plr2->war_char->class);
}

static bool check_same_team_team     ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2)
        return FALSE;
    else
        return (plr1->team == plr2->team);
}

static bool check_same_team_align    ( WAR_PLAYER * plr1, WAR_PLAYER * plr2 )
{
    if (!plr1 || !plr2 || !plr1->war_char || !plr2->war_char)
        return FALSE;
    else
        return ( (IS_EVIL ( plr1->war_char) && IS_EVIL ( plr2->war_char)) ||
                 (IS_GOOD ( plr1->war_char) && IS_GOOD ( plr2->war_char)) ||
                 (IS_NEUTRAL ( plr1->war_char) && IS_NEUTRAL ( plr2->war_char)) );
}

//-------------------------------------------------------------------------------
// check winner(s) and update status flags for functions end_*
//-------------------------------------------------------------------------------
static bool check_winner ( WAR_DATA * war, bool give_reward)
{
    WAR_PLAYER * player, * first_player;

    for (first_player = war->player; first_player; first_player = first_player->next )
        if (first_player->war_char && first_player->war_char->war_status == PS_ALIVE)
            break;

    if (!first_player) // no "live" players
        return TRUE;

    for (player = first_player->next; player; player = player->next )
    {
        if (!player->war_char) // bug avoiding - cancel war immediately
        {
            SET_BIT (war->flags, WS_NEED_CANCEL);
            war_cancel ( NULL, "auto" );
            return FALSE;
        }
        // skip dead players
        if (player->war_char->war_status != PS_ALIVE)
            continue;

        if (!war_desc[war->type].check_same_team_fn (first_player, player))
            return FALSE; // at least 2 teams
    }

    if (!give_reward)
        return TRUE;

    for (player = war->player; player; player = player->next)
    {
        if (war_desc[war->type].check_same_team_fn (first_player, player))
        {
            if (player->war_char->war_status == PS_ALIVE)
                player->war_char->war_status = PS_LIVE_WINNER;
            else
                player->war_char->war_status = PS_DEAD_WINNER;
        }
    }

    for (player = war->player; player; player = player->next)
    {
        if (player->war_char->war_status == PS_LIVE_WINNER ||
            player->war_char->war_status == PS_DEAD_WINNER)
        {
            player->war_char->stat_wars_win++;
            player->war_char->stat_type_wars_win[war->type]++;

            player->war_char->hit = player->war_char->max_hit;
            player->war_char->mana = player->war_char->max_mana;
            player->war_char->move = player->war_char->max_move;
            clear_bad_affects (player->war_char);

            if (war_desc[war->type].end_fn)
                war_desc[war->type].end_fn (player->war_char);
        }
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            player->war_char->hit = player->war_char->max_hit;
            player->war_char->mana = player->war_char->max_mana;
            player->war_char->move = player->war_char->max_move;
        }
    }

    if (war_desc[war->type].special_end_fn)
        war_desc[war->type].special_end_fn (war);

    ++stat_record.wars_completed;

    SET_BIT (war->flags, WS_NEED_CANCEL);
    war_cancel (NULL, "auto");
    return TRUE;
}

//-------------------------------------------------------------------------------
// add reward to war winner(s) and update statistics
//-------------------------------------------------------------------------------
static void end_genocid    ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    char_act ("\n{CCongratulations!{x You are victorious!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_clan_war   ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;
    
    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    char_act ("\n{CCongratulations!{x Your clan won the war!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_race_war  ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    char_act ("\n{CCongratulations!{x Your race won the war! It would rule the world one day!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_crusade   ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;
    char_act ("\n{CCongratulations!{x You've crushed the heretics!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_class_war ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    if (is_class_warrior(ch) && is_class_mage (ch))
        char_act ("\n{CCongratulations!{x An alloy of Might and Magic in your class aided you to win the war!", ch);
    else if (is_class_warrior(ch))
        char_act ("\nWho can resist pure MIGHT!? {CCongratulations!{x Your class won the war!", ch);
    else if (is_class_mage(ch))
        char_act ("\n{CCongratulations!{x Your mind power has confused these barbarians and your class has won the war!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_team_war  ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    char_act ("\n{CCongratulations!{x Your team won the war!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

static void end_align_war ( CHAR_DATA * ch )
{
    WAR_DATA * war = ch->in_war;

    ch->pcdata->bonuspoints += war->bp_reward;
    ch->pcdata->bp_earned += war->bp_reward;
    ch->pcdata->questpoints += war->qp_reward;
    ch->pcdata->qp_earned += war->qp_reward;
    ch->gold += war->gold_reward;

    if (IS_GOOD (ch))
        char_act ("\n{CCongratulations!{x Once again the good won the war!", ch);
    else if (IS_NEUTRAL (ch))
        char_act ("\n{CCongratulations!{x Neutrality was the right tactics and aided you to win the war!", ch);
    else if (IS_EVIL (ch))
        char_act ("\n{CCongratulations!{x Your rage was beyond comparison... You won the war!", ch);

    char_printf (ch, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    give_reward_obj (ch);
}

//-------------------------------------------------------------------------------
// create object (war reward) and give it to ch
//-------------------------------------------------------------------------------
static void give_reward_obj (CHAR_DATA * ch)
{
    WAR_DATA       * war = ch->in_war;
    OBJ_INDEX_DATA * obj_index;
    OBJ_DATA       * obj;

    if (war->obj_reward_vnum)
    {
        obj_index = get_obj_index (war->obj_reward_vnum);
        if (obj_index) 
        {
            obj = create_obj(obj_index, 0);
            obj_to_char (obj, ch);
            char_printf (ch, "\nThe {CImmortals{x grant you %s as the reward for your victory.\n", 
                mlstr_cval (obj->short_descr, ch));
        }
    }
}
//-------------------------------------------------------------------------------
// info about war end and some special actions if needed
//-------------------------------------------------------------------------------
static void sp_end_genocid ( WAR_DATA * war )
{
    WAR_PLAYER * player;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            war_showinfo (NULL, "In free for all war in the area %s won %s! Long live the champion!",
                war->war_area->name, player->war_char->name);
            return;
        }
}

static void sp_end_clan_war ( WAR_DATA * war )
{
    WAR_PLAYER * player;
    clan_t * clan;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            clan = clan_lookup (player->war_char->clan) ;
            if (!clan) // just to be sure
                return;

            war_showinfo (NULL, "In clan war in the area %s clan %s was victorious!",
                war->war_area->name, clan->name);
            return;
        }
}

static void sp_end_race_war ( WAR_DATA * war )
{
    WAR_PLAYER * player;
    race_t * race;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            race = race_lookup (ORG_RACE (player->war_char));
            if (!race) // just to be sure
                return;

            war_showinfo (NULL, "In race war in the area %s %ss were victorious! One day it will rule the world",
                war->war_area->name, race->name);
            return;
        }
}

static void sp_end_crusade ( WAR_DATA * war )
{
    WAR_PLAYER * player;
    religion_t * religion;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            religion = religion_lookup (player->war_char->religion) ;
            if (!religion) // just to be sure
                return;

            war_showinfo (NULL, "In religion war in the area %s the followers of %s have crushed all heretics!",
                war->war_area->name, religion->name);
            return;
        }
}

static void sp_end_class_war ( WAR_DATA * war )
{
    WAR_PLAYER * player;
    class_t * ch_class;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
        {
            ch_class = class_lookup (player->war_char->class) ;
            if (!ch_class) // just to be sure
                return;

            war_showinfo (NULL, "In class war in the area %s the %ss proved their superiority!",
                war->war_area->name, ch_class->name);
            return;
        }
}

static void sp_end_team_war ( WAR_DATA * war )
{
}

static void sp_end_align_war ( WAR_DATA * war )
{
    WAR_PLAYER * player;

    for (player = war->player; player; player = player->next)
        if (player->war_char->war_status == PS_LIVE_WINNER)
            break;

    if (!player) // just to be sure
        return;

    if (IS_GOOD (player->war_char))
        war_showinfo (NULL, "In the fierce war in the area %s once again the goodness proved their superiority!",
            war->war_area->name);
    else if (IS_NEUTRAL (player->war_char))
        war_showinfo (NULL, "In the fierce war in the area %s the neutrality was the right tactics!",
            war->war_area->name);
    else 
        war_showinfo (NULL, "In the fierce war in the area %s the outstanding rage has aided evils to win!",
            war->war_area->name);
}

//-------------------------------------------------------------------------------
// show detailed war stat for target char
//-------------------------------------------------------------------------------
DO_FUN (do_warstat)
{
    char arg[MAX_INPUT_LENGTH];
    BUFFER *output = NULL;
    CHAR_DATA *wch = NULL;
    bool loaded = FALSE;
    int i;
    int percent;

    argument = one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        char_act("You must provide a name.", ch);
        return;
    }

    if (!str_cmp("self", arg) || !str_cmp(arg, "") || !str_cmp(arg, ""))
        wch = ch;
    else
        wch = gq_findchar(arg);

    if (wch == NULL)
    {
        loaded = TRUE;
        wch = char_load_special(arg);
    }

    if (wch == NULL)
    {
        char_act("No chars with such name.", ch);
        return;
    }

    if (!str_cmp (argument, "clear") && IS_IMMORTAL(ch))
    {
        wch->stat_wars_all = 0;
        for (i = 0; i < WAR_MAX_TYPE; ++i)
            wch->stat_type_wars_all[i] = 0;

        wch->stat_wars_win = 0;
        for (i = 0; i < WAR_MAX_TYPE; ++i)
            wch->stat_type_wars_win[i] = 0;

        wch->total_war_kills = 0;
        wch->total_war_deaths = 0;

        char_printf (ch, "Warfare statictics for %s cleared.\n", wch->name);

        if (loaded)
            char_nuke(wch);
        return;
    }

    if (output == NULL)
        output = buf_new(ch->lang);

    buf_add(output,  "\n    {G/~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\\{x\n");
    buf_printf(output, "{x    {G|                    {CWarfare info for {W%-20s         {G|{x\n", wch->name);
    buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");
    
    buf_add(output,    "{x    {G|                       {RJoined{x           {CWon{G           {c% won{G       |{x\n");

    for (i = 0; i < WAR_MAX_TYPE; ++i)
    {
        if (wch->stat_type_wars_all[i])
            percent = wch->stat_type_wars_win[i] * 100 / wch->stat_type_wars_all[i];
        else
            percent = 0;
        buf_printf(output, "    {G|     %-11s:     {W[%6d]{x        {W[%6d]{G       {W[%6d]{G     |{x\n", 
            war_desc[i].type_str, wch->stat_type_wars_all[i], wch->stat_type_wars_win[i], percent);
    }

    buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

    if (wch->stat_wars_all)
        percent = wch->stat_wars_win * 100 / wch->stat_wars_all;
    else
        percent = 0;
    buf_printf(output, "{x    {G|     Total wars :     {W[%6d]{x        {W[%6d]{G       {W[%6d]{G     |{x\n", 
        wch->stat_wars_all, wch->stat_wars_win, percent);


    buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");

    buf_printf(output, "{x    {G|            {RK{rills{x :     {w%6d{x        {DDeaths{x :     {w%6d{G         |{x\n", 
        wch->total_war_kills, wch->total_war_deaths);

//        buf_printf(output, "    {G|     %-11s:     {W[%6d]{x        {W[%6d]{G       {W[%6d]{G     |{x\n", 
//            war_desc[i].type_str, wch->stat_type_wars_all[i], wch->stat_type_wars_win[i], percent);

    buf_add(output,    "{x    {G|__________________________________________________________________|{x\n");
    buf_add(output, "{x");

    page_to_char(buf_string(output), ch);
    buf_free(output);

    if (loaded)
        char_nuke(wch);
}

//-------------------------------------------------------------------------------
// check if char can move to room
//-------------------------------------------------------------------------------
bool check_war_move (CHAR_DATA * ch, ROOM_INDEX_DATA *to_room, bool silent)
{
    if (IS_NPC (ch) || IS_IMMORTAL (ch))
        return TRUE; // animals and immortals can move everywhere :)

    // char in war
    if (ch->in_war)
    {
        // "live" char
        if (ch->war_status == PS_ALIVE)
        {
            // the same battle-ground
            if (ch->in_war->war_area == to_room->area)
            {
                return TRUE; // can move
            }
            else
            {
                if (!silent)
                    char_act ("You can't leave battleground.", ch);
                return FALSE; // get lost!
            }
        }
        // "dead" char
        else
        {
            if (IS_SET(to_room->area->flags, AREA_WAR_ACTIVE))
            {
                if (!silent)
                    char_act ("You've been burnt by war. You can't go forth.", ch);
                return FALSE; // get lost!
            }
            else
                return TRUE; // can move
        }
    }
    // char is not in war
    else
    {
        // in to_room - war
        if (IS_SET(to_room->area->flags, AREA_WAR_ACTIVE)) 
        {
            if (!silent)
                char_act ("The flames of war blaze there. You can't go forth.", ch);
            return FALSE; // get lost!
        }
        else
            return TRUE; // can move
    }
}

//-------------------------------------------------------------------------------
// called from boot_db (db.c)
//-------------------------------------------------------------------------------
void war_init (void)
{
    war_root = NULL;
    war_enabled = FALSE;

    room_preparation = get_room_index(WAR_PREPARATION_ROOM);
    if (!room_preparation)
    {
        bug ("war_init: preparation room does not exist.", 0);
        return;
    }

    room_CR = get_room_index(ROOM_VNUM_CR);
    if (!room_CR)
    {
        bug ("war_init: ROOM_VNUM_CR does not exist", 0);
        return;
    }

    // all is OK:
    war_enabled = TRUE;
    time_to_next_war = RND_TIME_BETWEEN_WARS;
}

//-------------------------------------------------------------------------------
// get random room in war area to place players
//-------------------------------------------------------------------------------
static ROOM_INDEX_DATA * get_initial_room (AREA_DATA * pArea)
{
    int i, try_count;
    ROOM_INDEX_DATA * room; 
    flag64_t anti_flags = 0 ;

    if (!pArea)
        return 0;

    anti_flags = ROOM_GODS_ONLY | ROOM_IMP_ONLY | ROOM_NOEXPLORE |
                 ROOM_SAFE | ROOM_PEACE ;

    try_count = (pArea->max_vnum - pArea->min_vnum) * 10; // let it be
    for (i = 0; i < try_count; ++i)
    {
        room = get_room_index(number_range (pArea->min_vnum, pArea->max_vnum));
        if (room && !IS_SET (room->room_flags, anti_flags))
            return room;
    }
    return NULL;
}

//-------------------------------------------------------------------------------
// called from do_quit
//-------------------------------------------------------------------------------
void war_quit (CHAR_DATA * ch)
{
    // m.b. "auto" instead of str_empty...
    war_surrender (ch, str_empty);
}

//-------------------------------------------------------------------------------
// called from charm_parser (magic.c)
// to avoid charming during warfare
//-------------------------------------------------------------------------------
bool war_can_charm (CHAR_DATA * ch, CHAR_DATA * victim)
{
    // golem/archon creation
    if (ch && ch->in_war && ch->war_status == PS_ALIVE && 
        !IS_SET(ch->in_war->flags, WO_USE_PETS) && !victim)
    {
        char_act ("Your servants can't aid you. Fight yourself!", ch);
        return FALSE;
    }

    // Charm PC
    if (victim && victim->in_war && victim->war_status == PS_ALIVE && !IS_NPC (victim))
    {
        char_act ("You can't charm those who fight in war.", ch);
        return FALSE;
    }

    // Charm NPC
    if (ch && ch->in_war && ch->war_status == PS_ALIVE && 
        !IS_SET(ch->in_war->flags, WO_USE_PETS))
    {
        char_act ("You can't use charmies in this war.", ch);
        return FALSE;
    }

    return TRUE;
}

//-------------------------------------------------------------------------------
// called from interpreter (m.b. warset)
// during SET_PARMS phase ch can set all war parameters
//-------------------------------------------------------------------------------
DO_FUN (do_warparm)
{
    static char arg [MAX_INPUT_LENGTH] ;
    int           num, i, vnum;
    WAR_DATA *    war;
    OBJ_INDEX_DATA * obj;

    if (!war_root)
    {
        char_act("No active wars.", ch);
        return;
    }

    argument = one_argument (argument, arg, sizeof (arg)) ;
    num      = atoi (arg) ;

    if (arg[0] == '\0' || num <= 0)
    // all wars
    {
        char_act ("You must specify war number.", ch);
        war_list (ch, str_empty) ;
        return ;
    }
    else
    {
        int skip = num - 1 ;
        for (war = war_root ; war && skip > 0 ;
             war = war->next, --skip) ;

        if (war == NULL)
        {
            char_act ("No such war.", ch) ;
            war_list (ch, str_empty) ;
            return ;
        }

    }

    if (!IS_SET (war->flags, WS_SET_PARMS))
    {
        char_act ("Too late to set parameters... Cancel the war and start it again.", ch);
        return;
    }

    argument = one_argument (argument, arg, sizeof (arg)) ;

    if (arg[0] == '\0') 
    {
        do_help (ch, "'WARSET'" );        
        return;
    }

    // type
    if (!str_cmp (arg, "type")) 
    {
        // we scan war_desc table to find appropriate value
        for (i = 0; i < WAR_MAX_TYPE; ++i)
        {
            if (!str_prefix (argument, war_desc[i].type_str))
            {
                war->type = i;
                char_printf (ch, "War type set to '%s'.\n", war_desc[i].type_str);
                return;
            }
        }
        // no war type found - show possible types
        char_act ("Invalid war type. Possible types:", ch);
        for (i = 0; i < WAR_MAX_TYPE; ++i)
        {
             char_printf (ch, "    %s\n", war_desc[i].type_str);
        }
        char_act ( "", ch );
        return;
    }
    else if (!str_cmp (arg, "obj")) 
    {
        vnum = atoi (argument);

        // "obj 0" means "clear obj reward"
        if (!vnum)
        {
            char_act ("Obj reward removed.", ch);
            war->obj_reward_vnum = 0;
            return;
        }

        obj = get_obj_index (vnum);
        if (!obj) 
        {
            char_printf (ch, "Obj %d does not exist.\n", vnum);
            return;
        }
        else
        {
            if (obj->limit > -1 )
            {
                char_act ("You can't reward winners by limited items.", ch);
                return;
            }
            char_printf (ch, "Now winner will receive %s as reward.\n", mlstr_cval (obj->short_descr, ch));
            war->obj_reward_vnum = vnum;
            return;
        }
    }
    else if (!str_prefix (arg, "bpreward")) 
    {
        war->bp_reward = atoi (argument);
        char_printf (ch, "BP reward now: %d bp.\n", war->bp_reward);
        return;
    }
    else if (!str_prefix (arg, "qpreward")) 
    {
        war->qp_reward = atoi (argument);
        char_printf (ch, "QP reward now: %d qp.\n", war->qp_reward);
        return;
    }
    else if (!str_prefix (arg, "goldreward")) 
    {
        war->gold_reward = atoi (argument);
        char_printf (ch, "Gold reward now: %d gold.\n", war->gold_reward);
        return;
    }
    else if (!str_prefix (arg, "bpfee")) 
    {
        war->bp_join_fee = atoi (argument);
        char_printf (ch, "Join fee in BP now: %d bp.\n", war->bp_join_fee);
        return;
    }
    else if (!str_prefix (arg, "qpfee")) 
    {
        war->qp_join_fee = atoi (argument);
        char_printf (ch, "Join fee in QP now: %d qp.\n", war->qp_join_fee);
        return;
    }
    else if (!str_prefix (arg, "goldfee")) 
    {
        war->gold_join_fee = atoi (argument);
        char_printf (ch, "Join fee in gold now: %d gold.\n", war->gold_join_fee);
        return;
    }
    else if (!str_prefix (arg, "flag")) 
    {
        argument = one_argument (argument, arg, sizeof (arg)) ;
        if (!str_prefix (arg, "nopet")) 
        {
            TOGGLE_BIT (war->flags, WO_USE_PETS);
            if (IS_SET (war->flags, WO_USE_PETS))
                char_act ("Players {Gcan{x use pets and charmies in this war.", ch);
            else
                char_act ("Players {rcan't{x use pets and charmies in this war.", ch);
            return;
        }
        else if (!str_prefix (arg, "nopills")) 
        {
            TOGGLE_BIT (war->flags, WO_USE_PILLS);
            if (IS_SET (war->flags, WO_USE_PILLS))
                char_act ("Players {Gcan{x use pills/potions/scrolls/staves/wands this war.", ch);
            else
                char_act ("Players {rcan't{x use pills/potions/scrolls/staves/wands in this war.", ch);
            return;
        }
        else
        {
            char_act ("Available flags: nopet, nopills.", ch);
            return;
        }
    }
    else if (!str_prefix (arg, "time"))
    {
        num = atoi (argument);
        if (num < 0) 
            char_act ("Please specify a numder greater or equal to 0.", ch);
        else if (num == 0)
        {
            char_act ("Time limit for this war removed.", ch);
            war->time_limit = -1;
        }
        else
        {
            char_printf (ch, "Time limit for this war set to %d.", num);
            char_act ("", ch);
            war->time_limit = num;
        }
        return;
    }
    else if (!str_prefix (arg, "minlevel"))
    {
        num = atoi (argument);
        if (num < 10) 
            char_act ("Minimum level must not be less than 10.", ch);
        else if (num > war->maxlevel) 
            char_printf (ch, "Minimum level must not be greater than maximum level (%d).\n", war->maxlevel);
        else
        {
            char_printf (ch, "Minimum level for this war set to %d.\n", num);
            war->minlevel = num;
        }
        return;
    }
    else if (!str_prefix (arg, "maxlevel"))
    {
        num = atoi (argument);
        if (num > 91) 
            char_act ("Maximum level must not be greater than 91.", ch);
        else if (num < war->minlevel) 
            char_printf (ch, "Maximum level must not be less than minimum level (%d).\n", war->minlevel);
        else
        {
            char_printf (ch, "Maximum level for this war set to %d.\n", num);
            war->maxlevel = num;
        }
        return;
    }
    else if (!str_prefix (arg, "maxfighters"))
    {
        num = atoi (argument);
        if (num < 0) 
        {
            char_act ("Please specify a numder greater or equal to 0.", ch);
            return;
        }
        else if (num == 0)
        {
            char_act ("Participant limit for this war removed.", ch);
            war->max_fighters = 0;
            return;
        }
        else
        {
            char_printf (ch, "Participant limit for this war set to %d.\n", num);
            war->max_fighters = num;
            return;
        }
    }

    do_help (ch, "'WARSET'" );
}

//-------------------------------------------------------------------------------
// two functions are used to transfer pets into preparation room 
// and to CR after kill
//-------------------------------------------------------------------------------
static void transfer_pets (CHAR_DATA * ch, ROOM_INDEX_DATA * to_room)
{
    CHAR_DATA * wch, * vch_next;

    for (wch = ch->in_room->people; wch != NULL; wch = vch_next)
    {
        vch_next = wch->next_in_room;
        if ((wch != ch) && IS_AFFECTED(wch, AFF_CHARM) 
            && (wch->master == ch) && (IS_NPC(wch)))
        {
            if (wch->fighting != NULL) // ch must not be pumped, but... just to be sure
                stop_fighting(wch, TRUE);
            
            act("Obeying master's order $n leaves in {Ggreen{x flash of light!",wch,NULL,NULL,TO_ROOM);
            char_from_room(wch);
            char_to_room(wch, to_room);
            act("$n arrives in {Ggreen{x flash of light!",wch,NULL,NULL,TO_ROOM);

            wch->hit = wch->max_hit;
            wch->mana = wch->max_mana;
            wch->move = wch->max_move;
            clear_bad_affects (wch);

            do_look(wch, "auto");
        }
    }
}

static void transfer_pets_from_same_area (CHAR_DATA * ch, ROOM_INDEX_DATA * to_room)
{
    CHAR_DATA * wch;
    
    for (wch = char_list ; wch ; wch = wch->next)
    {
        if (!wch->in_room || !wch->in_room->area)
            continue;
        if ((wch->in_room->area == ch->in_room->area) &&
            (wch != ch) && IS_AFFECTED(wch, AFF_CHARM) && 
            (wch->master == ch) && (IS_NPC(wch)))
        {
            if (wch->fighting != NULL) // ch must not be pumped, but... just to be sure
                stop_fighting(wch, TRUE);
            act("Following master $n leaves in dim {Ggreen{x flash of light!",wch,NULL,NULL,TO_ROOM);
            char_from_room(wch);
            char_to_room(wch, to_room);
            act("$n arrives in dim {Ggreen{x flash of light!",wch,NULL,NULL,TO_ROOM);

            clear_bad_affects (wch);

            do_look(wch, "auto");
        }
    }
}

//-------------------------------------------------------------------------------
// (code from gquest.c)
// war_showinfo ((WAR_DATA *) -1, ...) - from immortal to all players in all wars
// war_showinfo (war, ...) - from player to players in the same war
// war_showinfo (NULL, ...) - to all connected players
//-------------------------------------------------------------------------------
static void war_showinfo (WAR_DATA * war, const char* fmt, ...)
{
    DESCRIPTOR_DATA * d;
    WAR_PLAYER      * player;

    va_list args;
    static char buf [2][MAX_STRING_LENGTH];
    static char buf2 [2][MAX_STRING_LENGTH];
    
    // form the string
    va_start  (args, fmt) ;
    vsnprintf (buf2[0], sizeof(buf2[0]), GETMSG(fmt,0), args) ;
    vsnprintf (buf2[1], sizeof(buf2[1]), GETMSG(fmt,1), args) ;
    va_end    (args) ;
    snprintf  (buf[0], sizeof(buf[0]), "{R[WAR]{x: %s\n", buf2[0]) ;
    snprintf  (buf[1], sizeof(buf[1]), "{R[WAR]{x: %s\n", buf2[1]) ;

    // message from immortals to all players currently involved in any war
    if (war == (WAR_DATA*) -1)
    {
        for (d = descriptor_list ; d ; d = d->next)
            if ((d->connected == CON_PLAYING) && d->character->in_war)
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
                else
                    char_printf (d->character, "%s", buf[d->character->lang]) ;
            }
    }
    else if (war)
    {
        for (player = war->player ; player ; player = player->next)
        {
            if (player->war_char)
            {
                if (IS_SET(player->war_char->comm, COMM_QUIET_EDITOR)
                    &&  player->war_char->desc->pString)
                        buf_add(player->war_char->pcdata->buffer, buf[player->war_char->lang]);
                else
                    char_printf (player->war_char, "%s", buf[player->war_char->lang]) ;
            }
        }
    }
    // message for everyone
    else
    {
        for (d = descriptor_list ; d ; d = d->next)
            if (d->connected == CON_PLAYING && IS_SET(d->character->comm,COMM_WARINFO)) 
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
                else
                    char_printf (d->character, "%s", buf[d->character->lang]) ;
            }
        return ;
    }

    // immortals
    for (d = descriptor_list ; d ; d = d->next)
    {
        if (d->connected == CON_PLAYING && IS_IMMORTAL (d->character) && IS_SET(d->character->comm,COMM_WARINFO))
        {
            if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                &&  d->character->desc->pString)
                    buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
            else
                char_printf (d->character, "%s", buf[d->character->lang]) ;
        }
    }
}

//-------------------------------------------------------------------------------
// remove all bad affects from ch. Used after start of war, victory or 
// player's death
//-------------------------------------------------------------------------------
void clear_bad_affects (CHAR_DATA * ch)
{
    AFFECT_DATA *paf,*paf_next;
    bool need_remove;
    
    // noaff
    for (paf = ch->affected; paf != NULL; paf = paf_next) 
    {
        paf_next = paf->next;

        need_remove = ((paf->type == gsn_plague) ||
                       (paf->type == gsn_poison) ||
                       (paf->type == gsn_sleep) ||
                       (paf->type == gsn_curse) ||
                       (paf->type == gsn_shielding) ||
                       (paf->type == gsn_deafen) ||
                       (paf->type == gsn_web) ||
                       (paf->type == gsn_nerve) ||
                       (paf->type == sn_lookup ("force grip")) ||
                       (paf->type == sn_lookup ("negative luck")) ||
                       (paf->type == gsn_strangle) ||
                       (paf->type == gsn_blackjack) ||
                       (paf->type == sn_lookup("vampiric touch")) ||
                       (paf->type == sn_lookup("chill touch")) ||
                       (paf->type == gsn_bleed) ||
                       (paf->type == sn_lookup ("bleed")) ||
                       (paf->type == sn_lookup ("garble")) ||
                       (paf->type == gsn_blindness) ||
                       (paf->type == gsn_critical) ||
                       (paf->type == gsn_dirt) ||
                       (paf->type == sn_lookup ("fire breath")) ||
                       (paf->type == gsn_chill_touch));

        if (paf->duration >= -1 && need_remove) 
        {
            dispel_message(ch, paf->type, FALSE);
            affect_remove(ch, paf);
        }
    }

    // wait/daze removing
    if (!IS_NPC(ch))
    {
        if (ch->daze > 0)
            ch->daze = 0;
        if (ch->wait > 0)
            ch->wait = 0;
    }
}

//-------------------------------------------------------------------------------
// function checks whether ch is in immortal's or rulers' prison cell
// (e.g. ch can join to the warfare)
//-------------------------------------------------------------------------------
static bool is_in_prison (CHAR_DATA * ch)
{
    if (!ch || !ch->in_room)
        return TRUE;
    else if ((ch->in_room->vnum == ROOM_VNUM_RULER_CELL) ||
        (ch->in_room->vnum == VNUM_PRISON_CELL1) ||
        (ch->in_room->vnum == VNUM_PRISON_CELL2) ||
        (ch->in_room->vnum == VNUM_PRISON_CELL3))

        return TRUE;
    else // (all is OK) 
        return FALSE;
}

//-------------------------------------------------------------------------------
// duels
//-------------------------------------------------------------------------------
// local functions for do_duel command
static void duel_help     ( CHAR_DATA *ch, const char *argument ); 
static void duel_provoke  ( CHAR_DATA *ch, const char *argument ); 
static void duel_accept   ( CHAR_DATA *ch, const char *argument ); 
static void duel_list     ( CHAR_DATA *ch, const char *argument ); 
static void duel_cancel   ( CHAR_DATA *ch, const char *argument ); 

static CMD_DATA duel_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra      
    { "help",     duel_help,        POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "provoke",  duel_provoke,     POS_STANDING, 10,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "accept",   duel_accept,      POS_STANDING, 10,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "list",     duel_list,        POS_DEAD,     1,              0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "cancel",   duel_cancel,      POS_DEAD,     10,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "surrender",war_surrender,    POS_DEAD,     10,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "allowed",  war_allowed,      POS_DEAD,     10,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "lock",     war_lock,         POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { "unlock",   war_unlock,       POS_DEAD,     93,             0,  0,   CONQ_ALL,        CONQCMD_KEEP_HIDE},
    { NULL }
};


//-------------------------------------------------------------------------------
// function called from interpreter
//-------------------------------------------------------------------------------
DO_FUN (do_duel)
{
    // this stuff is from conquer.c
    if (!parse_command(ch, argument, duel_cmd_table))
    {
        show_command_list (ch, duel_cmd_table);
        char_act (".", ch);
        char_act ("Use {CHELP DUEL{x for more information.", ch);
    }
}

static void duel_help     ( CHAR_DATA *ch, const char *argument )
{
    do_help (ch, "'1.DUEL'");
}

static void duel_provoke  ( CHAR_DATA *ch, const char *argument )
{
    int          area_count,   // count of free areas found
                 min, 
                 max;
    AREA_DATA *  free_areas[WAR_MAX_AREA_COUNT]; // list of "free" areas with WAR_ENABLED flag
    WAR_DATA  *  war;
    AREA_DATA *  curr_area;
    bool         can_start;
    WAR_PLAYER * war_player;

    if (IS_NPC (ch))
        return;

    if (!can_join_war (ch, TRUE))
        return; 

    if (ch->in_war)
    {
        char_act ("But you are already involved into warfare!", ch);
        return;
    }

    if (ch->last_duel_time)
    {
        char_printf (ch, "You should wait %d ticks to next duel.\n", ch->last_duel_time);
        return;
    }

    // find free area
    area_count = 0;
    for (curr_area = area_first; curr_area; curr_area = curr_area->next)
    {
        if (IS_SET(curr_area->flags, AREA_WAR_ENABLED)
            && !IS_SET(curr_area->flags, AREA_GQACTIVE))
        {
            can_start = TRUE;
            for (war = war_root; war; war = war->next)
                if (war->war_area == curr_area)
                {
                    can_start = FALSE; 
                    break;
                }
            if (can_start)
                free_areas[area_count++] = curr_area;
        }
    }

    if (area_count == 0) 
    {
        if (ch)
            char_act ("There are no free areas to start the duel.", ch);
        return;
    }
    curr_area = free_areas[number_range(0, area_count-1)]; // so: area found

    war = new_war_data();

    war->next = war_root;
    war_root = war;

    min = UMAX (PK_MIN_LEVEL, ch->level - pk_range (ch->level)/2);
    max = UMIN (LEVEL_HERO,   ch->level + pk_range (ch->level)/2);

    war->minlevel = min;
    war->maxlevel = max;
    war->war_area = curr_area;
    war->type = WAR_TYPE_DUEL;

    war->bp_reward = 0;
    war->qp_reward = 10; // to have fun :)
    war->gold_reward = 20;

    SET_BIT(war->flags, WS_PREPARING);
    war->time_to_begin = DUEL_WAIT_TIME; // time to find oponent

    war->time_limit = DUEL_DURATION; // we should limit duel duration to avoid
                                     // duel provoke - and no duel

    war_player = new_war_player ();
    if (!war_player) return;

    war_player->war_char = ch;
    war_player->next = war->player;
    war->player = war_player;
    war->max_fighters = 2;
    ch->war_status = PS_NONE;
    ch->in_war = war;
    war->curr_fighters = 1;

    war_showinfo (NULL, "%s proposes a duel in area %s!", 
         war->player->war_char->name, war->war_area->name);

    ++stat_record.duels_provoked;
    ++ch->duels_provoked;
}

static void duel_accept   ( CHAR_DATA *ch, const char *argument )
{
    WAR_DATA   * war;
    WAR_PLAYER * war_player;
    CHAR_DATA  * mount;

    if (IS_NPC (ch))
        return;

    if (!can_join_war (ch, TRUE))
        return;

    if (ch->in_war)
    {
        char_act ("But you are already involved into warfare!", ch);
        return;
    }

    if (argument[0] == '\0')
    {
        char_act ("Accept duel with whom?", ch);
        return;
    }

    // find opponent
    for (war = war_root; war; war = war->next)
    {
        if (war->type != WAR_TYPE_DUEL)
            continue;

        if (!war->player || !war->player->war_char) //let's be paranoid
            continue;

        if (!str_cmp (war->player->war_char->name, argument))
            break;
    }

    if (!war)
    {
        char_act ("No duels provoked by this person found.", ch);
        return;
    }

    if (war->curr_fighters >= 2 )
    {
        char_act ("All ranks are full. You are too late.", ch);
        return;
    }

    if (ch->level < war->minlevel)
    {
        char_act ("Sorry, you are too small for this duel.", ch);
        return;
    }
    if (ch->level > war->maxlevel)
    {
        char_act ("Sorry, you are too tough for this duel.", ch);
        return;
    }

    if (!can_join_war (war->player->war_char, FALSE))
    {
        char_act ("Your opponent is too busy now.", ch);
        return;
    }

    // all is ok - join
    // creator
    war_player = war->player;
    war_player->war_char->war_status = PS_ALIVE;

    if (IS_SET(war->flags, WO_USE_PETS)) // transport all charmies to preparation room
        transfer_pets (war_player->war_char, room_preparation);
    
    if (war_player->war_char->fighting != NULL)
        stop_fighting(war_player->war_char, TRUE);

    act("$n leaves in {Rred{x flash of light!",war_player->war_char,NULL,NULL,TO_ROOM);
    mount = MOUNTED (war_player->war_char);
    char_from_room (war_player->war_char);
    char_to_room (war_player->war_char, room_preparation);
    do_look (war_player->war_char, "auto");
    transfer_mount (war_player->war_char, mount);
    act("$n arrives in {Rred{x flash of light!",war_player->war_char,NULL,NULL,TO_ROOM);

    char_act ("\nThe {CImmortals{x allow you to fight in this war.\n", war_player->war_char);

    war_info (war_player->war_char, "auto");

    // opponent
    war_player = new_war_player ();
    if (!war_player) return;

    war_player->war_char = ch;
    war->player->next = war_player;
    war_player->next = NULL;
    ch->war_status = PS_ALIVE;

    if (IS_SET(war->flags, WO_USE_PETS)) // transport all charmies to preparation room
        transfer_pets (ch, room_preparation);
    
    if (ch->fighting != NULL)
        stop_fighting(ch, TRUE);

    act("$n leaves in {Rred{x flash of light!",ch,NULL,NULL,TO_ROOM);
    mount = MOUNTED (ch);
    char_from_room (ch);
    char_to_room (ch, room_preparation);
    do_look (ch, "auto");
    transfer_mount (ch, mount);
    act("$n arrives in {Rred{x flash of light!",ch,NULL,NULL,TO_ROOM);

    char_act ("\nThe {CImmortals{x allow you to fight in this war.\n", ch);

    ch->in_war = war;
    war->curr_fighters++;

    war_info (ch, "auto");

    war_showinfo (NULL, "%s accepts duel vs %s! Good luck!", 
        ch->name, war->player->war_char->name);

    war->time_to_begin = DUEL_PREPARATION_TIME;

    ++stat_record.duels_started;
    ++ch->duels_joined;
}

static void duel_list     ( CHAR_DATA *ch, const char *argument )
{
    war_list (ch, "duel");
}

static void duel_cancel   ( CHAR_DATA *ch, const char *argument )
{
    // immortals can't stop a duel this way
    if (IS_IMMORTAL (ch))
    {
        char_act ("You can't stop a duel this way. Use 'warfare cancel <number>' instead.", ch);
        return;
    }

    if (!ch->in_war)
    {
        char_act ("Duel? What a duel do you mean?", ch);
        return;
    }

    if (ch->in_war->type != WAR_TYPE_DUEL)
    {
        char_act ("No, no, no! It's not a duel.", ch);
        return;
    }

    // ONLY creator can stop a duel, but ONLY during preparation
    if (ch->in_war->player->war_char != ch)
    {
        char_printf (ch, "No, no, no! Only %s can cancel this duel. You should wait a bit and then surrender.\n", ch->in_war->player->war_char->name);
        return;
    }
    if (!IS_SET (ch->in_war->flags, WS_PREPARING))
    {
        char_act ("You can cancel a duel ONLY during preparation time.", ch);
        return;
    }

    // all is OK. We can cancel it. Message will be formed in war_cancel
    SET_BIT (ch->in_war->flags, WS_NEED_CANCEL);
    if (ch->in_war->player->next)
    {
        war_showinfo (NULL, "%s doesn't want to fight vs %s. Duel is cancelled.",
            ch->in_war->player->war_char->name, ch->in_war->player->next->war_char->name);
        char_printf (ch, "You cancel your duel vs %s.\n", ch->in_war->player->next->war_char->name);
    }
    else
    {
        char_printf (ch, "You cancel your duel.\n");
        war_showinfo (NULL, "%s doesn't want to fight. Duel is cancelled.",
            ch->in_war->player->war_char->name);
    }

    war_cancel (NULL, "auto");
    ch->last_duel_time = DUEL_INTERVAL;
}

static void end_duel ( WAR_DATA * war )
{
    CHAR_DATA * winner, * loser;

    if (war->player->war_char->war_status == PS_ALIVE)
    {
        winner = war->player->war_char;
        loser = war->player->next->war_char;
    }
    else
    {
        winner = war->player->next->war_char;
        loser = war->player->war_char;
    }

    winner->last_duel_time = DUEL_INTERVAL;
    loser->last_duel_time = DUEL_INTERVAL;

    winner->war_status = PS_LIVE_WINNER;

    war_showinfo (NULL, "In the duel in the area %s %s beats %s!",
        war->war_area->name, winner->name, loser->name);

    clear_bad_affects (winner);

    // small chance to receive bp during duel
    if (number_percent () < 6)
        war->bp_reward += 1;

    winner->pcdata->bonuspoints += war->bp_reward;
    winner->pcdata->bp_earned += war->bp_reward;
    winner->pcdata->questpoints += war->qp_reward;
    winner->pcdata->qp_earned += war->qp_reward;
    winner->gold += war->gold_reward;

    char_act ("\n{CCongratulations!{x You are victorious in this duel!", winner);

    char_printf (winner, "\nYou receive %d bp, %d qp and %d gold pieces as reward.\n",
        war->bp_reward, war->qp_reward, war->gold_reward);

    ++winner->duels_win;

    SET_BIT (war->flags, WS_NEED_CANCEL);
    war_cancel (NULL, "auto");
}

static bool can_join_war (CHAR_DATA * ch, bool not_silent)
{
    if (IS_SET (ch->comm2, COMM2_NOWAR))
    {
        if (not_silent)
            char_act ("The Immortals don't want you to fight.", ch);
        return FALSE;
    }

    if (ch->fighting)
    {
        if (not_silent)
            char_act ("You are fighting. You can't make so serious decision.", ch);
        return FALSE;
    }

    if (IS_AFFECTED(ch, AFF_SLEEP))
    {
        if (not_silent)
            char_act ("Do you want to fight in your dreams?", ch);
        return FALSE;
    }

    if (IS_SET (ch->plr_flags, PLR_GHOST))
    {
        if (not_silent)
            char_act ("You. Are. GHOST! And how do you want to fight?", ch);
        return FALSE;
    }

    if (IS_AFFECTED(ch, AFF_CHARM))
    {
        if (not_silent)
            char_act ("You must ask your beloved master first.", ch);
        return FALSE;
    }

    if (IS_AFFECTED(ch,AFF_FEAR))
    {
        if (not_silent)
            char_act("The war is very dangerous stuff. You are too scared to join it.", ch);
        return FALSE;
    }

    if (is_in_prison (ch))
    {
        if (not_silent)
            char_act ("In this somber place you can't even think about war!", ch);
        return FALSE;
    }

    // all is OK
    return TRUE;
}

void transfer_mount (CHAR_DATA * ch, CHAR_DATA  * mount)
{
    if (mount)
    {
        char_from_room(mount);
        char_to_room(mount, ch->in_room);
        ch->riding = TRUE;
        mount->riding = TRUE;
    }
}
