/* $Id: wquest.c,v 1.666 2004/09/20 10:49:54 shrike Exp $                            */

/************************************************************************************
 *    Copyright 2004 Astrum Metaphora consortium                                    *
 *                                                                                  *
 *    Licensed under the Apache License, Version 2.0 (the "License");               *
 *    you may not use this file except in compliance with the License.              *
 *    You may obtain a copy of the License at                                       *
 *                                                                                  *
 *    http://www.apache.org/licenses/LICENSE-2.0                                    *
 *                                                                                  *
 *    Unless required by applicable law or agreed to in writing, software           *
 *    distributed under the License is distributed on an "AS IS" BASIS,             *
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.      *
 *    See the License for the specific language governing permissions and           *
 *    limitations under the License.                                                *
 *                                                                                  *
 ************************************************************************************/

// Automated quest with windows. Idea by Ambulant.
// Part of this code uses global quests' code

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "merc.h"
#include "db/db.h"
#include "wquest.h"
#include "conquer.h"

extern STAT_DATA stat_record;
extern CHAR_DATA *questor_lookup(CHAR_DATA *ch);
extern void gain_exp(CHAR_DATA *ch, int gain);

#define ROOMS_RANGE number_range (7,15)

void wq_update (void);
void wq_quit (CHAR_DATA * ch);

static void wquest_help ( CHAR_DATA *ch, const char *argument ); 
static void wquest_info ( CHAR_DATA *ch, const char *argument );
static void wquest_list ( CHAR_DATA *ch, const char *argument );
static void wquest_request ( CHAR_DATA *ch, const char *argument );
static void wquest_join ( CHAR_DATA *ch, const char *argument );
static void wquest_complete ( CHAR_DATA *ch, const char *argument );
static void wquest_cancel ( CHAR_DATA *ch, const char *argument );
static void wquest_progress ( CHAR_DATA *ch, const char *argument );
static void wquest_hint ( CHAR_DATA *ch, const char *argument );
static void wquest_enable ( CHAR_DATA *ch, const char *argument );
static void wquest_disable ( CHAR_DATA *ch, const char *argument );
static void wquest_stat ( CHAR_DATA *ch, const char *argument );
static void wquest_start ( CHAR_DATA *ch, const char *argument );
static void wquest_stop ( CHAR_DATA *ch, const char *argument );

static WQ_QUESTDATA * wq_generate (int max_level, int count);
static void wq_freeplr (WQ_QUESTPLR* data);
static void wq_freequest (WQ_QUESTDATA* data);
static WQ_QUESTPLR * wq_newplayer (WQ_QUESTDATA * quest, CHAR_DATA * ch);
static void wq_showinfo (WQ_QUESTDATA * wq, const char* fmt, ...);

static WQ_MASTER s_wq_master ; // keep master information for WQ system

static CMD_DATA wquest_cmd_table[] = {
//    name        do_fn             min_pos       min_level       qp  gold min_clanstatus  extra      
    { "help",           wquest_help,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",        wquest_help,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "info",           wquest_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",           wquest_info,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "list",           wquest_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",         wquest_list,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "request",        wquest_request,   POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",       wquest_request,   POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "join",           wquest_join,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "", wquest_join,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "hint",           wquest_hint,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",      wquest_hint,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "complete",       wquest_complete,  POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",       wquest_complete,  POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "cancel",         wquest_cancel,    POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",         wquest_cancel,    POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "progress",       wquest_progress,  POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",       wquest_progress,  POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "enable",         wquest_enable,    POS_DEAD,     93,             0,  0,   CONQ_ALL,       0},
    { "",      wquest_enable,    POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "disable",        wquest_disable,   POS_DEAD,     93,             0,  0,   CONQ_ALL,       0},
    { "",      wquest_disable,   POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "stat",           wquest_stat,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       0},
    { "",     wquest_stat,      POS_DEAD,     1,              0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "start",          wquest_start,     POS_DEAD,     93,             0,  0,   CONQ_ALL,       0},
    { "",          wquest_start,     POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { "stop",           wquest_stop,      POS_DEAD,     93,             0,  0,   CONQ_ALL,       0},
    { "",           wquest_stop,      POS_DEAD,     93,             0,  0,   CONQ_ALL,       CONQCMD_RUS},
    { NULL }
};

// ----------------------------------------------------------------------
// initializer
// ----------------------------------------------------------------------
void wq_init (bool enable)
{
    s_wq_master.enable     = enable ;
    s_wq_master.quests     = NULL   ;
    s_wq_master.ticks_next = number_range (50, 60) ;
}

//-------------------------------------------------------------------------------
// function called from interpreter
//-------------------------------------------------------------------------------
DO_FUN (do_wquest)
{
    if (!ch || !ch->in_room)
        return;

    if (IS_NPC (ch))
        return;

    // this stuff is from conquer.c
    if (!parse_command(ch, argument, wquest_cmd_table))
    {
        show_command_list (ch, wquest_cmd_table);
        char_act (".", ch);
        char_act ("Use {CHELP WQUEST{x for more information.", ch);
    }
}

//--------------------------------------------------------------------------------
// Just help
//--------------------------------------------------------------------------------
static void wquest_help ( CHAR_DATA *ch, const char *argument )
{
    do_help (ch, "1.WQUEST");
}

//--------------------------------------------------------------------------------
// Detailed WQ information
// Immortals can select any quest
//--------------------------------------------------------------------------------
static void wquest_info ( CHAR_DATA *ch, const char *argument )
{
    BUFFER       * output;
    WQ_QUESTPLR  * plr = NULL, 
                 * curr_plr;
    int            num, i;
    WQ_QUESTDATA * quest = NULL;
    CHAR_DATA    * victim;
    OBJ_DATA     * win;
    ROOM_INDEX_DATA * finish;

    num      = atoi (argument);
    victim = ch;

    if ((argument[0] == '\0' || num <= 0) && !IS_IMMORTAL (ch))
    {
        quest = wq_findquest (ch);
        if (quest == NULL)
        {
            char_act ("But you have no quest.", ch);
            return;
        }
        victim = ch;
        plr = wq_findplr (ch);
        if (plr == NULL) // to avoid bugs
            return;

        if (plr->visited == -1)
        {
            char_act ("But you have cancelled this quest.", ch);
            return;
        }
    }
    // multiplayer wq by number
    else if (num > 0)
    {
        int skip = num - 1;
        for (quest = s_wq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip);

        if (quest == NULL)
        {
            char_act ("No such quest.", ch);
            wquest_list (ch, str_empty);
            return;
        }
        victim = NULL;
    }
    // multiplayer or personal wq by name
    if (IS_IMMORTAL (ch) && quest == NULL)
    {
        if (argument[0] == '\0')
            victim = ch;
        else if ((victim = gq_findchar (argument)) == NULL || IS_NPC (victim))
        {
            char_act("No chars with such name.", ch);
            return;
        }
        quest = wq_findquest (victim);
        if (quest == NULL)
        {
            char_act("This char has no quest.", ch);
            return;
        }
        plr = wq_findplr (victim);
        if (plr == NULL) // to avoid bugs
            return;

        if (plr->visited == -1)
        {
            char_act ("This char have cancelled the quest.", ch);
            return;
        }
    }

    if (quest == NULL) 
        return;

    plr = wq_findplr (victim);

    output = buf_new (ch->lang);
    buf_add (output, "Window quest\n");
    switch (quest->type)
    {
    case WQ_TYPE_PERSONAL:
        buf_add (output, "Type         : personal\n");
        break;
    case WQ_TYPE_MULTIPLAYER:
        buf_add (output, "Type         : multiplayer\n");
        break;
    }
    buf_printf (output, "Levels       : %d-%d\n", quest->level_min, quest->level_max);
    if (plr == NULL || plr->visited == 0)
        buf_printf (output, "First place  : %s (area %s)\n", 
                            mlstr_cval(((OBJ_DATA *)quest->windows[0])->in_room->name, ch),
                            ((OBJ_DATA *)quest->windows[0])->in_room->area->name);
    else
        buf_printf (output, "Current place  : %s (area %s)\n", 
                            mlstr_cval(((OBJ_DATA *)quest->windows[plr->visited-1])->in_room->name, ch),
                            ((OBJ_DATA *)quest->windows[plr->visited-1])->in_room->area->name);
    buf_printf (output, "Place count  : %d\n", quest->room_count);

    if (plr != NULL)
        buf_printf (output, "Places found : %d\n", plr->visited);

    buf_printf (output, "Time left    : %d\n", quest->ticks_left);

    if (quest->type != WQ_TYPE_PERSONAL)
    {
        buf_add (output, "Players      : ");
        for (curr_plr = quest->plr; curr_plr != NULL; curr_plr = curr_plr->next)
        {
            if (curr_plr->visited == -1)
                continue;

            buf_printf (output, "%s", curr_plr->name);
            if (curr_plr->next != NULL)
                buf_add (output, ", ");
        }
    }
    buf_add (output, "\n");
    // room info for immortals
    if (IS_IMMORTAL (ch))
    {
        buf_add (output, "Rooms:\n");
        for (i = 0; i < quest->room_count; ++i)
        {
            win = (OBJ_DATA *)quest->windows[i];
            buf_printf (output, "%s [%d], area %s\n", 
                mlstr_cval (win->in_room->name, ch), 
                win->in_room->vnum, 
                win->in_room->area->name);
        }
        finish = get_room_index (((OBJ_DATA *)quest->windows[quest->room_count-1])->value[0]);
        if (finish != NULL)
            buf_printf (output, "Final room: %s [%d], area %s\n", 
                mlstr_cval (finish->name, ch), 
                finish->vnum, 
                finish->area->name);
    }

    char_act(buf_string(output), ch);
    buf_free(output);
}

// ----------------------------------------------------------------------
// list currently active multiplayer WQs
// Immortals can also see personal WQs
// ----------------------------------------------------------------------
static void wquest_list ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTDATA    * quest;
    BUFFER          * output1, * output2;
    int               num = 0, num_personal;
    DESCRIPTOR_DATA * d;

    if (s_wq_master.quests == NULL && !IS_IMMORTAL (ch))
    {
        char_act ("There are no such quests at the moment.", ch);
        return;
    }

    output1 = buf_new (ch->lang);
    buf_printf (output1, "Current quests:\n");

    for (quest = s_wq_master.quests ; quest ; quest = quest->next)
    {
        buf_printf (output1, "{G%2d{x. Quest with level {W%d{x-{W%d{x areas",
                    ++num, quest->level_min, quest->level_max);
        buf_add (output1, "\n");
    }

    if (num > 0)
        char_act(buf_string (output1), ch);
    buf_free (output1);

    // immortals can see personal wquests
    if (IS_IMMORTAL (ch))
    {
        output2 = buf_new (ch->lang);

        num_personal = 0;
        buf_add (output2, "Personal quests for:\n");
        for (d = descriptor_list; d != NULL; d = d->next)
        {
            if ((d->character == NULL) || (d->character->pcdata == NULL) || (d->character->pcdata->wquest == NULL))
                continue;

            if (d->character->pcdata->wquest->type != WQ_TYPE_PERSONAL)
                continue;
            
            buf_printf (output2, "{G%2d{x. %s\n", ++num_personal, d->character->name);
        }
        if (num_personal > 0)
            char_act(buf_string (output2), ch);
        else if (num == 0)
            char_act ("There are no such quests at the moment.", ch);
        buf_free (output2);
    }
}

//--------------------------------------------------------------------------------
// char can request PERSONAL wquest from questor
//--------------------------------------------------------------------------------
static void wquest_request ( CHAR_DATA *ch, const char *argument )
{
    CHAR_DATA * questor;
    WQ_QUESTDATA * quest;

    if (!s_wq_master.enable)
    {
        char_act ("This type of quests is disabled now.", ch); 
        return;
    }

    quest = wq_findquest (ch);
    if (quest != NULL) 
    {
        char_act ("But you are already on a quest!", ch); 
        return;
    }

    if (ch->pcdata->wq_next_time > 0)
    {
        char_printf (ch, "You should wait %d ticks to next quest.\n", ch->pcdata->wq_next_time); 
        return;
    }

    if ((questor = questor_lookup(ch)) == NULL)
        return;

    act("$n asks $N for a quest.", ch, NULL, questor, TO_ROOM); 
    act_puts("You ask $N for a quest.", ch, NULL, questor, TO_CHAR, POS_DEAD); 

    ch->pcdata->wquest = wq_generate (ch->level + ch->level/5, ROOMS_RANGE);
    if (ch->pcdata->wquest != NULL) 
    {
        char_act ("Now try to complete!", ch); 
        ch->pcdata->wquest->type = WQ_TYPE_PERSONAL;
        ch->pcdata->wquest->ticks_start = 0;
        wq_newplayer (ch->pcdata->wquest, ch);
        wquest_info (ch, str_empty);
        ++ch->pcdata->wq_personal_join;
        ++stat_record.wq_pers_started;
    }
    else
    {
        char_act ("Sorry, I can't find a quest for you...", ch);
        return;
    }
}

//--------------------------------------------------------------------------------
// Join to a WQ. Char can join ANY multiplayer quest... to his own risque
//--------------------------------------------------------------------------------
static void wquest_join ( CHAR_DATA *ch, const char *argument )
{
    int            num;
    WQ_QUESTDATA * quest;

    quest = wq_findquest (ch);

    if (quest != NULL) 
    {
        char_act ("But you are already on a quest!", ch); 
        return;
    }

    num = atoi (argument);

    if (argument[0] == '\0' || num <= 0)
    {
        quest = s_wq_master.quests;
    }
    // multiplayer wq by number
    else if (num > 0)
    {
        int skip = num - 1;
        for (quest = s_wq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip);
    }
    if (quest == NULL)
    {
        char_act ("No such quest.", ch);
        wquest_list (ch, str_empty);
        return;
    }
    char_act ("Now try to complete!", ch); 
    wq_newplayer (quest, ch);
    wquest_info (ch, str_empty);
    ++ch->pcdata->wq_mult_join;
}

//--------------------------------------------------------------------------------
// Completion report. Char must be in the room, where points the last window
//--------------------------------------------------------------------------------
static void wquest_complete ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTPLR  * plr;
    int            qp_reward = 0, bp_reward = 0, gold_reward = 0, exp_reward = 0;
    WQ_QUESTDATA * quest;

    plr = wq_findplr (ch);
    quest = wq_findquest (ch);

    if ((quest == NULL) || (plr == NULL)) 
    {
        char_act ("But you are not in the quest!", ch); 
        return;
    }
    if (plr->visited < quest->room_count)
    {
        act ("Try to find ALL places to complete this quest.", ch, NULL, NULL, TO_CHAR); 
        return;
    }
    if (ch->in_room->vnum != ((OBJ_DATA *) quest->windows[plr->visited-1])->value[0])
    {
        char_act ("You should go to the place you've seen in the last window to complete this quest.", ch); 
        return;
    }
    // all is OK - we can grant small reward to the winner
    char_act ("Congratulations! Your quest is completed!", ch);
    if (quest->type != WQ_TYPE_PERSONAL)
        wq_showinfo (NULL, "%s has completed {WWQuest{x!", ch->name);

    switch (quest->type)
    {
    case WQ_TYPE_PERSONAL:
        qp_reward = number_range (plr->visited * 3, plr->visited * 7);
        gold_reward = number_range (plr->visited, plr->visited * 3);
        bp_reward = 0;
        ++ch->pcdata->wq_personal_win;
        ++stat_record.wq_pers_win;
        exp_reward = number_range (LVL(ch), LVL (ch) * (plr->visited));
        gain_exp (ch, exp_reward) ;
        act ("You receive {R$j{x experience points!", ch, (const void *) exp_reward, NULL, TO_CHAR);
        break;
    case WQ_TYPE_MULTIPLAYER:
        qp_reward = number_range (plr->visited * 6, plr->visited * 8);
        gold_reward = number_range (plr->visited, plr->visited * 7);
        if (number_percent () < 50)
            bp_reward = number_range (1,2);
        ++ch->pcdata->wq_mult_win;
        ++stat_record.wq_mult_win;
        exp_reward = number_range (LVL(ch), LVL (ch) * (plr->visited)+2);
        gain_exp (ch, exp_reward) ;
        act ("You receive {R$j{x experience points!", ch, (const void *) exp_reward, NULL, TO_CHAR);
        break;
    }

    ch->pcdata->questpoints += qp_reward;
    ch->pcdata->qp_earned += qp_reward;
    ch->pcdata->bonuspoints += bp_reward;
    ch->pcdata->bp_earned += bp_reward;
    ch->gold += gold_reward;
    if (bp_reward > 0) 
        char_printf (ch, "You receive {C%d{x bp, {C%d{x qp and %d {Ygold{x!\n\n",
            bp_reward, qp_reward, gold_reward);
    else
        char_printf (ch, "You receive {C%d{x qp and %d {Ygold{x!\n\n",
            qp_reward, gold_reward);

    wq_freequest (quest);
}

//--------------------------------------------------------------------------------
// Cancel quest.
// Personal WQ: we should set a timer
// Multiplayer WQ: we shouldn't free plr record so char cant't join another WQ
//--------------------------------------------------------------------------------
static void wquest_cancel ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTDATA * quest;
    WQ_QUESTPLR  * plr;

    quest = wq_findquest (ch);
    plr = wq_findplr (ch);
    if (quest == NULL) 
    {
        char_act ("But you are not in the quest!", ch); 
        return;
    }

    if (plr->visited == -1)
    {
        char_act ("But you have already cancelled this quest!", ch); 
        return;
    }

    switch (quest->type)
    {
    case WQ_TYPE_PERSONAL:
        wq_freequest (ch->pcdata->wquest);
        break;
    case WQ_TYPE_MULTIPLAYER:
        wq_findplr (ch)->visited = -1;
        break;
    }
    ch->pcdata->wq_next_time = 30;
    char_act ("You cancel your quest.", ch);
}

//--------------------------------------------------------------------------------
// Enable WQs. todo: save state into system.conf
//--------------------------------------------------------------------------------
static void wquest_enable ( CHAR_DATA *ch, const char *argument )
{
    if (s_wq_master.enable)
        char_act ("This type of quests is already enabled.", ch); 
    else
    {
        s_wq_master.enable = TRUE;
        char_act ("You enable WQs.", ch); 
    }
}

//--------------------------------------------------------------------------------
// Disable WQs. todo: save state into system.conf
//--------------------------------------------------------------------------------
static void wquest_disable ( CHAR_DATA *ch, const char *argument )
{
    if (!s_wq_master.enable)
        char_act ("This type of quests is already disabled.", ch); 
    else
    {
        s_wq_master.enable = FALSE;
        char_act ("You disable WQs.", ch); 
    }
}

//--------------------------------------------------------------------------------
// char's WQ statistics 
//--------------------------------------------------------------------------------
static void wquest_stat ( CHAR_DATA *ch, const char *argument )
{
    char arg[MAX_INPUT_LENGTH];
    BUFFER *output = NULL;
    CHAR_DATA *wch = NULL;
    bool loaded = FALSE;
    int percent;

    argument = one_argument(argument, arg, sizeof(arg));
    if (arg[0] == '\0')
    {
        char_act("You must provide a name.", ch);
        return;
    }

    if (!str_cmp("self", arg) || !str_cmp(arg, "") || !str_cmp(arg, ""))
        wch = ch;
    else
        wch = gq_findchar(arg);

    if (wch == NULL)
    {
        loaded = TRUE;
        wch = char_load_special(arg);
    }

    if (wch == NULL)
    {
        char_act("No chars with such name.", ch);
        return;
    }

    if (!str_cmp (argument, "clear"))
    {
        wch->pcdata->wq_mult_join = 0;
        wch->pcdata->wq_mult_win = 0;
        wch->pcdata->wq_personal_join = 0;
        wch->pcdata->wq_personal_win = 0;
        char_act ("Ok.", ch);

        if (loaded)
            char_nuke(wch);
        return;
    }

    if (output == NULL)
        output = buf_new(ch->lang);

    buf_add(output,  "\n    {G/~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\\{x\n");
    buf_printf(output, "{x    {G|                    {CWQuest info for {W%-20s          {G|{x\n", wch->name);
    buf_add(output,    "{x    {G|------------------------------------------------------------------|{x\n");
    
    buf_add(output,    "{x    {G|                       {RJoined{x           {CWon{G           {c% won{G       |{x\n");

    // Multiplayer
    if (wch->pcdata->wq_mult_join)
        percent = wch->pcdata->wq_mult_win * 100 / wch->pcdata->wq_mult_join;
    else
        percent = 0;
    buf_printf(output, "{x    {G|     Multiplayer :    {W[%6d]{x        {W[%6d]{G       {W[%6d]{G     |{x\n", 
        wch->pcdata->wq_mult_join, wch->pcdata->wq_mult_win, percent);

    // Personal
    if (wch->pcdata->wq_personal_join)
        percent = wch->pcdata->wq_personal_win * 100 / wch->pcdata->wq_personal_join;
    else
        percent = 0;
    buf_printf(output, "{x    {G|     Personal    :    {W[%6d]{x        {W[%6d]{G       {W[%6d]{G     |{x\n", 
        wch->pcdata->wq_personal_join, wch->pcdata->wq_personal_win, percent);


    buf_add(output,    "{x    {G|__________________________________________________________________|{x\n");
    buf_add(output, "{x");

    page_to_char(buf_string(output), ch);
    buf_free(output);

    if (loaded)
        char_nuke(wch);   
}

//--------------------------------------------------------------------------------
// Start multiplayer WQ (immortals' command)
//--------------------------------------------------------------------------------
static void wquest_start ( CHAR_DATA *ch, const char *argument )
{
    int            level, 
                   count;
    WQ_QUESTDATA * quest;
    static char    arg [MAX_INPUT_LENGTH];

    // manual
    if (str_cmp (argument, "auto"))
    {
        argument = one_argument (argument, arg, sizeof (arg));
        level    = atoi (arg);
        argument = one_argument (argument, arg, sizeof (arg));
        count    = atoi (arg);
        if (level == 0)
            level = LEVEL_HERO;
        if (count == 0)
            count = ROOMS_RANGE;
    }
    // auto
    else
    {
        level = number_range (10, LEVEL_HERO);
        count = ROOMS_RANGE;
    }
    quest = wq_generate (level, count);
    if (quest == NULL)
        return;
    quest->next = s_wq_master.quests;
    s_wq_master.quests = quest;
    wq_showinfo (NULL, "WQuest in level %d-%d areas will start in %d ticks.", 
        quest->level_min, quest->level_max, quest->ticks_start);
    ++stat_record.wq_mult_started;
}

//--------------------------------------------------------------------------------
// Start multiplayer (by number) or personal (by name) WQ
//--------------------------------------------------------------------------------
static void wquest_stop ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTDATA * quest;
    int            num;
    static char    arg [MAX_INPUT_LENGTH] ;

    argument = one_argument (argument, arg, sizeof (arg));
    num      = atoi (arg);

    if (arg[0] == '\0' || num <= 0)
    {
        char_act ("You must specify a number.", ch);
        wquest_list (ch, str_empty) ;
        return ;
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_wq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_act ("No such quest.", ch);
            wquest_list (ch, str_empty);
            return ;
        }
    }

    wq_showinfo (NULL, "WQuest has been cancelled by an {CImmortal{x.", ch->name);
    wq_freequest (quest) ;
}

// ----------------------------------------------------------------------
// comparator for qsort used in progress
// ----------------------------------------------------------------------
__inline int compare_wq_plr (const void* p, const void* q)
{
    return ((* (WQ_QUESTPLR**) q)->visited - (* (WQ_QUESTPLR**) p)->visited) ;
}

//--------------------------------------------------------------------------------
// Players' progress in the multiplayer WQ
//--------------------------------------------------------------------------------
static void wquest_progress ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTDATA * quest;
    WQ_QUESTPLR  * plr;
    BUFFER*        output;
    int            num, i;
    WQ_QUESTPLR ** sorted;

    static char arg [MAX_INPUT_LENGTH];

    argument = one_argument (argument, arg, sizeof (arg));
    num      = atoi (arg);

    if (arg[0] == '\0' || num <= 0)
    {
        // try the case when player already has a WQ
        if ((quest = ch->pcdata->wquest) == NULL)
        {
            char_printf (ch, "No such quest.\n");
            wquest_list (ch, str_empty);
            return;
        }
    }
    else
    {
        int skip = num - 1 ;
        for (quest = s_wq_master.quests ; quest && skip > 0 ;
             quest = quest->next, --skip) ;

        if (quest == NULL)
        {
            char_printf (ch, "No such quest.\n");
            wquest_list (ch, str_empty) ;
            return ;
        }
    }

    if (quest->plr_count == 0)
    {
        char_printf (ch, "No players in this quest.\n");
        return;
    }

    output = buf_new (ch->lang) ;
    buf_printf (output, "Players sorted by progress:\n");
    
    // allocate data for sorted list
    sorted = (WQ_QUESTPLR**) calloc (quest->plr_count, sizeof (WQ_QUESTPLR*)) ;

    // copy data
    num = 0;
    for (plr = quest->plr ; plr ; plr = plr->next)
    {
        if (plr->visited < 0) continue;
        sorted [num] = plr;
        ++num;
    }

    // sort data
    qsort (sorted, num, sizeof (WQ_QUESTPLR*), compare_wq_plr);

    for (i = 0 ; i < num ; ++i)
        buf_printf (output, "{w%2d. {W%s{x\n", i + 1, capitalize (sorted [i]->name)) ;

    free (sorted);

    char_act(buf_string (output), ch);
    buf_free (output);
}


//--------------------------------------------------------------------------------
// in personal WQ 
//--------------------------------------------------------------------------------
static void wquest_hint ( CHAR_DATA *ch, const char *argument )
{
    WQ_QUESTDATA    * quest;
    WQ_QUESTPLR     * plr; 
    ROOM_INDEX_DATA * room;

    if ((quest = wq_findquest (ch)) == NULL)
    {
        char_act ("But you have no quest.", ch);
        return;
    }
    if (quest->type != WQ_TYPE_PERSONAL)
    {
        char_act ("Hints are only available in personal quests.", ch);
        return;
    }

    if ((plr = wq_findplr (ch)) == NULL || plr->visited == -1) // to avoid bugs
        return;

    if (plr->hints > MAX_HINT - 1)
    {
        char_act ("You have no hints left.", ch);
        return;
    }

    // all is OK, we should show a hint and increase counter
    ++plr->hints;
    if (plr->visited < ch->pcdata->wquest->room_count)
    {
        char_printf (ch, "Next window is in the area named %s.\nHints left: %d.\n",
            ((OBJ_DATA *)quest->windows[plr->visited])->in_room->area->name,
            MAX_HINT - plr->hints);
    }
    else
    {
        room = get_room_index (((OBJ_DATA *) ch->pcdata->wquest->windows[plr->visited-1])->value[0]);
        if (room == NULL)
        {
            char_act ("{RBug!!! Please report it to Immortals!{x", ch);
        }
        char_printf (ch, "You can complete your quest in the area named %s.\n", room->area->name);
    }
}

// ----------------------------------------------------------------------
// update WQ system (called from handler.c)
// ----------------------------------------------------------------------
void wq_update (void)
{
    WQ_QUESTDATA * quest;
    WQ_QUESTDATA * next;
    DESCRIPTOR_DATA * d;

    // first step: update multiplayer WQs
    for (quest = s_wq_master.quests ; quest ; quest = next)
    {
        next = quest->next ;

        if (quest->ticks_start > 0) --quest->ticks_start ;
        else if (quest->ticks_left  > 0) --quest->ticks_left  ;

        if (quest->ticks_start == 0)
        {
            if (quest->plr_count < 1)
            {
                wq_showinfo (NULL, "Nobody wants to join the quest. It's cancelled.");
                wq_freequest (quest) ;
                continue ;
            }

            quest->ticks_start = -1 ;
            continue ;
        }

        if (quest->ticks_left == 0)
        {
            wq_showinfo  (NULL, "The time for WQ is over. Nobody can complete it.") ;
            wq_freequest (quest) ;
            continue ;
        }
        else if (quest->ticks_left < 6)
            wq_showinfo (quest, "Quest will end in {Y%d{w ticks!\n", quest->ticks_left) ;
    }
        
    // second step: update personal WQs
    for (d = descriptor_list; d != NULL; d = d->next)
    {
        if ((d->character == NULL) || (d->character->pcdata == NULL))
            continue;

        if (d->character->pcdata->wq_next_time > 0)
            --d->character->pcdata->wq_next_time;

        quest = d->character->pcdata->wquest;
        if (quest == NULL)
            continue;

        if (quest->type != WQ_TYPE_PERSONAL)
            continue;
        if (--quest->ticks_left <= 0)
        {
            wq_freequest (quest);
            char_act ("The time for your quest is over...", d->character);
        }
    }

    // third step: start autoquest
    if (s_wq_master.ticks_next > 0)
    {
        --s_wq_master.ticks_next;
        if (s_wq_master.ticks_next == 0)
        {
            wquest_start (NULL, "auto") ;
            s_wq_master.ticks_next = number_range (50, 60) ;
        }
    }
}

//--------------------------------------------------------------------------------
// generates new WQ
//
// max_level - areas with area->max_level > max level will not be selected
// count - number of rooms to select
//--------------------------------------------------------------------------------
static WQ_QUESTDATA * wq_generate (int max_level, int count)
{
    int i, curr, try_count_room, try_count_area;
    int min = 999;
    int max = 0;
    AREA_DATA * area;
    ROOM_INDEX_DATA * room; 
    int random_vnum;
    OBJ_INDEX_DATA * win_index;

    // the result will be stored in here
    WQ_QUESTDATA* ptr ;

    if ((win_index = get_obj_index (C_WINDOW_VNUM)) == NULL)
        return NULL;

    if (max_level <= 20)
        max_level = 20;
    if (count < 3)   
        count = 3;
    if (count > 64)  
        count = 64;

    // fill in WQ structure
    ptr = (WQ_QUESTDATA*) calloc (1, sizeof (WQ_QUESTDATA)) ;

    // allocate memory for internal lists
    ptr->windows  = (int*) calloc (count, sizeof (OBJ_DATA*)) ;

    for (curr = 0, try_count_area = 0; curr < (count+1) && try_count_area < 10000; try_count_area++)
    {
        area = area_lookup (number_range (2, area_last->vnum));
        if (IS_SET (area->flags, AREA_NOQUEST | AREA_NOWQUEST) || (area->max_level > max_level))
            continue;

        try_count_room = (area->max_vnum - area->min_vnum) * 10; // let it be
        for (i = 0; i < try_count_room; ++i)
        {
            random_vnum = number_range (area->min_vnum, area->max_vnum);
            if ((room = get_room_index(random_vnum)) != NULL)
            {
                // skip special rooms
                if (IS_SET (room->room_flags, ROOM_GODS_ONLY | ROOM_IMP_ONLY | ROOM_NOEXPLORE) ||
                    !room->exit) 
                    continue;

                // room found
                if (curr < count)
                {
                    ptr->windows[curr] = (int) create_obj (win_index, 0);
                    obj_to_room ( (OBJ_DATA*) ptr->windows[curr], room);
                }
                if (curr > 0)
                    ((OBJ_DATA*) ptr->windows[curr-1])->value[0] = random_vnum;
                ++curr;
                if (area->min_level < min)
                    min = area->min_level;
                if (area->max_level > max)
                    max = area->max_level;
                break;
            }
        }
    }

    if (try_count_area == 10000)
    {
        free (ptr);
        return NULL;
    }

    ptr->room_count = count ;
    ptr->plr = NULL;

    // levels
    ptr->level_min = min ;
    ptr->level_max = max ;

    // timing
    ptr->ticks_left  = number_range (count * 10, count * 12) ;
    //ptr->ticks_left = 2;
    ptr->ticks_total = ptr->ticks_left ;
    ptr->ticks_start = number_range (2, 4) ;

    ptr->type = WQ_TYPE_MULTIPLAYER;

    return ptr  ;
}

// ----------------------------------------------------------------------
// create new quest player
// ----------------------------------------------------------------------
static WQ_QUESTPLR * wq_newplayer (WQ_QUESTDATA * quest, CHAR_DATA * ch)
{
    WQ_QUESTPLR * ptr;
    
    // allocate memory
    ptr = (WQ_QUESTPLR*) calloc (1, sizeof (WQ_QUESTPLR));

    // other
    ptr->quest  = quest;
    ptr->visited = 0;
    ptr->name   = (char*) calloc (strlen (ch->name) + 1, sizeof (char));
    strnzcpy (ptr->name, strlen (ch->name) + 1, ch->name);

    // link to the list
    ptr->next  = quest->plr;
    quest->plr = ptr;

    // counter
    ++quest->plr_count;

    return ptr;
}

// ----------------------------------------------------------------------
// unlink and free GQ structure
// ----------------------------------------------------------------------
static void wq_freequest (WQ_QUESTDATA * data)
{
    WQ_QUESTDATA * quest;
    WQ_QUESTDATA * prev;
    WQ_QUESTPLR  * plr;
    WQ_QUESTPLR  * next;
    CHAR_DATA    * ch;

    int i;

    // find quest in the list and unlink it
    prev = NULL;
    for (quest = s_wq_master.quests ; quest ; prev = quest, quest = quest->next)
    {
        if (quest == data)
        {
            if (quest == s_wq_master.quests)
                s_wq_master.quests = quest->next;
            else 
                prev->next         = quest->next;
            break;
        }
    }

    // free players
    for (plr = data->plr ; plr ; plr = next)
    {
        ch = gq_findchar (plr->name);
        if (ch != NULL)
            ch->pcdata->wquest = NULL;
        next = plr->next;
        wq_freeplr (plr);
    }

    for (i = 0; i < data->room_count; ++i)
        extract_obj ((OBJ_DATA *)data->windows[i]);
    free (data->windows);
    free (data);
}

// ----------------------------------------------------------------------
// resets and free player structure
// ----------------------------------------------------------------------
static void wq_freeplr (WQ_QUESTPLR * data)
{
    free (data->name);
    free (data);
}

// ----------------------------------------------------------------------
// find plr record
// ----------------------------------------------------------------------
WQ_QUESTPLR * wq_findplr (CHAR_DATA* ch)
{
    WQ_QUESTPLR *  plr;
    WQ_QUESTDATA * quest;

    if (ch == NULL)
        return NULL;
    if (ch->pcdata == NULL)
        return NULL;

    if (ch->pcdata->wquest != NULL)
    {
        // use case-insensistive internal function
        for (plr = ch->pcdata->wquest->plr ; plr ; plr = plr->next)
            if (str_cmp (plr->name, ch->name) == 0) 
                return plr;
    }
    else
    {
        for (quest = s_wq_master.quests ; quest ; quest = quest->next)
        {
            // use case-insensistive internal function
            for (plr = quest->plr ; plr ; plr = plr->next)
                if (str_cmp (plr->name, ch->name) == 0) 
                    return plr ;
        }
    }

    // not in wquest
    return NULL ;
}

//-------------------------------------------------------------------------------
// (code from gquest.c)
// wq_showinfo ((WQ_QUESTDATA *) -1, ...) - from immortal to all players in all WQs
// wq_showinfo (wq, ...) - from player to players in the same wq
// wq_showinfo (NULL, ...) - to all connected players
//-------------------------------------------------------------------------------
static void wq_showinfo (WQ_QUESTDATA * wq, const char* fmt, ...)
{
    DESCRIPTOR_DATA * d;
    WQ_QUESTPLR     * player;
    CHAR_DATA       * ch;

    va_list           args;
    static char       buf  [2][MAX_STRING_LENGTH];
    static char       buf2 [2][MAX_STRING_LENGTH];
    
    // form the string
    va_start  (args, fmt) ;
    vsnprintf (buf2[0], sizeof(buf2[0]), GETMSG(fmt,0), args) ;
    vsnprintf (buf2[1], sizeof(buf2[1]), GETMSG(fmt,1), args) ;
    va_end    (args) ;
    snprintf  (buf[0], sizeof(buf[0]), GETMSG("{C[{cWQuest{C]{x: %s\n{x", 0), buf2[0]) ;
    snprintf  (buf[1], sizeof(buf[1]), GETMSG("{C[{cWQuest{C]{x: %s\n{x", 1), buf2[1]) ;

    // message from immortals to all players currently involved in any war
    if (wq == (WQ_QUESTDATA *) -1)
    {
        for (d = descriptor_list ; d ; d = d->next)
            if ((d->connected == CON_PLAYING) && d->character->pcdata->wquest)
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
                else
                    char_printf (d->character, "%s", buf[d->character->lang]) ;
            }
    }
    else if (wq != NULL)
    {
        for (player = wq->plr ; player ; player = player->next)
        {
            ch = gq_findchar (player->name);
            if (ch != NULL)
            {
                if (IS_SET(ch->comm, COMM_QUIET_EDITOR)
                    &&  ch->desc->pString)
                        buf_add(ch->pcdata->buffer, buf[ch->lang]);
                else
                    char_printf (ch, "%s", buf[ch->lang]) ;
            }
        }
    }
    // message for everyone
    else
    {
        for (d = descriptor_list ; d ; d = d->next)
            if (d->connected == CON_PLAYING) //&& IS_SET(d->character->comm,COMM_WARINFO)) 
            {
                if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                    &&  d->character->desc->pString)
                        buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
                else
                    char_printf (d->character, "%s", buf[d->character->lang]) ;
            }
        return ;
    }

    // immortals
    for (d = descriptor_list ; d ; d = d->next)
    {
        if (d->connected == CON_PLAYING && IS_IMMORTAL (d->character)) // && IS_SET(d->character->comm,COMM_WARINFO))
        {
            if (IS_SET(d->character->comm, COMM_QUIET_EDITOR)
                &&  d->character->desc->pString)
                    buf_add(d->character->pcdata->buffer, buf[d->character->lang]);
            else
                char_printf (d->character, "%s", buf[d->character->lang]) ;
        }
    }
}

void wq_quit (CHAR_DATA * ch)
{
    wquest_cancel (ch, str_empty);
}

WQ_QUESTDATA * wq_findquest (CHAR_DATA* ch)
{
    WQ_QUESTPLR * plr;

    if (ch->pcdata->wquest != NULL)
        return ch->pcdata->wquest;

    plr = wq_findplr (ch);

    if (plr)
        return plr->quest;
    else
        return NULL;
}
